"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_apigateway_1 = require("monocdk/aws-apigateway");
const aws_wafv2_1 = require("monocdk/aws-wafv2");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../../nag-pack");
/**
 * Rest API stages are associated with AWS WAFv2 web ACLs - (Control ID: 6.6)
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_apigateway_1.CfnStage) {
        const stageLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
        const stageName = nag_pack_1.resolveResourceFromInstrinsic(node, node.stageName);
        const restApiId = nag_pack_1.resolveResourceFromInstrinsic(node, node.restApiId);
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_wafv2_1.CfnWebACLAssociation) {
                if (isMatchingWebACLAssociation(child, stageLogicalId, stageName, restApiId)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to check whether a given Web ACL Association is associated with the given Rest API
 * @param node the CfnWebACLAssociation to check
 * @param stageLogicalId the Cfn Logical ID of the REST API Stage
 * @param stageName the name of the REST API Stage
 * @param restApiId the ID or Cfn Resource ID of the REST API associated with the Stage
 * returns whether the CfnWebACLAssociation is associates with the given Rest API
 */
function isMatchingWebACLAssociation(node, stageLogicalId, stageName, restApiId) {
    const resourceLogicalId = JSON.stringify(monocdk_1.Stack.of(node).resolve(node.resourceArn));
    const regexes = Array();
    regexes.push(`${restApiId}.+${stageLogicalId}(?![\\w])`);
    if (stageName !== undefined) {
        regexes.push(`${restApiId}.+${stageName}(?![\\w])`);
    }
    const regex = new RegExp(regexes.join('|'), 'gm');
    if (regex.test(resourceLogicalId)) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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