"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_wafv2_1 = require("monocdk/aws-wafv2");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../../nag-pack");
/**
 * WAFv2 web ACLs have logging enabled - (Control IDs: AC-4(26), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SI-4(17), SI-7(8))
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_wafv2_1.CfnWebACL) {
        const webAclLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
        const webAclName = monocdk_1.Stack.of(node).resolve(node.name);
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_wafv2_1.CfnLoggingConfiguration) {
                if (isMatchingLoggingConfiguration(child, webAclLogicalId, webAclName)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to check whether the Logging Configuration contains the given Web ACL
 * @param node the CfnLoggingConfiguration to check
 * @param webAclLogicalId the Cfn Logical ID of the Web ACL
 * @param webAclName the name of the Web ACL
 * returns whether the CfnLoggingConfiguration contains the given Web ACL
 */
function isMatchingLoggingConfiguration(node, webAclLogicalId, webAclName) {
    const resourceArn = JSON.stringify(monocdk_1.Stack.of(node).resolve(node.resourceArn));
    if (new RegExp(`${webAclLogicalId}(?![\\w])`).test(resourceArn) ||
        (webAclName != undefined &&
            new RegExp(`webacl\/${webAclName}(?![\\w\\-_\\.])`).test(resourceArn))) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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