"use strict";
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
Object.defineProperty(exports, "__esModule", { value: true });
const aws_secretsmanager_1 = require("monocdk/aws-secretsmanager");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../../nag-pack");
/**
 * Secrets have automatic rotation scheduled - (Control IDs: AC-4, AC-4(22), AC-24(1), AU-9(3), CA-9b, PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SI-1a.2, SI-1a.2, SI-1c.2)
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_secretsmanager_1.CfnSecret) {
        const secretLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
        const secretTargetAttachmentLogicalIds = Array();
        const cfnSecretTargetAttachments = Array();
        const cfnRotationSchedules = Array();
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_secretsmanager_1.CfnSecretTargetAttachment) {
                cfnSecretTargetAttachments.push(child);
            }
            else if (child instanceof aws_secretsmanager_1.CfnRotationSchedule) {
                cfnRotationSchedules.push(child);
            }
        }
        if (cfnRotationSchedules.length === 0) {
            return false;
        }
        let found = false;
        for (const child of cfnSecretTargetAttachments) {
            const attachmentLogicalId = getMatchingSecretTargetAttachment(child, secretLogicalId);
            if (attachmentLogicalId) {
                secretTargetAttachmentLogicalIds.push(attachmentLogicalId);
            }
        }
        for (const child of cfnRotationSchedules) {
            if (isMatchingRotationSchedule(child, secretLogicalId, secretTargetAttachmentLogicalIds)) {
                found = true;
                break;
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to check whether a given Secret Target Attachment is associated with the given secret.
 * @param node The CfnTargetAttachment to check.
 * @param secretLogicalId The Cfn Logical ID of the secret.
 * Returns the Logical ID if the attachment if is associated with the secret, otherwise and empty string.
 */
function getMatchingSecretTargetAttachment(node, secretLogicalId) {
    const resourceSecretId = nag_pack_1.resolveResourceFromInstrinsic(node, node.secretId);
    if (secretLogicalId === resourceSecretId) {
        return nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
    }
    return '';
}
/**
 * Helper function to check whether a given Rotation Schedule is associated with the given secret.
 * @param node The CfnRotationSchedule to check.
 * @param secretLogicalId The Cfn Logical ID of the secret.
 * @param secretTargetAttachmentLogicalIds The Cfn Logical IDs of any Secret Target Attachments associated with the given secret.
 * Returns whether the CfnRotationSchedule is associated with the given secret.
 */
function isMatchingRotationSchedule(node, secretLogicalId, secretTargetAttachmentLogicalIds) {
    const resourceSecretId = nag_pack_1.resolveResourceFromInstrinsic(node, node.secretId);
    if (secretLogicalId === resourceSecretId ||
        secretTargetAttachmentLogicalIds.includes(resourceSecretId)) {
        if (monocdk_1.Stack.of(node).resolve(node.hostedRotationLambda) === undefined &&
            monocdk_1.Stack.of(node).resolve(node.rotationLambdaArn) === undefined) {
            return false;
        }
        const rotationRules = monocdk_1.Stack.of(node).resolve(node.rotationRules);
        if (rotationRules !== undefined) {
            const automaticallyAfterDays = monocdk_1.Stack.of(node).resolve(rotationRules.automaticallyAfterDays);
            if (automaticallyAfterDays !== undefined) {
                return true;
            }
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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