"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_redshift_1 = require("monocdk/aws-redshift");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../../nag-pack");
/**
 * Redshift clusters require TLS/SSL encryption - (Control IDs: AC-4, AC-4(22), AC-24(1), AU-9(3), CA-9b, PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SI-1a.2, SI-1a.2, SI-1c.2)
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_redshift_1.CfnCluster) {
        const clusterParameterGroupName = nag_pack_1.resolveResourceFromInstrinsic(node, node.clusterParameterGroupName);
        if (clusterParameterGroupName === undefined) {
            return false;
        }
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_redshift_1.CfnClusterParameterGroup) {
                if (isMatchingParameterGroup(child, clusterParameterGroupName)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to check whether the Parameter Group enforces SSL and is associated with the given Redshift cluster
 * @param node the CfnClusterParameterGroup to check
 * @param parameterGroupName The name of the associated parameter group
 * returns whether the CfnClusterParameterGroup enforces SSL and is associated given Redshift cluster
 */
function isMatchingParameterGroup(node, parameterGroupName) {
    const parameterGroupLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
    if (parameterGroupName !== parameterGroupLogicalId ||
        node.parameters == undefined) {
        return false;
    }
    const parameters = monocdk_1.Stack.of(node).resolve(node.parameters);
    for (const parameter of parameters) {
        const resolvedParameter = monocdk_1.Stack.of(node).resolve(parameter);
        if (resolvedParameter.parameterName.toLowerCase() == 'require_ssl' &&
            resolvedParameter.parameterValue.toLowerCase() == 'true') {
            return true;
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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