"use strict";
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
Object.defineProperty(exports, "__esModule", { value: true });
const aws_ec2_1 = require("monocdk/aws-ec2");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../../nag-pack");
const EBS_OPTIMIZED_SUPPORTED = [
    'c1.xlarge',
    'c3.xlarge',
    'c3.2xlarge',
    'c3.4xlarge',
    'g2.2xlarge',
    'i2.xlarge',
    'i2.2xlarge',
    'i2.4xlarge',
    'm1.large',
    'm1.xlarge',
    'm2.2xlarge',
    'm2.4xlarge',
    'm3.xlarge',
    'm3.2xlarge',
    'r3.xlarge',
    'r3.2xlarge',
    'r3.4xlarge',
];
const DEFAULT_TYPE = 'm1.small';
/**
 * EC2 instance types that support EBS optimization and are not EBS optimized by default have EBS optimization enabled - (Control IDs: CP-2(5), CP-9a, CP-9b, CP-9c, CP-10, SC-5(2))
 * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-optimized.html#previous
 *  @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_ec2_1.CfnInstance) {
        const instanceType = node.instanceType
            ? nag_pack_1.resolveIfPrimitive(node, node.instanceType)
            : DEFAULT_TYPE;
        const ebsOptimized = monocdk_1.Stack.of(node).resolve(node.ebsOptimized);
        if (EBS_OPTIMIZED_SUPPORTED.includes(instanceType) &&
            ebsOptimized !== true) {
            return false;
        }
    }
    return true;
}
exports.default = default_1;
//# sourceMappingURL=data:application/json;base64,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