"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_apigateway_1 = require("monocdk/aws-apigateway");
const aws_wafv2_1 = require("monocdk/aws-wafv2");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../../nag-pack");
/**
 * Rest API stages are associated with AWS WAFv2 web ACLs - (Control ID: AC-4(21))
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_apigateway_1.CfnStage) {
        const stageLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
        const stageName = nag_pack_1.resolveResourceFromInstrinsic(node, node.stageName);
        const restApiId = nag_pack_1.resolveResourceFromInstrinsic(node, node.restApiId);
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_wafv2_1.CfnWebACLAssociation) {
                if (isMatchingWebACLAssociation(child, stageLogicalId, stageName, restApiId)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to check whether a given Web ACL Association is associated with the given Rest API
 * @param node the CfnWebACLAssociation to check
 * @param stageLogicalId the Cfn Logical ID of the REST API Stage
 * @param stageName the name of the REST API Stage
 * @param restApiId the ID or Cfn Resource ID of the REST API associated with the Stage
 * returns whether the CfnWebACLAssociation is associates with the given Rest API
 */
function isMatchingWebACLAssociation(node, stageLogicalId, stageName, restApiId) {
    const resourceLogicalId = JSON.stringify(monocdk_1.Stack.of(node).resolve(node.resourceArn));
    const regexes = Array();
    regexes.push(`${restApiId}.+${stageLogicalId}(?![\\w])`);
    if (stageName !== undefined) {
        regexes.push(`${restApiId}.+${stageName}(?![\\w])`);
    }
    const regex = new RegExp(regexes.join('|'), 'gm');
    if (regex.test(resourceLogicalId)) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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