"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_wafv2_1 = require("monocdk/aws-wafv2");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../../nag-pack");
/**
 * WAFv2 web ACLs have logging enabled - (Control IDs: AU-2(a)(d), AU-3, AU-12(a)(c), SC-7, SI-4(a)(b)(c))
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_wafv2_1.CfnWebACL) {
        const webAclLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
        const webAclName = monocdk_1.Stack.of(node).resolve(node.name);
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_wafv2_1.CfnLoggingConfiguration) {
                if (isMatchingLoggingConfiguration(child, webAclLogicalId, webAclName)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to check whether the Logging Configuration contains the given Web ACL
 * @param node the CfnLoggingConfiguration to check
 * @param webAclLogicalId the Cfn Logical ID of the Web ACL
 * @param webAclName the name of the Web ACL
 * returns whether the CfnLoggingConfiguration contains the given Web ACL
 */
function isMatchingLoggingConfiguration(node, webAclLogicalId, webAclName) {
    const resourceArn = JSON.stringify(monocdk_1.Stack.of(node).resolve(node.resourceArn));
    if (new RegExp(`${webAclLogicalId}(?![\\w])`).test(resourceArn) ||
        (webAclName != undefined &&
            new RegExp(`webacl\/${webAclName}(?![\\w\\-_\\.])`).test(resourceArn))) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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