"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_elasticloadbalancingv2_1 = require("monocdk/aws-elasticloadbalancingv2");
const aws_wafv2_1 = require("monocdk/aws-wafv2");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../../nag-pack");
/**
 * ALBs are associated with AWS WAFv2 web ACLs - (Control IDs: SC-7, SI-4(a)(b)(c))
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_elasticloadbalancingv2_1.CfnLoadBalancer) {
        const type = nag_pack_1.resolveIfPrimitive(node, node.type);
        if (type === undefined || type === 'application') {
            const loadBalancerLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
            let found = false;
            for (const child of monocdk_1.Stack.of(node).node.findAll()) {
                if (child instanceof aws_wafv2_1.CfnWebACLAssociation) {
                    if (isMatchingWebACLAssociation(child, loadBalancerLogicalId)) {
                        found = true;
                        break;
                    }
                }
            }
            if (!found) {
                return false;
            }
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to check whether a given Web ACL Association is associated with the given Load Balancer
 * @param node the CfnWebACLAssociation to check
 * @param loadBalancerLogicalId the Cfn Logical ID of the Load Balancer
 * returns whether the CfnWebACLAssociation is associates with the given Load Balancer
 */
function isMatchingWebACLAssociation(node, loadBalancerLogicalId) {
    const resourceLogicalId = JSON.stringify(monocdk_1.Stack.of(node).resolve(node.resourceArn));
    if (new RegExp(`${loadBalancerLogicalId}(?![\\w])`, 'gm').test(resourceLogicalId)) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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