"use strict";
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
Object.defineProperty(exports, "__esModule", { value: true });
const aws_secretsmanager_1 = require("monocdk/aws-secretsmanager");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../../nag-pack");
/**
 * Secrets have automatic rotation scheduled - (Control ID: 164.308(a)(4)(ii)(B))
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_secretsmanager_1.CfnSecret) {
        const secretLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
        const secretTargetAttachmentLogicalIds = Array();
        const cfnSecretTargetAttachments = Array();
        const cfnRotationSchedules = Array();
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_secretsmanager_1.CfnSecretTargetAttachment) {
                cfnSecretTargetAttachments.push(child);
            }
            else if (child instanceof aws_secretsmanager_1.CfnRotationSchedule) {
                cfnRotationSchedules.push(child);
            }
        }
        if (cfnRotationSchedules.length === 0) {
            return false;
        }
        let found = false;
        for (const child of cfnSecretTargetAttachments) {
            const attachmentLogicalId = getMatchingSecretTargetAttachment(child, secretLogicalId);
            if (attachmentLogicalId) {
                secretTargetAttachmentLogicalIds.push(attachmentLogicalId);
            }
        }
        for (const child of cfnRotationSchedules) {
            if (isMatchingRotationSchedule(child, secretLogicalId, secretTargetAttachmentLogicalIds)) {
                found = true;
                break;
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to check whether a given Secret Target Attachment is associated with the given secret.
 * @param node The CfnTargetAttachment to check.
 * @param secretLogicalId The Cfn Logical ID of the secret.
 * Returns the Logical ID if the attachment if is associated with the secret, otherwise and empty string.
 */
function getMatchingSecretTargetAttachment(node, secretLogicalId) {
    const resourceSecretId = nag_pack_1.resolveResourceFromInstrinsic(node, node.secretId);
    if (secretLogicalId === resourceSecretId) {
        return nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
    }
    return '';
}
/**
 * Helper function to check whether a given Rotation Schedule is associated with the given secret.
 * @param node The CfnRotationSchedule to check.
 * @param secretLogicalId The Cfn Logical ID of the secret.
 * @param secretTargetAttachmentLogicalIds The Cfn Logical IDs of any Secret Target Attachments associated with the given secret.
 * Returns whether the CfnRotationSchedule is associated with the given secret.
 */
function isMatchingRotationSchedule(node, secretLogicalId, secretTargetAttachmentLogicalIds) {
    const resourceSecretId = nag_pack_1.resolveResourceFromInstrinsic(node, node.secretId);
    if (secretLogicalId === resourceSecretId ||
        secretTargetAttachmentLogicalIds.includes(resourceSecretId)) {
        if (monocdk_1.Stack.of(node).resolve(node.hostedRotationLambda) === undefined &&
            monocdk_1.Stack.of(node).resolve(node.rotationLambdaArn) === undefined) {
            return false;
        }
        const rotationRules = monocdk_1.Stack.of(node).resolve(node.rotationRules);
        if (rotationRules !== undefined) {
            const automaticallyAfterDays = monocdk_1.Stack.of(node).resolve(rotationRules.automaticallyAfterDays);
            if (automaticallyAfterDays !== undefined) {
                return true;
            }
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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