"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_redshift_1 = require("monocdk/aws-redshift");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../../nag-pack");
/**
 * Redshift clusters require TLS/SSL encryption - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii))
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_redshift_1.CfnCluster) {
        const clusterParameterGroupName = nag_pack_1.resolveResourceFromInstrinsic(node, node.clusterParameterGroupName);
        if (clusterParameterGroupName === undefined) {
            return false;
        }
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_redshift_1.CfnClusterParameterGroup) {
                if (isMatchingParameterGroup(child, clusterParameterGroupName)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to check whether the Parameter Group enforces SSL and is associated with the given Redshift cluster
 * @param node the CfnClusterParameterGroup to check
 * @param parameterGroupName The name of the associated parameter group
 * returns whether the CfnClusterParameterGroup enforces SSL and is associated given Redshift cluster
 */
function isMatchingParameterGroup(node, parameterGroupName) {
    const parameterGroupLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
    if (parameterGroupName !== parameterGroupLogicalId ||
        node.parameters == undefined) {
        return false;
    }
    const parameters = monocdk_1.Stack.of(node).resolve(node.parameters);
    for (const parameter of parameters) {
        const resolvedParameter = monocdk_1.Stack.of(node).resolve(parameter);
        if (resolvedParameter.parameterName.toLowerCase() == 'require_ssl' &&
            resolvedParameter.parameterValue.toLowerCase() == 'true') {
            return true;
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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