"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_iam_1 = require("monocdk/aws-iam");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../../nag-pack");
/**
 * IAM Groups have at least one IAM User - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1))
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_iam_1.CfnGroup) {
        const groupLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
        const groupName = monocdk_1.Stack.of(node).resolve(node.groupName);
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_iam_1.CfnUser) {
                if (isMatchingUser(child, groupLogicalId, groupName)) {
                    found = true;
                    break;
                }
            }
            else if (child instanceof aws_iam_1.CfnUserToGroupAddition) {
                if (isMatchingGroupAddition(child, groupLogicalId, groupName)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to check whether the IAM User belongs to the IAM Group
 * @param node the CfnUser to check
 * @param groupLogicalId the Cfn Logical ID of the group
 * @param groupName the name of the group
 * returns whether the CfnUser is in the given group
 */
function isMatchingUser(node, groupLogicalId, groupName) {
    const groups = monocdk_1.Stack.of(node).resolve(node.groups);
    if (Array.isArray(groups)) {
        for (const group of groups) {
            const resolvedGroup = JSON.stringify(monocdk_1.Stack.of(node).resolve(group));
            if (new RegExp(`${groupLogicalId}(?![\\w])`).test(resolvedGroup) ||
                (groupName != undefined &&
                    new RegExp(`${groupName}(?![\\w\\-_\\.])`).test(resolvedGroup))) {
                return true;
            }
        }
    }
    return false;
}
/**
 * Helper function to check whether the User to Group Addition mentions the specified Group
 * @param node the CfnUserToGroupAddition to check
 * @param groupLogicalId the Cfn Logical ID of the group
 * @param groupName the name of the group
 * returns whether the CfnUserToGroupAddition references the given group
 */
function isMatchingGroupAddition(node, groupLogicalId, groupName) {
    const resolvedGroup = JSON.stringify(monocdk_1.Stack.of(node).resolve(node.groupName));
    if (new RegExp(`${groupLogicalId}(?![\\w])`).test(resolvedGroup) ||
        (groupName != undefined &&
            new RegExp(`${groupName}(?![\\w\\-_\\.])`).test(resolvedGroup))) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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