"use strict";
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
Object.defineProperty(exports, "__esModule", { value: true });
const aws_elasticloadbalancing_1 = require("monocdk/aws-elasticloadbalancing");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../../nag-pack");
/**
 * CLBs use ACM-managed certificates - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii))
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_elasticloadbalancing_1.CfnLoadBalancer) {
        //For each listener, ensure that it's utilizing an ACM SSL/HTTPS cert
        const listeners = monocdk_1.Stack.of(node).resolve(node.listeners);
        if (listeners != undefined) {
            //Iterate through listeners, checking if secured ACM certs are used
            for (const listener of listeners) {
                const resolvedListener = monocdk_1.Stack.of(node).resolve(listener);
                const listenerARN = nag_pack_1.resolveIfPrimitive(node, resolvedListener.sslCertificateId);
                //Use the ARN to check if this is an ACM managed cert
                if (listenerARN == undefined) {
                    return false;
                }
                else {
                    const acmRegex = /^arn:[^:]+:acm:.+$/;
                    if (!acmRegex.test(listenerARN)) {
                        return false;
                    }
                }
            }
        }
    }
    return true;
}
exports.default = default_1;
//# sourceMappingURL=data:application/json;base64,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