"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_backup_1 = require("monocdk/aws-backup");
const aws_dynamodb_1 = require("monocdk/aws-dynamodb");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../../nag-pack");
/**
 * DynamoDB tables are part of AWS Backup plan(s) - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B))
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_dynamodb_1.CfnTable) {
        const tableLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
        const tableName = monocdk_1.Stack.of(node).resolve(node.tableName);
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_backup_1.CfnBackupSelection) {
                if (isMatchingSelection(child, tableLogicalId, tableName)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to check whether the Backup Plan Selection contains the given Table
 * @param node the CfnBackupSelection to check
 * @param tableLogicalId the Cfn Logical ID of the table
 * @param tableName the name of the table
 * returns whether the CfnBackupSelection contains the given Table
 */
function isMatchingSelection(node, tableLogicalId, tableName) {
    const backupSelection = monocdk_1.Stack.of(node).resolve(node.backupSelection);
    const resources = monocdk_1.Stack.of(node).resolve(backupSelection.resources);
    if (Array.isArray(resources)) {
        for (const resource of resources) {
            const resolvedResource = JSON.stringify(monocdk_1.Stack.of(node).resolve(resource));
            if (new RegExp(`${tableLogicalId}(?![\\w])`).test(resolvedResource) ||
                (tableName != undefined &&
                    new RegExp(`table\/${tableName}(?![\\w\\-_\\.])`).test(resolvedResource))) {
                return true;
            }
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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