"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_ec2_1 = require("monocdk/aws-ec2");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../../nag-pack");
/**
 * VPCs have Flow Logs enabled
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_ec2_1.CfnVPC) {
        const vpcLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_ec2_1.CfnFlowLog) {
                if (isMatchingCompliantFlowLog(child, vpcLogicalId)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to check whether a given Flow Log is compliant and associated with the given VPC
 * @param node the CfnFlowLog to check
 * returns whether the CfnFlowLog is compliant
 */
function isMatchingCompliantFlowLog(node, vpcLogicalId) {
    const resourceLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.resourceId);
    if (node.resourceType === 'VPC' && resourceLogicalId === vpcLogicalId) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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