import lightgbm
import numpy as np
from ember import PEFeatureExtractor
from secml.array import CArray
from secml.ml.classifiers import CClassifier


class CClassifierEmber(CClassifier):
	"""
	The wrapper for the EMBER GBDT, by Anderson et al. https://arxiv.org/abs/1804.04637
	"""

	def __init__(self, tree_path: str = None):
		"""
		Create the EMBER tree.

		Parameters
		----------
		tree_path : str
			path to the tree parameters
		"""
		super(CClassifierEmber, self).__init__()
		self._lightgbm_model = self._load_tree(tree_path)

	def extract_features(self, x: CArray) -> CArray:
		"""
		Extract EMBER features

		Parameters
		----------
		x : CArray
			program sample
		Returns
		-------
		CArray
			EMBER features
		"""
		extractor = PEFeatureExtractor(2, print_feature_warning=False)
		x_bytes = bytes(x.astype(np.int).tolist()[0])
		features = CArray([np.array(extractor.feature_vector(x_bytes), dtype=np.float32)])
		return features

	def _backward(self, w):
		pass

	def _fit(self, x, y):
		raise NotImplementedError("Fit is not implemented.")

	def _load_tree(self, tree_path):
		booster = lightgbm.Booster(model_file=tree_path)
		self._classes = 2
		self._n_features = booster.num_feature()
		return booster

	def _forward(self, x):
		confidence = self._lightgbm_model.predict(x.tondarray())[0]
		confidence = CArray([1 - confidence, confidence])
		return confidence
