## Subresource Integrity

If you are loading Highlight.js via CDN you may wish to use [Subresource Integrity](https://developer.mozilla.org/en-US/docs/Web/Security/Subresource_Integrity) to guarantee that you are using a legimitate build of the library.

To do this you simply need to add the `integrity` attribute for each JavaScript file you download via CDN. These digests are used by the browser to confirm the files downloaded have not been modified.

```html
<script
  src="//cdnjs.cloudflare.com/ajax/libs/highlight.js/11.3.1/highlight.min.js"
  integrity="sha384-kCq9CJf66M6NoTFgSHAEGthqZ1X6afHQaIT+Mo236gH3FQ5uAWSIO5OWnM2H1fDA"></script>
<!-- including any other grammars you might need to load -->
<script
  src="//cdnjs.cloudflare.com/ajax/libs/highlight.js/11.3.1/languages/go.min.js"
  integrity="sha384-A6CJZ7OeaXXC4LutjeFo0FxlK24jyAHVL9XOabzyQq+HjDPuBg+18Zx9oyqMHheI"></script>
```

The full list of digests for every file can be found below.

### Digests

```
sha384-kCq9CJf66M6NoTFgSHAEGthqZ1X6afHQaIT+Mo236gH3FQ5uAWSIO5OWnM2H1fDA highlight.min.js
sha384-mo1bHgfdNdNaRMT44GUdGc1y+FoDhA/YudACQm3vZDGdKz+X+kjgtCahBI0RzF3B highlight.js
sha384-VnM2Afb+nAo7uieC+W1ICTv2ETQhE0cUxZRiOnPoEP4Vl6zqSjomcZbWwD5I5JNY es/highlight.min.js
sha384-JsQ71+wXJf89sC/CJ8FLgS0ZSjA8KZiPi7A7y0l6NyAZrl25edKBRRLmF8lO61jo es/highlight.js
sha384-VnM2Afb+nAo7uieC+W1ICTv2ETQhE0cUxZRiOnPoEP4Vl6zqSjomcZbWwD5I5JNY es/core.min.js
sha384-JsQ71+wXJf89sC/CJ8FLgS0ZSjA8KZiPi7A7y0l6NyAZrl25edKBRRLmF8lO61jo es/core.js
sha384-1rc4eQvZHQSmxJbc5aeGDsQM0d6ykj94yvDtqsBogqWJ8xVfguwckQvqpytwaTK3 languages/1c.min.js
sha384-sNO1nUn6nwU8vJgC0lo0eGmEL+mw6zgI+dv6WqW9xN/Qohzj6sKFe5LLYdmGPE32 languages/abnf.min.js
sha384-MXZZXf4XgDwSxRPrm77OjOV1bTjXyEKWJ9e4fn8CVh3ASLw/4R432SgPBfyciSQ7 languages/accesslog.min.js
sha384-rSbKs0BX4PmxGqW3J6XiPF661zaiiVG0kj6Aaq0S+P2kuDl2hcwpyJJgvp4S9aQw languages/ada.min.js
sha384-UfCplRHHhs6czVurKi3vcwNNSniF3rbO/J55XEm9mmaEygb424flw8K7N/NqJfRG languages/angelscript.min.js
sha384-aEUxHbUp5o3QKf+/B9FZ2rC57SYBP2pQVHnOhcQ8QXwEhCGLrEMHlPUvN0APn3aw languages/actionscript.min.js
sha384-rjQKCi9a3tpv+IwI0vlADLnomTro7bvuomQNCy/FbPD05PF4Rk8s08phTL50wXbm languages/apache.min.js
sha384-NXEuycJVyuQNDuqCHOdqHo4nAG8aPOXTz1Md+B9VCmWDQ/eSH0HCYFoIi21bj0U1 languages/applescript.min.js
sha384-o+UMrsCel3FJBOGOVFhY0hYDnEp1c/+c5b/f/L9zE6ZgcjzulT7XGQwbJbaIRLZh languages/arcade.min.js
sha384-9+MrenSD1Gd7qlxoWV2GOp1pluU5xIrWfisFjcQLH2f0qNUW8hd2rQEpvV2KyYCs languages/armasm.min.js
sha384-oMKwkruY4NvgpJoiRTNWXbXHVbc2/h9eMa7iKiFK/pG8MDpvirr5XuMydTgco58n languages/asciidoc.min.js
sha384-+M0c0KHAQ9ltYrYWkMGBHjwxIM0y1dGNvUTOia571ZG7RomdRH7SuMBIXNWTic8U languages/aspectj.min.js
sha384-Issjpt3YVGZIycMvqs+MzroEJoOZSqkhu+vW5b5LncSQHf+ohWFX46k/b8L0BBdV languages/autohotkey.min.js
sha384-yiFc4cI92W/F+7TnE/HV1QA38cD1tFtZ810L5X8nB3aelZVhnW38ZDeSi3Drh+Sk languages/autoit.min.js
sha384-qxvmXwsjBFYGFZhcjRP4vdAnzxlOhv8yEiHLcO3FiKsFyIqZmJPInjX9q3sVicY6 languages/avrasm.min.js
sha384-PRWLSnA821NWsRgeohAg/eEL3+pdex4MTRzRT6VgluwCkDD22usiQ0zgcA6icRzi languages/awk.min.js
sha384-PzXPeUt0hwBXths+RowTnljiyCB+F7L2lZn+l3a9uaOU2DmNAL4jn+zklcDezDjA languages/axapta.min.js
sha384-tQ9NsnEZ+z0A3Ys2I1e5sbiWE/Yed7e9Xxr0uwAqVb/xs2n1n1+Z40UqGnVhEjz2 languages/bash.min.js
sha384-Gp7mJd9undSSzXjEsN7GBJZOvSuydmZRN3FzMKNS5pvlLmMu5xqvGpHzsygbIxfn languages/basic.min.js
sha384-qsZefOAvVKamBn8KRQKGDJYksYpATe5m7bD+mDJrS8YBt5beQzOnGWvO4Mc/pnMm languages/bnf.min.js
sha384-TU5d9ajB0d8ao/XySmPBXdmr5JSe/pmbEJO1UjvdawUrL480DP5g82QZcLgclq55 languages/brainfuck.min.js
sha384-phI+yHWl49lQcRwa03IqJ5SMGGfARlUCzVyp8WMvOmOIkNnwzS7b5fOcTSoVNnAO languages/c.min.js
sha384-Tppmfbr0Cm2XYHMxXniZRCvbuMVZfZSmIRrM3aFhLe9m2hOceERVBQ4GPqLc9cvY languages/cal.min.js
sha384-+FVfqYSWWPb+wN1EFWQJ5vFK8vNu97JoM58fcU8gqso8gFYqZP/kB9csG3TWFq1V languages/capnproto.min.js
sha384-sNtj1afbKE3WzccvGHCJkNjl9ZYi9wt5v497faYAvWwHzCZrU4LNOyvE0tLCU5RJ languages/ceylon.min.js
sha384-cDZCNqj3MM6Ij7TLAvluKUzRSEmamR5k3aUlQP+7tCiLloQ895R7Sn4wTXbhX8m7 languages/clean.min.js
sha384-7RvwpnepgtE4k8reIWUIeqZ7zQLJRteZawus5jx74G2ECJhpVZaKvD5V0j3oD9fy languages/clojure-repl.min.js
sha384-s34HdHUocCL9N05lShcpUTcTo8Fexqorr9eO+KAEidlaETp7MVAsBnonQrsK4No9 languages/clojure.min.js
sha384-0NLz3IHYebVEjWHlDsuXhpoz3oJxfa5zsgAm5EOuNIR8zy5uKJb/EJ/So9MMtonL languages/cmake.min.js
sha384-zs3XNP4sE8JsP3CWR7IwmEWe43gqAiO2ugbga/pfzNEJETEuF5ljGPk8rfhQOZBq languages/coq.min.js
sha384-/s2W9G0jJeD6G5UgMHJWGF3k8wkCX6885j/ufscyIIPmp70fBIOoH3J6tg2iJLLV languages/cos.min.js
sha384-ICI5YgaBD2n+cUtNzFWhxbz/ywA8dWZKV6EjBI0S2UKOgvaM5/IlD/sjvlFb3Y2z languages/cpp.min.js
sha384-D72M27ynCFuIWOPzoSF61JO42vn9BBFfejHllUmZnlnFYBHsl7x6eQrNADs2HLIi languages/crmsh.min.js
sha384-D7uPcVXwS0H1H4OzZZAKn4Bfje5UYRRBCpKzeQUxOgP8S9JOuC7w5/ZuqLKNprzn languages/crystal.min.js
sha384-qfi2CUicVO+C3ChCHktHC2MgKbWr6KdhE8/Nupd+3Qs6hNsM43e7p1qlM6LqtfWh languages/csharp.min.js
sha384-meCoCBpwbE4Y9RGaUHH+EMB6UWlZzcZGO2dvKicEbOksln36Pk9V8VimG83ZHVgg languages/csp.min.js
sha384-DxvJ/j23W3bsvyliaiXS2VFH4YYkfE1EERLYxiVUHkSvyYG7h+E80kGWr7EQ27vS languages/d.min.js
sha384-iNPu5oARzTwGlHBW0he+xkjYJSu9f9qZy4XNyLSJ+l7/HtEHH4aNCpMTng3pWNeI languages/dart.min.js
sha384-M3fTp9w5ZAm/C2SGUnYlsbMFuCGKUJOuJ51H9xKdYCyx10jCHa5QbSZmRKE6st+W languages/delphi.min.js
sha384-BFrwq2/+5QGratoKXRRQZ34lDEgVpi6nN+kvjGrC1b/Z6O+xjq8HDZP18nGQOiXf languages/diff.min.js
sha384-zaDFdvjDjLZkgzKpO16p+oWgHXm5ua6e7K7tI7jmjdV00knJljkh6pPuTKFj4vop languages/django.min.js
sha384-H0gvrKGD84r9B6bhTbacT06+jD8VZehX5eOFFkq6tDFdG+YU3mTo5yZAyTyeZKvj languages/dns.min.js
sha384-3ilAVZi+IwUNRoAID8Q+ks9keix7cmed6fRtA8x9+PHtSRL8NA93caMUHxbhgb6G languages/dockerfile.min.js
sha384-zS9GXXJZ6hdyfoNtesv3Rw46d/Sfzr8VBo+C/gpsetNGbWOc47gi0J1zHt5g1Lwn languages/dos.min.js
sha384-SKzsynPQOY+WekN7xK4l8XKyqsPZ/9Rbg+6eBQa0MDieqOnFvllA0YMFkREpIVX8 languages/dsconfig.min.js
sha384-XIK2nUJKqdJxUrbuLFA9kmNS7D9SzCosrLJJmOPj8LXXQqweLRmdjhJfmRwZYPJk languages/dts.min.js
sha384-DZoQaDqeXN/BUm6zv9Q8THiamWuelp9se5gggF0GrfkVyXnzArVhE7Zz6CtyHbRx languages/dust.min.js
sha384-IPSIY/3T9r7yOIb9eNz3pUDyWNtv966UTbkfbxJiWvrWcuDdX1pW1HjD5+5DPbWI languages/ebnf.min.js
sha384-dG8FtU7XIU+rYGOfdmMe+NUh7WV1of9ppnF8trRQ2qRj7nfp16lMRrCzJ5x7yrAt languages/elixir.min.js
sha384-ei0he4eigaAL+ogWOhheIAeNVxaUhBZp6JBmU3JWmuunEXdCai4jJPi1QA1DODHw languages/elm.min.js
sha384-YrfNgyicSPvIuRK+Cnrn4Y3qbUNOThluDjUO9+Gf9IYsFhEmrJS1eJYZOfcVSSHt languages/erb.min.js
sha384-nZyPkY61svsmnRGApg1dhmO5MdadA8yhkEgTU4NleO1nyL6OaMtoonPPRrWJddvG languages/erlang-repl.min.js
sha384-Dd+0NbmPRIkRlUcvG3vkBfWY0GT7xPTuYsq7swBtMnMtQc9MZXARZ3mLpcDnF9Ka languages/erlang.min.js
sha384-kpgVcLgJy9tsR+z9Lxmb0EULiyo6xt+ACFv6C17rVyokRI0SRiWOzjMPNhX+m1Gm languages/excel.min.js
sha384-t7IiKSBll2ENnQ+nDttMtXyg6szkib6iUhJYdOqN0day4pvVesvxg66Nl1USSzD1 languages/fix.min.js
sha384-oH3tN+fvZD8SZMA6F2lD4g0JBJ5MLRJs3Y4a8OgLsP52EDQRi6bVaY9ME8+UQrRi languages/flix.min.js
sha384-Ah7GJ6GucygDvUeZxrGZUn6z3sBXZgJb4ouPtunkjgSdhBZbR6tt2PUJo5ztWuQa languages/fortran.min.js
sha384-w/nzPCaTAS5cdNcyNB3swwyM0wROkf15TI6loIjRwSaqP6V8smqDS1IopImm/K+t languages/gams.min.js
sha384-NpqU6jA+LNskrhgMDDLgAIoOYNhOp7PXcIpLGpNibaqfGRXmLWWTVz8FtiEfUz5Z languages/gauss.min.js
sha384-/mERM0SzGWKsvyoYSzqkomyiFVjhyNQgHXTqGNL4XNytDVPF1418rLpqDN4TFTzK languages/gcode.min.js
sha384-XCZWiSenx1dkOVtV62+XmsHeod2In0+bsOaYIKy5OMmdfaFfSi/S/ALiOR/KflgC languages/gherkin.min.js
sha384-TkZNBMmewXOXmWVck9G58nJfg1L03t6c251KmMF+WZIRVkrbZP9Q6oy9o0cWpOJL languages/glsl.min.js
sha384-avgB+dszAMGy9vaCe7R1yi9U1/+Mr7uLc72JQL3ENijPZzku6CKtlAqDrIQPoEZC languages/gml.min.js
sha384-A6CJZ7OeaXXC4LutjeFo0FxlK24jyAHVL9XOabzyQq+HjDPuBg+18Zx9oyqMHheI languages/go.min.js
sha384-bElC+3Su4gB/fSVF2krcAWbxmQWXJ/Jklk5DGWoQhXmeIOfIPSgMKlpVPB6ifcn6 languages/golo.min.js
sha384-lI9Ck004+Fe+FJmSNxSOg1aj+tytaAvvH51FmNALfgB9rh06fyPvUe1+4xtX1t3/ languages/gradle.min.js
sha384-bPjv86VE2tXQhwyLcYnjxyOuHiBxXOjs2r5piAvVasfp96Ssfhio4Ph/2OxAqQvN languages/groovy.min.js
sha384-2HOON1jg8mOgQ2SQECFbRljJAk9Q4OTehTez6Vv3OR+bitdYcB6YyNc/ZmF9bFL7 languages/haml.min.js
sha384-Cq9Sth31t/Un/1u06QdXTTxIQD1lr6rsx390kUb9VvHbnHu7/opAbotdKh/3CAun languages/handlebars.min.js
sha384-sZzwt6bdTNhSkV+2A1e5pX2EU4xs7/vhwEwV4CfMKXvdGm91waZsBtOPB/GZFQKt languages/haskell.min.js
sha384-cPXl2n3L2HiE+fvsadHtLqIOaMFobBf+mDZDlYiTgrvPNLTsAs2uFI+HbSwqurhH languages/haxe.min.js
sha384-SDpWwZVSp9+QmBiqPFS7tvg/TZsODggDyIIEoJ4KGaShoHKxJAdFT+Jvbo7GyPQU languages/hsp.min.js
sha384-P3TLmcdkvrR94wMeIiNbDVmMP4Dt7KHy+tzF5x94HidtyLDe/xjBmiQyvwvKMPoN languages/http.min.js
sha384-73URwtP7bVp1LTB5uJ6XsTvbJqQg/QJx1fwFBQBskCNDja/IzaiXojVPdX+LTWDP languages/hy.min.js
sha384-JdYR06KWl9zz5DyIGmQ/BSzBOro8OTmUnRzIs2b8iRnuT+cqa9zUUOfKI2dap2q0 languages/inform7.min.js
sha384-EBXzsPKkhuMSKzbXqvfI4SwhPOWwfNgfSacR7h10+xFtfkaI7HB3tPy5B1I6Qy66 languages/ini.min.js
sha384-XhZVtA4Q0tx4zIu7V+wTO0+gDqq8j9MuAEMfc5kX7pSx5B3wt54dSv6IHeElmIpd languages/irpf90.min.js
sha384-SFUtM6IMyvfjbR1tESNbmdTS5xAb9CnjG5d8ylOmQu1IzAkOHQMGPGK6AOjNXdRR languages/isbl.min.js
sha384-Tr7xlkXvImFzQFkbR2oC0VE3c9T96UuLQfiqmFyPJSNOgrdZbwKx/iKjGqFmiPlo languages/jboss-cli.min.js
sha384-2aAwT3z6eqFp7/aQRbdZ+ydDLhJDFHPq8H+4hK6B8nRgUgMe5/68iMBOlYGzJDzQ languages/json.min.js
sha384-qO1NCEojtLcTLTeYkdqk5jXp+zLlYsrY+kkixp/r5qBHepl6Z4m6FkfdVdGxHWBa languages/julia-repl.min.js
sha384-QQ7G3/n/90ikTPDhLQEI0bXQmLI/l9A0g+VJ/wyh9395uW2q0u5fYvwRdCOgJbB8 languages/julia.min.js
sha384-wKj+P87suFa6yN9kMuqDqerXYVtlO+5IQl2p+wtRIweWYfnZ3SXnGGSjZfhZ245G languages/lasso.min.js
sha384-FXftgKGqt+BZpDYWeBcOW3OSlkVsx2la/y7zjoS/MFCsiDjfbsOfrO8SiuKq085O languages/latex.min.js
sha384-+q6IhWKaOgLKMeU1oeOku0N1JKFqiraU5Zx1nm+nLhYI9wttEoWzDkobwlPCC4Jv languages/ldif.min.js
sha384-hc0n2iKpYo3iQg/axnLgOvQueYRdQ0j26Bh+6ovSsKPtFwoZOPJRar7nxyeb91wR languages/leaf.min.js
sha384-2GOFDz1bOaycIegH2K7PPNCn6p8E0iGu3w26aINCtQHubLyl4kZJUIBcqo+8uEDV languages/lisp.min.js
sha384-LIH3lY0ic3rCNvdLaCEYXLz5GIotOicvScdqMOwBKeVW1DA8y+zAl9DLtcLLdpLB languages/livecodeserver.min.js
sha384-ChUTbW3aPCrlKGvuSA/8/zc2rd+3betHSX35oaoXO+3MF/tgCP+92iExtLCAAZwI languages/llvm.min.js
sha384-psMPl0gN7ixeGHq/9mlOFMmuFaVMmBQYGLnDke0umqg+tzy/ZEraNTeKg3WY9/iv languages/lsl.min.js
sha384-Okga2AnAOnrmyqcfUkLbpEznqnZF8SmYlECpmaNLhxMZ9O2zxFXD8J2CZ3Io0FJY languages/lua.min.js
sha384-2UHHEbguEBJfN9zf5hV91zcB3qneTQaKSFNC9vezxPr4wdlNwjX9JaPXgZWI8JZI languages/makefile.min.js
sha384-QDB/m6D0ja26bHH14Uxpoulcmi48irQ+2P+9Nbx3g+Ql7eOpS7xcPa2ZtXN0Kwcd languages/markdown.min.js
sha384-7xwVvoqVBl5knIitqVuaTQS6gh5C8LFRm8+6PbfkMFLp5fo4vEhgcmMN7jJ1q5Ze languages/matlab.min.js
sha384-6bYTliyYDL/zRqSbQXd9kqufKomH7kmNi0Kx7550iIciP3pP5+DFJ0Y9K0dmOJQE languages/maxima.min.js
sha384-FNEyjwP7wl62lr2QlGsY9Z4WVy9BFFmlxDMf51NZxEdxOwNS74bHGloqp/+FGgwz languages/mel.min.js
sha384-iuGO/33Xlrtt9ACSCn+m9qxYXGaiSBXxwAMW/m/wNedFx8DFlMwIbBjBVvBf/mFo languages/mercury.min.js
sha384-j2D8wqsHdbueHjI/iW/lu2U5sui8VGgCNTbd/Xlj5rE4vtgTThMzv5JUgH6ig05Z languages/mipsasm.min.js
sha384-Ye4hPk/xMujodyprIxH54Ac0A8Ih3FXYfNb/A1YdRLZNmJgbkRqUU5TB9QyhWUFk languages/mizar.min.js
sha384-iVpeHvLuCheeoGb1Q3lMsIxELk1WEzxJXjnSSMINtO/0P1y8abTUYq8KbtszElmt languages/mojolicious.min.js
sha384-aFp7/DeZUi5V039s35406XotazxVBbdD+cDEIF80I9HDVLlwGSmCypfIUDKxm+DP languages/monkey.min.js
sha384-bNNNrni2Dl4/+UJNCqJUktBw/cACz9Wa3W+J3sNnulvMlxmd7bVbyqklCyHKswBQ languages/moonscript.min.js
sha384-i5TFH5u2IOw+N/60Lm52RY8MI9bhKaD1HqMveSIM/TouZWc+OxVF41IY1LEqnCYU languages/n1ql.min.js
sha384-MaIkuG2/MLduJVV+apuIZNK54RXwH1/FnumH92aiB2ZFGyDd6e7Fp77VoXx/MHRo languages/nginx.min.js
sha384-gHowdCTyBE4hqZGHJNVl1h2mqUGga9DYewhmfTCxx9M7opGyefwmlXznvRvhUW/C languages/nim.min.js
sha384-7EquNtJt81ladtZao6sAYu15OZ+aPY1CSYMY9gzrkwQhMvOBKf+XerPZaQa59nIL languages/nestedtext.min.js
sha384-J4EjzZ/yK1uxJazNEOMNkh19JCsi51qxhp3GBxZmC+CLTaieSO90ZPzzhqKrpbmT languages/nix.min.js
sha384-terN+8JddukHMyqz7EqsKCr/56VnRykPrkD7NA05QYVosPyoIWfljF/lzxs6Dm92 languages/node-repl.min.js
sha384-sfO+Cs4WEIaS3XARS0LUeiso002J4rCwZ/syAD4wJfkX4DVl2D7jkV9xntSKypCf languages/objectivec.min.js
sha384-lNXKavVdHqUSTEyXi3VUVo7ROEbcVvzF5axJy6x+rLsAG+XFaXYlOg3wZuBFTTgX languages/ocaml.min.js
sha384-ijmCim0XcI7Pd1o/2rUxJB9RSSEvGOqTHXwA1g2NQ3Uqd6bFpIOp1421kNuDq1TZ languages/openscad.min.js
sha384-nNJPejwjrqSR68iK5EOZ6rJ9wqeoWJutoffNGikooAduCSrub0XusIR1tG+wPHpi languages/oxygene.min.js
sha384-23CfqEADkai99NEZCke34GCy48MlKUVrSUE8jHWtWq3kSO/0BCLp7uvS6MBN77T8 languages/parser3.min.js
sha384-7NiKLen9GuZmNYBRm8B69jFzwyrAjoDxZmI//cp32PzlGGfP6IFfQv1IwQ4gr7Lw languages/perl.min.js
sha384-1pcotJbInwMIBLuDJ33ctBAhRahXk7pHIq9jOh1S+qc/c6sw+7ULIn83EmokG6tX languages/pf.min.js
sha384-AAxsEyPGXd4qn6FsyfuReqX6XJIbcS3RUI5CFUuXAmVb+CvCB/H8+b14CZljjVpv languages/pgsql.min.js
sha384-zFP5JIxLFPk44Rw1IUGDtuY3KYhN26Zcq2GNk8HoBinjc94orK+xrKi5Wq9PiOFa languages/php-template.min.js
sha384-DBDpWyTZt7hYzzK2I0FD48rL+lPmdd6tCKnHPqmWEDFyMsziow5TGcgl2TX+Ks4Q languages/php.min.js
sha384-a5Z2XiITlWKN0k/c74C2VcyXG0UVXxOjm2muVDNPY8LVL+j/mXieR5m1QBPj86Qe languages/plaintext.min.js
sha384-7P6FoyJLTbTl8u5tQp+EVUum6FQtXlrwQzcAEz9EEHXzMRQAkX83FoOuxe+U4S+Y languages/pony.min.js
sha384-t6rYG/DWFmAsafqUCXQ2K31dfnzDX8Vi2y+lIXCTo3tpDfaKrmWNucSul4JWJRll languages/powershell.min.js
sha384-+mrkzCBb5NUE7lzlOu5cAcPzgCOZ6vJI32ve6nQefk/IN6R9h4MhA0UVfL6hNzmh languages/processing.min.js
sha384-SwSPFU+n6P3iwk1svbZ+euWgWFlFeDxbvYzFukaGyWe1AoqgOHuUSuW2PHFQqVf1 languages/profile.min.js
sha384-G3hU9WH1Qah0QztjDLrJIXleky/z+kcCEA8tMDJ3T54IfgfHeamxOB2eKm1waM8f languages/prolog.min.js
sha384-wp6FXI4j+/UA/qNvlbTEOWnWSCQ+6pWPwhm1nkGTEHbUFtdgqq9MIMhTyYSEmX/v languages/properties.min.js
sha384-MdBaoV2FVHFdkoHURxWyJmQstQFV1I6iZx3zR4OYZcFRbhQN3Cr9Zd7QPPSKEsQd languages/protobuf.min.js
sha384-QaEhLwXA+UPn2xQ0g5uumqI94KB+pkQ/HvXvhi2bbwGDKmOaaIUQ4GLQu5uHlvz/ languages/puppet.min.js
sha384-ruUUMH/UVOV6BW1YC6q2kqlugBs2uvEyfhzi3IoqmQxwEoWU5GArLMFVICJNu8dz languages/purebasic.min.js
sha384-oHwytNSvTEDNyl8oTDIRkzG8lICzuvajqw4zQWi+IVxazKEAvtZDi2lYtpxgUogO languages/python-repl.min.js
sha384-3WTOwTFMCnnRLTCOAsTUXISorhChDkc2Qpjf1U8t8WHDzoMfvZdeP4lrS6TV4riM languages/python.min.js
sha384-DLbUd1SeWpPHDHJ6wpyPqy1r8nbRqkmTo9oJzIi2LOMoFXcnkxMrXRPmKSiVlgCE languages/q.min.js
sha384-MCBUCtwevdXtrcEb0xyAV5n1KD+zB/oJaNxe/S7EcpT+RkCquvv5jwFSvUiEA7ob languages/qml.min.js
sha384-y5FCsa/9GJQqbMqoNisnfyhvqzUlH4NpL/RBR5kLqstGzf+bEfNikQ42LTMEsmZQ languages/r.min.js
sha384-OCv0+grb9/VBgr98yYvLmNrdBa6jDOXObStQuGcjPSSHK5dKO9ZnYQwSIKr7/Djs languages/reasonml.min.js
sha384-oqXZJ540mWGZCc7ifi3c931wvE1GlMi/xdqYLjO08wAM5OKo8ZjTed3foAXCFr56 languages/rib.min.js
sha384-BOyx39FflvYEHLfu9WQuOOLaHmYyx5Md6/K2F2hZMpfoHjcUVSiskPxq7LLdmTmx languages/roboconf.min.js
sha384-nJK9xCrQrUQ/djbkg6Bj4VEeusGglQkmfef9z/xWkwYSiPhgvuxqUm8Gj+yYZVc6 languages/routeros.min.js
sha384-ZiisNPrqt5k1EUVCwx6JTojWBpxZc8w3Zxon3zjMbEPp0rnInjuugoNAVonO2vqR languages/rsl.min.js
sha384-o9Y+VPlfQ77kWbrndi+eg/UZEWnqHGXHOcWRmYESt91GFa/2laf+FkN7JJixUdOo languages/ruby.min.js
sha384-9rEL5EeUz5ftnzvRJMO1NFlhm1CDnJVtXbRihj+cHZvQT71T8iDpSHOAL4LUsqpZ languages/ruleslanguage.min.js
sha384-I4xoFwE0iYL5waCmNZNW91jXdGl52gAIAOiNSIr2QEgpccKrDWo/2NifWWv1hXlf languages/rust.min.js
sha384-Ej1yPk+6dc6uRMqjwTte6S7wfeAEJphbglRdrNMLk22REHjwGk6FQK/uq35c1+in languages/sas.min.js
sha384-3GmPRxZT/qivuIKFrF0btQk8Z2jeNUlpWOyZBiRlf1Yi2CD+EWISY5mi+2yflNz4 languages/scala.min.js
sha384-BfSb6Zt5lA82uoqRNdz3L9P1SPiCEYE8SAlZ2OY5bObD87RkXqb5AJj8JQ6sM6o4 languages/scheme.min.js
sha384-9iisHTGnBAXIWec6NPZyuDHGFdQW4Tf2hLbdtj5LIgseRQeMn5oAe/VLw7OfXt6h languages/scilab.min.js
sha384-c1UQcfDDyZTEhiK28uaNTLm+gUq74737rbyRnqOzWlkGl0DmMLWuWJ7fq2qvIjFT languages/shell.min.js
sha384-fc4pSM1V2t2lJhWpWaDwXbWhmN1WX13xwTRmEwe5RzNBL//gWjkyUdQm+S8bZvZu languages/smali.min.js
sha384-cmtG9/EUGS/v77+fRUuSu2Z4jRR9BBd5AhWaqtMA+6/26iDBTrLBB7kAKp64pYlh languages/smalltalk.min.js
sha384-pvbTAB6yQK+BbvU6+Y/vyu5CygWeLzeGqcbQzWHHOx1I7PvlnsUbLWVqO7rAE37l languages/sml.min.js
sha384-L1O5Yhl6soBJLqbZirtmvxQYCZKJ+jaSct8ExTTG/b3YRsM9JCHOHU4Ks5maVWL6 languages/sqf.min.js
sha384-DjNK+pT5U7Cn2eKym/l1D3Uoosy2dBkTUp5HPgV0YyToaVwDnvQHUPdkSyNoVHEJ languages/sql.min.js
sha384-KJh/jQDKcZcotn3F6toMIvb7SqZlLEf+Ky67Ep7Cei7cFW0DIKDdfh4zuWWLE/Bi languages/stan.min.js
sha384-0/aUop8aVRiuBPUyL+fmlU2emuK//UukL+N+cAPNhI+oewI1TEDanaMGom+WM/F5 languages/stata.min.js
sha384-fUeA8LWLkCdKS+sWf+pLkbvlVh1bu3RbJ62Tazhc79COY77yMf2e6+gfu/8aicaV languages/step21.min.js
sha384-soTQK6xJFiYvTeXdgg58IVjTpQ+mhWlPgClaoZSBR9Ug3h2O24RXbWEXAWJOJboF languages/subunit.min.js
sha384-nLughFrtiQNc7iHQeMAez7JSZBCsqSpMSyggNWxnucFq+Q71HxTuDdCpoPhIYbFz languages/taggerscript.min.js
sha384-KgFjJBWGEfnWbTBawNjqjX4/CDBzxZ00okJmKWbfZ2KmVaqpZRbl6T1dSnLjqp48 languages/tap.min.js
sha384-xYWkYWaYMMxFgCqJKKIagCeaTOlfvG9bE2RGtidsVf3dYMwXmrq552QY4yy1Cccl languages/tcl.min.js
sha384-wdi4XGIW+qjeYxDLr+KMoNCzD/p1tq9DrqEtsHdn974CfIuiZ1lxehIHp8zlxlm5 languages/thrift.min.js
sha384-Ax/9GCZPfWr6jPwnISFEWqsHLpKGvsHaL+Mj/hi1D53CV/nkW76a7TrnEY1Yqj8y languages/tp.min.js
sha384-uy5laLz2Fwfmdc71+YP+GRCRDb2KHocbHdmMMVI2caL3Xk7gX1Y6wXdZ0Ol/5NlD languages/twig.min.js
sha384-Rfo8tTbMHgwi4YghNa0YZwz9/1cHNeojRfcssZOFI6+UEGF8Dtp0JO0tDjeaEHMH languages/vala.min.js
sha384-QZzSUNVda+CvUm29FeSGDhxp6x9KXYhXh9/K7A9EuCQi0DiUCioAR50I9cKYXU4P languages/vbnet.min.js
sha384-5UKOXNVLIhK9k5wASNBtlqHnqadYVXD/7WuqOHS3bEEE63IDcYb5t2P/Hfymrryf languages/vbscript-html.min.js
sha384-F31pU9+rtzDCPCPUOdN49f5j+LL1QIbNCEDoygI5CPlLQViZEwkvyiJ1YcTS2BPb languages/vbscript.min.js
sha384-DUp0LMcPaYpXnWzeot4q1ueVbnM7uqYmg7vPMXCWxuhXiLZjFCzZonh2XktUeC8K languages/verilog.min.js
sha384-HUl49Vvav6HW3NMc9hWsoAAOq7oCtXO/Sxcqjb96xNTHUsP9154zi4zW+iI21it/ languages/vhdl.min.js
sha384-YXnwBG1rGTW/w+hTkIiRTv4VD89N7bx3TvFzYS9eMy4UeuUWZmR8Wb+CrX0EjgOG languages/vim.min.js
sha384-g297944pmKUQTMwYgeC17lSjBJGZ2A9Bq7iZES4tGQf18bxgpzVwU/QW/UqYBOD1 languages/wasm.min.js
sha384-3OKHE/K3D7X7Bby3LxrQmgM+85xhzkqxsXckWpXN5kKiFhyF/b5NPCXJA79XPDYY languages/wren.min.js
sha384-ypqk0MokYX9kag5v5NqFS5bqU7MAxQny5hTiDpxyIAByphCx7bspbsVUYU4ivE6G languages/x86asm.min.js
sha384-dPIylYkXShzD7u7fR3FM9F1RXpkJcnf1uhI6bMyvtLI8I6PFaCnQhcIJUqxsfEWx languages/xl.min.js
sha384-DTbsZ1fcyvKiuz9quVklNxPJ55VnLRBJ7NMVBC1+KAeIphkqtCyb8ZzI8dSieOGl languages/xml.min.js
sha384-DJeoX/Nj74VJ418tr6KOdbUW/5Rf8gWQgMlW6FZtlgRxRcfX1G151Kwr2VOiCvzi languages/xquery.min.js
sha384-poIejSSHtqMTnn8wHy0C85UlfYbGmbT/bkDBBFS/q4qP7KYTRwhZhgw+RKdxJFha languages/yaml.min.js
sha384-SkS7dLSafjbCS0wZwJHwcC6BO1m6uKXC9bS7Mg/aviwrgm3cn9DkUfcvLpmxA9Kq languages/zephir.min.js
sha384-LHE+oDF0I/eKR+faWc1Bk7XZ+egUwm8N8gj6YuysAxjkod81kVp7GKFWZqax7Mdo es/languages/abnf.min.js
sha384-WxlFfl8eaJ6oWE0CG+VxabINqfpPVA3xmIXMhXWt6mdllaNE/PiMuECOZiSKu+pk es/languages/accesslog.min.js
sha384-ayusnFwjLIVIe7UFz01WOaFEZ1gSRCrj4pvkIW5vn4raHCXbUSMywoxUluyNdRrA es/languages/ada.min.js
sha384-CKcPqn73Tx1+oMhLFA2PLcueY/uT0SIgE9l0/jGiSRNtK+uzzCzw1Mm5I84EdltQ es/languages/angelscript.min.js
sha384-dvBlxtws4cyewqB/HE+EjN+6OFfisWG0RDJhdJIpkb7cb2Q9dKjhsn0fIuHAV60/ es/languages/actionscript.min.js
sha384-/Ca/Sttz4a6MGq0yEmzwxMqLYq8fkVOY60T1JdjUNSrCnBRuIW5xPi4vrkL8PxSa es/languages/apache.min.js
sha384-1MNOTFg3NMCGIdL6t6I1au6lo6XEeryv0iKhaCqywLAITWTyIIZtdx6cn1WyKBZH es/languages/applescript.min.js
sha384-WFVZx3aKj5A4F6wsrRY8t1DCV6L9JpFR7PhCfFkNtowERvVXrnHLYWnX/Syky3Ea es/languages/arcade.min.js
sha384-8/pcinXI+ZDdEQXegUfoisr0FsmCwBHngUpHEL/NpE7I+Z18BXlbHkwPM4imk3Ij es/languages/armasm.min.js
sha384-tL24n62YTEBjp8zhtuxIbRBH1Po0369X6Ph2sXR0kNzc3L2Hq5uDEpTBuxkPeoGW es/languages/asciidoc.min.js
sha384-AAhllOZzQd1JB84SBAbb1W21BtVgOi9uEwxe8qMPXdHThtSkDUM5AXizSLrvz8Y4 es/languages/aspectj.min.js
sha384-0ZOrZFnxipiNj15nVGssH8Ll91mXjPQFDitMKPmE2NZrS+dD3XKmWOAK+NGzRAAD es/languages/autohotkey.min.js
sha384-BgK5p+gMhQfYxwceAkgRRfSt+oqSbQHt/hSvKMdrthCaH1gqJRO73AjsqDMIYtS3 es/languages/autoit.min.js
sha384-Ye8SDPtJ0QGTXYOgpu6HMVzWrsE8l3gf38kkdbNP5m/3/zdNHibB1bDR+dZm6zwL es/languages/avrasm.min.js
sha384-Aj6JtcQRzImIpq1zCWPkPA3anQT2nwW0otlds82rW4S4F+GvaqPvfhG2isa9OUyl es/languages/awk.min.js
sha384-zC2MCZQrfPFVfnguQ53bTurbHXdLwfn4vjxweL0NsmiLo/uqRdyzZlRQ5eGIxGWK es/languages/axapta.min.js
sha384-ZNQbyRl2KNY5jMqDq+03kJIM3FQAPokD86+43tRUh8zxEfAYUjrwRce6qtAvmbf3 es/languages/bash.min.js
sha384-sxUF455L3NCHLvBh/vzoUfMjp22voAvY/9y6pmPtZq/O1krapNOCC6+Y+nQCOiDr es/languages/basic.min.js
sha384-PB6ypFtHFTdNl9DFNOEJShETm7NJD0m/Cnq4sKhfusgZGaeRg1a895K0JV703LcR es/languages/bnf.min.js
sha384-gb0GDrjDV7q06grocKZ1gCmA5aQyFXzn+CWeVong6ATn7iGZCzm249x9tFznzVYn es/languages/brainfuck.min.js
sha384-ZosoGq/P40RU8hsUIjNu1V98BEJuZ7OMFccCyPem3bgq+9/EiqBeTqC74ioXjQdg es/languages/c.min.js
sha384-8ZYPouzOkuEbUgODPLwAsq/YMJVYXAdfCOsxjyoidQ+YCrkG7txPr7aLlQFGjazP es/languages/cal.min.js
sha384-AIOQaZwAkP7/JtuS59cHlWgxVNLSAfUhT003DZXdjsFXdSj+f7Z417k3WLC17qSE es/languages/capnproto.min.js
sha384-ihvM9E+z3gu0V4AEUI9+jPZ0UkFCtAPVcEjcbWWn20voWFh1q8M2+3c3yZ1nuNAq es/languages/ceylon.min.js
sha384-B6FdBh69/2OgglPAQDwYwBDTOfkjr+owgA23RozhiNWv99PoewHnYq6sRCdt8nYf es/languages/clean.min.js
sha384-ElvNAmy7eVsL+UnemwqRpaV8CiSXDFSLkC0vvVCqLoRuxCzaKi1ghWNEsjKIsq09 es/languages/clojure-repl.min.js
sha384-ZrKImURP7Mh7W4qzxPuEnjTMYVPdO65w42aXpVRrQ+iQbLqmA3N7MQUfcoqFYoi+ es/languages/clojure.min.js
sha384-rGsR/THWBWpt6tFAzCBvT+2KpvjxTZ5f4UBXN/jPIBu8+EaHMTJH5uVKnfgSP1+z es/languages/cmake.min.js
sha384-a1/Gqqoy87+U/QLSX5lIzT0pOhLCNxPQxnzbtR7d8hVNkwx9XHuMB8eHWMqqUl5v es/languages/coq.min.js
sha384-a61A5gGDkJYaVRaTLdT8O2yZvd4bcI6ac76TF8eNbXJIw1wWSszhqsDKA+E2MLb+ es/languages/cos.min.js
sha384-Z4lQX1I2jpSTIlcToQVfXvDxN4zvUqcunALsNmZxpTLp0EZcd4MX/MznSuNMrgyp es/languages/cpp.min.js
sha384-Cx2u5ILwAvu8k4+2CE7nmYfqOycMs5ARcF06NVE9XvM84MzfHyIvnLlf5Hkk9sKk es/languages/crmsh.min.js
sha384-UU0LEV8Uelz8ptYD6yVlzjdFH0WDl0ZJ+p3pm1hA9OwOh1vyZAh38rVcHVNlm1L3 es/languages/crystal.min.js
sha384-WWO/a3GAp0JeHR7DdScT9DmJadps2zJ7QwZqTiUNwpn3LH7cJOPSfujO/i/aUWbP es/languages/csharp.min.js
sha384-ag7TLEnJ67nwppphXqWAsf1tICQbRFriFJG32NtpXzCf9ZnYO74iay9JdX6iZNrK es/languages/csp.min.js
sha384-4G3TdN8xyGHtlqi/0ZcJ3oQDX3RsSJjdTNtgbamsuBaHXY0Rmu6GCSd6YLMAYBRz es/languages/d.min.js
sha384-FiPHZ1ZqmvdfyVaYXTKTrRgaDDXeyujp8PsEHNbvtmSQFy+Z9U9+5300SZMl2Chg es/languages/dart.min.js
sha384-DmVXSUfsr7sJ+f4JJyaXqvoY5OqxzdONaBhK4dx8hVa78qtkyA6nrbXb3kYgY/dd es/languages/delphi.min.js
sha384-f3s8ftnbXfqA0LSPsO6X6tFRnyzvCgO/yMYVv6j6ISKvXM1x+dBv2Xx/s0ZzDgaE es/languages/diff.min.js
sha384-ODmtyBaTyUcc4D4CYO1XR2hrsg0EbpUvnkgwmlCoukeendKjsbxqpoiEZF/mLZwK es/languages/django.min.js
sha384-0L1tlwkA1EPS9ueYm+jAZodhy49LcmjkVddvHEMe2ZX9OLkxNWLTXlm7N0wJlga9 es/languages/dns.min.js
sha384-eiYM+sJfOIQd1G1DpHYChHe6kCk/ZnZ/IjZ3N0nP7a0rawiVSFmUNfCmq2q32T7q es/languages/dockerfile.min.js
sha384-7tumswrNnoCz1k7mYiyrXEVgToc6jhcJ3LFv1xKBkM3vWWtxfwBvkfpDvMxr94w3 es/languages/dos.min.js
sha384-Feb+oyJmcJXV9jy8S3nf3Rc3ly0xohcw62ZySIeNkFQSbaFgWTVwEi9G29ahcacF es/languages/dsconfig.min.js
sha384-1WmgcZIlXDcqWgWSgw48s+7jFluq9C3BOxAh3+T44rCjZR3osBb+hDhnJ5E+9MZj es/languages/dts.min.js
sha384-3lxVrXGwTu0VcQKoavPHh+w/mFpxGxvtRoMno0BY8+kTpBi3U9wTJojoeiKZnk+h es/languages/dust.min.js
sha384-9fTlYrEvuxcVLlFrXBkZbKwyNkAs240exXUbdZZ2vBRCdwbJnwMKNOul0iWu46zF es/languages/ebnf.min.js
sha384-5tuOraZQ+kO5UIoLHW+lACRvXMAHfSOsglKOEBmg6cKp6dLEkuOcYSyMKkOlMUYp es/languages/elixir.min.js
sha384-rrbiE/rX80Eiiq7ejKRSV9H5oScvfkxrbh8/BbGN+rQ30aN3c58dFCdwN7gncV10 es/languages/elm.min.js
sha384-toFl9fYM4nod2MGr7GCuGIPan9hKW6FE0ETmG+0XJuu2oORMTm8PZgOQ8zOi7ezW es/languages/erb.min.js
sha384-lX9XBpYNWJneZhq9aFb+Co9axoRCzk41Lld4/FBDOBqelQxBkkidJ3KgREM5dMf/ es/languages/erlang-repl.min.js
sha384-IYOj7tAdI2vucyeHMc1C/ACUqgYUf2Ru3+I7K4leQ0DhIEgh6/htq8ws3JNUHbXC es/languages/erlang.min.js
sha384-S3Cws6p4SeOJp+dZlH6CY3uytQnsRRx9TJjaA7xBCe1ig9B3XNsbWWgSo+LZEiqg es/languages/excel.min.js
sha384-ijfsAiUXcugyRFBjm4nHcPsjtu6xLJXSPotCUIW5bFRbAZjjzqyu/Si0sthTfR4w es/languages/fix.min.js
sha384-PL/3c1sgbaJeQ3lH01pR+OLe0luVDNFhnwDg6g1d16exj0gh80U1kPBwV6PAb86p es/languages/flix.min.js
sha384-R9Dguo5lJT7wDbdpP+wPcfzqvoaL6tNFx5EJJcBmd4MiosSJjyCqKfwHlJ5GCCAw es/languages/fortran.min.js
sha384-pLdKXvCXgx1p3wCeqSDB8RzDEDn2TorHxd35EQ+Cx+WmR5uOM32phtB6pX4t0lea es/languages/gams.min.js
sha384-v6kSgaB+f17FUw9t2cIA320ERyr8gECwLGIlB5aipGNzfigDVyEx8Xy7Sp6ValWT es/languages/gauss.min.js
sha384-cggXcD5feZatj8qBw9lObSMGOS1v4LE65YaILy1dleC1ARQ+xflj3cf1XtSHl2XV es/languages/gcode.min.js
sha384-yCe6PH1/6KPERZF4LtDy6CJPOBMkOlQ6PnDJ33jKCNGGLx3IGqQrd54SpjRbjH20 es/languages/gherkin.min.js
sha384-TJxqlkbzaclRtH2NmWT5QUT0PscfOsxxxif9K80208mjXfiuMCh/U3/CpDjDYQ7o es/languages/glsl.min.js
sha384-YkJZmdK7vabBTjq01KGqAuD22OQI1skTEImmxcrUVfpP3VZCUMf8gmWKX+2a9dsa es/languages/go.min.js
sha384-KBC3OY6Z+iThYSrABgcsE/4VrKbyilPm7w7D14gUV77aB5We1NukEhkKf66vCKVD es/languages/golo.min.js
sha384-JfZgfBxdoPoIpMOo9PQjEsrNO+Kh+paI/ZwTkThLMzbRGZ0tz0T379TT1r6apsNc es/languages/gradle.min.js
sha384-9kvFAJWSrrFhIfdh3nDwk+w1A95sxr3VP9mGi41lmITq8Lk4GSdKacPQFcOyZXQ0 es/languages/groovy.min.js
sha384-xEjaIsg5Mc4Cvy4WU7q0FARHOIZHXNLp34VH2H5Ha8sWlN3Iy8EELr4XP/Oaf4Y6 es/languages/haml.min.js
sha384-nft1QOVEkAetlArWYLJW7SZgaMNvNcjXEG5gcRIWz1wHnKGFSRZijDBm9rf1BXeq es/languages/handlebars.min.js
sha384-lfQ9Alp4FCwUFEqSO2a33PqKJ0E+ZSM4oaFqVLbtZ22Nn7EOMR8apmOyx8WZ6tH1 es/languages/haskell.min.js
sha384-/WqEOO2r61I6iMLZ+xg4JkRf09GeA5CV/ELVIJks2WxF03G3JhADsicQfwe/Xyn4 es/languages/haxe.min.js
sha384-0p3bB1a+i3j2FQ0BuVyrE7Fw3d96mHLBYtF4liyBF/fqz+CmcnYliwFYNWP/uvux es/languages/hsp.min.js
sha384-yh6l0L76vEJYt4z0NI3ABMZZYk3egzZOK3OIhKVRMVhKz2o/7VaQ34spChku6Kml es/languages/http.min.js
sha384-TQeoFip+To4VibLYU9imRjxRD3acKnqN9k6PtIqmPeLbywFOybQF/Ck5zqdNwrdm es/languages/hy.min.js
sha384-90VXG5fPhhnO7XFzG3BVyMLtUn+I+lAJ1SL7U6oTWTNxnZbqpcvd8Aawp3lkZ7U3 es/languages/inform7.min.js
sha384-sTCqgHc4pdQ4fIJ/1B0t3a6uG0wxzItSxeQlLFsxRAetefkJGv3wg7BBd3+EoAyk es/languages/ini.min.js
sha384-iFMD8svDyGwsUsRX+Y44jXgOba8d6b8p6fR/xVoSP39pYqi+ET8z2WwYZmG/Bq6q es/languages/irpf90.min.js
sha384-eEn74gv8H4VuLNYINDsPyGmwuG16O2g4SSPCZP+oo5n6n5VVrvE2GkmkhHTIKbBm es/languages/jboss-cli.min.js
sha384-C8jbRVN+xx9h/D8SEjJNu9IFCCbZSd/JhaPcopTOwHyh5EAAGPUC/gH9xaDbC4Fs es/languages/json.min.js
sha384-54EGoEv3vo3LbcA/APD1huFDHUX6XbyKq4PlFTtvVfl8hhDop8ocWKTitrJHA6/W es/languages/julia-repl.min.js
sha384-k+bWRXpJ4t9Y07djXR7GhZt43U+8N33hLo3GvMSWppMCSSxaiao1nODNxYmNWZke es/languages/julia.min.js
sha384-6g6L4onhLNZ5/Kw+YLNnZJ8YyZe9DCVeLvk5to5B+6Iq+yzYVJnIjE0I2YZ7zEOX es/languages/lasso.min.js
sha384-Rt4sFjj0G+Pj4Q1SjUCdf0mPb9B3D8c6qEcajrU6ALHAFjGtcseOMGCnlCd7/+Ed es/languages/latex.min.js
sha384-M6Uzd83Tjl80685EX9qINKmmIOnWu6i+o8TvtW81CoGVS1EOsCHzXvKFv24mQeTF es/languages/ldif.min.js
sha384-+14A9y6+PAuQVUOr1gs0ovx2g0IQE03njDYqKBlTY8dfexwmDo9N8qY93U7qfw8M es/languages/leaf.min.js
sha384-/SyU8zm8nH5a9JI2oIgdGWJA6cC2ai5+iX9Q6T3KtOrGAYhkw8/yo8fVzaiKAlaL es/languages/lisp.min.js
sha384-wqib0RWo4+gsQbh0Awa8NZD/S0JWm72iqMpQFJnL9/UuE64VolEHdE5xg3ef5iFN es/languages/livecodeserver.min.js
sha384-5O3GPy5scxa4gH3nWl3NA4qHLWJnRZddJ+6liC/AsRJ8d9KO14hihFMqDBcFGm7m es/languages/llvm.min.js
sha384-i5SPJtb+BAgFAYwYEQ4OG/OwHGwBYxtAlcv3Adm4svyP8vzW33A9p1RsXRbP6sjR es/languages/lsl.min.js
sha384-Lhb8n5Na8sTYLAXi9aC6HsqIyndRtkTiUGGJjL2oL//oDxocNmxmGKMIu6QzUzsJ es/languages/lua.min.js
sha384-BLQYoykx+9juBKp9q1XsteC53jzQ6GUUWa1pFkQA1KYQn9RrU/rn2ZfbKubPQ/a1 es/languages/makefile.min.js
sha384-7HgGmhsyDq91dDXC+gtTLDqCAX14KTs64I18xn+GO6gtDVjnFI+UhelNu8Yo2qN/ es/languages/markdown.min.js
sha384-MwGBhmtkmnf1TCnzqGZQdzEWJ5v7pkYEb9LoOl4q8slRf+X6NJKTf0hLbP3rSM1P es/languages/matlab.min.js
sha384-Q/vz6NHTt+uuf9S/NCn9XBDLwP+AUNiLicT1XpqBCHs7tTfjFlbg2OA1Fw7paNtW es/languages/mercury.min.js
sha384-AvCaPz1vzUMRmR636JmmEi8/Xmyj0JwMhoiHDXSXs2N4BM+mzS+X1dzrxIaNgCKM es/languages/mipsasm.min.js
sha384-xmqqMq8/0k9LXLrc0eYxOe4tJZIb+6bv8EiUwgOnHiSYjh2/E8OQPQidCJ5oEdjs es/languages/mizar.min.js
sha384-jRMjcXaX7d7HZqa1/6MYRimWPo8xq965Y7pVVvF2WvxYpD337mtFWRKssIL9aFVf es/languages/mojolicious.min.js
sha384-nIIQ/ScNTOZzaw6oELiOg8FshlG75YHZ4uMaVIqLI4d9ME1XEFMMWx0FywNt02vt es/languages/monkey.min.js
sha384-KblzkOAhEwUWNhREdW4ZnsUM0YOGWO5xM+uGbuGXiZaFnnJU6DJ2ck1FCBKyjVtu es/languages/moonscript.min.js
sha384-lLzvHd9qZK8NrNTj3Dt3SmaYQfcG6BalR4lXHRU1uAwcJNenxA4s3mEFR7ZeBUUJ es/languages/n1ql.min.js
sha384-l9nvpsdIbZSfzxLt6OkLzYh1VmlLtrWmZQj51hja+A6hUl97ULXmj7hb5H2AWBki es/languages/nginx.min.js
sha384-aXwQ9FxIf7rE9d9Hq+mXoDMoYd+WaMTDIfgg5iU55BoQxK1jQYLUoVSUs3X9yJQv es/languages/nim.min.js
sha384-l4XFqUCQUOQmFMoVkO5TmzilODbi2GwpANZZ4Nu43CXOUYNhoc4QSiD80BmRyYub es/languages/nix.min.js
sha384-hjQwDdDqZ75zV26evlXvDfornpveqnz21convIWoE/dPE36CMcKupoZwO1tk5HK9 es/languages/node-repl.min.js
sha384-mJroiPX7owy6r09Z38Z9WIfkjqAUyHBoDNjIr10hS0SwnTMrhuLsotdpFmrV7Pwt es/languages/nestedtext.min.js
sha384-IHIFlZwAzdaNUUrTO5+OjH2ndizNWwPzEuwdVKdWoMWLBOIVhDD00DKUqeoxvYws es/languages/objectivec.min.js
sha384-6GvT9SaWnXGY+qyQEzC4sPartpPzyTCC1cZQZjQGdEVyQgQs5m5jlVX+fDJ3y0/x es/languages/ocaml.min.js
sha384-5HaTT/OL32s7fgBV2XwoEt03lDUMyQBxabrA3Gb2WE6NylyYHud0cIQ/uyo3wwmL es/languages/openscad.min.js
sha384-zJDs4hYHAaAQ9nwTuh3C3UOmUPzvdWZgs9DpdkI+ELBKPyRtbHpGxOLOeQwB0at1 es/languages/oxygene.min.js
sha384-aXaZUGJqVsFh+W3/T5Y3Ud/oQjc8SJWga9kKDKLqH3aSlZeiJSEYkYALW/aNr3ft es/languages/parser3.min.js
sha384-lPcL7dN3vL1ll1e6rG0MS7j6cYMusxeZYf7rWsl2v9F6etcWD8fqwPHT9Kfhn9dc es/languages/perl.min.js
sha384-fyYrRnmq4Hgqzbfh7ZPYGKumEQBLQXcM4ayCavHCmhBE8wd1OHHT12x/jB/D3G6Q es/languages/pf.min.js
sha384-noFxK7ADXySvgQBIJsBuYXU6H5ZEI/Wxk+EQFq8l0E/HjpHzwjFJM8AeqNtOwrSx es/languages/php-template.min.js
sha384-7KBupSyvzKFgnsJNK8UqTwv99nfFdI/gsJyZ/CgY48EwYwUKTdVsKfYjEsLc12G5 es/languages/php.min.js
sha384-xN32vlD/xclVY1z27oMVido9LDw2m7UffnnTBhOG7apaOBBT0F2jaCPoCBoN6LT6 es/languages/plaintext.min.js
sha384-Gkojuhmaq40KoaMKLkhD7IeYhSyYkSvylMH3JkLMkT3SDrAcGvm1pJsKaiJcqxSJ es/languages/pony.min.js
sha384-QUWUQR1BuxJG4+zV/CwdJmYGJ/kSbqYAN6Gd/sIHn+9xOb/XGg2ZqTJWeLFJb2gT es/languages/powershell.min.js
sha384-OBDIdTqkzG+oH1ss6ZbHWXs1rk+7J1Vnj4mCmmBD+kTP0WJT49wZIcxGlni8Kazy es/languages/processing.min.js
sha384-BZjIn7/gA9UBWRt6YvPIvW6uWacoG3+U/Wqb4ovO1G+z9hcrHq/mEzwC26qGhnWL es/languages/profile.min.js
sha384-3axapkKbINeJfDNmYji6PYM9tUodjVBwFAPUKQppPxxuvVKpPcUmfQ4isM99N7Lk es/languages/prolog.min.js
sha384-td8o4Fo1GeMK0f60YNUTNRftr3/PWnoK/9er3AQJVFIlftTeW73kcHMSEHqfBi1g es/languages/properties.min.js
sha384-alJ2u1gTuHtqts5SiU6LPMkTgteQa68GsHrUCMpLy+FJENz//4QM8Cthq1vBGXhW es/languages/protobuf.min.js
sha384-7MjnOm5PSrsafvqzsIihBlCOkFDekOywL9EW/HqHLv5eZ+mgQpjWHhcUTcgrWOXZ es/languages/puppet.min.js
sha384-LdFzTcM7Mr0MroDC0WnXPddX69xW/9TG7F6x7M6u9EWoIWrv9W33NAdG5np8SAMq es/languages/purebasic.min.js
sha384-oD3Il9cYzDaryo6xy22GLV0jCinwSI0CF9ljZiSbMVxu57Yaw0tbsSIsu1haWmE3 es/languages/python-repl.min.js
sha384-4M8Q2aYpoZ36SUVYPQRlb7KMl/5s7tPTnN9jzcpyz8hkYyKezUeFn0VAa3zW/4pq es/languages/python.min.js
sha384-YX5EZctppLhL1NhsTtxTX+pk33cC3/LJILlAyVm0+b5H8vSX33yMC8qNWWeX26G3 es/languages/q.min.js
sha384-3nDhCd8VmqwsfealNiD01udu3k3UDCEeAhbyMTmOwHO39GAJ8K4xDhNObVl7jJUi es/languages/qml.min.js
sha384-xCty3MNXVGE0kUWKDENktaNyZqxYCajbnKR1yP7e2rg1lcGLuVcbyWxIqferPl80 es/languages/r.min.js
sha384-TNCDDiL0k1ChZc+e2iuloHUmnLzT7iTlO7kDiA71LmmhFnGGCx3NqyPkSO9Qofa/ es/languages/reasonml.min.js
sha384-FG3AT1Tw3nBFMF1zXUEQn45SuPeC4LHtHV7FRdtOrwQd/pb+9QlEgQu7uR4V8ybd es/languages/rib.min.js
sha384-B4FmaO4vJwycsY1Cn3nMXmY6V+n4cnTWqhcLY1INRIKqbyTOz0isZ8OG1G+/g4hA es/languages/roboconf.min.js
sha384-q6G2BmrQCCS5p4M9N9Sg/ikjuWv7pM7IUF117wSdmRWH3QbAY1n/aoaDDGD7Yd0q es/languages/routeros.min.js
sha384-O0fSxQnFmgOhP1U2cT+Ofc88gClrz3Xz9aRe5vrtULvYv7/ffuHkaZVma6rXPWes es/languages/rsl.min.js
sha384-vNsOak1cSgImpBsiJ5E4zHUnJgT+HzLgwdKRZrZrQQX3/GvYEKNAlssQ9Epoxblu es/languages/ruby.min.js
sha384-rNX507dy4qvvHLPifn9rcdSW7A/xkCIjwzDj3lQXq3AhCBKooZ0+FVfWy5qQu2Qi es/languages/ruleslanguage.min.js
sha384-N3PPAyDQi0dSXOqZupd8F8mw2ZAD3vqq18csittS4pwDaV418iUMtl1Y7jlG/BH6 es/languages/rust.min.js
sha384-Is1oEDCxR51/Nd3bwXAvpJvZNjjsIHqU8+LxQHgnZG4antZ+vOU7bvDlJTjTFgY1 es/languages/sas.min.js
sha384-QOzXfaBvt/6KtfebDC9yejcxb1uAiwBwiOLt57IAk+sECol0X3ZnvyKGBLCs/VCm es/languages/scala.min.js
sha384-6zEZ0Aftfm0siYEXQ7NZchHT4e62OEVq8gAJm+As9L1XRVYhc9avmuuWRW6l/3ZJ es/languages/scheme.min.js
sha384-P+XbALgLzbXMnPyWHF5kJd1kdrK+Pa3jX4MY8kZQvfmThex1ZMyOS9/H+q4qEspf es/languages/scilab.min.js
sha384-5BKVfUi8omm1rFx9OoX8sZi1NYFBLKOATXpXi0Rg900SedHfXkFRsBVA+ZuCa7Pn es/languages/shell.min.js
sha384-bnQD9xAbNnnZ1xLlXemUejLrmDgZCO5Gv9SUu4ZbjqkhSE9iPhi8GDHF8CygN557 es/languages/smali.min.js
sha384-EwQitxpLCqDjbEKrK2IRELvoEnLl8RbAklkMRjNLNrE9FcJCk1fhRkYH1VfzEP1Q es/languages/smalltalk.min.js
sha384-tuHIHvuFs4BhESXqsqaaGbinYdi2zu6NrM3nE1cCHhZQaeDVUG50jyG56Q9UER5z es/languages/sml.min.js
sha384-lZmvQrhMRW2kdP0ChYBhHgobAzNvQyCiQ92z6P2nIwwn3SfxgLwrDyBYCd+OUg8C es/languages/sql.min.js
sha384-/llHT+n638JHZ/1cdZTo35lBN7DXUom+7laupHud+cPC3zbwm6x/3k8Dt3jfFNAl es/languages/stan.min.js
sha384-uOl4WV4E5C3DmiftqI4BnzvV2pbrp0MJEAKTDYWty7DdEmXxv+kFU/0Xzyh6QeOQ es/languages/step21.min.js
sha384-OEMQJlMITwCdTqu3YBVhgydfHjGpQ6tNcYmuUU5JqA+Y8bYECBVuuwIRHa/3q4So es/languages/subunit.min.js
sha384-FsdfBT0ADC0g/iyQp0kBdFrQ2WF7UyM+V3FkHi/MoMt6H0afOzktD25XuNWFkkqf es/languages/taggerscript.min.js
sha384-6YDZCGFdZtQxK7YTP/BN/vT3OCR9iQ5SYfrLYWj134xaeKbHEDsa1Ql/HarEm9Kg es/languages/tap.min.js
sha384-dtikXZDXWy/2PosCKZkoMKkv9fiiDG9cainK6Snmtj8e4GdrVJuKXA8BpubERN70 es/languages/tcl.min.js
sha384-gjx1YW1uTmVmoE+PxPK33MNkvlLDA29AzUEN8bKRdZ9CDkrUm863OhZ/9L9xehH4 es/languages/thrift.min.js
sha384-mi88JvVZOAuh6x++ND/NYIBII+kDJVhPjG2BrFGKKvnvvlg39JB5Hvl1BFtSDjnr es/languages/tp.min.js
sha384-NRGIhPw7i3ewVuJeU6MQdgmkiDOMrSj5yeQoI35VebJidp3NdQCagfYVYhxPNgBH es/languages/twig.min.js
sha384-ZsP0mQMaN7BwF/LO4XCxBH0sXgY00W0um7af8r0+1mhnbfKYTqMRO6MES7Iwdwdb es/languages/vala.min.js
sha384-Ut7xf3dOsMuWdK/nGk/N3MGq+TZB+lwFb+mATv9I2yoNq/Ac2zdh99ifv5b4s2Bj es/languages/vbnet.min.js
sha384-I+vIOQNypq9RspiH5kKZuXgtzFjmYp9DQPI162NRT5CuJq3+HWzU8v6X4rOzCifq es/languages/vbscript-html.min.js
sha384-ezTgxOQpf6gHn6aGOoXqiYbLboSrEBHt65ZlZxDPynCmJm5SgjOpdwhXI24ABjOu es/languages/vbscript.min.js
sha384-IkhbY9Hpvdq6om5wfENQNst7DdlC/q3h1U3eNN+Uw5EKigOURXj0iaj7AXjfOi1t es/languages/verilog.min.js
sha384-MH4cKoksHmBakn7O/kK2KM0g+BCptdQIWedgXAnTnCnbTuJAdtbcLPHAe5ttND0/ es/languages/vhdl.min.js
sha384-EvskxHJ15ksWVolpd0Rs049FjkTn8wi8ZL9Out3e/TyfChiwnMo7Pm7kJBhBvqYX es/languages/vim.min.js
sha384-hZKPeYMwb/EVxGxan9tMLE4BEFBh0CnjjLXyma4krYIgUSHChgJl8Yxd2Fg6W8hs es/languages/wasm.min.js
sha384-1MUhc3+QY5D0rXVvRlTDD97SXpoQkBm0nGmGuzMPF7m4d8z+pMqCOUCVZ321FVbC es/languages/wren.min.js
sha384-WW7dPtANXAeJC59kAJIlhOKJL8gTEK7ViYLpMlo+7OmcfsRu5/5v4YLxiLbEgt55 es/languages/xl.min.js
sha384-Ql6E/VEDAIYFA6C8NX6NJL4SUSk0LpKZc1N/Z3xtDG5aONdIPb3my3k27QQSUofH es/languages/xml.min.js
sha384-Te4ptKBKkWqKOXzG824vSTjpm0b0YQYtGwsvag8woo/9S99Xsxw+yJc8HYO9+Y7r es/languages/xquery.min.js
sha384-3S7COLf5Kn3uktVRMmGz5cdwyDEyjHzxRoj6Qvmz6w1Tj9YGz5cd1Hhfw/D/Pjlt es/languages/yaml.min.js
sha384-mpsGH9BaNDHGVtPn5Q/DtWRUgBY5qcv/WWsFBlE0hzv2k492+268qJQilFh4CTBr es/languages/zephir.min.js
sha384-Cd4qrRC3Ml0xmTZSFX6+kV/YF30eSj8aGgx94a3+On+mY927pv1uJbaqWRCr58pW es/languages/maxima.min.js
sha384-iEif9Xzcug74fkyP+32U9AByPQLmm7ed42bME7tIU+UBWKjLhSSFojxIaD9o+KRZ es/languages/mel.min.js
sha384-SRslf6zysLjUWPg4z8qQOpoIW9iMKqXZ6GpGI2Jc1zdPgumtiX/Xul27OjlxzL+n es/languages/pgsql.min.js
sha384-A7vZIQg3Pz5vm+q4UAK6ZoMJQYuwHJ5SWCxtVCGpwW3Xo4Rmw5D/i+wGsBqB1XcD es/languages/stata.min.js
sha384-EwGroHp4Z96UU6C4WOKfvdr33+S9uRnwGL5NhRPuzhsI/dVMVcmtV9LHvHdD68Q3 es/languages/x86asm.min.js
sha384-SC/adJ6M3pCb2b4mAUaOIL64d+PKH7eBURMeLr55VNJxkxt+8qKUR4hLP2lILBMJ es/languages/sqf.min.js
sha384-xs0tu/zPu9qlzueJ7LJZFdvljzT4udtqddJNFgFapNzatPl8ZI+soCp3iOahRV0n es/languages/gml.min.js
sha384-HEmaIvJ7lmR3/5U/TxuDkuZR2mK0zrScKRFrpxJiCXHgeQsrER1tmNxR0K0NS8AG es/languages/isbl.min.js
sha384-KGq6RUMFkWixaYQ5RokdcykUcXW6ZBMurMPpEvcok/yKmzsl6FtbHotsPNHhVxgP es/languages/1c.min.js
sha384-g7Xoc+g8TKqR89MsLpuskhY4qw1Y0LZOIZYPozP0jKVO4i2dtkHFiJpguSRUg99n languages/arduino.min.js
sha384-e5L8XpOCq1PymxKQ8gTzOPTQU4bj8Tbla5cWAbfdxZuVt37InIAgBsXb8igCrKGd languages/nsis.min.js
sha384-OGkN8/RcxRQEQoSZpew8fw4hFoe3kliptn0fWrzUHBAUcJKqjhGDkiYWftk6e1XZ languages/fsharp.min.js
sha384-KK/LFNrLC6mE7PVaYz+Bvogn+F5qF1eyDNg9IX7BUZ/pgxrqQqMccFbsF3V/9f8p languages/javascript.min.js
sha384-B3PtunlDsYZSdHX2/6GdQzRNtUN8U0MTLNzXucLZ5RinDbQLWK9F1rjQHOIjFtTz languages/livescript.min.js
sha384-VHQmPfaW8CFfx25Ud+c9+CtpI/0d7BFvUf/c7ghdCvbG0ohKsc3RBGQOfGIG4fTJ languages/coffeescript.min.js
sha384-C2vOFIepsBlxqkquZiX9KPrc58siBkfg5lBfJk2x+ZXOjl9FX+nqzgWbOYTMaKmG languages/less.min.js
sha384-ga6VZF1Eo83z0MY8IdPOMIRMJjInSh+HLJ1Ued8y7oeMCqsUiiB5N0GDQAPNd8as languages/scss.min.js
sha384-KftpUlDT96tJohPMbvJjXXNqC7iqim8TiLzWipvOAsCN1X4iwOkotlpwt+i6aYAd languages/stylus.min.js
sha384-t5lSKiOUytGuOQK0Y+jQjn9wxgQJ2D3Urj7iGIrTFJny5ojOAKvRQ1kqHgI8HQvg languages/css.min.js
sha384-cFGR/ZAYTHBFL18AapAt0TahYiWT1Rv4OEERh8c+lsHo0+nKXQVLZ+OI0Fdu2C5M languages/kotlin.min.js
sha384-gF5QKJdieotbGC54ucEjqKRIHC5Mas9W98zu4hHylWn2Nk0FjML/uyS4uKQwcH1x es/languages/arduino.min.js
sha384-Zjfhs5ojY24r1+shgibebC69BWPsIrRvA0PrfPmiysvIv9Ee1RW6tKXrOmijjDFu es/languages/nsis.min.js
sha384-CClLxMjCg0Nc1AVFWGWhIra+6vN6tCzwAaNXtTXbP+3P7TUyJV5BOOzeyQ7Xc5RS es/languages/fsharp.min.js
sha384-nsd9nKXKWqYf92fHQK76zEs3Cnhc9FQX0RQy6F0VylGqKz47sntvRJO1OhpX2dvn languages/java.min.js
sha384-wYfRQuFHePHKBbtXb4+5TPDQiUe4jHiEYJnSYqWzrqdimeBBeJFgX+dyCVy4ZDDB languages/mathematica.min.js
sha384-04yj4OvZPhkr0qGXvvIZHkzOObXNpNQ1zkArMgpieaYWrAwEUu8+y47lKiXuHMl9 es/languages/javascript.min.js
sha384-iRW+b+tvkaJ3QO+f9JzDGAJ48kkUOWXwx/CNKP79tGV6RSlhssmBotwEyIAH6Qa7 es/languages/livescript.min.js
sha384-WoXicMclVUL79ziBm2KCiYG0rzg5zQ1cGnxcbpQwaiht9DfWiy6TuQVSkcEvqk06 es/languages/coffeescript.min.js
sha384-qGqCb+ak4RPbd3LvgYkuVqIefOB47iQ/5yiC4jbr7rP9VP1DEd7zcwwRmtWKBlqn es/languages/less.min.js
sha384-OWSPtJB6DQC5arNQh5T9t8lK1K6LZg+pdJO7L+zspcG7hML7UdE68wSPVmA/brho es/languages/scss.min.js
sha384-catjosrAYYzFvdRRwEa417B1RcT/PSQ1+SG7FXxKpfiADsdb2dmE6ZbjxqG1X+Hl es/languages/stylus.min.js
sha384-3iCNN/yC+mN/DzqxZ9GbWu0n/oBM4RsFc0R97y/MU58ax1VpFkYWbW2XEGr8rDwx es/languages/css.min.js
sha384-3/b11FHGwx0Wm/h2E+3y6IxNhMQJ5Y+I6r391/3CyEXuMyTlDkGjNiQiBWKeL+DZ es/languages/kotlin.min.js
sha384-+vKw6ovTZOu73dBiPXPZ3S83BWlozw8y4F5nwvt2LFMoE133JbFzhcXgFr0ZRhjy es/languages/java.min.js
sha384-tOoYg2nM23MY+pNTSdtZCzTU/szobWjj93dfe4oJecTlHPnSLVO+/YnOFNKbCK/i es/languages/mathematica.min.js
sha384-i9x9IuDWLevOnB/5vxxItB0/q5y1pqdg7dhWE/Hi440rBc/teZ0lpGvVvT7JQTbF languages/swift.min.js
sha384-QoZgNAeajpdV4waMc96JTMiLSklx2XoNwQ+658rq97xjwI7LIiCTJUq+g7tMr1Sd languages/typescript.min.js
sha384-hAYisSVc2zuSJsJcI6cXOwq0zdY0F6ajptCp1D5p7mRCzv9PUR+AtsrRvF1ZSHpe es/languages/swift.min.js
sha384-hKqQidg2dE9Y2EImxc5u/5m2vayUR5cC+7ct2pg50QJioE9P2SPuoLOhtBHM8lHV es/languages/typescript.min.js
```
