import { ISecurityGroup, IVpc, SubnetSelection } from '@aws-cdk/aws-ec2';
import * as lambda from '@aws-cdk/aws-lambda';
import * as secretsmanager from '@aws-cdk/aws-secretsmanager';
import { StreamEventSource, StreamEventSourceProps } from './stream';
/**
 * Properties for a Kafka event source.
 *
 * @stability stable
 */
export interface KafkaEventSourceProps extends StreamEventSourceProps {
    /**
     * The Kafka topic to subscribe to.
     *
     * @stability stable
     */
    readonly topic: string;
    /**
     * The secret with the Kafka credentials, see https://docs.aws.amazon.com/msk/latest/developerguide/msk-password.html for details This field is required if your Kafka brokers are accessed over the Internet.
     *
     * @default none
     * @stability stable
     */
    readonly secret?: secretsmanager.ISecret;
}
/**
 * Properties for a MSK event source.
 *
 * @stability stable
 */
export interface ManagedKafkaEventSourceProps extends KafkaEventSourceProps {
    /**
     * An MSK cluster construct.
     *
     * @stability stable
     */
    readonly clusterArn: string;
}
/**
 * The authentication method to use with SelfManagedKafkaEventSource.
 *
 * @stability stable
 */
export declare enum AuthenticationMethod {
    /**
     * SASL_SCRAM_512_AUTH authentication method for your Kafka cluster.
     *
     * @stability stable
     */
    SASL_SCRAM_512_AUTH = "SASL_SCRAM_512_AUTH",
    /**
     * SASL_SCRAM_256_AUTH authentication method for your Kafka cluster.
     *
     * @stability stable
     */
    SASL_SCRAM_256_AUTH = "SASL_SCRAM_256_AUTH"
}
/**
 * Properties for a self managed Kafka cluster event source.
 *
 * If your Kafka cluster is only reachable via VPC make sure to configure it.
 *
 * @stability stable
 */
export interface SelfManagedKafkaEventSourceProps extends KafkaEventSourceProps {
    /**
     * The list of host and port pairs that are the addresses of the Kafka brokers in a "bootstrap" Kafka cluster that a Kafka client connects to initially to bootstrap itself.
     *
     * They are in the format `abc.xyz.com:xxxx`.
     *
     * @stability stable
     */
    readonly bootstrapServers: string[];
    /**
     * If your Kafka brokers are only reachable via VPC provide the VPC here.
     *
     * @default none
     * @stability stable
     */
    readonly vpc?: IVpc;
    /**
     * If your Kafka brokers are only reachable via VPC, provide the subnets selection here.
     *
     * @default - none, required if setting vpc
     * @stability stable
     */
    readonly vpcSubnets?: SubnetSelection;
    /**
     * If your Kafka brokers are only reachable via VPC, provide the security group here.
     *
     * @default - none, required if setting vpc
     * @stability stable
     */
    readonly securityGroup?: ISecurityGroup;
    /**
     * The authentication method for your Kafka cluster.
     *
     * @default AuthenticationMethod.SASL_SCRAM_512_AUTH
     * @stability stable
     */
    readonly authenticationMethod?: AuthenticationMethod;
}
/**
 * Use a MSK cluster as a streaming source for AWS Lambda.
 *
 * @stability stable
 */
export declare class ManagedKafkaEventSource extends StreamEventSource {
    private innerProps;
    /**
     * @stability stable
     */
    constructor(props: ManagedKafkaEventSourceProps);
    /**
     * Called by `lambda.addEventSource` to allow the event source to bind to this function.
     *
     * @stability stable
     */
    bind(target: lambda.IFunction): void;
    private sourceAccessConfigurations;
}
/**
 * Use a self hosted Kafka installation as a streaming source for AWS Lambda.
 *
 * @stability stable
 */
export declare class SelfManagedKafkaEventSource extends StreamEventSource {
    private innerProps;
    /**
     * @stability stable
     */
    constructor(props: SelfManagedKafkaEventSourceProps);
    /**
     * Called by `lambda.addEventSource` to allow the event source to bind to this function.
     *
     * @stability stable
     */
    bind(target: lambda.IFunction): void;
    private mappingId;
    private sourceAccessConfigurations;
}
