import numpy

from .integrator_types import ExplicitRungeKuttaIntegrator, ExplicitSymplecticIntegrator
from .. import backend as D

__all__ = [
    'RK1412Solver',
    'RK108Solver',
    'RK8713MSolver',
    'RK45CKSolver',
    'RK5Solver',
    'RK4Solver',
    'MidpointSolver',
    'HeunsSolver',
    'EulerSolver',
    'EulerTrapSolver',
    'HeunEulerSolver',
    'SymplecticEulerSolver',
    'BABs9o7HSolver',
    'ABAs5o6HSolver'
]

class RK1412Solver(ExplicitRungeKuttaIntegrator):
    """
    The derived class that implements the Adaptive Runge-Kutta 14(12) method using
    the coefficients defined by Feagin.
    
    References
    ----------
    [1] Feagin, Terry. "HIGH-ORDER EXPLICIT RUNGE-KUTTA METHODS USING W-SYMMETRY." Neural, Parallel & Scientific Computations 20, no. 3-4 (2012): 437-458.
    """
    
    @property
    def order(self):
        return 14.0
    
    __alt_names__ = ("Explicit RK1412", "RK1412", "Runge-Kutta 14(12)", "RK1412Feag")
    
    tableau = numpy.array(
        [[0.,                                     0.,                                         0.,                                      0.,                                     0.,                                   0.,                                    0.,                                    0.,                                     0.,                                     0.,                                      0.,                                      0.,                                       0.,                                     0.,                                         0.,                                         0.,                                       0.,                                        0.,                                      0.,                                        0.,                                       0.,                                      0.,                                      0.,                                      0.,                                    0.,                                       0.,                                      0.,                                       0.,                                      0.,                                       0.,                                     0.,                                      0.,                                     0.,        0.,                                    0.,                                      0., ],
    [11111111111111111./100000000000000000., 11111111111111111./100000000000000000.,     0.,                                      0.,                                     0.,                                   0.,                                    0.,                                    0.,                                     0.,                                     0.,                                      0.,                                      0.,                                       0.,                                     0.,                                         0.,                                         0.,                                       0.,                                        0.,                                      0.,                                        0.,                                       0.,                                      0.,                                      0.,                                      0.,                                    0.,                                       0.,                                      0.,                                       0.,                                      0.,                                       0.,                                     0.,                                      0.,                                     0.,        0.,                                    0.,                                      0., ],
    [13888888888888889./25000000000000000.,  -83333333333333333./100000000000000000.,    138888888888888889./100000000000000000., 0.,                                     0.,                                   0.,                                    0.,                                    0.,                                     0.,                                     0.,                                      0.,                                      0.,                                       0.,                                     0.,                                         0.,                                         0.,                                       0.,                                        0.,                                      0.,                                        0.,                                       0.,                                      0.,                                      0.,                                      0.,                                    0.,                                       0.,                                      0.,                                       0.,                                      0.,                                       0.,                                     0.,                                      0.,                                     0.,        0.,                                    0.,                                      0., ],
    [83333333333333333./100000000000000000., 20833333333333333./100000000000000000.,     0.,                                      5./8.,                                  0.,                                   0.,                                    0.,                                    0.,                                     0.,                                     0.,                                      0.,                                      0.,                                       0.,                                     0.,                                         0.,                                         0.,                                       0.,                                        0.,                                      0.,                                        0.,                                       0.,                                      0.,                                      0.,                                      0.,                                    0.,                                       0.,                                      0.,                                       0.,                                      0.,                                       0.,                                     0.,                                      0.,                                     0.,        0.,                                    0.,                                      0., ],
    [33333333333333333./100000000000000000., 19333333333333333./100000000000000000.,     0.,                                      11./50.,                                -2./25.,                              0.,                                    0.,                                    0.,                                     0.,                                     0.,                                      0.,                                      0.,                                       0.,                                     0.,                                         0.,                                         0.,                                       0.,                                        0.,                                      0.,                                        0.,                                       0.,                                      0.,                                      0.,                                      0.,                                    0.,                                       0.,                                      0.,                                       0.,                                      0.,                                       0.,                                     0.,                                      0.,                                     0.,        0.,                                    0.,                                      0., ],
    [1./1.,                                  1./10.,                                     0.,                                      0.,                                     2./5.,                                1./2.,                                 0.,                                    0.,                                     0.,                                     0.,                                      0.,                                      0.,                                       0.,                                     0.,                                         0.,                                         0.,                                       0.,                                        0.,                                      0.,                                        0.,                                       0.,                                      0.,                                      0.,                                      0.,                                    0.,                                       0.,                                      0.,                                       0.,                                      0.,                                       0.,                                     0.,                                      0.,                                     0.,        0.,                                    0.,                                      0., ],
    [16749674481819323./25000000000000000.,  5174228081833989./50000000000000000.,       0.,                                      0.,                                     6103444365320361./50000000000000000., 24128724516562331./50000000000000000., -381409600015607./10000000000000000.,  0.,                                     0.,                                     0.,                                      0.,                                      0.,                                       0.,                                     0.,                                         0.,                                         0.,                                       0.,                                        0.,                                      0.,                                        0.,                                       0.,                                      0.,                                      0.,                                      0.,                                    0.,                                       0.,                                      0.,                                       0.,                                      0.,                                       0.,                                     0.,                                      0.,                                     0.,        0.,                                    0.,                                      0., ],
    [7426709605345459./25000000000000000.,   12438052665409441./100000000000000000.,     0.,                                      0.,                                     0.,                                   2261202821975843./10000000000000000.,  1378858876180809./100000000000000000., -1680525334991711./25000000000000000.,  0.,                                     0.,                                      0.,                                      0.,                                       0.,                                     0.,                                         0.,                                         0.,                                       0.,                                        0.,                                      0.,                                        0.,                                       0.,                                      0.,                                      0.,                                      0.,                                    0.,                                       0.,                                      0.,                                       0.,                                      0.,                                       0.,                                     0.,                                      0.,                                     0.,        0.,                                    0.,                                      0., ],
    [72727272727272727./100000000000000000., 4684595328298369./50000000000000000.,       0.,                                      0.,                                     0.,                                   0.,                                    -613406843450511./100000000000000000., 10800991281275153./50000000000000000.,  21184753175788097./50000000000000000.,  0.,                                      0.,                                      0.,                                       0.,                                     0.,                                         0.,                                         0.,                                       0.,                                        0.,                                      0.,                                        0.,                                       0.,                                      0.,                                      0.,                                      0.,                                    0.,                                       0.,                                      0.,                                       0.,                                      0.,                                       0.,                                     0.,                                      0.,                                     0.,        0.,                                    0.,                                      0., ],
    [14015279904218877./100000000000000000., 8384798124090527./100000000000000000.,      0.,                                      0.,                                     0.,                                   0.,                                    -589746835504869./50000000000000000.,  -4945980411376253./20000000000000000.,  978080858367729./10000000000000000.,    21759068924342063./100000000000000000.,  0.,                                      0.,                                       0.,                                     0.,                                         0.,                                         0.,                                       0.,                                        0.,                                      0.,                                        0.,                                       0.,                                      0.,                                      0.,                                      0.,                                    0.,                                       0.,                                      0.,                                       0.,                                      0.,                                       0.,                                     0.,                                      0.,                                     0.,        0.,                                    0.,                                      0., ],
    [35035051988507537./50000000000000000.,  3076276798847141./50000000000000000.,       0.,                                      0.,                                     0.,                                   0.,                                    592232780324503./100000000000000000.,  47032615996384111./100000000000000000., 299688863848679./1000000000000000.,     -24765687759399491./100000000000000000., 1386187872142971./12500000000000000.,    0.,                                       0.,                                     0.,                                         0.,                                         0.,                                       0.,                                        0.,                                      0.,                                        0.,                                       0.,                                      0.,                                      0.,                                      0.,                                    0.,                                       0.,                                      0.,                                       0.,                                      0.,                                       0.,                                     0.,                                      0.,                                     0.,        0.,                                    0.,                                      0., ],
    [9090909090909091./25000000000000000.,   2098500366813913./50000000000000000.,       0.,                                      0.,                                     0.,                                   0.,                                    -63597539253241./20000000000000000.,   10079971436327401./12500000000000000.,  975983126412389./10000000000000000.,    77857557815839891./100000000000000000.,  20489042383159943./100000000000000000.,  -39065394906867047./25000000000000000.,   0.,                                     0.,                                         0.,                                         0.,                                       0.,                                        0.,                                      0.,                                        0.,                                       0.,                                      0.,                                      0.,                                      0.,                                    0.,                                       0.,                                      0.,                                       0.,                                      0.,                                       0.,                                     0.,                                      0.,                                     0.,        0.,                                    0.,                                      0., ],
    [26315789473684211./100000000000000000., 2188633911168651./50000000000000000.,       0.,                                      0.,                                     0.,                                   0.,                                    0.,                                    0.,                                     0.,                                     124873005504039./20000000000000000.,     20004309710957731./100000000000000000.,  -805328367804983./100000000000000000.,    423035056134793./20000000000000000.,    0.,                                         0.,                                         0.,                                       0.,                                        0.,                                      0.,                                        0.,                                       0.,                                      0.,                                      0.,                                      0.,                                    0.,                                       0.,                                      0.,                                       0.,                                      0.,                                       0.,                                     0.,                                      0.,                                     0.,        0.,                                    0.,                                      0., ],
    [3921722466502709./100000000000000000.,  1417496251817573./50000000000000000.,       0.,                                      0.,                                     0.,                                   0.,                                    0.,                                    0.,                                     0.,                                     249163204855817./100000000000000000.,    2301387878545931./100000000000000000.,   -322155956692977./100000000000000000.,    197688509889533./20000000000000000.,    -1065053856644437./50000000000000000.,      0.,                                         0.,                                       0.,                                        0.,                                      0.,                                        0.,                                       0.,                                      0.,                                      0.,                                      0.,                                    0.,                                       0.,                                      0.,                                       0.,                                      0.,                                       0.,                                     0.,                                      0.,                                     0.,        0.,                                    0.,                                      0., ],
    [20322937573209419./25000000000000000.,  343511894290243./1000000000000000.,         0.,                                      0.,                                     0.,                                   0.,                                    0.,                                    0.,                                     0.,                                     21045191202362739./100000000000000000.,  25856863014307603./25000000000000000.,   300151822932211./50000000000000000.,      42796906254980977./50000000000000000.,  -97723500503676681./100000000000000000.,    -66002698047929469./100000000000000000.,    0.,                                       0.,                                        0.,                                      0.,                                        0.,                                       0.,                                      0.,                                      0.,                                      0.,                                    0.,                                       0.,                                      0.,                                       0.,                                      0.,                                       0.,                                     0.,                                      0.,                                     0.,        0.,                                    0.,                                      0., ],
    [16666666666666667./100000000000000000., -1435740016721681./100000000000000000.,     0.,                                      0.,                                     0.,                                   0.,                                    0.,                                    0.,                                     0.,                                     -4578165875613./125000000000000.,        3502549756362137./100000000000000000.,   721892032724227./20000000000000000.,      -2652199675536811./100000000000000000., 4456990113056981./100000000000000000.,      777144333320989./6250000000000000.,         413829693239481./100000000000000000.,     0.,                                        0.,                                      0.,                                        0.,                                       0.,                                      0.,                                      0.,                                      0.,                                    0.,                                       0.,                                      0.,                                       0.,                                      0.,                                       0.,                                     0.,                                      0.,                                     0.,        0.,                                    0.,                                      0., ],
    [9./10.,                                 35603240442512029./100000000000000000.,     0.,                                      0.,                                     0.,                                   0.,                                    0.,                                    0.,                                     0.,                                     -2250963794737813./5000000000000000.,    4305279070837109./10000000000000000.,    3199831431318889./6250000000000000.,      45415181944320213./50000000000000000.,  -30980273342983483./25000000000000000.,     -64904866167176147./100000000000000000.,    25170890458681929./100000000000000000.,   974883087931983./1250000000000000.,        0.,                                      0.,                                        0.,                                       0.,                                      0.,                                      0.,                                      0.,                                    0.,                                       0.,                                      0.,                                       0.,                                      0.,                                       0.,                                     0.,                                      0.,                                     0.,        0.,                                    0.,                                      0., ],
    [6412992574519669./100000000000000000.,  1309356874065131./100000000000000000.,      0.,                                      0.,                                     0.,                                   0.,                                    0.,                                    0.,                                     0.,                                     0.,                                      0.,                                      0.,                                       0.,                                     -9320530679851./100000000000000000.,        2526871671311497./50000000000000000.,       40223517097./50000000000000000.,          59172602949417./100000000000000000.,       -8032294443./20000000000000000.,         0.,                                        0.,                                       0.,                                      0.,                                      0.,                                      0.,                                    0.,                                       0.,                                      0.,                                       0.,                                      0.,                                       0.,                                     0.,                                      0.,                                     0.,        0.,                                    0.,                                      0., ],
    [4082998185668577./20000000000000000.,   207926484466053./10000000000000000.,        0.,                                      0.,                                     0.,                                   0.,                                    0.,                                    0.,                                     0.,                                     0.,                                      0.,                                      0.,                                       0.,                                     58269591880009./100000000000000000.,        -25053147886213./3125000000000000.,         403847643847./100000000000000000.,        8546099980555061./100000000000000000.,     -25560810117./12500000000000000.,        10532857882443189./100000000000000000.,    0.,                                       0.,                                      0.,                                      0.,                                      0.,                                    0.,                                       0.,                                      0.,                                       0.,                                      0.,                                       0.,                                     0.,                                      0.,                                     0.,        0.,                                    0.,                                      0., ],
    [39535039104876057./100000000000000000., 140153449795736021./100000000000000000.,    0.,                                      0.,                                     0.,                                   0.,                                    0.,                                    0.,                                     0.,                                     0.,                                      0.,                                      0.,                                       0.,                                     -11512600049211063./50000000000000000.,     -360553420233456453./50000000000000000.,    93225390173709./25000000000000000.,       -471415495727125021./100000000000000000.,  -176367657545349./100000000000000000.,   382065274019349383./50000000000000000.,    70120408731950367./20000000000000000.,    0.,                                      0.,                                      0.,                                      0.,                                    0.,                                       0.,                                      0.,                                       0.,                                      0.,                                       0.,                                     0.,                                      0.,                                     0.,        0.,                                    0.,                                      0., ],
    [60464960895123943./100000000000000000., 298786626735301717./25000000000000000.,     0.,                                      0.,                                     0.,                                   0.,                                    0.,                                    0.,                                     0.,                                     0.,                                      0.,                                      0.,                                       0.,                                     779480932108175969./100000000000000000.,    -225800557546930317./4000000000000000.,     9123763069306449./100000000000000000.,    -636681399627174431./50000000000000000.,   -3968959219047197./100000000000000000.,  544392141883570887./10000000000000000.,    -364411637921569237./100000000000000000., -80450324991050991./100000000000000000., 0.,                                      0.,                                      0.,                                    0.,                                       0.,                                      0.,                                       0.,                                      0.,                                       0.,                                     0.,                                      0.,                                     0.,        0.,                                    0.,                                      0., ],
    [15917001814331423./20000000000000000.,  -14880942650710048843./100000000000000000., 0.,                                      0.,                                     0.,                                   0.,                                    0.,                                    0.,                                     0.,                                     0.,                                      0.,                                      0.,                                       0.,                                     -2293238195728141211./25000000000000000.,   70765614497159835983./100000000000000000.,  -110563611857482441./100000000000000000., 17613459188381137259./100000000000000000., 24569241210744033./50000000000000000.,   -17106950011245373609./25000000000000000., 279910604998398259./10000000000000000.,   659698550151411667./50000000000000000.,  25025756256796089./20000000000000000.,   0.,                                      0.,                                    0.,                                       0.,                                      0.,                                       0.,                                      0.,                                       0.,                                     0.,                                      0.,                                     0.,        0.,                                    0.,                                      0., ],
    [93587007425480331./100000000000000000., -241826986737049191./25000000000000000.,    0.,                                      0.,                                     0.,                                   0.,                                    0.,                                    0.,                                     0.,                                     0.,                                      0.,                                      0.,                                       0.,                                     -446990150858505531./100000000000000000.,   227563564345476341./5000000000000000.,      -7130850861838269./100000000000000000.,   140342017585515927./12500000000000000.,    394513677242571./3125000000000000.,      -272149587218427071./6250000000000000.,    39358715377152949./50000000000000000.,    26613234837234211./50000000000000000.,   42242273399632533./100000000000000000.,  8591312495030671./100000000000000000.,   0.,                                    0.,                                       0.,                                      0.,                                       0.,                                      0.,                                       0.,                                     0.,                                      0.,                                     0.,        0.,                                    0.,                                      0., ],
    [16666666666666667./100000000000000000., -62915020279409189./6250000000000000.,      0.,                                      0.,                                     0.,                                   0.,                                    0.,                                    0.,                                     0.,                                     -4578165875613./125000000000000.,        3502549756362137./100000000000000000.,   721892032724227./20000000000000000.,      -2652199675536811./100000000000000000., -39193060761341509./6250000000000000.,      4820792374425629891./100000000000000000.,   -868089611420207./12500000000000000.,     1268106902048502957./100000000000000000.,  598355844841619./50000000000000000.,     -58406220624060301./1250000000000000.,     133029613326626711./100000000000000000.,  50383393751699149./50000000000000000.,   2095120519336651./100000000000000000.,   1050673531656321./50000000000000000.,    476098007208561./50000000000000000.,   0.,                                       0.,                                      0.,                                       0.,                                      0.,                                       0.,                                     0.,                                      0.,                                     0.,        0.,                                    0.,                                      0., ],
    [20322937573209419./25000000000000000.,  -40947808167774370877./100000000000000000., 0.,                                      0.,                                     0.,                                   0.,                                    0.,                                    0.,                                     0.,                                     21045191202362739./100000000000000000.,  25856863014307603./25000000000000000.,   300151822932211./50000000000000000.,      42796906254980977./50000000000000000.,  -25051699854744786049./100000000000000000., 97321233326194213883./50000000000000000.,   -152251941051155183./50000000000000000.,  6132829744103521419./12500000000000000.,   156647589531270907./100000000000000000., -94098714497005586681./50000000000000000., 7525922247248471753./100000000000000000., 864335892450827669./25000000000000000.,  321147679440968961./100000000000000000., -46040804173841439./100000000000000000., -1741436679683621./20000000000000000., -184837953539575767./25000000000000000.,  0.,                                      0.,                                       0.,                                      0.,                                       0.,                                     0.,                                      0.,                                     0.,        0.,                                    0.,                                      0., ],
    [3921722466502709./100000000000000000.,  343347475853550879./100000000000000000.,    0.,                                      0.,                                     0.,                                   0.,                                    0.,                                    0.,                                     0.,                                     249163204855817./100000000000000000.,    2301387878545931./100000000000000000.,   -322155956692977./100000000000000000.,    197688509889533./20000000000000000.,    54063199844480627./25000000000000000.,      -1626998645464574213./100000000000000000.,  -2570690042410491./20000000000000000.,    -898915042666504253./100000000000000000.,  -13943814529281./4000000000000000.,      63174477645335923./4000000000000000.,      -57440333091409507./100000000000000000.,  -3456020390213933./10000000000000000.,   -132448298041317./20000000000000000.,    -194447032310551./25000000000000000.,    -14243367696091./4000000000000000.,    1198206266124827./250000000000000.,       7686273243653429./50000000000000000.,    0.,                                       0.,                                      0.,                                       0.,                                     0.,                                      0.,                                     0.,        0.,                                    0.,                                      0., ],
    [9090909090909091./25000000000000000.,   3230385208719854423./100000000000000000.,   0.,                                      0.,                                     0.,                                   0.,                                    -63597539253241./20000000000000000.,   10079971436327401./12500000000000000.,  975983126412389./10000000000000000.,    77857557815839891./100000000000000000.,  20489042383159943./100000000000000000.,  -39065394906867047./25000000000000000.,   0.,                                     817149459411552853./50000000000000000.,     -15454455529354362123./100000000000000000., 156971088703334873./100000000000000000.,  327685545087248131./100000000000000000.,   -1258723112984133./25000000000000000.,   15332115185804166507./100000000000000000., 44848011645482531./6250000000000000.,     -147018374337650241./50000000000000000., -6658459460768031./100000000000000000.,  -4623460549908437./100000000000000000.,  -1020993667928397./50000000000000000., -1067046212877471701./20000000000000000., -27109742943015731./20000000000000000.,  -9824767237577047./6250000000000000.,     0.,                                      0.,                                       0.,                                     0.,                                      0.,                                     0.,        0.,                                    0.,                                      0., ],
    [35035051988507537./50000000000000000.,  -1664514674863415129./100000000000000000.,  0.,                                      0.,                                     0.,                                   0.,                                    592232780324503./100000000000000000.,  47032615996384111./100000000000000000., 299688863848679./1000000000000000.,     -24765687759399491./100000000000000000., 1386187872142971./12500000000000000.,    0.,                                       -9834380876924583./20000000000000000.,  -114743154427289497./10000000000000000.,    321037266630492109./4000000000000000.,      -7682646079600857./20000000000000000.,    728147667468107583./100000000000000000.,   -6634969230612419./50000000000000000.,   -8107998325257307267./100000000000000000., -781484330222629./625000000000000.,       259263594969543681./100000000000000000., -15072014917320227./50000000000000000.,  11069223039491617./50000000000000000.,   8275772747718929./100000000000000000., 379921324081223041./20000000000000000.,   26923194640963969./100000000000000000.,  162674827447066537./100000000000000000.,  9834380876924583./20000000000000000.,    0.,                                       0.,                                     0.,                                      0.,                                     0.,        0.,                                    0.,                                      0., ],
    [14015279904218877./100000000000000000., 8384798124090527./100000000000000000.,      0.,                                      0.,                                     0.,                                   0.,                                    -589746835504869./50000000000000000.,  -4945980411376253./20000000000000000.,  978080858367729./10000000000000000.,    21759068924342063./100000000000000000.,  0.,                                      6879280338166261./50000000000000000.,     4398702297150467./100000000000000000.,  0.,                                         -25685040688409667./50000000000000000.,     82635569115131551./100000000000000000.,   1285090698599059163./50000000000000000.,   0.,                                      0.,                                        0.,                                       0.,                                      0.,                                      0.,                                      0.,                                    -1285090698599059163./50000000000000000., -82635569115131551./100000000000000000., 25685040688409667./50000000000000000.,    -4398702297150467./100000000000000000.,  -6879280338166261./50000000000000000.,    0.,                                     0.,                                      0.,                                     0.,        0.,                                    0.,                                      0., ],
    [7426709605345459./25000000000000000.,   12438052665409441./100000000000000000.,     0.,                                      0.,                                     0.,                                   2261202821975843./10000000000000000.,  1378858876180809./100000000000000000., -1680525334991711./25000000000000000.,  0.,                                     0.,                                      -17124779501708567./20000000000000000.,  -196337522866858909./100000000000000000., -1161664113620597./5000000000000000.,   0.,                                         430660719086453349./100000000000000000.,    -292722963249465483./100000000000000000., -1646263332795717889./20000000000000000.,  0.,                                      0.,                                        0.,                                       0.,                                      0.,                                      0.,                                      0.,                                    1646263332795717889./20000000000000000.,  292722963249465483./100000000000000000., -430660719086453349./100000000000000000., 1161664113620597./5000000000000000.,     196337522866858909./100000000000000000.,  17124779501708567./20000000000000000.,  0.,                                      0.,                                     0.,        0.,                                    0.,                                      0., ],
    [16749674481819323./25000000000000000.,  5174228081833989./50000000000000000.,       0.,                                      0.,                                     6103444365320361./50000000000000000., 24128724516562331./50000000000000000., -381409600015607./10000000000000000.,  0.,                                     -6881244066385029./12500000000000000.,  0.,                                      -71191581158518923./100000000000000000., -29206480283577567./50000000000000000.,   0.,                                     0.,                                         52761577031466233./25000000000000000.,      -8374947367395721./100000000000000000.,   255010749536160457./50000000000000000.,    0.,                                      0.,                                        0.,                                       0.,                                      0.,                                      0.,                                      0.,                                    -255010749536160457./50000000000000000.,  8374947367395721./100000000000000000.,   -52761577031466233./25000000000000000.,   0.,                                      29206480283577567./50000000000000000.,    71191581158518923./100000000000000000., 6881244066385029./12500000000000000.,    0.,                                     0.,        0.,                                    0.,                                      0., ],
    [33333333333333333./100000000000000000., 19333333333333333./100000000000000000.,     0.,                                      11./50.,                                -2./25.,                              0.,                                    0.,                                    1099934255807247./10000000000000000.,   -3178713100953377./12500000000000000.,  0.,                                      3462283108466777./4000000000000000.,     332416449114093083./100000000000000000.,  0.,                                     0.,                                         -1201022233159779339./100000000000000000.,  11915036656062331./25000000000000000.,    -580486022442072781./20000000000000000.,   0.,                                      0.,                                        0.,                                       0.,                                      0.,                                      0.,                                      0.,                                    580486022442072781./20000000000000000.,   -11915036656062331./25000000000000000.,  1201022233159779339./100000000000000000., 0.,                                      -332416449114093083./100000000000000000., -3462283108466777./4000000000000000.,   3178713100953377./12500000000000000.,    -1099934255807247./10000000000000000.,  0.,        0.,                                    0.,                                      0., ],
    [13888888888888889./25000000000000000.,  -83333333333333333./100000000000000000.,    138888888888888889./100000000000000000., 0.,                                     0.,                                   -3./4.,                                0.,                                    -4925295437180263./10000000000000000.,  0.,                                     0.,                                      0.,                                      0.,                                       0.,                                     0.,                                         0.,                                         0.,                                       0.,                                        0.,                                      0.,                                        0.,                                       0.,                                      0.,                                      0.,                                      0.,                                    0.,                                       0.,                                      0.,                                       0.,                                      0.,                                       0.,                                     0.,                                      4925295437180263./10000000000000000.,   3./4.,     0.,                                    0.,                                      0., ],
    [11111111111111111./100000000000000000., 11111111111111111./100000000000000000.,     0.,                                      -11111111111111111./50000000000000000., 0.,                                   0.,                                    0.,                                    0.,                                     0.,                                     0.,                                      0.,                                      0.,                                       0.,                                     0.,                                         0.,                                         0.,                                       0.,                                        0.,                                      0.,                                        0.,                                       0.,                                      0.,                                      0.,                                      0.,                                    0.,                                       0.,                                      0.,                                       0.,                                      0.,                                       0.,                                     0.,                                      0.,                                     0.,        11111111111111111./50000000000000000., 0.,                                      0., ],
    [1./1.,                                  7145878509724289./25000000000000000.,       29166666666666667./100000000000000000.,  7./32.,                                 0.,                                   21./128.,                              0.,                                    10909717747277833./50000000000000000.,  18039289847869777./100000000000000000., 0.,                                      10285691970242251./50000000000000000.,   94810856086629./390625000000000.,         24646578081362931./100000000000000000., -13799677631635633./4000000000000000.,      2860944527000451./12500000000000000.,       14164529985107571./50000000000000000.,    321085125837766641./100000000000000000.,   -2235387773648457./10000000000000000.,   -70712115720441907./100000000000000000.,   8028083628757177./2500000000000000.,      70477174154834883./50000000000000000.,   -15136205344374261./100000000000000000., 9308764363175357./25000000000000000.,    12648937320318067./50000000000000000., -321085125837766641./100000000000000000., -14164529985107571./50000000000000000.,  -2860944527000451./12500000000000000.,    -24646578081362931./100000000000000000., -94810856086629./390625000000000.,        -10285691970242251./50000000000000000., -18039289847869777./100000000000000000., -10909717747277833./50000000000000000., -21./128., -7./32.,                               -29166666666666667./100000000000000000., 0., ]], dtype=numpy.float64
    )

    final_state = numpy.array(
        [[0., 892857142857143./50000000000000000., 3./512., 3./256., 0., 9./512., 0., 3./128., 15./512., 0., 9./256., 21./512., 3./64., 0., 27./512., 15./256., 33./512., 0., 5267605678587651./50000000000000000., 8528067312087609./50000000000000000., 10311469866467597./50000000000000000., 10311469866467597./50000000000000000., 8528067312087609./50000000000000000., 5267605678587651./50000000000000000., -33./512., -15./256., -27./512., -3./64., -21./512., -9./256., -15./512., -3./128., -9./512., -3./256., -3./512., 892857142857143./50000000000000000.],
         [0., 892857142857143./50000000000000000., 242968749999999999./50000000000000000000., 3./256., 0., 9./512., 0., 3./128., 15./512., 0., 9./256., 21./512., 3./64., 0., 27./512., 15./256., 33./512., 0., 5267605678587651./50000000000000000., 8528067312087609./50000000000000000., 10311469866467597./50000000000000000., 10311469866467597./50000000000000000., 8528067312087609./50000000000000000., 5267605678587651./50000000000000000., -33./512., -15./256., -27./512., -3./64., -21./512., -9./256., -15./512., -3./128., -9./512., -3./256., -242968749999999999./50000000000000000000., 892857142857143./50000000000000000.]], dtype=numpy.float64
    )
        


class RK108Solver(ExplicitRungeKuttaIntegrator):
    """
    The derived class that implements the Adaptive Runge-Kutta 10(8) method using
    the coefficients defined by Feagin.
    
    References
    ----------
    [1] Feagin, Terry. "HIGH-ORDER EXPLICIT RUNGE-KUTTA METHODS USING W-SYMMETRY." Neural, Parallel & Scientific Computations 20, no. 3-4 (2012): 437-458.
    """
    
    @property
    def order(self):
        return 10.0
    
    __alt_names__ = ("Explicit RK108", "RK108", "Runge-Kutta 10(8)", "RK108Feag")

    tableau = numpy.array(
        [[0.,                                           0.,                                           0.,                                          0.,                                            0.,                                          0.,                                           0.,                                            0.,                                            0.,                                            0.,                                           0.,                                             0.,                                             0.,                                            0.,                                           0.,                                            0.,                                           0.,       0., ],
    [1./10.,                                       1./10.,                                       0.,                                          0.,                                            0.,                                          0.,                                           0.,                                            0.,                                            0.,                                            0.,                                           0.,                                             0.,                                             0.,                                            0.,                                           0.,                                            0.,                                           0.,       0., ],
    [53935784080298178753./100000000000000000000., -22879414034382286013./25000000000000000000., 29090688043565464561./20000000000000000000., 0.,                                            0.,                                          0.,                                           0.,                                            0.,                                            0.,                                            0.,                                           0.,                                             0.,                                             0.,                                            0.,                                           0.,                                            0.,                                           0.,       0., ],
    [8090367612044726813./10000000000000000000.,   2528239878763977129./12500000000000000000.,   0.,                                          60677757090335451097./100000000000000000000.,  0.,                                          0.,                                           0.,                                            0.,                                            0.,                                            0.,                                           0.,                                             0.,                                             0.,                                            0.,                                           0.,                                            0.,                                           0.,       0., ],
    [3090367612044726813./10000000000000000000.,   3680494294172871503./20000000000000000000.,   0.,                                          19796683122719236907./100000000000000000000.,  -1823869618284081573./25000000000000000000., 0.,                                           0.,                                            0.,                                            0.,                                            0.,                                           0.,                                             0.,                                             0.,                                            0.,                                           0.,                                            0.,                                           0.,       0., ],
    [3924296760879181073./4000000000000000000.,    8790073402066813373./100000000000000000000.,  0.,                                          0.,                                            10261492563006516133./25000000000000000000., 1206784384197166223./2500000000000000000.,    0.,                                            0.,                                            0.,                                            0.,                                           0.,                                             0.,                                             0.,                                            0.,                                           0.,                                            0.,                                           0.,       0., ],
    [83333333333333333333./100000000000000000000., 4298502524512301511./50000000000000000000.,   0.,                                          0.,                                            6617719260814443679./20000000000000000000.,  12241573932736254821./25000000000000000000.,  -914820468838563421./12500000000000000000.,    0.,                                            0.,                                            0.,                                           0.,                                             0.,                                             0.,                                            0.,                                           0.,                                            0.,                                           0.,       0., ],
    [35401736585680237633./100000000000000000000., 6046522456266686033./50000000000000000000.,   0.,                                          0.,                                            0.,                                          26012467575829562281./100000000000000000000., 1627013107745456651./50000000000000000000.,    -186181316192925313./3125000000000000000.,     0.,                                            0.,                                           0.,                                             0.,                                             0.,                                            0.,                                           0.,                                            0.,                                           0.,       0., ],
    [88252766196473234643./100000000000000000000., 11085437958039148351./100000000000000000000., 0.,                                          0.,                                            0.,                                          0.,                                           -1514403720637513969./25000000000000000000.,   3217637056017783901./10000000000000000000.,    25524286280403151579./50000000000000000000.,   0.,                                           0.,                                             0.,                                             0.,                                            0.,                                           0.,                                            0.,                                           0.,       0., ],
    [2008174244501007963./3125000000000000000.,    11205441475287900483./100000000000000000000., 0.,                                          0.,                                            0.,                                          0.,                                           -14494277590286591567./100000000000000000000., -33326971909625670659./100000000000000000000., 9985384591137601227./20000000000000000000.,    6368807611621076303./12500000000000000000.,   0.,                                             0.,                                             0.,                                            0.,                                           0.,                                            0.,                                           0.,       0., ],
    [1116825755498992037./3125000000000000000.,    5698839198209299307./50000000000000000000.,   0.,                                          0.,                                            0.,                                          0.,                                           -3844066821016784693./50000000000000000000.,   23952736032439064911./100000000000000000000.,  7955493247361892781./20000000000000000000.,    268897392184018639./25000000000000000000.,    -6555382483280377483./20000000000000000000.,    0.,                                             0.,                                            0.,                                           0.,                                            0.,                                           0.,       0., ],
    [11747233803526765357./100000000000000000000., 498946580175122529./6250000000000000000.,     0.,                                          0.,                                            0.,                                          0.,                                           -1040659373601206153./20000000000000000000.,   -5769541461685488817./100000000000000000000.,  9739095785605208249./50000000000000000000.,    14538492318832506973./100000000000000000000., -122334798492448559./1562500000000000000.,      -5725164968054945609./50000000000000000000.,    0.,                                            0.,                                           0.,                                            0.,                                           0.,       0., ],
    [83333333333333333333./100000000000000000000., 24627890254121432003./25000000000000000000.,  0.,                                          0.,                                            6617719260814443679./20000000000000000000.,  12241573932736254821./25000000000000000000.,  -68948243287421783791./50000000000000000000.,  -86116419502763566667./100000000000000000000., 578428813637537220023./100000000000000000000., 32880776198510356689./10000000000000000000.,  -238633905093136384013./100000000000000000000., -65095868496728783731./20000000000000000000.,   -108171770843211491177./50000000000000000000., 0.,                                           0.,                                            0.,                                           0.,       0., ],
    [3090367612044726813./10000000000000000000.,   17901605915432657821./20000000000000000000.,  0.,                                          19796683122719236907./100000000000000000000.,  -1823869618284081573./25000000000000000000., 0.,                                           -42561811983100380987./50000000000000000000.,  9958002807963332543./25000000000000000000.,    363937263181035606029./100000000000000000000., 30964575407966064473./20000000000000000000.,  -106110857352026858013./50000000000000000000.,  -158350398545326172713./100000000000000000000., -85780804142968132461./50000000000000000000.,  -2440364057501274521./100000000000000000000., 0.,                                            0.,                                           0.,       0., ],
    [53935784080298178753./100000000000000000000., -22879414034382286013./25000000000000000000., 29090688043565464561./20000000000000000000., 0.,                                            0.,                                          -38866682182248411677./50000000000000000000., 0.,                                            -1138619577693970087./12500000000000000000.,   0.,                                            0.,                                           0.,                                             0.,                                             0.,                                            1138619577693970087./12500000000000000000.,   38866682182248411677./50000000000000000000.,   0.,                                           0.,       0., ],
    [1./10.,                                       1./10.,                                       0.,                                          -15717866579977116337./100000000000000000000., 0.,                                          0.,                                           0.,                                            0.,                                            0.,                                            0.,                                           0.,                                             0.,                                             0.,                                            0.,                                           0.,                                            15717866579977116337./100000000000000000000., 0.,       0., ],
    [1./1.,                                        18178130070009528389./100000000000000000000., 27./40.,                                     17137907992359491997./50000000000000000000.,   0.,                                          25911121454832274451./100000000000000000000., -7165579334359041781./20000000000000000000.,   -20918979188176661219./20000000000000000000.,  93032784541562698329./100000000000000000000.,  88975479715854051223./50000000000000000000.,  1./10.,                                         -28254756953904408161./100000000000000000000.,  -15932735011997254917./100000000000000000000., -7275794732350075543./50000000000000000000.,  -25911121454832274451./100000000000000000000., -17137907992359491997./50000000000000000000., -27./40., 0., ]], dtype=numpy.float64
    )

    final_state = numpy.array(
        [[0., 3333333333333333333./100000000000000000000., 1./40., 3333333333333333333./100000000000000000000., 0., 1./20., 0., 1./25., 0., 2365468476861543627./12500000000000000000., 27742918851774317651./100000000000000000000., 27742918851774317651./100000000000000000000., 2365468476861543627./12500000000000000000., -1./25., -1./20., -3333333333333333333./100000000000000000000., -1./40., 3333333333333333333./100000000000000000000.],
         [0., 3333333333333333333./100000000000000000000., 1./36., 3333333333333333333./100000000000000000000., 0., 1./20., 0., 1./25., 0., 2365468476861543627./12500000000000000000., 27742918851774317651./100000000000000000000., 27742918851774317651./100000000000000000000., 2365468476861543627./12500000000000000000., -1./25., -1./20., -3333333333333333333./100000000000000000000., -1./36., 3333333333333333333./100000000000000000000.]], dtype=numpy.float64
    )
        
#     def get_error_estimate(self, dState, dTime, aux, tableau_idx_expand):
#         return (aux[0] - aux[-1]) / 360 + (dState - aux[-1]) / dTime # D.sum(self.final_state[1][tableau_idx_expand] * aux, axis=0) * dTime / 1000 # 

class RK8713MSolver(ExplicitRungeKuttaIntegrator):
    """
    The derived class that implements the Adaptive Runge-Kutta 8(7) method using
    the coefficients defined by Dormand and Prince.
    
    References
    ----------
    [1] Prince, P.J., and J.R. Dormand. ‘High Order Embedded Runge-Kutta Formulae’. Journal of Computational and Applied Mathematics 7, no. 1 (March 1981): 67–75. https://doi.org/10.1016/0771-050X(81)90010-3 .
    """
    
    @property
    def order(self):
        return 8.0
    
    __alt_names__ = ("Explicit RK8713M", "RK87", "Runge-Kutta 8(7)", "RK8713M")

    tableau = numpy.array(
        [[0.0,                    0.0,                  0.0,      0.0,       0.0,                      0.0,                    0.0,                     0.0,                     0.0,                     0.0,                     0.0,                   0.0,                  0.0, 0.0],
         [1/18,                   1/18,                 0.0,      0.0,       0.0,                      0.0,                    0.0,                     0.0,                     0.0,                     0.0,                     0.0,                   0.0,                  0.0, 0.0],
         [1/12,                   1/48,                 1/16,     0.0,       0.0,                      0.0,                    0.0,                     0.0,                     0.0,                     0.0,                     0.0,                   0.0,                  0.0, 0.0],
         [1/8,                    1/32,                 0.0,      3/32,      0.0,                      0.0,                    0.0,                     0.0,                     0.0,                     0.0,                     0.0,                   0.0,                  0.0, 0.0],
         [5/16,                   5/16,                 0.0,     -75/64,     75/64,                    0.0,                    0.0,                     0.0,                     0.0,                     0.0,                     0.0,                   0.0,                  0.0, 0.0],
         [3/8,                    3/80,                 0.0,      0.0,       3/16,                     3/20,                   0.0,                     0.0,                     0.0,                     0.0,                     0.0,                   0.0,                  0.0, 0.0],
         [59/400,                 29443841/614563906,   0.0,      0.0,       77736538/692538347,      -28693883/1125000000,    23124283/1800000000,     0.0,                     0.0,                     0.0,                     0.0,                   0.0,                  0.0, 0.0],
         [93/200,                 16016141/946692911,   0.0,      0.0,       61564180/158732637,       22789713/633445777,     545815736/2771057229,   -180193667/1043307555,    0.0,                     0.0,                     0.0,                   0.0,                  0.0, 0.0],
         [5490023248/9719169821,  39632708/573591083,   0.0,      0.0,      -433636366/683701615,     -421739975/2616292301,   100302831/723423059,     790204164/839813087,     800635310/3783071287,    0.0,                     0.0,                   0.0,                  0.0, 0.0],
         [13/20,                  246121993/1340847787, 0.0,      0.0,      -37695042795/15268766246, -309121744/1061227803,  -12992083/490766935,      6005943493/2108947869,   393006217/1396673457,    123872331/1001029789,    0.0,                   0.0,                  0.0, 0.0],
         [1201146811/1299019798, -1028468189/846180014, 0.0,      0.0,       8478235783/508512852,     1311729495/1432422823, -10304129995/1701304382, -48777925059/3047939560,  15336726248/1032824649, -45442868181/3398467696,  3065993473/597172653,  0.0,                  0.0, 0.0],
         [1,                      185892177/718116043,  0.0,      0.0,      -3185094517/667107341,    -477755414/1098053517,  -703635378/230739211,     5731566787/1027545527,   5232866602/850066563,   -4093664535/808688257,    3962137247/1805957418, 65686358/487910083,   0.0, 0.0],
         [1,                      403863854/491063109,  0.0,      0.0,      -5068492393/434740067,    -411421997/543043805,    652783627/914296604,     11173962825/925320556,  -13158990841/6184727034,  3936647629/1978049680,  -160528059/685178525,   248638103/1413531060, 0.0, 0.0]], dtype=numpy.float64
    )

    final_state = numpy.array(
        [[0., 13451932/455176623, 0.0, 0.0, 0.0, 0.0, -808719846/976000145, 1757004468/5645159321, 656045339/265891186, -3867574721/1518517206, 465885868/322736535,  53011238/667516719,   2/45,                 0.0],
         [0., 14005451/335480064, 0.0, 0.0, 0.0, 0.0, -59238493/1068277825, 181606767/758867731,   561292985/797845732, -1041891430/1371343529, 760417239/1151165299, 118820643/751138087, -528747749/2220607170, 1/4]], dtype=numpy.float64
    )
    
class RK45CKSolver(ExplicitRungeKuttaIntegrator):
    """
    The derived class that implements the Adaptive Runge-Kutta 4(5) method using
    the coefficients defined by Cash and Karp.
    
    References
    ----------
    [1] Cash, J. R., and Alan H. Karp. ‘A Variable Order Runge-Kutta Method for Initial Value Problems with Rapidly Varying Right-Hand Sides’. ACM Transactions on Mathematical Software 16, no. 3 (1 September 1990): 201–22. https://doi.org/10.1145/79505.79507 .
    """

    @property
    def order(self):
        return 5.0
    
    __alt_names__ = ("Explicit RK45CK", "RK45CK", "Runge-Kutta-Cash-Karp", "RK45")

    tableau = numpy.array(
        [[0.0,  0.0,        0.0,     0.0,       0.0,          0.0,      0.0],
         [1/5,  1/5,        0.0,     0.0,       0.0,          0.0,      0.0],
         [3/10, 3/40,       9/40,    0.0,       0.0,          0.0,      0.0],
         [3/5,  3/10,       -9/10,   6/5,       0.0,          0.0,      0.0],
         [1,    -11/54,     5/2,     -70/27,    35/27,        0.0,      0.0],
         [7/8,  1631/55296, 175/512, 575/13824, 44275/110592, 253/4096, 0.0]], dtype=numpy.float64
    )

    final_state = numpy.array(
        [[0., 37/378,     0, 250/621,     125/594,     0,         512/1771],
         [0., 2825/27648, 0, 18575/48384, 13525/55296, 277/14336, 1/4     ]], dtype=numpy.float64
    )
    
class RK5Solver(ExplicitRungeKuttaIntegrator):
    """
    The derived class that implements a 5th order Runge-Kutta method.
    This is simply the fifth order method embedded in RK45CK.
    
    References
    ----------
    [1] Cash, J. R., and Alan H. Karp. ‘A Variable Order Runge-Kutta Method for Initial Value Problems with Rapidly Varying Right-Hand Sides’. ACM Transactions on Mathematical Software 16, no. 3 (1 September 1990): 201–22. https://doi.org/10.1145/79505.79507 .
    """
    
    @property
    def order(self):
        return 5.0
    
    __alt_names__ = ("Explicit RK5", "RK5", "Runge-Kutta 5")

    tableau = numpy.copy(RK45CKSolver.tableau)

    final_state = numpy.array(
        [[0., 37/378,     0, 250/621,     125/594,     0,         512/1771]], dtype=numpy.float64
    )
    
class RK4Solver(ExplicitRungeKuttaIntegrator):
    """
    The derived class that implements the classic 4th order Runge-Kutta method.
    """
    
    @property
    def order(self):
        return 4.0
    
    __alt_names__ = ("Explicit RK4", "RK4", "Runge-Kutta 4")

    tableau = numpy.array(
        [[0,   0,   0,   0, 0],
         [1/2, 1/2, 0,   0, 0],
         [1/2, 0,   1/2, 0, 0],
         [1,   0,   0,   1, 0]], dtype=numpy.float64
    )

    final_state = numpy.array(
        [[0., 1/6, 1/3, 1/3, 1/6]], dtype=numpy.float64
    )

class MidpointSolver(ExplicitRungeKuttaIntegrator):
    """
    The derived class that implements the midpoint method.
    """
    
    @property
    def order(self):
        return 2.0
    
    __alt_names__ = ("Explicit Midpoint", "Midpoint")

    tableau = numpy.array(
        [[0,    0,   0],
         [1/2,  1/2, 0]], dtype=numpy.float64
    )

    final_state = numpy.array(
        [[0,    0,   1]], dtype=numpy.float64
    )

class HeunsSolver(ExplicitRungeKuttaIntegrator):
    """
    The derived class that implements Heun's method.
    """
    
    @property
    def order(self):
        return 2.0

    __alt_names__ = ("Explicit Heun's", "Heun's")

    tableau = numpy.array(
        [[0,    0,   0  ],
         [1,    1,   0  ]], dtype=numpy.float64
    )

    final_state = numpy.array(
        [[0,    1/4, 3/4]], dtype=numpy.float64
    )

class EulerSolver(ExplicitRungeKuttaIntegrator):
    """
    The derived class that implements the Euler method.
    """
    
    @property
    def order(self):
        return 1.0
    
    __alt_names__ = ("Explicit Euler", "Euler")

    tableau = numpy.array(
        [[0,    0]], dtype=numpy.float64
    )

    final_state = numpy.array(
        [[0,    1]], dtype=numpy.float64
    )

class EulerTrapSolver(ExplicitRungeKuttaIntegrator):
    """
    The derived class that implements the Euler-Trapezoidal method.
    """
    
    @property
    def order(self):
        return 2.0
    
    __alt_names__ = ("Explicit Euler-Trapezoidal", "Euler-Trapezoidal", "Euler-Trap", "Predictor-Corrector Euler")

    tableau = numpy.array(
        [[0,   0,   0,     0,   0  ],
         [1,   1,   0,     0,   0  ],
         [1,   1/2, 1/2,   0,   0  ],
         [1,   1/2, 0,     1/2, 0  ]], dtype=numpy.float64
    )

    final_state = numpy.array(
        [[0,   1/2, 0,    0,   1/2]], dtype=numpy.float64
    )

class HeunEulerSolver(ExplicitRungeKuttaIntegrator):
    """
    The derived class that implements the adaptive Heun-Euler method.
    This is a 1st order method (Euler) with an embedded 
    2nd order method (Heun) that does adaptive timestepping.
    """
    
    @property
    def order(self):
        return 2.0
    
    __alt_names__ = ("Explicit Adaptive Heun-Euler", "Adaptive Heun-Euler", "AHE")

    tableau = numpy.array(
        [[0,   0,   0],
         [1,   1,   0]], dtype=numpy.float64
    )

    final_state = numpy.array(
        [[0,    1/2, 1/2],
         [0,    1,   0   ]], dtype=numpy.float64
    )
    
class SymplecticEulerSolver(ExplicitSymplecticIntegrator):
    """
    The derived class that implements the symplectic Euler method.
    
    This is the simplest symplectic integration scheme.
    """
    
    @property
    def order(self):
        return 1.0
    
    __alt_names__ = ("Explicit Symplectic Forward Euler", "Symplectic Forward Euler")

    tableau = numpy.array(
        [[0.5, 0,   0.5],
         [0,   1.0, 0  ],
         [0.5, 0,   0.5]], dtype=numpy.float64
    )

class BABs9o7HSolver(ExplicitSymplecticIntegrator):
    """
    The derived class that implements the 7th order 
    BAB's9o7H symplectic integrator. This integrator
    is only applicable to systems that have a Hamiltonian
    that can be split such that: `H(p,q) = T(p) + V(q)`.
    
    References
    ----------
    [1] Nielsen, Kristian Mads Egeris. ‘Efficient Fourth Order Symplectic Integrators for Near-Harmonic Separable Hamiltonian Systems’. ArXiv:1501.04345 [Physics, Physics:Quant-Ph], 9 February 2015. http://arxiv.org/abs/1501.04345 .
    """
    
    @property
    def order(self):
        return 7.0
    
    __alt_names__ = ("Explicit BABS9O7H", "BABS9O7H")

    # Based on arXiv:1501.04345v2 - BAB's9o7H
    tableau = numpy.array(
       [[ 0.                  ,  0.                  ,  0.04649290043965892 ],
        [ 0.                  ,  0.1289555065927298  ,  0.                  ],
        [ 0.                  ,  0.                  ,  0.154901012702888   ],
        [ 0.                  ,  0.10907642985488271 ,  0.                  ],
        [ 0.                  ,  0.                  ,  0.31970548287359174 ],
        [ 0.                  , -0.013886035680471514,  0.                  ],
        [ 0.                  ,  0.                  , -0.19292000881571322 ],
        [ 0.                  ,  0.18375497456418036 ,  0.                  ],
        [ 0.                  ,  0.                  ,  0.17182061279957458 ],
        [ 0.                  ,  0.18419824933735726 ,  0.                  ],
        [ 0.                  ,  0.                  ,  0.17182061279957458 ],
        [ 0.                  ,  0.18375497456418036 ,  0.                  ],
        [ 0.                  ,  0.                  , -0.19292000881571322 ],
        [ 0.                  , -0.013886035680471514,  0.                  ],
        [ 0.                  ,  0.                  ,  0.31970548287359174 ],
        [ 0.                  ,  0.10907642985488271 ,  0.                  ],
        [ 0.                  ,  0.                  ,  0.154901012702888   ],
        [ 0.                  ,  0.1289555065927298  ,  0.                  ],
        [ 1.                  ,  0.                  ,  0.04649290043965892 ]], dtype=numpy.float64
    )

class ABAs5o6HSolver(ExplicitSymplecticIntegrator):
    """
    The derived class that implements the 6th order 
    ABAs5o6H symplectic integrator. This integrator
    is only applicable to systems that have a Hamiltonian
    that can be split such that `H(p,q) = T(p) + V(q)`.
    
    References
    ----------
    [1] Nielsen, Kristian Mads Egeris. ‘Efficient Fourth Order Symplectic Integrators for Near-Harmonic Separable Hamiltonian Systems’. ArXiv:1501.04345 [Physics, Physics:Quant-Ph], 9 February 2015. http://arxiv.org/abs/1501.04345 .
    """
    
    @property
    def order(self):
        return 6.0
    
    __alt_names__ = ("Explicit ABAS5O6H", "ABAS5O6H")

    # Based on arXiv:1501.04345v2 - ABAs5o6H
    tableau = numpy.array(
      [[ 0.                  ,  0.                  ,  0.15585935917621682 ],
       [ 0.                  , -0.6859195549562167  ,  0.                  ],
       [ 0.                  ,  0.                  , -0.007025499091957318],
       [ 0.                  ,  0.9966295909529364  ,  0.                  ],
       [ 0.                  ,  0.                  ,  0.35116613991574047 ],
       [ 0.                  ,  0.3785799280065607  ,  0.                  ],
       [ 0.                  ,  0.                  ,  0.35116613991574047 ],
       [ 0.                  ,  0.9966295909529364  ,  0.                  ],
       [ 0.                  ,  0.                  , -0.007025499091957318],
       [ 0.                  , -0.6859195549562167  ,  0.                  ],
       [ 1.                  ,  0.                  ,  0.15585935917621682 ]]
   )
