"""
Module defining the custom csv sampler object.
"""

import logging
from contextlib import suppress
from pathlib import Path

from scisample.base_sampler import BaseSampler
from scisample.utils import (log_and_raise_exception, read_csv,
                             test_for_uniform_lengths,
                             transpose_tabular)

LOG = logging.getLogger(__name__)


class CsvSampler(BaseSampler):
    """
    Class which reads samples from a csv file.

    .. code:: yaml

        sampler:
            type: csv
            csv_file: file_name.csv
            row_headers: True

    The ``csv_file`` entry gives the path to the csv file to read.
    the ``row_headers`` indicates whether the data is one entry per
    row (True) or one entry per column (False).
    """

    def __init__(self, data):
        super().__init__(data)
        self.path = Path(self.data['csv_file'])
        self._csv_data = None
        self.check_validity()

    def check_validity(self):
        super().check_validity()
        if not self.path.is_file():
            log_and_raise_exception(
                f"Could not find file {self.path} for CsvSampler.")

        for key, value in self.csv_data.items():
            if len(value) == 0:
                log_and_raise_exception(
                    f"No values associated with parameter "
                    f"{key} from file {self.path}.")
        with suppress(KeyError):
            test_for_uniform_lengths(self.data['parameters'].items())

    @property
    def csv_data(self):
        """
        The csv data as a dictionary of lists.
        """
        if self._csv_data is None:
            csv_data = read_csv(self.path)
            if not self.data['row_headers']:
                csv_data = transpose_tabular(csv_data)
            self._csv_data = {}
            for line in csv_data:
                self._csv_data[line[0]] = line[1:]
        return self._csv_data

    @property
    def parameters(self):
        """
        Return a of list of the parameters being generated by the
        sampler.
        """
        return list(self.csv_data.keys())

    def get_samples(self):
        """
        Get samples from the sampler.

        This returns samples as a list of dictionaries, with the
        sample variables as the keys:

        .. code:: python

            [{'b': 0.89856, 'a': 1}, {'b': 0.923223, 'a': 1}, ... ]
        """
        if self._samples is not None:
            return self._samples

        self._samples = []

        num_samples = 1

        for key, value in self.csv_data.items():
            num_samples = len(value)
            break

        for i in range(num_samples):
            new_sample = {}

            for key, value in self.csv_data.items():
                new_sample[key] = value[i]

            self._samples.append(new_sample)

        return self._samples
