"""
Module defining the column list object.
"""

import logging
from contextlib import suppress

from scisample.base_sampler import BaseSampler
from scisample.utils import log_and_raise_exception

LOG = logging.getLogger(__name__)


def process_row(row):
    """
    Take a row line and remove any comments,
    then split it.

    :param row: String to process.
    :returns: List of entries from the row.
    """
    row = row.strip()
    # Strip out comments
    with suppress(ValueError):
        row = row[:row.index('#')]
    return row.split()

class ColumnListSampler(BaseSampler):
    """
    Class defining basic column list sampling.

    .. code:: yaml

        sampler:
            type: column_list
            constants:
                X1: 20
            parameters: |
                X2       X3
                5        5
                10       10

    Entries in the ``constants`` dictionary will be added to all samples.
    Entries in the ``parameters`` block will be matched one to one.  The
    result of the above block would be:

    .. code:: python

        [{X1: 20, X2: 5, X3: 5}, {X1: 20, X2: 10, X3: 10}]
    
    Any text following a ``#`` in the parameters entry will be treated as
    a comment and ignored.  The following input would produce the same results
    as above.

    .. code:: yaml

        sampler:
            type: column_list
            constants:
                X1: 20
            parameters: |
                X2       X3
                5        5
                10       10
                #15      15  # Don't use this line
    """

    def __init__(self, data):
        """
        Initialize the sampler.

        :param data: Dictionary of sampler data.
        """
        super().__init__(data)
        self.check_validity()

    def check_validity(self):
        super().check_validity()
        self._check_variables_existence()
        self._check_variables_for_dups()
        LOG.error(f"parameter_block: {self.parameter_block}")
        # self.parameter_block must be called to check that
        # every row must have the same number of items
        self.parameter_block

    @property
    def parameters(self):
        """
        Return a of list of the parameters being generated by the
        sampler.
        """
        parameters = []
        with suppress(KeyError):
            parameters.extend(list(self.data['constants'].keys()))
        with suppress(KeyError):
            rows = self.data['parameters'].splitlines()
            headers = process_row(rows.pop(0))
            parameters.extend(headers)
        return parameters

    def get_samples(self):
        """
        Get samples from the sampler.

        This returns samples as a list of dictionaries, with the
        sample variables as the keys:

        .. code:: python

            [{'b': 0.89856, 'a': 1}, {'b': 0.923223, 'a': 1}, ... ]
        """
        LOG.info("Entering ColumnListSampler.get_samples()")
        if self._samples is not None:
            return self._samples

        self._samples = []

        parameter_samples = []


        with suppress(KeyError):
            rows = self.data['parameters'].splitlines()
            headers = process_row(rows.pop(0))

            for row in rows:
                data = process_row(row)
                if not data:
                    continue

                if len(data) != len(headers):
                    log_and_raise_exception(
                        "All rows must have the " +
                        "same number of values.\n"
                        f"  The header row has {len(headers)} values:\n"
                        f"    {headers}\n"
                        f"  The following row has {len(data)} values:\n"
                        f"    {data}.\n")
                    return False
                sample = {}
                for header, datum in zip(headers, data):
                    sample[header] = datum
                parameter_samples.append(sample)

        for i in range(len(parameter_samples)):
            new_sample = {}

            with suppress(KeyError):
                new_sample.update(self.data['constants'])

            with suppress(KeyError):
                for key, value in parameter_samples[i].items():
                    new_sample[key] = value

            self._samples.append(new_sample)

        return self._samples
