# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['snakefmt', 'snakefmt.parser']

package_data = \
{'': ['*']}

install_requires = \
['black>=19.10b0,<20.0', 'click>=7.1.1,<8.0.0', 'toml>=0.10.0,<0.11.0']

entry_points = \
{'console_scripts': ['snakefmt = snakefmt.snakefmt:main']}

setup_kwargs = {
    'name': 'snakefmt',
    'version': '0.1.1',
    'description': 'The uncompromising Snakemake code formatter',
    'long_description': '# Snakefmt\n\n[![GitHub Workflow Status](https://img.shields.io/github/workflow/status/snakemake/snakefmt/python_poetry_package)](https://github.com/snakemake/snakefmt/actions)\n[![codecov](https://codecov.io/gh/snakemake/snakefmt/branch/master/graph/badge.svg)](https://codecov.io/gh/snakemake/snakefmt)\n[![PyPI](https://img.shields.io/pypi/v/snakefmt)](https://pypi.org/project/snakefmt/)\n![PyPI - Python Version](https://img.shields.io/pypi/pyversions/snakefmt)\n[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n\nThis repository provides formatting for [Snakemake][snakemake] files. It follows the\ndesign and specifications of [Black][black].\n\n> **⚠️WARNING⚠️**: `snakefmt` modifies files in-place by default, thus we strongly\n> recommend ensuring your files are under version control before doing any formatting.\n> This is also to protect you from bugs as the project is still new. You\n> can also pipe the file in from stdin, which will print it to the screen, or use the\n> `--diff` or `--check` options. See [Usage](#usage) for more details.\n\n[TOC]: #\n\n# Table of Contents\n- [Install](#install)\n  - [PyPi](#pypi)\n  - [Conda](#conda)\n  - [Containers](#containers)\n  - [Local](#local)\n- [Example File](#example-file)\n- [Usage](#usage)\n  - [Basic Usage](#basic-usage)\n  - [Full Usage](#full-usage)\n- [Configuration](#configuration)\n- [Design](#design)\n  - [Syntax](#syntax)\n  - [Formatting](#formatting)\n- [Editor Integration](#editor-integration)\n- [Plug Us](#plug-us)\n- [Changes](#changes)\n- [Contributing](#contributing)\n\n\n## Install\n\n### PyPi\n\n```sh\npip install snakefmt\n```\n\n### Conda\n\n[![Conda (channel only)](https://img.shields.io/conda/vn/bioconda/snakefmt)](https://anaconda.org/bioconda/snakefmt)\n[![bioconda version](https://anaconda.org/bioconda/snakefmt/badges/platforms.svg)](https://anaconda.org/bioconda/snakefmt)\n\n```sh\nconda install -c bioconda snakefmt\n```\n\n### Containers\n\n#### Docker\n\n[![Docker Image Version (latest semver)](https://img.shields.io/docker/v/snakemake/snakefmt)](https://hub.docker.com/r/snakemake/snakefmt/tags)\n\n```sh\ndocker pull snakemake/snakefmt\ndocker run -it snakemake/snakefmt snakefmt --help\n```\n\nYou can find all the available tags on the [Docker Hub repository][dockerhub].\n\n[dockerhub]: https://hub.docker.com/r/snakemake/snakefmt\n\n#### Singularity\n\n```sh\nURI="docker://snakemake/snakefmt"\nsingularity exec "$URI" snakefmt --help\n```\n\nThe above will use the latest version. If you want to specify a version then use a [tag][dockerhub] like so.\n\n```sh\nVERSION="0.1.1"\nURI="docker://snakemake/snakefmt:${VERSION}"\n```\n\n### Local\n\nThese instructions include [installing `poetry`](https://python-poetry.org/docs/#installation).\n```sh\n# install poetry\ncurl -sSL https://raw.githubusercontent.com/python-poetry/poetry/master/get-poetry.py | python3\n\ngit clone https://github.com/snakemake/snakefmt && cd snakefmt\n# install snakefmt in a new environment\npoetry install\n# activate the environment so snakefmt is available on your PATH\npoetry shell\n```\n\n## Example File\n\nInput\n\n```python\nfrom snakemake.utils import min_version\nmin_version("5.14.0")\nconfigfile: "config.yaml" # snakemake keywords are treated like classes i.e. 2 newlines\nSAMPLES = [\'s1\', \'s2\'] # strings are normalised\nCONDITIONS = ["a", "b", "longlonglonglonglonglonglonglonglonglonglonglonglonglonglonglong"] # long lines are wrapped\ninclude: "rules/foo.smk" # 2 newlines\n\nrule all:\n    input: "data/results.txt" # newlines after keywords enforced and trailing comma\n\nrule gets_separated_by_two_newlines:\n    input:\n        files = expand("long/string/to/data/files/gets_broken_by_black/{sample}.{condition}",sample=SAMPLES, condition=CONDITIONS)\nif True:\n    rule can_be_inside_python_code:\n        input: "parameters", "get_indented"\n        threads: 4 # Numeric params stay unindented\n        params: key_val = "PEP8_formatted"\n        run:\n\n                print("weirdly_spaced_string_gets_respaced")\n\n```\n\n\nOutput\n\n```python\nfrom snakemake.utils import min_version\n\nmin_version("5.14.0")\n\n\nconfigfile: "config.yaml" # snakemake keywords are treated like classes i.e. 2 newlines\n\n\nSAMPLES = ["s1", "s2"] # strings are normalised\nCONDITIONS = [\n    "a",\n    "b",\n    "longlonglonglonglonglonglonglonglonglonglonglonglonglonglonglong",\n]  # long lines are wrapped\n\n\ninclude: "rules/foo.smk" # 2 newlines\n\n\nrule all:\n    input:\n        "data/results.txt", # newlines after keywords enforced and trailing comma\n\n\nrule gets_separated_by_two_newlines:\n    input:\n        files=expand(\n            "long/string/to/data/files/gets_broken_by_black/{sample}.{condition}",\n            sample=SAMPLES,\n            condition=CONDITIONS,\n        ),\n\n\nif True:\n\n    rule can_be_inside_python_code:\n        input:\n            "parameters",\n            "get_indented",\n        threads: 4 # Numeric params stay unindented\n        params:\n            key_val="PEP8_formatted",\n        run:\n            print("weirdly_spaced_string_gets_respaced")\n\n```\n\n\n## Usage\n\n### Basic Usage\n\nFormat a single Snakefile.\n\n```shell\nsnakefmt Snakefile\n```\n\nFormat all Snakefiles within a directory.\n\n```shell\nsnakefmt workflows/\n```\n\nFormat a file but write the output to stdout.\n\n```shell\nsnakefmt - < Snakefile\n```\n\n### Full Usage\n\n```\n$ snakefmt --help\nUsage: snakefmt [OPTIONS] [SRC]...\n\n  The uncompromising Snakemake code formatter.\n\n  SRC specifies directories and files to format. Directories will be\n  searched for file names that conform to the include/exclude patterns\n  provided.\n\n  Files are modified in-place by default; use diff, check, or  `snakefmt - <\n  Snakefile` to avoid this.\n\nOptions:\n  -l, --line-length INT  Lines longer than INT will be wrapped.  [default: 88]\n  --check                Don\'t write the files back, just return the status.\n                         Return code 0 means nothing would change. Return code\n                         1 means some files would be reformatted. Return code\n                         123 means there was an error.\n\n  -d, --diff             Don\'t write the files back, just output a diff for\n                         each file to stdout.\n\n  --compact-diff         Same as --diff but only shows lines that would change\n                         plus a few lines of context.\n\n  --include PATTERN      A regular expression that matches files and\n                         directories that should be included on recursive\n                         searches.  An empty value means all files are\n                         included regardless of the name.  Use forward slashes\n                         for directories on all platforms (Windows, too).\n                         Exclusions are calculated first, inclusions later.\n                         [default: (\\.smk$|^Snakefile)]\n\n  --exclude PATTERN      A regular expression that matches files and\n                         directories that should be excluded on recursive\n                         searches.  An empty value means no paths are\n                         excluded. Use forward slashes for directories on all\n                         platforms (Windows, too). Exclusions are calculated\n                         first, inclusions later.  [default: (\\.snakemake|\\.eg\n                         gs|\\.git|\\.hg|\\.mypy_cache|\\.nox|\\.tox|\\.venv|\\.svn|_\n                         build|buck-out|build|dist)]\n\n  -c, --config PATH      Read configuration from PATH. By default, will try to\n                         read from `./pyproject.toml`\n\n  -h, --help             Show this message and exit.\n  -V, --version          Show the version and exit.\n  -v, --verbose          Turns on debug-level logging.\n\n```\n\n#### Check\n\n##### `--check`\n\nDoes not write any formatted code back to file. It will instead check whether any\nchanges *would* be made. It returns one of three possible exit codes:\n\n**0** - indicates **no changes** would be made\n\n```\n$ echo \'include: "foo.txt"\' | snakefmt --check -                                        \n[INFO] 1 file(s) would be left unchanged 🎉\n$ echo "Exit code: $?"\nExit code: 0\n```\n\n**1** - indicates **changes** would be made\n\n```\n$ echo \'include:"foo.txt"\' | snakefmt --check - \n[INFO] 1 file(s) would be changed 😬\n$ echo "Exit code: $?"\nExit code: 1\n```\n\n**123** - indicates there was an **error** such as invalid syntax\n\n```\n$ echo \'include:\' | snakefmt --check -            \n[ERROR] L2: In include definition.\n[INFO] 1 file(s) contains errors 🤕\n$ echo "Exit code: $?"\nExit code: 123\n```\n\n#### Compact diff\n\n##### `--compact-diff`\n\nDoes not write any formatted code back to file. It will instead print a compact diff of\nhow the code looks before and after formatting. The diff is compact as it only prints\nthe lines that will change, with a few lines of surrounding context.\n\n```\n$ echo \'x = 1\\ny = 3\\n\\n\\nrule foo:\\n\\tinput: "foo.txt"\' | snakefmt --compact-diff -\n=====> Diff for stdin <=====\n\n--- original\n+++ new\n@@ -3,4 +3,5 @@\n \n \n rule foo:\n-       input: "foo.txt"\n+    input:\n+        "foo.txt",\n\n[INFO] All done 🎉\n```\n\nThe above example shows that the variable assignments at the beginning of the file are\nnot included in the compact diff (but would be included in a full diff).\n\n#### Diff\n\n##### `--diff`\n\nDoes not write any formatted code back to file. It will instead print a diff of how the\ncode looks before and after formatting.\n\n```\n$ echo \'rule foo:\\n\\tinput: "foo.txt"\' | snakefmt --diff -\n=====> Diff for stdin <=====\n\n  rule foo:\n-       input: "foo.txt"\n+     input:\n+         "foo.txt",\n\n[INFO] All done 🎉\n```\n\nIf multiple files are specified, a diff for each file is written to stdout, separated by\n`=====> Diff for <filepath> <=====`.\n\n## Configuration\n\n`snakefmt` is able to read project-specific default values for its command line options\nfrom a `pyproject.toml` file. In addition, it will also load any [`black`\nconfigurations][black-config] you have in the same file.\n\nBy default, `snakefmt` will search in your current directory for a file named\n`pyproject.toml`. If your configuration file is located somewhere else or called\nsomething different, then specify the location with `--config`.\n\nAny options you pass on the command line will take precedence over default values in the\nconfiguration file.\n\n#### Example\n\n`pyproject.toml`\n\n```toml\n[tool.snakefmt]\nline-length = 90\ninclude = \'\\.smk$|^Snakefile|\\.py$\'\n\n# snakefmt passes these options on to black\n[tool.black]\ntarget-version = ["py37"]\n```\n\nIn this example we increase the `--line-length` value and also include python (`*.py`)\nfiles for formatting - this effectively runs `black` on them. `snakefmt` will also pass\non the `[tool.black]` settings, internally, to `black`.\n\n## Design\n\n### Syntax\n\n`snakefmt`\'s parser will spot syntax errors in your snakefiles:\n\n* Unrecognised keywords\n* Duplicate keywords\n* Invalid parameters\n* Invalid python code\n\nBut `snakefmt` not complaining does not guarantee your file is entirely error-free.\n\n### Formatting\n\nPython code is `black`ed.\n\nSnakemake-specific syntax is formatted following the same principles: see [PEP8][PEP8].\n\n## Editor Integration\n\nFor instructions on how to integrate `snakefmt` into your editor of choice, refer to\n[`docs/editor_integration.md`](docs/editor_integration.md)\n\n## Plug Us\n\nIf you can\'t get enough of badges, then feel free to show others you\'re using `snakefmt`\nin your project.\n\n[![Code style: snakefmt](https://img.shields.io/badge/code%20style-snakefmt-000000.svg)](https://github.com/snakemake/snakefmt)\n\n#### Markdown\n\n```md\n[![Code style: snakefmt](https://img.shields.io/badge/code%20style-snakefmt-000000.svg)](https://github.com/snakemake/snakefmt)\n```\n\n#### ReStructuredText\n\n```rst\n.. image:: https://img.shields.io/badge/code%20style-snakefmt-000000.svg\n    :target: https://github.com/snakemake/snakefmt\n```\n\n## Changes\n\nSee [`CHANGELOG.md`][changes].\n\n## Contributing\n\nPlease refer to [CONTRIBUTING.md][contributing].\n\n\n[snakemake]: https://snakemake.readthedocs.io/\n[black]: https://black.readthedocs.io/en/stable/\n[PEP8]: https://www.python.org/dev/peps/pep-0008/\n[pyproject]: https://github.com/snakemake/snakefmt/blob/master/pyproject.toml\n[poetry]: https://python-poetry.org\n[contributing]: CONTRIBUTING.md\n[changes]: CHANGELOG.md\n[black-config]: https://github.com/psf/black#pyprojecttoml\n\n',
    'author': 'Michael Hall',
    'author_email': 'michael@mbh.sh',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/snakemake/snakefmt',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
