# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/baseWeightsPredictor.ipynb.

# %% auto 0
__all__ = ['BaseWeightsBasedPredictor', 'restructureWeightsDataList', 'summarizeWeightsData']

# %% ../nbs/baseWeightsPredictor.ipynb 5
from abc import ABC, abstractmethod

import pandas as pd
import numpy as np

import ipdb

# %% ../nbs/baseWeightsPredictor.ipynb 7
class BaseWeightsBasedPredictor(ABC):
    
    @abstractmethod
    def __init__(self):
        """Define weights-based predictor"""
    
    #---
    
    @abstractmethod
    def fit(self, X, Y):
        """Fit weights-based predictor on given training data"""
    
    #---
    
    @abstractmethod
    def getWeightsData(self, X, scalingList = None):
        """Compute weights of feature array X"""
    
    #---
    
    def predict(self, X, probs = [0.1, 0.5, 0.9], outputAsDf = False, scalingList = None):
        
        distributionDataList = self.getWeightsData(X = X,
                                                   outputType = 'cumulativeDistribution',
                                                   scalingList = scalingList)
        
        quantilesDict = {prob: [] for prob in probs}
        
        for probsDistributionFunction, YDistributionFunction in distributionDataList:
        
            for prob in probs:
                quantileIndex = np.where(probsDistributionFunction >= prob)[0][0]
                quantile = YDistributionFunction[quantileIndex]
                quantilesDict[prob].append(quantile)
        
        quantilesDf = pd.DataFrame(quantilesDict)
        
        # Just done to make the dictionary contain arrays rather than lists of the quantiles.
        quantilesDict = {prob: np.array(quantiles) for prob, quantiles in quantilesDict.items()}
        
        #---
        
        if outputAsDf:
            return quantilesDf
        
        else:
            return quantilesDict
    

# %% ../nbs/baseWeightsPredictor.ipynb 10
def restructureWeightsDataList(weightsDataList, outputType = 'onlyPositiveWeights', Y = None, scalingList = None, equalWeights = False):
    
    """
    Helper function. Creates weights-output by specifying considered
    neighbors of training observations for every test observation of interest.

    Parameters
    ----------
    neighborsList : {list}
        The i-th list-entry is supposed to correspond to the i-th test observation. 
        Every list-entry should be a array containing the indices of training observations
        which were selected as the neighbors of the considered test observation based on
        the selected Level-Set-Forecaster algorithm.     
    outputType : {"summarized", "onlyPositiveWeights", "all"}, default="onlyPositiveWeights"
        Specifies the structure of the output. 
        - If "all", then the weights are outputted as an array that is exactly as long as 
          the number of training observations. Consequently, also weights equal to zero are
          being computed. 
          NOTE: This can be take up lots of RAM for large datasets with
          > 10^6 observations.
        - If "onlyPositiveWeights", then weights equal to zero are truncated. In order to be 
          able to identify to which training observation each weight belongs, a tuple is
          outputted whose first entry are the weights and the second one are the corresponding
          training indices. 
        - If "summarized", then additionally to "onlyPositiveWeights", weights referencing to the
          same y-value are condensed to one single weight. In this case, the second entry of the
          outputted tuple contains the y-values to which each weight corresponds. 
          NOTE: Summarizing the weights can be very computationally burdensome if roughly the considered
          dataset has more than 10^6 observations and if ``binSize`` > 10^4.
        - If "cumulativeDistributionSummarized", then additionally to "summarized", the cumulative sum of the
          weights is computed, which can be interpreted as the empirical cumulative distribution
          function given the feature vector at hand.
          NOTE: This output type requires summarizing the weights, which can be very computationally 
          burdensome if roughly the considered dataset has more than 10^6 observations and if 
          ``binSize`` > 10^4.
    Y: array, default=None
        The target values of the training observations. Only needed when ``outputType`` is given as 
        "all" or "summarized"."""
    
    if outputType == 'all':
        
        weightsDataListAll = list()
        
        for weights, indicesPosWeight in weightsDataList:
            weightsAll = np.zeros(len(Y))
            weightsAll[indicesPosWeight] = weights
            weightsDataListAll.append(weightsAll)
        
        return weightsDataListAll
    
    #---
    
    elif outputType == 'onlyPositiveWeights':
        
        return weightsDataList
    
    #---
    
    elif outputType == 'summarized':
        
        weightsDataListSummarized = list()

        for i in range(len(weightsDataList)):
            weightsPos, YWeightPos = weightsDataList[i][0], Y[weightsDataList[i][1]]
            
            weightsSummarized, YUnique = summarizeWeightsData(weightsPos = weightsPos, 
                                                              YWeightPos = YWeightPos,
                                                              equalWeights = equalWeights)
            
            if not scalingList is None:
                YUnique = YUnique * scalingList[i]
                
            weightsDataListSummarized.append((weightsSummarized, YUnique))
            
        return weightsDataListSummarized
    
    #---
    
    elif outputType == 'cumulativeDistribution':
        
        distributionDataList = list()
        
        for i in range(len(weightsDataList)):
            weightsPos, YWeightPos = weightsDataList[i][0], Y[weightsDataList[i][1]]
            
            indicesSort = np.argsort(YWeightPos)
            
            weightsPosSorted = weightsPos[indicesSort]
            YWeightPosSorted = YWeightPos[indicesSort]
            
            cumulativeProbs = np.cumsum(weightsPosSorted)
            
            if not scalingList is None:
                YWeightPosSorted = YWeightPosSorted * scalingList[i]
                
            distributionDataList.append((cumulativeProbs, YWeightPosSorted))
            
        return distributionDataList
    
    #---
    
    elif outputType == 'cumulativeDistributionSummarized':
        
        distributionDataList = list()
        
        for i in range(len(weightsDataList)):
            weightsPos, YWeightPos = weightsDataList[i][0], Y[weightsDataList[i][1]]
            
            weightsSummarizedSorted, YPosWeightUniqueSorted = summarizeWeightsData(weightsPos = weightsPos, 
                                                                                   YWeightPos = YWeightPos,
                                                                                   equalWeights = equalWeights)
            
            cumulativeProbs = np.cumsum(weightsSummarizedSorted)
            
            if not scalingList is None:
                YPosWeightUniqueSorted = YPosWeightUniqueSorted * scalingList[i]
                
            distributionDataList.append((cumulativeProbs, YPosWeightUniqueSorted))
            
        return distributionDataList
    

# %% ../nbs/baseWeightsPredictor.ipynb 12
def summarizeWeightsData(weightsPos, YWeightPos, equalWeights = False):
    
    if equalWeights:
        counterDict = Counter(YWeightPos)
        YUniqueSorted = np.sort(list(counterDict.keys()))

        weightsSummarizedSorted = np.array([counterDict[value] / len(YWeightPos) for value in YUniqueSorted])
    
    else:
        duplicationDict = defaultdict(list)

        for i, item in enumerate(YWeightPos):
            duplicationDict[item].append(i)

        #---

        weightsSummarized = list()
        YUnique = list()

        for value, indices in duplicationDict.items():        

            weightsSummarized.append(weightsPos[indices].sum())
            YUnique.append(value)

        weightsSummarized, YUnique = np.array(weightsSummarized), np.array(YUnique)

        #---

        indicesSort = np.argsort(YUnique)
        weightsSummarizedSorted, YUniqueSorted = weightsSummarized[indicesSort], YUnique[indicesSort]
    
    return weightsSummarizedSorted, YUniqueSorted
