"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaLayerVersionCleaner = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const events = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const iam = require("aws-cdk-lib/aws-iam");
const logs = require("aws-cdk-lib/aws-logs");
const constructs_1 = require("constructs");
const delete_old_lambda_layer_versions_function_1 = require("./lambda/delete-old-lambda-layer-versions-function");
/**
 * Lambda Layer Version Cleaner Construct
 *
 * This construct creates a Lambda function that deletes old versions of a Lambda Layer. The function is
 * scheduled to run at a regular interval using an EventBridge rule. The number of versions to retain
 * must be specified as a positive integer using the `ILambdaLayerVersionCleanerProps` interface.
 * The function execution schedule is also required to be set in the `ILambdaLayerVersionCleanerProps` interface.
 */
class LambdaLayerVersionCleaner extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // Check if retainVersions is a positive integer
        if (props.retainVersions <= 0 || !Number.isInteger(props.retainVersions)) {
            throw new Error('Invalid retainVersions value. It should be a positive integer.');
        }
        // Create the Lambda function to delete old layer versions
        this.handler = new delete_old_lambda_layer_versions_function_1.DeleteOldLambdaLayerVersionsFunction(this, 'DeleteOldLambdaLayerVersionsFunction', {
            environment: { RETAIN_VERSIONS: props.retainVersions.toString() },
            timeout: props.handlerTimeout ?? cdk.Duration.minutes(15),
            memorySize: props.handlerMemorySize ?? 256,
        });
        // Create the LogGroup for the Lambda function
        new logs.LogGroup(this, `${this.handler.node.id}LogGroup`, {
            logGroupName: `/aws/lambda/${this.handler.functionName}`,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        // Add the necessary IAM permissions for the Lambda function
        const handlerPolicyStatement = new iam.PolicyStatement({
            actions: ['lambda:ListLayers', 'lambda:ListLayerVersions', 'lambda:DeleteLayerVersion'],
            effect: iam.Effect.ALLOW,
            resources: ['*'],
        });
        this.handler.addToRolePolicy(handlerPolicyStatement);
        // Create the EventBridge rule for the Lambda function execution schedule
        this.rule = new events.Rule(this, 'LayerCleanerScheduleRule', {
            schedule: props.layerCleanerSchedule,
        });
        this.rule.addTarget(new targets.LambdaFunction(this.handler));
        // Add the necessary permissions for the EventBridge rule to invoke the Lambda function
        targets.addLambdaPermission(this.rule, this.handler);
    }
}
exports.LambdaLayerVersionCleaner = LambdaLayerVersionCleaner;
_a = JSII_RTTI_SYMBOL_1;
LambdaLayerVersionCleaner[_a] = { fqn: "lambda-layer-version-cleaner.LambdaLayerVersionCleaner", version: "0.0.7" };
//# sourceMappingURL=data:application/json;base64,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