"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ParameterGroup = void 0;
const core_1 = require("@aws-cdk/core");
const rds_generated_1 = require("./rds.generated");
/**
 * A parameter group.
 *
 * Represents both a cluster parameter group,
 * and an instance parameter group.
 *
 * @stability stable
 * @resource AWS::RDS::DBParameterGroup
 */
class ParameterGroup extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const family = props.engine.parameterGroupFamily;
        if (!family) {
            throw new Error("ParameterGroup cannot be used with an engine that doesn't specify a version");
        }
        this.family = family;
        this.description = props.description;
        this.parameters = (_a = props.parameters) !== null && _a !== void 0 ? _a : {};
    }
    /**
     * Imports a parameter group.
     *
     * @stability stable
     */
    static fromParameterGroupName(scope, id, parameterGroupName) {
        class Import extends core_1.Resource {
            bindToCluster(_options) {
                return { parameterGroupName };
            }
            bindToInstance(_options) {
                return { parameterGroupName };
            }
            addParameter(_key, _value) {
                return false;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Method called when this Parameter Group is used when defining a database cluster.
     *
     * @stability stable
     */
    bindToCluster(_options) {
        if (!this.clusterCfnGroup) {
            const id = this.instanceCfnGroup ? 'ClusterParameterGroup' : 'Resource';
            this.clusterCfnGroup = new rds_generated_1.CfnDBClusterParameterGroup(this, id, {
                description: this.description || `Cluster parameter group for ${this.family}`,
                family: this.family,
                parameters: core_1.Lazy.any({ produce: () => this.parameters }),
            });
        }
        return {
            parameterGroupName: this.clusterCfnGroup.ref,
        };
    }
    /**
     * Method called when this Parameter Group is used when defining a database instance.
     *
     * @stability stable
     */
    bindToInstance(_options) {
        if (!this.instanceCfnGroup) {
            const id = this.clusterCfnGroup ? 'InstanceParameterGroup' : 'Resource';
            this.instanceCfnGroup = new rds_generated_1.CfnDBParameterGroup(this, id, {
                description: this.description || `Parameter group for ${this.family}`,
                family: this.family,
                parameters: core_1.Lazy.any({ produce: () => this.parameters }),
            });
        }
        return {
            parameterGroupName: this.instanceCfnGroup.ref,
        };
    }
    /**
     * Add a parameter to this parameter group.
     *
     * @param key The key of the parameter to be added.
     * @param value The value of the parameter to be added.
     * @stability stable
     */
    addParameter(key, value) {
        this.parameters[key] = value;
        return true;
    }
}
exports.ParameterGroup = ParameterGroup;
//# sourceMappingURL=data:application/json;base64,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