import * as iam from '@aws-cdk/aws-iam';
import * as secretsmanager from '@aws-cdk/aws-secretsmanager';
import * as core from '@aws-cdk/core';
import { IEngine } from './engine';
import { IParameterGroup } from './parameter-group';
/**
 * The extra options passed to the {@link IClusterEngine.bindToCluster} method.
 *
 * @stability stable
 */
export interface ClusterEngineBindOptions {
    /**
     * The role used for S3 importing.
     *
     * @default - none
     * @stability stable
     */
    readonly s3ImportRole?: iam.IRole;
    /**
     * The role used for S3 exporting.
     *
     * @default - none
     * @stability stable
     */
    readonly s3ExportRole?: iam.IRole;
    /**
     * The customer-provided ParameterGroup.
     *
     * @default - none
     * @stability stable
     */
    readonly parameterGroup?: IParameterGroup;
}
/**
 * The type returned from the {@link IClusterEngine.bindToCluster} method.
 *
 * @stability stable
 */
export interface ClusterEngineConfig {
    /**
     * The ParameterGroup to use for the cluster.
     *
     * @default - no ParameterGroup will be used
     * @stability stable
     */
    readonly parameterGroup?: IParameterGroup;
    /**
     * The port to use for this cluster, unless the customer specified the port directly.
     *
     * @default - use the default port for clusters (3306)
     * @stability stable
     */
    readonly port?: number;
    /**
     * Features supported by the database engine.
     *
     * @default - no features
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DBEngineVersion.html
     * @stability stable
     */
    readonly features?: ClusterEngineFeatures;
}
/**
 * Represents Database Engine features.
 *
 * @stability stable
 */
export interface ClusterEngineFeatures {
    /**
     * Feature name for the DB instance that the IAM role to access the S3 bucket for import is to be associated with.
     *
     * @default - no s3Import feature name
     * @stability stable
     */
    readonly s3Import?: string;
    /**
     * Feature name for the DB instance that the IAM role to export to S3 bucket is to be associated with.
     *
     * @default - no s3Export feature name
     * @stability stable
     */
    readonly s3Export?: string;
}
/**
 * The interface representing a database cluster (as opposed to instance) engine.
 *
 * @stability stable
 */
export interface IClusterEngine extends IEngine {
    /**
     * The application used by this engine to perform rotation for a single-user scenario.
     *
     * @stability stable
     */
    readonly singleUserRotationApplication: secretsmanager.SecretRotationApplication;
    /**
     * The application used by this engine to perform rotation for a multi-user scenario.
     *
     * @stability stable
     */
    readonly multiUserRotationApplication: secretsmanager.SecretRotationApplication;
    /**
     * The log types that are available with this engine type.
     *
     * @stability stable
     */
    readonly supportedLogTypes: string[];
    /**
     * Method called when the engine is used to create a new cluster.
     *
     * @stability stable
     */
    bindToCluster(scope: core.Construct, options: ClusterEngineBindOptions): ClusterEngineConfig;
}
/**
 * The versions for the Aurora cluster engine (those returned by {@link DatabaseClusterEngine.aurora}).
 *
 * @stability stable
 */
export declare class AuroraEngineVersion {
    /**
     * Version "5.6.10a".
     *
     * @stability stable
     */
    static readonly VER_10A: AuroraEngineVersion;
    /**
     * Version "5.6.mysql_aurora.1.17.9".
     *
     * @stability stable
     */
    static readonly VER_1_17_9: AuroraEngineVersion;
    /**
     * Version "5.6.mysql_aurora.1.19.0".
     *
     * @stability stable
     */
    static readonly VER_1_19_0: AuroraEngineVersion;
    /**
     * Version "5.6.mysql_aurora.1.19.1".
     *
     * @stability stable
     */
    static readonly VER_1_19_1: AuroraEngineVersion;
    /**
     * Version "5.6.mysql_aurora.1.19.2".
     *
     * @stability stable
     */
    static readonly VER_1_19_2: AuroraEngineVersion;
    /**
     * Version "5.6.mysql_aurora.1.19.5".
     *
     * @stability stable
     */
    static readonly VER_1_19_5: AuroraEngineVersion;
    /**
     * Version "5.6.mysql_aurora.1.19.6".
     *
     * @stability stable
     */
    static readonly VER_1_19_6: AuroraEngineVersion;
    /**
     * Version "5.6.mysql_aurora.1.20.0".
     *
     * @stability stable
     */
    static readonly VER_1_20_0: AuroraEngineVersion;
    /**
     * Version "5.6.mysql_aurora.1.20.1".
     *
     * @stability stable
     */
    static readonly VER_1_20_1: AuroraEngineVersion;
    /**
     * Version "5.6.mysql_aurora.1.21.0".
     *
     * @stability stable
     */
    static readonly VER_1_21_0: AuroraEngineVersion;
    /**
     * Version "5.6.mysql_aurora.1.22.0".
     *
     * @stability stable
     */
    static readonly VER_1_22_0: AuroraEngineVersion;
    /**
     * Version "5.6.mysql_aurora.1.22.1".
     *
     * @stability stable
     */
    static readonly VER_1_22_1: AuroraEngineVersion;
    /**
     * Version "5.6.mysql_aurora.1.22.1.3".
     *
     * @stability stable
     */
    static readonly VER_1_22_1_3: AuroraEngineVersion;
    /**
     * Version "5.6.mysql_aurora.1.22.2".
     *
     * @stability stable
     */
    static readonly VER_1_22_2: AuroraEngineVersion;
    /**
     * Create a new AuroraEngineVersion with an arbitrary version.
     *
     * @param auroraFullVersion the full version string, for example "5.6.mysql_aurora.1.78.3.6".
     * @param auroraMajorVersion the major version of the engine, defaults to "5.6".
     * @stability stable
     */
    static of(auroraFullVersion: string, auroraMajorVersion?: string): AuroraEngineVersion;
    private static builtIn_5_6;
    /**
     * The full version string, for example, "5.6.mysql_aurora.1.78.3.6".
     *
     * @stability stable
     */
    readonly auroraFullVersion: string;
    /**
     * The major version of the engine.
     *
     * Currently, it's always "5.6".
     *
     * @stability stable
     */
    readonly auroraMajorVersion: string;
    private constructor();
}
/**
 * Creation properties of the plain Aurora database cluster engine.
 *
 * Used in {@link DatabaseClusterEngine.aurora}.
 *
 * @stability stable
 */
export interface AuroraClusterEngineProps {
    /**
     * The version of the Aurora cluster engine.
     *
     * @stability stable
     */
    readonly version: AuroraEngineVersion;
}
/**
 * The versions for the Aurora MySQL cluster engine (those returned by {@link DatabaseClusterEngine.auroraMysql}).
 *
 * @stability stable
 */
export declare class AuroraMysqlEngineVersion {
    /**
     * Version "5.7.12".
     *
     * @stability stable
     */
    static readonly VER_5_7_12: AuroraMysqlEngineVersion;
    /**
     * Version "5.7.mysql_aurora.2.03.2".
     *
     * @stability stable
     */
    static readonly VER_2_03_2: AuroraMysqlEngineVersion;
    /**
     * Version "5.7.mysql_aurora.2.03.3".
     *
     * @stability stable
     */
    static readonly VER_2_03_3: AuroraMysqlEngineVersion;
    /**
     * Version "5.7.mysql_aurora.2.03.4".
     *
     * @stability stable
     */
    static readonly VER_2_03_4: AuroraMysqlEngineVersion;
    /**
     * Version "5.7.mysql_aurora.2.04.0".
     *
     * @stability stable
     */
    static readonly VER_2_04_0: AuroraMysqlEngineVersion;
    /**
     * Version "5.7.mysql_aurora.2.04.1".
     *
     * @stability stable
     */
    static readonly VER_2_04_1: AuroraMysqlEngineVersion;
    /**
     * Version "5.7.mysql_aurora.2.04.2".
     *
     * @stability stable
     */
    static readonly VER_2_04_2: AuroraMysqlEngineVersion;
    /**
     * Version "5.7.mysql_aurora.2.04.3".
     *
     * @stability stable
     */
    static readonly VER_2_04_3: AuroraMysqlEngineVersion;
    /**
     * Version "5.7.mysql_aurora.2.04.4".
     *
     * @stability stable
     */
    static readonly VER_2_04_4: AuroraMysqlEngineVersion;
    /**
     * Version "5.7.mysql_aurora.2.04.5".
     *
     * @stability stable
     */
    static readonly VER_2_04_5: AuroraMysqlEngineVersion;
    /**
     * Version "5.7.mysql_aurora.2.04.6".
     *
     * @stability stable
     */
    static readonly VER_2_04_6: AuroraMysqlEngineVersion;
    /**
     * Version "5.7.mysql_aurora.2.04.7".
     *
     * @stability stable
     */
    static readonly VER_2_04_7: AuroraMysqlEngineVersion;
    /**
     * Version "5.7.mysql_aurora.2.04.8".
     *
     * @stability stable
     */
    static readonly VER_2_04_8: AuroraMysqlEngineVersion;
    /**
     * Version "5.7.mysql_aurora.2.05.0".
     *
     * @stability stable
     */
    static readonly VER_2_05_0: AuroraMysqlEngineVersion;
    /**
     * Version "5.7.mysql_aurora.2.06.0".
     *
     * @stability stable
     */
    static readonly VER_2_06_0: AuroraMysqlEngineVersion;
    /**
     * Version "5.7.mysql_aurora.2.07.0".
     *
     * @stability stable
     */
    static readonly VER_2_07_0: AuroraMysqlEngineVersion;
    /**
     * Version "5.7.mysql_aurora.2.07.1".
     *
     * @stability stable
     */
    static readonly VER_2_07_1: AuroraMysqlEngineVersion;
    /**
     * Version "5.7.mysql_aurora.2.07.2".
     *
     * @stability stable
     */
    static readonly VER_2_07_2: AuroraMysqlEngineVersion;
    /**
     * Version "5.7.mysql_aurora.2.08.0".
     *
     * @stability stable
     */
    static readonly VER_2_08_0: AuroraMysqlEngineVersion;
    /**
     * Version "5.7.mysql_aurora.2.08.1".
     *
     * @stability stable
     */
    static readonly VER_2_08_1: AuroraMysqlEngineVersion;
    /**
     * Version "5.7.mysql_aurora.2.08.2".
     *
     * @stability stable
     */
    static readonly VER_2_08_2: AuroraMysqlEngineVersion;
    /**
     * Version "5.7.mysql_aurora.2.09.0".
     *
     * @stability stable
     */
    static readonly VER_2_09_0: AuroraMysqlEngineVersion;
    /**
     * Version "5.7.mysql_aurora.2.09.1".
     *
     * @stability stable
     */
    static readonly VER_2_09_1: AuroraMysqlEngineVersion;
    /**
     * Create a new AuroraMysqlEngineVersion with an arbitrary version.
     *
     * @param auroraMysqlFullVersion the full version string, for example "5.7.mysql_aurora.2.78.3.6".
     * @param auroraMysqlMajorVersion the major version of the engine, defaults to "5.7".
     * @stability stable
     */
    static of(auroraMysqlFullVersion: string, auroraMysqlMajorVersion?: string): AuroraMysqlEngineVersion;
    private static builtIn_5_7;
    /**
     * The full version string, for example, "5.7.mysql_aurora.1.78.3.6".
     *
     * @stability stable
     */
    readonly auroraMysqlFullVersion: string;
    /**
     * The major version of the engine.
     *
     * Currently, it's always "5.7".
     *
     * @stability stable
     */
    readonly auroraMysqlMajorVersion: string;
    private constructor();
}
/**
 * Creation properties of the Aurora MySQL database cluster engine.
 *
 * Used in {@link DatabaseClusterEngine.auroraMysql}.
 *
 * @stability stable
 */
export interface AuroraMysqlClusterEngineProps {
    /**
     * The version of the Aurora MySQL cluster engine.
     *
     * @stability stable
     */
    readonly version: AuroraMysqlEngineVersion;
}
/**
 * Features supported by this version of the Aurora Postgres cluster engine.
 *
 * @stability stable
 */
export interface AuroraPostgresEngineFeatures {
    /**
     * Whether this version of the Aurora Postgres cluster engine supports the S3 data import feature.
     *
     * @default false
     * @stability stable
     */
    readonly s3Import?: boolean;
    /**
     * Whether this version of the Aurora Postgres cluster engine supports the S3 data import feature.
     *
     * @default false
     * @stability stable
     */
    readonly s3Export?: boolean;
}
/**
 * The versions for the Aurora PostgreSQL cluster engine (those returned by {@link DatabaseClusterEngine.auroraPostgres}).
 *
 * @stability stable
 */
export declare class AuroraPostgresEngineVersion {
    /**
     * Version "9.6.8".
     *
     * @stability stable
     */
    static readonly VER_9_6_8: AuroraPostgresEngineVersion;
    /**
     * Version "9.6.9".
     *
     * @stability stable
     */
    static readonly VER_9_6_9: AuroraPostgresEngineVersion;
    /**
     * Version "9.6.11".
     *
     * @stability stable
     */
    static readonly VER_9_6_11: AuroraPostgresEngineVersion;
    /**
     * Version "9.6.12".
     *
     * @stability stable
     */
    static readonly VER_9_6_12: AuroraPostgresEngineVersion;
    /**
     * Version "9.6.16".
     *
     * @stability stable
     */
    static readonly VER_9_6_16: AuroraPostgresEngineVersion;
    /**
     * Version "9.6.17".
     *
     * @stability stable
     */
    static readonly VER_9_6_17: AuroraPostgresEngineVersion;
    /**
     * Version "10.4".
     *
     * @stability stable
     */
    static readonly VER_10_4: AuroraPostgresEngineVersion;
    /**
     * Version "10.5".
     *
     * @stability stable
     */
    static readonly VER_10_5: AuroraPostgresEngineVersion;
    /**
     * Version "10.6".
     *
     * @stability stable
     */
    static readonly VER_10_6: AuroraPostgresEngineVersion;
    /**
     * Version "10.7".
     *
     * @stability stable
     */
    static readonly VER_10_7: AuroraPostgresEngineVersion;
    /**
     * Version "10.11".
     *
     * @stability stable
     */
    static readonly VER_10_11: AuroraPostgresEngineVersion;
    /**
     * Version "10.12".
     *
     * @stability stable
     */
    static readonly VER_10_12: AuroraPostgresEngineVersion;
    /**
     * Version "11.4".
     *
     * @stability stable
     */
    static readonly VER_11_4: AuroraPostgresEngineVersion;
    /**
     * Version "11.6".
     *
     * @stability stable
     */
    static readonly VER_11_6: AuroraPostgresEngineVersion;
    /**
     * Version "11.7".
     *
     * @stability stable
     */
    static readonly VER_11_7: AuroraPostgresEngineVersion;
    /**
     * Version "11.8".
     *
     * @stability stable
     */
    static readonly VER_11_8: AuroraPostgresEngineVersion;
    /**
     * Version "11.9".
     *
     * @stability stable
     */
    static readonly VER_11_9: AuroraPostgresEngineVersion;
    /**
     * Create a new AuroraPostgresEngineVersion with an arbitrary version.
     *
     * @param auroraPostgresFullVersion the full version string, for example "9.6.25.1".
     * @param auroraPostgresMajorVersion the major version of the engine, for example "9.6".
     * @stability stable
     */
    static of(auroraPostgresFullVersion: string, auroraPostgresMajorVersion: string, auroraPostgresFeatures?: AuroraPostgresEngineFeatures): AuroraPostgresEngineVersion;
    /**
     * The full version string, for example, "9.6.25.1".
     *
     * @stability stable
     */
    readonly auroraPostgresFullVersion: string;
    /**
     * The major version of the engine, for example, "9.6".
     *
     * @stability stable
     */
    readonly auroraPostgresMajorVersion: string;
    /**
     * The supported features for the DB engine
     *
     * @internal
     */
    readonly _features: ClusterEngineFeatures;
    private constructor();
}
/**
 * Creation properties of the Aurora PostgreSQL database cluster engine.
 *
 * Used in {@link DatabaseClusterEngine.auroraPostgres}.
 *
 * @stability stable
 */
export interface AuroraPostgresClusterEngineProps {
    /**
     * The version of the Aurora PostgreSQL cluster engine.
     *
     * @stability stable
     */
    readonly version: AuroraPostgresEngineVersion;
}
/**
 * A database cluster engine.
 *
 * Provides mapping to the serverless application
 * used for secret rotation.
 *
 * @stability stable
 */
export declare class DatabaseClusterEngine {
    /**
     * The unversioned 'aurora' cluster engine.
     *
     * **Note**: we do not recommend using unversioned engines for non-serverless Clusters,
     *    as that can pose an availability risk.
     *    We recommend using versioned engines created using the {@link aurora()} method
     *
     * @stability stable
     */
    static readonly AURORA: IClusterEngine;
    /**
     * The unversioned 'aurora-msql' cluster engine.
     *
     * **Note**: we do not recommend using unversioned engines for non-serverless Clusters,
     *    as that can pose an availability risk.
     *    We recommend using versioned engines created using the {@link auroraMysql()} method
     *
     * @stability stable
     */
    static readonly AURORA_MYSQL: IClusterEngine;
    /**
     * The unversioned 'aurora-postgresql' cluster engine.
     *
     * **Note**: we do not recommend using unversioned engines for non-serverless Clusters,
     *    as that can pose an availability risk.
     *    We recommend using versioned engines created using the {@link auroraPostgres()} method
     *
     * @stability stable
     */
    static readonly AURORA_POSTGRESQL: IClusterEngine;
    /**
     * Creates a new plain Aurora database cluster engine.
     *
     * @stability stable
     */
    static aurora(props: AuroraClusterEngineProps): IClusterEngine;
    /**
     * Creates a new Aurora MySQL database cluster engine.
     *
     * @stability stable
     */
    static auroraMysql(props: AuroraMysqlClusterEngineProps): IClusterEngine;
    /**
     * Creates a new Aurora PostgreSQL database cluster engine.
     *
     * @stability stable
     */
    static auroraPostgres(props: AuroraPostgresClusterEngineProps): IClusterEngine;
}
