"""
# Amazon Relational Database Service Construct Library

<!--BEGIN STABILITY BANNER-->---


![cfn-resources: Stable](https://img.shields.io/badge/cfn--resources-stable-success.svg?style=for-the-badge)

![cdk-constructs: Stable](https://img.shields.io/badge/cdk--constructs-stable-success.svg?style=for-the-badge)

---
<!--END STABILITY BANNER-->

```python
# Example automatically generated. See https://github.com/aws/jsii/issues/826
import aws_cdk.aws_rds as rds
```

## Starting a clustered database

To set up a clustered database (like Aurora), define a `DatabaseCluster`. You must
always launch a database in a VPC. Use the `vpcSubnets` attribute to control whether
your instances will be launched privately or publicly:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
cluster = rds.DatabaseCluster(self, "Database",
    engine=rds.DatabaseClusterEngine.aurora_mysql(version=rds.AuroraMysqlEngineVersion.VER_2_08_1),
    credentials=rds.Credentials.from_generated_secret("clusteradmin"), # Optional - will default to 'admin' username and generated password
    instance_props={
        # optional , defaults to t3.medium
        "instance_type": ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
        "vpc_subnets": {
            "subnet_type": ec2.SubnetType.PRIVATE
        },
        "vpc": vpc
    }
)
```

If there isn't a constant for the exact version you want to use,
all of the `Version` classes have a static `of` method that can be used to create an arbitrary version.

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
custom_engine_version = rds.AuroraMysqlEngineVersion.of("5.7.mysql_aurora.2.08.1")
```

By default, the master password will be generated and stored in AWS Secrets Manager with auto-generated description.

Your cluster will be empty by default. To add a default database upon construction, specify the
`defaultDatabaseName` attribute.

Use `DatabaseClusterFromSnapshot` to create a cluster from a snapshot:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
rds.DatabaseClusterFromSnapshot(stack, "Database",
    engine=rds.DatabaseClusterEngine.aurora(version=rds.AuroraEngineVersion.VER_1_22_2),
    instance_props={
        "vpc": vpc
    },
    snapshot_identifier="mySnapshot"
)
```

## Starting an instance database

To set up a instance database, define a `DatabaseInstance`. You must
always launch a database in a VPC. Use the `vpcSubnets` attribute to control whether
your instances will be launched privately or publicly:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
instance = rds.DatabaseInstance(self, "Instance",
    engine=rds.DatabaseInstanceEngine.oracle_se2(version=rds.OracleEngineVersion.VER_19_0_0_0_2020_04_R1),
    # optional, defaults to m5.large
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE3, ec2.InstanceSize.SMALL),
    credentials=rds.Credentials.from_generated_secret("syscdk"), # Optional - will default to 'admin' username and generated password
    vpc=vpc,
    vpc_subnets={
        "subnet_type": ec2.SubnetType.PRIVATE
    }
)
```

If there isn't a constant for the exact engine version you want to use,
all of the `Version` classes have a static `of` method that can be used to create an arbitrary version.

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
custom_engine_version = rds.OracleEngineVersion.of("19.0.0.0.ru-2020-04.rur-2020-04.r1", "19")
```

By default, the master password will be generated and stored in AWS Secrets Manager.

To use the storage auto scaling option of RDS you can specify the maximum allocated storage.
This is the upper limit to which RDS can automatically scale the storage. More info can be found
[here](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PIOPS.StorageTypes.html#USER_PIOPS.Autoscaling)
Example for max storage configuration:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
instance = rds.DatabaseInstance(self, "Instance",
    engine=rds.DatabaseInstanceEngine.postgres(version=rds.PostgresEngineVersion.VER_12_3),
    # optional, defaults to m5.large
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
    vpc=vpc,
    max_allocated_storage=200
)
```

Use `DatabaseInstanceFromSnapshot` and `DatabaseInstanceReadReplica` to create an instance from snapshot or
a source database respectively:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
rds.DatabaseInstanceFromSnapshot(stack, "Instance",
    snapshot_identifier="my-snapshot",
    engine=rds.DatabaseInstanceEngine.postgres(version=rds.PostgresEngineVersion.VER_12_3),
    # optional, defaults to m5.large
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.LARGE),
    vpc=vpc
)

rds.DatabaseInstanceReadReplica(stack, "ReadReplica",
    source_database_instance=source_instance,
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.LARGE),
    vpc=vpc
)
```

Creating a "production" Oracle database instance with option and parameter groups:

```python
# Example automatically generated. See https://github.com/aws/jsii/issues/826
# Set open cursors with parameter group
parameter_group = rds.ParameterGroup(self, "ParameterGroup",
    engine=rds.DatabaseInstanceEngine.oracle_se2(version=rds.OracleEngineVersion.VER_19_0_0_0_2020_04_R1),
    parameters={
        "open_cursors": "2500"
    }
)

option_group = rds.OptionGroup(self, "OptionGroup",
    engine=rds.DatabaseInstanceEngine.oracle_se2(version=rds.OracleEngineVersion.VER_19_0_0_0_2020_04_R1),
    configurations=[OptionConfiguration(
        name="LOCATOR"
    ), OptionConfiguration(
        name="OEM",
        port=1158,
        vpc=vpc
    )
    ]
)

# Allow connections to OEM
option_group.option_connections.OEM.connections.allow_default_port_from_any_ipv4()

# Database instance with production values
instance = rds.DatabaseInstance(self, "Instance",
    engine=rds.DatabaseInstanceEngine.oracle_se2(version=rds.OracleEngineVersion.VER_19_0_0_0_2020_04_R1),
    license_model=rds.LicenseModel.BRING_YOUR_OWN_LICENSE,
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE3, ec2.InstanceSize.MEDIUM),
    multi_az=True,
    storage_type=rds.StorageType.IO1,
    credentials=rds.Credentials.from_username("syscdk"),
    vpc=vpc,
    database_name="ORCL",
    storage_encrypted=True,
    backup_retention=cdk.Duration.days(7),
    monitoring_interval=cdk.Duration.seconds(60),
    enable_performance_insights=True,
    cloudwatch_logs_exports=["trace", "audit", "alert", "listener"
    ],
    cloudwatch_logs_retention=logs.RetentionDays.ONE_MONTH,
    auto_minor_version_upgrade=False,
    option_group=option_group,
    parameter_group=parameter_group
)

# Allow connections on default port from any IPV4
instance.connections.allow_default_port_from_any_ipv4()

# Rotate the master user password every 30 days
instance.add_rotation_single_user()

# Add alarm for high CPU
cloudwatch.Alarm(self, "HighCPU",
    metric=instance.metric_cPUUtilization(),
    threshold=90,
    evaluation_periods=1
)

# Trigger Lambda function on instance availability events
fn = lambda_.Function(self, "Function",
    code=lambda_.Code.from_inline("exports.handler = (event) => console.log(event);"),
    handler="index.handler",
    runtime=lambda_.Runtime.NODEJS_10_X
)

availability_rule = instance.on_event("Availability", target=targets.LambdaFunction(fn))
availability_rule.add_event_pattern(
    detail={
        "EventCategories": ["availability"
        ]
    }
)
```

Add XMLDB and OEM with option group

```python
# Example automatically generated. See https://github.com/aws/jsii/issues/826
# Set open cursors with parameter group
parameter_group = rds.ParameterGroup(self, "ParameterGroup",
    engine=rds.DatabaseInstanceEngine.oracle_se2(version=rds.OracleEngineVersion.VER_19_0_0_0_2020_04_R1),
    parameters={
        "open_cursors": "2500"
    }
)

option_group = rds.OptionGroup(self, "OptionGroup",
    engine=rds.DatabaseInstanceEngine.oracle_se2(version=rds.OracleEngineVersion.VER_19_0_0_0_2020_04_R1),
    configurations=[OptionConfiguration(
        name="LOCATOR"
    ), OptionConfiguration(
        name="OEM",
        port=1158,
        vpc=vpc
    )
    ]
)

# Allow connections to OEM
option_group.option_connections.OEM.connections.allow_default_port_from_any_ipv4()

# Database instance with production values
instance = rds.DatabaseInstance(self, "Instance",
    engine=rds.DatabaseInstanceEngine.oracle_se2(version=rds.OracleEngineVersion.VER_19_0_0_0_2020_04_R1),
    license_model=rds.LicenseModel.BRING_YOUR_OWN_LICENSE,
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE3, ec2.InstanceSize.MEDIUM),
    multi_az=True,
    storage_type=rds.StorageType.IO1,
    credentials=rds.Credentials.from_username("syscdk"),
    vpc=vpc,
    database_name="ORCL",
    storage_encrypted=True,
    backup_retention=cdk.Duration.days(7),
    monitoring_interval=cdk.Duration.seconds(60),
    enable_performance_insights=True,
    cloudwatch_logs_exports=["trace", "audit", "alert", "listener"
    ],
    cloudwatch_logs_retention=logs.RetentionDays.ONE_MONTH,
    auto_minor_version_upgrade=False,
    option_group=option_group,
    parameter_group=parameter_group
)

# Allow connections on default port from any IPV4
instance.connections.allow_default_port_from_any_ipv4()

# Rotate the master user password every 30 days
instance.add_rotation_single_user()

# Add alarm for high CPU
cloudwatch.Alarm(self, "HighCPU",
    metric=instance.metric_cPUUtilization(),
    threshold=90,
    evaluation_periods=1
)

# Trigger Lambda function on instance availability events
fn = lambda_.Function(self, "Function",
    code=lambda_.Code.from_inline("exports.handler = (event) => console.log(event);"),
    handler="index.handler",
    runtime=lambda_.Runtime.NODEJS_10_X
)

availability_rule = instance.on_event("Availability", target=targets.LambdaFunction(fn))
availability_rule.add_event_pattern(
    detail={
        "EventCategories": ["availability"
        ]
    }
)
```

## Setting Public Accessibility

You can set public accessibility for the database instance or cluster using the `publiclyAccessible` property.
If you specify `true`, it creates an instance with a publicly resolvable DNS name, which resolves to a public IP address.
If you specify `false`, it creates an internal instance with a DNS name that resolves to a private IP address.
The default value depends on `vpcSubnets`.
It will be `true` if `vpcSubnets` is `subnetType: SubnetType.PUBLIC`, `false` otherwise.

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
# Setting public accessibility for DB instance
rds.DatabaseInstance(stack, "Instance",
    engine=rds.DatabaseInstanceEngine.mysql(
        version=rds.MysqlEngineVersion.VER_8_0_19
    ),
    vpc=vpc,
    vpc_subnets={
        "subnet_type": ec2.SubnetType.PRIVATE
    },
    publicly_accessible=True
)

# Setting public accessibility for DB cluster
rds.DatabaseCluster(stack, "DatabaseCluster",
    engine=DatabaseClusterEngine.AURORA,
    instance_props={
        "vpc": vpc,
        "vpc_subnets": {
            "subnet_type": ec2.SubnetType.PRIVATE
        },
        "publicly_accessible": True
    }
)
```

## Instance events

To define Amazon CloudWatch event rules for database instances, use the `onEvent`
method:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
rule = instance.on_event("InstanceEvent", target=targets.LambdaFunction(fn))
```

## Login credentials

By default, database instances and clusters will have `admin` user with an auto-generated password.
An alternative username (and password) may be specified for the admin user instead of the default.

The following examples use a `DatabaseInstance`, but the same usage is applicable to `DatabaseCluster`.

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
engine = rds.DatabaseInstanceEngine.postgres(version=rds.PostgresEngineVersion.VER_12_3)
rds.DatabaseInstance(self, "InstanceWithUsername",
    engine=engine,
    vpc=vpc,
    credentials=rds.Credentials.from_generated_secret("postgres")
)

rds.DatabaseInstance(self, "InstanceWithUsernameAndPassword",
    engine=engine,
    vpc=vpc,
    credentials=rds.Credentials.from_password("postgres", SecretValue.ssm_secure("/dbPassword", "1"))
)

my_secret = secretsmanager.Secret.from_secret_name(self, "DBSecret", "myDBLoginInfo")
rds.DatabaseInstance(self, "InstanceWithSecretLogin",
    engine=engine,
    vpc=vpc,
    credentials=rds.Credentials.from_secret(my_secret)
)
```

## Connecting

To control who can access the cluster or instance, use the `.connections` attribute. RDS databases have
a default port, so you don't need to specify the port:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
cluster.connections.allow_from_any_ipv4("Open to the world")
```

The endpoints to access your database cluster will be available as the `.clusterEndpoint` and `.readerEndpoint`
attributes:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
write_address = cluster.cluster_endpoint.socket_address
```

For an instance database:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
address = instance.instance_endpoint.socket_address
```

## Rotating credentials

When the master password is generated and stored in AWS Secrets Manager, it can be rotated automatically:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
instance.add_rotation_single_user(
    automatically_after=cdk.Duration.days(7), # defaults to 30 days
    exclude_characters="!@#$%^&*"
)
```

```python
# Example automatically generated. See https://github.com/aws/jsii/issues/826
cluster = rds.DatabaseCluster(stack, "Database",
    engine=rds.DatabaseClusterEngine.AURORA,
    instance_props={
        "instance_type": ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE3, ec2.InstanceSize.SMALL),
        "vpc": vpc
    }
)

cluster.add_rotation_single_user()
```

The multi user rotation scheme is also available:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
instance.add_rotation_multi_user("MyUser",
    secret=my_imported_secret
)
```

It's also possible to create user credentials together with the instance/cluster and add rotation:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
my_user_secret = rds.DatabaseSecret(self, "MyUserSecret",
    username="myuser",
    master_secret=instance.secret,
    exclude_characters="{}[]()'\"/\\"
)
my_user_secret_attached = my_user_secret.attach(instance)# Adds DB connections information in the secret

instance.add_rotation_multi_user("MyUser", # Add rotation using the multi user scheme
    secret=my_user_secret_attached)
```

**Note**: This user must be created manually in the database using the master credentials.
The rotation will start as soon as this user exists.

See also [@aws-cdk/aws-secretsmanager](https://github.com/aws/aws-cdk/blob/master/packages/%40aws-cdk/aws-secretsmanager/README.md) for credentials rotation of existing clusters/instances.

## IAM Authentication

You can also authenticate to a database instance using AWS Identity and Access Management (IAM) database authentication;
See [https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.IAMDBAuth.html](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.IAMDBAuth.html) for more information
and a list of supported versions and limitations.

The following example shows enabling IAM authentication for a database instance and granting connection access to an IAM role.

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
instance = rds.DatabaseInstance(stack, "Instance",
    engine=rds.DatabaseInstanceEngine.mysql(version=rds.MysqlEngineVersion.VER_8_0_19),
    vpc=vpc,
    iam_authentication=True
)
role = Role(stack, "DBRole", assumed_by=AccountPrincipal(stack.account))
instance.grant_connect(role)
```

The following example shows granting connection access for RDS Proxy to an IAM role.

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
cluster = rds.DatabaseCluster(stack, "Database",
    engine=rds.DatabaseClusterEngine.AURORA,
    instance_props={"vpc": vpc}
)

proxy = rds.DatabaseProxy(stack, "Proxy",
    proxy_target=rds.ProxyTarget.from_cluster(cluster),
    secrets=[cluster.secret],
    vpc=vpc
)

role = Role(stack, "DBProxyRole", assumed_by=AccountPrincipal(stack.account))
proxy.grant_connect(role)
```

**Note**: In addition to the setup above, a database user will need to be created to support IAM auth.
See [https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.IAMDBAuth.DBAccounts.html](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.IAMDBAuth.DBAccounts.html) for setup instructions.

## Kerberos Authentication

You can also authenticate using Kerberos to a database instance using AWS Managed Microsoft AD for authentication;
See [https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/kerberos-authentication.html](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/kerberos-authentication.html) for more information
and a list of supported versions and limitations.

The following example shows enabling domain support for a database instance and creating an IAM role to access
Directory Services.

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
role = iam.Role(stack, "RDSDirectoryServicesRole",
    assumed_by=iam.ServicePrincipal("rds.amazonaws.com"),
    managed_policies=[
        iam.ManagedPolicy.from_aws_managed_policy_name("service-role/AmazonRDSDirectoryServiceAccess")
    ]
)
instance = rds.DatabaseInstance(stack, "Instance",
    engine=rds.DatabaseInstanceEngine.mysql(version=rds.MysqlEngineVersion.VER_8_0_19),
    vpc=vpc,
    domain="d-????????", # The ID of the domain for the instance to join.
    domain_role=role
)
```

**Note**: In addition to the setup above, you need to make sure that the database instance has network connectivity
to the domain controllers. This includes enabling cross-VPC traffic if in a different VPC and setting up the
appropriate security groups/network ACL to allow traffic between the database instance and domain controllers.
Once configured, see [https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/kerberos-authentication.html](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/kerberos-authentication.html) for details
on configuring users for each available database engine.

## Metrics

Database instances and clusters both expose metrics (`cloudwatch.Metric`):

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
# The number of database connections in use (average over 5 minutes)
db_connections = instance.metric_database_connections()

# Average CPU utilization over 5 minutes
cpu_utilization = cluster.metric_cPUUtilization()

# The average amount of time taken per disk I/O operation (average over 1 minute)
read_latency = instance.metric("ReadLatency", statistic="Average", period_sec=60)
```

## Enabling S3 integration

Data in S3 buckets can be imported to and exported from certain database engines using SQL queries. To enable this
functionality, set the `s3ImportBuckets` and `s3ExportBuckets` properties for import and export respectively. When
configured, the CDK automatically creates and configures IAM roles as required.
Additionally, the `s3ImportRole` and `s3ExportRole` properties can be used to set this role directly.

You can read more about loading data to (or from) S3 here:

* Aurora MySQL - [import](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraMySQL.Integrating.LoadFromS3.html)
  and [export](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraMySQL.Integrating.SaveIntoS3.html).
* Aurora PostgreSQL - [import](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraPostgreSQL.Migrating.html)
  and [export](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/postgresql-s3-export.html).
* Microsoft SQL Server - [import & export](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/SQLServer.Procedural.Importing.html)
* PostgreSQL - [import](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/PostgreSQL.Procedural.Importing.html)
* Oracle - [import & export](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/oracle-s3-integration.html)

The following snippet sets up a database cluster with different S3 buckets where the data is imported and exported -

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
import aws_cdk.aws_s3 as s3

import_bucket = s3.Bucket(self, "importbucket")
export_bucket = s3.Bucket(self, "exportbucket")
rds.DatabaseCluster(self, "dbcluster",
    # ...
    s3_import_buckets=[import_bucket],
    s3_export_buckets=[export_bucket]
)
```

## Creating a Database Proxy

Amazon RDS Proxy sits between your application and your relational database to efficiently manage
connections to the database and improve scalability of the application. Learn more about at [Amazon RDS Proxy](https://aws.amazon.com/rds/proxy/)

The following code configures an RDS Proxy for a `DatabaseInstance`.

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
import aws_cdk.core as cdk
import aws_cdk.aws_ec2 as ec2
import aws_cdk.aws_rds as rds
import aws_cdk.aws_secretsmanager as secrets

vpc =
security_group =
secrets = [...]
db_instance =

proxy = db_instance.add_proxy("proxy",
    connection_borrow_timeout=cdk.Duration.seconds(30),
    max_connections_percent=50,
    secrets=secrets,
    vpc=vpc
)
```

## Exporting Logs

You can publish database logs to Amazon CloudWatch Logs. With CloudWatch Logs, you can perform real-time analysis of the log data,
store the data in highly durable storage, and manage the data with the CloudWatch Logs Agent. This is available for both database
instances and clusters; the types of logs available depend on the database type and engine being used.

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
# Exporting logs from a cluster
cluster = rds.DatabaseCluster(self, "Database",
    engine=rds.DatabaseClusterEngine.aurora({
        "version": rds.AuroraEngineVersion.VER_1_17_9
    }, cloudwatch_logs_exports, ["error", "general", "slowquery", "audit"], cloudwatch_logs_retention, logs.RetentionDays.THREE_MONTHS, cloudwatch_logs_retention_role, my_logs_publishing_role)
)

# Exporting logs from an instance
instance = rds.DatabaseInstance(self, "Instance",
    engine=rds.DatabaseInstanceEngine.postgres(
        version=rds.PostgresEngineVersion.VER_12_3
    ),
    # ...
    cloudwatch_logs_exports=["postgresql"]
)
```

## Option Groups

Some DB engines offer additional features that make it easier to manage data and databases, and to provide additional security for your database.
Amazon RDS uses option groups to enable and configure these features. An option group can specify features, called options,
that are available for a particular Amazon RDS DB instance.

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
vpc =
security_group =
rds.OptionGroup(stack, "Options",
    engine=rds.DatabaseInstanceEngine.oracle_se2(
        version=rds.OracleEngineVersion.VER_19
    ),
    configurations=[{
        "name": "OEM",
        "port": 5500,
        "vpc": vpc,
        "security_groups": [security_group]
    }
    ]
)
```

## Serverless

[Amazon Aurora Serverless]((https://aws.amazon.com/rds/aurora/serverless/)) is an on-demand, auto-scaling configuration for Amazon
Aurora. The database will automatically start up, shut down, and scale capacity
up or down based on your application's needs. It enables you to run your database
in the cloud without managing any database instances.

The following example initializes an Aurora Serverless PostgreSql cluster.
Aurora Serverless clusters can specify scaling properties which will be used to
automatically scale the database cluster seamlessly based on the workload.

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
import aws_cdk.aws_ec2 as ec2
import aws_cdk.aws_rds as rds

vpc = ec2.Vpc(self, "myrdsvpc")

cluster = rds.ServerlessCluster(self, "AnotherCluster",
    engine=rds.DatabaseClusterEngine.AURORA_POSTGRESQL,
    parameter_group=rds.ParameterGroup.from_parameter_group_name(self, "ParameterGroup", "default.aurora-postgresql10"),
    vpc=vpc,
    scaling=ServerlessScalingOptions(
        auto_pause=Duration.minutes(10), # default is to pause after 5 minutes of idle time
        min_capacity=rds.AuroraCapacityUnit.ACU_8, # default is 2 Aurora capacity units (ACUs)
        max_capacity=rds.AuroraCapacityUnit.ACU_32
    )
)
```

Aurora Serverless Clusters do not support the following features:

* Loading data from an Amazon S3 bucket
* Saving data to an Amazon S3 bucket
* Invoking an AWS Lambda function with an Aurora MySQL native function
* Aurora replicas
* Backtracking
* Multi-master clusters
* Database cloning
* IAM database cloning
* IAM database authentication
* Restoring a snapshot from MySQL DB instance
* Performance Insights
* RDS Proxy

Read more about the [limitations of Aurora Serverless](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html#aurora-serverless.limitations)

Learn more about using Amazon Aurora Serverless by reading the [documentation](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html)

### Data API

You can access your Aurora Serverless DB cluster using the built-in Data API. The Data API doesn't require a persistent connection to the DB cluster. Instead, it provides a secure HTTP endpoint and integration with AWS SDKs.

The following example shows granting Data API access to a Lamba function.

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
import aws_cdk.aws_ec2 as ec2
import aws_cdk.aws_lambda as lambda_
import aws_cdk.aws_rds as rds

vpc = ec2.Vpc(self, "MyVPC")

cluster = rds.ServerlessCluster(self, "AnotherCluster",
    engine=rds.DatabaseClusterEngine.AURORA_MYSQL,
    vpc=vpc,
    enable_data_api=True
)

fn = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_10_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler")),
    environment={
        "CLUSTER_ARN": cluster.cluster_arn,
        "SECRET_ARN": cluster.secret.secret_arn
    }
)
cluster.grant_data_api_access(fn)
```

**Note**: To invoke the Data API, the resource will need to read the secret associated with the cluster.

To learn more about using the Data API, see the [documentation](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/data-api.html).
"""
import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

from ._jsii import *

import aws_cdk.aws_cloudwatch
import aws_cdk.aws_ec2
import aws_cdk.aws_events
import aws_cdk.aws_iam
import aws_cdk.aws_kms
import aws_cdk.aws_logs
import aws_cdk.aws_s3
import aws_cdk.aws_secretsmanager
import aws_cdk.core
import constructs


@jsii.enum(jsii_type="@aws-cdk/aws-rds.AuroraCapacityUnit")
class AuroraCapacityUnit(enum.Enum):
    """(experimental) Aurora capacity units (ACUs).

    Each ACU is a combination of processing and memory capacity.

    :see: https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.how-it-works.html#aurora-serverless.architecture
    :stability: experimental
    """

    ACU_1 = "ACU_1"
    """(experimental) 1 Aurora Capacity Unit.

    :stability: experimental
    """
    ACU_2 = "ACU_2"
    """(experimental) 2 Aurora Capacity Units.

    :stability: experimental
    """
    ACU_4 = "ACU_4"
    """(experimental) 4 Aurora Capacity Units.

    :stability: experimental
    """
    ACU_8 = "ACU_8"
    """(experimental) 8 Aurora Capacity Units.

    :stability: experimental
    """
    ACU_16 = "ACU_16"
    """(experimental) 16 Aurora Capacity Units.

    :stability: experimental
    """
    ACU_32 = "ACU_32"
    """(experimental) 32 Aurora Capacity Units.

    :stability: experimental
    """
    ACU_64 = "ACU_64"
    """(experimental) 64 Aurora Capacity Units.

    :stability: experimental
    """
    ACU_128 = "ACU_128"
    """(experimental) 128 Aurora Capacity Units.

    :stability: experimental
    """
    ACU_192 = "ACU_192"
    """(experimental) 192 Aurora Capacity Units.

    :stability: experimental
    """
    ACU_256 = "ACU_256"
    """(experimental) 256 Aurora Capacity Units.

    :stability: experimental
    """
    ACU_384 = "ACU_384"
    """(experimental) 384 Aurora Capacity Units.

    :stability: experimental
    """


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.AuroraClusterEngineProps",
    jsii_struct_bases=[],
    name_mapping={"version": "version"},
)
class AuroraClusterEngineProps:
    def __init__(self, *, version: "AuroraEngineVersion") -> None:
        """Creation properties of the plain Aurora database cluster engine.

        Used in {@link DatabaseClusterEngine.aurora}.

        :param version: The version of the Aurora cluster engine.
        """
        self._values: typing.Dict[str, typing.Any] = {
            "version": version,
        }

    @builtins.property
    def version(self) -> "AuroraEngineVersion":
        """The version of the Aurora cluster engine."""
        result = self._values.get("version")
        assert result is not None, "Required property 'version' is missing"
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AuroraClusterEngineProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AuroraEngineVersion(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.AuroraEngineVersion",
):
    """The versions for the Aurora cluster engine (those returned by {@link DatabaseClusterEngine.aurora})."""

    @jsii.member(jsii_name="of")
    @builtins.classmethod
    def of(
        cls,
        aurora_full_version: builtins.str,
        aurora_major_version: typing.Optional[builtins.str] = None,
    ) -> "AuroraEngineVersion":
        """Create a new AuroraEngineVersion with an arbitrary version.

        :param aurora_full_version: the full version string, for example "5.6.mysql_aurora.1.78.3.6".
        :param aurora_major_version: the major version of the engine, defaults to "5.6".
        """
        return jsii.sinvoke(cls, "of", [aurora_full_version, aurora_major_version])

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_1_17_9")
    def VER_1_17_9(cls) -> "AuroraEngineVersion":
        """Version "5.6.mysql_aurora.1.17.9"."""
        return jsii.sget(cls, "VER_1_17_9")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_1_19_0")
    def VER_1_19_0(cls) -> "AuroraEngineVersion":
        """Version "5.6.mysql_aurora.1.19.0"."""
        return jsii.sget(cls, "VER_1_19_0")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_1_19_1")
    def VER_1_19_1(cls) -> "AuroraEngineVersion":
        """Version "5.6.mysql_aurora.1.19.1"."""
        return jsii.sget(cls, "VER_1_19_1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_1_19_2")
    def VER_1_19_2(cls) -> "AuroraEngineVersion":
        """Version "5.6.mysql_aurora.1.19.2"."""
        return jsii.sget(cls, "VER_1_19_2")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_1_19_5")
    def VER_1_19_5(cls) -> "AuroraEngineVersion":
        """Version "5.6.mysql_aurora.1.19.5"."""
        return jsii.sget(cls, "VER_1_19_5")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_1_19_6")
    def VER_1_19_6(cls) -> "AuroraEngineVersion":
        """Version "5.6.mysql_aurora.1.19.6"."""
        return jsii.sget(cls, "VER_1_19_6")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_1_20_0")
    def VER_1_20_0(cls) -> "AuroraEngineVersion":
        """Version "5.6.mysql_aurora.1.20.0"."""
        return jsii.sget(cls, "VER_1_20_0")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_1_20_1")
    def VER_1_20_1(cls) -> "AuroraEngineVersion":
        """Version "5.6.mysql_aurora.1.20.1"."""
        return jsii.sget(cls, "VER_1_20_1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_1_21_0")
    def VER_1_21_0(cls) -> "AuroraEngineVersion":
        """Version "5.6.mysql_aurora.1.21.0"."""
        return jsii.sget(cls, "VER_1_21_0")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_1_22_0")
    def VER_1_22_0(cls) -> "AuroraEngineVersion":
        """Version "5.6.mysql_aurora.1.22.0"."""
        return jsii.sget(cls, "VER_1_22_0")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_1_22_1")
    def VER_1_22_1(cls) -> "AuroraEngineVersion":
        """Version "5.6.mysql_aurora.1.22.1"."""
        return jsii.sget(cls, "VER_1_22_1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_1_22_1_3")
    def VER_1_22_1_3(cls) -> "AuroraEngineVersion":
        """Version "5.6.mysql_aurora.1.22.1.3"."""
        return jsii.sget(cls, "VER_1_22_1_3")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_1_22_2")
    def VER_1_22_2(cls) -> "AuroraEngineVersion":
        """Version "5.6.mysql_aurora.1.22.2"."""
        return jsii.sget(cls, "VER_1_22_2")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10A")
    def VER_10_A(cls) -> "AuroraEngineVersion":
        """Version "5.6.10a"."""
        return jsii.sget(cls, "VER_10A")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="auroraFullVersion")
    def aurora_full_version(self) -> builtins.str:
        """The full version string, for example, "5.6.mysql_aurora.1.78.3.6"."""
        return jsii.get(self, "auroraFullVersion")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="auroraMajorVersion")
    def aurora_major_version(self) -> builtins.str:
        """The major version of the engine.

        Currently, it's always "5.6".
        """
        return jsii.get(self, "auroraMajorVersion")


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.AuroraMysqlClusterEngineProps",
    jsii_struct_bases=[],
    name_mapping={"version": "version"},
)
class AuroraMysqlClusterEngineProps:
    def __init__(self, *, version: "AuroraMysqlEngineVersion") -> None:
        """Creation properties of the Aurora MySQL database cluster engine.

        Used in {@link DatabaseClusterEngine.auroraMysql}.

        :param version: The version of the Aurora MySQL cluster engine.
        """
        self._values: typing.Dict[str, typing.Any] = {
            "version": version,
        }

    @builtins.property
    def version(self) -> "AuroraMysqlEngineVersion":
        """The version of the Aurora MySQL cluster engine."""
        result = self._values.get("version")
        assert result is not None, "Required property 'version' is missing"
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AuroraMysqlClusterEngineProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AuroraMysqlEngineVersion(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.AuroraMysqlEngineVersion",
):
    """The versions for the Aurora MySQL cluster engine (those returned by {@link DatabaseClusterEngine.auroraMysql})."""

    @jsii.member(jsii_name="of")
    @builtins.classmethod
    def of(
        cls,
        aurora_mysql_full_version: builtins.str,
        aurora_mysql_major_version: typing.Optional[builtins.str] = None,
    ) -> "AuroraMysqlEngineVersion":
        """Create a new AuroraMysqlEngineVersion with an arbitrary version.

        :param aurora_mysql_full_version: the full version string, for example "5.7.mysql_aurora.2.78.3.6".
        :param aurora_mysql_major_version: the major version of the engine, defaults to "5.7".
        """
        return jsii.sinvoke(cls, "of", [aurora_mysql_full_version, aurora_mysql_major_version])

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_2_03_2")
    def VER_2_03_2(cls) -> "AuroraMysqlEngineVersion":
        """Version "5.7.mysql_aurora.2.03.2"."""
        return jsii.sget(cls, "VER_2_03_2")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_2_03_3")
    def VER_2_03_3(cls) -> "AuroraMysqlEngineVersion":
        """Version "5.7.mysql_aurora.2.03.3"."""
        return jsii.sget(cls, "VER_2_03_3")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_2_03_4")
    def VER_2_03_4(cls) -> "AuroraMysqlEngineVersion":
        """Version "5.7.mysql_aurora.2.03.4"."""
        return jsii.sget(cls, "VER_2_03_4")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_2_04_0")
    def VER_2_04_0(cls) -> "AuroraMysqlEngineVersion":
        """Version "5.7.mysql_aurora.2.04.0"."""
        return jsii.sget(cls, "VER_2_04_0")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_2_04_1")
    def VER_2_04_1(cls) -> "AuroraMysqlEngineVersion":
        """Version "5.7.mysql_aurora.2.04.1"."""
        return jsii.sget(cls, "VER_2_04_1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_2_04_2")
    def VER_2_04_2(cls) -> "AuroraMysqlEngineVersion":
        """Version "5.7.mysql_aurora.2.04.2"."""
        return jsii.sget(cls, "VER_2_04_2")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_2_04_3")
    def VER_2_04_3(cls) -> "AuroraMysqlEngineVersion":
        """Version "5.7.mysql_aurora.2.04.3"."""
        return jsii.sget(cls, "VER_2_04_3")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_2_04_4")
    def VER_2_04_4(cls) -> "AuroraMysqlEngineVersion":
        """Version "5.7.mysql_aurora.2.04.4"."""
        return jsii.sget(cls, "VER_2_04_4")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_2_04_5")
    def VER_2_04_5(cls) -> "AuroraMysqlEngineVersion":
        """Version "5.7.mysql_aurora.2.04.5"."""
        return jsii.sget(cls, "VER_2_04_5")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_2_04_6")
    def VER_2_04_6(cls) -> "AuroraMysqlEngineVersion":
        """Version "5.7.mysql_aurora.2.04.6"."""
        return jsii.sget(cls, "VER_2_04_6")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_2_04_7")
    def VER_2_04_7(cls) -> "AuroraMysqlEngineVersion":
        """Version "5.7.mysql_aurora.2.04.7"."""
        return jsii.sget(cls, "VER_2_04_7")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_2_04_8")
    def VER_2_04_8(cls) -> "AuroraMysqlEngineVersion":
        """Version "5.7.mysql_aurora.2.04.8"."""
        return jsii.sget(cls, "VER_2_04_8")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_2_05_0")
    def VER_2_05_0(cls) -> "AuroraMysqlEngineVersion":
        """Version "5.7.mysql_aurora.2.05.0"."""
        return jsii.sget(cls, "VER_2_05_0")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_2_06_0")
    def VER_2_06_0(cls) -> "AuroraMysqlEngineVersion":
        """Version "5.7.mysql_aurora.2.06.0"."""
        return jsii.sget(cls, "VER_2_06_0")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_2_07_0")
    def VER_2_07_0(cls) -> "AuroraMysqlEngineVersion":
        """Version "5.7.mysql_aurora.2.07.0"."""
        return jsii.sget(cls, "VER_2_07_0")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_2_07_1")
    def VER_2_07_1(cls) -> "AuroraMysqlEngineVersion":
        """Version "5.7.mysql_aurora.2.07.1"."""
        return jsii.sget(cls, "VER_2_07_1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_2_07_2")
    def VER_2_07_2(cls) -> "AuroraMysqlEngineVersion":
        """Version "5.7.mysql_aurora.2.07.2"."""
        return jsii.sget(cls, "VER_2_07_2")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_2_08_0")
    def VER_2_08_0(cls) -> "AuroraMysqlEngineVersion":
        """Version "5.7.mysql_aurora.2.08.0"."""
        return jsii.sget(cls, "VER_2_08_0")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_2_08_1")
    def VER_2_08_1(cls) -> "AuroraMysqlEngineVersion":
        """Version "5.7.mysql_aurora.2.08.1"."""
        return jsii.sget(cls, "VER_2_08_1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_2_08_2")
    def VER_2_08_2(cls) -> "AuroraMysqlEngineVersion":
        """Version "5.7.mysql_aurora.2.08.2"."""
        return jsii.sget(cls, "VER_2_08_2")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_2_09_0")
    def VER_2_09_0(cls) -> "AuroraMysqlEngineVersion":
        """Version "5.7.mysql_aurora.2.09.0"."""
        return jsii.sget(cls, "VER_2_09_0")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_2_09_1")
    def VER_2_09_1(cls) -> "AuroraMysqlEngineVersion":
        """Version "5.7.mysql_aurora.2.09.1"."""
        return jsii.sget(cls, "VER_2_09_1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_7_12")
    def VER_5_7_12(cls) -> "AuroraMysqlEngineVersion":
        """Version "5.7.12"."""
        return jsii.sget(cls, "VER_5_7_12")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="auroraMysqlFullVersion")
    def aurora_mysql_full_version(self) -> builtins.str:
        """The full version string, for example, "5.7.mysql_aurora.1.78.3.6"."""
        return jsii.get(self, "auroraMysqlFullVersion")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="auroraMysqlMajorVersion")
    def aurora_mysql_major_version(self) -> builtins.str:
        """The major version of the engine.

        Currently, it's always "5.7".
        """
        return jsii.get(self, "auroraMysqlMajorVersion")


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.AuroraPostgresClusterEngineProps",
    jsii_struct_bases=[],
    name_mapping={"version": "version"},
)
class AuroraPostgresClusterEngineProps:
    def __init__(self, *, version: "AuroraPostgresEngineVersion") -> None:
        """Creation properties of the Aurora PostgreSQL database cluster engine.

        Used in {@link DatabaseClusterEngine.auroraPostgres}.

        :param version: The version of the Aurora PostgreSQL cluster engine.
        """
        self._values: typing.Dict[str, typing.Any] = {
            "version": version,
        }

    @builtins.property
    def version(self) -> "AuroraPostgresEngineVersion":
        """The version of the Aurora PostgreSQL cluster engine."""
        result = self._values.get("version")
        assert result is not None, "Required property 'version' is missing"
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AuroraPostgresClusterEngineProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.AuroraPostgresEngineFeatures",
    jsii_struct_bases=[],
    name_mapping={"s3_export": "s3Export", "s3_import": "s3Import"},
)
class AuroraPostgresEngineFeatures:
    def __init__(
        self,
        *,
        s3_export: typing.Optional[builtins.bool] = None,
        s3_import: typing.Optional[builtins.bool] = None,
    ) -> None:
        """Features supported by this version of the Aurora Postgres cluster engine.

        :param s3_export: Whether this version of the Aurora Postgres cluster engine supports the S3 data import feature. Default: false
        :param s3_import: Whether this version of the Aurora Postgres cluster engine supports the S3 data import feature. Default: false
        """
        self._values: typing.Dict[str, typing.Any] = {}
        if s3_export is not None:
            self._values["s3_export"] = s3_export
        if s3_import is not None:
            self._values["s3_import"] = s3_import

    @builtins.property
    def s3_export(self) -> typing.Optional[builtins.bool]:
        """Whether this version of the Aurora Postgres cluster engine supports the S3 data import feature.

        :default: false
        """
        result = self._values.get("s3_export")
        return result

    @builtins.property
    def s3_import(self) -> typing.Optional[builtins.bool]:
        """Whether this version of the Aurora Postgres cluster engine supports the S3 data import feature.

        :default: false
        """
        result = self._values.get("s3_import")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AuroraPostgresEngineFeatures(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AuroraPostgresEngineVersion(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.AuroraPostgresEngineVersion",
):
    """The versions for the Aurora PostgreSQL cluster engine (those returned by {@link DatabaseClusterEngine.auroraPostgres})."""

    @jsii.member(jsii_name="of")
    @builtins.classmethod
    def of(
        cls,
        aurora_postgres_full_version: builtins.str,
        aurora_postgres_major_version: builtins.str,
        *,
        s3_export: typing.Optional[builtins.bool] = None,
        s3_import: typing.Optional[builtins.bool] = None,
    ) -> "AuroraPostgresEngineVersion":
        """Create a new AuroraPostgresEngineVersion with an arbitrary version.

        :param aurora_postgres_full_version: the full version string, for example "9.6.25.1".
        :param aurora_postgres_major_version: the major version of the engine, for example "9.6".
        :param s3_export: Whether this version of the Aurora Postgres cluster engine supports the S3 data import feature. Default: false
        :param s3_import: Whether this version of the Aurora Postgres cluster engine supports the S3 data import feature. Default: false
        """
        aurora_postgres_features = AuroraPostgresEngineFeatures(
            s3_export=s3_export, s3_import=s3_import
        )

        return jsii.sinvoke(cls, "of", [aurora_postgres_full_version, aurora_postgres_major_version, aurora_postgres_features])

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_11")
    def VER_10_11(cls) -> "AuroraPostgresEngineVersion":
        """Version "10.11"."""
        return jsii.sget(cls, "VER_10_11")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_12")
    def VER_10_12(cls) -> "AuroraPostgresEngineVersion":
        """Version "10.12"."""
        return jsii.sget(cls, "VER_10_12")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_4")
    def VER_10_4(cls) -> "AuroraPostgresEngineVersion":
        """Version "10.4"."""
        return jsii.sget(cls, "VER_10_4")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_5")
    def VER_10_5(cls) -> "AuroraPostgresEngineVersion":
        """Version "10.5"."""
        return jsii.sget(cls, "VER_10_5")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_6")
    def VER_10_6(cls) -> "AuroraPostgresEngineVersion":
        """Version "10.6"."""
        return jsii.sget(cls, "VER_10_6")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_7")
    def VER_10_7(cls) -> "AuroraPostgresEngineVersion":
        """Version "10.7"."""
        return jsii.sget(cls, "VER_10_7")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_4")
    def VER_11_4(cls) -> "AuroraPostgresEngineVersion":
        """Version "11.4"."""
        return jsii.sget(cls, "VER_11_4")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_6")
    def VER_11_6(cls) -> "AuroraPostgresEngineVersion":
        """Version "11.6"."""
        return jsii.sget(cls, "VER_11_6")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_7")
    def VER_11_7(cls) -> "AuroraPostgresEngineVersion":
        """Version "11.7"."""
        return jsii.sget(cls, "VER_11_7")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_8")
    def VER_11_8(cls) -> "AuroraPostgresEngineVersion":
        """Version "11.8"."""
        return jsii.sget(cls, "VER_11_8")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_9")
    def VER_11_9(cls) -> "AuroraPostgresEngineVersion":
        """Version "11.9"."""
        return jsii.sget(cls, "VER_11_9")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_6_11")
    def VER_9_6_11(cls) -> "AuroraPostgresEngineVersion":
        """Version "9.6.11"."""
        return jsii.sget(cls, "VER_9_6_11")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_6_12")
    def VER_9_6_12(cls) -> "AuroraPostgresEngineVersion":
        """Version "9.6.12"."""
        return jsii.sget(cls, "VER_9_6_12")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_6_16")
    def VER_9_6_16(cls) -> "AuroraPostgresEngineVersion":
        """Version "9.6.16"."""
        return jsii.sget(cls, "VER_9_6_16")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_6_17")
    def VER_9_6_17(cls) -> "AuroraPostgresEngineVersion":
        """Version "9.6.17"."""
        return jsii.sget(cls, "VER_9_6_17")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_6_8")
    def VER_9_6_8(cls) -> "AuroraPostgresEngineVersion":
        """Version "9.6.8"."""
        return jsii.sget(cls, "VER_9_6_8")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_6_9")
    def VER_9_6_9(cls) -> "AuroraPostgresEngineVersion":
        """Version "9.6.9"."""
        return jsii.sget(cls, "VER_9_6_9")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="auroraPostgresFullVersion")
    def aurora_postgres_full_version(self) -> builtins.str:
        """The full version string, for example, "9.6.25.1"."""
        return jsii.get(self, "auroraPostgresFullVersion")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="auroraPostgresMajorVersion")
    def aurora_postgres_major_version(self) -> builtins.str:
        """The major version of the engine, for example, "9.6"."""
        return jsii.get(self, "auroraPostgresMajorVersion")


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.BackupProps",
    jsii_struct_bases=[],
    name_mapping={"retention": "retention", "preferred_window": "preferredWindow"},
)
class BackupProps:
    def __init__(
        self,
        *,
        retention: aws_cdk.core.Duration,
        preferred_window: typing.Optional[builtins.str] = None,
    ) -> None:
        """Backup configuration for RDS databases.

        :param retention: How many days to retain the backup.
        :param preferred_window: A daily time range in 24-hours UTC format in which backups preferably execute. Must be at least 30 minutes long. Example: '01:00-02:00' Default: - a 30-minute window selected at random from an 8-hour block of time for each AWS Region. To see the time blocks available, see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow

        :default:

        - The retention period for automated backups is 1 day.
        The preferred backup window will be a 30-minute window selected at random
        from an 8-hour block of time for each AWS Region.

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow
        """
        self._values: typing.Dict[str, typing.Any] = {
            "retention": retention,
        }
        if preferred_window is not None:
            self._values["preferred_window"] = preferred_window

    @builtins.property
    def retention(self) -> aws_cdk.core.Duration:
        """How many days to retain the backup."""
        result = self._values.get("retention")
        assert result is not None, "Required property 'retention' is missing"
        return result

    @builtins.property
    def preferred_window(self) -> typing.Optional[builtins.str]:
        """A daily time range in 24-hours UTC format in which backups preferably execute.

        Must be at least 30 minutes long.

        Example: '01:00-02:00'

        :default:

        - a 30-minute window selected at random from an 8-hour block of
        time for each AWS Region. To see the time blocks available, see
        https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow
        """
        result = self._values.get("preferred_window")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BackupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnDBCluster(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.CfnDBCluster",
):
    """A CloudFormation ``AWS::RDS::DBCluster``.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html
    :cloudformationResource: AWS::RDS::DBCluster
    """

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        engine: builtins.str,
        associated_roles: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnDBCluster.DBClusterRoleProperty"]]]] = None,
        availability_zones: typing.Optional[typing.List[builtins.str]] = None,
        backtrack_window: typing.Optional[jsii.Number] = None,
        backup_retention_period: typing.Optional[jsii.Number] = None,
        database_name: typing.Optional[builtins.str] = None,
        db_cluster_identifier: typing.Optional[builtins.str] = None,
        db_cluster_parameter_group_name: typing.Optional[builtins.str] = None,
        db_subnet_group_name: typing.Optional[builtins.str] = None,
        deletion_protection: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        enable_cloudwatch_logs_exports: typing.Optional[typing.List[builtins.str]] = None,
        enable_http_endpoint: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        enable_iam_database_authentication: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        engine_mode: typing.Optional[builtins.str] = None,
        engine_version: typing.Optional[builtins.str] = None,
        global_cluster_identifier: typing.Optional[builtins.str] = None,
        kms_key_id: typing.Optional[builtins.str] = None,
        master_username: typing.Optional[builtins.str] = None,
        master_user_password: typing.Optional[builtins.str] = None,
        port: typing.Optional[jsii.Number] = None,
        preferred_backup_window: typing.Optional[builtins.str] = None,
        preferred_maintenance_window: typing.Optional[builtins.str] = None,
        replication_source_identifier: typing.Optional[builtins.str] = None,
        restore_type: typing.Optional[builtins.str] = None,
        scaling_configuration: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnDBCluster.ScalingConfigurationProperty"]] = None,
        snapshot_identifier: typing.Optional[builtins.str] = None,
        source_db_cluster_identifier: typing.Optional[builtins.str] = None,
        source_region: typing.Optional[builtins.str] = None,
        storage_encrypted: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        tags: typing.Optional[typing.List[aws_cdk.core.CfnTag]] = None,
        use_latest_restorable_time: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        vpc_security_group_ids: typing.Optional[typing.List[builtins.str]] = None,
    ) -> None:
        """Create a new ``AWS::RDS::DBCluster``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param engine: ``AWS::RDS::DBCluster.Engine``.
        :param associated_roles: ``AWS::RDS::DBCluster.AssociatedRoles``.
        :param availability_zones: ``AWS::RDS::DBCluster.AvailabilityZones``.
        :param backtrack_window: ``AWS::RDS::DBCluster.BacktrackWindow``.
        :param backup_retention_period: ``AWS::RDS::DBCluster.BackupRetentionPeriod``.
        :param database_name: ``AWS::RDS::DBCluster.DatabaseName``.
        :param db_cluster_identifier: ``AWS::RDS::DBCluster.DBClusterIdentifier``.
        :param db_cluster_parameter_group_name: ``AWS::RDS::DBCluster.DBClusterParameterGroupName``.
        :param db_subnet_group_name: ``AWS::RDS::DBCluster.DBSubnetGroupName``.
        :param deletion_protection: ``AWS::RDS::DBCluster.DeletionProtection``.
        :param enable_cloudwatch_logs_exports: ``AWS::RDS::DBCluster.EnableCloudwatchLogsExports``.
        :param enable_http_endpoint: ``AWS::RDS::DBCluster.EnableHttpEndpoint``.
        :param enable_iam_database_authentication: ``AWS::RDS::DBCluster.EnableIAMDatabaseAuthentication``.
        :param engine_mode: ``AWS::RDS::DBCluster.EngineMode``.
        :param engine_version: ``AWS::RDS::DBCluster.EngineVersion``.
        :param global_cluster_identifier: ``AWS::RDS::DBCluster.GlobalClusterIdentifier``.
        :param kms_key_id: ``AWS::RDS::DBCluster.KmsKeyId``.
        :param master_username: ``AWS::RDS::DBCluster.MasterUsername``.
        :param master_user_password: ``AWS::RDS::DBCluster.MasterUserPassword``.
        :param port: ``AWS::RDS::DBCluster.Port``.
        :param preferred_backup_window: ``AWS::RDS::DBCluster.PreferredBackupWindow``.
        :param preferred_maintenance_window: ``AWS::RDS::DBCluster.PreferredMaintenanceWindow``.
        :param replication_source_identifier: ``AWS::RDS::DBCluster.ReplicationSourceIdentifier``.
        :param restore_type: ``AWS::RDS::DBCluster.RestoreType``.
        :param scaling_configuration: ``AWS::RDS::DBCluster.ScalingConfiguration``.
        :param snapshot_identifier: ``AWS::RDS::DBCluster.SnapshotIdentifier``.
        :param source_db_cluster_identifier: ``AWS::RDS::DBCluster.SourceDBClusterIdentifier``.
        :param source_region: ``AWS::RDS::DBCluster.SourceRegion``.
        :param storage_encrypted: ``AWS::RDS::DBCluster.StorageEncrypted``.
        :param tags: ``AWS::RDS::DBCluster.Tags``.
        :param use_latest_restorable_time: ``AWS::RDS::DBCluster.UseLatestRestorableTime``.
        :param vpc_security_group_ids: ``AWS::RDS::DBCluster.VpcSecurityGroupIds``.
        """
        props = CfnDBClusterProps(
            engine=engine,
            associated_roles=associated_roles,
            availability_zones=availability_zones,
            backtrack_window=backtrack_window,
            backup_retention_period=backup_retention_period,
            database_name=database_name,
            db_cluster_identifier=db_cluster_identifier,
            db_cluster_parameter_group_name=db_cluster_parameter_group_name,
            db_subnet_group_name=db_subnet_group_name,
            deletion_protection=deletion_protection,
            enable_cloudwatch_logs_exports=enable_cloudwatch_logs_exports,
            enable_http_endpoint=enable_http_endpoint,
            enable_iam_database_authentication=enable_iam_database_authentication,
            engine_mode=engine_mode,
            engine_version=engine_version,
            global_cluster_identifier=global_cluster_identifier,
            kms_key_id=kms_key_id,
            master_username=master_username,
            master_user_password=master_user_password,
            port=port,
            preferred_backup_window=preferred_backup_window,
            preferred_maintenance_window=preferred_maintenance_window,
            replication_source_identifier=replication_source_identifier,
            restore_type=restore_type,
            scaling_configuration=scaling_configuration,
            snapshot_identifier=snapshot_identifier,
            source_db_cluster_identifier=source_db_cluster_identifier,
            source_region=source_region,
            storage_encrypted=storage_encrypted,
            tags=tags,
            use_latest_restorable_time=use_latest_restorable_time,
            vpc_security_group_ids=vpc_security_group_ids,
        )

        jsii.create(CfnDBCluster, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        """(experimental) Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.

        :stability: experimental
        """
        return jsii.invoke(self, "inspect", [inspector])

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        """
        :param props: -
        """
        return jsii.invoke(self, "renderProperties", [props])

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        """The CloudFormation resource type name for this resource class."""
        return jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="attrEndpointAddress")
    def attr_endpoint_address(self) -> builtins.str:
        """
        :cloudformationAttribute: Endpoint.Address
        """
        return jsii.get(self, "attrEndpointAddress")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="attrEndpointPort")
    def attr_endpoint_port(self) -> builtins.str:
        """
        :cloudformationAttribute: Endpoint.Port
        """
        return jsii.get(self, "attrEndpointPort")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="attrReadEndpointAddress")
    def attr_read_endpoint_address(self) -> builtins.str:
        """
        :cloudformationAttribute: ReadEndpoint.Address
        """
        return jsii.get(self, "attrReadEndpointAddress")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return jsii.get(self, "cfnProperties")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        """``AWS::RDS::DBCluster.Tags``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-tags
        """
        return jsii.get(self, "tags")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="engine")
    def engine(self) -> builtins.str:
        """``AWS::RDS::DBCluster.Engine``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-engine
        """
        return jsii.get(self, "engine")

    @engine.setter # type: ignore
    def engine(self, value: builtins.str) -> None:
        jsii.set(self, "engine", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="associatedRoles")
    def associated_roles(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnDBCluster.DBClusterRoleProperty"]]]]:
        """``AWS::RDS::DBCluster.AssociatedRoles``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-associatedroles
        """
        return jsii.get(self, "associatedRoles")

    @associated_roles.setter # type: ignore
    def associated_roles(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnDBCluster.DBClusterRoleProperty"]]]],
    ) -> None:
        jsii.set(self, "associatedRoles", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="availabilityZones")
    def availability_zones(self) -> typing.Optional[typing.List[builtins.str]]:
        """``AWS::RDS::DBCluster.AvailabilityZones``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-availabilityzones
        """
        return jsii.get(self, "availabilityZones")

    @availability_zones.setter # type: ignore
    def availability_zones(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        jsii.set(self, "availabilityZones", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="backtrackWindow")
    def backtrack_window(self) -> typing.Optional[jsii.Number]:
        """``AWS::RDS::DBCluster.BacktrackWindow``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-backtrackwindow
        """
        return jsii.get(self, "backtrackWindow")

    @backtrack_window.setter # type: ignore
    def backtrack_window(self, value: typing.Optional[jsii.Number]) -> None:
        jsii.set(self, "backtrackWindow", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="backupRetentionPeriod")
    def backup_retention_period(self) -> typing.Optional[jsii.Number]:
        """``AWS::RDS::DBCluster.BackupRetentionPeriod``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-backuprententionperiod
        """
        return jsii.get(self, "backupRetentionPeriod")

    @backup_retention_period.setter # type: ignore
    def backup_retention_period(self, value: typing.Optional[jsii.Number]) -> None:
        jsii.set(self, "backupRetentionPeriod", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="databaseName")
    def database_name(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.DatabaseName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-databasename
        """
        return jsii.get(self, "databaseName")

    @database_name.setter # type: ignore
    def database_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "databaseName", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="dbClusterIdentifier")
    def db_cluster_identifier(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.DBClusterIdentifier``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-dbclusteridentifier
        """
        return jsii.get(self, "dbClusterIdentifier")

    @db_cluster_identifier.setter # type: ignore
    def db_cluster_identifier(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "dbClusterIdentifier", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="dbClusterParameterGroupName")
    def db_cluster_parameter_group_name(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.DBClusterParameterGroupName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-dbclusterparametergroupname
        """
        return jsii.get(self, "dbClusterParameterGroupName")

    @db_cluster_parameter_group_name.setter # type: ignore
    def db_cluster_parameter_group_name(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        jsii.set(self, "dbClusterParameterGroupName", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="dbSubnetGroupName")
    def db_subnet_group_name(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.DBSubnetGroupName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-dbsubnetgroupname
        """
        return jsii.get(self, "dbSubnetGroupName")

    @db_subnet_group_name.setter # type: ignore
    def db_subnet_group_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "dbSubnetGroupName", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="deletionProtection")
    def deletion_protection(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBCluster.DeletionProtection``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-deletionprotection
        """
        return jsii.get(self, "deletionProtection")

    @deletion_protection.setter # type: ignore
    def deletion_protection(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "deletionProtection", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="enableCloudwatchLogsExports")
    def enable_cloudwatch_logs_exports(
        self,
    ) -> typing.Optional[typing.List[builtins.str]]:
        """``AWS::RDS::DBCluster.EnableCloudwatchLogsExports``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-enablecloudwatchlogsexports
        """
        return jsii.get(self, "enableCloudwatchLogsExports")

    @enable_cloudwatch_logs_exports.setter # type: ignore
    def enable_cloudwatch_logs_exports(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        jsii.set(self, "enableCloudwatchLogsExports", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="enableHttpEndpoint")
    def enable_http_endpoint(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBCluster.EnableHttpEndpoint``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-enablehttpendpoint
        """
        return jsii.get(self, "enableHttpEndpoint")

    @enable_http_endpoint.setter # type: ignore
    def enable_http_endpoint(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "enableHttpEndpoint", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="enableIamDatabaseAuthentication")
    def enable_iam_database_authentication(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBCluster.EnableIAMDatabaseAuthentication``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-enableiamdatabaseauthentication
        """
        return jsii.get(self, "enableIamDatabaseAuthentication")

    @enable_iam_database_authentication.setter # type: ignore
    def enable_iam_database_authentication(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "enableIamDatabaseAuthentication", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="engineMode")
    def engine_mode(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.EngineMode``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-enginemode
        """
        return jsii.get(self, "engineMode")

    @engine_mode.setter # type: ignore
    def engine_mode(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "engineMode", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="engineVersion")
    def engine_version(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.EngineVersion``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-engineversion
        """
        return jsii.get(self, "engineVersion")

    @engine_version.setter # type: ignore
    def engine_version(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "engineVersion", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="globalClusterIdentifier")
    def global_cluster_identifier(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.GlobalClusterIdentifier``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-globalclusteridentifier
        """
        return jsii.get(self, "globalClusterIdentifier")

    @global_cluster_identifier.setter # type: ignore
    def global_cluster_identifier(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "globalClusterIdentifier", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="kmsKeyId")
    def kms_key_id(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.KmsKeyId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-kmskeyid
        """
        return jsii.get(self, "kmsKeyId")

    @kms_key_id.setter # type: ignore
    def kms_key_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "kmsKeyId", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="masterUsername")
    def master_username(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.MasterUsername``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-masterusername
        """
        return jsii.get(self, "masterUsername")

    @master_username.setter # type: ignore
    def master_username(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "masterUsername", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="masterUserPassword")
    def master_user_password(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.MasterUserPassword``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-masteruserpassword
        """
        return jsii.get(self, "masterUserPassword")

    @master_user_password.setter # type: ignore
    def master_user_password(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "masterUserPassword", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="port")
    def port(self) -> typing.Optional[jsii.Number]:
        """``AWS::RDS::DBCluster.Port``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-port
        """
        return jsii.get(self, "port")

    @port.setter # type: ignore
    def port(self, value: typing.Optional[jsii.Number]) -> None:
        jsii.set(self, "port", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="preferredBackupWindow")
    def preferred_backup_window(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.PreferredBackupWindow``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-preferredbackupwindow
        """
        return jsii.get(self, "preferredBackupWindow")

    @preferred_backup_window.setter # type: ignore
    def preferred_backup_window(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "preferredBackupWindow", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="preferredMaintenanceWindow")
    def preferred_maintenance_window(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.PreferredMaintenanceWindow``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-preferredmaintenancewindow
        """
        return jsii.get(self, "preferredMaintenanceWindow")

    @preferred_maintenance_window.setter # type: ignore
    def preferred_maintenance_window(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        jsii.set(self, "preferredMaintenanceWindow", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="replicationSourceIdentifier")
    def replication_source_identifier(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.ReplicationSourceIdentifier``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-replicationsourceidentifier
        """
        return jsii.get(self, "replicationSourceIdentifier")

    @replication_source_identifier.setter # type: ignore
    def replication_source_identifier(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        jsii.set(self, "replicationSourceIdentifier", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="restoreType")
    def restore_type(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.RestoreType``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-restoretype
        """
        return jsii.get(self, "restoreType")

    @restore_type.setter # type: ignore
    def restore_type(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "restoreType", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="scalingConfiguration")
    def scaling_configuration(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnDBCluster.ScalingConfigurationProperty"]]:
        """``AWS::RDS::DBCluster.ScalingConfiguration``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-scalingconfiguration
        """
        return jsii.get(self, "scalingConfiguration")

    @scaling_configuration.setter # type: ignore
    def scaling_configuration(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnDBCluster.ScalingConfigurationProperty"]],
    ) -> None:
        jsii.set(self, "scalingConfiguration", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="snapshotIdentifier")
    def snapshot_identifier(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.SnapshotIdentifier``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-snapshotidentifier
        """
        return jsii.get(self, "snapshotIdentifier")

    @snapshot_identifier.setter # type: ignore
    def snapshot_identifier(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "snapshotIdentifier", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="sourceDbClusterIdentifier")
    def source_db_cluster_identifier(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.SourceDBClusterIdentifier``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-sourcedbclusteridentifier
        """
        return jsii.get(self, "sourceDbClusterIdentifier")

    @source_db_cluster_identifier.setter # type: ignore
    def source_db_cluster_identifier(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        jsii.set(self, "sourceDbClusterIdentifier", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="sourceRegion")
    def source_region(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.SourceRegion``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-sourceregion
        """
        return jsii.get(self, "sourceRegion")

    @source_region.setter # type: ignore
    def source_region(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "sourceRegion", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="storageEncrypted")
    def storage_encrypted(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBCluster.StorageEncrypted``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-storageencrypted
        """
        return jsii.get(self, "storageEncrypted")

    @storage_encrypted.setter # type: ignore
    def storage_encrypted(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "storageEncrypted", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="useLatestRestorableTime")
    def use_latest_restorable_time(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBCluster.UseLatestRestorableTime``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-uselatestrestorabletime
        """
        return jsii.get(self, "useLatestRestorableTime")

    @use_latest_restorable_time.setter # type: ignore
    def use_latest_restorable_time(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "useLatestRestorableTime", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="vpcSecurityGroupIds")
    def vpc_security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        """``AWS::RDS::DBCluster.VpcSecurityGroupIds``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-vpcsecuritygroupids
        """
        return jsii.get(self, "vpcSecurityGroupIds")

    @vpc_security_group_ids.setter # type: ignore
    def vpc_security_group_ids(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        jsii.set(self, "vpcSecurityGroupIds", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-rds.CfnDBCluster.DBClusterRoleProperty",
        jsii_struct_bases=[],
        name_mapping={"role_arn": "roleArn", "feature_name": "featureName"},
    )
    class DBClusterRoleProperty:
        def __init__(
            self,
            *,
            role_arn: builtins.str,
            feature_name: typing.Optional[builtins.str] = None,
        ) -> None:
            """
            :param role_arn: ``CfnDBCluster.DBClusterRoleProperty.RoleArn``.
            :param feature_name: ``CfnDBCluster.DBClusterRoleProperty.FeatureName``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-dbclusterrole.html
            """
            self._values: typing.Dict[str, typing.Any] = {
                "role_arn": role_arn,
            }
            if feature_name is not None:
                self._values["feature_name"] = feature_name

        @builtins.property
        def role_arn(self) -> builtins.str:
            """``CfnDBCluster.DBClusterRoleProperty.RoleArn``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-dbclusterrole.html#cfn-rds-dbcluster-dbclusterrole-rolearn
            """
            result = self._values.get("role_arn")
            assert result is not None, "Required property 'role_arn' is missing"
            return result

        @builtins.property
        def feature_name(self) -> typing.Optional[builtins.str]:
            """``CfnDBCluster.DBClusterRoleProperty.FeatureName``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-dbclusterrole.html#cfn-rds-dbcluster-dbclusterrole-featurename
            """
            result = self._values.get("feature_name")
            return result

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DBClusterRoleProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-rds.CfnDBCluster.ScalingConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "auto_pause": "autoPause",
            "max_capacity": "maxCapacity",
            "min_capacity": "minCapacity",
            "seconds_until_auto_pause": "secondsUntilAutoPause",
        },
    )
    class ScalingConfigurationProperty:
        def __init__(
            self,
            *,
            auto_pause: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
            max_capacity: typing.Optional[jsii.Number] = None,
            min_capacity: typing.Optional[jsii.Number] = None,
            seconds_until_auto_pause: typing.Optional[jsii.Number] = None,
        ) -> None:
            """
            :param auto_pause: ``CfnDBCluster.ScalingConfigurationProperty.AutoPause``.
            :param max_capacity: ``CfnDBCluster.ScalingConfigurationProperty.MaxCapacity``.
            :param min_capacity: ``CfnDBCluster.ScalingConfigurationProperty.MinCapacity``.
            :param seconds_until_auto_pause: ``CfnDBCluster.ScalingConfigurationProperty.SecondsUntilAutoPause``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-scalingconfiguration.html
            """
            self._values: typing.Dict[str, typing.Any] = {}
            if auto_pause is not None:
                self._values["auto_pause"] = auto_pause
            if max_capacity is not None:
                self._values["max_capacity"] = max_capacity
            if min_capacity is not None:
                self._values["min_capacity"] = min_capacity
            if seconds_until_auto_pause is not None:
                self._values["seconds_until_auto_pause"] = seconds_until_auto_pause

        @builtins.property
        def auto_pause(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            """``CfnDBCluster.ScalingConfigurationProperty.AutoPause``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-scalingconfiguration.html#cfn-rds-dbcluster-scalingconfiguration-autopause
            """
            result = self._values.get("auto_pause")
            return result

        @builtins.property
        def max_capacity(self) -> typing.Optional[jsii.Number]:
            """``CfnDBCluster.ScalingConfigurationProperty.MaxCapacity``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-scalingconfiguration.html#cfn-rds-dbcluster-scalingconfiguration-maxcapacity
            """
            result = self._values.get("max_capacity")
            return result

        @builtins.property
        def min_capacity(self) -> typing.Optional[jsii.Number]:
            """``CfnDBCluster.ScalingConfigurationProperty.MinCapacity``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-scalingconfiguration.html#cfn-rds-dbcluster-scalingconfiguration-mincapacity
            """
            result = self._values.get("min_capacity")
            return result

        @builtins.property
        def seconds_until_auto_pause(self) -> typing.Optional[jsii.Number]:
            """``CfnDBCluster.ScalingConfigurationProperty.SecondsUntilAutoPause``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-scalingconfiguration.html#cfn-rds-dbcluster-scalingconfiguration-secondsuntilautopause
            """
            result = self._values.get("seconds_until_auto_pause")
            return result

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ScalingConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnDBClusterParameterGroup(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.CfnDBClusterParameterGroup",
):
    """A CloudFormation ``AWS::RDS::DBClusterParameterGroup``.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html
    :cloudformationResource: AWS::RDS::DBClusterParameterGroup
    """

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        description: builtins.str,
        family: builtins.str,
        parameters: typing.Any,
        tags: typing.Optional[typing.List[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        """Create a new ``AWS::RDS::DBClusterParameterGroup``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param description: ``AWS::RDS::DBClusterParameterGroup.Description``.
        :param family: ``AWS::RDS::DBClusterParameterGroup.Family``.
        :param parameters: ``AWS::RDS::DBClusterParameterGroup.Parameters``.
        :param tags: ``AWS::RDS::DBClusterParameterGroup.Tags``.
        """
        props = CfnDBClusterParameterGroupProps(
            description=description, family=family, parameters=parameters, tags=tags
        )

        jsii.create(CfnDBClusterParameterGroup, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        """(experimental) Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.

        :stability: experimental
        """
        return jsii.invoke(self, "inspect", [inspector])

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        """
        :param props: -
        """
        return jsii.invoke(self, "renderProperties", [props])

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        """The CloudFormation resource type name for this resource class."""
        return jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return jsii.get(self, "cfnProperties")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        """``AWS::RDS::DBClusterParameterGroup.Tags``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html#cfn-rds-dbclusterparametergroup-tags
        """
        return jsii.get(self, "tags")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="description")
    def description(self) -> builtins.str:
        """``AWS::RDS::DBClusterParameterGroup.Description``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html#cfn-rds-dbclusterparametergroup-description
        """
        return jsii.get(self, "description")

    @description.setter # type: ignore
    def description(self, value: builtins.str) -> None:
        jsii.set(self, "description", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="family")
    def family(self) -> builtins.str:
        """``AWS::RDS::DBClusterParameterGroup.Family``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html#cfn-rds-dbclusterparametergroup-family
        """
        return jsii.get(self, "family")

    @family.setter # type: ignore
    def family(self, value: builtins.str) -> None:
        jsii.set(self, "family", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="parameters")
    def parameters(self) -> typing.Any:
        """``AWS::RDS::DBClusterParameterGroup.Parameters``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html#cfn-rds-dbclusterparametergroup-parameters
        """
        return jsii.get(self, "parameters")

    @parameters.setter # type: ignore
    def parameters(self, value: typing.Any) -> None:
        jsii.set(self, "parameters", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.CfnDBClusterParameterGroupProps",
    jsii_struct_bases=[],
    name_mapping={
        "description": "description",
        "family": "family",
        "parameters": "parameters",
        "tags": "tags",
    },
)
class CfnDBClusterParameterGroupProps:
    def __init__(
        self,
        *,
        description: builtins.str,
        family: builtins.str,
        parameters: typing.Any,
        tags: typing.Optional[typing.List[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        """Properties for defining a ``AWS::RDS::DBClusterParameterGroup``.

        :param description: ``AWS::RDS::DBClusterParameterGroup.Description``.
        :param family: ``AWS::RDS::DBClusterParameterGroup.Family``.
        :param parameters: ``AWS::RDS::DBClusterParameterGroup.Parameters``.
        :param tags: ``AWS::RDS::DBClusterParameterGroup.Tags``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html
        """
        self._values: typing.Dict[str, typing.Any] = {
            "description": description,
            "family": family,
            "parameters": parameters,
        }
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def description(self) -> builtins.str:
        """``AWS::RDS::DBClusterParameterGroup.Description``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html#cfn-rds-dbclusterparametergroup-description
        """
        result = self._values.get("description")
        assert result is not None, "Required property 'description' is missing"
        return result

    @builtins.property
    def family(self) -> builtins.str:
        """``AWS::RDS::DBClusterParameterGroup.Family``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html#cfn-rds-dbclusterparametergroup-family
        """
        result = self._values.get("family")
        assert result is not None, "Required property 'family' is missing"
        return result

    @builtins.property
    def parameters(self) -> typing.Any:
        """``AWS::RDS::DBClusterParameterGroup.Parameters``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html#cfn-rds-dbclusterparametergroup-parameters
        """
        result = self._values.get("parameters")
        assert result is not None, "Required property 'parameters' is missing"
        return result

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        """``AWS::RDS::DBClusterParameterGroup.Tags``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html#cfn-rds-dbclusterparametergroup-tags
        """
        result = self._values.get("tags")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDBClusterParameterGroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.CfnDBClusterProps",
    jsii_struct_bases=[],
    name_mapping={
        "engine": "engine",
        "associated_roles": "associatedRoles",
        "availability_zones": "availabilityZones",
        "backtrack_window": "backtrackWindow",
        "backup_retention_period": "backupRetentionPeriod",
        "database_name": "databaseName",
        "db_cluster_identifier": "dbClusterIdentifier",
        "db_cluster_parameter_group_name": "dbClusterParameterGroupName",
        "db_subnet_group_name": "dbSubnetGroupName",
        "deletion_protection": "deletionProtection",
        "enable_cloudwatch_logs_exports": "enableCloudwatchLogsExports",
        "enable_http_endpoint": "enableHttpEndpoint",
        "enable_iam_database_authentication": "enableIamDatabaseAuthentication",
        "engine_mode": "engineMode",
        "engine_version": "engineVersion",
        "global_cluster_identifier": "globalClusterIdentifier",
        "kms_key_id": "kmsKeyId",
        "master_username": "masterUsername",
        "master_user_password": "masterUserPassword",
        "port": "port",
        "preferred_backup_window": "preferredBackupWindow",
        "preferred_maintenance_window": "preferredMaintenanceWindow",
        "replication_source_identifier": "replicationSourceIdentifier",
        "restore_type": "restoreType",
        "scaling_configuration": "scalingConfiguration",
        "snapshot_identifier": "snapshotIdentifier",
        "source_db_cluster_identifier": "sourceDbClusterIdentifier",
        "source_region": "sourceRegion",
        "storage_encrypted": "storageEncrypted",
        "tags": "tags",
        "use_latest_restorable_time": "useLatestRestorableTime",
        "vpc_security_group_ids": "vpcSecurityGroupIds",
    },
)
class CfnDBClusterProps:
    def __init__(
        self,
        *,
        engine: builtins.str,
        associated_roles: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnDBCluster.DBClusterRoleProperty]]]] = None,
        availability_zones: typing.Optional[typing.List[builtins.str]] = None,
        backtrack_window: typing.Optional[jsii.Number] = None,
        backup_retention_period: typing.Optional[jsii.Number] = None,
        database_name: typing.Optional[builtins.str] = None,
        db_cluster_identifier: typing.Optional[builtins.str] = None,
        db_cluster_parameter_group_name: typing.Optional[builtins.str] = None,
        db_subnet_group_name: typing.Optional[builtins.str] = None,
        deletion_protection: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        enable_cloudwatch_logs_exports: typing.Optional[typing.List[builtins.str]] = None,
        enable_http_endpoint: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        enable_iam_database_authentication: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        engine_mode: typing.Optional[builtins.str] = None,
        engine_version: typing.Optional[builtins.str] = None,
        global_cluster_identifier: typing.Optional[builtins.str] = None,
        kms_key_id: typing.Optional[builtins.str] = None,
        master_username: typing.Optional[builtins.str] = None,
        master_user_password: typing.Optional[builtins.str] = None,
        port: typing.Optional[jsii.Number] = None,
        preferred_backup_window: typing.Optional[builtins.str] = None,
        preferred_maintenance_window: typing.Optional[builtins.str] = None,
        replication_source_identifier: typing.Optional[builtins.str] = None,
        restore_type: typing.Optional[builtins.str] = None,
        scaling_configuration: typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnDBCluster.ScalingConfigurationProperty]] = None,
        snapshot_identifier: typing.Optional[builtins.str] = None,
        source_db_cluster_identifier: typing.Optional[builtins.str] = None,
        source_region: typing.Optional[builtins.str] = None,
        storage_encrypted: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        tags: typing.Optional[typing.List[aws_cdk.core.CfnTag]] = None,
        use_latest_restorable_time: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        vpc_security_group_ids: typing.Optional[typing.List[builtins.str]] = None,
    ) -> None:
        """Properties for defining a ``AWS::RDS::DBCluster``.

        :param engine: ``AWS::RDS::DBCluster.Engine``.
        :param associated_roles: ``AWS::RDS::DBCluster.AssociatedRoles``.
        :param availability_zones: ``AWS::RDS::DBCluster.AvailabilityZones``.
        :param backtrack_window: ``AWS::RDS::DBCluster.BacktrackWindow``.
        :param backup_retention_period: ``AWS::RDS::DBCluster.BackupRetentionPeriod``.
        :param database_name: ``AWS::RDS::DBCluster.DatabaseName``.
        :param db_cluster_identifier: ``AWS::RDS::DBCluster.DBClusterIdentifier``.
        :param db_cluster_parameter_group_name: ``AWS::RDS::DBCluster.DBClusterParameterGroupName``.
        :param db_subnet_group_name: ``AWS::RDS::DBCluster.DBSubnetGroupName``.
        :param deletion_protection: ``AWS::RDS::DBCluster.DeletionProtection``.
        :param enable_cloudwatch_logs_exports: ``AWS::RDS::DBCluster.EnableCloudwatchLogsExports``.
        :param enable_http_endpoint: ``AWS::RDS::DBCluster.EnableHttpEndpoint``.
        :param enable_iam_database_authentication: ``AWS::RDS::DBCluster.EnableIAMDatabaseAuthentication``.
        :param engine_mode: ``AWS::RDS::DBCluster.EngineMode``.
        :param engine_version: ``AWS::RDS::DBCluster.EngineVersion``.
        :param global_cluster_identifier: ``AWS::RDS::DBCluster.GlobalClusterIdentifier``.
        :param kms_key_id: ``AWS::RDS::DBCluster.KmsKeyId``.
        :param master_username: ``AWS::RDS::DBCluster.MasterUsername``.
        :param master_user_password: ``AWS::RDS::DBCluster.MasterUserPassword``.
        :param port: ``AWS::RDS::DBCluster.Port``.
        :param preferred_backup_window: ``AWS::RDS::DBCluster.PreferredBackupWindow``.
        :param preferred_maintenance_window: ``AWS::RDS::DBCluster.PreferredMaintenanceWindow``.
        :param replication_source_identifier: ``AWS::RDS::DBCluster.ReplicationSourceIdentifier``.
        :param restore_type: ``AWS::RDS::DBCluster.RestoreType``.
        :param scaling_configuration: ``AWS::RDS::DBCluster.ScalingConfiguration``.
        :param snapshot_identifier: ``AWS::RDS::DBCluster.SnapshotIdentifier``.
        :param source_db_cluster_identifier: ``AWS::RDS::DBCluster.SourceDBClusterIdentifier``.
        :param source_region: ``AWS::RDS::DBCluster.SourceRegion``.
        :param storage_encrypted: ``AWS::RDS::DBCluster.StorageEncrypted``.
        :param tags: ``AWS::RDS::DBCluster.Tags``.
        :param use_latest_restorable_time: ``AWS::RDS::DBCluster.UseLatestRestorableTime``.
        :param vpc_security_group_ids: ``AWS::RDS::DBCluster.VpcSecurityGroupIds``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html
        """
        self._values: typing.Dict[str, typing.Any] = {
            "engine": engine,
        }
        if associated_roles is not None:
            self._values["associated_roles"] = associated_roles
        if availability_zones is not None:
            self._values["availability_zones"] = availability_zones
        if backtrack_window is not None:
            self._values["backtrack_window"] = backtrack_window
        if backup_retention_period is not None:
            self._values["backup_retention_period"] = backup_retention_period
        if database_name is not None:
            self._values["database_name"] = database_name
        if db_cluster_identifier is not None:
            self._values["db_cluster_identifier"] = db_cluster_identifier
        if db_cluster_parameter_group_name is not None:
            self._values["db_cluster_parameter_group_name"] = db_cluster_parameter_group_name
        if db_subnet_group_name is not None:
            self._values["db_subnet_group_name"] = db_subnet_group_name
        if deletion_protection is not None:
            self._values["deletion_protection"] = deletion_protection
        if enable_cloudwatch_logs_exports is not None:
            self._values["enable_cloudwatch_logs_exports"] = enable_cloudwatch_logs_exports
        if enable_http_endpoint is not None:
            self._values["enable_http_endpoint"] = enable_http_endpoint
        if enable_iam_database_authentication is not None:
            self._values["enable_iam_database_authentication"] = enable_iam_database_authentication
        if engine_mode is not None:
            self._values["engine_mode"] = engine_mode
        if engine_version is not None:
            self._values["engine_version"] = engine_version
        if global_cluster_identifier is not None:
            self._values["global_cluster_identifier"] = global_cluster_identifier
        if kms_key_id is not None:
            self._values["kms_key_id"] = kms_key_id
        if master_username is not None:
            self._values["master_username"] = master_username
        if master_user_password is not None:
            self._values["master_user_password"] = master_user_password
        if port is not None:
            self._values["port"] = port
        if preferred_backup_window is not None:
            self._values["preferred_backup_window"] = preferred_backup_window
        if preferred_maintenance_window is not None:
            self._values["preferred_maintenance_window"] = preferred_maintenance_window
        if replication_source_identifier is not None:
            self._values["replication_source_identifier"] = replication_source_identifier
        if restore_type is not None:
            self._values["restore_type"] = restore_type
        if scaling_configuration is not None:
            self._values["scaling_configuration"] = scaling_configuration
        if snapshot_identifier is not None:
            self._values["snapshot_identifier"] = snapshot_identifier
        if source_db_cluster_identifier is not None:
            self._values["source_db_cluster_identifier"] = source_db_cluster_identifier
        if source_region is not None:
            self._values["source_region"] = source_region
        if storage_encrypted is not None:
            self._values["storage_encrypted"] = storage_encrypted
        if tags is not None:
            self._values["tags"] = tags
        if use_latest_restorable_time is not None:
            self._values["use_latest_restorable_time"] = use_latest_restorable_time
        if vpc_security_group_ids is not None:
            self._values["vpc_security_group_ids"] = vpc_security_group_ids

    @builtins.property
    def engine(self) -> builtins.str:
        """``AWS::RDS::DBCluster.Engine``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-engine
        """
        result = self._values.get("engine")
        assert result is not None, "Required property 'engine' is missing"
        return result

    @builtins.property
    def associated_roles(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnDBCluster.DBClusterRoleProperty]]]]:
        """``AWS::RDS::DBCluster.AssociatedRoles``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-associatedroles
        """
        result = self._values.get("associated_roles")
        return result

    @builtins.property
    def availability_zones(self) -> typing.Optional[typing.List[builtins.str]]:
        """``AWS::RDS::DBCluster.AvailabilityZones``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-availabilityzones
        """
        result = self._values.get("availability_zones")
        return result

    @builtins.property
    def backtrack_window(self) -> typing.Optional[jsii.Number]:
        """``AWS::RDS::DBCluster.BacktrackWindow``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-backtrackwindow
        """
        result = self._values.get("backtrack_window")
        return result

    @builtins.property
    def backup_retention_period(self) -> typing.Optional[jsii.Number]:
        """``AWS::RDS::DBCluster.BackupRetentionPeriod``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-backuprententionperiod
        """
        result = self._values.get("backup_retention_period")
        return result

    @builtins.property
    def database_name(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.DatabaseName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-databasename
        """
        result = self._values.get("database_name")
        return result

    @builtins.property
    def db_cluster_identifier(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.DBClusterIdentifier``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-dbclusteridentifier
        """
        result = self._values.get("db_cluster_identifier")
        return result

    @builtins.property
    def db_cluster_parameter_group_name(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.DBClusterParameterGroupName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-dbclusterparametergroupname
        """
        result = self._values.get("db_cluster_parameter_group_name")
        return result

    @builtins.property
    def db_subnet_group_name(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.DBSubnetGroupName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-dbsubnetgroupname
        """
        result = self._values.get("db_subnet_group_name")
        return result

    @builtins.property
    def deletion_protection(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBCluster.DeletionProtection``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-deletionprotection
        """
        result = self._values.get("deletion_protection")
        return result

    @builtins.property
    def enable_cloudwatch_logs_exports(
        self,
    ) -> typing.Optional[typing.List[builtins.str]]:
        """``AWS::RDS::DBCluster.EnableCloudwatchLogsExports``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-enablecloudwatchlogsexports
        """
        result = self._values.get("enable_cloudwatch_logs_exports")
        return result

    @builtins.property
    def enable_http_endpoint(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBCluster.EnableHttpEndpoint``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-enablehttpendpoint
        """
        result = self._values.get("enable_http_endpoint")
        return result

    @builtins.property
    def enable_iam_database_authentication(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBCluster.EnableIAMDatabaseAuthentication``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-enableiamdatabaseauthentication
        """
        result = self._values.get("enable_iam_database_authentication")
        return result

    @builtins.property
    def engine_mode(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.EngineMode``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-enginemode
        """
        result = self._values.get("engine_mode")
        return result

    @builtins.property
    def engine_version(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.EngineVersion``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-engineversion
        """
        result = self._values.get("engine_version")
        return result

    @builtins.property
    def global_cluster_identifier(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.GlobalClusterIdentifier``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-globalclusteridentifier
        """
        result = self._values.get("global_cluster_identifier")
        return result

    @builtins.property
    def kms_key_id(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.KmsKeyId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-kmskeyid
        """
        result = self._values.get("kms_key_id")
        return result

    @builtins.property
    def master_username(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.MasterUsername``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-masterusername
        """
        result = self._values.get("master_username")
        return result

    @builtins.property
    def master_user_password(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.MasterUserPassword``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-masteruserpassword
        """
        result = self._values.get("master_user_password")
        return result

    @builtins.property
    def port(self) -> typing.Optional[jsii.Number]:
        """``AWS::RDS::DBCluster.Port``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-port
        """
        result = self._values.get("port")
        return result

    @builtins.property
    def preferred_backup_window(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.PreferredBackupWindow``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-preferredbackupwindow
        """
        result = self._values.get("preferred_backup_window")
        return result

    @builtins.property
    def preferred_maintenance_window(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.PreferredMaintenanceWindow``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-preferredmaintenancewindow
        """
        result = self._values.get("preferred_maintenance_window")
        return result

    @builtins.property
    def replication_source_identifier(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.ReplicationSourceIdentifier``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-replicationsourceidentifier
        """
        result = self._values.get("replication_source_identifier")
        return result

    @builtins.property
    def restore_type(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.RestoreType``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-restoretype
        """
        result = self._values.get("restore_type")
        return result

    @builtins.property
    def scaling_configuration(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnDBCluster.ScalingConfigurationProperty]]:
        """``AWS::RDS::DBCluster.ScalingConfiguration``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-scalingconfiguration
        """
        result = self._values.get("scaling_configuration")
        return result

    @builtins.property
    def snapshot_identifier(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.SnapshotIdentifier``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-snapshotidentifier
        """
        result = self._values.get("snapshot_identifier")
        return result

    @builtins.property
    def source_db_cluster_identifier(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.SourceDBClusterIdentifier``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-sourcedbclusteridentifier
        """
        result = self._values.get("source_db_cluster_identifier")
        return result

    @builtins.property
    def source_region(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBCluster.SourceRegion``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-sourceregion
        """
        result = self._values.get("source_region")
        return result

    @builtins.property
    def storage_encrypted(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBCluster.StorageEncrypted``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-storageencrypted
        """
        result = self._values.get("storage_encrypted")
        return result

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        """``AWS::RDS::DBCluster.Tags``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-tags
        """
        result = self._values.get("tags")
        return result

    @builtins.property
    def use_latest_restorable_time(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBCluster.UseLatestRestorableTime``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-uselatestrestorabletime
        """
        result = self._values.get("use_latest_restorable_time")
        return result

    @builtins.property
    def vpc_security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        """``AWS::RDS::DBCluster.VpcSecurityGroupIds``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-vpcsecuritygroupids
        """
        result = self._values.get("vpc_security_group_ids")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDBClusterProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnDBInstance(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.CfnDBInstance",
):
    """A CloudFormation ``AWS::RDS::DBInstance``.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html
    :cloudformationResource: AWS::RDS::DBInstance
    """

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        db_instance_class: builtins.str,
        allocated_storage: typing.Optional[builtins.str] = None,
        allow_major_version_upgrade: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        associated_roles: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnDBInstance.DBInstanceRoleProperty"]]]] = None,
        auto_minor_version_upgrade: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        availability_zone: typing.Optional[builtins.str] = None,
        backup_retention_period: typing.Optional[jsii.Number] = None,
        ca_certificate_identifier: typing.Optional[builtins.str] = None,
        character_set_name: typing.Optional[builtins.str] = None,
        copy_tags_to_snapshot: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        db_cluster_identifier: typing.Optional[builtins.str] = None,
        db_instance_identifier: typing.Optional[builtins.str] = None,
        db_name: typing.Optional[builtins.str] = None,
        db_parameter_group_name: typing.Optional[builtins.str] = None,
        db_security_groups: typing.Optional[typing.List[builtins.str]] = None,
        db_snapshot_identifier: typing.Optional[builtins.str] = None,
        db_subnet_group_name: typing.Optional[builtins.str] = None,
        delete_automated_backups: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        deletion_protection: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        domain: typing.Optional[builtins.str] = None,
        domain_iam_role_name: typing.Optional[builtins.str] = None,
        enable_cloudwatch_logs_exports: typing.Optional[typing.List[builtins.str]] = None,
        enable_iam_database_authentication: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        enable_performance_insights: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        engine: typing.Optional[builtins.str] = None,
        engine_version: typing.Optional[builtins.str] = None,
        iops: typing.Optional[jsii.Number] = None,
        kms_key_id: typing.Optional[builtins.str] = None,
        license_model: typing.Optional[builtins.str] = None,
        master_username: typing.Optional[builtins.str] = None,
        master_user_password: typing.Optional[builtins.str] = None,
        max_allocated_storage: typing.Optional[jsii.Number] = None,
        monitoring_interval: typing.Optional[jsii.Number] = None,
        monitoring_role_arn: typing.Optional[builtins.str] = None,
        multi_az: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        option_group_name: typing.Optional[builtins.str] = None,
        performance_insights_kms_key_id: typing.Optional[builtins.str] = None,
        performance_insights_retention_period: typing.Optional[jsii.Number] = None,
        port: typing.Optional[builtins.str] = None,
        preferred_backup_window: typing.Optional[builtins.str] = None,
        preferred_maintenance_window: typing.Optional[builtins.str] = None,
        processor_features: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnDBInstance.ProcessorFeatureProperty"]]]] = None,
        promotion_tier: typing.Optional[jsii.Number] = None,
        publicly_accessible: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        source_db_instance_identifier: typing.Optional[builtins.str] = None,
        source_region: typing.Optional[builtins.str] = None,
        storage_encrypted: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        storage_type: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.List[aws_cdk.core.CfnTag]] = None,
        timezone: typing.Optional[builtins.str] = None,
        use_default_processor_features: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        vpc_security_groups: typing.Optional[typing.List[builtins.str]] = None,
    ) -> None:
        """Create a new ``AWS::RDS::DBInstance``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param db_instance_class: ``AWS::RDS::DBInstance.DBInstanceClass``.
        :param allocated_storage: ``AWS::RDS::DBInstance.AllocatedStorage``.
        :param allow_major_version_upgrade: ``AWS::RDS::DBInstance.AllowMajorVersionUpgrade``.
        :param associated_roles: ``AWS::RDS::DBInstance.AssociatedRoles``.
        :param auto_minor_version_upgrade: ``AWS::RDS::DBInstance.AutoMinorVersionUpgrade``.
        :param availability_zone: ``AWS::RDS::DBInstance.AvailabilityZone``.
        :param backup_retention_period: ``AWS::RDS::DBInstance.BackupRetentionPeriod``.
        :param ca_certificate_identifier: ``AWS::RDS::DBInstance.CACertificateIdentifier``.
        :param character_set_name: ``AWS::RDS::DBInstance.CharacterSetName``.
        :param copy_tags_to_snapshot: ``AWS::RDS::DBInstance.CopyTagsToSnapshot``.
        :param db_cluster_identifier: ``AWS::RDS::DBInstance.DBClusterIdentifier``.
        :param db_instance_identifier: ``AWS::RDS::DBInstance.DBInstanceIdentifier``.
        :param db_name: ``AWS::RDS::DBInstance.DBName``.
        :param db_parameter_group_name: ``AWS::RDS::DBInstance.DBParameterGroupName``.
        :param db_security_groups: ``AWS::RDS::DBInstance.DBSecurityGroups``.
        :param db_snapshot_identifier: ``AWS::RDS::DBInstance.DBSnapshotIdentifier``.
        :param db_subnet_group_name: ``AWS::RDS::DBInstance.DBSubnetGroupName``.
        :param delete_automated_backups: ``AWS::RDS::DBInstance.DeleteAutomatedBackups``.
        :param deletion_protection: ``AWS::RDS::DBInstance.DeletionProtection``.
        :param domain: ``AWS::RDS::DBInstance.Domain``.
        :param domain_iam_role_name: ``AWS::RDS::DBInstance.DomainIAMRoleName``.
        :param enable_cloudwatch_logs_exports: ``AWS::RDS::DBInstance.EnableCloudwatchLogsExports``.
        :param enable_iam_database_authentication: ``AWS::RDS::DBInstance.EnableIAMDatabaseAuthentication``.
        :param enable_performance_insights: ``AWS::RDS::DBInstance.EnablePerformanceInsights``.
        :param engine: ``AWS::RDS::DBInstance.Engine``.
        :param engine_version: ``AWS::RDS::DBInstance.EngineVersion``.
        :param iops: ``AWS::RDS::DBInstance.Iops``.
        :param kms_key_id: ``AWS::RDS::DBInstance.KmsKeyId``.
        :param license_model: ``AWS::RDS::DBInstance.LicenseModel``.
        :param master_username: ``AWS::RDS::DBInstance.MasterUsername``.
        :param master_user_password: ``AWS::RDS::DBInstance.MasterUserPassword``.
        :param max_allocated_storage: ``AWS::RDS::DBInstance.MaxAllocatedStorage``.
        :param monitoring_interval: ``AWS::RDS::DBInstance.MonitoringInterval``.
        :param monitoring_role_arn: ``AWS::RDS::DBInstance.MonitoringRoleArn``.
        :param multi_az: ``AWS::RDS::DBInstance.MultiAZ``.
        :param option_group_name: ``AWS::RDS::DBInstance.OptionGroupName``.
        :param performance_insights_kms_key_id: ``AWS::RDS::DBInstance.PerformanceInsightsKMSKeyId``.
        :param performance_insights_retention_period: ``AWS::RDS::DBInstance.PerformanceInsightsRetentionPeriod``.
        :param port: ``AWS::RDS::DBInstance.Port``.
        :param preferred_backup_window: ``AWS::RDS::DBInstance.PreferredBackupWindow``.
        :param preferred_maintenance_window: ``AWS::RDS::DBInstance.PreferredMaintenanceWindow``.
        :param processor_features: ``AWS::RDS::DBInstance.ProcessorFeatures``.
        :param promotion_tier: ``AWS::RDS::DBInstance.PromotionTier``.
        :param publicly_accessible: ``AWS::RDS::DBInstance.PubliclyAccessible``.
        :param source_db_instance_identifier: ``AWS::RDS::DBInstance.SourceDBInstanceIdentifier``.
        :param source_region: ``AWS::RDS::DBInstance.SourceRegion``.
        :param storage_encrypted: ``AWS::RDS::DBInstance.StorageEncrypted``.
        :param storage_type: ``AWS::RDS::DBInstance.StorageType``.
        :param tags: ``AWS::RDS::DBInstance.Tags``.
        :param timezone: ``AWS::RDS::DBInstance.Timezone``.
        :param use_default_processor_features: ``AWS::RDS::DBInstance.UseDefaultProcessorFeatures``.
        :param vpc_security_groups: ``AWS::RDS::DBInstance.VPCSecurityGroups``.
        """
        props = CfnDBInstanceProps(
            db_instance_class=db_instance_class,
            allocated_storage=allocated_storage,
            allow_major_version_upgrade=allow_major_version_upgrade,
            associated_roles=associated_roles,
            auto_minor_version_upgrade=auto_minor_version_upgrade,
            availability_zone=availability_zone,
            backup_retention_period=backup_retention_period,
            ca_certificate_identifier=ca_certificate_identifier,
            character_set_name=character_set_name,
            copy_tags_to_snapshot=copy_tags_to_snapshot,
            db_cluster_identifier=db_cluster_identifier,
            db_instance_identifier=db_instance_identifier,
            db_name=db_name,
            db_parameter_group_name=db_parameter_group_name,
            db_security_groups=db_security_groups,
            db_snapshot_identifier=db_snapshot_identifier,
            db_subnet_group_name=db_subnet_group_name,
            delete_automated_backups=delete_automated_backups,
            deletion_protection=deletion_protection,
            domain=domain,
            domain_iam_role_name=domain_iam_role_name,
            enable_cloudwatch_logs_exports=enable_cloudwatch_logs_exports,
            enable_iam_database_authentication=enable_iam_database_authentication,
            enable_performance_insights=enable_performance_insights,
            engine=engine,
            engine_version=engine_version,
            iops=iops,
            kms_key_id=kms_key_id,
            license_model=license_model,
            master_username=master_username,
            master_user_password=master_user_password,
            max_allocated_storage=max_allocated_storage,
            monitoring_interval=monitoring_interval,
            monitoring_role_arn=monitoring_role_arn,
            multi_az=multi_az,
            option_group_name=option_group_name,
            performance_insights_kms_key_id=performance_insights_kms_key_id,
            performance_insights_retention_period=performance_insights_retention_period,
            port=port,
            preferred_backup_window=preferred_backup_window,
            preferred_maintenance_window=preferred_maintenance_window,
            processor_features=processor_features,
            promotion_tier=promotion_tier,
            publicly_accessible=publicly_accessible,
            source_db_instance_identifier=source_db_instance_identifier,
            source_region=source_region,
            storage_encrypted=storage_encrypted,
            storage_type=storage_type,
            tags=tags,
            timezone=timezone,
            use_default_processor_features=use_default_processor_features,
            vpc_security_groups=vpc_security_groups,
        )

        jsii.create(CfnDBInstance, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        """(experimental) Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.

        :stability: experimental
        """
        return jsii.invoke(self, "inspect", [inspector])

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        """
        :param props: -
        """
        return jsii.invoke(self, "renderProperties", [props])

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        """The CloudFormation resource type name for this resource class."""
        return jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="attrEndpointAddress")
    def attr_endpoint_address(self) -> builtins.str:
        """
        :cloudformationAttribute: Endpoint.Address
        """
        return jsii.get(self, "attrEndpointAddress")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="attrEndpointPort")
    def attr_endpoint_port(self) -> builtins.str:
        """
        :cloudformationAttribute: Endpoint.Port
        """
        return jsii.get(self, "attrEndpointPort")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return jsii.get(self, "cfnProperties")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        """``AWS::RDS::DBInstance.Tags``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-tags
        """
        return jsii.get(self, "tags")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="dbInstanceClass")
    def db_instance_class(self) -> builtins.str:
        """``AWS::RDS::DBInstance.DBInstanceClass``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbinstanceclass
        """
        return jsii.get(self, "dbInstanceClass")

    @db_instance_class.setter # type: ignore
    def db_instance_class(self, value: builtins.str) -> None:
        jsii.set(self, "dbInstanceClass", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="allocatedStorage")
    def allocated_storage(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.AllocatedStorage``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-allocatedstorage
        """
        return jsii.get(self, "allocatedStorage")

    @allocated_storage.setter # type: ignore
    def allocated_storage(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "allocatedStorage", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="allowMajorVersionUpgrade")
    def allow_major_version_upgrade(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBInstance.AllowMajorVersionUpgrade``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-allowmajorversionupgrade
        """
        return jsii.get(self, "allowMajorVersionUpgrade")

    @allow_major_version_upgrade.setter # type: ignore
    def allow_major_version_upgrade(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "allowMajorVersionUpgrade", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="associatedRoles")
    def associated_roles(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnDBInstance.DBInstanceRoleProperty"]]]]:
        """``AWS::RDS::DBInstance.AssociatedRoles``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-associatedroles
        """
        return jsii.get(self, "associatedRoles")

    @associated_roles.setter # type: ignore
    def associated_roles(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnDBInstance.DBInstanceRoleProperty"]]]],
    ) -> None:
        jsii.set(self, "associatedRoles", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="autoMinorVersionUpgrade")
    def auto_minor_version_upgrade(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBInstance.AutoMinorVersionUpgrade``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-autominorversionupgrade
        """
        return jsii.get(self, "autoMinorVersionUpgrade")

    @auto_minor_version_upgrade.setter # type: ignore
    def auto_minor_version_upgrade(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "autoMinorVersionUpgrade", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="availabilityZone")
    def availability_zone(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.AvailabilityZone``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-availabilityzone
        """
        return jsii.get(self, "availabilityZone")

    @availability_zone.setter # type: ignore
    def availability_zone(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "availabilityZone", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="backupRetentionPeriod")
    def backup_retention_period(self) -> typing.Optional[jsii.Number]:
        """``AWS::RDS::DBInstance.BackupRetentionPeriod``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-backupretentionperiod
        """
        return jsii.get(self, "backupRetentionPeriod")

    @backup_retention_period.setter # type: ignore
    def backup_retention_period(self, value: typing.Optional[jsii.Number]) -> None:
        jsii.set(self, "backupRetentionPeriod", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="caCertificateIdentifier")
    def ca_certificate_identifier(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.CACertificateIdentifier``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-cacertificateidentifier
        """
        return jsii.get(self, "caCertificateIdentifier")

    @ca_certificate_identifier.setter # type: ignore
    def ca_certificate_identifier(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "caCertificateIdentifier", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="characterSetName")
    def character_set_name(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.CharacterSetName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-charactersetname
        """
        return jsii.get(self, "characterSetName")

    @character_set_name.setter # type: ignore
    def character_set_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "characterSetName", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="copyTagsToSnapshot")
    def copy_tags_to_snapshot(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBInstance.CopyTagsToSnapshot``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-copytagstosnapshot
        """
        return jsii.get(self, "copyTagsToSnapshot")

    @copy_tags_to_snapshot.setter # type: ignore
    def copy_tags_to_snapshot(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "copyTagsToSnapshot", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="dbClusterIdentifier")
    def db_cluster_identifier(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.DBClusterIdentifier``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbclusteridentifier
        """
        return jsii.get(self, "dbClusterIdentifier")

    @db_cluster_identifier.setter # type: ignore
    def db_cluster_identifier(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "dbClusterIdentifier", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="dbInstanceIdentifier")
    def db_instance_identifier(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.DBInstanceIdentifier``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbinstanceidentifier
        """
        return jsii.get(self, "dbInstanceIdentifier")

    @db_instance_identifier.setter # type: ignore
    def db_instance_identifier(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "dbInstanceIdentifier", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="dbName")
    def db_name(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.DBName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbname
        """
        return jsii.get(self, "dbName")

    @db_name.setter # type: ignore
    def db_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "dbName", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="dbParameterGroupName")
    def db_parameter_group_name(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.DBParameterGroupName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbparametergroupname
        """
        return jsii.get(self, "dbParameterGroupName")

    @db_parameter_group_name.setter # type: ignore
    def db_parameter_group_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "dbParameterGroupName", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="dbSecurityGroups")
    def db_security_groups(self) -> typing.Optional[typing.List[builtins.str]]:
        """``AWS::RDS::DBInstance.DBSecurityGroups``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbsecuritygroups
        """
        return jsii.get(self, "dbSecurityGroups")

    @db_security_groups.setter # type: ignore
    def db_security_groups(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        jsii.set(self, "dbSecurityGroups", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="dbSnapshotIdentifier")
    def db_snapshot_identifier(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.DBSnapshotIdentifier``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbsnapshotidentifier
        """
        return jsii.get(self, "dbSnapshotIdentifier")

    @db_snapshot_identifier.setter # type: ignore
    def db_snapshot_identifier(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "dbSnapshotIdentifier", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="dbSubnetGroupName")
    def db_subnet_group_name(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.DBSubnetGroupName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbsubnetgroupname
        """
        return jsii.get(self, "dbSubnetGroupName")

    @db_subnet_group_name.setter # type: ignore
    def db_subnet_group_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "dbSubnetGroupName", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="deleteAutomatedBackups")
    def delete_automated_backups(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBInstance.DeleteAutomatedBackups``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-deleteautomatedbackups
        """
        return jsii.get(self, "deleteAutomatedBackups")

    @delete_automated_backups.setter # type: ignore
    def delete_automated_backups(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "deleteAutomatedBackups", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="deletionProtection")
    def deletion_protection(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBInstance.DeletionProtection``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-deletionprotection
        """
        return jsii.get(self, "deletionProtection")

    @deletion_protection.setter # type: ignore
    def deletion_protection(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "deletionProtection", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="domain")
    def domain(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.Domain``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-domain
        """
        return jsii.get(self, "domain")

    @domain.setter # type: ignore
    def domain(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "domain", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="domainIamRoleName")
    def domain_iam_role_name(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.DomainIAMRoleName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-domainiamrolename
        """
        return jsii.get(self, "domainIamRoleName")

    @domain_iam_role_name.setter # type: ignore
    def domain_iam_role_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "domainIamRoleName", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="enableCloudwatchLogsExports")
    def enable_cloudwatch_logs_exports(
        self,
    ) -> typing.Optional[typing.List[builtins.str]]:
        """``AWS::RDS::DBInstance.EnableCloudwatchLogsExports``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-enablecloudwatchlogsexports
        """
        return jsii.get(self, "enableCloudwatchLogsExports")

    @enable_cloudwatch_logs_exports.setter # type: ignore
    def enable_cloudwatch_logs_exports(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        jsii.set(self, "enableCloudwatchLogsExports", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="enableIamDatabaseAuthentication")
    def enable_iam_database_authentication(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBInstance.EnableIAMDatabaseAuthentication``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-enableiamdatabaseauthentication
        """
        return jsii.get(self, "enableIamDatabaseAuthentication")

    @enable_iam_database_authentication.setter # type: ignore
    def enable_iam_database_authentication(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "enableIamDatabaseAuthentication", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="enablePerformanceInsights")
    def enable_performance_insights(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBInstance.EnablePerformanceInsights``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-enableperformanceinsights
        """
        return jsii.get(self, "enablePerformanceInsights")

    @enable_performance_insights.setter # type: ignore
    def enable_performance_insights(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "enablePerformanceInsights", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="engine")
    def engine(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.Engine``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-engine
        """
        return jsii.get(self, "engine")

    @engine.setter # type: ignore
    def engine(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "engine", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="engineVersion")
    def engine_version(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.EngineVersion``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-engineversion
        """
        return jsii.get(self, "engineVersion")

    @engine_version.setter # type: ignore
    def engine_version(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "engineVersion", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="iops")
    def iops(self) -> typing.Optional[jsii.Number]:
        """``AWS::RDS::DBInstance.Iops``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-iops
        """
        return jsii.get(self, "iops")

    @iops.setter # type: ignore
    def iops(self, value: typing.Optional[jsii.Number]) -> None:
        jsii.set(self, "iops", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="kmsKeyId")
    def kms_key_id(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.KmsKeyId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-kmskeyid
        """
        return jsii.get(self, "kmsKeyId")

    @kms_key_id.setter # type: ignore
    def kms_key_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "kmsKeyId", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="licenseModel")
    def license_model(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.LicenseModel``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-licensemodel
        """
        return jsii.get(self, "licenseModel")

    @license_model.setter # type: ignore
    def license_model(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "licenseModel", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="masterUsername")
    def master_username(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.MasterUsername``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-masterusername
        """
        return jsii.get(self, "masterUsername")

    @master_username.setter # type: ignore
    def master_username(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "masterUsername", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="masterUserPassword")
    def master_user_password(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.MasterUserPassword``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-masteruserpassword
        """
        return jsii.get(self, "masterUserPassword")

    @master_user_password.setter # type: ignore
    def master_user_password(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "masterUserPassword", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="maxAllocatedStorage")
    def max_allocated_storage(self) -> typing.Optional[jsii.Number]:
        """``AWS::RDS::DBInstance.MaxAllocatedStorage``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-maxallocatedstorage
        """
        return jsii.get(self, "maxAllocatedStorage")

    @max_allocated_storage.setter # type: ignore
    def max_allocated_storage(self, value: typing.Optional[jsii.Number]) -> None:
        jsii.set(self, "maxAllocatedStorage", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="monitoringInterval")
    def monitoring_interval(self) -> typing.Optional[jsii.Number]:
        """``AWS::RDS::DBInstance.MonitoringInterval``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-monitoringinterval
        """
        return jsii.get(self, "monitoringInterval")

    @monitoring_interval.setter # type: ignore
    def monitoring_interval(self, value: typing.Optional[jsii.Number]) -> None:
        jsii.set(self, "monitoringInterval", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="monitoringRoleArn")
    def monitoring_role_arn(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.MonitoringRoleArn``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-monitoringrolearn
        """
        return jsii.get(self, "monitoringRoleArn")

    @monitoring_role_arn.setter # type: ignore
    def monitoring_role_arn(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "monitoringRoleArn", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="multiAz")
    def multi_az(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBInstance.MultiAZ``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-multiaz
        """
        return jsii.get(self, "multiAz")

    @multi_az.setter # type: ignore
    def multi_az(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "multiAz", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="optionGroupName")
    def option_group_name(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.OptionGroupName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-optiongroupname
        """
        return jsii.get(self, "optionGroupName")

    @option_group_name.setter # type: ignore
    def option_group_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "optionGroupName", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="performanceInsightsKmsKeyId")
    def performance_insights_kms_key_id(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.PerformanceInsightsKMSKeyId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-performanceinsightskmskeyid
        """
        return jsii.get(self, "performanceInsightsKmsKeyId")

    @performance_insights_kms_key_id.setter # type: ignore
    def performance_insights_kms_key_id(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        jsii.set(self, "performanceInsightsKmsKeyId", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="performanceInsightsRetentionPeriod")
    def performance_insights_retention_period(self) -> typing.Optional[jsii.Number]:
        """``AWS::RDS::DBInstance.PerformanceInsightsRetentionPeriod``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-performanceinsightsretentionperiod
        """
        return jsii.get(self, "performanceInsightsRetentionPeriod")

    @performance_insights_retention_period.setter # type: ignore
    def performance_insights_retention_period(
        self,
        value: typing.Optional[jsii.Number],
    ) -> None:
        jsii.set(self, "performanceInsightsRetentionPeriod", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="port")
    def port(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.Port``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-port
        """
        return jsii.get(self, "port")

    @port.setter # type: ignore
    def port(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "port", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="preferredBackupWindow")
    def preferred_backup_window(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.PreferredBackupWindow``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-preferredbackupwindow
        """
        return jsii.get(self, "preferredBackupWindow")

    @preferred_backup_window.setter # type: ignore
    def preferred_backup_window(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "preferredBackupWindow", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="preferredMaintenanceWindow")
    def preferred_maintenance_window(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.PreferredMaintenanceWindow``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-preferredmaintenancewindow
        """
        return jsii.get(self, "preferredMaintenanceWindow")

    @preferred_maintenance_window.setter # type: ignore
    def preferred_maintenance_window(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        jsii.set(self, "preferredMaintenanceWindow", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="processorFeatures")
    def processor_features(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnDBInstance.ProcessorFeatureProperty"]]]]:
        """``AWS::RDS::DBInstance.ProcessorFeatures``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-processorfeatures
        """
        return jsii.get(self, "processorFeatures")

    @processor_features.setter # type: ignore
    def processor_features(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnDBInstance.ProcessorFeatureProperty"]]]],
    ) -> None:
        jsii.set(self, "processorFeatures", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="promotionTier")
    def promotion_tier(self) -> typing.Optional[jsii.Number]:
        """``AWS::RDS::DBInstance.PromotionTier``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-promotiontier
        """
        return jsii.get(self, "promotionTier")

    @promotion_tier.setter # type: ignore
    def promotion_tier(self, value: typing.Optional[jsii.Number]) -> None:
        jsii.set(self, "promotionTier", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="publiclyAccessible")
    def publicly_accessible(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBInstance.PubliclyAccessible``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-publiclyaccessible
        """
        return jsii.get(self, "publiclyAccessible")

    @publicly_accessible.setter # type: ignore
    def publicly_accessible(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "publiclyAccessible", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="sourceDbInstanceIdentifier")
    def source_db_instance_identifier(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.SourceDBInstanceIdentifier``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-sourcedbinstanceidentifier
        """
        return jsii.get(self, "sourceDbInstanceIdentifier")

    @source_db_instance_identifier.setter # type: ignore
    def source_db_instance_identifier(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        jsii.set(self, "sourceDbInstanceIdentifier", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="sourceRegion")
    def source_region(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.SourceRegion``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-sourceregion
        """
        return jsii.get(self, "sourceRegion")

    @source_region.setter # type: ignore
    def source_region(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "sourceRegion", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="storageEncrypted")
    def storage_encrypted(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBInstance.StorageEncrypted``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-storageencrypted
        """
        return jsii.get(self, "storageEncrypted")

    @storage_encrypted.setter # type: ignore
    def storage_encrypted(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "storageEncrypted", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="storageType")
    def storage_type(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.StorageType``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-storagetype
        """
        return jsii.get(self, "storageType")

    @storage_type.setter # type: ignore
    def storage_type(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "storageType", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="timezone")
    def timezone(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.Timezone``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-timezone
        """
        return jsii.get(self, "timezone")

    @timezone.setter # type: ignore
    def timezone(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "timezone", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="useDefaultProcessorFeatures")
    def use_default_processor_features(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBInstance.UseDefaultProcessorFeatures``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-usedefaultprocessorfeatures
        """
        return jsii.get(self, "useDefaultProcessorFeatures")

    @use_default_processor_features.setter # type: ignore
    def use_default_processor_features(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "useDefaultProcessorFeatures", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="vpcSecurityGroups")
    def vpc_security_groups(self) -> typing.Optional[typing.List[builtins.str]]:
        """``AWS::RDS::DBInstance.VPCSecurityGroups``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-vpcsecuritygroups
        """
        return jsii.get(self, "vpcSecurityGroups")

    @vpc_security_groups.setter # type: ignore
    def vpc_security_groups(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        jsii.set(self, "vpcSecurityGroups", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-rds.CfnDBInstance.DBInstanceRoleProperty",
        jsii_struct_bases=[],
        name_mapping={"feature_name": "featureName", "role_arn": "roleArn"},
    )
    class DBInstanceRoleProperty:
        def __init__(
            self,
            *,
            feature_name: builtins.str,
            role_arn: builtins.str,
        ) -> None:
            """
            :param feature_name: ``CfnDBInstance.DBInstanceRoleProperty.FeatureName``.
            :param role_arn: ``CfnDBInstance.DBInstanceRoleProperty.RoleArn``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-dbinstancerole.html
            """
            self._values: typing.Dict[str, typing.Any] = {
                "feature_name": feature_name,
                "role_arn": role_arn,
            }

        @builtins.property
        def feature_name(self) -> builtins.str:
            """``CfnDBInstance.DBInstanceRoleProperty.FeatureName``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-dbinstancerole.html#cfn-rds-dbinstance-dbinstancerole-featurename
            """
            result = self._values.get("feature_name")
            assert result is not None, "Required property 'feature_name' is missing"
            return result

        @builtins.property
        def role_arn(self) -> builtins.str:
            """``CfnDBInstance.DBInstanceRoleProperty.RoleArn``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-dbinstancerole.html#cfn-rds-dbinstance-dbinstancerole-rolearn
            """
            result = self._values.get("role_arn")
            assert result is not None, "Required property 'role_arn' is missing"
            return result

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DBInstanceRoleProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-rds.CfnDBInstance.ProcessorFeatureProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "value": "value"},
    )
    class ProcessorFeatureProperty:
        def __init__(
            self,
            *,
            name: typing.Optional[builtins.str] = None,
            value: typing.Optional[builtins.str] = None,
        ) -> None:
            """
            :param name: ``CfnDBInstance.ProcessorFeatureProperty.Name``.
            :param value: ``CfnDBInstance.ProcessorFeatureProperty.Value``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-processorfeature.html
            """
            self._values: typing.Dict[str, typing.Any] = {}
            if name is not None:
                self._values["name"] = name
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            """``CfnDBInstance.ProcessorFeatureProperty.Name``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-processorfeature.html#cfn-rds-dbinstance-processorfeature-name
            """
            result = self._values.get("name")
            return result

        @builtins.property
        def value(self) -> typing.Optional[builtins.str]:
            """``CfnDBInstance.ProcessorFeatureProperty.Value``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-processorfeature.html#cfn-rds-dbinstance-processorfeature-value
            """
            result = self._values.get("value")
            return result

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ProcessorFeatureProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.CfnDBInstanceProps",
    jsii_struct_bases=[],
    name_mapping={
        "db_instance_class": "dbInstanceClass",
        "allocated_storage": "allocatedStorage",
        "allow_major_version_upgrade": "allowMajorVersionUpgrade",
        "associated_roles": "associatedRoles",
        "auto_minor_version_upgrade": "autoMinorVersionUpgrade",
        "availability_zone": "availabilityZone",
        "backup_retention_period": "backupRetentionPeriod",
        "ca_certificate_identifier": "caCertificateIdentifier",
        "character_set_name": "characterSetName",
        "copy_tags_to_snapshot": "copyTagsToSnapshot",
        "db_cluster_identifier": "dbClusterIdentifier",
        "db_instance_identifier": "dbInstanceIdentifier",
        "db_name": "dbName",
        "db_parameter_group_name": "dbParameterGroupName",
        "db_security_groups": "dbSecurityGroups",
        "db_snapshot_identifier": "dbSnapshotIdentifier",
        "db_subnet_group_name": "dbSubnetGroupName",
        "delete_automated_backups": "deleteAutomatedBackups",
        "deletion_protection": "deletionProtection",
        "domain": "domain",
        "domain_iam_role_name": "domainIamRoleName",
        "enable_cloudwatch_logs_exports": "enableCloudwatchLogsExports",
        "enable_iam_database_authentication": "enableIamDatabaseAuthentication",
        "enable_performance_insights": "enablePerformanceInsights",
        "engine": "engine",
        "engine_version": "engineVersion",
        "iops": "iops",
        "kms_key_id": "kmsKeyId",
        "license_model": "licenseModel",
        "master_username": "masterUsername",
        "master_user_password": "masterUserPassword",
        "max_allocated_storage": "maxAllocatedStorage",
        "monitoring_interval": "monitoringInterval",
        "monitoring_role_arn": "monitoringRoleArn",
        "multi_az": "multiAz",
        "option_group_name": "optionGroupName",
        "performance_insights_kms_key_id": "performanceInsightsKmsKeyId",
        "performance_insights_retention_period": "performanceInsightsRetentionPeriod",
        "port": "port",
        "preferred_backup_window": "preferredBackupWindow",
        "preferred_maintenance_window": "preferredMaintenanceWindow",
        "processor_features": "processorFeatures",
        "promotion_tier": "promotionTier",
        "publicly_accessible": "publiclyAccessible",
        "source_db_instance_identifier": "sourceDbInstanceIdentifier",
        "source_region": "sourceRegion",
        "storage_encrypted": "storageEncrypted",
        "storage_type": "storageType",
        "tags": "tags",
        "timezone": "timezone",
        "use_default_processor_features": "useDefaultProcessorFeatures",
        "vpc_security_groups": "vpcSecurityGroups",
    },
)
class CfnDBInstanceProps:
    def __init__(
        self,
        *,
        db_instance_class: builtins.str,
        allocated_storage: typing.Optional[builtins.str] = None,
        allow_major_version_upgrade: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        associated_roles: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnDBInstance.DBInstanceRoleProperty]]]] = None,
        auto_minor_version_upgrade: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        availability_zone: typing.Optional[builtins.str] = None,
        backup_retention_period: typing.Optional[jsii.Number] = None,
        ca_certificate_identifier: typing.Optional[builtins.str] = None,
        character_set_name: typing.Optional[builtins.str] = None,
        copy_tags_to_snapshot: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        db_cluster_identifier: typing.Optional[builtins.str] = None,
        db_instance_identifier: typing.Optional[builtins.str] = None,
        db_name: typing.Optional[builtins.str] = None,
        db_parameter_group_name: typing.Optional[builtins.str] = None,
        db_security_groups: typing.Optional[typing.List[builtins.str]] = None,
        db_snapshot_identifier: typing.Optional[builtins.str] = None,
        db_subnet_group_name: typing.Optional[builtins.str] = None,
        delete_automated_backups: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        deletion_protection: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        domain: typing.Optional[builtins.str] = None,
        domain_iam_role_name: typing.Optional[builtins.str] = None,
        enable_cloudwatch_logs_exports: typing.Optional[typing.List[builtins.str]] = None,
        enable_iam_database_authentication: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        enable_performance_insights: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        engine: typing.Optional[builtins.str] = None,
        engine_version: typing.Optional[builtins.str] = None,
        iops: typing.Optional[jsii.Number] = None,
        kms_key_id: typing.Optional[builtins.str] = None,
        license_model: typing.Optional[builtins.str] = None,
        master_username: typing.Optional[builtins.str] = None,
        master_user_password: typing.Optional[builtins.str] = None,
        max_allocated_storage: typing.Optional[jsii.Number] = None,
        monitoring_interval: typing.Optional[jsii.Number] = None,
        monitoring_role_arn: typing.Optional[builtins.str] = None,
        multi_az: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        option_group_name: typing.Optional[builtins.str] = None,
        performance_insights_kms_key_id: typing.Optional[builtins.str] = None,
        performance_insights_retention_period: typing.Optional[jsii.Number] = None,
        port: typing.Optional[builtins.str] = None,
        preferred_backup_window: typing.Optional[builtins.str] = None,
        preferred_maintenance_window: typing.Optional[builtins.str] = None,
        processor_features: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnDBInstance.ProcessorFeatureProperty]]]] = None,
        promotion_tier: typing.Optional[jsii.Number] = None,
        publicly_accessible: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        source_db_instance_identifier: typing.Optional[builtins.str] = None,
        source_region: typing.Optional[builtins.str] = None,
        storage_encrypted: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        storage_type: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.List[aws_cdk.core.CfnTag]] = None,
        timezone: typing.Optional[builtins.str] = None,
        use_default_processor_features: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        vpc_security_groups: typing.Optional[typing.List[builtins.str]] = None,
    ) -> None:
        """Properties for defining a ``AWS::RDS::DBInstance``.

        :param db_instance_class: ``AWS::RDS::DBInstance.DBInstanceClass``.
        :param allocated_storage: ``AWS::RDS::DBInstance.AllocatedStorage``.
        :param allow_major_version_upgrade: ``AWS::RDS::DBInstance.AllowMajorVersionUpgrade``.
        :param associated_roles: ``AWS::RDS::DBInstance.AssociatedRoles``.
        :param auto_minor_version_upgrade: ``AWS::RDS::DBInstance.AutoMinorVersionUpgrade``.
        :param availability_zone: ``AWS::RDS::DBInstance.AvailabilityZone``.
        :param backup_retention_period: ``AWS::RDS::DBInstance.BackupRetentionPeriod``.
        :param ca_certificate_identifier: ``AWS::RDS::DBInstance.CACertificateIdentifier``.
        :param character_set_name: ``AWS::RDS::DBInstance.CharacterSetName``.
        :param copy_tags_to_snapshot: ``AWS::RDS::DBInstance.CopyTagsToSnapshot``.
        :param db_cluster_identifier: ``AWS::RDS::DBInstance.DBClusterIdentifier``.
        :param db_instance_identifier: ``AWS::RDS::DBInstance.DBInstanceIdentifier``.
        :param db_name: ``AWS::RDS::DBInstance.DBName``.
        :param db_parameter_group_name: ``AWS::RDS::DBInstance.DBParameterGroupName``.
        :param db_security_groups: ``AWS::RDS::DBInstance.DBSecurityGroups``.
        :param db_snapshot_identifier: ``AWS::RDS::DBInstance.DBSnapshotIdentifier``.
        :param db_subnet_group_name: ``AWS::RDS::DBInstance.DBSubnetGroupName``.
        :param delete_automated_backups: ``AWS::RDS::DBInstance.DeleteAutomatedBackups``.
        :param deletion_protection: ``AWS::RDS::DBInstance.DeletionProtection``.
        :param domain: ``AWS::RDS::DBInstance.Domain``.
        :param domain_iam_role_name: ``AWS::RDS::DBInstance.DomainIAMRoleName``.
        :param enable_cloudwatch_logs_exports: ``AWS::RDS::DBInstance.EnableCloudwatchLogsExports``.
        :param enable_iam_database_authentication: ``AWS::RDS::DBInstance.EnableIAMDatabaseAuthentication``.
        :param enable_performance_insights: ``AWS::RDS::DBInstance.EnablePerformanceInsights``.
        :param engine: ``AWS::RDS::DBInstance.Engine``.
        :param engine_version: ``AWS::RDS::DBInstance.EngineVersion``.
        :param iops: ``AWS::RDS::DBInstance.Iops``.
        :param kms_key_id: ``AWS::RDS::DBInstance.KmsKeyId``.
        :param license_model: ``AWS::RDS::DBInstance.LicenseModel``.
        :param master_username: ``AWS::RDS::DBInstance.MasterUsername``.
        :param master_user_password: ``AWS::RDS::DBInstance.MasterUserPassword``.
        :param max_allocated_storage: ``AWS::RDS::DBInstance.MaxAllocatedStorage``.
        :param monitoring_interval: ``AWS::RDS::DBInstance.MonitoringInterval``.
        :param monitoring_role_arn: ``AWS::RDS::DBInstance.MonitoringRoleArn``.
        :param multi_az: ``AWS::RDS::DBInstance.MultiAZ``.
        :param option_group_name: ``AWS::RDS::DBInstance.OptionGroupName``.
        :param performance_insights_kms_key_id: ``AWS::RDS::DBInstance.PerformanceInsightsKMSKeyId``.
        :param performance_insights_retention_period: ``AWS::RDS::DBInstance.PerformanceInsightsRetentionPeriod``.
        :param port: ``AWS::RDS::DBInstance.Port``.
        :param preferred_backup_window: ``AWS::RDS::DBInstance.PreferredBackupWindow``.
        :param preferred_maintenance_window: ``AWS::RDS::DBInstance.PreferredMaintenanceWindow``.
        :param processor_features: ``AWS::RDS::DBInstance.ProcessorFeatures``.
        :param promotion_tier: ``AWS::RDS::DBInstance.PromotionTier``.
        :param publicly_accessible: ``AWS::RDS::DBInstance.PubliclyAccessible``.
        :param source_db_instance_identifier: ``AWS::RDS::DBInstance.SourceDBInstanceIdentifier``.
        :param source_region: ``AWS::RDS::DBInstance.SourceRegion``.
        :param storage_encrypted: ``AWS::RDS::DBInstance.StorageEncrypted``.
        :param storage_type: ``AWS::RDS::DBInstance.StorageType``.
        :param tags: ``AWS::RDS::DBInstance.Tags``.
        :param timezone: ``AWS::RDS::DBInstance.Timezone``.
        :param use_default_processor_features: ``AWS::RDS::DBInstance.UseDefaultProcessorFeatures``.
        :param vpc_security_groups: ``AWS::RDS::DBInstance.VPCSecurityGroups``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html
        """
        self._values: typing.Dict[str, typing.Any] = {
            "db_instance_class": db_instance_class,
        }
        if allocated_storage is not None:
            self._values["allocated_storage"] = allocated_storage
        if allow_major_version_upgrade is not None:
            self._values["allow_major_version_upgrade"] = allow_major_version_upgrade
        if associated_roles is not None:
            self._values["associated_roles"] = associated_roles
        if auto_minor_version_upgrade is not None:
            self._values["auto_minor_version_upgrade"] = auto_minor_version_upgrade
        if availability_zone is not None:
            self._values["availability_zone"] = availability_zone
        if backup_retention_period is not None:
            self._values["backup_retention_period"] = backup_retention_period
        if ca_certificate_identifier is not None:
            self._values["ca_certificate_identifier"] = ca_certificate_identifier
        if character_set_name is not None:
            self._values["character_set_name"] = character_set_name
        if copy_tags_to_snapshot is not None:
            self._values["copy_tags_to_snapshot"] = copy_tags_to_snapshot
        if db_cluster_identifier is not None:
            self._values["db_cluster_identifier"] = db_cluster_identifier
        if db_instance_identifier is not None:
            self._values["db_instance_identifier"] = db_instance_identifier
        if db_name is not None:
            self._values["db_name"] = db_name
        if db_parameter_group_name is not None:
            self._values["db_parameter_group_name"] = db_parameter_group_name
        if db_security_groups is not None:
            self._values["db_security_groups"] = db_security_groups
        if db_snapshot_identifier is not None:
            self._values["db_snapshot_identifier"] = db_snapshot_identifier
        if db_subnet_group_name is not None:
            self._values["db_subnet_group_name"] = db_subnet_group_name
        if delete_automated_backups is not None:
            self._values["delete_automated_backups"] = delete_automated_backups
        if deletion_protection is not None:
            self._values["deletion_protection"] = deletion_protection
        if domain is not None:
            self._values["domain"] = domain
        if domain_iam_role_name is not None:
            self._values["domain_iam_role_name"] = domain_iam_role_name
        if enable_cloudwatch_logs_exports is not None:
            self._values["enable_cloudwatch_logs_exports"] = enable_cloudwatch_logs_exports
        if enable_iam_database_authentication is not None:
            self._values["enable_iam_database_authentication"] = enable_iam_database_authentication
        if enable_performance_insights is not None:
            self._values["enable_performance_insights"] = enable_performance_insights
        if engine is not None:
            self._values["engine"] = engine
        if engine_version is not None:
            self._values["engine_version"] = engine_version
        if iops is not None:
            self._values["iops"] = iops
        if kms_key_id is not None:
            self._values["kms_key_id"] = kms_key_id
        if license_model is not None:
            self._values["license_model"] = license_model
        if master_username is not None:
            self._values["master_username"] = master_username
        if master_user_password is not None:
            self._values["master_user_password"] = master_user_password
        if max_allocated_storage is not None:
            self._values["max_allocated_storage"] = max_allocated_storage
        if monitoring_interval is not None:
            self._values["monitoring_interval"] = monitoring_interval
        if monitoring_role_arn is not None:
            self._values["monitoring_role_arn"] = monitoring_role_arn
        if multi_az is not None:
            self._values["multi_az"] = multi_az
        if option_group_name is not None:
            self._values["option_group_name"] = option_group_name
        if performance_insights_kms_key_id is not None:
            self._values["performance_insights_kms_key_id"] = performance_insights_kms_key_id
        if performance_insights_retention_period is not None:
            self._values["performance_insights_retention_period"] = performance_insights_retention_period
        if port is not None:
            self._values["port"] = port
        if preferred_backup_window is not None:
            self._values["preferred_backup_window"] = preferred_backup_window
        if preferred_maintenance_window is not None:
            self._values["preferred_maintenance_window"] = preferred_maintenance_window
        if processor_features is not None:
            self._values["processor_features"] = processor_features
        if promotion_tier is not None:
            self._values["promotion_tier"] = promotion_tier
        if publicly_accessible is not None:
            self._values["publicly_accessible"] = publicly_accessible
        if source_db_instance_identifier is not None:
            self._values["source_db_instance_identifier"] = source_db_instance_identifier
        if source_region is not None:
            self._values["source_region"] = source_region
        if storage_encrypted is not None:
            self._values["storage_encrypted"] = storage_encrypted
        if storage_type is not None:
            self._values["storage_type"] = storage_type
        if tags is not None:
            self._values["tags"] = tags
        if timezone is not None:
            self._values["timezone"] = timezone
        if use_default_processor_features is not None:
            self._values["use_default_processor_features"] = use_default_processor_features
        if vpc_security_groups is not None:
            self._values["vpc_security_groups"] = vpc_security_groups

    @builtins.property
    def db_instance_class(self) -> builtins.str:
        """``AWS::RDS::DBInstance.DBInstanceClass``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbinstanceclass
        """
        result = self._values.get("db_instance_class")
        assert result is not None, "Required property 'db_instance_class' is missing"
        return result

    @builtins.property
    def allocated_storage(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.AllocatedStorage``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-allocatedstorage
        """
        result = self._values.get("allocated_storage")
        return result

    @builtins.property
    def allow_major_version_upgrade(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBInstance.AllowMajorVersionUpgrade``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-allowmajorversionupgrade
        """
        result = self._values.get("allow_major_version_upgrade")
        return result

    @builtins.property
    def associated_roles(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnDBInstance.DBInstanceRoleProperty]]]]:
        """``AWS::RDS::DBInstance.AssociatedRoles``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-associatedroles
        """
        result = self._values.get("associated_roles")
        return result

    @builtins.property
    def auto_minor_version_upgrade(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBInstance.AutoMinorVersionUpgrade``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-autominorversionupgrade
        """
        result = self._values.get("auto_minor_version_upgrade")
        return result

    @builtins.property
    def availability_zone(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.AvailabilityZone``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-availabilityzone
        """
        result = self._values.get("availability_zone")
        return result

    @builtins.property
    def backup_retention_period(self) -> typing.Optional[jsii.Number]:
        """``AWS::RDS::DBInstance.BackupRetentionPeriod``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-backupretentionperiod
        """
        result = self._values.get("backup_retention_period")
        return result

    @builtins.property
    def ca_certificate_identifier(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.CACertificateIdentifier``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-cacertificateidentifier
        """
        result = self._values.get("ca_certificate_identifier")
        return result

    @builtins.property
    def character_set_name(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.CharacterSetName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-charactersetname
        """
        result = self._values.get("character_set_name")
        return result

    @builtins.property
    def copy_tags_to_snapshot(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBInstance.CopyTagsToSnapshot``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-copytagstosnapshot
        """
        result = self._values.get("copy_tags_to_snapshot")
        return result

    @builtins.property
    def db_cluster_identifier(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.DBClusterIdentifier``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbclusteridentifier
        """
        result = self._values.get("db_cluster_identifier")
        return result

    @builtins.property
    def db_instance_identifier(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.DBInstanceIdentifier``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbinstanceidentifier
        """
        result = self._values.get("db_instance_identifier")
        return result

    @builtins.property
    def db_name(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.DBName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbname
        """
        result = self._values.get("db_name")
        return result

    @builtins.property
    def db_parameter_group_name(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.DBParameterGroupName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbparametergroupname
        """
        result = self._values.get("db_parameter_group_name")
        return result

    @builtins.property
    def db_security_groups(self) -> typing.Optional[typing.List[builtins.str]]:
        """``AWS::RDS::DBInstance.DBSecurityGroups``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbsecuritygroups
        """
        result = self._values.get("db_security_groups")
        return result

    @builtins.property
    def db_snapshot_identifier(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.DBSnapshotIdentifier``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbsnapshotidentifier
        """
        result = self._values.get("db_snapshot_identifier")
        return result

    @builtins.property
    def db_subnet_group_name(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.DBSubnetGroupName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbsubnetgroupname
        """
        result = self._values.get("db_subnet_group_name")
        return result

    @builtins.property
    def delete_automated_backups(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBInstance.DeleteAutomatedBackups``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-deleteautomatedbackups
        """
        result = self._values.get("delete_automated_backups")
        return result

    @builtins.property
    def deletion_protection(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBInstance.DeletionProtection``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-deletionprotection
        """
        result = self._values.get("deletion_protection")
        return result

    @builtins.property
    def domain(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.Domain``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-domain
        """
        result = self._values.get("domain")
        return result

    @builtins.property
    def domain_iam_role_name(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.DomainIAMRoleName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-domainiamrolename
        """
        result = self._values.get("domain_iam_role_name")
        return result

    @builtins.property
    def enable_cloudwatch_logs_exports(
        self,
    ) -> typing.Optional[typing.List[builtins.str]]:
        """``AWS::RDS::DBInstance.EnableCloudwatchLogsExports``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-enablecloudwatchlogsexports
        """
        result = self._values.get("enable_cloudwatch_logs_exports")
        return result

    @builtins.property
    def enable_iam_database_authentication(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBInstance.EnableIAMDatabaseAuthentication``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-enableiamdatabaseauthentication
        """
        result = self._values.get("enable_iam_database_authentication")
        return result

    @builtins.property
    def enable_performance_insights(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBInstance.EnablePerformanceInsights``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-enableperformanceinsights
        """
        result = self._values.get("enable_performance_insights")
        return result

    @builtins.property
    def engine(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.Engine``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-engine
        """
        result = self._values.get("engine")
        return result

    @builtins.property
    def engine_version(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.EngineVersion``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-engineversion
        """
        result = self._values.get("engine_version")
        return result

    @builtins.property
    def iops(self) -> typing.Optional[jsii.Number]:
        """``AWS::RDS::DBInstance.Iops``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-iops
        """
        result = self._values.get("iops")
        return result

    @builtins.property
    def kms_key_id(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.KmsKeyId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-kmskeyid
        """
        result = self._values.get("kms_key_id")
        return result

    @builtins.property
    def license_model(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.LicenseModel``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-licensemodel
        """
        result = self._values.get("license_model")
        return result

    @builtins.property
    def master_username(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.MasterUsername``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-masterusername
        """
        result = self._values.get("master_username")
        return result

    @builtins.property
    def master_user_password(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.MasterUserPassword``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-masteruserpassword
        """
        result = self._values.get("master_user_password")
        return result

    @builtins.property
    def max_allocated_storage(self) -> typing.Optional[jsii.Number]:
        """``AWS::RDS::DBInstance.MaxAllocatedStorage``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-maxallocatedstorage
        """
        result = self._values.get("max_allocated_storage")
        return result

    @builtins.property
    def monitoring_interval(self) -> typing.Optional[jsii.Number]:
        """``AWS::RDS::DBInstance.MonitoringInterval``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-monitoringinterval
        """
        result = self._values.get("monitoring_interval")
        return result

    @builtins.property
    def monitoring_role_arn(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.MonitoringRoleArn``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-monitoringrolearn
        """
        result = self._values.get("monitoring_role_arn")
        return result

    @builtins.property
    def multi_az(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBInstance.MultiAZ``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-multiaz
        """
        result = self._values.get("multi_az")
        return result

    @builtins.property
    def option_group_name(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.OptionGroupName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-optiongroupname
        """
        result = self._values.get("option_group_name")
        return result

    @builtins.property
    def performance_insights_kms_key_id(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.PerformanceInsightsKMSKeyId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-performanceinsightskmskeyid
        """
        result = self._values.get("performance_insights_kms_key_id")
        return result

    @builtins.property
    def performance_insights_retention_period(self) -> typing.Optional[jsii.Number]:
        """``AWS::RDS::DBInstance.PerformanceInsightsRetentionPeriod``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-performanceinsightsretentionperiod
        """
        result = self._values.get("performance_insights_retention_period")
        return result

    @builtins.property
    def port(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.Port``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-port
        """
        result = self._values.get("port")
        return result

    @builtins.property
    def preferred_backup_window(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.PreferredBackupWindow``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-preferredbackupwindow
        """
        result = self._values.get("preferred_backup_window")
        return result

    @builtins.property
    def preferred_maintenance_window(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.PreferredMaintenanceWindow``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-preferredmaintenancewindow
        """
        result = self._values.get("preferred_maintenance_window")
        return result

    @builtins.property
    def processor_features(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnDBInstance.ProcessorFeatureProperty]]]]:
        """``AWS::RDS::DBInstance.ProcessorFeatures``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-processorfeatures
        """
        result = self._values.get("processor_features")
        return result

    @builtins.property
    def promotion_tier(self) -> typing.Optional[jsii.Number]:
        """``AWS::RDS::DBInstance.PromotionTier``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-promotiontier
        """
        result = self._values.get("promotion_tier")
        return result

    @builtins.property
    def publicly_accessible(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBInstance.PubliclyAccessible``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-publiclyaccessible
        """
        result = self._values.get("publicly_accessible")
        return result

    @builtins.property
    def source_db_instance_identifier(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.SourceDBInstanceIdentifier``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-sourcedbinstanceidentifier
        """
        result = self._values.get("source_db_instance_identifier")
        return result

    @builtins.property
    def source_region(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.SourceRegion``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-sourceregion
        """
        result = self._values.get("source_region")
        return result

    @builtins.property
    def storage_encrypted(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBInstance.StorageEncrypted``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-storageencrypted
        """
        result = self._values.get("storage_encrypted")
        return result

    @builtins.property
    def storage_type(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.StorageType``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-storagetype
        """
        result = self._values.get("storage_type")
        return result

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        """``AWS::RDS::DBInstance.Tags``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-tags
        """
        result = self._values.get("tags")
        return result

    @builtins.property
    def timezone(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBInstance.Timezone``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-timezone
        """
        result = self._values.get("timezone")
        return result

    @builtins.property
    def use_default_processor_features(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBInstance.UseDefaultProcessorFeatures``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-usedefaultprocessorfeatures
        """
        result = self._values.get("use_default_processor_features")
        return result

    @builtins.property
    def vpc_security_groups(self) -> typing.Optional[typing.List[builtins.str]]:
        """``AWS::RDS::DBInstance.VPCSecurityGroups``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-vpcsecuritygroups
        """
        result = self._values.get("vpc_security_groups")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDBInstanceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnDBParameterGroup(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.CfnDBParameterGroup",
):
    """A CloudFormation ``AWS::RDS::DBParameterGroup``.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbparametergroup.html
    :cloudformationResource: AWS::RDS::DBParameterGroup
    """

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        description: builtins.str,
        family: builtins.str,
        parameters: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Mapping[builtins.str, builtins.str]]] = None,
        tags: typing.Optional[typing.List[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        """Create a new ``AWS::RDS::DBParameterGroup``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param description: ``AWS::RDS::DBParameterGroup.Description``.
        :param family: ``AWS::RDS::DBParameterGroup.Family``.
        :param parameters: ``AWS::RDS::DBParameterGroup.Parameters``.
        :param tags: ``AWS::RDS::DBParameterGroup.Tags``.
        """
        props = CfnDBParameterGroupProps(
            description=description, family=family, parameters=parameters, tags=tags
        )

        jsii.create(CfnDBParameterGroup, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        """(experimental) Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.

        :stability: experimental
        """
        return jsii.invoke(self, "inspect", [inspector])

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        """
        :param props: -
        """
        return jsii.invoke(self, "renderProperties", [props])

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        """The CloudFormation resource type name for this resource class."""
        return jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return jsii.get(self, "cfnProperties")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        """``AWS::RDS::DBParameterGroup.Tags``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbparametergroup.html#cfn-rds-dbparametergroup-tags
        """
        return jsii.get(self, "tags")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="description")
    def description(self) -> builtins.str:
        """``AWS::RDS::DBParameterGroup.Description``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbparametergroup.html#cfn-rds-dbparametergroup-description
        """
        return jsii.get(self, "description")

    @description.setter # type: ignore
    def description(self, value: builtins.str) -> None:
        jsii.set(self, "description", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="family")
    def family(self) -> builtins.str:
        """``AWS::RDS::DBParameterGroup.Family``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbparametergroup.html#cfn-rds-dbparametergroup-family
        """
        return jsii.get(self, "family")

    @family.setter # type: ignore
    def family(self, value: builtins.str) -> None:
        jsii.set(self, "family", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="parameters")
    def parameters(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Mapping[builtins.str, builtins.str]]]:
        """``AWS::RDS::DBParameterGroup.Parameters``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbparametergroup.html#cfn-rds-dbparametergroup-parameters
        """
        return jsii.get(self, "parameters")

    @parameters.setter # type: ignore
    def parameters(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Mapping[builtins.str, builtins.str]]],
    ) -> None:
        jsii.set(self, "parameters", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.CfnDBParameterGroupProps",
    jsii_struct_bases=[],
    name_mapping={
        "description": "description",
        "family": "family",
        "parameters": "parameters",
        "tags": "tags",
    },
)
class CfnDBParameterGroupProps:
    def __init__(
        self,
        *,
        description: builtins.str,
        family: builtins.str,
        parameters: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Mapping[builtins.str, builtins.str]]] = None,
        tags: typing.Optional[typing.List[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        """Properties for defining a ``AWS::RDS::DBParameterGroup``.

        :param description: ``AWS::RDS::DBParameterGroup.Description``.
        :param family: ``AWS::RDS::DBParameterGroup.Family``.
        :param parameters: ``AWS::RDS::DBParameterGroup.Parameters``.
        :param tags: ``AWS::RDS::DBParameterGroup.Tags``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbparametergroup.html
        """
        self._values: typing.Dict[str, typing.Any] = {
            "description": description,
            "family": family,
        }
        if parameters is not None:
            self._values["parameters"] = parameters
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def description(self) -> builtins.str:
        """``AWS::RDS::DBParameterGroup.Description``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbparametergroup.html#cfn-rds-dbparametergroup-description
        """
        result = self._values.get("description")
        assert result is not None, "Required property 'description' is missing"
        return result

    @builtins.property
    def family(self) -> builtins.str:
        """``AWS::RDS::DBParameterGroup.Family``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbparametergroup.html#cfn-rds-dbparametergroup-family
        """
        result = self._values.get("family")
        assert result is not None, "Required property 'family' is missing"
        return result

    @builtins.property
    def parameters(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Mapping[builtins.str, builtins.str]]]:
        """``AWS::RDS::DBParameterGroup.Parameters``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbparametergroup.html#cfn-rds-dbparametergroup-parameters
        """
        result = self._values.get("parameters")
        return result

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        """``AWS::RDS::DBParameterGroup.Tags``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbparametergroup.html#cfn-rds-dbparametergroup-tags
        """
        result = self._values.get("tags")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDBParameterGroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnDBProxy(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.CfnDBProxy",
):
    """A CloudFormation ``AWS::RDS::DBProxy``.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html
    :cloudformationResource: AWS::RDS::DBProxy
    """

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        auth: typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union["CfnDBProxy.AuthFormatProperty", aws_cdk.core.IResolvable]]],
        db_proxy_name: builtins.str,
        engine_family: builtins.str,
        role_arn: builtins.str,
        vpc_subnet_ids: typing.List[builtins.str],
        debug_logging: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        idle_client_timeout: typing.Optional[jsii.Number] = None,
        require_tls: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        tags: typing.Optional[typing.List["CfnDBProxy.TagFormatProperty"]] = None,
        vpc_security_group_ids: typing.Optional[typing.List[builtins.str]] = None,
    ) -> None:
        """Create a new ``AWS::RDS::DBProxy``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param auth: ``AWS::RDS::DBProxy.Auth``.
        :param db_proxy_name: ``AWS::RDS::DBProxy.DBProxyName``.
        :param engine_family: ``AWS::RDS::DBProxy.EngineFamily``.
        :param role_arn: ``AWS::RDS::DBProxy.RoleArn``.
        :param vpc_subnet_ids: ``AWS::RDS::DBProxy.VpcSubnetIds``.
        :param debug_logging: ``AWS::RDS::DBProxy.DebugLogging``.
        :param idle_client_timeout: ``AWS::RDS::DBProxy.IdleClientTimeout``.
        :param require_tls: ``AWS::RDS::DBProxy.RequireTLS``.
        :param tags: ``AWS::RDS::DBProxy.Tags``.
        :param vpc_security_group_ids: ``AWS::RDS::DBProxy.VpcSecurityGroupIds``.
        """
        props = CfnDBProxyProps(
            auth=auth,
            db_proxy_name=db_proxy_name,
            engine_family=engine_family,
            role_arn=role_arn,
            vpc_subnet_ids=vpc_subnet_ids,
            debug_logging=debug_logging,
            idle_client_timeout=idle_client_timeout,
            require_tls=require_tls,
            tags=tags,
            vpc_security_group_ids=vpc_security_group_ids,
        )

        jsii.create(CfnDBProxy, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        """(experimental) Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.

        :stability: experimental
        """
        return jsii.invoke(self, "inspect", [inspector])

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        """
        :param props: -
        """
        return jsii.invoke(self, "renderProperties", [props])

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        """The CloudFormation resource type name for this resource class."""
        return jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="attrDbProxyArn")
    def attr_db_proxy_arn(self) -> builtins.str:
        """
        :cloudformationAttribute: DBProxyArn
        """
        return jsii.get(self, "attrDbProxyArn")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="attrEndpoint")
    def attr_endpoint(self) -> builtins.str:
        """
        :cloudformationAttribute: Endpoint
        """
        return jsii.get(self, "attrEndpoint")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return jsii.get(self, "cfnProperties")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="auth")
    def auth(
        self,
    ) -> typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union["CfnDBProxy.AuthFormatProperty", aws_cdk.core.IResolvable]]]:
        """``AWS::RDS::DBProxy.Auth``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-auth
        """
        return jsii.get(self, "auth")

    @auth.setter # type: ignore
    def auth(
        self,
        value: typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union["CfnDBProxy.AuthFormatProperty", aws_cdk.core.IResolvable]]],
    ) -> None:
        jsii.set(self, "auth", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="dbProxyName")
    def db_proxy_name(self) -> builtins.str:
        """``AWS::RDS::DBProxy.DBProxyName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-dbproxyname
        """
        return jsii.get(self, "dbProxyName")

    @db_proxy_name.setter # type: ignore
    def db_proxy_name(self, value: builtins.str) -> None:
        jsii.set(self, "dbProxyName", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="engineFamily")
    def engine_family(self) -> builtins.str:
        """``AWS::RDS::DBProxy.EngineFamily``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-enginefamily
        """
        return jsii.get(self, "engineFamily")

    @engine_family.setter # type: ignore
    def engine_family(self, value: builtins.str) -> None:
        jsii.set(self, "engineFamily", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="roleArn")
    def role_arn(self) -> builtins.str:
        """``AWS::RDS::DBProxy.RoleArn``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-rolearn
        """
        return jsii.get(self, "roleArn")

    @role_arn.setter # type: ignore
    def role_arn(self, value: builtins.str) -> None:
        jsii.set(self, "roleArn", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="vpcSubnetIds")
    def vpc_subnet_ids(self) -> typing.List[builtins.str]:
        """``AWS::RDS::DBProxy.VpcSubnetIds``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-vpcsubnetids
        """
        return jsii.get(self, "vpcSubnetIds")

    @vpc_subnet_ids.setter # type: ignore
    def vpc_subnet_ids(self, value: typing.List[builtins.str]) -> None:
        jsii.set(self, "vpcSubnetIds", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="debugLogging")
    def debug_logging(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBProxy.DebugLogging``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-debuglogging
        """
        return jsii.get(self, "debugLogging")

    @debug_logging.setter # type: ignore
    def debug_logging(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "debugLogging", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="idleClientTimeout")
    def idle_client_timeout(self) -> typing.Optional[jsii.Number]:
        """``AWS::RDS::DBProxy.IdleClientTimeout``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-idleclienttimeout
        """
        return jsii.get(self, "idleClientTimeout")

    @idle_client_timeout.setter # type: ignore
    def idle_client_timeout(self, value: typing.Optional[jsii.Number]) -> None:
        jsii.set(self, "idleClientTimeout", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="requireTls")
    def require_tls(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBProxy.RequireTLS``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-requiretls
        """
        return jsii.get(self, "requireTls")

    @require_tls.setter # type: ignore
    def require_tls(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "requireTls", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="tags")
    def tags(self) -> typing.Optional[typing.List["CfnDBProxy.TagFormatProperty"]]:
        """``AWS::RDS::DBProxy.Tags``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-tags
        """
        return jsii.get(self, "tags")

    @tags.setter # type: ignore
    def tags(
        self,
        value: typing.Optional[typing.List["CfnDBProxy.TagFormatProperty"]],
    ) -> None:
        jsii.set(self, "tags", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="vpcSecurityGroupIds")
    def vpc_security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        """``AWS::RDS::DBProxy.VpcSecurityGroupIds``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-vpcsecuritygroupids
        """
        return jsii.get(self, "vpcSecurityGroupIds")

    @vpc_security_group_ids.setter # type: ignore
    def vpc_security_group_ids(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        jsii.set(self, "vpcSecurityGroupIds", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-rds.CfnDBProxy.AuthFormatProperty",
        jsii_struct_bases=[],
        name_mapping={
            "auth_scheme": "authScheme",
            "description": "description",
            "iam_auth": "iamAuth",
            "secret_arn": "secretArn",
            "user_name": "userName",
        },
    )
    class AuthFormatProperty:
        def __init__(
            self,
            *,
            auth_scheme: typing.Optional[builtins.str] = None,
            description: typing.Optional[builtins.str] = None,
            iam_auth: typing.Optional[builtins.str] = None,
            secret_arn: typing.Optional[builtins.str] = None,
            user_name: typing.Optional[builtins.str] = None,
        ) -> None:
            """
            :param auth_scheme: ``CfnDBProxy.AuthFormatProperty.AuthScheme``.
            :param description: ``CfnDBProxy.AuthFormatProperty.Description``.
            :param iam_auth: ``CfnDBProxy.AuthFormatProperty.IAMAuth``.
            :param secret_arn: ``CfnDBProxy.AuthFormatProperty.SecretArn``.
            :param user_name: ``CfnDBProxy.AuthFormatProperty.UserName``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxy-authformat.html
            """
            self._values: typing.Dict[str, typing.Any] = {}
            if auth_scheme is not None:
                self._values["auth_scheme"] = auth_scheme
            if description is not None:
                self._values["description"] = description
            if iam_auth is not None:
                self._values["iam_auth"] = iam_auth
            if secret_arn is not None:
                self._values["secret_arn"] = secret_arn
            if user_name is not None:
                self._values["user_name"] = user_name

        @builtins.property
        def auth_scheme(self) -> typing.Optional[builtins.str]:
            """``CfnDBProxy.AuthFormatProperty.AuthScheme``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxy-authformat.html#cfn-rds-dbproxy-authformat-authscheme
            """
            result = self._values.get("auth_scheme")
            return result

        @builtins.property
        def description(self) -> typing.Optional[builtins.str]:
            """``CfnDBProxy.AuthFormatProperty.Description``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxy-authformat.html#cfn-rds-dbproxy-authformat-description
            """
            result = self._values.get("description")
            return result

        @builtins.property
        def iam_auth(self) -> typing.Optional[builtins.str]:
            """``CfnDBProxy.AuthFormatProperty.IAMAuth``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxy-authformat.html#cfn-rds-dbproxy-authformat-iamauth
            """
            result = self._values.get("iam_auth")
            return result

        @builtins.property
        def secret_arn(self) -> typing.Optional[builtins.str]:
            """``CfnDBProxy.AuthFormatProperty.SecretArn``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxy-authformat.html#cfn-rds-dbproxy-authformat-secretarn
            """
            result = self._values.get("secret_arn")
            return result

        @builtins.property
        def user_name(self) -> typing.Optional[builtins.str]:
            """``CfnDBProxy.AuthFormatProperty.UserName``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxy-authformat.html#cfn-rds-dbproxy-authformat-username
            """
            result = self._values.get("user_name")
            return result

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AuthFormatProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-rds.CfnDBProxy.TagFormatProperty",
        jsii_struct_bases=[],
        name_mapping={"key": "key", "value": "value"},
    )
    class TagFormatProperty:
        def __init__(
            self,
            *,
            key: typing.Optional[builtins.str] = None,
            value: typing.Optional[builtins.str] = None,
        ) -> None:
            """
            :param key: ``CfnDBProxy.TagFormatProperty.Key``.
            :param value: ``CfnDBProxy.TagFormatProperty.Value``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxy-tagformat.html
            """
            self._values: typing.Dict[str, typing.Any] = {}
            if key is not None:
                self._values["key"] = key
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def key(self) -> typing.Optional[builtins.str]:
            """``CfnDBProxy.TagFormatProperty.Key``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxy-tagformat.html#cfn-rds-dbproxy-tagformat-key
            """
            result = self._values.get("key")
            return result

        @builtins.property
        def value(self) -> typing.Optional[builtins.str]:
            """``CfnDBProxy.TagFormatProperty.Value``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxy-tagformat.html#cfn-rds-dbproxy-tagformat-value
            """
            result = self._values.get("value")
            return result

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TagFormatProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.CfnDBProxyProps",
    jsii_struct_bases=[],
    name_mapping={
        "auth": "auth",
        "db_proxy_name": "dbProxyName",
        "engine_family": "engineFamily",
        "role_arn": "roleArn",
        "vpc_subnet_ids": "vpcSubnetIds",
        "debug_logging": "debugLogging",
        "idle_client_timeout": "idleClientTimeout",
        "require_tls": "requireTls",
        "tags": "tags",
        "vpc_security_group_ids": "vpcSecurityGroupIds",
    },
)
class CfnDBProxyProps:
    def __init__(
        self,
        *,
        auth: typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[CfnDBProxy.AuthFormatProperty, aws_cdk.core.IResolvable]]],
        db_proxy_name: builtins.str,
        engine_family: builtins.str,
        role_arn: builtins.str,
        vpc_subnet_ids: typing.List[builtins.str],
        debug_logging: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        idle_client_timeout: typing.Optional[jsii.Number] = None,
        require_tls: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        tags: typing.Optional[typing.List[CfnDBProxy.TagFormatProperty]] = None,
        vpc_security_group_ids: typing.Optional[typing.List[builtins.str]] = None,
    ) -> None:
        """Properties for defining a ``AWS::RDS::DBProxy``.

        :param auth: ``AWS::RDS::DBProxy.Auth``.
        :param db_proxy_name: ``AWS::RDS::DBProxy.DBProxyName``.
        :param engine_family: ``AWS::RDS::DBProxy.EngineFamily``.
        :param role_arn: ``AWS::RDS::DBProxy.RoleArn``.
        :param vpc_subnet_ids: ``AWS::RDS::DBProxy.VpcSubnetIds``.
        :param debug_logging: ``AWS::RDS::DBProxy.DebugLogging``.
        :param idle_client_timeout: ``AWS::RDS::DBProxy.IdleClientTimeout``.
        :param require_tls: ``AWS::RDS::DBProxy.RequireTLS``.
        :param tags: ``AWS::RDS::DBProxy.Tags``.
        :param vpc_security_group_ids: ``AWS::RDS::DBProxy.VpcSecurityGroupIds``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html
        """
        self._values: typing.Dict[str, typing.Any] = {
            "auth": auth,
            "db_proxy_name": db_proxy_name,
            "engine_family": engine_family,
            "role_arn": role_arn,
            "vpc_subnet_ids": vpc_subnet_ids,
        }
        if debug_logging is not None:
            self._values["debug_logging"] = debug_logging
        if idle_client_timeout is not None:
            self._values["idle_client_timeout"] = idle_client_timeout
        if require_tls is not None:
            self._values["require_tls"] = require_tls
        if tags is not None:
            self._values["tags"] = tags
        if vpc_security_group_ids is not None:
            self._values["vpc_security_group_ids"] = vpc_security_group_ids

    @builtins.property
    def auth(
        self,
    ) -> typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[CfnDBProxy.AuthFormatProperty, aws_cdk.core.IResolvable]]]:
        """``AWS::RDS::DBProxy.Auth``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-auth
        """
        result = self._values.get("auth")
        assert result is not None, "Required property 'auth' is missing"
        return result

    @builtins.property
    def db_proxy_name(self) -> builtins.str:
        """``AWS::RDS::DBProxy.DBProxyName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-dbproxyname
        """
        result = self._values.get("db_proxy_name")
        assert result is not None, "Required property 'db_proxy_name' is missing"
        return result

    @builtins.property
    def engine_family(self) -> builtins.str:
        """``AWS::RDS::DBProxy.EngineFamily``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-enginefamily
        """
        result = self._values.get("engine_family")
        assert result is not None, "Required property 'engine_family' is missing"
        return result

    @builtins.property
    def role_arn(self) -> builtins.str:
        """``AWS::RDS::DBProxy.RoleArn``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-rolearn
        """
        result = self._values.get("role_arn")
        assert result is not None, "Required property 'role_arn' is missing"
        return result

    @builtins.property
    def vpc_subnet_ids(self) -> typing.List[builtins.str]:
        """``AWS::RDS::DBProxy.VpcSubnetIds``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-vpcsubnetids
        """
        result = self._values.get("vpc_subnet_ids")
        assert result is not None, "Required property 'vpc_subnet_ids' is missing"
        return result

    @builtins.property
    def debug_logging(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBProxy.DebugLogging``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-debuglogging
        """
        result = self._values.get("debug_logging")
        return result

    @builtins.property
    def idle_client_timeout(self) -> typing.Optional[jsii.Number]:
        """``AWS::RDS::DBProxy.IdleClientTimeout``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-idleclienttimeout
        """
        result = self._values.get("idle_client_timeout")
        return result

    @builtins.property
    def require_tls(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::DBProxy.RequireTLS``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-requiretls
        """
        result = self._values.get("require_tls")
        return result

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[CfnDBProxy.TagFormatProperty]]:
        """``AWS::RDS::DBProxy.Tags``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-tags
        """
        result = self._values.get("tags")
        return result

    @builtins.property
    def vpc_security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        """``AWS::RDS::DBProxy.VpcSecurityGroupIds``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-vpcsecuritygroupids
        """
        result = self._values.get("vpc_security_group_ids")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDBProxyProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnDBProxyTargetGroup(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.CfnDBProxyTargetGroup",
):
    """A CloudFormation ``AWS::RDS::DBProxyTargetGroup``.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html
    :cloudformationResource: AWS::RDS::DBProxyTargetGroup
    """

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        db_proxy_name: builtins.str,
        target_group_name: builtins.str,
        connection_pool_configuration_info: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty"]] = None,
        db_cluster_identifiers: typing.Optional[typing.List[builtins.str]] = None,
        db_instance_identifiers: typing.Optional[typing.List[builtins.str]] = None,
    ) -> None:
        """Create a new ``AWS::RDS::DBProxyTargetGroup``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param db_proxy_name: ``AWS::RDS::DBProxyTargetGroup.DBProxyName``.
        :param target_group_name: ``AWS::RDS::DBProxyTargetGroup.TargetGroupName``.
        :param connection_pool_configuration_info: ``AWS::RDS::DBProxyTargetGroup.ConnectionPoolConfigurationInfo``.
        :param db_cluster_identifiers: ``AWS::RDS::DBProxyTargetGroup.DBClusterIdentifiers``.
        :param db_instance_identifiers: ``AWS::RDS::DBProxyTargetGroup.DBInstanceIdentifiers``.
        """
        props = CfnDBProxyTargetGroupProps(
            db_proxy_name=db_proxy_name,
            target_group_name=target_group_name,
            connection_pool_configuration_info=connection_pool_configuration_info,
            db_cluster_identifiers=db_cluster_identifiers,
            db_instance_identifiers=db_instance_identifiers,
        )

        jsii.create(CfnDBProxyTargetGroup, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        """(experimental) Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.

        :stability: experimental
        """
        return jsii.invoke(self, "inspect", [inspector])

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        """
        :param props: -
        """
        return jsii.invoke(self, "renderProperties", [props])

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        """The CloudFormation resource type name for this resource class."""
        return jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="attrTargetGroupArn")
    def attr_target_group_arn(self) -> builtins.str:
        """
        :cloudformationAttribute: TargetGroupArn
        """
        return jsii.get(self, "attrTargetGroupArn")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return jsii.get(self, "cfnProperties")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="dbProxyName")
    def db_proxy_name(self) -> builtins.str:
        """``AWS::RDS::DBProxyTargetGroup.DBProxyName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#cfn-rds-dbproxytargetgroup-dbproxyname
        """
        return jsii.get(self, "dbProxyName")

    @db_proxy_name.setter # type: ignore
    def db_proxy_name(self, value: builtins.str) -> None:
        jsii.set(self, "dbProxyName", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="targetGroupName")
    def target_group_name(self) -> builtins.str:
        """``AWS::RDS::DBProxyTargetGroup.TargetGroupName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#cfn-rds-dbproxytargetgroup-targetgroupname
        """
        return jsii.get(self, "targetGroupName")

    @target_group_name.setter # type: ignore
    def target_group_name(self, value: builtins.str) -> None:
        jsii.set(self, "targetGroupName", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="connectionPoolConfigurationInfo")
    def connection_pool_configuration_info(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty"]]:
        """``AWS::RDS::DBProxyTargetGroup.ConnectionPoolConfigurationInfo``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#cfn-rds-dbproxytargetgroup-connectionpoolconfigurationinfo
        """
        return jsii.get(self, "connectionPoolConfigurationInfo")

    @connection_pool_configuration_info.setter # type: ignore
    def connection_pool_configuration_info(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty"]],
    ) -> None:
        jsii.set(self, "connectionPoolConfigurationInfo", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="dbClusterIdentifiers")
    def db_cluster_identifiers(self) -> typing.Optional[typing.List[builtins.str]]:
        """``AWS::RDS::DBProxyTargetGroup.DBClusterIdentifiers``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#cfn-rds-dbproxytargetgroup-dbclusteridentifiers
        """
        return jsii.get(self, "dbClusterIdentifiers")

    @db_cluster_identifiers.setter # type: ignore
    def db_cluster_identifiers(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        jsii.set(self, "dbClusterIdentifiers", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="dbInstanceIdentifiers")
    def db_instance_identifiers(self) -> typing.Optional[typing.List[builtins.str]]:
        """``AWS::RDS::DBProxyTargetGroup.DBInstanceIdentifiers``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#cfn-rds-dbproxytargetgroup-dbinstanceidentifiers
        """
        return jsii.get(self, "dbInstanceIdentifiers")

    @db_instance_identifiers.setter # type: ignore
    def db_instance_identifiers(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        jsii.set(self, "dbInstanceIdentifiers", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-rds.CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty",
        jsii_struct_bases=[],
        name_mapping={
            "connection_borrow_timeout": "connectionBorrowTimeout",
            "init_query": "initQuery",
            "max_connections_percent": "maxConnectionsPercent",
            "max_idle_connections_percent": "maxIdleConnectionsPercent",
            "session_pinning_filters": "sessionPinningFilters",
        },
    )
    class ConnectionPoolConfigurationInfoFormatProperty:
        def __init__(
            self,
            *,
            connection_borrow_timeout: typing.Optional[jsii.Number] = None,
            init_query: typing.Optional[builtins.str] = None,
            max_connections_percent: typing.Optional[jsii.Number] = None,
            max_idle_connections_percent: typing.Optional[jsii.Number] = None,
            session_pinning_filters: typing.Optional[typing.List[builtins.str]] = None,
        ) -> None:
            """
            :param connection_borrow_timeout: ``CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty.ConnectionBorrowTimeout``.
            :param init_query: ``CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty.InitQuery``.
            :param max_connections_percent: ``CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty.MaxConnectionsPercent``.
            :param max_idle_connections_percent: ``CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty.MaxIdleConnectionsPercent``.
            :param session_pinning_filters: ``CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty.SessionPinningFilters``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat.html
            """
            self._values: typing.Dict[str, typing.Any] = {}
            if connection_borrow_timeout is not None:
                self._values["connection_borrow_timeout"] = connection_borrow_timeout
            if init_query is not None:
                self._values["init_query"] = init_query
            if max_connections_percent is not None:
                self._values["max_connections_percent"] = max_connections_percent
            if max_idle_connections_percent is not None:
                self._values["max_idle_connections_percent"] = max_idle_connections_percent
            if session_pinning_filters is not None:
                self._values["session_pinning_filters"] = session_pinning_filters

        @builtins.property
        def connection_borrow_timeout(self) -> typing.Optional[jsii.Number]:
            """``CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty.ConnectionBorrowTimeout``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat.html#cfn-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat-connectionborrowtimeout
            """
            result = self._values.get("connection_borrow_timeout")
            return result

        @builtins.property
        def init_query(self) -> typing.Optional[builtins.str]:
            """``CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty.InitQuery``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat.html#cfn-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat-initquery
            """
            result = self._values.get("init_query")
            return result

        @builtins.property
        def max_connections_percent(self) -> typing.Optional[jsii.Number]:
            """``CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty.MaxConnectionsPercent``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat.html#cfn-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat-maxconnectionspercent
            """
            result = self._values.get("max_connections_percent")
            return result

        @builtins.property
        def max_idle_connections_percent(self) -> typing.Optional[jsii.Number]:
            """``CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty.MaxIdleConnectionsPercent``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat.html#cfn-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat-maxidleconnectionspercent
            """
            result = self._values.get("max_idle_connections_percent")
            return result

        @builtins.property
        def session_pinning_filters(self) -> typing.Optional[typing.List[builtins.str]]:
            """``CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty.SessionPinningFilters``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat.html#cfn-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat-sessionpinningfilters
            """
            result = self._values.get("session_pinning_filters")
            return result

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ConnectionPoolConfigurationInfoFormatProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.CfnDBProxyTargetGroupProps",
    jsii_struct_bases=[],
    name_mapping={
        "db_proxy_name": "dbProxyName",
        "target_group_name": "targetGroupName",
        "connection_pool_configuration_info": "connectionPoolConfigurationInfo",
        "db_cluster_identifiers": "dbClusterIdentifiers",
        "db_instance_identifiers": "dbInstanceIdentifiers",
    },
)
class CfnDBProxyTargetGroupProps:
    def __init__(
        self,
        *,
        db_proxy_name: builtins.str,
        target_group_name: builtins.str,
        connection_pool_configuration_info: typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty]] = None,
        db_cluster_identifiers: typing.Optional[typing.List[builtins.str]] = None,
        db_instance_identifiers: typing.Optional[typing.List[builtins.str]] = None,
    ) -> None:
        """Properties for defining a ``AWS::RDS::DBProxyTargetGroup``.

        :param db_proxy_name: ``AWS::RDS::DBProxyTargetGroup.DBProxyName``.
        :param target_group_name: ``AWS::RDS::DBProxyTargetGroup.TargetGroupName``.
        :param connection_pool_configuration_info: ``AWS::RDS::DBProxyTargetGroup.ConnectionPoolConfigurationInfo``.
        :param db_cluster_identifiers: ``AWS::RDS::DBProxyTargetGroup.DBClusterIdentifiers``.
        :param db_instance_identifiers: ``AWS::RDS::DBProxyTargetGroup.DBInstanceIdentifiers``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html
        """
        self._values: typing.Dict[str, typing.Any] = {
            "db_proxy_name": db_proxy_name,
            "target_group_name": target_group_name,
        }
        if connection_pool_configuration_info is not None:
            self._values["connection_pool_configuration_info"] = connection_pool_configuration_info
        if db_cluster_identifiers is not None:
            self._values["db_cluster_identifiers"] = db_cluster_identifiers
        if db_instance_identifiers is not None:
            self._values["db_instance_identifiers"] = db_instance_identifiers

    @builtins.property
    def db_proxy_name(self) -> builtins.str:
        """``AWS::RDS::DBProxyTargetGroup.DBProxyName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#cfn-rds-dbproxytargetgroup-dbproxyname
        """
        result = self._values.get("db_proxy_name")
        assert result is not None, "Required property 'db_proxy_name' is missing"
        return result

    @builtins.property
    def target_group_name(self) -> builtins.str:
        """``AWS::RDS::DBProxyTargetGroup.TargetGroupName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#cfn-rds-dbproxytargetgroup-targetgroupname
        """
        result = self._values.get("target_group_name")
        assert result is not None, "Required property 'target_group_name' is missing"
        return result

    @builtins.property
    def connection_pool_configuration_info(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty]]:
        """``AWS::RDS::DBProxyTargetGroup.ConnectionPoolConfigurationInfo``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#cfn-rds-dbproxytargetgroup-connectionpoolconfigurationinfo
        """
        result = self._values.get("connection_pool_configuration_info")
        return result

    @builtins.property
    def db_cluster_identifiers(self) -> typing.Optional[typing.List[builtins.str]]:
        """``AWS::RDS::DBProxyTargetGroup.DBClusterIdentifiers``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#cfn-rds-dbproxytargetgroup-dbclusteridentifiers
        """
        result = self._values.get("db_cluster_identifiers")
        return result

    @builtins.property
    def db_instance_identifiers(self) -> typing.Optional[typing.List[builtins.str]]:
        """``AWS::RDS::DBProxyTargetGroup.DBInstanceIdentifiers``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#cfn-rds-dbproxytargetgroup-dbinstanceidentifiers
        """
        result = self._values.get("db_instance_identifiers")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDBProxyTargetGroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnDBSecurityGroup(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.CfnDBSecurityGroup",
):
    """A CloudFormation ``AWS::RDS::DBSecurityGroup``.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group.html
    :cloudformationResource: AWS::RDS::DBSecurityGroup
    """

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        db_security_group_ingress: typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnDBSecurityGroup.IngressProperty"]]],
        group_description: builtins.str,
        ec2_vpc_id: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.List[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        """Create a new ``AWS::RDS::DBSecurityGroup``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param db_security_group_ingress: ``AWS::RDS::DBSecurityGroup.DBSecurityGroupIngress``.
        :param group_description: ``AWS::RDS::DBSecurityGroup.GroupDescription``.
        :param ec2_vpc_id: ``AWS::RDS::DBSecurityGroup.EC2VpcId``.
        :param tags: ``AWS::RDS::DBSecurityGroup.Tags``.
        """
        props = CfnDBSecurityGroupProps(
            db_security_group_ingress=db_security_group_ingress,
            group_description=group_description,
            ec2_vpc_id=ec2_vpc_id,
            tags=tags,
        )

        jsii.create(CfnDBSecurityGroup, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        """(experimental) Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.

        :stability: experimental
        """
        return jsii.invoke(self, "inspect", [inspector])

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        """
        :param props: -
        """
        return jsii.invoke(self, "renderProperties", [props])

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        """The CloudFormation resource type name for this resource class."""
        return jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return jsii.get(self, "cfnProperties")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        """``AWS::RDS::DBSecurityGroup.Tags``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group.html#cfn-rds-dbsecuritygroup-tags
        """
        return jsii.get(self, "tags")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="dbSecurityGroupIngress")
    def db_security_group_ingress(
        self,
    ) -> typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnDBSecurityGroup.IngressProperty"]]]:
        """``AWS::RDS::DBSecurityGroup.DBSecurityGroupIngress``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group.html#cfn-rds-dbsecuritygroup-dbsecuritygroupingress
        """
        return jsii.get(self, "dbSecurityGroupIngress")

    @db_security_group_ingress.setter # type: ignore
    def db_security_group_ingress(
        self,
        value: typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnDBSecurityGroup.IngressProperty"]]],
    ) -> None:
        jsii.set(self, "dbSecurityGroupIngress", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="groupDescription")
    def group_description(self) -> builtins.str:
        """``AWS::RDS::DBSecurityGroup.GroupDescription``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group.html#cfn-rds-dbsecuritygroup-groupdescription
        """
        return jsii.get(self, "groupDescription")

    @group_description.setter # type: ignore
    def group_description(self, value: builtins.str) -> None:
        jsii.set(self, "groupDescription", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="ec2VpcId")
    def ec2_vpc_id(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBSecurityGroup.EC2VpcId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group.html#cfn-rds-dbsecuritygroup-ec2vpcid
        """
        return jsii.get(self, "ec2VpcId")

    @ec2_vpc_id.setter # type: ignore
    def ec2_vpc_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "ec2VpcId", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-rds.CfnDBSecurityGroup.IngressProperty",
        jsii_struct_bases=[],
        name_mapping={
            "cidrip": "cidrip",
            "ec2_security_group_id": "ec2SecurityGroupId",
            "ec2_security_group_name": "ec2SecurityGroupName",
            "ec2_security_group_owner_id": "ec2SecurityGroupOwnerId",
        },
    )
    class IngressProperty:
        def __init__(
            self,
            *,
            cidrip: typing.Optional[builtins.str] = None,
            ec2_security_group_id: typing.Optional[builtins.str] = None,
            ec2_security_group_name: typing.Optional[builtins.str] = None,
            ec2_security_group_owner_id: typing.Optional[builtins.str] = None,
        ) -> None:
            """
            :param cidrip: ``CfnDBSecurityGroup.IngressProperty.CIDRIP``.
            :param ec2_security_group_id: ``CfnDBSecurityGroup.IngressProperty.EC2SecurityGroupId``.
            :param ec2_security_group_name: ``CfnDBSecurityGroup.IngressProperty.EC2SecurityGroupName``.
            :param ec2_security_group_owner_id: ``CfnDBSecurityGroup.IngressProperty.EC2SecurityGroupOwnerId``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group-rule.html
            """
            self._values: typing.Dict[str, typing.Any] = {}
            if cidrip is not None:
                self._values["cidrip"] = cidrip
            if ec2_security_group_id is not None:
                self._values["ec2_security_group_id"] = ec2_security_group_id
            if ec2_security_group_name is not None:
                self._values["ec2_security_group_name"] = ec2_security_group_name
            if ec2_security_group_owner_id is not None:
                self._values["ec2_security_group_owner_id"] = ec2_security_group_owner_id

        @builtins.property
        def cidrip(self) -> typing.Optional[builtins.str]:
            """``CfnDBSecurityGroup.IngressProperty.CIDRIP``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group-rule.html#cfn-rds-securitygroup-cidrip
            """
            result = self._values.get("cidrip")
            return result

        @builtins.property
        def ec2_security_group_id(self) -> typing.Optional[builtins.str]:
            """``CfnDBSecurityGroup.IngressProperty.EC2SecurityGroupId``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group-rule.html#cfn-rds-securitygroup-ec2securitygroupid
            """
            result = self._values.get("ec2_security_group_id")
            return result

        @builtins.property
        def ec2_security_group_name(self) -> typing.Optional[builtins.str]:
            """``CfnDBSecurityGroup.IngressProperty.EC2SecurityGroupName``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group-rule.html#cfn-rds-securitygroup-ec2securitygroupname
            """
            result = self._values.get("ec2_security_group_name")
            return result

        @builtins.property
        def ec2_security_group_owner_id(self) -> typing.Optional[builtins.str]:
            """``CfnDBSecurityGroup.IngressProperty.EC2SecurityGroupOwnerId``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group-rule.html#cfn-rds-securitygroup-ec2securitygroupownerid
            """
            result = self._values.get("ec2_security_group_owner_id")
            return result

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "IngressProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnDBSecurityGroupIngress(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.CfnDBSecurityGroupIngress",
):
    """A CloudFormation ``AWS::RDS::DBSecurityGroupIngress``.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html
    :cloudformationResource: AWS::RDS::DBSecurityGroupIngress
    """

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        db_security_group_name: builtins.str,
        cidrip: typing.Optional[builtins.str] = None,
        ec2_security_group_id: typing.Optional[builtins.str] = None,
        ec2_security_group_name: typing.Optional[builtins.str] = None,
        ec2_security_group_owner_id: typing.Optional[builtins.str] = None,
    ) -> None:
        """Create a new ``AWS::RDS::DBSecurityGroupIngress``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param db_security_group_name: ``AWS::RDS::DBSecurityGroupIngress.DBSecurityGroupName``.
        :param cidrip: ``AWS::RDS::DBSecurityGroupIngress.CIDRIP``.
        :param ec2_security_group_id: ``AWS::RDS::DBSecurityGroupIngress.EC2SecurityGroupId``.
        :param ec2_security_group_name: ``AWS::RDS::DBSecurityGroupIngress.EC2SecurityGroupName``.
        :param ec2_security_group_owner_id: ``AWS::RDS::DBSecurityGroupIngress.EC2SecurityGroupOwnerId``.
        """
        props = CfnDBSecurityGroupIngressProps(
            db_security_group_name=db_security_group_name,
            cidrip=cidrip,
            ec2_security_group_id=ec2_security_group_id,
            ec2_security_group_name=ec2_security_group_name,
            ec2_security_group_owner_id=ec2_security_group_owner_id,
        )

        jsii.create(CfnDBSecurityGroupIngress, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        """(experimental) Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.

        :stability: experimental
        """
        return jsii.invoke(self, "inspect", [inspector])

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        """
        :param props: -
        """
        return jsii.invoke(self, "renderProperties", [props])

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        """The CloudFormation resource type name for this resource class."""
        return jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return jsii.get(self, "cfnProperties")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="dbSecurityGroupName")
    def db_security_group_name(self) -> builtins.str:
        """``AWS::RDS::DBSecurityGroupIngress.DBSecurityGroupName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html#cfn-rds-securitygroup-ingress-dbsecuritygroupname
        """
        return jsii.get(self, "dbSecurityGroupName")

    @db_security_group_name.setter # type: ignore
    def db_security_group_name(self, value: builtins.str) -> None:
        jsii.set(self, "dbSecurityGroupName", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="cidrip")
    def cidrip(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBSecurityGroupIngress.CIDRIP``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html#cfn-rds-securitygroup-ingress-cidrip
        """
        return jsii.get(self, "cidrip")

    @cidrip.setter # type: ignore
    def cidrip(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "cidrip", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="ec2SecurityGroupId")
    def ec2_security_group_id(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBSecurityGroupIngress.EC2SecurityGroupId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html#cfn-rds-securitygroup-ingress-ec2securitygroupid
        """
        return jsii.get(self, "ec2SecurityGroupId")

    @ec2_security_group_id.setter # type: ignore
    def ec2_security_group_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "ec2SecurityGroupId", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="ec2SecurityGroupName")
    def ec2_security_group_name(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBSecurityGroupIngress.EC2SecurityGroupName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html#cfn-rds-securitygroup-ingress-ec2securitygroupname
        """
        return jsii.get(self, "ec2SecurityGroupName")

    @ec2_security_group_name.setter # type: ignore
    def ec2_security_group_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "ec2SecurityGroupName", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="ec2SecurityGroupOwnerId")
    def ec2_security_group_owner_id(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBSecurityGroupIngress.EC2SecurityGroupOwnerId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html#cfn-rds-securitygroup-ingress-ec2securitygroupownerid
        """
        return jsii.get(self, "ec2SecurityGroupOwnerId")

    @ec2_security_group_owner_id.setter # type: ignore
    def ec2_security_group_owner_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "ec2SecurityGroupOwnerId", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.CfnDBSecurityGroupIngressProps",
    jsii_struct_bases=[],
    name_mapping={
        "db_security_group_name": "dbSecurityGroupName",
        "cidrip": "cidrip",
        "ec2_security_group_id": "ec2SecurityGroupId",
        "ec2_security_group_name": "ec2SecurityGroupName",
        "ec2_security_group_owner_id": "ec2SecurityGroupOwnerId",
    },
)
class CfnDBSecurityGroupIngressProps:
    def __init__(
        self,
        *,
        db_security_group_name: builtins.str,
        cidrip: typing.Optional[builtins.str] = None,
        ec2_security_group_id: typing.Optional[builtins.str] = None,
        ec2_security_group_name: typing.Optional[builtins.str] = None,
        ec2_security_group_owner_id: typing.Optional[builtins.str] = None,
    ) -> None:
        """Properties for defining a ``AWS::RDS::DBSecurityGroupIngress``.

        :param db_security_group_name: ``AWS::RDS::DBSecurityGroupIngress.DBSecurityGroupName``.
        :param cidrip: ``AWS::RDS::DBSecurityGroupIngress.CIDRIP``.
        :param ec2_security_group_id: ``AWS::RDS::DBSecurityGroupIngress.EC2SecurityGroupId``.
        :param ec2_security_group_name: ``AWS::RDS::DBSecurityGroupIngress.EC2SecurityGroupName``.
        :param ec2_security_group_owner_id: ``AWS::RDS::DBSecurityGroupIngress.EC2SecurityGroupOwnerId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html
        """
        self._values: typing.Dict[str, typing.Any] = {
            "db_security_group_name": db_security_group_name,
        }
        if cidrip is not None:
            self._values["cidrip"] = cidrip
        if ec2_security_group_id is not None:
            self._values["ec2_security_group_id"] = ec2_security_group_id
        if ec2_security_group_name is not None:
            self._values["ec2_security_group_name"] = ec2_security_group_name
        if ec2_security_group_owner_id is not None:
            self._values["ec2_security_group_owner_id"] = ec2_security_group_owner_id

    @builtins.property
    def db_security_group_name(self) -> builtins.str:
        """``AWS::RDS::DBSecurityGroupIngress.DBSecurityGroupName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html#cfn-rds-securitygroup-ingress-dbsecuritygroupname
        """
        result = self._values.get("db_security_group_name")
        assert result is not None, "Required property 'db_security_group_name' is missing"
        return result

    @builtins.property
    def cidrip(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBSecurityGroupIngress.CIDRIP``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html#cfn-rds-securitygroup-ingress-cidrip
        """
        result = self._values.get("cidrip")
        return result

    @builtins.property
    def ec2_security_group_id(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBSecurityGroupIngress.EC2SecurityGroupId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html#cfn-rds-securitygroup-ingress-ec2securitygroupid
        """
        result = self._values.get("ec2_security_group_id")
        return result

    @builtins.property
    def ec2_security_group_name(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBSecurityGroupIngress.EC2SecurityGroupName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html#cfn-rds-securitygroup-ingress-ec2securitygroupname
        """
        result = self._values.get("ec2_security_group_name")
        return result

    @builtins.property
    def ec2_security_group_owner_id(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBSecurityGroupIngress.EC2SecurityGroupOwnerId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html#cfn-rds-securitygroup-ingress-ec2securitygroupownerid
        """
        result = self._values.get("ec2_security_group_owner_id")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDBSecurityGroupIngressProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.CfnDBSecurityGroupProps",
    jsii_struct_bases=[],
    name_mapping={
        "db_security_group_ingress": "dbSecurityGroupIngress",
        "group_description": "groupDescription",
        "ec2_vpc_id": "ec2VpcId",
        "tags": "tags",
    },
)
class CfnDBSecurityGroupProps:
    def __init__(
        self,
        *,
        db_security_group_ingress: typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnDBSecurityGroup.IngressProperty]]],
        group_description: builtins.str,
        ec2_vpc_id: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.List[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        """Properties for defining a ``AWS::RDS::DBSecurityGroup``.

        :param db_security_group_ingress: ``AWS::RDS::DBSecurityGroup.DBSecurityGroupIngress``.
        :param group_description: ``AWS::RDS::DBSecurityGroup.GroupDescription``.
        :param ec2_vpc_id: ``AWS::RDS::DBSecurityGroup.EC2VpcId``.
        :param tags: ``AWS::RDS::DBSecurityGroup.Tags``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group.html
        """
        self._values: typing.Dict[str, typing.Any] = {
            "db_security_group_ingress": db_security_group_ingress,
            "group_description": group_description,
        }
        if ec2_vpc_id is not None:
            self._values["ec2_vpc_id"] = ec2_vpc_id
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def db_security_group_ingress(
        self,
    ) -> typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnDBSecurityGroup.IngressProperty]]]:
        """``AWS::RDS::DBSecurityGroup.DBSecurityGroupIngress``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group.html#cfn-rds-dbsecuritygroup-dbsecuritygroupingress
        """
        result = self._values.get("db_security_group_ingress")
        assert result is not None, "Required property 'db_security_group_ingress' is missing"
        return result

    @builtins.property
    def group_description(self) -> builtins.str:
        """``AWS::RDS::DBSecurityGroup.GroupDescription``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group.html#cfn-rds-dbsecuritygroup-groupdescription
        """
        result = self._values.get("group_description")
        assert result is not None, "Required property 'group_description' is missing"
        return result

    @builtins.property
    def ec2_vpc_id(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBSecurityGroup.EC2VpcId``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group.html#cfn-rds-dbsecuritygroup-ec2vpcid
        """
        result = self._values.get("ec2_vpc_id")
        return result

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        """``AWS::RDS::DBSecurityGroup.Tags``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group.html#cfn-rds-dbsecuritygroup-tags
        """
        result = self._values.get("tags")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDBSecurityGroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnDBSubnetGroup(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.CfnDBSubnetGroup",
):
    """A CloudFormation ``AWS::RDS::DBSubnetGroup``.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbsubnet-group.html
    :cloudformationResource: AWS::RDS::DBSubnetGroup
    """

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        db_subnet_group_description: builtins.str,
        subnet_ids: typing.List[builtins.str],
        db_subnet_group_name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.List[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        """Create a new ``AWS::RDS::DBSubnetGroup``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param db_subnet_group_description: ``AWS::RDS::DBSubnetGroup.DBSubnetGroupDescription``.
        :param subnet_ids: ``AWS::RDS::DBSubnetGroup.SubnetIds``.
        :param db_subnet_group_name: ``AWS::RDS::DBSubnetGroup.DBSubnetGroupName``.
        :param tags: ``AWS::RDS::DBSubnetGroup.Tags``.
        """
        props = CfnDBSubnetGroupProps(
            db_subnet_group_description=db_subnet_group_description,
            subnet_ids=subnet_ids,
            db_subnet_group_name=db_subnet_group_name,
            tags=tags,
        )

        jsii.create(CfnDBSubnetGroup, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        """(experimental) Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.

        :stability: experimental
        """
        return jsii.invoke(self, "inspect", [inspector])

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        """
        :param props: -
        """
        return jsii.invoke(self, "renderProperties", [props])

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        """The CloudFormation resource type name for this resource class."""
        return jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return jsii.get(self, "cfnProperties")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        """``AWS::RDS::DBSubnetGroup.Tags``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbsubnet-group.html#cfn-rds-dbsubnetgroup-tags
        """
        return jsii.get(self, "tags")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="dbSubnetGroupDescription")
    def db_subnet_group_description(self) -> builtins.str:
        """``AWS::RDS::DBSubnetGroup.DBSubnetGroupDescription``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbsubnet-group.html#cfn-rds-dbsubnetgroup-dbsubnetgroupdescription
        """
        return jsii.get(self, "dbSubnetGroupDescription")

    @db_subnet_group_description.setter # type: ignore
    def db_subnet_group_description(self, value: builtins.str) -> None:
        jsii.set(self, "dbSubnetGroupDescription", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="subnetIds")
    def subnet_ids(self) -> typing.List[builtins.str]:
        """``AWS::RDS::DBSubnetGroup.SubnetIds``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbsubnet-group.html#cfn-rds-dbsubnetgroup-subnetids
        """
        return jsii.get(self, "subnetIds")

    @subnet_ids.setter # type: ignore
    def subnet_ids(self, value: typing.List[builtins.str]) -> None:
        jsii.set(self, "subnetIds", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="dbSubnetGroupName")
    def db_subnet_group_name(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBSubnetGroup.DBSubnetGroupName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbsubnet-group.html#cfn-rds-dbsubnetgroup-dbsubnetgroupname
        """
        return jsii.get(self, "dbSubnetGroupName")

    @db_subnet_group_name.setter # type: ignore
    def db_subnet_group_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "dbSubnetGroupName", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.CfnDBSubnetGroupProps",
    jsii_struct_bases=[],
    name_mapping={
        "db_subnet_group_description": "dbSubnetGroupDescription",
        "subnet_ids": "subnetIds",
        "db_subnet_group_name": "dbSubnetGroupName",
        "tags": "tags",
    },
)
class CfnDBSubnetGroupProps:
    def __init__(
        self,
        *,
        db_subnet_group_description: builtins.str,
        subnet_ids: typing.List[builtins.str],
        db_subnet_group_name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.List[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        """Properties for defining a ``AWS::RDS::DBSubnetGroup``.

        :param db_subnet_group_description: ``AWS::RDS::DBSubnetGroup.DBSubnetGroupDescription``.
        :param subnet_ids: ``AWS::RDS::DBSubnetGroup.SubnetIds``.
        :param db_subnet_group_name: ``AWS::RDS::DBSubnetGroup.DBSubnetGroupName``.
        :param tags: ``AWS::RDS::DBSubnetGroup.Tags``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbsubnet-group.html
        """
        self._values: typing.Dict[str, typing.Any] = {
            "db_subnet_group_description": db_subnet_group_description,
            "subnet_ids": subnet_ids,
        }
        if db_subnet_group_name is not None:
            self._values["db_subnet_group_name"] = db_subnet_group_name
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def db_subnet_group_description(self) -> builtins.str:
        """``AWS::RDS::DBSubnetGroup.DBSubnetGroupDescription``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbsubnet-group.html#cfn-rds-dbsubnetgroup-dbsubnetgroupdescription
        """
        result = self._values.get("db_subnet_group_description")
        assert result is not None, "Required property 'db_subnet_group_description' is missing"
        return result

    @builtins.property
    def subnet_ids(self) -> typing.List[builtins.str]:
        """``AWS::RDS::DBSubnetGroup.SubnetIds``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbsubnet-group.html#cfn-rds-dbsubnetgroup-subnetids
        """
        result = self._values.get("subnet_ids")
        assert result is not None, "Required property 'subnet_ids' is missing"
        return result

    @builtins.property
    def db_subnet_group_name(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::DBSubnetGroup.DBSubnetGroupName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbsubnet-group.html#cfn-rds-dbsubnetgroup-dbsubnetgroupname
        """
        result = self._values.get("db_subnet_group_name")
        return result

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        """``AWS::RDS::DBSubnetGroup.Tags``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbsubnet-group.html#cfn-rds-dbsubnetgroup-tags
        """
        result = self._values.get("tags")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDBSubnetGroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnEventSubscription(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.CfnEventSubscription",
):
    """A CloudFormation ``AWS::RDS::EventSubscription``.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html
    :cloudformationResource: AWS::RDS::EventSubscription
    """

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        sns_topic_arn: builtins.str,
        enabled: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        event_categories: typing.Optional[typing.List[builtins.str]] = None,
        source_ids: typing.Optional[typing.List[builtins.str]] = None,
        source_type: typing.Optional[builtins.str] = None,
    ) -> None:
        """Create a new ``AWS::RDS::EventSubscription``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param sns_topic_arn: ``AWS::RDS::EventSubscription.SnsTopicArn``.
        :param enabled: ``AWS::RDS::EventSubscription.Enabled``.
        :param event_categories: ``AWS::RDS::EventSubscription.EventCategories``.
        :param source_ids: ``AWS::RDS::EventSubscription.SourceIds``.
        :param source_type: ``AWS::RDS::EventSubscription.SourceType``.
        """
        props = CfnEventSubscriptionProps(
            sns_topic_arn=sns_topic_arn,
            enabled=enabled,
            event_categories=event_categories,
            source_ids=source_ids,
            source_type=source_type,
        )

        jsii.create(CfnEventSubscription, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        """(experimental) Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.

        :stability: experimental
        """
        return jsii.invoke(self, "inspect", [inspector])

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        """
        :param props: -
        """
        return jsii.invoke(self, "renderProperties", [props])

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        """The CloudFormation resource type name for this resource class."""
        return jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return jsii.get(self, "cfnProperties")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="snsTopicArn")
    def sns_topic_arn(self) -> builtins.str:
        """``AWS::RDS::EventSubscription.SnsTopicArn``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-snstopicarn
        """
        return jsii.get(self, "snsTopicArn")

    @sns_topic_arn.setter # type: ignore
    def sns_topic_arn(self, value: builtins.str) -> None:
        jsii.set(self, "snsTopicArn", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="enabled")
    def enabled(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::EventSubscription.Enabled``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-enabled
        """
        return jsii.get(self, "enabled")

    @enabled.setter # type: ignore
    def enabled(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "enabled", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="eventCategories")
    def event_categories(self) -> typing.Optional[typing.List[builtins.str]]:
        """``AWS::RDS::EventSubscription.EventCategories``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-eventcategories
        """
        return jsii.get(self, "eventCategories")

    @event_categories.setter # type: ignore
    def event_categories(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        jsii.set(self, "eventCategories", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="sourceIds")
    def source_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        """``AWS::RDS::EventSubscription.SourceIds``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-sourceids
        """
        return jsii.get(self, "sourceIds")

    @source_ids.setter # type: ignore
    def source_ids(self, value: typing.Optional[typing.List[builtins.str]]) -> None:
        jsii.set(self, "sourceIds", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="sourceType")
    def source_type(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::EventSubscription.SourceType``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-sourcetype
        """
        return jsii.get(self, "sourceType")

    @source_type.setter # type: ignore
    def source_type(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "sourceType", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.CfnEventSubscriptionProps",
    jsii_struct_bases=[],
    name_mapping={
        "sns_topic_arn": "snsTopicArn",
        "enabled": "enabled",
        "event_categories": "eventCategories",
        "source_ids": "sourceIds",
        "source_type": "sourceType",
    },
)
class CfnEventSubscriptionProps:
    def __init__(
        self,
        *,
        sns_topic_arn: builtins.str,
        enabled: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        event_categories: typing.Optional[typing.List[builtins.str]] = None,
        source_ids: typing.Optional[typing.List[builtins.str]] = None,
        source_type: typing.Optional[builtins.str] = None,
    ) -> None:
        """Properties for defining a ``AWS::RDS::EventSubscription``.

        :param sns_topic_arn: ``AWS::RDS::EventSubscription.SnsTopicArn``.
        :param enabled: ``AWS::RDS::EventSubscription.Enabled``.
        :param event_categories: ``AWS::RDS::EventSubscription.EventCategories``.
        :param source_ids: ``AWS::RDS::EventSubscription.SourceIds``.
        :param source_type: ``AWS::RDS::EventSubscription.SourceType``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html
        """
        self._values: typing.Dict[str, typing.Any] = {
            "sns_topic_arn": sns_topic_arn,
        }
        if enabled is not None:
            self._values["enabled"] = enabled
        if event_categories is not None:
            self._values["event_categories"] = event_categories
        if source_ids is not None:
            self._values["source_ids"] = source_ids
        if source_type is not None:
            self._values["source_type"] = source_type

    @builtins.property
    def sns_topic_arn(self) -> builtins.str:
        """``AWS::RDS::EventSubscription.SnsTopicArn``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-snstopicarn
        """
        result = self._values.get("sns_topic_arn")
        assert result is not None, "Required property 'sns_topic_arn' is missing"
        return result

    @builtins.property
    def enabled(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::EventSubscription.Enabled``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-enabled
        """
        result = self._values.get("enabled")
        return result

    @builtins.property
    def event_categories(self) -> typing.Optional[typing.List[builtins.str]]:
        """``AWS::RDS::EventSubscription.EventCategories``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-eventcategories
        """
        result = self._values.get("event_categories")
        return result

    @builtins.property
    def source_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        """``AWS::RDS::EventSubscription.SourceIds``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-sourceids
        """
        result = self._values.get("source_ids")
        return result

    @builtins.property
    def source_type(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::EventSubscription.SourceType``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-sourcetype
        """
        result = self._values.get("source_type")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnEventSubscriptionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnGlobalCluster(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.CfnGlobalCluster",
):
    """A CloudFormation ``AWS::RDS::GlobalCluster``.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html
    :cloudformationResource: AWS::RDS::GlobalCluster
    """

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        deletion_protection: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        engine: typing.Optional[builtins.str] = None,
        engine_version: typing.Optional[builtins.str] = None,
        global_cluster_identifier: typing.Optional[builtins.str] = None,
        source_db_cluster_identifier: typing.Optional[builtins.str] = None,
        storage_encrypted: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
    ) -> None:
        """Create a new ``AWS::RDS::GlobalCluster``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param deletion_protection: ``AWS::RDS::GlobalCluster.DeletionProtection``.
        :param engine: ``AWS::RDS::GlobalCluster.Engine``.
        :param engine_version: ``AWS::RDS::GlobalCluster.EngineVersion``.
        :param global_cluster_identifier: ``AWS::RDS::GlobalCluster.GlobalClusterIdentifier``.
        :param source_db_cluster_identifier: ``AWS::RDS::GlobalCluster.SourceDBClusterIdentifier``.
        :param storage_encrypted: ``AWS::RDS::GlobalCluster.StorageEncrypted``.
        """
        props = CfnGlobalClusterProps(
            deletion_protection=deletion_protection,
            engine=engine,
            engine_version=engine_version,
            global_cluster_identifier=global_cluster_identifier,
            source_db_cluster_identifier=source_db_cluster_identifier,
            storage_encrypted=storage_encrypted,
        )

        jsii.create(CfnGlobalCluster, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        """(experimental) Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.

        :stability: experimental
        """
        return jsii.invoke(self, "inspect", [inspector])

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        """
        :param props: -
        """
        return jsii.invoke(self, "renderProperties", [props])

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        """The CloudFormation resource type name for this resource class."""
        return jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return jsii.get(self, "cfnProperties")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="deletionProtection")
    def deletion_protection(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::GlobalCluster.DeletionProtection``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-deletionprotection
        """
        return jsii.get(self, "deletionProtection")

    @deletion_protection.setter # type: ignore
    def deletion_protection(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "deletionProtection", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="engine")
    def engine(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::GlobalCluster.Engine``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-engine
        """
        return jsii.get(self, "engine")

    @engine.setter # type: ignore
    def engine(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "engine", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="engineVersion")
    def engine_version(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::GlobalCluster.EngineVersion``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-engineversion
        """
        return jsii.get(self, "engineVersion")

    @engine_version.setter # type: ignore
    def engine_version(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "engineVersion", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="globalClusterIdentifier")
    def global_cluster_identifier(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::GlobalCluster.GlobalClusterIdentifier``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-globalclusteridentifier
        """
        return jsii.get(self, "globalClusterIdentifier")

    @global_cluster_identifier.setter # type: ignore
    def global_cluster_identifier(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "globalClusterIdentifier", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="sourceDbClusterIdentifier")
    def source_db_cluster_identifier(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::GlobalCluster.SourceDBClusterIdentifier``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-sourcedbclusteridentifier
        """
        return jsii.get(self, "sourceDbClusterIdentifier")

    @source_db_cluster_identifier.setter # type: ignore
    def source_db_cluster_identifier(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        jsii.set(self, "sourceDbClusterIdentifier", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="storageEncrypted")
    def storage_encrypted(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::GlobalCluster.StorageEncrypted``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-storageencrypted
        """
        return jsii.get(self, "storageEncrypted")

    @storage_encrypted.setter # type: ignore
    def storage_encrypted(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "storageEncrypted", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.CfnGlobalClusterProps",
    jsii_struct_bases=[],
    name_mapping={
        "deletion_protection": "deletionProtection",
        "engine": "engine",
        "engine_version": "engineVersion",
        "global_cluster_identifier": "globalClusterIdentifier",
        "source_db_cluster_identifier": "sourceDbClusterIdentifier",
        "storage_encrypted": "storageEncrypted",
    },
)
class CfnGlobalClusterProps:
    def __init__(
        self,
        *,
        deletion_protection: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        engine: typing.Optional[builtins.str] = None,
        engine_version: typing.Optional[builtins.str] = None,
        global_cluster_identifier: typing.Optional[builtins.str] = None,
        source_db_cluster_identifier: typing.Optional[builtins.str] = None,
        storage_encrypted: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
    ) -> None:
        """Properties for defining a ``AWS::RDS::GlobalCluster``.

        :param deletion_protection: ``AWS::RDS::GlobalCluster.DeletionProtection``.
        :param engine: ``AWS::RDS::GlobalCluster.Engine``.
        :param engine_version: ``AWS::RDS::GlobalCluster.EngineVersion``.
        :param global_cluster_identifier: ``AWS::RDS::GlobalCluster.GlobalClusterIdentifier``.
        :param source_db_cluster_identifier: ``AWS::RDS::GlobalCluster.SourceDBClusterIdentifier``.
        :param storage_encrypted: ``AWS::RDS::GlobalCluster.StorageEncrypted``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html
        """
        self._values: typing.Dict[str, typing.Any] = {}
        if deletion_protection is not None:
            self._values["deletion_protection"] = deletion_protection
        if engine is not None:
            self._values["engine"] = engine
        if engine_version is not None:
            self._values["engine_version"] = engine_version
        if global_cluster_identifier is not None:
            self._values["global_cluster_identifier"] = global_cluster_identifier
        if source_db_cluster_identifier is not None:
            self._values["source_db_cluster_identifier"] = source_db_cluster_identifier
        if storage_encrypted is not None:
            self._values["storage_encrypted"] = storage_encrypted

    @builtins.property
    def deletion_protection(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::GlobalCluster.DeletionProtection``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-deletionprotection
        """
        result = self._values.get("deletion_protection")
        return result

    @builtins.property
    def engine(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::GlobalCluster.Engine``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-engine
        """
        result = self._values.get("engine")
        return result

    @builtins.property
    def engine_version(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::GlobalCluster.EngineVersion``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-engineversion
        """
        result = self._values.get("engine_version")
        return result

    @builtins.property
    def global_cluster_identifier(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::GlobalCluster.GlobalClusterIdentifier``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-globalclusteridentifier
        """
        result = self._values.get("global_cluster_identifier")
        return result

    @builtins.property
    def source_db_cluster_identifier(self) -> typing.Optional[builtins.str]:
        """``AWS::RDS::GlobalCluster.SourceDBClusterIdentifier``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-sourcedbclusteridentifier
        """
        result = self._values.get("source_db_cluster_identifier")
        return result

    @builtins.property
    def storage_encrypted(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        """``AWS::RDS::GlobalCluster.StorageEncrypted``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-storageencrypted
        """
        result = self._values.get("storage_encrypted")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnGlobalClusterProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnOptionGroup(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.CfnOptionGroup",
):
    """A CloudFormation ``AWS::RDS::OptionGroup``.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-optiongroup.html
    :cloudformationResource: AWS::RDS::OptionGroup
    """

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        engine_name: builtins.str,
        major_engine_version: builtins.str,
        option_configurations: typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnOptionGroup.OptionConfigurationProperty"]]],
        option_group_description: builtins.str,
        tags: typing.Optional[typing.List[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        """Create a new ``AWS::RDS::OptionGroup``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param engine_name: ``AWS::RDS::OptionGroup.EngineName``.
        :param major_engine_version: ``AWS::RDS::OptionGroup.MajorEngineVersion``.
        :param option_configurations: ``AWS::RDS::OptionGroup.OptionConfigurations``.
        :param option_group_description: ``AWS::RDS::OptionGroup.OptionGroupDescription``.
        :param tags: ``AWS::RDS::OptionGroup.Tags``.
        """
        props = CfnOptionGroupProps(
            engine_name=engine_name,
            major_engine_version=major_engine_version,
            option_configurations=option_configurations,
            option_group_description=option_group_description,
            tags=tags,
        )

        jsii.create(CfnOptionGroup, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        """(experimental) Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.

        :stability: experimental
        """
        return jsii.invoke(self, "inspect", [inspector])

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        """
        :param props: -
        """
        return jsii.invoke(self, "renderProperties", [props])

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        """The CloudFormation resource type name for this resource class."""
        return jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return jsii.get(self, "cfnProperties")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        """``AWS::RDS::OptionGroup.Tags``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-optiongroup.html#cfn-rds-optiongroup-tags
        """
        return jsii.get(self, "tags")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="engineName")
    def engine_name(self) -> builtins.str:
        """``AWS::RDS::OptionGroup.EngineName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-optiongroup.html#cfn-rds-optiongroup-enginename
        """
        return jsii.get(self, "engineName")

    @engine_name.setter # type: ignore
    def engine_name(self, value: builtins.str) -> None:
        jsii.set(self, "engineName", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="majorEngineVersion")
    def major_engine_version(self) -> builtins.str:
        """``AWS::RDS::OptionGroup.MajorEngineVersion``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-optiongroup.html#cfn-rds-optiongroup-majorengineversion
        """
        return jsii.get(self, "majorEngineVersion")

    @major_engine_version.setter # type: ignore
    def major_engine_version(self, value: builtins.str) -> None:
        jsii.set(self, "majorEngineVersion", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="optionConfigurations")
    def option_configurations(
        self,
    ) -> typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnOptionGroup.OptionConfigurationProperty"]]]:
        """``AWS::RDS::OptionGroup.OptionConfigurations``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-optiongroup.html#cfn-rds-optiongroup-optionconfigurations
        """
        return jsii.get(self, "optionConfigurations")

    @option_configurations.setter # type: ignore
    def option_configurations(
        self,
        value: typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnOptionGroup.OptionConfigurationProperty"]]],
    ) -> None:
        jsii.set(self, "optionConfigurations", value)

    @builtins.property # type: ignore
    @jsii.member(jsii_name="optionGroupDescription")
    def option_group_description(self) -> builtins.str:
        """``AWS::RDS::OptionGroup.OptionGroupDescription``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-optiongroup.html#cfn-rds-optiongroup-optiongroupdescription
        """
        return jsii.get(self, "optionGroupDescription")

    @option_group_description.setter # type: ignore
    def option_group_description(self, value: builtins.str) -> None:
        jsii.set(self, "optionGroupDescription", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-rds.CfnOptionGroup.OptionConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "option_name": "optionName",
            "db_security_group_memberships": "dbSecurityGroupMemberships",
            "option_settings": "optionSettings",
            "option_version": "optionVersion",
            "port": "port",
            "vpc_security_group_memberships": "vpcSecurityGroupMemberships",
        },
    )
    class OptionConfigurationProperty:
        def __init__(
            self,
            *,
            option_name: builtins.str,
            db_security_group_memberships: typing.Optional[typing.List[builtins.str]] = None,
            option_settings: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnOptionGroup.OptionSettingProperty"]]]] = None,
            option_version: typing.Optional[builtins.str] = None,
            port: typing.Optional[jsii.Number] = None,
            vpc_security_group_memberships: typing.Optional[typing.List[builtins.str]] = None,
        ) -> None:
            """
            :param option_name: ``CfnOptionGroup.OptionConfigurationProperty.OptionName``.
            :param db_security_group_memberships: ``CfnOptionGroup.OptionConfigurationProperty.DBSecurityGroupMemberships``.
            :param option_settings: ``CfnOptionGroup.OptionConfigurationProperty.OptionSettings``.
            :param option_version: ``CfnOptionGroup.OptionConfigurationProperty.OptionVersion``.
            :param port: ``CfnOptionGroup.OptionConfigurationProperty.Port``.
            :param vpc_security_group_memberships: ``CfnOptionGroup.OptionConfigurationProperty.VpcSecurityGroupMemberships``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-optiongroup-optionconfigurations.html
            """
            self._values: typing.Dict[str, typing.Any] = {
                "option_name": option_name,
            }
            if db_security_group_memberships is not None:
                self._values["db_security_group_memberships"] = db_security_group_memberships
            if option_settings is not None:
                self._values["option_settings"] = option_settings
            if option_version is not None:
                self._values["option_version"] = option_version
            if port is not None:
                self._values["port"] = port
            if vpc_security_group_memberships is not None:
                self._values["vpc_security_group_memberships"] = vpc_security_group_memberships

        @builtins.property
        def option_name(self) -> builtins.str:
            """``CfnOptionGroup.OptionConfigurationProperty.OptionName``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-optiongroup-optionconfigurations.html#cfn-rds-optiongroup-optionconfigurations-optionname
            """
            result = self._values.get("option_name")
            assert result is not None, "Required property 'option_name' is missing"
            return result

        @builtins.property
        def db_security_group_memberships(
            self,
        ) -> typing.Optional[typing.List[builtins.str]]:
            """``CfnOptionGroup.OptionConfigurationProperty.DBSecurityGroupMemberships``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-optiongroup-optionconfigurations.html#cfn-rds-optiongroup-optionconfigurations-dbsecuritygroupmemberships
            """
            result = self._values.get("db_security_group_memberships")
            return result

        @builtins.property
        def option_settings(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnOptionGroup.OptionSettingProperty"]]]]:
            """``CfnOptionGroup.OptionConfigurationProperty.OptionSettings``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-optiongroup-optionconfigurations.html#cfn-rds-optiongroup-optionconfigurations-optionsettings
            """
            result = self._values.get("option_settings")
            return result

        @builtins.property
        def option_version(self) -> typing.Optional[builtins.str]:
            """``CfnOptionGroup.OptionConfigurationProperty.OptionVersion``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-optiongroup-optionconfigurations.html#cfn-rds-optiongroup-optionconfiguration-optionversion
            """
            result = self._values.get("option_version")
            return result

        @builtins.property
        def port(self) -> typing.Optional[jsii.Number]:
            """``CfnOptionGroup.OptionConfigurationProperty.Port``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-optiongroup-optionconfigurations.html#cfn-rds-optiongroup-optionconfigurations-port
            """
            result = self._values.get("port")
            return result

        @builtins.property
        def vpc_security_group_memberships(
            self,
        ) -> typing.Optional[typing.List[builtins.str]]:
            """``CfnOptionGroup.OptionConfigurationProperty.VpcSecurityGroupMemberships``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-optiongroup-optionconfigurations.html#cfn-rds-optiongroup-optionconfigurations-vpcsecuritygroupmemberships
            """
            result = self._values.get("vpc_security_group_memberships")
            return result

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "OptionConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-rds.CfnOptionGroup.OptionSettingProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "value": "value"},
    )
    class OptionSettingProperty:
        def __init__(
            self,
            *,
            name: typing.Optional[builtins.str] = None,
            value: typing.Optional[builtins.str] = None,
        ) -> None:
            """
            :param name: ``CfnOptionGroup.OptionSettingProperty.Name``.
            :param value: ``CfnOptionGroup.OptionSettingProperty.Value``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-optiongroup-optionconfigurations-optionsettings.html
            """
            self._values: typing.Dict[str, typing.Any] = {}
            if name is not None:
                self._values["name"] = name
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            """``CfnOptionGroup.OptionSettingProperty.Name``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-optiongroup-optionconfigurations-optionsettings.html#cfn-rds-optiongroup-optionconfigurations-optionsettings-name
            """
            result = self._values.get("name")
            return result

        @builtins.property
        def value(self) -> typing.Optional[builtins.str]:
            """``CfnOptionGroup.OptionSettingProperty.Value``.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-optiongroup-optionconfigurations-optionsettings.html#cfn-rds-optiongroup-optionconfigurations-optionsettings-value
            """
            result = self._values.get("value")
            return result

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "OptionSettingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.CfnOptionGroupProps",
    jsii_struct_bases=[],
    name_mapping={
        "engine_name": "engineName",
        "major_engine_version": "majorEngineVersion",
        "option_configurations": "optionConfigurations",
        "option_group_description": "optionGroupDescription",
        "tags": "tags",
    },
)
class CfnOptionGroupProps:
    def __init__(
        self,
        *,
        engine_name: builtins.str,
        major_engine_version: builtins.str,
        option_configurations: typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnOptionGroup.OptionConfigurationProperty]]],
        option_group_description: builtins.str,
        tags: typing.Optional[typing.List[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        """Properties for defining a ``AWS::RDS::OptionGroup``.

        :param engine_name: ``AWS::RDS::OptionGroup.EngineName``.
        :param major_engine_version: ``AWS::RDS::OptionGroup.MajorEngineVersion``.
        :param option_configurations: ``AWS::RDS::OptionGroup.OptionConfigurations``.
        :param option_group_description: ``AWS::RDS::OptionGroup.OptionGroupDescription``.
        :param tags: ``AWS::RDS::OptionGroup.Tags``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-optiongroup.html
        """
        self._values: typing.Dict[str, typing.Any] = {
            "engine_name": engine_name,
            "major_engine_version": major_engine_version,
            "option_configurations": option_configurations,
            "option_group_description": option_group_description,
        }
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def engine_name(self) -> builtins.str:
        """``AWS::RDS::OptionGroup.EngineName``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-optiongroup.html#cfn-rds-optiongroup-enginename
        """
        result = self._values.get("engine_name")
        assert result is not None, "Required property 'engine_name' is missing"
        return result

    @builtins.property
    def major_engine_version(self) -> builtins.str:
        """``AWS::RDS::OptionGroup.MajorEngineVersion``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-optiongroup.html#cfn-rds-optiongroup-majorengineversion
        """
        result = self._values.get("major_engine_version")
        assert result is not None, "Required property 'major_engine_version' is missing"
        return result

    @builtins.property
    def option_configurations(
        self,
    ) -> typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnOptionGroup.OptionConfigurationProperty]]]:
        """``AWS::RDS::OptionGroup.OptionConfigurations``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-optiongroup.html#cfn-rds-optiongroup-optionconfigurations
        """
        result = self._values.get("option_configurations")
        assert result is not None, "Required property 'option_configurations' is missing"
        return result

    @builtins.property
    def option_group_description(self) -> builtins.str:
        """``AWS::RDS::OptionGroup.OptionGroupDescription``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-optiongroup.html#cfn-rds-optiongroup-optiongroupdescription
        """
        result = self._values.get("option_group_description")
        assert result is not None, "Required property 'option_group_description' is missing"
        return result

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        """``AWS::RDS::OptionGroup.Tags``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-optiongroup.html#cfn-rds-optiongroup-tags
        """
        result = self._values.get("tags")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnOptionGroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.ClusterEngineBindOptions",
    jsii_struct_bases=[],
    name_mapping={
        "parameter_group": "parameterGroup",
        "s3_export_role": "s3ExportRole",
        "s3_import_role": "s3ImportRole",
    },
)
class ClusterEngineBindOptions:
    def __init__(
        self,
        *,
        parameter_group: typing.Optional["IParameterGroup"] = None,
        s3_export_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        s3_import_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
    ) -> None:
        """The extra options passed to the {@link IClusterEngine.bindToCluster} method.

        :param parameter_group: The customer-provided ParameterGroup. Default: - none
        :param s3_export_role: The role used for S3 exporting. Default: - none
        :param s3_import_role: The role used for S3 importing. Default: - none
        """
        self._values: typing.Dict[str, typing.Any] = {}
        if parameter_group is not None:
            self._values["parameter_group"] = parameter_group
        if s3_export_role is not None:
            self._values["s3_export_role"] = s3_export_role
        if s3_import_role is not None:
            self._values["s3_import_role"] = s3_import_role

    @builtins.property
    def parameter_group(self) -> typing.Optional["IParameterGroup"]:
        """The customer-provided ParameterGroup.

        :default: - none
        """
        result = self._values.get("parameter_group")
        return result

    @builtins.property
    def s3_export_role(self) -> typing.Optional[aws_cdk.aws_iam.IRole]:
        """The role used for S3 exporting.

        :default: - none
        """
        result = self._values.get("s3_export_role")
        return result

    @builtins.property
    def s3_import_role(self) -> typing.Optional[aws_cdk.aws_iam.IRole]:
        """The role used for S3 importing.

        :default: - none
        """
        result = self._values.get("s3_import_role")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ClusterEngineBindOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.ClusterEngineConfig",
    jsii_struct_bases=[],
    name_mapping={
        "features": "features",
        "parameter_group": "parameterGroup",
        "port": "port",
    },
)
class ClusterEngineConfig:
    def __init__(
        self,
        *,
        features: typing.Optional["ClusterEngineFeatures"] = None,
        parameter_group: typing.Optional["IParameterGroup"] = None,
        port: typing.Optional[jsii.Number] = None,
    ) -> None:
        """The type returned from the {@link IClusterEngine.bindToCluster} method.

        :param features: Features supported by the database engine. Default: - no features
        :param parameter_group: The ParameterGroup to use for the cluster. Default: - no ParameterGroup will be used
        :param port: The port to use for this cluster, unless the customer specified the port directly. Default: - use the default port for clusters (3306)
        """
        if isinstance(features, dict):
            features = ClusterEngineFeatures(**features)
        self._values: typing.Dict[str, typing.Any] = {}
        if features is not None:
            self._values["features"] = features
        if parameter_group is not None:
            self._values["parameter_group"] = parameter_group
        if port is not None:
            self._values["port"] = port

    @builtins.property
    def features(self) -> typing.Optional["ClusterEngineFeatures"]:
        """Features supported by the database engine.

        :default: - no features

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DBEngineVersion.html
        """
        result = self._values.get("features")
        return result

    @builtins.property
    def parameter_group(self) -> typing.Optional["IParameterGroup"]:
        """The ParameterGroup to use for the cluster.

        :default: - no ParameterGroup will be used
        """
        result = self._values.get("parameter_group")
        return result

    @builtins.property
    def port(self) -> typing.Optional[jsii.Number]:
        """The port to use for this cluster, unless the customer specified the port directly.

        :default: - use the default port for clusters (3306)
        """
        result = self._values.get("port")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ClusterEngineConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.ClusterEngineFeatures",
    jsii_struct_bases=[],
    name_mapping={"s3_export": "s3Export", "s3_import": "s3Import"},
)
class ClusterEngineFeatures:
    def __init__(
        self,
        *,
        s3_export: typing.Optional[builtins.str] = None,
        s3_import: typing.Optional[builtins.str] = None,
    ) -> None:
        """Represents Database Engine features.

        :param s3_export: Feature name for the DB instance that the IAM role to export to S3 bucket is to be associated with. Default: - no s3Export feature name
        :param s3_import: Feature name for the DB instance that the IAM role to access the S3 bucket for import is to be associated with. Default: - no s3Import feature name
        """
        self._values: typing.Dict[str, typing.Any] = {}
        if s3_export is not None:
            self._values["s3_export"] = s3_export
        if s3_import is not None:
            self._values["s3_import"] = s3_import

    @builtins.property
    def s3_export(self) -> typing.Optional[builtins.str]:
        """Feature name for the DB instance that the IAM role to export to S3 bucket is to be associated with.

        :default: - no s3Export feature name
        """
        result = self._values.get("s3_export")
        return result

    @builtins.property
    def s3_import(self) -> typing.Optional[builtins.str]:
        """Feature name for the DB instance that the IAM role to access the S3 bucket for import is to be associated with.

        :default: - no s3Import feature name
        """
        result = self._values.get("s3_import")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ClusterEngineFeatures(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class Credentials(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@aws-cdk/aws-rds.Credentials",
):
    """Username and password combination."""

    @builtins.staticmethod
    def __jsii_proxy_class__():
        return _CredentialsProxy

    def __init__(self) -> None:
        jsii.create(Credentials, self, [])

    @jsii.member(jsii_name="fromGeneratedSecret")
    @builtins.classmethod
    def from_generated_secret(
        cls,
        username: builtins.str,
        *,
        encryption_key: typing.Optional[aws_cdk.aws_kms.IKey] = None,
        exclude_characters: typing.Optional[builtins.str] = None,
    ) -> "Credentials":
        """Creates Credentials with a password generated and stored in Secrets Manager.

        :param username: -
        :param encryption_key: KMS encryption key to encrypt the generated secret. Default: - default master key
        :param exclude_characters: The characters to exclude from the generated password. Has no effect if {@link password} has been provided. Default: - the DatabaseSecret default exclude character set (" %+~`#$&*()|[]{}:;<>?!'/
        """
        options = CredentialsBaseOptions(
            encryption_key=encryption_key, exclude_characters=exclude_characters
        )

        return jsii.sinvoke(cls, "fromGeneratedSecret", [username, options])

    @jsii.member(jsii_name="fromPassword")
    @builtins.classmethod
    def from_password(
        cls,
        username: builtins.str,
        password: aws_cdk.core.SecretValue,
    ) -> "Credentials":
        """Creates Credentials from a password.

        Do not put passwords in your CDK code directly.

        :param username: -
        :param password: -
        """
        return jsii.sinvoke(cls, "fromPassword", [username, password])

    @jsii.member(jsii_name="fromSecret")
    @builtins.classmethod
    def from_secret(
        cls,
        secret: aws_cdk.aws_secretsmanager.ISecret,
        username: typing.Optional[builtins.str] = None,
    ) -> "Credentials":
        """Creates Credentials from an existing Secrets Manager ``Secret`` (or ``DatabaseSecret``).

        The Secret must be a JSON string with a ``username`` and ``password`` field::

           {
              ...
              "username": <required: username>,
              "password": <required: password>,
           }

        :param secret: The secret where the credentials are stored.
        :param username: The username defined in the secret. If specified the username will be referenced as a string and not a dynamic reference to the username field in the secret. This allows to replace the secret without replacing the instance or cluster.
        """
        return jsii.sinvoke(cls, "fromSecret", [secret, username])

    @jsii.member(jsii_name="fromUsername")
    @builtins.classmethod
    def from_username(
        cls,
        username: builtins.str,
        *,
        password: typing.Optional[aws_cdk.core.SecretValue] = None,
        encryption_key: typing.Optional[aws_cdk.aws_kms.IKey] = None,
        exclude_characters: typing.Optional[builtins.str] = None,
    ) -> "Credentials":
        """(deprecated) Creates Credentials for the given username, and optional password and key.

        If no password is provided, one will be generated and stored in Secrets Manager.

        :param username: -
        :param password: Password. Do not put passwords in your CDK code directly. Default: - a Secrets Manager generated password
        :param encryption_key: KMS encryption key to encrypt the generated secret. Default: - default master key
        :param exclude_characters: The characters to exclude from the generated password. Has no effect if {@link password} has been provided. Default: - the DatabaseSecret default exclude character set (" %+~`#$&*()|[]{}:;<>?!'/

        :deprecated:

        use ``fromGeneratedSecret()`` or ``fromPassword()`` for new Clusters and Instances.
        Note that switching from ``fromUsername()`` to ``fromGeneratedSecret()`` or ``fromPassword()`` for already deployed
        Clusters or Instances will result in their replacement!

        :stability: deprecated
        """
        options = CredentialsFromUsernameOptions(
            password=password,
            encryption_key=encryption_key,
            exclude_characters=exclude_characters,
        )

        return jsii.sinvoke(cls, "fromUsername", [username, options])

    @builtins.property # type: ignore
    @jsii.member(jsii_name="username")
    @abc.abstractmethod
    def username(self) -> builtins.str:
        """Username."""
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="encryptionKey")
    @abc.abstractmethod
    def encryption_key(self) -> typing.Optional[aws_cdk.aws_kms.IKey]:
        """KMS encryption key to encrypt the generated secret.

        :default: - default master key
        """
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="excludeCharacters")
    @abc.abstractmethod
    def exclude_characters(self) -> typing.Optional[builtins.str]:
        """The characters to exclude from the generated password.

        Only used if {@link password} has not been set.

        :default: - the DatabaseSecret default exclude character set (" %+~`#$&*()|[]{}:;<>?!'/

        :: "\\")
        """
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="password")
    @abc.abstractmethod
    def password(self) -> typing.Optional[aws_cdk.core.SecretValue]:
        """Password.

        Do not put passwords in your CDK code directly.

        :default: - a Secrets Manager generated password
        """
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="secret")
    @abc.abstractmethod
    def secret(self) -> typing.Optional[aws_cdk.aws_secretsmanager.ISecret]:
        """Secret used to instantiate this Login.

        :default: - none
        """
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="usernameAsString")
    @abc.abstractmethod
    def username_as_string(self) -> typing.Optional[builtins.bool]:
        """Whether the username should be referenced as a string and not as a dynamic reference to the username in the secret.

        :default: false
        """
        ...


class _CredentialsProxy(Credentials):
    @builtins.property # type: ignore
    @jsii.member(jsii_name="username")
    def username(self) -> builtins.str:
        """Username."""
        return jsii.get(self, "username")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="encryptionKey")
    def encryption_key(self) -> typing.Optional[aws_cdk.aws_kms.IKey]:
        """KMS encryption key to encrypt the generated secret.

        :default: - default master key
        """
        return jsii.get(self, "encryptionKey")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="excludeCharacters")
    def exclude_characters(self) -> typing.Optional[builtins.str]:
        """The characters to exclude from the generated password.

        Only used if {@link password} has not been set.

        :default: - the DatabaseSecret default exclude character set (" %+~`#$&*()|[]{}:;<>?!'/

        :: "\\")
        """
        return jsii.get(self, "excludeCharacters")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="password")
    def password(self) -> typing.Optional[aws_cdk.core.SecretValue]:
        """Password.

        Do not put passwords in your CDK code directly.

        :default: - a Secrets Manager generated password
        """
        return jsii.get(self, "password")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="secret")
    def secret(self) -> typing.Optional[aws_cdk.aws_secretsmanager.ISecret]:
        """Secret used to instantiate this Login.

        :default: - none
        """
        return jsii.get(self, "secret")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="usernameAsString")
    def username_as_string(self) -> typing.Optional[builtins.bool]:
        """Whether the username should be referenced as a string and not as a dynamic reference to the username in the secret.

        :default: false
        """
        return jsii.get(self, "usernameAsString")


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.CredentialsBaseOptions",
    jsii_struct_bases=[],
    name_mapping={
        "encryption_key": "encryptionKey",
        "exclude_characters": "excludeCharacters",
    },
)
class CredentialsBaseOptions:
    def __init__(
        self,
        *,
        encryption_key: typing.Optional[aws_cdk.aws_kms.IKey] = None,
        exclude_characters: typing.Optional[builtins.str] = None,
    ) -> None:
        """Base options for creating Credentials.

        :param encryption_key: KMS encryption key to encrypt the generated secret. Default: - default master key
        :param exclude_characters: The characters to exclude from the generated password. Has no effect if {@link password} has been provided. Default: - the DatabaseSecret default exclude character set (" %+~`#$&*()|[]{}:;<>?!'/
        """
        self._values: typing.Dict[str, typing.Any] = {}
        if encryption_key is not None:
            self._values["encryption_key"] = encryption_key
        if exclude_characters is not None:
            self._values["exclude_characters"] = exclude_characters

    @builtins.property
    def encryption_key(self) -> typing.Optional[aws_cdk.aws_kms.IKey]:
        """KMS encryption key to encrypt the generated secret.

        :default: - default master key
        """
        result = self._values.get("encryption_key")
        return result

    @builtins.property
    def exclude_characters(self) -> typing.Optional[builtins.str]:
        """The characters to exclude from the generated password.

        Has no effect if {@link password} has been provided.

        :default: - the DatabaseSecret default exclude character set (" %+~`#$&*()|[]{}:;<>?!'/

        :: "\\")
        """
        result = self._values.get("exclude_characters")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CredentialsBaseOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.CredentialsFromUsernameOptions",
    jsii_struct_bases=[CredentialsBaseOptions],
    name_mapping={
        "encryption_key": "encryptionKey",
        "exclude_characters": "excludeCharacters",
        "password": "password",
    },
)
class CredentialsFromUsernameOptions(CredentialsBaseOptions):
    def __init__(
        self,
        *,
        encryption_key: typing.Optional[aws_cdk.aws_kms.IKey] = None,
        exclude_characters: typing.Optional[builtins.str] = None,
        password: typing.Optional[aws_cdk.core.SecretValue] = None,
    ) -> None:
        """Options for creating Credentials from a username.

        :param encryption_key: KMS encryption key to encrypt the generated secret. Default: - default master key
        :param exclude_characters: The characters to exclude from the generated password. Has no effect if {@link password} has been provided. Default: - the DatabaseSecret default exclude character set (" %+~`#$&*()|[]{}:;<>?!'/
        :param password: Password. Do not put passwords in your CDK code directly. Default: - a Secrets Manager generated password
        """
        self._values: typing.Dict[str, typing.Any] = {}
        if encryption_key is not None:
            self._values["encryption_key"] = encryption_key
        if exclude_characters is not None:
            self._values["exclude_characters"] = exclude_characters
        if password is not None:
            self._values["password"] = password

    @builtins.property
    def encryption_key(self) -> typing.Optional[aws_cdk.aws_kms.IKey]:
        """KMS encryption key to encrypt the generated secret.

        :default: - default master key
        """
        result = self._values.get("encryption_key")
        return result

    @builtins.property
    def exclude_characters(self) -> typing.Optional[builtins.str]:
        """The characters to exclude from the generated password.

        Has no effect if {@link password} has been provided.

        :default: - the DatabaseSecret default exclude character set (" %+~`#$&*()|[]{}:;<>?!'/

        :: "\\")
        """
        result = self._values.get("exclude_characters")
        return result

    @builtins.property
    def password(self) -> typing.Optional[aws_cdk.core.SecretValue]:
        """Password.

        Do not put passwords in your CDK code directly.

        :default: - a Secrets Manager generated password
        """
        result = self._values.get("password")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CredentialsFromUsernameOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.DatabaseClusterAttributes",
    jsii_struct_bases=[],
    name_mapping={
        "cluster_identifier": "clusterIdentifier",
        "cluster_endpoint_address": "clusterEndpointAddress",
        "engine": "engine",
        "instance_endpoint_addresses": "instanceEndpointAddresses",
        "instance_identifiers": "instanceIdentifiers",
        "port": "port",
        "reader_endpoint_address": "readerEndpointAddress",
        "security_groups": "securityGroups",
    },
)
class DatabaseClusterAttributes:
    def __init__(
        self,
        *,
        cluster_identifier: builtins.str,
        cluster_endpoint_address: typing.Optional[builtins.str] = None,
        engine: typing.Optional["IClusterEngine"] = None,
        instance_endpoint_addresses: typing.Optional[typing.List[builtins.str]] = None,
        instance_identifiers: typing.Optional[typing.List[builtins.str]] = None,
        port: typing.Optional[jsii.Number] = None,
        reader_endpoint_address: typing.Optional[builtins.str] = None,
        security_groups: typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]] = None,
    ) -> None:
        """Properties that describe an existing cluster instance.

        :param cluster_identifier: Identifier for the cluster.
        :param cluster_endpoint_address: Cluster endpoint address. Default: - no endpoint address
        :param engine: The engine of the existing Cluster. Default: - the imported Cluster's engine is unknown
        :param instance_endpoint_addresses: Endpoint addresses of individual instances. Default: - no instance endpoints
        :param instance_identifiers: Identifier for the instances. Default: - no instance identifiers
        :param port: The database port. Default: - none
        :param reader_endpoint_address: Reader endpoint address. Default: - no reader address
        :param security_groups: The security groups of the database cluster. Default: - no security groups
        """
        self._values: typing.Dict[str, typing.Any] = {
            "cluster_identifier": cluster_identifier,
        }
        if cluster_endpoint_address is not None:
            self._values["cluster_endpoint_address"] = cluster_endpoint_address
        if engine is not None:
            self._values["engine"] = engine
        if instance_endpoint_addresses is not None:
            self._values["instance_endpoint_addresses"] = instance_endpoint_addresses
        if instance_identifiers is not None:
            self._values["instance_identifiers"] = instance_identifiers
        if port is not None:
            self._values["port"] = port
        if reader_endpoint_address is not None:
            self._values["reader_endpoint_address"] = reader_endpoint_address
        if security_groups is not None:
            self._values["security_groups"] = security_groups

    @builtins.property
    def cluster_identifier(self) -> builtins.str:
        """Identifier for the cluster."""
        result = self._values.get("cluster_identifier")
        assert result is not None, "Required property 'cluster_identifier' is missing"
        return result

    @builtins.property
    def cluster_endpoint_address(self) -> typing.Optional[builtins.str]:
        """Cluster endpoint address.

        :default: - no endpoint address
        """
        result = self._values.get("cluster_endpoint_address")
        return result

    @builtins.property
    def engine(self) -> typing.Optional["IClusterEngine"]:
        """The engine of the existing Cluster.

        :default: - the imported Cluster's engine is unknown
        """
        result = self._values.get("engine")
        return result

    @builtins.property
    def instance_endpoint_addresses(self) -> typing.Optional[typing.List[builtins.str]]:
        """Endpoint addresses of individual instances.

        :default: - no instance endpoints
        """
        result = self._values.get("instance_endpoint_addresses")
        return result

    @builtins.property
    def instance_identifiers(self) -> typing.Optional[typing.List[builtins.str]]:
        """Identifier for the instances.

        :default: - no instance identifiers
        """
        result = self._values.get("instance_identifiers")
        return result

    @builtins.property
    def port(self) -> typing.Optional[jsii.Number]:
        """The database port.

        :default: - none
        """
        result = self._values.get("port")
        return result

    @builtins.property
    def reader_endpoint_address(self) -> typing.Optional[builtins.str]:
        """Reader endpoint address.

        :default: - no reader address
        """
        result = self._values.get("reader_endpoint_address")
        return result

    @builtins.property
    def security_groups(
        self,
    ) -> typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]]:
        """The security groups of the database cluster.

        :default: - no security groups
        """
        result = self._values.get("security_groups")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DatabaseClusterAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class DatabaseClusterEngine(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.DatabaseClusterEngine",
):
    """A database cluster engine.

    Provides mapping to the serverless application
    used for secret rotation.
    """

    def __init__(self) -> None:
        jsii.create(DatabaseClusterEngine, self, [])

    @jsii.member(jsii_name="aurora")
    @builtins.classmethod
    def aurora(cls, *, version: AuroraEngineVersion) -> "IClusterEngine":
        """Creates a new plain Aurora database cluster engine.

        :param version: The version of the Aurora cluster engine.
        """
        props = AuroraClusterEngineProps(version=version)

        return jsii.sinvoke(cls, "aurora", [props])

    @jsii.member(jsii_name="auroraMysql")
    @builtins.classmethod
    def aurora_mysql(cls, *, version: AuroraMysqlEngineVersion) -> "IClusterEngine":
        """Creates a new Aurora MySQL database cluster engine.

        :param version: The version of the Aurora MySQL cluster engine.
        """
        props = AuroraMysqlClusterEngineProps(version=version)

        return jsii.sinvoke(cls, "auroraMysql", [props])

    @jsii.member(jsii_name="auroraPostgres")
    @builtins.classmethod
    def aurora_postgres(
        cls,
        *,
        version: AuroraPostgresEngineVersion,
    ) -> "IClusterEngine":
        """Creates a new Aurora PostgreSQL database cluster engine.

        :param version: The version of the Aurora PostgreSQL cluster engine.
        """
        props = AuroraPostgresClusterEngineProps(version=version)

        return jsii.sinvoke(cls, "auroraPostgres", [props])

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="AURORA")
    def AURORA(cls) -> "IClusterEngine":
        """The unversioned 'aurora' cluster engine.

        **Note**: we do not recommend using unversioned engines for non-serverless Clusters,
        as that can pose an availability risk.
        We recommend using versioned engines created using the {@link aurora()} method
        """
        return jsii.sget(cls, "AURORA")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="AURORA_MYSQL")
    def AURORA_MYSQL(cls) -> "IClusterEngine":
        """The unversioned 'aurora-msql' cluster engine.

        **Note**: we do not recommend using unversioned engines for non-serverless Clusters,
        as that can pose an availability risk.
        We recommend using versioned engines created using the {@link auroraMysql()} method
        """
        return jsii.sget(cls, "AURORA_MYSQL")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="AURORA_POSTGRESQL")
    def AURORA_POSTGRESQL(cls) -> "IClusterEngine":
        """The unversioned 'aurora-postgresql' cluster engine.

        **Note**: we do not recommend using unversioned engines for non-serverless Clusters,
        as that can pose an availability risk.
        We recommend using versioned engines created using the {@link auroraPostgres()} method
        """
        return jsii.sget(cls, "AURORA_POSTGRESQL")


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.DatabaseClusterFromSnapshotProps",
    jsii_struct_bases=[],
    name_mapping={
        "engine": "engine",
        "instance_props": "instanceProps",
        "snapshot_identifier": "snapshotIdentifier",
        "backup": "backup",
        "cloudwatch_logs_exports": "cloudwatchLogsExports",
        "cloudwatch_logs_retention": "cloudwatchLogsRetention",
        "cloudwatch_logs_retention_role": "cloudwatchLogsRetentionRole",
        "cluster_identifier": "clusterIdentifier",
        "default_database_name": "defaultDatabaseName",
        "deletion_protection": "deletionProtection",
        "instance_identifier_base": "instanceIdentifierBase",
        "instances": "instances",
        "monitoring_interval": "monitoringInterval",
        "monitoring_role": "monitoringRole",
        "parameter_group": "parameterGroup",
        "port": "port",
        "preferred_maintenance_window": "preferredMaintenanceWindow",
        "removal_policy": "removalPolicy",
        "s3_export_buckets": "s3ExportBuckets",
        "s3_export_role": "s3ExportRole",
        "s3_import_buckets": "s3ImportBuckets",
        "s3_import_role": "s3ImportRole",
        "subnet_group": "subnetGroup",
    },
)
class DatabaseClusterFromSnapshotProps:
    def __init__(
        self,
        *,
        engine: "IClusterEngine",
        instance_props: "InstanceProps",
        snapshot_identifier: builtins.str,
        backup: typing.Optional[BackupProps] = None,
        cloudwatch_logs_exports: typing.Optional[typing.List[builtins.str]] = None,
        cloudwatch_logs_retention: typing.Optional[aws_cdk.aws_logs.RetentionDays] = None,
        cloudwatch_logs_retention_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        cluster_identifier: typing.Optional[builtins.str] = None,
        default_database_name: typing.Optional[builtins.str] = None,
        deletion_protection: typing.Optional[builtins.bool] = None,
        instance_identifier_base: typing.Optional[builtins.str] = None,
        instances: typing.Optional[jsii.Number] = None,
        monitoring_interval: typing.Optional[aws_cdk.core.Duration] = None,
        monitoring_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        parameter_group: typing.Optional["IParameterGroup"] = None,
        port: typing.Optional[jsii.Number] = None,
        preferred_maintenance_window: typing.Optional[builtins.str] = None,
        removal_policy: typing.Optional[aws_cdk.core.RemovalPolicy] = None,
        s3_export_buckets: typing.Optional[typing.List[aws_cdk.aws_s3.IBucket]] = None,
        s3_export_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        s3_import_buckets: typing.Optional[typing.List[aws_cdk.aws_s3.IBucket]] = None,
        s3_import_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        subnet_group: typing.Optional["ISubnetGroup"] = None,
    ) -> None:
        """Properties for ``DatabaseClusterFromSnapshot``.

        :param engine: What kind of database to start.
        :param instance_props: Settings for the individual instances that are launched.
        :param snapshot_identifier: The identifier for the DB instance snapshot or DB cluster snapshot to restore from. You can use either the name or the Amazon Resource Name (ARN) to specify a DB cluster snapshot. However, you can use only the ARN to specify a DB instance snapshot.
        :param backup: Backup settings. Default: - Backup retention period for automated backups is 1 day. Backup preferred window is set to a 30-minute window selected at random from an 8-hour block of time for each AWS Region, occurring on a random day of the week.
        :param cloudwatch_logs_exports: The list of log types that need to be enabled for exporting to CloudWatch Logs. Default: - no log exports
        :param cloudwatch_logs_retention: The number of days log events are kept in CloudWatch Logs. When updating this property, unsetting it doesn't remove the log retention policy. To remove the retention policy, set the value to ``Infinity``. Default: - logs never expire
        :param cloudwatch_logs_retention_role: The IAM role for the Lambda function associated with the custom resource that sets the retention policy. Default: - a new role is created.
        :param cluster_identifier: An optional identifier for the cluster. Default: - A name is automatically generated.
        :param default_database_name: Name of a database which is automatically created inside the cluster. Default: - Database is not created in cluster.
        :param deletion_protection: Indicates whether the DB cluster should have deletion protection enabled. Default: - true if ``removalPolicy`` is RETAIN, false otherwise
        :param instance_identifier_base: Base identifier for instances. Every replica is named by appending the replica number to this string, 1-based. Default: - clusterIdentifier is used with the word "Instance" appended. If clusterIdentifier is not provided, the identifier is automatically generated.
        :param instances: How many replicas/instances to create. Has to be at least 1. Default: 2
        :param monitoring_interval: The interval, in seconds, between points when Amazon RDS collects enhanced monitoring metrics for the DB instances. Default: no enhanced monitoring
        :param monitoring_role: Role that will be used to manage DB instances monitoring. Default: - A role is automatically created for you
        :param parameter_group: Additional parameters to pass to the database engine. Default: - No parameter group.
        :param port: What port to listen on. Default: - The default for the engine is used.
        :param preferred_maintenance_window: A preferred maintenance window day/time range. Should be specified as a range ddd:hh24:mi-ddd:hh24:mi (24H Clock UTC). Example: 'Sun:23:45-Mon:00:15' Default: - 30-minute window selected at random from an 8-hour block of time for each AWS Region, occurring on a random day of the week.
        :param removal_policy: The removal policy to apply when the cluster and its instances are removed from the stack or replaced during an update. Default: - RemovalPolicy.SNAPSHOT (remove the cluster and instances, but retain a snapshot of the data)
        :param s3_export_buckets: S3 buckets that you want to load data into. This feature is only supported by the Aurora database engine. This property must not be used if ``s3ExportRole`` is used. For MySQL: Default: - None
        :param s3_export_role: Role that will be associated with this DB cluster to enable S3 export. This feature is only supported by the Aurora database engine. This property must not be used if ``s3ExportBuckets`` is used. For MySQL: Default: - New role is created if ``s3ExportBuckets`` is set, no role is defined otherwise
        :param s3_import_buckets: S3 buckets that you want to load data from. This feature is only supported by the Aurora database engine. This property must not be used if ``s3ImportRole`` is used. For MySQL: Default: - None
        :param s3_import_role: Role that will be associated with this DB cluster to enable S3 import. This feature is only supported by the Aurora database engine. This property must not be used if ``s3ImportBuckets`` is used. For MySQL: Default: - New role is created if ``s3ImportBuckets`` is set, no role is defined otherwise
        :param subnet_group: Existing subnet group for the cluster. Default: - a new subnet group will be created.
        """
        if isinstance(instance_props, dict):
            instance_props = InstanceProps(**instance_props)
        if isinstance(backup, dict):
            backup = BackupProps(**backup)
        self._values: typing.Dict[str, typing.Any] = {
            "engine": engine,
            "instance_props": instance_props,
            "snapshot_identifier": snapshot_identifier,
        }
        if backup is not None:
            self._values["backup"] = backup
        if cloudwatch_logs_exports is not None:
            self._values["cloudwatch_logs_exports"] = cloudwatch_logs_exports
        if cloudwatch_logs_retention is not None:
            self._values["cloudwatch_logs_retention"] = cloudwatch_logs_retention
        if cloudwatch_logs_retention_role is not None:
            self._values["cloudwatch_logs_retention_role"] = cloudwatch_logs_retention_role
        if cluster_identifier is not None:
            self._values["cluster_identifier"] = cluster_identifier
        if default_database_name is not None:
            self._values["default_database_name"] = default_database_name
        if deletion_protection is not None:
            self._values["deletion_protection"] = deletion_protection
        if instance_identifier_base is not None:
            self._values["instance_identifier_base"] = instance_identifier_base
        if instances is not None:
            self._values["instances"] = instances
        if monitoring_interval is not None:
            self._values["monitoring_interval"] = monitoring_interval
        if monitoring_role is not None:
            self._values["monitoring_role"] = monitoring_role
        if parameter_group is not None:
            self._values["parameter_group"] = parameter_group
        if port is not None:
            self._values["port"] = port
        if preferred_maintenance_window is not None:
            self._values["preferred_maintenance_window"] = preferred_maintenance_window
        if removal_policy is not None:
            self._values["removal_policy"] = removal_policy
        if s3_export_buckets is not None:
            self._values["s3_export_buckets"] = s3_export_buckets
        if s3_export_role is not None:
            self._values["s3_export_role"] = s3_export_role
        if s3_import_buckets is not None:
            self._values["s3_import_buckets"] = s3_import_buckets
        if s3_import_role is not None:
            self._values["s3_import_role"] = s3_import_role
        if subnet_group is not None:
            self._values["subnet_group"] = subnet_group

    @builtins.property
    def engine(self) -> "IClusterEngine":
        """What kind of database to start."""
        result = self._values.get("engine")
        assert result is not None, "Required property 'engine' is missing"
        return result

    @builtins.property
    def instance_props(self) -> "InstanceProps":
        """Settings for the individual instances that are launched."""
        result = self._values.get("instance_props")
        assert result is not None, "Required property 'instance_props' is missing"
        return result

    @builtins.property
    def snapshot_identifier(self) -> builtins.str:
        """The identifier for the DB instance snapshot or DB cluster snapshot to restore from.

        You can use either the name or the Amazon Resource Name (ARN) to specify a DB cluster snapshot.
        However, you can use only the ARN to specify a DB instance snapshot.
        """
        result = self._values.get("snapshot_identifier")
        assert result is not None, "Required property 'snapshot_identifier' is missing"
        return result

    @builtins.property
    def backup(self) -> typing.Optional[BackupProps]:
        """Backup settings.

        :default:

        - Backup retention period for automated backups is 1 day.
        Backup preferred window is set to a 30-minute window selected at random from an
        8-hour block of time for each AWS Region, occurring on a random day of the week.

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow
        """
        result = self._values.get("backup")
        return result

    @builtins.property
    def cloudwatch_logs_exports(self) -> typing.Optional[typing.List[builtins.str]]:
        """The list of log types that need to be enabled for exporting to CloudWatch Logs.

        :default: - no log exports
        """
        result = self._values.get("cloudwatch_logs_exports")
        return result

    @builtins.property
    def cloudwatch_logs_retention(
        self,
    ) -> typing.Optional[aws_cdk.aws_logs.RetentionDays]:
        """The number of days log events are kept in CloudWatch Logs.

        When updating
        this property, unsetting it doesn't remove the log retention policy. To
        remove the retention policy, set the value to ``Infinity``.

        :default: - logs never expire
        """
        result = self._values.get("cloudwatch_logs_retention")
        return result

    @builtins.property
    def cloudwatch_logs_retention_role(self) -> typing.Optional[aws_cdk.aws_iam.IRole]:
        """The IAM role for the Lambda function associated with the custom resource that sets the retention policy.

        :default: - a new role is created.
        """
        result = self._values.get("cloudwatch_logs_retention_role")
        return result

    @builtins.property
    def cluster_identifier(self) -> typing.Optional[builtins.str]:
        """An optional identifier for the cluster.

        :default: - A name is automatically generated.
        """
        result = self._values.get("cluster_identifier")
        return result

    @builtins.property
    def default_database_name(self) -> typing.Optional[builtins.str]:
        """Name of a database which is automatically created inside the cluster.

        :default: - Database is not created in cluster.
        """
        result = self._values.get("default_database_name")
        return result

    @builtins.property
    def deletion_protection(self) -> typing.Optional[builtins.bool]:
        """Indicates whether the DB cluster should have deletion protection enabled.

        :default: - true if ``removalPolicy`` is RETAIN, false otherwise
        """
        result = self._values.get("deletion_protection")
        return result

    @builtins.property
    def instance_identifier_base(self) -> typing.Optional[builtins.str]:
        """Base identifier for instances.

        Every replica is named by appending the replica number to this string, 1-based.

        :default:

        - clusterIdentifier is used with the word "Instance" appended.
        If clusterIdentifier is not provided, the identifier is automatically generated.
        """
        result = self._values.get("instance_identifier_base")
        return result

    @builtins.property
    def instances(self) -> typing.Optional[jsii.Number]:
        """How many replicas/instances to create.

        Has to be at least 1.

        :default: 2
        """
        result = self._values.get("instances")
        return result

    @builtins.property
    def monitoring_interval(self) -> typing.Optional[aws_cdk.core.Duration]:
        """The interval, in seconds, between points when Amazon RDS collects enhanced monitoring metrics for the DB instances.

        :default: no enhanced monitoring
        """
        result = self._values.get("monitoring_interval")
        return result

    @builtins.property
    def monitoring_role(self) -> typing.Optional[aws_cdk.aws_iam.IRole]:
        """Role that will be used to manage DB instances monitoring.

        :default: - A role is automatically created for you
        """
        result = self._values.get("monitoring_role")
        return result

    @builtins.property
    def parameter_group(self) -> typing.Optional["IParameterGroup"]:
        """Additional parameters to pass to the database engine.

        :default: - No parameter group.
        """
        result = self._values.get("parameter_group")
        return result

    @builtins.property
    def port(self) -> typing.Optional[jsii.Number]:
        """What port to listen on.

        :default: - The default for the engine is used.
        """
        result = self._values.get("port")
        return result

    @builtins.property
    def preferred_maintenance_window(self) -> typing.Optional[builtins.str]:
        """A preferred maintenance window day/time range. Should be specified as a range ddd:hh24:mi-ddd:hh24:mi (24H Clock UTC).

        Example: 'Sun:23:45-Mon:00:15'

        :default:

        - 30-minute window selected at random from an 8-hour block of time for
        each AWS Region, occurring on a random day of the week.

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_UpgradeDBInstance.Maintenance.html#Concepts.DBMaintenance
        """
        result = self._values.get("preferred_maintenance_window")
        return result

    @builtins.property
    def removal_policy(self) -> typing.Optional[aws_cdk.core.RemovalPolicy]:
        """The removal policy to apply when the cluster and its instances are removed from the stack or replaced during an update.

        :default: - RemovalPolicy.SNAPSHOT (remove the cluster and instances, but retain a snapshot of the data)
        """
        result = self._values.get("removal_policy")
        return result

    @builtins.property
    def s3_export_buckets(self) -> typing.Optional[typing.List[aws_cdk.aws_s3.IBucket]]:
        """S3 buckets that you want to load data into. This feature is only supported by the Aurora database engine.

        This property must not be used if ``s3ExportRole`` is used.

        For MySQL:

        :default: - None

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/postgresql-s3-export.html
        """
        result = self._values.get("s3_export_buckets")
        return result

    @builtins.property
    def s3_export_role(self) -> typing.Optional[aws_cdk.aws_iam.IRole]:
        """Role that will be associated with this DB cluster to enable S3 export.

        This feature is only supported by the Aurora database engine.

        This property must not be used if ``s3ExportBuckets`` is used.

        For MySQL:

        :default: - New role is created if ``s3ExportBuckets`` is set, no role is defined otherwise

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/postgresql-s3-export.html
        """
        result = self._values.get("s3_export_role")
        return result

    @builtins.property
    def s3_import_buckets(self) -> typing.Optional[typing.List[aws_cdk.aws_s3.IBucket]]:
        """S3 buckets that you want to load data from. This feature is only supported by the Aurora database engine.

        This property must not be used if ``s3ImportRole`` is used.

        For MySQL:

        :default: - None

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraPostgreSQL.Migrating.html
        """
        result = self._values.get("s3_import_buckets")
        return result

    @builtins.property
    def s3_import_role(self) -> typing.Optional[aws_cdk.aws_iam.IRole]:
        """Role that will be associated with this DB cluster to enable S3 import.

        This feature is only supported by the Aurora database engine.

        This property must not be used if ``s3ImportBuckets`` is used.

        For MySQL:

        :default: - New role is created if ``s3ImportBuckets`` is set, no role is defined otherwise

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraPostgreSQL.Migrating.html
        """
        result = self._values.get("s3_import_role")
        return result

    @builtins.property
    def subnet_group(self) -> typing.Optional["ISubnetGroup"]:
        """Existing subnet group for the cluster.

        :default: - a new subnet group will be created.
        """
        result = self._values.get("subnet_group")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DatabaseClusterFromSnapshotProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.DatabaseClusterProps",
    jsii_struct_bases=[],
    name_mapping={
        "engine": "engine",
        "instance_props": "instanceProps",
        "backup": "backup",
        "cloudwatch_logs_exports": "cloudwatchLogsExports",
        "cloudwatch_logs_retention": "cloudwatchLogsRetention",
        "cloudwatch_logs_retention_role": "cloudwatchLogsRetentionRole",
        "cluster_identifier": "clusterIdentifier",
        "credentials": "credentials",
        "default_database_name": "defaultDatabaseName",
        "deletion_protection": "deletionProtection",
        "instance_identifier_base": "instanceIdentifierBase",
        "instances": "instances",
        "monitoring_interval": "monitoringInterval",
        "monitoring_role": "monitoringRole",
        "parameter_group": "parameterGroup",
        "port": "port",
        "preferred_maintenance_window": "preferredMaintenanceWindow",
        "removal_policy": "removalPolicy",
        "s3_export_buckets": "s3ExportBuckets",
        "s3_export_role": "s3ExportRole",
        "s3_import_buckets": "s3ImportBuckets",
        "s3_import_role": "s3ImportRole",
        "storage_encrypted": "storageEncrypted",
        "storage_encryption_key": "storageEncryptionKey",
        "subnet_group": "subnetGroup",
    },
)
class DatabaseClusterProps:
    def __init__(
        self,
        *,
        engine: "IClusterEngine",
        instance_props: "InstanceProps",
        backup: typing.Optional[BackupProps] = None,
        cloudwatch_logs_exports: typing.Optional[typing.List[builtins.str]] = None,
        cloudwatch_logs_retention: typing.Optional[aws_cdk.aws_logs.RetentionDays] = None,
        cloudwatch_logs_retention_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        cluster_identifier: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[Credentials] = None,
        default_database_name: typing.Optional[builtins.str] = None,
        deletion_protection: typing.Optional[builtins.bool] = None,
        instance_identifier_base: typing.Optional[builtins.str] = None,
        instances: typing.Optional[jsii.Number] = None,
        monitoring_interval: typing.Optional[aws_cdk.core.Duration] = None,
        monitoring_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        parameter_group: typing.Optional["IParameterGroup"] = None,
        port: typing.Optional[jsii.Number] = None,
        preferred_maintenance_window: typing.Optional[builtins.str] = None,
        removal_policy: typing.Optional[aws_cdk.core.RemovalPolicy] = None,
        s3_export_buckets: typing.Optional[typing.List[aws_cdk.aws_s3.IBucket]] = None,
        s3_export_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        s3_import_buckets: typing.Optional[typing.List[aws_cdk.aws_s3.IBucket]] = None,
        s3_import_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        storage_encrypted: typing.Optional[builtins.bool] = None,
        storage_encryption_key: typing.Optional[aws_cdk.aws_kms.IKey] = None,
        subnet_group: typing.Optional["ISubnetGroup"] = None,
    ) -> None:
        """Properties for a new database cluster.

        :param engine: What kind of database to start.
        :param instance_props: Settings for the individual instances that are launched.
        :param backup: Backup settings. Default: - Backup retention period for automated backups is 1 day. Backup preferred window is set to a 30-minute window selected at random from an 8-hour block of time for each AWS Region, occurring on a random day of the week.
        :param cloudwatch_logs_exports: The list of log types that need to be enabled for exporting to CloudWatch Logs. Default: - no log exports
        :param cloudwatch_logs_retention: The number of days log events are kept in CloudWatch Logs. When updating this property, unsetting it doesn't remove the log retention policy. To remove the retention policy, set the value to ``Infinity``. Default: - logs never expire
        :param cloudwatch_logs_retention_role: The IAM role for the Lambda function associated with the custom resource that sets the retention policy. Default: - a new role is created.
        :param cluster_identifier: An optional identifier for the cluster. Default: - A name is automatically generated.
        :param credentials: Credentials for the administrative user. Default: - A username of 'admin' (or 'postgres' for PostgreSQL) and SecretsManager-generated password
        :param default_database_name: Name of a database which is automatically created inside the cluster. Default: - Database is not created in cluster.
        :param deletion_protection: Indicates whether the DB cluster should have deletion protection enabled. Default: - true if ``removalPolicy`` is RETAIN, false otherwise
        :param instance_identifier_base: Base identifier for instances. Every replica is named by appending the replica number to this string, 1-based. Default: - clusterIdentifier is used with the word "Instance" appended. If clusterIdentifier is not provided, the identifier is automatically generated.
        :param instances: How many replicas/instances to create. Has to be at least 1. Default: 2
        :param monitoring_interval: The interval, in seconds, between points when Amazon RDS collects enhanced monitoring metrics for the DB instances. Default: no enhanced monitoring
        :param monitoring_role: Role that will be used to manage DB instances monitoring. Default: - A role is automatically created for you
        :param parameter_group: Additional parameters to pass to the database engine. Default: - No parameter group.
        :param port: What port to listen on. Default: - The default for the engine is used.
        :param preferred_maintenance_window: A preferred maintenance window day/time range. Should be specified as a range ddd:hh24:mi-ddd:hh24:mi (24H Clock UTC). Example: 'Sun:23:45-Mon:00:15' Default: - 30-minute window selected at random from an 8-hour block of time for each AWS Region, occurring on a random day of the week.
        :param removal_policy: The removal policy to apply when the cluster and its instances are removed from the stack or replaced during an update. Default: - RemovalPolicy.SNAPSHOT (remove the cluster and instances, but retain a snapshot of the data)
        :param s3_export_buckets: S3 buckets that you want to load data into. This feature is only supported by the Aurora database engine. This property must not be used if ``s3ExportRole`` is used. For MySQL: Default: - None
        :param s3_export_role: Role that will be associated with this DB cluster to enable S3 export. This feature is only supported by the Aurora database engine. This property must not be used if ``s3ExportBuckets`` is used. For MySQL: Default: - New role is created if ``s3ExportBuckets`` is set, no role is defined otherwise
        :param s3_import_buckets: S3 buckets that you want to load data from. This feature is only supported by the Aurora database engine. This property must not be used if ``s3ImportRole`` is used. For MySQL: Default: - None
        :param s3_import_role: Role that will be associated with this DB cluster to enable S3 import. This feature is only supported by the Aurora database engine. This property must not be used if ``s3ImportBuckets`` is used. For MySQL: Default: - New role is created if ``s3ImportBuckets`` is set, no role is defined otherwise
        :param storage_encrypted: Whether to enable storage encryption. Default: - true if storageEncryptionKey is provided, false otherwise
        :param storage_encryption_key: The KMS key for storage encryption. If specified, {@link storageEncrypted} will be set to ``true``. Default: - if storageEncrypted is true then the default master key, no key otherwise
        :param subnet_group: Existing subnet group for the cluster. Default: - a new subnet group will be created.
        """
        if isinstance(instance_props, dict):
            instance_props = InstanceProps(**instance_props)
        if isinstance(backup, dict):
            backup = BackupProps(**backup)
        self._values: typing.Dict[str, typing.Any] = {
            "engine": engine,
            "instance_props": instance_props,
        }
        if backup is not None:
            self._values["backup"] = backup
        if cloudwatch_logs_exports is not None:
            self._values["cloudwatch_logs_exports"] = cloudwatch_logs_exports
        if cloudwatch_logs_retention is not None:
            self._values["cloudwatch_logs_retention"] = cloudwatch_logs_retention
        if cloudwatch_logs_retention_role is not None:
            self._values["cloudwatch_logs_retention_role"] = cloudwatch_logs_retention_role
        if cluster_identifier is not None:
            self._values["cluster_identifier"] = cluster_identifier
        if credentials is not None:
            self._values["credentials"] = credentials
        if default_database_name is not None:
            self._values["default_database_name"] = default_database_name
        if deletion_protection is not None:
            self._values["deletion_protection"] = deletion_protection
        if instance_identifier_base is not None:
            self._values["instance_identifier_base"] = instance_identifier_base
        if instances is not None:
            self._values["instances"] = instances
        if monitoring_interval is not None:
            self._values["monitoring_interval"] = monitoring_interval
        if monitoring_role is not None:
            self._values["monitoring_role"] = monitoring_role
        if parameter_group is not None:
            self._values["parameter_group"] = parameter_group
        if port is not None:
            self._values["port"] = port
        if preferred_maintenance_window is not None:
            self._values["preferred_maintenance_window"] = preferred_maintenance_window
        if removal_policy is not None:
            self._values["removal_policy"] = removal_policy
        if s3_export_buckets is not None:
            self._values["s3_export_buckets"] = s3_export_buckets
        if s3_export_role is not None:
            self._values["s3_export_role"] = s3_export_role
        if s3_import_buckets is not None:
            self._values["s3_import_buckets"] = s3_import_buckets
        if s3_import_role is not None:
            self._values["s3_import_role"] = s3_import_role
        if storage_encrypted is not None:
            self._values["storage_encrypted"] = storage_encrypted
        if storage_encryption_key is not None:
            self._values["storage_encryption_key"] = storage_encryption_key
        if subnet_group is not None:
            self._values["subnet_group"] = subnet_group

    @builtins.property
    def engine(self) -> "IClusterEngine":
        """What kind of database to start."""
        result = self._values.get("engine")
        assert result is not None, "Required property 'engine' is missing"
        return result

    @builtins.property
    def instance_props(self) -> "InstanceProps":
        """Settings for the individual instances that are launched."""
        result = self._values.get("instance_props")
        assert result is not None, "Required property 'instance_props' is missing"
        return result

    @builtins.property
    def backup(self) -> typing.Optional[BackupProps]:
        """Backup settings.

        :default:

        - Backup retention period for automated backups is 1 day.
        Backup preferred window is set to a 30-minute window selected at random from an
        8-hour block of time for each AWS Region, occurring on a random day of the week.

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow
        """
        result = self._values.get("backup")
        return result

    @builtins.property
    def cloudwatch_logs_exports(self) -> typing.Optional[typing.List[builtins.str]]:
        """The list of log types that need to be enabled for exporting to CloudWatch Logs.

        :default: - no log exports
        """
        result = self._values.get("cloudwatch_logs_exports")
        return result

    @builtins.property
    def cloudwatch_logs_retention(
        self,
    ) -> typing.Optional[aws_cdk.aws_logs.RetentionDays]:
        """The number of days log events are kept in CloudWatch Logs.

        When updating
        this property, unsetting it doesn't remove the log retention policy. To
        remove the retention policy, set the value to ``Infinity``.

        :default: - logs never expire
        """
        result = self._values.get("cloudwatch_logs_retention")
        return result

    @builtins.property
    def cloudwatch_logs_retention_role(self) -> typing.Optional[aws_cdk.aws_iam.IRole]:
        """The IAM role for the Lambda function associated with the custom resource that sets the retention policy.

        :default: - a new role is created.
        """
        result = self._values.get("cloudwatch_logs_retention_role")
        return result

    @builtins.property
    def cluster_identifier(self) -> typing.Optional[builtins.str]:
        """An optional identifier for the cluster.

        :default: - A name is automatically generated.
        """
        result = self._values.get("cluster_identifier")
        return result

    @builtins.property
    def credentials(self) -> typing.Optional[Credentials]:
        """Credentials for the administrative user.

        :default: - A username of 'admin' (or 'postgres' for PostgreSQL) and SecretsManager-generated password
        """
        result = self._values.get("credentials")
        return result

    @builtins.property
    def default_database_name(self) -> typing.Optional[builtins.str]:
        """Name of a database which is automatically created inside the cluster.

        :default: - Database is not created in cluster.
        """
        result = self._values.get("default_database_name")
        return result

    @builtins.property
    def deletion_protection(self) -> typing.Optional[builtins.bool]:
        """Indicates whether the DB cluster should have deletion protection enabled.

        :default: - true if ``removalPolicy`` is RETAIN, false otherwise
        """
        result = self._values.get("deletion_protection")
        return result

    @builtins.property
    def instance_identifier_base(self) -> typing.Optional[builtins.str]:
        """Base identifier for instances.

        Every replica is named by appending the replica number to this string, 1-based.

        :default:

        - clusterIdentifier is used with the word "Instance" appended.
        If clusterIdentifier is not provided, the identifier is automatically generated.
        """
        result = self._values.get("instance_identifier_base")
        return result

    @builtins.property
    def instances(self) -> typing.Optional[jsii.Number]:
        """How many replicas/instances to create.

        Has to be at least 1.

        :default: 2
        """
        result = self._values.get("instances")
        return result

    @builtins.property
    def monitoring_interval(self) -> typing.Optional[aws_cdk.core.Duration]:
        """The interval, in seconds, between points when Amazon RDS collects enhanced monitoring metrics for the DB instances.

        :default: no enhanced monitoring
        """
        result = self._values.get("monitoring_interval")
        return result

    @builtins.property
    def monitoring_role(self) -> typing.Optional[aws_cdk.aws_iam.IRole]:
        """Role that will be used to manage DB instances monitoring.

        :default: - A role is automatically created for you
        """
        result = self._values.get("monitoring_role")
        return result

    @builtins.property
    def parameter_group(self) -> typing.Optional["IParameterGroup"]:
        """Additional parameters to pass to the database engine.

        :default: - No parameter group.
        """
        result = self._values.get("parameter_group")
        return result

    @builtins.property
    def port(self) -> typing.Optional[jsii.Number]:
        """What port to listen on.

        :default: - The default for the engine is used.
        """
        result = self._values.get("port")
        return result

    @builtins.property
    def preferred_maintenance_window(self) -> typing.Optional[builtins.str]:
        """A preferred maintenance window day/time range. Should be specified as a range ddd:hh24:mi-ddd:hh24:mi (24H Clock UTC).

        Example: 'Sun:23:45-Mon:00:15'

        :default:

        - 30-minute window selected at random from an 8-hour block of time for
        each AWS Region, occurring on a random day of the week.

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_UpgradeDBInstance.Maintenance.html#Concepts.DBMaintenance
        """
        result = self._values.get("preferred_maintenance_window")
        return result

    @builtins.property
    def removal_policy(self) -> typing.Optional[aws_cdk.core.RemovalPolicy]:
        """The removal policy to apply when the cluster and its instances are removed from the stack or replaced during an update.

        :default: - RemovalPolicy.SNAPSHOT (remove the cluster and instances, but retain a snapshot of the data)
        """
        result = self._values.get("removal_policy")
        return result

    @builtins.property
    def s3_export_buckets(self) -> typing.Optional[typing.List[aws_cdk.aws_s3.IBucket]]:
        """S3 buckets that you want to load data into. This feature is only supported by the Aurora database engine.

        This property must not be used if ``s3ExportRole`` is used.

        For MySQL:

        :default: - None

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/postgresql-s3-export.html
        """
        result = self._values.get("s3_export_buckets")
        return result

    @builtins.property
    def s3_export_role(self) -> typing.Optional[aws_cdk.aws_iam.IRole]:
        """Role that will be associated with this DB cluster to enable S3 export.

        This feature is only supported by the Aurora database engine.

        This property must not be used if ``s3ExportBuckets`` is used.

        For MySQL:

        :default: - New role is created if ``s3ExportBuckets`` is set, no role is defined otherwise

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/postgresql-s3-export.html
        """
        result = self._values.get("s3_export_role")
        return result

    @builtins.property
    def s3_import_buckets(self) -> typing.Optional[typing.List[aws_cdk.aws_s3.IBucket]]:
        """S3 buckets that you want to load data from. This feature is only supported by the Aurora database engine.

        This property must not be used if ``s3ImportRole`` is used.

        For MySQL:

        :default: - None

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraPostgreSQL.Migrating.html
        """
        result = self._values.get("s3_import_buckets")
        return result

    @builtins.property
    def s3_import_role(self) -> typing.Optional[aws_cdk.aws_iam.IRole]:
        """Role that will be associated with this DB cluster to enable S3 import.

        This feature is only supported by the Aurora database engine.

        This property must not be used if ``s3ImportBuckets`` is used.

        For MySQL:

        :default: - New role is created if ``s3ImportBuckets`` is set, no role is defined otherwise

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraPostgreSQL.Migrating.html
        """
        result = self._values.get("s3_import_role")
        return result

    @builtins.property
    def storage_encrypted(self) -> typing.Optional[builtins.bool]:
        """Whether to enable storage encryption.

        :default: - true if storageEncryptionKey is provided, false otherwise
        """
        result = self._values.get("storage_encrypted")
        return result

    @builtins.property
    def storage_encryption_key(self) -> typing.Optional[aws_cdk.aws_kms.IKey]:
        """The KMS key for storage encryption.

        If specified, {@link storageEncrypted} will be set to ``true``.

        :default: - if storageEncrypted is true then the default master key, no key otherwise
        """
        result = self._values.get("storage_encryption_key")
        return result

    @builtins.property
    def subnet_group(self) -> typing.Optional["ISubnetGroup"]:
        """Existing subnet group for the cluster.

        :default: - a new subnet group will be created.
        """
        result = self._values.get("subnet_group")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DatabaseClusterProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.DatabaseInstanceAttributes",
    jsii_struct_bases=[],
    name_mapping={
        "instance_endpoint_address": "instanceEndpointAddress",
        "instance_identifier": "instanceIdentifier",
        "port": "port",
        "security_groups": "securityGroups",
        "engine": "engine",
    },
)
class DatabaseInstanceAttributes:
    def __init__(
        self,
        *,
        instance_endpoint_address: builtins.str,
        instance_identifier: builtins.str,
        port: jsii.Number,
        security_groups: typing.List[aws_cdk.aws_ec2.ISecurityGroup],
        engine: typing.Optional["IInstanceEngine"] = None,
    ) -> None:
        """Properties that describe an existing instance.

        :param instance_endpoint_address: The endpoint address.
        :param instance_identifier: The instance identifier.
        :param port: The database port.
        :param security_groups: The security groups of the instance.
        :param engine: The engine of the existing database Instance. Default: - the imported Instance's engine is unknown
        """
        self._values: typing.Dict[str, typing.Any] = {
            "instance_endpoint_address": instance_endpoint_address,
            "instance_identifier": instance_identifier,
            "port": port,
            "security_groups": security_groups,
        }
        if engine is not None:
            self._values["engine"] = engine

    @builtins.property
    def instance_endpoint_address(self) -> builtins.str:
        """The endpoint address."""
        result = self._values.get("instance_endpoint_address")
        assert result is not None, "Required property 'instance_endpoint_address' is missing"
        return result

    @builtins.property
    def instance_identifier(self) -> builtins.str:
        """The instance identifier."""
        result = self._values.get("instance_identifier")
        assert result is not None, "Required property 'instance_identifier' is missing"
        return result

    @builtins.property
    def port(self) -> jsii.Number:
        """The database port."""
        result = self._values.get("port")
        assert result is not None, "Required property 'port' is missing"
        return result

    @builtins.property
    def security_groups(self) -> typing.List[aws_cdk.aws_ec2.ISecurityGroup]:
        """The security groups of the instance."""
        result = self._values.get("security_groups")
        assert result is not None, "Required property 'security_groups' is missing"
        return result

    @builtins.property
    def engine(self) -> typing.Optional["IInstanceEngine"]:
        """The engine of the existing database Instance.

        :default: - the imported Instance's engine is unknown
        """
        result = self._values.get("engine")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DatabaseInstanceAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class DatabaseInstanceEngine(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.DatabaseInstanceEngine",
):
    """A database instance engine.

    Provides mapping to DatabaseEngine used for
    secret rotation.
    """

    def __init__(self) -> None:
        jsii.create(DatabaseInstanceEngine, self, [])

    @jsii.member(jsii_name="mariaDb")
    @builtins.classmethod
    def maria_db(cls, *, version: "MariaDbEngineVersion") -> "IInstanceEngine":
        """Creates a new MariaDB instance engine.

        :param version: The exact version of the engine to use.
        """
        props = MariaDbInstanceEngineProps(version=version)

        return jsii.sinvoke(cls, "mariaDb", [props])

    @jsii.member(jsii_name="mysql")
    @builtins.classmethod
    def mysql(cls, *, version: "MysqlEngineVersion") -> "IInstanceEngine":
        """Creates a new MySQL instance engine.

        :param version: The exact version of the engine to use.
        """
        props = MySqlInstanceEngineProps(version=version)

        return jsii.sinvoke(cls, "mysql", [props])

    @jsii.member(jsii_name="oracleEe")
    @builtins.classmethod
    def oracle_ee(cls, *, version: "OracleEngineVersion") -> "IInstanceEngine":
        """Creates a new Oracle Enterprise Edition instance engine.

        :param version: The exact version of the engine to use.
        """
        props = OracleEeInstanceEngineProps(version=version)

        return jsii.sinvoke(cls, "oracleEe", [props])

    @jsii.member(jsii_name="oracleSe")
    @builtins.classmethod
    def oracle_se(cls, *, version: "OracleLegacyEngineVersion") -> "IInstanceEngine":
        """(deprecated) Creates a new Oracle Standard Edition instance engine.

        :param version: (deprecated) The exact version of the engine to use.

        :deprecated: instances can no longer be created with this engine. See https://forums.aws.amazon.com/ann.jspa?annID=7341

        :stability: deprecated
        """
        props = OracleSeInstanceEngineProps(version=version)

        return jsii.sinvoke(cls, "oracleSe", [props])

    @jsii.member(jsii_name="oracleSe1")
    @builtins.classmethod
    def oracle_se1(cls, *, version: "OracleLegacyEngineVersion") -> "IInstanceEngine":
        """(deprecated) Creates a new Oracle Standard Edition 1 instance engine.

        :param version: (deprecated) The exact version of the engine to use.

        :deprecated: instances can no longer be created with this engine. See https://forums.aws.amazon.com/ann.jspa?annID=7341

        :stability: deprecated
        """
        props = OracleSe1InstanceEngineProps(version=version)

        return jsii.sinvoke(cls, "oracleSe1", [props])

    @jsii.member(jsii_name="oracleSe2")
    @builtins.classmethod
    def oracle_se2(cls, *, version: "OracleEngineVersion") -> "IInstanceEngine":
        """Creates a new Oracle Standard Edition 1 instance engine.

        :param version: The exact version of the engine to use.
        """
        props = OracleSe2InstanceEngineProps(version=version)

        return jsii.sinvoke(cls, "oracleSe2", [props])

    @jsii.member(jsii_name="postgres")
    @builtins.classmethod
    def postgres(cls, *, version: "PostgresEngineVersion") -> "IInstanceEngine":
        """Creates a new PostgreSQL instance engine.

        :param version: The exact version of the engine to use.
        """
        props = PostgresInstanceEngineProps(version=version)

        return jsii.sinvoke(cls, "postgres", [props])

    @jsii.member(jsii_name="sqlServerEe")
    @builtins.classmethod
    def sql_server_ee(cls, *, version: "SqlServerEngineVersion") -> "IInstanceEngine":
        """Creates a new SQL Server Enterprise Edition instance engine.

        :param version: The exact version of the engine to use.
        """
        props = SqlServerEeInstanceEngineProps(version=version)

        return jsii.sinvoke(cls, "sqlServerEe", [props])

    @jsii.member(jsii_name="sqlServerEx")
    @builtins.classmethod
    def sql_server_ex(cls, *, version: "SqlServerEngineVersion") -> "IInstanceEngine":
        """Creates a new SQL Server Express Edition instance engine.

        :param version: The exact version of the engine to use.
        """
        props = SqlServerExInstanceEngineProps(version=version)

        return jsii.sinvoke(cls, "sqlServerEx", [props])

    @jsii.member(jsii_name="sqlServerSe")
    @builtins.classmethod
    def sql_server_se(cls, *, version: "SqlServerEngineVersion") -> "IInstanceEngine":
        """Creates a new SQL Server Standard Edition instance engine.

        :param version: The exact version of the engine to use.
        """
        props = SqlServerSeInstanceEngineProps(version=version)

        return jsii.sinvoke(cls, "sqlServerSe", [props])

    @jsii.member(jsii_name="sqlServerWeb")
    @builtins.classmethod
    def sql_server_web(cls, *, version: "SqlServerEngineVersion") -> "IInstanceEngine":
        """Creates a new SQL Server Web Edition instance engine.

        :param version: The exact version of the engine to use.
        """
        props = SqlServerWebInstanceEngineProps(version=version)

        return jsii.sinvoke(cls, "sqlServerWeb", [props])

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="MARIADB")
    def MARIADB(cls) -> "IInstanceEngine":
        """(deprecated) The unversioned 'mariadb' instance engine.

        :deprecated:

        using unversioned engines is an availability risk.
        We recommend using versioned engines created using the {@link mariaDb()} method

        :stability: deprecated
        """
        return jsii.sget(cls, "MARIADB")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="MYSQL")
    def MYSQL(cls) -> "IInstanceEngine":
        """(deprecated) The unversioned 'mysql' instance engine.

        :deprecated:

        using unversioned engines is an availability risk.
        We recommend using versioned engines created using the {@link mysql()} method

        :stability: deprecated
        """
        return jsii.sget(cls, "MYSQL")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="ORACLE_EE")
    def ORACLE_EE(cls) -> "IInstanceEngine":
        """(deprecated) The unversioned 'oracle-ee' instance engine.

        :deprecated:

        using unversioned engines is an availability risk.
        We recommend using versioned engines created using the {@link oracleEe()} method

        :stability: deprecated
        """
        return jsii.sget(cls, "ORACLE_EE")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="ORACLE_SE")
    def ORACLE_SE(cls) -> "IInstanceEngine":
        """(deprecated) The unversioned 'oracle-se' instance engine.

        :deprecated: instances can no longer be created with this engine. See https://forums.aws.amazon.com/ann.jspa?annID=7341

        :stability: deprecated
        """
        return jsii.sget(cls, "ORACLE_SE")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="ORACLE_SE1")
    def ORACLE_SE1(cls) -> "IInstanceEngine":
        """(deprecated) The unversioned 'oracle-se1' instance engine.

        :deprecated: instances can no longer be created with this engine. See https://forums.aws.amazon.com/ann.jspa?annID=7341

        :stability: deprecated
        """
        return jsii.sget(cls, "ORACLE_SE1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="ORACLE_SE2")
    def ORACLE_SE2(cls) -> "IInstanceEngine":
        """(deprecated) The unversioned 'oracle-se2' instance engine.

        :deprecated:

        using unversioned engines is an availability risk.
        We recommend using versioned engines created using the {@link oracleSe2()} method

        :stability: deprecated
        """
        return jsii.sget(cls, "ORACLE_SE2")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="POSTGRES")
    def POSTGRES(cls) -> "IInstanceEngine":
        """(deprecated) The unversioned 'postgres' instance engine.

        :deprecated:

        using unversioned engines is an availability risk.
        We recommend using versioned engines created using the {@link postgres()} method

        :stability: deprecated
        """
        return jsii.sget(cls, "POSTGRES")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="SQL_SERVER_EE")
    def SQL_SERVER_EE(cls) -> "IInstanceEngine":
        """(deprecated) The unversioned 'sqlserver-ee' instance engine.

        :deprecated:

        using unversioned engines is an availability risk.
        We recommend using versioned engines created using the {@link sqlServerEe()} method

        :stability: deprecated
        """
        return jsii.sget(cls, "SQL_SERVER_EE")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="SQL_SERVER_EX")
    def SQL_SERVER_EX(cls) -> "IInstanceEngine":
        """(deprecated) The unversioned 'sqlserver-ex' instance engine.

        :deprecated:

        using unversioned engines is an availability risk.
        We recommend using versioned engines created using the {@link sqlServerEx()} method

        :stability: deprecated
        """
        return jsii.sget(cls, "SQL_SERVER_EX")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="SQL_SERVER_SE")
    def SQL_SERVER_SE(cls) -> "IInstanceEngine":
        """(deprecated) The unversioned 'sqlserver-se' instance engine.

        :deprecated:

        using unversioned engines is an availability risk.
        We recommend using versioned engines created using the {@link sqlServerSe()} method

        :stability: deprecated
        """
        return jsii.sget(cls, "SQL_SERVER_SE")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="SQL_SERVER_WEB")
    def SQL_SERVER_WEB(cls) -> "IInstanceEngine":
        """(deprecated) The unversioned 'sqlserver-web' instance engine.

        :deprecated:

        using unversioned engines is an availability risk.
        We recommend using versioned engines created using the {@link sqlServerWeb()} method

        :stability: deprecated
        """
        return jsii.sget(cls, "SQL_SERVER_WEB")


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.DatabaseInstanceNewProps",
    jsii_struct_bases=[],
    name_mapping={
        "vpc": "vpc",
        "auto_minor_version_upgrade": "autoMinorVersionUpgrade",
        "availability_zone": "availabilityZone",
        "backup_retention": "backupRetention",
        "cloudwatch_logs_exports": "cloudwatchLogsExports",
        "cloudwatch_logs_retention": "cloudwatchLogsRetention",
        "cloudwatch_logs_retention_role": "cloudwatchLogsRetentionRole",
        "copy_tags_to_snapshot": "copyTagsToSnapshot",
        "delete_automated_backups": "deleteAutomatedBackups",
        "deletion_protection": "deletionProtection",
        "domain": "domain",
        "domain_role": "domainRole",
        "enable_performance_insights": "enablePerformanceInsights",
        "iam_authentication": "iamAuthentication",
        "instance_identifier": "instanceIdentifier",
        "iops": "iops",
        "max_allocated_storage": "maxAllocatedStorage",
        "monitoring_interval": "monitoringInterval",
        "monitoring_role": "monitoringRole",
        "multi_az": "multiAz",
        "option_group": "optionGroup",
        "performance_insight_encryption_key": "performanceInsightEncryptionKey",
        "performance_insight_retention": "performanceInsightRetention",
        "port": "port",
        "preferred_backup_window": "preferredBackupWindow",
        "preferred_maintenance_window": "preferredMaintenanceWindow",
        "processor_features": "processorFeatures",
        "publicly_accessible": "publiclyAccessible",
        "removal_policy": "removalPolicy",
        "s3_export_buckets": "s3ExportBuckets",
        "s3_export_role": "s3ExportRole",
        "s3_import_buckets": "s3ImportBuckets",
        "s3_import_role": "s3ImportRole",
        "security_groups": "securityGroups",
        "storage_type": "storageType",
        "subnet_group": "subnetGroup",
        "vpc_placement": "vpcPlacement",
        "vpc_subnets": "vpcSubnets",
    },
)
class DatabaseInstanceNewProps:
    def __init__(
        self,
        *,
        vpc: aws_cdk.aws_ec2.IVpc,
        auto_minor_version_upgrade: typing.Optional[builtins.bool] = None,
        availability_zone: typing.Optional[builtins.str] = None,
        backup_retention: typing.Optional[aws_cdk.core.Duration] = None,
        cloudwatch_logs_exports: typing.Optional[typing.List[builtins.str]] = None,
        cloudwatch_logs_retention: typing.Optional[aws_cdk.aws_logs.RetentionDays] = None,
        cloudwatch_logs_retention_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        copy_tags_to_snapshot: typing.Optional[builtins.bool] = None,
        delete_automated_backups: typing.Optional[builtins.bool] = None,
        deletion_protection: typing.Optional[builtins.bool] = None,
        domain: typing.Optional[builtins.str] = None,
        domain_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        enable_performance_insights: typing.Optional[builtins.bool] = None,
        iam_authentication: typing.Optional[builtins.bool] = None,
        instance_identifier: typing.Optional[builtins.str] = None,
        iops: typing.Optional[jsii.Number] = None,
        max_allocated_storage: typing.Optional[jsii.Number] = None,
        monitoring_interval: typing.Optional[aws_cdk.core.Duration] = None,
        monitoring_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        multi_az: typing.Optional[builtins.bool] = None,
        option_group: typing.Optional["IOptionGroup"] = None,
        performance_insight_encryption_key: typing.Optional[aws_cdk.aws_kms.IKey] = None,
        performance_insight_retention: typing.Optional["PerformanceInsightRetention"] = None,
        port: typing.Optional[jsii.Number] = None,
        preferred_backup_window: typing.Optional[builtins.str] = None,
        preferred_maintenance_window: typing.Optional[builtins.str] = None,
        processor_features: typing.Optional["ProcessorFeatures"] = None,
        publicly_accessible: typing.Optional[builtins.bool] = None,
        removal_policy: typing.Optional[aws_cdk.core.RemovalPolicy] = None,
        s3_export_buckets: typing.Optional[typing.List[aws_cdk.aws_s3.IBucket]] = None,
        s3_export_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        s3_import_buckets: typing.Optional[typing.List[aws_cdk.aws_s3.IBucket]] = None,
        s3_import_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        security_groups: typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]] = None,
        storage_type: typing.Optional["StorageType"] = None,
        subnet_group: typing.Optional["ISubnetGroup"] = None,
        vpc_placement: typing.Optional[aws_cdk.aws_ec2.SubnetSelection] = None,
        vpc_subnets: typing.Optional[aws_cdk.aws_ec2.SubnetSelection] = None,
    ) -> None:
        """Construction properties for a DatabaseInstanceNew.

        :param vpc: The VPC network where the DB subnet group should be created.
        :param auto_minor_version_upgrade: Indicates that minor engine upgrades are applied automatically to the DB instance during the maintenance window. Default: true
        :param availability_zone: The name of the Availability Zone where the DB instance will be located. Default: - no preference
        :param backup_retention: The number of days during which automatic DB snapshots are retained. Set to zero to disable backups. When creating a read replica, you must enable automatic backups on the source database instance by setting the backup retention to a value other than zero. Default: Duration.days(1)
        :param cloudwatch_logs_exports: The list of log types that need to be enabled for exporting to CloudWatch Logs. Default: - no log exports
        :param cloudwatch_logs_retention: The number of days log events are kept in CloudWatch Logs. When updating this property, unsetting it doesn't remove the log retention policy. To remove the retention policy, set the value to ``Infinity``. Default: - logs never expire
        :param cloudwatch_logs_retention_role: The IAM role for the Lambda function associated with the custom resource that sets the retention policy. Default: - a new role is created.
        :param copy_tags_to_snapshot: Indicates whether to copy all of the user-defined tags from the DB instance to snapshots of the DB instance. Default: true
        :param delete_automated_backups: Indicates whether automated backups should be deleted or retained when you delete a DB instance. Default: false
        :param deletion_protection: Indicates whether the DB instance should have deletion protection enabled. Default: - true if ``removalPolicy`` is RETAIN, false otherwise
        :param domain: The Active Directory directory ID to create the DB instance in. Default: - Do not join domain
        :param domain_role: The IAM role to be used when making API calls to the Directory Service. The role needs the AWS-managed policy AmazonRDSDirectoryServiceAccess or equivalent. Default: - The role will be created for you if {@link DatabaseInstanceNewProps#domain} is specified
        :param enable_performance_insights: Whether to enable Performance Insights for the DB instance. Default: - false, unless ``performanceInsightRentention`` or ``performanceInsightEncryptionKey`` is set.
        :param iam_authentication: Whether to enable mapping of AWS Identity and Access Management (IAM) accounts to database accounts. Default: false
        :param instance_identifier: A name for the DB instance. If you specify a name, AWS CloudFormation converts it to lowercase. Default: - a CloudFormation generated name
        :param iops: The number of I/O operations per second (IOPS) that the database provisions. The value must be equal to or greater than 1000. Default: - no provisioned iops
        :param max_allocated_storage: Upper limit to which RDS can scale the storage in GiB(Gibibyte). Default: - No autoscaling of RDS instance
        :param monitoring_interval: The interval, in seconds, between points when Amazon RDS collects enhanced monitoring metrics for the DB instance. Default: - no enhanced monitoring
        :param monitoring_role: Role that will be used to manage DB instance monitoring. Default: - A role is automatically created for you
        :param multi_az: Specifies if the database instance is a multiple Availability Zone deployment. Default: false
        :param option_group: The option group to associate with the instance. Default: - no option group
        :param performance_insight_encryption_key: The AWS KMS key for encryption of Performance Insights data. Default: - default master key
        :param performance_insight_retention: The amount of time, in days, to retain Performance Insights data. Default: 7
        :param port: The port for the instance. Default: - the default port for the chosen engine.
        :param preferred_backup_window: The daily time range during which automated backups are performed. Constraints: - Must be in the format ``hh24:mi-hh24:mi``. - Must be in Universal Coordinated Time (UTC). - Must not conflict with the preferred maintenance window. - Must be at least 30 minutes. Default: - a 30-minute window selected at random from an 8-hour block of time for each AWS Region. To see the time blocks available, see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow
        :param preferred_maintenance_window: The weekly time range (in UTC) during which system maintenance can occur. Format: ``ddd:hh24:mi-ddd:hh24:mi`` Constraint: Minimum 30-minute window Default: - a 30-minute window selected at random from an 8-hour block of time for each AWS Region, occurring on a random day of the week. To see the time blocks available, see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_UpgradeDBInstance.Maintenance.html#Concepts.DBMaintenance
        :param processor_features: The number of CPU cores and the number of threads per core. Default: - the default number of CPU cores and threads per core for the chosen instance class. See https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html#USER_ConfigureProcessor
        :param publicly_accessible: Indicates whether the DB instance is an internet-facing instance. Default: - ``true`` if ``vpcSubnets`` is ``subnetType: SubnetType.PUBLIC``, ``false`` otherwise
        :param removal_policy: The CloudFormation policy to apply when the instance is removed from the stack or replaced during an update. Default: - RemovalPolicy.SNAPSHOT (remove the resource, but retain a snapshot of the data)
        :param s3_export_buckets: S3 buckets that you want to load data into. This feature is only supported by the Microsoft SQL Server and Oracle engines. This property must not be used if ``s3ExportRole`` is used. For Microsoft SQL Server: Default: - None
        :param s3_export_role: Role that will be associated with this DB instance to enable S3 export. This feature is only supported by the Microsoft SQL Server and Oracle engines. This property must not be used if ``s3ExportBuckets`` is used. For Microsoft SQL Server: Default: - New role is created if ``s3ExportBuckets`` is set, no role is defined otherwise
        :param s3_import_buckets: S3 buckets that you want to load data from. This feature is only supported by the Microsoft SQL Server, Oracle, and PostgreSQL engines. This property must not be used if ``s3ImportRole`` is used. For Microsoft SQL Server: Default: - None
        :param s3_import_role: Role that will be associated with this DB instance to enable S3 import. This feature is only supported by the Microsoft SQL Server, Oracle, and PostgreSQL engines. This property must not be used if ``s3ImportBuckets`` is used. For Microsoft SQL Server: Default: - New role is created if ``s3ImportBuckets`` is set, no role is defined otherwise
        :param security_groups: The security groups to assign to the DB instance. Default: - a new security group is created
        :param storage_type: The storage type. Default: GP2
        :param subnet_group: Existing subnet group for the instance. Default: - a new subnet group will be created.
        :param vpc_placement: (deprecated) The type of subnets to add to the created DB subnet group. Default: - private subnets
        :param vpc_subnets: The type of subnets to add to the created DB subnet group. Default: - private subnets
        """
        if isinstance(processor_features, dict):
            processor_features = ProcessorFeatures(**processor_features)
        if isinstance(vpc_placement, dict):
            vpc_placement = aws_cdk.aws_ec2.SubnetSelection(**vpc_placement)
        if isinstance(vpc_subnets, dict):
            vpc_subnets = aws_cdk.aws_ec2.SubnetSelection(**vpc_subnets)
        self._values: typing.Dict[str, typing.Any] = {
            "vpc": vpc,
        }
        if auto_minor_version_upgrade is not None:
            self._values["auto_minor_version_upgrade"] = auto_minor_version_upgrade
        if availability_zone is not None:
            self._values["availability_zone"] = availability_zone
        if backup_retention is not None:
            self._values["backup_retention"] = backup_retention
        if cloudwatch_logs_exports is not None:
            self._values["cloudwatch_logs_exports"] = cloudwatch_logs_exports
        if cloudwatch_logs_retention is not None:
            self._values["cloudwatch_logs_retention"] = cloudwatch_logs_retention
        if cloudwatch_logs_retention_role is not None:
            self._values["cloudwatch_logs_retention_role"] = cloudwatch_logs_retention_role
        if copy_tags_to_snapshot is not None:
            self._values["copy_tags_to_snapshot"] = copy_tags_to_snapshot
        if delete_automated_backups is not None:
            self._values["delete_automated_backups"] = delete_automated_backups
        if deletion_protection is not None:
            self._values["deletion_protection"] = deletion_protection
        if domain is not None:
            self._values["domain"] = domain
        if domain_role is not None:
            self._values["domain_role"] = domain_role
        if enable_performance_insights is not None:
            self._values["enable_performance_insights"] = enable_performance_insights
        if iam_authentication is not None:
            self._values["iam_authentication"] = iam_authentication
        if instance_identifier is not None:
            self._values["instance_identifier"] = instance_identifier
        if iops is not None:
            self._values["iops"] = iops
        if max_allocated_storage is not None:
            self._values["max_allocated_storage"] = max_allocated_storage
        if monitoring_interval is not None:
            self._values["monitoring_interval"] = monitoring_interval
        if monitoring_role is not None:
            self._values["monitoring_role"] = monitoring_role
        if multi_az is not None:
            self._values["multi_az"] = multi_az
        if option_group is not None:
            self._values["option_group"] = option_group
        if performance_insight_encryption_key is not None:
            self._values["performance_insight_encryption_key"] = performance_insight_encryption_key
        if performance_insight_retention is not None:
            self._values["performance_insight_retention"] = performance_insight_retention
        if port is not None:
            self._values["port"] = port
        if preferred_backup_window is not None:
            self._values["preferred_backup_window"] = preferred_backup_window
        if preferred_maintenance_window is not None:
            self._values["preferred_maintenance_window"] = preferred_maintenance_window
        if processor_features is not None:
            self._values["processor_features"] = processor_features
        if publicly_accessible is not None:
            self._values["publicly_accessible"] = publicly_accessible
        if removal_policy is not None:
            self._values["removal_policy"] = removal_policy
        if s3_export_buckets is not None:
            self._values["s3_export_buckets"] = s3_export_buckets
        if s3_export_role is not None:
            self._values["s3_export_role"] = s3_export_role
        if s3_import_buckets is not None:
            self._values["s3_import_buckets"] = s3_import_buckets
        if s3_import_role is not None:
            self._values["s3_import_role"] = s3_import_role
        if security_groups is not None:
            self._values["security_groups"] = security_groups
        if storage_type is not None:
            self._values["storage_type"] = storage_type
        if subnet_group is not None:
            self._values["subnet_group"] = subnet_group
        if vpc_placement is not None:
            self._values["vpc_placement"] = vpc_placement
        if vpc_subnets is not None:
            self._values["vpc_subnets"] = vpc_subnets

    @builtins.property
    def vpc(self) -> aws_cdk.aws_ec2.IVpc:
        """The VPC network where the DB subnet group should be created."""
        result = self._values.get("vpc")
        assert result is not None, "Required property 'vpc' is missing"
        return result

    @builtins.property
    def auto_minor_version_upgrade(self) -> typing.Optional[builtins.bool]:
        """Indicates that minor engine upgrades are applied automatically to the DB instance during the maintenance window.

        :default: true
        """
        result = self._values.get("auto_minor_version_upgrade")
        return result

    @builtins.property
    def availability_zone(self) -> typing.Optional[builtins.str]:
        """The name of the Availability Zone where the DB instance will be located.

        :default: - no preference
        """
        result = self._values.get("availability_zone")
        return result

    @builtins.property
    def backup_retention(self) -> typing.Optional[aws_cdk.core.Duration]:
        """The number of days during which automatic DB snapshots are retained.

        Set to zero to disable backups.
        When creating a read replica, you must enable automatic backups on the source
        database instance by setting the backup retention to a value other than zero.

        :default: Duration.days(1)
        """
        result = self._values.get("backup_retention")
        return result

    @builtins.property
    def cloudwatch_logs_exports(self) -> typing.Optional[typing.List[builtins.str]]:
        """The list of log types that need to be enabled for exporting to CloudWatch Logs.

        :default: - no log exports
        """
        result = self._values.get("cloudwatch_logs_exports")
        return result

    @builtins.property
    def cloudwatch_logs_retention(
        self,
    ) -> typing.Optional[aws_cdk.aws_logs.RetentionDays]:
        """The number of days log events are kept in CloudWatch Logs.

        When updating
        this property, unsetting it doesn't remove the log retention policy. To
        remove the retention policy, set the value to ``Infinity``.

        :default: - logs never expire
        """
        result = self._values.get("cloudwatch_logs_retention")
        return result

    @builtins.property
    def cloudwatch_logs_retention_role(self) -> typing.Optional[aws_cdk.aws_iam.IRole]:
        """The IAM role for the Lambda function associated with the custom resource that sets the retention policy.

        :default: - a new role is created.
        """
        result = self._values.get("cloudwatch_logs_retention_role")
        return result

    @builtins.property
    def copy_tags_to_snapshot(self) -> typing.Optional[builtins.bool]:
        """Indicates whether to copy all of the user-defined tags from the DB instance to snapshots of the DB instance.

        :default: true
        """
        result = self._values.get("copy_tags_to_snapshot")
        return result

    @builtins.property
    def delete_automated_backups(self) -> typing.Optional[builtins.bool]:
        """Indicates whether automated backups should be deleted or retained when you delete a DB instance.

        :default: false
        """
        result = self._values.get("delete_automated_backups")
        return result

    @builtins.property
    def deletion_protection(self) -> typing.Optional[builtins.bool]:
        """Indicates whether the DB instance should have deletion protection enabled.

        :default: - true if ``removalPolicy`` is RETAIN, false otherwise
        """
        result = self._values.get("deletion_protection")
        return result

    @builtins.property
    def domain(self) -> typing.Optional[builtins.str]:
        """The Active Directory directory ID to create the DB instance in.

        :default: - Do not join domain
        """
        result = self._values.get("domain")
        return result

    @builtins.property
    def domain_role(self) -> typing.Optional[aws_cdk.aws_iam.IRole]:
        """The IAM role to be used when making API calls to the Directory Service.

        The role needs the AWS-managed policy
        AmazonRDSDirectoryServiceAccess or equivalent.

        :default: - The role will be created for you if {@link DatabaseInstanceNewProps#domain} is specified
        """
        result = self._values.get("domain_role")
        return result

    @builtins.property
    def enable_performance_insights(self) -> typing.Optional[builtins.bool]:
        """Whether to enable Performance Insights for the DB instance.

        :default: - false, unless ``performanceInsightRentention`` or ``performanceInsightEncryptionKey`` is set.
        """
        result = self._values.get("enable_performance_insights")
        return result

    @builtins.property
    def iam_authentication(self) -> typing.Optional[builtins.bool]:
        """Whether to enable mapping of AWS Identity and Access Management (IAM) accounts to database accounts.

        :default: false
        """
        result = self._values.get("iam_authentication")
        return result

    @builtins.property
    def instance_identifier(self) -> typing.Optional[builtins.str]:
        """A name for the DB instance.

        If you specify a name, AWS CloudFormation
        converts it to lowercase.

        :default: - a CloudFormation generated name
        """
        result = self._values.get("instance_identifier")
        return result

    @builtins.property
    def iops(self) -> typing.Optional[jsii.Number]:
        """The number of I/O operations per second (IOPS) that the database provisions.

        The value must be equal to or greater than 1000.

        :default: - no provisioned iops
        """
        result = self._values.get("iops")
        return result

    @builtins.property
    def max_allocated_storage(self) -> typing.Optional[jsii.Number]:
        """Upper limit to which RDS can scale the storage in GiB(Gibibyte).

        :default: - No autoscaling of RDS instance

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PIOPS.StorageTypes.html#USER_PIOPS.Autoscaling
        """
        result = self._values.get("max_allocated_storage")
        return result

    @builtins.property
    def monitoring_interval(self) -> typing.Optional[aws_cdk.core.Duration]:
        """The interval, in seconds, between points when Amazon RDS collects enhanced monitoring metrics for the DB instance.

        :default: - no enhanced monitoring
        """
        result = self._values.get("monitoring_interval")
        return result

    @builtins.property
    def monitoring_role(self) -> typing.Optional[aws_cdk.aws_iam.IRole]:
        """Role that will be used to manage DB instance monitoring.

        :default: - A role is automatically created for you
        """
        result = self._values.get("monitoring_role")
        return result

    @builtins.property
    def multi_az(self) -> typing.Optional[builtins.bool]:
        """Specifies if the database instance is a multiple Availability Zone deployment.

        :default: false
        """
        result = self._values.get("multi_az")
        return result

    @builtins.property
    def option_group(self) -> typing.Optional["IOptionGroup"]:
        """The option group to associate with the instance.

        :default: - no option group
        """
        result = self._values.get("option_group")
        return result

    @builtins.property
    def performance_insight_encryption_key(
        self,
    ) -> typing.Optional[aws_cdk.aws_kms.IKey]:
        """The AWS KMS key for encryption of Performance Insights data.

        :default: - default master key
        """
        result = self._values.get("performance_insight_encryption_key")
        return result

    @builtins.property
    def performance_insight_retention(
        self,
    ) -> typing.Optional["PerformanceInsightRetention"]:
        """The amount of time, in days, to retain Performance Insights data.

        :default: 7
        """
        result = self._values.get("performance_insight_retention")
        return result

    @builtins.property
    def port(self) -> typing.Optional[jsii.Number]:
        """The port for the instance.

        :default: - the default port for the chosen engine.
        """
        result = self._values.get("port")
        return result

    @builtins.property
    def preferred_backup_window(self) -> typing.Optional[builtins.str]:
        """The daily time range during which automated backups are performed.

        Constraints:

        - Must be in the format ``hh24:mi-hh24:mi``.
        - Must be in Universal Coordinated Time (UTC).
        - Must not conflict with the preferred maintenance window.
        - Must be at least 30 minutes.

        :default:

        - a 30-minute window selected at random from an 8-hour block of
        time for each AWS Region. To see the time blocks available, see
        https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow
        """
        result = self._values.get("preferred_backup_window")
        return result

    @builtins.property
    def preferred_maintenance_window(self) -> typing.Optional[builtins.str]:
        """The weekly time range (in UTC) during which system maintenance can occur.

        Format: ``ddd:hh24:mi-ddd:hh24:mi``
        Constraint: Minimum 30-minute window

        :default:

        - a 30-minute window selected at random from an 8-hour block of
        time for each AWS Region, occurring on a random day of the week. To see
        the time blocks available, see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_UpgradeDBInstance.Maintenance.html#Concepts.DBMaintenance
        """
        result = self._values.get("preferred_maintenance_window")
        return result

    @builtins.property
    def processor_features(self) -> typing.Optional["ProcessorFeatures"]:
        """The number of CPU cores and the number of threads per core.

        :default:

        - the default number of CPU cores and threads per core for the
        chosen instance class.

        See https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html#USER_ConfigureProcessor
        """
        result = self._values.get("processor_features")
        return result

    @builtins.property
    def publicly_accessible(self) -> typing.Optional[builtins.bool]:
        """Indicates whether the DB instance is an internet-facing instance.

        :default: - ``true`` if ``vpcSubnets`` is ``subnetType: SubnetType.PUBLIC``, ``false`` otherwise
        """
        result = self._values.get("publicly_accessible")
        return result

    @builtins.property
    def removal_policy(self) -> typing.Optional[aws_cdk.core.RemovalPolicy]:
        """The CloudFormation policy to apply when the instance is removed from the stack or replaced during an update.

        :default: - RemovalPolicy.SNAPSHOT (remove the resource, but retain a snapshot of the data)
        """
        result = self._values.get("removal_policy")
        return result

    @builtins.property
    def s3_export_buckets(self) -> typing.Optional[typing.List[aws_cdk.aws_s3.IBucket]]:
        """S3 buckets that you want to load data into.

        This feature is only supported by the Microsoft SQL Server and Oracle engines.

        This property must not be used if ``s3ExportRole`` is used.

        For Microsoft SQL Server:

        :default: - None

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/oracle-s3-integration.html
        """
        result = self._values.get("s3_export_buckets")
        return result

    @builtins.property
    def s3_export_role(self) -> typing.Optional[aws_cdk.aws_iam.IRole]:
        """Role that will be associated with this DB instance to enable S3 export.

        This feature is only supported by the Microsoft SQL Server and Oracle engines.

        This property must not be used if ``s3ExportBuckets`` is used.

        For Microsoft SQL Server:

        :default: - New role is created if ``s3ExportBuckets`` is set, no role is defined otherwise

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/oracle-s3-integration.html
        """
        result = self._values.get("s3_export_role")
        return result

    @builtins.property
    def s3_import_buckets(self) -> typing.Optional[typing.List[aws_cdk.aws_s3.IBucket]]:
        """S3 buckets that you want to load data from.

        This feature is only supported by the Microsoft SQL Server, Oracle, and PostgreSQL engines.

        This property must not be used if ``s3ImportRole`` is used.

        For Microsoft SQL Server:

        :default: - None

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/PostgreSQL.Procedural.Importing.html
        """
        result = self._values.get("s3_import_buckets")
        return result

    @builtins.property
    def s3_import_role(self) -> typing.Optional[aws_cdk.aws_iam.IRole]:
        """Role that will be associated with this DB instance to enable S3 import.

        This feature is only supported by the Microsoft SQL Server, Oracle, and PostgreSQL engines.

        This property must not be used if ``s3ImportBuckets`` is used.

        For Microsoft SQL Server:

        :default: - New role is created if ``s3ImportBuckets`` is set, no role is defined otherwise

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/PostgreSQL.Procedural.Importing.html
        """
        result = self._values.get("s3_import_role")
        return result

    @builtins.property
    def security_groups(
        self,
    ) -> typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]]:
        """The security groups to assign to the DB instance.

        :default: - a new security group is created
        """
        result = self._values.get("security_groups")
        return result

    @builtins.property
    def storage_type(self) -> typing.Optional["StorageType"]:
        """The storage type.

        :default: GP2
        """
        result = self._values.get("storage_type")
        return result

    @builtins.property
    def subnet_group(self) -> typing.Optional["ISubnetGroup"]:
        """Existing subnet group for the instance.

        :default: - a new subnet group will be created.
        """
        result = self._values.get("subnet_group")
        return result

    @builtins.property
    def vpc_placement(self) -> typing.Optional[aws_cdk.aws_ec2.SubnetSelection]:
        """(deprecated) The type of subnets to add to the created DB subnet group.

        :default: - private subnets

        :deprecated: use ``vpcSubnets``

        :stability: deprecated
        """
        result = self._values.get("vpc_placement")
        return result

    @builtins.property
    def vpc_subnets(self) -> typing.Optional[aws_cdk.aws_ec2.SubnetSelection]:
        """The type of subnets to add to the created DB subnet group.

        :default: - private subnets
        """
        result = self._values.get("vpc_subnets")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DatabaseInstanceNewProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.DatabaseInstanceReadReplicaProps",
    jsii_struct_bases=[DatabaseInstanceNewProps],
    name_mapping={
        "vpc": "vpc",
        "auto_minor_version_upgrade": "autoMinorVersionUpgrade",
        "availability_zone": "availabilityZone",
        "backup_retention": "backupRetention",
        "cloudwatch_logs_exports": "cloudwatchLogsExports",
        "cloudwatch_logs_retention": "cloudwatchLogsRetention",
        "cloudwatch_logs_retention_role": "cloudwatchLogsRetentionRole",
        "copy_tags_to_snapshot": "copyTagsToSnapshot",
        "delete_automated_backups": "deleteAutomatedBackups",
        "deletion_protection": "deletionProtection",
        "domain": "domain",
        "domain_role": "domainRole",
        "enable_performance_insights": "enablePerformanceInsights",
        "iam_authentication": "iamAuthentication",
        "instance_identifier": "instanceIdentifier",
        "iops": "iops",
        "max_allocated_storage": "maxAllocatedStorage",
        "monitoring_interval": "monitoringInterval",
        "monitoring_role": "monitoringRole",
        "multi_az": "multiAz",
        "option_group": "optionGroup",
        "performance_insight_encryption_key": "performanceInsightEncryptionKey",
        "performance_insight_retention": "performanceInsightRetention",
        "port": "port",
        "preferred_backup_window": "preferredBackupWindow",
        "preferred_maintenance_window": "preferredMaintenanceWindow",
        "processor_features": "processorFeatures",
        "publicly_accessible": "publiclyAccessible",
        "removal_policy": "removalPolicy",
        "s3_export_buckets": "s3ExportBuckets",
        "s3_export_role": "s3ExportRole",
        "s3_import_buckets": "s3ImportBuckets",
        "s3_import_role": "s3ImportRole",
        "security_groups": "securityGroups",
        "storage_type": "storageType",
        "subnet_group": "subnetGroup",
        "vpc_placement": "vpcPlacement",
        "vpc_subnets": "vpcSubnets",
        "instance_type": "instanceType",
        "source_database_instance": "sourceDatabaseInstance",
        "storage_encrypted": "storageEncrypted",
        "storage_encryption_key": "storageEncryptionKey",
    },
)
class DatabaseInstanceReadReplicaProps(DatabaseInstanceNewProps):
    def __init__(
        self,
        *,
        vpc: aws_cdk.aws_ec2.IVpc,
        auto_minor_version_upgrade: typing.Optional[builtins.bool] = None,
        availability_zone: typing.Optional[builtins.str] = None,
        backup_retention: typing.Optional[aws_cdk.core.Duration] = None,
        cloudwatch_logs_exports: typing.Optional[typing.List[builtins.str]] = None,
        cloudwatch_logs_retention: typing.Optional[aws_cdk.aws_logs.RetentionDays] = None,
        cloudwatch_logs_retention_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        copy_tags_to_snapshot: typing.Optional[builtins.bool] = None,
        delete_automated_backups: typing.Optional[builtins.bool] = None,
        deletion_protection: typing.Optional[builtins.bool] = None,
        domain: typing.Optional[builtins.str] = None,
        domain_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        enable_performance_insights: typing.Optional[builtins.bool] = None,
        iam_authentication: typing.Optional[builtins.bool] = None,
        instance_identifier: typing.Optional[builtins.str] = None,
        iops: typing.Optional[jsii.Number] = None,
        max_allocated_storage: typing.Optional[jsii.Number] = None,
        monitoring_interval: typing.Optional[aws_cdk.core.Duration] = None,
        monitoring_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        multi_az: typing.Optional[builtins.bool] = None,
        option_group: typing.Optional["IOptionGroup"] = None,
        performance_insight_encryption_key: typing.Optional[aws_cdk.aws_kms.IKey] = None,
        performance_insight_retention: typing.Optional["PerformanceInsightRetention"] = None,
        port: typing.Optional[jsii.Number] = None,
        preferred_backup_window: typing.Optional[builtins.str] = None,
        preferred_maintenance_window: typing.Optional[builtins.str] = None,
        processor_features: typing.Optional["ProcessorFeatures"] = None,
        publicly_accessible: typing.Optional[builtins.bool] = None,
        removal_policy: typing.Optional[aws_cdk.core.RemovalPolicy] = None,
        s3_export_buckets: typing.Optional[typing.List[aws_cdk.aws_s3.IBucket]] = None,
        s3_export_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        s3_import_buckets: typing.Optional[typing.List[aws_cdk.aws_s3.IBucket]] = None,
        s3_import_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        security_groups: typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]] = None,
        storage_type: typing.Optional["StorageType"] = None,
        subnet_group: typing.Optional["ISubnetGroup"] = None,
        vpc_placement: typing.Optional[aws_cdk.aws_ec2.SubnetSelection] = None,
        vpc_subnets: typing.Optional[aws_cdk.aws_ec2.SubnetSelection] = None,
        instance_type: aws_cdk.aws_ec2.InstanceType,
        source_database_instance: "IDatabaseInstance",
        storage_encrypted: typing.Optional[builtins.bool] = None,
        storage_encryption_key: typing.Optional[aws_cdk.aws_kms.IKey] = None,
    ) -> None:
        """Construction properties for a DatabaseInstanceReadReplica.

        :param vpc: The VPC network where the DB subnet group should be created.
        :param auto_minor_version_upgrade: Indicates that minor engine upgrades are applied automatically to the DB instance during the maintenance window. Default: true
        :param availability_zone: The name of the Availability Zone where the DB instance will be located. Default: - no preference
        :param backup_retention: The number of days during which automatic DB snapshots are retained. Set to zero to disable backups. When creating a read replica, you must enable automatic backups on the source database instance by setting the backup retention to a value other than zero. Default: Duration.days(1)
        :param cloudwatch_logs_exports: The list of log types that need to be enabled for exporting to CloudWatch Logs. Default: - no log exports
        :param cloudwatch_logs_retention: The number of days log events are kept in CloudWatch Logs. When updating this property, unsetting it doesn't remove the log retention policy. To remove the retention policy, set the value to ``Infinity``. Default: - logs never expire
        :param cloudwatch_logs_retention_role: The IAM role for the Lambda function associated with the custom resource that sets the retention policy. Default: - a new role is created.
        :param copy_tags_to_snapshot: Indicates whether to copy all of the user-defined tags from the DB instance to snapshots of the DB instance. Default: true
        :param delete_automated_backups: Indicates whether automated backups should be deleted or retained when you delete a DB instance. Default: false
        :param deletion_protection: Indicates whether the DB instance should have deletion protection enabled. Default: - true if ``removalPolicy`` is RETAIN, false otherwise
        :param domain: The Active Directory directory ID to create the DB instance in. Default: - Do not join domain
        :param domain_role: The IAM role to be used when making API calls to the Directory Service. The role needs the AWS-managed policy AmazonRDSDirectoryServiceAccess or equivalent. Default: - The role will be created for you if {@link DatabaseInstanceNewProps#domain} is specified
        :param enable_performance_insights: Whether to enable Performance Insights for the DB instance. Default: - false, unless ``performanceInsightRentention`` or ``performanceInsightEncryptionKey`` is set.
        :param iam_authentication: Whether to enable mapping of AWS Identity and Access Management (IAM) accounts to database accounts. Default: false
        :param instance_identifier: A name for the DB instance. If you specify a name, AWS CloudFormation converts it to lowercase. Default: - a CloudFormation generated name
        :param iops: The number of I/O operations per second (IOPS) that the database provisions. The value must be equal to or greater than 1000. Default: - no provisioned iops
        :param max_allocated_storage: Upper limit to which RDS can scale the storage in GiB(Gibibyte). Default: - No autoscaling of RDS instance
        :param monitoring_interval: The interval, in seconds, between points when Amazon RDS collects enhanced monitoring metrics for the DB instance. Default: - no enhanced monitoring
        :param monitoring_role: Role that will be used to manage DB instance monitoring. Default: - A role is automatically created for you
        :param multi_az: Specifies if the database instance is a multiple Availability Zone deployment. Default: false
        :param option_group: The option group to associate with the instance. Default: - no option group
        :param performance_insight_encryption_key: The AWS KMS key for encryption of Performance Insights data. Default: - default master key
        :param performance_insight_retention: The amount of time, in days, to retain Performance Insights data. Default: 7
        :param port: The port for the instance. Default: - the default port for the chosen engine.
        :param preferred_backup_window: The daily time range during which automated backups are performed. Constraints: - Must be in the format ``hh24:mi-hh24:mi``. - Must be in Universal Coordinated Time (UTC). - Must not conflict with the preferred maintenance window. - Must be at least 30 minutes. Default: - a 30-minute window selected at random from an 8-hour block of time for each AWS Region. To see the time blocks available, see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow
        :param preferred_maintenance_window: The weekly time range (in UTC) during which system maintenance can occur. Format: ``ddd:hh24:mi-ddd:hh24:mi`` Constraint: Minimum 30-minute window Default: - a 30-minute window selected at random from an 8-hour block of time for each AWS Region, occurring on a random day of the week. To see the time blocks available, see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_UpgradeDBInstance.Maintenance.html#Concepts.DBMaintenance
        :param processor_features: The number of CPU cores and the number of threads per core. Default: - the default number of CPU cores and threads per core for the chosen instance class. See https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html#USER_ConfigureProcessor
        :param publicly_accessible: Indicates whether the DB instance is an internet-facing instance. Default: - ``true`` if ``vpcSubnets`` is ``subnetType: SubnetType.PUBLIC``, ``false`` otherwise
        :param removal_policy: The CloudFormation policy to apply when the instance is removed from the stack or replaced during an update. Default: - RemovalPolicy.SNAPSHOT (remove the resource, but retain a snapshot of the data)
        :param s3_export_buckets: S3 buckets that you want to load data into. This feature is only supported by the Microsoft SQL Server and Oracle engines. This property must not be used if ``s3ExportRole`` is used. For Microsoft SQL Server: Default: - None
        :param s3_export_role: Role that will be associated with this DB instance to enable S3 export. This feature is only supported by the Microsoft SQL Server and Oracle engines. This property must not be used if ``s3ExportBuckets`` is used. For Microsoft SQL Server: Default: - New role is created if ``s3ExportBuckets`` is set, no role is defined otherwise
        :param s3_import_buckets: S3 buckets that you want to load data from. This feature is only supported by the Microsoft SQL Server, Oracle, and PostgreSQL engines. This property must not be used if ``s3ImportRole`` is used. For Microsoft SQL Server: Default: - None
        :param s3_import_role: Role that will be associated with this DB instance to enable S3 import. This feature is only supported by the Microsoft SQL Server, Oracle, and PostgreSQL engines. This property must not be used if ``s3ImportBuckets`` is used. For Microsoft SQL Server: Default: - New role is created if ``s3ImportBuckets`` is set, no role is defined otherwise
        :param security_groups: The security groups to assign to the DB instance. Default: - a new security group is created
        :param storage_type: The storage type. Default: GP2
        :param subnet_group: Existing subnet group for the instance. Default: - a new subnet group will be created.
        :param vpc_placement: (deprecated) The type of subnets to add to the created DB subnet group. Default: - private subnets
        :param vpc_subnets: The type of subnets to add to the created DB subnet group. Default: - private subnets
        :param instance_type: The name of the compute and memory capacity classes.
        :param source_database_instance: The source database instance. Each DB instance can have a limited number of read replicas. For more information, see https://docs.aws.amazon.com/AmazonRDS/latest/DeveloperGuide/USER_ReadRepl.html.
        :param storage_encrypted: Indicates whether the DB instance is encrypted. Default: - true if storageEncryptionKey has been provided, false otherwise
        :param storage_encryption_key: The KMS key that's used to encrypt the DB instance. Default: - default master key if storageEncrypted is true, no key otherwise
        """
        if isinstance(processor_features, dict):
            processor_features = ProcessorFeatures(**processor_features)
        if isinstance(vpc_placement, dict):
            vpc_placement = aws_cdk.aws_ec2.SubnetSelection(**vpc_placement)
        if isinstance(vpc_subnets, dict):
            vpc_subnets = aws_cdk.aws_ec2.SubnetSelection(**vpc_subnets)
        self._values: typing.Dict[str, typing.Any] = {
            "vpc": vpc,
            "instance_type": instance_type,
            "source_database_instance": source_database_instance,
        }
        if auto_minor_version_upgrade is not None:
            self._values["auto_minor_version_upgrade"] = auto_minor_version_upgrade
        if availability_zone is not None:
            self._values["availability_zone"] = availability_zone
        if backup_retention is not None:
            self._values["backup_retention"] = backup_retention
        if cloudwatch_logs_exports is not None:
            self._values["cloudwatch_logs_exports"] = cloudwatch_logs_exports
        if cloudwatch_logs_retention is not None:
            self._values["cloudwatch_logs_retention"] = cloudwatch_logs_retention
        if cloudwatch_logs_retention_role is not None:
            self._values["cloudwatch_logs_retention_role"] = cloudwatch_logs_retention_role
        if copy_tags_to_snapshot is not None:
            self._values["copy_tags_to_snapshot"] = copy_tags_to_snapshot
        if delete_automated_backups is not None:
            self._values["delete_automated_backups"] = delete_automated_backups
        if deletion_protection is not None:
            self._values["deletion_protection"] = deletion_protection
        if domain is not None:
            self._values["domain"] = domain
        if domain_role is not None:
            self._values["domain_role"] = domain_role
        if enable_performance_insights is not None:
            self._values["enable_performance_insights"] = enable_performance_insights
        if iam_authentication is not None:
            self._values["iam_authentication"] = iam_authentication
        if instance_identifier is not None:
            self._values["instance_identifier"] = instance_identifier
        if iops is not None:
            self._values["iops"] = iops
        if max_allocated_storage is not None:
            self._values["max_allocated_storage"] = max_allocated_storage
        if monitoring_interval is not None:
            self._values["monitoring_interval"] = monitoring_interval
        if monitoring_role is not None:
            self._values["monitoring_role"] = monitoring_role
        if multi_az is not None:
            self._values["multi_az"] = multi_az
        if option_group is not None:
            self._values["option_group"] = option_group
        if performance_insight_encryption_key is not None:
            self._values["performance_insight_encryption_key"] = performance_insight_encryption_key
        if performance_insight_retention is not None:
            self._values["performance_insight_retention"] = performance_insight_retention
        if port is not None:
            self._values["port"] = port
        if preferred_backup_window is not None:
            self._values["preferred_backup_window"] = preferred_backup_window
        if preferred_maintenance_window is not None:
            self._values["preferred_maintenance_window"] = preferred_maintenance_window
        if processor_features is not None:
            self._values["processor_features"] = processor_features
        if publicly_accessible is not None:
            self._values["publicly_accessible"] = publicly_accessible
        if removal_policy is not None:
            self._values["removal_policy"] = removal_policy
        if s3_export_buckets is not None:
            self._values["s3_export_buckets"] = s3_export_buckets
        if s3_export_role is not None:
            self._values["s3_export_role"] = s3_export_role
        if s3_import_buckets is not None:
            self._values["s3_import_buckets"] = s3_import_buckets
        if s3_import_role is not None:
            self._values["s3_import_role"] = s3_import_role
        if security_groups is not None:
            self._values["security_groups"] = security_groups
        if storage_type is not None:
            self._values["storage_type"] = storage_type
        if subnet_group is not None:
            self._values["subnet_group"] = subnet_group
        if vpc_placement is not None:
            self._values["vpc_placement"] = vpc_placement
        if vpc_subnets is not None:
            self._values["vpc_subnets"] = vpc_subnets
        if storage_encrypted is not None:
            self._values["storage_encrypted"] = storage_encrypted
        if storage_encryption_key is not None:
            self._values["storage_encryption_key"] = storage_encryption_key

    @builtins.property
    def vpc(self) -> aws_cdk.aws_ec2.IVpc:
        """The VPC network where the DB subnet group should be created."""
        result = self._values.get("vpc")
        assert result is not None, "Required property 'vpc' is missing"
        return result

    @builtins.property
    def auto_minor_version_upgrade(self) -> typing.Optional[builtins.bool]:
        """Indicates that minor engine upgrades are applied automatically to the DB instance during the maintenance window.

        :default: true
        """
        result = self._values.get("auto_minor_version_upgrade")
        return result

    @builtins.property
    def availability_zone(self) -> typing.Optional[builtins.str]:
        """The name of the Availability Zone where the DB instance will be located.

        :default: - no preference
        """
        result = self._values.get("availability_zone")
        return result

    @builtins.property
    def backup_retention(self) -> typing.Optional[aws_cdk.core.Duration]:
        """The number of days during which automatic DB snapshots are retained.

        Set to zero to disable backups.
        When creating a read replica, you must enable automatic backups on the source
        database instance by setting the backup retention to a value other than zero.

        :default: Duration.days(1)
        """
        result = self._values.get("backup_retention")
        return result

    @builtins.property
    def cloudwatch_logs_exports(self) -> typing.Optional[typing.List[builtins.str]]:
        """The list of log types that need to be enabled for exporting to CloudWatch Logs.

        :default: - no log exports
        """
        result = self._values.get("cloudwatch_logs_exports")
        return result

    @builtins.property
    def cloudwatch_logs_retention(
        self,
    ) -> typing.Optional[aws_cdk.aws_logs.RetentionDays]:
        """The number of days log events are kept in CloudWatch Logs.

        When updating
        this property, unsetting it doesn't remove the log retention policy. To
        remove the retention policy, set the value to ``Infinity``.

        :default: - logs never expire
        """
        result = self._values.get("cloudwatch_logs_retention")
        return result

    @builtins.property
    def cloudwatch_logs_retention_role(self) -> typing.Optional[aws_cdk.aws_iam.IRole]:
        """The IAM role for the Lambda function associated with the custom resource that sets the retention policy.

        :default: - a new role is created.
        """
        result = self._values.get("cloudwatch_logs_retention_role")
        return result

    @builtins.property
    def copy_tags_to_snapshot(self) -> typing.Optional[builtins.bool]:
        """Indicates whether to copy all of the user-defined tags from the DB instance to snapshots of the DB instance.

        :default: true
        """
        result = self._values.get("copy_tags_to_snapshot")
        return result

    @builtins.property
    def delete_automated_backups(self) -> typing.Optional[builtins.bool]:
        """Indicates whether automated backups should be deleted or retained when you delete a DB instance.

        :default: false
        """
        result = self._values.get("delete_automated_backups")
        return result

    @builtins.property
    def deletion_protection(self) -> typing.Optional[builtins.bool]:
        """Indicates whether the DB instance should have deletion protection enabled.

        :default: - true if ``removalPolicy`` is RETAIN, false otherwise
        """
        result = self._values.get("deletion_protection")
        return result

    @builtins.property
    def domain(self) -> typing.Optional[builtins.str]:
        """The Active Directory directory ID to create the DB instance in.

        :default: - Do not join domain
        """
        result = self._values.get("domain")
        return result

    @builtins.property
    def domain_role(self) -> typing.Optional[aws_cdk.aws_iam.IRole]:
        """The IAM role to be used when making API calls to the Directory Service.

        The role needs the AWS-managed policy
        AmazonRDSDirectoryServiceAccess or equivalent.

        :default: - The role will be created for you if {@link DatabaseInstanceNewProps#domain} is specified
        """
        result = self._values.get("domain_role")
        return result

    @builtins.property
    def enable_performance_insights(self) -> typing.Optional[builtins.bool]:
        """Whether to enable Performance Insights for the DB instance.

        :default: - false, unless ``performanceInsightRentention`` or ``performanceInsightEncryptionKey`` is set.
        """
        result = self._values.get("enable_performance_insights")
        return result

    @builtins.property
    def iam_authentication(self) -> typing.Optional[builtins.bool]:
        """Whether to enable mapping of AWS Identity and Access Management (IAM) accounts to database accounts.

        :default: false
        """
        result = self._values.get("iam_authentication")
        return result

    @builtins.property
    def instance_identifier(self) -> typing.Optional[builtins.str]:
        """A name for the DB instance.

        If you specify a name, AWS CloudFormation
        converts it to lowercase.

        :default: - a CloudFormation generated name
        """
        result = self._values.get("instance_identifier")
        return result

    @builtins.property
    def iops(self) -> typing.Optional[jsii.Number]:
        """The number of I/O operations per second (IOPS) that the database provisions.

        The value must be equal to or greater than 1000.

        :default: - no provisioned iops
        """
        result = self._values.get("iops")
        return result

    @builtins.property
    def max_allocated_storage(self) -> typing.Optional[jsii.Number]:
        """Upper limit to which RDS can scale the storage in GiB(Gibibyte).

        :default: - No autoscaling of RDS instance

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PIOPS.StorageTypes.html#USER_PIOPS.Autoscaling
        """
        result = self._values.get("max_allocated_storage")
        return result

    @builtins.property
    def monitoring_interval(self) -> typing.Optional[aws_cdk.core.Duration]:
        """The interval, in seconds, between points when Amazon RDS collects enhanced monitoring metrics for the DB instance.

        :default: - no enhanced monitoring
        """
        result = self._values.get("monitoring_interval")
        return result

    @builtins.property
    def monitoring_role(self) -> typing.Optional[aws_cdk.aws_iam.IRole]:
        """Role that will be used to manage DB instance monitoring.

        :default: - A role is automatically created for you
        """
        result = self._values.get("monitoring_role")
        return result

    @builtins.property
    def multi_az(self) -> typing.Optional[builtins.bool]:
        """Specifies if the database instance is a multiple Availability Zone deployment.

        :default: false
        """
        result = self._values.get("multi_az")
        return result

    @builtins.property
    def option_group(self) -> typing.Optional["IOptionGroup"]:
        """The option group to associate with the instance.

        :default: - no option group
        """
        result = self._values.get("option_group")
        return result

    @builtins.property
    def performance_insight_encryption_key(
        self,
    ) -> typing.Optional[aws_cdk.aws_kms.IKey]:
        """The AWS KMS key for encryption of Performance Insights data.

        :default: - default master key
        """
        result = self._values.get("performance_insight_encryption_key")
        return result

    @builtins.property
    def performance_insight_retention(
        self,
    ) -> typing.Optional["PerformanceInsightRetention"]:
        """The amount of time, in days, to retain Performance Insights data.

        :default: 7
        """
        result = self._values.get("performance_insight_retention")
        return result

    @builtins.property
    def port(self) -> typing.Optional[jsii.Number]:
        """The port for the instance.

        :default: - the default port for the chosen engine.
        """
        result = self._values.get("port")
        return result

    @builtins.property
    def preferred_backup_window(self) -> typing.Optional[builtins.str]:
        """The daily time range during which automated backups are performed.

        Constraints:

        - Must be in the format ``hh24:mi-hh24:mi``.
        - Must be in Universal Coordinated Time (UTC).
        - Must not conflict with the preferred maintenance window.
        - Must be at least 30 minutes.

        :default:

        - a 30-minute window selected at random from an 8-hour block of
        time for each AWS Region. To see the time blocks available, see
        https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow
        """
        result = self._values.get("preferred_backup_window")
        return result

    @builtins.property
    def preferred_maintenance_window(self) -> typing.Optional[builtins.str]:
        """The weekly time range (in UTC) during which system maintenance can occur.

        Format: ``ddd:hh24:mi-ddd:hh24:mi``
        Constraint: Minimum 30-minute window

        :default:

        - a 30-minute window selected at random from an 8-hour block of
        time for each AWS Region, occurring on a random day of the week. To see
        the time blocks available, see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_UpgradeDBInstance.Maintenance.html#Concepts.DBMaintenance
        """
        result = self._values.get("preferred_maintenance_window")
        return result

    @builtins.property
    def processor_features(self) -> typing.Optional["ProcessorFeatures"]:
        """The number of CPU cores and the number of threads per core.

        :default:

        - the default number of CPU cores and threads per core for the
        chosen instance class.

        See https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html#USER_ConfigureProcessor
        """
        result = self._values.get("processor_features")
        return result

    @builtins.property
    def publicly_accessible(self) -> typing.Optional[builtins.bool]:
        """Indicates whether the DB instance is an internet-facing instance.

        :default: - ``true`` if ``vpcSubnets`` is ``subnetType: SubnetType.PUBLIC``, ``false`` otherwise
        """
        result = self._values.get("publicly_accessible")
        return result

    @builtins.property
    def removal_policy(self) -> typing.Optional[aws_cdk.core.RemovalPolicy]:
        """The CloudFormation policy to apply when the instance is removed from the stack or replaced during an update.

        :default: - RemovalPolicy.SNAPSHOT (remove the resource, but retain a snapshot of the data)
        """
        result = self._values.get("removal_policy")
        return result

    @builtins.property
    def s3_export_buckets(self) -> typing.Optional[typing.List[aws_cdk.aws_s3.IBucket]]:
        """S3 buckets that you want to load data into.

        This feature is only supported by the Microsoft SQL Server and Oracle engines.

        This property must not be used if ``s3ExportRole`` is used.

        For Microsoft SQL Server:

        :default: - None

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/oracle-s3-integration.html
        """
        result = self._values.get("s3_export_buckets")
        return result

    @builtins.property
    def s3_export_role(self) -> typing.Optional[aws_cdk.aws_iam.IRole]:
        """Role that will be associated with this DB instance to enable S3 export.

        This feature is only supported by the Microsoft SQL Server and Oracle engines.

        This property must not be used if ``s3ExportBuckets`` is used.

        For Microsoft SQL Server:

        :default: - New role is created if ``s3ExportBuckets`` is set, no role is defined otherwise

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/oracle-s3-integration.html
        """
        result = self._values.get("s3_export_role")
        return result

    @builtins.property
    def s3_import_buckets(self) -> typing.Optional[typing.List[aws_cdk.aws_s3.IBucket]]:
        """S3 buckets that you want to load data from.

        This feature is only supported by the Microsoft SQL Server, Oracle, and PostgreSQL engines.

        This property must not be used if ``s3ImportRole`` is used.

        For Microsoft SQL Server:

        :default: - None

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/PostgreSQL.Procedural.Importing.html
        """
        result = self._values.get("s3_import_buckets")
        return result

    @builtins.property
    def s3_import_role(self) -> typing.Optional[aws_cdk.aws_iam.IRole]:
        """Role that will be associated with this DB instance to enable S3 import.

        This feature is only supported by the Microsoft SQL Server, Oracle, and PostgreSQL engines.

        This property must not be used if ``s3ImportBuckets`` is used.

        For Microsoft SQL Server:

        :default: - New role is created if ``s3ImportBuckets`` is set, no role is defined otherwise

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/PostgreSQL.Procedural.Importing.html
        """
        result = self._values.get("s3_import_role")
        return result

    @builtins.property
    def security_groups(
        self,
    ) -> typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]]:
        """The security groups to assign to the DB instance.

        :default: - a new security group is created
        """
        result = self._values.get("security_groups")
        return result

    @builtins.property
    def storage_type(self) -> typing.Optional["StorageType"]:
        """The storage type.

        :default: GP2
        """
        result = self._values.get("storage_type")
        return result

    @builtins.property
    def subnet_group(self) -> typing.Optional["ISubnetGroup"]:
        """Existing subnet group for the instance.

        :default: - a new subnet group will be created.
        """
        result = self._values.get("subnet_group")
        return result

    @builtins.property
    def vpc_placement(self) -> typing.Optional[aws_cdk.aws_ec2.SubnetSelection]:
        """(deprecated) The type of subnets to add to the created DB subnet group.

        :default: - private subnets

        :deprecated: use ``vpcSubnets``

        :stability: deprecated
        """
        result = self._values.get("vpc_placement")
        return result

    @builtins.property
    def vpc_subnets(self) -> typing.Optional[aws_cdk.aws_ec2.SubnetSelection]:
        """The type of subnets to add to the created DB subnet group.

        :default: - private subnets
        """
        result = self._values.get("vpc_subnets")
        return result

    @builtins.property
    def instance_type(self) -> aws_cdk.aws_ec2.InstanceType:
        """The name of the compute and memory capacity classes."""
        result = self._values.get("instance_type")
        assert result is not None, "Required property 'instance_type' is missing"
        return result

    @builtins.property
    def source_database_instance(self) -> "IDatabaseInstance":
        """The source database instance.

        Each DB instance can have a limited number of read replicas. For more
        information, see https://docs.aws.amazon.com/AmazonRDS/latest/DeveloperGuide/USER_ReadRepl.html.
        """
        result = self._values.get("source_database_instance")
        assert result is not None, "Required property 'source_database_instance' is missing"
        return result

    @builtins.property
    def storage_encrypted(self) -> typing.Optional[builtins.bool]:
        """Indicates whether the DB instance is encrypted.

        :default: - true if storageEncryptionKey has been provided, false otherwise
        """
        result = self._values.get("storage_encrypted")
        return result

    @builtins.property
    def storage_encryption_key(self) -> typing.Optional[aws_cdk.aws_kms.IKey]:
        """The KMS key that's used to encrypt the DB instance.

        :default: - default master key if storageEncrypted is true, no key otherwise
        """
        result = self._values.get("storage_encryption_key")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DatabaseInstanceReadReplicaProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.DatabaseInstanceSourceProps",
    jsii_struct_bases=[DatabaseInstanceNewProps],
    name_mapping={
        "vpc": "vpc",
        "auto_minor_version_upgrade": "autoMinorVersionUpgrade",
        "availability_zone": "availabilityZone",
        "backup_retention": "backupRetention",
        "cloudwatch_logs_exports": "cloudwatchLogsExports",
        "cloudwatch_logs_retention": "cloudwatchLogsRetention",
        "cloudwatch_logs_retention_role": "cloudwatchLogsRetentionRole",
        "copy_tags_to_snapshot": "copyTagsToSnapshot",
        "delete_automated_backups": "deleteAutomatedBackups",
        "deletion_protection": "deletionProtection",
        "domain": "domain",
        "domain_role": "domainRole",
        "enable_performance_insights": "enablePerformanceInsights",
        "iam_authentication": "iamAuthentication",
        "instance_identifier": "instanceIdentifier",
        "iops": "iops",
        "max_allocated_storage": "maxAllocatedStorage",
        "monitoring_interval": "monitoringInterval",
        "monitoring_role": "monitoringRole",
        "multi_az": "multiAz",
        "option_group": "optionGroup",
        "performance_insight_encryption_key": "performanceInsightEncryptionKey",
        "performance_insight_retention": "performanceInsightRetention",
        "port": "port",
        "preferred_backup_window": "preferredBackupWindow",
        "preferred_maintenance_window": "preferredMaintenanceWindow",
        "processor_features": "processorFeatures",
        "publicly_accessible": "publiclyAccessible",
        "removal_policy": "removalPolicy",
        "s3_export_buckets": "s3ExportBuckets",
        "s3_export_role": "s3ExportRole",
        "s3_import_buckets": "s3ImportBuckets",
        "s3_import_role": "s3ImportRole",
        "security_groups": "securityGroups",
        "storage_type": "storageType",
        "subnet_group": "subnetGroup",
        "vpc_placement": "vpcPlacement",
        "vpc_subnets": "vpcSubnets",
        "engine": "engine",
        "allocated_storage": "allocatedStorage",
        "allow_major_version_upgrade": "allowMajorVersionUpgrade",
        "database_name": "databaseName",
        "instance_type": "instanceType",
        "license_model": "licenseModel",
        "parameter_group": "parameterGroup",
        "timezone": "timezone",
    },
)
class DatabaseInstanceSourceProps(DatabaseInstanceNewProps):
    def __init__(
        self,
        *,
        vpc: aws_cdk.aws_ec2.IVpc,
        auto_minor_version_upgrade: typing.Optional[builtins.bool] = None,
        availability_zone: typing.Optional[builtins.str] = None,
        backup_retention: typing.Optional[aws_cdk.core.Duration] = None,
        cloudwatch_logs_exports: typing.Optional[typing.List[builtins.str]] = None,
        cloudwatch_logs_retention: typing.Optional[aws_cdk.aws_logs.RetentionDays] = None,
        cloudwatch_logs_retention_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        copy_tags_to_snapshot: typing.Optional[builtins.bool] = None,
        delete_automated_backups: typing.Optional[builtins.bool] = None,
        deletion_protection: typing.Optional[builtins.bool] = None,
        domain: typing.Optional[builtins.str] = None,
        domain_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        enable_performance_insights: typing.Optional[builtins.bool] = None,
        iam_authentication: typing.Optional[builtins.bool] = None,
        instance_identifier: typing.Optional[builtins.str] = None,
        iops: typing.Optional[jsii.Number] = None,
        max_allocated_storage: typing.Optional[jsii.Number] = None,
        monitoring_interval: typing.Optional[aws_cdk.core.Duration] = None,
        monitoring_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        multi_az: typing.Optional[builtins.bool] = None,
        option_group: typing.Optional["IOptionGroup"] = None,
        performance_insight_encryption_key: typing.Optional[aws_cdk.aws_kms.IKey] = None,
        performance_insight_retention: typing.Optional["PerformanceInsightRetention"] = None,
        port: typing.Optional[jsii.Number] = None,
        preferred_backup_window: typing.Optional[builtins.str] = None,
        preferred_maintenance_window: typing.Optional[builtins.str] = None,
        processor_features: typing.Optional["ProcessorFeatures"] = None,
        publicly_accessible: typing.Optional[builtins.bool] = None,
        removal_policy: typing.Optional[aws_cdk.core.RemovalPolicy] = None,
        s3_export_buckets: typing.Optional[typing.List[aws_cdk.aws_s3.IBucket]] = None,
        s3_export_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        s3_import_buckets: typing.Optional[typing.List[aws_cdk.aws_s3.IBucket]] = None,
        s3_import_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        security_groups: typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]] = None,
        storage_type: typing.Optional["StorageType"] = None,
        subnet_group: typing.Optional["ISubnetGroup"] = None,
        vpc_placement: typing.Optional[aws_cdk.aws_ec2.SubnetSelection] = None,
        vpc_subnets: typing.Optional[aws_cdk.aws_ec2.SubnetSelection] = None,
        engine: "IInstanceEngine",
        allocated_storage: typing.Optional[jsii.Number] = None,
        allow_major_version_upgrade: typing.Optional[builtins.bool] = None,
        database_name: typing.Optional[builtins.str] = None,
        instance_type: typing.Optional[aws_cdk.aws_ec2.InstanceType] = None,
        license_model: typing.Optional["LicenseModel"] = None,
        parameter_group: typing.Optional["IParameterGroup"] = None,
        timezone: typing.Optional[builtins.str] = None,
    ) -> None:
        """Construction properties for a DatabaseInstanceSource.

        :param vpc: The VPC network where the DB subnet group should be created.
        :param auto_minor_version_upgrade: Indicates that minor engine upgrades are applied automatically to the DB instance during the maintenance window. Default: true
        :param availability_zone: The name of the Availability Zone where the DB instance will be located. Default: - no preference
        :param backup_retention: The number of days during which automatic DB snapshots are retained. Set to zero to disable backups. When creating a read replica, you must enable automatic backups on the source database instance by setting the backup retention to a value other than zero. Default: Duration.days(1)
        :param cloudwatch_logs_exports: The list of log types that need to be enabled for exporting to CloudWatch Logs. Default: - no log exports
        :param cloudwatch_logs_retention: The number of days log events are kept in CloudWatch Logs. When updating this property, unsetting it doesn't remove the log retention policy. To remove the retention policy, set the value to ``Infinity``. Default: - logs never expire
        :param cloudwatch_logs_retention_role: The IAM role for the Lambda function associated with the custom resource that sets the retention policy. Default: - a new role is created.
        :param copy_tags_to_snapshot: Indicates whether to copy all of the user-defined tags from the DB instance to snapshots of the DB instance. Default: true
        :param delete_automated_backups: Indicates whether automated backups should be deleted or retained when you delete a DB instance. Default: false
        :param deletion_protection: Indicates whether the DB instance should have deletion protection enabled. Default: - true if ``removalPolicy`` is RETAIN, false otherwise
        :param domain: The Active Directory directory ID to create the DB instance in. Default: - Do not join domain
        :param domain_role: The IAM role to be used when making API calls to the Directory Service. The role needs the AWS-managed policy AmazonRDSDirectoryServiceAccess or equivalent. Default: - The role will be created for you if {@link DatabaseInstanceNewProps#domain} is specified
        :param enable_performance_insights: Whether to enable Performance Insights for the DB instance. Default: - false, unless ``performanceInsightRentention`` or ``performanceInsightEncryptionKey`` is set.
        :param iam_authentication: Whether to enable mapping of AWS Identity and Access Management (IAM) accounts to database accounts. Default: false
        :param instance_identifier: A name for the DB instance. If you specify a name, AWS CloudFormation converts it to lowercase. Default: - a CloudFormation generated name
        :param iops: The number of I/O operations per second (IOPS) that the database provisions. The value must be equal to or greater than 1000. Default: - no provisioned iops
        :param max_allocated_storage: Upper limit to which RDS can scale the storage in GiB(Gibibyte). Default: - No autoscaling of RDS instance
        :param monitoring_interval: The interval, in seconds, between points when Amazon RDS collects enhanced monitoring metrics for the DB instance. Default: - no enhanced monitoring
        :param monitoring_role: Role that will be used to manage DB instance monitoring. Default: - A role is automatically created for you
        :param multi_az: Specifies if the database instance is a multiple Availability Zone deployment. Default: false
        :param option_group: The option group to associate with the instance. Default: - no option group
        :param performance_insight_encryption_key: The AWS KMS key for encryption of Performance Insights data. Default: - default master key
        :param performance_insight_retention: The amount of time, in days, to retain Performance Insights data. Default: 7
        :param port: The port for the instance. Default: - the default port for the chosen engine.
        :param preferred_backup_window: The daily time range during which automated backups are performed. Constraints: - Must be in the format ``hh24:mi-hh24:mi``. - Must be in Universal Coordinated Time (UTC). - Must not conflict with the preferred maintenance window. - Must be at least 30 minutes. Default: - a 30-minute window selected at random from an 8-hour block of time for each AWS Region. To see the time blocks available, see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow
        :param preferred_maintenance_window: The weekly time range (in UTC) during which system maintenance can occur. Format: ``ddd:hh24:mi-ddd:hh24:mi`` Constraint: Minimum 30-minute window Default: - a 30-minute window selected at random from an 8-hour block of time for each AWS Region, occurring on a random day of the week. To see the time blocks available, see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_UpgradeDBInstance.Maintenance.html#Concepts.DBMaintenance
        :param processor_features: The number of CPU cores and the number of threads per core. Default: - the default number of CPU cores and threads per core for the chosen instance class. See https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html#USER_ConfigureProcessor
        :param publicly_accessible: Indicates whether the DB instance is an internet-facing instance. Default: - ``true`` if ``vpcSubnets`` is ``subnetType: SubnetType.PUBLIC``, ``false`` otherwise
        :param removal_policy: The CloudFormation policy to apply when the instance is removed from the stack or replaced during an update. Default: - RemovalPolicy.SNAPSHOT (remove the resource, but retain a snapshot of the data)
        :param s3_export_buckets: S3 buckets that you want to load data into. This feature is only supported by the Microsoft SQL Server and Oracle engines. This property must not be used if ``s3ExportRole`` is used. For Microsoft SQL Server: Default: - None
        :param s3_export_role: Role that will be associated with this DB instance to enable S3 export. This feature is only supported by the Microsoft SQL Server and Oracle engines. This property must not be used if ``s3ExportBuckets`` is used. For Microsoft SQL Server: Default: - New role is created if ``s3ExportBuckets`` is set, no role is defined otherwise
        :param s3_import_buckets: S3 buckets that you want to load data from. This feature is only supported by the Microsoft SQL Server, Oracle, and PostgreSQL engines. This property must not be used if ``s3ImportRole`` is used. For Microsoft SQL Server: Default: - None
        :param s3_import_role: Role that will be associated with this DB instance to enable S3 import. This feature is only supported by the Microsoft SQL Server, Oracle, and PostgreSQL engines. This property must not be used if ``s3ImportBuckets`` is used. For Microsoft SQL Server: Default: - New role is created if ``s3ImportBuckets`` is set, no role is defined otherwise
        :param security_groups: The security groups to assign to the DB instance. Default: - a new security group is created
        :param storage_type: The storage type. Default: GP2
        :param subnet_group: Existing subnet group for the instance. Default: - a new subnet group will be created.
        :param vpc_placement: (deprecated) The type of subnets to add to the created DB subnet group. Default: - private subnets
        :param vpc_subnets: The type of subnets to add to the created DB subnet group. Default: - private subnets
        :param engine: The database engine.
        :param allocated_storage: The allocated storage size, specified in gigabytes (GB). Default: 100
        :param allow_major_version_upgrade: Whether to allow major version upgrades. Default: false
        :param database_name: The name of the database. Default: - no name
        :param instance_type: The name of the compute and memory capacity for the instance. Default: - m5.large (or, more specifically, db.m5.large)
        :param license_model: The license model. Default: - RDS default license model
        :param parameter_group: The DB parameter group to associate with the instance. Default: - no parameter group
        :param timezone: The time zone of the instance. This is currently supported only by Microsoft Sql Server. Default: - RDS default timezone
        """
        if isinstance(processor_features, dict):
            processor_features = ProcessorFeatures(**processor_features)
        if isinstance(vpc_placement, dict):
            vpc_placement = aws_cdk.aws_ec2.SubnetSelection(**vpc_placement)
        if isinstance(vpc_subnets, dict):
            vpc_subnets = aws_cdk.aws_ec2.SubnetSelection(**vpc_subnets)
        self._values: typing.Dict[str, typing.Any] = {
            "vpc": vpc,
            "engine": engine,
        }
        if auto_minor_version_upgrade is not None:
            self._values["auto_minor_version_upgrade"] = auto_minor_version_upgrade
        if availability_zone is not None:
            self._values["availability_zone"] = availability_zone
        if backup_retention is not None:
            self._values["backup_retention"] = backup_retention
        if cloudwatch_logs_exports is not None:
            self._values["cloudwatch_logs_exports"] = cloudwatch_logs_exports
        if cloudwatch_logs_retention is not None:
            self._values["cloudwatch_logs_retention"] = cloudwatch_logs_retention
        if cloudwatch_logs_retention_role is not None:
            self._values["cloudwatch_logs_retention_role"] = cloudwatch_logs_retention_role
        if copy_tags_to_snapshot is not None:
            self._values["copy_tags_to_snapshot"] = copy_tags_to_snapshot
        if delete_automated_backups is not None:
            self._values["delete_automated_backups"] = delete_automated_backups
        if deletion_protection is not None:
            self._values["deletion_protection"] = deletion_protection
        if domain is not None:
            self._values["domain"] = domain
        if domain_role is not None:
            self._values["domain_role"] = domain_role
        if enable_performance_insights is not None:
            self._values["enable_performance_insights"] = enable_performance_insights
        if iam_authentication is not None:
            self._values["iam_authentication"] = iam_authentication
        if instance_identifier is not None:
            self._values["instance_identifier"] = instance_identifier
        if iops is not None:
            self._values["iops"] = iops
        if max_allocated_storage is not None:
            self._values["max_allocated_storage"] = max_allocated_storage
        if monitoring_interval is not None:
            self._values["monitoring_interval"] = monitoring_interval
        if monitoring_role is not None:
            self._values["monitoring_role"] = monitoring_role
        if multi_az is not None:
            self._values["multi_az"] = multi_az
        if option_group is not None:
            self._values["option_group"] = option_group
        if performance_insight_encryption_key is not None:
            self._values["performance_insight_encryption_key"] = performance_insight_encryption_key
        if performance_insight_retention is not None:
            self._values["performance_insight_retention"] = performance_insight_retention
        if port is not None:
            self._values["port"] = port
        if preferred_backup_window is not None:
            self._values["preferred_backup_window"] = preferred_backup_window
        if preferred_maintenance_window is not None:
            self._values["preferred_maintenance_window"] = preferred_maintenance_window
        if processor_features is not None:
            self._values["processor_features"] = processor_features
        if publicly_accessible is not None:
            self._values["publicly_accessible"] = publicly_accessible
        if removal_policy is not None:
            self._values["removal_policy"] = removal_policy
        if s3_export_buckets is not None:
            self._values["s3_export_buckets"] = s3_export_buckets
        if s3_export_role is not None:
            self._values["s3_export_role"] = s3_export_role
        if s3_import_buckets is not None:
            self._values["s3_import_buckets"] = s3_import_buckets
        if s3_import_role is not None:
            self._values["s3_import_role"] = s3_import_role
        if security_groups is not None:
            self._values["security_groups"] = security_groups
        if storage_type is not None:
            self._values["storage_type"] = storage_type
        if subnet_group is not None:
            self._values["subnet_group"] = subnet_group
        if vpc_placement is not None:
            self._values["vpc_placement"] = vpc_placement
        if vpc_subnets is not None:
            self._values["vpc_subnets"] = vpc_subnets
        if allocated_storage is not None:
            self._values["allocated_storage"] = allocated_storage
        if allow_major_version_upgrade is not None:
            self._values["allow_major_version_upgrade"] = allow_major_version_upgrade
        if database_name is not None:
            self._values["database_name"] = database_name
        if instance_type is not None:
            self._values["instance_type"] = instance_type
        if license_model is not None:
            self._values["license_model"] = license_model
        if parameter_group is not None:
            self._values["parameter_group"] = parameter_group
        if timezone is not None:
            self._values["timezone"] = timezone

    @builtins.property
    def vpc(self) -> aws_cdk.aws_ec2.IVpc:
        """The VPC network where the DB subnet group should be created."""
        result = self._values.get("vpc")
        assert result is not None, "Required property 'vpc' is missing"
        return result

    @builtins.property
    def auto_minor_version_upgrade(self) -> typing.Optional[builtins.bool]:
        """Indicates that minor engine upgrades are applied automatically to the DB instance during the maintenance window.

        :default: true
        """
        result = self._values.get("auto_minor_version_upgrade")
        return result

    @builtins.property
    def availability_zone(self) -> typing.Optional[builtins.str]:
        """The name of the Availability Zone where the DB instance will be located.

        :default: - no preference
        """
        result = self._values.get("availability_zone")
        return result

    @builtins.property
    def backup_retention(self) -> typing.Optional[aws_cdk.core.Duration]:
        """The number of days during which automatic DB snapshots are retained.

        Set to zero to disable backups.
        When creating a read replica, you must enable automatic backups on the source
        database instance by setting the backup retention to a value other than zero.

        :default: Duration.days(1)
        """
        result = self._values.get("backup_retention")
        return result

    @builtins.property
    def cloudwatch_logs_exports(self) -> typing.Optional[typing.List[builtins.str]]:
        """The list of log types that need to be enabled for exporting to CloudWatch Logs.

        :default: - no log exports
        """
        result = self._values.get("cloudwatch_logs_exports")
        return result

    @builtins.property
    def cloudwatch_logs_retention(
        self,
    ) -> typing.Optional[aws_cdk.aws_logs.RetentionDays]:
        """The number of days log events are kept in CloudWatch Logs.

        When updating
        this property, unsetting it doesn't remove the log retention policy. To
        remove the retention policy, set the value to ``Infinity``.

        :default: - logs never expire
        """
        result = self._values.get("cloudwatch_logs_retention")
        return result

    @builtins.property
    def cloudwatch_logs_retention_role(self) -> typing.Optional[aws_cdk.aws_iam.IRole]:
        """The IAM role for the Lambda function associated with the custom resource that sets the retention policy.

        :default: - a new role is created.
        """
        result = self._values.get("cloudwatch_logs_retention_role")
        return result

    @builtins.property
    def copy_tags_to_snapshot(self) -> typing.Optional[builtins.bool]:
        """Indicates whether to copy all of the user-defined tags from the DB instance to snapshots of the DB instance.

        :default: true
        """
        result = self._values.get("copy_tags_to_snapshot")
        return result

    @builtins.property
    def delete_automated_backups(self) -> typing.Optional[builtins.bool]:
        """Indicates whether automated backups should be deleted or retained when you delete a DB instance.

        :default: false
        """
        result = self._values.get("delete_automated_backups")
        return result

    @builtins.property
    def deletion_protection(self) -> typing.Optional[builtins.bool]:
        """Indicates whether the DB instance should have deletion protection enabled.

        :default: - true if ``removalPolicy`` is RETAIN, false otherwise
        """
        result = self._values.get("deletion_protection")
        return result

    @builtins.property
    def domain(self) -> typing.Optional[builtins.str]:
        """The Active Directory directory ID to create the DB instance in.

        :default: - Do not join domain
        """
        result = self._values.get("domain")
        return result

    @builtins.property
    def domain_role(self) -> typing.Optional[aws_cdk.aws_iam.IRole]:
        """The IAM role to be used when making API calls to the Directory Service.

        The role needs the AWS-managed policy
        AmazonRDSDirectoryServiceAccess or equivalent.

        :default: - The role will be created for you if {@link DatabaseInstanceNewProps#domain} is specified
        """
        result = self._values.get("domain_role")
        return result

    @builtins.property
    def enable_performance_insights(self) -> typing.Optional[builtins.bool]:
        """Whether to enable Performance Insights for the DB instance.

        :default: - false, unless ``performanceInsightRentention`` or ``performanceInsightEncryptionKey`` is set.
        """
        result = self._values.get("enable_performance_insights")
        return result

    @builtins.property
    def iam_authentication(self) -> typing.Optional[builtins.bool]:
        """Whether to enable mapping of AWS Identity and Access Management (IAM) accounts to database accounts.

        :default: false
        """
        result = self._values.get("iam_authentication")
        return result

    @builtins.property
    def instance_identifier(self) -> typing.Optional[builtins.str]:
        """A name for the DB instance.

        If you specify a name, AWS CloudFormation
        converts it to lowercase.

        :default: - a CloudFormation generated name
        """
        result = self._values.get("instance_identifier")
        return result

    @builtins.property
    def iops(self) -> typing.Optional[jsii.Number]:
        """The number of I/O operations per second (IOPS) that the database provisions.

        The value must be equal to or greater than 1000.

        :default: - no provisioned iops
        """
        result = self._values.get("iops")
        return result

    @builtins.property
    def max_allocated_storage(self) -> typing.Optional[jsii.Number]:
        """Upper limit to which RDS can scale the storage in GiB(Gibibyte).

        :default: - No autoscaling of RDS instance

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PIOPS.StorageTypes.html#USER_PIOPS.Autoscaling
        """
        result = self._values.get("max_allocated_storage")
        return result

    @builtins.property
    def monitoring_interval(self) -> typing.Optional[aws_cdk.core.Duration]:
        """The interval, in seconds, between points when Amazon RDS collects enhanced monitoring metrics for the DB instance.

        :default: - no enhanced monitoring
        """
        result = self._values.get("monitoring_interval")
        return result

    @builtins.property
    def monitoring_role(self) -> typing.Optional[aws_cdk.aws_iam.IRole]:
        """Role that will be used to manage DB instance monitoring.

        :default: - A role is automatically created for you
        """
        result = self._values.get("monitoring_role")
        return result

    @builtins.property
    def multi_az(self) -> typing.Optional[builtins.bool]:
        """Specifies if the database instance is a multiple Availability Zone deployment.

        :default: false
        """
        result = self._values.get("multi_az")
        return result

    @builtins.property
    def option_group(self) -> typing.Optional["IOptionGroup"]:
        """The option group to associate with the instance.

        :default: - no option group
        """
        result = self._values.get("option_group")
        return result

    @builtins.property
    def performance_insight_encryption_key(
        self,
    ) -> typing.Optional[aws_cdk.aws_kms.IKey]:
        """The AWS KMS key for encryption of Performance Insights data.

        :default: - default master key
        """
        result = self._values.get("performance_insight_encryption_key")
        return result

    @builtins.property
    def performance_insight_retention(
        self,
    ) -> typing.Optional["PerformanceInsightRetention"]:
        """The amount of time, in days, to retain Performance Insights data.

        :default: 7
        """
        result = self._values.get("performance_insight_retention")
        return result

    @builtins.property
    def port(self) -> typing.Optional[jsii.Number]:
        """The port for the instance.

        :default: - the default port for the chosen engine.
        """
        result = self._values.get("port")
        return result

    @builtins.property
    def preferred_backup_window(self) -> typing.Optional[builtins.str]:
        """The daily time range during which automated backups are performed.

        Constraints:

        - Must be in the format ``hh24:mi-hh24:mi``.
        - Must be in Universal Coordinated Time (UTC).
        - Must not conflict with the preferred maintenance window.
        - Must be at least 30 minutes.

        :default:

        - a 30-minute window selected at random from an 8-hour block of
        time for each AWS Region. To see the time blocks available, see
        https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow
        """
        result = self._values.get("preferred_backup_window")
        return result

    @builtins.property
    def preferred_maintenance_window(self) -> typing.Optional[builtins.str]:
        """The weekly time range (in UTC) during which system maintenance can occur.

        Format: ``ddd:hh24:mi-ddd:hh24:mi``
        Constraint: Minimum 30-minute window

        :default:

        - a 30-minute window selected at random from an 8-hour block of
        time for each AWS Region, occurring on a random day of the week. To see
        the time blocks available, see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_UpgradeDBInstance.Maintenance.html#Concepts.DBMaintenance
        """
        result = self._values.get("preferred_maintenance_window")
        return result

    @builtins.property
    def processor_features(self) -> typing.Optional["ProcessorFeatures"]:
        """The number of CPU cores and the number of threads per core.

        :default:

        - the default number of CPU cores and threads per core for the
        chosen instance class.

        See https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html#USER_ConfigureProcessor
        """
        result = self._values.get("processor_features")
        return result

    @builtins.property
    def publicly_accessible(self) -> typing.Optional[builtins.bool]:
        """Indicates whether the DB instance is an internet-facing instance.

        :default: - ``true`` if ``vpcSubnets`` is ``subnetType: SubnetType.PUBLIC``, ``false`` otherwise
        """
        result = self._values.get("publicly_accessible")
        return result

    @builtins.property
    def removal_policy(self) -> typing.Optional[aws_cdk.core.RemovalPolicy]:
        """The CloudFormation policy to apply when the instance is removed from the stack or replaced during an update.

        :default: - RemovalPolicy.SNAPSHOT (remove the resource, but retain a snapshot of the data)
        """
        result = self._values.get("removal_policy")
        return result

    @builtins.property
    def s3_export_buckets(self) -> typing.Optional[typing.List[aws_cdk.aws_s3.IBucket]]:
        """S3 buckets that you want to load data into.

        This feature is only supported by the Microsoft SQL Server and Oracle engines.

        This property must not be used if ``s3ExportRole`` is used.

        For Microsoft SQL Server:

        :default: - None

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/oracle-s3-integration.html
        """
        result = self._values.get("s3_export_buckets")
        return result

    @builtins.property
    def s3_export_role(self) -> typing.Optional[aws_cdk.aws_iam.IRole]:
        """Role that will be associated with this DB instance to enable S3 export.

        This feature is only supported by the Microsoft SQL Server and Oracle engines.

        This property must not be used if ``s3ExportBuckets`` is used.

        For Microsoft SQL Server:

        :default: - New role is created if ``s3ExportBuckets`` is set, no role is defined otherwise

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/oracle-s3-integration.html
        """
        result = self._values.get("s3_export_role")
        return result

    @builtins.property
    def s3_import_buckets(self) -> typing.Optional[typing.List[aws_cdk.aws_s3.IBucket]]:
        """S3 buckets that you want to load data from.

        This feature is only supported by the Microsoft SQL Server, Oracle, and PostgreSQL engines.

        This property must not be used if ``s3ImportRole`` is used.

        For Microsoft SQL Server:

        :default: - None

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/PostgreSQL.Procedural.Importing.html
        """
        result = self._values.get("s3_import_buckets")
        return result

    @builtins.property
    def s3_import_role(self) -> typing.Optional[aws_cdk.aws_iam.IRole]:
        """Role that will be associated with this DB instance to enable S3 import.

        This feature is only supported by the Microsoft SQL Server, Oracle, and PostgreSQL engines.

        This property must not be used if ``s3ImportBuckets`` is used.

        For Microsoft SQL Server:

        :default: - New role is created if ``s3ImportBuckets`` is set, no role is defined otherwise

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/PostgreSQL.Procedural.Importing.html
        """
        result = self._values.get("s3_import_role")
        return result

    @builtins.property
    def security_groups(
        self,
    ) -> typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]]:
        """The security groups to assign to the DB instance.

        :default: - a new security group is created
        """
        result = self._values.get("security_groups")
        return result

    @builtins.property
    def storage_type(self) -> typing.Optional["StorageType"]:
        """The storage type.

        :default: GP2
        """
        result = self._values.get("storage_type")
        return result

    @builtins.property
    def subnet_group(self) -> typing.Optional["ISubnetGroup"]:
        """Existing subnet group for the instance.

        :default: - a new subnet group will be created.
        """
        result = self._values.get("subnet_group")
        return result

    @builtins.property
    def vpc_placement(self) -> typing.Optional[aws_cdk.aws_ec2.SubnetSelection]:
        """(deprecated) The type of subnets to add to the created DB subnet group.

        :default: - private subnets

        :deprecated: use ``vpcSubnets``

        :stability: deprecated
        """
        result = self._values.get("vpc_placement")
        return result

    @builtins.property
    def vpc_subnets(self) -> typing.Optional[aws_cdk.aws_ec2.SubnetSelection]:
        """The type of subnets to add to the created DB subnet group.

        :default: - private subnets
        """
        result = self._values.get("vpc_subnets")
        return result

    @builtins.property
    def engine(self) -> "IInstanceEngine":
        """The database engine."""
        result = self._values.get("engine")
        assert result is not None, "Required property 'engine' is missing"
        return result

    @builtins.property
    def allocated_storage(self) -> typing.Optional[jsii.Number]:
        """The allocated storage size, specified in gigabytes (GB).

        :default: 100
        """
        result = self._values.get("allocated_storage")
        return result

    @builtins.property
    def allow_major_version_upgrade(self) -> typing.Optional[builtins.bool]:
        """Whether to allow major version upgrades.

        :default: false
        """
        result = self._values.get("allow_major_version_upgrade")
        return result

    @builtins.property
    def database_name(self) -> typing.Optional[builtins.str]:
        """The name of the database.

        :default: - no name
        """
        result = self._values.get("database_name")
        return result

    @builtins.property
    def instance_type(self) -> typing.Optional[aws_cdk.aws_ec2.InstanceType]:
        """The name of the compute and memory capacity for the instance.

        :default: - m5.large (or, more specifically, db.m5.large)
        """
        result = self._values.get("instance_type")
        return result

    @builtins.property
    def license_model(self) -> typing.Optional["LicenseModel"]:
        """The license model.

        :default: - RDS default license model
        """
        result = self._values.get("license_model")
        return result

    @builtins.property
    def parameter_group(self) -> typing.Optional["IParameterGroup"]:
        """The DB parameter group to associate with the instance.

        :default: - no parameter group
        """
        result = self._values.get("parameter_group")
        return result

    @builtins.property
    def timezone(self) -> typing.Optional[builtins.str]:
        """The time zone of the instance.

        This is currently supported only by Microsoft Sql Server.

        :default: - RDS default timezone
        """
        result = self._values.get("timezone")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DatabaseInstanceSourceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.DatabaseProxyAttributes",
    jsii_struct_bases=[],
    name_mapping={
        "db_proxy_arn": "dbProxyArn",
        "db_proxy_name": "dbProxyName",
        "endpoint": "endpoint",
        "security_groups": "securityGroups",
    },
)
class DatabaseProxyAttributes:
    def __init__(
        self,
        *,
        db_proxy_arn: builtins.str,
        db_proxy_name: builtins.str,
        endpoint: builtins.str,
        security_groups: typing.List[aws_cdk.aws_ec2.ISecurityGroup],
    ) -> None:
        """Properties that describe an existing DB Proxy.

        :param db_proxy_arn: DB Proxy ARN.
        :param db_proxy_name: DB Proxy Name.
        :param endpoint: Endpoint.
        :param security_groups: The security groups of the instance.
        """
        self._values: typing.Dict[str, typing.Any] = {
            "db_proxy_arn": db_proxy_arn,
            "db_proxy_name": db_proxy_name,
            "endpoint": endpoint,
            "security_groups": security_groups,
        }

    @builtins.property
    def db_proxy_arn(self) -> builtins.str:
        """DB Proxy ARN."""
        result = self._values.get("db_proxy_arn")
        assert result is not None, "Required property 'db_proxy_arn' is missing"
        return result

    @builtins.property
    def db_proxy_name(self) -> builtins.str:
        """DB Proxy Name."""
        result = self._values.get("db_proxy_name")
        assert result is not None, "Required property 'db_proxy_name' is missing"
        return result

    @builtins.property
    def endpoint(self) -> builtins.str:
        """Endpoint."""
        result = self._values.get("endpoint")
        assert result is not None, "Required property 'endpoint' is missing"
        return result

    @builtins.property
    def security_groups(self) -> typing.List[aws_cdk.aws_ec2.ISecurityGroup]:
        """The security groups of the instance."""
        result = self._values.get("security_groups")
        assert result is not None, "Required property 'security_groups' is missing"
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DatabaseProxyAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.DatabaseProxyOptions",
    jsii_struct_bases=[],
    name_mapping={
        "secrets": "secrets",
        "vpc": "vpc",
        "borrow_timeout": "borrowTimeout",
        "db_proxy_name": "dbProxyName",
        "debug_logging": "debugLogging",
        "iam_auth": "iamAuth",
        "idle_client_timeout": "idleClientTimeout",
        "init_query": "initQuery",
        "max_connections_percent": "maxConnectionsPercent",
        "max_idle_connections_percent": "maxIdleConnectionsPercent",
        "require_tls": "requireTLS",
        "role": "role",
        "security_groups": "securityGroups",
        "session_pinning_filters": "sessionPinningFilters",
        "vpc_subnets": "vpcSubnets",
    },
)
class DatabaseProxyOptions:
    def __init__(
        self,
        *,
        secrets: typing.List[aws_cdk.aws_secretsmanager.ISecret],
        vpc: aws_cdk.aws_ec2.IVpc,
        borrow_timeout: typing.Optional[aws_cdk.core.Duration] = None,
        db_proxy_name: typing.Optional[builtins.str] = None,
        debug_logging: typing.Optional[builtins.bool] = None,
        iam_auth: typing.Optional[builtins.bool] = None,
        idle_client_timeout: typing.Optional[aws_cdk.core.Duration] = None,
        init_query: typing.Optional[builtins.str] = None,
        max_connections_percent: typing.Optional[jsii.Number] = None,
        max_idle_connections_percent: typing.Optional[jsii.Number] = None,
        require_tls: typing.Optional[builtins.bool] = None,
        role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        security_groups: typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]] = None,
        session_pinning_filters: typing.Optional[typing.List["SessionPinningFilter"]] = None,
        vpc_subnets: typing.Optional[aws_cdk.aws_ec2.SubnetSelection] = None,
    ) -> None:
        """Options for a new DatabaseProxy.

        :param secrets: The secret that the proxy uses to authenticate to the RDS DB instance or Aurora DB cluster. These secrets are stored within Amazon Secrets Manager. One or more secrets are required.
        :param vpc: The VPC to associate with the new proxy.
        :param borrow_timeout: The duration for a proxy to wait for a connection to become available in the connection pool. Only applies when the proxy has opened its maximum number of connections and all connections are busy with client sessions. Value must be between 1 second and 1 hour, or ``Duration.seconds(0)`` to represent unlimited. Default: cdk.Duration.seconds(120)
        :param db_proxy_name: The identifier for the proxy. This name must be unique for all proxies owned by your AWS account in the specified AWS Region. An identifier must begin with a letter and must contain only ASCII letters, digits, and hyphens; it can't end with a hyphen or contain two consecutive hyphens. Default: - Generated by CloudFormation (recommended)
        :param debug_logging: Whether the proxy includes detailed information about SQL statements in its logs. This information helps you to debug issues involving SQL behavior or the performance and scalability of the proxy connections. The debug information includes the text of SQL statements that you submit through the proxy. Thus, only enable this setting when needed for debugging, and only when you have security measures in place to safeguard any sensitive information that appears in the logs. Default: false
        :param iam_auth: Whether to require or disallow AWS Identity and Access Management (IAM) authentication for connections to the proxy. Default: false
        :param idle_client_timeout: The number of seconds that a connection to the proxy can be inactive before the proxy disconnects it. You can set this value higher or lower than the connection timeout limit for the associated database. Default: cdk.Duration.minutes(30)
        :param init_query: One or more SQL statements for the proxy to run when opening each new database connection. Typically used with SET statements to make sure that each connection has identical settings such as time zone and character set. For multiple statements, use semicolons as the separator. You can also include multiple variables in a single SET statement, such as SET x=1, y=2. not currently supported for PostgreSQL. Default: - no initialization query
        :param max_connections_percent: The maximum size of the connection pool for each target in a target group. For Aurora MySQL, it is expressed as a percentage of the max_connections setting for the RDS DB instance or Aurora DB cluster used by the target group. 1-100 Default: 100
        :param max_idle_connections_percent: Controls how actively the proxy closes idle database connections in the connection pool. A high value enables the proxy to leave a high percentage of idle connections open. A low value causes the proxy to close idle client connections and return the underlying database connections to the connection pool. For Aurora MySQL, it is expressed as a percentage of the max_connections setting for the RDS DB instance or Aurora DB cluster used by the target group. between 0 and MaxConnectionsPercent Default: 50
        :param require_tls: A Boolean parameter that specifies whether Transport Layer Security (TLS) encryption is required for connections to the proxy. By enabling this setting, you can enforce encrypted TLS connections to the proxy. Default: true
        :param role: IAM role that the proxy uses to access secrets in AWS Secrets Manager. Default: - A role will automatically be created
        :param security_groups: One or more VPC security groups to associate with the new proxy. Default: - No security groups
        :param session_pinning_filters: Each item in the list represents a class of SQL operations that normally cause all later statements in a session using a proxy to be pinned to the same underlying database connection. Including an item in the list exempts that class of SQL operations from the pinning behavior. Default: - no session pinning filters
        :param vpc_subnets: The subnets used by the proxy. Default: - the VPC default strategy if not specified.
        """
        if isinstance(vpc_subnets, dict):
            vpc_subnets = aws_cdk.aws_ec2.SubnetSelection(**vpc_subnets)
        self._values: typing.Dict[str, typing.Any] = {
            "secrets": secrets,
            "vpc": vpc,
        }
        if borrow_timeout is not None:
            self._values["borrow_timeout"] = borrow_timeout
        if db_proxy_name is not None:
            self._values["db_proxy_name"] = db_proxy_name
        if debug_logging is not None:
            self._values["debug_logging"] = debug_logging
        if iam_auth is not None:
            self._values["iam_auth"] = iam_auth
        if idle_client_timeout is not None:
            self._values["idle_client_timeout"] = idle_client_timeout
        if init_query is not None:
            self._values["init_query"] = init_query
        if max_connections_percent is not None:
            self._values["max_connections_percent"] = max_connections_percent
        if max_idle_connections_percent is not None:
            self._values["max_idle_connections_percent"] = max_idle_connections_percent
        if require_tls is not None:
            self._values["require_tls"] = require_tls
        if role is not None:
            self._values["role"] = role
        if security_groups is not None:
            self._values["security_groups"] = security_groups
        if session_pinning_filters is not None:
            self._values["session_pinning_filters"] = session_pinning_filters
        if vpc_subnets is not None:
            self._values["vpc_subnets"] = vpc_subnets

    @builtins.property
    def secrets(self) -> typing.List[aws_cdk.aws_secretsmanager.ISecret]:
        """The secret that the proxy uses to authenticate to the RDS DB instance or Aurora DB cluster.

        These secrets are stored within Amazon Secrets Manager.
        One or more secrets are required.
        """
        result = self._values.get("secrets")
        assert result is not None, "Required property 'secrets' is missing"
        return result

    @builtins.property
    def vpc(self) -> aws_cdk.aws_ec2.IVpc:
        """The VPC to associate with the new proxy."""
        result = self._values.get("vpc")
        assert result is not None, "Required property 'vpc' is missing"
        return result

    @builtins.property
    def borrow_timeout(self) -> typing.Optional[aws_cdk.core.Duration]:
        """The duration for a proxy to wait for a connection to become available in the connection pool.

        Only applies when the proxy has opened its maximum number of connections and all connections are busy with client
        sessions.

        Value must be between 1 second and 1 hour, or ``Duration.seconds(0)`` to represent unlimited.

        :default: cdk.Duration.seconds(120)
        """
        result = self._values.get("borrow_timeout")
        return result

    @builtins.property
    def db_proxy_name(self) -> typing.Optional[builtins.str]:
        """The identifier for the proxy.

        This name must be unique for all proxies owned by your AWS account in the specified AWS Region.
        An identifier must begin with a letter and must contain only ASCII letters, digits, and hyphens;
        it can't end with a hyphen or contain two consecutive hyphens.

        :default: - Generated by CloudFormation (recommended)
        """
        result = self._values.get("db_proxy_name")
        return result

    @builtins.property
    def debug_logging(self) -> typing.Optional[builtins.bool]:
        """Whether the proxy includes detailed information about SQL statements in its logs.

        This information helps you to debug issues involving SQL behavior or the performance and scalability of the proxy connections.
        The debug information includes the text of SQL statements that you submit through the proxy.
        Thus, only enable this setting when needed for debugging, and only when you have security measures in place to safeguard any sensitive
        information that appears in the logs.

        :default: false
        """
        result = self._values.get("debug_logging")
        return result

    @builtins.property
    def iam_auth(self) -> typing.Optional[builtins.bool]:
        """Whether to require or disallow AWS Identity and Access Management (IAM) authentication for connections to the proxy.

        :default: false
        """
        result = self._values.get("iam_auth")
        return result

    @builtins.property
    def idle_client_timeout(self) -> typing.Optional[aws_cdk.core.Duration]:
        """The number of seconds that a connection to the proxy can be inactive before the proxy disconnects it.

        You can set this value higher or lower than the connection timeout limit for the associated database.

        :default: cdk.Duration.minutes(30)
        """
        result = self._values.get("idle_client_timeout")
        return result

    @builtins.property
    def init_query(self) -> typing.Optional[builtins.str]:
        """One or more SQL statements for the proxy to run when opening each new database connection.

        Typically used with SET statements to make sure that each connection has identical settings such as time zone
        and character set.
        For multiple statements, use semicolons as the separator.
        You can also include multiple variables in a single SET statement, such as SET x=1, y=2.

        not currently supported for PostgreSQL.

        :default: - no initialization query
        """
        result = self._values.get("init_query")
        return result

    @builtins.property
    def max_connections_percent(self) -> typing.Optional[jsii.Number]:
        """The maximum size of the connection pool for each target in a target group.

        For Aurora MySQL, it is expressed as a percentage of the max_connections setting for the RDS DB instance or Aurora DB
        cluster used by the target group.

        1-100

        :default: 100
        """
        result = self._values.get("max_connections_percent")
        return result

    @builtins.property
    def max_idle_connections_percent(self) -> typing.Optional[jsii.Number]:
        """Controls how actively the proxy closes idle database connections in the connection pool.

        A high value enables the proxy to leave a high percentage of idle connections open.
        A low value causes the proxy to close idle client connections and return the underlying database connections
        to the connection pool.
        For Aurora MySQL, it is expressed as a percentage of the max_connections setting for the RDS DB instance
        or Aurora DB cluster used by the target group.

        between 0 and MaxConnectionsPercent

        :default: 50
        """
        result = self._values.get("max_idle_connections_percent")
        return result

    @builtins.property
    def require_tls(self) -> typing.Optional[builtins.bool]:
        """A Boolean parameter that specifies whether Transport Layer Security (TLS) encryption is required for connections to the proxy.

        By enabling this setting, you can enforce encrypted TLS connections to the proxy.

        :default: true
        """
        result = self._values.get("require_tls")
        return result

    @builtins.property
    def role(self) -> typing.Optional[aws_cdk.aws_iam.IRole]:
        """IAM role that the proxy uses to access secrets in AWS Secrets Manager.

        :default: - A role will automatically be created
        """
        result = self._values.get("role")
        return result

    @builtins.property
    def security_groups(
        self,
    ) -> typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]]:
        """One or more VPC security groups to associate with the new proxy.

        :default: - No security groups
        """
        result = self._values.get("security_groups")
        return result

    @builtins.property
    def session_pinning_filters(
        self,
    ) -> typing.Optional[typing.List["SessionPinningFilter"]]:
        """Each item in the list represents a class of SQL operations that normally cause all later statements in a session using a proxy to be pinned to the same underlying database connection.

        Including an item in the list exempts that class of SQL operations from the pinning behavior.

        :default: - no session pinning filters
        """
        result = self._values.get("session_pinning_filters")
        return result

    @builtins.property
    def vpc_subnets(self) -> typing.Optional[aws_cdk.aws_ec2.SubnetSelection]:
        """The subnets used by the proxy.

        :default: - the VPC default strategy if not specified.
        """
        result = self._values.get("vpc_subnets")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DatabaseProxyOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.DatabaseProxyProps",
    jsii_struct_bases=[DatabaseProxyOptions],
    name_mapping={
        "secrets": "secrets",
        "vpc": "vpc",
        "borrow_timeout": "borrowTimeout",
        "db_proxy_name": "dbProxyName",
        "debug_logging": "debugLogging",
        "iam_auth": "iamAuth",
        "idle_client_timeout": "idleClientTimeout",
        "init_query": "initQuery",
        "max_connections_percent": "maxConnectionsPercent",
        "max_idle_connections_percent": "maxIdleConnectionsPercent",
        "require_tls": "requireTLS",
        "role": "role",
        "security_groups": "securityGroups",
        "session_pinning_filters": "sessionPinningFilters",
        "vpc_subnets": "vpcSubnets",
        "proxy_target": "proxyTarget",
    },
)
class DatabaseProxyProps(DatabaseProxyOptions):
    def __init__(
        self,
        *,
        secrets: typing.List[aws_cdk.aws_secretsmanager.ISecret],
        vpc: aws_cdk.aws_ec2.IVpc,
        borrow_timeout: typing.Optional[aws_cdk.core.Duration] = None,
        db_proxy_name: typing.Optional[builtins.str] = None,
        debug_logging: typing.Optional[builtins.bool] = None,
        iam_auth: typing.Optional[builtins.bool] = None,
        idle_client_timeout: typing.Optional[aws_cdk.core.Duration] = None,
        init_query: typing.Optional[builtins.str] = None,
        max_connections_percent: typing.Optional[jsii.Number] = None,
        max_idle_connections_percent: typing.Optional[jsii.Number] = None,
        require_tls: typing.Optional[builtins.bool] = None,
        role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        security_groups: typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]] = None,
        session_pinning_filters: typing.Optional[typing.List["SessionPinningFilter"]] = None,
        vpc_subnets: typing.Optional[aws_cdk.aws_ec2.SubnetSelection] = None,
        proxy_target: "ProxyTarget",
    ) -> None:
        """Construction properties for a DatabaseProxy.

        :param secrets: The secret that the proxy uses to authenticate to the RDS DB instance or Aurora DB cluster. These secrets are stored within Amazon Secrets Manager. One or more secrets are required.
        :param vpc: The VPC to associate with the new proxy.
        :param borrow_timeout: The duration for a proxy to wait for a connection to become available in the connection pool. Only applies when the proxy has opened its maximum number of connections and all connections are busy with client sessions. Value must be between 1 second and 1 hour, or ``Duration.seconds(0)`` to represent unlimited. Default: cdk.Duration.seconds(120)
        :param db_proxy_name: The identifier for the proxy. This name must be unique for all proxies owned by your AWS account in the specified AWS Region. An identifier must begin with a letter and must contain only ASCII letters, digits, and hyphens; it can't end with a hyphen or contain two consecutive hyphens. Default: - Generated by CloudFormation (recommended)
        :param debug_logging: Whether the proxy includes detailed information about SQL statements in its logs. This information helps you to debug issues involving SQL behavior or the performance and scalability of the proxy connections. The debug information includes the text of SQL statements that you submit through the proxy. Thus, only enable this setting when needed for debugging, and only when you have security measures in place to safeguard any sensitive information that appears in the logs. Default: false
        :param iam_auth: Whether to require or disallow AWS Identity and Access Management (IAM) authentication for connections to the proxy. Default: false
        :param idle_client_timeout: The number of seconds that a connection to the proxy can be inactive before the proxy disconnects it. You can set this value higher or lower than the connection timeout limit for the associated database. Default: cdk.Duration.minutes(30)
        :param init_query: One or more SQL statements for the proxy to run when opening each new database connection. Typically used with SET statements to make sure that each connection has identical settings such as time zone and character set. For multiple statements, use semicolons as the separator. You can also include multiple variables in a single SET statement, such as SET x=1, y=2. not currently supported for PostgreSQL. Default: - no initialization query
        :param max_connections_percent: The maximum size of the connection pool for each target in a target group. For Aurora MySQL, it is expressed as a percentage of the max_connections setting for the RDS DB instance or Aurora DB cluster used by the target group. 1-100 Default: 100
        :param max_idle_connections_percent: Controls how actively the proxy closes idle database connections in the connection pool. A high value enables the proxy to leave a high percentage of idle connections open. A low value causes the proxy to close idle client connections and return the underlying database connections to the connection pool. For Aurora MySQL, it is expressed as a percentage of the max_connections setting for the RDS DB instance or Aurora DB cluster used by the target group. between 0 and MaxConnectionsPercent Default: 50
        :param require_tls: A Boolean parameter that specifies whether Transport Layer Security (TLS) encryption is required for connections to the proxy. By enabling this setting, you can enforce encrypted TLS connections to the proxy. Default: true
        :param role: IAM role that the proxy uses to access secrets in AWS Secrets Manager. Default: - A role will automatically be created
        :param security_groups: One or more VPC security groups to associate with the new proxy. Default: - No security groups
        :param session_pinning_filters: Each item in the list represents a class of SQL operations that normally cause all later statements in a session using a proxy to be pinned to the same underlying database connection. Including an item in the list exempts that class of SQL operations from the pinning behavior. Default: - no session pinning filters
        :param vpc_subnets: The subnets used by the proxy. Default: - the VPC default strategy if not specified.
        :param proxy_target: DB proxy target: Instance or Cluster.
        """
        if isinstance(vpc_subnets, dict):
            vpc_subnets = aws_cdk.aws_ec2.SubnetSelection(**vpc_subnets)
        self._values: typing.Dict[str, typing.Any] = {
            "secrets": secrets,
            "vpc": vpc,
            "proxy_target": proxy_target,
        }
        if borrow_timeout is not None:
            self._values["borrow_timeout"] = borrow_timeout
        if db_proxy_name is not None:
            self._values["db_proxy_name"] = db_proxy_name
        if debug_logging is not None:
            self._values["debug_logging"] = debug_logging
        if iam_auth is not None:
            self._values["iam_auth"] = iam_auth
        if idle_client_timeout is not None:
            self._values["idle_client_timeout"] = idle_client_timeout
        if init_query is not None:
            self._values["init_query"] = init_query
        if max_connections_percent is not None:
            self._values["max_connections_percent"] = max_connections_percent
        if max_idle_connections_percent is not None:
            self._values["max_idle_connections_percent"] = max_idle_connections_percent
        if require_tls is not None:
            self._values["require_tls"] = require_tls
        if role is not None:
            self._values["role"] = role
        if security_groups is not None:
            self._values["security_groups"] = security_groups
        if session_pinning_filters is not None:
            self._values["session_pinning_filters"] = session_pinning_filters
        if vpc_subnets is not None:
            self._values["vpc_subnets"] = vpc_subnets

    @builtins.property
    def secrets(self) -> typing.List[aws_cdk.aws_secretsmanager.ISecret]:
        """The secret that the proxy uses to authenticate to the RDS DB instance or Aurora DB cluster.

        These secrets are stored within Amazon Secrets Manager.
        One or more secrets are required.
        """
        result = self._values.get("secrets")
        assert result is not None, "Required property 'secrets' is missing"
        return result

    @builtins.property
    def vpc(self) -> aws_cdk.aws_ec2.IVpc:
        """The VPC to associate with the new proxy."""
        result = self._values.get("vpc")
        assert result is not None, "Required property 'vpc' is missing"
        return result

    @builtins.property
    def borrow_timeout(self) -> typing.Optional[aws_cdk.core.Duration]:
        """The duration for a proxy to wait for a connection to become available in the connection pool.

        Only applies when the proxy has opened its maximum number of connections and all connections are busy with client
        sessions.

        Value must be between 1 second and 1 hour, or ``Duration.seconds(0)`` to represent unlimited.

        :default: cdk.Duration.seconds(120)
        """
        result = self._values.get("borrow_timeout")
        return result

    @builtins.property
    def db_proxy_name(self) -> typing.Optional[builtins.str]:
        """The identifier for the proxy.

        This name must be unique for all proxies owned by your AWS account in the specified AWS Region.
        An identifier must begin with a letter and must contain only ASCII letters, digits, and hyphens;
        it can't end with a hyphen or contain two consecutive hyphens.

        :default: - Generated by CloudFormation (recommended)
        """
        result = self._values.get("db_proxy_name")
        return result

    @builtins.property
    def debug_logging(self) -> typing.Optional[builtins.bool]:
        """Whether the proxy includes detailed information about SQL statements in its logs.

        This information helps you to debug issues involving SQL behavior or the performance and scalability of the proxy connections.
        The debug information includes the text of SQL statements that you submit through the proxy.
        Thus, only enable this setting when needed for debugging, and only when you have security measures in place to safeguard any sensitive
        information that appears in the logs.

        :default: false
        """
        result = self._values.get("debug_logging")
        return result

    @builtins.property
    def iam_auth(self) -> typing.Optional[builtins.bool]:
        """Whether to require or disallow AWS Identity and Access Management (IAM) authentication for connections to the proxy.

        :default: false
        """
        result = self._values.get("iam_auth")
        return result

    @builtins.property
    def idle_client_timeout(self) -> typing.Optional[aws_cdk.core.Duration]:
        """The number of seconds that a connection to the proxy can be inactive before the proxy disconnects it.

        You can set this value higher or lower than the connection timeout limit for the associated database.

        :default: cdk.Duration.minutes(30)
        """
        result = self._values.get("idle_client_timeout")
        return result

    @builtins.property
    def init_query(self) -> typing.Optional[builtins.str]:
        """One or more SQL statements for the proxy to run when opening each new database connection.

        Typically used with SET statements to make sure that each connection has identical settings such as time zone
        and character set.
        For multiple statements, use semicolons as the separator.
        You can also include multiple variables in a single SET statement, such as SET x=1, y=2.

        not currently supported for PostgreSQL.

        :default: - no initialization query
        """
        result = self._values.get("init_query")
        return result

    @builtins.property
    def max_connections_percent(self) -> typing.Optional[jsii.Number]:
        """The maximum size of the connection pool for each target in a target group.

        For Aurora MySQL, it is expressed as a percentage of the max_connections setting for the RDS DB instance or Aurora DB
        cluster used by the target group.

        1-100

        :default: 100
        """
        result = self._values.get("max_connections_percent")
        return result

    @builtins.property
    def max_idle_connections_percent(self) -> typing.Optional[jsii.Number]:
        """Controls how actively the proxy closes idle database connections in the connection pool.

        A high value enables the proxy to leave a high percentage of idle connections open.
        A low value causes the proxy to close idle client connections and return the underlying database connections
        to the connection pool.
        For Aurora MySQL, it is expressed as a percentage of the max_connections setting for the RDS DB instance
        or Aurora DB cluster used by the target group.

        between 0 and MaxConnectionsPercent

        :default: 50
        """
        result = self._values.get("max_idle_connections_percent")
        return result

    @builtins.property
    def require_tls(self) -> typing.Optional[builtins.bool]:
        """A Boolean parameter that specifies whether Transport Layer Security (TLS) encryption is required for connections to the proxy.

        By enabling this setting, you can enforce encrypted TLS connections to the proxy.

        :default: true
        """
        result = self._values.get("require_tls")
        return result

    @builtins.property
    def role(self) -> typing.Optional[aws_cdk.aws_iam.IRole]:
        """IAM role that the proxy uses to access secrets in AWS Secrets Manager.

        :default: - A role will automatically be created
        """
        result = self._values.get("role")
        return result

    @builtins.property
    def security_groups(
        self,
    ) -> typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]]:
        """One or more VPC security groups to associate with the new proxy.

        :default: - No security groups
        """
        result = self._values.get("security_groups")
        return result

    @builtins.property
    def session_pinning_filters(
        self,
    ) -> typing.Optional[typing.List["SessionPinningFilter"]]:
        """Each item in the list represents a class of SQL operations that normally cause all later statements in a session using a proxy to be pinned to the same underlying database connection.

        Including an item in the list exempts that class of SQL operations from the pinning behavior.

        :default: - no session pinning filters
        """
        result = self._values.get("session_pinning_filters")
        return result

    @builtins.property
    def vpc_subnets(self) -> typing.Optional[aws_cdk.aws_ec2.SubnetSelection]:
        """The subnets used by the proxy.

        :default: - the VPC default strategy if not specified.
        """
        result = self._values.get("vpc_subnets")
        return result

    @builtins.property
    def proxy_target(self) -> "ProxyTarget":
        """DB proxy target: Instance or Cluster."""
        result = self._values.get("proxy_target")
        assert result is not None, "Required property 'proxy_target' is missing"
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DatabaseProxyProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class DatabaseSecret(
    aws_cdk.aws_secretsmanager.Secret,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.DatabaseSecret",
):
    """A database secret.

    :resource: AWS::SecretsManager::Secret
    """

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        username: builtins.str,
        encryption_key: typing.Optional[aws_cdk.aws_kms.IKey] = None,
        exclude_characters: typing.Optional[builtins.str] = None,
        master_secret: typing.Optional[aws_cdk.aws_secretsmanager.ISecret] = None,
        replace_on_password_criteria_changes: typing.Optional[builtins.bool] = None,
    ) -> None:
        """
        :param scope: -
        :param id: -
        :param username: The username.
        :param encryption_key: The KMS key to use to encrypt the secret. Default: default master key
        :param exclude_characters: Characters to not include in the generated password. Default: " %+~`#$&*()|[]{}:;<>?!'/
        :param master_secret: The master secret which will be used to rotate this secret. Default: - no master secret information will be included
        :param replace_on_password_criteria_changes: Whether to replace this secret when the criteria for the password change. This is achieved by overriding the logical id of the AWS::SecretsManager::Secret with a hash of the options that influence the password generation. This way a new secret will be created when the password is regenerated and the cluster or instance consuming this secret will have its credentials updated. Default: false
        """
        props = DatabaseSecretProps(
            username=username,
            encryption_key=encryption_key,
            exclude_characters=exclude_characters,
            master_secret=master_secret,
            replace_on_password_criteria_changes=replace_on_password_criteria_changes,
        )

        jsii.create(DatabaseSecret, self, [scope, id, props])


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.DatabaseSecretProps",
    jsii_struct_bases=[],
    name_mapping={
        "username": "username",
        "encryption_key": "encryptionKey",
        "exclude_characters": "excludeCharacters",
        "master_secret": "masterSecret",
        "replace_on_password_criteria_changes": "replaceOnPasswordCriteriaChanges",
    },
)
class DatabaseSecretProps:
    def __init__(
        self,
        *,
        username: builtins.str,
        encryption_key: typing.Optional[aws_cdk.aws_kms.IKey] = None,
        exclude_characters: typing.Optional[builtins.str] = None,
        master_secret: typing.Optional[aws_cdk.aws_secretsmanager.ISecret] = None,
        replace_on_password_criteria_changes: typing.Optional[builtins.bool] = None,
    ) -> None:
        """Construction properties for a DatabaseSecret.

        :param username: The username.
        :param encryption_key: The KMS key to use to encrypt the secret. Default: default master key
        :param exclude_characters: Characters to not include in the generated password. Default: " %+~`#$&*()|[]{}:;<>?!'/
        :param master_secret: The master secret which will be used to rotate this secret. Default: - no master secret information will be included
        :param replace_on_password_criteria_changes: Whether to replace this secret when the criteria for the password change. This is achieved by overriding the logical id of the AWS::SecretsManager::Secret with a hash of the options that influence the password generation. This way a new secret will be created when the password is regenerated and the cluster or instance consuming this secret will have its credentials updated. Default: false
        """
        self._values: typing.Dict[str, typing.Any] = {
            "username": username,
        }
        if encryption_key is not None:
            self._values["encryption_key"] = encryption_key
        if exclude_characters is not None:
            self._values["exclude_characters"] = exclude_characters
        if master_secret is not None:
            self._values["master_secret"] = master_secret
        if replace_on_password_criteria_changes is not None:
            self._values["replace_on_password_criteria_changes"] = replace_on_password_criteria_changes

    @builtins.property
    def username(self) -> builtins.str:
        """The username."""
        result = self._values.get("username")
        assert result is not None, "Required property 'username' is missing"
        return result

    @builtins.property
    def encryption_key(self) -> typing.Optional[aws_cdk.aws_kms.IKey]:
        """The KMS key to use to encrypt the secret.

        :default: default master key
        """
        result = self._values.get("encryption_key")
        return result

    @builtins.property
    def exclude_characters(self) -> typing.Optional[builtins.str]:
        """Characters to not include in the generated password.

        :default: " %+~`#$&*()|[]{}:;<>?!'/

        :: "\\"
        """
        result = self._values.get("exclude_characters")
        return result

    @builtins.property
    def master_secret(self) -> typing.Optional[aws_cdk.aws_secretsmanager.ISecret]:
        """The master secret which will be used to rotate this secret.

        :default: - no master secret information will be included
        """
        result = self._values.get("master_secret")
        return result

    @builtins.property
    def replace_on_password_criteria_changes(self) -> typing.Optional[builtins.bool]:
        """Whether to replace this secret when the criteria for the password change.

        This is achieved by overriding the logical id of the AWS::SecretsManager::Secret
        with a hash of the options that influence the password generation. This
        way a new secret will be created when the password is regenerated and the
        cluster or instance consuming this secret will have its credentials updated.

        :default: false
        """
        result = self._values.get("replace_on_password_criteria_changes")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DatabaseSecretProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class Endpoint(metaclass=jsii.JSIIMeta, jsii_type="@aws-cdk/aws-rds.Endpoint"):
    """Connection endpoint of a database cluster or instance.

    Consists of a combination of hostname and port.
    """

    def __init__(self, address: builtins.str, port: jsii.Number) -> None:
        """
        :param address: -
        :param port: -
        """
        jsii.create(Endpoint, self, [address, port])

    @builtins.property # type: ignore
    @jsii.member(jsii_name="hostname")
    def hostname(self) -> builtins.str:
        """The hostname of the endpoint."""
        return jsii.get(self, "hostname")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="port")
    def port(self) -> jsii.Number:
        """The port of the endpoint."""
        return jsii.get(self, "port")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="socketAddress")
    def socket_address(self) -> builtins.str:
        """The combination of "HOSTNAME:PORT" for this endpoint."""
        return jsii.get(self, "socketAddress")


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.EngineVersion",
    jsii_struct_bases=[],
    name_mapping={"major_version": "majorVersion", "full_version": "fullVersion"},
)
class EngineVersion:
    def __init__(
        self,
        *,
        major_version: builtins.str,
        full_version: typing.Optional[builtins.str] = None,
    ) -> None:
        """A version of an engine - for either a cluster, or instance.

        :param major_version: The major version of the engine, for example, "5.6". Used in specifying the ParameterGroup family and OptionGroup version for this engine.
        :param full_version: The full version string of the engine, for example, "5.6.mysql_aurora.1.22.1". It can be undefined, which means RDS should use whatever version it deems appropriate for the given engine type. Default: - no version specified
        """
        self._values: typing.Dict[str, typing.Any] = {
            "major_version": major_version,
        }
        if full_version is not None:
            self._values["full_version"] = full_version

    @builtins.property
    def major_version(self) -> builtins.str:
        """The major version of the engine, for example, "5.6". Used in specifying the ParameterGroup family and OptionGroup version for this engine."""
        result = self._values.get("major_version")
        assert result is not None, "Required property 'major_version' is missing"
        return result

    @builtins.property
    def full_version(self) -> typing.Optional[builtins.str]:
        """The full version string of the engine, for example, "5.6.mysql_aurora.1.22.1". It can be undefined, which means RDS should use whatever version it deems appropriate for the given engine type.

        :default: - no version specified
        """
        result = self._values.get("full_version")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EngineVersion(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.interface(jsii_type="@aws-cdk/aws-rds.IDatabaseCluster")
class IDatabaseCluster(
    aws_cdk.core.IResource,
    aws_cdk.aws_ec2.IConnectable,
    aws_cdk.aws_secretsmanager.ISecretAttachmentTarget,
    typing_extensions.Protocol,
):
    """Create a clustered database with a given number of instances."""

    @builtins.staticmethod
    def __jsii_proxy_class__():
        return _IDatabaseClusterProxy

    @builtins.property # type: ignore
    @jsii.member(jsii_name="clusterEndpoint")
    def cluster_endpoint(self) -> Endpoint:
        """The endpoint to use for read/write operations.

        :attribute: EndpointAddress,EndpointPort
        """
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="clusterIdentifier")
    def cluster_identifier(self) -> builtins.str:
        """Identifier of the cluster."""
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="clusterReadEndpoint")
    def cluster_read_endpoint(self) -> Endpoint:
        """Endpoint to use for load-balanced read-only operations.

        :attribute: ReadEndpointAddress
        """
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="instanceEndpoints")
    def instance_endpoints(self) -> typing.List[Endpoint]:
        """Endpoints which address each individual replica."""
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="instanceIdentifiers")
    def instance_identifiers(self) -> typing.List[builtins.str]:
        """Identifiers of the replicas."""
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="engine")
    def engine(self) -> typing.Optional["IClusterEngine"]:
        """The engine of this Cluster.

        May be not known for imported Clusters if it wasn't provided explicitly.
        """
        ...

    @jsii.member(jsii_name="addProxy")
    def add_proxy(
        self,
        id: builtins.str,
        *,
        secrets: typing.List[aws_cdk.aws_secretsmanager.ISecret],
        vpc: aws_cdk.aws_ec2.IVpc,
        borrow_timeout: typing.Optional[aws_cdk.core.Duration] = None,
        db_proxy_name: typing.Optional[builtins.str] = None,
        debug_logging: typing.Optional[builtins.bool] = None,
        iam_auth: typing.Optional[builtins.bool] = None,
        idle_client_timeout: typing.Optional[aws_cdk.core.Duration] = None,
        init_query: typing.Optional[builtins.str] = None,
        max_connections_percent: typing.Optional[jsii.Number] = None,
        max_idle_connections_percent: typing.Optional[jsii.Number] = None,
        require_tls: typing.Optional[builtins.bool] = None,
        role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        security_groups: typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]] = None,
        session_pinning_filters: typing.Optional[typing.List["SessionPinningFilter"]] = None,
        vpc_subnets: typing.Optional[aws_cdk.aws_ec2.SubnetSelection] = None,
    ) -> "DatabaseProxy":
        """Add a new db proxy to this cluster.

        :param id: -
        :param secrets: The secret that the proxy uses to authenticate to the RDS DB instance or Aurora DB cluster. These secrets are stored within Amazon Secrets Manager. One or more secrets are required.
        :param vpc: The VPC to associate with the new proxy.
        :param borrow_timeout: The duration for a proxy to wait for a connection to become available in the connection pool. Only applies when the proxy has opened its maximum number of connections and all connections are busy with client sessions. Value must be between 1 second and 1 hour, or ``Duration.seconds(0)`` to represent unlimited. Default: cdk.Duration.seconds(120)
        :param db_proxy_name: The identifier for the proxy. This name must be unique for all proxies owned by your AWS account in the specified AWS Region. An identifier must begin with a letter and must contain only ASCII letters, digits, and hyphens; it can't end with a hyphen or contain two consecutive hyphens. Default: - Generated by CloudFormation (recommended)
        :param debug_logging: Whether the proxy includes detailed information about SQL statements in its logs. This information helps you to debug issues involving SQL behavior or the performance and scalability of the proxy connections. The debug information includes the text of SQL statements that you submit through the proxy. Thus, only enable this setting when needed for debugging, and only when you have security measures in place to safeguard any sensitive information that appears in the logs. Default: false
        :param iam_auth: Whether to require or disallow AWS Identity and Access Management (IAM) authentication for connections to the proxy. Default: false
        :param idle_client_timeout: The number of seconds that a connection to the proxy can be inactive before the proxy disconnects it. You can set this value higher or lower than the connection timeout limit for the associated database. Default: cdk.Duration.minutes(30)
        :param init_query: One or more SQL statements for the proxy to run when opening each new database connection. Typically used with SET statements to make sure that each connection has identical settings such as time zone and character set. For multiple statements, use semicolons as the separator. You can also include multiple variables in a single SET statement, such as SET x=1, y=2. not currently supported for PostgreSQL. Default: - no initialization query
        :param max_connections_percent: The maximum size of the connection pool for each target in a target group. For Aurora MySQL, it is expressed as a percentage of the max_connections setting for the RDS DB instance or Aurora DB cluster used by the target group. 1-100 Default: 100
        :param max_idle_connections_percent: Controls how actively the proxy closes idle database connections in the connection pool. A high value enables the proxy to leave a high percentage of idle connections open. A low value causes the proxy to close idle client connections and return the underlying database connections to the connection pool. For Aurora MySQL, it is expressed as a percentage of the max_connections setting for the RDS DB instance or Aurora DB cluster used by the target group. between 0 and MaxConnectionsPercent Default: 50
        :param require_tls: A Boolean parameter that specifies whether Transport Layer Security (TLS) encryption is required for connections to the proxy. By enabling this setting, you can enforce encrypted TLS connections to the proxy. Default: true
        :param role: IAM role that the proxy uses to access secrets in AWS Secrets Manager. Default: - A role will automatically be created
        :param security_groups: One or more VPC security groups to associate with the new proxy. Default: - No security groups
        :param session_pinning_filters: Each item in the list represents a class of SQL operations that normally cause all later statements in a session using a proxy to be pinned to the same underlying database connection. Including an item in the list exempts that class of SQL operations from the pinning behavior. Default: - no session pinning filters
        :param vpc_subnets: The subnets used by the proxy. Default: - the VPC default strategy if not specified.
        """
        ...

    @jsii.member(jsii_name="metric")
    def metric(
        self,
        metric_name: builtins.str,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """Return the given named metric for this DBCluster.

        :param metric_name: -
        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        ...

    @jsii.member(jsii_name="metricCPUUtilization")
    def metric_cpu_utilization(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The percentage of CPU utilization.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        ...

    @jsii.member(jsii_name="metricDatabaseConnections")
    def metric_database_connections(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The number of database connections in use.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        ...

    @jsii.member(jsii_name="metricDeadlocks")
    def metric_deadlocks(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The average number of deadlocks in the database per second.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        ...

    @jsii.member(jsii_name="metricEngineUptime")
    def metric_engine_uptime(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The amount of time that the instance has been running, in seconds.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        ...

    @jsii.member(jsii_name="metricFreeableMemory")
    def metric_freeable_memory(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The amount of available random access memory, in bytes.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        ...

    @jsii.member(jsii_name="metricFreeLocalStorage")
    def metric_free_local_storage(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The amount of local storage available, in bytes.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        ...

    @jsii.member(jsii_name="metricNetworkReceiveThroughput")
    def metric_network_receive_throughput(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The amount of network throughput received from clients by each instance, in bytes per second.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        ...

    @jsii.member(jsii_name="metricNetworkThroughput")
    def metric_network_throughput(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The amount of network throughput both received from and transmitted to clients by each instance, in bytes per second.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        ...

    @jsii.member(jsii_name="metricNetworkTransmitThroughput")
    def metric_network_transmit_throughput(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The amount of network throughput sent to clients by each instance, in bytes per second.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        ...

    @jsii.member(jsii_name="metricSnapshotStorageUsed")
    def metric_snapshot_storage_used(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The total amount of backup storage in bytes consumed by all Aurora snapshots outside its backup retention window.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        ...

    @jsii.member(jsii_name="metricTotalBackupStorageBilled")
    def metric_total_backup_storage_billed(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The total amount of backup storage in bytes for which you are billed.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        ...

    @jsii.member(jsii_name="metricVolumeBytesUsed")
    def metric_volume_bytes_used(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The amount of storage used by your Aurora DB instance, in bytes.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        ...

    @jsii.member(jsii_name="metricVolumeReadIOPs")
    def metric_volume_read_io_ps(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The number of billed read I/O operations from a cluster volume, reported at 5-minute intervals.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        ...

    @jsii.member(jsii_name="metricVolumeWriteIOPs")
    def metric_volume_write_io_ps(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The number of write disk I/O operations to the cluster volume, reported at 5-minute intervals.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        ...


class _IDatabaseClusterProxy(
    jsii.proxy_for(aws_cdk.core.IResource), # type: ignore
    jsii.proxy_for(aws_cdk.aws_ec2.IConnectable), # type: ignore
    jsii.proxy_for(aws_cdk.aws_secretsmanager.ISecretAttachmentTarget), # type: ignore
):
    """Create a clustered database with a given number of instances."""

    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-rds.IDatabaseCluster"

    @builtins.property # type: ignore
    @jsii.member(jsii_name="clusterEndpoint")
    def cluster_endpoint(self) -> Endpoint:
        """The endpoint to use for read/write operations.

        :attribute: EndpointAddress,EndpointPort
        """
        return jsii.get(self, "clusterEndpoint")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="clusterIdentifier")
    def cluster_identifier(self) -> builtins.str:
        """Identifier of the cluster."""
        return jsii.get(self, "clusterIdentifier")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="clusterReadEndpoint")
    def cluster_read_endpoint(self) -> Endpoint:
        """Endpoint to use for load-balanced read-only operations.

        :attribute: ReadEndpointAddress
        """
        return jsii.get(self, "clusterReadEndpoint")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="instanceEndpoints")
    def instance_endpoints(self) -> typing.List[Endpoint]:
        """Endpoints which address each individual replica."""
        return jsii.get(self, "instanceEndpoints")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="instanceIdentifiers")
    def instance_identifiers(self) -> typing.List[builtins.str]:
        """Identifiers of the replicas."""
        return jsii.get(self, "instanceIdentifiers")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="engine")
    def engine(self) -> typing.Optional["IClusterEngine"]:
        """The engine of this Cluster.

        May be not known for imported Clusters if it wasn't provided explicitly.
        """
        return jsii.get(self, "engine")

    @jsii.member(jsii_name="addProxy")
    def add_proxy(
        self,
        id: builtins.str,
        *,
        secrets: typing.List[aws_cdk.aws_secretsmanager.ISecret],
        vpc: aws_cdk.aws_ec2.IVpc,
        borrow_timeout: typing.Optional[aws_cdk.core.Duration] = None,
        db_proxy_name: typing.Optional[builtins.str] = None,
        debug_logging: typing.Optional[builtins.bool] = None,
        iam_auth: typing.Optional[builtins.bool] = None,
        idle_client_timeout: typing.Optional[aws_cdk.core.Duration] = None,
        init_query: typing.Optional[builtins.str] = None,
        max_connections_percent: typing.Optional[jsii.Number] = None,
        max_idle_connections_percent: typing.Optional[jsii.Number] = None,
        require_tls: typing.Optional[builtins.bool] = None,
        role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        security_groups: typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]] = None,
        session_pinning_filters: typing.Optional[typing.List["SessionPinningFilter"]] = None,
        vpc_subnets: typing.Optional[aws_cdk.aws_ec2.SubnetSelection] = None,
    ) -> "DatabaseProxy":
        """Add a new db proxy to this cluster.

        :param id: -
        :param secrets: The secret that the proxy uses to authenticate to the RDS DB instance or Aurora DB cluster. These secrets are stored within Amazon Secrets Manager. One or more secrets are required.
        :param vpc: The VPC to associate with the new proxy.
        :param borrow_timeout: The duration for a proxy to wait for a connection to become available in the connection pool. Only applies when the proxy has opened its maximum number of connections and all connections are busy with client sessions. Value must be between 1 second and 1 hour, or ``Duration.seconds(0)`` to represent unlimited. Default: cdk.Duration.seconds(120)
        :param db_proxy_name: The identifier for the proxy. This name must be unique for all proxies owned by your AWS account in the specified AWS Region. An identifier must begin with a letter and must contain only ASCII letters, digits, and hyphens; it can't end with a hyphen or contain two consecutive hyphens. Default: - Generated by CloudFormation (recommended)
        :param debug_logging: Whether the proxy includes detailed information about SQL statements in its logs. This information helps you to debug issues involving SQL behavior or the performance and scalability of the proxy connections. The debug information includes the text of SQL statements that you submit through the proxy. Thus, only enable this setting when needed for debugging, and only when you have security measures in place to safeguard any sensitive information that appears in the logs. Default: false
        :param iam_auth: Whether to require or disallow AWS Identity and Access Management (IAM) authentication for connections to the proxy. Default: false
        :param idle_client_timeout: The number of seconds that a connection to the proxy can be inactive before the proxy disconnects it. You can set this value higher or lower than the connection timeout limit for the associated database. Default: cdk.Duration.minutes(30)
        :param init_query: One or more SQL statements for the proxy to run when opening each new database connection. Typically used with SET statements to make sure that each connection has identical settings such as time zone and character set. For multiple statements, use semicolons as the separator. You can also include multiple variables in a single SET statement, such as SET x=1, y=2. not currently supported for PostgreSQL. Default: - no initialization query
        :param max_connections_percent: The maximum size of the connection pool for each target in a target group. For Aurora MySQL, it is expressed as a percentage of the max_connections setting for the RDS DB instance or Aurora DB cluster used by the target group. 1-100 Default: 100
        :param max_idle_connections_percent: Controls how actively the proxy closes idle database connections in the connection pool. A high value enables the proxy to leave a high percentage of idle connections open. A low value causes the proxy to close idle client connections and return the underlying database connections to the connection pool. For Aurora MySQL, it is expressed as a percentage of the max_connections setting for the RDS DB instance or Aurora DB cluster used by the target group. between 0 and MaxConnectionsPercent Default: 50
        :param require_tls: A Boolean parameter that specifies whether Transport Layer Security (TLS) encryption is required for connections to the proxy. By enabling this setting, you can enforce encrypted TLS connections to the proxy. Default: true
        :param role: IAM role that the proxy uses to access secrets in AWS Secrets Manager. Default: - A role will automatically be created
        :param security_groups: One or more VPC security groups to associate with the new proxy. Default: - No security groups
        :param session_pinning_filters: Each item in the list represents a class of SQL operations that normally cause all later statements in a session using a proxy to be pinned to the same underlying database connection. Including an item in the list exempts that class of SQL operations from the pinning behavior. Default: - no session pinning filters
        :param vpc_subnets: The subnets used by the proxy. Default: - the VPC default strategy if not specified.
        """
        options = DatabaseProxyOptions(
            secrets=secrets,
            vpc=vpc,
            borrow_timeout=borrow_timeout,
            db_proxy_name=db_proxy_name,
            debug_logging=debug_logging,
            iam_auth=iam_auth,
            idle_client_timeout=idle_client_timeout,
            init_query=init_query,
            max_connections_percent=max_connections_percent,
            max_idle_connections_percent=max_idle_connections_percent,
            require_tls=require_tls,
            role=role,
            security_groups=security_groups,
            session_pinning_filters=session_pinning_filters,
            vpc_subnets=vpc_subnets,
        )

        return jsii.invoke(self, "addProxy", [id, options])

    @jsii.member(jsii_name="metric")
    def metric(
        self,
        metric_name: builtins.str,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """Return the given named metric for this DBCluster.

        :param metric_name: -
        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metric", [metric_name, props])

    @jsii.member(jsii_name="metricCPUUtilization")
    def metric_cpu_utilization(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The percentage of CPU utilization.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricCPUUtilization", [props])

    @jsii.member(jsii_name="metricDatabaseConnections")
    def metric_database_connections(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The number of database connections in use.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricDatabaseConnections", [props])

    @jsii.member(jsii_name="metricDeadlocks")
    def metric_deadlocks(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The average number of deadlocks in the database per second.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricDeadlocks", [props])

    @jsii.member(jsii_name="metricEngineUptime")
    def metric_engine_uptime(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The amount of time that the instance has been running, in seconds.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricEngineUptime", [props])

    @jsii.member(jsii_name="metricFreeableMemory")
    def metric_freeable_memory(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The amount of available random access memory, in bytes.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricFreeableMemory", [props])

    @jsii.member(jsii_name="metricFreeLocalStorage")
    def metric_free_local_storage(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The amount of local storage available, in bytes.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricFreeLocalStorage", [props])

    @jsii.member(jsii_name="metricNetworkReceiveThroughput")
    def metric_network_receive_throughput(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The amount of network throughput received from clients by each instance, in bytes per second.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricNetworkReceiveThroughput", [props])

    @jsii.member(jsii_name="metricNetworkThroughput")
    def metric_network_throughput(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The amount of network throughput both received from and transmitted to clients by each instance, in bytes per second.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricNetworkThroughput", [props])

    @jsii.member(jsii_name="metricNetworkTransmitThroughput")
    def metric_network_transmit_throughput(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The amount of network throughput sent to clients by each instance, in bytes per second.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricNetworkTransmitThroughput", [props])

    @jsii.member(jsii_name="metricSnapshotStorageUsed")
    def metric_snapshot_storage_used(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The total amount of backup storage in bytes consumed by all Aurora snapshots outside its backup retention window.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricSnapshotStorageUsed", [props])

    @jsii.member(jsii_name="metricTotalBackupStorageBilled")
    def metric_total_backup_storage_billed(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The total amount of backup storage in bytes for which you are billed.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricTotalBackupStorageBilled", [props])

    @jsii.member(jsii_name="metricVolumeBytesUsed")
    def metric_volume_bytes_used(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The amount of storage used by your Aurora DB instance, in bytes.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricVolumeBytesUsed", [props])

    @jsii.member(jsii_name="metricVolumeReadIOPs")
    def metric_volume_read_io_ps(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The number of billed read I/O operations from a cluster volume, reported at 5-minute intervals.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricVolumeReadIOPs", [props])

    @jsii.member(jsii_name="metricVolumeWriteIOPs")
    def metric_volume_write_io_ps(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The number of write disk I/O operations to the cluster volume, reported at 5-minute intervals.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricVolumeWriteIOPs", [props])


@jsii.interface(jsii_type="@aws-cdk/aws-rds.IDatabaseInstance")
class IDatabaseInstance(
    aws_cdk.core.IResource,
    aws_cdk.aws_ec2.IConnectable,
    aws_cdk.aws_secretsmanager.ISecretAttachmentTarget,
    typing_extensions.Protocol,
):
    """A database instance."""

    @builtins.staticmethod
    def __jsii_proxy_class__():
        return _IDatabaseInstanceProxy

    @builtins.property # type: ignore
    @jsii.member(jsii_name="dbInstanceEndpointAddress")
    def db_instance_endpoint_address(self) -> builtins.str:
        """The instance endpoint address.

        :attribute: EndpointAddress
        """
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="dbInstanceEndpointPort")
    def db_instance_endpoint_port(self) -> builtins.str:
        """The instance endpoint port.

        :attribute: EndpointPort
        """
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="instanceArn")
    def instance_arn(self) -> builtins.str:
        """The instance arn."""
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="instanceEndpoint")
    def instance_endpoint(self) -> Endpoint:
        """The instance endpoint."""
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="instanceIdentifier")
    def instance_identifier(self) -> builtins.str:
        """The instance identifier."""
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="engine")
    def engine(self) -> typing.Optional["IInstanceEngine"]:
        """The engine of this database Instance.

        May be not known for imported Instances if it wasn't provided explicitly,
        or for read replicas.
        """
        ...

    @jsii.member(jsii_name="addProxy")
    def add_proxy(
        self,
        id: builtins.str,
        *,
        secrets: typing.List[aws_cdk.aws_secretsmanager.ISecret],
        vpc: aws_cdk.aws_ec2.IVpc,
        borrow_timeout: typing.Optional[aws_cdk.core.Duration] = None,
        db_proxy_name: typing.Optional[builtins.str] = None,
        debug_logging: typing.Optional[builtins.bool] = None,
        iam_auth: typing.Optional[builtins.bool] = None,
        idle_client_timeout: typing.Optional[aws_cdk.core.Duration] = None,
        init_query: typing.Optional[builtins.str] = None,
        max_connections_percent: typing.Optional[jsii.Number] = None,
        max_idle_connections_percent: typing.Optional[jsii.Number] = None,
        require_tls: typing.Optional[builtins.bool] = None,
        role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        security_groups: typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]] = None,
        session_pinning_filters: typing.Optional[typing.List["SessionPinningFilter"]] = None,
        vpc_subnets: typing.Optional[aws_cdk.aws_ec2.SubnetSelection] = None,
    ) -> "DatabaseProxy":
        """Add a new db proxy to this instance.

        :param id: -
        :param secrets: The secret that the proxy uses to authenticate to the RDS DB instance or Aurora DB cluster. These secrets are stored within Amazon Secrets Manager. One or more secrets are required.
        :param vpc: The VPC to associate with the new proxy.
        :param borrow_timeout: The duration for a proxy to wait for a connection to become available in the connection pool. Only applies when the proxy has opened its maximum number of connections and all connections are busy with client sessions. Value must be between 1 second and 1 hour, or ``Duration.seconds(0)`` to represent unlimited. Default: cdk.Duration.seconds(120)
        :param db_proxy_name: The identifier for the proxy. This name must be unique for all proxies owned by your AWS account in the specified AWS Region. An identifier must begin with a letter and must contain only ASCII letters, digits, and hyphens; it can't end with a hyphen or contain two consecutive hyphens. Default: - Generated by CloudFormation (recommended)
        :param debug_logging: Whether the proxy includes detailed information about SQL statements in its logs. This information helps you to debug issues involving SQL behavior or the performance and scalability of the proxy connections. The debug information includes the text of SQL statements that you submit through the proxy. Thus, only enable this setting when needed for debugging, and only when you have security measures in place to safeguard any sensitive information that appears in the logs. Default: false
        :param iam_auth: Whether to require or disallow AWS Identity and Access Management (IAM) authentication for connections to the proxy. Default: false
        :param idle_client_timeout: The number of seconds that a connection to the proxy can be inactive before the proxy disconnects it. You can set this value higher or lower than the connection timeout limit for the associated database. Default: cdk.Duration.minutes(30)
        :param init_query: One or more SQL statements for the proxy to run when opening each new database connection. Typically used with SET statements to make sure that each connection has identical settings such as time zone and character set. For multiple statements, use semicolons as the separator. You can also include multiple variables in a single SET statement, such as SET x=1, y=2. not currently supported for PostgreSQL. Default: - no initialization query
        :param max_connections_percent: The maximum size of the connection pool for each target in a target group. For Aurora MySQL, it is expressed as a percentage of the max_connections setting for the RDS DB instance or Aurora DB cluster used by the target group. 1-100 Default: 100
        :param max_idle_connections_percent: Controls how actively the proxy closes idle database connections in the connection pool. A high value enables the proxy to leave a high percentage of idle connections open. A low value causes the proxy to close idle client connections and return the underlying database connections to the connection pool. For Aurora MySQL, it is expressed as a percentage of the max_connections setting for the RDS DB instance or Aurora DB cluster used by the target group. between 0 and MaxConnectionsPercent Default: 50
        :param require_tls: A Boolean parameter that specifies whether Transport Layer Security (TLS) encryption is required for connections to the proxy. By enabling this setting, you can enforce encrypted TLS connections to the proxy. Default: true
        :param role: IAM role that the proxy uses to access secrets in AWS Secrets Manager. Default: - A role will automatically be created
        :param security_groups: One or more VPC security groups to associate with the new proxy. Default: - No security groups
        :param session_pinning_filters: Each item in the list represents a class of SQL operations that normally cause all later statements in a session using a proxy to be pinned to the same underlying database connection. Including an item in the list exempts that class of SQL operations from the pinning behavior. Default: - no session pinning filters
        :param vpc_subnets: The subnets used by the proxy. Default: - the VPC default strategy if not specified.
        """
        ...

    @jsii.member(jsii_name="grantConnect")
    def grant_connect(
        self,
        grantee: aws_cdk.aws_iam.IGrantable,
    ) -> aws_cdk.aws_iam.Grant:
        """Grant the given identity connection access to the database.

        :param grantee: -
        """
        ...

    @jsii.member(jsii_name="metric")
    def metric(
        self,
        metric_name: builtins.str,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """Return the given named metric for this DBInstance.

        :param metric_name: -
        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        ...

    @jsii.member(jsii_name="metricCPUUtilization")
    def metric_cpu_utilization(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The percentage of CPU utilization.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        ...

    @jsii.member(jsii_name="metricDatabaseConnections")
    def metric_database_connections(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The number of database connections in use.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        ...

    @jsii.member(jsii_name="metricFreeableMemory")
    def metric_freeable_memory(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The amount of available random access memory.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        ...

    @jsii.member(jsii_name="metricFreeStorageSpace")
    def metric_free_storage_space(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The amount of available storage space.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        ...

    @jsii.member(jsii_name="metricReadIOPS")
    def metric_read_iops(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The average number of disk write I/O operations per second.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        ...

    @jsii.member(jsii_name="metricWriteIOPS")
    def metric_write_iops(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The average number of disk read I/O operations per second.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        ...

    @jsii.member(jsii_name="onEvent")
    def on_event(
        self,
        id: builtins.str,
        *,
        description: typing.Optional[builtins.str] = None,
        event_pattern: typing.Optional[aws_cdk.aws_events.EventPattern] = None,
        rule_name: typing.Optional[builtins.str] = None,
        target: typing.Optional[aws_cdk.aws_events.IRuleTarget] = None,
    ) -> aws_cdk.aws_events.Rule:
        """Defines a CloudWatch event rule which triggers for instance events.

        Use
        ``rule.addEventPattern(pattern)`` to specify a filter.

        :param id: -
        :param description: A description of the rule's purpose. Default: - No description
        :param event_pattern: Additional restrictions for the event to route to the specified target. The method that generates the rule probably imposes some type of event filtering. The filtering implied by what you pass here is added on top of that filtering. Default: - No additional filtering based on an event pattern.
        :param rule_name: A name for the rule. Default: AWS CloudFormation generates a unique physical ID.
        :param target: The target to register for the event. Default: - No target is added to the rule. Use ``addTarget()`` to add a target.
        """
        ...


class _IDatabaseInstanceProxy(
    jsii.proxy_for(aws_cdk.core.IResource), # type: ignore
    jsii.proxy_for(aws_cdk.aws_ec2.IConnectable), # type: ignore
    jsii.proxy_for(aws_cdk.aws_secretsmanager.ISecretAttachmentTarget), # type: ignore
):
    """A database instance."""

    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-rds.IDatabaseInstance"

    @builtins.property # type: ignore
    @jsii.member(jsii_name="dbInstanceEndpointAddress")
    def db_instance_endpoint_address(self) -> builtins.str:
        """The instance endpoint address.

        :attribute: EndpointAddress
        """
        return jsii.get(self, "dbInstanceEndpointAddress")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="dbInstanceEndpointPort")
    def db_instance_endpoint_port(self) -> builtins.str:
        """The instance endpoint port.

        :attribute: EndpointPort
        """
        return jsii.get(self, "dbInstanceEndpointPort")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="instanceArn")
    def instance_arn(self) -> builtins.str:
        """The instance arn."""
        return jsii.get(self, "instanceArn")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="instanceEndpoint")
    def instance_endpoint(self) -> Endpoint:
        """The instance endpoint."""
        return jsii.get(self, "instanceEndpoint")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="instanceIdentifier")
    def instance_identifier(self) -> builtins.str:
        """The instance identifier."""
        return jsii.get(self, "instanceIdentifier")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="engine")
    def engine(self) -> typing.Optional["IInstanceEngine"]:
        """The engine of this database Instance.

        May be not known for imported Instances if it wasn't provided explicitly,
        or for read replicas.
        """
        return jsii.get(self, "engine")

    @jsii.member(jsii_name="addProxy")
    def add_proxy(
        self,
        id: builtins.str,
        *,
        secrets: typing.List[aws_cdk.aws_secretsmanager.ISecret],
        vpc: aws_cdk.aws_ec2.IVpc,
        borrow_timeout: typing.Optional[aws_cdk.core.Duration] = None,
        db_proxy_name: typing.Optional[builtins.str] = None,
        debug_logging: typing.Optional[builtins.bool] = None,
        iam_auth: typing.Optional[builtins.bool] = None,
        idle_client_timeout: typing.Optional[aws_cdk.core.Duration] = None,
        init_query: typing.Optional[builtins.str] = None,
        max_connections_percent: typing.Optional[jsii.Number] = None,
        max_idle_connections_percent: typing.Optional[jsii.Number] = None,
        require_tls: typing.Optional[builtins.bool] = None,
        role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        security_groups: typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]] = None,
        session_pinning_filters: typing.Optional[typing.List["SessionPinningFilter"]] = None,
        vpc_subnets: typing.Optional[aws_cdk.aws_ec2.SubnetSelection] = None,
    ) -> "DatabaseProxy":
        """Add a new db proxy to this instance.

        :param id: -
        :param secrets: The secret that the proxy uses to authenticate to the RDS DB instance or Aurora DB cluster. These secrets are stored within Amazon Secrets Manager. One or more secrets are required.
        :param vpc: The VPC to associate with the new proxy.
        :param borrow_timeout: The duration for a proxy to wait for a connection to become available in the connection pool. Only applies when the proxy has opened its maximum number of connections and all connections are busy with client sessions. Value must be between 1 second and 1 hour, or ``Duration.seconds(0)`` to represent unlimited. Default: cdk.Duration.seconds(120)
        :param db_proxy_name: The identifier for the proxy. This name must be unique for all proxies owned by your AWS account in the specified AWS Region. An identifier must begin with a letter and must contain only ASCII letters, digits, and hyphens; it can't end with a hyphen or contain two consecutive hyphens. Default: - Generated by CloudFormation (recommended)
        :param debug_logging: Whether the proxy includes detailed information about SQL statements in its logs. This information helps you to debug issues involving SQL behavior or the performance and scalability of the proxy connections. The debug information includes the text of SQL statements that you submit through the proxy. Thus, only enable this setting when needed for debugging, and only when you have security measures in place to safeguard any sensitive information that appears in the logs. Default: false
        :param iam_auth: Whether to require or disallow AWS Identity and Access Management (IAM) authentication for connections to the proxy. Default: false
        :param idle_client_timeout: The number of seconds that a connection to the proxy can be inactive before the proxy disconnects it. You can set this value higher or lower than the connection timeout limit for the associated database. Default: cdk.Duration.minutes(30)
        :param init_query: One or more SQL statements for the proxy to run when opening each new database connection. Typically used with SET statements to make sure that each connection has identical settings such as time zone and character set. For multiple statements, use semicolons as the separator. You can also include multiple variables in a single SET statement, such as SET x=1, y=2. not currently supported for PostgreSQL. Default: - no initialization query
        :param max_connections_percent: The maximum size of the connection pool for each target in a target group. For Aurora MySQL, it is expressed as a percentage of the max_connections setting for the RDS DB instance or Aurora DB cluster used by the target group. 1-100 Default: 100
        :param max_idle_connections_percent: Controls how actively the proxy closes idle database connections in the connection pool. A high value enables the proxy to leave a high percentage of idle connections open. A low value causes the proxy to close idle client connections and return the underlying database connections to the connection pool. For Aurora MySQL, it is expressed as a percentage of the max_connections setting for the RDS DB instance or Aurora DB cluster used by the target group. between 0 and MaxConnectionsPercent Default: 50
        :param require_tls: A Boolean parameter that specifies whether Transport Layer Security (TLS) encryption is required for connections to the proxy. By enabling this setting, you can enforce encrypted TLS connections to the proxy. Default: true
        :param role: IAM role that the proxy uses to access secrets in AWS Secrets Manager. Default: - A role will automatically be created
        :param security_groups: One or more VPC security groups to associate with the new proxy. Default: - No security groups
        :param session_pinning_filters: Each item in the list represents a class of SQL operations that normally cause all later statements in a session using a proxy to be pinned to the same underlying database connection. Including an item in the list exempts that class of SQL operations from the pinning behavior. Default: - no session pinning filters
        :param vpc_subnets: The subnets used by the proxy. Default: - the VPC default strategy if not specified.
        """
        options = DatabaseProxyOptions(
            secrets=secrets,
            vpc=vpc,
            borrow_timeout=borrow_timeout,
            db_proxy_name=db_proxy_name,
            debug_logging=debug_logging,
            iam_auth=iam_auth,
            idle_client_timeout=idle_client_timeout,
            init_query=init_query,
            max_connections_percent=max_connections_percent,
            max_idle_connections_percent=max_idle_connections_percent,
            require_tls=require_tls,
            role=role,
            security_groups=security_groups,
            session_pinning_filters=session_pinning_filters,
            vpc_subnets=vpc_subnets,
        )

        return jsii.invoke(self, "addProxy", [id, options])

    @jsii.member(jsii_name="grantConnect")
    def grant_connect(
        self,
        grantee: aws_cdk.aws_iam.IGrantable,
    ) -> aws_cdk.aws_iam.Grant:
        """Grant the given identity connection access to the database.

        :param grantee: -
        """
        return jsii.invoke(self, "grantConnect", [grantee])

    @jsii.member(jsii_name="metric")
    def metric(
        self,
        metric_name: builtins.str,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """Return the given named metric for this DBInstance.

        :param metric_name: -
        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metric", [metric_name, props])

    @jsii.member(jsii_name="metricCPUUtilization")
    def metric_cpu_utilization(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The percentage of CPU utilization.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricCPUUtilization", [props])

    @jsii.member(jsii_name="metricDatabaseConnections")
    def metric_database_connections(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The number of database connections in use.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricDatabaseConnections", [props])

    @jsii.member(jsii_name="metricFreeableMemory")
    def metric_freeable_memory(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The amount of available random access memory.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricFreeableMemory", [props])

    @jsii.member(jsii_name="metricFreeStorageSpace")
    def metric_free_storage_space(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The amount of available storage space.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricFreeStorageSpace", [props])

    @jsii.member(jsii_name="metricReadIOPS")
    def metric_read_iops(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The average number of disk write I/O operations per second.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricReadIOPS", [props])

    @jsii.member(jsii_name="metricWriteIOPS")
    def metric_write_iops(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The average number of disk read I/O operations per second.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricWriteIOPS", [props])

    @jsii.member(jsii_name="onEvent")
    def on_event(
        self,
        id: builtins.str,
        *,
        description: typing.Optional[builtins.str] = None,
        event_pattern: typing.Optional[aws_cdk.aws_events.EventPattern] = None,
        rule_name: typing.Optional[builtins.str] = None,
        target: typing.Optional[aws_cdk.aws_events.IRuleTarget] = None,
    ) -> aws_cdk.aws_events.Rule:
        """Defines a CloudWatch event rule which triggers for instance events.

        Use
        ``rule.addEventPattern(pattern)`` to specify a filter.

        :param id: -
        :param description: A description of the rule's purpose. Default: - No description
        :param event_pattern: Additional restrictions for the event to route to the specified target. The method that generates the rule probably imposes some type of event filtering. The filtering implied by what you pass here is added on top of that filtering. Default: - No additional filtering based on an event pattern.
        :param rule_name: A name for the rule. Default: AWS CloudFormation generates a unique physical ID.
        :param target: The target to register for the event. Default: - No target is added to the rule. Use ``addTarget()`` to add a target.
        """
        options = aws_cdk.aws_events.OnEventOptions(
            description=description,
            event_pattern=event_pattern,
            rule_name=rule_name,
            target=target,
        )

        return jsii.invoke(self, "onEvent", [id, options])


@jsii.interface(jsii_type="@aws-cdk/aws-rds.IDatabaseProxy")
class IDatabaseProxy(aws_cdk.core.IResource, typing_extensions.Protocol):
    """DB Proxy."""

    @builtins.staticmethod
    def __jsii_proxy_class__():
        return _IDatabaseProxyProxy

    @builtins.property # type: ignore
    @jsii.member(jsii_name="dbProxyArn")
    def db_proxy_arn(self) -> builtins.str:
        """DB Proxy ARN.

        :attribute: true
        """
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="dbProxyName")
    def db_proxy_name(self) -> builtins.str:
        """DB Proxy Name.

        :attribute: true
        """
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="endpoint")
    def endpoint(self) -> builtins.str:
        """Endpoint.

        :attribute: true
        """
        ...

    @jsii.member(jsii_name="grantConnect")
    def grant_connect(
        self,
        grantee: aws_cdk.aws_iam.IGrantable,
    ) -> aws_cdk.aws_iam.Grant:
        """Grant the given identity connection access to the proxy.

        :param grantee: -
        """
        ...


class _IDatabaseProxyProxy(
    jsii.proxy_for(aws_cdk.core.IResource) # type: ignore
):
    """DB Proxy."""

    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-rds.IDatabaseProxy"

    @builtins.property # type: ignore
    @jsii.member(jsii_name="dbProxyArn")
    def db_proxy_arn(self) -> builtins.str:
        """DB Proxy ARN.

        :attribute: true
        """
        return jsii.get(self, "dbProxyArn")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="dbProxyName")
    def db_proxy_name(self) -> builtins.str:
        """DB Proxy Name.

        :attribute: true
        """
        return jsii.get(self, "dbProxyName")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="endpoint")
    def endpoint(self) -> builtins.str:
        """Endpoint.

        :attribute: true
        """
        return jsii.get(self, "endpoint")

    @jsii.member(jsii_name="grantConnect")
    def grant_connect(
        self,
        grantee: aws_cdk.aws_iam.IGrantable,
    ) -> aws_cdk.aws_iam.Grant:
        """Grant the given identity connection access to the proxy.

        :param grantee: -
        """
        return jsii.invoke(self, "grantConnect", [grantee])


@jsii.interface(jsii_type="@aws-cdk/aws-rds.IEngine")
class IEngine(typing_extensions.Protocol):
    """A common interface for database engines.

    Don't implement this interface directly,
    instead implement one of the known sub-interfaces,
    like IClusterEngine and IInstanceEngine.
    """

    @builtins.staticmethod
    def __jsii_proxy_class__():
        return _IEngineProxy

    @builtins.property # type: ignore
    @jsii.member(jsii_name="engineType")
    def engine_type(self) -> builtins.str:
        """The type of the engine, for example "mysql"."""
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="defaultUsername")
    def default_username(self) -> typing.Optional[builtins.str]:
        """The default name of the master database user if one was not provided explicitly.

        The global default of 'admin' will be used if this is ``undefined``.
        Note that 'admin' is a reserved word in PostgreSQL and cannot be used.
        """
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="engineFamily")
    def engine_family(self) -> typing.Optional[builtins.str]:
        """The family this engine belongs to, like "MYSQL", or "POSTGRESQL".

        This property is used when creating a Database Proxy.
        Most engines don't belong to any family
        (and because of that, you can't create Database Proxies for their Clusters or Instances).

        :default: - the engine doesn't belong to any family
        """
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="engineVersion")
    def engine_version(self) -> typing.Optional[EngineVersion]:
        """The exact version of the engine that is used, for example "5.1.42".

        :default: - use the default version for this engine type
        """
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="parameterGroupFamily")
    def parameter_group_family(self) -> typing.Optional[builtins.str]:
        """The family to use for ParameterGroups using this engine.

        This is usually equal to "",
        but can sometimes be a variation of that.
        You can pass this property when creating new ParameterGroup.

        :default:

        - the ParameterGroup family is not known
        (which means the major version of the engine is also not known)
        """
        ...


class _IEngineProxy:
    """A common interface for database engines.

    Don't implement this interface directly,
    instead implement one of the known sub-interfaces,
    like IClusterEngine and IInstanceEngine.
    """

    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-rds.IEngine"

    @builtins.property # type: ignore
    @jsii.member(jsii_name="engineType")
    def engine_type(self) -> builtins.str:
        """The type of the engine, for example "mysql"."""
        return jsii.get(self, "engineType")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="defaultUsername")
    def default_username(self) -> typing.Optional[builtins.str]:
        """The default name of the master database user if one was not provided explicitly.

        The global default of 'admin' will be used if this is ``undefined``.
        Note that 'admin' is a reserved word in PostgreSQL and cannot be used.
        """
        return jsii.get(self, "defaultUsername")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="engineFamily")
    def engine_family(self) -> typing.Optional[builtins.str]:
        """The family this engine belongs to, like "MYSQL", or "POSTGRESQL".

        This property is used when creating a Database Proxy.
        Most engines don't belong to any family
        (and because of that, you can't create Database Proxies for their Clusters or Instances).

        :default: - the engine doesn't belong to any family
        """
        return jsii.get(self, "engineFamily")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="engineVersion")
    def engine_version(self) -> typing.Optional[EngineVersion]:
        """The exact version of the engine that is used, for example "5.1.42".

        :default: - use the default version for this engine type
        """
        return jsii.get(self, "engineVersion")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="parameterGroupFamily")
    def parameter_group_family(self) -> typing.Optional[builtins.str]:
        """The family to use for ParameterGroups using this engine.

        This is usually equal to "",
        but can sometimes be a variation of that.
        You can pass this property when creating new ParameterGroup.

        :default:

        - the ParameterGroup family is not known
        (which means the major version of the engine is also not known)
        """
        return jsii.get(self, "parameterGroupFamily")


@jsii.interface(jsii_type="@aws-cdk/aws-rds.IInstanceEngine")
class IInstanceEngine(IEngine, typing_extensions.Protocol):
    """Interface representing a database instance (as opposed to cluster) engine."""

    @builtins.staticmethod
    def __jsii_proxy_class__():
        return _IInstanceEngineProxy

    @builtins.property # type: ignore
    @jsii.member(jsii_name="multiUserRotationApplication")
    def multi_user_rotation_application(
        self,
    ) -> aws_cdk.aws_secretsmanager.SecretRotationApplication:
        """The application used by this engine to perform rotation for a multi-user scenario."""
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="singleUserRotationApplication")
    def single_user_rotation_application(
        self,
    ) -> aws_cdk.aws_secretsmanager.SecretRotationApplication:
        """The application used by this engine to perform rotation for a single-user scenario."""
        ...

    @jsii.member(jsii_name="bindToInstance")
    def bind_to_instance(
        self,
        scope: aws_cdk.core.Construct,
        *,
        domain: typing.Optional[builtins.str] = None,
        option_group: typing.Optional["IOptionGroup"] = None,
        s3_export_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        s3_import_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        timezone: typing.Optional[builtins.str] = None,
    ) -> "InstanceEngineConfig":
        """Method called when the engine is used to create a new instance.

        :param scope: -
        :param domain: The Active Directory directory ID to create the DB instance in. Default: - none (it's an optional field)
        :param option_group: The option group of the database. Default: - none
        :param s3_export_role: The role used for S3 exporting. Default: - none
        :param s3_import_role: The role used for S3 importing. Default: - none
        :param timezone: The timezone of the database, set by the customer. Default: - none (it's an optional field)
        """
        ...


class _IInstanceEngineProxy(
    jsii.proxy_for(IEngine) # type: ignore
):
    """Interface representing a database instance (as opposed to cluster) engine."""

    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-rds.IInstanceEngine"

    @builtins.property # type: ignore
    @jsii.member(jsii_name="multiUserRotationApplication")
    def multi_user_rotation_application(
        self,
    ) -> aws_cdk.aws_secretsmanager.SecretRotationApplication:
        """The application used by this engine to perform rotation for a multi-user scenario."""
        return jsii.get(self, "multiUserRotationApplication")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="singleUserRotationApplication")
    def single_user_rotation_application(
        self,
    ) -> aws_cdk.aws_secretsmanager.SecretRotationApplication:
        """The application used by this engine to perform rotation for a single-user scenario."""
        return jsii.get(self, "singleUserRotationApplication")

    @jsii.member(jsii_name="bindToInstance")
    def bind_to_instance(
        self,
        scope: aws_cdk.core.Construct,
        *,
        domain: typing.Optional[builtins.str] = None,
        option_group: typing.Optional["IOptionGroup"] = None,
        s3_export_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        s3_import_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        timezone: typing.Optional[builtins.str] = None,
    ) -> "InstanceEngineConfig":
        """Method called when the engine is used to create a new instance.

        :param scope: -
        :param domain: The Active Directory directory ID to create the DB instance in. Default: - none (it's an optional field)
        :param option_group: The option group of the database. Default: - none
        :param s3_export_role: The role used for S3 exporting. Default: - none
        :param s3_import_role: The role used for S3 importing. Default: - none
        :param timezone: The timezone of the database, set by the customer. Default: - none (it's an optional field)
        """
        options = InstanceEngineBindOptions(
            domain=domain,
            option_group=option_group,
            s3_export_role=s3_export_role,
            s3_import_role=s3_import_role,
            timezone=timezone,
        )

        return jsii.invoke(self, "bindToInstance", [scope, options])


@jsii.interface(jsii_type="@aws-cdk/aws-rds.IOptionGroup")
class IOptionGroup(aws_cdk.core.IResource, typing_extensions.Protocol):
    """An option group."""

    @builtins.staticmethod
    def __jsii_proxy_class__():
        return _IOptionGroupProxy

    @builtins.property # type: ignore
    @jsii.member(jsii_name="optionGroupName")
    def option_group_name(self) -> builtins.str:
        """The name of the option group.

        :attribute: true
        """
        ...

    @jsii.member(jsii_name="addConfiguration")
    def add_configuration(
        self,
        *,
        name: builtins.str,
        port: typing.Optional[jsii.Number] = None,
        security_groups: typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]] = None,
        settings: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        version: typing.Optional[builtins.str] = None,
        vpc: typing.Optional[aws_cdk.aws_ec2.IVpc] = None,
    ) -> builtins.bool:
        """Adds a configuration to this OptionGroup.

        This method is a no-op for an imported OptionGroup.

        :param name: The name of the option.
        :param port: The port number that this option uses. If ``port`` is specified then ``vpc`` must also be specified. Default: - no port
        :param security_groups: Optional list of security groups to use for this option, if ``vpc`` is specified. If no groups are provided, a default one will be created. Default: - a default group will be created if ``port`` or ``vpc`` are specified.
        :param settings: The settings for the option. Default: - no settings
        :param version: The version for the option. Default: - no version
        :param vpc: The VPC where a security group should be created for this option. If ``vpc`` is specified then ``port`` must also be specified. Default: - no VPC

        :return: true if the OptionConfiguration was successfully added.
        """
        ...


class _IOptionGroupProxy(
    jsii.proxy_for(aws_cdk.core.IResource) # type: ignore
):
    """An option group."""

    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-rds.IOptionGroup"

    @builtins.property # type: ignore
    @jsii.member(jsii_name="optionGroupName")
    def option_group_name(self) -> builtins.str:
        """The name of the option group.

        :attribute: true
        """
        return jsii.get(self, "optionGroupName")

    @jsii.member(jsii_name="addConfiguration")
    def add_configuration(
        self,
        *,
        name: builtins.str,
        port: typing.Optional[jsii.Number] = None,
        security_groups: typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]] = None,
        settings: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        version: typing.Optional[builtins.str] = None,
        vpc: typing.Optional[aws_cdk.aws_ec2.IVpc] = None,
    ) -> builtins.bool:
        """Adds a configuration to this OptionGroup.

        This method is a no-op for an imported OptionGroup.

        :param name: The name of the option.
        :param port: The port number that this option uses. If ``port`` is specified then ``vpc`` must also be specified. Default: - no port
        :param security_groups: Optional list of security groups to use for this option, if ``vpc`` is specified. If no groups are provided, a default one will be created. Default: - a default group will be created if ``port`` or ``vpc`` are specified.
        :param settings: The settings for the option. Default: - no settings
        :param version: The version for the option. Default: - no version
        :param vpc: The VPC where a security group should be created for this option. If ``vpc`` is specified then ``port`` must also be specified. Default: - no VPC

        :return: true if the OptionConfiguration was successfully added.
        """
        configuration = OptionConfiguration(
            name=name,
            port=port,
            security_groups=security_groups,
            settings=settings,
            version=version,
            vpc=vpc,
        )

        return jsii.invoke(self, "addConfiguration", [configuration])


@jsii.interface(jsii_type="@aws-cdk/aws-rds.IParameterGroup")
class IParameterGroup(aws_cdk.core.IResource, typing_extensions.Protocol):
    """A parameter group.

    Represents both a cluster parameter group,
    and an instance parameter group.
    """

    @builtins.staticmethod
    def __jsii_proxy_class__():
        return _IParameterGroupProxy

    @jsii.member(jsii_name="addParameter")
    def add_parameter(self, key: builtins.str, value: builtins.str) -> builtins.bool:
        """Adds a parameter to this group.

        If this is an imported parameter group,
        this method does nothing.

        :param key: -
        :param value: -

        :return:

        true if the parameter was actually added
        (i.e., this ParameterGroup is not imported),
        false otherwise
        """
        ...

    @jsii.member(jsii_name="bindToCluster")
    def bind_to_cluster(self) -> "ParameterGroupClusterConfig":
        """Method called when this Parameter Group is used when defining a database cluster."""
        ...

    @jsii.member(jsii_name="bindToInstance")
    def bind_to_instance(self) -> "ParameterGroupInstanceConfig":
        """Method called when this Parameter Group is used when defining a database instance."""
        ...


class _IParameterGroupProxy(
    jsii.proxy_for(aws_cdk.core.IResource) # type: ignore
):
    """A parameter group.

    Represents both a cluster parameter group,
    and an instance parameter group.
    """

    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-rds.IParameterGroup"

    @jsii.member(jsii_name="addParameter")
    def add_parameter(self, key: builtins.str, value: builtins.str) -> builtins.bool:
        """Adds a parameter to this group.

        If this is an imported parameter group,
        this method does nothing.

        :param key: -
        :param value: -

        :return:

        true if the parameter was actually added
        (i.e., this ParameterGroup is not imported),
        false otherwise
        """
        return jsii.invoke(self, "addParameter", [key, value])

    @jsii.member(jsii_name="bindToCluster")
    def bind_to_cluster(self) -> "ParameterGroupClusterConfig":
        """Method called when this Parameter Group is used when defining a database cluster."""
        options = ParameterGroupClusterBindOptions()

        return jsii.invoke(self, "bindToCluster", [options])

    @jsii.member(jsii_name="bindToInstance")
    def bind_to_instance(self) -> "ParameterGroupInstanceConfig":
        """Method called when this Parameter Group is used when defining a database instance."""
        options = ParameterGroupInstanceBindOptions()

        return jsii.invoke(self, "bindToInstance", [options])


@jsii.interface(jsii_type="@aws-cdk/aws-rds.IServerlessCluster")
class IServerlessCluster(
    aws_cdk.core.IResource,
    aws_cdk.aws_ec2.IConnectable,
    aws_cdk.aws_secretsmanager.ISecretAttachmentTarget,
    typing_extensions.Protocol,
):
    """(experimental) Interface representing a serverless database cluster.

    :stability: experimental
    """

    @builtins.staticmethod
    def __jsii_proxy_class__():
        return _IServerlessClusterProxy

    @builtins.property # type: ignore
    @jsii.member(jsii_name="clusterArn")
    def cluster_arn(self) -> builtins.str:
        """(experimental) The ARN of the cluster.

        :stability: experimental
        """
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="clusterEndpoint")
    def cluster_endpoint(self) -> Endpoint:
        """(experimental) The endpoint to use for read/write operations.

        :stability: experimental
        :attribute: EndpointAddress,EndpointPort
        """
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="clusterIdentifier")
    def cluster_identifier(self) -> builtins.str:
        """(experimental) Identifier of the cluster.

        :stability: experimental
        """
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="clusterReadEndpoint")
    def cluster_read_endpoint(self) -> Endpoint:
        """(experimental) Endpoint to use for load-balanced read-only operations.

        :stability: experimental
        :attribute: ReadEndpointAddress
        """
        ...

    @jsii.member(jsii_name="grantDataApiAccess")
    def grant_data_api_access(
        self,
        grantee: aws_cdk.aws_iam.IGrantable,
    ) -> aws_cdk.aws_iam.Grant:
        """(experimental) Grant the given identity to access to the Data API.

        :param grantee: The principal to grant access to.

        :stability: experimental
        """
        ...


class _IServerlessClusterProxy(
    jsii.proxy_for(aws_cdk.core.IResource), # type: ignore
    jsii.proxy_for(aws_cdk.aws_ec2.IConnectable), # type: ignore
    jsii.proxy_for(aws_cdk.aws_secretsmanager.ISecretAttachmentTarget), # type: ignore
):
    """(experimental) Interface representing a serverless database cluster.

    :stability: experimental
    """

    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-rds.IServerlessCluster"

    @builtins.property # type: ignore
    @jsii.member(jsii_name="clusterArn")
    def cluster_arn(self) -> builtins.str:
        """(experimental) The ARN of the cluster.

        :stability: experimental
        """
        return jsii.get(self, "clusterArn")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="clusterEndpoint")
    def cluster_endpoint(self) -> Endpoint:
        """(experimental) The endpoint to use for read/write operations.

        :stability: experimental
        :attribute: EndpointAddress,EndpointPort
        """
        return jsii.get(self, "clusterEndpoint")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="clusterIdentifier")
    def cluster_identifier(self) -> builtins.str:
        """(experimental) Identifier of the cluster.

        :stability: experimental
        """
        return jsii.get(self, "clusterIdentifier")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="clusterReadEndpoint")
    def cluster_read_endpoint(self) -> Endpoint:
        """(experimental) Endpoint to use for load-balanced read-only operations.

        :stability: experimental
        :attribute: ReadEndpointAddress
        """
        return jsii.get(self, "clusterReadEndpoint")

    @jsii.member(jsii_name="grantDataApiAccess")
    def grant_data_api_access(
        self,
        grantee: aws_cdk.aws_iam.IGrantable,
    ) -> aws_cdk.aws_iam.Grant:
        """(experimental) Grant the given identity to access to the Data API.

        :param grantee: The principal to grant access to.

        :stability: experimental
        """
        return jsii.invoke(self, "grantDataApiAccess", [grantee])


@jsii.interface(jsii_type="@aws-cdk/aws-rds.ISubnetGroup")
class ISubnetGroup(aws_cdk.core.IResource, typing_extensions.Protocol):
    """Interface for a subnet group."""

    @builtins.staticmethod
    def __jsii_proxy_class__():
        return _ISubnetGroupProxy

    @builtins.property # type: ignore
    @jsii.member(jsii_name="subnetGroupName")
    def subnet_group_name(self) -> builtins.str:
        """The name of the subnet group.

        :attribute: true
        """
        ...


class _ISubnetGroupProxy(
    jsii.proxy_for(aws_cdk.core.IResource) # type: ignore
):
    """Interface for a subnet group."""

    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-rds.ISubnetGroup"

    @builtins.property # type: ignore
    @jsii.member(jsii_name="subnetGroupName")
    def subnet_group_name(self) -> builtins.str:
        """The name of the subnet group.

        :attribute: true
        """
        return jsii.get(self, "subnetGroupName")


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.InstanceEngineBindOptions",
    jsii_struct_bases=[],
    name_mapping={
        "domain": "domain",
        "option_group": "optionGroup",
        "s3_export_role": "s3ExportRole",
        "s3_import_role": "s3ImportRole",
        "timezone": "timezone",
    },
)
class InstanceEngineBindOptions:
    def __init__(
        self,
        *,
        domain: typing.Optional[builtins.str] = None,
        option_group: typing.Optional[IOptionGroup] = None,
        s3_export_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        s3_import_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        timezone: typing.Optional[builtins.str] = None,
    ) -> None:
        """The options passed to {@link IInstanceEngine.bind}.

        :param domain: The Active Directory directory ID to create the DB instance in. Default: - none (it's an optional field)
        :param option_group: The option group of the database. Default: - none
        :param s3_export_role: The role used for S3 exporting. Default: - none
        :param s3_import_role: The role used for S3 importing. Default: - none
        :param timezone: The timezone of the database, set by the customer. Default: - none (it's an optional field)
        """
        self._values: typing.Dict[str, typing.Any] = {}
        if domain is not None:
            self._values["domain"] = domain
        if option_group is not None:
            self._values["option_group"] = option_group
        if s3_export_role is not None:
            self._values["s3_export_role"] = s3_export_role
        if s3_import_role is not None:
            self._values["s3_import_role"] = s3_import_role
        if timezone is not None:
            self._values["timezone"] = timezone

    @builtins.property
    def domain(self) -> typing.Optional[builtins.str]:
        """The Active Directory directory ID to create the DB instance in.

        :default: - none (it's an optional field)
        """
        result = self._values.get("domain")
        return result

    @builtins.property
    def option_group(self) -> typing.Optional[IOptionGroup]:
        """The option group of the database.

        :default: - none
        """
        result = self._values.get("option_group")
        return result

    @builtins.property
    def s3_export_role(self) -> typing.Optional[aws_cdk.aws_iam.IRole]:
        """The role used for S3 exporting.

        :default: - none
        """
        result = self._values.get("s3_export_role")
        return result

    @builtins.property
    def s3_import_role(self) -> typing.Optional[aws_cdk.aws_iam.IRole]:
        """The role used for S3 importing.

        :default: - none
        """
        result = self._values.get("s3_import_role")
        return result

    @builtins.property
    def timezone(self) -> typing.Optional[builtins.str]:
        """The timezone of the database, set by the customer.

        :default: - none (it's an optional field)
        """
        result = self._values.get("timezone")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "InstanceEngineBindOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.InstanceEngineConfig",
    jsii_struct_bases=[],
    name_mapping={"features": "features", "option_group": "optionGroup"},
)
class InstanceEngineConfig:
    def __init__(
        self,
        *,
        features: typing.Optional["InstanceEngineFeatures"] = None,
        option_group: typing.Optional[IOptionGroup] = None,
    ) -> None:
        """The type returned from the {@link IInstanceEngine.bind} method.

        :param features: Features supported by the database engine. Default: - no features
        :param option_group: Option group of the database. Default: - none
        """
        if isinstance(features, dict):
            features = InstanceEngineFeatures(**features)
        self._values: typing.Dict[str, typing.Any] = {}
        if features is not None:
            self._values["features"] = features
        if option_group is not None:
            self._values["option_group"] = option_group

    @builtins.property
    def features(self) -> typing.Optional["InstanceEngineFeatures"]:
        """Features supported by the database engine.

        :default: - no features

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DBEngineVersion.html
        """
        result = self._values.get("features")
        return result

    @builtins.property
    def option_group(self) -> typing.Optional[IOptionGroup]:
        """Option group of the database.

        :default: - none
        """
        result = self._values.get("option_group")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "InstanceEngineConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.InstanceEngineFeatures",
    jsii_struct_bases=[],
    name_mapping={"s3_export": "s3Export", "s3_import": "s3Import"},
)
class InstanceEngineFeatures:
    def __init__(
        self,
        *,
        s3_export: typing.Optional[builtins.str] = None,
        s3_import: typing.Optional[builtins.str] = None,
    ) -> None:
        """Represents Database Engine features.

        :param s3_export: Feature name for the DB instance that the IAM role to export to S3 bucket is to be associated with. Default: - no s3Export feature name
        :param s3_import: Feature name for the DB instance that the IAM role to access the S3 bucket for import is to be associated with. Default: - no s3Import feature name
        """
        self._values: typing.Dict[str, typing.Any] = {}
        if s3_export is not None:
            self._values["s3_export"] = s3_export
        if s3_import is not None:
            self._values["s3_import"] = s3_import

    @builtins.property
    def s3_export(self) -> typing.Optional[builtins.str]:
        """Feature name for the DB instance that the IAM role to export to S3 bucket is to be associated with.

        :default: - no s3Export feature name
        """
        result = self._values.get("s3_export")
        return result

    @builtins.property
    def s3_import(self) -> typing.Optional[builtins.str]:
        """Feature name for the DB instance that the IAM role to access the S3 bucket for import is to be associated with.

        :default: - no s3Import feature name
        """
        result = self._values.get("s3_import")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "InstanceEngineFeatures(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.InstanceProps",
    jsii_struct_bases=[],
    name_mapping={
        "vpc": "vpc",
        "allow_major_version_upgrade": "allowMajorVersionUpgrade",
        "auto_minor_version_upgrade": "autoMinorVersionUpgrade",
        "delete_automated_backups": "deleteAutomatedBackups",
        "enable_performance_insights": "enablePerformanceInsights",
        "instance_type": "instanceType",
        "parameter_group": "parameterGroup",
        "performance_insight_encryption_key": "performanceInsightEncryptionKey",
        "performance_insight_retention": "performanceInsightRetention",
        "publicly_accessible": "publiclyAccessible",
        "security_groups": "securityGroups",
        "vpc_subnets": "vpcSubnets",
    },
)
class InstanceProps:
    def __init__(
        self,
        *,
        vpc: aws_cdk.aws_ec2.IVpc,
        allow_major_version_upgrade: typing.Optional[builtins.bool] = None,
        auto_minor_version_upgrade: typing.Optional[builtins.bool] = None,
        delete_automated_backups: typing.Optional[builtins.bool] = None,
        enable_performance_insights: typing.Optional[builtins.bool] = None,
        instance_type: typing.Optional[aws_cdk.aws_ec2.InstanceType] = None,
        parameter_group: typing.Optional[IParameterGroup] = None,
        performance_insight_encryption_key: typing.Optional[aws_cdk.aws_kms.IKey] = None,
        performance_insight_retention: typing.Optional["PerformanceInsightRetention"] = None,
        publicly_accessible: typing.Optional[builtins.bool] = None,
        security_groups: typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]] = None,
        vpc_subnets: typing.Optional[aws_cdk.aws_ec2.SubnetSelection] = None,
    ) -> None:
        """Instance properties for database instances.

        :param vpc: What subnets to run the RDS instances in. Must be at least 2 subnets in two different AZs.
        :param allow_major_version_upgrade: Whether to allow upgrade of major version for the DB instance. Default: - false
        :param auto_minor_version_upgrade: Whether to enable automatic upgrade of minor version for the DB instance. Default: - true
        :param delete_automated_backups: Whether to remove automated backups immediately after the DB instance is deleted for the DB instance. Default: - true
        :param enable_performance_insights: Whether to enable Performance Insights for the DB instance. Default: - false, unless ``performanceInsightRentention`` or ``performanceInsightEncryptionKey`` is set.
        :param instance_type: What type of instance to start for the replicas. Default: - t3.medium (or, more precisely, db.t3.medium)
        :param parameter_group: The DB parameter group to associate with the instance. Default: no parameter group
        :param performance_insight_encryption_key: The AWS KMS key for encryption of Performance Insights data. Default: - default master key
        :param performance_insight_retention: The amount of time, in days, to retain Performance Insights data. Default: 7
        :param publicly_accessible: Indicates whether the DB instance is an internet-facing instance. Default: - ``true`` if ``vpcSubnets`` is ``subnetType: SubnetType.PUBLIC``, ``false`` otherwise
        :param security_groups: Security group. Default: a new security group is created.
        :param vpc_subnets: Where to place the instances within the VPC. Default: - the Vpc default strategy if not specified.
        """
        if isinstance(vpc_subnets, dict):
            vpc_subnets = aws_cdk.aws_ec2.SubnetSelection(**vpc_subnets)
        self._values: typing.Dict[str, typing.Any] = {
            "vpc": vpc,
        }
        if allow_major_version_upgrade is not None:
            self._values["allow_major_version_upgrade"] = allow_major_version_upgrade
        if auto_minor_version_upgrade is not None:
            self._values["auto_minor_version_upgrade"] = auto_minor_version_upgrade
        if delete_automated_backups is not None:
            self._values["delete_automated_backups"] = delete_automated_backups
        if enable_performance_insights is not None:
            self._values["enable_performance_insights"] = enable_performance_insights
        if instance_type is not None:
            self._values["instance_type"] = instance_type
        if parameter_group is not None:
            self._values["parameter_group"] = parameter_group
        if performance_insight_encryption_key is not None:
            self._values["performance_insight_encryption_key"] = performance_insight_encryption_key
        if performance_insight_retention is not None:
            self._values["performance_insight_retention"] = performance_insight_retention
        if publicly_accessible is not None:
            self._values["publicly_accessible"] = publicly_accessible
        if security_groups is not None:
            self._values["security_groups"] = security_groups
        if vpc_subnets is not None:
            self._values["vpc_subnets"] = vpc_subnets

    @builtins.property
    def vpc(self) -> aws_cdk.aws_ec2.IVpc:
        """What subnets to run the RDS instances in.

        Must be at least 2 subnets in two different AZs.
        """
        result = self._values.get("vpc")
        assert result is not None, "Required property 'vpc' is missing"
        return result

    @builtins.property
    def allow_major_version_upgrade(self) -> typing.Optional[builtins.bool]:
        """Whether to allow upgrade of major version for the DB instance.

        :default: - false
        """
        result = self._values.get("allow_major_version_upgrade")
        return result

    @builtins.property
    def auto_minor_version_upgrade(self) -> typing.Optional[builtins.bool]:
        """Whether to enable automatic upgrade of minor version for the DB instance.

        :default: - true
        """
        result = self._values.get("auto_minor_version_upgrade")
        return result

    @builtins.property
    def delete_automated_backups(self) -> typing.Optional[builtins.bool]:
        """Whether to remove automated backups immediately after the DB instance is deleted for the DB instance.

        :default: - true
        """
        result = self._values.get("delete_automated_backups")
        return result

    @builtins.property
    def enable_performance_insights(self) -> typing.Optional[builtins.bool]:
        """Whether to enable Performance Insights for the DB instance.

        :default: - false, unless ``performanceInsightRentention`` or ``performanceInsightEncryptionKey`` is set.
        """
        result = self._values.get("enable_performance_insights")
        return result

    @builtins.property
    def instance_type(self) -> typing.Optional[aws_cdk.aws_ec2.InstanceType]:
        """What type of instance to start for the replicas.

        :default: - t3.medium (or, more precisely, db.t3.medium)
        """
        result = self._values.get("instance_type")
        return result

    @builtins.property
    def parameter_group(self) -> typing.Optional[IParameterGroup]:
        """The DB parameter group to associate with the instance.

        :default: no parameter group
        """
        result = self._values.get("parameter_group")
        return result

    @builtins.property
    def performance_insight_encryption_key(
        self,
    ) -> typing.Optional[aws_cdk.aws_kms.IKey]:
        """The AWS KMS key for encryption of Performance Insights data.

        :default: - default master key
        """
        result = self._values.get("performance_insight_encryption_key")
        return result

    @builtins.property
    def performance_insight_retention(
        self,
    ) -> typing.Optional["PerformanceInsightRetention"]:
        """The amount of time, in days, to retain Performance Insights data.

        :default: 7
        """
        result = self._values.get("performance_insight_retention")
        return result

    @builtins.property
    def publicly_accessible(self) -> typing.Optional[builtins.bool]:
        """Indicates whether the DB instance is an internet-facing instance.

        :default: - ``true`` if ``vpcSubnets`` is ``subnetType: SubnetType.PUBLIC``, ``false`` otherwise
        """
        result = self._values.get("publicly_accessible")
        return result

    @builtins.property
    def security_groups(
        self,
    ) -> typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]]:
        """Security group.

        :default: a new security group is created.
        """
        result = self._values.get("security_groups")
        return result

    @builtins.property
    def vpc_subnets(self) -> typing.Optional[aws_cdk.aws_ec2.SubnetSelection]:
        """Where to place the instances within the VPC.

        :default: - the Vpc default strategy if not specified.
        """
        result = self._values.get("vpc_subnets")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "InstanceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="@aws-cdk/aws-rds.LicenseModel")
class LicenseModel(enum.Enum):
    """The license model."""

    LICENSE_INCLUDED = "LICENSE_INCLUDED"
    """License included."""
    BRING_YOUR_OWN_LICENSE = "BRING_YOUR_OWN_LICENSE"
    """Bring your own licencse."""
    GENERAL_PUBLIC_LICENSE = "GENERAL_PUBLIC_LICENSE"
    """General public license."""


class MariaDbEngineVersion(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.MariaDbEngineVersion",
):
    """The versions for the MariaDB instance engines (those returned by {@link DatabaseInstanceEngine.mariaDb})."""

    @jsii.member(jsii_name="of")
    @builtins.classmethod
    def of(
        cls,
        maria_db_full_version: builtins.str,
        maria_db_major_version: builtins.str,
    ) -> "MariaDbEngineVersion":
        """Create a new MariaDbEngineVersion with an arbitrary version.

        :param maria_db_full_version: the full version string, for example "10.5.28".
        :param maria_db_major_version: the major version of the engine, for example "10.5".
        """
        return jsii.sinvoke(cls, "of", [maria_db_full_version, maria_db_major_version])

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_0")
    def VER_10_0(cls) -> "MariaDbEngineVersion":
        """Version "10.0" (only a major version, without a specific minor version)."""
        return jsii.sget(cls, "VER_10_0")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_0_17")
    def VER_10_0_17(cls) -> "MariaDbEngineVersion":
        """Version "10.0.17"."""
        return jsii.sget(cls, "VER_10_0_17")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_0_24")
    def VER_10_0_24(cls) -> "MariaDbEngineVersion":
        """Version "10.0.24"."""
        return jsii.sget(cls, "VER_10_0_24")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_0_28")
    def VER_10_0_28(cls) -> "MariaDbEngineVersion":
        """Version "10.0.28"."""
        return jsii.sget(cls, "VER_10_0_28")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_0_31")
    def VER_10_0_31(cls) -> "MariaDbEngineVersion":
        """Version "10.0.31"."""
        return jsii.sget(cls, "VER_10_0_31")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_0_32")
    def VER_10_0_32(cls) -> "MariaDbEngineVersion":
        """Version "10.0.32"."""
        return jsii.sget(cls, "VER_10_0_32")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_0_34")
    def VER_10_0_34(cls) -> "MariaDbEngineVersion":
        """Version "10.0.34"."""
        return jsii.sget(cls, "VER_10_0_34")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_0_35")
    def VER_10_0_35(cls) -> "MariaDbEngineVersion":
        """Version "10.0.35"."""
        return jsii.sget(cls, "VER_10_0_35")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_1")
    def VER_10_1(cls) -> "MariaDbEngineVersion":
        """Version "10.1" (only a major version, without a specific minor version)."""
        return jsii.sget(cls, "VER_10_1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_1_14")
    def VER_10_1_14(cls) -> "MariaDbEngineVersion":
        """Version "10.1.14"."""
        return jsii.sget(cls, "VER_10_1_14")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_1_19")
    def VER_10_1_19(cls) -> "MariaDbEngineVersion":
        """Version "10.1.19"."""
        return jsii.sget(cls, "VER_10_1_19")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_1_23")
    def VER_10_1_23(cls) -> "MariaDbEngineVersion":
        """Version "10.1.23"."""
        return jsii.sget(cls, "VER_10_1_23")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_1_26")
    def VER_10_1_26(cls) -> "MariaDbEngineVersion":
        """Version "10.1.26"."""
        return jsii.sget(cls, "VER_10_1_26")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_1_31")
    def VER_10_1_31(cls) -> "MariaDbEngineVersion":
        """Version "10.1.31"."""
        return jsii.sget(cls, "VER_10_1_31")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_1_34")
    def VER_10_1_34(cls) -> "MariaDbEngineVersion":
        """Version "10.1.34"."""
        return jsii.sget(cls, "VER_10_1_34")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_2")
    def VER_10_2(cls) -> "MariaDbEngineVersion":
        """Version "10.2" (only a major version, without a specific minor version)."""
        return jsii.sget(cls, "VER_10_2")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_2_11")
    def VER_10_2_11(cls) -> "MariaDbEngineVersion":
        """Version "10.2.11"."""
        return jsii.sget(cls, "VER_10_2_11")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_2_12")
    def VER_10_2_12(cls) -> "MariaDbEngineVersion":
        """Version "10.2.12"."""
        return jsii.sget(cls, "VER_10_2_12")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_2_15")
    def VER_10_2_15(cls) -> "MariaDbEngineVersion":
        """Version "10.2.15"."""
        return jsii.sget(cls, "VER_10_2_15")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_2_21")
    def VER_10_2_21(cls) -> "MariaDbEngineVersion":
        """Version "10.2.21"."""
        return jsii.sget(cls, "VER_10_2_21")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_3")
    def VER_10_3(cls) -> "MariaDbEngineVersion":
        """Version "10.3" (only a major version, without a specific minor version)."""
        return jsii.sget(cls, "VER_10_3")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_3_13")
    def VER_10_3_13(cls) -> "MariaDbEngineVersion":
        """Version "10.3.13"."""
        return jsii.sget(cls, "VER_10_3_13")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_3_20")
    def VER_10_3_20(cls) -> "MariaDbEngineVersion":
        """Version "10.3.20"."""
        return jsii.sget(cls, "VER_10_3_20")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_3_23")
    def VER_10_3_23(cls) -> "MariaDbEngineVersion":
        """Version "10.3.23"."""
        return jsii.sget(cls, "VER_10_3_23")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_3_8")
    def VER_10_3_8(cls) -> "MariaDbEngineVersion":
        """Version "10.3.8"."""
        return jsii.sget(cls, "VER_10_3_8")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_4")
    def VER_10_4(cls) -> "MariaDbEngineVersion":
        """Version "10.4" (only a major version, without a specific minor version)."""
        return jsii.sget(cls, "VER_10_4")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_4_13")
    def VER_10_4_13(cls) -> "MariaDbEngineVersion":
        """Version "10.4.13"."""
        return jsii.sget(cls, "VER_10_4_13")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_4_8")
    def VER_10_4_8(cls) -> "MariaDbEngineVersion":
        """Version "10.4.8"."""
        return jsii.sget(cls, "VER_10_4_8")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="mariaDbFullVersion")
    def maria_db_full_version(self) -> builtins.str:
        """The full version string, for example, "10.5.28"."""
        return jsii.get(self, "mariaDbFullVersion")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="mariaDbMajorVersion")
    def maria_db_major_version(self) -> builtins.str:
        """The major version of the engine, for example, "10.5"."""
        return jsii.get(self, "mariaDbMajorVersion")


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.MariaDbInstanceEngineProps",
    jsii_struct_bases=[],
    name_mapping={"version": "version"},
)
class MariaDbInstanceEngineProps:
    def __init__(self, *, version: MariaDbEngineVersion) -> None:
        """Properties for MariaDB instance engines.

        Used in {@link DatabaseInstanceEngine.mariaDb}.

        :param version: The exact version of the engine to use.
        """
        self._values: typing.Dict[str, typing.Any] = {
            "version": version,
        }

    @builtins.property
    def version(self) -> MariaDbEngineVersion:
        """The exact version of the engine to use."""
        result = self._values.get("version")
        assert result is not None, "Required property 'version' is missing"
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "MariaDbInstanceEngineProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.MySqlInstanceEngineProps",
    jsii_struct_bases=[],
    name_mapping={"version": "version"},
)
class MySqlInstanceEngineProps:
    def __init__(self, *, version: "MysqlEngineVersion") -> None:
        """Properties for MySQL instance engines.

        Used in {@link DatabaseInstanceEngine.mysql}.

        :param version: The exact version of the engine to use.
        """
        self._values: typing.Dict[str, typing.Any] = {
            "version": version,
        }

    @builtins.property
    def version(self) -> "MysqlEngineVersion":
        """The exact version of the engine to use."""
        result = self._values.get("version")
        assert result is not None, "Required property 'version' is missing"
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "MySqlInstanceEngineProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class MysqlEngineVersion(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.MysqlEngineVersion",
):
    """The versions for the MySQL instance engines (those returned by {@link DatabaseInstanceEngine.mysql})."""

    @jsii.member(jsii_name="of")
    @builtins.classmethod
    def of(
        cls,
        mysql_full_version: builtins.str,
        mysql_major_version: builtins.str,
    ) -> "MysqlEngineVersion":
        """Create a new MysqlEngineVersion with an arbitrary version.

        :param mysql_full_version: the full version string, for example "8.1.43".
        :param mysql_major_version: the major version of the engine, for example "8.1".
        """
        return jsii.sinvoke(cls, "of", [mysql_full_version, mysql_major_version])

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_5")
    def VER_5_5(cls) -> "MysqlEngineVersion":
        """Version "5.5" (only a major version, without a specific minor version)."""
        return jsii.sget(cls, "VER_5_5")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_5_46")
    def VER_5_5_46(cls) -> "MysqlEngineVersion":
        """Version "5.5.46"."""
        return jsii.sget(cls, "VER_5_5_46")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_5_53")
    def VER_5_5_53(cls) -> "MysqlEngineVersion":
        """Version "5.5.53"."""
        return jsii.sget(cls, "VER_5_5_53")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_5_57")
    def VER_5_5_57(cls) -> "MysqlEngineVersion":
        """Version "5.5.57"."""
        return jsii.sget(cls, "VER_5_5_57")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_5_59")
    def VER_5_5_59(cls) -> "MysqlEngineVersion":
        """Version "5.5.59"."""
        return jsii.sget(cls, "VER_5_5_59")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_5_61")
    def VER_5_5_61(cls) -> "MysqlEngineVersion":
        """Version "5.5.61"."""
        return jsii.sget(cls, "VER_5_5_61")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_6")
    def VER_5_6(cls) -> "MysqlEngineVersion":
        """Version "5.6" (only a major version, without a specific minor version)."""
        return jsii.sget(cls, "VER_5_6")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_6_34")
    def VER_5_6_34(cls) -> "MysqlEngineVersion":
        """Version "5.6.34"."""
        return jsii.sget(cls, "VER_5_6_34")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_6_35")
    def VER_5_6_35(cls) -> "MysqlEngineVersion":
        """Version "5.6.35"."""
        return jsii.sget(cls, "VER_5_6_35")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_6_37")
    def VER_5_6_37(cls) -> "MysqlEngineVersion":
        """Version "5.6.37"."""
        return jsii.sget(cls, "VER_5_6_37")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_6_39")
    def VER_5_6_39(cls) -> "MysqlEngineVersion":
        """Version "5.6.39"."""
        return jsii.sget(cls, "VER_5_6_39")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_6_40")
    def VER_5_6_40(cls) -> "MysqlEngineVersion":
        """Version "5.6.40"."""
        return jsii.sget(cls, "VER_5_6_40")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_6_41")
    def VER_5_6_41(cls) -> "MysqlEngineVersion":
        """Version "5.6.41"."""
        return jsii.sget(cls, "VER_5_6_41")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_6_43")
    def VER_5_6_43(cls) -> "MysqlEngineVersion":
        """Version "5.6.43"."""
        return jsii.sget(cls, "VER_5_6_43")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_6_44")
    def VER_5_6_44(cls) -> "MysqlEngineVersion":
        """Version "5.6.44"."""
        return jsii.sget(cls, "VER_5_6_44")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_6_46")
    def VER_5_6_46(cls) -> "MysqlEngineVersion":
        """Version "5.6.46"."""
        return jsii.sget(cls, "VER_5_6_46")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_6_48")
    def VER_5_6_48(cls) -> "MysqlEngineVersion":
        """Version "5.6.48"."""
        return jsii.sget(cls, "VER_5_6_48")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_7")
    def VER_5_7(cls) -> "MysqlEngineVersion":
        """Version "5.7" (only a major version, without a specific minor version)."""
        return jsii.sget(cls, "VER_5_7")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_7_16")
    def VER_5_7_16(cls) -> "MysqlEngineVersion":
        """Version "5.7.16"."""
        return jsii.sget(cls, "VER_5_7_16")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_7_17")
    def VER_5_7_17(cls) -> "MysqlEngineVersion":
        """Version "5.7.17"."""
        return jsii.sget(cls, "VER_5_7_17")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_7_19")
    def VER_5_7_19(cls) -> "MysqlEngineVersion":
        """Version "5.7.19"."""
        return jsii.sget(cls, "VER_5_7_19")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_7_21")
    def VER_5_7_21(cls) -> "MysqlEngineVersion":
        """Version "5.7.21"."""
        return jsii.sget(cls, "VER_5_7_21")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_7_22")
    def VER_5_7_22(cls) -> "MysqlEngineVersion":
        """Version "5.7.22"."""
        return jsii.sget(cls, "VER_5_7_22")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_7_23")
    def VER_5_7_23(cls) -> "MysqlEngineVersion":
        """Version "5.7.23"."""
        return jsii.sget(cls, "VER_5_7_23")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_7_24")
    def VER_5_7_24(cls) -> "MysqlEngineVersion":
        """Version "5.7.24"."""
        return jsii.sget(cls, "VER_5_7_24")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_7_25")
    def VER_5_7_25(cls) -> "MysqlEngineVersion":
        """Version "5.7.25"."""
        return jsii.sget(cls, "VER_5_7_25")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_7_26")
    def VER_5_7_26(cls) -> "MysqlEngineVersion":
        """Version "5.7.26"."""
        return jsii.sget(cls, "VER_5_7_26")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_7_28")
    def VER_5_7_28(cls) -> "MysqlEngineVersion":
        """Version "5.7.28"."""
        return jsii.sget(cls, "VER_5_7_28")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_7_30")
    def VER_5_7_30(cls) -> "MysqlEngineVersion":
        """Version "5.7.30"."""
        return jsii.sget(cls, "VER_5_7_30")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_5_7_31")
    def VER_5_7_31(cls) -> "MysqlEngineVersion":
        """Version "5.7.31"."""
        return jsii.sget(cls, "VER_5_7_31")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_8_0")
    def VER_8_0(cls) -> "MysqlEngineVersion":
        """Version "8.0" (only a major version, without a specific minor version)."""
        return jsii.sget(cls, "VER_8_0")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_8_0_11")
    def VER_8_0_11(cls) -> "MysqlEngineVersion":
        """Version "8.0.11"."""
        return jsii.sget(cls, "VER_8_0_11")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_8_0_13")
    def VER_8_0_13(cls) -> "MysqlEngineVersion":
        """Version "8.0.13"."""
        return jsii.sget(cls, "VER_8_0_13")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_8_0_15")
    def VER_8_0_15(cls) -> "MysqlEngineVersion":
        """Version "8.0.15"."""
        return jsii.sget(cls, "VER_8_0_15")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_8_0_16")
    def VER_8_0_16(cls) -> "MysqlEngineVersion":
        """Version "8.0.16"."""
        return jsii.sget(cls, "VER_8_0_16")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_8_0_17")
    def VER_8_0_17(cls) -> "MysqlEngineVersion":
        """Version "8.0.17"."""
        return jsii.sget(cls, "VER_8_0_17")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_8_0_19")
    def VER_8_0_19(cls) -> "MysqlEngineVersion":
        """Version "8.0.19"."""
        return jsii.sget(cls, "VER_8_0_19")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_8_0_20")
    def VER_8_0_20(cls) -> "MysqlEngineVersion":
        """Version "8.0.20 "."""
        return jsii.sget(cls, "VER_8_0_20")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_8_0_21")
    def VER_8_0_21(cls) -> "MysqlEngineVersion":
        """Version "8.0.21 "."""
        return jsii.sget(cls, "VER_8_0_21")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="mysqlFullVersion")
    def mysql_full_version(self) -> builtins.str:
        """The full version string, for example, "10.5.28"."""
        return jsii.get(self, "mysqlFullVersion")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="mysqlMajorVersion")
    def mysql_major_version(self) -> builtins.str:
        """The major version of the engine, for example, "10.5"."""
        return jsii.get(self, "mysqlMajorVersion")


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.OptionConfiguration",
    jsii_struct_bases=[],
    name_mapping={
        "name": "name",
        "port": "port",
        "security_groups": "securityGroups",
        "settings": "settings",
        "version": "version",
        "vpc": "vpc",
    },
)
class OptionConfiguration:
    def __init__(
        self,
        *,
        name: builtins.str,
        port: typing.Optional[jsii.Number] = None,
        security_groups: typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]] = None,
        settings: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        version: typing.Optional[builtins.str] = None,
        vpc: typing.Optional[aws_cdk.aws_ec2.IVpc] = None,
    ) -> None:
        """Configuration properties for an option.

        :param name: The name of the option.
        :param port: The port number that this option uses. If ``port`` is specified then ``vpc`` must also be specified. Default: - no port
        :param security_groups: Optional list of security groups to use for this option, if ``vpc`` is specified. If no groups are provided, a default one will be created. Default: - a default group will be created if ``port`` or ``vpc`` are specified.
        :param settings: The settings for the option. Default: - no settings
        :param version: The version for the option. Default: - no version
        :param vpc: The VPC where a security group should be created for this option. If ``vpc`` is specified then ``port`` must also be specified. Default: - no VPC
        """
        self._values: typing.Dict[str, typing.Any] = {
            "name": name,
        }
        if port is not None:
            self._values["port"] = port
        if security_groups is not None:
            self._values["security_groups"] = security_groups
        if settings is not None:
            self._values["settings"] = settings
        if version is not None:
            self._values["version"] = version
        if vpc is not None:
            self._values["vpc"] = vpc

    @builtins.property
    def name(self) -> builtins.str:
        """The name of the option."""
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return result

    @builtins.property
    def port(self) -> typing.Optional[jsii.Number]:
        """The port number that this option uses.

        If ``port`` is specified then ``vpc``
        must also be specified.

        :default: - no port
        """
        result = self._values.get("port")
        return result

    @builtins.property
    def security_groups(
        self,
    ) -> typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]]:
        """Optional list of security groups to use for this option, if ``vpc`` is specified.

        If no groups are provided, a default one will be created.

        :default: - a default group will be created if ``port`` or ``vpc`` are specified.
        """
        result = self._values.get("security_groups")
        return result

    @builtins.property
    def settings(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        """The settings for the option.

        :default: - no settings
        """
        result = self._values.get("settings")
        return result

    @builtins.property
    def version(self) -> typing.Optional[builtins.str]:
        """The version for the option.

        :default: - no version
        """
        result = self._values.get("version")
        return result

    @builtins.property
    def vpc(self) -> typing.Optional[aws_cdk.aws_ec2.IVpc]:
        """The VPC where a security group should be created for this option.

        If ``vpc``
        is specified then ``port`` must also be specified.

        :default: - no VPC
        """
        result = self._values.get("vpc")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "OptionConfiguration(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IOptionGroup)
class OptionGroup(
    aws_cdk.core.Resource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.OptionGroup",
):
    """An option group."""

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        configurations: typing.List[OptionConfiguration],
        engine: IInstanceEngine,
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        """
        :param scope: -
        :param id: -
        :param configurations: The configurations for this option group.
        :param engine: The database engine that this option group is associated with.
        :param description: A description of the option group. Default: a CDK generated description
        """
        props = OptionGroupProps(
            configurations=configurations, engine=engine, description=description
        )

        jsii.create(OptionGroup, self, [scope, id, props])

    @jsii.member(jsii_name="fromOptionGroupName")
    @builtins.classmethod
    def from_option_group_name(
        cls,
        scope: constructs.Construct,
        id: builtins.str,
        option_group_name: builtins.str,
    ) -> IOptionGroup:
        """Import an existing option group.

        :param scope: -
        :param id: -
        :param option_group_name: -
        """
        return jsii.sinvoke(cls, "fromOptionGroupName", [scope, id, option_group_name])

    @jsii.member(jsii_name="addConfiguration")
    def add_configuration(
        self,
        *,
        name: builtins.str,
        port: typing.Optional[jsii.Number] = None,
        security_groups: typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]] = None,
        settings: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        version: typing.Optional[builtins.str] = None,
        vpc: typing.Optional[aws_cdk.aws_ec2.IVpc] = None,
    ) -> builtins.bool:
        """Adds a configuration to this OptionGroup.

        This method is a no-op for an imported OptionGroup.

        :param name: The name of the option.
        :param port: The port number that this option uses. If ``port`` is specified then ``vpc`` must also be specified. Default: - no port
        :param security_groups: Optional list of security groups to use for this option, if ``vpc`` is specified. If no groups are provided, a default one will be created. Default: - a default group will be created if ``port`` or ``vpc`` are specified.
        :param settings: The settings for the option. Default: - no settings
        :param version: The version for the option. Default: - no version
        :param vpc: The VPC where a security group should be created for this option. If ``vpc`` is specified then ``port`` must also be specified. Default: - no VPC
        """
        configuration = OptionConfiguration(
            name=name,
            port=port,
            security_groups=security_groups,
            settings=settings,
            version=version,
            vpc=vpc,
        )

        return jsii.invoke(self, "addConfiguration", [configuration])

    @builtins.property # type: ignore
    @jsii.member(jsii_name="optionConnections")
    def option_connections(
        self,
    ) -> typing.Mapping[builtins.str, aws_cdk.aws_ec2.Connections]:
        """The connections object for the options."""
        return jsii.get(self, "optionConnections")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="optionGroupName")
    def option_group_name(self) -> builtins.str:
        """The name of the option group."""
        return jsii.get(self, "optionGroupName")


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.OptionGroupProps",
    jsii_struct_bases=[],
    name_mapping={
        "configurations": "configurations",
        "engine": "engine",
        "description": "description",
    },
)
class OptionGroupProps:
    def __init__(
        self,
        *,
        configurations: typing.List[OptionConfiguration],
        engine: IInstanceEngine,
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        """Construction properties for an OptionGroup.

        :param configurations: The configurations for this option group.
        :param engine: The database engine that this option group is associated with.
        :param description: A description of the option group. Default: a CDK generated description
        """
        self._values: typing.Dict[str, typing.Any] = {
            "configurations": configurations,
            "engine": engine,
        }
        if description is not None:
            self._values["description"] = description

    @builtins.property
    def configurations(self) -> typing.List[OptionConfiguration]:
        """The configurations for this option group."""
        result = self._values.get("configurations")
        assert result is not None, "Required property 'configurations' is missing"
        return result

    @builtins.property
    def engine(self) -> IInstanceEngine:
        """The database engine that this option group is associated with."""
        result = self._values.get("engine")
        assert result is not None, "Required property 'engine' is missing"
        return result

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        """A description of the option group.

        :default: a CDK generated description
        """
        result = self._values.get("description")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "OptionGroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.OracleEeInstanceEngineProps",
    jsii_struct_bases=[],
    name_mapping={"version": "version"},
)
class OracleEeInstanceEngineProps:
    def __init__(self, *, version: "OracleEngineVersion") -> None:
        """Properties for Oracle Enterprise Edition instance engines.

        Used in {@link DatabaseInstanceEngine.oracleEe}.

        :param version: The exact version of the engine to use.
        """
        self._values: typing.Dict[str, typing.Any] = {
            "version": version,
        }

    @builtins.property
    def version(self) -> "OracleEngineVersion":
        """The exact version of the engine to use."""
        result = self._values.get("version")
        assert result is not None, "Required property 'version' is missing"
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "OracleEeInstanceEngineProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class OracleEngineVersion(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.OracleEngineVersion",
):
    """The versions for the Oracle instance engines (those returned by {@link DatabaseInstanceEngine.oracleSe2} and {@link DatabaseInstanceEngine.oracleEe})."""

    @jsii.member(jsii_name="of")
    @builtins.classmethod
    def of(
        cls,
        oracle_full_version: builtins.str,
        oracle_major_version: builtins.str,
    ) -> "OracleEngineVersion":
        """Creates a new OracleEngineVersion with an arbitrary version.

        :param oracle_full_version: the full version string, for example "19.0.0.0.ru-2019-10.rur-2019-10.r1".
        :param oracle_major_version: the major version of the engine, for example "19".
        """
        return jsii.sinvoke(cls, "of", [oracle_full_version, oracle_major_version])

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_1")
    def VER_12_1(cls) -> "OracleEngineVersion":
        """Version "12.1" (only a major version, without a specific minor version)."""
        return jsii.sget(cls, "VER_12_1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_1_0_2_V1")
    def VER_12_1_0_2_V1(cls) -> "OracleEngineVersion":
        """Version "12.1.0.2.v1"."""
        return jsii.sget(cls, "VER_12_1_0_2_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_1_0_2_V10")
    def VER_12_1_0_2_V10(cls) -> "OracleEngineVersion":
        """Version "12.1.0.2.v10"."""
        return jsii.sget(cls, "VER_12_1_0_2_V10")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_1_0_2_V11")
    def VER_12_1_0_2_V11(cls) -> "OracleEngineVersion":
        """Version "12.1.0.2.v11"."""
        return jsii.sget(cls, "VER_12_1_0_2_V11")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_1_0_2_V12")
    def VER_12_1_0_2_V12(cls) -> "OracleEngineVersion":
        """Version "12.1.0.2.v12"."""
        return jsii.sget(cls, "VER_12_1_0_2_V12")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_1_0_2_V13")
    def VER_12_1_0_2_V13(cls) -> "OracleEngineVersion":
        """Version "12.1.0.2.v13"."""
        return jsii.sget(cls, "VER_12_1_0_2_V13")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_1_0_2_V14")
    def VER_12_1_0_2_V14(cls) -> "OracleEngineVersion":
        """Version "12.1.0.2.v14"."""
        return jsii.sget(cls, "VER_12_1_0_2_V14")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_1_0_2_V15")
    def VER_12_1_0_2_V15(cls) -> "OracleEngineVersion":
        """Version "12.1.0.2.v15"."""
        return jsii.sget(cls, "VER_12_1_0_2_V15")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_1_0_2_V16")
    def VER_12_1_0_2_V16(cls) -> "OracleEngineVersion":
        """Version "12.1.0.2.v16"."""
        return jsii.sget(cls, "VER_12_1_0_2_V16")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_1_0_2_V17")
    def VER_12_1_0_2_V17(cls) -> "OracleEngineVersion":
        """Version "12.1.0.2.v17"."""
        return jsii.sget(cls, "VER_12_1_0_2_V17")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_1_0_2_V18")
    def VER_12_1_0_2_V18(cls) -> "OracleEngineVersion":
        """Version "12.1.0.2.v18"."""
        return jsii.sget(cls, "VER_12_1_0_2_V18")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_1_0_2_V19")
    def VER_12_1_0_2_V19(cls) -> "OracleEngineVersion":
        """Version "12.1.0.2.v19"."""
        return jsii.sget(cls, "VER_12_1_0_2_V19")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_1_0_2_V2")
    def VER_12_1_0_2_V2(cls) -> "OracleEngineVersion":
        """Version "12.1.0.2.v2"."""
        return jsii.sget(cls, "VER_12_1_0_2_V2")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_1_0_2_V20")
    def VER_12_1_0_2_V20(cls) -> "OracleEngineVersion":
        """Version "12.1.0.2.v20"."""
        return jsii.sget(cls, "VER_12_1_0_2_V20")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_1_0_2_V21")
    def VER_12_1_0_2_V21(cls) -> "OracleEngineVersion":
        """Version "12.1.0.2.v21"."""
        return jsii.sget(cls, "VER_12_1_0_2_V21")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_1_0_2_V3")
    def VER_12_1_0_2_V3(cls) -> "OracleEngineVersion":
        """Version "12.1.0.2.v3"."""
        return jsii.sget(cls, "VER_12_1_0_2_V3")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_1_0_2_V4")
    def VER_12_1_0_2_V4(cls) -> "OracleEngineVersion":
        """Version "12.1.0.2.v4"."""
        return jsii.sget(cls, "VER_12_1_0_2_V4")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_1_0_2_V5")
    def VER_12_1_0_2_V5(cls) -> "OracleEngineVersion":
        """Version "12.1.0.2.v5"."""
        return jsii.sget(cls, "VER_12_1_0_2_V5")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_1_0_2_V6")
    def VER_12_1_0_2_V6(cls) -> "OracleEngineVersion":
        """Version "12.1.0.2.v6"."""
        return jsii.sget(cls, "VER_12_1_0_2_V6")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_1_0_2_V7")
    def VER_12_1_0_2_V7(cls) -> "OracleEngineVersion":
        """Version "12.1.0.2.v7"."""
        return jsii.sget(cls, "VER_12_1_0_2_V7")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_1_0_2_V8")
    def VER_12_1_0_2_V8(cls) -> "OracleEngineVersion":
        """Version "12.1.0.2.v8"."""
        return jsii.sget(cls, "VER_12_1_0_2_V8")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_1_0_2_V9")
    def VER_12_1_0_2_V9(cls) -> "OracleEngineVersion":
        """Version "12.1.0.2.v9"."""
        return jsii.sget(cls, "VER_12_1_0_2_V9")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_2")
    def VER_12_2(cls) -> "OracleEngineVersion":
        """Version "12.2" (only a major version, without a specific minor version)."""
        return jsii.sget(cls, "VER_12_2")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_2_0_1_2018_10_R1")
    def VER_12_2_0_1_2018_10_R1(cls) -> "OracleEngineVersion":
        """Version "12.2.0.1.ru-2018-10.rur-2018-10.r1"."""
        return jsii.sget(cls, "VER_12_2_0_1_2018_10_R1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_2_0_1_2019_01_R1")
    def VER_12_2_0_1_2019_01_R1(cls) -> "OracleEngineVersion":
        """Version "12.2.0.1.ru-2019-01.rur-2019-01.r1"."""
        return jsii.sget(cls, "VER_12_2_0_1_2019_01_R1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_2_0_1_2019_04_R1")
    def VER_12_2_0_1_2019_04_R1(cls) -> "OracleEngineVersion":
        """Version "12.2.0.1.ru-2019-04.rur-2019-04.r1"."""
        return jsii.sget(cls, "VER_12_2_0_1_2019_04_R1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_2_0_1_2019_07_R1")
    def VER_12_2_0_1_2019_07_R1(cls) -> "OracleEngineVersion":
        """Version "12.2.0.1.ru-2019-07.rur-2019-07.r1"."""
        return jsii.sget(cls, "VER_12_2_0_1_2019_07_R1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_2_0_1_2019_10_R1")
    def VER_12_2_0_1_2019_10_R1(cls) -> "OracleEngineVersion":
        """Version "12.2.0.1.ru-2019-10.rur-2019-10.r1"."""
        return jsii.sget(cls, "VER_12_2_0_1_2019_10_R1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_2_0_1_2020_01_R1")
    def VER_12_2_0_1_2020_01_R1(cls) -> "OracleEngineVersion":
        """Version "12.2.0.1.ru-2020-01.rur-2020-01.r1"."""
        return jsii.sget(cls, "VER_12_2_0_1_2020_01_R1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_2_0_1_2020_04_R1")
    def VER_12_2_0_1_2020_04_R1(cls) -> "OracleEngineVersion":
        """Version "12.2.0.1.ru-2020-04.rur-2020-04.r1"."""
        return jsii.sget(cls, "VER_12_2_0_1_2020_04_R1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_2_0_1_2020_07_R1")
    def VER_12_2_0_1_2020_07_R1(cls) -> "OracleEngineVersion":
        """Version "12.2.0.1.ru-2020-07.rur-2020-07.r1"."""
        return jsii.sget(cls, "VER_12_2_0_1_2020_07_R1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_18")
    def VER_18(cls) -> "OracleEngineVersion":
        """Version "18" (only a major version, without a specific minor version)."""
        return jsii.sget(cls, "VER_18")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_18_0_0_0_2019_07_R1")
    def VER_18_0_0_0_2019_07_R1(cls) -> "OracleEngineVersion":
        """Version "18.0.0.0.ru-2019-07.rur-2019-07.r1"."""
        return jsii.sget(cls, "VER_18_0_0_0_2019_07_R1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_18_0_0_0_2019_10_R1")
    def VER_18_0_0_0_2019_10_R1(cls) -> "OracleEngineVersion":
        """Version "18.0.0.0.ru-2019-10.rur-2019-10.r1"."""
        return jsii.sget(cls, "VER_18_0_0_0_2019_10_R1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_18_0_0_0_2020_01_R1")
    def VER_18_0_0_0_2020_01_R1(cls) -> "OracleEngineVersion":
        """Version "18.0.0.0.ru-2020-01.rur-2020-01.r1"."""
        return jsii.sget(cls, "VER_18_0_0_0_2020_01_R1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_18_0_0_0_2020_04_R1")
    def VER_18_0_0_0_2020_04_R1(cls) -> "OracleEngineVersion":
        """Version "18.0.0.0.ru-2020-04.rur-2020-04.r1"."""
        return jsii.sget(cls, "VER_18_0_0_0_2020_04_R1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_18_0_0_0_2020_07_R1")
    def VER_18_0_0_0_2020_07_R1(cls) -> "OracleEngineVersion":
        """Version "18.0.0.0.ru-2020-07.rur-2020-07.r1"."""
        return jsii.sget(cls, "VER_18_0_0_0_2020_07_R1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_19")
    def VER_19(cls) -> "OracleEngineVersion":
        """Version "19" (only a major version, without a specific minor version)."""
        return jsii.sget(cls, "VER_19")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_19_0_0_0_2019_07_R1")
    def VER_19_0_0_0_2019_07_R1(cls) -> "OracleEngineVersion":
        """Version "19.0.0.0.ru-2019-07.rur-2019-07.r1"."""
        return jsii.sget(cls, "VER_19_0_0_0_2019_07_R1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_19_0_0_0_2019_10_R1")
    def VER_19_0_0_0_2019_10_R1(cls) -> "OracleEngineVersion":
        """Version "19.0.0.0.ru-2019-10.rur-2019-10.r1"."""
        return jsii.sget(cls, "VER_19_0_0_0_2019_10_R1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_19_0_0_0_2020_01_R1")
    def VER_19_0_0_0_2020_01_R1(cls) -> "OracleEngineVersion":
        """Version "19.0.0.0.ru-2020-01.rur-2020-01.r1"."""
        return jsii.sget(cls, "VER_19_0_0_0_2020_01_R1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_19_0_0_0_2020_04_R1")
    def VER_19_0_0_0_2020_04_R1(cls) -> "OracleEngineVersion":
        """Version "19.0.0.0.ru-2020-04.rur-2020-04.r1"."""
        return jsii.sget(cls, "VER_19_0_0_0_2020_04_R1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_19_0_0_0_2020_07_R1")
    def VER_19_0_0_0_2020_07_R1(cls) -> "OracleEngineVersion":
        """Version "19.0.0.0.ru-2020-07.rur-2020-07.r1"."""
        return jsii.sget(cls, "VER_19_0_0_0_2020_07_R1")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="oracleFullVersion")
    def oracle_full_version(self) -> builtins.str:
        """The full version string, for example, "19.0.0.0.ru-2019-10.rur-2019-10.r1"."""
        return jsii.get(self, "oracleFullVersion")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="oracleMajorVersion")
    def oracle_major_version(self) -> builtins.str:
        """The major version of the engine, for example, "19"."""
        return jsii.get(self, "oracleMajorVersion")


class OracleLegacyEngineVersion(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.OracleLegacyEngineVersion",
):
    """(deprecated) The versions for the legacy Oracle instance engines (those returned by {@link DatabaseInstanceEngine.oracleSe} and {@link DatabaseInstanceEngine.oracleSe1}). Note: RDS will stop allowing creating new databases with this version in August 2020.

    :deprecated: instances can no longer be created with these engine versions. See https://forums.aws.amazon.com/ann.jspa?annID=7341

    :stability: deprecated
    """

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_2")
    def VER_11_2(cls) -> "OracleLegacyEngineVersion":
        """(deprecated) Version "11.2" (only a major version, without a specific minor version).

        :stability: deprecated
        """
        return jsii.sget(cls, "VER_11_2")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_2_0_2_V2")
    def VER_11_2_0_2_V2(cls) -> "OracleLegacyEngineVersion":
        """(deprecated) Version "11.2.0.2.v2".

        :stability: deprecated
        """
        return jsii.sget(cls, "VER_11_2_0_2_V2")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_2_0_4_V1")
    def VER_11_2_0_4_V1(cls) -> "OracleLegacyEngineVersion":
        """(deprecated) Version "11.2.0.4.v1".

        :stability: deprecated
        """
        return jsii.sget(cls, "VER_11_2_0_4_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_2_0_4_V10")
    def VER_11_2_0_4_V10(cls) -> "OracleLegacyEngineVersion":
        """(deprecated) Version "11.2.0.4.v10".

        :stability: deprecated
        """
        return jsii.sget(cls, "VER_11_2_0_4_V10")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_2_0_4_V11")
    def VER_11_2_0_4_V11(cls) -> "OracleLegacyEngineVersion":
        """(deprecated) Version "11.2.0.4.v11".

        :stability: deprecated
        """
        return jsii.sget(cls, "VER_11_2_0_4_V11")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_2_0_4_V12")
    def VER_11_2_0_4_V12(cls) -> "OracleLegacyEngineVersion":
        """(deprecated) Version "11.2.0.4.v12".

        :stability: deprecated
        """
        return jsii.sget(cls, "VER_11_2_0_4_V12")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_2_0_4_V13")
    def VER_11_2_0_4_V13(cls) -> "OracleLegacyEngineVersion":
        """(deprecated) Version "11.2.0.4.v13".

        :stability: deprecated
        """
        return jsii.sget(cls, "VER_11_2_0_4_V13")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_2_0_4_V14")
    def VER_11_2_0_4_V14(cls) -> "OracleLegacyEngineVersion":
        """(deprecated) Version "11.2.0.4.v14".

        :stability: deprecated
        """
        return jsii.sget(cls, "VER_11_2_0_4_V14")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_2_0_4_V15")
    def VER_11_2_0_4_V15(cls) -> "OracleLegacyEngineVersion":
        """(deprecated) Version "11.2.0.4.v15".

        :stability: deprecated
        """
        return jsii.sget(cls, "VER_11_2_0_4_V15")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_2_0_4_V16")
    def VER_11_2_0_4_V16(cls) -> "OracleLegacyEngineVersion":
        """(deprecated) Version "11.2.0.4.v16".

        :stability: deprecated
        """
        return jsii.sget(cls, "VER_11_2_0_4_V16")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_2_0_4_V17")
    def VER_11_2_0_4_V17(cls) -> "OracleLegacyEngineVersion":
        """(deprecated) Version "11.2.0.4.v17".

        :stability: deprecated
        """
        return jsii.sget(cls, "VER_11_2_0_4_V17")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_2_0_4_V18")
    def VER_11_2_0_4_V18(cls) -> "OracleLegacyEngineVersion":
        """(deprecated) Version "11.2.0.4.v18".

        :stability: deprecated
        """
        return jsii.sget(cls, "VER_11_2_0_4_V18")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_2_0_4_V19")
    def VER_11_2_0_4_V19(cls) -> "OracleLegacyEngineVersion":
        """(deprecated) Version "11.2.0.4.v19".

        :stability: deprecated
        """
        return jsii.sget(cls, "VER_11_2_0_4_V19")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_2_0_4_V20")
    def VER_11_2_0_4_V20(cls) -> "OracleLegacyEngineVersion":
        """(deprecated) Version "11.2.0.4.v20".

        :stability: deprecated
        """
        return jsii.sget(cls, "VER_11_2_0_4_V20")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_2_0_4_V21")
    def VER_11_2_0_4_V21(cls) -> "OracleLegacyEngineVersion":
        """(deprecated) Version "11.2.0.4.v21".

        :stability: deprecated
        """
        return jsii.sget(cls, "VER_11_2_0_4_V21")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_2_0_4_V22")
    def VER_11_2_0_4_V22(cls) -> "OracleLegacyEngineVersion":
        """(deprecated) Version "11.2.0.4.v22".

        :stability: deprecated
        """
        return jsii.sget(cls, "VER_11_2_0_4_V22")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_2_0_4_V23")
    def VER_11_2_0_4_V23(cls) -> "OracleLegacyEngineVersion":
        """(deprecated) Version "11.2.0.4.v23".

        :stability: deprecated
        """
        return jsii.sget(cls, "VER_11_2_0_4_V23")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_2_0_4_V24")
    def VER_11_2_0_4_V24(cls) -> "OracleLegacyEngineVersion":
        """(deprecated) Version "11.2.0.4.v24".

        :stability: deprecated
        """
        return jsii.sget(cls, "VER_11_2_0_4_V24")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_2_0_4_V25")
    def VER_11_2_0_4_V25(cls) -> "OracleLegacyEngineVersion":
        """(deprecated) Version "11.2.0.4.v25".

        :stability: deprecated
        """
        return jsii.sget(cls, "VER_11_2_0_4_V25")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_2_0_4_V3")
    def VER_11_2_0_4_V3(cls) -> "OracleLegacyEngineVersion":
        """(deprecated) Version "11.2.0.4.v3".

        :stability: deprecated
        """
        return jsii.sget(cls, "VER_11_2_0_4_V3")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_2_0_4_V4")
    def VER_11_2_0_4_V4(cls) -> "OracleLegacyEngineVersion":
        """(deprecated) Version "11.2.0.4.v4".

        :stability: deprecated
        """
        return jsii.sget(cls, "VER_11_2_0_4_V4")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_2_0_4_V5")
    def VER_11_2_0_4_V5(cls) -> "OracleLegacyEngineVersion":
        """(deprecated) Version "11.2.0.4.v5".

        :stability: deprecated
        """
        return jsii.sget(cls, "VER_11_2_0_4_V5")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_2_0_4_V6")
    def VER_11_2_0_4_V6(cls) -> "OracleLegacyEngineVersion":
        """(deprecated) Version "11.2.0.4.v6".

        :stability: deprecated
        """
        return jsii.sget(cls, "VER_11_2_0_4_V6")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_2_0_4_V7")
    def VER_11_2_0_4_V7(cls) -> "OracleLegacyEngineVersion":
        """(deprecated) Version "11.2.0.4.v7".

        :stability: deprecated
        """
        return jsii.sget(cls, "VER_11_2_0_4_V7")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_2_0_4_V8")
    def VER_11_2_0_4_V8(cls) -> "OracleLegacyEngineVersion":
        """(deprecated) Version "11.2.0.4.v8".

        :stability: deprecated
        """
        return jsii.sget(cls, "VER_11_2_0_4_V8")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_2_0_4_V9")
    def VER_11_2_0_4_V9(cls) -> "OracleLegacyEngineVersion":
        """(deprecated) Version "11.2.0.4.v9".

        :stability: deprecated
        """
        return jsii.sget(cls, "VER_11_2_0_4_V9")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="oracleLegacyFullVersion")
    def oracle_legacy_full_version(self) -> builtins.str:
        """(deprecated) The full version string, for example, "11.2.0.4.v24".

        :stability: deprecated
        """
        return jsii.get(self, "oracleLegacyFullVersion")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="oracleLegacyMajorVersion")
    def oracle_legacy_major_version(self) -> builtins.str:
        """(deprecated) The major version of the engine, for example, "11.2".

        :stability: deprecated
        """
        return jsii.get(self, "oracleLegacyMajorVersion")


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.OracleSe1InstanceEngineProps",
    jsii_struct_bases=[],
    name_mapping={"version": "version"},
)
class OracleSe1InstanceEngineProps:
    def __init__(self, *, version: OracleLegacyEngineVersion) -> None:
        """(deprecated) Properties for Oracle Standard Edition 1 instance engines.

        Used in {@link DatabaseInstanceEngine.oracleSe1}.

        :param version: (deprecated) The exact version of the engine to use.

        :deprecated: instances can no longer be created with this engine. See https://forums.aws.amazon.com/ann.jspa?annID=7341

        :stability: deprecated
        """
        self._values: typing.Dict[str, typing.Any] = {
            "version": version,
        }

    @builtins.property
    def version(self) -> OracleLegacyEngineVersion:
        """(deprecated) The exact version of the engine to use.

        :stability: deprecated
        """
        result = self._values.get("version")
        assert result is not None, "Required property 'version' is missing"
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "OracleSe1InstanceEngineProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.OracleSe2InstanceEngineProps",
    jsii_struct_bases=[],
    name_mapping={"version": "version"},
)
class OracleSe2InstanceEngineProps:
    def __init__(self, *, version: OracleEngineVersion) -> None:
        """Properties for Oracle Standard Edition 2 instance engines.

        Used in {@link DatabaseInstanceEngine.oracleSe2}.

        :param version: The exact version of the engine to use.
        """
        self._values: typing.Dict[str, typing.Any] = {
            "version": version,
        }

    @builtins.property
    def version(self) -> OracleEngineVersion:
        """The exact version of the engine to use."""
        result = self._values.get("version")
        assert result is not None, "Required property 'version' is missing"
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "OracleSe2InstanceEngineProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.OracleSeInstanceEngineProps",
    jsii_struct_bases=[],
    name_mapping={"version": "version"},
)
class OracleSeInstanceEngineProps:
    def __init__(self, *, version: OracleLegacyEngineVersion) -> None:
        """(deprecated) Properties for Oracle Standard Edition instance engines.

        Used in {@link DatabaseInstanceEngine.oracleSe}.

        :param version: (deprecated) The exact version of the engine to use.

        :deprecated: instances can no longer be created with this engine. See https://forums.aws.amazon.com/ann.jspa?annID=7341

        :stability: deprecated
        """
        self._values: typing.Dict[str, typing.Any] = {
            "version": version,
        }

    @builtins.property
    def version(self) -> OracleLegacyEngineVersion:
        """(deprecated) The exact version of the engine to use.

        :stability: deprecated
        """
        result = self._values.get("version")
        assert result is not None, "Required property 'version' is missing"
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "OracleSeInstanceEngineProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IParameterGroup)
class ParameterGroup(
    aws_cdk.core.Resource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.ParameterGroup",
):
    """A parameter group.

    Represents both a cluster parameter group,
    and an instance parameter group.

    :resource: AWS::RDS::DBParameterGroup
    """

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        engine: IEngine,
        description: typing.Optional[builtins.str] = None,
        parameters: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    ) -> None:
        """
        :param scope: -
        :param id: -
        :param engine: The database engine for this parameter group.
        :param description: Description for this parameter group. Default: a CDK generated description
        :param parameters: The parameters in this parameter group. Default: - None
        """
        props = ParameterGroupProps(
            engine=engine, description=description, parameters=parameters
        )

        jsii.create(ParameterGroup, self, [scope, id, props])

    @jsii.member(jsii_name="fromParameterGroupName")
    @builtins.classmethod
    def from_parameter_group_name(
        cls,
        scope: constructs.Construct,
        id: builtins.str,
        parameter_group_name: builtins.str,
    ) -> IParameterGroup:
        """Imports a parameter group.

        :param scope: -
        :param id: -
        :param parameter_group_name: -
        """
        return jsii.sinvoke(cls, "fromParameterGroupName", [scope, id, parameter_group_name])

    @jsii.member(jsii_name="addParameter")
    def add_parameter(self, key: builtins.str, value: builtins.str) -> builtins.bool:
        """Add a parameter to this parameter group.

        :param key: The key of the parameter to be added.
        :param value: The value of the parameter to be added.
        """
        return jsii.invoke(self, "addParameter", [key, value])

    @jsii.member(jsii_name="bindToCluster")
    def bind_to_cluster(self) -> "ParameterGroupClusterConfig":
        """Method called when this Parameter Group is used when defining a database cluster."""
        _options = ParameterGroupClusterBindOptions()

        return jsii.invoke(self, "bindToCluster", [_options])

    @jsii.member(jsii_name="bindToInstance")
    def bind_to_instance(self) -> "ParameterGroupInstanceConfig":
        """Method called when this Parameter Group is used when defining a database instance."""
        _options = ParameterGroupInstanceBindOptions()

        return jsii.invoke(self, "bindToInstance", [_options])


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.ParameterGroupClusterBindOptions",
    jsii_struct_bases=[],
    name_mapping={},
)
class ParameterGroupClusterBindOptions:
    def __init__(self) -> None:
        """Options for {@link IParameterGroup.bindToCluster}. Empty for now, but can be extended later."""
        self._values: typing.Dict[str, typing.Any] = {}

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ParameterGroupClusterBindOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.ParameterGroupClusterConfig",
    jsii_struct_bases=[],
    name_mapping={"parameter_group_name": "parameterGroupName"},
)
class ParameterGroupClusterConfig:
    def __init__(self, *, parameter_group_name: builtins.str) -> None:
        """The type returned from {@link IParameterGroup.bindToCluster}.

        :param parameter_group_name: The name of this parameter group.
        """
        self._values: typing.Dict[str, typing.Any] = {
            "parameter_group_name": parameter_group_name,
        }

    @builtins.property
    def parameter_group_name(self) -> builtins.str:
        """The name of this parameter group."""
        result = self._values.get("parameter_group_name")
        assert result is not None, "Required property 'parameter_group_name' is missing"
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ParameterGroupClusterConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.ParameterGroupInstanceBindOptions",
    jsii_struct_bases=[],
    name_mapping={},
)
class ParameterGroupInstanceBindOptions:
    def __init__(self) -> None:
        """Options for {@link IParameterGroup.bindToInstance}. Empty for now, but can be extended later."""
        self._values: typing.Dict[str, typing.Any] = {}

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ParameterGroupInstanceBindOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.ParameterGroupInstanceConfig",
    jsii_struct_bases=[],
    name_mapping={"parameter_group_name": "parameterGroupName"},
)
class ParameterGroupInstanceConfig:
    def __init__(self, *, parameter_group_name: builtins.str) -> None:
        """The type returned from {@link IParameterGroup.bindToInstance}.

        :param parameter_group_name: The name of this parameter group.
        """
        self._values: typing.Dict[str, typing.Any] = {
            "parameter_group_name": parameter_group_name,
        }

    @builtins.property
    def parameter_group_name(self) -> builtins.str:
        """The name of this parameter group."""
        result = self._values.get("parameter_group_name")
        assert result is not None, "Required property 'parameter_group_name' is missing"
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ParameterGroupInstanceConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.ParameterGroupProps",
    jsii_struct_bases=[],
    name_mapping={
        "engine": "engine",
        "description": "description",
        "parameters": "parameters",
    },
)
class ParameterGroupProps:
    def __init__(
        self,
        *,
        engine: IEngine,
        description: typing.Optional[builtins.str] = None,
        parameters: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    ) -> None:
        """Properties for a parameter group.

        :param engine: The database engine for this parameter group.
        :param description: Description for this parameter group. Default: a CDK generated description
        :param parameters: The parameters in this parameter group. Default: - None
        """
        self._values: typing.Dict[str, typing.Any] = {
            "engine": engine,
        }
        if description is not None:
            self._values["description"] = description
        if parameters is not None:
            self._values["parameters"] = parameters

    @builtins.property
    def engine(self) -> IEngine:
        """The database engine for this parameter group."""
        result = self._values.get("engine")
        assert result is not None, "Required property 'engine' is missing"
        return result

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        """Description for this parameter group.

        :default: a CDK generated description
        """
        result = self._values.get("description")
        return result

    @builtins.property
    def parameters(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        """The parameters in this parameter group.

        :default: - None
        """
        result = self._values.get("parameters")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ParameterGroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="@aws-cdk/aws-rds.PerformanceInsightRetention")
class PerformanceInsightRetention(enum.Enum):
    """The retention period for Performance Insight."""

    DEFAULT = "DEFAULT"
    """Default retention period of 7 days."""
    LONG_TERM = "LONG_TERM"
    """Long term retention period of 2 years."""


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.PostgresEngineFeatures",
    jsii_struct_bases=[],
    name_mapping={"s3_import": "s3Import"},
)
class PostgresEngineFeatures:
    def __init__(self, *, s3_import: typing.Optional[builtins.bool] = None) -> None:
        """Features supported by the Postgres database engine.

        :param s3_import: Whether this version of the Postgres engine supports the S3 data import feature. Default: false
        """
        self._values: typing.Dict[str, typing.Any] = {}
        if s3_import is not None:
            self._values["s3_import"] = s3_import

    @builtins.property
    def s3_import(self) -> typing.Optional[builtins.bool]:
        """Whether this version of the Postgres engine supports the S3 data import feature.

        :default: false
        """
        result = self._values.get("s3_import")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "PostgresEngineFeatures(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class PostgresEngineVersion(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.PostgresEngineVersion",
):
    """The versions for the PostgreSQL instance engines (those returned by {@link DatabaseInstanceEngine.postgres})."""

    @jsii.member(jsii_name="of")
    @builtins.classmethod
    def of(
        cls,
        postgres_full_version: builtins.str,
        postgres_major_version: builtins.str,
        *,
        s3_import: typing.Optional[builtins.bool] = None,
    ) -> "PostgresEngineVersion":
        """Create a new PostgresEngineVersion with an arbitrary version.

        :param postgres_full_version: the full version string, for example "13.11".
        :param postgres_major_version: the major version of the engine, for example "13".
        :param s3_import: Whether this version of the Postgres engine supports the S3 data import feature. Default: false
        """
        postgres_features = PostgresEngineFeatures(s3_import=s3_import)

        return jsii.sinvoke(cls, "of", [postgres_full_version, postgres_major_version, postgres_features])

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10")
    def VER_10(cls) -> "PostgresEngineVersion":
        """Version "10" (only a major version, without a specific minor version)."""
        return jsii.sget(cls, "VER_10")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_1")
    def VER_10_1(cls) -> "PostgresEngineVersion":
        """Version "10.1"."""
        return jsii.sget(cls, "VER_10_1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_10")
    def VER_10_10(cls) -> "PostgresEngineVersion":
        """Version "10.10"."""
        return jsii.sget(cls, "VER_10_10")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_11")
    def VER_10_11(cls) -> "PostgresEngineVersion":
        """Version "10.11"."""
        return jsii.sget(cls, "VER_10_11")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_12")
    def VER_10_12(cls) -> "PostgresEngineVersion":
        """Version "10.12"."""
        return jsii.sget(cls, "VER_10_12")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_13")
    def VER_10_13(cls) -> "PostgresEngineVersion":
        """Version "10.13"."""
        return jsii.sget(cls, "VER_10_13")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_14")
    def VER_10_14(cls) -> "PostgresEngineVersion":
        """Version "10.14"."""
        return jsii.sget(cls, "VER_10_14")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_3")
    def VER_10_3(cls) -> "PostgresEngineVersion":
        """Version "10.3"."""
        return jsii.sget(cls, "VER_10_3")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_4")
    def VER_10_4(cls) -> "PostgresEngineVersion":
        """Version "10.4"."""
        return jsii.sget(cls, "VER_10_4")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_5")
    def VER_10_5(cls) -> "PostgresEngineVersion":
        """Version "10.5"."""
        return jsii.sget(cls, "VER_10_5")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_6")
    def VER_10_6(cls) -> "PostgresEngineVersion":
        """Version "10.6"."""
        return jsii.sget(cls, "VER_10_6")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_7")
    def VER_10_7(cls) -> "PostgresEngineVersion":
        """Version "10.7"."""
        return jsii.sget(cls, "VER_10_7")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_10_9")
    def VER_10_9(cls) -> "PostgresEngineVersion":
        """Version "10.9"."""
        return jsii.sget(cls, "VER_10_9")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11")
    def VER_11(cls) -> "PostgresEngineVersion":
        """Version "11" (only a major version, without a specific minor version)."""
        return jsii.sget(cls, "VER_11")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_1")
    def VER_11_1(cls) -> "PostgresEngineVersion":
        """Version "11.1"."""
        return jsii.sget(cls, "VER_11_1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_2")
    def VER_11_2(cls) -> "PostgresEngineVersion":
        """Version "11.2"."""
        return jsii.sget(cls, "VER_11_2")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_4")
    def VER_11_4(cls) -> "PostgresEngineVersion":
        """Version "11.4"."""
        return jsii.sget(cls, "VER_11_4")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_5")
    def VER_11_5(cls) -> "PostgresEngineVersion":
        """Version "11.5"."""
        return jsii.sget(cls, "VER_11_5")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_6")
    def VER_11_6(cls) -> "PostgresEngineVersion":
        """Version "11.6"."""
        return jsii.sget(cls, "VER_11_6")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_7")
    def VER_11_7(cls) -> "PostgresEngineVersion":
        """Version "11.7"."""
        return jsii.sget(cls, "VER_11_7")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_8")
    def VER_11_8(cls) -> "PostgresEngineVersion":
        """Version "11.8"."""
        return jsii.sget(cls, "VER_11_8")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_9")
    def VER_11_9(cls) -> "PostgresEngineVersion":
        """Version "11.9"."""
        return jsii.sget(cls, "VER_11_9")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12")
    def VER_12(cls) -> "PostgresEngineVersion":
        """Version "12" (only a major version, without a specific minor version)."""
        return jsii.sget(cls, "VER_12")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_2")
    def VER_12_2(cls) -> "PostgresEngineVersion":
        """Version "12.2"."""
        return jsii.sget(cls, "VER_12_2")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_3")
    def VER_12_3(cls) -> "PostgresEngineVersion":
        """Version "12.3"."""
        return jsii.sget(cls, "VER_12_3")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_4")
    def VER_12_4(cls) -> "PostgresEngineVersion":
        """Version "12.4"."""
        return jsii.sget(cls, "VER_12_4")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_5")
    def VER_9_5(cls) -> "PostgresEngineVersion":
        """Version "9.5" (only a major version, without a specific minor version)."""
        return jsii.sget(cls, "VER_9_5")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_5_10")
    def VER_9_5_10(cls) -> "PostgresEngineVersion":
        """Version "9.5.10"."""
        return jsii.sget(cls, "VER_9_5_10")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_5_12")
    def VER_9_5_12(cls) -> "PostgresEngineVersion":
        """Version "9.5.12"."""
        return jsii.sget(cls, "VER_9_5_12")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_5_13")
    def VER_9_5_13(cls) -> "PostgresEngineVersion":
        """Version "9.5.13"."""
        return jsii.sget(cls, "VER_9_5_13")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_5_14")
    def VER_9_5_14(cls) -> "PostgresEngineVersion":
        """Version "9.5.14"."""
        return jsii.sget(cls, "VER_9_5_14")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_5_15")
    def VER_9_5_15(cls) -> "PostgresEngineVersion":
        """Version "9.5.15"."""
        return jsii.sget(cls, "VER_9_5_15")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_5_16")
    def VER_9_5_16(cls) -> "PostgresEngineVersion":
        """Version "9.5.16"."""
        return jsii.sget(cls, "VER_9_5_16")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_5_18")
    def VER_9_5_18(cls) -> "PostgresEngineVersion":
        """Version "9.5.18"."""
        return jsii.sget(cls, "VER_9_5_18")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_5_19")
    def VER_9_5_19(cls) -> "PostgresEngineVersion":
        """Version "9.5.19"."""
        return jsii.sget(cls, "VER_9_5_19")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_5_2")
    def VER_9_5_2(cls) -> "PostgresEngineVersion":
        """Version "9.5.2"."""
        return jsii.sget(cls, "VER_9_5_2")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_5_20")
    def VER_9_5_20(cls) -> "PostgresEngineVersion":
        """Version "9.5.20"."""
        return jsii.sget(cls, "VER_9_5_20")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_5_21")
    def VER_9_5_21(cls) -> "PostgresEngineVersion":
        """Version "9.5.21"."""
        return jsii.sget(cls, "VER_9_5_21")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_5_22")
    def VER_9_5_22(cls) -> "PostgresEngineVersion":
        """Version "9.5.22"."""
        return jsii.sget(cls, "VER_9_5_22")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_5_23")
    def VER_9_5_23(cls) -> "PostgresEngineVersion":
        """Version "9.5.23"."""
        return jsii.sget(cls, "VER_9_5_23")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_5_4")
    def VER_9_5_4(cls) -> "PostgresEngineVersion":
        """Version "9.5.4"."""
        return jsii.sget(cls, "VER_9_5_4")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_5_6")
    def VER_9_5_6(cls) -> "PostgresEngineVersion":
        """Version "9.5.6"."""
        return jsii.sget(cls, "VER_9_5_6")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_5_7")
    def VER_9_5_7(cls) -> "PostgresEngineVersion":
        """Version "9.5.7"."""
        return jsii.sget(cls, "VER_9_5_7")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_5_9")
    def VER_9_5_9(cls) -> "PostgresEngineVersion":
        """Version "9.5.9"."""
        return jsii.sget(cls, "VER_9_5_9")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_6")
    def VER_9_6(cls) -> "PostgresEngineVersion":
        """Version "9.6" (only a major version, without a specific minor version)."""
        return jsii.sget(cls, "VER_9_6")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_6_1")
    def VER_9_6_1(cls) -> "PostgresEngineVersion":
        """Version "9.6.1"."""
        return jsii.sget(cls, "VER_9_6_1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_6_10")
    def VER_9_6_10(cls) -> "PostgresEngineVersion":
        """Version "9.6.10"."""
        return jsii.sget(cls, "VER_9_6_10")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_6_11")
    def VER_9_6_11(cls) -> "PostgresEngineVersion":
        """Version "9.6.11"."""
        return jsii.sget(cls, "VER_9_6_11")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_6_12")
    def VER_9_6_12(cls) -> "PostgresEngineVersion":
        """Version "9.6.12"."""
        return jsii.sget(cls, "VER_9_6_12")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_6_14")
    def VER_9_6_14(cls) -> "PostgresEngineVersion":
        """Version "9.6.14"."""
        return jsii.sget(cls, "VER_9_6_14")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_6_15")
    def VER_9_6_15(cls) -> "PostgresEngineVersion":
        """Version "9.6.15"."""
        return jsii.sget(cls, "VER_9_6_15")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_6_16")
    def VER_9_6_16(cls) -> "PostgresEngineVersion":
        """Version "9.6.16"."""
        return jsii.sget(cls, "VER_9_6_16")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_6_17")
    def VER_9_6_17(cls) -> "PostgresEngineVersion":
        """Version "9.6.17"."""
        return jsii.sget(cls, "VER_9_6_17")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_6_18")
    def VER_9_6_18(cls) -> "PostgresEngineVersion":
        """Version "9.6.18"."""
        return jsii.sget(cls, "VER_9_6_18")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_6_19")
    def VER_9_6_19(cls) -> "PostgresEngineVersion":
        """Version "9.6.19"."""
        return jsii.sget(cls, "VER_9_6_19")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_6_2")
    def VER_9_6_2(cls) -> "PostgresEngineVersion":
        """Version "9.6.2"."""
        return jsii.sget(cls, "VER_9_6_2")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_6_3")
    def VER_9_6_3(cls) -> "PostgresEngineVersion":
        """Version "9.6.3"."""
        return jsii.sget(cls, "VER_9_6_3")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_6_5")
    def VER_9_6_5(cls) -> "PostgresEngineVersion":
        """Version "9.6.5"."""
        return jsii.sget(cls, "VER_9_6_5")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_6_6")
    def VER_9_6_6(cls) -> "PostgresEngineVersion":
        """Version "9.6.6"."""
        return jsii.sget(cls, "VER_9_6_6")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_6_8")
    def VER_9_6_8(cls) -> "PostgresEngineVersion":
        """Version "9.6.8"."""
        return jsii.sget(cls, "VER_9_6_8")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_9_6_9")
    def VER_9_6_9(cls) -> "PostgresEngineVersion":
        """Version "9.6.9"."""
        return jsii.sget(cls, "VER_9_6_9")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="postgresFullVersion")
    def postgres_full_version(self) -> builtins.str:
        """The full version string, for example, "13.11"."""
        return jsii.get(self, "postgresFullVersion")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="postgresMajorVersion")
    def postgres_major_version(self) -> builtins.str:
        """The major version of the engine, for example, "13"."""
        return jsii.get(self, "postgresMajorVersion")


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.PostgresInstanceEngineProps",
    jsii_struct_bases=[],
    name_mapping={"version": "version"},
)
class PostgresInstanceEngineProps:
    def __init__(self, *, version: PostgresEngineVersion) -> None:
        """Properties for PostgreSQL instance engines.

        Used in {@link DatabaseInstanceEngine.postgres}.

        :param version: The exact version of the engine to use.
        """
        self._values: typing.Dict[str, typing.Any] = {
            "version": version,
        }

    @builtins.property
    def version(self) -> PostgresEngineVersion:
        """The exact version of the engine to use."""
        result = self._values.get("version")
        assert result is not None, "Required property 'version' is missing"
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "PostgresInstanceEngineProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.ProcessorFeatures",
    jsii_struct_bases=[],
    name_mapping={"core_count": "coreCount", "threads_per_core": "threadsPerCore"},
)
class ProcessorFeatures:
    def __init__(
        self,
        *,
        core_count: typing.Optional[jsii.Number] = None,
        threads_per_core: typing.Optional[jsii.Number] = None,
    ) -> None:
        """The processor features.

        :param core_count: The number of CPU core. Default: - the default number of CPU cores for the chosen instance class.
        :param threads_per_core: The number of threads per core. Default: - the default number of threads per core for the chosen instance class.
        """
        self._values: typing.Dict[str, typing.Any] = {}
        if core_count is not None:
            self._values["core_count"] = core_count
        if threads_per_core is not None:
            self._values["threads_per_core"] = threads_per_core

    @builtins.property
    def core_count(self) -> typing.Optional[jsii.Number]:
        """The number of CPU core.

        :default: - the default number of CPU cores for the chosen instance class.
        """
        result = self._values.get("core_count")
        return result

    @builtins.property
    def threads_per_core(self) -> typing.Optional[jsii.Number]:
        """The number of threads per core.

        :default: - the default number of threads per core for the chosen instance class.
        """
        result = self._values.get("threads_per_core")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ProcessorFeatures(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class ProxyTarget(metaclass=jsii.JSIIMeta, jsii_type="@aws-cdk/aws-rds.ProxyTarget"):
    """Proxy target: Instance or Cluster.

    A target group is a collection of databases that the proxy can connect to.
    Currently, you can specify only one RDS DB instance or Aurora DB cluster.
    """

    @jsii.member(jsii_name="fromCluster")
    @builtins.classmethod
    def from_cluster(cls, cluster: IDatabaseCluster) -> "ProxyTarget":
        """From cluster.

        :param cluster: RDS database cluster.
        """
        return jsii.sinvoke(cls, "fromCluster", [cluster])

    @jsii.member(jsii_name="fromInstance")
    @builtins.classmethod
    def from_instance(cls, instance: IDatabaseInstance) -> "ProxyTarget":
        """From instance.

        :param instance: RDS database instance.
        """
        return jsii.sinvoke(cls, "fromInstance", [instance])

    @jsii.member(jsii_name="bind")
    def bind(self, _: "DatabaseProxy") -> "ProxyTargetConfig":
        """Bind this target to the specified database proxy.

        :param _: -
        """
        return jsii.invoke(self, "bind", [_])


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.ProxyTargetConfig",
    jsii_struct_bases=[],
    name_mapping={
        "engine_family": "engineFamily",
        "db_clusters": "dbClusters",
        "db_instances": "dbInstances",
    },
)
class ProxyTargetConfig:
    def __init__(
        self,
        *,
        engine_family: builtins.str,
        db_clusters: typing.Optional[typing.List[IDatabaseCluster]] = None,
        db_instances: typing.Optional[typing.List[IDatabaseInstance]] = None,
    ) -> None:
        """The result of binding a ``ProxyTarget`` to a ``DatabaseProxy``.

        :param engine_family: The engine family of the database instance or cluster this proxy connects with.
        :param db_clusters: The database clusters to which this proxy connects. Either this or ``dbInstances`` will be set and the other ``undefined``. Default: - ``undefined`` if ``dbInstances`` is set.
        :param db_instances: The database instances to which this proxy connects. Either this or ``dbClusters`` will be set and the other ``undefined``. Default: - ``undefined`` if ``dbClusters`` is set.
        """
        self._values: typing.Dict[str, typing.Any] = {
            "engine_family": engine_family,
        }
        if db_clusters is not None:
            self._values["db_clusters"] = db_clusters
        if db_instances is not None:
            self._values["db_instances"] = db_instances

    @builtins.property
    def engine_family(self) -> builtins.str:
        """The engine family of the database instance or cluster this proxy connects with."""
        result = self._values.get("engine_family")
        assert result is not None, "Required property 'engine_family' is missing"
        return result

    @builtins.property
    def db_clusters(self) -> typing.Optional[typing.List[IDatabaseCluster]]:
        """The database clusters to which this proxy connects.

        Either this or ``dbInstances`` will be set and the other ``undefined``.

        :default: - ``undefined`` if ``dbInstances`` is set.
        """
        result = self._values.get("db_clusters")
        return result

    @builtins.property
    def db_instances(self) -> typing.Optional[typing.List[IDatabaseInstance]]:
        """The database instances to which this proxy connects.

        Either this or ``dbClusters`` will be set and the other ``undefined``.

        :default: - ``undefined`` if ``dbClusters`` is set.
        """
        result = self._values.get("db_instances")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ProxyTargetConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.RotationMultiUserOptions",
    jsii_struct_bases=[],
    name_mapping={
        "secret": "secret",
        "automatically_after": "automaticallyAfter",
        "exclude_characters": "excludeCharacters",
    },
)
class RotationMultiUserOptions:
    def __init__(
        self,
        *,
        secret: aws_cdk.aws_secretsmanager.ISecret,
        automatically_after: typing.Optional[aws_cdk.core.Duration] = None,
        exclude_characters: typing.Optional[builtins.str] = None,
    ) -> None:
        """Options to add the multi user rotation.

        :param secret: The secret to rotate. It must be a JSON string with the following format:: { "engine": <required: database engine>, "host": <required: instance host name>, "username": <required: username>, "password": <required: password>, "dbname": <optional: database name>, "port": <optional: if not specified, default port will be used>, "masterarn": <required: the arn of the master secret which will be used to create users/change passwords> }
        :param automatically_after: Specifies the number of days after the previous rotation before Secrets Manager triggers the next automatic rotation. Default: - 30 days
        :param exclude_characters: Specifies characters to not include in generated passwords. Default: " %+~`#$&*()|[]{}:;<>?!'/
        """
        self._values: typing.Dict[str, typing.Any] = {
            "secret": secret,
        }
        if automatically_after is not None:
            self._values["automatically_after"] = automatically_after
        if exclude_characters is not None:
            self._values["exclude_characters"] = exclude_characters

    @builtins.property
    def secret(self) -> aws_cdk.aws_secretsmanager.ISecret:
        """The secret to rotate.

        It must be a JSON string with the following format::

           {
              "engine": <required: database engine>,
              "host": <required: instance host name>,
              "username": <required: username>,
              "password": <required: password>,
              "dbname": <optional: database name>,
              "port": <optional: if not specified, default port will be used>,
              "masterarn": <required: the arn of the master secret which will be used to create users/change passwords>
           }
        """
        result = self._values.get("secret")
        assert result is not None, "Required property 'secret' is missing"
        return result

    @builtins.property
    def automatically_after(self) -> typing.Optional[aws_cdk.core.Duration]:
        """Specifies the number of days after the previous rotation before Secrets Manager triggers the next automatic rotation.

        :default: - 30 days
        """
        result = self._values.get("automatically_after")
        return result

    @builtins.property
    def exclude_characters(self) -> typing.Optional[builtins.str]:
        """Specifies characters to not include in generated passwords.

        :default: " %+~`#$&*()|[]{}:;<>?!'/

        :: "\\"
        """
        result = self._values.get("exclude_characters")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RotationMultiUserOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.RotationSingleUserOptions",
    jsii_struct_bases=[],
    name_mapping={
        "automatically_after": "automaticallyAfter",
        "exclude_characters": "excludeCharacters",
    },
)
class RotationSingleUserOptions:
    def __init__(
        self,
        *,
        automatically_after: typing.Optional[aws_cdk.core.Duration] = None,
        exclude_characters: typing.Optional[builtins.str] = None,
    ) -> None:
        """Options to add the multi user rotation.

        :param automatically_after: Specifies the number of days after the previous rotation before Secrets Manager triggers the next automatic rotation. Default: - 30 days
        :param exclude_characters: Specifies characters to not include in generated passwords. Default: " %+~`#$&*()|[]{}:;<>?!'/
        """
        self._values: typing.Dict[str, typing.Any] = {}
        if automatically_after is not None:
            self._values["automatically_after"] = automatically_after
        if exclude_characters is not None:
            self._values["exclude_characters"] = exclude_characters

    @builtins.property
    def automatically_after(self) -> typing.Optional[aws_cdk.core.Duration]:
        """Specifies the number of days after the previous rotation before Secrets Manager triggers the next automatic rotation.

        :default: - 30 days
        """
        result = self._values.get("automatically_after")
        return result

    @builtins.property
    def exclude_characters(self) -> typing.Optional[builtins.str]:
        """Specifies characters to not include in generated passwords.

        :default: " %+~`#$&*()|[]{}:;<>?!'/

        :: "\\"
        """
        result = self._values.get("exclude_characters")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RotationSingleUserOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IServerlessCluster)
class ServerlessCluster(
    aws_cdk.core.Resource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.ServerlessCluster",
):
    """(experimental) Create an Aurora Serverless Cluster.

    :stability: experimental
    :resource: AWS::RDS::DBCluster
    """

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        engine: "IClusterEngine",
        vpc: aws_cdk.aws_ec2.IVpc,
        backup_retention: typing.Optional[aws_cdk.core.Duration] = None,
        cluster_identifier: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[Credentials] = None,
        default_database_name: typing.Optional[builtins.str] = None,
        deletion_protection: typing.Optional[builtins.bool] = None,
        enable_data_api: typing.Optional[builtins.bool] = None,
        parameter_group: typing.Optional[IParameterGroup] = None,
        removal_policy: typing.Optional[aws_cdk.core.RemovalPolicy] = None,
        scaling: typing.Optional["ServerlessScalingOptions"] = None,
        security_groups: typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]] = None,
        storage_encryption_key: typing.Optional[aws_cdk.aws_kms.IKey] = None,
        subnet_group: typing.Optional[ISubnetGroup] = None,
        vpc_subnets: typing.Optional[aws_cdk.aws_ec2.SubnetSelection] = None,
    ) -> None:
        """
        :param scope: -
        :param id: -
        :param engine: (experimental) What kind of database to start.
        :param vpc: (experimental) The VPC that this Aurora Serverless cluster has been created in.
        :param backup_retention: (experimental) The number of days during which automatic DB snapshots are retained. Automatic backup retention cannot be disabled on serverless clusters. Must be a value from 1 day to 35 days. Default: Duration.days(1)
        :param cluster_identifier: (experimental) An optional identifier for the cluster. Default: - A name is automatically generated.
        :param credentials: (experimental) Credentials for the administrative user. Default: - A username of 'admin' and SecretsManager-generated password
        :param default_database_name: (experimental) Name of a database which is automatically created inside the cluster. Default: - Database is not created in cluster.
        :param deletion_protection: (experimental) Indicates whether the DB cluster should have deletion protection enabled. Default: - true if removalPolicy is RETAIN, false otherwise
        :param enable_data_api: (experimental) Whether to enable the Data API. Default: false
        :param parameter_group: (experimental) Additional parameters to pass to the database engine. Default: - no parameter group.
        :param removal_policy: (experimental) The removal policy to apply when the cluster and its instances are removed from the stack or replaced during an update. Default: - RemovalPolicy.SNAPSHOT (remove the cluster and instances, but retain a snapshot of the data)
        :param scaling: (experimental) Scaling configuration of an Aurora Serverless database cluster. Default: - Serverless cluster is automatically paused after 5 minutes of being idle. minimum capacity: 2 ACU maximum capacity: 16 ACU
        :param security_groups: (experimental) Security group. Default: - a new security group is created.
        :param storage_encryption_key: (experimental) The KMS key for storage encryption. Default: - the default master key will be used for storage encryption
        :param subnet_group: (experimental) Existing subnet group for the cluster. Default: - a new subnet group will be created.
        :param vpc_subnets: (experimental) Where to place the instances within the VPC. Default: - the VPC default strategy if not specified.

        :stability: experimental
        """
        props = ServerlessClusterProps(
            engine=engine,
            vpc=vpc,
            backup_retention=backup_retention,
            cluster_identifier=cluster_identifier,
            credentials=credentials,
            default_database_name=default_database_name,
            deletion_protection=deletion_protection,
            enable_data_api=enable_data_api,
            parameter_group=parameter_group,
            removal_policy=removal_policy,
            scaling=scaling,
            security_groups=security_groups,
            storage_encryption_key=storage_encryption_key,
            subnet_group=subnet_group,
            vpc_subnets=vpc_subnets,
        )

        jsii.create(ServerlessCluster, self, [scope, id, props])

    @jsii.member(jsii_name="fromServerlessClusterAttributes")
    @builtins.classmethod
    def from_serverless_cluster_attributes(
        cls,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        cluster_identifier: builtins.str,
        cluster_endpoint_address: typing.Optional[builtins.str] = None,
        port: typing.Optional[jsii.Number] = None,
        reader_endpoint_address: typing.Optional[builtins.str] = None,
        secret: typing.Optional[aws_cdk.aws_secretsmanager.ISecret] = None,
        security_groups: typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]] = None,
    ) -> IServerlessCluster:
        """(experimental) Import an existing DatabaseCluster from properties.

        :param scope: -
        :param id: -
        :param cluster_identifier: (experimental) Identifier for the cluster.
        :param cluster_endpoint_address: (experimental) Cluster endpoint address. Default: - no endpoint address
        :param port: (experimental) The database port. Default: - none
        :param reader_endpoint_address: (experimental) Reader endpoint address. Default: - no reader address
        :param secret: (experimental) The secret attached to the database cluster. Default: - no secret
        :param security_groups: (experimental) The security groups of the database cluster. Default: - no security groups

        :stability: experimental
        """
        attrs = ServerlessClusterAttributes(
            cluster_identifier=cluster_identifier,
            cluster_endpoint_address=cluster_endpoint_address,
            port=port,
            reader_endpoint_address=reader_endpoint_address,
            secret=secret,
            security_groups=security_groups,
        )

        return jsii.sinvoke(cls, "fromServerlessClusterAttributes", [scope, id, attrs])

    @jsii.member(jsii_name="addRotationMultiUser")
    def add_rotation_multi_user(
        self,
        id: builtins.str,
        *,
        secret: aws_cdk.aws_secretsmanager.ISecret,
        automatically_after: typing.Optional[aws_cdk.core.Duration] = None,
        exclude_characters: typing.Optional[builtins.str] = None,
    ) -> aws_cdk.aws_secretsmanager.SecretRotation:
        """(experimental) Adds the multi user rotation to this cluster.

        :param id: -
        :param secret: The secret to rotate. It must be a JSON string with the following format:: { "engine": <required: database engine>, "host": <required: instance host name>, "username": <required: username>, "password": <required: password>, "dbname": <optional: database name>, "port": <optional: if not specified, default port will be used>, "masterarn": <required: the arn of the master secret which will be used to create users/change passwords> }
        :param automatically_after: Specifies the number of days after the previous rotation before Secrets Manager triggers the next automatic rotation. Default: - 30 days
        :param exclude_characters: Specifies characters to not include in generated passwords. Default: " %+~`#$&*()|[]{}:;<>?!'/

        :stability: experimental
        """
        options = RotationMultiUserOptions(
            secret=secret,
            automatically_after=automatically_after,
            exclude_characters=exclude_characters,
        )

        return jsii.invoke(self, "addRotationMultiUser", [id, options])

    @jsii.member(jsii_name="addRotationSingleUser")
    def add_rotation_single_user(
        self,
        *,
        automatically_after: typing.Optional[aws_cdk.core.Duration] = None,
        exclude_characters: typing.Optional[builtins.str] = None,
    ) -> aws_cdk.aws_secretsmanager.SecretRotation:
        """(experimental) Adds the single user rotation of the master password to this cluster.

        :param automatically_after: Specifies the number of days after the previous rotation before Secrets Manager triggers the next automatic rotation. Default: - 30 days
        :param exclude_characters: Specifies characters to not include in generated passwords. Default: " %+~`#$&*()|[]{}:;<>?!'/

        :stability: experimental
        """
        options = RotationSingleUserOptions(
            automatically_after=automatically_after,
            exclude_characters=exclude_characters,
        )

        return jsii.invoke(self, "addRotationSingleUser", [options])

    @jsii.member(jsii_name="asSecretAttachmentTarget")
    def as_secret_attachment_target(
        self,
    ) -> aws_cdk.aws_secretsmanager.SecretAttachmentTargetProps:
        """(experimental) Renders the secret attachment target specifications.

        :stability: experimental
        """
        return jsii.invoke(self, "asSecretAttachmentTarget", [])

    @jsii.member(jsii_name="grantDataApiAccess")
    def grant_data_api_access(
        self,
        grantee: aws_cdk.aws_iam.IGrantable,
    ) -> aws_cdk.aws_iam.Grant:
        """(experimental) Grant the given identity to access to the Data API, including read access to the secret attached to the cluster if present.

        :param grantee: The principal to grant access to.

        :stability: experimental
        """
        return jsii.invoke(self, "grantDataApiAccess", [grantee])

    @builtins.property # type: ignore
    @jsii.member(jsii_name="clusterArn")
    def cluster_arn(self) -> builtins.str:
        """(experimental) The ARN of the cluster.

        :stability: experimental
        """
        return jsii.get(self, "clusterArn")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="clusterEndpoint")
    def cluster_endpoint(self) -> Endpoint:
        """(experimental) The endpoint to use for read/write operations.

        :stability: experimental
        """
        return jsii.get(self, "clusterEndpoint")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="clusterIdentifier")
    def cluster_identifier(self) -> builtins.str:
        """(experimental) Identifier of the cluster.

        :stability: experimental
        """
        return jsii.get(self, "clusterIdentifier")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="clusterReadEndpoint")
    def cluster_read_endpoint(self) -> Endpoint:
        """(experimental) The endpoint to use for read/write operations.

        :stability: experimental
        """
        return jsii.get(self, "clusterReadEndpoint")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="connections")
    def connections(self) -> aws_cdk.aws_ec2.Connections:
        """(experimental) Access to the network connections.

        :stability: experimental
        """
        return jsii.get(self, "connections")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="secret")
    def secret(self) -> typing.Optional[aws_cdk.aws_secretsmanager.ISecret]:
        """(experimental) The secret attached to this cluster.

        :stability: experimental
        """
        return jsii.get(self, "secret")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="enableDataApi")
    def _enable_data_api(self) -> typing.Optional[builtins.bool]:
        """
        :stability: experimental
        """
        return jsii.get(self, "enableDataApi")

    @_enable_data_api.setter # type: ignore
    def _enable_data_api(self, value: typing.Optional[builtins.bool]) -> None:
        jsii.set(self, "enableDataApi", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.ServerlessClusterAttributes",
    jsii_struct_bases=[],
    name_mapping={
        "cluster_identifier": "clusterIdentifier",
        "cluster_endpoint_address": "clusterEndpointAddress",
        "port": "port",
        "reader_endpoint_address": "readerEndpointAddress",
        "secret": "secret",
        "security_groups": "securityGroups",
    },
)
class ServerlessClusterAttributes:
    def __init__(
        self,
        *,
        cluster_identifier: builtins.str,
        cluster_endpoint_address: typing.Optional[builtins.str] = None,
        port: typing.Optional[jsii.Number] = None,
        reader_endpoint_address: typing.Optional[builtins.str] = None,
        secret: typing.Optional[aws_cdk.aws_secretsmanager.ISecret] = None,
        security_groups: typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]] = None,
    ) -> None:
        """(experimental) Properties that describe an existing cluster instance.

        :param cluster_identifier: (experimental) Identifier for the cluster.
        :param cluster_endpoint_address: (experimental) Cluster endpoint address. Default: - no endpoint address
        :param port: (experimental) The database port. Default: - none
        :param reader_endpoint_address: (experimental) Reader endpoint address. Default: - no reader address
        :param secret: (experimental) The secret attached to the database cluster. Default: - no secret
        :param security_groups: (experimental) The security groups of the database cluster. Default: - no security groups

        :stability: experimental
        """
        self._values: typing.Dict[str, typing.Any] = {
            "cluster_identifier": cluster_identifier,
        }
        if cluster_endpoint_address is not None:
            self._values["cluster_endpoint_address"] = cluster_endpoint_address
        if port is not None:
            self._values["port"] = port
        if reader_endpoint_address is not None:
            self._values["reader_endpoint_address"] = reader_endpoint_address
        if secret is not None:
            self._values["secret"] = secret
        if security_groups is not None:
            self._values["security_groups"] = security_groups

    @builtins.property
    def cluster_identifier(self) -> builtins.str:
        """(experimental) Identifier for the cluster.

        :stability: experimental
        """
        result = self._values.get("cluster_identifier")
        assert result is not None, "Required property 'cluster_identifier' is missing"
        return result

    @builtins.property
    def cluster_endpoint_address(self) -> typing.Optional[builtins.str]:
        """(experimental) Cluster endpoint address.

        :default: - no endpoint address

        :stability: experimental
        """
        result = self._values.get("cluster_endpoint_address")
        return result

    @builtins.property
    def port(self) -> typing.Optional[jsii.Number]:
        """(experimental) The database port.

        :default: - none

        :stability: experimental
        """
        result = self._values.get("port")
        return result

    @builtins.property
    def reader_endpoint_address(self) -> typing.Optional[builtins.str]:
        """(experimental) Reader endpoint address.

        :default: - no reader address

        :stability: experimental
        """
        result = self._values.get("reader_endpoint_address")
        return result

    @builtins.property
    def secret(self) -> typing.Optional[aws_cdk.aws_secretsmanager.ISecret]:
        """(experimental) The secret attached to the database cluster.

        :default: - no secret

        :stability: experimental
        """
        result = self._values.get("secret")
        return result

    @builtins.property
    def security_groups(
        self,
    ) -> typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]]:
        """(experimental) The security groups of the database cluster.

        :default: - no security groups

        :stability: experimental
        """
        result = self._values.get("security_groups")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ServerlessClusterAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.ServerlessClusterProps",
    jsii_struct_bases=[],
    name_mapping={
        "engine": "engine",
        "vpc": "vpc",
        "backup_retention": "backupRetention",
        "cluster_identifier": "clusterIdentifier",
        "credentials": "credentials",
        "default_database_name": "defaultDatabaseName",
        "deletion_protection": "deletionProtection",
        "enable_data_api": "enableDataApi",
        "parameter_group": "parameterGroup",
        "removal_policy": "removalPolicy",
        "scaling": "scaling",
        "security_groups": "securityGroups",
        "storage_encryption_key": "storageEncryptionKey",
        "subnet_group": "subnetGroup",
        "vpc_subnets": "vpcSubnets",
    },
)
class ServerlessClusterProps:
    def __init__(
        self,
        *,
        engine: "IClusterEngine",
        vpc: aws_cdk.aws_ec2.IVpc,
        backup_retention: typing.Optional[aws_cdk.core.Duration] = None,
        cluster_identifier: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[Credentials] = None,
        default_database_name: typing.Optional[builtins.str] = None,
        deletion_protection: typing.Optional[builtins.bool] = None,
        enable_data_api: typing.Optional[builtins.bool] = None,
        parameter_group: typing.Optional[IParameterGroup] = None,
        removal_policy: typing.Optional[aws_cdk.core.RemovalPolicy] = None,
        scaling: typing.Optional["ServerlessScalingOptions"] = None,
        security_groups: typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]] = None,
        storage_encryption_key: typing.Optional[aws_cdk.aws_kms.IKey] = None,
        subnet_group: typing.Optional[ISubnetGroup] = None,
        vpc_subnets: typing.Optional[aws_cdk.aws_ec2.SubnetSelection] = None,
    ) -> None:
        """(experimental) Properties to configure an Aurora Serverless Cluster.

        :param engine: (experimental) What kind of database to start.
        :param vpc: (experimental) The VPC that this Aurora Serverless cluster has been created in.
        :param backup_retention: (experimental) The number of days during which automatic DB snapshots are retained. Automatic backup retention cannot be disabled on serverless clusters. Must be a value from 1 day to 35 days. Default: Duration.days(1)
        :param cluster_identifier: (experimental) An optional identifier for the cluster. Default: - A name is automatically generated.
        :param credentials: (experimental) Credentials for the administrative user. Default: - A username of 'admin' and SecretsManager-generated password
        :param default_database_name: (experimental) Name of a database which is automatically created inside the cluster. Default: - Database is not created in cluster.
        :param deletion_protection: (experimental) Indicates whether the DB cluster should have deletion protection enabled. Default: - true if removalPolicy is RETAIN, false otherwise
        :param enable_data_api: (experimental) Whether to enable the Data API. Default: false
        :param parameter_group: (experimental) Additional parameters to pass to the database engine. Default: - no parameter group.
        :param removal_policy: (experimental) The removal policy to apply when the cluster and its instances are removed from the stack or replaced during an update. Default: - RemovalPolicy.SNAPSHOT (remove the cluster and instances, but retain a snapshot of the data)
        :param scaling: (experimental) Scaling configuration of an Aurora Serverless database cluster. Default: - Serverless cluster is automatically paused after 5 minutes of being idle. minimum capacity: 2 ACU maximum capacity: 16 ACU
        :param security_groups: (experimental) Security group. Default: - a new security group is created.
        :param storage_encryption_key: (experimental) The KMS key for storage encryption. Default: - the default master key will be used for storage encryption
        :param subnet_group: (experimental) Existing subnet group for the cluster. Default: - a new subnet group will be created.
        :param vpc_subnets: (experimental) Where to place the instances within the VPC. Default: - the VPC default strategy if not specified.

        :stability: experimental
        """
        if isinstance(scaling, dict):
            scaling = ServerlessScalingOptions(**scaling)
        if isinstance(vpc_subnets, dict):
            vpc_subnets = aws_cdk.aws_ec2.SubnetSelection(**vpc_subnets)
        self._values: typing.Dict[str, typing.Any] = {
            "engine": engine,
            "vpc": vpc,
        }
        if backup_retention is not None:
            self._values["backup_retention"] = backup_retention
        if cluster_identifier is not None:
            self._values["cluster_identifier"] = cluster_identifier
        if credentials is not None:
            self._values["credentials"] = credentials
        if default_database_name is not None:
            self._values["default_database_name"] = default_database_name
        if deletion_protection is not None:
            self._values["deletion_protection"] = deletion_protection
        if enable_data_api is not None:
            self._values["enable_data_api"] = enable_data_api
        if parameter_group is not None:
            self._values["parameter_group"] = parameter_group
        if removal_policy is not None:
            self._values["removal_policy"] = removal_policy
        if scaling is not None:
            self._values["scaling"] = scaling
        if security_groups is not None:
            self._values["security_groups"] = security_groups
        if storage_encryption_key is not None:
            self._values["storage_encryption_key"] = storage_encryption_key
        if subnet_group is not None:
            self._values["subnet_group"] = subnet_group
        if vpc_subnets is not None:
            self._values["vpc_subnets"] = vpc_subnets

    @builtins.property
    def engine(self) -> "IClusterEngine":
        """(experimental) What kind of database to start.

        :stability: experimental
        """
        result = self._values.get("engine")
        assert result is not None, "Required property 'engine' is missing"
        return result

    @builtins.property
    def vpc(self) -> aws_cdk.aws_ec2.IVpc:
        """(experimental) The VPC that this Aurora Serverless cluster has been created in.

        :stability: experimental
        """
        result = self._values.get("vpc")
        assert result is not None, "Required property 'vpc' is missing"
        return result

    @builtins.property
    def backup_retention(self) -> typing.Optional[aws_cdk.core.Duration]:
        """(experimental) The number of days during which automatic DB snapshots are retained.

        Automatic backup retention cannot be disabled on serverless clusters.
        Must be a value from 1 day to 35 days.

        :default: Duration.days(1)

        :stability: experimental
        """
        result = self._values.get("backup_retention")
        return result

    @builtins.property
    def cluster_identifier(self) -> typing.Optional[builtins.str]:
        """(experimental) An optional identifier for the cluster.

        :default: - A name is automatically generated.

        :stability: experimental
        """
        result = self._values.get("cluster_identifier")
        return result

    @builtins.property
    def credentials(self) -> typing.Optional[Credentials]:
        """(experimental) Credentials for the administrative user.

        :default: - A username of 'admin' and SecretsManager-generated password

        :stability: experimental
        """
        result = self._values.get("credentials")
        return result

    @builtins.property
    def default_database_name(self) -> typing.Optional[builtins.str]:
        """(experimental) Name of a database which is automatically created inside the cluster.

        :default: - Database is not created in cluster.

        :stability: experimental
        """
        result = self._values.get("default_database_name")
        return result

    @builtins.property
    def deletion_protection(self) -> typing.Optional[builtins.bool]:
        """(experimental) Indicates whether the DB cluster should have deletion protection enabled.

        :default: - true if removalPolicy is RETAIN, false otherwise

        :stability: experimental
        """
        result = self._values.get("deletion_protection")
        return result

    @builtins.property
    def enable_data_api(self) -> typing.Optional[builtins.bool]:
        """(experimental) Whether to enable the Data API.

        :default: false

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/data-api.html
        :stability: experimental
        """
        result = self._values.get("enable_data_api")
        return result

    @builtins.property
    def parameter_group(self) -> typing.Optional[IParameterGroup]:
        """(experimental) Additional parameters to pass to the database engine.

        :default: - no parameter group.

        :stability: experimental
        """
        result = self._values.get("parameter_group")
        return result

    @builtins.property
    def removal_policy(self) -> typing.Optional[aws_cdk.core.RemovalPolicy]:
        """(experimental) The removal policy to apply when the cluster and its instances are removed from the stack or replaced during an update.

        :default: - RemovalPolicy.SNAPSHOT (remove the cluster and instances, but retain a snapshot of the data)

        :stability: experimental
        """
        result = self._values.get("removal_policy")
        return result

    @builtins.property
    def scaling(self) -> typing.Optional["ServerlessScalingOptions"]:
        """(experimental) Scaling configuration of an Aurora Serverless database cluster.

        :default:

        - Serverless cluster is automatically paused after 5 minutes of being idle.
        minimum capacity: 2 ACU
        maximum capacity: 16 ACU

        :stability: experimental
        """
        result = self._values.get("scaling")
        return result

    @builtins.property
    def security_groups(
        self,
    ) -> typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]]:
        """(experimental) Security group.

        :default: - a new security group is created.

        :stability: experimental
        """
        result = self._values.get("security_groups")
        return result

    @builtins.property
    def storage_encryption_key(self) -> typing.Optional[aws_cdk.aws_kms.IKey]:
        """(experimental) The KMS key for storage encryption.

        :default: - the default master key will be used for storage encryption

        :stability: experimental
        """
        result = self._values.get("storage_encryption_key")
        return result

    @builtins.property
    def subnet_group(self) -> typing.Optional[ISubnetGroup]:
        """(experimental) Existing subnet group for the cluster.

        :default: - a new subnet group will be created.

        :stability: experimental
        """
        result = self._values.get("subnet_group")
        return result

    @builtins.property
    def vpc_subnets(self) -> typing.Optional[aws_cdk.aws_ec2.SubnetSelection]:
        """(experimental) Where to place the instances within the VPC.

        :default: - the VPC default strategy if not specified.

        :stability: experimental
        """
        result = self._values.get("vpc_subnets")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ServerlessClusterProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.ServerlessScalingOptions",
    jsii_struct_bases=[],
    name_mapping={
        "auto_pause": "autoPause",
        "max_capacity": "maxCapacity",
        "min_capacity": "minCapacity",
    },
)
class ServerlessScalingOptions:
    def __init__(
        self,
        *,
        auto_pause: typing.Optional[aws_cdk.core.Duration] = None,
        max_capacity: typing.Optional[AuroraCapacityUnit] = None,
        min_capacity: typing.Optional[AuroraCapacityUnit] = None,
    ) -> None:
        """(experimental) Options for configuring scaling on an Aurora Serverless cluster.

        :param auto_pause: (experimental) The time before an Aurora Serverless database cluster is paused. A database cluster can be paused only when it is idle (it has no connections). Auto pause time must be between 5 minutes and 1 day. If a DB cluster is paused for more than seven days, the DB cluster might be backed up with a snapshot. In this case, the DB cluster is restored when there is a request to connect to it. Set to 0 to disable Default: - automatic pause enabled after 5 minutes
        :param max_capacity: (experimental) The maximum capacity for an Aurora Serverless database cluster. Default: - determined by Aurora based on database engine
        :param min_capacity: (experimental) The minimum capacity for an Aurora Serverless database cluster. Default: - determined by Aurora based on database engine

        :stability: experimental
        """
        self._values: typing.Dict[str, typing.Any] = {}
        if auto_pause is not None:
            self._values["auto_pause"] = auto_pause
        if max_capacity is not None:
            self._values["max_capacity"] = max_capacity
        if min_capacity is not None:
            self._values["min_capacity"] = min_capacity

    @builtins.property
    def auto_pause(self) -> typing.Optional[aws_cdk.core.Duration]:
        """(experimental) The time before an Aurora Serverless database cluster is paused.

        A database cluster can be paused only when it is idle (it has no connections).
        Auto pause time must be between 5 minutes and 1 day.

        If a DB cluster is paused for more than seven days, the DB cluster might be
        backed up with a snapshot. In this case, the DB cluster is restored when there
        is a request to connect to it.

        Set to 0 to disable

        :default: - automatic pause enabled after 5 minutes

        :stability: experimental
        """
        result = self._values.get("auto_pause")
        return result

    @builtins.property
    def max_capacity(self) -> typing.Optional[AuroraCapacityUnit]:
        """(experimental) The maximum capacity for an Aurora Serverless database cluster.

        :default: - determined by Aurora based on database engine

        :stability: experimental
        """
        result = self._values.get("max_capacity")
        return result

    @builtins.property
    def min_capacity(self) -> typing.Optional[AuroraCapacityUnit]:
        """(experimental) The minimum capacity for an Aurora Serverless database cluster.

        :default: - determined by Aurora based on database engine

        :stability: experimental
        """
        result = self._values.get("min_capacity")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ServerlessScalingOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class SessionPinningFilter(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.SessionPinningFilter",
):
    """SessionPinningFilter.

    :see: https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html#rds-proxy-pinning
    """

    @jsii.member(jsii_name="of")
    @builtins.classmethod
    def of(cls, filter_name: builtins.str) -> "SessionPinningFilter":
        """custom filter.

        :param filter_name: -
        """
        return jsii.sinvoke(cls, "of", [filter_name])

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="EXCLUDE_VARIABLE_SETS")
    def EXCLUDE_VARIABLE_SETS(cls) -> "SessionPinningFilter":
        """You can opt out of session pinning for the following kinds of application statements:.

        - Setting session variables and configuration settings.
        """
        return jsii.sget(cls, "EXCLUDE_VARIABLE_SETS")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="filterName")
    def filter_name(self) -> builtins.str:
        """Filter name."""
        return jsii.get(self, "filterName")


class SnapshotCredentials(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@aws-cdk/aws-rds.SnapshotCredentials",
):
    """Credentials to update the password for a ``DatabaseInstanceFromSnapshot``."""

    @builtins.staticmethod
    def __jsii_proxy_class__():
        return _SnapshotCredentialsProxy

    def __init__(self) -> None:
        jsii.create(SnapshotCredentials, self, [])

    @jsii.member(jsii_name="fromGeneratedPassword")
    @builtins.classmethod
    def from_generated_password(
        cls,
        username: builtins.str,
        *,
        encryption_key: typing.Optional[aws_cdk.aws_kms.IKey] = None,
        exclude_characters: typing.Optional[builtins.str] = None,
    ) -> "SnapshotCredentials":
        """(deprecated) Generate a new password for the snapshot, using the existing username and an optional encryption key.

        Note - The username must match the existing master username of the snapshot.

        :param username: -
        :param encryption_key: KMS encryption key to encrypt the generated secret. Default: - default master key
        :param exclude_characters: The characters to exclude from the generated password. Default: - the DatabaseSecret default exclude character set (" %+~`#$&*()|[]{}:;<>?!'/

        :deprecated:

        use ``fromGeneratedSecret()`` for new Clusters and Instances.
        Note that switching from ``fromGeneratedPassword()`` to ``fromGeneratedSecret()`` for already deployed
        Clusters or Instances will update their master password.

        :stability: deprecated
        """
        options = SnapshotCredentialsFromGeneratedPasswordOptions(
            encryption_key=encryption_key, exclude_characters=exclude_characters
        )

        return jsii.sinvoke(cls, "fromGeneratedPassword", [username, options])

    @jsii.member(jsii_name="fromGeneratedSecret")
    @builtins.classmethod
    def from_generated_secret(
        cls,
        username: builtins.str,
        *,
        encryption_key: typing.Optional[aws_cdk.aws_kms.IKey] = None,
        exclude_characters: typing.Optional[builtins.str] = None,
    ) -> "SnapshotCredentials":
        """Generate a new password for the snapshot, using the existing username and an optional encryption key.

        The new credentials are stored in Secrets Manager.

        Note - The username must match the existing master username of the snapshot.

        :param username: -
        :param encryption_key: KMS encryption key to encrypt the generated secret. Default: - default master key
        :param exclude_characters: The characters to exclude from the generated password. Default: - the DatabaseSecret default exclude character set (" %+~`#$&*()|[]{}:;<>?!'/
        """
        options = SnapshotCredentialsFromGeneratedPasswordOptions(
            encryption_key=encryption_key, exclude_characters=exclude_characters
        )

        return jsii.sinvoke(cls, "fromGeneratedSecret", [username, options])

    @jsii.member(jsii_name="fromPassword")
    @builtins.classmethod
    def from_password(cls, password: aws_cdk.core.SecretValue) -> "SnapshotCredentials":
        """Update the snapshot login with an existing password.

        :param password: -
        """
        return jsii.sinvoke(cls, "fromPassword", [password])

    @jsii.member(jsii_name="fromSecret")
    @builtins.classmethod
    def from_secret(
        cls,
        secret: aws_cdk.aws_secretsmanager.Secret,
    ) -> "SnapshotCredentials":
        """Update the snapshot login with an existing password from a Secret.

        The Secret must be a JSON string with a ``password`` field::

           {
              ...
              "password": <required: password>,
           }

        :param secret: -
        """
        return jsii.sinvoke(cls, "fromSecret", [secret])

    @builtins.property # type: ignore
    @jsii.member(jsii_name="generatePassword")
    @abc.abstractmethod
    def generate_password(self) -> builtins.bool:
        """Whether a new password should be generated."""
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="encryptionKey")
    @abc.abstractmethod
    def encryption_key(self) -> typing.Optional[aws_cdk.aws_kms.IKey]:
        """KMS encryption key to encrypt the generated secret.

        :default: - default master key
        """
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="excludeCharacters")
    @abc.abstractmethod
    def exclude_characters(self) -> typing.Optional[builtins.str]:
        """The characters to exclude from the generated password.

        Only used if {@link generatePassword} if true.

        :default: - the DatabaseSecret default exclude character set (" %+~`#$&*()|[]{}:;<>?!'/

        :: "\\")
        """
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="password")
    @abc.abstractmethod
    def password(self) -> typing.Optional[aws_cdk.core.SecretValue]:
        """The master user password.

        Do not put passwords in your CDK code directly.

        :default: - the existing password from the snapshot
        """
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="replaceOnPasswordCriteriaChanges")
    @abc.abstractmethod
    def replace_on_password_criteria_changes(self) -> typing.Optional[builtins.bool]:
        """Whether to replace the generated secret when the criteria for the password change.

        :default: false
        """
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="secret")
    @abc.abstractmethod
    def secret(self) -> typing.Optional[aws_cdk.aws_secretsmanager.Secret]:
        """Secret used to instantiate this Login.

        :default: - none
        """
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="username")
    @abc.abstractmethod
    def username(self) -> typing.Optional[builtins.str]:
        """The master user name.

        Must be the **current** master user name of the snapshot.
        It is not possible to change the master user name of a RDS instance.

        :default: - the existing username from the snapshot
        """
        ...


class _SnapshotCredentialsProxy(SnapshotCredentials):
    @builtins.property # type: ignore
    @jsii.member(jsii_name="generatePassword")
    def generate_password(self) -> builtins.bool:
        """Whether a new password should be generated."""
        return jsii.get(self, "generatePassword")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="encryptionKey")
    def encryption_key(self) -> typing.Optional[aws_cdk.aws_kms.IKey]:
        """KMS encryption key to encrypt the generated secret.

        :default: - default master key
        """
        return jsii.get(self, "encryptionKey")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="excludeCharacters")
    def exclude_characters(self) -> typing.Optional[builtins.str]:
        """The characters to exclude from the generated password.

        Only used if {@link generatePassword} if true.

        :default: - the DatabaseSecret default exclude character set (" %+~`#$&*()|[]{}:;<>?!'/

        :: "\\")
        """
        return jsii.get(self, "excludeCharacters")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="password")
    def password(self) -> typing.Optional[aws_cdk.core.SecretValue]:
        """The master user password.

        Do not put passwords in your CDK code directly.

        :default: - the existing password from the snapshot
        """
        return jsii.get(self, "password")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="replaceOnPasswordCriteriaChanges")
    def replace_on_password_criteria_changes(self) -> typing.Optional[builtins.bool]:
        """Whether to replace the generated secret when the criteria for the password change.

        :default: false
        """
        return jsii.get(self, "replaceOnPasswordCriteriaChanges")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="secret")
    def secret(self) -> typing.Optional[aws_cdk.aws_secretsmanager.Secret]:
        """Secret used to instantiate this Login.

        :default: - none
        """
        return jsii.get(self, "secret")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="username")
    def username(self) -> typing.Optional[builtins.str]:
        """The master user name.

        Must be the **current** master user name of the snapshot.
        It is not possible to change the master user name of a RDS instance.

        :default: - the existing username from the snapshot
        """
        return jsii.get(self, "username")


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.SnapshotCredentialsFromGeneratedPasswordOptions",
    jsii_struct_bases=[],
    name_mapping={
        "encryption_key": "encryptionKey",
        "exclude_characters": "excludeCharacters",
    },
)
class SnapshotCredentialsFromGeneratedPasswordOptions:
    def __init__(
        self,
        *,
        encryption_key: typing.Optional[aws_cdk.aws_kms.IKey] = None,
        exclude_characters: typing.Optional[builtins.str] = None,
    ) -> None:
        """Options used in the {@link SnapshotCredentials.fromGeneratedPassword} method.

        :param encryption_key: KMS encryption key to encrypt the generated secret. Default: - default master key
        :param exclude_characters: The characters to exclude from the generated password. Default: - the DatabaseSecret default exclude character set (" %+~`#$&*()|[]{}:;<>?!'/
        """
        self._values: typing.Dict[str, typing.Any] = {}
        if encryption_key is not None:
            self._values["encryption_key"] = encryption_key
        if exclude_characters is not None:
            self._values["exclude_characters"] = exclude_characters

    @builtins.property
    def encryption_key(self) -> typing.Optional[aws_cdk.aws_kms.IKey]:
        """KMS encryption key to encrypt the generated secret.

        :default: - default master key
        """
        result = self._values.get("encryption_key")
        return result

    @builtins.property
    def exclude_characters(self) -> typing.Optional[builtins.str]:
        """The characters to exclude from the generated password.

        :default: - the DatabaseSecret default exclude character set (" %+~`#$&*()|[]{}:;<>?!'/

        :: "\\")
        """
        result = self._values.get("exclude_characters")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "SnapshotCredentialsFromGeneratedPasswordOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.SqlServerEeInstanceEngineProps",
    jsii_struct_bases=[],
    name_mapping={"version": "version"},
)
class SqlServerEeInstanceEngineProps:
    def __init__(self, *, version: "SqlServerEngineVersion") -> None:
        """Properties for SQL Server Enterprise Edition instance engines.

        Used in {@link DatabaseInstanceEngine.sqlServerEe}.

        :param version: The exact version of the engine to use.
        """
        self._values: typing.Dict[str, typing.Any] = {
            "version": version,
        }

    @builtins.property
    def version(self) -> "SqlServerEngineVersion":
        """The exact version of the engine to use."""
        result = self._values.get("version")
        assert result is not None, "Required property 'version' is missing"
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "SqlServerEeInstanceEngineProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class SqlServerEngineVersion(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.SqlServerEngineVersion",
):
    """The versions for the SQL Server instance engines (those returned by {@link DatabaseInstanceEngine.sqlServerSe}, {@link DatabaseInstanceEngine.sqlServerEx}, {@link DatabaseInstanceEngine.sqlServerWeb} and {@link DatabaseInstanceEngine.sqlServerEe})."""

    @jsii.member(jsii_name="of")
    @builtins.classmethod
    def of(
        cls,
        sql_server_full_version: builtins.str,
        sql_server_major_version: builtins.str,
    ) -> "SqlServerEngineVersion":
        """Create a new SqlServerEngineVersion with an arbitrary version.

        :param sql_server_full_version: the full version string, for example "15.00.3049.1.v1".
        :param sql_server_major_version: the major version of the engine, for example "15.00".
        """
        return jsii.sinvoke(cls, "of", [sql_server_full_version, sql_server_major_version])

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11")
    def VER_11(cls) -> "SqlServerEngineVersion":
        """Version "11.00" (only a major version, without a specific minor version)."""
        return jsii.sget(cls, "VER_11")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_00_5058_0_V1")
    def VER_11_00_5058_0_V1(cls) -> "SqlServerEngineVersion":
        """Version "11.00.5058.0.v1"."""
        return jsii.sget(cls, "VER_11_00_5058_0_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_00_6020_0_V1")
    def VER_11_00_6020_0_V1(cls) -> "SqlServerEngineVersion":
        """Version "11.00.6020.0.v1"."""
        return jsii.sget(cls, "VER_11_00_6020_0_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_00_6594_0_V1")
    def VER_11_00_6594_0_V1(cls) -> "SqlServerEngineVersion":
        """Version "11.00.6594.0.v1"."""
        return jsii.sget(cls, "VER_11_00_6594_0_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_00_7462_6_V1")
    def VER_11_00_7462_6_V1(cls) -> "SqlServerEngineVersion":
        """Version "11.00.7462.6.v1"."""
        return jsii.sget(cls, "VER_11_00_7462_6_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_11_00_7493_4_V1")
    def VER_11_00_7493_4_V1(cls) -> "SqlServerEngineVersion":
        """Version "11.00.7493.4.v1"."""
        return jsii.sget(cls, "VER_11_00_7493_4_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12")
    def VER_12(cls) -> "SqlServerEngineVersion":
        """Version "12.00" (only a major version, without a specific minor version)."""
        return jsii.sget(cls, "VER_12")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_00_5000_0_V1")
    def VER_12_00_5000_0_V1(cls) -> "SqlServerEngineVersion":
        """Version "12.00.5000.0.v1"."""
        return jsii.sget(cls, "VER_12_00_5000_0_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_00_5546_0_V1")
    def VER_12_00_5546_0_V1(cls) -> "SqlServerEngineVersion":
        """Version "12.00.5546.0.v1"."""
        return jsii.sget(cls, "VER_12_00_5546_0_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_00_5571_0_V1")
    def VER_12_00_5571_0_V1(cls) -> "SqlServerEngineVersion":
        """Version "12.00.5571.0.v1"."""
        return jsii.sget(cls, "VER_12_00_5571_0_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_00_6293_0_V1")
    def VER_12_00_6293_0_V1(cls) -> "SqlServerEngineVersion":
        """Version "12.00.6293.0.v1"."""
        return jsii.sget(cls, "VER_12_00_6293_0_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_12_00_6329_1_V1")
    def VER_12_00_6329_1_V1(cls) -> "SqlServerEngineVersion":
        """Version "12.00.6329.1.v1"."""
        return jsii.sget(cls, "VER_12_00_6329_1_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_13")
    def VER_13(cls) -> "SqlServerEngineVersion":
        """Version "13.00" (only a major version, without a specific minor version)."""
        return jsii.sget(cls, "VER_13")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_13_00_2164_0_V1")
    def VER_13_00_2164_0_V1(cls) -> "SqlServerEngineVersion":
        """Version "13.00.2164.0.v1"."""
        return jsii.sget(cls, "VER_13_00_2164_0_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_13_00_4422_0_V1")
    def VER_13_00_4422_0_V1(cls) -> "SqlServerEngineVersion":
        """Version "13.00.4422.0.v1"."""
        return jsii.sget(cls, "VER_13_00_4422_0_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_13_00_4451_0_V1")
    def VER_13_00_4451_0_V1(cls) -> "SqlServerEngineVersion":
        """Version "13.00.4451.0.v1"."""
        return jsii.sget(cls, "VER_13_00_4451_0_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_13_00_4466_4_V1")
    def VER_13_00_4466_4_V1(cls) -> "SqlServerEngineVersion":
        """Version "13.00.4466.4.v1"."""
        return jsii.sget(cls, "VER_13_00_4466_4_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_13_00_4522_0_V1")
    def VER_13_00_4522_0_V1(cls) -> "SqlServerEngineVersion":
        """Version "13.00.4522.0.v1"."""
        return jsii.sget(cls, "VER_13_00_4522_0_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_13_00_5216_0_V1")
    def VER_13_00_5216_0_V1(cls) -> "SqlServerEngineVersion":
        """Version "13.00.5216.0.v1"."""
        return jsii.sget(cls, "VER_13_00_5216_0_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_13_00_5292_0_V1")
    def VER_13_00_5292_0_V1(cls) -> "SqlServerEngineVersion":
        """Version "13.00.5292.0.v1"."""
        return jsii.sget(cls, "VER_13_00_5292_0_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_13_00_5366_0_V1")
    def VER_13_00_5366_0_V1(cls) -> "SqlServerEngineVersion":
        """Version "13.00.5366.0.v1"."""
        return jsii.sget(cls, "VER_13_00_5366_0_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_13_00_5426_0_V1")
    def VER_13_00_5426_0_V1(cls) -> "SqlServerEngineVersion":
        """Version "13.00.5426.0.v1"."""
        return jsii.sget(cls, "VER_13_00_5426_0_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_13_00_5598_27_V1")
    def VER_13_00_5598_27_V1(cls) -> "SqlServerEngineVersion":
        """Version "13.00.5598.27.v1"."""
        return jsii.sget(cls, "VER_13_00_5598_27_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_13_00_5820_21_V1")
    def VER_13_00_5820_21_V1(cls) -> "SqlServerEngineVersion":
        """Version "13.00.5820.21.v1"."""
        return jsii.sget(cls, "VER_13_00_5820_21_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_14")
    def VER_14(cls) -> "SqlServerEngineVersion":
        """Version "14.00" (only a major version, without a specific minor version)."""
        return jsii.sget(cls, "VER_14")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_14_00_1000_169_V1")
    def VER_14_00_1000_169_V1(cls) -> "SqlServerEngineVersion":
        """Version "14.00.1000.169.v1"."""
        return jsii.sget(cls, "VER_14_00_1000_169_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_14_00_3015_40_V1")
    def VER_14_00_3015_40_V1(cls) -> "SqlServerEngineVersion":
        """Version "14.00.3015.40.v1"."""
        return jsii.sget(cls, "VER_14_00_3015_40_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_14_00_3035_2_V1")
    def VER_14_00_3035_2_V1(cls) -> "SqlServerEngineVersion":
        """Version "14.00.3035.2.v1"."""
        return jsii.sget(cls, "VER_14_00_3035_2_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_14_00_3049_1_V1")
    def VER_14_00_3049_1_V1(cls) -> "SqlServerEngineVersion":
        """Version "14.00.3049.1.v1"."""
        return jsii.sget(cls, "VER_14_00_3049_1_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_14_00_3192_2_V1")
    def VER_14_00_3192_2_V1(cls) -> "SqlServerEngineVersion":
        """Version "14.00.3192.2.v1"."""
        return jsii.sget(cls, "VER_14_00_3192_2_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_14_00_3223_3_V1")
    def VER_14_00_3223_3_V1(cls) -> "SqlServerEngineVersion":
        """Version "14.00.3223.3.v1"."""
        return jsii.sget(cls, "VER_14_00_3223_3_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_14_00_3281_6_V1")
    def VER_14_00_3281_6_V1(cls) -> "SqlServerEngineVersion":
        """Version "14.00.3281.6.v1"."""
        return jsii.sget(cls, "VER_14_00_3281_6_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_14_00_3294_2_V1")
    def VER_14_00_3294_2_V1(cls) -> "SqlServerEngineVersion":
        """Version "14.00.3294.2.v1"."""
        return jsii.sget(cls, "VER_14_00_3294_2_V1")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_15")
    def VER_15(cls) -> "SqlServerEngineVersion":
        """Version "15.00" (only a major version, without a specific minor version)."""
        return jsii.sget(cls, "VER_15")

    @jsii.python.classproperty # type: ignore
    @jsii.member(jsii_name="VER_15_00_4043_16_V1")
    def VER_15_00_4043_16_V1(cls) -> "SqlServerEngineVersion":
        """Version "15.00.4043.16.v1"."""
        return jsii.sget(cls, "VER_15_00_4043_16_V1")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="sqlServerFullVersion")
    def sql_server_full_version(self) -> builtins.str:
        """The full version string, for example, "15.00.3049.1.v1"."""
        return jsii.get(self, "sqlServerFullVersion")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="sqlServerMajorVersion")
    def sql_server_major_version(self) -> builtins.str:
        """The major version of the engine, for example, "15.00"."""
        return jsii.get(self, "sqlServerMajorVersion")


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.SqlServerExInstanceEngineProps",
    jsii_struct_bases=[],
    name_mapping={"version": "version"},
)
class SqlServerExInstanceEngineProps:
    def __init__(self, *, version: SqlServerEngineVersion) -> None:
        """Properties for SQL Server Express Edition instance engines.

        Used in {@link DatabaseInstanceEngine.sqlServerEx}.

        :param version: The exact version of the engine to use.
        """
        self._values: typing.Dict[str, typing.Any] = {
            "version": version,
        }

    @builtins.property
    def version(self) -> SqlServerEngineVersion:
        """The exact version of the engine to use."""
        result = self._values.get("version")
        assert result is not None, "Required property 'version' is missing"
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "SqlServerExInstanceEngineProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.SqlServerSeInstanceEngineProps",
    jsii_struct_bases=[],
    name_mapping={"version": "version"},
)
class SqlServerSeInstanceEngineProps:
    def __init__(self, *, version: SqlServerEngineVersion) -> None:
        """Properties for SQL Server Standard Edition instance engines.

        Used in {@link DatabaseInstanceEngine.sqlServerSe}.

        :param version: The exact version of the engine to use.
        """
        self._values: typing.Dict[str, typing.Any] = {
            "version": version,
        }

    @builtins.property
    def version(self) -> SqlServerEngineVersion:
        """The exact version of the engine to use."""
        result = self._values.get("version")
        assert result is not None, "Required property 'version' is missing"
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "SqlServerSeInstanceEngineProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.SqlServerWebInstanceEngineProps",
    jsii_struct_bases=[],
    name_mapping={"version": "version"},
)
class SqlServerWebInstanceEngineProps:
    def __init__(self, *, version: SqlServerEngineVersion) -> None:
        """Properties for SQL Server Web Edition instance engines.

        Used in {@link DatabaseInstanceEngine.sqlServerWeb}.

        :param version: The exact version of the engine to use.
        """
        self._values: typing.Dict[str, typing.Any] = {
            "version": version,
        }

    @builtins.property
    def version(self) -> SqlServerEngineVersion:
        """The exact version of the engine to use."""
        result = self._values.get("version")
        assert result is not None, "Required property 'version' is missing"
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "SqlServerWebInstanceEngineProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="@aws-cdk/aws-rds.StorageType")
class StorageType(enum.Enum):
    """The type of storage."""

    STANDARD = "STANDARD"
    """Standard."""
    GP2 = "GP2"
    """General purpose (SSD)."""
    IO1 = "IO1"
    """Provisioned IOPS (SSD)."""


@jsii.implements(ISubnetGroup)
class SubnetGroup(
    aws_cdk.core.Resource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.SubnetGroup",
):
    """Class for creating a RDS DB subnet group.

    :resource: AWS::RDS::DBSubnetGroup
    """

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        description: builtins.str,
        vpc: aws_cdk.aws_ec2.IVpc,
        removal_policy: typing.Optional[aws_cdk.core.RemovalPolicy] = None,
        subnet_group_name: typing.Optional[builtins.str] = None,
        vpc_subnets: typing.Optional[aws_cdk.aws_ec2.SubnetSelection] = None,
    ) -> None:
        """
        :param scope: -
        :param id: -
        :param description: Description of the subnet group.
        :param vpc: The VPC to place the subnet group in.
        :param removal_policy: The removal policy to apply when the subnet group are removed from the stack or replaced during an update. Default: RemovalPolicy.DESTROY
        :param subnet_group_name: The name of the subnet group. Default: - a name is generated
        :param vpc_subnets: Which subnets within the VPC to associate with this group. Default: - private subnets
        """
        props = SubnetGroupProps(
            description=description,
            vpc=vpc,
            removal_policy=removal_policy,
            subnet_group_name=subnet_group_name,
            vpc_subnets=vpc_subnets,
        )

        jsii.create(SubnetGroup, self, [scope, id, props])

    @jsii.member(jsii_name="fromSubnetGroupName")
    @builtins.classmethod
    def from_subnet_group_name(
        cls,
        scope: constructs.Construct,
        id: builtins.str,
        subnet_group_name: builtins.str,
    ) -> ISubnetGroup:
        """Imports an existing subnet group by name.

        :param scope: -
        :param id: -
        :param subnet_group_name: -
        """
        return jsii.sinvoke(cls, "fromSubnetGroupName", [scope, id, subnet_group_name])

    @builtins.property # type: ignore
    @jsii.member(jsii_name="subnetGroupName")
    def subnet_group_name(self) -> builtins.str:
        """The name of the subnet group."""
        return jsii.get(self, "subnetGroupName")


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.SubnetGroupProps",
    jsii_struct_bases=[],
    name_mapping={
        "description": "description",
        "vpc": "vpc",
        "removal_policy": "removalPolicy",
        "subnet_group_name": "subnetGroupName",
        "vpc_subnets": "vpcSubnets",
    },
)
class SubnetGroupProps:
    def __init__(
        self,
        *,
        description: builtins.str,
        vpc: aws_cdk.aws_ec2.IVpc,
        removal_policy: typing.Optional[aws_cdk.core.RemovalPolicy] = None,
        subnet_group_name: typing.Optional[builtins.str] = None,
        vpc_subnets: typing.Optional[aws_cdk.aws_ec2.SubnetSelection] = None,
    ) -> None:
        """Properties for creating a SubnetGroup.

        :param description: Description of the subnet group.
        :param vpc: The VPC to place the subnet group in.
        :param removal_policy: The removal policy to apply when the subnet group are removed from the stack or replaced during an update. Default: RemovalPolicy.DESTROY
        :param subnet_group_name: The name of the subnet group. Default: - a name is generated
        :param vpc_subnets: Which subnets within the VPC to associate with this group. Default: - private subnets
        """
        if isinstance(vpc_subnets, dict):
            vpc_subnets = aws_cdk.aws_ec2.SubnetSelection(**vpc_subnets)
        self._values: typing.Dict[str, typing.Any] = {
            "description": description,
            "vpc": vpc,
        }
        if removal_policy is not None:
            self._values["removal_policy"] = removal_policy
        if subnet_group_name is not None:
            self._values["subnet_group_name"] = subnet_group_name
        if vpc_subnets is not None:
            self._values["vpc_subnets"] = vpc_subnets

    @builtins.property
    def description(self) -> builtins.str:
        """Description of the subnet group."""
        result = self._values.get("description")
        assert result is not None, "Required property 'description' is missing"
        return result

    @builtins.property
    def vpc(self) -> aws_cdk.aws_ec2.IVpc:
        """The VPC to place the subnet group in."""
        result = self._values.get("vpc")
        assert result is not None, "Required property 'vpc' is missing"
        return result

    @builtins.property
    def removal_policy(self) -> typing.Optional[aws_cdk.core.RemovalPolicy]:
        """The removal policy to apply when the subnet group are removed from the stack or replaced during an update.

        :default: RemovalPolicy.DESTROY
        """
        result = self._values.get("removal_policy")
        return result

    @builtins.property
    def subnet_group_name(self) -> typing.Optional[builtins.str]:
        """The name of the subnet group.

        :default: - a name is generated
        """
        result = self._values.get("subnet_group_name")
        return result

    @builtins.property
    def vpc_subnets(self) -> typing.Optional[aws_cdk.aws_ec2.SubnetSelection]:
        """Which subnets within the VPC to associate with this group.

        :default: - private subnets
        """
        result = self._values.get("vpc_subnets")
        return result

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "SubnetGroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IDatabaseCluster)
class DatabaseClusterBase(
    aws_cdk.core.Resource,
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@aws-cdk/aws-rds.DatabaseClusterBase",
):
    """A new or imported clustered database."""

    @builtins.staticmethod
    def __jsii_proxy_class__():
        return _DatabaseClusterBaseProxy

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        account: typing.Optional[builtins.str] = None,
        physical_name: typing.Optional[builtins.str] = None,
        region: typing.Optional[builtins.str] = None,
    ) -> None:
        """
        :param scope: -
        :param id: -
        :param account: The AWS account ID this resource belongs to. Default: - the resource is in the same account as the stack it belongs to
        :param physical_name: The value passed in by users to the physical name prop of the resource. - ``undefined`` implies that a physical name will be allocated by CloudFormation during deployment. - a concrete value implies a specific physical name - ``PhysicalName.GENERATE_IF_NEEDED`` is a marker that indicates that a physical will only be generated by the CDK if it is needed for cross-environment references. Otherwise, it will be allocated by CloudFormation. Default: - The physical name will be allocated by CloudFormation at deployment time
        :param region: The AWS region this resource belongs to. Default: - the resource is in the same region as the stack it belongs to
        """
        props = aws_cdk.core.ResourceProps(
            account=account, physical_name=physical_name, region=region
        )

        jsii.create(DatabaseClusterBase, self, [scope, id, props])

    @jsii.member(jsii_name="addProxy")
    def add_proxy(
        self,
        id: builtins.str,
        *,
        secrets: typing.List[aws_cdk.aws_secretsmanager.ISecret],
        vpc: aws_cdk.aws_ec2.IVpc,
        borrow_timeout: typing.Optional[aws_cdk.core.Duration] = None,
        db_proxy_name: typing.Optional[builtins.str] = None,
        debug_logging: typing.Optional[builtins.bool] = None,
        iam_auth: typing.Optional[builtins.bool] = None,
        idle_client_timeout: typing.Optional[aws_cdk.core.Duration] = None,
        init_query: typing.Optional[builtins.str] = None,
        max_connections_percent: typing.Optional[jsii.Number] = None,
        max_idle_connections_percent: typing.Optional[jsii.Number] = None,
        require_tls: typing.Optional[builtins.bool] = None,
        role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        security_groups: typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]] = None,
        session_pinning_filters: typing.Optional[typing.List[SessionPinningFilter]] = None,
        vpc_subnets: typing.Optional[aws_cdk.aws_ec2.SubnetSelection] = None,
    ) -> "DatabaseProxy":
        """Add a new db proxy to this cluster.

        :param id: -
        :param secrets: The secret that the proxy uses to authenticate to the RDS DB instance or Aurora DB cluster. These secrets are stored within Amazon Secrets Manager. One or more secrets are required.
        :param vpc: The VPC to associate with the new proxy.
        :param borrow_timeout: The duration for a proxy to wait for a connection to become available in the connection pool. Only applies when the proxy has opened its maximum number of connections and all connections are busy with client sessions. Value must be between 1 second and 1 hour, or ``Duration.seconds(0)`` to represent unlimited. Default: cdk.Duration.seconds(120)
        :param db_proxy_name: The identifier for the proxy. This name must be unique for all proxies owned by your AWS account in the specified AWS Region. An identifier must begin with a letter and must contain only ASCII letters, digits, and hyphens; it can't end with a hyphen or contain two consecutive hyphens. Default: - Generated by CloudFormation (recommended)
        :param debug_logging: Whether the proxy includes detailed information about SQL statements in its logs. This information helps you to debug issues involving SQL behavior or the performance and scalability of the proxy connections. The debug information includes the text of SQL statements that you submit through the proxy. Thus, only enable this setting when needed for debugging, and only when you have security measures in place to safeguard any sensitive information that appears in the logs. Default: false
        :param iam_auth: Whether to require or disallow AWS Identity and Access Management (IAM) authentication for connections to the proxy. Default: false
        :param idle_client_timeout: The number of seconds that a connection to the proxy can be inactive before the proxy disconnects it. You can set this value higher or lower than the connection timeout limit for the associated database. Default: cdk.Duration.minutes(30)
        :param init_query: One or more SQL statements for the proxy to run when opening each new database connection. Typically used with SET statements to make sure that each connection has identical settings such as time zone and character set. For multiple statements, use semicolons as the separator. You can also include multiple variables in a single SET statement, such as SET x=1, y=2. not currently supported for PostgreSQL. Default: - no initialization query
        :param max_connections_percent: The maximum size of the connection pool for each target in a target group. For Aurora MySQL, it is expressed as a percentage of the max_connections setting for the RDS DB instance or Aurora DB cluster used by the target group. 1-100 Default: 100
        :param max_idle_connections_percent: Controls how actively the proxy closes idle database connections in the connection pool. A high value enables the proxy to leave a high percentage of idle connections open. A low value causes the proxy to close idle client connections and return the underlying database connections to the connection pool. For Aurora MySQL, it is expressed as a percentage of the max_connections setting for the RDS DB instance or Aurora DB cluster used by the target group. between 0 and MaxConnectionsPercent Default: 50
        :param require_tls: A Boolean parameter that specifies whether Transport Layer Security (TLS) encryption is required for connections to the proxy. By enabling this setting, you can enforce encrypted TLS connections to the proxy. Default: true
        :param role: IAM role that the proxy uses to access secrets in AWS Secrets Manager. Default: - A role will automatically be created
        :param security_groups: One or more VPC security groups to associate with the new proxy. Default: - No security groups
        :param session_pinning_filters: Each item in the list represents a class of SQL operations that normally cause all later statements in a session using a proxy to be pinned to the same underlying database connection. Including an item in the list exempts that class of SQL operations from the pinning behavior. Default: - no session pinning filters
        :param vpc_subnets: The subnets used by the proxy. Default: - the VPC default strategy if not specified.
        """
        options = DatabaseProxyOptions(
            secrets=secrets,
            vpc=vpc,
            borrow_timeout=borrow_timeout,
            db_proxy_name=db_proxy_name,
            debug_logging=debug_logging,
            iam_auth=iam_auth,
            idle_client_timeout=idle_client_timeout,
            init_query=init_query,
            max_connections_percent=max_connections_percent,
            max_idle_connections_percent=max_idle_connections_percent,
            require_tls=require_tls,
            role=role,
            security_groups=security_groups,
            session_pinning_filters=session_pinning_filters,
            vpc_subnets=vpc_subnets,
        )

        return jsii.invoke(self, "addProxy", [id, options])

    @jsii.member(jsii_name="asSecretAttachmentTarget")
    def as_secret_attachment_target(
        self,
    ) -> aws_cdk.aws_secretsmanager.SecretAttachmentTargetProps:
        """Renders the secret attachment target specifications."""
        return jsii.invoke(self, "asSecretAttachmentTarget", [])

    @jsii.member(jsii_name="metric")
    def metric(
        self,
        metric_name: builtins.str,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """Return the given named metric for this DBCluster.

        :param metric_name: -
        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metric", [metric_name, props])

    @jsii.member(jsii_name="metricCPUUtilization")
    def metric_cpu_utilization(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The percentage of CPU utilization.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricCPUUtilization", [props])

    @jsii.member(jsii_name="metricDatabaseConnections")
    def metric_database_connections(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The number of database connections in use.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricDatabaseConnections", [props])

    @jsii.member(jsii_name="metricDeadlocks")
    def metric_deadlocks(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The average number of deadlocks in the database per second.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricDeadlocks", [props])

    @jsii.member(jsii_name="metricEngineUptime")
    def metric_engine_uptime(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The amount of time that the instance has been running, in seconds.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricEngineUptime", [props])

    @jsii.member(jsii_name="metricFreeableMemory")
    def metric_freeable_memory(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The amount of available random access memory, in bytes.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricFreeableMemory", [props])

    @jsii.member(jsii_name="metricFreeLocalStorage")
    def metric_free_local_storage(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The amount of local storage available, in bytes.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricFreeLocalStorage", [props])

    @jsii.member(jsii_name="metricNetworkReceiveThroughput")
    def metric_network_receive_throughput(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The amount of network throughput received from clients by each instance, in bytes per second.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricNetworkReceiveThroughput", [props])

    @jsii.member(jsii_name="metricNetworkThroughput")
    def metric_network_throughput(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The amount of network throughput both received from and transmitted to clients by each instance, in bytes per second.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricNetworkThroughput", [props])

    @jsii.member(jsii_name="metricNetworkTransmitThroughput")
    def metric_network_transmit_throughput(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The amount of network throughput sent to clients by each instance, in bytes per second.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricNetworkTransmitThroughput", [props])

    @jsii.member(jsii_name="metricSnapshotStorageUsed")
    def metric_snapshot_storage_used(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The total amount of backup storage in bytes consumed by all Aurora snapshots outside its backup retention window.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricSnapshotStorageUsed", [props])

    @jsii.member(jsii_name="metricTotalBackupStorageBilled")
    def metric_total_backup_storage_billed(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The total amount of backup storage in bytes for which you are billed.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricTotalBackupStorageBilled", [props])

    @jsii.member(jsii_name="metricVolumeBytesUsed")
    def metric_volume_bytes_used(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The amount of storage used by your Aurora DB instance, in bytes.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricVolumeBytesUsed", [props])

    @jsii.member(jsii_name="metricVolumeReadIOPs")
    def metric_volume_read_io_ps(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The number of billed read I/O operations from a cluster volume, reported at 5-minute intervals.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricVolumeReadIOPs", [props])

    @jsii.member(jsii_name="metricVolumeWriteIOPs")
    def metric_volume_write_io_ps(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The number of write disk I/O operations to the cluster volume, reported at 5-minute intervals.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricVolumeWriteIOPs", [props])

    @builtins.property # type: ignore
    @jsii.member(jsii_name="clusterEndpoint")
    @abc.abstractmethod
    def cluster_endpoint(self) -> Endpoint:
        """The endpoint to use for read/write operations."""
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="clusterIdentifier")
    @abc.abstractmethod
    def cluster_identifier(self) -> builtins.str:
        """Identifier of the cluster."""
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="clusterReadEndpoint")
    @abc.abstractmethod
    def cluster_read_endpoint(self) -> Endpoint:
        """Endpoint to use for load-balanced read-only operations."""
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="connections")
    @abc.abstractmethod
    def connections(self) -> aws_cdk.aws_ec2.Connections:
        """Access to the network connections."""
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="instanceEndpoints")
    @abc.abstractmethod
    def instance_endpoints(self) -> typing.List[Endpoint]:
        """Endpoints which address each individual replica."""
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="instanceIdentifiers")
    @abc.abstractmethod
    def instance_identifiers(self) -> typing.List[builtins.str]:
        """Identifiers of the replicas."""
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="engine")
    @abc.abstractmethod
    def engine(self) -> typing.Optional["IClusterEngine"]:
        """The engine of this Cluster.

        May be not known for imported Clusters if it wasn't provided explicitly.
        """
        ...


class _DatabaseClusterBaseProxy(
    DatabaseClusterBase, jsii.proxy_for(aws_cdk.core.Resource) # type: ignore
):
    @builtins.property # type: ignore
    @jsii.member(jsii_name="clusterEndpoint")
    def cluster_endpoint(self) -> Endpoint:
        """The endpoint to use for read/write operations."""
        return jsii.get(self, "clusterEndpoint")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="clusterIdentifier")
    def cluster_identifier(self) -> builtins.str:
        """Identifier of the cluster."""
        return jsii.get(self, "clusterIdentifier")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="clusterReadEndpoint")
    def cluster_read_endpoint(self) -> Endpoint:
        """Endpoint to use for load-balanced read-only operations."""
        return jsii.get(self, "clusterReadEndpoint")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="connections")
    def connections(self) -> aws_cdk.aws_ec2.Connections:
        """Access to the network connections."""
        return jsii.get(self, "connections")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="instanceEndpoints")
    def instance_endpoints(self) -> typing.List[Endpoint]:
        """Endpoints which address each individual replica."""
        return jsii.get(self, "instanceEndpoints")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="instanceIdentifiers")
    def instance_identifiers(self) -> typing.List[builtins.str]:
        """Identifiers of the replicas."""
        return jsii.get(self, "instanceIdentifiers")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="engine")
    def engine(self) -> typing.Optional["IClusterEngine"]:
        """The engine of this Cluster.

        May be not known for imported Clusters if it wasn't provided explicitly.
        """
        return jsii.get(self, "engine")


class DatabaseClusterFromSnapshot(
    DatabaseClusterBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.DatabaseClusterFromSnapshot",
):
    """A database cluster restored from a snapshot.

    :resource: AWS::RDS::DBInstance
    """

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        engine: "IClusterEngine",
        instance_props: InstanceProps,
        snapshot_identifier: builtins.str,
        backup: typing.Optional[BackupProps] = None,
        cloudwatch_logs_exports: typing.Optional[typing.List[builtins.str]] = None,
        cloudwatch_logs_retention: typing.Optional[aws_cdk.aws_logs.RetentionDays] = None,
        cloudwatch_logs_retention_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        cluster_identifier: typing.Optional[builtins.str] = None,
        default_database_name: typing.Optional[builtins.str] = None,
        deletion_protection: typing.Optional[builtins.bool] = None,
        instance_identifier_base: typing.Optional[builtins.str] = None,
        instances: typing.Optional[jsii.Number] = None,
        monitoring_interval: typing.Optional[aws_cdk.core.Duration] = None,
        monitoring_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        parameter_group: typing.Optional[IParameterGroup] = None,
        port: typing.Optional[jsii.Number] = None,
        preferred_maintenance_window: typing.Optional[builtins.str] = None,
        removal_policy: typing.Optional[aws_cdk.core.RemovalPolicy] = None,
        s3_export_buckets: typing.Optional[typing.List[aws_cdk.aws_s3.IBucket]] = None,
        s3_export_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        s3_import_buckets: typing.Optional[typing.List[aws_cdk.aws_s3.IBucket]] = None,
        s3_import_role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        subnet_group: typing.Optional[ISubnetGroup] = None,
    ) -> None:
        """
        :param scope: -
        :param id: -
        :param engine: What kind of database to start.
        :param instance_props: Settings for the individual instances that are launched.
        :param snapshot_identifier: The identifier for the DB instance snapshot or DB cluster snapshot to restore from. You can use either the name or the Amazon Resource Name (ARN) to specify a DB cluster snapshot. However, you can use only the ARN to specify a DB instance snapshot.
        :param backup: Backup settings. Default: - Backup retention period for automated backups is 1 day. Backup preferred window is set to a 30-minute window selected at random from an 8-hour block of time for each AWS Region, occurring on a random day of the week.
        :param cloudwatch_logs_exports: The list of log types that need to be enabled for exporting to CloudWatch Logs. Default: - no log exports
        :param cloudwatch_logs_retention: The number of days log events are kept in CloudWatch Logs. When updating this property, unsetting it doesn't remove the log retention policy. To remove the retention policy, set the value to ``Infinity``. Default: - logs never expire
        :param cloudwatch_logs_retention_role: The IAM role for the Lambda function associated with the custom resource that sets the retention policy. Default: - a new role is created.
        :param cluster_identifier: An optional identifier for the cluster. Default: - A name is automatically generated.
        :param default_database_name: Name of a database which is automatically created inside the cluster. Default: - Database is not created in cluster.
        :param deletion_protection: Indicates whether the DB cluster should have deletion protection enabled. Default: - true if ``removalPolicy`` is RETAIN, false otherwise
        :param instance_identifier_base: Base identifier for instances. Every replica is named by appending the replica number to this string, 1-based. Default: - clusterIdentifier is used with the word "Instance" appended. If clusterIdentifier is not provided, the identifier is automatically generated.
        :param instances: How many replicas/instances to create. Has to be at least 1. Default: 2
        :param monitoring_interval: The interval, in seconds, between points when Amazon RDS collects enhanced monitoring metrics for the DB instances. Default: no enhanced monitoring
        :param monitoring_role: Role that will be used to manage DB instances monitoring. Default: - A role is automatically created for you
        :param parameter_group: Additional parameters to pass to the database engine. Default: - No parameter group.
        :param port: What port to listen on. Default: - The default for the engine is used.
        :param preferred_maintenance_window: A preferred maintenance window day/time range. Should be specified as a range ddd:hh24:mi-ddd:hh24:mi (24H Clock UTC). Example: 'Sun:23:45-Mon:00:15' Default: - 30-minute window selected at random from an 8-hour block of time for each AWS Region, occurring on a random day of the week.
        :param removal_policy: The removal policy to apply when the cluster and its instances are removed from the stack or replaced during an update. Default: - RemovalPolicy.SNAPSHOT (remove the cluster and instances, but retain a snapshot of the data)
        :param s3_export_buckets: S3 buckets that you want to load data into. This feature is only supported by the Aurora database engine. This property must not be used if ``s3ExportRole`` is used. For MySQL: Default: - None
        :param s3_export_role: Role that will be associated with this DB cluster to enable S3 export. This feature is only supported by the Aurora database engine. This property must not be used if ``s3ExportBuckets`` is used. For MySQL: Default: - New role is created if ``s3ExportBuckets`` is set, no role is defined otherwise
        :param s3_import_buckets: S3 buckets that you want to load data from. This feature is only supported by the Aurora database engine. This property must not be used if ``s3ImportRole`` is used. For MySQL: Default: - None
        :param s3_import_role: Role that will be associated with this DB cluster to enable S3 import. This feature is only supported by the Aurora database engine. This property must not be used if ``s3ImportBuckets`` is used. For MySQL: Default: - New role is created if ``s3ImportBuckets`` is set, no role is defined otherwise
        :param subnet_group: Existing subnet group for the cluster. Default: - a new subnet group will be created.
        """
        props = DatabaseClusterFromSnapshotProps(
            engine=engine,
            instance_props=instance_props,
            snapshot_identifier=snapshot_identifier,
            backup=backup,
            cloudwatch_logs_exports=cloudwatch_logs_exports,
            cloudwatch_logs_retention=cloudwatch_logs_retention,
            cloudwatch_logs_retention_role=cloudwatch_logs_retention_role,
            cluster_identifier=cluster_identifier,
            default_database_name=default_database_name,
            deletion_protection=deletion_protection,
            instance_identifier_base=instance_identifier_base,
            instances=instances,
            monitoring_interval=monitoring_interval,
            monitoring_role=monitoring_role,
            parameter_group=parameter_group,
            port=port,
            preferred_maintenance_window=preferred_maintenance_window,
            removal_policy=removal_policy,
            s3_export_buckets=s3_export_buckets,
            s3_export_role=s3_export_role,
            s3_import_buckets=s3_import_buckets,
            s3_import_role=s3_import_role,
            subnet_group=subnet_group,
        )

        jsii.create(DatabaseClusterFromSnapshot, self, [scope, id, props])

    @builtins.property # type: ignore
    @jsii.member(jsii_name="clusterEndpoint")
    def cluster_endpoint(self) -> Endpoint:
        """The endpoint to use for read/write operations."""
        return jsii.get(self, "clusterEndpoint")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="clusterIdentifier")
    def cluster_identifier(self) -> builtins.str:
        """Identifier of the cluster."""
        return jsii.get(self, "clusterIdentifier")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="clusterReadEndpoint")
    def cluster_read_endpoint(self) -> Endpoint:
        """Endpoint to use for load-balanced read-only operations."""
        return jsii.get(self, "clusterReadEndpoint")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="connections")
    def connections(self) -> aws_cdk.aws_ec2.Connections:
        """Access to the network connections."""
        return jsii.get(self, "connections")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="instanceEndpoints")
    def instance_endpoints(self) -> typing.List[Endpoint]:
        """Endpoints which address each individual replica."""
        return jsii.get(self, "instanceEndpoints")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="instanceIdentifiers")
    def instance_identifiers(self) -> typing.List[builtins.str]:
        """Identifiers of the replicas."""
        return jsii.get(self, "instanceIdentifiers")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="newCfnProps")
    def _new_cfn_props(self) -> CfnDBClusterProps:
        return jsii.get(self, "newCfnProps")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="securityGroups")
    def _security_groups(self) -> typing.List[aws_cdk.aws_ec2.ISecurityGroup]:
        return jsii.get(self, "securityGroups")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="subnetGroup")
    def _subnet_group(self) -> ISubnetGroup:
        return jsii.get(self, "subnetGroup")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="engine")
    def engine(self) -> typing.Optional["IClusterEngine"]:
        """The engine for this Cluster.

        Never undefined.
        """
        return jsii.get(self, "engine")


@jsii.implements(IDatabaseInstance)
class DatabaseInstanceBase(
    aws_cdk.core.Resource,
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@aws-cdk/aws-rds.DatabaseInstanceBase",
):
    """A new or imported database instance."""

    @builtins.staticmethod
    def __jsii_proxy_class__():
        return _DatabaseInstanceBaseProxy

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        account: typing.Optional[builtins.str] = None,
        physical_name: typing.Optional[builtins.str] = None,
        region: typing.Optional[builtins.str] = None,
    ) -> None:
        """
        :param scope: -
        :param id: -
        :param account: The AWS account ID this resource belongs to. Default: - the resource is in the same account as the stack it belongs to
        :param physical_name: The value passed in by users to the physical name prop of the resource. - ``undefined`` implies that a physical name will be allocated by CloudFormation during deployment. - a concrete value implies a specific physical name - ``PhysicalName.GENERATE_IF_NEEDED`` is a marker that indicates that a physical will only be generated by the CDK if it is needed for cross-environment references. Otherwise, it will be allocated by CloudFormation. Default: - The physical name will be allocated by CloudFormation at deployment time
        :param region: The AWS region this resource belongs to. Default: - the resource is in the same region as the stack it belongs to
        """
        props = aws_cdk.core.ResourceProps(
            account=account, physical_name=physical_name, region=region
        )

        jsii.create(DatabaseInstanceBase, self, [scope, id, props])

    @jsii.member(jsii_name="fromDatabaseInstanceAttributes")
    @builtins.classmethod
    def from_database_instance_attributes(
        cls,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        instance_endpoint_address: builtins.str,
        instance_identifier: builtins.str,
        port: jsii.Number,
        security_groups: typing.List[aws_cdk.aws_ec2.ISecurityGroup],
        engine: typing.Optional[IInstanceEngine] = None,
    ) -> IDatabaseInstance:
        """Import an existing database instance.

        :param scope: -
        :param id: -
        :param instance_endpoint_address: The endpoint address.
        :param instance_identifier: The instance identifier.
        :param port: The database port.
        :param security_groups: The security groups of the instance.
        :param engine: The engine of the existing database Instance. Default: - the imported Instance's engine is unknown
        """
        attrs = DatabaseInstanceAttributes(
            instance_endpoint_address=instance_endpoint_address,
            instance_identifier=instance_identifier,
            port=port,
            security_groups=security_groups,
            engine=engine,
        )

        return jsii.sinvoke(cls, "fromDatabaseInstanceAttributes", [scope, id, attrs])

    @jsii.member(jsii_name="addProxy")
    def add_proxy(
        self,
        id: builtins.str,
        *,
        secrets: typing.List[aws_cdk.aws_secretsmanager.ISecret],
        vpc: aws_cdk.aws_ec2.IVpc,
        borrow_timeout: typing.Optional[aws_cdk.core.Duration] = None,
        db_proxy_name: typing.Optional[builtins.str] = None,
        debug_logging: typing.Optional[builtins.bool] = None,
        iam_auth: typing.Optional[builtins.bool] = None,
        idle_client_timeout: typing.Optional[aws_cdk.core.Duration] = None,
        init_query: typing.Optional[builtins.str] = None,
        max_connections_percent: typing.Optional[jsii.Number] = None,
        max_idle_connections_percent: typing.Optional[jsii.Number] = None,
        require_tls: typing.Optional[builtins.bool] = None,
        role: typing.Optional[aws_cdk.aws_iam.IRole] = None,
        security_groups: typing.Optional[typing.List[aws_cdk.aws_ec2.ISecurityGroup]] = None,
        session_pinning_filters: typing.Optional[typing.List[SessionPinningFilter]] = None,
        vpc_subnets: typing.Optional[aws_cdk.aws_ec2.SubnetSelection] = None,
    ) -> "DatabaseProxy":
        """Add a new db proxy to this instance.

        :param id: -
        :param secrets: The secret that the proxy uses to authenticate to the RDS DB instance or Aurora DB cluster. These secrets are stored within Amazon Secrets Manager. One or more secrets are required.
        :param vpc: The VPC to associate with the new proxy.
        :param borrow_timeout: The duration for a proxy to wait for a connection to become available in the connection pool. Only applies when the proxy has opened its maximum number of connections and all connections are busy with client sessions. Value must be between 1 second and 1 hour, or ``Duration.seconds(0)`` to represent unlimited. Default: cdk.Duration.seconds(120)
        :param db_proxy_name: The identifier for the proxy. This name must be unique for all proxies owned by your AWS account in the specified AWS Region. An identifier must begin with a letter and must contain only ASCII letters, digits, and hyphens; it can't end with a hyphen or contain two consecutive hyphens. Default: - Generated by CloudFormation (recommended)
        :param debug_logging: Whether the proxy includes detailed information about SQL statements in its logs. This information helps you to debug issues involving SQL behavior or the performance and scalability of the proxy connections. The debug information includes the text of SQL statements that you submit through the proxy. Thus, only enable this setting when needed for debugging, and only when you have security measures in place to safeguard any sensitive information that appears in the logs. Default: false
        :param iam_auth: Whether to require or disallow AWS Identity and Access Management (IAM) authentication for connections to the proxy. Default: false
        :param idle_client_timeout: The number of seconds that a connection to the proxy can be inactive before the proxy disconnects it. You can set this value higher or lower than the connection timeout limit for the associated database. Default: cdk.Duration.minutes(30)
        :param init_query: One or more SQL statements for the proxy to run when opening each new database connection. Typically used with SET statements to make sure that each connection has identical settings such as time zone and character set. For multiple statements, use semicolons as the separator. You can also include multiple variables in a single SET statement, such as SET x=1, y=2. not currently supported for PostgreSQL. Default: - no initialization query
        :param max_connections_percent: The maximum size of the connection pool for each target in a target group. For Aurora MySQL, it is expressed as a percentage of the max_connections setting for the RDS DB instance or Aurora DB cluster used by the target group. 1-100 Default: 100
        :param max_idle_connections_percent: Controls how actively the proxy closes idle database connections in the connection pool. A high value enables the proxy to leave a high percentage of idle connections open. A low value causes the proxy to close idle client connections and return the underlying database connections to the connection pool. For Aurora MySQL, it is expressed as a percentage of the max_connections setting for the RDS DB instance or Aurora DB cluster used by the target group. between 0 and MaxConnectionsPercent Default: 50
        :param require_tls: A Boolean parameter that specifies whether Transport Layer Security (TLS) encryption is required for connections to the proxy. By enabling this setting, you can enforce encrypted TLS connections to the proxy. Default: true
        :param role: IAM role that the proxy uses to access secrets in AWS Secrets Manager. Default: - A role will automatically be created
        :param security_groups: One or more VPC security groups to associate with the new proxy. Default: - No security groups
        :param session_pinning_filters: Each item in the list represents a class of SQL operations that normally cause all later statements in a session using a proxy to be pinned to the same underlying database connection. Including an item in the list exempts that class of SQL operations from the pinning behavior. Default: - no session pinning filters
        :param vpc_subnets: The subnets used by the proxy. Default: - the VPC default strategy if not specified.
        """
        options = DatabaseProxyOptions(
            secrets=secrets,
            vpc=vpc,
            borrow_timeout=borrow_timeout,
            db_proxy_name=db_proxy_name,
            debug_logging=debug_logging,
            iam_auth=iam_auth,
            idle_client_timeout=idle_client_timeout,
            init_query=init_query,
            max_connections_percent=max_connections_percent,
            max_idle_connections_percent=max_idle_connections_percent,
            require_tls=require_tls,
            role=role,
            security_groups=security_groups,
            session_pinning_filters=session_pinning_filters,
            vpc_subnets=vpc_subnets,
        )

        return jsii.invoke(self, "addProxy", [id, options])

    @jsii.member(jsii_name="asSecretAttachmentTarget")
    def as_secret_attachment_target(
        self,
    ) -> aws_cdk.aws_secretsmanager.SecretAttachmentTargetProps:
        """Renders the secret attachment target specifications."""
        return jsii.invoke(self, "asSecretAttachmentTarget", [])

    @jsii.member(jsii_name="grantConnect")
    def grant_connect(
        self,
        grantee: aws_cdk.aws_iam.IGrantable,
    ) -> aws_cdk.aws_iam.Grant:
        """Grant the given identity connection access to the database.

        :param grantee: -
        """
        return jsii.invoke(self, "grantConnect", [grantee])

    @jsii.member(jsii_name="metric")
    def metric(
        self,
        metric_name: builtins.str,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """Return the given named metric for this DBInstance.

        :param metric_name: -
        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metric", [metric_name, props])

    @jsii.member(jsii_name="metricCPUUtilization")
    def metric_cpu_utilization(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The percentage of CPU utilization.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricCPUUtilization", [props])

    @jsii.member(jsii_name="metricDatabaseConnections")
    def metric_database_connections(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The number of database connections in use.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricDatabaseConnections", [props])

    @jsii.member(jsii_name="metricFreeableMemory")
    def metric_freeable_memory(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The amount of available random access memory.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricFreeableMemory", [props])

    @jsii.member(jsii_name="metricFreeStorageSpace")
    def metric_free_storage_space(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The amount of available storage space.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricFreeStorageSpace", [props])

    @jsii.member(jsii_name="metricReadIOPS")
    def metric_read_iops(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The average number of disk write I/O operations per second.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricReadIOPS", [props])

    @jsii.member(jsii_name="metricWriteIOPS")
    def metric_write_iops(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[aws_cdk.core.Duration] = None,
        region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[aws_cdk.aws_cloudwatch.Unit] = None,
    ) -> aws_cdk.aws_cloudwatch.Metric:
        """The average number of disk read I/O operations per second.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        """
        props = aws_cdk.aws_cloudwatch.MetricOptions(
            account=account,
            color=color,
            dimensions=dimensions,
            label=label,
            period=period,
            region=region,
            statistic=statistic,
            unit=unit,
        )

        return jsii.invoke(self, "metricWriteIOPS", [props])

    @jsii.member(jsii_name="onEvent")
    def on_event(
        self,
        id: builtins.str,
        *,
        description: typing.Optional[builtins.str] = None,
        event_pattern: typing.Optional[aws_cdk.aws_events.EventPattern] = None,
        rule_name: typing.Optional[builtins.str] = None,
        target: typing.Optional[aws_cdk.aws_events.IRuleTarget] = None,
    ) -> aws_cdk.aws_events.Rule:
        """Defines a CloudWatch event rule which triggers for instance events.

        Use
        ``rule.addEventPattern(pattern)`` to specify a filter.

        :param id: -
        :param description: A description of the rule's purpose. Default: - No description
        :param event_pattern: Additional restrictions for the event to route to the specified target. The method that generates the rule probably imposes some type of event filtering. The filtering implied by what you pass here is added on top of that filtering. Default: - No additional filtering based on an event pattern.
        :param rule_name: A name for the rule. Default: AWS CloudFormation generates a unique physical ID.
        :param target: The target to register for the event. Default: - No target is added to the rule. Use ``addTarget()`` to add a target.
        """
        options = aws_cdk.aws_events.OnEventOptions(
            description=description,
            event_pattern=event_pattern,
            rule_name=rule_name,
            target=target,
        )

        return jsii.invoke(self, "onEvent", [id, options])

    @builtins.property # type: ignore
    @jsii.member(jsii_name="connections")
    @abc.abstractmethod
    def connections(self) -> aws_cdk.aws_ec2.Connections:
        """Access to network connections."""
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="dbInstanceEndpointAddress")
    @abc.abstractmethod
    def db_instance_endpoint_address(self) -> builtins.str:
        """The instance endpoint address."""
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="dbInstanceEndpointPort")
    @abc.abstractmethod
    def db_instance_endpoint_port(self) -> builtins.str:
        """The instance endpoint port."""
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="instanceArn")
    def instance_arn(self) -> builtins.str:
        """The instance arn."""
        return jsii.get(self, "instanceArn")

    @builtins.property # type: ignore
    @jsii.member(jsii_name="instanceEndpoint")
    @abc.abstractmethod
    def instance_endpoint(self) -> Endpoint:
        """The instance endpoint."""
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="instanceIdentifier")
    @abc.abstractmethod
    def instance_identifier(self) -> builtins.str:
        """The instance identifier."""
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="engine")
    @abc.abstractmethod
    def engine(self) -> typing.Optional[IInstanceEngine]:
        """The engine of this database Instance.

        May be not known for imported Instances if it wasn't provided explicitly,
        or for read replicas.
        """
        ...

    @builtins.property # type: ignore
    @jsii.member(jsii_name="enableIamAuthentication")
    @abc.abstractmethod
    def _enable_iam_authentication(self) -> typing.Optional[builtins.bool]:
        ...

    @_enable_iam_authentication.setter # type: ignore
    @abc.abstractmethod
    def _enable_iam_authentication(self, value: typing.Optional[builtins.bool]) -> None:
        ...


class _DatabaseInstanceBaseProxy(
    DatabaseInstanceBase, jsii.proxy_for(aws_cdk.core.Resource) # type: ignore
):
    @builtins.property # type: ignore
    @jsii.member(jsii_name="connections")
    def connections(self) -> aws_cdk.aws_ec2.Connections:
   