""" Provides the :class:`EnvironmentManager` which ensures the build environment is up to date and allows for locking
the requirements in place."""

from __future__ import annotations

import contextlib
import dataclasses
import enum
import hashlib
import logging
import shutil
import time
from pathlib import Path
from typing import Any, Iterator, Sequence, cast

from packaging.requirements import Requirement

from kraken.cli import __version__

from .inspect import get_environment_state
from .lockfile import Lockfile, LockfileMetadata
from .pex import PEX, PEXBuildConfig, PEXLayout, PEXVariables, activate_pex
from .requirements import LocalRequirement, RequirementSpec

logger = logging.getLogger(__name__)


def get_implied_requirements(develop: bool) -> list[str]:
    """Returns a list of requirements that are implied for build environments managed by Kraken CLI.

    :param develop: If set to `True`, it is assumed that the current Kraken CLI is installed in develop mode
        using `slap link` or `slap install --link` and will be installed from the local project directory on
        the file system instead of from PyPI. Otherwise, Kraken CLI will be picked up from PyPI.
    """

    if develop:
        import kraken.cli

        init_path = Path(kraken.cli.__file__).resolve()
        kraken_path = init_path.parent.parent.parent
        project_root = kraken_path.parent
        pyproject = project_root / "pyproject.toml"
        if not pyproject.is_file():
            raise RuntimeError(
                "kraken-cli does not seem to be installed in development mode (expected kraken-cli's "
                'pyproject.toml at "%s")' % pyproject
            )

        # TODO (@NiklasRosenstein): It would be nice if we could tell Pip to install kraken-cli in
        #       development mode, but `pip install -e DIR` does not currently work for projects using
        #       Poetry.
        return [f"kraken-cli@{project_root}"]

    # Determine the next Kraken CLI release that may ship with breaking changes.
    version: tuple[int, int, int] = cast(Any, tuple(map(int, __version__.split("."))))
    min_version = f"0.{version[1]}.0"
    if version[0] == 0:
        # While we're in 0 major land, let's assume potential breaks with the next minor version.
        max_version = f"0.{version[1]+1}.0"
    else:
        max_version = f"{version[0]}.0.0"

    return [f"kraken-cli>={min_version},<{max_version}"]


@dataclasses.dataclass
class CalculateLockfileResult:
    lockfile: Lockfile
    extra_distributions: set[str]


class EnvironmentType(enum.Enum):
    """The environment type to create."""

    PEX_FILE = enum.auto()
    PEX_PACKED = enum.auto()
    PEX_LOOSE = enum.auto()
    # VENV = enum.auto()

    def use_pex(self) -> bool:
        return self.name.startswith("PEX_")

    def to_pex_layout(self) -> PEXLayout:
        if self == EnvironmentType.PEX_FILE:
            return PEXLayout.ZIPAPP
        elif self == EnvironmentType.PEX_PACKED:
            return PEXLayout.PACKED
        elif self == EnvironmentType.PEX_LOOSE:
            return PEXLayout.LOOSE
        else:
            raise ValueError(f"cannot get PEX layout type for {self!r}")


class BuildEnvironment:
    """Represents a separate Python environment that we install build time requirements into."""

    def __init__(
        self,
        project_path: Path,
        path: Path,
        verbosity: int,
        hash_algorithm: str = "sha256",
        develop: bool = False,
        type_: EnvironmentType = EnvironmentType.PEX_FILE,
    ) -> None:
        """
        :param project_path: The directory that relative paths should be assumed relative to.
        :param path: The directory at which the environment should be located.
        :param verbosity: 1 for showing Pip output, 2 for making Pip output verbose.
        :param hash_algorithm: The hashing algorithm for the installation source.
        :param develop: Set to `True` to install kraken-cli from source (requires that it is installed in development
            mode using [Slap][]).

        [Slap]: https://github.com/python-slap/slap-cli
        """

        assert hash_algorithm in hashlib.algorithms_available

        if type_ == EnvironmentType.PEX_FILE:
            path = path.parent / (path.name + ".pex")
        elif type_ == EnvironmentType.PEX_PACKED:
            path = path.parent / (path.name + ".pexpacked")
        elif type_ == EnvironmentType.PEX_LOOSE:
            path = path.parent / (path.name + ".pexenv")
        # elif type_ == EnvironmentType.VENV:
        #     path = path.parent / (path.name + ".venv")
        else:
            assert False, type_

        self._project_path = project_path
        self._path = path
        self._verbosity = verbosity
        self._hash_algorithm = hash_algorithm
        self._develop = develop
        self._type = type_

        self._hashes: list[str] | None = None
        self._hash_file = path.parent / (path.name + "." + hash_algorithm)
        self._install_log_file = path.parent / (path.name + ".log")

    @property
    def path(self) -> Path:
        return self._path

    @property
    def hash_algorithm(self) -> str:
        return self._hash_algorithm

    @property
    def hash_file(self) -> Path:
        return self._hash_file

    @property
    def install_log_file(self) -> Path:
        return self._install_log_file

    @property
    def hashes(self) -> Sequence[str]:
        """Returns the hash code(s) of the environment. There can be multiple as the environment may be installed from
        a lockfile which has a hash for it's original requirements as well as it's pinned requirements."""

        if self._hashes is None:
            if self._hash_file.exists():
                self._hashes = list(filter(bool, map(str.strip, self._hash_file.read_text().splitlines())))
            else:
                self._hashes = []
        return self._hashes

    @hashes.setter
    def hashes(self, value: list[str]) -> None:
        """Writes the hash code of the environment."""

        assert isinstance(value, list)
        self._hash_file.write_text("\n".join(value) + "\n")
        self._hashes = list(value)

    def set_hashes(self, source: RequirementSpec | Lockfile) -> None:
        if isinstance(source, RequirementSpec):
            hashes = [source.to_hash(self._hash_algorithm)]
        else:
            hashes = [
                source.requirements.to_hash(self._hash_algorithm),
                source.to_pinned_requirement_spec().to_hash(self._hash_algorithm),
            ]
        self.hashes = hashes

    def changed(self, requirements: RequirementSpec) -> bool:
        return requirements.to_hash(self._hash_algorithm) not in self.hashes

    def exists(self) -> bool:
        """Returns `True` if the environment exists."""

        return self._path.exists()

    def remove(self) -> None:
        """Remove the virtual environment."""

        # # Sanity check if this is really a virtual environment.
        # python_bin = self.get_program("python")
        # if not python_bin.exists():
        #     raise RuntimeError(
        #         f"would remove directory {self._path} but after a sanity check this doesn't look "
        #         "like a virtual environment!"
        #     )

        if self._path.is_file():
            self._path.unlink()
        elif self._path.is_dir():
            shutil.rmtree(self._path)
        if self._hash_file.is_file():
            self._hash_file.unlink()
        if self._install_log_file.is_file():
            self._install_log_file.unlink()

    def install(self, requirements: RequirementSpec) -> None:
        """Install the build environment using the given requirements.

        :param requirements: The requirements to install into the environment.
        """

        logger.debug("Installing %s (%s)", requirements, requirements.to_hash(self._hash_algorithm))

        if self._type.use_pex():
            config = self._get_pex_build_config(requirements)

            logger.info("Resolving ...")
            tstart = time.perf_counter()
            resolved = config.resolve()
            logger.info("Resolve complete (%fs)", time.perf_counter() - tstart)

            logger.info("Building ...")
            tstart = time.perf_counter()
            builder = config.builder(resolved)
            builder.build(str(self._path), layout=self._type.to_pex_layout())
            logger.info("Build complete (%fs)", time.perf_counter() - tstart)

        else:
            raise RuntimeError(f"Not supported: {self._type!r}")

    def _get_pex_build_config(self, requirements: RequirementSpec) -> PEXBuildConfig:

        req_strings: list[str] = []
        for req in requirements.requirements:
            if isinstance(req, LocalRequirement):
                req_strings.append(req.to_arg(self._project_path))
            else:
                req_strings.append(str(req))

        interpreter_constraints = [requirements.interpreter_constraint] if requirements.interpreter_constraint else []

        # Ensure with the preamble that the local python paths are importable.
        preamble = "# kraken-cli BEGIN\nimport os, sys\n"
        preamble += "os.environ.setdefault('KRAKEN_MANAGED', '1')\n"
        for path in (self._project_path / x for x in requirements.pythonpath):
            preamble += f"sys.path.append({str(path.absolute())!r})\n"
        preamble += "# kraken-cli END\n"

        return PEXBuildConfig(
            preamble=preamble,
            interpreter_constraints=interpreter_constraints,
            script="kraken",
            requirements=req_strings,
            index_url=requirements.index_url,
            extra_index_urls=requirements.extra_index_urls,
            platforms=None,
            preserve_log=False,
        )

    @contextlib.contextmanager
    def activate(self) -> Iterator[None]:
        # NOTE (@NiklasRosenstein): Need to point PEX_EXTRA_SYS_PATH at the PEX itself, otherwise activate9) fails.
        pex = PEX(self._path, env=PEXVariables({"PEX_EXTRA_SYS_PATH": str(self._path)}))
        with activate_pex(pex):
            yield

    def calculate_lockfile(self, requirements: RequirementSpec) -> CalculateLockfileResult:
        """Calculate the lockfile of the environment.

        :param requirements: The requirements that were used to install the environment. These requirements
            will be embedded as part of the returned lockfile.
        """

        assert self._type.use_pex()

        with self.activate():

            # python = self.get_program("python")
            env = get_environment_state()

            # Convert all distribution names to lowercase.
            # TODO (@NiklasRosenstein): Further changes may be needed to correctly normalize all distribution names.
            env.distributions = {k.lower(): v for k, v in env.distributions.items()}

            # Collect only the package name and version for required packages.
            distributions = {}
            requirements_stack = list(requirements.requirements)

            while requirements_stack:
                package_req = requirements_stack.pop(0)
                package_name = package_req.name.lower()

                if package_name in distributions:
                    continue
                if package_name not in env.distributions:
                    # NOTE (@NiklasRosenstein): We may be missing the package because it's a requirement that is only
                    #       installed under certain conditions (e.g. markers/extras).
                    continue

                dist = env.distributions[package_name]

                # Pin the package version.
                distributions[package_name] = dist.version

                # Filter the requirements of the distribution down to the ones required according to markers and the
                # current package requirement's extras.
                for req in map(Requirement, dist.requirements):
                    if isinstance(package_req, Requirement) and req.marker:
                        satisfied = any(req.marker.evaluate({"extra": extra}) for extra in package_req.extras)
                    else:
                        satisfied = True
                    if satisfied:
                        requirements_stack.append(req)

            metadata = LockfileMetadata.new()
            metadata.kraken_cli_version = f"{env.kraken_cli_version} (instrumented by {metadata.kraken_cli_version})"
            metadata.python_version = f"{env.python_version} (instrumented by {env.python_version})"

            extra_distributions = env.distributions.keys() - distributions.keys() - {"pip"}
            return CalculateLockfileResult(Lockfile(metadata, requirements, distributions), extra_distributions)
