"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseProxy = exports.ProxyTarget = exports.SessionPinningFilter = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const cdk = require("@aws-cdk/core");
const util_1 = require("./private/util");
const rds_generated_1 = require("./rds.generated");
/**
 * SessionPinningFilter.
 *
 * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html#rds-proxy-pinning
 * @stability stable
 */
class SessionPinningFilter {
    constructor(
    /**
     * Filter name
     */
    filterName) {
        this.filterName = filterName;
    }
    /**
     * custom filter.
     *
     * @stability stable
     */
    static of(filterName) {
        return new SessionPinningFilter(filterName);
    }
}
exports.SessionPinningFilter = SessionPinningFilter;
_a = JSII_RTTI_SYMBOL_1;
SessionPinningFilter[_a] = { fqn: "@aws-cdk/aws-rds.SessionPinningFilter", version: "1.139.0" };
/**
 * You can opt out of session pinning for the following kinds of application statements:.
 *
 * - Setting session variables and configuration settings.
 *
 * @stability stable
 */
SessionPinningFilter.EXCLUDE_VARIABLE_SETS = new SessionPinningFilter('EXCLUDE_VARIABLE_SETS');
/**
 * Proxy target: Instance or Cluster.
 *
 * A target group is a collection of databases that the proxy can connect to.
 * Currently, you can specify only one RDS DB instance or Aurora DB cluster.
 *
 * @stability stable
 */
class ProxyTarget {
    constructor(dbInstance, dbCluster) {
        this.dbInstance = dbInstance;
        this.dbCluster = dbCluster;
    }
    /**
     * From instance.
     *
     * @param instance RDS database instance.
     * @stability stable
     */
    static fromInstance(instance) {
        jsiiDeprecationWarnings._aws_cdk_aws_rds_IDatabaseInstance(instance);
        return new ProxyTarget(instance, undefined);
    }
    /**
     * From cluster.
     *
     * @param cluster RDS database cluster.
     * @stability stable
     */
    static fromCluster(cluster) {
        jsiiDeprecationWarnings._aws_cdk_aws_rds_IDatabaseCluster(cluster);
        return new ProxyTarget(undefined, cluster);
    }
    /**
     * Bind this target to the specified database proxy.
     *
     * @stability stable
     */
    bind(proxy) {
        var _d, _e, _f, _g, _h, _j;
        jsiiDeprecationWarnings._aws_cdk_aws_rds_DatabaseProxy(proxy);
        const engine = (_e = (_d = this.dbInstance) === null || _d === void 0 ? void 0 : _d.engine) !== null && _e !== void 0 ? _e : (_f = this.dbCluster) === null || _f === void 0 ? void 0 : _f.engine;
        if (!engine) {
            const errorResource = (_g = this.dbCluster) !== null && _g !== void 0 ? _g : this.dbInstance;
            throw new Error(`Could not determine engine for proxy target '${errorResource === null || errorResource === void 0 ? void 0 : errorResource.node.path}'. ` +
                'Please provide it explicitly when importing the resource');
        }
        const engineFamily = engine.engineFamily;
        if (!engineFamily) {
            throw new Error(`Engine '${util_1.engineDescription(engine)}' does not support proxies`);
        }
        // allow connecting to the Cluster/Instance from the Proxy
        (_h = this.dbCluster) === null || _h === void 0 ? void 0 : _h.connections.allowDefaultPortFrom(proxy, 'Allow connections to the database Cluster from the Proxy');
        (_j = this.dbInstance) === null || _j === void 0 ? void 0 : _j.connections.allowDefaultPortFrom(proxy, 'Allow connections to the database Instance from the Proxy');
        return {
            engineFamily,
            dbClusters: this.dbCluster ? [this.dbCluster] : undefined,
            dbInstances: this.dbInstance ? [this.dbInstance] : undefined,
        };
    }
}
exports.ProxyTarget = ProxyTarget;
_b = JSII_RTTI_SYMBOL_1;
ProxyTarget[_b] = { fqn: "@aws-cdk/aws-rds.ProxyTarget", version: "1.139.0" };
/**
 * Represents an RDS Database Proxy.
 *
 */
class DatabaseProxyBase extends cdk.Resource {
    grantConnect(grantee, dbUser) {
        if (!dbUser) {
            throw new Error('For imported Database Proxies, the dbUser is required in grantConnect()');
        }
        const scopeStack = cdk.Stack.of(this);
        const proxyGeneratedId = scopeStack.splitArn(this.dbProxyArn, cdk.ArnFormat.COLON_RESOURCE_NAME).resourceName;
        const userArn = scopeStack.formatArn({
            service: 'rds-db',
            resource: 'dbuser',
            resourceName: `${proxyGeneratedId}/${dbUser}`,
            arnFormat: cdk.ArnFormat.COLON_RESOURCE_NAME,
        });
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['rds-db:connect'],
            resourceArns: [userArn],
        });
    }
}
/**
 * RDS Database Proxy.
 *
 * @stability stable
 * @resource AWS::RDS::DBProxy
 */
class DatabaseProxy extends DatabaseProxyBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _d, _e, _f, _g;
        super(scope, id, { physicalName: props.dbProxyName || id });
        jsiiDeprecationWarnings._aws_cdk_aws_rds_DatabaseProxyProps(props);
        const role = props.role || new iam.Role(this, 'IAMRole', {
            assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
        });
        for (const secret of props.secrets) {
            secret.grantRead(role);
        }
        const securityGroups = (_d = props.securityGroups) !== null && _d !== void 0 ? _d : [
            new ec2.SecurityGroup(this, 'ProxySecurityGroup', {
                description: 'SecurityGroup for Database Proxy',
                vpc: props.vpc,
            }),
        ];
        this.connections = new ec2.Connections({ securityGroups });
        const bindResult = props.proxyTarget.bind(this);
        if (props.secrets.length < 1) {
            throw new Error('One or more secrets are required.');
        }
        this.secrets = props.secrets;
        this.resource = new rds_generated_1.CfnDBProxy(this, 'Resource', {
            auth: props.secrets.map(_ => {
                return {
                    authScheme: 'SECRETS',
                    iamAuth: props.iamAuth ? 'REQUIRED' : 'DISABLED',
                    secretArn: _.secretArn,
                };
            }),
            dbProxyName: this.physicalName,
            debugLogging: props.debugLogging,
            engineFamily: bindResult.engineFamily,
            idleClientTimeout: (_e = props.idleClientTimeout) === null || _e === void 0 ? void 0 : _e.toSeconds(),
            requireTls: (_f = props.requireTLS) !== null && _f !== void 0 ? _f : true,
            roleArn: role.roleArn,
            vpcSecurityGroupIds: cdk.Lazy.list({ produce: () => this.connections.securityGroups.map(_ => _.securityGroupId) }),
            vpcSubnetIds: props.vpc.selectSubnets(props.vpcSubnets).subnetIds,
        });
        this.dbProxyName = this.resource.ref;
        this.dbProxyArn = this.resource.attrDbProxyArn;
        this.endpoint = this.resource.attrEndpoint;
        let dbInstanceIdentifiers;
        if (bindResult.dbInstances) {
            // support for only single instance
            dbInstanceIdentifiers = [bindResult.dbInstances[0].instanceIdentifier];
        }
        let dbClusterIdentifiers;
        if (bindResult.dbClusters) {
            dbClusterIdentifiers = bindResult.dbClusters.map((c) => c.clusterIdentifier);
        }
        if (!!dbInstanceIdentifiers && !!dbClusterIdentifiers) {
            throw new Error('Cannot specify both dbInstanceIdentifiers and dbClusterIdentifiers');
        }
        const proxyTargetGroup = new rds_generated_1.CfnDBProxyTargetGroup(this, 'ProxyTargetGroup', {
            targetGroupName: 'default',
            dbProxyName: this.dbProxyName,
            dbInstanceIdentifiers,
            dbClusterIdentifiers,
            connectionPoolConfigurationInfo: toConnectionPoolConfigurationInfo(props),
        });
        (_g = bindResult.dbClusters) === null || _g === void 0 ? void 0 : _g.forEach((c) => proxyTargetGroup.node.addDependency(c));
    }
    /**
     * Import an existing database proxy.
     *
     * @stability stable
     */
    static fromDatabaseProxyAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_rds_DatabaseProxyAttributes(attrs);
        class Import extends DatabaseProxyBase {
            constructor() {
                super(...arguments);
                this.dbProxyName = attrs.dbProxyName;
                this.dbProxyArn = attrs.dbProxyArn;
                this.endpoint = attrs.endpoint;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Renders the secret attachment target specifications.
     *
     * @stability stable
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.dbProxyName,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_PROXY,
        };
    }
    /**
     * Grant the given identity connection access to the proxy.
     *
     * @stability stable
     */
    grantConnect(grantee, dbUser) {
        if (!dbUser) {
            if (this.secrets.length > 1) {
                throw new Error('When the Proxy contains multiple Secrets, you must pass a dbUser explicitly to grantConnect()');
            }
            // 'username' is the field RDS uses here,
            // see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-proxy.html#rds-proxy-secrets-arns
            dbUser = this.secrets[0].secretValueFromJson('username').toString();
        }
        return super.grantConnect(grantee, dbUser);
    }
}
exports.DatabaseProxy = DatabaseProxy;
_c = JSII_RTTI_SYMBOL_1;
DatabaseProxy[_c] = { fqn: "@aws-cdk/aws-rds.DatabaseProxy", version: "1.139.0" };
/**
 * ConnectionPoolConfiguration (L2 => L1)
 */
function toConnectionPoolConfigurationInfo(props) {
    var _d, _e;
    return {
        connectionBorrowTimeout: (_d = props.borrowTimeout) === null || _d === void 0 ? void 0 : _d.toSeconds(),
        initQuery: props.initQuery,
        maxConnectionsPercent: props.maxConnectionsPercent,
        maxIdleConnectionsPercent: props.maxIdleConnectionsPercent,
        sessionPinningFilters: (_e = props.sessionPinningFilters) === null || _e === void 0 ? void 0 : _e.map(_ => _.filterName),
    };
}
//# sourceMappingURL=data:application/json;base64,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