"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderUnless = exports.helperRemovalPolicy = exports.renderCredentials = exports.defaultDeletionProtection = exports.engineDescription = exports.setupS3ImportExport = exports.DEFAULT_PASSWORD_EXCLUDE_CHARS = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const database_secret_1 = require("../database-secret");
const props_1 = require("../props");
/**
 * The default set of characters we exclude from generated passwords for database users.
 * It's a combination of characters that have a tendency to cause problems in shell scripts,
 * some engine-specific characters (for example, Oracle doesn't like '@' in its passwords),
 * and some that trip up other services, like DMS.
 *
 * This constant is private to the RDS module.
 */
exports.DEFAULT_PASSWORD_EXCLUDE_CHARS = " %+~`#$&*()|[]{}:;<>?!'/@\"\\";
/**
 * Validates the S3 import/export props and returns the import/export roles, if any.
 * If `combineRoles` is true, will reuse the import role for export (or vice versa) if possible.
 *
 * Notably, `combineRoles` is set to true for instances, but false for clusters.
 * This is because the `combineRoles` functionality is most applicable to instances and didn't exist
 * for the initial clusters implementation. To maintain backwards compatibility, it is set to false for clusters.
 */
function setupS3ImportExport(scope, props, combineRoles) {
    let s3ImportRole = props.s3ImportRole;
    let s3ExportRole = props.s3ExportRole;
    if (props.s3ImportBuckets && props.s3ImportBuckets.length > 0) {
        if (props.s3ImportRole) {
            throw new Error('Only one of s3ImportRole or s3ImportBuckets must be specified, not both.');
        }
        s3ImportRole = (combineRoles && s3ExportRole) ? s3ExportRole : new iam.Role(scope, 'S3ImportRole', {
            assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
        });
        for (const bucket of props.s3ImportBuckets) {
            bucket.grantRead(s3ImportRole);
        }
    }
    if (props.s3ExportBuckets && props.s3ExportBuckets.length > 0) {
        if (props.s3ExportRole) {
            throw new Error('Only one of s3ExportRole or s3ExportBuckets must be specified, not both.');
        }
        s3ExportRole = (combineRoles && s3ImportRole) ? s3ImportRole : new iam.Role(scope, 'S3ExportRole', {
            assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
        });
        for (const bucket of props.s3ExportBuckets) {
            bucket.grantReadWrite(s3ExportRole);
        }
    }
    return { s3ImportRole, s3ExportRole };
}
exports.setupS3ImportExport = setupS3ImportExport;
function engineDescription(engine) {
    var _a;
    return engine.engineType + (((_a = engine.engineVersion) === null || _a === void 0 ? void 0 : _a.fullVersion) ? `-${engine.engineVersion.fullVersion}` : '');
}
exports.engineDescription = engineDescription;
/**
 * By default, deletion protection is disabled.
 * Enable if explicitly provided or if the RemovalPolicy has been set to RETAIN
 */
function defaultDeletionProtection(deletionProtection, removalPolicy) {
    return deletionProtection !== null && deletionProtection !== void 0 ? deletionProtection : (removalPolicy === core_1.RemovalPolicy.RETAIN ? true : undefined);
}
exports.defaultDeletionProtection = defaultDeletionProtection;
/**
 * Renders the credentials for an instance or cluster
 */
function renderCredentials(scope, engine, credentials) {
    var _a;
    let renderedCredentials = credentials !== null && credentials !== void 0 ? credentials : props_1.Credentials.fromUsername((_a = engine.defaultUsername) !== null && _a !== void 0 ? _a : 'admin'); // For backwards compatibilty
    if (!renderedCredentials.secret && !renderedCredentials.password) {
        renderedCredentials = props_1.Credentials.fromSecret(new database_secret_1.DatabaseSecret(scope, 'Secret', {
            username: renderedCredentials.username,
            secretName: renderedCredentials.secretName,
            encryptionKey: renderedCredentials.encryptionKey,
            excludeCharacters: renderedCredentials.excludeCharacters,
            // if username must be referenced as a string we can safely replace the
            // secret when customization options are changed without risking a replacement
            replaceOnPasswordCriteriaChanges: credentials === null || credentials === void 0 ? void 0 : credentials.usernameAsString,
            replicaRegions: renderedCredentials.replicaRegions,
        }), 
        // pass username if it must be referenced as a string
        (credentials === null || credentials === void 0 ? void 0 : credentials.usernameAsString) ? renderedCredentials.username : undefined);
    }
    return renderedCredentials;
}
exports.renderCredentials = renderCredentials;
/**
 * The RemovalPolicy that should be applied to a "helper" resource, if the base resource has the given removal policy
 *
 * - For Clusters, this determines the RemovalPolicy for Instances/SubnetGroups.
 * - For Instances, this determines the RemovalPolicy for SubnetGroups.
 *
 * If the basePolicy is:
 *
 *  DESTROY or SNAPSHOT -> DESTROY (snapshot is good enough to recreate)
 *  RETAIN              -> RETAIN  (anything else will lose data or fail to deploy)
 *  (undefined)         -> DESTROY (base policy is assumed to be SNAPSHOT)
 */
function helperRemovalPolicy(basePolicy) {
    return basePolicy === core_1.RemovalPolicy.RETAIN
        ? core_1.RemovalPolicy.RETAIN
        : core_1.RemovalPolicy.DESTROY;
}
exports.helperRemovalPolicy = helperRemovalPolicy;
/**
 * Return a given value unless it's the same as another value
 */
function renderUnless(value, suppressValue) {
    return value === suppressValue ? undefined : value;
}
exports.renderUnless = renderUnless;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidXRpbC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInV0aWwudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsd0NBQXdDO0FBRXhDLHdDQUE4QztBQUM5Qyx3REFBb0Q7QUFFcEQsb0NBQXVDO0FBTXZDOzs7Ozs7O0dBT0c7QUFDVSxRQUFBLDhCQUE4QixHQUFHLCtCQUErQixDQUFDO0FBVTlFOzs7Ozs7O0dBT0c7QUFDSCxTQUFnQixtQkFBbUIsQ0FDakMsS0FBZ0IsRUFDaEIsS0FBa0MsRUFDbEMsWUFBcUI7SUFFckIsSUFBSSxZQUFZLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQztJQUN0QyxJQUFJLFlBQVksR0FBRyxLQUFLLENBQUMsWUFBWSxDQUFDO0lBRXRDLElBQUksS0FBSyxDQUFDLGVBQWUsSUFBSSxLQUFLLENBQUMsZUFBZSxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7UUFDN0QsSUFBSSxLQUFLLENBQUMsWUFBWSxFQUFFO1lBQ3RCLE1BQU0sSUFBSSxLQUFLLENBQUMsMEVBQTBFLENBQUMsQ0FBQztTQUM3RjtRQUVELFlBQVksR0FBRyxDQUFDLFlBQVksSUFBSSxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRTtZQUNqRyxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsbUJBQW1CLENBQUM7U0FDekQsQ0FBQyxDQUFDO1FBQ0gsS0FBSyxNQUFNLE1BQU0sSUFBSSxLQUFLLENBQUMsZUFBZSxFQUFFO1lBQzFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsWUFBWSxDQUFDLENBQUM7U0FDaEM7S0FDRjtJQUVELElBQUksS0FBSyxDQUFDLGVBQWUsSUFBSSxLQUFLLENBQUMsZUFBZSxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7UUFDN0QsSUFBSSxLQUFLLENBQUMsWUFBWSxFQUFFO1lBQ3RCLE1BQU0sSUFBSSxLQUFLLENBQUMsMEVBQTBFLENBQUMsQ0FBQztTQUM3RjtRQUVELFlBQVksR0FBRyxDQUFDLFlBQVksSUFBSSxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRTtZQUNqRyxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsbUJBQW1CLENBQUM7U0FDekQsQ0FBQyxDQUFDO1FBQ0gsS0FBSyxNQUFNLE1BQU0sSUFBSSxLQUFLLENBQUMsZUFBZSxFQUFFO1lBQzFDLE1BQU0sQ0FBQyxjQUFjLENBQUMsWUFBWSxDQUFDLENBQUM7U0FDckM7S0FDRjtJQUVELE9BQU8sRUFBRSxZQUFZLEVBQUUsWUFBWSxFQUFFLENBQUM7QUFDeEMsQ0FBQztBQW5DRCxrREFtQ0M7QUFFRCxTQUFnQixpQkFBaUIsQ0FBQyxNQUFlOztJQUMvQyxPQUFPLE1BQU0sQ0FBQyxVQUFVLEdBQUcsQ0FBQyxPQUFBLE1BQU0sQ0FBQyxhQUFhLDBDQUFFLFdBQVcsRUFBQyxDQUFDLENBQUMsSUFBSSxNQUFNLENBQUMsYUFBYSxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQztBQUMvRyxDQUFDO0FBRkQsOENBRUM7QUFFRDs7O0dBR0c7QUFDSCxTQUFnQix5QkFBeUIsQ0FBQyxrQkFBNEIsRUFBRSxhQUE2QjtJQUNuRyxPQUFPLGtCQUFrQixhQUFsQixrQkFBa0IsY0FBbEIsa0JBQWtCLEdBQUksQ0FBQyxhQUFhLEtBQUssb0JBQWEsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUM7QUFDM0YsQ0FBQztBQUZELDhEQUVDO0FBRUQ7O0dBRUc7QUFDSCxTQUFnQixpQkFBaUIsQ0FBQyxLQUFnQixFQUFFLE1BQWUsRUFBRSxXQUF5Qjs7SUFDNUYsSUFBSSxtQkFBbUIsR0FBRyxXQUFXLGFBQVgsV0FBVyxjQUFYLFdBQVcsR0FBSSxtQkFBVyxDQUFDLFlBQVksT0FBQyxNQUFNLENBQUMsZUFBZSxtQ0FBSSxPQUFPLENBQUMsQ0FBQyxDQUFDLDZCQUE2QjtJQUVuSSxJQUFJLENBQUMsbUJBQW1CLENBQUMsTUFBTSxJQUFJLENBQUMsbUJBQW1CLENBQUMsUUFBUSxFQUFFO1FBQ2hFLG1CQUFtQixHQUFHLG1CQUFXLENBQUMsVUFBVSxDQUMxQyxJQUFJLGdDQUFjLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtZQUNsQyxRQUFRLEVBQUUsbUJBQW1CLENBQUMsUUFBUTtZQUN0QyxVQUFVLEVBQUUsbUJBQW1CLENBQUMsVUFBVTtZQUMxQyxhQUFhLEVBQUUsbUJBQW1CLENBQUMsYUFBYTtZQUNoRCxpQkFBaUIsRUFBRSxtQkFBbUIsQ0FBQyxpQkFBaUI7WUFDeEQsdUVBQXVFO1lBQ3ZFLDhFQUE4RTtZQUM5RSxnQ0FBZ0MsRUFBRSxXQUFXLGFBQVgsV0FBVyx1QkFBWCxXQUFXLENBQUUsZ0JBQWdCO1lBQy9ELGNBQWMsRUFBRSxtQkFBbUIsQ0FBQyxjQUFjO1NBQ25ELENBQUM7UUFDRixxREFBcUQ7UUFDckQsQ0FBQSxXQUFXLGFBQVgsV0FBVyx1QkFBWCxXQUFXLENBQUUsZ0JBQWdCLEVBQUMsQ0FBQyxDQUFDLG1CQUFtQixDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUN6RSxDQUFDO0tBQ0g7SUFFRCxPQUFPLG1CQUFtQixDQUFDO0FBQzdCLENBQUM7QUFyQkQsOENBcUJDO0FBRUQ7Ozs7Ozs7Ozs7O0dBV0c7QUFDSCxTQUFnQixtQkFBbUIsQ0FBQyxVQUEwQjtJQUM1RCxPQUFPLFVBQVUsS0FBSyxvQkFBYSxDQUFDLE1BQU07UUFDeEMsQ0FBQyxDQUFDLG9CQUFhLENBQUMsTUFBTTtRQUN0QixDQUFDLENBQUMsb0JBQWEsQ0FBQyxPQUFPLENBQUM7QUFDNUIsQ0FBQztBQUpELGtEQUlDO0FBRUQ7O0dBRUc7QUFDSCxTQUFnQixZQUFZLENBQUksS0FBUSxFQUFFLGFBQWdCO0lBQ3hELE9BQU8sS0FBSyxLQUFLLGFBQWEsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUM7QUFDckQsQ0FBQztBQUZELG9DQUVDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgaWFtIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgczMgZnJvbSAnQGF3cy1jZGsvYXdzLXMzJztcbmltcG9ydCB7IFJlbW92YWxQb2xpY3kgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IERhdGFiYXNlU2VjcmV0IH0gZnJvbSAnLi4vZGF0YWJhc2Utc2VjcmV0JztcbmltcG9ydCB7IElFbmdpbmUgfSBmcm9tICcuLi9lbmdpbmUnO1xuaW1wb3J0IHsgQ3JlZGVudGlhbHMgfSBmcm9tICcuLi9wcm9wcyc7XG5cbi8vIGtlZXAgdGhpcyBpbXBvcnQgc2VwYXJhdGUgZnJvbSBvdGhlciBpbXBvcnRzIHRvIHJlZHVjZSBjaGFuY2UgZm9yIG1lcmdlIGNvbmZsaWN0cyB3aXRoIHYyLW1haW5cbi8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBuby1kdXBsaWNhdGUtaW1wb3J0cywgaW1wb3J0L29yZGVyXG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcblxuLyoqXG4gKiBUaGUgZGVmYXVsdCBzZXQgb2YgY2hhcmFjdGVycyB3ZSBleGNsdWRlIGZyb20gZ2VuZXJhdGVkIHBhc3N3b3JkcyBmb3IgZGF0YWJhc2UgdXNlcnMuXG4gKiBJdCdzIGEgY29tYmluYXRpb24gb2YgY2hhcmFjdGVycyB0aGF0IGhhdmUgYSB0ZW5kZW5jeSB0byBjYXVzZSBwcm9ibGVtcyBpbiBzaGVsbCBzY3JpcHRzLFxuICogc29tZSBlbmdpbmUtc3BlY2lmaWMgY2hhcmFjdGVycyAoZm9yIGV4YW1wbGUsIE9yYWNsZSBkb2Vzbid0IGxpa2UgJ0AnIGluIGl0cyBwYXNzd29yZHMpLFxuICogYW5kIHNvbWUgdGhhdCB0cmlwIHVwIG90aGVyIHNlcnZpY2VzLCBsaWtlIERNUy5cbiAqXG4gKiBUaGlzIGNvbnN0YW50IGlzIHByaXZhdGUgdG8gdGhlIFJEUyBtb2R1bGUuXG4gKi9cbmV4cG9ydCBjb25zdCBERUZBVUxUX1BBU1NXT1JEX0VYQ0xVREVfQ0hBUlMgPSBcIiAlK35gIyQmKigpfFtde306Ozw+PyEnL0BcXFwiXFxcXFwiO1xuXG4vKiogQ29tbW9uIGJhc2Ugb2YgYERhdGFiYXNlSW5zdGFuY2VQcm9wc2AgYW5kIGBEYXRhYmFzZUNsdXN0ZXJCYXNlUHJvcHNgIHRoYXQgaGFzIG9ubHkgdGhlIFMzIHByb3BzICovXG5leHBvcnQgaW50ZXJmYWNlIERhdGFiYXNlUzNJbXBvcnRFeHBvcnRQcm9wcyB7XG4gIHJlYWRvbmx5IHMzSW1wb3J0Um9sZT86IGlhbS5JUm9sZTtcbiAgcmVhZG9ubHkgczNJbXBvcnRCdWNrZXRzPzogczMuSUJ1Y2tldFtdO1xuICByZWFkb25seSBzM0V4cG9ydFJvbGU/OiBpYW0uSVJvbGU7XG4gIHJlYWRvbmx5IHMzRXhwb3J0QnVja2V0cz86IHMzLklCdWNrZXRbXTtcbn1cblxuLyoqXG4gKiBWYWxpZGF0ZXMgdGhlIFMzIGltcG9ydC9leHBvcnQgcHJvcHMgYW5kIHJldHVybnMgdGhlIGltcG9ydC9leHBvcnQgcm9sZXMsIGlmIGFueS5cbiAqIElmIGBjb21iaW5lUm9sZXNgIGlzIHRydWUsIHdpbGwgcmV1c2UgdGhlIGltcG9ydCByb2xlIGZvciBleHBvcnQgKG9yIHZpY2UgdmVyc2EpIGlmIHBvc3NpYmxlLlxuICpcbiAqIE5vdGFibHksIGBjb21iaW5lUm9sZXNgIGlzIHNldCB0byB0cnVlIGZvciBpbnN0YW5jZXMsIGJ1dCBmYWxzZSBmb3IgY2x1c3RlcnMuXG4gKiBUaGlzIGlzIGJlY2F1c2UgdGhlIGBjb21iaW5lUm9sZXNgIGZ1bmN0aW9uYWxpdHkgaXMgbW9zdCBhcHBsaWNhYmxlIHRvIGluc3RhbmNlcyBhbmQgZGlkbid0IGV4aXN0XG4gKiBmb3IgdGhlIGluaXRpYWwgY2x1c3RlcnMgaW1wbGVtZW50YXRpb24uIFRvIG1haW50YWluIGJhY2t3YXJkcyBjb21wYXRpYmlsaXR5LCBpdCBpcyBzZXQgdG8gZmFsc2UgZm9yIGNsdXN0ZXJzLlxuICovXG5leHBvcnQgZnVuY3Rpb24gc2V0dXBTM0ltcG9ydEV4cG9ydChcbiAgc2NvcGU6IENvbnN0cnVjdCxcbiAgcHJvcHM6IERhdGFiYXNlUzNJbXBvcnRFeHBvcnRQcm9wcyxcbiAgY29tYmluZVJvbGVzOiBib29sZWFuKTogeyBzM0ltcG9ydFJvbGU/OiBpYW0uSVJvbGUsIHMzRXhwb3J0Um9sZT86IGlhbS5JUm9sZSB9IHtcblxuICBsZXQgczNJbXBvcnRSb2xlID0gcHJvcHMuczNJbXBvcnRSb2xlO1xuICBsZXQgczNFeHBvcnRSb2xlID0gcHJvcHMuczNFeHBvcnRSb2xlO1xuXG4gIGlmIChwcm9wcy5zM0ltcG9ydEJ1Y2tldHMgJiYgcHJvcHMuczNJbXBvcnRCdWNrZXRzLmxlbmd0aCA+IDApIHtcbiAgICBpZiAocHJvcHMuczNJbXBvcnRSb2xlKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ09ubHkgb25lIG9mIHMzSW1wb3J0Um9sZSBvciBzM0ltcG9ydEJ1Y2tldHMgbXVzdCBiZSBzcGVjaWZpZWQsIG5vdCBib3RoLicpO1xuICAgIH1cblxuICAgIHMzSW1wb3J0Um9sZSA9IChjb21iaW5lUm9sZXMgJiYgczNFeHBvcnRSb2xlKSA/IHMzRXhwb3J0Um9sZSA6IG5ldyBpYW0uUm9sZShzY29wZSwgJ1MzSW1wb3J0Um9sZScsIHtcbiAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdyZHMuYW1hem9uYXdzLmNvbScpLFxuICAgIH0pO1xuICAgIGZvciAoY29uc3QgYnVja2V0IG9mIHByb3BzLnMzSW1wb3J0QnVja2V0cykge1xuICAgICAgYnVja2V0LmdyYW50UmVhZChzM0ltcG9ydFJvbGUpO1xuICAgIH1cbiAgfVxuXG4gIGlmIChwcm9wcy5zM0V4cG9ydEJ1Y2tldHMgJiYgcHJvcHMuczNFeHBvcnRCdWNrZXRzLmxlbmd0aCA+IDApIHtcbiAgICBpZiAocHJvcHMuczNFeHBvcnRSb2xlKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ09ubHkgb25lIG9mIHMzRXhwb3J0Um9sZSBvciBzM0V4cG9ydEJ1Y2tldHMgbXVzdCBiZSBzcGVjaWZpZWQsIG5vdCBib3RoLicpO1xuICAgIH1cblxuICAgIHMzRXhwb3J0Um9sZSA9IChjb21iaW5lUm9sZXMgJiYgczNJbXBvcnRSb2xlKSA/IHMzSW1wb3J0Um9sZSA6IG5ldyBpYW0uUm9sZShzY29wZSwgJ1MzRXhwb3J0Um9sZScsIHtcbiAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdyZHMuYW1hem9uYXdzLmNvbScpLFxuICAgIH0pO1xuICAgIGZvciAoY29uc3QgYnVja2V0IG9mIHByb3BzLnMzRXhwb3J0QnVja2V0cykge1xuICAgICAgYnVja2V0LmdyYW50UmVhZFdyaXRlKHMzRXhwb3J0Um9sZSk7XG4gICAgfVxuICB9XG5cbiAgcmV0dXJuIHsgczNJbXBvcnRSb2xlLCBzM0V4cG9ydFJvbGUgfTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGVuZ2luZURlc2NyaXB0aW9uKGVuZ2luZTogSUVuZ2luZSkge1xuICByZXR1cm4gZW5naW5lLmVuZ2luZVR5cGUgKyAoZW5naW5lLmVuZ2luZVZlcnNpb24/LmZ1bGxWZXJzaW9uID8gYC0ke2VuZ2luZS5lbmdpbmVWZXJzaW9uLmZ1bGxWZXJzaW9ufWAgOiAnJyk7XG59XG5cbi8qKlxuICogQnkgZGVmYXVsdCwgZGVsZXRpb24gcHJvdGVjdGlvbiBpcyBkaXNhYmxlZC5cbiAqIEVuYWJsZSBpZiBleHBsaWNpdGx5IHByb3ZpZGVkIG9yIGlmIHRoZSBSZW1vdmFsUG9saWN5IGhhcyBiZWVuIHNldCB0byBSRVRBSU5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGRlZmF1bHREZWxldGlvblByb3RlY3Rpb24oZGVsZXRpb25Qcm90ZWN0aW9uPzogYm9vbGVhbiwgcmVtb3ZhbFBvbGljeT86IFJlbW92YWxQb2xpY3kpOiBib29sZWFuIHwgdW5kZWZpbmVkIHtcbiAgcmV0dXJuIGRlbGV0aW9uUHJvdGVjdGlvbiA/PyAocmVtb3ZhbFBvbGljeSA9PT0gUmVtb3ZhbFBvbGljeS5SRVRBSU4gPyB0cnVlIDogdW5kZWZpbmVkKTtcbn1cblxuLyoqXG4gKiBSZW5kZXJzIHRoZSBjcmVkZW50aWFscyBmb3IgYW4gaW5zdGFuY2Ugb3IgY2x1c3RlclxuICovXG5leHBvcnQgZnVuY3Rpb24gcmVuZGVyQ3JlZGVudGlhbHMoc2NvcGU6IENvbnN0cnVjdCwgZW5naW5lOiBJRW5naW5lLCBjcmVkZW50aWFscz86IENyZWRlbnRpYWxzKTogQ3JlZGVudGlhbHMge1xuICBsZXQgcmVuZGVyZWRDcmVkZW50aWFscyA9IGNyZWRlbnRpYWxzID8/IENyZWRlbnRpYWxzLmZyb21Vc2VybmFtZShlbmdpbmUuZGVmYXVsdFVzZXJuYW1lID8/ICdhZG1pbicpOyAvLyBGb3IgYmFja3dhcmRzIGNvbXBhdGliaWx0eVxuXG4gIGlmICghcmVuZGVyZWRDcmVkZW50aWFscy5zZWNyZXQgJiYgIXJlbmRlcmVkQ3JlZGVudGlhbHMucGFzc3dvcmQpIHtcbiAgICByZW5kZXJlZENyZWRlbnRpYWxzID0gQ3JlZGVudGlhbHMuZnJvbVNlY3JldChcbiAgICAgIG5ldyBEYXRhYmFzZVNlY3JldChzY29wZSwgJ1NlY3JldCcsIHtcbiAgICAgICAgdXNlcm5hbWU6IHJlbmRlcmVkQ3JlZGVudGlhbHMudXNlcm5hbWUsXG4gICAgICAgIHNlY3JldE5hbWU6IHJlbmRlcmVkQ3JlZGVudGlhbHMuc2VjcmV0TmFtZSxcbiAgICAgICAgZW5jcnlwdGlvbktleTogcmVuZGVyZWRDcmVkZW50aWFscy5lbmNyeXB0aW9uS2V5LFxuICAgICAgICBleGNsdWRlQ2hhcmFjdGVyczogcmVuZGVyZWRDcmVkZW50aWFscy5leGNsdWRlQ2hhcmFjdGVycyxcbiAgICAgICAgLy8gaWYgdXNlcm5hbWUgbXVzdCBiZSByZWZlcmVuY2VkIGFzIGEgc3RyaW5nIHdlIGNhbiBzYWZlbHkgcmVwbGFjZSB0aGVcbiAgICAgICAgLy8gc2VjcmV0IHdoZW4gY3VzdG9taXphdGlvbiBvcHRpb25zIGFyZSBjaGFuZ2VkIHdpdGhvdXQgcmlza2luZyBhIHJlcGxhY2VtZW50XG4gICAgICAgIHJlcGxhY2VPblBhc3N3b3JkQ3JpdGVyaWFDaGFuZ2VzOiBjcmVkZW50aWFscz8udXNlcm5hbWVBc1N0cmluZyxcbiAgICAgICAgcmVwbGljYVJlZ2lvbnM6IHJlbmRlcmVkQ3JlZGVudGlhbHMucmVwbGljYVJlZ2lvbnMsXG4gICAgICB9KSxcbiAgICAgIC8vIHBhc3MgdXNlcm5hbWUgaWYgaXQgbXVzdCBiZSByZWZlcmVuY2VkIGFzIGEgc3RyaW5nXG4gICAgICBjcmVkZW50aWFscz8udXNlcm5hbWVBc1N0cmluZyA/IHJlbmRlcmVkQ3JlZGVudGlhbHMudXNlcm5hbWUgOiB1bmRlZmluZWQsXG4gICAgKTtcbiAgfVxuXG4gIHJldHVybiByZW5kZXJlZENyZWRlbnRpYWxzO1xufVxuXG4vKipcbiAqIFRoZSBSZW1vdmFsUG9saWN5IHRoYXQgc2hvdWxkIGJlIGFwcGxpZWQgdG8gYSBcImhlbHBlclwiIHJlc291cmNlLCBpZiB0aGUgYmFzZSByZXNvdXJjZSBoYXMgdGhlIGdpdmVuIHJlbW92YWwgcG9saWN5XG4gKlxuICogLSBGb3IgQ2x1c3RlcnMsIHRoaXMgZGV0ZXJtaW5lcyB0aGUgUmVtb3ZhbFBvbGljeSBmb3IgSW5zdGFuY2VzL1N1Ym5ldEdyb3Vwcy5cbiAqIC0gRm9yIEluc3RhbmNlcywgdGhpcyBkZXRlcm1pbmVzIHRoZSBSZW1vdmFsUG9saWN5IGZvciBTdWJuZXRHcm91cHMuXG4gKlxuICogSWYgdGhlIGJhc2VQb2xpY3kgaXM6XG4gKlxuICogIERFU1RST1kgb3IgU05BUFNIT1QgLT4gREVTVFJPWSAoc25hcHNob3QgaXMgZ29vZCBlbm91Z2ggdG8gcmVjcmVhdGUpXG4gKiAgUkVUQUlOICAgICAgICAgICAgICAtPiBSRVRBSU4gIChhbnl0aGluZyBlbHNlIHdpbGwgbG9zZSBkYXRhIG9yIGZhaWwgdG8gZGVwbG95KVxuICogICh1bmRlZmluZWQpICAgICAgICAgLT4gREVTVFJPWSAoYmFzZSBwb2xpY3kgaXMgYXNzdW1lZCB0byBiZSBTTkFQU0hPVClcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGhlbHBlclJlbW92YWxQb2xpY3koYmFzZVBvbGljeT86IFJlbW92YWxQb2xpY3kpOiBSZW1vdmFsUG9saWN5IHtcbiAgcmV0dXJuIGJhc2VQb2xpY3kgPT09IFJlbW92YWxQb2xpY3kuUkVUQUlOXG4gICAgPyBSZW1vdmFsUG9saWN5LlJFVEFJTlxuICAgIDogUmVtb3ZhbFBvbGljeS5ERVNUUk9ZO1xufVxuXG4vKipcbiAqIFJldHVybiBhIGdpdmVuIHZhbHVlIHVubGVzcyBpdCdzIHRoZSBzYW1lIGFzIGFub3RoZXIgdmFsdWVcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIHJlbmRlclVubGVzczxBPih2YWx1ZTogQSwgc3VwcHJlc3NWYWx1ZTogQSk6IEEgfCB1bmRlZmluZWQge1xuICByZXR1cm4gdmFsdWUgPT09IHN1cHByZXNzVmFsdWUgPyB1bmRlZmluZWQgOiB2YWx1ZTtcbn1cbiJdfQ==