from pydantic import BaseModel, PrivateAttr
from typing import List, TextIO, Union, Optional
import os

class Type(BaseModel):
    new_type: str
    old_type: str

class Enum (BaseModel):
    name: str
    values: List[str]
    type: Optional[str]
    start_value: Optional[int]


class Constant (BaseModel):
    name: str
    value: Union[int, str]


class Outputer(BaseModel):
    path: str
    _output: TextIO = PrivateAttr()
    _comment_mark: str = PrivateAttr()
    _comment_indentation: int = PrivateAttr() # doesn't apply to the comment in output_header()

    def __init__(self, *args, comment_mark="#", comment_indentation=0, **kwargs):
        super().__init__(*args, **kwargs)
        self._create_dir_if_not_exists(self.path)
        self._output = open(self.path, "w")
        self._comment_mark = comment_mark
        self._comment_indentation = comment_indentation

    def __del__(self):
        self._output.close()

    @staticmethod
    def _create_dir_if_not_exists(path):
        os.makedirs(os.path.dirname(path), exist_ok=True)
        
    def output_enum(self, enum: Enum, prefix="", assignment="=", suffix=""):
        start_value = enum.start_value if enum.start_value else 0
        for (i, value) in enumerate(enum.values):
            self._output.write(f"{prefix}{value} {assignment} {i+start_value}{suffix}\n")

    def output_comment(self, comment):
        indent = '\t' * self._comment_indentation
        self._output.write(f"\n{indent}{self._comment_mark} {comment}\n")

    def output_constant(self, constant: Constant, prefix="", assignment="=", suffix=""):
        if type(constant.value) == int:
            value = constant.value
        elif type(constant.value) == str:
            value = f'"{constant.value}"'
        else:
            raise Exception("Internal error - illegal constant type. %s", type(constant.value))
        self._output.write(f"{prefix}{constant.name} {assignment} {value}{suffix}\n")

    def output_type(self, type: Type, def_command="", suffix=""):
        self._output.write(f"{def_command} {type.new_type} {type.old_type}{suffix}\n")

    def output_header(self):
        self._output.write(f"{self._comment_mark} autogenerated by reconstant - do not edit!\n")

    def output_footer(self):
        pass

