"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFormationTemplate = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const s3_assets = require("@aws-cdk/aws-s3-assets");
const util_1 = require("./private/util");
/**
 * Represents the Product Provisioning Artifact Template.
 */
class CloudFormationTemplate {
    /**
     * Template from URL
     * @param url The url that points to the provisioning artifacts template
     */
    static fromUrl(url) {
        return new CloudFormationUrlTemplate(url);
    }
    /**
     * Loads the provisioning artifacts template from a local disk path.
     *
     * @param path A file containing the provisioning artifacts
     */
    static fromAsset(path, options) {
        return new CloudFormationAssetTemplate(path, options);
    }
    /**
     * Creates a product with the resources defined in the given product stack.
     */
    static fromProductStack(productStack) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_servicecatalog_ProductStack(productStack);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromProductStack);
            }
            throw error;
        }
        return new CloudFormationProductStackTemplate(productStack);
    }
}
exports.CloudFormationTemplate = CloudFormationTemplate;
_a = JSII_RTTI_SYMBOL_1;
CloudFormationTemplate[_a] = { fqn: "@aws-cdk/aws-servicecatalog.CloudFormationTemplate", version: "1.174.0" };
/**
 * Template code from a Url.
 */
class CloudFormationUrlTemplate extends CloudFormationTemplate {
    constructor(url) {
        super();
        this.url = url;
    }
    bind(_scope) {
        return {
            httpUrl: this.url,
        };
    }
}
/**
 * Template from a local file.
 */
class CloudFormationAssetTemplate extends CloudFormationTemplate {
    /**
     * @param path The path to the asset file.
     */
    constructor(path, options = {}) {
        super();
        this.path = path;
        this.options = options;
    }
    bind(scope) {
        // If the same AssetCode is used multiple times, retain only the first instantiation.
        if (!this.asset) {
            this.asset = new s3_assets.Asset(scope, `Template${util_1.hashValues(this.path)}`, {
                path: this.path,
                ...this.options,
            });
        }
        return {
            httpUrl: this.asset.httpUrl,
        };
    }
}
/**
 * Template from a CDK defined product stack.
 */
class CloudFormationProductStackTemplate extends CloudFormationTemplate {
    /**
     * @param productStack A service catalog product stack.
     */
    constructor(productStack) {
        super();
        this.productStack = productStack;
    }
    bind(_scope) {
        return {
            httpUrl: this.productStack._getTemplateUrl(),
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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