from .context import gpytoolbox as gpy
from .context import numpy as np
from .context import unittest

class TestTipAngles(unittest.TestCase):

    def test_single_triangle_2d(self):
        v = np.array([[0.0,0.0],[1.0,0.0],[0.0,1.0]])
        f = np.array([[0,1,2]],dtype=int)
        alpha = gpy.tip_angles(v,f)
        alpha_gt = np.array([[np.pi/2., np.pi/4., np.pi/4.]])
        self.assertTrue(np.isclose(alpha, alpha_gt).all())
    
    def test_single_triangle_3d(self):
        v = np.array([[0.0,0.0,0.0],[1.0,0.0,0.0],[0.0,0.0,1.0]])
        f = np.array([[0,1,2]],dtype=int)
        alpha = gpy.tip_angles(v,f)
        alpha_gt = np.array([[np.pi/2., np.pi/4., np.pi/4.]])
        self.assertTrue(np.isclose(alpha, alpha_gt).all())

    def test_bunny_oded(self):
        v,f = gpy.read_mesh("test/unit_tests_data/bunny_oded.obj")

        alpha = gpy.tip_angles(v,f)
        alpha_gt = np.array([[0.86029597, 1.63838384, 0.64291284],
       [1.20782338, 0.72059966, 1.21316962],
       [1.50321674, 0.94347058, 0.69490534],
       [1.18826541, 1.03479611, 0.91853114],
       [0.95263444, 1.16227662, 1.02668159],
       [1.05950815, 1.06481427, 1.01727023],
       [0.76752026, 1.37473446, 0.99933794],
       [0.98991619, 1.02784168, 1.12383478],
       [0.92392304, 1.16861338, 1.04905624],
       [0.9565451 , 1.07762188, 1.10742567],
       [1.07146829, 0.98670153, 1.08342283],
       [1.0278015 , 1.03002578, 1.08376538],
       [0.92619175, 1.05820939, 1.15719151],
       [1.31165664, 0.91187293, 0.91806308],
       [1.14053249, 0.95162407, 1.04943609],
       [1.05376994, 0.98634721, 1.10147551],
       [1.0865005 , 1.04929526, 1.00579689],
       [0.91424005, 1.12342307, 1.10392954],
       [1.07684674, 1.07018424, 0.99456167],
       [0.9999804 , 1.03786952, 1.10374273],
       [1.05887004, 1.09464968, 0.98807294],
       [1.07956625, 1.08016988, 0.98185653],
       [0.93317133, 1.06017688, 1.14824445],
       [0.89105763, 1.19142295, 1.05911207],
       [1.03355677, 0.98289287, 1.12514302],
       [1.09606076, 1.056643  , 0.98888889],
       [1.10585386, 1.01800653, 1.01773227],
       [1.08742272, 1.00759613, 1.0465738 ],
       [1.10049651, 0.92805097, 1.11304517],
       [1.03243159, 1.0904845 , 1.01867656],
       [1.01827631, 1.0809699 , 1.04234645],
       [1.03858888, 1.10396029, 0.99904349],
       [1.10501051, 0.96572415, 1.07085799],
       [1.03745487, 1.07801152, 1.02612627],
       [0.95466775, 1.0944505 , 1.09247441],
       [1.16610164, 1.03615087, 0.93934014],
       [1.19296986, 1.058941  , 0.88968179],
       [0.91656194, 1.12980897, 1.09522175],
       [0.99250186, 1.14055773, 1.00853306],
       [1.00549851, 1.06902645, 1.06706769],
       [1.03813026, 1.17283926, 0.93062314],
       [1.03098466, 1.04448327, 1.06612472],
       [1.0488913 , 1.08561343, 1.00708793],
       [1.07661852, 1.00735735, 1.05761678],
       [1.0151768 , 1.06731997, 1.05909589],
       [1.07233874, 1.00685397, 1.06239995],
       [1.13332346, 1.00090422, 1.00736498],
       [1.00667238, 1.01307982, 1.12184044],
       [1.03488425, 1.13881952, 0.96788888],
       [1.11793848, 0.96222276, 1.06143141],
       [0.97158981, 1.07398602, 1.09601682],
       [1.1917723 , 1.10077291, 0.84904745],
       [0.87173839, 1.11214276, 1.1577115 ],
       [0.90258293, 1.10153014, 1.13747958],
       [1.14518915, 0.94588203, 1.05052147],
       [0.98410761, 1.16467965, 0.9928054 ],
       [0.97672356, 1.09576358, 1.06910551],
       [0.85674036, 1.11978695, 1.16506534],
       [0.97604098, 1.10491398, 1.06063769],
       [1.03869828, 1.01690476, 1.08598962],
       [1.05162137, 1.03922358, 1.0507477 ],
       [1.08968179, 1.01681246, 1.0350984 ],
       [1.03030547, 1.06517483, 1.04611235],
       [1.06086679, 1.07404298, 1.00668288],
       [0.94707734, 1.04595816, 1.14855715],
       [1.15520694, 0.95996221, 1.02642351],
       [1.01797822, 1.119592  , 1.00402243],
       [1.1515394 , 0.89789303, 1.09216022],
       [1.00745676, 1.08986949, 1.0442664 ],
       [1.08085407, 1.1149573 , 0.94578128],
       [1.02813117, 0.93342455, 1.18003693],
       [1.08751799, 1.09848156, 0.9555931 ],
       [1.09228653, 1.11496983, 0.93433629],
       [0.93221864, 1.0601551 , 1.14921891],
       [1.17021885, 1.00169368, 0.96968012],
       [0.82635958, 0.97113872, 1.34409435],
       [1.06638616, 1.07500294, 1.00020355],
       [1.13772658, 0.90614032, 1.09772575],
       [0.95638843, 0.98639521, 1.19880901],
       [1.03717058, 1.04409501, 1.06032706],
       [1.17287543, 1.01210687, 0.95661035],
       [1.01368804, 1.04200243, 1.08590218],
       [1.07756948, 1.06328102, 1.00074215],
       [1.03648889, 1.02612657, 1.07897719],
       [1.04097421, 0.9554863 , 1.14513215],
       [1.18474281, 0.95117803, 1.00567182],
       [1.02073692, 0.99880341, 1.12205232],
       [1.06079858, 1.11678755, 0.96400653],
       [1.04301031, 1.00696605, 1.0916163 ],
       [1.05938193, 1.16125651, 0.92095421],
       [1.10313226, 1.08636674, 0.95209366],
       [1.06950386, 0.96809782, 1.10399098],
       [0.82234343, 1.25973363, 1.05951559],
       [1.26274862, 0.90366522, 0.97517882],
       [1.17809751, 0.91663506, 1.04686008],
       [1.06034577, 1.14804575, 0.93320114],
       [1.0833086 , 1.15969738, 0.89858668],
       [1.07251356, 0.87564689, 1.1934322 ],
       [1.37276845, 0.72114522, 1.04767898],
       [1.03282372, 1.00194705, 1.10682188],
       [0.87932699, 1.13158869, 1.13067698],
       [0.97551743, 1.16055156, 1.00552366],
       [1.17859455, 0.95658797, 1.00641013],
       [1.03154987, 1.06355694, 1.04648584],
       [0.98864041, 1.15446063, 0.99849161],
       [1.08657005, 1.02942213, 1.02560047],
       [1.08277536, 1.00002387, 1.05879343],
       [1.0765905 , 1.01256501, 1.05243715],
       [1.10555061, 1.02371049, 1.01233156],
       [0.86141041, 1.09633188, 1.18385037],
       [0.97486644, 1.03988181, 1.1268444 ],
       [1.0143452 , 1.18887515, 0.93837231],
       [0.97641467, 1.05707717, 1.10810082],
       [1.09041779, 1.03863913, 1.01253573],
       [1.12882179, 1.20031428, 0.81245658],
       [1.03830291, 1.05946719, 1.04382255],
       [1.01774034, 1.07798784, 1.04586448],
       [1.14203394, 1.0627847 , 0.93677402],
       [1.07537885, 1.14302642, 0.92318739],
       [1.03703796, 1.09275262, 1.01180208],
       [0.98264967, 1.05825647, 1.10068652],
       [0.96212096, 1.07590479, 1.1035669 ],
       [0.63884542, 0.86497416, 1.63777308],
       [0.9995846 , 1.05009856, 1.0919095 ],
       [1.24427888, 0.85810163, 1.03921214],
       [1.04624041, 1.01149237, 1.08385988],
       [1.05863625, 0.95861138, 1.12434503],
       [1.25650963, 0.92150357, 0.96357945],
       [0.92341943, 1.42171983, 0.79645339],
       [1.0347281 , 1.04938262, 1.05748193],
       [1.22904254, 0.94101957, 0.97153055],
       [1.01983597, 1.0348969 , 1.08685979],
       [1.0106686 , 1.04552514, 1.08539891],
       [1.09357111, 1.05203514, 0.9959864 ],
       [1.01919879, 0.99573363, 1.12666023],
       [1.02525394, 1.13133675, 0.98500196],
       [0.78819655, 0.86282161, 1.49057449],
       [1.09544005, 1.02422706, 1.02192555],
       [1.01299058, 0.98303432, 1.14556775],
       [1.0769974 , 0.97462985, 1.0899654 ],
       [1.10961089, 0.99125906, 1.0407227 ],
       [1.08054321, 0.99064681, 1.07040263],
       [0.8634827 , 0.7223662 , 1.55574376],
       [1.05099771, 0.9420874 , 1.14850755],
       [1.06903934, 1.03655111, 1.0360022 ],
       [0.9680118 , 0.78361786, 1.389963  ],
       [1.00523964, 0.99805165, 1.13830136],
       [0.98689475, 1.05009616, 1.10460174],
       [1.04870766, 1.11966201, 0.97322298],
       [1.12084264, 0.9557853 , 1.06496472],
       [1.07581074, 1.12373317, 0.94204875],
       [1.50379762, 0.94800788, 0.68978716],
       [1.04055701, 1.05216276, 1.04887288],
       [1.1599638 , 1.2360829 , 0.74554596],
       [1.08607374, 1.03867227, 1.01684664],
       [0.99003809, 1.0807939 , 1.07076066],
       [1.15751647, 0.9633602 , 1.02071598],
       [1.81494543, 0.66348708, 0.66316014],
       [1.2050208 , 0.84805598, 1.08851587],
       [0.63805072, 1.27409066, 1.22945127],
       [1.01144991, 1.07985601, 1.05028673],
       [0.94288182, 1.05703715, 1.14167369],
       [1.0572208 , 1.08495635, 0.99941551],
       [1.0987066 , 1.00157884, 1.04130721],
       [1.012601  , 1.08926985, 1.03972181],
       [1.09203882, 1.04032459, 1.00922924],
       [0.98890045, 1.06528925, 1.08740295],
       [0.86626067, 1.14334822, 1.13198376],
       [1.05430039, 1.04912952, 1.03816274],
       [0.92383324, 1.00382669, 1.21393272],
       [1.03596054, 1.22454361, 0.88108851],
       [1.130211  , 0.97991176, 1.0314699 ],
       [0.99151952, 1.09364001, 1.05643312],
       [1.17925596, 1.11450092, 0.84783577],
       [1.04262286, 1.07718578, 1.02178401],
       [1.06730898, 1.0445214 , 1.02976227],
       [1.01742387, 1.07147784, 1.05269095],
       [0.89073958, 0.44526653, 1.80558655],
       [0.98446684, 1.06851978, 1.08860604],
       [1.16100515, 0.92916942, 1.05141808],
       [1.02818884, 1.03424763, 1.07915618],
       [1.00496312, 1.05101467, 1.08561486],
       [0.99022945, 1.06428572, 1.08707748],
       [1.09022661, 1.09547648, 0.95588956],
       [1.01759216, 1.09601511, 1.02798538],
       [1.03317179, 1.04836604, 1.06005483],
       [0.97962355, 1.15295176, 1.00901734],
       [1.03682161, 1.06843566, 1.03633537],
       [1.084108  , 1.07168758, 0.98579708],
       [1.03089871, 0.99835782, 1.11233613],
       [0.99175112, 1.19643331, 0.95340823],
       [0.91536855, 1.05528238, 1.17094172],
       [0.87742668, 1.17528081, 1.08888517],
       [1.05909389, 1.05868496, 1.0238138 ],
       [1.13407148, 0.93234631, 1.07517486],
       [0.99749372, 1.05510765, 1.08899128],
       [1.00378435, 1.04935406, 1.08845425],
       [1.08259856, 1.07666891, 0.98232518],
       [1.02864152, 1.03104659, 1.08190454],
       [1.0648174 , 1.07781387, 0.99896139],
       [1.07870634, 0.93836219, 1.12452412],
       [0.98059424, 1.20535023, 0.95564818],
       [1.08500567, 1.05360678, 1.0029802 ],
       [0.95817071, 1.23228968, 0.95113227],
       [0.58586132, 1.20893136, 1.34679998],
       [1.15653883, 1.06456513, 0.92048869],
       [1.08934807, 1.02692469, 1.02531989],
       [0.98915951, 1.08543948, 1.06699367],
       [0.91207976, 0.98810867, 1.24140423],
       [0.94877102, 1.08210865, 1.11071299],
       [1.00148271, 1.05818936, 1.08192059],
       [0.95158625, 1.00232558, 1.18768083],
       [1.33187845, 1.32343226, 0.48628194],
       [1.06612502, 1.09347615, 0.98199149],
       [1.02240352, 1.01887386, 1.10031527],
       [0.89000776, 1.20197428, 1.04961062],
       [1.04833752, 1.09564886, 0.99760627],
       [1.09921381, 0.98768232, 1.05469653],
       [1.09588164, 0.97667495, 1.06903606],
       [1.01344367, 1.05063428, 1.07751471],
       [1.07897095, 1.01673679, 1.04588491],
       [1.06328855, 1.0443089 , 1.0339952 ],
       [1.08231854, 1.01429212, 1.044982  ],
       [1.0860676 , 1.02109199, 1.03443306],
       [1.07164519, 1.01433495, 1.05561252],
       [1.11889985, 1.0149993 , 1.0076935 ],
       [1.1315851 , 1.06924923, 0.94075832],
       [1.16310708, 0.90766782, 1.07081776],
       [1.14755591, 0.92970374, 1.064333  ],
       [1.10909645, 1.0766426 , 0.9558536 ],
       [1.05657259, 1.00317931, 1.08184075],
       [0.93689216, 1.09161831, 1.11308219],
       [1.04559724, 1.09896373, 0.99703168],
       [1.05157103, 1.10130951, 0.98871211],
       [1.00417063, 1.0408008 , 1.09662123],
       [1.09093517, 1.02743669, 1.02322079],
       [1.04848204, 1.01968477, 1.07342585],
       [0.94451878, 1.14057963, 1.05649425],
       [1.14153267, 0.96412448, 1.0359355 ],
       [1.00450567, 1.06183414, 1.07525284],
       [1.03494297, 1.08660702, 1.02004266],
       [1.24355544, 0.72130722, 1.17672999],
       [0.4515745 , 1.64830219, 1.04171597],
       [1.17978309, 1.00272202, 0.95908754],
       [0.95417189, 1.07901391, 1.10840685],
       [1.02685451, 1.04265447, 1.07208368],
       [1.10304912, 1.02844456, 1.01009897],
       [1.0058616 , 1.08185154, 1.05387951],
       [1.05540475, 1.06483777, 1.02135013],
       [1.05355688, 1.14934487, 0.93869091],
       [1.13609958, 1.08541908, 0.920074  ],
       [1.12458425, 0.97547187, 1.04153653],
       [1.01397737, 1.07425981, 1.05335547],
       [1.02965447, 0.93778167, 1.17415652],
       [1.07656984, 1.02673475, 1.03828807],
       [1.07961931, 1.01497962, 1.04699373],
       [1.07545918, 1.01205255, 1.05408093],
       [1.03947136, 1.0791122 , 1.0230091 ],
       [1.04590767, 1.04039005, 1.05529493],
       [1.04774912, 1.05746259, 1.03638095],
       [1.08367156, 1.03674728, 1.02117381],
       [1.00841443, 1.0811173 , 1.05206092],
       [1.04026326, 1.0176353 , 1.0836941 ],
       [1.07492572, 1.01500968, 1.05165725],
       [1.07143177, 0.9919314 , 1.07822948],
       [1.00290657, 1.09367152, 1.04501456],
       [1.0332115 , 1.14060585, 0.96777531],
       [1.05698617, 1.13851118, 0.94609531],
       [1.10208708, 0.96816155, 1.07134403],
       [1.06316605, 1.06756039, 1.01086621],
       [0.94023489, 1.10078471, 1.10057305],
       [1.03614443, 1.0157774 , 1.08967083],
       [1.00143897, 1.05695382, 1.08319986],
       [1.04216175, 1.09867036, 1.00076055],
       [1.12768455, 0.97399486, 1.03991325],
       [1.05234674, 1.11033568, 0.97891023],
       [1.15062349, 1.03318521, 0.95778395],
       [1.05819958, 0.93275517, 1.15063791],
       [0.94210943, 1.00392883, 1.19555439],
       [0.99418048, 1.12369399, 1.02371818],
       [1.05618644, 1.01442104, 1.07098517],
       [1.12883409, 1.05238004, 0.96037852],
       [0.589671  , 0.83304068, 1.71888098],
       [1.12235359, 1.00174634, 1.01749272],
       [1.03876649, 1.03181429, 1.07101187],
       [1.00983404, 1.0424726 , 1.08928601],
       [1.05330072, 1.12750632, 0.96078562],
       [1.10791326, 0.94338293, 1.09029647],
       [0.99536625, 1.07529148, 1.07093493],
       [1.09426731, 1.14205161, 0.90527374],
       [1.12404614, 1.18206703, 0.83547948],
       [0.98847837, 1.01926451, 1.13384977],
       [1.07074314, 1.05960482, 1.01124469],
       [1.02801312, 1.09793077, 1.01564876],
       [1.13232671, 1.08449805, 0.92476789],
       [1.00836744, 1.04252074, 1.09070447],
       [1.04077788, 1.10041826, 1.00039651],
       [1.05402648, 1.07748417, 1.010082  ],
       [1.06001539, 1.05080604, 1.03077122],
       [1.06265292, 1.0229194 , 1.05602033],
       [1.04510289, 1.03484078, 1.06164898],
       [1.00937442, 1.04752908, 1.08468915],
       [1.05096095, 1.02245425, 1.06817745],
       [1.09376975, 1.01484897, 1.03297393],
       [1.04720984, 1.07922774, 1.01515507],
       [1.05404118, 1.01011353, 1.07743794],
       [0.99212447, 1.06401017, 1.08545801],
       [0.97733504, 1.00779457, 1.15646304],
       [0.99920888, 1.03189045, 1.11049332],
       [1.05927043, 1.11730663, 0.96501559],
       [1.06868637, 1.00760205, 1.06530423],
       [1.05849668, 1.0097658 , 1.07333018],
       [1.09095239, 0.9566534 , 1.09398686],
       [1.02464745, 1.04092311, 1.0760221 ],
       [1.10890466, 0.97751064, 1.05517735],
       [0.97618826, 1.05186596, 1.11353844],
       [1.0158767 , 1.05089395, 1.074822  ],
       [1.11663589, 0.97578496, 1.0491718 ],
       [0.97094928, 1.15798112, 1.01266226],
       [0.9005674 , 1.12795396, 1.1130713 ],
       [1.08554005, 1.00072501, 1.0553276 ],
       [1.07936335, 1.00337576, 1.05885355],
       [1.04438954, 1.06726908, 1.02993404],
       [1.03789586, 1.05689584, 1.04680095],
       [0.89610261, 1.16984214, 1.0756479 ],
       [0.86995057, 1.33838855, 0.93325353],
       [1.10378434, 1.01622615, 1.02158216],
       [1.00976567, 1.11661815, 1.01520882],
       [1.05821905, 1.03973725, 1.04363635],
       [1.06309252, 1.00668131, 1.07181883],
       [1.07505248, 1.04645409, 1.02008608],
       [1.01221338, 1.07286545, 1.05651382],
       [0.91335893, 0.9337443 , 1.29448943],
       [1.17928671, 0.95862178, 1.00368416],
       [1.1873364 , 0.89669335, 1.05756291],
       [1.03035112, 1.10820001, 1.00304152],
       [1.06394628, 1.01445507, 1.0631913 ],
       [1.03279636, 1.02358882, 1.08520747],
       [0.96170519, 1.10736019, 1.07252728],
       [1.08035114, 1.02772115, 1.03352036],
       [1.08781965, 1.01748876, 1.03628424],
       [1.03227225, 1.03556794, 1.07375246],
       [1.06195544, 1.05617892, 1.02345829],
       [1.04195559, 1.06830672, 1.03133034],
       [1.04134275, 1.07900832, 1.02124158],
       [1.06468408, 1.0246983 , 1.05221027],
       [1.08107071, 1.00894887, 1.05157308],
       [1.08203752, 1.02692763, 1.03262751],
       [1.05789397, 1.09239806, 0.99130062],
       [1.03497077, 1.025998  , 1.08062389],
       [1.05761436, 1.07790253, 1.00607576],
       [1.05883177, 1.06201383, 1.02074705],
       [1.00202289, 1.13434971, 1.00522006],
       [1.10485498, 1.03833954, 0.99839813],
       [1.10096251, 0.98651317, 1.05411697],
       [1.0414407 , 1.10276949, 0.99738247],
       [1.06540626, 1.05948713, 1.01669926],
       [1.09929504, 1.07978877, 0.96250884],
       [1.02440987, 1.07559419, 1.04158859],
       [1.06371775, 1.05728575, 1.02058916],
       [1.03021407, 1.08250899, 1.02886959],
       [1.09810863, 1.01562458, 1.02785945],
       [1.12347146, 1.01216948, 1.00595171],
       [0.99361842, 1.03630736, 1.11166686],
       [1.03795281, 0.90374052, 1.19989932],
       [0.90615908, 1.12733529, 1.10809829],
       [0.97320764, 1.0436574 , 1.12472762],
       [1.15524793, 0.98742387, 0.99892085],
       [1.07221208, 1.06690057, 1.00248   ],
       [1.05001113, 1.081735  , 1.00984652],
       [1.1811858 , 0.98128769, 0.97911917],
       [0.89601332, 1.05905838, 1.18652095],
       [1.13636926, 0.94426148, 1.06096191],
       [1.07985015, 0.96754849, 1.09419402],
       [1.06128558, 1.07485757, 1.0054495 ],
       [0.98898518, 1.0750777 , 1.07752978],
       [1.12768354, 1.11065509, 0.90325402],
       [1.08005275, 1.0510959 , 1.010444  ],
       [1.15417782, 1.01359163, 0.97382321],
       [0.93176038, 1.06755442, 1.14227786],
       [0.95447621, 1.02516812, 1.16194832],
       [1.06507153, 1.0168187 , 1.05970242],
       [1.02742632, 1.00285993, 1.1113064 ],
       [1.01824487, 1.09101295, 1.03233483],
       [1.03002323, 1.09454163, 1.01702779],
       [1.07686585, 1.09023707, 0.97448973],
       [1.03025828, 1.09614746, 1.01518692],
       [1.01552611, 1.03907334, 1.0869932 ],
       [1.08291781, 1.02631217, 1.03236267],
       [1.01809095, 1.05127186, 1.07222984],
       [1.01318769, 1.0455738 , 1.08283116],
       [1.0335121 , 1.08936874, 1.01871182],
       [1.00868032, 1.05086597, 1.08204636],
       [1.03831501, 1.09296541, 1.01031224],
       [1.0204282 , 1.0314723 , 1.08969215],
       [1.05626725, 1.0671121 , 1.0182133 ],
       [1.02308531, 1.08415345, 1.03435389],
       [1.06492898, 1.01323691, 1.06342676],
       [1.08039328, 1.01551747, 1.04568191],
       [0.99636184, 1.11470856, 1.03052225],
       [1.04433417, 0.97729759, 1.11996089],
       [1.04679018, 1.06222865, 1.03257383],
       [1.00669562, 1.03439952, 1.10049751],
       [1.02285855, 1.04083176, 1.07790234],
       [1.01435913, 1.07258937, 1.05464415],
       [1.06952921, 1.13321057, 0.93885287],
       [1.03734831, 1.06433826, 1.03990608],
       [1.05715504, 0.9932804 , 1.09115721],
       [1.08245875, 1.04845509, 1.01067881],
       [1.18306741, 0.96441911, 0.99410613],
       [1.04683103, 1.07863516, 1.01612646],
       [0.97382893, 1.08416514, 1.08359858],
       [1.15706669, 1.13211856, 0.8524074 ],
       [1.13313431, 0.91336602, 1.09509233],
       [1.03995163, 1.01200967, 1.08963136],
       [1.09022351, 1.03976586, 1.01160328],
       [1.08634893, 1.03812072, 1.017123  ],
       [1.00528596, 1.02873712, 1.10756957],
       [1.11612846, 1.00747818, 1.01798601],
       [1.28265897, 0.87407558, 0.9848581 ],
       [0.94763721, 0.97051561, 1.22343983],
       [0.94649243, 1.12076062, 1.07433961],
       [1.07473693, 0.93083702, 1.1360187 ],
       [1.09184854, 0.97386542, 1.07587869],
       [1.04525029, 1.00260643, 1.09373593],
       [1.05666362, 1.00209633, 1.0828327 ],
       [1.040371  , 1.02317281, 1.07804884],
       [0.97173065, 1.22550517, 0.94435683],
       [0.96036424, 1.19777422, 0.98345419],
       [0.93150806, 1.18632641, 1.02375819],
       [1.13544053, 0.96071804, 1.04543409],
       [1.01628586, 1.05322898, 1.07207781],
       [1.08833158, 1.0341221 , 1.01913897],
       [1.04880147, 1.0098896 , 1.08290158],
       [1.09068731, 1.03487762, 1.01602772],
       [0.99117614, 1.06630859, 1.08410793],
       [1.03305076, 1.02061793, 1.08792397],
       [1.01967546, 1.08908267, 1.03283453],
       [1.00703785, 1.09084681, 1.04370799],
       [1.07333257, 1.03052812, 1.03773197],
       [1.03454294, 1.10032512, 1.00672459],
       [1.01178632, 1.02081593, 1.1089904 ],
       [1.09079287, 1.01637315, 1.03442663],
       [1.03205046, 1.08136776, 1.02817444],
       [1.08221393, 1.0209504 , 1.03842833],
       [1.07783818, 1.03649548, 1.02725899],
       [1.01319188, 1.0057815 , 1.12261926],
       [1.04958922, 1.00494966, 1.08705378],
       [1.05671732, 1.06580332, 1.01907201],
       [1.05026796, 1.07188095, 1.01944375],
       [1.00577765, 1.02700724, 1.10880776],
       [1.0042153 , 0.99654527, 1.14083209],
       [1.09629692, 0.98950032, 1.05579541],
       [0.99258939, 1.06745889, 1.08154437],
       [1.03198007, 1.08335883, 1.02625375],
       [1.04364155, 1.02481214, 1.07313896],
       [1.16931478, 0.98021391, 0.99206397],
       [1.06476105, 0.98776143, 1.08907017],
       [1.08548884, 1.01973374, 1.03637007],
       [1.03018113, 1.11257433, 0.9988372 ],
       [1.05760431, 1.15986691, 0.92412143],
       [0.86846543, 1.1091598 , 1.16396742],
       [1.03027496, 1.11638821, 0.99492948],
       [1.05784338, 1.09500571, 0.98874356],
       [1.13832892, 0.91215872, 1.09110502],
       [1.21702023, 1.06489858, 0.85967385],
       [1.05205165, 1.01303782, 1.07650318],
       [1.0184485 , 1.03115282, 1.09199133],
       [0.96018599, 1.10449765, 1.07690902],
       [1.08072712, 0.99275604, 1.06810949],
       [0.98945756, 1.01647447, 1.13566062],
       [0.84529373, 1.19427224, 1.10202668],
       [0.94113439, 1.33166835, 0.86878992],
       [0.94042836, 1.17363712, 1.02752718],
       [1.20305287, 0.9837283 , 0.95481148],
       [1.03895786, 1.08957644, 1.01305835],
       [1.07533815, 1.06188999, 1.00436451],
       [1.18775407, 0.90391599, 1.04992259],
       [1.07712693, 1.05022098, 1.01424474],
       [0.98546647, 1.112785  , 1.04334118],
       [1.12431295, 1.03089218, 0.98638752],
       [1.10486167, 1.03455241, 1.00217857],
       [1.0020025 , 1.02740466, 1.11218549],
       [1.04960258, 1.07048773, 1.02150235],
       [1.0505327 , 1.01970912, 1.07135083],
       [1.04832507, 1.02138902, 1.07187857],
       [1.01853716, 1.08994536, 1.03311013],
       [1.02741603, 1.01430424, 1.09987238],
       [1.04849638, 1.07756398, 1.01553229],
       [1.09704965, 1.02141735, 1.02312566],
       [1.02351715, 1.01859407, 1.09948143],
       [1.00936436, 1.08144598, 1.05078231],
       [1.08515701, 1.03367677, 1.02275887],
       [1.04895104, 1.11294599, 0.97969562],
       [1.1112644 , 1.02392983, 1.00639842],
       [1.02144783, 1.03320802, 1.0869368 ],
       [1.05051932, 1.08543061, 1.00564273],
       [1.0202204 , 1.09396137, 1.02741089],
       [1.03695315, 1.03603115, 1.06860835],
       [1.09783581, 1.02574889, 1.01800796],
       [1.06937975, 1.06205088, 1.01016202],
       [1.01151302, 1.06760718, 1.06247245],
       [1.02369333, 1.03752287, 1.08037646],
       [1.09541769, 1.0409462 , 1.00522877],
       [1.09445166, 1.02891167, 1.01822933],
       [1.03564978, 1.09378559, 1.01215729],
       [1.10658753, 1.0158339 , 1.01917122],
       [1.05992387, 1.08483102, 0.99683777],
       [1.07769122, 1.02941074, 1.0344907 ],
       [1.01886428, 1.09411122, 1.02861715],
       [1.04124738, 0.92477035, 1.17557493],
       [1.00525638, 1.08419132, 1.05214495],
       [1.02784681, 1.10925991, 1.00448593],
       [1.05887644, 1.0474438 , 1.03527241],
       [1.08201604, 1.11333427, 0.94624235],
       [1.18988264, 0.85434091, 1.0973691 ],
       [1.10489172, 1.00093811, 1.03576282],
       [1.09093527, 0.98376985, 1.06688754],
       [1.47318929, 1.11114479, 0.55725857],
       [1.27238576, 1.00384606, 0.86536083],
       [1.0229332 , 1.12568708, 0.99297237],
       [1.08929371, 1.02131846, 1.03098048],
       [0.96514491, 1.08569585, 1.09075189],
       [0.97715686, 0.98501201, 1.17942378],
       [0.99371813, 1.18210026, 0.96577426],
       [0.97756334, 0.99194349, 1.17208582],
       [1.1429149 , 1.01997839, 0.97869936],
       [0.85017383, 1.38814781, 0.90327102],
       [1.01992702, 0.93294979, 1.18871584],
       [1.02236142, 1.1393684 , 0.97986284],
       [1.02289485, 1.01933841, 1.0993594 ],
       [1.03388478, 1.01365908, 1.0940488 ],
       [1.15429847, 0.9859499 , 1.00134429],
       [1.02652765, 1.00743588, 1.10762912],
       [1.12750274, 0.98245376, 1.03163615],
       [1.101838  , 0.97747933, 1.06227533],
       [1.01363806, 1.13864898, 0.98930562],
       [1.14399277, 0.98782703, 1.00977285],
       [1.00563237, 1.03377817, 1.10218211],
       [1.06955787, 1.02220345, 1.04983134],
       [1.01991676, 1.06905113, 1.05262477],
       [1.0556353 , 1.04819768, 1.03775967],
       [1.04475741, 1.01014736, 1.08668788],
       [1.00733449, 1.09500166, 1.0392565 ],
       [1.0232281 , 1.04807538, 1.07028917],
       [1.0360338 , 1.00476872, 1.10079014],
       [0.97255289, 1.11526661, 1.05377315],
       [1.07729446, 0.96426857, 1.10002961],
       [1.03065745, 1.07737413, 1.03356107],
       [1.16998158, 0.97906758, 0.99254349],
       [1.12088336, 1.00661057, 1.01409872],
       [1.00307821, 1.10470961, 1.03380483],
       [1.0137451 , 1.03823225, 1.0896153 ],
       [1.08742794, 1.03244711, 1.02171761],
       [1.090352  , 1.03315047, 1.01809019],
       [1.01882768, 1.06311109, 1.05965388],
       [1.01093463, 1.0987337 , 1.03192432],
       [1.00950676, 1.0504246 , 1.08166129],
       [1.02277664, 1.06310079, 1.05571522],
       [1.04707248, 1.07497338, 1.01954679],
       [0.99950322, 1.00918297, 1.13290646],
       [1.05589909, 1.07776422, 1.00792935],
       [1.01662687, 1.0311119 , 1.09385388],
       [1.04211229, 1.08185756, 1.01762281],
       [1.08872408, 0.95703644, 1.09583214],
       [1.04194694, 1.0627403 , 1.03690541],
       [1.00662212, 1.04757503, 1.0873955 ],
       [0.99142779, 0.89226157, 1.25790329],
       [1.00238642, 1.10984825, 1.02935799],
       [1.03060502, 0.97728637, 1.13370126],
       [1.06056415, 0.98638239, 1.09464611],
       [1.08103606, 1.09145743, 0.96909917],
       [1.70821565, 0.73162077, 0.70175623],
       [0.847109  , 1.37227838, 0.92220527],
       [1.01199882, 1.00271426, 1.12687957],
       [1.04968113, 1.05588666, 1.03602486],
       [0.55220684, 1.28461285, 1.30477296],
       [0.50695269, 0.96620475, 1.66843521],
       [0.9416251 , 1.01657941, 1.18338815],
       [1.03319356, 1.06097244, 1.04742665],
       [1.03355479, 1.04810301, 1.05993486],
       [1.07645229, 0.86743187, 1.1977085 ],
       [1.08152254, 1.05835254, 1.00171758],
       [0.99026198, 0.99890613, 1.15242455],
       [1.16040786, 0.98068824, 1.00049655],
       [0.96684864, 1.03524267, 1.13950135],
       [1.06993939, 0.98372396, 1.0879293 ],
       [1.03585224, 1.02074246, 1.08499796],
       [1.10152689, 1.08037448, 0.95969128],
       [1.03828377, 1.03175   , 1.07155888],
       [0.85572868, 1.0730014 , 1.21286258],
       [1.04562188, 1.01066478, 1.08530599],
       [1.01305287, 1.02848925, 1.10005053],
       [1.02040792, 1.06547593, 1.0557088 ],
       [1.02539841, 1.09763447, 1.01855977],
       [1.08049619, 1.05981345, 1.00128302],
       [1.08855004, 1.03075676, 1.02228585],
       [1.02693899, 1.0523594 , 1.06229426],
       [1.07108099, 1.04905218, 1.02145949],
       [1.08639673, 1.03662452, 1.01857141],
       [1.04520729, 1.05135667, 1.04502869],
       [1.09599043, 1.03795434, 1.00764788],
       [1.02390295, 1.01056557, 1.10712414],
       [1.06898649, 1.03997142, 1.03263475],
       [1.10744759, 1.03103791, 1.00310715],
       [1.05811252, 0.97953762, 1.10394251],
       [1.04738597, 1.12433139, 0.96987529],
       [1.04610684, 1.00839451, 1.0870913 ],
       [1.01229688, 1.1750053 , 0.95429048],
       [1.12355067, 1.1199402 , 0.89810179],
       [1.07754244, 0.96190841, 1.1021418 ],
       [1.06391252, 1.05282417, 1.02485596],
       [1.02257161, 1.04663107, 1.07238997],
       [1.01143147, 1.11834053, 1.01182064],
       [1.07295358, 1.03873635, 1.02990273],
       [1.01205447, 1.11352322, 1.01601496],
       [1.0496228 , 1.02809562, 1.06387423],
       [1.07031994, 1.04719875, 1.02407396],
       [1.06063055, 1.03710921, 1.04385289],
       [1.02079215, 1.11921214, 1.00158836],
       [1.09403121, 1.0010951 , 1.04646634],
       [0.93968499, 1.02267907, 1.17922859],
       [1.02655875, 1.1055718 , 1.00946211],
       [1.08073019, 1.0335668 , 1.02729566],
       [1.10341278, 1.19460399, 0.84357589],
       [0.99345683, 1.05952955, 1.08860627],
       [1.02886279, 1.06806713, 1.04466273],
       [1.18682735, 0.98530924, 0.96945606],
       [1.04269384, 1.03699792, 1.06190089],
       [1.05348333, 1.09127328, 0.99683604],
       [1.06004158, 1.0644169 , 1.01713417],
       [1.08043748, 1.00245331, 1.05870187],
       [1.12495872, 0.87089777, 1.14573617],
       [0.94216796, 1.18609131, 1.01333338],
       [1.03185419, 1.01385316, 1.0958853 ],
       [1.01539245, 1.04563179, 1.08056842],
       [1.14076148, 1.04204473, 0.95878645],
       [0.60701708, 1.09859711, 1.43597846],
       [1.16640314, 0.9671254 , 1.00806411],
       [1.17711593, 0.99586296, 0.96861376],
       [1.04178983, 1.09866112, 1.0011417 ],
       [1.08061567, 1.01541115, 1.04556583],
       [0.88669459, 0.74834156, 1.50655651],
       [0.9486628 , 1.21221827, 0.98071159],
       [1.18676256, 0.96797801, 0.98685208],
       [0.9973172 , 1.12618485, 1.0180906 ],
       [1.00619681, 1.12441238, 1.01098347],
       [1.06886385, 0.9994952 , 1.0732336 ],
       [1.0405241 , 1.00204774, 1.09902081],
       [1.03842827, 1.10164484, 1.00151954],
       [1.01391391, 1.15359336, 0.97408538],
       [1.07354485, 1.0882745 , 0.97977331],
       [1.14101097, 1.06316917, 0.93741251],
       [0.98680111, 1.05390473, 1.10088681],
       [1.25022856, 0.93425607, 0.95710803],
       [1.10816485, 0.99568189, 1.03774591],
       [0.92083717, 1.17316096, 1.04759452],
       [1.03039594, 1.08400241, 1.0271943 ],
       [1.04966087, 1.02816983, 1.06376195],
       [1.01364338, 1.07546087, 1.0524884 ],
       [1.0905473 , 1.04436052, 1.00668483],
       [1.05201274, 1.03793455, 1.05164536],
       [1.03749023, 1.00439589, 1.09970653],
       [1.11345671, 1.03774859, 0.99038735],
       [1.03984699, 1.03528515, 1.06646051],
       [1.05023783, 0.96103572, 1.13031911],
       [0.89109748, 1.17295325, 1.07754192],
       [0.91938779, 1.14585732, 1.07634755],
       [1.08287577, 1.03802506, 1.02069182],
       [1.24622431, 0.95340767, 0.94196068],
       [0.87927805, 0.98572659, 1.27658801],
       [1.02323903, 1.08129675, 1.03705688],
       [1.08516855, 0.9952164 , 1.06120771],
       [1.03466188, 1.02801721, 1.07891356],
       [1.004117  , 1.08525065, 1.052225  ],
       [1.03448323, 0.99073089, 1.11637854],
       [1.01243858, 1.07220993, 1.05694415],
       [1.02426157, 0.99651323, 1.12081785],
       [1.02047429, 1.07382981, 1.04728855],
       [1.0144642 , 1.0645238 , 1.06260466],
       [1.04642734, 1.03367761, 1.06148771],
       [1.11053123, 1.02070588, 1.01035554],
       [1.02768733, 1.095992  , 1.01791333],
       [0.85109731, 1.17512937, 1.11536598],
       [1.09805301, 1.01693674, 1.0266029 ],
       [1.03053883, 1.07857698, 1.03247684],
       [0.73614465, 1.25035671, 1.15509129],
       [1.04245059, 1.00394159, 1.09520048],
       [1.10450156, 0.94252137, 1.09456973],
       [0.93910164, 1.19452774, 1.00796327],
       [1.03224069, 1.06008832, 1.04926365],
       [1.00951319, 1.0994198 , 1.03265967],
       [1.02112296, 0.97563707, 1.14483263],
       [1.00420661, 1.11544323, 1.02194282],
       [0.97979606, 0.92233549, 1.23946111],
       [1.28060601, 0.97024318, 0.89074347],
       [1.12323193, 1.05757258, 0.96078814],
       [1.03780306, 1.00372431, 1.10006529],
       [1.17733054, 0.92981103, 1.03445108],
       [1.33216214, 0.89114925, 0.91828127],
       [0.97969711, 1.16569131, 0.99620423],
       [1.09208035, 1.0497634 , 0.99974891],
       [1.01783994, 1.02669888, 1.09705383],
       [1.06770986, 1.05309482, 1.02078797],
       [1.05013452, 1.33876108, 0.75269705],
       [1.06185147, 1.01848288, 1.0612583 ],
       [1.07616931, 1.09471676, 0.97070659],
       [1.12313624, 0.99774058, 1.02071583],
       [1.02610355, 1.05881584, 1.05667326],
       [1.03751632, 1.12345595, 0.98062039],
       [1.05835086, 1.00339289, 1.0798489 ],
       [1.08041457, 1.01601904, 1.04515905],
       [1.12118778, 1.07747893, 0.94292595],
       [1.11781172, 0.98351868, 1.04026225],
       [1.09071304, 1.12081909, 0.93006052],
       [1.07937464, 1.02016971, 1.0420483 ],
       [1.13816633, 1.14677015, 0.85665617],
       [1.0432943 , 1.02630113, 1.07199722],
       [1.06842572, 1.09970841, 0.97345852],
       [1.0702684 , 1.04729746, 1.02402679],
       [1.04291234, 1.03773466, 1.06094565],
       [1.06694497, 1.0498073 , 1.02484038],
       [1.01471815, 1.08310428, 1.04377022],
       [1.03789852, 1.01171039, 1.09198374],
       [1.0599773 , 1.05401225, 1.0276031 ],
       [1.00907138, 1.09660441, 1.03591686],
       [1.03354792, 0.99720012, 1.11084462],
       [1.06544183, 1.04768408, 1.02846674],
       [1.14943455, 1.02156995, 0.97058816],
       [0.85431743, 0.92156142, 1.36571381],
       [1.11412993, 1.00641208, 1.02105064],
       [1.05420719, 0.97215206, 1.1152334 ],
       [0.83076602, 0.44101961, 1.86980702],
       [1.28978762, 1.25844297, 0.59336206],
       [1.00907999, 1.04523317, 1.08727949],
       [0.93857376, 1.07578131, 1.12723759],
       [1.03301671, 1.09614926, 1.01242669],
       [1.0609492 , 1.06596398, 1.01467947],
       [1.13235629, 1.02201498, 0.98722138],
       [1.02268762, 1.10266226, 1.01624277],
       [1.12525534, 1.01530687, 1.00103044],
       [0.98110835, 1.06627051, 1.09421379],
       [1.0473344 , 0.98735031, 1.10690794],
       [1.07245005, 1.06266866, 1.00647394],
       [1.03134243, 1.03645035, 1.07379987],
       [1.02026216, 1.01510093, 1.10622957],
       [1.03880124, 1.0025081 , 1.10028331],
       [0.8082005 , 1.51143811, 0.82195404],
       [1.01720781, 1.08881171, 1.03557314],
       [1.05315389, 1.02586834, 1.06257042],
       [1.22166986, 1.00904975, 0.91087304],
       [0.83083328, 0.7089698 , 1.60178958],
       [1.03720253, 1.0461079 , 1.05828222],
       [1.27396227, 0.93142809, 0.93620229],
       [0.99059868, 0.95416324, 1.19683074],
       [1.03338921, 1.08204249, 1.02616095],
       [1.05399951, 1.05518195, 1.0324112 ],
       [0.98179658, 1.09123306, 1.06856302],
       [1.06994593, 1.0524298 , 1.01921692],
       [1.17170234, 1.01082687, 0.95906345],
       [0.96794784, 1.23629266, 0.93735215],
       [1.04840802, 1.02799476, 1.06518987],
       [1.06919505, 1.12559981, 0.9467978 ],
       [1.03980847, 1.09876169, 1.00302249],
       [0.93326258, 0.92452609, 1.28380399],
       [0.98245322, 1.14472678, 1.01441264],
       [1.1106326 , 1.08029344, 0.95066661],
       [1.03216952, 1.04500006, 1.06442308],
       [1.01159703, 1.06319836, 1.06679726],
       [1.0330536 , 1.07186568, 1.03667337],
       [0.99968843, 1.05645001, 1.08545422],
       [1.19688179, 0.98009753, 0.96461333],
       [0.9491777 , 1.14823791, 1.04417704],
       [0.96163029, 1.0379709 , 1.14199146],
       [0.94078787, 1.08364719, 1.11715758],
       [1.16278016, 0.88058662, 1.09822587],
       [0.70137212, 1.2952614 , 1.14495914],
       [0.97321786, 1.0694691 , 1.09890569],
       [1.26380764, 0.88995303, 0.98783198],
       [1.0443862 , 1.08866037, 1.00854609],
       [0.92726186, 1.00488876, 1.20944203],
       [1.83823889, 0.45536552, 0.84798824],
       [1.07387463, 1.03108661, 1.03663141],
       [1.01592071, 1.02054836, 1.10512359],
       [1.03084614, 1.07183132, 1.0389152 ],
       [1.06474565, 1.02999162, 1.04685538],
       [1.03999878, 1.02920507, 1.0723888 ],
       [1.08655166, 1.03649408, 1.01854692],
       [1.01053804, 1.09546905, 1.03558556],
       [1.05548907, 1.02535796, 1.06074562],
       [1.11246196, 1.03210448, 0.99702622],
       [1.13420629, 1.039195  , 0.96819136],
       [1.03128202, 1.0642346 , 1.04607603],
       [1.20586775, 1.02853793, 0.90718697],
       [1.38301751, 0.68548116, 1.07309398],
       [1.04673975, 1.10907651, 0.9857764 ],
       [0.96070455, 1.10259443, 1.07829367],
       [1.22424558, 1.097979  , 0.81936808],
       [0.99066672, 1.25967524, 0.8912507 ],
       [1.05474444, 1.02314214, 1.06370607],
       [1.02979633, 1.16839347, 0.94340285],
       [1.26059565, 0.8949655 , 0.98603151],
       [0.96051331, 1.06804869, 1.11303065],
       [1.07639272, 1.10675565, 0.95844428],
       [0.96850433, 1.13137473, 1.0417136 ],
       [1.11100304, 1.02715643, 1.00343318],
       [1.0014078 , 1.11776913, 1.02241572],
       [1.04837828, 1.09170443, 1.00150995],
       [1.08601557, 1.09980989, 0.95576719],
       [0.94554146, 1.12505334, 1.07099786],
       [1.08589921, 1.03470351, 1.02098994],
       [1.01123324, 1.10568837, 1.02467104],
       [1.0461743 , 1.09965133, 0.99576703],
       [1.1563102 , 0.9042489 , 1.08103356],
       [1.1021303 , 1.03666562, 1.00279674],
       [1.05566039, 1.03144763, 1.05448463],
       [0.95302042, 1.16603725, 1.02253498],
       [1.00650124, 0.93342117, 1.20167024],
       [1.02643011, 1.04620492, 1.06895762],
       [1.11013952, 1.0436618 , 0.98779133],
       [1.00079229, 0.98671592, 1.15408444],
       [1.04407501, 1.01651779, 1.08099985],
       [1.13745761, 1.03757206, 0.96656298],
       [1.01609864, 1.02411506, 1.10137896],
       [1.01951541, 0.9820195 , 1.14005774],
       [0.97044985, 0.96035175, 1.21079105],
       [1.2629001 , 0.94272746, 0.93596509],
       [1.10577455, 0.98507804, 1.05074006],
       [1.02086402, 1.02251537, 1.09821326],
       [1.14186582, 1.02257482, 0.97715202],
       [1.25113494, 0.95078184, 0.93967587],
       [1.09919466, 1.00637155, 1.03602644],
       [1.1221724 , 0.99883845, 1.02058181],
       [1.01445873, 1.08279292, 1.044341  ],
       [1.01628649, 1.05535753, 1.06994864],
       [1.11638664, 1.02502784, 1.00017817],
       [1.03012427, 1.016511  , 1.09495737],
       [0.70119662, 1.28293569, 1.15746035],
       [0.76070903, 1.31704684, 1.06383678],
       [1.08208123, 0.99677453, 1.0627369 ],
       [1.22093414, 0.93681508, 0.98384343],
       [0.89569669, 1.30022372, 0.94567224],
       [0.49573004, 0.82952688, 1.81633573],
       [1.08225538, 0.99954081, 1.05979646],
       [1.00847261, 1.13533723, 0.99778282],
       [1.03381612, 1.0726893 , 1.03508724],
       [1.03042346, 1.12692307, 0.98424612],
       [1.19405484, 1.35572777, 0.59181004],
       [1.04620973, 1.05009453, 1.0452884 ],
       [1.16511935, 0.96531336, 1.01115994],
       [1.03768608, 1.05570813, 1.04819844],
       [1.03530769, 1.04683362, 1.05945134],
       [1.05977968, 1.05166447, 1.03014851],
       [1.0313275 , 1.07337855, 1.0368866 ],
       [1.03587348, 1.02452063, 1.08119854],
       [1.0181205 , 1.09830633, 1.02516582],
       [1.06711231, 1.05018715, 1.02429319],
       [0.99686288, 1.12037791, 1.02435186],
       [0.99219723, 0.97659171, 1.17280372],
       [1.03780371, 1.06988011, 1.03390883],
       [0.9165292 , 1.06425495, 1.1608085 ],
       [0.96662929, 1.50709516, 0.66786821],
       [1.06613311, 0.94129996, 1.13415957],
       [1.08092452, 0.88384403, 1.17682411],
       [1.0831219 , 0.90264804, 1.15582271],
       [0.87662611, 1.16239841, 1.10256813],
       [1.00633265, 0.97174643, 1.16351357],
       [0.974488  , 1.06595077, 1.10115388],
       [0.9988054 , 0.98378422, 1.15900304],
       [1.05213255, 1.07716251, 1.01229759],
       [0.81012854, 0.75878187, 1.57268224],
       [1.00494321, 0.99476584, 1.14188359],
       [0.97946938, 1.18164532, 0.98047796],
       [1.09560662, 1.02713911, 1.01884692],
       [1.07274101, 1.12662377, 0.94222788],
       [0.90855033, 1.20069077, 1.03235156],
       [0.99973664, 0.98245535, 1.15940067],
       [1.06032668, 1.04202161, 1.03924436],
       [1.05970522, 1.01906384, 1.06282359],
       [1.0385126 , 1.00021982, 1.10286024],
       [1.00158511, 1.07117509, 1.06883246],
       [0.9450204 , 1.15700596, 1.0395663 ],
       [1.0704888 , 1.06687368, 1.00423017],
       [0.9700468 , 1.0740279 , 1.09751795],
       [1.16921183, 1.01176591, 0.96061491],
       [1.18266018, 1.03680727, 0.92212519],
       [1.06565291, 1.04442354, 1.0315162 ],
       [0.95090588, 0.96618439, 1.22450238],
       [0.99004437, 1.15106332, 1.00048496],
       [1.09364206, 1.07161115, 0.97633944],
       [1.08916769, 0.95998346, 1.0924415 ],
       [1.08771692, 1.03849693, 1.01537881],
       [0.97193621, 1.11379757, 1.05585887],
       [1.13999526, 1.02499656, 0.97660083],
       [0.97131444, 0.93787982, 1.23239838],
       [1.00504352, 1.08856115, 1.04798798],
       [1.05227326, 0.99971608, 1.08960331],
       [1.00115723, 1.01163817, 1.12879725],
       [0.96102738, 1.23813812, 0.94242716],
       [1.02866007, 1.03760851, 1.07532407],
       [1.0188142 , 1.0442731 , 1.07850535],
       [1.04561224, 1.02954585, 1.06643457],
       [1.11438139, 1.00999631, 1.01721495],
       [1.08652985, 1.00321258, 1.05185022],
       [1.23810212, 0.90241302, 1.00107751],
       [0.46515086, 0.72740676, 1.94903504],
       [0.99950649, 1.31662854, 0.82545762],
       [0.76778773, 1.37389818, 0.99990674],
       [0.6370191 , 0.89401996, 1.61055359],
       [1.12116052, 0.96035428, 1.06007784],
       [0.9835726 , 1.07064038, 1.08737967],
       [0.97586432, 0.97530398, 1.19042436],
       [1.05663476, 1.0341342 , 1.05082369],
       [1.06856235, 1.05701325, 1.01601706],
       [0.90790823, 1.28118225, 0.95250217],
       [1.06032822, 1.03542575, 1.04583869],
       [1.0871426 , 1.07148338, 0.98296667],
       [1.06659303, 1.03716639, 1.03783323],
       [1.03839406, 1.06942531, 1.03377328],
       [1.0302318 , 1.05261333, 1.05874752],
       [1.03799185, 1.02538433, 1.07821648],
       [1.01864879, 1.08448023, 1.03846363],
       [1.03183849, 1.00936368, 1.10039048],
       [1.02927132, 1.05919208, 1.05312925],
       [1.02592883, 0.98965   , 1.12601383],
       [0.97747276, 1.16837114, 0.99574875],
       [1.06770954, 1.03387857, 1.04000455],
       [1.15338296, 0.70508755, 1.28312214],
       [1.63449748, 0.88524262, 0.62185255],
       [1.04611072, 1.1131739 , 0.98230804],
       [0.77182122, 1.57336663, 0.79640481],
       [1.08662873, 0.90443773, 1.15052619],
       [1.10462409, 1.09551233, 0.94145623],
       [0.76299771, 1.14171369, 1.23688126],
       [1.04335699, 1.13790446, 0.96033121],
       [1.1526331 , 0.9613073 , 1.02765226],
       [1.16518215, 1.00352919, 0.97288131],
       [0.95994748, 1.09503493, 1.08661024],
       [0.9787177 , 1.13502284, 1.02785211],
       [1.17174765, 0.93835729, 1.03148772],
       [0.94580672, 1.12771874, 1.06806719],
       [0.98155047, 1.04641267, 1.11362952],
       [1.09468118, 0.88191659, 1.16499488],
       [1.37285793, 0.95516129, 0.81357344],
       [1.02817477, 1.0503754 , 1.06304249],
       [1.04688521, 1.04200221, 1.05270523],
       [1.02116916, 1.11599893, 1.00442457],
       [0.98910171, 1.05376099, 1.09872995],
       [1.15553115, 1.04203672, 0.94402478],
       [1.02030006, 1.07987818, 1.04141441],
       [1.05057848, 1.0069684 , 1.08404577],
       [1.0452201 , 1.09435101, 1.00202154],
       [1.03141675, 0.99320999, 1.11696591],
       [1.05796656, 1.03941404, 1.04421205],
       [0.9836558 , 1.18645217, 0.97148468],
       [0.98775127, 1.06324873, 1.09059265],
       [1.00717313, 0.98710613, 1.14731339],
       [1.05058842, 0.93939886, 1.15160537],
       [1.14867679, 1.02207908, 0.97083678],
       [1.0045283 , 1.01784043, 1.11922392],
       [0.98074253, 1.16929879, 0.99155133],
       [0.96880187, 0.9660383 , 1.20675248],
       [1.02056893, 1.08760809, 1.03341563],
       [1.07942069, 1.06032728, 1.00184468],
       [0.99635706, 1.13590214, 1.00933346],
       [0.97474197, 1.20654373, 0.96030695],
       [1.00421981, 1.09274897, 1.04462387],
       [0.98050586, 1.10741472, 1.05367208],
       [1.05782853, 0.97349547, 1.11026865],
       [1.03511671, 0.99648556, 1.10999038],
       [0.98016925, 1.09078552, 1.07063788],
       [0.92219047, 1.09329796, 1.12610423],
       [1.1308779 , 1.08765094, 0.92306381],
       [1.20237297, 0.89144058, 1.04777911],
       [0.96215309, 1.20295181, 0.97648776],
       [0.6096454 , 0.91345112, 1.61849613],
       [1.01585287, 1.04220545, 1.08353433],
       [1.06312785, 1.05235077, 1.02611403],
       [1.10147006, 1.05975777, 0.98036483],
       [1.04371721, 1.0560443 , 1.04183115],
       [1.10923197, 0.98738186, 1.04497883],
       [0.95416834, 1.13101406, 1.05641025],
       [1.05162494, 1.04813713, 1.04183058],
       [1.00865108, 1.15204344, 0.98089814],
       [1.03786993, 1.04734763, 1.0563751 ],
       [1.05463616, 1.03284103, 1.05411546],
       [1.06748559, 1.03590333, 1.03820373],
       [1.04681749, 1.03070542, 1.06406974],
       [1.07181786, 1.0393207 , 1.03045409],
       [1.09121823, 1.02859673, 1.02177769],
       [1.09911031, 1.02492366, 1.01755869],
       [1.03964915, 1.03215332, 1.06979018],
       [1.14407364, 0.99406496, 1.00345406],
       [1.23824868, 0.93309267, 0.97025131],
       [1.03456231, 1.04243992, 1.06459043],
       [0.89919241, 1.05613372, 1.18626652],
       [1.07597652, 0.96999343, 1.0956227 ],
       [1.09172169, 0.97218658, 1.07768439],
       [0.81558232, 1.16258072, 1.1634296 ],
       [1.12021221, 1.08882726, 0.93255319],
       [0.93382899, 1.14590251, 1.06186115],
       [1.26167921, 0.87213203, 1.00778141],
       [1.06433537, 0.99729115, 1.07996614],
       [1.03143366, 1.08733848, 1.02282051],
       [1.05238176, 0.97540518, 1.11380571],
       [1.09308747, 1.04293291, 1.00557227],
       [0.98173285, 1.1431543 , 1.0167055 ],
       [0.97391019, 1.03717308, 1.13050938],
       [0.95244835, 1.09635488, 1.09278943],
       [1.10865658, 0.95529093, 1.07764514],
       [1.29944632, 0.97050756, 0.87163878],
       [0.74519847, 1.13885312, 1.25754106],
       [1.04455621, 1.05536405, 1.04167239],
       [1.00657552, 1.07998068, 1.05503645],
       [1.06685278, 1.06465375, 1.01008612],
       [1.04067473, 1.01203663, 1.08888129],
       [0.97161698, 1.11468602, 1.05528966],
       [0.95552528, 1.12297132, 1.06309606],
       [1.07789602, 1.08392324, 0.97977339],
       [0.93599765, 1.04885649, 1.15673851],
       [1.03594456, 1.09261384, 1.01303426],
       [1.11389592, 1.02630332, 1.00139341],
       [0.95871967, 1.043876  , 1.13899698],
       [1.06490311, 1.01896449, 1.05772505],
       [1.03883841, 1.07444305, 1.0283112 ],
       [1.03170978, 0.95061755, 1.15926533],
       [1.09767603, 1.12950424, 0.91441238],
       [1.10160294, 1.08801866, 0.95197105],
       [1.11895443, 1.02182328, 1.00081495],
       [0.95613822, 1.1197423 , 1.06571213],
       [1.19518029, 0.98339266, 0.9630197 ],
       [1.07253838, 1.02396518, 1.04508909],
       [1.01350154, 1.09174587, 1.03634524],
       [1.1165166 , 0.9903151 , 1.03476096],
       [1.17008764, 0.95935444, 1.01215057],
       [1.00427378, 1.04036727, 1.0969516 ],
       [1.06461829, 1.09817497, 0.97879939],
       [1.06127866, 1.07430308, 1.00601092],
       [1.10394859, 1.00881293, 1.02883113],
       [1.06662719, 0.97272663, 1.10223883],
       [1.14207055, 1.05461183, 0.94491028],
       [0.60571083, 1.09336904, 1.44251278],
       [1.07502572, 1.02286829, 1.04369864],
       [1.52307802, 0.97610266, 0.64241197],
       [1.06747891, 1.1008799 , 0.97323385],
       [1.07122708, 0.99451091, 1.07585465],
       [1.04626658, 1.02822554, 1.06710053],
       [1.06834867, 1.03030368, 1.04294031],
       [1.03336911, 1.05475162, 1.05347192],
       [1.09405276, 1.03267244, 1.01486745],
       [1.05192336, 1.03808336, 1.05158593],
       [1.05960481, 1.03336639, 1.04862145],
       [1.0043712 , 0.98348006, 1.15374139],
       [1.0483076 , 1.06188299, 1.03140206],
       [1.02370849, 1.05431125, 1.06357292],
       [1.07545726, 1.02374284, 1.04239255],
       [1.05166154, 1.02946332, 1.0604678 ],
       [1.04237252, 1.02733312, 1.07188701],
       [1.0896741 , 1.03781473, 1.01410383],
       [1.09994815, 1.0348486 , 1.00679591],
       [1.03947145, 1.05925782, 1.04286339],
       [1.01236195, 1.13166856, 0.99756214],
       [0.92546802, 1.02829121, 1.18783342],
       [1.05184821, 1.05203759, 1.03770685],
       [1.16095008, 0.94666988, 1.03397269],
       [1.08506314, 0.98668414, 1.06984537],
       [1.07393406, 1.00325424, 1.06440435],
       [0.95610573, 0.97560855, 1.20987838],
       [0.95405577, 1.10157138, 1.08596551],
       [0.87368416, 1.15136641, 1.11654208],
       [1.01683538, 1.13840785, 0.98634942],
       [0.93999778, 1.07275127, 1.12884361],
       [1.08924954, 1.08971099, 0.96263213],
       [1.0234361 , 1.13561345, 0.9825431 ],
       [1.12035921, 1.05550206, 0.96573138],
       [1.04418916, 0.97103141, 1.12637208],
       [1.1135011 , 0.82821412, 1.19987743],
       [1.01888516, 1.16213273, 0.96057477],
       [1.08090918, 0.9528898 , 1.10779367],
       [0.99722846, 1.05154618, 1.09281801],
       [1.14258206, 1.17440902, 0.82460157],
       [0.8280693 , 0.73590544, 1.57761792],
       [1.05617913, 1.01719891, 1.06821461],
       [1.03520565, 0.99826463, 1.10812237],
       [1.06800909, 1.01283611, 1.06074745],
       [1.02525982, 1.0276128 , 1.08872003],
       [1.13342863, 1.06188285, 0.94628117],
       [1.03912124, 1.01820662, 1.0842648 ],
       [1.02531439, 1.09114094, 1.02513732],
       [1.12709401, 1.10424336, 0.91025528],
       [1.04408871, 0.99572461, 1.10177933],
       [1.06874514, 1.0663612 , 1.00648632],
       [1.10448911, 1.07917155, 0.957932  ],
       [1.07910608, 1.05220456, 1.01028201],
       [1.03110818, 0.99147786, 1.11900661],
       [0.89444599, 1.15329405, 1.09385261],
       [1.18878226, 1.00342342, 0.94938698],
       [1.05123534, 0.92997076, 1.16038655],
       [0.97419171, 1.02272841, 1.14467253],
       [1.07083221, 1.02979544, 1.040965  ],
       [0.98552105, 1.00190357, 1.15416803],
       [1.0464269 , 1.0204187 , 1.07474705],
       [1.02550382, 1.05290968, 1.06317916],
       [0.99159559, 1.02285519, 1.12714187],
       [1.00817131, 1.1484149 , 0.98500645],
       [0.95621662, 1.18066616, 1.00470988],
       [1.03440723, 1.05847569, 1.04870974],
       [0.94822312, 1.05348569, 1.13988384],
       [0.95782038, 1.1406612 , 1.04311106],
       [1.05211832, 1.03307402, 1.05640031],
       [0.93962156, 1.00622159, 1.19574951],
       [1.05311763, 1.05078446, 1.03769056],
       [1.13447634, 0.89746851, 1.10964781],
       [1.07811671, 1.01826321, 1.04521274],
       [1.02706774, 1.06489683, 1.04962809],
       [1.07151796, 1.03779594, 1.03227876],
       [1.01975056, 1.021069  , 1.10077309],
       [1.05664437, 1.02773804, 1.05721025],
       [1.01735008, 1.12102647, 1.0032161 ],
       [1.12825024, 0.9959007 , 1.01744171],
       [1.03977745, 1.04703006, 1.05478514],
       [1.02730372, 1.0268446 , 1.08744434],
       [1.12185397, 1.01928097, 1.00045771],
       [1.04431146, 1.05176241, 1.04551878],
       [1.03081274, 1.07334249, 1.03743742],
       [1.05877129, 1.05454514, 1.02827622],
       [1.03993223, 1.03992328, 1.06173715],
       [1.07548077, 1.02478449, 1.04132739],
       [1.02836856, 1.07532359, 1.0379005 ],
       [1.0193632 , 1.09577068, 1.02645878],
       [1.00813124, 1.02309446, 1.11036695],
       [1.02779804, 1.05068368, 1.06311093],
       [1.02632318, 1.15463776, 0.96063171],
       [1.05403508, 1.07601374, 1.01154383],
       [1.12607219, 0.93943043, 1.07609003],
       [1.00151251, 1.08955629, 1.05052386],
       [1.09820829, 1.04828541, 0.99509896],
       [0.98035393, 1.09050492, 1.0707338 ],
       [0.95509256, 1.05326031, 1.13323978],
       [1.11065311, 1.07979395, 0.9511456 ],
       [1.06407144, 1.13184312, 0.9456781 ],
       [0.97255178, 1.02547768, 1.14356319],
       [1.08102157, 1.08486151, 0.97570957],
       [1.07728003, 0.97681059, 1.08750204],
       [1.01010046, 1.05161365, 1.07987855],
       [1.01067475, 1.0704012 , 1.0605167 ],
       [1.19745181, 0.98348001, 0.96066083],
       [0.70954387, 1.86358892, 0.56845986],
       [1.03859985, 1.10151551, 1.00147729],
       [0.99842656, 1.20601669, 0.9371494 ],
       [1.09251689, 0.97344245, 1.07563331],
       [1.08506489, 0.9892155 , 1.06731226],
       [0.76182709, 0.91565049, 1.46411507],
       [1.07629782, 1.02335631, 1.04193852],
       [1.01741684, 1.0150305 , 1.10914531],
       [1.11592598, 1.05003204, 0.97563463],
       [1.07174016, 1.07798364, 0.99186885],
       [1.07884883, 1.02970524, 1.03303858],
       [1.03514194, 1.06237763, 1.04407308],
       [0.97503952, 1.10419007, 1.06236307],
       [1.07414914, 1.03592749, 1.03151603],
       [1.08253833, 0.92742008, 1.13163425],
       [1.19858152, 0.85365651, 1.08935462],
       [1.07048923, 1.04930482, 1.02179861],
       [1.0614479 , 1.01885202, 1.06129274],
       [1.27955288, 0.85634499, 1.00569478],
       [1.04477319, 1.10113381, 0.99568566],
       [1.07492973, 1.04261777, 1.02404515],
       [1.18068048, 0.85866207, 1.1022501 ],
       [1.03407307, 1.13161302, 0.97590656],
       [1.12554092, 1.09937148, 0.91668025],
       [0.97277971, 1.15209572, 1.01671723],
       [1.00691605, 1.05946318, 1.07521343],
       [1.14251888, 1.02252535, 0.97654841],
       [0.98277356, 1.0958893 , 1.06292979],
       [1.0624248 , 1.00642785, 1.07274   ],
       [1.0893485 , 0.99738524, 1.05485891],
       [1.11561087, 1.00196549, 1.02401629],
       [0.9329022 , 0.99736308, 1.21132737],
       [1.0289891 , 0.90042926, 1.21217429],
       [1.08791114, 1.06759861, 0.98608291],
       [1.07579845, 0.9701428 , 1.09565141],
       [1.22414059, 0.95995084, 0.95750123],
       [1.10307469, 1.00902157, 1.02949639],
       [0.97740421, 1.07023043, 1.09395801],
       [1.08481848, 1.05709963, 0.99967454],
       [1.03245084, 1.0204722 , 1.08866961],
       [1.05488006, 1.01471519, 1.0719974 ],
       [1.06820789, 1.06623442, 1.00715034],
       [1.04734692, 1.06979523, 1.0244505 ],
       [1.08700222, 1.02373027, 1.03086017],
       [1.0376377 , 1.00583937, 1.09811559],
       [1.05978252, 1.04901717, 1.03279296],
       [1.07192491, 1.12923123, 0.94043651],
       [1.02601324, 1.00225249, 1.11332693],
       [1.04034544, 1.03430509, 1.06694212],
       [1.06968155, 1.04742947, 1.02448163],
       [1.01147694, 1.06550563, 1.06461009],
       [1.04120454, 1.07508619, 1.02530192],
       [1.04031669, 1.06731081, 1.03396516],
       [1.0367224 , 1.02535389, 1.07951637],
       [1.02663982, 1.02429125, 1.09066158],
       [1.0347466 , 1.08394278, 1.02290327],
       [1.03044222, 1.05022887, 1.06092156],
       [0.97791117, 1.05132199, 1.11235949],
       [1.07248128, 1.0163869 , 1.05272448],
       [1.06404395, 1.10203593, 0.97551278],
       [1.08514368, 1.04761079, 1.00883818],
       [1.00719892, 1.06426614, 1.07012759],
       [1.06029107, 1.0605008 , 1.02080079],
       [1.0652847 , 1.0084638 , 1.06784415],
       [0.98740258, 1.07053604, 1.08365403],
       [1.09897998, 0.98099922, 1.06161346],
       [0.99444924, 1.05150023, 1.09564318],
       [1.09412062, 1.16202345, 0.88544858],
       [0.90954151, 1.11524754, 1.11680361],
       [1.1288489 , 1.00270203, 1.01004172],
       [1.10562022, 1.00753588, 1.02843655],
       [1.03866537, 1.01017184, 1.09275544],
       [1.25763989, 1.1752543 , 0.70869846],
       [1.00428978, 1.01546238, 1.1218405 ],
       [1.02405649, 1.02476866, 1.0927675 ],
       [1.10700862, 1.04766942, 0.98691461],
       [1.03021693, 1.02830561, 1.08307012],
       [1.18728093, 0.99499888, 0.95931284],
       [0.85039457, 1.16910654, 1.12209154],
       [1.08939661, 1.05937798, 0.99281806],
       [0.97613714, 1.09330076, 1.07215475],
       [1.06795935, 1.07552411, 0.99810919],
       [1.04908247, 1.01188124, 1.08062895],
       [1.07422966, 1.11969965, 0.94766335],
       [1.10317141, 1.03415246, 1.00426878],
       [1.07404098, 1.04896893, 1.01858275],
       [1.15109325, 1.02157789, 0.96892152],
       [1.20030269, 1.15597814, 0.78531183],
       [1.01161187, 1.12372125, 1.00625953],
       [1.00322168, 1.10843384, 1.02993713],
       [1.21993145, 1.09699915, 0.82466206],
       [1.00611648, 1.0559451 , 1.07953108],
       [0.99686631, 1.14835928, 0.99636706],
       [1.02674668, 1.22243613, 0.89240984],
       [0.89935422, 1.15071519, 1.09152325],
       [1.0726762 , 1.05394861, 1.01496784],
       [0.99998995, 1.07577286, 1.06582984],
       [1.02024392, 1.07252016, 1.04882858],
       [1.00727947, 1.09040296, 1.04391022],
       [1.0124061 , 1.04721996, 1.0819666 ],
       [1.04130824, 1.07492996, 1.02535445],
       [1.08402421, 1.09873243, 0.95883602],
       [1.02672798, 1.01324503, 1.10161964],
       [0.50967254, 1.84610179, 0.78581833],
       [0.86263098, 1.2928548 , 0.98610688],
       [0.90950275, 1.07416464, 1.15792527],
       [0.94898779, 1.13623942, 1.05636544],
       [1.07980551, 1.14193699, 0.91985015],
       [1.03194846, 1.09657299, 1.01307121],
       [1.02432144, 1.06644492, 1.0508263 ],
       [1.08680083, 1.0258357 , 1.02895613],
       [0.99404705, 1.081198  , 1.0663476 ],
       [1.04828357, 1.08570563, 1.00760345],
       [1.06366489, 1.02804431, 1.04988346],
       [0.99850907, 1.02798854, 1.11509504],
       [1.02333356, 1.09613817, 1.02212093],
       [1.04197471, 1.02784513, 1.07177281],
       [1.16339664, 0.9232816 , 1.05491441],
       [0.996463  , 0.99383465, 1.151295  ],
       [1.12018115, 1.01946001, 1.0019515 ],
       [1.08127926, 1.01958663, 1.04072677],
       [1.00737192, 1.04427072, 1.08995002],
       [1.08660035, 1.02551924, 1.02947307],
       [1.03867366, 1.03326366, 1.06965533],
       [1.04944383, 1.04876732, 1.04338151],
       [1.0484423 , 1.08121914, 1.01193122],
       [1.07704518, 1.03434477, 1.0302027 ],
       [1.0840816 , 1.02452077, 1.03299029],
       [1.10722159, 0.99161265, 1.04275842],
       [1.05086621, 1.07977866, 1.01094778],
       [1.01123131, 1.03701198, 1.09334936],
       [1.0741098 , 0.99280896, 1.0746739 ],
       [1.0290863 , 1.03512347, 1.07738288],
       [1.04248928, 1.08656502, 1.01253835],
       [1.00479877, 1.09667123, 1.04012265],
       [1.01596815, 1.0523853 , 1.0732392 ],
       [0.98049547, 0.96580051, 1.19529667],
       [1.03178356, 1.05834997, 1.05145912],
       [1.0190068 , 1.11232005, 1.0102658 ],
       [1.00992882, 1.06874935, 1.06291449],
       [0.95944907, 0.89533592, 1.28680767],
       [1.00093504, 1.22255208, 0.91810553],
       [1.03024349, 1.07246219, 1.03888698],
       [1.09129744, 1.02977001, 1.0205252 ],
       [1.07401532, 1.02514797, 1.04242937],
       [1.28953102, 0.95566222, 0.89639941],
       [1.18116638, 0.94233044, 1.01809584],
       [1.01916998, 1.07439367, 1.04802901],
       [0.97442957, 1.00722895, 1.15993413],
       [1.05173033, 1.0898307 , 1.00003162],
       [1.01390663, 1.06291934, 1.06476668],
       [0.70342605, 0.83610679, 1.60205981],
       [1.01143513, 1.05087685, 1.07928067],
       [0.97871773, 1.0807857 , 1.08208922],
       [1.1083928 , 1.05337663, 0.97982322],
       [1.00457233, 1.07160123, 1.0654191 ],
       [0.96178616, 1.13852299, 1.0412835 ],
       [0.99615992, 0.9685259 , 1.17690683],
       [1.05096629, 1.02506371, 1.06556266],
       [1.0730379 , 1.04201731, 1.02653744],
       [0.96395475, 1.07861043, 1.09902748],
       [0.88829096, 0.76806506, 1.48523663],
       [1.02818398, 1.02327204, 1.09013663],
       [1.08321907, 1.03437799, 1.02399559],
       [0.79286192, 1.16471013, 1.1840206 ],
       [0.84458616, 0.70010107, 1.59690543],
       [1.09053012, 0.93673079, 1.11433175],
       [1.10565436, 0.99939567, 1.03654262],
       [1.04810323, 1.18722429, 0.90626513],
       [1.12954288, 0.89383008, 1.1182197 ],
       [0.9073817 , 1.2381053 , 0.99610566],
       [1.12726346, 1.19367822, 0.82065098],
       [1.12907134, 1.00165294, 1.01086837],
       [1.08653476, 0.95898244, 1.09607546],
       [1.0926462 , 1.08047742, 0.96846904],
       [0.97927848, 1.09094985, 1.07136432],
       [1.15095477, 1.01116319, 0.97947469],
       [1.03547695, 1.05194888, 1.05416682],
       [1.11058326, 1.23270373, 0.79830566],
       [0.81406357, 0.95660816, 1.37092092],
       [1.10650913, 1.07887759, 0.95620594],
       [1.27879511, 0.90399642, 0.95880112],
       [0.94820189, 1.0710031 , 1.12238766],
       [1.05873521, 1.00737753, 1.07547992],
       [0.9706423 , 1.08331198, 1.08763837],
       [1.03089932, 1.07938297, 1.03131036],
       [1.057317  , 1.01807207, 1.06620358],
       [1.06466227, 1.07784734, 0.99908305],
       [1.00712695, 1.07239438, 1.06207132],
       [1.03218443, 1.08335208, 1.02605614],
       [1.08988256, 1.02018686, 1.03152324],
       [1.05271662, 1.00535609, 1.08351994],
       [1.07924363, 1.03387897, 1.02847006],
       [1.12339749, 0.87642327, 1.14177189],
       [1.0943614 , 1.09552726, 0.951704  ],
       [1.03589651, 1.09375359, 1.01194256],
       [1.01293178, 1.02625495, 1.10240593],
       [1.00920441, 1.09678909, 1.03559916],
       [1.07940654, 1.06054968, 1.00163643],
       [1.05492121, 1.06561157, 1.02105987],
       [1.00958285, 1.04528321, 1.0867266 ],
       [1.07788227, 1.02569438, 1.038016  ],
       [1.04123341, 1.07258941, 1.02776983],
       [1.07788458, 1.00281715, 1.06089093],
       [1.00692762, 1.05481589, 1.07984914],
       [1.01054506, 1.04396627, 1.08708132],
       [1.07917362, 1.0518508 , 1.01056823],
       [1.06104104, 1.04454342, 1.03600819],
       [1.03017343, 1.04372268, 1.06769655],
       [1.0550468 , 1.0447692 , 1.04177665],
       [1.04555998, 1.07024024, 1.02579244],
       [0.96096166, 1.12324569, 1.05738531],
       [1.01222567, 1.06024857, 1.0691184 ],
       [1.05827215, 1.05677518, 1.02654533],
       [1.14211369, 1.07715595, 0.92232301],
       [1.01270109, 1.11848482, 1.01040675],
       [1.3636726 , 1.28479232, 0.49312774],
       [1.07295667, 1.02361244, 1.04502354],
       [0.90715386, 1.10006358, 1.13437522],
       [1.08010522, 1.02706131, 1.03442612],
       [1.00415344, 1.11183884, 1.02560038],
       [0.94736329, 1.09847376, 1.0957556 ],
       [0.98554768, 0.95985101, 1.19619396],
       [1.0790317 , 1.04973898, 1.01282197],
       [1.11374557, 1.0294363 , 0.99841079],
       [1.03135937, 1.01895075, 1.09128253],
       [1.07858065, 1.05825163, 1.00476037],
       [1.53957612, 0.87161655, 0.73039998],
       [0.9985749 , 1.06065564, 1.08236212],
       [1.01410209, 1.06966608, 1.05782449],
       [1.01611752, 1.00526782, 1.12020732],
       [1.03459751, 1.04079149, 1.06620365],
       [1.12305422, 1.08172957, 0.93680886],
       [1.04274267, 1.20116067, 0.89768932],
       [1.02349961, 1.05758461, 1.06050843],
       [1.02229799, 1.01347645, 1.10581821],
       [1.06310656, 1.03924571, 1.03924039],
       [1.01901092, 0.9606993 , 1.16188244],
       [1.13207458, 0.99271701, 1.01680107],
       [1.03477336, 1.12440527, 0.98241402],
       [1.18344627, 1.05044042, 0.90770597],
       [1.18144711, 1.18615353, 0.77399202],
       [0.99837235, 1.10809308, 1.03512722],
       [1.08075938, 1.08955722, 0.97127605],
       [1.12280514, 0.88942684, 1.12936068],
       [1.2618063 , 0.91448898, 0.96529737],
       [1.32392597, 0.84616484, 0.97150184],
       [0.95983864, 1.35671266, 0.82504135],
       [1.05513435, 0.96558993, 1.12086837],
       [1.0082356 , 1.10388722, 1.02946983],
       [1.13350262, 1.03386533, 0.9742247 ],
       [0.98033488, 0.9106387 , 1.25061907],
       [1.04026338, 0.98778314, 1.11354614],
       [1.07503742, 1.0195839 , 1.04697133],
       [0.9813013 , 1.10248986, 1.0578015 ],
       [1.07199858, 0.79983603, 1.26975804],
       [1.85271373, 0.70341052, 0.5854684 ],
       [1.15120682, 0.82485879, 1.16552704],
       [1.06443403, 1.0544376 , 1.02272103],
       [1.04997724, 1.01611665, 1.07549877],
       [1.06739109, 1.04768079, 1.02652077],
       [0.99557031, 1.08875882, 1.05726352],
       [1.0713391 , 1.07318854, 0.99706501],
       [1.0405882 , 1.09176909, 1.00923537],
       [1.02365404, 1.08733104, 1.03060758],
       [1.02179763, 1.03610565, 1.08368937],
       [1.08756711, 1.02533692, 1.02868862],
       [1.02099164, 1.03295192, 1.0876491 ],
       [0.88396836, 0.93644135, 1.32118294],
       [1.07301312, 1.01336533, 1.0552142 ],
       [1.02427506, 1.10705061, 1.01026698],
       [1.02596058, 1.12313222, 0.99249986],
       [1.03915081, 1.12683787, 0.97560397],
       [1.00640065, 1.01996205, 1.11522996],
       [1.02845047, 1.02726608, 1.08587611],
       [1.04514357, 1.0521447 , 1.04430438],
       [1.05204605, 1.11025397, 0.97929263],
       [1.03951745, 1.0397126 , 1.0623626 ],
       [1.05483005, 1.06285706, 1.02390555],
       [1.10334749, 0.98795164, 1.05029352],
       [1.10812281, 0.99709486, 1.03637499],
       [1.04044414, 1.0549441 , 1.04620441],
       [1.04635037, 1.05814781, 1.03709447],
       [1.04100244, 1.06228267, 1.03830754],
       [1.02469172, 1.04289891, 1.07400202],
       [1.03262715, 1.04235036, 1.06661514],
       [1.04797056, 1.08546011, 1.00816199],
       [1.02607781, 0.73837388, 1.37714097],
       [1.07804896, 1.0747511 , 0.9887926 ],
       [1.05206089, 1.005302  , 1.08422977],
       [1.33382501, 0.88774101, 0.92002663],
       [1.05122798, 1.10254788, 0.9878168 ],
       [0.36153042, 0.86670207, 1.91336016],
       [1.01346432, 1.0710898 , 1.05703853],
       [1.02561988, 0.98223315, 1.13373963],
       [0.90573125, 1.06386616, 1.17199524],
       [0.98044755, 1.15727952, 1.00386558],
       [1.02070111, 1.11124822, 1.00964332],
       [0.95212899, 1.31360434, 0.87585932],
       [1.09643907, 1.02789585, 1.01725774],
       [0.95032532, 1.02516194, 1.16610539],
       [0.9889718 , 1.05926652, 1.09335433],
       [1.04331839, 1.0077828 , 1.09049146],
       [1.09073487, 1.04037445, 1.01048334],
       [1.07160981, 0.76820697, 1.30177587],
       [1.07061331, 1.06795611, 1.00302323],
       [1.11223269, 0.98277811, 1.04658186],
       [1.06304032, 1.05041696, 1.02813537],
       [1.03881328, 1.06824975, 1.03452963],
       [1.03743648, 0.91268886, 1.19146731],
       [1.03525208, 0.83438974, 1.27195083],
       [1.00777093, 1.06330053, 1.07052119],
       [1.07160709, 1.03807976, 1.0319058 ],
       [1.02419776, 1.10547672, 1.01191818],
       [0.98215355, 1.12032591, 1.0391132 ],
       [0.99629483, 1.08821592, 1.0570819 ],
       [1.10963064, 1.00184767, 1.03011435],
       [1.10077462, 1.01529037, 1.02552766],
       [1.02347927, 0.94998832, 1.16812506],
       [1.08804923, 0.9322344 , 1.12130902],
       [0.9875096 , 1.05755864, 1.09652442],
       [0.89818557, 1.10876211, 1.13464497],
       [0.9865509 , 0.96252498, 1.19251678],
       [0.98585156, 0.85251555, 1.30322554],
       [1.2077063 , 0.90732132, 1.02656503],
       [0.97317845, 0.95355857, 1.21485563],
       [1.16165551, 0.94835135, 1.0315858 ],
       [1.00712332, 1.10391201, 1.03055732],
       [1.12849368, 0.9591457 , 1.05395327],
       [1.09765612, 1.06033747, 0.98359907],
       [1.0584659 , 1.11866461, 0.96446214],
       [1.07834023, 1.16760547, 0.89564695],
       [0.97200263, 1.12194015, 1.04764988],
       [1.21133891, 1.21694502, 0.71330873],
       [1.14501313, 1.03706511, 0.9595144 ],
       [1.02523157, 1.06430972, 1.05205137],
       [1.01980951, 1.0618838 , 1.05989934],
       [1.06235654, 1.02426867, 1.05496744],
       [1.03834841, 1.02290415, 1.0803401 ],
       [0.99697747, 1.06945315, 1.07516203],
       [0.98567339, 1.04845192, 1.10746735],
       [1.02244569, 1.02186515, 1.09728181],
       [1.09591516, 1.03249217, 1.01318533],
       [1.02122301, 1.09536541, 1.02500423],
       [1.06405926, 1.01005799, 1.0674754 ],
       [1.11391116, 1.00509742, 1.02258407],
       [1.03809982, 1.1022636 , 1.00122924],
       [1.00733294, 1.03354973, 1.10070999],
       [0.99933945, 1.00069754, 1.14155566],
       [1.14085393, 0.97789676, 1.02284196],
       [1.02134515, 1.00572285, 1.11452466],
       [1.00626779, 1.08444915, 1.05087571],
       [1.02262571, 1.04958817, 1.06937878],
       [1.10417922, 0.9827532 , 1.05466023],
       [1.04802162, 1.06756597, 1.02600506],
       [1.04275057, 1.05163531, 1.04720677],
       [1.06483448, 1.08776046, 0.98899771],
       [1.03922865, 1.09760253, 1.00476148],
       [1.02045811, 1.05255801, 1.06857654],
       [1.04141502, 1.04078414, 1.05939349],
       [1.05132985, 1.06693094, 1.02333187],
       [1.05691077, 1.00557556, 1.07910632],
       [1.05676421, 1.05398722, 1.03084123],
       [0.98372312, 1.06796994, 1.08989959],
       [1.72388957, 0.82272362, 0.59497946],
       [1.1403944 , 1.09280908, 0.90838917],
       [0.94695077, 1.03587397, 1.15876792],
       [1.21068998, 1.00635698, 0.92454569],
       [0.86153583, 1.17951361, 1.10054322],
       [1.04569068, 1.09254638, 1.0033556 ],
       [1.05463339, 1.04290192, 1.04405734],
       [1.27993907, 0.98681343, 0.87484015],
       [1.2695637 , 0.90920902, 0.96281993],
       [1.04419956, 1.06828977, 1.02910332],
       [0.9314774 , 0.96459645, 1.2455188 ],
       [1.0051692 , 1.08146172, 1.05496173],
       [0.95366549, 1.10683039, 1.08109677],
       [0.9928405 , 1.09911603, 1.04963612],
       [1.06716416, 0.97828647, 1.09614202],
       [1.03211146, 1.09371792, 1.01576327],
       [0.89097982, 1.01313222, 1.23748062],
       [1.0578474 , 1.01933274, 1.06441252],
       [1.0624997 , 1.07283857, 1.00625438],
       [1.05096492, 1.04877069, 1.04185704],
       [1.01763893, 1.06386113, 1.06009259],
       [1.07927306, 1.2198536 , 0.84246599],
       [0.7198311 , 1.23143684, 1.19032471],
       [0.9281167 , 1.04815615, 1.1653198 ],
       [1.01535684, 1.09579444, 1.03044138],
       [1.01171194, 1.1279518 , 1.00192891],
       [1.03015126, 1.002905  , 1.10853639],
       [1.02059077, 1.05595485, 1.06504703],
       [0.97871742, 1.02334391, 1.13953133],
       [0.97907738, 1.03316582, 1.12934946],
       [0.91595566, 1.05744113, 1.16819586],
       [1.04719499, 1.01801144, 1.07638622],
       [1.10302244, 0.96174908, 1.07682114],
       [1.12690151, 0.9137718 , 1.10091935],
       [0.98579396, 1.17203747, 0.98376122],
       [0.88333571, 1.17522937, 1.08302758],
       [1.02997254, 1.0804773 , 1.03114281],
       [0.88935165, 1.17723533, 1.07500567],
       [1.09402986, 1.128493  , 0.91906979],
       [1.09754558, 0.99652714, 1.04751993],
       [1.03624184, 1.01199529, 1.09335553],
       [1.00381053, 1.08061265, 1.05716947],
       [1.10273583, 1.04237074, 0.99648609],
       [1.04640666, 1.00720684, 1.08797915],
       [0.96698627, 1.11926261, 1.05534377],
       [1.14495178, 1.0444962 , 0.95214467],
       [0.9901136 , 0.96169297, 1.18978609],
       [0.95323906, 1.12355573, 1.06479787],
       [1.05784001, 1.02730505, 1.0564476 ],
       [1.05392231, 1.04158806, 1.04608228],
       [1.04417667, 1.04991296, 1.04750302],
       [0.98149858, 1.08438566, 1.07570841],
       [1.07726397, 0.97929321, 1.08503548],
       [1.08904316, 0.99680544, 1.05574406],
       [1.0872559 , 1.02207302, 1.03226374],
       [1.03494709, 1.00164984, 1.10499572],
       [0.9992182 , 1.04750742, 1.09486703],
       [1.06089799, 1.02350694, 1.05718772],
       [1.01534201, 1.10924391, 1.01700674],
       [1.06807946, 1.04222538, 1.03128781],
       [1.10871668, 0.99725841, 1.03561757],
       [1.00687802, 1.15152983, 0.98318481],
       [1.17353194, 0.9531982 , 1.01486252],
       [0.92039362, 1.04950873, 1.17169031],
       [1.02955148, 1.1358429 , 0.97619827],
       [1.010061  , 1.04562826, 1.08590339],
       [1.03297387, 1.05685949, 1.05175929],
       [0.94471367, 1.06133343, 1.13554555],
       [1.06619026, 1.05219003, 1.02321236],
       [0.97580388, 1.05833844, 1.10745033],
       [1.04888463, 1.10275478, 0.98995325],
       [1.00563753, 1.06881476, 1.06714036],
       [1.0412519 , 1.04877291, 1.05156784],
       [1.04816263, 1.03789728, 1.05553275],
       [1.03655696, 1.02694278, 1.07809291],
       [1.01605962, 1.12117213, 1.0043609 ],
       [1.0439657 , 1.06780369, 1.02982326],
       [1.0540453 , 1.04987713, 1.03767023],
       [0.79233966, 1.11733313, 1.23191986],
       [1.00841881, 1.11986753, 1.01330631],
       [0.96172211, 1.1866007 , 0.99326984],
       [0.98785666, 1.12053163, 1.03320436],
       [0.95761131, 1.27209042, 0.91189092],
       [1.03302939, 1.02868326, 1.07988   ],
       [1.08250867, 1.06208254, 0.99700145],
       [1.75878508, 0.87728921, 0.50551836],
       [0.6167985 , 1.19897199, 1.32582216],
       [1.00373592, 1.04565829, 1.09219844],
       [1.15352744, 0.9621328 , 1.02593242],
       [1.04968396, 1.00406886, 1.08783984],
       [1.00294661, 1.13298468, 1.00566137],
       [0.94059333, 1.1153818 , 1.08561752],
       [1.07538368, 1.12258732, 0.94362166],
       [1.02697352, 1.04676723, 1.0678519 ],
       [1.08470317, 1.03209639, 1.0247931 ],
       [1.04761499, 1.12810437, 0.9658733 ],
       [1.01541682, 1.06326469, 1.06291115],
       [1.0330915 , 1.0486271 , 1.05987405],
       [1.06310238, 1.0626293 , 1.01586098],
       [1.04086394, 1.01914595, 1.08158277],
       [1.0369876 , 1.02696268, 1.07764238],
       [0.89925327, 0.83267714, 1.40966225],
       [0.72275704, 1.71373626, 0.70509935],
       [1.00907293, 1.19127336, 0.94124636],
       [1.08605503, 1.07636096, 0.97917666],
       [1.01550334, 1.02779187, 1.09829744],
       [1.0030557 , 1.01801071, 1.12052624],
       [0.95548772, 1.12938632, 1.05671862],
       [1.03871084, 1.11602445, 0.98685736],
       [1.01538871, 1.13006941, 0.99613454],
       [1.03208746, 1.12368552, 0.98581967],
       [0.9878846 , 1.09325681, 1.06045125],
       [1.09312454, 1.08471523, 0.96375288],
       [0.93818156, 1.09292759, 1.11048351],
       [1.13048352, 1.0059191 , 1.00519004],
       [0.99541714, 1.06300953, 1.08316598],
       [1.06564522, 1.05465011, 1.02129732],
       [1.21667989, 0.99987764, 0.92503513],
       [1.06216311, 0.98058686, 1.09884268],
       [1.02229165, 1.07891131, 1.04038969],
       [1.04501768, 1.06693376, 1.02964121],
       [0.95896732, 1.1297804 , 1.05284493],
       [1.06166274, 1.02651679, 1.05341313],
       [0.95620234, 1.04137712, 1.1440132 ],
       [1.04787408, 1.03804752, 1.05567106],
       [1.0582135 , 0.942378  , 1.14100115],
       [1.09404249, 1.04294465, 1.00460552],
       [1.04123717, 1.04490369, 1.05545179],
       [1.02748896, 1.07993342, 1.03417027],
       [1.03912072, 1.05509092, 1.04738101],
       [1.04530405, 1.06360808, 1.03268052],
       [1.06107525, 1.07399233, 1.00652508],
       [0.98512626, 1.03427904, 1.12218736],
       [0.99271152, 1.1505523 , 0.99832883],
       [0.99630464, 1.03215561, 1.1131324 ],
       [1.02929995, 0.98661701, 1.12567569],
       [1.04282178, 1.08586655, 1.01290433],
       [1.06077755, 1.04088285, 1.03993226],
       [1.11142099, 1.01989514, 1.01027653],
       [1.00924744, 1.07570485, 1.05664036],
       [1.02088579, 1.01429564, 1.10641122],
       [0.9902891 , 0.99415038, 1.15715317],
       [0.9459689 , 0.98590814, 1.2097156 ],
       [0.92470052, 1.01991785, 1.19697428],
       [0.99976343, 1.01545332, 1.1263759 ],
       [1.04511172, 1.04007693, 1.056404  ],
       [1.04809166, 1.03812455, 1.05537644],
       [1.08224756, 1.10399537, 0.95534972],
       [1.04158295, 1.03313299, 1.06687672],
       [1.0990193 , 0.99258277, 1.04999059],
       [1.10548523, 1.0145333 , 1.02157412],
       [1.02624977, 1.02805912, 1.08728377],
       [1.03908728, 1.06973845, 1.03276692],
       [1.05110843, 1.05626994, 1.03421428],
       [1.06482283, 1.03761617, 1.03915365],
       [1.06074652, 1.02161324, 1.05923289],
       [1.02571879, 1.04350875, 1.07236512],
       [1.0316953 , 1.0775469 , 1.03235046],
       [1.17069381, 0.9940823 , 0.97681655],
       [1.14181163, 1.03716605, 0.96261497],
       [1.04492941, 1.01682975, 1.07983349],
       [0.96354344, 0.96707797, 1.21097125],
       [0.91786467, 0.92772245, 1.29600553],
       [1.00472772, 1.0424073 , 1.09445764],
       [0.99179043, 1.09749276, 1.05230946],
       [0.99162233, 1.07256007, 1.07741025],
       [1.1551524 , 1.09801001, 0.88843024],
       [0.96959454, 0.96237534, 1.20962277],
       [1.08101658, 1.04696683, 1.01360924],
       [1.13545257, 1.02529973, 0.98084035],
       [0.95349397, 1.06670398, 1.1213947 ],
       [1.09189794, 1.00351783, 1.04617688],
       [1.01676801, 1.05745973, 1.06736491],
       [1.01062735, 0.99857688, 1.13238843],
       [1.05249609, 1.01678665, 1.07230991],
       [1.04627367, 1.07908671, 1.01623228],
       [0.9894408 , 1.00159051, 1.15056134],
       [1.02142965, 1.07001813, 1.05014488],
       [1.03708901, 1.05718601, 1.04731763],
       [1.04858921, 1.01438322, 1.07862021],
       [1.05550091, 1.04134465, 1.0447471 ],
       [1.09404548, 1.03483415, 1.01271303],
       [1.08268371, 1.0730425 , 0.98586644],
       [1.1373943 , 1.04981281, 0.95438554],
       [0.96620758, 1.04077897, 1.1346061 ],
       [1.05154997, 1.13271372, 0.95732896],
       [1.11761028, 1.01024629, 1.01373609],
       [1.01310829, 1.13852244, 0.98996193],
       [1.07797009, 0.9945214 , 1.06910116],
       [1.10940123, 0.9626768 , 1.06951462],
       [1.02451043, 1.14922662, 0.9678556 ],
       [1.05067578, 1.13273466, 0.95818221],
       [1.07068443, 1.03672045, 1.03418778],
       [1.10931511, 0.93182911, 1.10044843],
       [1.08334873, 1.12779495, 0.93044898],
       [1.10084705, 1.01167615, 1.02906945],
       [1.08153302, 1.04881598, 1.01124365],
       [1.05518332, 0.97240436, 1.11400498],
       [1.10011585, 1.22266734, 0.81880946],
       [0.86376179, 1.0909195 , 1.18691137],
       [1.07717141, 1.02725325, 1.037168  ],
       [0.95889701, 1.20240735, 0.9802883 ],
       [1.06716577, 1.04082923, 1.03359765],
       [1.13749302, 0.99062813, 1.0134715 ],
       [1.00040699, 1.1430144 , 0.99817126],
       [1.07247938, 1.12367061, 0.94544266],
       [1.0784099 , 1.06106065, 1.0021221 ],
       [1.09619495, 0.96091296, 1.08448475],
       [1.05049421, 1.04690399, 1.04419445],
       [1.04898176, 1.07427852, 1.01833236],
       [1.02542376, 1.07743057, 1.03873833],
       [1.00269138, 1.07048974, 1.06841153],
       [1.08195294, 0.97888811, 1.08075161],
       [1.09754669, 1.1151931 , 0.92885286],
       [1.17599813, 0.80910537, 1.15648915],
       [0.99041666, 1.12457245, 1.02660354],
       [0.93367415, 1.14236547, 1.06555303],
       [1.10145823, 1.08744733, 0.9526871 ],
       [1.05706783, 1.05137559, 1.03314923],
       [1.02212268, 1.00001547, 1.1194545 ],
       [1.05018679, 1.05621321, 1.03519266],
       [1.10642537, 1.0105912 , 1.02457609],
       [1.00168221, 1.18462876, 0.95528168],
       [0.98222683, 1.2033639 , 0.95600193],
       [1.23326861, 0.88568412, 1.02263992],
       [1.23590944, 0.82865122, 1.07703199],
       [1.05764288, 1.13088177, 0.953068  ],
       [1.09937602, 1.06942283, 0.97279381],
       [1.00967728, 1.10132119, 1.03059419],
       [1.13187803, 0.94155876, 1.06815586],
       [0.98933739, 1.05329225, 1.09896302],
       [1.01873479, 1.03545803, 1.08739984],
       [1.0511459 , 1.04865657, 1.04179018],
       [1.0393524 , 1.0438768 , 1.05836346],
       [1.03681702, 1.01626415, 1.08851149],
       [1.03828661, 1.07514421, 1.02816184],
       [1.03219487, 1.08490395, 1.02449383],
       [1.00330381, 1.0518621 , 1.08642675],
       [1.02026535, 1.0811074 , 1.0402199 ],
       [1.04229203, 1.11261587, 0.98668475],
       [1.01840512, 0.88347848, 1.23970905],
       [0.97158601, 1.04126057, 1.12874607],
       [1.10262802, 1.08214991, 0.95681473],
       [1.06182232, 1.05551506, 1.02425528],
       [0.97203724, 1.16340471, 1.0061507 ],
       [0.9649149 , 1.14715626, 1.0295215 ],
       [1.12230011, 1.05154913, 0.96774341],
       [1.09761563, 0.9659884 , 1.07798863],
       [1.06366716, 0.98394946, 1.09397603],
       [1.08357379, 1.099642  , 0.95837686],
       [1.10469367, 1.02096188, 1.01593711],
       [0.98919989, 1.14328279, 1.00910997],
       [1.09576074, 1.06449691, 0.98133499],
       [0.92910753, 1.05238365, 1.16010147],
       [1.05131809, 1.08766292, 1.00261165],
       [1.06208873, 1.02844317, 1.05106075],
       [1.02725759, 1.10658403, 1.00775103],
       [1.05272887, 1.07421236, 1.01465143],
       [1.03443012, 1.03062804, 1.0765345 ],
       [1.15496809, 0.990735  , 0.99588956],
       [1.05641038, 1.05015377, 1.0350285 ],
       [1.0478885 , 1.02227805, 1.07142611],
       [1.03859769, 1.06014528, 1.04284969],
       [1.02283496, 1.07987186, 1.03888583],
       [1.08022936, 1.03167547, 1.02968782],
       [1.02232013, 1.10006985, 1.01920267],
       [1.0477806 , 1.0142977 , 1.07951435],
       [1.01944288, 1.14562385, 0.97652592],
       [0.94089361, 1.079063  , 1.12163604],
       [0.91886092, 1.11454203, 1.1081897 ],
       [1.11884688, 1.00095512, 1.02179066],
       [0.99027851, 1.00543173, 1.14588242],
       [1.03336407, 1.05796559, 1.050263  ],
       [0.96481455, 1.03647   , 1.1403081 ],
       [0.96833025, 1.05486271, 1.1183997 ],
       [1.03476299, 1.02287004, 1.08395963],
       [1.14905502, 0.96633129, 1.02620634],
       [0.8774135 , 1.17325707, 1.09092209],
       [1.08791362, 1.01188381, 1.04179522],
       [1.09547413, 1.02570326, 1.02041527],
       [1.0655626 , 1.04673207, 1.02929798],
       [1.10948081, 0.94512318, 1.08698866],
       [0.89743965, 1.19626074, 1.04789226],
       [1.04125797, 0.96610728, 1.13422741],
       [1.18429151, 1.03193205, 0.9253691 ],
       [1.11959428, 0.98181243, 1.04018594],
       [1.00546601, 0.95646474, 1.1796619 ],
       [1.03845396, 1.17643718, 0.92670151],
       [1.0438371 , 1.0204591 , 1.07729646],
       [1.01427068, 1.06117552, 1.06614645],
       [0.95009317, 1.09577062, 1.09572886],
       [1.0799894 , 1.01368116, 1.0479221 ],
       [1.0641161 , 1.01912933, 1.05834723],
       [1.06658836, 1.02448578, 1.05051851],
       [1.08194112, 1.0294312 , 1.03022034],
       [1.10171763, 1.04325772, 0.99661731],
       [0.9961327 , 1.06638276, 1.07907719],
       [1.54658822, 0.80162779, 0.79337664],
       [1.22276468, 0.89964969, 1.01917828],
       [0.93539911, 0.97916781, 1.22702573],
       [1.09441828, 1.08745124, 0.95972314],
       [1.02853285, 1.02472527, 1.08833454],
       [1.05226391, 1.06808691, 1.02124183],
       [1.06298455, 0.97717358, 1.10143453],
       [0.97323283, 1.11089992, 1.05745991],
       [1.13461501, 0.98058642, 1.02639122],
       [1.19291203, 0.96364215, 0.98503848],
       [0.82633607, 1.54469081, 0.77056577],
       [1.54924547, 0.85893927, 0.73340791],
       [1.01363528, 1.03817642, 1.08978095],
       [1.01923866, 1.04285029, 1.0795037 ],
       [1.02876107, 1.06464028, 1.04819131],
       [0.95610277, 1.11114165, 1.07434824],
       [1.00607478, 1.04418338, 1.09133449],
       [1.03124136, 1.01615754, 1.09419376],
       [1.03389089, 1.01514747, 1.09255429],
       [1.07549625, 1.04560724, 1.02048916],
       [1.08792476, 1.04474924, 1.00891866],
       [1.03498946, 1.04028981, 1.06631338],
       [1.03323703, 1.04553351, 1.06282212],
       [1.02785313, 1.07876274, 1.03497678],
       [1.02304213, 1.06622374, 1.05232679],
       [1.09689202, 1.00945539, 1.03524525],
       [0.77974226, 1.57894306, 0.78290734],
       [1.07362096, 1.13012496, 0.93784672],
       [1.15944768, 0.84023862, 1.14190636],
       [1.0635866 , 1.03987691, 1.03812914],
       [1.03677848, 1.05704361, 1.04777056],
       [1.09287671, 0.95775708, 1.09095886],
       [1.10341619, 1.09806203, 0.94011443],
       [1.04111619, 1.0969611 , 1.00351536],
       [1.01798332, 1.03957177, 1.08403757],
       [1.07785813, 0.97861539, 1.08511913],
       [0.9924035 , 1.05814466, 1.0910445 ],
       [0.98224321, 1.0091635 , 1.15018594],
       [1.1114596 , 0.98526281, 1.04487024],
       [1.08664707, 1.10160782, 0.95333777],
       [1.04173462, 1.08559644, 1.0142616 ],
       [1.0412465 , 1.06676586, 1.03358029],
       [1.09455772, 1.04301798, 1.00401695],
       [1.08548149, 1.02322453, 1.03288664],
       [1.03244354, 1.05638269, 1.05276642],
       [1.02536957, 1.10793695, 1.00828613],
       [1.04298439, 1.03441428, 1.06419398],
       [1.07639032, 1.04105691, 1.02414542],
       [1.03506538, 1.05661933, 1.04990794],
       [1.01929767, 1.08522046, 1.03707452],
       [1.02132969, 1.0549141 , 1.06534887],
       [1.01725269, 1.10577284, 1.01856712],
       [1.05173556, 1.07473829, 1.01511881],
       [1.03462654, 1.03513302, 1.0718331 ],
       [1.11102225, 0.959349  , 1.07122141],
       [1.10140156, 0.96957744, 1.07061366],
       [0.99543489, 1.01430118, 1.13185658],
       [1.01290895, 1.14493713, 0.98374657],
       [1.05364126, 1.00944661, 1.07850478],
       [1.12163937, 0.94492469, 1.07502859],
       [1.10587765, 1.12457741, 0.9111376 ],
       [1.04985658, 1.05788141, 1.03385467],
       [1.03682174, 1.12726727, 0.97750365],
       [1.45743579, 0.85511507, 0.8290418 ],
       [1.05350256, 1.09098759, 0.9971025 ],
       [0.99288461, 1.03588051, 1.11282753],
       [1.05827633, 1.00450027, 1.07881605],
       [1.2495571 , 0.86171718, 1.03031837],
       [0.91443741, 0.97514853, 1.25200671],
       [0.84603653, 1.40706279, 0.88849333],
       [1.00204775, 1.05368254, 1.08586236],
       [1.21512303, 0.92305472, 1.0034149 ],
       [1.00779043, 1.13878857, 0.99501366],
       [1.04117431, 1.10715578, 0.99326257],
       [0.93758823, 1.08314149, 1.12086293],
       [1.10075543, 0.72771056, 1.31312666],
       [1.08905465, 0.98898825, 1.06354976],
       [1.04225534, 1.14063816, 0.95869915],
       [1.03734084, 1.0845728 , 1.01967901],
       [1.06026246, 1.0521551 , 1.02917509],
       [0.99475378, 1.07199821, 1.07484066],
       [1.04257518, 1.10116352, 0.99785395],
       [1.19808505, 0.82142301, 1.1220846 ],
       [1.08405978, 0.93380739, 1.12372549],
       [1.56276119, 0.78906309, 0.78976837],
       [0.97654325, 1.01032591, 1.15472349],
       [1.23085564, 0.70785241, 1.20288461],
       [1.06043584, 0.99198218, 1.08917463],
       [1.11566128, 1.04540561, 0.98052577],
       [1.04326822, 1.03490292, 1.06342151],
       [1.00402421, 1.10972519, 1.02784325],
       [0.9990404 , 1.08636794, 1.05618431],
       [0.97961211, 1.19720479, 0.96477576],
       [1.19597634, 0.91979452, 1.0258218 ],
       [1.09624198, 0.80825778, 1.23709289],
       [1.13048079, 0.86652604, 1.14458582],
       [1.1516388 , 0.8660148 , 1.12393906],
       [1.13399173, 1.05187104, 0.95572989],
       [1.0678671 , 1.07479354, 0.99893202],
       [1.04841593, 1.08271032, 1.01046641],
       [1.03516064, 1.04870724, 1.05772478],
       [1.07110436, 1.06165773, 1.00883056],
       [1.02998136, 1.08474751, 1.02686378],
       [1.0637478 , 1.02481034, 1.05303451],
       [1.04352162, 1.01152857, 1.08654247],
       [1.07177327, 1.03614345, 1.03367593],
       [1.0803569 , 1.02380759, 1.03742816],
       [1.04563462, 1.10188288, 0.99407516],
       [1.04912526, 0.99869704, 1.09377036],
       [0.97574254, 1.05312215, 1.11272797],
       [0.92446809, 1.40641811, 0.81070645],
       [0.89082372, 1.10691334, 1.14385559],
       [1.1955724 , 0.85658423, 1.08943602],
       [0.97989124, 1.05298447, 1.10871695],
       [1.09777936, 1.02445306, 1.01936023],
       [1.07383972, 1.10178677, 0.96596617],
       [0.96869724, 1.05408012, 1.1188153 ],
       [0.97000148, 1.03118302, 1.14040816],
       [1.03735704, 1.09013476, 1.01410086],
       [1.10197399, 0.95271   , 1.08690866],
       [1.0515407 , 0.99906381, 1.09098814],
       [1.04859934, 1.09561343, 0.99737988],
       [0.93833724, 1.12450081, 1.0787546 ],
       [1.07431497, 1.00644285, 1.06083483],
       [1.0618104 , 1.04549966, 1.03428259],
       [1.00071119, 1.06835987, 1.0725216 ],
       [1.0400305 , 1.00556866, 1.09599349],
       [1.05935678, 1.0671841 , 1.01505178],
       [1.06436054, 1.04140495, 1.03582717],
       [1.12978297, 1.0091635 , 1.00264618],
       [1.03813963, 1.0652552 , 1.03819782],
       [1.04541214, 1.04327355, 1.05290696],
       [1.03706432, 1.03227006, 1.07225828],
       [1.04460831, 1.06698149, 1.03000286],
       [1.09001955, 1.0230874 , 1.02848571],
       [1.03909046, 1.0938977 , 1.00860449],
       [1.01088153, 1.01814752, 1.1125636 ],
       [1.01306827, 1.06996694, 1.05855743],
       [1.07959784, 1.09622239, 0.96577242],
       [0.90706506, 1.12769247, 1.10683513],
       [1.05570059, 1.07709426, 1.0087978 ],
       [1.13824863, 0.97988108, 1.02346294],
       [0.9455631 , 1.05445861, 1.14157095],
       [1.13425322, 0.92690776, 1.08043168],
       [0.99876962, 1.04684876, 1.09597427],
       [1.04187101, 1.02784248, 1.07187917],
       [1.03158539, 0.9750141 , 1.13499317],
       [1.60626217, 0.78279169, 0.75253879],
       [1.09853537, 1.10304384, 0.94001344],
       [0.82034771, 1.09727427, 1.22397067],
       [1.02102744, 1.04291135, 1.07765387],
       [1.24133679, 0.8485365 , 1.05171937],
       [0.76766824, 1.32354284, 1.05038158],
       [0.82932372, 0.99816554, 1.31410339],
       [1.15366108, 0.97190056, 1.01603102],
       [0.85509168, 1.32810085, 0.95840012],
       [0.96195912, 1.1641114 , 1.01552213],
       [1.20909303, 0.94430053, 0.98819909],
       [1.11528958, 1.04708167, 0.97922141],
       [0.96572993, 1.48956287, 0.68629985],
       [1.04305814, 0.94649644, 1.15203808],
       [1.09952582, 1.01624919, 1.02581765],
       [1.00669293, 1.01586832, 1.1190314 ],
       [1.01894991, 1.03728997, 1.08535278],
       [1.04292449, 1.04424815, 1.05442002],
       [1.03570152, 1.01064126, 1.09524987],
       [0.91024624, 1.24947939, 0.98186702],
       [0.80886414, 1.77347492, 0.5592536 ],
       [0.76851626, 1.20027189, 1.17280451],
       [1.02668525, 1.17287058, 0.94203682],
       [1.26713116, 0.93585384, 0.93860765],
       [0.98428014, 0.98038308, 1.17692943],
       [1.08355274, 0.98587253, 1.07216738],
       [1.04751723, 1.08475904, 1.00931638],
       [1.11807592, 1.1116728 , 0.91184393],
       [1.09384187, 0.94275877, 1.10499202],
       [0.99290149, 1.01959527, 1.12909589],
       [1.07428755, 1.09373025, 0.97357485],
       [0.95521916, 1.02497649, 1.16139701],
       [1.18496077, 0.96222728, 0.9944046 ],
       [0.9945686 , 1.16838307, 0.97864099],
       [1.11606048, 0.98500043, 1.04053175],
       [1.07487687, 1.00245099, 1.0642648 ],
       [1.09290109, 1.0109047 , 1.03778686],
       [1.06927057, 1.04150137, 1.03082072],
       [0.95634444, 1.13551513, 1.04973308],
       [1.01824929, 1.06468062, 1.05866274],
       [1.05805491, 1.03041523, 1.05312252],
       [1.06649114, 1.06921742, 1.00588409],
       [1.03792848, 1.06378277, 1.0398814 ],
       [1.07410805, 1.03430643, 1.03317817],
       [1.08719455, 1.03298959, 1.02140852],
       [0.9980164 , 1.05185667, 1.09171958],
       [1.08545174, 1.00693471, 1.0492062 ],
       [0.80499593, 1.23343361, 1.10316312],
       [1.01144982, 1.20439806, 0.92574478],
       [1.08246736, 1.1060571 , 0.95306819],
       [1.08877078, 1.00382812, 1.04899376],
       [1.05386642, 1.04073012, 1.04699612],
       [1.08253612, 1.08854228, 0.97051425],
       [1.07505338, 0.98210446, 1.08443481],
       [1.07393915, 1.05003772, 1.01761578],
       [1.08925623, 0.95973864, 1.09259778],
       [1.09743063, 1.06357127, 0.98059076],
       [0.97880042, 1.06157557, 1.10121666],
       [0.95696152, 1.08639104, 1.09824009],
       [1.04674902, 0.99114658, 1.10369706],
       [1.00434139, 1.05854358, 1.07870768],
       [1.0276831 , 1.05173413, 1.06217542],
       [1.0795317 , 1.08142876, 0.9806322 ],
       [0.98525017, 1.10168179, 1.0546607 ],
       [1.08628901, 0.99561715, 1.05968649],
       [1.03801917, 1.05816234, 1.04541114],
       [1.01408322, 1.106074  , 1.02143543],
       [1.03932074, 1.03859576, 1.06367615],
       [1.05982239, 1.04461656, 1.0371537 ],
       [1.06871531, 1.03903656, 1.03384078],
       [1.05929893, 1.01380352, 1.06849021],
       [1.02729097, 1.07913819, 1.0351635 ],
       [1.09577673, 0.99579963, 1.05001628],
       [1.07858933, 1.02320763, 1.0397957 ],
       [1.01325391, 1.04202993, 1.08630881],
       [1.05722963, 1.10912163, 0.97524139],
       [1.07216466, 1.13424156, 0.93518644],
       [1.02830502, 1.04684473, 1.0664429 ],
       [1.01919051, 1.00269732, 1.11970482],
       [0.92133448, 1.16601621, 1.05424197],
       [0.98464645, 1.05502621, 1.10191999],
       [1.04199263, 0.90999358, 1.18960645],
       [1.06526717, 1.02505925, 1.05126623],
       [1.10314198, 0.96651374, 1.07193694],
       [1.22208677, 0.70085214, 1.21865374],
       [0.79259388, 1.50311635, 0.84588243],
       [1.21689164, 1.00584718, 0.91885384],
       [1.02951059, 0.98246142, 1.12962065],
       [1.091378  , 0.89929484, 1.15091981],
       [1.50678337, 0.88139595, 0.75341334],
       [1.16261604, 1.02260235, 0.95637426],
       [0.94451554, 1.08231992, 1.1147572 ],
       [0.99232141, 1.05010832, 1.09916292],
       [1.01737111, 1.02950579, 1.09471576],
       [1.17319616, 0.96207572, 1.00632078],
       [1.06003959, 1.00927895, 1.07227412],
       [1.06921446, 1.12509065, 0.94728754],
       [1.1046456 , 1.13890681, 0.89804025],
       [1.1104844 , 1.0025056 , 1.02860266],
       [1.04017002, 1.0195298 , 1.08189283],
       [1.04910379, 1.02504915, 1.06743971],
       [1.09500421, 1.03710932, 1.00947913],
       [1.16635022, 1.05620121, 0.91904123],
       [0.84536611, 0.96063533, 1.33559121],
       [0.99604975, 1.02658591, 1.11895698],
       [1.11186409, 1.00358133, 1.02614723],
       [0.94510698, 1.13202002, 1.06446566],
       [1.14086559, 0.99494527, 1.00578179],
       [1.08267378, 1.04428516, 1.01463371],
       [1.08288555, 1.02723835, 1.03146876],
       [0.78303084, 0.7829456 , 1.57561621],
       [0.81706655, 1.48703292, 0.83749318],
       [0.98972084, 1.18717104, 0.96470078],
       [1.18454464, 1.0138838 , 0.94316421],
       [0.70790774, 1.02764938, 1.40603553],
       [0.97335494, 1.06168   , 1.10655771],
       [1.00977128, 1.0610093 , 1.07081208],
       [0.85392997, 1.20885895, 1.07880373],
       [1.00680914, 1.06184037, 1.07294314],
       [1.02116222, 1.05131883, 1.0691116 ],
       [1.06531661, 1.05437995, 1.0218961 ],
       [1.03604735, 1.1266565 , 0.9788888 ],
       [1.05146168, 1.0529156 , 1.03721537],
       [1.05466008, 1.012511  , 1.07442157],
       [1.06256901, 1.04107208, 1.03795157],
       [1.07538684, 1.05833683, 1.00786898],
       [1.03796341, 1.06005353, 1.04357571],
       [1.02445737, 1.04659596, 1.07053933],
       [1.04855999, 1.08255877, 1.0104739 ],
       [1.04403326, 1.02420098, 1.07335842],
       [1.01368068, 1.07420505, 1.05370692],
       [0.88676644, 0.98962624, 1.26519998],
       [1.0142876 , 1.05873227, 1.06857278],
       [0.96032899, 1.02821571, 1.15304796],
       [1.00250242, 1.10130263, 1.03778761],
       [0.91849202, 1.09532904, 1.1277716 ],
       [1.11464566, 0.98757927, 1.03936773],
       [1.05883558, 1.09532674, 0.98743033],
       [1.1576611 , 0.96419833, 1.01973322],
       [0.99387114, 1.14025398, 1.00746754],
       [1.11251584, 0.93556101, 1.09351581],
       [0.97822807, 1.04269128, 1.1206733 ],
       [0.96814076, 1.06696288, 1.10648901],
       [1.06841441, 1.09328169, 0.97989656],
       [1.08682413, 1.00229639, 1.05247213],
       [1.06643563, 1.02667566, 1.04848137],
       [1.01614732, 0.99652852, 1.12891681],
       [0.96585058, 1.10453366, 1.07120841],
       [1.02510268, 0.98816312, 1.12832685],
       [1.10164813, 0.9944823 , 1.04546223],
       [1.02811713, 1.06184555, 1.05162997],
       [1.01470676, 1.12764455, 0.99924134],
       [1.04869568, 1.05872356, 1.03417341],
       [1.04944504, 1.05886411, 1.0332835 ],
       [1.03897719, 1.05838569, 1.04422977],
       [1.06282873, 1.02827483, 1.0504891 ],
       [0.99350365, 1.04806053, 1.10002847],
       [1.0049902 , 1.05178481, 1.08481764],
       [1.11679434, 0.97616268, 1.04863563],
       [1.0336016 , 1.03139915, 1.0765919 ],
       [1.10143193, 0.98600305, 1.05415767],
       [0.90668941, 1.10646425, 1.12843899],
       [1.07729527, 1.0118472 , 1.05245018],
       [1.01380092, 1.08239396, 1.04539778],
       [0.91792625, 0.9737036 , 1.2499628 ],
       [1.06250101, 1.19068927, 0.88840237],
       [1.09512521, 1.04556194, 1.0009055 ],
       [1.05945218, 1.06504742, 1.01709305],
       [0.94239841, 1.05477068, 1.14442357],
       [0.89306801, 0.83318245, 1.41534219],
       [1.10618092, 0.98192816, 1.05348357],
       [0.94371754, 1.11292493, 1.08495019],
       [0.99850312, 0.89901041, 1.24407913],
       [1.26407568, 0.93218519, 0.94533178],
       [1.15721255, 0.95304219, 1.03133792],
       [1.05186654, 1.04457294, 1.04515318],
       [0.9654024 , 1.11713256, 1.05905769],
       [1.02817964, 1.09677108, 1.01664193],
       [0.89385437, 1.41964836, 0.82808992],
       [1.07790866, 1.00755767, 1.05612633],
       [1.02162233, 1.11377724, 1.00619308],
       [0.92889558, 1.16726088, 1.0454362 ],
       [1.09770326, 0.9203797 , 1.1235097 ],
       [0.97660016, 0.96946486, 1.19552764],
       [1.13431646, 1.01512874, 0.99214746],
       [1.02820068, 1.08821041, 1.02518156],
       [1.04201308, 1.06781562, 1.03176395],
       [1.01721183, 0.91470281, 1.20967801],
       [1.1646736 , 1.12638801, 0.85053103],
       [1.05024921, 1.01658231, 1.07476113],
       [0.95557726, 1.02311082, 1.16290458],
       [1.11579675, 0.98329537, 1.04250054],
       [1.02809361, 1.09136409, 1.02213496],
       [1.03797115, 1.11339624, 0.99022526],
       [1.19436762, 1.12810255, 0.81912248],
       [1.19035903, 0.81885954, 1.13237408],
       [0.97074077, 1.08309312, 1.08775876],
       [1.02081528, 1.028499  , 1.09227837],
       [0.78705292, 1.40044812, 0.95409161],
       [1.03886959, 1.11245359, 0.99026948],
       [1.06214732, 1.0243486 , 1.05509673],
       [0.98282313, 0.88766882, 1.2711007 ],
       [1.08860257, 1.01139022, 1.04159987],
       [1.03747629, 1.0915795 , 1.01253686],
       [1.03160983, 1.05576541, 1.05421742],
       [1.12171451, 0.96094647, 1.05893167],
       [1.04430629, 1.04459401, 1.05269235],
       [1.04842111, 1.08005899, 1.01311255],
       [1.0436538 , 1.04418849, 1.05375036],
       [1.10766208, 1.00508706, 1.02884351],
       [1.04851579, 1.03800714, 1.05506972],
       [1.02701896, 1.07343227, 1.04114142],
       [1.03695127, 1.02827864, 1.07636275],
       [0.98344353, 1.14634518, 1.01180394],
       [1.02516559, 1.06857321, 1.04785386],
       [0.98077161, 1.05227501, 1.10854603],
       [1.04351707, 1.0270928 , 1.07098278],
       [0.94998638, 1.14335285, 1.04825343],
       [1.05288113, 1.05219256, 1.03651897],
       [0.97697306, 1.01976611, 1.14485348],
       [1.01458592, 1.09422426, 1.03278248],
       [1.13692451, 1.0011962 , 1.00347194],
       [1.03752772, 1.04501532, 1.05904961],
       [1.09710695, 1.09197737, 0.95250833],
       [1.05862313, 1.05134089, 1.03162864],
       [1.10342085, 0.9901569 , 1.0480149 ],
       [1.08534443, 0.98720801, 1.06904021],
       [1.02992606, 1.00884511, 1.10282149],
       [1.0169603 , 1.05377123, 1.07086112],
       [1.09861355, 1.04266372, 1.00031538],
       [1.13111492, 0.97695545, 1.03352228],
       [0.87469373, 1.12076679, 1.14613212],
       [1.15459262, 1.03352183, 0.9534782 ],
       [0.9829793 , 1.06021766, 1.0983957 ],
       [1.07116603, 1.03385461, 1.03657201],
       [1.01916111, 0.97372263, 1.14870891],
       [1.0454496 , 1.04853343, 1.04760963],
       [1.0473666 , 1.05447117, 1.03975489],
       [1.02840371, 1.0752522 , 1.03793674],
       [1.04879434, 1.04649651, 1.04630181],
       [1.10135823, 1.07838463, 0.9618498 ],
       [1.03976451, 1.06352088, 1.03830727],
       [0.96590886, 1.05239122, 1.12329258],
       [1.07486956, 1.04833772, 1.01838538],
       [1.03341304, 1.11420791, 0.99397171],
       [1.0695257 , 1.1567399 , 0.91532705],
       [1.0450727 , 1.05692133, 1.03959862],
       [0.94703625, 1.06963028, 1.12492612],
       [1.25983709, 0.97005107, 0.91170449],
       [0.74549014, 1.26375883, 1.13234368],
       [1.11725698, 1.09313329, 0.93120239],
       [1.02370423, 1.06110309, 1.05678534],
       [0.98886313, 1.06463603, 1.08809349],
       [0.85145368, 1.28123048, 1.00890849],
       [0.88740728, 1.18143554, 1.07274983],
       [0.87423529, 1.16094495, 1.10641242],
       [1.33189182, 0.89356806, 0.91613278],
       [1.03824125, 0.97306159, 1.13028982],
       [1.05248482, 1.05653413, 1.0325737 ],
       [1.08396127, 1.02977845, 1.02785294],
       [1.0604775 , 0.96330459, 1.11781056],
       [1.06294402, 1.02213463, 1.05651401],
       [0.6823798 , 1.2117588 , 1.24745405],
       [0.99980367, 1.06976354, 1.07202544],
       [0.99478198, 0.99033548, 1.15647519],
       [0.79339258, 0.8519646 , 1.49623547],
       [0.84775434, 0.94800326, 1.34583505],
       [1.01254233, 0.99376421, 1.13528611],
       [1.00436696, 1.10845029, 1.0287754 ],
       [1.06697191, 1.05836548, 1.01625527],
       [1.02927868, 1.03470172, 1.07761226],
       [1.14945655, 1.05396223, 0.93817387],
       [1.09432672, 0.96505694, 1.08220899],
       [1.02150553, 1.0700046 , 1.05008252],
       [1.02867421, 1.17190802, 0.94101043],
       [0.99579097, 1.03986601, 1.10593567],
       [1.18296554, 0.89839302, 1.06023409],
       [0.99710422, 1.06829932, 1.07618911],
       [0.89579792, 0.94939883, 1.2963959 ],
       [0.94126195, 1.27486997, 0.92546073],
       [0.94320117, 1.17215323, 1.02623825],
       [1.14539029, 0.99934582, 0.99685654],
       [1.12596522, 1.02365614, 0.9919713 ],
       [1.03610517, 1.06253556, 1.04295192],
       [1.06060004, 1.01570885, 1.06528377],
       [1.00948905, 1.05035034, 1.08175327],
       [1.05272531, 1.07051548, 1.01835186],
       [1.0370248 , 1.0312021 , 1.07336575],
       [1.10855921, 0.99687564, 1.0361578 ],
       [1.03489496, 1.04415522, 1.06254247],
       [1.05148366, 1.04564287, 1.04446613],
       [1.04813992, 1.08620427, 1.00724847],
       [1.05663297, 1.04335908, 1.0416006 ],
       [1.01385382, 1.03003535, 1.09770349],
       [1.0651368 , 1.03401383, 1.04244203],
       [1.0276166 , 1.04524346, 1.06873259],
       [1.06756833, 1.02702297, 1.04700136],
       [1.01128174, 1.03222047, 1.09809044],
       [1.01205738, 1.11194963, 1.01758564],
       [1.11898459, 1.08672627, 0.93588179],
       [0.99632801, 1.15638997, 0.98887468],
       [1.0624923 , 0.97880282, 1.10029753],
       [1.13196552, 0.96088309, 1.04874404],
       [0.91100173, 1.26121106, 0.96937986],
       [1.04871735, 1.01612768, 1.07674763],
       [1.01710396, 1.10941704, 1.01507166],
       [1.03755983, 1.04132674, 1.06270608],
       [0.82055508, 1.13226674, 1.18877083],
       [1.15312948, 0.95886091, 1.02960226],
       [1.05866897, 1.09315781, 0.98976588],
       [0.98375719, 1.06307621, 1.09475925],
       [1.1780336 , 1.01612804, 0.94743101],
       [1.13024562, 0.98151578, 1.02983125],
       [1.0459706 , 1.09495453, 1.00066752],
       [1.00531995, 1.03829901, 1.09797369],
       [0.92203541, 0.85485825, 1.36469899],
       [1.17691015, 0.95910952, 1.00557298],
       [0.95484307, 1.15763003, 1.02911955],
       [1.02443396, 1.11136932, 1.00578938],
       [1.03903473, 1.01988137, 1.08267655],
       [1.17481504, 0.98257524, 0.98420237],
       [1.07039188, 1.03284724, 1.03835353],
       [1.03269461, 1.06416912, 1.04472892],
       [1.06090402, 1.01868427, 1.06200436],
       [1.07963654, 1.03971098, 1.02224513],
       [1.10183066, 1.01325158, 1.02651041],
       [1.14771923, 0.92687964, 1.06699378],
       [1.04451013, 1.03489795, 1.06218457],
       [1.01147834, 1.00204163, 1.12807269],
       [1.00739945, 1.10381163, 1.03038157],
       [0.9154827 , 1.074315  , 1.15179495],
       [1.06366211, 1.05097094, 1.0269596 ],
       [0.94078776, 1.14688015, 1.05392474],
       [1.24231426, 0.81790601, 1.08137238],
       [1.75219366, 0.67340533, 0.71599367],
       [1.0626443 , 1.05921508, 1.01973327],
       [1.05392657, 1.01460277, 1.07306331],
       [0.94986968, 1.21848122, 0.97324175],
       [0.9645635 , 1.17480515, 1.002224  ],
       [1.10685138, 0.95344915, 1.08129212],
       [1.15444281, 0.93116713, 1.05598272],
       [1.75773447, 0.66787392, 0.71598426],
       [0.97340814, 1.15271942, 1.01546509],
       [1.01856938, 1.05773715, 1.06528613],
       [1.04031458, 1.02223444, 1.07904364],
       [1.09848211, 1.03232999, 1.01078055],
       [1.06776773, 1.0608743 , 1.01295062],
       [1.05707919, 0.95988251, 1.12463095],
       [1.06820566, 1.07237661, 1.00101038],
       [0.8452017 , 1.14925272, 1.14713823],
       [1.15798584, 0.90072675, 1.08288006],
       [0.83674526, 0.77834569, 1.5265017 ],
       [1.30713842, 1.09740702, 0.73704721],
       [0.97502294, 0.91240066, 1.25416905],
       [0.97942202, 1.17664242, 0.98552821],
       [1.18187442, 0.9705018 , 0.98921643],
       [1.02931428, 0.98276654, 1.12951184],
       [1.02109496, 1.07313329, 1.0473644 ],
       [1.02323043, 1.00068256, 1.11767966],
       [1.09808129, 1.06352863, 0.97998274],
       [1.00168968, 1.05075303, 1.08914994],
       [1.00805815, 1.24338823, 0.89014627],
       [1.18153887, 0.95052367, 1.00953012],
       [1.03837336, 1.06564376, 1.03757553],
       [0.83353708, 1.28141049, 1.02664509],
       [1.18605412, 0.93963653, 1.015902  ],
       [0.98533118, 1.07320247, 1.083059  ],
       [1.05688425, 0.97578047, 1.10892793],
       [1.10830637, 1.03115232, 1.00213396],
       [1.03624672, 1.0643966 , 1.04094933],
       [1.04904309, 1.05096088, 1.04158869],
       [1.0371966 , 1.08283094, 1.02156511],
       [1.02319697, 1.0059339 , 1.11246178],
       [1.01792478, 1.01627427, 1.10739361],
       [1.03498642, 1.09042977, 1.01617647],
       [1.07160614, 1.03650247, 1.03348405],
       [1.04665415, 1.04660613, 1.04833237],
       [1.00597553, 1.01890464, 1.11671248],
       [1.05615784, 1.03631598, 1.04911884],
       [1.01445005, 0.98236844, 1.14477416],
       [1.05958957, 1.01693551, 1.06506757],
       [1.00925724, 1.07479708, 1.05753834],
       [1.09398589, 1.06646894, 0.98113783],
       [1.05201953, 0.96480262, 1.1247705 ],
       [1.07410247, 1.14082543, 0.92666476],
       [1.04643132, 0.9552295 , 1.13993183],
       [1.0171982 , 1.01531298, 1.10908148],
       [1.11669978, 1.02453059, 1.00036228],
       [1.13078129, 1.04339239, 0.96741898],
       [1.19522006, 0.68961683, 1.25675576],
       [1.02682777, 1.13223904, 0.98252585],
       [1.03754943, 1.05669263, 1.04735059],
       [1.07384192, 1.04038311, 1.02736762],
       [1.81273914, 0.58352691, 0.7453266 ],
       [1.13028282, 1.02914425, 0.98216559],
       [1.08778016, 0.99178034, 1.06203215],
       [1.03096819, 1.09881056, 1.0118139 ],
       [0.86545215, 1.20396693, 1.07217357],
       [1.08855437, 1.05764361, 0.99539467],
       [1.04026157, 1.01639089, 1.08494019],
       [1.12507467, 1.1055449 , 0.91097308],
       [0.69656144, 0.67756828, 1.76746293],
       [0.94800325, 1.02068191, 1.17290749],
       [1.01985261, 0.9834407 , 1.13829934],
       [1.00147061, 1.03613017, 1.10399188],
       [1.03220274, 1.07610675, 1.03328316],
       [0.98565822, 1.00267785, 1.15325658],
       [1.04126744, 1.07217955, 1.02814566],
       [1.04350184, 1.0387228 , 1.05936802],
       [1.01967755, 1.04517136, 1.07674375],
       [0.96220507, 1.1673016 , 1.01208598],
       [1.00518919, 1.11081978, 1.02558369],
       [0.89609515, 1.1048567 , 1.1406408 ],
       [0.9927559 , 1.08860988, 1.06022687],
       [0.99902737, 1.12410155, 1.01846373],
       [1.04503754, 0.9920692 , 1.10448591],
       [1.0597168 , 1.17683734, 0.90503851],
       [1.06174646, 1.03387485, 1.04597135],
       [0.97904988, 0.99563506, 1.16690771],
       [1.0700822 , 1.1714259 , 0.90008455],
       [1.28628423, 0.88364573, 0.9716627 ],
       [1.04445188, 1.04199329, 1.05514748],
       [1.03295868, 1.06556688, 1.04306709],
       [1.11656654, 1.02076546, 1.00426065],
       [1.10340432, 1.03156749, 1.00662084],
       [1.10150384, 0.98568123, 1.05440758],
       [0.99821591, 1.08842387, 1.05495288],
       [1.07822691, 1.1471596 , 0.91620614],
       [1.05138817, 1.05089241, 1.03931208],
       [1.06712849, 1.0325338 , 1.04193037],
       [1.08644454, 1.03086153, 1.02428658],
       [1.0379186 , 1.03467699, 1.06899707],
       [1.1275941 , 1.01561839, 0.99838017],
       [1.06556308, 1.02159683, 1.05443275],
       [1.0888572 , 1.17356212, 0.87917334],
       [1.00877026, 1.3615264 , 0.771296  ],
       [1.06189829, 1.23962875, 0.84006561],
       [0.91375289, 1.0587187 , 1.16912106],
       [1.28884782, 1.02758875, 0.82515609],
       [1.02324519, 1.01261321, 1.10573425],
       [1.03150708, 1.02124993, 1.08883564],
       [1.05991672, 1.00013786, 1.08153808],
       [1.04876804, 1.08718329, 1.00564133],
       [1.08638166, 1.03740315, 1.01780784],
       [1.03776738, 1.01825828, 1.08556699],
       [1.0616655 , 1.04246298, 1.03746418],
       [0.86530999, 1.05142097, 1.22486169],
       [1.00583242, 1.16135001, 0.97441022],
       [1.09301283, 1.07986661, 0.96871321],
       [0.66455846, 1.84362946, 0.63340474],
       [0.92293496, 0.95857659, 1.2600811 ],
       [1.01971364, 1.16180277, 0.96007625],
       [1.05365729, 1.03211042, 1.05582495],
       [1.04490345, 1.05619706, 1.04049214],
       [1.06635851, 1.03005119, 1.04518295],
       [1.033679  , 1.05006197, 1.05785168],
       [1.04290569, 1.07060204, 1.02808492],
       [1.0156994 , 1.10911871, 1.01677454],
       [1.11358619, 1.00602245, 1.02198402],
       [1.05442259, 1.05107102, 1.03609905],
       [1.02062471, 1.08632103, 1.03464691],
       [1.06440743, 1.03954525, 1.03763997],
       [1.04998269, 1.05342345, 1.03818651],
       [1.10695113, 1.02400722, 1.0106343 ],
       [1.04772452, 1.04750285, 1.04636529],
       [0.97439909, 1.08220579, 1.08498778],
       [1.06933465, 1.0408951 , 1.0313629 ],
       [1.02507658, 1.08086347, 1.0356526 ],
       [1.09242431, 0.99036051, 1.05880783],
       [1.02605644, 1.12024617, 0.99529004],
       [1.41224396, 0.85592103, 0.87342766],
       [1.08979556, 1.05718025, 0.99461683],
       [0.98752333, 1.07512945, 1.07893988],
       [1.05132468, 1.06185016, 1.02841782],
       [1.14660561, 0.97747625, 1.0175108 ],
       [1.21104414, 0.91947929, 1.01106922],
       [1.00397544, 0.96361332, 1.17400389],
       [1.0213898 , 1.07591287, 1.04428998],
       [1.01935092, 1.10285113, 1.0193906 ],
       [0.89526008, 0.97140409, 1.27492848],
       [1.05522712, 1.10664989, 0.97971565],
       [1.08208047, 1.00015585, 1.05935634],
       [1.06876313, 1.10433964, 0.96848988],
       [1.00882442, 1.01867244, 1.11409579],
       [1.05007778, 0.98590144, 1.10561343],
       [1.03319562, 1.12356762, 0.98482942],
       [1.25105084, 1.18556993, 0.70497188],
       [1.18648437, 0.97992925, 0.97517904],
       [1.34828431, 0.92864676, 0.86466159],
       [1.14972485, 0.99440175, 0.99746606],
       [1.01892722, 1.11357174, 1.00909369],
       [1.0235988 , 1.03178456, 1.08620929],
       [1.14094637, 1.01105941, 0.98958688],
       [1.11081788, 1.01857296, 1.01220181],
       [1.0253929 , 1.08449747, 1.03170228],
       [1.07898634, 1.06085604, 1.00175027],
       [0.95497975, 1.06759187, 1.11902103],
       [1.11765091, 0.92364209, 1.10029964],
       [1.09190861, 1.06980522, 0.97987882],
       [1.07644128, 0.90626844, 1.15888294],
       [1.09965169, 1.02653566, 1.0154053 ],
       [0.91387116, 1.06025776, 1.16746373],
       [1.00199965, 1.0550346 , 1.0845584 ],
       [1.05819747, 1.07499082, 1.00840437],
       [0.97716873, 0.99136874, 1.17305518],
       [1.10854806, 1.33375691, 0.69928768],
       [1.0331203 , 0.93219379, 1.17627856],
       [1.03170621, 0.99826399, 1.11162245],
       [1.02864019, 0.9305779 , 1.18237456],
       [1.06033604, 0.90910535, 1.17215126],
       [0.95541083, 1.05248602, 1.1336958 ],
       [1.10042383, 1.08392974, 0.95723908],
       [1.0202289 , 1.03061985, 1.09074391],
       [1.03000754, 1.07810897, 1.03347614],
       [1.06929367, 1.03314423, 1.03915476],
       [1.0555922 , 1.02271022, 1.06329024],
       [1.05421962, 1.01642185, 1.07095118],
       [1.00457182, 0.94735536, 1.18966548],
       [1.06855337, 1.01511805, 1.05792124],
       [1.17816265, 0.75569295, 1.20773706],
       [1.01857895, 0.91138561, 1.21162809],
       [1.14472848, 1.04232192, 0.95454224],
       [0.79894295, 1.62583414, 0.71681557],
       [0.87755671, 1.1710207 , 1.09301524],
       [0.98049026, 1.12081841, 1.04028398],
       [1.11480642, 1.00268574, 1.02410049],
       [1.09622287, 1.01832334, 1.02704644],
       [0.98936015, 1.11796187, 1.03427063],
       [1.01775455, 1.02979833, 1.09403977],
       [1.03220828, 1.08125292, 1.02813146],
       [1.12506061, 1.02735953, 0.98917251],
       [0.94829167, 1.26998398, 0.92331701],
       [1.47993833, 0.83128862, 0.8303657 ],
       [0.97100909, 0.98016751, 1.19041606],
       [1.00269275, 1.10164927, 1.03725063],
       [0.62600773, 1.3562549 , 1.15933003],
       [1.03425756, 0.98084024, 1.12649485],
       [1.07499014, 1.0361768 , 1.03042571],
       [0.99432636, 1.04015433, 1.10711196],
       [1.07836372, 1.02312832, 1.04010061],
       [1.04053821, 1.05191098, 1.04914346],
       [1.08840025, 1.05751316, 0.99567925],
       [1.13634311, 1.02720625, 0.9780433 ],
       [1.06906275, 1.14595872, 0.92657118],
       [1.09300483, 0.97188126, 1.07670656],
       [1.0789489 , 1.04673626, 1.01590749],
       [1.03236654, 1.06452906, 1.04469705],
       [1.03870739, 1.05164449, 1.05124078],
       [0.98601011, 1.11514052, 1.04044202],
       [1.02929702, 1.05816068, 1.05413496],
       [1.11559471, 1.06380934, 0.9621886 ],
       [1.07339353, 1.06116582, 1.00703331],
       [1.06798964, 0.97243298, 1.10117004],
       [1.17057206, 1.13217759, 0.838843  ],
       [0.99648954, 1.08457076, 1.06053235],
       [0.78544263, 0.67915485, 1.67699517],
       [0.97797068, 1.11883882, 1.04478315],
       [1.11570165, 1.02730612, 0.99858488],
       [1.07569234, 1.01869866, 1.04720165],
       [1.18359371, 1.13582446, 0.82217448],
       [1.02478118, 1.07446669, 1.04234479],
       [1.10365803, 1.07306091, 0.96487372],
       [1.00296622, 1.03709364, 1.10153279],
       [1.03241143, 1.04957676, 1.05960446],
       [1.0588398 , 1.12717603, 0.95557682],
       [1.00487132, 1.04255931, 1.09416203],
       [1.10735769, 1.02742605, 1.00680892],
       [1.04380839, 0.9901987 , 1.10758556],
       [1.02214899, 1.09566921, 1.02377445],
       [0.98592114, 1.08578471, 1.0698868 ],
       [1.01327005, 1.01566126, 1.11266134],
       [1.18311043, 0.93570904, 1.02277318],
       [0.99188826, 1.13347572, 1.01622868],
       [0.87554166, 1.15592428, 1.11012671],
       [1.01629633, 1.13460866, 0.99068766],
       [1.08804082, 1.02027608, 1.03327575],
       [1.04324029, 1.06886408, 1.02948829],
       [1.02320263, 1.11854554, 0.99984448],
       [1.10737122, 1.01975296, 1.01446847],
       [1.09266289, 1.02478074, 1.02414902],
       [1.08718923, 1.04742242, 1.00698101],
       [1.05969208, 1.11762968, 0.96427089],
       [0.95661296, 1.03328562, 1.15169408],
       [1.08745189, 1.07220623, 0.98193454],
       [0.96142265, 1.05912532, 1.12104468],
       [1.10150517, 1.00500753, 1.03507996],
       [1.17908702, 0.9093401 , 1.05316553],
       [0.98700772, 1.13104575, 1.02353918],
       [1.07276819, 1.00676325, 1.06206121],
       [0.96857662, 0.85744635, 1.31556969],
       [0.82920488, 0.65861976, 1.65376802],
       [1.03633378, 1.01007003, 1.09518885],
       [1.02229335, 1.04823993, 1.07105937],
       [1.07685562, 1.07080245, 0.99393459],
       [1.02550852, 1.04626783, 1.06981631],
       [1.03741593, 1.03407075, 1.07010597],
       [1.100144  , 1.01107963, 1.03036902],
       [0.973605  , 1.12817918, 1.03980847],
       [1.00249753, 1.0576064 , 1.08148872],
       [1.06092598, 1.05346221, 1.02720446],
       [0.94756295, 1.00279765, 1.19123205],
       [1.02763153, 1.06181682, 1.05214431],
       [0.84626278, 1.02634381, 1.26898607],
       [1.03764401, 1.04530648, 1.05864216],
       [0.95008005, 1.17953861, 1.01197399],
       [1.15797138, 0.95714933, 1.02647194],
       [1.1938668 , 1.13641039, 0.81131546],
       [0.88890223, 0.74837818, 1.50431224],
       [1.20383433, 0.98891525, 0.94884307],
       [1.03671773, 1.05742013, 1.0474548 ],
       [1.06735277, 1.06861564, 1.00562424],
       [1.03207203, 1.05323443, 1.0562862 ],
       [1.03577873, 1.09187594, 1.01393798],
       [1.01943003, 1.04910386, 1.07305877],
       [1.01721832, 1.04138935, 1.08298498],
       [1.07659909, 1.04818266, 1.01681091],
       [1.21180346, 0.96784275, 0.96194644],
       [0.84447297, 1.20816664, 1.08895304],
       [0.99684411, 1.17750033, 0.96724821],
       [1.13612653, 1.06108906, 0.94437706],
       [1.0612815 , 1.11592121, 0.96438994],
       [0.72498004, 1.12481806, 1.29179455],
       [1.08472761, 1.06719488, 0.98967016],
       [1.05793588, 1.07006794, 1.01358884],
       [1.05462028, 1.05454605, 1.03242633],
       [1.04339789, 1.05722629, 1.04096847],
       [1.00198203, 1.10922117, 1.03038945],
       [0.9262102 , 0.95634707, 1.25903539],
       [1.09153276, 1.08994485, 0.96011504],
       [1.10080772, 1.0088031 , 1.03198183],
       [1.04426579, 1.0122367 , 1.08509017],
       [1.02689488, 1.07237634, 1.04232144],
       [1.04755621, 1.03257263, 1.06146382],
       [1.05270968, 1.04055061, 1.04833236],
       [1.07664778, 0.96316875, 1.10177612],
       [1.03425588, 1.06033047, 1.04700631],
       [1.10098159, 0.97760971, 1.06300136],
       [1.02826826, 1.01470215, 1.09862224],
       [0.97080171, 1.1645771 , 1.00621384],
       [1.10094381, 1.24856659, 0.79208225],
       [1.05012503, 1.06299092, 1.02847671],
       [1.02001332, 1.09372955, 1.02784979],
       [1.0748847 , 1.03143012, 1.03527783],
       [0.96533316, 1.04918421, 1.12707528],
       [1.04129566, 1.06727892, 1.03301807],
       [1.01167466, 1.07419492, 1.05572307],
       [1.07983764, 0.94991477, 1.11184025],
       [1.04149412, 1.06094482, 1.03915372],
       [0.98840722, 1.02197282, 1.13121261],
       [1.00301207, 1.05079011, 1.08779048],
       [1.05699281, 1.08885291, 0.99574694],
       [0.99042365, 1.17825174, 0.97291726],
       [1.07073373, 1.02371139, 1.04714753],
       [1.09062822, 1.02885981, 1.02210462],
       [1.05519491, 1.0947791 , 0.99161864],
       [1.08353479, 1.08659927, 0.97145859],
       [0.95197028, 1.08374528, 1.10587709],
       [1.0051662 , 0.9597598 , 1.17666666],
       [0.99643939, 0.99908038, 1.14607288],
       [1.10657082, 0.93862682, 1.09639502],
       [1.00308698, 1.00737766, 1.13112802],
       [1.10147695, 1.00220276, 1.03791295],
       [1.03366181, 1.00500209, 1.10292875],
       [1.01237386, 1.00605412, 1.12316467],
       [1.04289651, 0.97212511, 1.12657103],
       [0.99433197, 1.10518066, 1.04208002],
       [0.93304307, 1.10692336, 1.10162623],
       [1.14480451, 0.97376867, 1.02301948],
       [1.11608398, 0.9728841 , 1.05262457],
       [0.99633284, 0.97625836, 1.16900144],
       [1.16774612, 0.90200252, 1.07184401],
       [1.06736845, 1.13500921, 0.93921499],
       [1.00235328, 1.0537567 , 1.08548267],
       [0.83586192, 1.24587018, 1.05986056],
       [1.17769483, 0.78966918, 1.17422865],
       [1.01356058, 1.11370468, 1.0143274 ],
       [1.09292031, 1.01745501, 1.03121733],
       [1.08806437, 0.94539222, 1.10813607],
       [1.00039499, 1.0069166 , 1.13428106],
       [1.10641706, 1.07226263, 0.96291297],
       [1.0019812 , 1.09905867, 1.04055278],
       [1.06404362, 1.02039251, 1.05715652],
       [1.07394164, 1.02645316, 1.04119785],
       [1.02374891, 1.07594154, 1.04190221],
       [1.1462015 , 1.08817759, 0.90721356],
       [1.04162581, 1.02817946, 1.07178738],
       [0.82063102, 1.3370511 , 0.98391053],
       [1.0428973 , 1.05666778, 1.04202757],
       [1.05155981, 1.01975216, 1.07028069],
       [1.15294413, 1.03545097, 0.95319756],
       [1.00116749, 0.98406395, 1.15636121],
       [1.01298767, 1.21660537, 0.91199961],
       [1.04713737, 0.92580615, 1.16864913],
       [0.90245091, 1.25090387, 0.98823787],
       [0.99523302, 1.05454728, 1.09181236],
       [1.05214321, 0.98671316, 1.10273629],
       [1.06389989, 1.05849611, 1.01919665],
       [1.06866155, 1.03272642, 1.04020468],
       [1.07266048, 1.02336073, 1.04557144],
       [1.07214965, 1.01721039, 1.05223261],
       [1.03531603, 1.04319399, 1.06308264],
       [0.95070847, 0.94307168, 1.24781251],
       [0.98402194, 1.23338222, 0.92418849],
       [1.03348438, 0.92815353, 1.17995474],
       [0.88266468, 0.97840231, 1.28052566],
       [1.58042507, 0.87328515, 0.68788243],
       [0.94657994, 1.192594  , 1.00241872],
       [1.08486096, 1.04390761, 1.01282408],
       [1.02639805, 1.05652042, 1.05867418],
       [1.08000507, 0.9905142 , 1.07107339],
       [1.07396743, 1.05628007, 1.01134516],
       [1.07978158, 1.01763757, 1.04417351],
       [1.12208552, 1.08716097, 0.93234617],
       [0.78650392, 1.29007445, 1.06501428],
       [0.83747651, 1.15899954, 1.1451166 ],
       [0.98186799, 1.087865  , 1.07185967],
       [0.99198005, 1.10183538, 1.04777723],
       [1.06636627, 1.04811031, 1.02711607],
       [1.04703195, 1.03699884, 1.05756186],
       [1.04896448, 1.03174353, 1.06088465],
       [1.1250921 , 1.06056863, 0.95593193],
       [1.06020696, 1.00802619, 1.0733595 ],
       [0.96864139, 1.10323272, 1.06971855],
       [0.98717345, 1.10615822, 1.04826098],
       [1.03650924, 0.94667578, 1.15840764],
       [1.22110265, 0.96981533, 0.95067468],
       [1.04432681, 1.07551459, 1.02175125],
       [1.01805255, 1.08903809, 1.03450201],
       [1.0224583 , 1.07292688, 1.04620747],
       [1.0689283 , 0.96928971, 1.10337465],
       [1.12489495, 1.00973647, 1.00696123],
       [0.9721218 , 0.96623119, 1.20323967],
       [1.08413467, 1.07210553, 0.98535245],
       [0.94291483, 1.1393131 , 1.05936473],
       [1.0404619 , 1.02279388, 1.07833687],
       [1.04186301, 1.07894879, 1.02078086],
       [1.03142911, 1.02198756, 1.08817598],
       [1.02535919, 1.09896849, 1.01726497],
       [1.04929432, 1.07060519, 1.02169314],
       [1.05537358, 1.01485918, 1.0713599 ],
       [1.08176749, 1.00716248, 1.05266268],
       [1.08326307, 0.98548803, 1.07284155],
       [0.97868075, 1.0676822 , 1.0952297 ],
       [0.96181415, 1.12118998, 1.05858852],
       [1.08897235, 1.03372978, 1.01889053],
       [1.09929742, 1.09324831, 0.94904693],
       [1.1030737 , 0.98456214, 1.05395681],
       [1.01913455, 1.03946983, 1.08298827],
       [1.01673212, 1.03193224, 1.09292829],
       [1.00291568, 1.12462417, 1.0140528 ],
       [1.013595  , 1.09323155, 1.0347661 ],
       [1.03100087, 0.99500644, 1.11558534],
       [1.06102492, 0.95140628, 1.12916145],
       [1.00074867, 1.01303025, 1.12781374],
       [0.98293555, 1.04900315, 1.10965396],
       [1.02655892, 1.16676089, 0.94827284],
       [1.05134227, 1.19466101, 0.89558937],
       [0.96545227, 1.0390551 , 1.13708529],
       [0.99419809, 1.07449526, 1.07289931],
       [1.06386703, 1.15487765, 0.92284797],
       [0.994194  , 1.28382321, 0.86357545],
       [1.02600804, 0.99162993, 1.12395469],
       [1.02805661, 1.07230881, 1.04122723],
       [1.02498773, 0.99383453, 1.1227704 ],
       [1.16588945, 1.04691263, 0.92879057],
       [1.08327899, 0.93991069, 1.11840298],
       [1.07615905, 0.90488381, 1.16054979],
       [1.12812084, 1.03903971, 0.97443211],
       [1.0615643 , 1.05754948, 1.02247887],
       [1.04053093, 1.07010057, 1.03096115],
       [0.91575932, 1.07212374, 1.1537096 ],
       [1.06015437, 1.00442791, 1.07701037],
       [1.00997001, 1.03491912, 1.09670352],
       [0.82759101, 1.36090443, 0.95309721],
       [1.03127836, 1.10199004, 1.00832426],
       [1.09937376, 1.03457798, 1.00764091],
       [1.04149509, 1.05938839, 1.04070918],
       [0.94580014, 1.1446748 , 1.05111771],
       [1.15280124, 1.01029842, 0.97849299],
       [1.24043894, 1.21571475, 0.68543896],
       [0.80125876, 0.51918134, 1.82115255],
       [1.04726932, 0.99947145, 1.09485189],
       [1.04410837, 1.02385611, 1.07362818],
       [1.02949845, 1.02717385, 1.08492036],
       [1.04374113, 1.06427362, 1.0335779 ],
       [1.03631212, 1.05447734, 1.05080319],
       [1.05392866, 1.0850401 , 1.00262389],
       [1.12820002, 0.9793764 , 1.03401624],
       [0.98560444, 1.14210401, 1.0138842 ],
       [1.1183704 , 0.96244356, 1.0607787 ],
       [1.20320123, 1.01552179, 0.92286963],
       [0.88192796, 1.18485355, 1.07481114],
       [0.88183268, 1.26658644, 0.99317353],
       [1.07750736, 1.1175656 , 0.94651969],
       [1.09009399, 0.95999113, 1.09150754],
       [0.9836372 , 1.09041433, 1.06754113],
       [1.05870448, 0.99896472, 1.08392345],
       [1.04750626, 1.01377338, 1.08031302],
       [1.03598797, 1.08421405, 1.02139063],
       [1.06749964, 0.93340609, 1.14068693],
       [0.67311872, 1.61289786, 0.85557607],
       [0.97115578, 0.9662194 , 1.20421747],
       [1.00419493, 1.02550322, 1.11189451],
       [1.11420571, 1.05976599, 0.96762095],
       [1.04584131, 1.01188256, 1.08386879],
       [1.03875164, 1.06448044, 1.03836057],
       [1.05780741, 1.04600958, 1.03777566],
       [1.06639239, 1.04023118, 1.03496909],
       [1.07810488, 0.96586157, 1.0976262 ],
       [1.09045453, 1.03775735, 1.01338078],
       [0.99043674, 1.13630946, 1.01484645],
       [1.01472041, 1.1332309 , 0.99364135],
       [1.04883407, 1.04754616, 1.04521242],
       [1.15769855, 0.97825079, 1.00564332],
       [1.08589331, 1.06938121, 0.98631814],
       [1.06213829, 0.97202423, 1.10743013],
       [1.15251669, 0.87560588, 1.11347009],
       [0.97095027, 1.14015683, 1.03048555],
       [0.95948796, 1.13365845, 1.04844624],
       [0.94674184, 1.17043509, 1.02441572],
       [0.99364055, 1.13876895, 1.00918315],
       [1.07746685, 1.00350385, 1.06062195],
       [1.05423933, 1.0921292 , 0.99522413],
       [1.0377111 , 1.03441796, 1.06946359],
       [1.04973447, 1.03667924, 1.05517895],
       [1.05898755, 1.04799077, 1.03461434],
       [1.01040984, 1.06409434, 1.06708848],
       [1.05013816, 1.08233857, 1.00911593],
       [0.97467623, 1.07294443, 1.09397199],
       [1.07959814, 0.99112677, 1.07086774],
       [1.11547442, 1.04999243, 0.9761258 ],
       [1.00998331, 0.99828714, 1.1333222 ],
       [1.11711025, 0.98731519, 1.03716721],
       [0.97396518, 1.08373417, 1.0838933 ],
       [1.01739563, 1.08129005, 1.04290698],
       [1.02276394, 1.09158622, 1.02724249],
       [1.15529257, 0.99481263, 0.99148745],
       [1.13380024, 1.02276201, 0.9850304 ],
       [1.15625617, 0.97887042, 1.00646607],
       [1.11736316, 1.07740223, 0.94682726],
       [1.11917848, 1.03724746, 0.98516671],
       [1.08525878, 1.00703506, 1.04929881],
       [0.86998802, 1.09187891, 1.17972573],
       [1.1111553 , 0.96306326, 1.0673741 ],
       [1.06561012, 1.07653695, 0.99944559],
       [1.14401474, 1.02927889, 0.96829902],
       [0.92829685, 1.16773519, 1.04556062],
       [0.97857008, 1.09934302, 1.06367956],
       [1.03177542, 1.08094629, 1.02887094],
       [1.04930951, 1.01416764, 1.07811551],
       [0.98750188, 1.03343362, 1.12065715],
       [1.14538565, 1.04622804, 0.94997896],
       [1.00956193, 1.06751923, 1.06451149],
       [0.91590006, 1.08599314, 1.13969945],
       [0.9134278 , 1.03961044, 1.18855441],
       [1.0883976 , 1.00230048, 1.05089458],
       [1.05627096, 1.03160037, 1.05372132],
       [1.04461132, 1.06203684, 1.0349445 ],
       [0.96049986, 1.08190959, 1.0991832 ],
       [1.11325121, 1.08437135, 0.9439701 ],
       [0.83000165, 1.06519788, 1.24639312],
       [1.10418786, 0.95168653, 1.08571827],
       [0.98280577, 1.09595958, 1.0628273 ],
       [1.05753878, 1.00027308, 1.08378079],
       [1.08778615, 1.05273101, 1.00107549],
       [1.17483926, 1.05055989, 0.9161935 ],
       [0.93704819, 1.18397754, 1.02056692],
       [1.09930154, 1.25148007, 0.79081104],
       [0.99765765, 1.09424839, 1.04968662],
       [1.08903072, 1.08510869, 0.96745325],
       [1.08059795, 1.04180623, 1.01918847],
       [1.05431815, 1.01635908, 1.07091542],
       [1.06649989, 1.06917706, 1.00591571],
       [1.06479228, 1.02085299, 1.05594739],
       [0.97338513, 1.10195588, 1.06625164],
       [1.16865877, 1.00638729, 0.96654659],
       [0.98465255, 1.03367066, 1.12326945],
       [1.0624982 , 0.99278208, 1.08631238],
       [1.18086744, 0.76440235, 1.19632287],
       [1.04650143, 1.0487832 , 1.04630802],
       [1.62749847, 0.75454626, 0.75954792],
       [1.03050141, 1.03141739, 1.07967386],
       [1.05188247, 1.04959855, 1.04011163],
       [0.98928268, 1.08578105, 1.06652893],
       [1.0080975 , 1.08613447, 1.04736068],
       [1.09930539, 0.99317181, 1.04911546],
       [1.15996743, 0.80956492, 1.17206031],
       [0.7445181 , 0.91785505, 1.47921951],
       [1.08570264, 1.00986268, 1.04602733],
       [1.0560954 , 1.04384473, 1.04165253],
       [0.97039689, 1.1041404 , 1.06705537],
       [1.05646726, 1.00582623, 1.07929916],
       [1.04852846, 1.03093194, 1.06213225],
       [1.04391389, 1.04361917, 1.0540596 ],
       [1.06208975, 1.00559782, 1.07390508],
       [1.0693131 , 0.97195165, 1.10032791],
       [1.09633382, 1.0497878 , 0.99547103],
       [1.02325094, 0.99838266, 1.11995905],
       [1.10860334, 1.02432441, 1.00866491],
       [1.02641459, 1.0502416 , 1.06493646],
       [0.68484637, 1.29543032, 1.16131597],
       [1.01760122, 1.06550553, 1.0584859 ],
       [1.14869694, 1.02109126, 0.97180445],
       [0.84743004, 1.73758317, 0.55657945],
       [1.02153499, 0.96860109, 1.15145658],
       [1.14007641, 1.01885752, 0.98265873],
       [0.98129854, 1.00507542, 1.1552187 ],
       [1.03461825, 1.05205157, 1.05492284],
       [1.04641489, 1.06298268, 1.03219508],
       [1.0277536 , 1.00850161, 1.10533744],
       [0.9855707 , 1.09152614, 1.06449582],
       [1.07411158, 1.0069415 , 1.06053958],
       [1.0215193 , 1.03532308, 1.08475027],
       [1.08543896, 1.00284891, 1.05330479],
       [1.06982561, 1.03043605, 1.04133099],
       [1.09359261, 0.98739621, 1.06060383],
       [1.00429194, 1.06634606, 1.07095466],
       [1.05648006, 1.00090548, 1.08420712],
       [1.08999814, 1.05188153, 0.99971299],
       [1.03892625, 1.0925612 , 1.0101052 ],
       [1.08028363, 1.0579222 , 1.00338682],
       [1.05661033, 1.02544905, 1.05953327],
       [1.08003862, 1.00132965, 1.06022438],
       [1.03033826, 0.94743337, 1.16382103],
       [1.12458087, 1.03615849, 0.98085329],
       [1.16986337, 0.94661844, 1.02511084],
       [0.96924801, 1.14085662, 1.03148802],
       [1.02930216, 1.06877503, 1.04351547],
       [1.03227431, 1.03568591, 1.07363244],
       [1.08207672, 1.18266007, 0.87685586],
       [0.98931674, 1.12482085, 1.02745507],
       [1.07206858, 0.98176423, 1.08775985],
       [1.03104089, 1.02930808, 1.08124368],
       [1.0143071 , 1.13976373, 0.98752183],
       [1.1514577 , 1.02147481, 0.96866014],
       [1.09930593, 0.90527624, 1.13701048],
       [1.11474965, 0.99836429, 1.02847871],
       [0.94483399, 1.12551092, 1.07124774],
       [1.1953663 , 0.91317483, 1.03305152],
       [0.89463829, 1.23233208, 1.01462228],
       [1.11003762, 1.12990392, 0.90165112],
       [1.08542648, 1.10119539, 0.95497079],
       [0.88940068, 1.04549535, 1.20669662],
       [1.04006792, 1.07269992, 1.02882481],
       [1.06946901, 0.96246881, 1.10965483],
       [0.94571619, 1.16464096, 1.0312355 ],
       [1.10231473, 1.07871218, 0.96056574],
       [1.03657534, 0.94875558, 1.15626173],
       [1.12410706, 1.05631897, 0.96116662],
       [1.00366743, 0.92785869, 1.21006653],
       [0.99515679, 1.08299283, 1.06344304],
       [1.03414467, 1.1484158 , 0.95903218],
       [1.08965162, 1.01774855, 1.03419249],
       [1.11486707, 1.09327054, 0.93345504],
       [1.05893286, 0.9646806 , 1.1179792 ],
       [1.05272053, 0.89004143, 1.1988307 ],
       [1.15340873, 1.05961319, 0.92857073],
       [1.00297279, 1.11482102, 1.02379884],
       [0.97276369, 1.14144662, 1.02738234],
       [1.03838736, 1.0355701 , 1.06763519],
       [1.05343101, 1.0560633 , 1.03209834],
       [1.06379723, 1.02343797, 1.05435745],
       [0.98265039, 1.06170715, 1.09723512],
       [1.0867925 , 0.94260972, 1.11219044],
       [0.98194345, 1.09291616, 1.06673304],
       [1.06984296, 1.04388216, 1.02786753],
       [1.08853849, 1.00027358, 1.05278058],
       [1.25607929, 0.93714139, 0.94837198],
       [1.09139158, 0.96994836, 1.08025272],
       [1.52153025, 0.80298807, 0.81707433],
       [1.11950929, 1.04522022, 0.97686314],
       [1.08639346, 1.07384772, 0.98135147],
       [1.08908207, 1.05883249, 0.99367809],
       [1.0391295 , 0.99955262, 1.10291053],
       [1.00287299, 1.15232727, 0.98639239],
       [0.9407438 , 1.14715041, 1.05369844],
       [1.04505032, 1.09698984, 0.99955249],
       [1.05551437, 1.07760741, 1.00847088],
       [1.06465945, 1.00049642, 1.07643679],
       [1.10112345, 1.06449701, 0.97597219],
       [1.10022229, 1.05264883, 0.98872154],
       [1.01813976, 1.08002609, 1.0434268 ],
       [1.04352985, 1.05780852, 1.04025428],
       [1.04505458, 1.05052005, 1.04601802],
       [1.02276235, 1.07434344, 1.04448686],
       [1.00466814, 1.09963317, 1.03729134],
       [1.0522482 , 0.98726137, 1.10208308],
       [1.01903371, 1.07249167, 1.05006728],
       [1.26300638, 0.96142068, 0.9171656 ],
       [1.0757028 , 0.93576114, 1.13012871],
       [0.96716768, 1.1508034 , 1.02362157],
       [0.87994424, 0.90726259, 1.35438582],
       [1.13123829, 1.13460363, 0.87575074],
       [1.06291477, 1.0006835 , 1.07799438],
       [1.01270735, 0.98648909, 1.14239622],
       [1.0686534 , 1.05686907, 1.01607019],
       [0.99799783, 1.03234628, 1.11124855],
       [1.02174899, 1.07219754, 1.04764612],
       [1.02203628, 1.03648812, 1.08306825],
       [1.06973573, 1.06893949, 1.00291743],
       [1.04929483, 1.08865555, 1.00364227],
       [1.03013513, 1.08592473, 1.02553279],
       [1.04480482, 1.03812757, 1.05866026],
       [1.04354514, 1.08975952, 1.00828799],
       [1.04479506, 1.046392  , 1.0504056 ],
       [0.98455614, 1.05122861, 1.10580791],
       [1.05841307, 1.05162722, 1.03155237],
       [1.00047263, 1.11941294, 1.02170708],
       [1.09573915, 1.04568197, 1.00017153],
       [1.11260436, 0.99896162, 1.03002668],
       [1.06217596, 1.01585749, 1.06355921],
       [1.05303435, 1.03018038, 1.05837792],
       [1.02732388, 1.04224164, 1.07202713],
       [0.99238604, 1.01747565, 1.13173096],
       [0.9521779 , 1.13449297, 1.05492179],
       [1.05700267, 0.9843332 , 1.10025678],
       [1.04955543, 1.05235672, 1.0396805 ],
       [1.08346765, 1.05180721, 1.00631779],
       [1.08751066, 1.01568091, 1.03840109],
       [0.82166586, 1.14517595, 1.17475085],
       [1.08361642, 1.04458085, 1.01339538],
       [1.03281514, 1.06101765, 1.04775986],
       [1.05012014, 1.06195379, 1.02951873],
       [1.03160676, 1.01486378, 1.09512211],
       [0.76918662, 1.12880204, 1.24360399],
       [1.15144722, 1.09527015, 0.89487528],
       [1.11178095, 1.04082464, 0.98898707],
       [1.17422941, 1.00501671, 0.96234654],
       [1.1110581 , 1.21438514, 0.81614941],
       [0.86814515, 1.01200728, 1.26144023],
       [1.06597213, 1.0615158 , 1.01410472],
       [0.90387604, 1.18893031, 1.0487863 ],
       [0.89333355, 1.17247603, 1.07578307],
       [1.05803753, 1.0583103 , 1.02524482],
       [0.96175782, 1.09541528, 1.08441956],
       [1.07780346, 1.02673591, 1.03705329],
       [1.09783601, 1.09638637, 0.94737027],
       [1.31140381, 0.8628805 , 0.96730835],
       [0.93417297, 1.15144972, 1.05596997],
       [1.1551946 , 1.08928603, 0.89711202],
       [1.08931353, 1.15571189, 0.89656724],
       [1.06541122, 0.9633619 , 1.11281953],
       [1.05519611, 0.89320197, 1.19319457],
       [1.06725422, 1.0964555 , 0.97788293],
       [1.04907992, 1.01759449, 1.07491824],
       [0.90499488, 1.0930384 , 1.14355938],
       [0.96786229, 1.09022923, 1.08350113],
       [0.82407806, 1.23890852, 1.07860608],
       [1.10770609, 1.03258495, 1.00130162],
       [1.02601991, 1.03034141, 1.08523133],
       [1.03187577, 1.05921086, 1.05050602],
       [0.99870512, 1.07163891, 1.07124863],
       [1.07829596, 1.08940989, 0.9738868 ],
       [1.00949304, 1.05383891, 1.07826071],
       [0.93213993, 1.1160222 , 1.09343053],
       [1.03701258, 1.04526653, 1.05931354],
       [1.05472316, 1.06821968, 1.01864981],
       [1.08757631, 1.06171329, 0.99230305],
       [0.9639897 , 1.19102364, 0.98657931],
       [1.09203447, 1.20590347, 0.84365471],
       [1.06301981, 0.99401576, 1.08455708],
       [1.07231241, 1.04023994, 1.0290403 ],
       [1.05852305, 0.99326742, 1.08980218],
       [1.10094382, 1.03453196, 1.00611687],
       [1.11303923, 1.00447084, 1.02408259],
       [1.00419515, 1.13885976, 0.99853775],
       [1.03615703, 1.07253345, 1.03290217],
       [1.03983975, 1.02357081, 1.0781821 ],
       [1.03137014, 1.08897249, 1.02125003],
       [0.9841258 , 1.0906587 , 1.06680815],
       [0.99471897, 1.08235361, 1.06452008],
       [1.05152335, 1.01870405, 1.07136526],
       [1.04781594, 1.04418625, 1.04959047],
       [1.06120294, 1.02175078, 1.05863893],
       [1.07003284, 1.00524342, 1.0663164 ],
       [1.05975527, 0.99256749, 1.08926989],
       [1.08416776, 1.05881642, 0.99860847],
       [1.02411593, 1.05892373, 1.05855299],
       [1.21089228, 0.99429603, 0.93640434],
       [1.01788642, 0.99351975, 1.13018648],
       [1.01405428, 1.1041417 , 1.02339667],
       [0.93954694, 1.08072746, 1.12131825],
       [1.03712924, 1.04441918, 1.06004423],
       [1.02626405, 1.09541325, 1.01991535],
       [0.98590709, 1.10625448, 1.04943109],
       [1.03352565, 1.02066471, 1.08740229],
       [1.04431628, 1.08129576, 1.01598062],
       [1.11360415, 1.00023797, 1.02775054],
       [0.96915293, 1.07398837, 1.09845136],
       [1.00061522, 1.05094033, 1.0900371 ],
       [1.0209584 , 1.09739341, 1.02324085],
       [1.00954176, 1.09256816, 1.03948273],
       [1.02153353, 1.02910379, 1.09095533],
       [1.05871325, 1.06022312, 1.02265628],
       [1.11221433, 1.01626523, 1.0131131 ],
       [1.03252509, 1.03478215, 1.07428541],
       [1.09041272, 1.02364148, 1.02753845],
       [1.02611318, 1.0686171 , 1.04686237],
       [1.01637867, 1.05044396, 1.07477002],
       [1.05220403, 1.03337428, 1.05601435],
       [0.97100615, 1.17482068, 0.99576583],
       [0.96415245, 1.10724499, 1.07019521],
       [1.05873438, 1.11843954, 0.96441874],
       [1.0833163 , 1.00646676, 1.05180959],
       [0.99539758, 1.06981476, 1.07638031],
       [1.07179131, 1.01887198, 1.05092936],
       [1.04012204, 1.09382757, 1.00764303],
       [1.07939501, 1.04994662, 1.01225102],
       [1.09776189, 0.85592419, 1.18790657],
       [1.066648  , 1.45693646, 0.6180082 ],
       [0.92014609, 1.11524018, 1.10620638],
       [1.1421163 , 1.10316194, 0.89631441],
       [1.10585509, 1.02016176, 1.0155758 ],
       [0.69255965, 0.87192438, 1.57710862],
       [1.2675953 , 1.12302447, 0.75097289],
       [0.97403571, 1.22111006, 0.94644689],
       [1.30755844, 0.89480204, 0.93923217],
       [1.1169053 , 0.88719763, 1.13748972],
       [1.68260443, 0.66439205, 0.79459617],
       [1.04809193, 1.09199182, 1.00150891],
       [0.92391978, 0.98042169, 1.23725118],
       [1.04094139, 1.10615228, 0.99449899],
       [1.00309093, 1.08466101, 1.05384071],
       [0.92478894, 1.19850001, 1.0183037 ],
       [1.06445424, 0.94231269, 1.13482572],
       [1.02157112, 1.09325111, 1.02677043],
       [0.9559303 , 1.11842921, 1.06723313],
       [1.07574695, 1.12686394, 0.93898177],
       [1.31537837, 1.03759212, 0.78862216],
       [1.07391644, 0.89440701, 1.1732692 ],
       [0.96554207, 1.1442748 , 1.03177578],
       [1.02310598, 0.9665976 , 1.15188908],
       [1.03526725, 0.99165755, 1.11466785],
       [0.99342033, 1.22675568, 0.92141665],
       [0.97628668, 1.00566703, 1.15963895],
       [0.9225234 , 1.09857841, 1.12049085],
       [1.06565236, 1.1317244 , 0.94421589],
       [1.11885539, 0.88798771, 1.13474956],
       [1.11368647, 1.0962149 , 0.93169128],
       [1.01444507, 1.01214015, 1.11500743],
       [1.08236093, 1.01824945, 1.04098227],
       [1.04282851, 1.04791907, 1.05084507],
       [1.01776629, 1.0395277 , 1.08429866],
       [1.07552329, 0.99354072, 1.07252865],
       [1.0625096 , 1.11140657, 0.96767649],
       [1.00069445, 1.09264259, 1.04825561],
       [1.06036478, 1.0730611 , 1.00816676],
       [1.08283345, 1.08876284, 0.96999636],
       [1.1104732 , 1.05379397, 0.97732548],
       [1.00530613, 0.98435835, 1.15192817],
       [1.18126844, 1.02874623, 0.93157799],
       [1.07312154, 0.97270489, 1.09576623],
       [1.00289457, 1.09580522, 1.04289286],
       [1.04913918, 0.99303503, 1.09941844],
       [1.02376154, 1.00044605, 1.11738506],
       [1.02600889, 1.01118951, 1.10439425],
       [1.09044288, 1.04537749, 1.00577229],
       [1.03107625, 1.05917704, 1.05133936],
       [1.07944051, 1.03153095, 1.03062119],
       [1.0899289 , 1.02743153, 1.02423222],
       [1.04821931, 0.99860271, 1.09477064],
       [1.08993062, 1.04171134, 1.0099507 ],
       [1.03581492, 1.07167045, 1.03410728],
       [1.05586948, 1.0530499 , 1.03267327],
       [1.03033597, 1.04098616, 1.07027053],
       [1.06426512, 1.05950989, 1.01781765],
       [1.08122027, 1.08279001, 0.97758237],
       [0.99037858, 1.08056197, 1.0706521 ],
       [0.99714622, 1.17648525, 0.96796118],
       [1.00381751, 1.10559245, 1.03218269],
       [1.10094462, 0.97949871, 1.06114932],
       [1.02107012, 1.10522445, 1.01529808],
       [1.09743608, 0.97691235, 1.06724422],
       [1.04767412, 1.04174496, 1.05217357],
       [1.07993555, 1.04997767, 1.01167943],
       [1.04648621, 1.06365624, 1.0314502 ],
       [0.99765513, 1.09542861, 1.04850891],
       [1.06858315, 0.99863626, 1.07437325],
       [1.06847799, 1.09062634, 0.98248832],
       [1.09112038, 0.98739346, 1.06307882],
       [1.06025474, 1.09592835, 0.98540956],
       [0.99504185, 1.05082229, 1.09572851],
       [1.02279448, 1.02948674, 1.08931143],
       [1.04117429, 1.03130054, 1.06911782],
       [1.01690169, 1.08120374, 1.04348722],
       [1.06726026, 0.99956314, 1.07476926],
       [1.10541154, 0.99265406, 1.04352705],
       [1.07115784, 1.02061646, 1.04981835],
       [1.01424803, 1.02546466, 1.10187996],
       [1.12060305, 0.98027481, 1.0407148 ],
       [1.03442019, 1.07695126, 1.03022121],
       [1.03029498, 1.0724275 , 1.03887017],
       [0.99088207, 1.05268148, 1.0980291 ],
       [1.07316259, 1.09890934, 0.96952072],
       [1.07045476, 1.06533096, 1.00580693],
       [1.07233489, 0.95287076, 1.116387  ],
       [0.92258764, 1.13235688, 1.08664813],
       [1.08890769, 0.97603243, 1.07665253],
       [0.97905736, 1.18740013, 0.97513516],
       [1.06160383, 0.9945699 , 1.08541892],
       [1.08309962, 1.18515734, 0.87333569],
       [1.14303099, 0.6194608 , 1.37910087],
       [1.6846636 , 0.90810118, 0.54882787],
       [0.84751306, 0.88168981, 1.41238978],
       [1.04808677, 1.01470418, 1.0788017 ],
       [1.13699993, 1.12895958, 0.87563315],
       [1.56445768, 0.87939355, 0.69774143],
       [0.98732271, 1.0162394 , 1.13803054],
       [1.0250117 , 1.07945017, 1.03713079],
       [1.15077548, 0.98155761, 1.00925957],
       [1.16734987, 1.03829752, 0.93594526],
       [1.1764755 , 1.00039324, 0.96472391],
       [0.89718204, 1.26482293, 0.97958768],
       [0.91808893, 1.05324753, 1.17025619],
       [1.05330549, 1.00032737, 1.08795979],
       [0.99041945, 1.00175719, 1.14941602],
       [0.95212372, 1.11311537, 1.07635356],
       [1.19118171, 0.92963534, 1.0207756 ],
       [1.07241823, 0.94383938, 1.12533504],
       [0.96046726, 0.88136743, 1.29975796],
       [0.89813969, 1.10225395, 1.14119902],
       [0.91611369, 1.1318523 , 1.09362667],
       [1.08643827, 1.07567685, 0.97947754],
       [1.15720933, 1.05032613, 0.9340572 ],
       [1.10999359, 1.05066582, 0.98093324],
       [0.98509139, 1.14389029, 1.01261097],
       [0.90021702, 1.25180421, 0.98957143],
       [1.15696965, 1.02986269, 0.95476032],
       [1.05990435, 1.04664797, 1.03504033],
       [0.91430461, 1.00732945, 1.21995859],
       [1.02470599, 1.05980514, 1.05708152],
       [0.87202616, 1.03045349, 1.239113  ],
       [1.03767449, 1.08043413, 1.02348403],
       [1.06211224, 1.02723341, 1.052247  ],
       [1.0537597 , 1.03625059, 1.05158236],
       [1.10698014, 0.94737126, 1.08724125],
       [0.92086292, 1.1187942 , 1.10193553],
       [1.05495782, 0.98649456, 1.10014028],
       [1.00237803, 0.99129859, 1.14791603],
       [1.02746576, 1.06634211, 1.04778478],
       [0.98960419, 1.08679158, 1.06519688],
       [0.96199197, 1.13884297, 1.04075772],
       [1.09026729, 1.02531142, 1.02601394],
       [0.97430804, 1.0641006 , 1.10318401],
       [1.02548106, 1.0845685 , 1.03154309],
       [1.02275733, 1.03157914, 1.08725619],
       [0.99693753, 1.09919349, 1.04546163],
       [1.00821587, 1.1089918 , 1.02438498],
       [1.09848392, 1.02697668, 1.01613206],
       [1.09987946, 1.01243354, 1.02927965],
       [1.07583593, 1.03192464, 1.03383209],
       [1.03080713, 1.08363287, 1.02715265],
       [1.09899473, 0.98906384, 1.05353409],
       [1.08418984, 1.05340166, 1.00400115],
       [1.01748707, 1.07555758, 1.048548  ],
       [1.05917151, 1.04462346, 1.03779768],
       [1.04561111, 1.05909008, 1.03689146],
       [1.08435671, 1.00794315, 1.04929279],
       [1.0782323 , 0.9905286 , 1.07283174],
       [0.99138818, 1.06214636, 1.08805811],
       [1.07194111, 1.01985159, 1.04979995],
       [1.02052763, 1.03116537, 1.08989966],
       [1.06874201, 1.01067921, 1.06217144],
       [1.058484  , 1.03287005, 1.0502386 ],
       [1.02366823, 1.10158101, 1.01634342],
       [1.04410346, 1.01858693, 1.07890226],
       [1.06317282, 1.02117245, 1.05724738],
       [1.03260932, 1.01206315, 1.09692018],
       [1.0772991 , 1.07656827, 0.98772528],
       [1.00520152, 1.05871951, 1.07767163],
       [0.99230098, 1.04705652, 1.10223515],
       [0.96296206, 1.07413413, 1.10449646],
       [1.05576037, 1.12022545, 0.96560683],
       [1.03558731, 1.10260014, 1.00340521],
       [1.02120869, 1.08270239, 1.03768157],
       [1.02705233, 1.02271672, 1.0918236 ],
       [1.03350469, 1.08634138, 1.02174658],
       [1.09617017, 1.03177851, 1.01364397],
       [0.98577278, 0.95913976, 1.19668012],
       [1.01956615, 1.08586647, 1.03616004],
       [1.02419624, 1.05812155, 1.05927486],
       [1.07765939, 1.06078889, 1.00314438],
       [1.02867621, 1.04896678, 1.06394966],
       [1.05093934, 1.04548006, 1.04517325],
       [1.31971836, 0.91755703, 0.90431727],
       [1.02064226, 0.94701303, 1.17393736],
       [1.02543895, 1.0834831 , 1.0326706 ],
       [1.11686181, 1.05532196, 0.96940888],
       [1.06757123, 0.98395985, 1.09006157],
       [1.02497785, 1.03124533, 1.08536948],
       [1.10240705, 1.00315538, 1.03603022],
       [1.02466786, 1.04106604, 1.07585875],
       [1.11875505, 1.08889957, 0.93393803],
       [0.85847336, 0.88540413, 1.39771516],
       [0.50971014, 1.66870749, 0.96317502],
       [0.77844881, 0.74022833, 1.62291552],
       [0.89289098, 1.15711302, 1.09158866],
       [1.021829  , 0.96663094, 1.15313272],
       [1.0500144 , 1.15745684, 0.93412141],
       [1.12711575, 0.96998233, 1.04449457],
       [0.8373617 , 1.41052045, 0.8937105 ],
       [1.15013109, 0.98884168, 1.00261988],
       [1.00729656, 1.08807686, 1.04621923],
       [1.02093661, 1.0429235 , 1.07773255],
       [0.59848869, 1.86021324, 0.68289072],
       [1.05992823, 0.94411593, 1.13754849],
       [1.04863321, 1.10342928, 0.98953017],
       [1.09229062, 0.9933718 , 1.05593024],
       [0.95327834, 1.15472522, 1.0335891 ],
       [1.1184389 , 1.07953815, 0.9436156 ],
       [1.08778092, 0.90620607, 1.14760566],
       [0.96500628, 1.1312241 , 1.04536227],
       [1.23906287, 0.82224555, 1.08028423],
       [0.98433574, 1.19328308, 0.96397384],
       [0.92105831, 1.24301264, 0.9775217 ],
       [1.07253266, 0.86140938, 1.20765062],
       [1.14137936, 0.92376005, 1.07645325],
       [1.0608642 , 1.04482533, 1.03590313],
       [1.00294226, 0.99134098, 1.14730942],
       [1.02014693, 1.12237536, 0.99907036],
       [1.09635396, 1.00831045, 1.03692825],
       [0.97120198, 0.97310923, 1.19728145],
       [0.92247856, 1.1239018 , 1.09521229],
       [1.00014254, 1.23042903, 0.91102108],
       [0.96353055, 1.24434546, 0.93371664],
       [1.12385996, 0.98202882, 1.03570387],
       [1.00835576, 1.08310741, 1.05012948],
       [1.02410943, 1.03672951, 1.08075372],
       [1.03813113, 1.06674722, 1.0367143 ],
       [0.88295156, 1.35249563, 0.90614547],
       [1.17527531, 0.91988294, 1.0464344 ],
       [0.9527408 , 1.09312263, 1.09572922],
       [0.92761724, 1.15738022, 1.0565952 ],
       [1.0422118 , 1.00686878, 1.09251208],
       [1.05632927, 0.99466608, 1.0905973 ],
       [1.04986444, 0.99824755, 1.09348066],
       [1.0199949 , 1.0324577 , 1.08914005],
       [1.08901154, 0.96798332, 1.08459779],
       [1.1244619 , 1.00853091, 1.00859984],
       [1.08483558, 1.0305331 , 1.02622398],
       [1.0362584 , 1.00301211, 1.10232214],
       [1.02278222, 1.02338698, 1.09542346],
       [1.04205816, 1.02917589, 1.0703586 ],
       [1.02459917, 1.10081702, 1.01617647],
       [1.04927852, 1.04009005, 1.05222408],
       [1.0642216 , 1.0380008 , 1.03937026],
       [1.00617642, 1.06317775, 1.07223848],
       [1.09090868, 1.03149497, 1.01918901],
       [1.0260562 , 1.03189752, 1.08363894],
       [1.06528346, 1.02587817, 1.05043102],
       [1.07576054, 1.03827574, 1.02755637],
       [1.05135028, 1.07386591, 1.01637647],
       [1.09942611, 0.9664615 , 1.07570505],
       [1.07006953, 1.06237419, 1.00914894],
       [1.07881326, 1.03500075, 1.02777864],
       [1.04541591, 1.01771377, 1.07846297],
       [0.94489081, 1.08633155, 1.11037028],
       [1.07658455, 1.0094522 , 1.05555591],
       [1.06607401, 1.03445119, 1.04106745],
       [1.06727464, 1.07991926, 0.99439875],
       [1.07197011, 1.04575565, 1.02386689],
       [1.08951113, 0.97825371, 1.07382781],
       [0.99165552, 1.0838894 , 1.06604774],
       [1.07737058, 1.06874562, 0.99547646],
       [1.05903749, 1.10917829, 0.97337687],
       [0.95832484, 1.08198825, 1.10127956],
       [0.95508707, 1.07260681, 1.11389877],
       [1.083633  , 1.0112074 , 1.04675225],
       [1.0681357 , 0.97801206, 1.09544489],
       [0.97743594, 1.09806457, 1.06609214],
       [1.07218103, 1.05193757, 1.01747405],
       [1.09449229, 0.99714929, 1.04995108],
       [1.03682457, 1.22197671, 0.88279138],
       [1.08063119, 1.03857211, 1.02238936],
       [1.08705233, 0.9953669 , 1.05917342],
       [1.04780182, 1.08945493, 1.00433591],
       [1.07091465, 1.01723141, 1.0534466 ],
       [1.00717922, 1.08614326, 1.04827018],
       [0.73627706, 0.67991585, 1.72539975],
       [1.20663703, 1.0213024 , 0.91365322],
       [0.87461162, 1.06329344, 1.20368759],
       [1.15717947, 1.03423847, 0.95017471],
       [1.12098972, 0.92679406, 1.09380888],
       [0.90233564, 1.07558305, 1.16367397],
       [1.01216966, 1.14899671, 0.98042628],
       [1.01441362, 1.09405141, 1.03312762],
       [0.92442086, 0.92259272, 1.29457907],
       [0.92971339, 1.05150103, 1.16037824],
       [0.9521783 , 1.04142399, 1.14799037],
       [1.39271507, 0.885155  , 0.86372258],
       [1.16349512, 1.20178715, 0.77631038],
       [1.07902974, 0.98517143, 1.07739148],
       [1.09289527, 1.03031122, 1.01838617],
       [1.06013511, 1.09851862, 0.98293892],
       [1.09876968, 1.03239695, 1.01042603],
       [1.122587  , 1.02051868, 0.99848697],
       [1.28153439, 0.6980764 , 1.16198186],
       [0.96457381, 1.14494825, 1.0320706 ],
       [1.06128905, 1.05945315, 1.02085046],
       [1.03886425, 1.11367917, 0.98904924],
       [0.88622393, 1.10809661, 1.14727212],
       [1.09655976, 0.9011672 , 1.1438657 ],
       [1.06752534, 1.1392843 , 0.93478302],
       [1.09315707, 1.08086486, 0.96757072],
       [1.11235176, 1.02058332, 1.00865757],
       [0.9598034 , 1.11661552, 1.06517374],
       [1.05763234, 1.07544738, 1.00851293],
       [1.02111327, 0.93444702, 1.18603237],
       [0.95613173, 1.08599099, 1.09946993],
       [0.9845916 , 1.0067124 , 1.15028865],
       [1.01005369, 1.11191502, 1.01962395],
       [1.14904551, 0.97469787, 1.01784928],
       [1.27625438, 0.90651389, 0.95882438],
       [1.17531229, 0.98567299, 0.98060738],
       [1.00809335, 1.12833664, 1.00516267],
       [1.11264824, 1.01520733, 1.01373708],
       [1.03632199, 1.00251134, 1.10275933],
       [1.02489977, 1.16331274, 0.95338014],
       [0.94517593, 1.12482041, 1.07159631],
       [1.07474014, 1.12724106, 0.93961146],
       [0.99046092, 1.08322685, 1.06790489],
       [0.9898882 , 1.05975239, 1.09195207],
       [1.03189089, 1.02784972, 1.08185204],
       [1.05521961, 1.07139101, 1.01498203],
       [1.1480237 , 0.90967976, 1.08388919],
       [1.41689336, 0.69116517, 1.03353413],
       [1.24380972, 1.18862206, 0.70916088],
       [0.99039552, 0.97277575, 1.17842139],
       [1.11898366, 1.0605726 , 0.96203639],
       [1.07401981, 1.03550678, 1.03206606],
       [1.0821611 , 1.06249514, 0.99693642],
       [1.11449966, 1.05424361, 0.97284938],
       [1.01079226, 1.09555818, 1.03524221],
       [1.09389412, 1.00526232, 1.04243622],
       [1.03945639, 1.00437661, 1.09775965],
       [1.07861618, 1.01722886, 1.04574761],
       [1.1109731 , 1.02507217, 1.00554739],
       [1.02742318, 1.08675174, 1.02741773],
       [1.04487077, 1.07141976, 1.02530213],
       [1.01639184, 1.04861076, 1.07659006],
       [1.05398963, 1.06174418, 1.02585884],
       [1.0484957 , 1.072396  , 1.02070095],
       [1.07263447, 1.0584656 , 1.01049259],
       [1.03685877, 1.02538296, 1.07935092],
       [1.04082848, 1.06539476, 1.03536942],
       [1.06304028, 1.04346158, 1.03509079],
       [1.00712957, 1.04277983, 1.09168325],
       [0.98943822, 1.0572239 , 1.09493054],
       [1.09581182, 0.99737846, 1.04840237],
       [0.99693055, 1.11039607, 1.03426604],
       [1.09299119, 1.03763072, 1.01097074],
       [1.04428881, 1.03389393, 1.06340991],
       [0.95653534, 1.18062251, 1.0044348 ],
       [0.98602987, 0.9354795 , 1.22008328],
       [1.00629126, 1.11866856, 1.01663284],
       [1.08693538, 1.01037285, 1.04428443],
       [1.05354733, 1.05646146, 1.03158387],
       [1.06552288, 1.01753592, 1.05853386],
       [1.04256471, 1.07090587, 1.02812207],
       [0.9917657 , 1.0780501 , 1.07177686],
       [1.10913051, 1.06321545, 0.96924669],
       [1.08678522, 1.08048538, 0.97432205],
       [1.11318692, 0.98721638, 1.04118935],
       [1.01476419, 1.02877959, 1.09804887],
       [1.30529597, 0.84281779, 0.99347889],
       [0.93647082, 1.28426405, 0.92085779],
       [1.03789695, 1.02824471, 1.07545099],
       [1.03662491, 1.06441508, 1.04055267],
       [1.04431954, 1.0755072 , 1.02176591],
       [1.07502621, 1.05070777, 1.01585868],
       [1.08965519, 1.03465512, 1.01728234],
       [1.0593933 , 1.00595352, 1.07624584],
       [1.07256105, 1.06077469, 1.00825691],
       [1.04268183, 1.00264732, 1.0962635 ],
       [1.19753822, 1.04426846, 0.89978598],
       [0.99061486, 1.09641739, 1.0545604 ],
       [1.06517963, 1.01401055, 1.06240248],
       [0.87342966, 1.19647875, 1.07168425],
       [1.11835176, 1.23588187, 0.78735902],
       [0.69208995, 1.190808  , 1.2586947 ],
       [1.1460202 , 0.80352806, 1.19204439],
       [0.97031407, 1.08828978, 1.0829888 ],
       [1.2431541 , 0.9520578 , 0.94638076],
       [0.95524087, 1.06628647, 1.12006532],
       [1.14133811, 0.98850553, 1.01174901],
       [1.13061366, 0.87484494, 1.13613406],
       [1.0541018 , 0.98301781, 1.10447304],
       [1.05282308, 1.03637978, 1.05238978],
       [1.04935401, 0.98420438, 1.10803426],
       [1.01146827, 0.98715001, 1.14297438],
       [1.01707044, 1.04201783, 1.08250438],
       [1.09769018, 0.97434083, 1.06956164],
       [1.01701522, 1.07404453, 1.0505329 ],
       [0.94926976, 1.20865132, 0.98367157],
       [1.10749872, 1.05771462, 0.97637932],
       [1.04039017, 0.98122643, 1.11997605],
       [1.05844505, 1.00421428, 1.07893333],
       [1.00044918, 1.07779916, 1.06334431],
       [1.09752857, 1.01969163, 1.02437245],
       [0.93179869, 1.06259089, 1.14720308],
       [1.1278743 , 1.0826262 , 0.93109215],
       [0.96776071, 1.10986599, 1.06396596],
       [1.12105424, 1.08492204, 0.93561637],
       [1.03466243, 1.00765047, 1.09927976],
       [1.14991745, 1.00837902, 0.98329619],
       [1.05987492, 1.11033051, 0.97138722],
       [1.06863799, 1.1290834 , 0.94387126],
       [1.03999546, 1.10846551, 0.99313168],
       [1.06063322, 0.99087856, 1.09008088],
       [1.06294651, 1.03602961, 1.04261654],
       [0.88830519, 1.04323312, 1.21005434],
       [1.0969488 , 0.80682078, 1.23782307],
       [1.02557701, 0.97944775, 1.13656789],
       [1.14426732, 0.97719475, 1.02013058],
       [1.09486762, 0.96421752, 1.08250751],
       [1.06481538, 0.95495692, 1.12182035],
       [1.06594825, 0.95382172, 1.12182269],
       [1.11651618, 0.95730752, 1.06776895],
       [1.03964597, 0.97130133, 1.13064535],
       [1.0782846 , 1.04101153, 1.02229652],
       [1.05442739, 1.02208502, 1.06508025],
       [0.94176403, 1.20903997, 0.99078865],
       [0.69309864, 0.94377881, 1.50471521],
       [1.6368504 , 0.86233107, 0.64241118],
       [1.02085672, 1.09889401, 1.02184193],
       [1.1240321 , 1.0851157 , 0.93244485],
       [1.06270732, 1.01936107, 1.05952427],
       [1.02360573, 1.02111595, 1.09687097],
       [1.00147854, 1.10121401, 1.03890011],
       [1.01600947, 1.08344411, 1.04213907],
       [0.99849698, 1.06302438, 1.08007129],
       [1.05478526, 1.01152702, 1.07528038],
       [1.03791049, 1.07054557, 1.03313659],
       [1.00775879, 1.03798972, 1.09584414],
       [1.05260428, 1.02328771, 1.06570066],
       [1.03164658, 1.04282102, 1.06712506],
       [1.06714884, 1.03759392, 1.0368499 ],
       [1.03339843, 1.04457384, 1.06362039],
       [1.0562949 , 1.02711721, 1.05818055],
       [1.06109181, 1.02366488, 1.05683597],
       [1.03733917, 1.08615921, 1.01809428],
       [1.01737658, 1.05479106, 1.06942502],
       [1.01702433, 1.07781332, 1.046755  ],
       [1.03620448, 1.09311968, 1.01226849],
       [1.07257988, 1.09817502, 0.97083776],
       [1.05943156, 1.05889945, 1.02326165],
       [1.02853823, 0.9996026 , 1.11345182],
       [1.05076678, 1.04486663, 1.04595924],
       [1.24213864, 0.69586847, 1.20358555],
       [0.96290498, 1.09810552, 1.08058215],
       [1.05153057, 1.01311887, 1.07694321],
       [1.04610629, 1.05160909, 1.04387727],
       [1.0522226 , 1.06001331, 1.02935675],
       [1.03935356, 1.07350989, 1.0287292 ],
       [1.00800029, 1.09420231, 1.03939005],
       [0.98089926, 1.15168758, 1.00900582],
       [0.94962445, 1.08074197, 1.11122624],
       [1.06329185, 1.00043677, 1.07786404],
       [1.112114  , 0.94745779, 1.08202086],
       [1.09724339, 1.06983861, 0.97451066],
       [1.18925095, 1.06937813, 0.88296357],
       [0.8140706 , 0.6615813 , 1.66594075],
       [1.11859641, 1.08310057, 0.93989568],
       [1.19572129, 1.14368064, 0.80219072],
       [1.01528979, 1.0526722 , 1.07363066],
       [1.10866488, 1.00312474, 1.02980304],
       [0.92668138, 1.04503938, 1.16987189],
       [1.04077074, 1.02563645, 1.07518546],
       [1.00628827, 1.05181699, 1.08348739],
       [1.00732481, 1.05868862, 1.07557923],
       [1.1065874 , 1.04882043, 0.98618482],
       [0.98722945, 1.10570602, 1.04865718],
       [0.95294631, 1.24019208, 0.94845426],
       [0.95221357, 1.1400256 , 1.04935348],
       [1.0130068 , 1.01612333, 1.11246252],
       [1.39776774, 0.98372866, 0.76009626],
       [1.26388375, 0.7862855 , 1.0914234 ],
       [1.61268524, 0.82318731, 0.7057201 ],
       [0.53966682, 1.84846065, 0.75346518],
       [1.0137844 , 1.14239826, 0.98540999],
       [1.06784197, 1.00754312, 1.06620756],
       [1.12913237, 1.08883581, 0.92362447],
       [1.12289245, 1.05316297, 0.96553723],
       [1.14297859, 0.98355461, 1.01505946],
       [1.0485756 , 1.01837477, 1.07464228],
       [0.97037223, 1.05442577, 1.11679465],
       [0.92656619, 1.14645671, 1.06856976],
       [1.14294898, 1.03374465, 0.96489901],
       [1.01963953, 1.08625108, 1.03570205],
       [1.00866062, 1.10212866, 1.03080338],
       [1.16181547, 0.96725166, 1.01252552],
       [1.01038214, 0.99635215, 1.13485836],
       [1.035867  , 0.98685335, 1.1188723 ],
       [0.95444588, 1.09053548, 1.09661129],
       [1.08388384, 1.02365966, 1.03404916],
       [1.05877162, 0.96602506, 1.11679597],
       [1.0568531 , 1.04568208, 1.03905747],
       [1.26118823, 0.96752528, 0.91287914],
       [1.07443508, 0.98069077, 1.08646681],
       [1.09696656, 0.96280395, 1.08182214],
       [1.10027463, 0.99315011, 1.04816791],
       [1.01538678, 1.01142835, 1.11477753],
       [1.09591937, 0.97660221, 1.06907107],
       [0.95682484, 1.22721055, 0.95755727],
       [1.80606759, 0.6606973 , 0.67482777],
       [1.15691563, 1.0073432 , 0.97733382],
       [0.96757037, 1.17719461, 0.99682767],
       [1.12012431, 0.91591043, 1.10555791],
       [1.00279751, 1.09801778, 1.04077737],
       [1.07830295, 1.01704074, 1.04624897],
       [1.16938695, 0.99088332, 0.98132239],
       [1.08531889, 1.14238694, 0.91388683],
       [1.10732582, 0.98591724, 1.0483496 ],
       [1.06355859, 1.0594924 , 1.01854167],
       [0.95261272, 1.13157138, 1.05740855],
       [1.04879056, 1.02634674, 1.06645536],
       [1.08220985, 1.06542207, 0.99396073],
       [1.09302236, 1.03997684, 1.00859345],
       [1.07778667, 1.03177482, 1.03203116],
       [1.07984694, 1.03818368, 1.02356203],
       [1.02901815, 1.04887863, 1.06369587],
       [1.05937987, 1.02251419, 1.05969859],
       [1.0633865 , 1.06554332, 1.01266283],
       [1.0932573 , 1.04646728, 1.00186808],
       [1.01537064, 1.05718495, 1.06903707],
       [1.08440663, 1.05340257, 1.00378346],
       [1.07102879, 0.99481647, 1.0757474 ],
       [1.07566431, 1.04537805, 1.02055029],
       [1.03848418, 1.02819238, 1.0749161 ],
       [1.09997096, 1.00982434, 1.03179734],
       [0.98082272, 1.04542227, 1.11534766],
       [1.00224826, 1.04161168, 1.09773272],
       [1.02855129, 0.98087657, 1.13216479],
       [1.20381173, 0.95147467, 0.98630626],
       [1.03105757, 1.08228203, 1.02825305],
       [0.78119566, 1.86504549, 0.49535151],
       [1.04021926, 0.86889804, 1.23247535],
       [1.03835086, 0.96547977, 1.13776203],
       [1.04285509, 1.03408081, 1.06465676],
       [1.03775075, 1.05038701, 1.0534549 ],
       [1.05280957, 1.0112498 , 1.07753328],
       [1.10156431, 1.04372017, 0.99630818],
       [1.08999872, 1.03074524, 1.0208487 ],
       [1.07440054, 0.99642977, 1.07076234],
       [1.07960633, 1.05805556, 1.00393076],
       [1.00609626, 1.07444547, 1.06105092],
       [0.93307146, 1.16025463, 1.04826656],
       [1.07974494, 1.17872682, 0.88312089],
       [1.35772563, 1.03208317, 0.75178386],
       [1.04197031, 1.06318318, 1.03643916],
       [0.98204018, 1.0071734 , 1.15237907],
       [1.04950186, 1.05140489, 1.0406859 ],
       [1.01472951, 1.08159497, 1.04526818],
       [0.8142856 , 1.20631304, 1.12099401],
       [1.11062057, 1.03306669, 0.99790539],
       [1.01288471, 1.06657132, 1.06213662],
       [1.01164962, 1.07363912, 1.05630391],
       [0.99872736, 1.0759256 , 1.06693969],
       [1.05526345, 0.97661312, 1.10971608],
       [1.10894388, 1.04396104, 0.98868773],
       [1.01752736, 1.16283338, 0.96123191],
       [1.05309784, 1.02779611, 1.06069871],
       [1.0424805 , 1.0510253 , 1.04808685],
       [1.09005084, 0.99784977, 1.05369205],
       [1.01889799, 1.09265934, 1.03003532],
       [1.30192206, 0.79240584, 1.04726475],
       [0.99458   , 0.96911049, 1.17790216],
       [0.99511398, 1.11755991, 1.02891876],
       [0.99776928, 0.97200451, 1.17181886],
       [0.99527212, 1.04003132, 1.10628921],
       [0.96382551, 1.16989259, 1.00787455],
       [0.89742946, 1.13711009, 1.1070531 ],
       [1.15864315, 1.00431043, 0.97863907],
       [0.96123448, 1.05221883, 1.12813935],
       [0.85956303, 1.1144196 , 1.16761002],
       [0.95967321, 1.18850987, 0.99340957],
       [1.02201141, 1.04833176, 1.07124948],
       [0.96625107, 0.97675626, 1.19858532],
       [0.98105886, 1.16108013, 0.99945366],
       [0.93005934, 1.16684342, 1.04468989],
       [1.11070095, 1.0138022 , 1.01708951],
       [0.98866274, 1.12777577, 1.02515414],
       [0.99423018, 0.99578481, 1.15157766],
       [0.99199762, 1.06643764, 1.0831574 ],
       [1.14233573, 0.92151468, 1.07774225],
       [1.04288308, 1.07331723, 1.02539234],
       [0.93904096, 1.04545339, 1.1570983 ],
       [0.97761922, 1.06967393, 1.09429951],
       [1.10730933, 1.00039815, 1.03388517],
       [0.97310259, 1.09087336, 1.0776167 ],
       [1.00535555, 1.05757158, 1.07866553],
       [1.05122287, 1.09271688, 0.9976529 ],
       [1.01293926, 1.10253215, 1.02612124],
       [1.17929822, 1.00131094, 0.96098349],
       [1.05068601, 1.29879547, 0.79211118],
       [1.47605604, 0.81107976, 0.85445685],
       [1.1776941 , 1.00067622, 0.96322233],
       [1.0782597 , 0.88199025, 1.1813427 ],
       [1.23339371, 0.96809558, 0.94010337],
       [1.01371449, 1.06632634, 1.06155182],
       [1.06372108, 1.02938207, 1.0484895 ],
       [0.97361233, 0.92954604, 1.23843428],
       [0.93501757, 1.15182384, 1.05475124],
       [1.05790705, 1.05571308, 1.02797252],
       [0.93659647, 1.12517697, 1.07981921],
       [0.94987379, 0.99599951, 1.19571936],
       [1.05597104, 1.06386411, 1.0217575 ],
       [1.00693406, 1.07687183, 1.05778677],
       [1.0593857 , 1.00705679, 1.07515016],
       [1.02471793, 1.07353766, 1.04333706],
       [1.01955498, 1.06418011, 1.05785756],
       [1.04348251, 1.05175202, 1.04635812],
       [1.06786798, 1.06714043, 1.00658424],
       [1.03817816, 1.02590307, 1.07751143],
       [1.02589324, 1.01172631, 1.10397311],
       [1.02782364, 1.01280876, 1.10096025],
       [1.03713756, 1.06599783, 1.03845726],
       [1.08811108, 1.03633032, 1.01715125],
       [1.01928528, 1.08432301, 1.03798436],
       [1.03536002, 1.0337593 , 1.07247333],
       [1.08882516, 1.04390995, 1.00885754],
       [1.01650374, 1.10906157, 1.01602734],
       [1.05982066, 1.10805028, 0.97372171],
       [1.11267959, 1.01415866, 1.0147544 ],
       [1.05621167, 1.00507335, 1.08030764],
       [1.24771906, 1.11579998, 0.77807361],
       [0.89195476, 1.15070827, 1.09892962],
       [0.95482229, 1.11701034, 1.06976002],
       [1.06976613, 1.05872184, 1.01310469],
       [1.05106803, 1.05851594, 1.03200868],
       [1.06241974, 1.05072613, 1.02844678],
       [0.98783113, 1.00868051, 1.14508102],
       [1.0149474 , 1.13727545, 0.98936981],
       [1.08094864, 0.97857038, 1.08207363],
       [1.0752062 , 1.09364171, 0.97274474],
       [1.00200874, 1.06650869, 1.07307522],
       [0.88290737, 1.05740219, 1.2012831 ],
       [1.02594183, 1.14280044, 0.97285039],
       [1.05932981, 1.06756941, 1.01469343],
       [1.07162005, 1.01734125, 1.05263135],
       [1.06804182, 1.04989055, 1.02366028],
       [0.95888105, 1.15369736, 1.02901424],
       [1.01743391, 1.04724142, 1.07691732],
       [1.04052519, 1.09204197, 1.00902549],
       [1.06712657, 0.92802796, 1.14643813],
       [1.03558379, 1.00939033, 1.09661854],
       [1.05934129, 1.06561416, 1.01663721],
       [1.04493225, 1.09615061, 1.0005098 ],
       [1.06361164, 0.97644955, 1.10153147],
       [0.97182366, 1.12653189, 1.0432371 ],
       [0.98149625, 1.09393561, 1.0661608 ],
       [1.06226053, 0.94239713, 1.136935  ],
       [1.12722641, 0.98608059, 1.02828566],
       [1.07239199, 1.1087916 , 0.96040906],
       [1.02429034, 1.01523786, 1.10206445],
       [1.05665188, 1.09918753, 0.98575325],
       [0.96827816, 1.05260858, 1.12070591],
       [1.05676631, 1.07047269, 1.01435365],
       [1.01595461, 1.09455841, 1.03107964],
       [1.13441915, 1.01720988, 0.98996363],
       [1.07031879, 1.06228341, 1.00899045],
       [0.92811986, 1.10934262, 1.10413017],
       [1.05137343, 0.99844445, 1.09177478],
       [1.02530957, 1.01140696, 1.10487613],
       [0.98323594, 1.22502211, 0.9333346 ],
       [1.07974659, 1.20927208, 0.85257398],
       [0.8373263 , 1.15821761, 1.14604875],
       [1.01430602, 1.05525154, 1.07203509],
       [1.19665366, 0.93605848, 1.00888051],
       [1.03291288, 1.15017161, 0.95850817],
       [1.04201223, 1.07341896, 1.02616146],
       [1.06014107, 1.12935587, 0.95209571],
       [1.13625865, 1.03267816, 0.97265585],
       [1.05233446, 1.05060594, 1.03865226],
       [1.01443107, 1.06634661, 1.06081498],
       [1.02496195, 1.06931394, 1.04731676],
       [0.88482075, 1.47667997, 0.78009193],
       [0.97539968, 1.09916516, 1.06702782],
       [1.08171036, 1.06370998, 0.99617232],
       [0.97514165, 1.09820557, 1.06824543],
       [1.06841628, 1.08435393, 0.98882245],
       [1.1249367 , 1.05601813, 0.96063783],
       [0.95677809, 0.99236601, 1.19244855],
       [0.9853242 , 0.97949791, 1.17677055],
       [1.10589235, 0.82280708, 1.21289323],
       [1.01937828, 1.13860697, 0.98360741],
       [1.06891988, 1.0527669 , 1.01990588],
       [1.00703518, 1.09262175, 1.04193573],
       [1.05267756, 1.00893797, 1.07997713],
       [1.04069904, 0.99934152, 1.1015521 ],
       [1.09880668, 0.99666304, 1.04612294],
       [1.06347792, 1.03712409, 1.04099065],
       [1.03948392, 1.07347549, 1.02863324],
       [1.1193378 , 1.10392643, 0.91832842],
       [1.14445145, 0.97932794, 1.01781327],
       [1.0883318 , 1.00713035, 1.0461305 ],
       [1.06776236, 1.06609272, 1.00773758],
       [1.00805838, 1.08037225, 1.05316202],
       [1.03941454, 1.07514705, 1.02703107],
       [1.05829823, 1.02840343, 1.05489099],
       [1.03694575, 1.0682862 , 1.03636071],
       [1.08580413, 1.04957272, 1.0062158 ],
       [1.03519024, 0.99461378, 1.11178863],
       [0.99193018, 1.11314615, 1.03651632],
       [1.08767505, 1.04302288, 1.01089472],
       [0.99289547, 1.11015244, 1.03854474],
       [1.09682948, 1.0426864 , 1.00207677],
       [1.03751718, 1.07578928, 1.02828619],
       [1.01182221, 1.0229399 , 1.10683055],
       [0.98565563, 1.03421352, 1.1217235 ],
       [1.00512203, 1.03669831, 1.09977231],
       [1.02449685, 1.08039642, 1.03669938],
       [1.04384713, 0.98445965, 1.11328588],
       [0.94290067, 1.02345321, 1.17523877],
       [1.02499015, 0.93386744, 1.18273507],
       [1.22134749, 0.97415768, 0.94608748],
       [1.03343022, 1.06609378, 1.04206865],
       [1.05092041, 1.02322761, 1.06744463],
       [0.9944652 , 1.11008862, 1.03703883],
       [1.00599468, 1.02730606, 1.10829191],
       [1.05225227, 1.0184059 , 1.07093448],
       [1.06112671, 1.0607806 , 1.01968534],
       [1.03710316, 1.04529695, 1.05919254],
       [1.0857495 , 1.0127143 , 1.04312886],
       [1.08884926, 0.9812193 , 1.0715241 ],
       [0.8215174 , 0.9663844 , 1.35369085],
       [1.03454219, 1.13610541, 0.97094506],
       [1.01976926, 1.06217145, 1.05965194],
       [1.04538045, 1.00470642, 1.09150578],
       [1.10974388, 1.03747692, 0.99437186],
       [1.03425432, 1.02952842, 1.07780991],
       [1.02574116, 1.02233745, 1.09351405],
       [0.95396641, 1.11802215, 1.06960409],
       [1.0171338 , 1.10478654, 1.01967232],
       [1.03078714, 1.00875858, 1.10204693],
       [1.06067833, 1.07826811, 1.00264622],
       [1.08234137, 0.98557256, 1.07367872],
       [1.10080837, 1.06799017, 0.97279412],
       [0.97236975, 1.12335635, 1.04586655],
       [1.00489199, 1.09349876, 1.04320191],
       [0.93731295, 1.11347303, 1.09080667],
       [1.07785311, 1.06208821, 1.00165133],
       [1.08282136, 1.00327893, 1.05549236],
       [1.051363  , 1.06483869, 1.02539096],
       [1.09584263, 0.97579864, 1.06995138],
       [0.99601705, 1.10822056, 1.03735504],
       [1.10421354, 0.98578276, 1.05159635],
       [1.05198583, 1.04010051, 1.04950631],
       [1.09179074, 1.03667717, 1.01312474],
       [1.09482808, 1.06114745, 0.98561713],
       [1.04834227, 0.95284922, 1.14040117],
       [1.09386544, 1.01612029, 1.03160692],
       [1.0668267 , 1.11512466, 0.95964129],
       [1.17767617, 1.07188193, 0.89203455],
       [1.0279024 , 0.96981479, 1.14387546],
       [0.79849528, 0.78705289, 1.55604449],
       [1.17788133, 1.04564682, 0.91806451],
       [0.96293966, 1.02005122, 1.15860177],
       [0.99683504, 0.96362201, 1.1811356 ],
       [0.98448709, 1.08313999, 1.07396557],
       [1.35649937, 0.86037592, 0.92471736],
       [1.19055163, 0.9391645 , 1.01187652],
       [0.99909573, 1.04393157, 1.09856536],
       [1.03467347, 1.21279529, 0.89412389],
       [1.07468463, 0.9962824 , 1.07062562],
       [0.88125574, 1.26672672, 0.9936102 ],
       [1.05371729, 1.08462643, 1.00324893],
       [1.10278911, 1.05537782, 0.98342573],
       [1.07385754, 1.03600113, 1.03173398],
       [1.06961489, 0.97920311, 1.09277466],
       [1.07173028, 0.99566707, 1.0741953 ],
       [1.05628096, 1.09368547, 0.99162623],
       [1.02800617, 1.04350451, 1.07008198],
       [1.10159072, 1.02704024, 1.0129617 ],
       [1.25830367, 0.98993777, 0.89335121],
       [0.97372439, 1.00273767, 1.16513059],
       [1.12489556, 1.1323973 , 0.88429979],
       [1.08889947, 1.05188197, 1.00081121],
       [1.04849147, 1.08494917, 1.00815201],
       [1.0246824 , 1.08674677, 1.03016348],
       [0.83242085, 1.34836454, 0.96080727],
       [1.00799012, 1.07987338, 1.05372916],
       [1.10938059, 0.99146997, 1.0407421 ],
       [1.08843239, 0.95197804, 1.10118223],
       [1.10408069, 1.04974306, 0.98776891],
       [1.02945537, 1.0950572 , 1.01708009],
       [1.01395782, 1.06316667, 1.06446816],
       [1.00878592, 1.06152299, 1.07128374],
       [1.04839843, 1.05523898, 1.03795524],
       [1.06519956, 1.02111313, 1.05527997],
       [1.06153558, 1.04045321, 1.03960387],
       [1.01108   , 1.0973795 , 1.03313316],
       [1.11828744, 1.02226261, 1.0010426 ],
       [1.00103525, 1.11836423, 1.02219317],
       [1.11746762, 1.01603852, 1.00808652],
       [1.0286548 , 1.02863958, 1.08429827],
       [1.01797072, 1.01430022, 1.10932171],
       [1.0255343 , 1.04201109, 1.07404726],
       [1.00541108, 1.11266137, 1.02352021],
       [1.03822278, 1.10928043, 0.99408945],
       [0.97484956, 1.0722067 , 1.09453639],
       [1.10290055, 1.06457845, 0.97411365],
       [1.08092419, 1.05485615, 1.00581231],
       [0.93080885, 1.22630317, 0.98448064],
       [0.8313467 , 1.21575588, 1.09449007],
       [1.05624374, 1.01000597, 1.07534295],
       [1.04945807, 1.04248241, 1.04965218],
       [1.03053396, 1.09649566, 1.01456304],
       [1.1040622 , 0.9900946 , 1.04743585],
       [1.0147888 , 1.03696508, 1.08983877],
       [1.07014617, 1.02319272, 1.04825376],
       [0.99313945, 1.06832495, 1.08012826],
       [0.96539533, 1.00368217, 1.17251516],
       [1.08897837, 1.01029915, 1.04231513],
       [0.93372186, 1.21675464, 0.99111615],
       [0.80697189, 1.19923004, 1.13539072],
       [1.01184387, 1.11598777, 1.01376102],
       [1.06075863, 1.04485286, 1.03598116],
       [1.11273755, 1.01116077, 1.01769433],
       [1.05030096, 1.06657352, 1.02471818],
       [1.12405347, 0.9876769 , 1.02986229],
       [1.04695393, 1.07844045, 1.01619828],
       [1.0933801 , 1.02731747, 1.02089508],
       [0.92123238, 1.23574862, 0.98461166],
       [1.00939473, 0.99512584, 1.13707208],
       [1.10659648, 1.04630483, 0.98869134],
       [1.07076192, 1.01973685, 1.05109388],
       [0.95731789, 1.10004876, 1.084226  ],
       [1.13179717, 1.06132876, 0.94846673],
       [1.13237301, 1.03739037, 0.97182927],
       [1.06372885, 0.99949491, 1.07836889],
       [1.0045351 , 0.93438295, 1.20267461],
       [1.12608701, 0.92033126, 1.09517438],
       [0.93116349, 1.09611166, 1.1143175 ],
       [1.01925193, 1.03104584, 1.09129488],
       [1.00146391, 1.0756016 , 1.06452715],
       [1.00069183, 1.0760261 , 1.06487472],
       [1.01674158, 1.08135694, 1.04349413],
       [1.11535756, 1.00026564, 1.02596946],
       [0.97423175, 1.08568039, 1.08168052],
       [0.9097568 , 1.06541795, 1.1664179 ],
       [1.04625625, 1.0613508 , 1.0339856 ],
       [1.12231721, 1.1017748 , 0.91750064],
       [0.84031117, 1.05827804, 1.24300344],
       [1.06852348, 1.12087118, 0.95219799],
       [1.01847382, 1.20200972, 0.92110911],
       [0.96557002, 1.08257018, 1.09345245],
       [1.0698371 , 1.18844202, 0.88331354],
       [1.0015804 , 1.08217901, 1.05783325],
       [0.95618507, 1.12998406, 1.05542352],
       [0.90742146, 1.4134903 , 0.82068089],
       [1.00942625, 0.97310757, 1.15905884],
       [1.05869338, 1.07026053, 1.01263875],
       [1.01169823, 1.00356022, 1.12633421],
       [1.03433622, 1.07665035, 1.03060609],
       [1.04795623, 0.89421962, 1.19941681],
       [1.05632507, 1.00691204, 1.07835554],
       [0.98360589, 1.09894893, 1.05903783],
       [1.03221125, 1.06254218, 1.04683922],
       [1.0800678 , 1.06875975, 0.9927651 ],
       [1.08001351, 1.07864951, 0.98292963],
       [1.00606423, 1.13283409, 1.00269434],
       [1.10812984, 1.06675872, 0.9667041 ],
       [1.07619429, 1.04503586, 1.0203625 ],
       [1.10145459, 0.97032095, 1.06981712],
       [1.25352837, 0.90339516, 0.98466913],
       [1.14160537, 0.9848644 , 1.01512289],
       [0.99334374, 1.10745082, 1.04079809],
       [0.88034741, 1.38415208, 0.87709316],
       [1.07595273, 0.89422817, 1.17141175],
       [1.03713837, 1.11425323, 0.99020105],
       [1.00315264, 1.0777913 , 1.06064871],
       [1.01488198, 0.97117198, 1.15553869],
       [1.10913723, 1.04535265, 0.98710277],
       [1.06005191, 1.00743355, 1.0741072 ],
       [1.02526907, 1.06071524, 1.05560834],
       [1.05346275, 1.01915873, 1.06897118],
       [1.03849206, 1.01708403, 1.08601657],
       [1.04596719, 1.03602764, 1.05959782],
       [1.04561776, 1.06517948, 1.03079541],
       [1.03916064, 1.00570728, 1.09672474],
       [0.99732908, 1.1195435 , 1.02472007],
       [1.12034725, 1.02905653, 0.99218887],
       [1.02437589, 0.99667531, 1.12054145],
       [1.02542033, 1.02294546, 1.09322686],
       [1.03350948, 0.99536359, 1.11271958],
       [1.07806906, 1.02842656, 1.03509704],
       [1.02087993, 1.12259332, 0.99811941],
       [1.1123389 , 1.01359942, 1.01565433],
       [0.97988546, 1.01827853, 1.14342867],
       [0.94579856, 1.09855656, 1.09723754],
       [0.9138823 , 1.03118092, 1.19652943],
       [0.92809562, 1.30029963, 0.91319741],
       [0.69982368, 1.58851678, 0.85325219],
       [1.02444888, 1.06039189, 1.05675188],
       [1.01575006, 1.05250799, 1.07333461],
       [1.06596953, 1.01071454, 1.06490859],
       [1.08178208, 1.07763702, 0.98217355],
       [1.09954857, 1.0100138 , 1.03203028],
       [1.04375802, 1.0517287 , 1.04610593],
       [1.06526968, 1.03937682, 1.03694615],
       [1.12870949, 1.04893817, 0.96394499],
       [1.02395418, 1.01991475, 1.09772372],
       [1.00428767, 0.98749789, 1.1498071 ],
       [0.99823232, 1.00741805, 1.13594229],
       [0.99105477, 1.0346152 , 1.11592269],
       [1.02766725, 1.06208767, 1.05183772],
       [1.05734647, 1.0045188 , 1.07972738],
       [1.02608451, 1.07383412, 1.04167402],
       [1.04124954, 1.00537257, 1.09497054],
       [1.07797211, 1.02730057, 1.03631998],
       [1.02067981, 1.0863127 , 1.03460015],
       [1.18823487, 1.01239836, 0.94095942],
       [0.97077402, 0.97718294, 1.19363569],
       [1.13393067, 1.02104637, 0.98661562],
       [1.06425061, 0.99690036, 1.08044168],
       [1.01047749, 1.05760884, 1.07350633],
       [1.08887927, 0.96029392, 1.09241946],
       [1.09729661, 0.94566267, 1.09863337],
       [1.06045193, 0.97565998, 1.10548075],
       [1.07177657, 1.05570509, 1.014111  ],
       [1.16347224, 1.05688367, 0.92123674],
       [0.94151471, 1.09126039, 1.10881756],
       [1.11176085, 1.07299452, 0.95683729],
       [1.05245296, 1.06036248, 1.02877721],
       [1.0649011 , 0.91359416, 1.16309739],
       [1.14051604, 1.10427439, 0.89680221],
       [0.99102833, 1.03343962, 1.1171247 ],
       [1.05456247, 0.9836746 , 1.10335558],
       [1.01613428, 1.00127696, 1.12418141],
       [0.85246442, 1.02083144, 1.26829679],
       [1.38323226, 0.94004851, 0.81831188],
       [0.8536037 , 1.26616268, 1.02182627],
       [1.02952096, 1.03470592, 1.07736577],
       [0.92547777, 0.92133677, 1.29477811],
       [1.13972849, 0.93366782, 1.06819634],
       [1.03838157, 0.94773192, 1.15547917],
       [1.04930005, 0.82043475, 1.27185785],
       [1.18170682, 0.99908987, 0.96079596],
       [1.06921245, 0.98631226, 1.08606795],
       [1.04735047, 1.02110898, 1.0731332 ],
       [1.11527794, 1.0203769 , 1.00593781],
       [1.06452616, 1.06319236, 1.01387413],
       [1.06455523, 1.03515095, 1.04188647],
       [1.04610673, 0.99560967, 1.09987626],
       [1.06679703, 1.06258589, 1.01220973],
       [1.08108489, 0.98962548, 1.07088228],
       [1.08564758, 1.04343209, 1.01251299],
       [1.04917638, 1.04449957, 1.0479167 ],
       [0.9450614 , 1.10765977, 1.08887148],
       [1.04420015, 1.02036241, 1.0770301 ],
       [1.00165536, 1.00063019, 1.1393071 ],
       [1.07679578, 1.08610394, 0.97869294],
       [1.03038218, 1.12714279, 0.98406768],
       [1.25416226, 0.84752674, 1.03990366],
       [1.08956505, 0.95924049, 1.09278711],
       [1.03344516, 1.00046603, 1.10768147],
       [1.10182665, 0.95979867, 1.07996734],
       [0.93779514, 1.10450904, 1.09928848],
       [1.04987256, 1.04266998, 1.04905012],
       [1.01606709, 1.04026792, 1.08525765],
       [0.9962325 , 1.08768816, 1.05767199],
       [1.0537213 , 1.07508309, 1.01278826],
       [1.01918126, 1.07743001, 1.04498138],
       [1.03401973, 1.0633426 , 1.04423032],
       [1.05306751, 1.06874962, 1.01977552],
       [0.99974866, 1.12263431, 1.01920969],
       [1.10420659, 1.02218707, 1.01519899],
       [1.01603902, 0.99650899, 1.12904464],
       [1.00441127, 1.12480263, 1.01237875],
       [1.12690964, 1.00537734, 1.00930568],
       [1.08467443, 1.02517516, 1.03174306],
       [1.02288307, 1.04763613, 1.07107346],
       [1.00323672, 1.0188871 , 1.11946883],
       [1.26957143, 0.90645328, 0.96556795],
       [1.0750584 , 0.97044077, 1.09609348],
       [1.1197565 , 0.91090795, 1.11092821],
       [0.95922888, 1.13786227, 1.0445015 ],
       [1.0463351 , 1.26376024, 0.83149731],
       [0.88518649, 1.08330955, 1.17309661],
       [1.04792758, 1.0941594 , 0.99950568],
       [1.03889281, 1.01946867, 1.08323117],
       [0.98875237, 1.07562735, 1.07721294],
       [1.07560757, 1.06457786, 1.00140723],
       [1.06835057, 1.05542535, 1.01781673],
       [1.03324933, 1.04754564, 1.06079768],
       [1.06370171, 1.05416247, 1.02372847],
       [1.01828665, 1.02966205, 1.09364395],
       [1.03382014, 1.07078423, 1.03698828],
       [1.01504904, 1.11783546, 1.00870816],
       [1.16789931, 0.98958793, 0.98410541],
       [1.03480026, 1.12593673, 0.98085566],
       [1.00437232, 1.08429966, 1.05292067],
       [1.07098043, 1.0442753 , 1.02633693],
       [1.04525352, 1.03037557, 1.06596356],
       [1.1271406 , 1.04631957, 0.96813248],
       [1.02073728, 1.0442743 , 1.07658108],
       [1.0837193 , 1.02063329, 1.03724006],
       [1.01928561, 1.04030508, 1.08200196],
       [1.01552167, 1.07950416, 1.04656682],
       [0.96451507, 0.93473455, 1.24234304],
       [1.18960867, 1.01716865, 0.93481533],
       [1.05043686, 0.9343314 , 1.15682439],
       [0.97121426, 1.10042099, 1.0699574 ],
       [1.02782935, 1.04657503, 1.06718827],
       [0.92590721, 1.10760646, 1.10807898],
       [0.9112478 , 1.13612071, 1.09422414],
       [0.96614835, 1.11502318, 1.06042112],
       [1.08387031, 1.06265709, 0.99506525],
       [1.161275  , 0.96624938, 1.01406828],
       [1.13896721, 0.83506324, 1.1675622 ],
       [1.01548   , 1.0730519 , 1.05306076],
       [1.11426793, 1.08180024, 0.94552448],
       [0.90735223, 1.20057668, 1.03366374],
       [0.98225388, 1.12733646, 1.03200231],
       [1.15498228, 0.98029155, 1.00631882],
       [0.94233877, 1.06329709, 1.1359568 ],
       [1.04943163, 1.10410597, 0.98805506],
       [0.97942932, 0.95851211, 1.20365123],
       [0.92441028, 1.07996855, 1.13721383],
       [1.29964529, 1.07325219, 0.76869518],
       [1.19312609, 1.0248834 , 0.92358316],
       [0.97551776, 1.14791101, 1.01816389],
       [1.01197797, 1.02668477, 1.10292992],
       [1.03962873, 1.05691562, 1.0450483 ],
       [1.07457469, 1.06671906, 1.0002989 ],
       [1.26319178, 0.93605832, 0.94234255],
       [1.05314165, 1.01248348, 1.07596753],
       [1.03887445, 1.06771389, 1.03500431],
       [1.06113852, 1.0850987 , 0.99535544],
       [1.05621129, 1.06701792, 1.01836345],
       [1.062798  , 1.09121578, 0.98757888],
       [1.02053015, 1.07743248, 1.04363002],
       [1.16187305, 1.03052088, 0.94919872],
       [0.98737951, 0.94909776, 1.20511538],
       [0.98775165, 1.09013871, 1.0637023 ],
       [1.00208213, 1.13245439, 1.00705613],
       [1.15549284, 1.03121248, 0.95488733],
       [1.04993764, 1.08077947, 1.01087555],
       [0.96698503, 1.07458762, 1.10002001],
       [1.03579031, 1.1132723 , 0.99253004],
       [0.94675248, 1.08899983, 1.10584035],
       [1.14081485, 0.89202293, 1.10875488],
       [1.06050266, 1.08657901, 0.99451099],
       [1.04054632, 1.05232215, 1.04872418],
       [1.04661468, 1.03379649, 1.06118148],
       [1.05837447, 1.03404631, 1.04917187],
       [1.07193776, 1.02280504, 1.04684985],
       [1.02652823, 1.06382205, 1.05124238],
       [1.06309354, 1.02842925, 1.05006986],
       [1.11735827, 1.01621514, 1.00801924],
       [1.01363205, 1.1094384 , 1.0185222 ],
       [1.12737147, 1.01621257, 0.99800861],
       [1.12332964, 1.02101661, 0.9972464 ],
       [0.99784799, 1.12315642, 1.02058824],
       [1.0298105 , 1.09557594, 1.01620621],
       [1.00262616, 1.04476501, 1.09420148],
       [1.10789744, 0.82517124, 1.20852398],
       [1.17208803, 0.89972138, 1.06978324],
       [1.05312324, 1.13383752, 0.95463189],
       [1.06688152, 1.08986247, 0.98484866],
       [1.2368904 , 0.76269752, 1.14200473],
       [1.14349394, 1.13369894, 0.86439978],
       [1.03698291, 1.06872325, 1.03588649],
       [1.04921068, 1.07596482, 1.01641716],
       [1.07012419, 1.00225216, 1.0692163 ],
       [1.01002333, 1.07313653, 1.0584328 ],
       [1.03469671, 1.05863897, 1.04825697],
       [1.06001026, 1.0247235 , 1.0568589 ],
       [1.03622159, 1.06414744, 1.04122362],
       [1.09218789, 1.04006191, 1.00934286],
       [1.03576219, 1.01372297, 1.09210749],
       [1.0147675 , 0.99529192, 1.13153323],
       [1.00518942, 1.13702443, 0.9993788 ],
       [1.04057949, 1.0229041 , 1.07810907],
       [1.01304792, 1.05408021, 1.07446453],
       [1.05622282, 1.07123609, 1.01413375],
       [1.06233325, 1.04044188, 1.03881753],
       [1.01377956, 1.13191658, 0.99589652],
       [1.04381186, 1.0614033 , 1.03637749],
       [1.08056998, 1.0441277 , 1.01689498],
       [1.04144705, 1.06330995, 1.03683565],
       [1.05361706, 1.06021641, 1.02775919],
       [1.05127355, 1.08643614, 1.00388297],
       [1.28498959, 0.96441577, 0.8921873 ],
       [1.1029237 , 0.96735337, 1.07131558],
       [1.0804442 , 0.96041934, 1.1007291 ],
       [1.03577291, 1.08146549, 1.02435425],
       [1.0716458 , 1.00705625, 1.06289061],
       [0.92915746, 1.06260301, 1.14983219],
       [0.92065135, 1.05142003, 1.16952127],
       [0.96351574, 1.17236836, 1.00570855],
       [1.00377261, 1.08348733, 1.05433271],
       [1.74166418, 0.76984876, 0.63007972],
       [0.98366423, 1.16052633, 0.9974021 ],
       [1.03859788, 1.15792386, 0.94507092],
       [1.34161708, 0.75096488, 1.04901069],
       [0.98953594, 0.97108976, 1.18096695],
       [1.03085878, 0.97010927, 1.1406246 ],
       [1.02292602, 1.10214849, 1.01651815],
       [1.07685545, 1.14458539, 0.92015181],
       [0.97951969, 0.95939405, 1.20267891],
       [0.66694944, 0.66118984, 1.81345337],
       [1.16461209, 1.00671534, 0.97026523],
       [0.978034  , 1.02994157, 1.13361708],
       [0.95542521, 1.12601804, 1.06014941],
       [1.06150166, 1.09987939, 0.9802116 ],
       [1.15295395, 1.16081153, 0.82782718],
       [0.98591171, 1.07223671, 1.08344424],
       [1.06733818, 1.01954558, 1.0547089 ],
       [1.03767448, 1.03590913, 1.06800904],
       [1.08632405, 0.9867887 , 1.06847991],
       [1.05773656, 0.96660661, 1.11724948],
       [1.08949393, 1.05476947, 0.99732925],
       [1.02019056, 1.02049083, 1.10091126],
       [1.23169969, 1.07817179, 0.83172117],
       [1.00598649, 1.06263812, 1.07296804],
       [0.99514016, 1.03088626, 1.11556623],
       [0.83632313, 0.46148581, 1.84378371],
       [1.03330309, 1.09573167, 1.01255789],
       [0.99337001, 1.05351641, 1.09470624],
       [1.03515157, 1.07400399, 1.03243709],
       [1.05177823, 0.99380451, 1.09600991],
       [1.10660291, 0.96050782, 1.07448192],
       [0.93574392, 1.09047654, 1.1153722 ],
       [1.00917344, 1.03885361, 1.09356561],
       [1.12980108, 0.99400704, 1.01778453],
       [0.95677016, 1.0741483 , 1.11067419],
       [1.04475115, 1.06823544, 1.02860606],
       [1.09196945, 0.98020508, 1.06941812],
       [1.06033414, 1.00601702, 1.0752415 ],
       [1.05041719, 1.08240379, 1.00877167],
       [1.01396071, 1.00593487, 1.12169707],
       [1.0149729 , 0.99841918, 1.12820057],
       [1.1043009 , 1.02221412, 1.01507764],
       [1.01304611, 0.9948148 , 1.13373175],
       [1.01335582, 0.99885111, 1.12938572],
       [1.0958896 , 1.01996253, 1.02574052],
       [1.03620462, 1.09001275, 1.01537528],
       [1.938008  , 0.3897055 , 0.81387916],
       [0.98832755, 1.0513399 , 1.1019252 ],
       [0.97364878, 1.02233214, 1.14561173],
       [1.16557557, 0.92696151, 1.04905558],
       [0.94592533, 1.14109186, 1.05457546],
       [0.97236185, 0.9580124 , 1.2112184 ],
       [1.17251161, 0.89928918, 1.06979186],
       [0.99289021, 1.02894545, 1.11975699],
       [1.05758133, 1.0038121 , 1.08019922],
       [0.98848959, 1.0682551 , 1.08484796],
       [1.06113535, 1.07459817, 1.00585913],
       [1.05380142, 1.02021438, 1.06757686],
       [1.01801517, 1.06645307, 1.05712441],
       [1.02191884, 1.02012259, 1.09955122],
       [1.06681565, 1.0506827 , 1.0240943 ],
       [1.01002817, 1.01860797, 1.11295651],
       [1.14319873, 1.02896481, 0.96942912],
       [1.00852813, 1.13516516, 0.99789937],
       [1.03118653, 1.08154204, 1.02886408],
       [1.0811224 , 1.00675378, 1.05371648],
       [1.00966599, 1.07488692, 1.05703975],
       [1.07790946, 1.02162535, 1.04205785],
       [0.97033488, 1.05607634, 1.11518144],
       [1.07359016, 1.03830737, 1.02969513],
       [0.94282171, 0.96972411, 1.22904683],
       [1.03155448, 1.04069955, 1.06933863],
       [1.06676051, 1.01868359, 1.05614855],
       [1.04431605, 1.06320143, 1.03407518],
       [1.03906904, 1.03122377, 1.07129984],
       [1.15727395, 1.05124397, 0.93307474],
       [0.89754717, 0.95807001, 1.28597547],
       [1.16699049, 0.9400776 , 1.03452456],
       [0.91719096, 1.05522716, 1.16917453],
       [1.06798111, 0.9304129 , 1.14319865],
       [1.05293371, 1.07038408, 1.01827486],
       [1.14840817, 0.91526274, 1.07792175],
       [1.21153972, 1.19830864, 0.73174429],
       [1.12088653, 0.99740567, 1.02330046],
       [1.05319296, 0.99810224, 1.09029745],
       [1.01493339, 0.97297439, 1.15368487],
       [1.01328893, 0.97890577, 1.14939796],
       [0.91679901, 1.21177998, 1.01301366],
       [0.94220499, 1.14471194, 1.05467572],
       [1.2168521 , 0.98881814, 0.93592241],
       [1.08372068, 1.08178673, 0.97608524],
       [1.09178542, 0.99032535, 1.05948188],
       [1.09833096, 1.05885009, 0.9844116 ],
       [0.81940515, 0.93435295, 1.38783456],
       [0.96441234, 1.2124035 , 0.96477681],
       [1.12468541, 0.94297953, 1.07392772],
       [1.07298532, 1.00822412, 1.06038321],
       [1.05253875, 1.05387224, 1.03518165],
       [1.00146078, 1.09646615, 1.04366573],
       [1.06865593, 1.08438524, 0.98855148],
       [0.89711986, 0.99803767, 1.24643513],
       [1.05973815, 0.9985863 , 1.0832682 ],
       [1.08866205, 1.0367992 , 1.0161314 ],
       [0.90859099, 1.28557937, 0.94742229],
       [0.98648808, 1.09177852, 1.06332605],
       [0.98533357, 1.08956924, 1.06668984],
       [1.29392337, 1.23710446, 0.61056482],
       [0.9623054 , 1.00323009, 1.17605716],
       [1.06182851, 1.04175465, 1.03800949],
       [0.98387062, 1.07241822, 1.08530381],
       [1.05732482, 1.10750668, 0.97676115],
       [1.04706545, 1.19235791, 0.90216929],
       [1.04485623, 1.11975642, 0.97698   ],
       [1.1400997 , 0.97232716, 1.02916579],
       [1.02229426, 1.09992881, 1.01936959],
       [1.08216721, 1.01514137, 1.04428408],
       [1.10434961, 1.01612755, 1.0211155 ],
       [1.00475305, 1.07252386, 1.06431574],
       [1.10045536, 0.99610771, 1.04502958],
       [1.11064277, 1.0005198 , 1.03043008],
       [0.99422982, 1.0173784 , 1.12998443],
       [1.03382195, 1.09004689, 1.01772382],
       [1.14362497, 0.99039414, 1.00757354],
       [1.03801805, 1.09062557, 1.01294903],
       [0.99892237, 1.04456565, 1.09810463],
       [1.31497127, 1.29692547, 0.52969591],
       [1.0064033 , 1.17782282, 0.95736653],
       [1.26436575, 0.94228785, 0.93493905],
       [0.90307728, 0.97386377, 1.2646516 ],
       [0.97133382, 0.99195407, 1.17830476],
       [1.05853494, 1.06753449, 1.01552322],
       [1.16845662, 0.71706312, 1.25607292],
       [1.08273792, 1.05101698, 1.00783775],
       [1.07099016, 1.08194897, 0.98865352],
       [0.99311615, 1.06129816, 1.08717834],
       [1.07178748, 1.02951546, 1.04028971],
       [0.99632797, 1.0592345 , 1.08603019],
       [0.99322917, 1.08534402, 1.06301946],
       [1.02518371, 1.03720219, 1.07920675],
       [1.02308555, 1.02874868, 1.08975843],
       [0.98563395, 1.1162786 , 1.0396801 ],
       [1.12928417, 1.00900746, 1.00330103],
       [0.994024  , 1.14331777, 1.00425088],
       [1.07850227, 1.00012681, 1.06296357],
       [1.06131252, 1.04828016, 1.03199998],
       [1.055908  , 1.05909215, 1.0265925 ],
       [1.07101942, 1.03454903, 1.03602421],
       [0.98924947, 1.14839816, 1.00394503],
       [1.06488149, 1.00759912, 1.06911205],
       [0.91713155, 1.09723935, 1.12722176],
       [1.06233781, 0.94495627, 1.13429857],
       [1.10537036, 1.06831238, 0.96790991],
       [1.03104496, 1.04867545, 1.06187225],
       [1.06010471, 1.03814421, 1.04334373],
       [1.05056622, 1.08698835, 1.00403808],
       [1.074631  , 1.13836631, 0.92859534],
       [0.9580637 , 1.04310594, 1.14042301],
       [1.13250731, 1.10316259, 0.90592275],
       [1.0665899 , 0.94138781, 1.13361495],
       [1.06660212, 0.98717655, 1.08781398],
       [0.96873515, 1.26317079, 0.90968672],
       [0.79341943, 0.55700676, 1.79116646],
       [0.95158796, 1.04314595, 1.14685874],
       [1.00716178, 1.05569174, 1.07873913],
       [1.18397459, 0.99595753, 0.96166054],
       [0.95399452, 1.20383314, 0.98376499],
       [1.01015992, 0.89127223, 1.24016051],
       [1.21877614, 0.96867216, 0.95414436],
       [1.05800249, 1.04267704, 1.04091312],
       [1.03200469, 1.00115603, 1.10843193],
       [1.05447456, 1.13283456, 0.95428353],
       [1.02011306, 1.04163602, 1.07984357],
       [1.07485234, 1.02113775, 1.04560256],
       [1.05298899, 1.02765191, 1.06095176],
       [1.14080582, 1.16772157, 0.83306526],
       [1.09193212, 1.05235407, 0.99730646],
       [1.14585436, 0.94402586, 1.05171244],
       [1.00346977, 1.09495626, 1.04316662],
       [1.12715313, 1.01565134, 0.99878819],
       [1.00904494, 0.9056925 , 1.22685521],
       [0.98634253, 1.1336007 , 1.02164943],
       [1.18551374, 0.93459566, 1.02148326],
       [0.93428038, 0.92544186, 1.28187041],
       [1.10659085, 0.93969208, 1.09530972],
       [1.05486835, 1.0701578 , 1.01656651],
       [1.13525811, 1.00943144, 0.9969031 ],
       [1.13658096, 0.98647835, 1.01853335],
       [1.02559539, 1.09149539, 1.02450187],
       [1.00752559, 0.95982536, 1.1742417 ],
       [0.81317543, 1.20555836, 1.12285886],
       [0.99295347, 1.08154972, 1.06708946],
       [0.82446524, 1.13066613, 1.18646128],
       [1.07919612, 0.98370292, 1.07869361],
       [1.00812997, 1.04554013, 1.08792255],
       [1.01729667, 1.03277334, 1.09152265],
       [1.11559249, 1.00764843, 1.01835174],
       [1.0032101 , 1.1271494 , 1.01123316],
       [1.0278132 , 1.021338  , 1.09244146],
       [1.02077177, 1.10119671, 1.01962418],
       [1.03835923, 1.09741057, 1.00582285],
       [0.99604661, 1.27780834, 0.8677377 ],
       [0.95218247, 1.16134388, 1.0280663 ],
       [1.03194778, 1.2345285 , 0.87511637],
       [1.0284923 , 1.14581375, 0.96728661],
       [0.99247258, 1.07452722, 1.07459286],
       [1.05959803, 0.95549524, 1.12649939],
       [1.03506046, 1.11010033, 0.99643186],
       [1.07335517, 1.38615685, 0.68208063],
       [1.11052717, 1.03772697, 0.99333851],
       [1.07950831, 1.07636975, 0.98571459],
       [1.09817289, 0.99520233, 1.04821743],
       [1.02898762, 1.02202004, 1.09058499],
       [1.07475923, 0.98650808, 1.08032534],
       [1.01832258, 1.10476282, 1.01850725],
       [1.04834094, 1.00721524, 1.08603647],
       [0.97461564, 1.11494373, 1.05203328],
       [0.97658069, 1.12899644, 1.03601553],
       [0.98647439, 1.02864518, 1.12647308],
       [1.03300955, 1.08303208, 1.02555103],
       [1.02178168, 1.06947995, 1.05033103],
       [1.0563196 , 1.0143366 , 1.07093645],
       [1.01675854, 1.07590484, 1.04892927],
       [0.99547155, 1.02944691, 1.11667419],
       [1.02662669, 1.04613462, 1.06883135],
       [0.99325864, 1.03089572, 1.11743829],
       [0.95781931, 1.04743021, 1.13634313],
       [1.05720509, 1.00923188, 1.07515568],
       [1.0967042 , 1.06588029, 0.97900816],
       [1.04444267, 1.00347415, 1.09367583],
       [1.00728887, 1.07101908, 1.0632847 ],
       [1.14523705, 1.06224222, 0.93411338],
       [0.885076  , 1.08331101, 1.17320565],
       [1.15328401, 0.95671369, 1.03159495],
       [0.94613553, 1.21689578, 0.97856135],
       [1.08537876, 0.96415542, 1.09205848],
       [0.96826647, 1.13116347, 1.04216272],
       [1.07700489, 1.08139093, 0.98319683],
       [1.1119038 , 1.25432401, 0.77536485],
       [1.08365499, 0.97468901, 1.08324866],
       [1.08277461, 0.96432715, 1.09449089],
       [1.17003173, 1.01426919, 0.95729174],
       [1.05992942, 0.9334447 , 1.14821854],
       [1.04502305, 1.25410709, 0.84246252],
       [1.43561654, 0.90288526, 0.80309085],
       [1.03265499, 1.00774197, 1.1011957 ],
       [1.04906255, 1.0290724 , 1.0634577 ],
       [1.05179948, 1.05084601, 1.03894716],
       [1.03526054, 1.10304584, 1.00328628],
       [0.80233854, 0.80831376, 1.53094035],
       [1.0580945 , 1.00487573, 1.07862243],
       [0.92076864, 1.01791492, 1.20290909],
       [1.01251372, 1.07783898, 1.05123995],
       [1.06353493, 0.9877123 , 1.09034542],
       [1.18768484, 1.04538166, 0.90852615],
       [1.05789838, 1.01413426, 1.06956001],
       [1.15447738, 1.14933365, 0.83778162],
       [1.11440225, 1.08736456, 0.93982584],
       [1.07628731, 0.97666796, 1.08863738],
       [0.93112934, 1.09220367, 1.11825965],
       [1.0479124 , 1.04176001, 1.05192025],
       [0.97140084, 0.99019369, 1.17999812],
       [1.01183432, 1.00679436, 1.12296397],
       [0.99309216, 1.2057825 , 0.94271799],
       [0.69295391, 0.76748804, 1.6811507 ],
       [1.07326432, 1.11174138, 0.95658695],
       [1.04910302, 1.10073496, 0.99175468],
       [0.991891  , 1.0337595 , 1.11594216],
       [0.99700218, 1.06192882, 1.08266165],
       [1.07025686, 1.10927355, 0.96206225],
       [1.12127767, 0.96653083, 1.05378416],
       [1.03985441, 1.07853299, 1.02320525],
       [1.08857113, 1.02221783, 1.0308037 ],
       [1.00797805, 1.0273046 , 1.10631   ],
       [0.92917164, 0.99832846, 1.21409255],
       [1.13491896, 1.02526347, 0.98141023],
       [0.96549773, 1.21824597, 0.95784895],
       [1.03990785, 1.12964684, 0.97203796],
       [1.07462745, 0.97769698, 1.08926822],
       [0.92176325, 1.26335712, 0.95647229],
       [1.02457137, 1.02385605, 1.09316523],
       [1.76193238, 0.52261672, 0.85704356],
       [1.08210274, 0.97428413, 1.08520578],
       [1.10139311, 0.97701494, 1.0631846 ],
       [1.03469706, 1.09848195, 1.00841364],
       [1.01636697, 1.07094591, 1.05427977],
       [1.08650223, 0.99357046, 1.06151996],
       [0.94522956, 1.09985892, 1.09650417],
       [0.95380893, 1.12699363, 1.06079009],
       [1.09119551, 1.06776488, 0.98263227],
       [1.0110797 , 1.01790289, 1.11261006],
       [1.11445963, 1.03640163, 0.9907314 ],
       [1.01222545, 1.05074847, 1.07861874],
       [1.09123466, 1.0571877 , 0.99317029],
       [1.04822041, 1.05758786, 1.03578438],
       [1.02681894, 1.0322089 , 1.08256481],
       [0.96904415, 1.12630837, 1.04624013],
       [1.07315065, 0.993222  , 1.07522001],
       [1.10854165, 0.99914696, 1.03390405],
       [1.29734845, 1.06163757, 0.78260663],
       [1.20240546, 1.01783426, 0.92135293],
       [0.94469087, 1.12430254, 1.07259925],
       [1.03748097, 1.01286396, 1.09124773],
       [1.02174803, 1.07289734, 1.04694729],
       [0.93239419, 1.17256544, 1.03663303],
       [0.98871866, 1.23777118, 0.91510281],
       [1.10488365, 1.12655841, 0.91015059],
       [1.05926541, 0.93912304, 1.1432042 ],
       [1.15273125, 1.03398911, 0.95487229],
       [1.10988785, 1.08453824, 0.94716656],
       [0.94159835, 1.26214545, 0.93784885],
       [1.04632442, 1.12915239, 0.96611584],
       [1.05870034, 1.1487946 , 0.93409771],
       [0.97765979, 1.17586379, 0.98806907],
       [0.97147498, 1.15429566, 1.01582202],
       [1.0821559 , 0.98662833, 1.07280842],
       [1.1708983 , 0.77866897, 1.19202538],
       [1.01974661, 1.04800882, 1.07383722],
       [1.08530487, 1.05382798, 1.0024598 ],
       [1.06366103, 1.04116505, 1.03676657],
       [1.05484181, 0.9979818 , 1.08876904],
       [1.11213554, 1.11900214, 0.91045498],
       [0.78089714, 1.56314749, 0.79754802],
       [1.09174197, 1.04392526, 1.00592543],
       [0.81674155, 1.12713103, 1.19772007],
       [1.19902497, 1.05027254, 0.89229515],
       [1.04863593, 1.02591512, 1.06704161],
       [0.99726682, 1.10475655, 1.03956929],
       [1.03405632, 1.01015113, 1.0973852 ],
       [0.91962007, 0.92508267, 1.29688992],
       [0.77171795, 0.80170267, 1.56817203],
       [1.06885052, 1.09571261, 0.97702952],
       [1.05080544, 1.08215631, 1.0086309 ],
       [1.11864971, 1.06197474, 0.9609682 ],
       [1.07421246, 1.067916  , 0.99946419],
       [1.11168945, 0.98946021, 1.04044299],
       [0.82655764, 1.40982768, 0.90520733],
       [1.17794455, 1.10234933, 0.86129877],
       [0.99751494, 1.03893276, 1.10514496],
       [1.03471466, 1.1028239 , 1.0040541 ],
       [1.05822779, 1.04130422, 1.04206065],
       [1.05308685, 1.04814838, 1.04035743],
       [1.05128811, 1.02062986, 1.06967468],
       [1.01274071, 1.07082553, 1.05802642],
       [1.01040677, 1.09242432, 1.03876156],
       [1.08990978, 1.02911831, 1.02256456],
       [0.94180145, 1.22302912, 0.97676208],
       [1.02443887, 0.98978451, 1.12736927],
       [1.06697742, 1.02702823, 1.047587  ],
       [1.05426732, 0.99913143, 1.0881939 ],
       [1.0023512 , 1.12560124, 1.01364022],
       [1.04334561, 0.99670993, 1.10153711],
       [1.31716737, 1.22248855, 0.60193674],
       [1.11068263, 1.04781062, 0.98309941],
       [0.77581223, 1.04624129, 1.31953913],
       [0.98659115, 1.09601237, 1.05898913],
       [1.07750349, 1.05902854, 1.00506062],
       [1.04059556, 1.03372636, 1.06727073],
       [1.02059064, 1.0383423 , 1.08265971],
       [0.99175785, 1.03184632, 1.11798848],
       [0.93060009, 1.21106246, 0.9999301 ],
       [1.08233645, 1.05977977, 0.99947644],
       [1.00075681, 1.03458202, 1.10625382],
       [1.01734742, 1.14866312, 0.97558212],
       [1.10726196, 1.00833596, 1.02599474],
       [1.07404479, 1.00925404, 1.05829383],
       [1.02256734, 1.02067635, 1.09834897],
       [1.10919199, 1.03934013, 0.99306053],
       [1.0502842 , 1.01475655, 1.07655191],
       [1.40977875, 0.95732183, 0.77449207],
       [1.00390202, 1.11465938, 1.02303125],
       [0.99169145, 1.07836086, 1.07154034],
       [1.05765521, 1.089433  , 0.99450445],
       [0.99395356, 0.94985259, 1.19778651],
       [1.73179108, 0.63866415, 0.77113742],
       [1.03950994, 1.24270859, 0.85937412],
       [0.97678426, 1.09556634, 1.06924205],
       [0.9762204 , 1.0640485 , 1.10132376],
       [1.18861138, 1.05083402, 0.90214725],
       [0.96702286, 1.15547223, 1.01909756],
       [0.95504948, 1.03071058, 1.1558326 ],
       [1.1546598 , 0.95542292, 1.03150993],
       [1.09275078, 1.08097452, 0.96786735],
       [1.05308396, 1.00341161, 1.08509708],
       [0.91905403, 1.10374886, 1.11878976],
       [1.08736896, 1.03670247, 1.01752123],
       [1.05389193, 0.98890636, 1.09879437],
       [1.10456257, 1.13416568, 0.9028644 ],
       [1.12011182, 1.02345396, 0.99802687],
       [1.0956118 , 0.94959311, 1.09638775],
       [1.07646325, 1.06983758, 0.99529182],
       [1.01160703, 1.01987254, 1.11011308],
       [1.05229902, 1.00770035, 1.08159329],
       [1.03065393, 1.06091611, 1.05002261],
       [1.07599792, 1.04596289, 1.01963184],
       [1.0667286 , 0.982004  , 1.09286005],
       [0.89919257, 1.15325896, 1.08914113],
       [1.05662603, 1.00319806, 1.08176856],
       [1.01002306, 0.97653859, 1.155031  ],
       [1.02252164, 0.96118304, 1.15788798],
       [1.01531098, 1.06158417, 1.0646975 ],
       [1.08626487, 0.99924754, 1.05608024],
       [1.00795345, 1.105141  , 1.0284982 ],
       [1.03486215, 1.11321404, 0.99351647],
       [0.82704586, 1.47152716, 0.84301964],
       [1.01066924, 1.05740334, 1.07352008],
       [1.06808952, 1.09484422, 0.97865892],
       [1.08271407, 1.1161427 , 0.94273588],
       [0.98441737, 1.07327513, 1.08390016],
       [0.98537445, 1.06454543, 1.09167278],
       [0.91322923, 1.11278201, 1.11558141],
       [0.98540516, 1.04811752, 1.10806998],
       [1.11007342, 1.00233175, 1.02918748],
       [1.08102753, 1.02274762, 1.03781751],
       [1.02392111, 1.0914456 , 1.02622595],
       [1.0066179 , 1.03998499, 1.09498976],
       [1.0865074 , 1.05636186, 0.99872339],
       [1.08323572, 1.03766697, 1.02068996],
       [1.01921237, 1.09594147, 1.02643881],
       [1.18910274, 0.85218393, 1.10030598],
       [1.39184401, 0.88367586, 0.86607279],
       [0.97524114, 0.9567038 , 1.20964771],
       [1.28074338, 0.91376393, 0.94708534],
       [1.02109024, 0.99094002, 1.12956239],
       [0.98880937, 1.15332955, 0.99945373],
       [0.90978692, 1.10669777, 1.12510797],
       [1.02970461, 0.96452689, 1.14736115],
       [1.10763772, 0.73125771, 1.30269723],
       [1.10569476, 1.13389578, 0.90200211],
       [0.99979563, 1.05462701, 1.08717002],
       [1.08616161, 1.01877479, 1.03665625],
       [1.07248161, 1.02372303, 1.04538801],
       [1.02313894, 1.08698925, 1.03146446],
       [1.09645111, 1.0376575 , 1.00748405],
       [1.07972343, 1.01975384, 1.04211538],
       [0.85543199, 1.14860408, 1.13755658],
       [1.08620089, 1.06219971, 0.99319205],
       [1.15425656, 1.03517645, 0.95215964],
       [1.08282678, 1.03577416, 1.02299171],
       [1.05187527, 1.00281764, 1.08689974],
       [1.03388106, 1.13391092, 0.97380068],
       [1.01444021, 1.03561983, 1.09153262],
       [1.00493382, 1.11948899, 1.01716985],
       [1.18309563, 0.81253163, 1.14596539],
       [1.12023826, 1.05180861, 0.96954579],
       [1.0579738 , 0.98138451, 1.10223434],
       [1.06456081, 1.00167769, 1.07535415],
       [0.9256736 , 1.2151475 , 1.00077155],
       [1.05323753, 1.26944459, 0.81891054],
       [1.07003238, 1.00519864, 1.06636164],
       [1.63522437, 0.84158012, 0.66478817],
       [1.07493955, 0.97151954, 1.09513356],
       [1.1280501 , 0.97627989, 1.03726266],
       [0.91519127, 1.20257439, 1.02382699],
       [1.12707353, 1.02766737, 0.98685175],
       [0.96035786, 1.09469009, 1.0865447 ],
       [0.99376781, 1.07424583, 1.07357902],
       [1.08299885, 1.04037887, 1.01821493],
       [1.25000316, 0.96706594, 0.92452355],
       [0.95214215, 1.04015718, 1.14929332],
       [1.02948532, 0.96857737, 1.14352996],
       [0.8862274 , 1.12674537, 1.12861989],
       [1.01617956, 1.00880531, 1.11660779],
       [0.98998517, 1.09042351, 1.06118397],
       [1.05985278, 1.01379916, 1.06794071],
       [1.01699112, 1.07842068, 1.04618085],
       [1.04230244, 1.02498847, 1.07430174],
       [1.00594922, 1.09042592, 1.04521751],
       [1.10609812, 1.06455377, 0.97094076],
       [1.12410309, 1.0218615 , 0.99562806],
       [1.06456918, 1.05946742, 1.01755605],
       [0.93513072, 1.16607108, 1.04039086],
       [0.96265441, 1.12632887, 1.05260938],
       [0.90106679, 1.17658973, 1.06393613],
       [1.04909256, 1.02528033, 1.06721976],
       [1.05031467, 0.99115554, 1.10012244],
       [0.96500611, 0.98917827, 1.18740827],
       [0.8580644 , 1.11998065, 1.1635476 ],
       [1.01628713, 1.01997493, 1.10533059],
       [1.06508351, 1.00704332, 1.06946581],
       [0.99892816, 1.04545419, 1.09721031],
       [0.99314288, 1.06043347, 1.0880163 ],
       [1.03553909, 1.07573811, 1.03031545],
       [1.09565058, 1.00119096, 1.04475111],
       [1.09829069, 0.9971115 , 1.04619047],
       [1.01535171, 1.02958648, 1.09665446],
       [1.07081524, 1.0181776 , 1.05259981],
       [1.04668547, 1.03576281, 1.05914437],
       [1.0551182 , 1.07468788, 1.01178657],
       [0.9882694 , 1.07560806, 1.07771519],
       [1.05411859, 1.01012958, 1.07734448],
       [1.05140832, 1.0149667 , 1.07521763],
       [0.83115049, 1.21474155, 1.09570061],
       [1.16304358, 0.87782604, 1.10072304],
       [1.87166532, 0.55936946, 0.71055788],
       [1.41611691, 0.82426997, 0.90120578],
       [1.0930761 , 0.93526392, 1.11325263],
       [1.09721634, 1.02463443, 1.01974188],
       [1.08124828, 1.02124347, 1.03910091],
       [1.00367602, 1.04477888, 1.09313775],
       [1.05889646, 1.10704901, 0.97564719],
       [0.88367602, 1.2650468 , 0.99286983],
       [1.0431685 , 0.91050306, 1.1879211 ],
       [1.02988886, 0.99671775, 1.11498604],
       [1.04881721, 1.04286253, 1.04991291],
       [1.01804436, 1.01193553, 1.11161277],
       [1.12822557, 1.05935645, 0.95401063],
       [0.98153596, 1.08434181, 1.07571489],
       [0.95429941, 1.16375547, 1.02353778],
       [1.18371883, 1.18105601, 0.77681781],
       [0.95764562, 1.01897438, 1.16497265],
       [1.0476385 , 0.98103265, 1.11292151],
       [1.01022273, 1.04253486, 1.08883506],
       [1.05308004, 1.06007808, 1.02843453],
       [0.93067507, 1.03168537, 1.17923222],
       [1.16350139, 0.9923989 , 0.98569236],
       [0.99570202, 1.0701112 , 1.07577943],
       [0.9517603 , 1.05517326, 1.1346591 ],
       [0.97976346, 1.04528434, 1.11654486],
       [1.0392367 , 1.01041499, 1.09194096],
       [1.10766552, 1.04699998, 0.98692715],
       [1.08338504, 1.05577668, 1.00243093],
       [0.93670473, 1.133844  , 1.07104392],
       [0.95411496, 0.94370512, 1.24377257],
       [1.03373651, 1.07945515, 1.028401  ],
       [1.26138699, 0.85317689, 1.02702878],
       [0.95121247, 1.0404282 , 1.14995199],
       [1.03507453, 1.19526254, 0.91125559],
       [0.98527962, 1.15042853, 1.00588451],
       [1.03025833, 1.02184859, 1.08948574],
       [1.03857643, 1.01891847, 1.08409776],
       [1.08424416, 1.03526542, 1.02208307],
       [0.99614006, 0.96780756, 1.17764503],
       [0.94165576, 1.15297036, 1.04696653],
       [1.11541235, 0.99042247, 1.03575783],
       [1.20928333, 0.93749914, 0.99481018],
       [1.08063122, 1.04908785, 1.01187358],
       [1.06371807, 1.02211489, 1.05575969],
       [1.04625291, 1.03749122, 1.05784852],
       [1.06928713, 1.05038123, 1.02192429],
       [1.07203819, 1.04517948, 1.02437498],
       [1.08059637, 1.0251838 , 1.03581249],
       [1.02037677, 1.09180643, 1.02940945],
       [1.06229074, 0.97617728, 1.10312464],
       [1.05794449, 1.02341192, 1.06023624],
       [1.00937171, 0.9971394 , 1.13508154],
       [1.09799048, 1.01243668, 1.0311655 ],
       [0.99685614, 0.99379065, 1.15094586],
       [1.02101553, 1.07948594, 1.04109119],
       [1.06185085, 1.04818237, 1.03155943],
       [0.99430087, 1.12920036, 1.01809143],
       [1.07339083, 1.04947369, 1.01872813],
       [1.05563935, 1.08712526, 0.99882804],
       [1.07400205, 1.02121974, 1.04637086],
       [1.05013704, 1.01580563, 1.07564998],
       [1.01562162, 1.12527612, 1.00069492],
       [1.04433112, 1.10066394, 0.99659759],
       [1.03170522, 1.09853116, 1.01135627],
       [0.99351689, 1.06811576, 1.07996   ],
       [1.0695732 , 1.04407917, 1.02794028],
       [1.08124864, 0.98844876, 1.07189525],
       [1.08855711, 1.07666985, 0.97636569],
       [0.99687966, 1.08300455, 1.06170845],
       [1.04876715, 1.03431666, 1.05850885],
       [0.89798257, 1.27228865, 0.97132142],
       [1.01982607, 1.04211153, 1.07965505],
       [0.91069945, 1.46427137, 0.76662183],
       [1.28378667, 1.02333082, 0.83447516],
       [1.04631646, 1.09752094, 0.99775525],
       [1.05539748, 1.06241334, 1.02378184],
       [0.97504973, 0.97842363, 1.1881193 ],
       [1.13050922, 0.97580126, 1.03528218],
       [0.99214895, 1.03251797, 1.11692574],
       [1.25090539, 0.98241883, 0.90826843],
       [1.01353362, 0.98311001, 1.14494902],
       [1.02591311, 1.10370894, 1.0119706 ],
       [1.03554042, 1.11210458, 0.99394766],
       [1.09143784, 1.0005053 , 1.04964952],
       [0.96925281, 1.17396183, 0.99837801],
       [1.04712362, 0.97533214, 1.11913689],
       [1.08661826, 1.00619294, 1.04878145],
       [1.09176629, 0.91020979, 1.13961658],
       [1.13155098, 0.79889745, 1.21114422],
       [1.03637875, 1.00279857, 1.10241533],
       [0.97309555, 1.12000173, 1.04849537],
       [0.98651584, 1.02752779, 1.12754902],
       [1.10724021, 0.91611056, 1.11824189],
       [0.81999021, 1.03679334, 1.28480911],
       [1.17657821, 1.0015983 , 0.96341614],
       [1.0471425 , 0.96167977, 1.13277039],
       [1.13682137, 0.94853108, 1.05624021],
       [1.08097583, 1.02253349, 1.03808333],
       [1.11180935, 1.02461955, 1.00516376],
       [1.03555766, 0.98732277, 1.11871222],
       [1.02570026, 1.05824376, 1.05764864],
       [1.03628088, 0.99012237, 1.11518941],
       [1.01123701, 1.15616418, 0.97419146],
       [0.98205709, 1.08524156, 1.074294  ],
       [1.08997385, 0.84590535, 1.20571345],
       [0.99572944, 1.16812501, 0.97773821],
       [1.58865064, 0.91746395, 0.63547807],
       [1.00960274, 1.00533263, 1.12665729],
       [1.04616648, 1.07712328, 1.01830289],
       [1.03096066, 1.05912719, 1.0515048 ],
       [1.11715897, 0.98882467, 1.03560901],
       [1.004374  , 1.14426669, 0.99295196],
       [1.04460362, 0.97306672, 1.12392232],
       [1.07284232, 1.09348236, 0.97526797],
       [1.05626697, 1.1005628 , 0.98476288],
       [1.08536569, 1.04025348, 1.01597349],
       [1.03577745, 1.06651384, 1.03930137],
       [1.04809707, 1.03929704, 1.05419854],
       [1.03870947, 1.02955579, 1.0733274 ],
       [1.03332808, 1.04646416, 1.06180042],
       [1.05120855, 1.00186211, 1.088522  ],
       [1.05952861, 1.07647271, 1.00559134],
       [1.03778347, 1.05307436, 1.05073482],
       [1.12002281, 1.04647376, 0.97509608],
       [1.10544854, 1.06266677, 0.97347735],
       [1.06281255, 0.99733834, 1.08144177],
       [0.96464872, 1.13869832, 1.03824561],
       [1.18364343, 0.93171403, 1.0262352 ],
       [1.1470322 , 1.07014929, 0.92441116],
       [1.00046119, 1.07906717, 1.06206429],
       [1.08815323, 1.00948844, 1.04395099],
       [1.03316126, 1.07456817, 1.03386323],
       [1.02845642, 1.07150883, 1.04162741],
       [1.08722405, 1.01242237, 1.04194623],
       [1.02719565, 1.08501895, 1.02937805],
       [0.98873308, 1.04455291, 1.10830666],
       [1.05835634, 1.04505159, 1.03818473],
       [1.07110802, 1.08668121, 0.98380343],
       [1.09283062, 0.97363293, 1.0751291 ],
       [1.05908362, 1.08300806, 0.99950098],
       [1.05679679, 1.06398756, 1.02080831],
       [1.13132933, 1.08611907, 0.92414425],
       [1.0269754 , 1.10485981, 1.00975745],
       [0.82193555, 1.43892362, 0.88073349],
       [1.11495576, 1.02910988, 0.997527  ],
       [1.09587482, 1.00823323, 1.0374846 ],
       [1.06859951, 1.12370036, 0.94929278],
       [1.08759208, 0.9791688 , 1.07483178],
       [1.09458538, 1.08770005, 0.95930722],
       [0.95077834, 0.92069949, 1.27011483],
       [1.14796402, 1.02186181, 0.97176683],
       [0.97134708, 1.1745421 , 0.99570348],
       [1.06903731, 1.03129988, 1.04125546],
       [1.05500775, 1.14200323, 0.94458168],
       [1.20662921, 1.01890183, 0.91606161],
       [1.09835745, 0.92334861, 1.11988659],
       [1.06474789, 1.09651061, 0.98033414],
       [1.01651064, 1.15025644, 0.97482557],
       [1.09850658, 1.06394457, 0.9791415 ],
       [0.87541542, 1.22643841, 1.03973882],
       [1.0821169 , 1.03422438, 1.02525137],
       [0.95891147, 1.06439012, 1.11829106],
       [0.83535042, 1.16558467, 1.14065756],
       [0.97960647, 1.0365688 , 1.12541738],
       [0.80907927, 1.32048963, 1.01202376],
       [0.91974647, 1.16134298, 1.06050321],
       [1.03946268, 1.09485584, 1.00727414],
       [1.04744896, 1.00176089, 1.09238281],
       [1.07844358, 1.02178967, 1.0413594 ],
       [1.03521081, 1.02002891, 1.08635293],
       [1.01793587, 1.00387556, 1.11978123],
       [1.13865468, 1.04185312, 0.96108485],
       [0.97277102, 1.07815851, 1.09066313],
       [0.95398243, 1.11005941, 1.07755081],
       [0.9987572 , 1.15803699, 0.98479847],
       [1.10204445, 1.10135419, 0.93819401],
       [1.10145777, 1.09048515, 0.94964973],
       [1.29392193, 1.14895414, 0.69871658],
       [0.64824961, 0.94038214, 1.5529609 ],
       [1.10751759, 1.00005069, 1.03402437],
       [1.00735237, 1.08738292, 1.04685737],
       [1.56213081, 0.96476807, 0.61469377],
       [1.10724097, 0.99000008, 1.0443516 ],
       [1.09911138, 1.018404  , 1.02407728],
       [1.00822704, 1.03131852, 1.1020471 ],
       [1.01906855, 1.10990535, 1.01261875],
       [1.02860744, 0.99124167, 1.12174354],
       [1.06724392, 1.05611308, 1.01823565],
       [1.01322748, 1.07692716, 1.05143802],
       [1.08205077, 1.03829681, 1.02124508],
       [1.04619369, 1.03664783, 1.05875113],
       [1.02391036, 1.06756601, 1.05011629],
       [1.07386299, 1.04050147, 1.02722819],
       [1.03214063, 1.06915663, 1.04029539],
       [1.04943666, 1.0339772 , 1.05817879],
       [0.99384303, 1.07430182, 1.0734478 ],
       [1.10451746, 1.02768759, 1.0093876 ],
       [0.64951948, 1.15801108, 1.33406209],
       [0.98713342, 1.11690229, 1.03755694],
       [1.06021171, 1.03259873, 1.04878221],
       [1.03468938, 1.04416143, 1.06274185],
       [1.02167707, 1.0177876 , 1.10212798],
       [1.07911161, 1.06369276, 0.99878828],
       [1.00929985, 1.03981543, 1.09247738],
       [1.08234984, 1.02079241, 1.0384504 ],
       [1.05737302, 1.098804  , 0.98541564],
       [1.02782212, 1.03213079, 1.08163974],
       [0.97667282, 1.07735871, 1.08756112],
       [1.08619323, 0.98268254, 1.07271689],
       [1.07106053, 1.00612797, 1.06440415],
       [1.04377627, 1.02890526, 1.06891113],
       [1.04826564, 0.94102943, 1.15229759],
       [1.08272128, 1.07056618, 0.98830519],
       [1.28507672, 0.88993175, 0.96658419],
       [1.09766463, 1.03291071, 1.01101731],
       [0.98939002, 1.1114051 , 1.04079754],
       [1.03970361, 1.17507273, 0.92681632],
       [1.11978555, 1.10591308, 0.91589403],
       [1.09494555, 1.05941557, 0.98723153],
       [0.99713137, 1.1241518 , 1.02030948],
       [1.22686037, 0.94121411, 0.97351817],
       [1.0990081 , 1.03482042, 1.00776413],
       [0.96520317, 1.01593384, 1.16045564],
       [1.04185621, 0.98806159, 1.11167485],
       [0.92691777, 0.91652158, 1.2981533 ],
       [1.22196292, 0.9885962 , 0.93103354],
       [0.99681461, 1.06706579, 1.07771225],
       [1.03440712, 1.08019455, 1.02699099],
       [0.97797961, 1.04243667, 1.12117638],
       [1.07654472, 1.08150297, 0.98354496],
       [1.03229228, 1.10621401, 1.00308637],
       [1.01662559, 1.00673715, 1.11822991],
       [1.04979633, 1.00970157, 1.08209475],
       [0.994797  , 1.14143023, 1.00536543],
       [0.97147081, 1.18041081, 0.98971103],
       [1.03370269, 1.105187  , 1.00270296],
       [1.09330699, 0.91064734, 1.13763832],
       [1.05497979, 1.13939006, 0.9472228 ],
       [1.15518201, 0.98742016, 0.99899049],
       [1.08495983, 1.0109818 , 1.04565102],
       [1.00320658, 1.08237925, 1.05600683],
       [1.04989589, 1.1298299 , 0.96186686],
       [0.99969641, 1.13435078, 1.00754546],
       [1.02601571, 0.98076978, 1.13480716],
       [1.0392481 , 1.0306738 , 1.07167075],
       [1.08070878, 1.03888849, 1.02199538],
       [1.05509878, 0.99443469, 1.09205918],
       [0.91725946, 1.08986649, 1.1344667 ],
       [0.93023053, 1.11358824, 1.09777388],
       [1.07825737, 1.17053908, 0.89279621],
       [0.67667798, 1.10031101, 1.36460366],
       [1.0580495 , 1.10053506, 0.9830081 ],
       [0.97198216, 1.1438405 , 1.02577   ],
       [0.63015995, 1.38537599, 1.12605671],
       [0.95316479, 1.57943745, 0.60899041],
       [1.00705663, 1.03972715, 1.09480888],
       [1.09045923, 1.0493159 , 1.00181752],
       [1.062872  , 1.00759995, 1.07112071],
       [1.07302891, 1.05110354, 1.0174602 ],
       [1.05935752, 1.00992446, 1.07231067],
       [1.06695543, 1.05721222, 1.017425  ],
       [1.04372895, 1.061965  , 1.03589871],
       [1.0460946 , 1.01491563, 1.08058242],
       [1.07658143, 1.02534634, 1.03966489],
       [1.06018576, 1.00717439, 1.0742325 ],
       [1.01898517, 1.07332285, 1.04928463],
       [1.05655084, 1.06546497, 1.01957684],
       [0.96734396, 1.10964145, 1.06460724],
       [1.18235419, 1.10149603, 0.85774243],
       [1.10418084, 0.87491692, 1.1624949 ],
       [1.01938342, 1.05561302, 1.06659622],
       [1.04944783, 1.06810569, 1.02403913],
       [1.061144  , 1.02225504, 1.05819362],
       [1.04452793, 1.04698892, 1.0500758 ],
       [1.06926404, 1.03793171, 1.0343969 ],
       [1.04041316, 1.06293552, 1.03824397],
       [1.07603532, 1.00135291, 1.06420443],
       [1.02045325, 1.05585235, 1.06528705],
       [0.97895655, 1.06903919, 1.09359691],
       [1.06458793, 1.08973198, 0.98727275],
       [1.03827377, 1.08721736, 1.01610153],
       [1.03351438, 1.03571998, 1.07235829],
       [0.96509544, 1.13190252, 1.04459469],
       [0.91385866, 1.25514815, 0.97258584],
       [1.02940037, 0.99528882, 1.11690346],
       [1.08939461, 1.03310905, 1.01908899],
       [1.23242398, 0.8973747 , 1.01179398],
       [0.99622017, 1.08322241, 1.06215007],
       [1.00527282, 1.10739785, 1.02892198],
       [1.1417768 , 0.9852616 , 1.01455426],
       [1.01554138, 0.94112822, 1.18492306],
       [0.98284914, 1.13927717, 1.01946634],
       [1.2206041 , 0.97470796, 0.9462806 ],
       [1.07628259, 1.10251602, 0.96279405],
       [1.77370327, 0.58528184, 0.78260754],
       [0.75386731, 1.25034322, 1.13738213],
       [1.09096891, 1.0871966 , 0.96342714],
       [1.07727253, 0.99311665, 1.07120347],
       [1.11363603, 1.01114355, 1.01681308],
       [0.98968949, 1.04468751, 1.10721565],
       [1.07672151, 1.01074731, 1.05412383],
       [1.06436668, 0.99722506, 1.08000091],
       [1.11899737, 0.97240685, 1.05018843],
       [0.95076979, 1.17795797, 1.01286489],
       [0.8421634 , 1.09785093, 1.20157832],
       [1.0410623 , 1.01157539, 1.08895496],
       [0.81759957, 1.18207259, 1.1419205 ],
       [0.96303075, 0.94303432, 1.23552759],
       [0.90149638, 1.1295803 , 1.11051597],
       [1.19032846, 0.93400531, 1.01725888],
       [1.03904377, 0.99809385, 1.10445503],
       [1.01857307, 1.09190045, 1.03111914],
       [1.05342437, 1.02089766, 1.06727062],
       [1.10427643, 1.08702536, 0.95029086],
       [1.11405631, 0.97107779, 1.05645855],
       [1.14853939, 0.9937719 , 0.99928136],
       [1.14143473, 1.02522505, 0.97493287],
       [0.95276563, 1.13683036, 1.05199666],
       [1.00218315, 1.08057601, 1.05883349],
       [1.12241828, 1.01485043, 1.00432394],
       [1.08668049, 0.86842643, 1.18648574],
       [1.07138779, 1.09160264, 0.97860222],
       [1.10593818, 0.91227734, 1.12337713],
       [0.99933658, 1.1727172 , 0.96953887],
       [1.0959832 , 1.07006464, 0.97554481],
       [1.26669764, 1.00281652, 0.87207849],
       [1.06404485, 1.07066617, 1.00688162],
       [1.04874557, 1.04043559, 1.0524115 ],
       [1.06338447, 1.03847476, 1.03973343],
       [1.06344307, 1.01429542, 1.06385416],
       [1.00057914, 1.0740475 , 1.06696601],
       [1.04309393, 1.0189099 , 1.07958883],
       [1.01578257, 1.04946031, 1.07634977],
       [1.0049055 , 1.08281672, 1.05387043],
       [1.04144688, 1.08561735, 1.01452843],
       [1.0668831 , 1.0400576 , 1.03465195],
       [1.0997679 , 0.9986759 , 1.04314886],
       [0.98311227, 1.06595355, 1.09252684],
       [1.04802246, 1.06452935, 1.02904084],
       [1.06910404, 0.97006686, 1.10242175],
       [1.05010767, 1.06821256, 1.02327242],
       [1.00349494, 1.03744137, 1.10065633],
       [1.00766903, 1.11052473, 1.02339889],
       [1.02259991, 1.06396326, 1.05502949],
       [1.04153189, 1.02846094, 1.07159983],
       [1.00223063, 1.06342138, 1.07594064],
       [1.0308651 , 1.08090827, 1.02981928],
       [1.05786533, 1.08070799, 1.00301933],
       [1.08082412, 0.98866935, 1.07209919],
       [1.0710613 , 1.02100579, 1.04952556],
       [1.0161708 , 1.06809531, 1.05732654],
       [1.04444793, 0.98660005, 1.11054468],
       [1.19230727, 0.96116192, 0.98812346],
       [1.04470276, 1.05747063, 1.03941926],
       [1.02939953, 1.0753625 , 1.03683062],
       [0.92788935, 1.04349464, 1.17020866],
       [1.073407  , 0.88768875, 1.18049691],
       [1.11904312, 1.0025725 , 1.01997703],
       [1.05006771, 1.01331166, 1.07821328],
       [1.02206133, 1.13913337, 0.98039795],
       [0.97578719, 1.01643984, 1.14936562],
       [1.05401042, 0.97230532, 1.11527691],
       [0.90125356, 0.88312733, 1.35721176],
       [0.96770577, 0.90254294, 1.27134395],
       [1.21046936, 1.14704994, 0.78407335],
       [1.10237635, 0.99216978, 1.04704652],
       [1.02564226, 1.06414681, 1.05180359],
       [1.05969737, 1.05542369, 1.0264716 ],
       [1.02730935, 1.03155579, 1.08272751],
       [1.08859144, 1.00059449, 1.05240673],
       [0.98579044, 1.05213764, 1.10366458],
       [1.08596781, 1.03901364, 1.0166112 ],
       [1.09324622, 1.03060882, 1.01773762],
       [1.24634662, 0.93699166, 0.95825437],
       [0.88430142, 1.16133049, 1.09596074],
       [1.07190809, 1.08623847, 0.9834461 ],
       [1.10533043, 0.92239472, 1.1138675 ],
       [1.08609743, 0.86037401, 1.19512121],
       [1.11315888, 1.09411883, 0.93431494],
       [1.06049424, 1.07666836, 1.00443006],
       [1.05790703, 0.99632362, 1.08736201],
       [1.12714015, 1.08408684, 0.93036566],
       [1.0066135 , 1.00869769, 1.12628147],
       [1.05101553, 1.06642249, 1.02415464],
       [1.05868168, 1.06671277, 1.0161982 ],
       [0.97844989, 1.07536635, 1.08777641],
       [1.01210334, 1.19718463, 0.93230468],
       [0.96226496, 0.95985668, 1.21947102],
       [1.00915325, 0.98140316, 1.15103625],
       [1.01072413, 1.04420829, 1.08666023],
       [1.07273691, 1.12216708, 0.94668866],
       [0.93077774, 1.07880309, 1.13201182],
       [1.08477226, 0.98380294, 1.07301746],
       [1.19195935, 0.90482054, 1.04481276],
       [0.96860508, 1.08882035, 1.08416723],
       [0.89802412, 1.1370015 , 1.10656704],
       [1.05803948, 1.04436683, 1.03918634],
       [0.98500992, 1.16123247, 0.99535026],
       [1.13938308, 0.99838542, 1.00382416],
       [1.04365459, 1.03472819, 1.06320987],
       [1.04694226, 1.05704804, 1.03760236],
       [1.05405024, 1.05193006, 1.03561235],
       [1.07129887, 0.98842509, 1.08186869],
       [1.05933548, 1.02702482, 1.05523236],
       [1.00742451, 1.03917709, 1.09499105],
       [1.07608373, 1.01164687, 1.05386206],
       [1.10415633, 0.98353332, 1.053903  ],
       [1.06467868, 1.08673413, 0.99017984],
       [1.07757201, 1.09317776, 0.97084289],
       [1.0615073 , 1.04764174, 1.03244362],
       [1.07771394, 1.08308035, 0.98079836],
       [1.07471492, 1.03845245, 1.02842529],
       [1.01726074, 1.07416198, 1.05016993],
       [1.03744478, 1.05324465, 1.05090323],
       [1.00887317, 1.06180134, 1.07091815],
       [1.03896539, 1.03669202, 1.06593524],
       [1.00442412, 1.06081282, 1.07635571],
       [1.00162052, 1.06047854, 1.07949359],
       [1.03326822, 1.07893712, 1.02938731],
       [1.05035053, 1.0683402 , 1.02290193],
       [0.91183348, 1.20842422, 1.02133496],
       [0.92761656, 1.16721754, 1.04675855],
       [1.00330048, 1.01981936, 1.11847281],
       [1.05283272, 0.98173084, 1.10702909],
       [0.86409319, 1.3408533 , 0.93664616],
       [1.06144256, 1.07362419, 1.0065259 ],
       [0.99702331, 1.08947584, 1.0550935 ],
       [1.1055753 , 0.98926457, 1.04675278],
       [1.05136789, 1.00166685, 1.08855791],
       [1.09810513, 1.05868687, 0.98480065],
       [1.10773513, 1.03178038, 1.00207715],
       [0.67107754, 1.10450709, 1.36600802],
       [1.13535041, 0.99268642, 1.01355582],
       [1.0271283 , 1.00486093, 1.10960342],
       [0.98878443, 1.07731525, 1.07549297],
       [1.04108835, 1.0101988 , 1.0903055 ],
       [1.05338241, 1.04602979, 1.04218045],
       [1.03143353, 1.06308619, 1.04707293],
       [1.01817979, 1.05805675, 1.06535611],
       [1.07377398, 1.08439496, 0.98342371],
       [1.13655135, 0.91529865, 1.08974265],
       [1.05858473, 1.07492426, 1.00808366],
       [0.99690485, 1.05268996, 1.09199784],
       [1.19434028, 0.9596468 , 0.98760558],
       [1.15707598, 1.02999936, 0.95451732],
       [1.02338384, 0.85949424, 1.25871457],
       [0.99134963, 1.07021247, 1.08003056],
       [1.09256403, 0.88602394, 1.16300468],
       [0.89691515, 1.14720232, 1.09747519],
       [0.86884866, 1.21367297, 1.05907102],
       [1.18513526, 0.84569383, 1.11076356],
       [0.96217633, 1.03839419, 1.14102214],
       [1.05374129, 1.14806975, 0.93978162],
       [1.1306175 , 0.92511715, 1.085858  ],
       [1.05823503, 1.0528545 , 1.03050312],
       [1.05116137, 1.02736515, 1.06306613],
       [0.99276568, 1.06309822, 1.08572875],
       [0.99043571, 1.07037236, 1.08078458],
       [1.12072256, 0.94049629, 1.08037381],
       [0.94111708, 0.99497436, 1.20550122],
       [0.9422942 , 1.13615515, 1.0631433 ],
       [1.11244686, 1.05255513, 0.97659067],
       [1.03808457, 1.06947661, 1.03403147],
       [0.96285471, 1.03757736, 1.14116058],
       [1.09703215, 1.03970958, 1.00485092],
       [1.02684203, 1.04090058, 1.07385004],
       [1.04138393, 0.9874509 , 1.11275782],
       [1.02347198, 1.09214248, 1.02597819],
       [0.86925763, 1.18201978, 1.09031524],
       [0.99302932, 1.00994841, 1.13861492],
       [1.05830747, 1.00636059, 1.07692459],
       [1.00990797, 1.0807853 , 1.05089938],
       [1.07825553, 1.05185777, 1.01147935],
       [1.06022408, 0.99076211, 1.09060646],
       [1.03505965, 0.99693667, 1.10959633],
       [1.12938645, 1.02676809, 0.98543811],
       [1.0385881 , 1.03445467, 1.06854988],
       [0.96775503, 1.02720359, 1.14663403],
       [1.11836842, 0.95854491, 1.06467933],
       [1.03786585, 1.05838779, 1.04533902],
       [1.0553871 , 1.03600309, 1.05020247],
       [0.94701349, 1.19304254, 1.00153663],
       [1.06280244, 0.99849265, 1.08029757],
       [1.0464942 , 1.05786564, 1.03723282],
       [1.03889772, 1.06668852, 1.03600641],
       [1.06090224, 1.05768321, 1.0230072 ],
       [1.04820092, 1.08253693, 1.0108548 ],
       [1.07745802, 1.00345306, 1.06068158],
       [1.05950899, 1.03902242, 1.04306124],
       [1.05002837, 1.06156088, 1.0300034 ],
       [0.97846442, 1.26626847, 0.89685976],
       [1.02102532, 0.99567472, 1.12489262],
       [1.05086444, 1.01691281, 1.0738154 ],
       [0.56182281, 0.87086261, 1.70890724],
       [1.05055564, 0.94831554, 1.14272148],
       [1.04225822, 0.99974215, 1.09959228],
       [1.03389558, 1.06311498, 1.04458209],
       [1.10103561, 1.07154014, 0.9690169 ],
       [1.00426558, 1.06748876, 1.06983832],
       [1.08206517, 1.05561211, 1.00391537],
       [0.97532753, 1.01171824, 1.15454689],
       [1.77558259, 0.82774437, 0.5382657 ],
       [0.9458884 , 1.13275126, 1.06295299],
       [1.08682097, 1.00926247, 1.04550921],
       [1.06400147, 1.03664466, 1.04094652],
       [1.04471139, 1.07158425, 1.02529701],
       [1.04231007, 1.04945051, 1.04983208],
       [1.04867753, 1.03385707, 1.05905806],
       [1.05433455, 1.0216698 , 1.0655883 ],
       [1.00836497, 1.05423186, 1.07899583],
       [1.0232295 , 1.01158849, 1.10677467],
       [1.06875885, 1.01739034, 1.05544347],
       [1.09183192, 1.02931741, 1.02044333],
       [1.13393054, 0.90108443, 1.10657768],
       [1.12337999, 0.92139726, 1.09681541],
       [1.24666403, 1.06316624, 0.83176238],
       [0.97787458, 1.04781503, 1.11590304],
       [1.14621534, 1.07542967, 0.91994765],
       [1.14955712, 1.09399391, 0.89804162],
       [1.24598589, 1.06344345, 0.83216332],
       [1.10777724, 0.89785828, 1.13595713],
       [1.34855149, 1.01029455, 0.78274661],
       [0.99967602, 0.85530872, 1.28660792],
       [0.9650639 , 1.20985985, 0.9666689 ],
       [1.04574591, 1.02795231, 1.06789443],
       [1.02370449, 1.10706211, 1.01082605],
       [1.01387086, 1.03987056, 1.08785123],
       [1.13454279, 0.98306246, 1.0239874 ],
       [1.07500169, 1.01812692, 1.04846405],
       [1.09677081, 1.05316934, 0.99165251],
       [0.98221306, 1.28491134, 0.87446826],
       [1.21885298, 0.94889816, 0.97384151],
       [0.98092256, 1.04497428, 1.11569581],
       [1.04614983, 1.04767077, 1.04777206],
       [1.07626818, 1.04109943, 1.02422504],
       [1.05045313, 1.09735579, 0.99378373],
       [1.09974871, 1.03300964, 1.0088343 ],
       [1.08547085, 1.05691976, 0.99920204],
       [1.09444023, 1.01906743, 1.028085  ],
       [0.7469667 , 0.87022191, 1.52440405],
       [0.99781768, 1.1813401 , 0.96243487],
       [1.07941822, 1.05282511, 1.00934932],
       [1.04537885, 1.0085585 , 1.0876553 ],
       [0.96818647, 1.09290072, 1.08050546],
       [0.99068735, 1.11663428, 1.03427102],
       [1.0291931 , 1.05197286, 1.0604267 ],
       [0.97763119, 0.98653417, 1.17742729],
       [1.06908137, 1.04659732, 1.02591396],
       [1.04100884, 1.15447128, 0.94611253],
       [1.0890491 , 1.09352941, 0.95901414],
       [1.03218097, 1.06767844, 1.04173324],
       [1.19326334, 0.83001545, 1.11831386],
       [1.00366558, 1.02505425, 1.11287282],
       [1.02024267, 1.05723253, 1.06411746],
       [1.03600327, 1.04076652, 1.06482286],
       [1.05440462, 1.03501242, 1.05217561],
       [1.01534444, 1.0475955 , 1.07865271],
       [1.00591512, 1.06460446, 1.07107308],
       [1.04133736, 1.02199766, 1.07825763],
       [1.04600184, 1.03832297, 1.05726784],
       [1.23995229, 0.86818851, 1.03345185],
       [1.10707251, 1.06605427, 0.96846588],
       [0.99597333, 1.07407103, 1.07154829],
       [0.97403264, 1.22394176, 0.94361826],
       [1.06583721, 1.04544654, 1.03030891],
       [1.04727709, 1.02657732, 1.06773825],
       [0.99676562, 1.04867027, 1.09615676],
       [1.04959154, 1.03560915, 1.05639196],
       [1.01731443, 1.02894124, 1.09533698],
       [1.10008224, 1.03298625, 1.00852417],
       [1.01835611, 1.07059959, 1.05263695],
       [1.20498634, 1.32881909, 0.60778722],
       [1.18530609, 1.02066806, 0.9356185 ],
       [1.02755292, 1.13184879, 0.98219094],
       [1.01724581, 1.10233023, 1.02201662],
       [1.07670743, 1.04865885, 1.01622637],
       [1.02896836, 1.04655346, 1.06607084],
       [1.04285395, 1.03141629, 1.06732241],
       [1.03357139, 1.04140743, 1.06661383],
       [1.0788025 , 1.07031983, 0.99247032],
       [1.09837448, 1.12719181, 0.91602636],
       [0.75713242, 1.1940063 , 1.19045393],
       [1.04489507, 1.07239532, 1.02430227],
       [1.01080838, 1.03716923, 1.09361504],
       [1.10371505, 0.98379447, 1.05408313],
       [1.08594219, 1.12511221, 0.93053825],
       [0.90417398, 1.21226912, 1.02514955],
       [1.06855853, 1.08733373, 0.9857004 ],
       [0.85461366, 1.16055972, 1.12641927],
       [1.03556355, 1.10419404, 1.00183507],
       [0.92551856, 1.16192358, 1.05415052],
       [1.19119858, 0.85768752, 1.09270655],
       [1.14177783, 1.09931801, 0.90049681],
       [1.02784716, 0.86200921, 1.25173629],
       [0.8942479 , 1.0774697 , 1.16987505],
       [1.16548674, 0.77948025, 1.19662567],
       [0.81014983, 1.21353989, 1.11790294],
       [0.92675939, 1.03526285, 1.17957042],
       [1.0739818 , 1.10328341, 0.96432745],
       [1.02736684, 1.07354694, 1.04067888],
       [1.03379791, 1.10764425, 1.00015049],
       [1.05459131, 1.04906757, 1.03793377],
       [0.99610909, 1.11054531, 1.03493825],
       [1.02024096, 1.10041607, 1.02093563],
       [1.07552457, 0.87263845, 1.19342963],
       [1.01644277, 1.15925601, 0.96589387],
       [1.08847923, 0.96803841, 1.08507501],
       [1.05719517, 1.06586168, 1.01853581],
       [1.03185868, 1.02766804, 1.08206593],
       [1.06493433, 1.01194861, 1.06470972],
       [1.08208348, 1.03221729, 1.02729188],
       [1.10256028, 1.19190287, 0.8471295 ],
       [0.78002001, 1.52640293, 0.83516972],
       [0.95957253, 1.0598205 , 1.12219963],
       [1.04124134, 1.02408697, 1.07626434],
       [1.0041938 , 1.07893207, 1.05846679],
       [1.09356653, 1.06366628, 0.98435985],
       [0.91072677, 1.16409272, 1.06677316],
       [1.03350973, 0.99489734, 1.11318558],
       [0.77325514, 1.21669589, 1.15164163],
       [1.02555835, 1.10195398, 1.01408032],
       [1.08452962, 1.17611193, 0.8809511 ],
       [1.10970252, 1.09509898, 0.93679116],
       [1.03144014, 1.05584349, 1.05430902],
       [1.04859541, 1.04527804, 1.0477192 ],
       [1.01266626, 1.07133922, 1.05758717],
       [1.05980055, 1.08041851, 1.00137359],
       [1.12387812, 0.99280479, 1.02490974],
       [1.02609633, 1.09350882, 1.0219875 ],
       [1.05869388, 1.08365305, 0.99924573],
       [1.06912258, 1.0375784 , 1.03489168],
       [1.07039043, 1.03640626, 1.03479596],
       [1.02888128, 0.96427797, 1.1484334 ],
       [1.08347931, 1.05844276, 0.99967058],
       [1.04564119, 1.04809974, 1.04785172],
       [1.06494519, 1.08741485, 0.98923262],
       [1.03056671, 1.05659651, 1.05442943],
       [1.04635543, 1.04035243, 1.0548848 ],
       [0.96875199, 1.12190992, 1.05093074],
       [1.03711895, 1.02905289, 1.07542082],
       [1.01067843, 1.06743864, 1.06347559],
       [1.0772121 , 1.03985035, 1.0245302 ],
       [1.08047748, 0.99965391, 1.06146126],
       [0.99563093, 1.10376041, 1.04220132],
       [0.82504052, 1.15018096, 1.16637117],
       [1.25585538, 0.9416354 , 0.94410188],
       [1.01450468, 1.02547022, 1.10161775],
       [1.03217238, 1.07320983, 1.03621044],
       [1.03986324, 1.04835252, 1.05337689],
       [1.04787864, 1.04653576, 1.04717826],
       [1.08181033, 1.04150719, 1.01827513],
       [1.144481  , 1.03343437, 0.96367728],
       [0.70617212, 0.95737323, 1.4780473 ],
       [1.76518354, 0.76281388, 0.61359523],
       [1.04967253, 1.01790174, 1.07401838],
       [1.0274496 , 1.08871157, 1.02543148],
       [1.27713922, 0.86058779, 1.00386564],
       [1.11449583, 1.07192146, 0.95517537],
       [1.13685328, 1.10022166, 0.90451771],
       [1.12123267, 0.98342572, 1.03693427],
       [0.88412989, 1.10877235, 1.14869041],
       [1.11407559, 1.00534744, 1.02216963],
       [1.12257476, 1.05725256, 0.96176534],
       [1.13301321, 0.97659387, 1.03198557],
       [0.92188441, 1.13332199, 1.08638625],
       [1.24738222, 1.00028182, 0.89392861],
       [1.13445102, 0.94167952, 1.06546212],
       [1.23253641, 1.00497365, 0.90408259],
       [1.69665747, 0.7524093 , 0.69252588],
       [1.10339172, 1.08593193, 0.952269  ],
       [1.15490926, 0.98295176, 1.00373163],
       [1.03607007, 1.06975361, 1.03576898],
       [1.02810287, 1.07174671, 1.04174307],
       [1.10121644, 1.02598659, 1.01438962],
       [1.02437012, 1.01716998, 1.10005256],
       [1.00585267, 1.09027485, 1.04546514],
       [1.01441754, 1.0633548 , 1.06382031],
       [1.02920955, 0.97143741, 1.14094569],
       [0.89738091, 0.92075427, 1.32345747],
       [1.06487376, 1.03207535, 1.04464354],
       [1.03001044, 1.05019565, 1.06138656],
       [1.0504696 , 1.0475097 , 1.04361336],
       [1.04757093, 1.00020927, 1.09381245],
       [0.99678377, 1.05236509, 1.09244379],
       [1.13479423, 0.90348414, 1.10331429],
       [1.02766939, 1.0903827 , 1.02354056],
       [1.06616382, 1.05889748, 1.01653135],
       [0.97697027, 1.09782027, 1.06680212],
       [1.06733423, 1.11062334, 0.96363508],
       [1.02350749, 1.22836039, 0.88972477],
       [1.0620706 , 1.01371073, 1.06581133],
       [0.73723242, 0.75523144, 1.6491288 ],
       [1.11475031, 0.9227318 , 1.10411055],
       [0.96361962, 1.12048614, 1.05748689],
       [1.03984364, 1.03391411, 1.0678349 ],
       [1.01542196, 1.05473832, 1.07143237],
       [1.0620606 , 1.07498918, 1.00454288],
       [1.1064402 , 0.99694769, 1.03820476],
       [0.99085053, 1.07861739, 1.07212473],
       [1.02086441, 1.04361094, 1.07711731],
       [0.99498305, 1.06969025, 1.07691935],
       [1.03027781, 1.07091618, 1.04039866],
       [1.03800687, 1.05245463, 1.05113116],
       [1.25092675, 0.8689134 , 1.02175251],
       [1.03792316, 1.05531503, 1.04835446],
       [1.05664665, 1.03337332, 1.05157268],
       [1.03855908, 1.04688192, 1.05615165],
       [1.04517369, 1.03914738, 1.05727159],
       [1.07345778, 1.0667305 , 1.00140438],
       [1.09055356, 1.01882003, 1.03221907],
       [1.06012509, 0.99258122, 1.08888635],
       [1.10633171, 1.05495093, 0.98031001],
       [1.04671812, 0.98539037, 1.10948417],
       [1.07348092, 1.05662838, 1.01148335],
       [0.63337674, 1.2468489 , 1.26136701],
       [1.19373321, 1.02958592, 0.91827352],
       [0.9858884 , 1.13449436, 1.0212099 ],
       [1.03874589, 1.02058364, 1.08226312],
       [1.03655852, 1.0550242 , 1.05000994],
       [1.05145581, 1.03407775, 1.0560591 ],
       [1.04718133, 1.08314903, 1.01126229],
       [1.0109995 , 1.02506841, 1.10552474],
       [1.18726685, 1.15257493, 0.80175087],
       [1.66352889, 0.84349989, 0.63456387],
       [1.19667753, 0.98773011, 0.95718502],
       [0.77249957, 0.99268385, 1.37640923],
       [1.05782243, 1.04361703, 1.0401532 ],
       [1.00453995, 1.1571966 , 0.9798561 ],
       [1.02323611, 0.88140789, 1.23694865],
       [0.93101705, 1.05126643, 1.15930918],
       [1.09140567, 0.94263135, 1.10755563],
       [0.96825115, 1.1210958 , 1.0522457 ],
       [1.02384906, 1.02301248, 1.0947311 ],
       [1.14185994, 0.95818197, 1.04155074],
       [1.04489484, 0.9913107 , 1.10538712],
       [1.07875379, 1.1032867 , 0.95955217],
       [0.87883095, 1.05667582, 1.20608589],
       [1.00900533, 1.1762642 , 0.95632312],
       [1.02621681, 1.19636207, 0.91901378],
       [0.94393006, 0.91724403, 1.28041857],
       [0.95795178, 1.05159018, 1.13205069],
       [1.09912722, 0.90917231, 1.13329312],
       [0.95197294, 1.07292697, 1.11669274],
       [0.8868743 , 1.44493843, 0.80977993],
       [0.82317191, 1.09603293, 1.22238782],
       [1.14259652, 0.97291577, 1.02608036],
       [1.08009466, 1.00707668, 1.05442131],
       [1.03267133, 1.04716542, 1.06175591],
       [1.09337149, 1.04657551, 1.00164565],
       [1.02860793, 1.08393964, 1.02904508],
       [1.04891032, 1.03615579, 1.05652655],
       [1.01109676, 1.12026774, 1.01022815],
       [0.86107474, 1.37551813, 0.90499979],
       [0.92236084, 1.01400166, 1.20523016],
       [1.11937222, 1.01451406, 1.00770637],
       [1.02237065, 1.07163849, 1.04758351],
       [1.07584571, 1.05763849, 1.00810846],
       [0.97076513, 1.05372152, 1.117106  ],
       [1.1209991 , 0.98868115, 1.03191241],
       [1.083572  , 1.03835211, 1.01966855],
       [1.0702571 , 0.99057676, 1.08075879],
       [1.050714  , 0.99634494, 1.09453372],
       [1.10876122, 0.98593228, 1.04689915],
       [0.82269991, 1.50214382, 0.81674893],
       [0.90760596, 1.15533311, 1.07865358],
       [1.05469778, 0.9625375 , 1.12435737],
       [1.03132389, 1.05830041, 1.05196835],
       [0.98456926, 1.02192058, 1.13510281],
       [1.07858653, 1.00707963, 1.05592649],
       [0.94681437, 0.98908273, 1.20569555],
       [1.20171748, 0.96120292, 0.97867225],
       [1.0347119 , 1.11857637, 0.98830438],
       [1.05922387, 1.08367075, 0.99869803],
       [1.04019855, 1.0328853 , 1.0685088 ],
       [1.03622305, 1.0755506 , 1.029819  ],
       [1.00605535, 1.12907548, 1.00646183],
       [1.08130091, 0.99919238, 1.06109936],
       [1.0521441 , 1.04737072, 1.04207783],
       [1.02214013, 1.07964908, 1.03980344],
       [1.02522119, 1.06655641, 1.04981505],
       [0.98985397, 1.11397807, 1.03776061],
       [1.04200716, 1.02435317, 1.07523233],
       [1.02541898, 1.10245047, 1.0137232 ],
       [1.00105037, 1.08410582, 1.05643646],
       [1.11654057, 1.0249677 , 1.00008438],
       [1.0016371 , 1.12970631, 1.01024924],
       [0.87313871, 1.41591124, 0.8525427 ],
       [0.92577983, 1.17883144, 1.03698138],
       [1.11040889, 1.01285417, 1.0183296 ],
       [1.04842964, 1.08332627, 1.00983674],
       [1.02271838, 1.06204494, 1.05682933],
       [1.0381826 , 1.04776494, 1.05564512],
       [1.01825344, 1.03573961, 1.0875996 ],
       [1.11467861, 1.04593292, 0.98098112],
       [0.93858571, 1.16668932, 1.03631762],
       [1.32962433, 1.01580682, 0.7961615 ],
       [0.81850419, 1.11607007, 1.20701839],
       [0.94823083, 1.02287023, 1.17049159],
       [1.03496154, 1.01871655, 1.08791457],
       [0.99546637, 1.01578794, 1.13033834],
       [0.91785395, 1.10275189, 1.12098682],
       [1.14183051, 1.00537707, 0.99438507],
       [1.07558165, 1.13154053, 0.93447048],
       [1.08278045, 1.11045384, 0.94835836],
       [1.11896102, 1.01710868, 1.00552296],
       [1.04567954, 1.14628857, 0.94962455],
       [1.02076776, 0.9750357 , 1.14578919],
       [1.12367178, 1.00936121, 1.00855967],
       [0.98658643, 1.13118554, 1.02382068],
       [1.06551972, 0.91206469, 1.16400825],
       [0.97701868, 1.00022145, 1.16435253],
       [1.00420538, 0.83491663, 1.30247064],
       [1.01615713, 1.11984573, 1.00558979],
       [0.86087609, 1.22427006, 1.0564465 ],
       [1.08904224, 0.97171626, 1.08083415],
       [1.07155892, 0.98028997, 1.08974377],
       [0.95346727, 1.18918278, 0.9989426 ],
       [1.02419162, 1.00603141, 1.11136962],
       [1.02194842, 1.06078419, 1.05886004],
       [1.07986291, 1.04550873, 1.01622101],
       [1.04196387, 1.00965425, 1.08997453],
       [1.02993271, 1.10643132, 1.00522862],
       [1.10315934, 1.00800582, 1.0304275 ],
       [0.99880392, 0.99163248, 1.15115626],
       [0.99805388, 0.94583355, 1.19770522],
       [0.98868852, 1.14943719, 1.00346695],
       [1.10769781, 1.01078629, 1.02310855],
       [1.02992932, 1.0368164 , 1.07484694],
       [1.01024514, 1.08256354, 1.04878398],
       [1.09490324, 1.0312521 , 1.01543732],
       [0.99326279, 1.08109347, 1.0672364 ],
       [0.99177682, 1.09765954, 1.0521563 ],
       [0.97428106, 1.0325137 , 1.1347979 ],
       [0.96939093, 0.97286905, 1.19933268],
       [0.81546826, 1.16227219, 1.16385221],
       [1.14506805, 1.02866985, 0.96785475],
       [1.08365667, 0.98853106, 1.06940493],
       [1.04392321, 1.03755113, 1.06011831],
       [1.04632175, 1.01392998, 1.08134092],
       [0.97736409, 1.17498702, 0.98924155],
       [1.12555232, 1.1527893 , 0.86325103],
       [0.98596028, 1.07777362, 1.07785875],
       [1.06024598, 1.00288533, 1.07846134],
       [1.04252437, 1.04241919, 1.05664909],
       [1.03512821, 1.06854707, 1.03791737],
       [1.1865814 , 0.94906315, 1.00594811],
       [1.09197784, 1.06523287, 0.98438194],
       [1.03746052, 1.04325155, 1.06088058],
       [1.0666049 , 1.05618615, 1.0188016 ],
       [1.03329697, 1.05310846, 1.05518722],
       [1.03406175, 1.02306207, 1.08446884],
       [1.05860745, 1.0676937 , 1.0152915 ],
       [1.00841967, 1.06097624, 1.07219675],
       [1.08389538, 1.04947912, 1.00821815],
       [1.01151536, 1.09132161, 1.03875569],
       [1.07171906, 1.04383647, 1.02603712],
       [1.03843735, 0.92359055, 1.17956475],
       [1.03357287, 0.9164273 , 1.19159248],
       [0.98092722, 1.11578364, 1.0448818 ],
       [1.08060813, 1.03246414, 1.02852038],
       [1.05835046, 1.03962602, 1.04361617],
       [0.99516839, 1.06048428, 1.08593998],
       [1.05251259, 1.05423301, 1.03484706],
       [1.0126291 , 1.14532624, 0.98363731],
       [0.93151935, 1.09112881, 1.1189445 ],
       [0.87128641, 1.33107606, 0.93923018],
       [0.82031089, 1.20346   , 1.11782177],
       [0.97413557, 1.10459352, 1.06286357],
       [1.26340297, 0.93966643, 0.93852326],
       [1.12758075, 1.0283309 , 0.98568101],
       [1.01811835, 1.08160594, 1.04186836],
       [1.18536812, 0.96311283, 0.9931117 ],
       [0.95421749, 1.05851944, 1.12885573],
       [1.00247617, 1.16715058, 0.9719659 ],
       [0.98180385, 1.03899616, 1.12079264],
       [0.97451443, 1.07918865, 1.08788956],
       [1.0103834 , 1.05597364, 1.07523561],
       [1.10298306, 1.05286106, 0.98574853],
       [1.02694922, 0.96563152, 1.14901191],
       [0.99223232, 1.09634674, 1.05301359],
       [1.0112011 , 1.04196232, 1.08842922],
       [1.22704486, 1.01072104, 0.90382676],
       [0.95551277, 0.95820983, 1.22787006],
       [1.05685721, 0.93235016, 1.15238528],
       [1.1912019 , 0.96558704, 0.98480372],
       [1.20105736, 1.25924273, 0.68129256],
       [1.12868666, 1.00011497, 1.01279103],
       [1.04737788, 1.11449083, 0.97972395],
       [1.01635659, 1.02902345, 1.09621261],
       [1.03224029, 0.99675514, 1.11259722],
       [1.0676392 , 1.03654028, 1.03741317],
       [1.1011531 , 1.04015719, 1.00028236],
       [1.01753393, 1.02061855, 1.10344017],
       [1.04920702, 1.07551748, 1.01686816],
       [1.03915688, 1.05519509, 1.04724068],
       [1.05495184, 1.02536344, 1.06127737],
       [1.08881733, 1.02942489, 1.02335044],
       [1.01016402, 1.0579632 , 1.07346544],
       [1.09105474, 1.06816288, 0.98237503],
       [1.09126823, 1.03273913, 1.01758529],
       [1.0714404 , 1.05805864, 1.01209362],
       [1.05229728, 1.02154269, 1.06775268],
       [1.06570531, 1.04371345, 1.0321739 ],
       [1.0251379 , 1.12728902, 0.98916573],
       [1.03578148, 1.07662525, 1.02918593],
       [1.04629234, 1.03847845, 1.05682187],
       [1.10489119, 0.98064518, 1.05605628],
       [0.99512403, 1.00405772, 1.1424109 ],
       [1.13348724, 1.0346769 , 0.97342852],
       [1.09143463, 1.05871335, 0.99144468],
       [1.2696599 , 0.93445083, 0.93748193],
       [1.50646053, 0.79635337, 0.83877875],
       [1.05102508, 1.12258784, 0.96797973],
       [1.01754634, 1.05477217, 1.06927415],
       [1.01967539, 1.04658251, 1.07533475],
       [1.13203396, 0.96018245, 1.04937624],
       [0.67456578, 1.36600397, 1.1010229 ],
       [1.11178098, 1.03930423, 0.99050744],
       [1.06656915, 1.05563115, 1.01939236],
       [1.04274675, 1.02110529, 1.07774061],
       [1.09447012, 1.01261707, 1.03450546],
       [1.03373516, 1.01605572, 1.09180177],
       [1.08990189, 1.03184156, 1.0198492 ],
       [1.04551616, 1.02436615, 1.07171033],
       [1.09929015, 1.02381049, 1.01849201],
       [1.00950314, 1.02174605, 1.11034346],
       [1.03033008, 1.1012266 , 1.01003597],
       [1.3029441 , 0.7207941 , 1.11785445],
       [0.98166118, 1.04295744, 1.11697403],
       [1.02536967, 1.01897698, 1.097246  ],
       [1.00071655, 1.06039071, 1.08048539],
       [1.06041639, 1.01714019, 1.06403607],
       [1.06469687, 1.06550506, 1.01139073],
       [1.04155342, 1.05187227, 1.04816696],
       [0.97639206, 1.04025735, 1.12494325],
       [1.02410983, 0.99390696, 1.12357586],
       [0.78691684, 1.08873305, 1.26594276],
       [0.8084697 , 1.33002693, 1.00309602],
       [1.08626643, 0.99102367, 1.06430255],
       [1.0788885 , 0.95369194, 1.10901222],
       [1.00570364, 1.08528537, 1.05060364],
       [0.88467751, 1.13335144, 1.1235637 ],
       [1.01210532, 1.08041491, 1.04907243],
       [1.01143334, 0.9610513 , 1.16910801],
       [1.01133401, 1.12741379, 1.00284485],
       [0.96824226, 1.1427867 , 1.0305637 ],
       [0.96100461, 1.05511193, 1.12547611],
       [0.96827527, 1.07951688, 1.0938005 ],
       [0.98661679, 1.03733235, 1.11764351],
       [1.02124065, 0.98060555, 1.13974645],
       [0.98535557, 0.98400475, 1.17223233],
       [1.1593588 , 1.02918792, 0.95304593],
       [1.02990572, 0.95093053, 1.1607564 ],
       [1.11582391, 0.90611739, 1.11965135],
       [0.9628364 , 1.16761339, 1.01114286],
       [0.87218098, 1.1694697 , 1.09994197],
       [1.16265785, 1.02214098, 0.95679383],
       [0.60425923, 1.46244542, 1.074888  ],
       [1.04114093, 0.97493318, 1.12551854],
       [0.93413715, 1.09598808, 1.11146742],
       [1.24765209, 0.93443717, 0.9595034 ],
       [1.13803108, 0.98319329, 1.02036828],
       [1.05000335, 1.07540745, 1.01618185],
       [1.0727548 , 0.984782  , 1.08405586],
       [1.09565794, 1.02879793, 1.01713679],
       [1.06419462, 1.04979434, 1.02760369],
       [1.03808385, 1.05702634, 1.04648247],
       [1.0338247 , 1.05862787, 1.04914008],
       [1.01214459, 1.06766453, 1.06178354],
       [1.08017826, 0.97506886, 1.08634553],
       [0.99114407, 1.2306516 , 0.91979699],
       [0.97681707, 1.0154922 , 1.14928338],
       [1.01011155, 1.1002621 , 1.03121901],
       [1.02949592, 1.08605945, 1.02603729],
       [1.04610292, 1.04911018, 1.04637956],
       [1.06486453, 1.04652385, 1.03020428],
       [1.04418471, 1.07238245, 1.0250255 ],
       [1.03933848, 1.09957335, 1.00268082],
       [1.07386812, 0.98081915, 1.08690538],
       [1.02169072, 1.03276165, 1.08714028],
       [1.19794039, 1.15661511, 0.78703715],
       [1.56241561, 0.81054877, 0.76862827],
       [1.12567659, 1.12635882, 0.88955724],
       [1.0632459 , 0.97159211, 1.10675465],
       [1.01420316, 1.0772006 , 1.05018889],
       [0.99664484, 1.07119278, 1.07375504],
       [1.64733431, 0.85945311, 0.63480523],
       [1.08406879, 1.01067519, 1.04684868],
       [1.0687735 , 1.06409917, 1.00871999],
       [1.04128653, 1.06372374, 1.03658239],
       [1.08744958, 0.9851944 , 1.06894868],
       [1.02666241, 1.061301  , 1.05362924],
       [1.05946694, 1.04993502, 1.03219069],
       [1.05264078, 1.04333176, 1.04562011],
       [1.04961586, 1.04836604, 1.04361075],
       [1.05838267, 1.04465931, 1.03855067],
       [1.07267182, 0.9871802 , 1.08174063],
       [0.95440067, 1.49426826, 0.69292373],
       [1.00354459, 1.04081774, 1.09723032],
       [1.08645007, 1.05125975, 1.00388283],
       [1.02393723, 1.02926396, 1.08839146],
       [1.04829527, 1.05877527, 1.03452212],
       [1.08048805, 1.00809443, 1.05301017],
       [1.07001259, 1.09394476, 0.97763531],
       [1.02716757, 1.05831667, 1.05610841],
       [1.05731756, 1.16407788, 0.92019721],
       [0.86866779, 1.18601341, 1.08691145],
       [0.97191773, 1.22159026, 0.94808466],
       [0.93261314, 1.04794837, 1.16103114],
       [1.02767005, 1.0588082 , 1.0551144 ],
       [1.14202321, 0.98450445, 1.01506499],
       [1.0283744 , 1.11921953, 0.99399872],
       [1.0588768 , 1.02639295, 1.05632291],
       [1.03737714, 1.02924105, 1.07497446],
       [1.12624312, 1.00664836, 1.00870118],
       [1.15634773, 0.99035915, 0.99488578],
       [1.0437655 , 1.10294897, 0.99487819],
       [0.95851305, 1.14890583, 1.03417378],
       [1.07088752, 1.07802327, 0.99268186],
       [1.08276627, 1.04123603, 1.01759036],
       [0.97614182, 1.117975  , 1.04747584],
       [1.14665285, 1.03718291, 0.95775689],
       [1.12134393, 1.02944518, 0.99080355],
       [1.01674583, 1.15990464, 0.96494218],
       [0.82272962, 1.18103708, 1.13782595],
       [1.15969285, 1.03941558, 0.94248422],
       [0.94625292, 1.21722548, 0.97811426],
       [1.0972073 , 1.22921746, 0.81516789],
       [0.99349472, 0.96441441, 1.18368352],
       [0.74054788, 1.42609249, 0.97495228],
       [0.55553711, 0.65004825, 1.9360073 ],
       [1.09508771, 1.05537917, 0.99112577],
       [1.04240121, 1.08846956, 1.01072189],
       [0.98362924, 1.06831606, 1.08964736],
       [0.98308871, 1.07812368, 1.08038027],
       [1.05810769, 1.01460289, 1.06888207],
       [1.03654931, 1.03717125, 1.06787209],
       [1.05152552, 1.04216186, 1.04790527],
       [1.05741847, 0.99550693, 1.08866725],
       [0.83901305, 1.15101848, 1.15156112],
       [1.09895711, 0.98198458, 1.06065097],
       [1.08825304, 1.01438953, 1.03895008],
       [1.07308524, 1.04058351, 1.0279239 ],
       [1.04258594, 1.06455573, 1.03445099],
       [1.00855992, 1.08354975, 1.04948298],
       [1.05885948, 1.02937152, 1.05336165],
       [1.07361027, 0.95544921, 1.11253317],
       [0.89697257, 1.11703979, 1.1275803 ],
       [1.10221357, 0.96067295, 1.07870613],
       [1.00293349, 1.08071009, 1.05794907],
       [1.05306441, 1.04744923, 1.04107901],
       [1.0007572 , 1.14537497, 0.99546048],
       [1.18003394, 0.96219485, 0.99936386],
       [1.0270861 , 1.08692386, 1.02758269],
       [1.05910454, 0.99783884, 1.08464928],
       [1.12048608, 1.01157976, 1.00952681],
       [0.97658147, 1.02984908, 1.1351621 ],
       [1.02115537, 1.07318246, 1.04725483],
       [1.05126822, 1.04863772, 1.04168671],
       [1.06356316, 1.02789156, 1.05013793],
       [1.06394689, 1.06267686, 1.01496891],
       [0.95208774, 1.12857375, 1.06093117],
       [0.98007647, 1.05683236, 1.10468383],
       [1.0804425 , 0.96526874, 1.09588142],
       [0.9777552 , 1.01349793, 1.15033953],
       [1.11606708, 0.97926381, 1.04626176],
       [1.06436429, 1.08212291, 0.99510545],
       [1.06329096, 0.98969994, 1.08860176],
       [1.07436374, 1.02378927, 1.04343964],
       [1.08403862, 0.99809963, 1.0594544 ],
       [1.04674148, 1.06172314, 1.03312804],
       [0.88571866, 1.13997757, 1.11589642],
       [1.08074164, 1.10726044, 0.95359057],
       [0.92365067, 1.11874052, 1.09920147],
       [1.27268074, 0.89118071, 0.9777312 ],
       [0.97715991, 0.9758875 , 1.18854524],
       [1.03762485, 1.0344788 , 1.069489  ],
       [1.05490537, 0.98454439, 1.10214289],
       [1.11066253, 1.03671564, 0.99421449],
       [1.06817563, 1.05240079, 1.02101624],
       [1.1302915 , 1.01974617, 0.99155498],
       [1.10683322, 0.98329832, 1.05146112],
       [1.01007843, 0.96307398, 1.16844024],
       [0.98289595, 1.078311  , 1.0803857 ],
       [1.01211532, 1.1049757 , 1.02450164],
       [1.13848859, 1.03686043, 0.96624363],
       [0.96833287, 1.18166706, 0.99159273],
       [0.9554425 , 1.15481044, 1.03133971],
       [1.15495497, 0.99471543, 0.99192225],
       [1.20662857, 0.87787152, 1.05709256],
       [1.1487086 , 1.03415648, 0.95872757],
       [1.04637027, 0.91132591, 1.18389648],
       [0.89047463, 1.19242609, 1.05869193],
       [0.93785103, 1.15619234, 1.04754928],
       [1.23708058, 1.0020869 , 0.90242518],
       [1.07976734, 1.18654604, 0.87527927],
       [1.01274298, 1.05137572, 1.07747395],
       [1.03027417, 1.11541482, 0.99590366],
       [1.10203978, 1.09518702, 0.94436585],
       [1.04562987, 1.08572821, 1.01023457],
       [1.02886891, 1.04475242, 1.06797132],
       [1.02255199, 1.05199371, 1.06704695],
       [1.0508339 , 1.0625638 , 1.02819496],
       [0.97058568, 1.18139155, 0.98961543],
       [0.68711296, 0.82929742, 1.62518227],
       [0.95667112, 0.99472685, 1.19019468],
       [1.07707897, 1.02713072, 1.03738297],
       [0.98344325, 1.06343737, 1.09471204],
       [1.026694  , 1.10006193, 1.01483672],
       [1.08028902, 1.08533805, 0.97596559],
       [1.0455439 , 1.06302961, 1.03301914],
       [1.11247498, 1.08453527, 0.94458241],
       [1.09906794, 1.04519557, 0.99732914],
       [1.01424539, 1.01632738, 1.11101989],
       [1.04045515, 1.06649419, 1.0346433 ],
       [0.99361092, 1.05085411, 1.09712763],
       [0.9722148 , 1.1271632 , 1.04221465],
       [1.09569029, 0.99016271, 1.05573965],
       [1.14962349, 0.90456253, 1.08740664],
       [1.06800088, 1.08617755, 0.98741422],
       [1.0653386 , 0.99472442, 1.08152964],
       [1.11238614, 1.00567279, 1.02353372],
       [1.06975696, 1.11258551, 0.95925018],
       [0.61061422, 1.36531021, 1.16566823],
       [0.82657883, 1.14403505, 1.17097878],
       [1.06876115, 1.09409384, 0.97873766],
       [0.9886731 , 0.9826674 , 1.17025216],
       [0.99437522, 1.14494111, 1.00227632],
       [1.09759647, 1.0088255 , 1.03517068],
       [1.00242197, 1.04668843, 1.09248226],
       [1.00037326, 1.05136706, 1.08985233],
       [1.04243349, 1.08392037, 1.01523879],
       [0.9372013 , 1.12063122, 1.08376013],
       [1.07748408, 1.00899624, 1.05511233],
       [1.03135665, 0.92684411, 1.18339189],
       [1.01217273, 1.00162204, 1.12779789],
       [1.04968648, 1.06712562, 1.02478056],
       [1.00125551, 0.95743515, 1.182902  ],
       [1.00494707, 1.07846842, 1.05817716],
       [1.15403145, 1.031152  , 0.95640921],
       [0.98541856, 1.07298557, 1.08318852],
       [1.13681096, 1.0598492 , 0.9449325 ],
       [1.05734717, 0.92376809, 1.16047739],
       [1.05506812, 1.02814527, 1.05837926],
       [1.04140588, 0.98936078, 1.110826  ],
       [0.95162553, 1.15292483, 1.0370423 ],
       [1.09519184, 1.00161302, 1.04478779],
       [1.16841362, 1.04738577, 0.92579326],
       [1.06311194, 1.06164036, 1.01684035],
       [1.05646867, 0.99794588, 1.0871781 ],
       [0.98249557, 1.12246268, 1.0366344 ],
       [0.99658072, 0.9992601 , 1.14575184],
       [0.79076349, 1.14285017, 1.20797899],
       [1.21674307, 1.15953874, 0.76531084],
       [1.00399048, 1.06888787, 1.06871431],
       [1.0358581 , 0.90092872, 1.20480584],
       [1.19176246, 1.01676176, 0.93306843],
       [1.14510038, 1.06494408, 0.9315482 ],
       [0.7103389 , 0.5072072 , 1.92404656],
       [1.00774157, 0.82361246, 1.31023862],
       [0.88294943, 1.21253281, 1.04611041],
       [1.11870532, 1.05940343, 0.9634839 ],
       [1.09782608, 0.91225457, 1.13151201],
       [1.07668675, 1.09270834, 0.97219757],
       [0.96408583, 1.08693717, 1.09056965],
       [1.09388197, 1.05156815, 0.99614253],
       [1.05880313, 1.01238463, 1.07040489],
       [1.04908443, 1.02750876, 1.06499946],
       [1.11368474, 1.21876763, 0.80914028],
       [0.82084376, 1.45508252, 0.86566637],
       [0.87669949, 1.15137567, 1.11351749],
       [1.02342737, 1.07946001, 1.03870527],
       [1.0393784 , 1.0653    , 1.03691425],
       [0.98208391, 1.2263233 , 0.93318544],
       [1.12429763, 0.99719932, 1.0200957 ],
       [0.99843426, 1.06026013, 1.08289826],
       [1.08587011, 0.93266699, 1.12305555],
       [1.08731726, 0.85476226, 1.19951313],
       [1.10788357, 1.05336054, 0.98034854],
       [1.06154534, 1.08315291, 0.9968944 ],
       [1.05497536, 1.09218825, 0.99442905],
       [0.96825253, 0.99008213, 1.18325799],
       [0.91928676, 1.39206347, 0.83024242],
       [1.01869757, 1.08177681, 1.04111828],
       [0.97833462, 1.05314339, 1.11011465],
       [1.0464719 , 1.02072616, 1.0743946 ],
       [0.88629389, 0.94121011, 1.31408865],
       [0.46001365, 0.83735837, 1.84422064],
       [0.89338966, 1.29479162, 0.95341137],
       [1.10462444, 0.91720841, 1.1197598 ],
       [1.39867879, 0.8482266 , 0.89468725],
       [1.19007386, 1.11209765, 0.83942114],
       [1.12346132, 0.93552108, 1.08261025],
       [1.09005214, 1.06924308, 0.98229743],
       [1.01935607, 1.07945212, 1.04278446],
       [1.02776471, 1.02361072, 1.09021723],
       [1.09375298, 1.03608472, 1.01175495],
       [1.02866912, 1.08894646, 1.02397708],
       [1.00766191, 0.96440273, 1.16952801],
       [1.01358352, 1.13818589, 0.98982324],
       [1.03846761, 1.07500291, 1.02812213],
       [1.03503371, 1.06045664, 1.04610231],
       [1.18434671, 1.00696097, 0.95028498],
       [1.06413835, 1.06671841, 1.01073589],
       [1.02851152, 0.97785776, 1.13522338],
       [1.07065846, 0.99105745, 1.07987675],
       [1.04269014, 0.94664168, 1.15226084],
       [1.22512887, 1.01108527, 0.90537851],
       [1.0496613 , 1.04701656, 1.0449148 ],
       [1.10756782, 0.98832059, 1.04570425],
       [1.03577015, 0.95193901, 1.15388349],
       [1.0019734 , 1.038652  , 1.10096726],
       [1.03393667, 0.9339792 , 1.17367678],
       [1.0877613 , 1.0060649 , 1.04776645],
       [0.99156001, 1.09710735, 1.0529253 ],
       [1.12051491, 1.04512398, 0.97595376],
       [1.00093952, 0.9947378 , 1.14591534],
       [1.53879159, 0.88670622, 0.71609485],
       [1.20881938, 0.99205335, 0.94071992],
       [1.11299261, 0.9714882 , 1.05711184],
       [1.1917543 , 1.06827043, 0.88156792],
       [1.17643216, 1.07370617, 0.89145432],
       [0.97557859, 1.18568272, 0.98033134],
       [1.20728297, 1.20355083, 0.73075885],
       [1.24903656, 1.13488755, 0.75766854],
       [1.11651816, 1.11609396, 0.90898054],
       [1.06424092, 0.7962188 , 1.28113293],
       [1.03450523, 1.03431788, 1.07276954],
       [0.88981469, 1.09241766, 1.1593603 ],
       [1.07504555, 1.0076745 , 1.05887261],
       [1.05523604, 1.107276  , 0.97908061],
       [0.98744684, 1.05260286, 1.10154294],
       [1.08205237, 1.01986563, 1.03967466],
       [0.98440508, 1.07652675, 1.08066082],
       [1.09287208, 0.90420108, 1.14451949],
       [1.09664373, 0.97443592, 1.070513  ],
       [0.76993544, 1.26180638, 1.10985083],
       [0.97656438, 1.23090099, 0.93412729],
       [0.97105944, 1.20055713, 0.96997609],
       [0.75930301, 1.50156983, 0.88071981],
       [1.05384606, 1.00095974, 1.08678686],
       [0.97996279, 1.00840909, 1.15322078],
       [1.03743013, 1.00173722, 1.10242531],
       [1.10245955, 1.17274818, 0.86638492],
       [1.10948157, 0.90752986, 1.12458122],
       [0.79648771, 0.65070582, 1.69439912],
       [1.11389273, 1.04869646, 0.97900346],
       [0.77224691, 0.95171967, 1.41762607],
       [1.14319074, 1.04093725, 0.95746467],
       [1.1066563 , 1.11036192, 0.92457444],
       [1.13530666, 1.1437123 , 0.86257369],
       [1.0934321 , 1.01827171, 1.02988885],
       [0.97584524, 1.16061586, 1.00513156],
       [1.73194315, 0.65251604, 0.75713346],
       [1.09056387, 1.03408864, 1.01694015],
       [1.08540137, 1.01653907, 1.03965222],
       [1.02890392, 1.10018951, 1.01249922],
       [0.9790884 , 1.0958812 , 1.06662306],
       [1.01466668, 1.03584194, 1.09108404],
       [0.87620946, 0.99706376, 1.26831944],
       [0.97351188, 1.09063804, 1.07744274],
       [1.00602125, 1.12845758, 1.00711383],
       [1.11505591, 1.00136223, 1.02517451],
       [0.9255307 , 1.15388365, 1.0621783 ],
       [0.98447158, 1.04228519, 1.11483588],
       [1.0566261 , 1.11299698, 0.97196957],
       [1.11172129, 1.07627715, 0.95359421],
       [1.11228933, 1.07777385, 0.95152948],
       [0.94893657, 1.16498329, 1.02767279],
       [0.9399771 , 1.22253366, 0.97908189],
       [1.0105587 , 1.08056876, 1.0504652 ],
       [0.94118619, 1.1427237 , 1.05768276],
       [0.91168244, 1.16764292, 1.0622673 ],
       [1.03561162, 1.09008785, 1.01589318],
       [1.06913167, 1.04176109, 1.03069989],
       [1.05820924, 0.99891635, 1.08446707],
       [1.10762619, 1.00986651, 1.02409995],
       [0.91468809, 1.03767572, 1.18922884],
       [1.38461236, 0.68405405, 1.07292624],
       [1.04309919, 0.96293777, 1.13555569],
       [1.11458736, 0.99762554, 1.02937975],
       [1.0073656 , 0.87437305, 1.25985401],
       [1.06223453, 0.90035102, 1.17900711],
       [1.24831541, 0.91426699, 0.97901025],
       [0.90439181, 1.22828451, 1.00891633],
       [1.23081372, 0.80784456, 1.10293437],
       [1.13840573, 1.10134119, 0.90184573],
       [0.85392487, 1.04646737, 1.24120041],
       [1.26121334, 0.9588237 , 0.92155561],
       [1.06783084, 1.08613946, 0.98762235],
       [1.07993134, 0.93485808, 1.12680324],
       [0.96608205, 0.94536262, 1.23014798],
       [0.98204651, 1.15889156, 1.00065459],
       [1.08029849, 1.06483372, 0.99646044],
       [1.02007636, 1.08763788, 1.03387841],
       [1.04026441, 1.02186322, 1.07946502],
       [1.00534405, 0.9870435 , 1.14920511],
       [0.94530683, 1.07813521, 1.11815061],
       [0.77191565, 1.0468824 , 1.32279461],
       [1.12388396, 0.92549936, 1.09220933],
       [1.07870428, 0.94795169, 1.11493669],
       [1.18063597, 1.14377994, 0.81717675],
       [0.77339458, 1.20084767, 1.1673504 ],
       [1.72400985, 0.77485076, 0.64273204],
       [1.25704094, 0.8308953 , 1.05365641],
       [1.03657919, 1.00911308, 1.09590038],
       [0.89702184, 1.19520685, 1.04936396],
       [0.96915488, 0.95560151, 1.21683626],
       [1.13830861, 1.06982857, 0.93345548],
       [0.98640158, 1.0896621 , 1.06552897],
       [1.04570868, 1.04602727, 1.0498567 ],
       [1.0389134 , 0.99029037, 1.11238889],
       [1.15586247, 1.01327937, 0.97245082],
       [1.02718273, 1.11986867, 0.99454126],
       [1.20904708, 1.1476586 , 0.78488697],
       [1.01925373, 1.08815551, 1.03418342],
       [1.17437731, 1.01831273, 0.94890261],
       [1.0686698 , 0.97278111, 1.10014174],
       [1.08790897, 1.04749961, 1.00618408],
       [1.04825375, 0.97096999, 1.12236891],
       [1.0312938 , 1.08994363, 1.02035522],
       [1.02371344, 1.03968313, 1.07819608],
       [0.99417349, 1.08418481, 1.06323435],
       [1.04772219, 0.99186293, 1.10200753],
       [1.00366963, 1.00441642, 1.1335066 ],
       [1.02560102, 1.09304736, 1.02294428],
       [1.07733597, 0.95291294, 1.11134374],
       [1.13673273, 1.07515815, 0.92970177],
       [1.09373741, 1.01534362, 1.03251163],
       [1.02285301, 1.01435376, 1.10438588],
       [1.07824818, 1.072595  , 0.99074947],
       [1.01665065, 1.09095553, 1.03398647],
       [1.00872511, 0.94955753, 1.18331001],
       [1.1575344 , 1.07757799, 0.90648026],
       [1.12808794, 1.04800929, 0.96549542],
       [0.96281492, 1.11809366, 1.06068407],
       [0.82731131, 1.24511976, 1.06916159],
       [0.87577823, 1.18179679, 1.08401764],
       [1.19845044, 0.98860209, 0.95454012],
       [1.08787325, 1.15810423, 0.89561517],
       [0.9755691 , 1.20431999, 0.96170357],
       [0.98324025, 1.01091821, 1.14743419],
       [0.38557823, 0.86272789, 1.89328654],
       [0.52072257, 1.29465885, 1.32621123],
       [1.13574006, 0.91764634, 1.08820625],
       [1.05559435, 1.01828551, 1.06771279],
       [0.78649459, 1.11014403, 1.24495403],
       [0.97876621, 1.13179275, 1.03103369],
       [1.02840997, 1.03418832, 1.07899436],
       [1.07148747, 1.02013957, 1.04996561],
       [1.22244886, 0.98834871, 0.93079508],
       [1.04585323, 0.96543405, 1.13030538],
       [1.08992298, 1.11534293, 0.93632674],
       [1.12381052, 1.06667384, 0.95110829],
       [0.94488394, 1.15353226, 1.04317646],
       [1.20695461, 1.00260012, 0.93203792],
       [1.39002169, 1.01211906, 0.7394519 ],
       [0.89078743, 0.99796728, 1.25283794],
       [0.99609021, 1.00392996, 1.14157249],
       [0.98497076, 1.13339479, 1.0232271 ],
       [0.9933758 , 1.0931271 , 1.05508976],
       [0.99684031, 1.07061422, 1.07413812],
       [1.114215  , 1.05117232, 0.97620533],
       [1.00359544, 1.06936197, 1.06863525],
       [1.04912377, 1.01800614, 1.07446274],
       [1.00116484, 0.98661559, 1.15381222],
       [1.15942901, 0.89644615, 1.08571749],
       [1.03743588, 0.93476821, 1.16938856],
       [1.00160588, 0.87932889, 1.26065788],
       [1.15935643, 0.99242274, 0.98981348],
       [1.0460769 , 0.96406118, 1.13145457],
       [1.05291789, 0.96941159, 1.11926318],
       [1.03877204, 1.00216671, 1.1006539 ],
       [1.10243852, 1.08950252, 0.9496516 ],
       [1.13256423, 1.00522442, 1.00380401],
       [0.99485033, 1.01373727, 1.13300505],
       [1.07938593, 1.08289366, 0.97931306],
       [1.06461242, 1.11015045, 0.96682978],
       [0.99334767, 1.09570452, 1.05254046],
       [1.08368921, 0.96360819, 1.09429525],
       [1.04599331, 1.12805731, 0.96754203],
       [0.97743758, 1.1370297 , 1.02712537],
       [1.14560143, 1.07646202, 0.91952921],
       [1.0446064 , 1.10224034, 0.99474591],
       [0.99725807, 1.08953561, 1.05479897],
       [1.0648767 , 1.04793357, 1.02878238],
       [1.04538166, 1.09426198, 1.00194901],
       [1.0470043 , 1.12789193, 0.96669642],
       [0.91274958, 0.97152925, 1.25731382],
       [1.11048487, 1.04252671, 0.98858108],
       [1.11318272, 0.96403779, 1.06437214],
       [1.09367046, 1.07851386, 0.96940833],
       [1.22332159, 1.08749938, 0.83077169],
       [1.18320435, 1.07239107, 0.88599724],
       [0.94587761, 1.23723978, 0.95847526],
       [0.98561801, 1.06197081, 1.09400384],
       [1.0695184 , 1.0923722 , 0.97970206],
       [1.28160719, 1.02424186, 0.8357436 ],
       [0.91495871, 1.32237135, 0.9042626 ],
       [1.06482655, 1.02285668, 1.05390943],
       [0.99301999, 1.09972349, 1.04884917],
       [0.6042282 , 1.72187477, 0.81548969],
       [1.13727679, 1.04991395, 0.95440192],
       [0.98727969, 1.10559397, 1.048719  ],
       [0.92750272, 1.12553737, 1.08855256],
       [0.99664971, 0.97243894, 1.172504  ],
       [1.00557429, 1.11219499, 1.02382337],
       [1.11466679, 1.08049205, 0.94643381],
       [0.87683281, 0.96642273, 1.29833711],
       [0.97683701, 1.20155231, 0.96320334],
       [1.22078344, 0.95161281, 0.96919641],
       [1.09747464, 1.06686701, 0.97725101],
       [1.01969689, 1.0769057 , 1.04499006],
       [1.03862878, 0.98603437, 1.11692951],
       [1.07844702, 0.97382444, 1.08932119],
       [1.11064814, 1.06419138, 0.96675313],
       [1.07741857, 1.0242531 , 1.03992099],
       [1.08081662, 1.02022603, 1.04055001],
       [1.06511166, 1.1321518 , 0.94432919],
       [0.89599459, 1.17972101, 1.06587706],
       [1.11178974, 0.97211928, 1.05768363],
       [1.04270048, 1.08041906, 1.01847311],
       [1.1141964 , 1.08073707, 0.94665918],
       [0.96118781, 1.19296682, 0.98743802],
       [1.09875361, 1.09087138, 0.95196766],
       [1.09543569, 1.05448361, 0.99167336],
       [0.97758481, 1.08420807, 1.07979977],
       [0.99794498, 1.11163577, 1.0320119 ],
       [1.10768862, 1.03313649, 1.00076755],
       [1.01610124, 1.06759118, 1.05790023],
       [1.09892199, 0.99341023, 1.04926043],
       [1.02824666, 1.03713607, 1.07620993],
       [1.06753482, 1.15333115, 0.92072669],
       [1.04830088, 1.04045522, 1.05283655],
       [1.08638447, 1.05681455, 0.99839364],
       [0.93663696, 1.12236933, 1.08258636],
       [1.08024434, 1.00435189, 1.05699642],
       [1.09433687, 1.05626225, 0.99099353],
       [1.0716072 , 1.01272173, 1.05726373],
       [1.05018069, 1.09874738, 0.99266459],
       [1.10461459, 1.0179743 , 1.01900376],
       [0.89259052, 1.19610669, 1.05289544],
       [1.05055758, 0.92563223, 1.16540284],
       [1.08677298, 1.00715685, 1.04766283],
       [0.98174526, 1.08877189, 1.07107551],
       [0.82306005, 1.24184842, 1.07668419],
       [1.06656201, 0.87411105, 1.20091959],
       [1.20407595, 1.03310195, 0.90441476],
       [0.9768093 , 1.01564573, 1.14913762],
       [0.98153707, 1.10694588, 1.0531097 ],
       [1.01898618, 1.00133669, 1.12126978],
       [0.91395409, 1.231808  , 0.99583056],
       [0.94889868, 0.99975277, 1.1929412 ],
       [1.0850339 , 1.06889902, 0.98765973],
       [0.97251553, 1.11715978, 1.05191735],
       [0.88585194, 0.9893358 , 1.26640491],
       [0.76305351, 1.19600774, 1.1825314 ],
       [1.01262093, 1.00268916, 1.12628256],
       [1.16318075, 0.93008819, 1.04832371],
       [0.98193191, 1.11182526, 1.04783548],
       [1.14497645, 0.99836531, 0.9982509 ],
       [1.14294575, 1.02302042, 0.97562647],
       [0.99236342, 1.19981249, 0.94941675],
       [1.05277138, 0.96425821, 1.12456306],
       [1.02865006, 1.09294035, 1.02000224],
       [1.03492673, 1.03937628, 1.06728964],
       [1.05369239, 1.05014201, 1.03775826],
       [1.12226261, 0.96243518, 1.05689487],
       [0.93414828, 1.07878145, 1.12866292],
       [1.09405225, 1.09787834, 0.94966206],
       [1.15292281, 1.02903401, 0.95963584],
       [1.01994257, 1.04088838, 1.0807617 ],
       [1.10577355, 1.0098427 , 1.02597641],
       [1.0541472 , 0.97922265, 1.1082228 ],
       [0.99926698, 1.16872316, 0.97360252],
       [1.08086683, 0.94118984, 1.11953598],
       [1.02792082, 0.9904478 , 1.12322403],
       [0.94649752, 1.07641835, 1.11867679],
       [1.01459703, 0.97936033, 1.14763529],
       [0.99023993, 1.12083468, 1.03051804],
       [0.99914667, 1.04726169, 1.0951843 ],
       [1.05411771, 1.07028293, 1.01719202],
       [1.04758941, 1.12740672, 0.96659652],
       [0.90842364, 1.12029778, 1.11287124],
       [1.08972973, 0.96280455, 1.08905837],
       [1.15054206, 0.99511969, 0.9959309 ],
       [1.05166856, 0.9416499 , 1.1482742 ],
       [1.00405725, 1.02855052, 1.10898488],
       [1.0081473 , 1.04695419, 1.08649116],
       [1.00519504, 1.08700499, 1.04939262],
       [1.04920137, 1.04251469, 1.0498766 ],
       [1.10230885, 0.90469325, 1.13459055],
       [1.18742604, 0.78190481, 1.1722618 ],
       [1.17222066, 1.03510711, 0.93426488],
       [0.92484682, 1.18347323, 1.03327261],
       [1.03346159, 1.02924622, 1.07888484],
       [1.0426965 , 1.11151723, 0.98737892],
       [1.0005056 , 1.1324677 , 1.00861935],
       [1.19136764, 1.11008983, 0.84013519],
       [0.88768765, 1.15999075, 1.09391425],
       [0.98927104, 1.12206569, 1.03025592],
       [1.00989485, 1.13057716, 1.00112064],
       [1.09578677, 1.07327325, 0.97253264],
       [0.9799984 , 0.98234179, 1.17925247],
       [1.10352981, 0.99338769, 1.04467516],
       [1.07062009, 0.976765  , 1.09420756],
       [0.97641378, 1.05245324, 1.11272563],
       [0.87865691, 1.12220856, 1.14072718],
       [1.14379501, 0.93430305, 1.06349459],
       [0.95702524, 1.1252137 , 1.05935371],
       [0.98657492, 1.06834227, 1.08667546],
       [1.06160033, 0.90516174, 1.17483058],
       [1.05297496, 1.08090492, 1.00771278],
       [1.11843501, 1.00897282, 1.01418482],
       [1.04186837, 1.01882208, 1.0809022 ],
       [1.06256377, 1.03067855, 1.04835034],
       [1.06719712, 1.10794697, 0.96644856],
       [1.1338569 , 0.93933983, 1.06839592],
       [1.05435805, 0.96888406, 1.11835054],
       [1.15335501, 0.98839464, 0.999843  ],
       [1.07809285, 0.96838558, 1.09511422],
       [1.18989126, 0.96408157, 0.98761982],
       [0.99320593, 1.16200821, 0.98637851],
       [0.91919719, 1.10575376, 1.1166417 ],
       [1.15698181, 0.96530653, 1.01930431],
       [1.09662815, 1.10143919, 0.94352532],
       [0.98670045, 1.12129871, 1.0335935 ],
       [1.05880997, 1.10613427, 0.97664841],
       [0.98311616, 1.13682626, 1.02165024],
       [1.04608471, 0.98419713, 1.11131081],
       [1.03473408, 1.08598124, 1.02087733],
       [1.065889  , 1.03711569, 1.03858796],
       [0.97322583, 1.04288227, 1.12548455],
       [1.16163031, 0.88295457, 1.09700778],
       [0.90693615, 1.02460191, 1.21005459],
       [0.97646905, 1.00244806, 1.16267554],
       [1.0794287 , 0.9597159 , 1.10244805],
       [1.0492534 , 1.04179585, 1.05054341],
       [1.05681753, 1.02100584, 1.06376928],
       [1.06969756, 1.06238612, 1.00950897],
       [1.0369459 , 1.06547078, 1.03917597],
       [0.92192701, 1.32706721, 0.89259844],
       [0.80870684, 1.62699958, 0.70588624],
       [1.16178757, 1.09216876, 0.88763632],
       [1.07176957, 0.90156628, 1.1682568 ],
       [0.9331415 , 1.10955058, 1.09890057],
       [1.01366917, 1.0670672 , 1.06085629],
       [1.09941926, 1.03362127, 1.00855212],
       [0.84811808, 1.22194074, 1.07153382],
       [1.18550932, 1.06432755, 0.89175578],
       [1.11736823, 1.02611416, 0.99811026],
       [1.1643233 , 0.94235589, 1.03491346],
       [1.08690834, 1.12311704, 0.93156727],
       [1.07561298, 0.9436223 , 1.12235737],
       [1.06486832, 1.09034468, 0.98637965],
       [1.07557254, 0.98362772, 1.08239239],
       [1.12198042, 1.05058141, 0.96903083],
       [1.13706384, 1.06337253, 0.94115628],
       [1.00501732, 1.12041024, 1.0161651 ],
       [0.95648988, 1.099113  , 1.08598978],
       [0.94966783, 1.07193389, 1.11999093],
       [1.01264412, 1.16434412, 0.96460441],
       [1.15296468, 0.87166744, 1.11696054],
       [1.12847164, 1.03936057, 0.97376045],
       [1.02008966, 1.15287546, 0.96862754],
       [0.95897962, 1.11563698, 1.06697606],
       [1.14257076, 1.00367626, 0.99534564],
       [1.12981748, 0.95023665, 1.06153852],
       [1.15735646, 1.00502068, 0.97921551],
       [1.13391403, 1.02996262, 0.977716  ],
       [1.08689457, 1.02212575, 1.03257233],
       [1.03151245, 1.08542581, 1.0246544 ],
       [0.86775724, 1.14470765, 1.12912777],
       [0.96921738, 1.05154358, 1.12083169],
       [1.12124332, 1.14185185, 0.87849748],
       [0.97683947, 1.12733357, 1.03741962],
       [1.01255618, 1.06752466, 1.06151182],
       [1.0069439 , 1.07704565, 1.0576031 ],
       [1.04207903, 1.03665509, 1.06285853],
       [0.92105129, 1.10209054, 1.11845082],
       [1.19807491, 0.79962472, 1.14389303],
       [1.08084706, 0.93740495, 1.12334064],
       [0.93659944, 1.10026745, 1.10472577],
       [1.08978077, 1.10703535, 0.94477653],
       [1.02997682, 1.0017112 , 1.10990463],
       [1.14137937, 1.10978173, 0.89043156],
       [1.00182606, 1.10442311, 1.03534348],
       [0.91009595, 1.14235373, 1.08914297],
       [1.03866857, 0.99943352, 1.10349056],
       [1.09381433, 1.09087096, 0.95690736],
       [0.95343207, 1.07460861, 1.11355197],
       [1.14532338, 0.97701415, 1.01925512],
       [0.85821242, 1.61705921, 0.66632102],
       [1.0312031 , 1.0643342 , 1.04605535],
       [1.52453071, 0.91369285, 0.70336909],
       [1.0755444 , 0.94988347, 1.11616479],
       [0.98838661, 1.14956598, 1.00364006],
       [1.13205185, 0.97051992, 1.03902088],
       [1.01483116, 0.96751498, 1.15924651],
       [1.13167421, 1.03526779, 0.97465066],
       [1.0357688 , 1.08398449, 1.02183937],
       [1.0203175 , 1.03274975, 1.0885254 ],
       [1.09682226, 0.86838462, 1.17638578],
       [1.02374399, 1.13569456, 0.9821541 ],
       [0.86037483, 1.10582781, 1.17539002],
       [1.08225517, 1.06788549, 0.991452  ],
       [1.03132549, 1.02861187, 1.0816553 ],
       [1.02507462, 1.07605048, 1.04046755],
       [1.0625953 , 1.02996978, 1.04902758],
       [1.00064219, 0.98468321, 1.15626725],
       [0.92982885, 1.31661415, 0.89514966],
       [0.93320474, 1.13925105, 1.06913686],
       [1.09648094, 0.92638906, 1.11872266],
       [1.08105198, 1.10992203, 0.95061865],
       [1.04581899, 1.06054357, 1.0352301 ],
       [1.18970094, 1.05296775, 0.89892396],
       [1.09495744, 1.02283223, 1.02380299],
       [1.05132499, 0.91613809, 1.17412957],
       [1.04221464, 1.00917474, 1.09020327],
       [1.0109592 , 1.03558582, 1.09504763],
       [1.01496921, 1.15483164, 0.9717918 ],
       [1.31538999, 0.78521241, 1.04099026],
       [1.08623535, 0.99931458, 1.05604272],
       [0.76756673, 1.17948489, 1.19454103],
       [0.98506191, 1.15367078, 1.00285997],
       [1.1285084 , 0.98909469, 1.02398957],
       [0.96178318, 1.05910967, 1.12069979],
       [1.03391383, 0.97436323, 1.1333156 ],
       [0.97298633, 1.13523068, 1.03337565],
       [1.01839109, 1.03339784, 1.08980372],
       [1.04263099, 1.07281081, 1.02615086],
       [1.1486061 , 1.10232889, 0.89065766],
       [1.19000879, 1.12209766, 0.8294862 ],
       [1.10600659, 0.9901088 , 1.04547726],
       [1.01875676, 1.1001065 , 1.0227294 ],
       [1.04192776, 1.01135403, 1.08831086],
       [1.02753983, 1.02150562, 1.0925472 ],
       [1.08163599, 1.02340452, 1.03655214],
       [1.03516934, 1.06358195, 1.04284136],
       [1.19796698, 0.88611664, 1.05750904],
       [1.07903637, 1.08293706, 0.97961922],
       [1.10288763, 0.97281524, 1.06588978],
       [1.14157233, 0.96821724, 1.03180308],
       [1.05650759, 1.00095082, 1.08413424],
       [0.95311698, 1.09351104, 1.09496462],
       [0.9508108 , 1.11328293, 1.07749892],
       [1.02323562, 1.08061869, 1.03773834],
       [1.01884922, 1.0919161 , 1.03082733],
       [0.94281474, 1.21746058, 0.98131733],
       [1.17004992, 1.02478122, 0.94676152],
       [1.14249249, 0.91282316, 1.08627701],
       [1.22128916, 0.9608125 , 0.959491  ],
       [1.0139036 , 1.13946756, 0.98822149],
       [1.04407283, 0.93173576, 1.16578406],
       [1.14725122, 1.00157144, 0.99277   ],
       [1.01536104, 1.13662164, 0.98960997],
       [1.14226316, 1.02764689, 0.9716826 ],
       [1.13429422, 1.04339101, 0.96390742],
       [1.02318614, 1.09130769, 1.02709882],
       [1.03374185, 1.0330148 , 1.074836  ],
       [1.17095752, 1.0881342 , 0.88250093],
       [1.2426734 , 1.05951156, 0.83940769],
       [1.00096308, 1.03160376, 1.10902581],
       [1.05422772, 1.04414603, 1.0432189 ],
       [1.0018776 , 1.10112643, 1.03858862],
       [1.00240025, 1.08587892, 1.05331348],
       [1.04946101, 1.04887481, 1.04325683],
       [1.04346911, 1.08166206, 1.01646148],
       [1.68239112, 0.78474013, 0.67446141],
       [1.06289799, 0.98531822, 1.09337644],
       [1.06951425, 1.12916669, 0.94291171],
       [0.95280868, 0.96369453, 1.22508944],
       [1.06821863, 1.0283966 , 1.04497743],
       [1.04865772, 0.95399444, 1.1389405 ],
       [1.02652069, 0.97034923, 1.14472273],
       [1.00247393, 1.0615886 , 1.07753012],
       [1.0529162 , 1.03646568, 1.05221077],
       [1.22934516, 0.90125704, 1.01099045],
       [0.86669001, 1.06881234, 1.2060903 ],
       [1.13192136, 1.08780961, 0.92186169],
       [1.03738606, 0.98925277, 1.11495382],
       [1.03135817, 1.05895699, 1.0512775 ],
       [1.01155895, 1.08742972, 1.04260398],
       [1.04984884, 1.08815257, 1.00359124],
       [0.99859128, 1.0498208 , 1.09318057],
       [1.06053099, 0.96782664, 1.11323502],
       [0.99883681, 1.09115307, 1.05160277],
       [1.02921815, 1.02714773, 1.08522677],
       [1.03388656, 1.07399892, 1.03370717],
       [0.90685595, 1.13175041, 1.10298629],
       [0.86226869, 1.20101195, 1.07831202],
       [1.01531698, 1.06809773, 1.05817795],
       [1.07392311, 1.07095054, 0.99671901],
       [1.12084405, 1.03434431, 0.9864043 ],
       [1.09394134, 1.04777241, 0.99987891],
       [0.97419523, 1.05780393, 1.10959349],
       [0.75100845, 1.09141466, 1.29916955],
       [0.8218263 , 1.3526792 , 0.96708716],
       [1.09596415, 1.03029137, 1.01533714],
       [1.18784396, 1.06733173, 0.88641697],
       [1.04228689, 1.08763134, 1.01167442],
       [1.04657366, 1.03572585, 1.05929314],
       [1.05617552, 1.07576507, 1.00965206],
       [1.09928376, 1.021726  , 1.02058289],
       [1.02947754, 1.07020696, 1.04190815],
       [0.98519124, 1.14297303, 1.01342838],
       [0.96462326, 1.15674309, 1.0202263 ],
       [0.99769174, 1.14807051, 0.9958304 ],
       [0.98114395, 1.02047703, 1.13997166],
       [1.11182498, 0.99167703, 1.03809065],
       [1.10708133, 1.082781  , 0.95173032],
       [1.11487179, 1.03976023, 0.98696063],
       [1.02546565, 1.08961493, 1.02651207],
       [0.97561156, 1.12142763, 1.04455346],
       [1.01748548, 1.01457358, 1.10953359],
       [1.02457581, 1.08763984, 1.029377  ],
       [1.03709525, 1.02490231, 1.0795951 ],
       [1.10484515, 0.89654553, 1.14020197],
       [1.11689828, 0.82555782, 1.19913656],
       [1.05994801, 1.05517741, 1.02646724],
       [1.02089487, 1.01963852, 1.10105926],
       [1.02984016, 0.96166484, 1.15008766],
       [0.96011816, 1.11998457, 1.06148992],
       [1.0638347 , 0.98498748, 1.09277047],
       [0.74298796, 0.69377997, 1.70482473],
       [1.41027939, 0.7676035 , 0.96370976],
       [1.05931125, 0.99942991, 1.08285149],
       [1.18154495, 0.89305053, 1.06699717],
       [1.22362258, 0.96293106, 0.95503901],
       [1.04364179, 1.00699867, 1.0909522 ],
       [1.04971702, 1.08161261, 1.01026302],
       [1.00100488, 1.09522514, 1.04536263],
       [0.99091816, 1.03414212, 1.11653238],
       [1.17198481, 0.99698577, 0.97262208],
       [1.20905137, 1.0023629 , 0.93017839],
       [1.05555527, 1.113079  , 0.97295838],
       [1.01016235, 1.03709199, 1.09433832],
       [1.11647588, 1.03286135, 0.99225542],
       [1.05021051, 1.02465277, 1.06672937],
       [1.09174898, 1.03820854, 1.01163513],
       [1.02640212, 1.03854363, 1.0766469 ],
       [1.08952879, 1.05910252, 0.99296134],
       [1.07137283, 1.04002006, 1.03019976],
       [1.06386978, 1.02566032, 1.05206255],
       [1.04300203, 1.04198982, 1.05660081],
       [1.11545071, 1.11677493, 0.90936702],
       [1.17171895, 1.11165498, 0.85821872],
       [1.00698325, 1.05133554, 1.08327387],
       [0.97675769, 1.16475907, 1.00007589],
       [1.01112943, 0.96858691, 1.16187631],
       [1.05942885, 1.09946301, 0.9827008 ],
       [1.00563328, 0.97815001, 1.15780936],
       [0.89520624, 1.35759381, 0.88879261],
       [0.97659892, 0.88502531, 1.27996842],
       [0.96067188, 1.03003096, 1.15088981],
       [0.81630008, 1.18616336, 1.13912921],
       [0.8486065 , 1.00373922, 1.28924694],
       [0.95961692, 1.07913286, 1.10284288],
       [0.97132776, 0.98489134, 1.18537355],
       [1.05096898, 1.11941931, 0.97120437],
       [1.06345143, 0.93224725, 1.14589397],
       [1.04959318, 0.95157098, 1.14042849],
       [1.0631675 , 1.01553708, 1.06288808],
       [1.07733274, 1.05036524, 1.01389467],
       [1.06006608, 0.99112746, 1.09039911],
       [1.02590187, 1.03089838, 1.08479241],
       [1.03797605, 1.01727095, 1.08634566],
       [1.05355657, 1.04562629, 1.04240979],
       [1.03372987, 1.00561255, 1.10225023],
       [1.04604291, 1.02407646, 1.07147329],
       [1.00074704, 1.09016754, 1.05067807],
       [1.03610631, 1.06077371, 1.04471264],
       [1.12994698, 1.10409932, 0.90754635],
       [0.83052568, 1.14892134, 1.16214564],
       [1.01294947, 1.02759537, 1.10104782],
       [1.00458617, 0.96331264, 1.17369384],
       [1.18305995, 1.08423613, 0.87429658],
       [1.15541204, 0.90709028, 1.07909034],
       [0.97159413, 1.03335093, 1.13664759],
       [1.20748   , 1.01674822, 0.91736443],
       [1.01870852, 0.89314068, 1.22974345],
       [1.12008157, 1.01355809, 1.007953  ],
       [0.63921345, 0.7841221 , 1.7182571 ],
       [1.43492297, 0.85610115, 0.85056853],
       [1.24716726, 0.9499586 , 0.94446679],
       [0.97462331, 1.17462845, 0.99234089],
       [1.02739896, 1.11571018, 0.99848351],
       [1.12516535, 1.04816991, 0.9682574 ],
       [1.0455511 , 0.9237249 , 1.17231666],
       [0.99437014, 1.13739405, 1.00982846],
       [1.129479  , 1.01749729, 0.99461637],
       [1.03944611, 0.97811349, 1.12403305],
       [1.08029897, 1.01156695, 1.04972673],
       [1.00204408, 1.03250793, 1.10704064],
       [1.07327144, 1.06951406, 0.99880715],
       [1.06414998, 1.04807896, 1.02936372],
       [1.08294021, 1.03304312, 1.02560932],
       [1.03744039, 1.05726327, 1.046889  ],
       [1.02237089, 1.12560018, 0.99362158],
       [0.93745862, 1.07257729, 1.13155674],
       [1.13847112, 0.83043082, 1.17269071],
       [1.04469854, 1.11880108, 0.97809303],
       [0.93449795, 0.93363092, 1.27346378],
       [0.94095033, 1.07989894, 1.12074338],
       [1.22877508, 0.9808124 , 0.93200517],
       [1.0506151 , 0.78480963, 1.30616792],
       [0.84933249, 1.22752692, 1.06473325],
       [0.95954644, 1.11961659, 1.06242963],
       [1.03724572, 1.14937606, 0.95497087],
       [1.11180418, 0.94501402, 1.08477445],
       [1.12804497, 1.05510647, 0.95844121],
       [1.15016016, 1.0663352 , 0.92509729],
       [1.0152987 , 1.02171795, 1.104576  ],
       [1.10313722, 0.99754804, 1.0409074 ],
       [1.05700136, 1.0908378 , 0.99375349],
       [1.0012101 , 1.07941926, 1.0609633 ],
       [1.07947623, 1.015053  , 1.04706342],
       [1.08356304, 1.00808457, 1.04994504],
       [1.03056207, 1.03513124, 1.07589934],
       [1.0132818 , 1.03411931, 1.09419154],
       [1.06064002, 1.06200816, 1.01894447],
       [1.14442294, 0.99641858, 1.00075113],
       [1.1364048 , 1.04133656, 0.9638513 ],
       [0.8445904 , 1.14995437, 1.14704788],
       [1.16211566, 0.95673175, 1.02274524],
       [0.79336132, 1.19941595, 1.14881538],
       [1.17177553, 0.82197454, 1.14784258],
       [0.85446977, 1.04456152, 1.24256136],
       [0.63798592, 1.72118567, 0.78242107],
       [1.3558349 , 0.85350276, 0.93225499],
       [1.07932963, 1.09579183, 0.96647119],
       [1.17570501, 0.94960335, 1.01628429],
       [0.98572057, 1.13908383, 1.01678825],
       [1.03997611, 0.92398666, 1.17762989],
       [1.15786597, 1.00818565, 0.97554104],
       [0.97975814, 1.14813843, 1.01369608],
       [1.10327582, 1.06102149, 0.97729534],
       [1.04851545, 1.10395547, 0.98912173],
       [1.22030469, 0.76197781, 1.15931016],
       [0.99360172, 1.00312169, 1.14486925],
       [1.03308725, 1.06223394, 1.04627146],
       [1.0762751 , 1.01862774, 1.0466898 ],
       [1.06946686, 1.0035212 , 1.06860459],
       [0.99935334, 1.06972418, 1.07251514],
       [0.98365617, 1.17945231, 0.97848418],
       [1.14642904, 0.81519003, 1.17997359],
       [0.98256319, 0.96853381, 1.19049565],
       [1.24901204, 0.84978127, 1.04279934],
       [1.14370158, 1.03337964, 0.96451144],
       [0.91845022, 1.1828895 , 1.04025293],
       [1.14291374, 1.00594242, 0.9927365 ],
       [1.00651954, 1.00428709, 1.13078602],
       [1.02772091, 1.08011838, 1.03375336],
       [1.07716903, 1.05886938, 1.00555425],
       [0.84471583, 1.52822492, 0.7686519 ],
       [0.95248741, 1.1970898 , 0.99201544],
       [1.05520048, 1.02051621, 1.06587596],
       [1.08346338, 0.99744911, 1.06068017],
       [0.96956947, 1.08326672, 1.08875646],
       [0.97461424, 1.05096598, 1.11601243],
       [1.16205567, 0.97823634, 1.00130064],
       [0.87217713, 1.18990356, 1.07951197],
       [0.96290823, 1.18780281, 0.99088161],
       [0.98026835, 0.96880429, 1.19252002],
       [1.19141124, 1.03269801, 0.9174834 ],
       [0.96555709, 1.17622663, 0.99980894],
       [1.17155338, 1.00597958, 0.96405969],
       [1.10269884, 1.05959606, 0.97929775],
       [0.94182769, 1.01227507, 1.1874899 ],
       [1.23702589, 0.85068857, 1.05387819],
       [0.93580443, 1.10974242, 1.09604581],
       [0.9959868 , 1.07142028, 1.07418557],
       [1.09444642, 1.037871  , 1.00927524],
       [0.9640123 , 1.09621147, 1.08136888],
       [1.06443149, 1.09766908, 0.97949208],
       [1.22509326, 1.08129646, 0.83520293],
       [1.03710965, 0.93607027, 1.16841274],
       [1.08422806, 1.13961328, 0.91775132],
       [0.91536568, 1.19336255, 1.03286442],
       [1.20087125, 1.01831021, 0.92241119],
       [0.99303337, 0.97899538, 1.1695639 ],
       [1.00521509, 0.98669004, 1.14968752],
       [1.01826651, 1.07993524, 1.0433909 ],
       [0.99639259, 1.0494888 , 1.09571126],
       [1.08507759, 1.06323801, 0.99327706],
       [1.0739758 , 1.06046351, 1.00715334],
       [0.99515548, 1.08628094, 1.06015623],
       [1.11602378, 1.08533817, 0.94023071],
       [0.90137998, 1.11237531, 1.12783737],
       [1.0261418 , 0.87444206, 1.24100879],
       [1.11565311, 1.07484971, 0.95108982],
       [1.07642506, 0.94477255, 1.12039504],
       [1.21495094, 0.99594824, 0.93069348],
       [1.03580137, 0.9352916 , 1.17049969],
       [1.00544842, 0.97869579, 1.15744844],
       [1.01211697, 0.97597151, 1.15350417],
       [1.10184315, 1.05710433, 0.98264517],
       [1.09520162, 1.05480904, 0.99158199],
       [0.94538173, 0.98348355, 1.21272737],
       [1.09827046, 0.99272775, 1.05059445],
       [0.93962856, 1.1317991 , 1.07016499],
       [0.96044232, 1.07833367, 1.10281667],
       [1.13763124, 1.09764287, 0.90631854],
       [1.24646906, 1.03263638, 0.86248722],
       [0.84567069, 1.10770451, 1.18821745],
       [0.95815482, 0.92372846, 1.25970937],
       [1.21421327, 1.04245143, 0.88492795],
       [1.17091153, 1.0287025 , 0.94197863],
       [1.13829378, 1.00503011, 0.99826876],
       [0.99924428, 1.14818225, 0.99416613],
       [1.02778069, 1.10586355, 1.00794842],
       [1.00066023, 1.09869546, 1.04223697],
       [1.05035786, 0.94888073, 1.14235406],
       [1.11425939, 1.03197894, 0.99535433],
       [0.88462109, 1.13940402, 1.11756755],
       [1.30144921, 0.96186924, 0.8782742 ],
       [0.89349238, 1.31631769, 0.93178258],
       [1.08518519, 1.11577956, 0.9406279 ],
       [0.90420453, 1.19914636, 1.03824177],
       [1.08728882, 0.9542705 , 1.10003334],
       [1.01053722, 1.06592177, 1.06513367],
       [1.09554218, 1.01259593, 1.03345455],
       [1.05934134, 0.99296222, 1.08928909],
       [1.17672177, 1.03852761, 0.92634327],
       [0.92931274, 1.08936345, 1.12291647],
       [0.88072026, 1.21098398, 1.04988841],
       [1.03335113, 0.93114637, 1.17709515],
       [0.92843101, 1.19496255, 1.01819909],
       [0.96368512, 0.9175192 , 1.26038833],
       [1.14051352, 1.02072712, 0.98035202],
       [1.1898685 , 1.09710499, 0.85461917],
       [0.96218914, 1.07668602, 1.10271749],
       [1.04283413, 1.06888217, 1.02987635],
       [1.16406644, 1.03961043, 0.93791578],
       [1.12601639, 1.00410626, 1.01147001],
       [1.13043224, 1.1232269 , 0.88793351],
       [0.82849033, 1.18371103, 1.1293913 ],
       [0.83759563, 0.90406623, 1.3999308 ],
       [1.03226758, 1.10175226, 1.00757281],
       [1.06238624, 1.18997632, 0.88923009],
       [0.84474825, 1.1414923 , 1.1553521 ],
       [1.1996023 , 0.97014596, 0.97184439],
       [0.96315719, 1.0952825 , 1.08315296],
       [1.01259006, 1.11098764, 1.01801496],
       [0.93040977, 0.85747585, 1.35370703],
       [0.83740871, 0.73774902, 1.56643493],
       [0.97439114, 1.13811213, 1.02908939],
       [0.98182353, 1.24145076, 0.91831836],
       [1.07700329, 1.23714041, 0.82744895],
       [1.04848536, 0.94007209, 1.1530352 ],
       [0.75843819, 1.34645335, 1.03670112],
       [0.66850228, 1.67567359, 0.79741679],
       [1.4649255 , 0.94923468, 0.72743247],
       [0.83523764, 0.89108972, 1.41526529],
       [0.73739405, 1.15313536, 1.25106325],
       [0.8586696 , 1.24544   , 1.03748305],
       [1.36882914, 0.89801188, 0.87475164]])
        self.assertTrue(np.isclose(alpha, alpha_gt).all())

    

if __name__ == '__main__':
    unittest.main()
