from .context import gpytoolbox
from .context import numpy as np
from .context import unittest

class TestPerFaceNormals(unittest.TestCase):
    def test_single_triangle(self):
        v = np.array([[0.0,0.0,0.0],[1.0,0.0,0.0],[0.0,1.0,0.0]])
        f_up = np.array([[0,1,2]],dtype=int)
        n = gpytoolbox.per_face_normals(v,f_up)
        n_up = np.array([[0,0,1]])
        # the normal should be pointing up
        self.assertTrue(np.isclose(n - n_up,0.0).all())
        # Now if we change the ordering convention, it should point down:
        f_down = np.array([[0,2,1]],dtype=int)
        n = gpytoolbox.per_face_normals(v,f_down)
        n_down = np.array([[0,0,-1]])
        self.assertTrue(np.isclose(n - n_down,0.0).all())

    # TODO WRITE THIS WITHOUT IGL
    def test_bunny(self):
        v,f = gpytoolbox.read_mesh("test/unit_tests_data/bunny_oded.obj")
        # n_gt = igl.per_face_normals(v,f,np.array([0.,0.,0.]))
        # n_gt =  n_gt/np.tile(np.linalg.norm(n_gt,axis=1)[:,None],(1,3))
        n = gpytoolbox.per_face_normals(v,f,unit_norm=True)
        n_gt = np.array([[0.00020002,0.00016287,0.0010518],[1.5867e-05,7.5189e-05,0.0011368],[0.00020001,0.0001629,0.0010518],[0.00016705,-0.00032133,0.0017132],[-2.1965e-05,0.00015131,0.00166],[-4.6961e-05,0.00069003,0.0012876],[0.00039569,0.00025275,0.00090556],[0.0016099,-0.00060523,0.00086469],[-0.00021189,8.094e-05,0.0017295],[0.00063453,-9.2134e-05,0.0015129],[0.00016399,0.00024339,0.0017488],[-2.6035e-05,0.00015535,0.0018155],[-0.00024678,0.00063527,0.0014776],[0.00054707,-8.8083e-07,0.0010944],[0.0013796,-0.00075764,0.0010363],[0.0014999,-0.000507,0.00080755],[-0.00028183,-7.1771e-05,0.0018977],[0.001269,-0.00069389,0.00095304],[0.00043681,-0.00078018,0.0013917],[0.00099292,-0.00042047,0.0016312],[0.00017903,0.00045709,0.0017295],[0.00088674,-0.00069969,0.0015666],[-0.00030698,0.0010208,0.0014462],[0.00091634,0.00019666,0.00093588],[0.0014029,-0.00045432,0.00073021],[0.0011953,-0.00059173,0.0012163],[0.0013313,-0.00034796,0.001014],[0.00064212,-0.00096857,0.0015229],[-0.00027774,-7.1087e-05,0.0017709],[0.0011215,-0.00083919,0.0011067],[0.00097208,-0.0005787,0.0012817],[0.00091644,-0.00045378,0.001666],[0.00086597,-0.00067816,0.0015209],[0.00022681,0.0005128,0.0018646],[0.00078528,-0.00089914,0.0014691],[-0.00042024,0.0010813,0.0016483],[0.00092906,8.0894e-05,0.00093974],[0.0012503,-8.5607e-05,0.00067983],[0.0011564,-0.00049565,0.0011735],[0.0010903,-0.00064639,0.0012612],[0.0012483,-0.00016222,0.00095665],[0.00053569,-0.0012075,0.0013982],[-0.00043783,-0.00048296,0.0017507],[0.0010636,-0.00078419,0.0012383],[0.0011128,-0.0007125,0.0010959],[0.0008312,-0.00099708,0.001136],[0.0012345,-0.0007753,0.0012848],[0.0010373,-5.6636e-05,0.0017238],[0.0011004,-0.00091114,0.0012008],[0.00022743,0.00081042,0.0017791],[0.00097899,-0.0010788,0.0011469],[-0.00042086,0.0012495,0.0013981],[0.0010158,0.00012481,0.00089827],[0.0012168,-0.00014506,0.00077735],[0.0013324,-4.9402e-05,0.00080183],[0.00094071,-0.00027061,0.0014278],[0.0010278,-0.00057908,0.0014254],[0.00096696,2.3616e-05,0.0010414],[0.000281,-0.0012063,0.001356],[0.0008003,-0.0014041,0.0010058],[0.00022703,-0.00034365,0.0017257],[0.00092866,-0.00083667,0.0012753],[0.0011189,-0.00070005,0.0010812],[0.001239,-0.0007735,0.00091816],[0.0010999,-0.00089307,0.0011786],[0.0012301,-0.00074199,0.0012344],[0.0010641,-4.5422e-05,0.0017135],[0.00093903,-0.00099077,0.0010567],[0.00032116,0.00092636,0.0019045],[0.00078824,-0.0013439,0.0009627],[-0.00043258,0.0014168,0.0015332],[0.0011001,0.0001484,0.00097223],[0.0012569,-8.2743e-05,0.0008066],[0.0013213,-0.00014322,0.0007898],[0.0010098,-0.00021244,0.0013617],[0.00078439,-0.00011494,0.0011789],[0.00097742,-0.00060462,0.0014419],[0.0010081,2.8038e-05,0.0010862],[0.00023348,-0.0013095,0.0012934],[0.00068669,-0.0014369,0.00086304],[2.785e-06,-0.00094508,0.0015645],[0.00099815,-0.00058563,0.0013413],[0.00090296,-0.00081625,0.0013552],[0.0011204,-0.00065952,0.0010837],[0.0012271,-0.00091564,0.00089299],[0.0011772,-0.00097954,0.00098552],[0.0014081,-0.00040999,0.0013227],[0.0012962,-0.00082109,0.0011083],[0.0011275,0.00033673,0.0016802],[0.0010029,-0.0010504,0.00083038],[0.00029185,0.0011746,0.001714],[0.00082934,-0.0013561,0.00080551],[-0.00041288,0.0014002,0.0012513],[0.0012053,0.00020368,0.00092761],[0.0013029,-5.8373e-05,0.00078318],[0.0013305,-0.000131,0.0007728],[0.0010799,8.633e-05,0.0010297],[0.001066,-0.00016805,0.0016019],[0.00060051,-5.2543e-06,0.0010653],[0.00094405,-0.00057833,0.0014658],[0.00083012,0.00013634,0.0011464],[0.00051327,-0.0015939,0.00096715],[0.00030397,-0.0012764,0.0012567],[0.00076049,-0.0014832,0.00069551],[0.0010466,-0.00069589,0.0012305],[0.00078976,-0.00069091,0.0014262],[0.00094582,-0.00076586,0.0013032],[0.0011563,-0.00059419,0.0010004],[0.0012057,-0.0008196,0.00078856],[0.00093726,-0.0010267,0.00081383],[0.0012175,-0.001062,0.0010728],[0.0014163,-0.00031615,0.0011187],[0.0015105,-0.00068893,0.00095085],[0.0012166,0.00037579,0.0016123],[0.00077176,-0.0011019,0.00065314],[0.00044031,0.0013219,0.001728],[0.00075103,-0.0015207,0.00071624],[-0.00021222,0.0016547,0.0011317],[0.0014544,0.00032653,0.0011036],[0.0014406,-4.7109e-05,0.00086594],[0.0014088,-0.00010896,0.00081995],[0.001168,-0.00013543,0.001089],[0.00069002,6.7144e-05,0.00099252],[0.0010609,-0.00018246,0.0016309],[0.00070721,3.2299e-05,0.001258],[0.0011476,-0.00042364,0.0013367],[0.00094572,-0.00024426,0.0012196],[0.00042527,-0.0011722,0.00079575],[0.00037248,-0.0014378,0.0014823],[0.00065102,-0.0015228,0.00057385],[0.00088432,-0.0012354,0.0010788],[0.00086464,-0.00032007,0.0014884],[0.00094778,-0.00081803,0.001309],[0.0010394,-0.00084052,0.001192],[0.0011039,-0.00070901,0.00094829],[0.0011568,-0.0010648,0.00073138],[0.0010162,-0.0011156,0.00048322],[0.001264,-0.0011555,0.00064497],[0.0014326,-0.0004241,0.00089558],[0.0015696,-0.00010188,0.0011713],[0.0014867,-0.0007879,0.00053734],[0.001248,0.00075941,0.001523],[0.00083556,-0.0011418,0.00043838],[0.00030152,0.0015402,0.0012406],[0.00082497,-0.0015429,0.00043231],[-0.00017847,0.0013043,0.00079814],[0.0015265,5.7562e-05,0.00071212],[0.0015671,0.00038335,0.0010931],[0.0014441,-8.5265e-05,0.00079491],[0.001305,-9.1342e-06,0.0009494],[0.0011347,-0.00010981,0.0011165],[0.00069002,6.7164e-05,0.00099254],[0.00095631,-0.00014397,0.0015637],[0.00061484,8.7256e-06,0.0010462],[0.0011246,-0.00040881,0.0013448],[0.00083372,-0.00014893,0.0014076],[0.00059487,-0.0013153,0.000491],[0.00026713,-0.0012069,0.00078529],[0.00090179,-0.0012579,0.0010526],[0.00070288,-0.0018327,0.00099303],[0.00071555,-0.0015504,0.00037632],[0.0011191,-0.0010218,0.00078679],[0.0012011,-0.00011339,0.0012714],[0.00078123,-0.00088005,0.001341],[0.0012332,-0.00072887,0.0011209],[0.0013004,-0.00055253,0.00074572],[0.0012407,-0.00093632,0.00058793],[0.0014042,-0.0015753,0.00069635],[0.0015161,-0.0010612,0.00050224],[0.0015885,-0.00022838,0.00064037],[0.0014536,-3.3832e-05,0.00084902],[0.0015233,-0.00059791,0.00035507],[0.0013552,0.00079103,0.0013319],[0.0011751,-0.0018838,0.00049782],[0.00034772,0.0015991,0.0012298],[0.0007939,-0.001629,0.00038579],[-3.3498e-05,0.0015056,0.00064293],[0.0005976,0.00012627,0.00038583],[0.0016411,-9.1509e-05,0.00077194],[0.0015557,0.00021121,0.001105],[0.0015148,-5.3664e-05,0.00083555],[0.0013636,-6.2173e-05,0.00098724],[0.0011869,-9.3375e-05,0.0011708],[0.0011178,8.9597e-05,0.0015384],[0.00098982,-1.039e-05,0.0016264],[0.0011788,1.4458e-05,0.0014313],[0.00085455,-0.00013825,0.0014076],[0.0010114,-0.00049364,0.001414],[0.00082906,-0.00039469,0.0013592],[0.00054735,-0.0014162,0.00044347],[0.00040633,-0.0013139,0.00043521],[0.00049117,-0.0014866,0.00070079],[0.00036201,-0.0013737,0.00060968],[0.00069051,-0.0015388,0.0003557],[0.00080349,-0.001435,0.00054296],[0.00096117,8.4573e-05,0.0014678],[0.0011821,-0.00032294,0.0012755],[0.0010964,-0.0011223,0.00093728],[0.0009873,-0.00054515,0.0013768],[0.0012356,-0.00091906,0.00067386],[0.0010241,-0.0013779,0.00041602],[0.0013829,-0.0015535,0.00034247],[0.0014721,-0.0010666,0.000279],[0.0016036,-0.0002727,0.00048881],[0.00084926,-0.00014971,0.00033503],[0.0017753,0.00019557,0.00091782],[0.0016136,-0.00066098,0.00019211],[0.0012973,0.0011625,0.0011084],[0.0011533,-0.0018173,0.00034769],[0.00013992,0.0017316,0.00075283],[0.00081837,-0.0016106,0.00022384],[-2.5092e-05,0.0015623,0.00068746],[0.00078809,0.0002573,2.4982e-05],[0.0016713,-3.3972e-05,0.00068322],[0.001535,-3.7736e-05,0.00081833],[0.0016926,0.00060396,0.00034135],[0.0014331,5.8137e-06,0.00091223],[0.0012842,1.1127e-06,0.001074],[0.0012003,-0.00010964,0.0011386],[0.001059,-0.00017958,0.0014147],[0.0011121,-0.00011348,0.0014089],[0.00096627,0.00014326,0.0015394],[0.0010294,-0.00020899,0.0014463],[0.00066229,-0.00024869,0.0016664],[0.0010643,-0.00055089,0.0012012],[0.00058993,-0.001458,0.0003526],[0.00043031,-0.0014288,0.00046363],[0.00051696,-0.0015239,0.00062313],[0.00079483,-0.001418,0.00052332],[0.00041942,-0.0014693,0.00048535],[0.00073555,-0.0015495,0.00018393],[0.00098955,-0.0013312,0.00039652],[0.00070225,-6.7655e-05,0.0015878],[0.0012274,-0.0002756,0.0012054],[0.00080446,-0.001245,0.0010073],[0.00051834,-0.00079951,0.0015048],[0.0012922,-0.00086218,0.00060607],[0.0010617,-0.001328,0.00036845],[0.0012684,-0.0016731,0.00023363],[0.0014226,-0.0013233,0.00018645],[0.0015448,-0.00085652,0.00010331],[0.0011684,-0.00017544,0.00032574],[0.0010677,-8.5907e-05,0.00027697],[0.0018018,0.00020453,0.00077278],[0.001542,-0.00054095,0.00011128],[0.0014876,0.0012207,0.00084189],[0.00092733,-0.0016749,0.00020944],[0.00036303,0.0018838,0.00043144],[0.00076647,-0.001594,0.00018628],[0.00013341,0.0017193,0.00041193],[0.0017255,0.00045566,6.0114e-05],[0.001712,-7.5708e-05,0.00070156],[0.0015796,-7.1578e-05,0.00084108],[0.0016808,-4.1918e-06,0.00037966],[0.001467,-0.00011204,0.00092383],[0.0013178,-8.6205e-05,0.001092],[0.0012321,-0.00018652,0.0011574],[0.0012128,-0.0002734,0.001265],[0.0009607,-0.00022872,0.0015058],[0.0010822,-0.00036945,0.0013357],[0.00067047,-5.8257e-05,0.0016858],[0.00095473,-0.00063694,0.0012654],[0.00063596,-0.00024995,0.001632],[0.0010829,-0.00011557,0.0012932],[0.00055631,-0.0015186,0.00031734],[0.00048648,-0.0015104,0.00033669],[0.00033988,-0.0014931,0.00046382],[0.00055712,-0.0015405,0.00034185],[0.00032597,-0.0014908,0.00040723],[0.00072582,-0.0015377,0.00017924],[0.00060597,-0.0015812,0.00013356],[0.001034,-0.0003218,0.0013096],[0.00070783,0.00013273,0.0015587],[0.0011485,-0.00064133,0.001174],[0.00044639,-0.00094423,0.00135],[0.00053957,-0.00080077,0.0014164],[0.0010804,-0.0013521,0.00045008],[0.00053724,-0.0016797,5.684e-05],[0.0012106,-0.0015814,6.9986e-05],[0.0013481,-0.0012665,3.761e-05],[0.0015683,-0.0008835,2.7881e-06],[0.0015862,-0.00024627,0.00023923],[0.0011619,-6.9175e-05,0.00035947],[0.0017199,0.00056595,0.00052246],[0.001677,-0.00060037,-4.2085e-05],[0.0011782,0.0015715,0.00033221],[0.00092573,-0.0016226,5.0424e-05],[0.00032989,0.0016914,0.00039382],[0.00078917,-0.0015762,3.4971e-05],[0.0001155,0.00164,0.00037042],[0.0016975,0.00039129,0.00018053],[0.0017627,4.8943e-05,0.00050802],[0.0016045,-1.9635e-05,0.00076622],[0.0015335,-5.057e-05,0.00085828],[0.0017831,0.00012766,0.00014003],[0.0013752,-3.4451e-05,0.0010429],[0.0013369,-9.6659e-05,0.001082],[0.0011076,-9.9704e-05,0.0012865],[0.0011608,-0.00047994,0.0011716],[0.0010577,-0.00030126,0.0013839],[0.0011159,-0.00039271,0.0012895],[0.00096957,-0.00027352,0.0014411],[0.00096024,-0.00064064,0.0012944],[0.00068487,-0.0002866,0.0015685],[0.0010206,-7.2835e-05,0.0013565],[0.00059225,-0.0015589,0.00022645],[0.00044532,-0.0015539,0.00030005],[0.00054111,-0.0015088,0.00038047],[0.00036453,-0.0015608,0.0004168],[0.00057438,-0.0015398,0.00033307],[0.00035878,-0.0015691,0.00033661],[0.00075866,-0.0015504,5.2476e-05],[0.00051076,-0.0016262,0.00019998],[0.00057749,-0.0005551,0.001415],[0.00063963,0.0001012,0.0015354],[0.0011869,-0.000595,0.0011117],[0.00079942,-0.00072321,0.0012192],[0.00099232,-0.00054149,0.0012964],[0.0010864,-0.0013359,0.00043714],[0.00040163,-0.0017383,0.00015295],[0.0010481,-0.0015574,-1.7389e-05],[0.0012874,-0.0014038,-4.0981e-05],[0.0014228,-0.0010631,-0.00010406],[0.0015913,-0.00072365,-0.0001077],[0.001509,-0.00021183,0.00021002],[0.0014911,-5.6162e-05,0.00027359],[0.0017801,0.0005536,0.00032417],[0.0016158,-0.00049494,-9.4689e-05],[0.0011577,0.0014747,0.00018912],[0.00084002,-0.0015597,1.3373e-05],[0.00044832,0.0018425,0.00020893],[0.00078234,-0.0015402,4.2352e-05],[0.0015772,0.00028361,-1.4977e-05],[0.00038603,0.0015652,-0.00028831],[0.0015966,0.00012909,0.00015534],[0.0017444,9.0639e-05,0.00050168],[0.0016179,2.3093e-05,0.00077378],[0.0015536,2.9073e-05,0.00087541],[0.0017013,0.00015769,0.00013303],[0.0013912,-1.6291e-05,0.0010571],[0.0013456,7.9428e-05,0.0011129],[0.0011124,-8.6353e-05,0.0012944],[0.0011114,-0.00044764,0.0012249],[0.0010393,-0.00045648,0.0013409],[0.00097028,-0.00036662,0.0014346],[0.0010839,-0.00046765,0.0012385],[0.00068152,-0.00045923,0.0015558],[0.00095903,-0.00011221,0.0013687],[0.0010063,-8.6802e-05,0.0014729],[0.001002,0.0001231,0.001364],[0.00053746,-0.0015938,0.0001753],[0.00048108,-0.0016086,0.00019849],[0.00015852,-0.0015866,6.0241e-05],[8.6607e-05,-0.0015665,0.00017764],[0.00015734,-0.0016265,3.5443e-05],[0.00017687,-0.0016426,0.00018553],[0.0006362,-0.0015896,-5.1588e-05],[0.00012572,-0.0016987,-3.9551e-05],[0.00080257,-0.00070656,0.0012257],[0.0006316,8.0265e-05,0.001451],[0.00060607,-8.2322e-05,0.001505],[0.00094917,-0.0011587,0.00097602],[0.00093262,-0.00082452,0.0011727],[0.00096324,-0.00051578,0.0013662],[0.00044292,-0.0018755,7.0408e-05],[3.3096e-06,-0.0017755,-9.089e-05],[0.0010107,-0.0014779,-0.00013395],[0.0011769,-0.0012834,-0.00019899],[0.0013534,-0.0010323,-0.00025827],[0.0015849,-0.00073519,-0.00023123],[0.0017226,-0.00023696,2.4326e-05],[0.0017477,0.00017224,0.00017862],[0.0015966,0.00086559,1.2668e-05],[0.0016835,-0.00052759,-0.00034893],[0.0010616,0.0014704,9.9945e-05],[0.00085191,-0.0015057,-0.00019289],[8.4097e-05,0.0017517,-0.00025256],[0.00081551,-0.0015278,-0.0001484],[0.0015663,0.00025743,2.8081e-05],[0.00055763,0.0014144,6.5623e-05],[0.0016838,0.00016617,9.9548e-05],[0.001683,0.0001464,0.00059658],[0.0017766,0.00023146,0.00026555],[0.001575,6.1305e-05,0.00083463],[0.0014851,8.2176e-05,0.00095624],[0.0017604,0.00016632,0.00013156],[0.0013178,4.3247e-05,0.0011534],[0.0013122,0.00010003,0.0011417],[0.00091293,3.172e-05,0.0014508],[0.0011005,-0.00021882,0.0012538],[0.00082567,-0.00031544,0.0015075],[0.00090538,-0.0003187,0.0015056],[0.00089538,-0.00034387,0.0014174],[0.00072944,-0.00049231,0.0015539],[0.0011074,-0.00020807,0.0012411],[0.00081999,4.1909e-05,0.0015951],[0.0011307,4.9093e-05,0.0012719],[0.00058408,-0.0016136,-2.8989e-05],[0.00031808,-0.0016694,5.7879e-05],[0.00015215,-0.001603,6.4615e-05],[0.0001484,-0.0016043,0.00015696],[8.5636e-05,-0.0016404,0.00022395],[0.00011579,-0.0016696,6.0382e-05],[0.00019604,-0.0016959,0.0001094],[0.00066756,-0.001565,-0.00026467],[-1.0057e-05,-0.0017475,4.1072e-05],[0.00045304,-0.0008581,0.0012543],[0.00047008,-1.2772e-05,0.0015091],[0.00053822,-0.0001067,0.0014889],[0.00094341,-0.0010227,0.00084477],[0.00083022,-0.00088873,0.0012012],[0.00075137,-0.00067718,0.0014825],[0.00029909,-0.0019466,0.00017174],[-0.00010445,-0.0018355,-2.5806e-05],[0.00087238,-0.0014636,-0.00020667],[0.0011112,-0.0012866,-0.00022691],[0.0012466,-0.0010934,-0.00030792],[0.0014042,-0.00091944,-0.0003395],[0.0016362,-0.00054835,-0.00035771],[0.0015499,-1.6655e-06,-0.00011268],[0.0016208,0.00015191,-6.8689e-05],[0.0013561,0.00073784,2.2238e-05],[0.0016962,-8.8176e-05,-0.00057406],[0.0011547,0.0014894,-7.797e-05],[0.00085838,-0.0014706,-0.00017156],[0.00012491,0.0018085,-0.00040993],[0.00066314,-0.0015563,-0.00026289],[0.0014572,0.00015824,-0.00011862],[0.001629,0.00011426,1.5042e-05],[0.00073861,0.0010562,-0.00067636],[0.0016787,0.00014397,9.8558e-05],[0.0016841,0.00024599,0.00059936],[0.0017556,0.00011521,0.00026378],[0.0015702,0.00024301,0.00084397],[0.0014824,0.00016914,0.00096294],[0.0017263,0.00015247,0.00012899],[0.0013357,-4.1112e-05,0.0011578],[0.0013229,-8.538e-05,0.0011245],[0.00090771,-3.1509e-06,0.0014355],[0.00093359,-0.00010906,0.0014282],[0.00080431,-0.00040989,0.0014564],[0.0008879,-0.00027103,0.0014796],[0.00073738,-0.00043149,0.0015734],[0.00089716,-0.00014116,0.0014533],[0.00079298,-0.00044155,0.0015128],[0.0011126,3.3089e-05,0.0012835],[0.0003303,-0.00026465,0.0017461],[0.0011196,0.00011399,0.0012687],[0.00045675,-0.0016368,-0.00013341],[0.0003445,-0.0016798,-8.6072e-05],[-8.0694e-05,-0.0016566,-0.00012268],[-0.00032396,-0.0016452,-0.00025825],[-0.00048392,-0.0016603,-0.00022129],[6.7097e-05,-0.0017038,2.5528e-05],[-0.0001718,-0.0016831,-0.00019573],[0.00054503,-0.0015719,-0.00035219],[-0.0001142,-0.001643,-3.3501e-05],[0.00076132,-0.0010699,0.0010582],[0.00054309,-8.1773e-05,0.0013968],[0.00045639,-4.884e-05,0.0014751],[0.00042422,-0.00045019,0.0013087],[0.00024562,-0.001868,0.00042801],[0.00070246,-0.00081391,0.0013634],[0.00033219,-0.00037999,0.0017798],[-9.8378e-05,-0.0018242,-8.3917e-05],[-0.0011586,-0.0014103,-0.00070832],[0.00086899,-0.0013976,-0.0004001],[0.0010469,-0.0012039,-0.00027506],[0.0011159,-0.00098282,-0.00031399],[0.0013054,-0.00087697,-0.00045722],[0.001496,-0.00055395,-0.00074322],[0.0015814,5.9029e-06,-0.00048612],[0.0014341,0.00020114,-0.00010989],[0.0014054,0.00093845,-0.00014108],[0.0015451,-8.5091e-05,-0.00074201],[0.00070638,0.0016623,-0.00058134],[0.00086566,-0.0013457,-0.00051455],[-8.1009e-06,0.0016392,-0.00049012],[0.00067239,-0.0014704,-0.00049903],[0.0014689,0.00015569,-0.00011137],[0.0015974,0.00011154,1.5744e-05],[0.0009733,0.0008422,-0.00011289],[0.0017248,0.00014507,0.00010696],[0.0016981,0.00027538,0.0005566],[0.001608,0.00032228,0.00073516],[0.001796,0.00017639,0.00016608],[0.0015375,0.00025481,0.00085949],[0.001496,0.00015959,0.00094782],[0.0017652,0.00016499,0.0001146],[0.0013606,-6.3118e-05,0.0011168],[0.0010331,9.8055e-05,0.0013955],[0.00096875,-3.7006e-05,0.0013897],[0.00088945,0.00022183,0.0014161],[0.00065071,-0.00031092,0.0015174],[0.00077225,-0.00018586,0.0015904],[0.0007025,-0.00040753,0.0015993],[0.00073909,-5.8454e-05,0.0015025],[0.00086437,-0.00048799,0.0015036],[0.00083956,0.00017771,0.0014467],[0.0006541,-0.00048955,0.0015756],[0.0009303,0.00021294,0.0013901],[0.00048316,-0.0016076,-0.00032651],[0.00031309,-0.0016751,-0.00010994],[-4.7977e-05,-0.0016445,-0.00037631],[9.1496e-05,-0.0016688,-0.00020429],[-0.00033663,-0.0017296,-0.00030036],[-0.00050574,-0.0016494,-5.375e-05],[-0.00013247,-0.0018468,0.00013077],[-0.00021938,-0.0016961,0.0001159],[0.000566,-0.0015522,-0.00050858],[-0.0011462,-0.0012269,0.00064646],[0.00083404,-0.0010507,0.0010659],[-2.7218e-06,-0.00036572,0.0014816],[0.00025478,-0.00013506,0.0014336],[0.00055581,-0.00028236,0.0010164],[0.00026668,-0.0018765,0.00041828],[0.00062877,-0.00084802,0.0013648],[0.00063303,-0.00022406,0.0017362],[-1.4841e-05,-0.0010732,-7.6591e-05],[-0.0012721,-0.0012477,-0.00054039],[0.00086969,-0.0012849,-0.0004031],[0.00075631,-0.0013992,-0.00046175],[0.0010784,-0.0009992,-0.00033238],[0.0011682,-0.0007928,-0.00041372],[0.0014259,-0.00052033,-0.00070333],[0.0015115,-0.00029772,-0.00088139],[0.0017553,0.00023662,-0.00062909],[0.0013563,0.00018979,-0.00048069],[0.0012732,0.00061147,-0.00055689],[0.0015324,0.00014524,-0.00083493],[0.00072177,0.001526,-0.0008119],[0.00080129,-0.0013459,-0.00055129],[2.4554e-05,0.0015651,-0.00086385],[0.00063447,-0.0014872,-0.00053324],[0.001591,2.699e-05,-0.00011637],[0.0014838,0.00019374,-6.0469e-05],[0.0016816,8.8628e-05,1.6033e-05],[0.00099124,0.00017993,-0.00083551],[0.0017176,0.00015766,0.00010663],[0.0017067,0.00020929,0.00055773],[0.0016308,0.00020559,0.00073805],[0.0017472,0.00028901,0.00016129],[0.0015441,0.0002465,0.00086239],[0.001506,0.00014177,0.0009522],[0.0017415,0.00023424,0.00011265],[0.0013313,0.00017608,0.0011274],[0.0010029,0.00016546,0.0013686],[0.00092517,0.00015432,0.0013611],[0.00075838,0.00028304,0.0014628],[0.00072032,-0.00025694,0.0014807],[0.00064495,-0.00012817,0.0015329],[0.0006999,-0.00022751,0.0015962],[0.0008842,-0.00029882,0.001551],[0.00070419,0.00023475,0.0014991],[0.0006431,-0.00061695,0.0015521],[0.00085069,0.00013453,0.0014561],[0.00028736,-0.00067525,0.0016011],[0.00095128,0.00020858,0.0014199],[0.00043251,-0.0015917,-0.00035661],[0.00034172,-0.0016526,-0.00032116],[0.0003067,-0.0017387,-9.039e-05],[0.00023304,-0.0017101,-0.00010035],[0.00010895,-0.0017732,6.4877e-05],[-0.00016148,-0.0017022,0.00023486],[-9.2485e-05,-0.0018294,0.00015575],[-0.00020394,-0.0016831,0.000126],[0.00045008,-0.0015088,-0.00055833],[-0.0010166,-0.0013383,0.00080875],[0.0006071,-0.00093277,0.0013064],[0.00080941,-0.00092509,0.0010564],[3.8569e-05,2.0344e-05,0.001548],[0.00025324,-6.6865e-05,0.0013942],[0.00014558,-0.0011138,0.00032312],[-5.8907e-06,-0.0016238,0.00017479],[0.00054501,-0.00081018,0.0015031],[0.00039281,-4.7817e-05,0.0018505],[-0.00071945,-0.00071692,-0.0004856],[-1.4822e-05,-0.0010732,-7.6631e-05],[-0.00039968,-0.0016618,-9.8175e-05],[0.00080824,-0.0010995,-0.00070066],[0.00073403,-0.0012571,-0.00074315],[0.00076793,-0.00069864,-0.00046752],[0.0010757,-0.00077181,-0.00061795],[0.0013622,-0.00051915,-0.00082216],[0.0014942,-0.00029519,-0.00088171],[0.0015533,0.00023242,-0.00076967],[0.0013394,0.00074757,-0.00079989],[0.0012397,0.00017192,-0.0012484],[0.00040628,0.0014818,-0.001047],[0.00078555,-0.0011773,-0.00085729],[-8.3114e-05,0.0013475,-0.00083869],[0.00062873,-0.0012872,-0.0008536],[0.001597,5.6448e-05,-0.00018372],[0.0014417,0.00028238,-0.0002991],[0.0016622,0.00013495,-8.1897e-05],[0.0013307,0.00018515,-0.00075287],[0.0017441,0.00015508,0.00011805],[0.0017583,0.00037357,0.00030261],[0.0016639,0.00026529,0.00065909],[0.001569,0.00026708,0.0008479],[0.0017682,0.0002801,0.00018522],[0.0015618,0.00023461,0.00084249],[0.0013845,0.00022717,0.0010928],[0.0017737,0.00024246,0.00010746],[0.0011345,0.00029243,0.0013079],[0.00096568,0.00017881,0.001377],[0.00087797,0.00017178,0.0013732],[0.00072757,0.00039509,0.0014304],[0.00059376,-0.00017059,0.0014989],[0.00046162,-5.1845e-05,0.0013769],[0.00062132,-0.00016452,0.0017343],[0.00038036,3.6465e-05,0.001802],[0.00059717,0.00026999,0.0014873],[0.000438,-0.00047681,0.0016786],[0.00057557,0.00025026,0.0015468],[9.8942e-05,-0.00053467,0.0016698],[0.00056841,0.00035782,0.0015454],[0.00045529,-0.0015406,-0.00056794],[0.00031776,-0.0016201,-0.000328],[0.00029982,-0.0017694,-1.7152e-05],[0.00022218,-0.001682,-5.4803e-05],[-0.0001017,-0.0019704,4.4916e-05],[9.8136e-05,-0.0017472,0.00012605],[-0.00014212,-0.0017196,0.00010172],[-0.0011972,-0.0015806,0.00074993],[-0.00016122,-0.0016363,-0.00011722],[0.0004367,-0.0011765,-0.0010371],[-0.00036062,-0.0016535,0.00023777],[-1.2441e-05,-0.0011662,0.0012312],[1.2257e-05,-0.0012306,0.0010012],[0.0002029,0.00010633,0.0015376],[0.00029544,-5.1039e-05,0.001445],[0.00010981,-0.0011629,0.00037003],[8.9013e-05,-0.0015185,8.2416e-05],[7.1352e-05,-0.00099958,0.0014421],[0.0004232,-4.2894e-05,0.0019224],[-0.0001199,-0.00097078,-0.0013401],[-0.00018055,-0.00095535,-0.00017069],[-0.00034884,-0.001698,-0.0001521],[0.00077757,-0.0008648,-0.00063955],[0.0007816,-0.0011316,-0.00074363],[0.0006516,-0.0012717,-0.00079932],[0.00065447,-0.0005186,-0.00040525],[0.0012085,-0.00058259,-0.00081231],[0.0013937,-0.0003953,-0.00090918],[0.0012544,0.00037975,-0.0010639],[0.0012831,0.00065953,-0.00083569],[0.00086892,0.00072627,-0.0012693],[0.00038671,0.0012679,-0.0012648],[0.00073333,-0.0011708,-0.00087831],[-9.9275e-05,0.0013284,-0.0010937],[0.00030919,-0.0011788,-0.00094974],[0.0016299,-5.6294e-06,-0.00028089],[0.0016489,6.4186e-05,-0.00019005],[0.0015137,-3.5297e-05,-0.00074604],[0.0017135,2.9046e-05,-8.1554e-05],[0.0012814,-2.9601e-05,-0.00090175],[0.001742,0.00024555,0.0001175],[0.0017174,0.0004489,0.00029602],[0.0016151,0.00045504,0.0006508],[0.0015384,0.00037894,0.00084215],[0.0017179,0.00045103,0.00018044],[0.00151,0.00044771,0.0008346],[0.0013059,0.00050741,0.0010727],[0.0017379,0.00043234,0.0001038],[0.0010656,0.00043543,0.0012612],[0.00083426,0.00041313,0.0012409],[0.0010043,0.00039855,0.0015921],[0.00062617,0.00042958,0.0014409],[0.00052886,-0.00020356,0.0014846],[0.00053007,-4.9672e-08,0.0013503],[0.00047684,0.00017012,0.0015423],[0.00036426,-0.00032342,0.0017777],[0.00049119,9.7414e-05,0.0018126],[0.00056688,0.00037683,0.0014488],[8.3666e-05,-0.00063854,0.0016265],[0.0005555,0.00034716,0.0015358],[-0.00012725,-0.00062468,0.0016149],[0.00054173,0.00058912,0.0015807],[0.00028284,-0.0014643,-0.00064537],[0.0003504,-0.001493,-0.00068965],[0.00024024,-0.0017367,-6.2616e-05],[0.00021481,-0.0016479,-5.5992e-05],[8.421e-05,-0.0019278,0.00017312],[-2.1036e-05,-0.0017394,2.6607e-05],[-0.00013443,-0.0017223,0.00010846],[-0.0008043,-0.0016131,0.00099607],[0.00013814,-0.0016167,0.00011852],[0.00038025,-0.0011254,-0.0010201],[-0.00022442,-0.0016113,0.00029522],[-4.5098e-05,-0.0012855,0.0011104],[1.8123e-05,-0.0012223,0.001251],[0.00025912,8.2099e-05,0.0015971],[0.00020299,5.3774e-06,0.0015608],[0.00013629,-0.0013646,0.00043859],[-6.2071e-05,-0.0015559,5.0848e-06],[0.00021541,-0.0011565,0.0014554],[1.3212e-05,0.00026724,0.0019654],[-0.00035599,-0.00084299,-0.0014491],[-0.00014136,-0.0014143,-0.00041005],[3.5381e-05,-0.0016743,2.8839e-05],[0.00069849,-0.0007466,-0.00063433],[0.0007532,-0.0010745,-0.00075825],[0.00063349,-0.0011836,-0.00092011],[0.00063704,-0.00056146,-0.00054403],[0.0010125,-0.00053427,-0.00091712],[0.0010322,-0.00034993,-0.001112],[0.0010332,0.0003724,-0.0012176],[0.00091547,0.00075327,-0.0013531],[0.00058852,0.00072633,-0.0014626],[0.00023136,0.0011292,-0.0012143],[0.00070274,-0.0010311,-0.0010184],[-8.3818e-06,0.0012133,-0.00093415],[0.00028715,-0.00096381,-0.00116],[0.0016916,2.9418e-05,-0.00039425],[0.0016462,0.00013617,-0.00035428],[0.0017453,-0.00010969,-0.00055316],[0.0017278,9.7656e-05,-0.00022864],[0.0015671,-2.366e-05,-0.00080373],[0.0017396,0.00035474,-9.4117e-05],[0.0017311,0.00048855,0.00023592],[0.0016449,0.00056043,0.00049378],[0.0015816,0.0004834,0.00070269],[0.0014915,0.00040586,0.00088913],[0.0017376,0.00046849,0.00016073],[0.0013369,0.00055612,0.0010228],[0.0011704,0.00056877,0.0011703],[0.0017456,0.00047148,3.5586e-05],[0.00092779,0.00048209,0.0013181],[0.00088464,0.00038358,0.0011963],[0.00050221,0.00055784,0.0017198],[0.00062077,0.0003305,0.0014005],[0.00018831,-1.9819e-05,0.00089883],[0.00027729,4.4889e-05,0.00086401],[0.00044147,0.00018197,0.0015394],[0.00011899,-0.000187,0.0019608],[0.00030196,0.00022458,0.0018766],[0.00073499,0.00029393,0.0013691],[-4.0636e-05,-0.00054794,0.0016653],[0.00061979,0.00031391,0.0014919],[-0.00011438,-0.00063042,0.0016019],[0.00055655,0.00056524,0.0015349],[0.00030274,-0.0012576,-0.0010662],[0.00048208,-0.0014537,-0.00056009],[0.00021996,-0.0016899,-5.5067e-05],[0.00024484,-0.0017436,-8.6931e-05],[8.8491e-05,-0.00153,-3.0927e-06],[-0.00017487,-0.0013073,0.00017401],[-2.7167e-05,-0.0017563,6.5995e-05],[-0.00015737,-0.0017378,0.00024702],[-0.00019323,-0.0015356,0.00041777],[-0.00055678,-0.001158,0.00083182],[0.00024733,-0.001547,-0.00048946],[0.00036101,-0.00083882,-0.0012158],[3.8978e-05,-0.0016173,5.9438e-05],[3.426e-05,-0.00122,0.0010913],[-8.0966e-05,-0.0012205,0.0011926],[0.00031855,0.00010774,0.0016832],[0.00027885,4.3473e-05,0.0016321],[8.1459e-05,-0.0013103,0.00046781],[-4.9104e-06,-0.0014147,-4.8655e-05],[-0.00013127,-0.0012729,0.0013656],[-0.00041867,8.5444e-05,0.0018704],[-0.00050422,-0.00091041,-0.0013858],[-0.00034818,-0.0012876,-0.00048727],[-5.1256e-05,-0.0016716,0.00010553],[0.00079176,-0.00084143,-0.00071199],[0.00064077,-0.0011568,-0.00091421],[0.00054341,-0.0011915,-0.00097866],[0.00092542,-0.00061921,-0.00092899],[0.0010635,-0.00033031,-0.0011176],[0.00086559,0.00012689,-0.0012346],[0.00056511,0.00089623,-0.0011143],[0.00021942,0.001044,-0.0013224],[0.00028431,-0.00095965,-0.0011535],[-3.288e-05,0.00090677,-0.0011577],[0.00010113,-0.00065699,-0.00085166],[0.0015991,0.00010671,-0.00038201],[0.001437,-3.0193e-05,-0.00041913],[0.0016763,7.3022e-06,-0.00034925],[0.0015091,-0.00026499,-0.00062069],[0.00078935,-8.8864e-05,-0.00022936],[0.0017357,0.00025672,-0.00024012],[0.0015331,-0.00011303,-0.00084896],[0.0017508,0.00038656,-9.5849e-05],[0.0016674,0.00065546,0.0002283],[0.0016271,0.000581,0.00048945],[0.0015559,0.0005466,0.00069662],[0.0014324,0.00058092,0.00087499],[0.0016807,0.00063561,0.00015687],[0.0012993,0.00058513,0.0010006],[0.0011298,0.00057869,0.0011354],[0.0016969,0.00063467,3.1403e-05],[0.00083487,0.00054555,0.0012198],[0.00066874,0.00044441,0.00085554],[0.00056688,0.00041778,0.0017928],[0.00085203,0.00023382,0.0013228],[0.00030028,-1.2659e-05,0.0021286],[0.00046389,0.00019579,0.0015685],[0.00045496,5.3212e-05,0.0015186],[0.00027614,-8.1589e-05,0.0018716],[0.00013593,-0.00011946,0.0021156],[4.0092e-05,8.4835e-05,0.0018685],[0.00073842,0.00013713,0.0013486],[-0.00010443,-0.00056428,0.0016114],[0.00054456,0.00055447,0.001433],[0.00022606,-0.00047337,0.0016297],[0.00060252,0.00044875,0.0015646],[0.00048474,-0.0013127,-0.00097606],[0.00054842,-0.0013226,-0.0010127],[0.00026368,-0.0016761,-1.8811e-05],[5.0004e-05,-0.001735,-0.00024861],[0.00013336,-0.0015394,2.8693e-05],[-0.00014376,-0.0013294,0.00019961],[5.2771e-05,-0.0017469,0.00013339],[0.0001177,-0.0016903,0.00046753],[-0.00028489,-0.0015355,0.00036858],[-0.00063068,-0.0011618,0.00079173],[0.00022646,-0.0015055,-0.00048829],[0.00041248,-0.00099214,-0.0014595],[7.5263e-05,-0.0016118,7.7575e-05],[-7.5805e-05,-0.0011715,0.0011685],[0.00029537,0.00013123,0.001739],[-0.00012607,-0.0012024,0.0012246],[0.0003124,3.1114e-05,0.0016792],[8.6327e-05,-0.0013675,0.00048892],[-8.306e-05,-0.0013854,-8.3988e-05],[-0.00058112,-0.00093932,0.0014942],[0.00024018,-0.00041579,0.0016829],[-0.00026223,-0.0010241,-0.0013062],[-0.00019048,-0.0012394,-0.00063978],[5.7995e-05,-0.001696,0.00015623],[0.00081333,-0.00063101,-0.0010343],[0.00056854,-0.00080687,-0.0013106],[0.00042196,-0.0006941,-0.0014083],[0.00083049,-0.00060957,-0.0010316],[0.00081291,-0.00029039,-0.0012435],[0.00036138,7.2252e-05,-0.00091993],[0.00014557,0.0010088,-0.0016576],[2.1361e-05,0.000964,-0.0013192],[0.00021547,-0.00059393,-0.0013696],[0.00050415,0.00087001,-0.000804],[9.2448e-05,-0.00050547,-0.0010806],[0.0015991,0.00013795,-0.00046452],[0.001399,2.346e-05,-0.00059261],[0.0017095,-1.4079e-05,-0.00030377],[0.0016996,-0.00030705,-0.00054496],[0.00073228,-4.9113e-05,-0.00036269],[0.0016876,0.00030896,-0.00035699],[0.0016475,-0.00011268,-0.0006005],[0.0016932,0.00047482,-0.0002874],[0.0016402,0.00069564,0.00029912],[0.0016657,0.0006657,0.00020892],[0.0015762,0.00062716,0.00057463],[0.0014479,0.00061727,0.00082687],[0.0013309,0.00062975,0.0009632],[0.0016747,0.00066891,9.4098e-05],[0.0011823,0.0006444,0.0011021],[0.00092167,0.00064511,0.0012384],[0.0016499,0.00070309,-0.00012339],[0.00084854,0.00053606,0.0012044],[0.00056458,0.00046506,0.00087579],[0.00057226,0.00039795,0.0017303],[0.00077239,0.00024939,0.0012043],[0.00028749,-0.0001767,0.0019024],[0.00075162,0.00016459,0.0017431],[0.00072211,-4.4387e-05,0.0014931],[0.00063743,-0.00027755,0.0016964],[0.00064422,-0.00032642,0.00175],[0.00010067,3.8491e-05,0.0017557],[0.00052222,0.0002221,0.0014146],[-0.0002396,-0.00043493,0.0015752],[0.00096311,0.00038945,0.0012942],[-1.0487e-05,-0.00028594,0.0016549],[0.00049629,0.00045925,0.0015303],[0.00051721,-0.00092548,-0.0014469],[0.00063608,-0.0012881,-0.00090497],[0.00014284,-0.0016504,3.2158e-05],[0.00030607,-0.0016768,-0.00035478],[-4.5388e-06,-0.0017511,0.00015797],[-0.00015553,-0.0014867,0.00017355],[-0.00070357,-0.0013405,0.00056285],[-5.9889e-07,-0.0017362,0.00052873],[0.00023528,-0.0016118,-0.00051978],[7.4856e-05,-0.0015836,8.379e-05],[-0.00025106,-0.0014535,0.0005411],[0.00026181,-0.0014045,-0.00077859],[0.00042621,-0.00065858,-0.0016408],[4.9057e-05,-0.0016056,9.8585e-05],[5.8391e-05,-0.0010922,0.0011672],[0.00044546,0.0001991,0.001787],[-0.00011339,-0.001169,0.0011968],[0.00037306,5.9801e-05,0.0017141],[6.299e-05,-0.0013537,0.00050696],[-1.9602e-05,-0.0013937,-0.00015435],[-0.00029469,-0.00080131,0.0014969],[-0.00053051,-0.00069736,0.0014404],[-0.00036151,-0.0010306,-0.0011849],[-0.00027511,-0.0012285,-0.00068171],[1.3832e-06,-0.0016927,0.00020187],[0.00067951,-0.00069043,-0.0013251],[0.00034974,-0.00079555,-0.0014283],[0.00021799,-0.00069015,-0.0015327],[0.00082573,-0.00031875,-0.0012907],[0.00062226,0.00019297,-0.0013376],[7.6669e-05,0.00053151,-0.00084713],[2.0863e-05,0.00095864,-0.0013157],[0.00021415,-0.00076881,-0.0018025],[0.00036803,0.00014753,-0.00089681],[0.0014714,-7.4169e-06,-0.00061837],[0.001641,-0.00012193,-0.00043964],[0.0012799,-5.255e-05,-0.0006379],[0.0016764,0.00032186,-0.00033265],[0.0015679,-0.00027197,-0.00049417],[0.0012672,4.1594e-05,-0.00055438],[0.0016832,0.00041687,-0.0003671],[0.0016775,-0.00040895,-0.00082139],[0.0016764,0.00059905,-0.00029493],[0.0016367,0.00065033,0.00029776],[0.0016511,0.00066365,0.00020715],[0.0015943,0.00054702,0.00057587],[0.0014625,0.00055935,0.00082802],[0.0013,0.00069411,0.00095308],[0.0016537,0.00070525,9.2973e-05],[0.0011442,0.00071015,0.0010842],[0.00090637,0.00068887,0.001237],[0.001662,0.0006748,-0.0001228],[0.00061213,0.00046115,0.00087595],[0.00056461,0.00046509,0.00087583],[0.00041346,0.00067252,0.0016052],[0.00052434,0.00034276,0.001266],[0.0003022,-0.00018107,0.0018942],[0.00072341,-7.8428e-05,0.0015991],[0.00092147,0.00039609,0.0019533],[3.714e-05,-0.00061837,0.0016482],[0.00066027,-0.0003193,0.0017566],[0.0006459,6.7569e-05,0.0018047],[-0.00020918,-0.00011922,0.0017136],[-1.6414e-05,-0.00032132,0.0015735],[0.00080155,0.00083847,0.0012045],[-0.00046736,-0.00045825,0.001446],[0.00046934,0.00055146,0.0015436],[0.00089923,-0.0010437,-0.0012265],[0.00072989,-0.0012898,-0.0013687],[2.9709e-05,-0.0016505,-4.9734e-05],[0.00025935,-0.0017026,-0.00039991],[3.3037e-05,-0.0018201,0.00019643],[-0.00026983,-0.0014991,9.8137e-05],[-0.00083447,-0.0013008,0.00045369],[-1.9395e-05,-0.0017928,0.00052925],[0.00058355,-0.001692,-0.00024637],[5.8479e-05,-0.0016249,7.6133e-05],[-0.00048786,-0.0014761,0.00041487],[0.0010197,-0.0012854,-5.1105e-05],[0.0012558,-0.0010903,-0.00090714],[5.1269e-05,-0.0016435,0.00010144],[-8.5913e-05,-0.0009801,0.0012225],[0.00033978,0.00027958,0.001867],[0.00042918,3.866e-05,0.001767],[-0.0003215,-0.0010194,0.001278],[0.00014334,-0.001388,0.00055343],[-0.00013737,-0.0013981,-0.0002058],[-0.0005161,-0.00059336,0.0014973],[-0.00053084,-0.00069455,0.0014373],[-0.00017161,-0.0011144,-0.0011251],[-0.00023195,-0.0012054,-0.00071397],[5.0991e-05,-0.0017382,0.00022818],[0.00051095,-0.00014974,-0.0014943],[0.00010301,-2.7555e-05,-0.0015176],[6.9233e-05,-9.3928e-05,-0.0015107],[0.00043614,-0.00024184,-0.0015079],[0.00027421,0.00031821,-0.0016146],[2.8798e-05,0.00073092,-0.0014314],[0.00016002,0.00090223,-0.0012209],[0.0001869,-0.00051574,-0.0016871],[0.00089589,-4.6458e-05,-0.0005141],[0.0015526,-4.3857e-05,-0.00052962],[0.0016204,-7.0539e-05,-0.00058376],[0.0013019,-6.0508e-05,-0.00061887],[0.0016022,0.00039945,-0.00051276],[0.0014937,-0.00024537,-0.0007837],[0.0012169,8.5454e-05,-0.00076775],[0.001605,0.00048782,-0.00053578],[0.0016645,-0.00040723,-0.00066155],[0.0016402,0.00061493,-0.00034319],[0.0016357,0.00068204,0.00024196],[0.001612,0.00068311,0.0003564],[0.0016386,0.00071261,0.00011258],[0.001486,0.00062862,0.00073016],[0.0012885,0.0006562,0.0010098],[0.0011649,0.00074557,0.001052],[0.0016309,0.00074654,4.8174e-06],[0.0009474,0.00076267,0.0011787],[0.00088485,0.00068461,0.0012286],[0.0016418,0.00069827,-0.00017819],[0.00086645,0.00084403,0.0014083],[0.00057851,0.00089725,0.0014716],[0.00052395,0.0006213,0.0015236],[0.000403,0.0011052,0.0016936],[0.00027049,1.5258e-05,0.0018416],[0.00033946,6.7788e-05,0.001781],[0.00054703,0.00055643,0.0021038],[-0.00020951,-0.00045094,0.0016139],[0.00016831,-8.6354e-05,0.0017655],[0.00026384,0.00019752,0.0017918],[-0.00018877,-0.00011935,0.0016214],[-0.00034992,-4.6287e-05,0.0015584],[0.00039448,0.00097855,0.0013045],[-0.00053191,-0.00039189,0.0014309],[0.00034388,0.00055665,0.0015055],[0.00084733,-0.0012616,-0.00098509],[0.0016127,-0.0013165,-0.00056854],[-0.00028655,-0.0015976,9.4249e-05],[5.2559e-05,-0.0017099,-0.00030985],[0.00018938,-0.0017278,0.00011711],[-1.9174e-05,-0.0018098,0.00053412],[-0.00087717,-0.001376,0.00042609],[-0.00049105,-0.0014934,0.00025573],[7.5098e-05,-0.001864,-7.2797e-05],[0.00053325,-0.0017708,9.0874e-05],[5.0446e-05,-0.0016437,8.2997e-05],[5.6996e-05,-0.0015908,6.7148e-05],[0.0010631,-0.0011594,-0.00040636],[0.0011403,-0.0011042,-0.00076372],[4.0454e-05,-0.001695,0.00011363],[0.00013506,-0.00086009,0.0012405],[0.0006213,0.0004055,0.0018712],[0.00056222,0.00010133,0.0018181],[-9.8042e-05,-0.00093168,0.0013344],[0.00016457,-0.0014476,0.00056042],[-6.0375e-05,-0.0014001,-0.00028693],[-0.00045471,-0.00058051,0.0015453],[-0.0005537,-0.00071634,0.0014736],[-0.0002695,-0.00111,-0.0010063],[-0.0002281,-0.0012204,-0.00071929],[2.8645e-05,-0.0017284,0.0002438],[8.4301e-05,-4.7177e-05,-0.0015093],[0.00012759,-3.4682e-05,-0.001495],[0.00044428,-0.00022075,-0.0014455],[0.00023269,0.00026734,-0.0015269],[-0.00017893,0.0009002,-0.0016927],[-2.3309e-05,0.00027771,-0.0011244],[0.0012233,-0.00096588,-0.00070313],[0.00089589,-4.6489e-05,-0.00051411],[0.0014496,-0.00017985,-0.00066434],[0.0015932,-4.5927e-05,-0.00054331],[0.0015701,0.00030336,-0.00063165],[0.0012955,-0.00017309,-0.00073808],[0.0015959,0.00049084,-0.00052417],[0.0015752,-0.00013552,-0.0007159],[0.0013402,0.00010503,-0.00083426],[0.0016268,0.00047349,-0.00054007],[0.0016094,-0.00049848,-0.00072379],[0.0016618,0.00059476,-0.00034487],[0.0015661,0.00079099,0.00023432],[0.001584,0.00070826,0.00035122],[0.0015761,0.00082111,0.00010887],[0.0014936,0.00060809,0.00073154],[0.0012829,0.00067491,0.0010095],[0.0012346,0.00058517,0.0010704],[0.0015613,0.00087061,3.2424e-06],[0.00096625,0.00076241,0.001197],[0.00078322,0.0009525,0.0012385],[0.0015135,0.00093216,-0.00018139],[0.00088358,0.00092377,0.0011943],[0.00052252,0.00097149,0.0014285],[0.0004437,0.00080945,0.001506],[0.00060139,0.00104,0.0016574],[0.00025628,2.104e-05,0.0018585],[0.00033046,0.00039564,0.0018934],[0.00036768,0.00084267,0.0017476],[-0.00020411,-0.0003201,0.0016414],[-0.00021457,-0.00044314,0.0015597],[0.00015962,0.00012068,0.0017697],[0.00028513,4.4966e-05,0.0018204],[-0.00037688,-0.00021424,0.0015665],[-0.00040006,-8.8571e-05,0.0013119],[0.00048893,0.00073873,0.0013573],[-0.00055788,-0.00040326,0.0014395],[0.00014831,0.00093267,0.001387],[0.0012558,-0.0014729,-0.00069645],[0.0011183,-0.00089444,-0.00045375],[-0.00051779,-0.001536,-7.8764e-05],[0.00021303,-0.0016606,-0.00018896],[0.00034545,-0.0016853,0.0002355],[-8.0254e-05,-0.0018567,0.00049301],[-0.0010417,-0.0012732,0.00026083],[-0.00041448,-0.0015626,0.00033227],[6.6566e-05,-0.0019981,-9.0803e-05],[0.00036051,-0.0019008,-0.00013391],[3.3467e-05,-0.0016787,7.5101e-05],[3.0982e-05,-0.0016365,5.2636e-05],[0.001292,-0.0011335,-0.00018534],[0.0012908,-0.0010955,-0.00037045],[3.8443e-05,-0.0017669,0.00011665],[0.00038583,0.0005598,0.0019563],[-6.6678e-05,-0.0007127,0.0013285],[0.00060006,9.4088e-05,0.0018712],[-0.00043152,-0.00062793,0.0013824],[0.00012431,-0.0015323,0.00057503],[-0.00011978,-0.0014016,-0.0003128],[-0.00053471,-0.0005377,0.0015916],[-0.00063539,-0.00062527,0.0014557],[-0.00011478,-0.0011717,-0.00095815],[-0.0001932,-0.001221,-0.00075793],[5.0976e-05,-0.0018958,0.00027474],[-2.6958e-05,0.00025618,-0.0015523],[0.00011578,5.1908e-06,-0.0014936],[0.00050427,-0.00067075,-0.0013892],[2.0434e-05,0.00031689,-0.0015708],[-0.00016867,0.00085209,-0.0016007],[0.001294,-0.00085536,-0.00082765],[0.0013148,-0.00017899,-0.00090755],[0.001489,-0.00018206,-0.0006952],[0.0015737,-6.2639e-06,-0.00066444],[0.0015572,0.0003062,-0.00063923],[0.0012331,-0.00013061,-0.00090846],[0.001479,0.00057113,-0.00072844],[0.0014167,-8.2285e-05,-0.00098033],[0.0013427,0.0001145,-0.00088363],[0.0015925,0.00050582,-0.00061902],[0.0015948,-0.00025568,-0.00079366],[0.0014296,-0.00043913,-0.0010083],[0.0014489,0.00074685,-0.0007441],[0.0015401,0.00082863,0.00014369],[0.0015666,0.00077659,0.00021388],[0.001528,0.00077519,0.00046996],[0.0015312,0.0008568,7.4464e-06],[0.001303,0.00072073,0.00095387],[0.0012819,0.0006665,0.00099516],[0.00091796,0.00068041,0.0012576],[0.0014895,0.00093167,-0.0001582],[0.0007026,0.00081114,0.0012899],[0.0013582,0.0010422,-0.00048705],[0.00077414,0.0011493,0.0011657],[0.00054713,0.00094588,0.0013915],[0.00065774,0.0011582,0.001189],[6.0556e-05,0.00083089,0.0014967],[0.00045545,0.00094315,0.0013273],[0.00020853,0.00034096,0.0018395],[0.00029174,0.00041094,0.0019069],[0.00065536,0.0007402,0.0016949],[-0.0001719,-0.00031953,0.001573],[-0.00036016,-0.00033845,0.0015298],[8.3272e-05,0.00014665,0.0017218],[-0.00012307,0.00018811,0.0018313],[-0.00043286,-0.00016941,0.0015578],[-0.00035927,-0.00013762,0.0013492],[0.00032096,0.00077456,0.0013627],[-0.00079238,-0.00015741,0.0013873],[-0.00021696,0.00092419,0.0013244],[0.00095689,-0.0012868,-0.00046074],[0.0011367,-0.00097784,-0.00044358],[0.0011396,-0.00081774,-0.00030331],[-0.00056313,-0.0016217,0.00013662],[-0.0010661,-0.0013121,0.00020445],[0.00017399,-0.0016641,0.00030452],[0.0003539,-0.0017559,0.00029292],[2.166e-05,-0.0019796,-7.3787e-05],[-0.00038611,-0.0015608,-8.0219e-05],[2.9324e-05,-0.0016223,9.3327e-05],[5.4307e-05,-0.0018955,-4.1808e-05],[0.00033587,-0.0019381,3.3184e-05],[3.7674e-05,-0.0017401,7.6005e-05],[3.3394e-05,-0.0016515,5.187e-05],[0.0013916,-0.0011291,-0.00039348],[0.0012832,-0.00094781,-0.00045487],[4.436e-05,-0.0018575,0.00012003],[0.00064414,0.0006754,0.0019303],[0.00042789,-0.0004825,0.0014471],[0.00051006,4.1051e-05,0.0019655],[-0.00025822,-0.00056917,0.0014505],[0.0001304,-0.0015957,0.00059785],[-9.0237e-05,-0.0014467,-0.00035531],[-0.0007372,-0.00062622,0.0016371],[-0.0007951,-0.00069365,0.0014876],[-0.00021381,-0.001174,-0.00085378],[-0.00019697,-0.0011983,-0.00074768],[6.3247e-05,-0.0018342,0.00025512],[0.00027965,0.00016087,-0.0012798],[0.00096533,-0.00030042,-0.0012272],[0.001205,-0.00088012,-0.00082296],[0.0010716,0.00042966,-0.00089164],[0.0013612,-0.00081056,-0.00039018],[0.0012956,-0.00040539,-0.00088341],[0.0015197,-7.9873e-05,-0.00073565],[0.0015461,0.00025267,-0.00070951],[0.0014904,0.00061504,-0.00067246],[0.0013528,-2.15e-05,-0.00084735],[0.0015425,0.000405,-0.00072353],[0.0013675,4.182e-05,-0.0009883],[0.0013968,-0.00017775,-0.0010709],[0.0014328,0.00082051,-0.00061667],[0.0012919,-0.00054787,-0.0009722],[0.0014567,-0.00042465,-0.0010064],[0.0013803,0.00088091,-0.00074628],[0.0014487,0.00094384,0.00013731],[0.0014772,0.00089718,0.00020331],[0.0014439,0.0008796,0.00045051],[0.0014295,0.00098535,5.244e-06],[0.0012856,0.00074075,0.00094652],[0.0012686,0.00071566,0.00099591],[0.0010013,0.00058792,0.0013098],[0.0014657,0.00095886,-0.00015786],[0.00087531,0.00061673,0.0013937],[0.0014774,0.00084341,-0.00048827],[0.00049592,0.00067937,0.0015154],[0.00025663,0.0013272,0.0011524],[0.00073815,0.0010021,0.0012238],[-0.00014441,0.0011294,0.0013154],[0.00052151,0.00091378,0.0013081],[-1.08e-05,0.00040992,0.001841],[0.0002698,0.00043829,0.0018198],[0.00048333,0.00088737,0.0013743],[-0.00016419,-5.6508e-05,0.0015458],[-0.00035083,-0.00042416,0.0014388],[-0.00047616,-0.00041351,0.0016347],[6.5939e-05,0.00032694,0.0017427],[-0.00014063,0.00029404,0.0018154],[-0.0011794,-0.00020153,0.0021011],[5.8847e-05,0.0012041,0.00099756],[-0.0011741,-0.00023226,0.0013713],[-0.00027372,0.00099147,0.0012346],[0.0010607,-0.0013469,-0.00034682],[0.0012687,-0.0010666,-0.00044524],[0.0014242,-0.00097508,-0.00043145],[-0.001116,-0.0014021,-0.00029499],[-0.0010046,-0.001305,0.00023471],[-0.0010721,-0.0014091,-0.00056117],[-0.0010282,-0.0013684,-0.0007316],[-0.00088457,-0.0014545,-0.00083271],[-0.00020896,-0.0016525,6.8632e-05],[-3.2126e-07,-0.0016574,7.5748e-05],[-0.00049593,-0.0015867,-0.0006388],[0.00029305,-0.0019535,-2.3081e-05],[4.3977e-05,-0.0017531,7.9836e-05],[4.8331e-05,-0.0016782,6.1172e-05],[0.0015155,-0.0011379,-0.00028736],[0.0012717,-0.00096628,-0.00058169],[4.1362e-05,-0.0019046,0.00012103],[0.00048368,0.00072759,0.0018887],[0.00061773,-1.384e-05,0.0019393],[-0.0002356,-3.0525e-05,0.0015083],[-0.0007025,-0.00036842,0.0017035],[0.00011142,-0.0016052,0.00059448],[-0.0001168,-0.0014159,-0.00035979],[-0.00082127,-0.0005684,0.0016574],[-0.0011731,-0.00022558,0.0013745],[-3.6232e-05,-0.0012897,-0.0008364],[-0.00018652,-0.0012122,-0.00076932],[8.4645e-05,-0.0013565,0.00019983],[0.00039701,-0.00014434,-0.0012937],[0.001058,-0.00065594,-0.0012211],[0.0012405,-0.00086507,-0.00087134],[0.0013197,-0.00045696,-0.00059648],[0.0014347,-0.00046575,-0.00077662],[0.0014851,-3.6678e-05,-0.00088169],[0.0013386,0.00037418,-0.0010191],[0.0013035,0.00068916,-0.00089844],[0.0012508,3.4297e-05,-0.0010476],[0.0014496,0.00045538,-0.00086129],[0.001091,0.00012515,-0.001282],[0.0012694,-0.00013984,-0.0011698],[0.0015014,0.00078648,-0.00051802],[0.0013239,-0.00038159,-0.0010618],[0.0014877,-0.00062066,-0.00096332],[0.0013502,-0.00038845,-0.0011568],[0.0014678,0.00082526,-0.00059004],[0.0013861,0.00096906,2.368e-05],[0.0014298,0.00091568,0.00011458],[0.001422,0.0009453,0.00029937],[0.0013247,0.00094934,0.00059772],[0.0014001,0.00099455,-4.0274e-05],[0.0012705,0.00074031,0.00094881],[0.0010973,0.00077394,0.0011175],[0.00086528,0.00061447,0.0013645],[0.001433,0.00098656,-0.00023107],[0.0015006,0.00085922,-0.00050054],[0.00044374,0.00080581,0.0015349],[7.8093e-05,0.0010577,0.0013565],[-1.4217e-05,0.0013293,0.0011745],[0.00058339,0.00076841,0.0014585],[-0.00025832,0.0010769,0.0012569],[0.00024121,0.0011638,0.00083995],[-5.7152e-06,0.00036755,0.0017517],[0.00010932,0.00048178,0.0018008],[0.00026942,0.00094301,0.0013731],[-0.00025571,-8.6673e-06,0.0014656],[-0.0003793,-0.00039014,0.0013987],[-0.00035077,-0.00049006,0.0016103],[0.00011553,0.00030331,0.0017183],[0.0001322,0.00021021,0.0018609],[-0.00060827,-0.00072525,0.0021975],[0.00019056,0.0011648,0.00098586],[-0.00096256,-0.00045054,0.0014167],[-0.00019294,0.00096393,0.0012019],[0.0012049,-0.0012111,-0.00029825],[0.0011761,-0.0010388,-0.00061261],[0.00091616,-0.00064482,-0.00030573],[-0.00097783,-0.0013481,-0.00031311],[-0.0010824,-0.0013723,-0.00053406],[-0.00018337,-0.0016352,-0.00017521],[-0.0010446,-0.0013785,-0.00073464],[-0.0011315,-0.0015066,-0.00083616],[-0.00093381,-0.001473,-0.00055204],[8.0579e-06,-0.0016288,-3.3841e-05],[4.8472e-05,-0.00166,5.0746e-05],[-0.0006282,-0.0014124,0.00024281],[0.00021808,-0.0015227,7.1558e-05],[4.3584e-05,-0.0017135,6.5596e-05],[4.0924e-05,-0.0018344,8.639e-05],[0.0009847,-0.00063139,-0.00030809],[0.00063245,-0.00097548,-0.00026299],[0.0012518,-0.00049278,-0.00083042],[4.8047e-05,-0.0021524,0.000136],[0.00063751,0.00078392,0.0017834],[0.00056596,-4.8613e-05,0.002018],[0.00066811,0.0003938,0.001524],[0.00044878,0.00012597,0.0016063],[0.00016046,-0.0016993,0.00059052],[-8.5982e-05,-0.0014959,-0.00041294],[-0.001321,-0.00069933,0.0015138],[-0.0015649,-0.00027862,0.0011483],[-0.00017567,-0.001256,-0.00067433],[-0.00018271,-0.0011798,-0.00074933],[6.5778e-05,-0.0010522,-0.00048867],[4.8306e-05,-0.0023091,0.0002427],[0.0013508,-0.00069696,-0.00077931],[0.0010761,-0.00074573,-0.00073736],[0.0012651,-0.00044076,-0.00079481],[0.0014533,-6.4093e-05,-0.00089927],[0.001316,-0.00061621,-0.00089475],[0.0013649,0.00044356,-0.00093542],[0.0012767,0.00058518,-0.0010402],[0.001411,0.0004118,-0.00088618],[0.0011368,-0.00016116,-0.0012195],[0.0014353,0.00050222,-0.00086459],[0.001196,0.00024128,-0.0012362],[0.0012898,-0.00013723,-0.0011827],[0.0014707,0.00087511,-0.00051998],[0.0015059,-0.00061263,-0.00096228],[0.0014103,-0.00035828,-0.0011588],[0.0015008,0.00079413,-0.00059568],[0.0013566,0.00098774,2.2937e-05],[0.0013812,0.00094881,0.00011083],[0.0013946,0.00090417,0.00029307],[0.0013308,0.00085922,0.00059296],[0.0014107,0.00098707,-4.0386e-05],[0.0012129,0.00084646,0.00093199],[0.0011112,0.00078496,0.001132],[0.00078948,0.00081647,0.0013787],[0.0014216,0.0010176,-0.00023156],[0.0014193,0.00099406,-0.00049615],[0.0004691,0.00084538,0.0014608],[0.00022944,0.00085247,0.0014633],[0.00015242,0.0010968,0.0013039],[0.00051302,0.00087124,0.0014212],[-0.00021827,0.0010572,0.0013504],[0.00038839,0.0011504,0.00085269],[0.00032789,0.00025252,0.0017213],[1.2948e-05,0.00076628,0.0015992],[0.00017784,0.00094358,0.0010725],[-0.0002964,-6.651e-05,0.0017592],[-0.00016906,-4.5047e-05,0.00076552],[-0.00035691,-0.00041787,0.0016856],[-0.00051259,-0.00062113,0.0018496],[0.00014281,0.00021885,0.0018462],[0.0001422,8.3186e-05,0.0017283],[-0.001078,-0.00069095,0.0014456],[5.6347e-05,0.0012931,0.00078909],[-0.0015002,-0.00055715,0.0010277],[-0.00028091,0.0010981,0.0011768],[0.00095506,-0.001251,-0.00088236],[0.00095986,-0.00096809,-0.0010079],[0.00091615,-0.00064482,-0.00030574],[-0.00043275,-0.0016595,0.00014119],[-0.00064978,-0.0016128,-0.00021538],[-8.534e-06,-0.0017299,-3.1181e-05],[-0.00071688,-0.0015951,-0.00048083],[-0.00090266,-0.0017674,-0.00064335],[-0.0010317,-0.0016808,-0.00059959],[2.0212e-05,-0.0016826,-2.6419e-05],[1.5343e-05,-0.0016835,2.9952e-05],[-0.00076039,-0.0014432,0.00013834],[0.00010152,-0.001519,-7.9302e-05],[1.8657e-05,-0.0017187,5.1118e-05],[4.752e-05,-0.0018004,8.9047e-05],[0.0011336,-0.00075196,-0.00047142],[0.00073658,-0.001009,-0.00016807],[0.0013733,-0.00054127,-0.00083514],[4.8132e-05,-0.001952,0.00012625],[0.00069741,-0.0001151,0.0019849],[0.00070729,0.00066452,0.0016399],[0.00038919,0.00050234,0.0014228],[-0.0013894,0.0008248,0.0012815],[0.00013779,-0.0017144,0.00058771],[-8.4845e-05,-0.0014762,-0.0004075],[-0.0015954,-0.00012036,0.0012637],[-0.001258,0.00010007,0.0019411],[-0.0014936,-0.00034017,0.0011501],[-0.00014238,-0.001214,-0.0008113],[3.7092e-05,-0.0013285,-0.00059062],[9.5093e-05,-0.0016337,0.00016156],[0.0009557,-0.00052819,-0.00077859],[0.0011066,-0.00037211,-0.00070351],[0.0013158,1.3153e-05,-0.0011275],[0.00117,-0.00055296,-0.0011468],[0.0014052,0.00043868,-0.00092078],[0.0011765,0.00059196,-0.0010784],[0.0013208,0.00044117,-0.00098128],[0.0014133,-0.00025654,-0.00097332],[0.0014628,0.00050105,-0.00085479],[0.0013934,0.00016818,-0.0010084],[0.0011694,-0.00010177,-0.0012383],[0.0013591,0.00092519,-0.00067674],[0.0017255,-0.00074195,-0.0013391],[0.001073,-0.00022609,-0.0013342],[0.0014949,0.00081816,-0.00064563],[0.0013271,0.00096234,1.7394e-05],[0.0013287,0.00096985,1.8338e-05],[0.0013399,0.00094498,0.00013783],[0.0013244,0.00095217,0.00039122],[0.0012224,0.00094102,0.00074161],[0.0013554,0.0010192,-0.00015006],[0.0011414,0.00087433,0.00099068],[0.00080118,0.00085242,0.0012915],[0.0013745,0.0010589,-0.00033936],[0.0013821,0.0010251,-0.00057557],[0.00046834,0.00085864,0.0014702],[0.00024246,0.00088215,0.0014776],[2.1615e-07,0.00085243,0.0014956],[-0.00020123,0.0010801,0.0013253],[0.00049715,0.00084466,0.0014405],[-0.00035632,0.0010045,0.001301],[0.00018192,0.00099486,0.00042972],[0.00021479,0.00065286,0.0016155],[0.00025131,0.00067501,0.0015629],[-2.7676e-06,0.00096859,0.0010497],[-0.00024957,-7.6999e-05,0.0016858],[-0.00014546,-6.1511e-05,0.00078795],[-0.00039179,-0.00036878,0.0016169],[-0.00074278,-0.00040147,0.0017874],[-4.223e-05,0.00031901,0.0021307],[1.124e-06,0.00014132,0.0017712],[-0.0012081,-0.00053143,0.0013845],[0.00017421,0.0012328,0.00076153],[-0.0014769,-0.0005383,0.0010049],[0.00018978,0.0010681,0.0011247],[0.0015609,-0.0007355,-0.00070645],[0.00096602,-0.0009567,-0.001001],[0.00095078,-0.00094148,-0.0010049],[0.0009527,-0.00071036,-0.00033118],[-0.00037602,-0.0016711,-0.00035228],[-2.9083e-06,-0.0017416,-7.5598e-05],[-0.00060939,-0.0015785,-0.00051296],[2.1979e-05,-0.001668,-4.3802e-05],[-0.00083128,-0.0018706,-0.00070265],[-0.0010159,-0.0019869,-0.00071434],[-0.0010694,-0.0013028,0.0002131],[2.0788e-05,-0.0016565,-2.6426e-05],[2.0333e-05,-0.0016841,2.7519e-05],[-0.00069377,-0.0013143,0.00012982],[9.5676e-05,-0.0016508,-3.9813e-05],[5.038e-05,-0.0017617,3.6788e-05],[4.8022e-05,-0.0018502,9.191e-05],[0.00092479,-0.0008797,-0.00048126],[0.0013363,-0.00021863,-0.00095666],[4.1992e-05,-0.0019414,0.00012807],[0.00067011,-0.00013878,0.0020437],[0.00058088,0.00062316,0.0017662],[0.00056208,0.0006342,0.0016922],[-0.00026186,0.0017048,0.00084622],[0.00019588,-0.0018697,0.00060318],[-0.00010192,-0.0016241,-0.00044208],[-0.0016104,-0.00011882,0.0011161],[-0.0016607,0.00010094,0.0010624],[-0.0016309,-0.0003482,0.0009668],[3.464e-05,-0.0012771,-0.00075413],[0.00016534,-0.0013523,-0.00052527],[4.1514e-05,-0.0019747,0.00014118],[0.001716,-0.00080758,-0.0013933],[0.0011757,-0.0003664,-0.0010179],[0.0012723,0.00020685,-0.0011504],[0.0011577,-0.00017403,-0.0012627],[0.0013903,-0.00027985,-0.00094855],[0.0012707,0.00076149,-0.00091837],[0.0012737,0.00034857,-0.0010705],[0.001373,0.00034383,-0.00098599],[0.001439,-0.00020419,-0.00092904],[0.0013297,0.00078308,-0.00084909],[0.0011737,-0.00014041,-0.0012297],[0.0010839,-0.00023485,-0.0013457],[0.0014801,0.00077087,-0.00069671],[0.001245,-4.5907e-05,-0.0012398],[0.0013955,0.00099696,-0.00063866],[0.0012569,0.0010229,1.5613e-05],[0.0012579,0.0010478,1.8933e-05],[0.0012368,0.001031,0.00012783],[0.0012536,0.00097743,0.00037327],[0.0011613,0.00097757,0.0007147],[0.0012925,0.0010881,-0.00014684],[0.0010316,0.0010272,0.00094469],[0.00068952,0.0010476,0.0012557],[0.0013424,0.0010988,-0.00033654],[0.0012342,0.0011873,-0.00055092],[0.00057293,0.0010392,0.0012726],[0.00030634,0.0007877,0.0015099],[0.0001443,0.00063999,0.0015686],[0.00020838,0.00049987,0.0015601],[0.00057915,0.000724,0.0014799],[0.0002396,0.00032074,0.0016216],[-2.6988e-05,0.0010544,0.0004299],[-7.5329e-05,0.00068471,0.0014685],[5.7698e-05,0.0013586,0.0013905],[-0.00033038,3.0448e-05,0.0020539],[-0.00045307,-0.00018861,0.0019951],[-0.00070137,-0.00051618,0.0016248],[-0.0011649,-0.00050972,0.0013365],[-2.0939e-05,0.00011598,0.0017323],[-0.00010336,0.00063592,0.0014367],[-0.0014781,-0.00056386,0.00099062],[0.00028775,0.0012338,0.00092728],[-0.0015705,-0.00054993,0.00090654],[0.00055857,0.00066707,0.0014162],[0.0014411,-0.00070709,-0.00092992],[0.00073879,-0.00082825,-0.0012057],[0.0009854,-0.00097237,-0.0010269],[0.0010511,-0.00083694,-0.00098687],[4.3409e-05,-0.0017145,-3.7265e-05],[4.5415e-05,-0.0016964,-3.6752e-05],[-3.832e-06,-0.0017255,-5.3713e-05],[3.5359e-05,-0.0017017,-3.5023e-05],[-5.6375e-05,-0.0020649,-8.8773e-05],[-0.00020776,-0.0022719,1.9847e-06],[-0.00061973,-0.0014317,0.00070571],[2.8567e-05,-0.0016872,-2.185e-05],[1.487e-05,-0.0017051,2.4309e-05],[-0.00010492,-0.0013558,0.00077123],[0.00015214,-0.0017237,2.1515e-05],[5.9957e-05,-0.001767,4.2529e-05],[4.3267e-05,-0.0018247,8.8091e-05],[0.00079055,-0.00093913,-0.00098328],[0.0013498,-0.00019815,-0.0011064],[4.8756e-05,-0.0018933,0.00013008],[0.0006858,-0.00014451,0.0020654],[0.00059901,0.00063403,0.0018055],[-0.00022987,0.0012801,0.0017134],[-0.00015181,0.0016265,0.00089049],[0.0001164,-0.0019371,0.00059759],[0.0001695,-0.0017524,-0.00034789],[-0.0016633,-5.3824e-05,0.0011936],[-0.0016378,-9.9888e-05,0.0011195],[-0.00015184,0.0014839,0.00092935],[-0.0015849,-0.0004267,0.00099466],[0.0001312,-0.0013542,-0.00086374],[7.208e-05,-0.0015649,-0.00055557],[0.00016817,-0.0019682,9.6087e-05],[0.0011348,-0.00052647,-0.00099247],[0.0011998,-0.00049515,-0.00098989],[0.0011267,0.00026843,-0.0013028],[0.0011318,-0.00016058,-0.0013101],[0.0010432,-0.0001318,-0.0013573],[0.0010486,0.00082487,-0.0011011],[0.0011035,0.00038551,-0.0011899],[0.001128,0.00040635,-0.001225],[0.0012034,-0.00011338,-0.0012023],[0.0012019,0.00083786,-0.0010196],[0.001217,-0.00016523,-0.0011049],[0.0010915,-0.00024957,-0.001212],[0.0011899,0.00087268,-0.0010754],[0.0011508,-2.5726e-05,-0.0012191],[0.0011466,0.0010661,-0.00093466],[0.0012159,0.0010413,7.5748e-05],[0.0012202,0.0010018,-3.7792e-07],[0.001176,0.0010341,-7.1801e-05],[0.0011884,0.0010228,0.00015875],[0.001143,0.0010443,0.0005115],[0.0010118,0.0010352,0.00084171],[0.0012472,0.0010926,-0.00020447],[0.00071162,0.001097,0.0011283],[0.0012189,0.0011679,-0.00055111],[0.0010686,0.0012158,-0.00073222],[0.00084463,0.00066088,0.0014091],[0.00027761,0.00074004,0.0015631],[0.00022114,0.00079134,0.001531],[0.00027636,0.00063923,0.0015423],[0.00031279,0.00049216,0.0015158],[0.00056424,0.0006944,0.0015292],[0.00041694,0.0003385,0.001651],[-0.00010065,0.0017465,0.00063006],[-0.00025297,0.0011408,0.0013736],[-8.4079e-05,0.0013421,0.0013305],[-0.00013118,-5.2736e-07,0.0020546],[0.00022926,-0.00047637,0.0016856],[-0.00049869,-0.00069515,0.0016674],[-0.0010457,-0.0006179,0.0013614],[-0.00021918,0.00017751,0.0009921],[-2.8153e-05,0.00039926,0.00095243],[-0.0014323,-0.00055664,0.00096706],[-0.00029169,0.0011745,0.000866],[-0.0015525,-0.00055815,0.00090471],[-0.00014317,0.00060959,0.001427],[0.0013483,-0.00034724,-0.0010029],[0.0011689,-0.0010215,-0.0010729],[0.00072518,-0.0011517,-0.0010798],[0.001059,-0.00050948,-0.0012851],[0.00082505,-0.0010045,-0.0010187],[4.6134e-05,-0.0017162,-3.8471e-05],[4.8708e-05,-0.0017101,-7.5309e-05],[3.4865e-05,-0.0016968,-3.2043e-05],[1.2728e-06,-0.0016642,-9.1642e-05],[2.9712e-05,-0.0016763,-3.2224e-05],[-4.4488e-05,-0.0015208,-0.00011825],[-0.0001726,-0.0014656,1.6055e-06],[-0.00035995,-0.0014974,0.00064767],[-0.00062867,-0.0012015,0.00087757],[1.9405e-05,-0.0016827,-1.7626e-05],[6.2545e-05,-0.0017182,1.767e-06],[0.00011677,-0.0017084,2.7612e-05],[0.00050021,-0.0011765,-0.001016],[4.6597e-05,-0.0017868,4.9611e-05],[5.2107e-05,-0.0018367,8.4826e-05],[0.0013292,-0.00037561,-0.0010261],[0.00010599,-0.0019297,0.00011028],[0.00053371,-0.00022654,0.0021356],[0.00031064,0.00050172,0.0018816],[-0.00024773,0.0010931,0.0015052],[-0.0004991,0.0014453,0.0009656],[0.00012965,-0.0021151,0.0006507],[4.7473e-06,-0.0019246,-0.00028365],[-0.0016969,-5.646e-05,0.0010229],[-0.0016109,-9.818e-05,0.0010899],[-0.00045343,0.0014402,0.00065003],[-0.0014963,-0.00041194,0.0010152],[9.3607e-05,-0.0014028,-0.00092106],[0.0001959,-0.0017015,-0.00052292],[0.00011408,-0.0019788,5.8097e-05],[0.0011964,-0.00048707,-0.0010833],[0.0010709,0.00018835,-0.0013693],[0.00096247,0.00066675,-0.0012705],[0.0010613,-0.00022937,-0.0013429],[0.0011656,4.346e-05,-0.0012165],[0.001171,0.00049381,-0.0010972],[0.0010885,0.00039379,-0.0011791],[0.0010561,0.00053949,-0.001215],[0.0012075,-5.8257e-05,-0.001131],[0.0011755,0.00089039,-0.0010176],[0.0011073,-0.00030904,-0.0012108],[0.0011777,-0.00019048,-0.0011981],[0.0011252,0.00095696,-0.0010612],[0.0011965,-7.2977e-06,-0.0012366],[0.0010624,0.001097,-0.00089766],[0.0011182,0.0011226,7.1642e-05],[0.0011247,0.0011103,1.6706e-06],[0.001086,0.0011056,-6.6623e-05],[0.0010641,0.0010923,0.00014219],[0.0010546,0.0010538,0.00047856],[0.00096036,0.0010417,0.00080917],[0.0010999,0.0012175,-0.00018858],[0.00082774,0.00091614,0.0011786],[0.0010789,0.0012615,-0.0005228],[0.0010285,0.001187,-0.00070866],[0.00097003,0.00090288,0.0011218],[0.00023727,0.00079931,0.0015445],[0.00031877,0.0006568,0.0015863],[0.00041901,0.00044091,0.0016068],[0.0004888,0.00029698,0.001641],[0.00035737,0.0009621,0.0014086],[0.0003338,0.00043876,0.0015978],[4.2665e-05,0.0017587,0.0006559],[-0.00015663,0.0011021,0.001359],[-0.00025245,0.0015295,0.00092929],[-0.00028346,0.00051164,0.001991],[0.00010685,0.0002353,0.0016875],[-0.00047265,-0.00075434,0.0014933],[-0.0010595,-0.00084082,0.00124],[-0.0014512,-0.000685,0.000907],[-0.0004253,0.0003748,0.0020945],[-0.00024638,0.0010009,0.0013909],[-0.0015586,-0.00058238,0.00089455],[6.7697e-07,0.00091548,0.0012123],[-0.0016612,-0.00059314,0.00094023],[9.4079e-05,0.00033422,0.001593],[0.00087028,-9.5336e-05,-0.00156],[0.00057801,-0.0006749,-0.0014861],[0.00059967,-0.001109,-0.0012374],[0.0010319,-0.00049652,-0.0013569],[0.00028798,-0.00069955,-0.0014065],[4.5221e-05,-0.0017039,-3.864e-05],[6.044e-05,-0.0017022,-6.563e-05],[4.0779e-05,-0.0016992,-2.7698e-05],[6.5523e-05,-0.0016661,-4.0148e-05],[3.4431e-05,-0.0017036,-2.9731e-05],[5.6633e-05,-0.0015215,-3.5755e-05],[-5.5971e-05,-0.0014636,9.83e-05],[-0.00028413,-0.00145,0.00068287],[-0.0002876,-0.0011467,0.001177],[3.6155e-05,-0.0017079,-6.9732e-06],[6.7136e-05,-0.0017233,4.5135e-06],[9.091e-05,-0.0018326,-3.7383e-06],[0.00052513,-0.0012167,-0.0010387],[0.0001084,-0.0017548,8.7657e-05],[0.00019422,-0.0017742,0.00017299],[0.0010641,-0.00022544,-0.0014017],[0.00035487,-0.0018553,0.000274],[0.00034425,-0.00012227,0.00214],[-0.0002714,0.00084714,0.0018335],[-0.00054581,0.0011958,0.0012894],[-0.00057617,0.001396,0.00083785],[2.585e-05,-0.0022891,0.00066339],[0.00029688,-0.0018705,-0.00011888],[-0.0016816,-0.0001654,0.0010956],[-0.00051032,0.0011253,0.0010499],[-0.0015033,-0.00025725,0.0011319],[-0.00051217,0.0014054,0.00065929],[-0.0016143,-0.00036997,0.0010491],[0.00032666,-0.0013353,-0.0010231],[0.00065215,-0.0014781,-0.00069977],[0.00067551,-0.0017856,-0.0001438],[0.00085162,0.00025884,-0.0015068],[0.0010561,0.00061704,-0.0011583],[0.00084451,-0.00014497,-0.0015355],[0.00115,7.4168e-05,-0.0013463],[0.0010209,0.00051769,-0.0011721],[0.00088873,0.00038542,-0.001182],[0.0011886,0.00070312,-0.0016654],[0.0011613,-2.6198e-05,-0.0012975],[0.0010124,0.00098465,-0.0012911],[0.0010016,-0.00027649,-0.0011566],[0.00096213,-9.735e-05,-0.0013487],[0.00098924,0.00098528,-0.0011991],[0.0010896,4.4686e-05,-0.0013353],[0.00094077,0.001096,-0.00098637],[0.0010622,0.0011071,0.0001109],[0.0011142,0.001136,4.1667e-05],[0.0010397,0.001061,-6.0813e-05],[0.00094003,0.0010301,-0.00017636],[0.0010045,0.0011171,0.0002192],[0.00092847,0.0010765,0.00057736],[0.00088692,0.0010714,0.00087164],[0.0010132,0.0012779,-0.00036117],[0.0010244,0.0012697,-0.00058564],[0.00090685,0.001184,-0.00081049],[0.00073439,0.0011902,0.00098547],[0.00039203,0.0007935,0.0015006],[0.00032338,0.00094827,0.0013955],[0.00052555,0.00064696,0.0015236],[0.00055782,0.0004396,0.0015779],[0.00027102,0.00029836,0.0017024],[0.00050298,0.001232,0.0010986],[0.00013256,0.00042595,0.0016009],[2.107e-05,0.0013521,0.00047913],[-0.00037164,0.0015368,0.0010793],[-0.00023434,0.0015153,0.00092396],[-0.00068279,0.00047456,0.0016893],[-0.00065025,0.00027949,0.0014939],[-0.0009315,-0.00030782,0.0014493],[-0.0012136,-0.00068204,0.0011859],[-0.0014045,-0.00068267,0.00089102],[-0.00059732,0.00084737,0.0016259],[-0.00035126,0.0011248,0.0015437],[-0.0015396,-0.00053008,0.00085687],[-0.00035809,0.00085895,0.0011778],[-0.0016278,-0.00056098,0.00091126],[-0.00016302,0.00030499,0.0015562],[0.00062305,-0.00030375,-0.0016815],[0.00078863,-0.00094295,-0.00117],[0.00029395,-0.00080845,-0.0014391],[0.00065354,-0.00077613,-0.0014413],[0.0010326,-0.00026085,-0.0014278],[0.00061967,-0.00049371,-0.0013986],[4.8627e-05,-0.0017042,-6.089e-05],[4.2295e-05,-0.0017013,-3.738e-05],[5.5292e-05,-0.0016785,-3.6181e-05],[3.3589e-05,-0.0016975,-2.4597e-05],[6.3769e-05,-0.0016198,-3.9488e-05],[3.9132e-05,-0.0016922,-3.1713e-05],[3.7248e-05,-0.0015685,6.8915e-05],[-0.00020443,-0.0014559,0.00065675],[-0.00039771,-0.0011109,0.0011998],[-1.6631e-05,-0.0017532,1.9911e-05],[6.9726e-05,-0.0017041,-2.2166e-05],[0.00011949,-0.0017067,-2.0007e-05],[0.00036212,-0.0013205,-0.0010385],[0.00020832,-0.0017636,4.4813e-05],[0.00035855,-0.0017838,0.00010651],[0.0010302,-3.5307e-05,-0.0014112],[0.0006724,-0.0017101,0.00013244],[-0.00040312,-0.00053121,0.0027354],[-0.00048615,0.0006618,0.0017236],[-0.00072059,0.0010594,0.0012751],[-0.00091133,0.0013422,0.00092516],[-0.00014193,-0.0026803,0.00087222],[3.9959e-05,-0.0018981,-2.0934e-05],[-0.0013189,-0.0001361,0.0016012],[-0.00071638,0.0010615,0.00081954],[-0.0010064,-0.00022062,0.0013783],[-0.00039028,0.0013541,0.00073195],[-0.00033332,-0.0001167,0.0015663],[0.00063353,-0.0013594,-0.00084697],[0.00067167,-0.0015071,-0.00070932],[0.00072534,-0.0017408,-9.5753e-05],[0.00077465,0.00016367,-0.0015637],[0.00083105,0.00026473,-0.0014765],[0.0010674,0.0005702,-0.0011547],[0.0011105,0.00017894,-0.0013378],[0.0011214,8.6411e-05,-0.0012906],[0.00099868,0.00050992,-0.0011479],[0.00083357,0.00049021,-0.001171],[0.001073,0.00088074,-0.0016272],[0.00098764,-9.0778e-05,-0.0012015],[0.0010003,0.00096258,-0.0012714],[0.00098836,-0.00037791,-0.0012971],[0.0010936,-6.5956e-07,-0.0013261],[0.00092098,0.00096745,-0.0011376],[0.00087949,0.0010416,-0.00092829],[0.000972,0.0011403,0.00010485],[0.0010189,0.0012064,4.3303e-05],[0.00096082,0.001162,-5.667e-05],[0.00088494,0.0010827,-0.00017022],[0.00087318,0.001094,0.00019204],[0.00083894,0.0010592,0.00053019],[0.00085171,0.0010704,0.00084438],[0.00097949,0.0012699,-0.00035258],[0.00097799,0.0012673,-0.00056953],[0.00086378,0.0011909,-0.00079211],[0.00069529,0.0011251,0.00095588],[0.0004822,0.00069862,0.0015759],[0.00038604,0.00087703,0.0014444],[0.00035518,0.00089396,0.0014671],[0.00036327,0.00075515,0.001564],[0.00013177,0.00050111,0.0016884],[0.00053032,0.0011982,0.0011155],[5.7356e-05,0.00053284,0.0016058],[2.8529e-05,0.0012936,0.00045927],[-0.00047509,0.0015189,0.0010422],[-0.00031149,0.0018211,0.00095169],[-0.00074325,0.00034617,0.0016568],[-0.00083371,0.00079155,0.0015132],[-0.0011862,-0.00034889,0.0013816],[-0.0014496,-0.00071097,0.0009157],[-0.001565,-0.00071348,0.00076195],[-0.00080726,0.00081956,0.0015015],[-0.0005206,0.0014759,0.0010686],[-0.0015841,-0.0005462,0.00088662],[-6.2469e-05,0.00053513,0.0013887],[-0.00014222,-0.00035053,0.001631],[-0.00012739,0.00027423,0.0016723],[0.00079301,-0.00039681,-0.0013944],[0.00062701,-0.00086396,-0.0013058],[6.4424e-05,-0.00063901,-0.0014739],[0.00059149,-0.00074014,-0.0014638],[0.00088935,-0.00019035,-0.0015378],[0.00023495,-0.0002502,-0.001546],[5.8041e-05,-0.0017051,-5.3555e-05],[4.9934e-05,-0.0016971,-3.1405e-05],[5.2071e-05,-0.001694,-3.9542e-05],[4.7216e-05,-0.001704,-1.479e-05],[5.5956e-05,-0.0016409,-4.7164e-05],[5.4013e-05,-0.0017015,-2.1788e-05],[7.1449e-05,-0.0015645,9.7019e-05],[-0.00016265,-0.0014336,0.00068037],[-0.00023207,-0.0010622,0.0013114],[3.2561e-05,-0.0018531,6.2935e-05],[7.5692e-05,-0.0017163,-1.8776e-05],[0.00021118,-0.0016812,3.8341e-05],[0.00029837,-0.0013289,-0.0011258],[0.00041694,-0.0016371,0.00017815],[0.00075906,-0.0014804,0.00037813],[0.00088001,2.8506e-05,-0.0014765],[0.00098605,-0.0014912,0.00037299],[-0.00055098,-0.00019418,0.0020492],[-0.00074291,0.00076197,0.00147],[-0.0008991,0.0010966,0.0010586],[-0.00080715,0.001285,0.00097346],[-0.00018456,-0.0023115,0.00070502],[-7.7908e-06,-0.001672,-5.3442e-05],[-0.00070119,0.00041658,0.0013329],[-0.0010734,-0.00026786,0.001433],[-0.00038094,0.0012151,0.00074138],[-0.00059992,-0.00072421,0.0014016],[-0.00074245,0.0013109,0.00082957],[-0.00013363,-0.0003291,0.0015526],[0.00016663,-0.0011025,-0.00048604],[0.00057758,-0.0017383,-0.00071553],[0.00096513,-0.0015135,-0.00018954],[0.00093863,0.00010552,-0.0014609],[0.00097646,0.00021493,-0.0013685],[0.00082897,0.00061371,-0.0012615],[0.001109,0.00020297,-0.0014207],[0.001182,0.00017552,-0.0017724],[0.00082414,0.00050095,-0.0011437],[0.00083735,0.00049057,-0.0011541],[0.0007688,0.00084101,-0.0015901],[0.00060839,-4.7458e-05,-0.00094056],[0.00062262,0.00094471,-0.0014129],[0.0010101,-0.00039198,-0.0012059],[0.0012864,-9.7605e-05,-0.0011505],[0.00083349,0.00093926,-0.0011402],[0.00076685,0.0010142,-0.00097133],[0.0010043,0.0012085,6.2643e-05],[0.00086223,0.0010895,0.00017365],[0.0012596,0.0015583,-4.906e-05],[0.0012534,0.0016879,-0.000351],[0.00074979,0.0010206,0.0002365],[0.00082945,0.0010914,0.00056456],[0.00097563,0.0013559,-0.00044735],[0.0008663,0.0012789,-0.00069028],[0.00076662,0.0011691,-0.00085013],[0.00053564,0.0013015,0.00085324],[0.00058427,0.00088917,0.0014166],[0.00025513,0.00069214,0.0016151],[0.00057685,0.0012046,0.0011093],[0.00043366,0.00089938,0.0014587],[0.00024852,0.00075958,0.0016014],[3.2992e-05,0.00048263,0.0016551],[0.00060223,0.0013099,0.00083677],[-2.7779e-05,0.00051289,0.0015701],[-9.9158e-05,0.00053082,0.0013858],[-0.00061949,0.0017786,0.00072999],[-0.00074312,0.0016992,0.0008116],[-0.0011694,0.00032088,0.0013402],[-0.0010499,0.00072942,0.0012877],[-0.0013388,-0.00010762,0.0012567],[-0.0015617,-0.00050116,0.00080192],[-0.0014927,-0.00067513,0.00072373],[-0.00093929,0.001166,0.0010812],[-0.00070409,0.0014038,0.00095856],[-0.0014897,-0.0006005,0.00087804],[5.5438e-05,-0.00054627,0.0015575],[-0.0001455,0.00026016,0.0016025],[5.4432e-05,-0.00081381,-0.0014559],[0.00076991,-0.00074293,-0.001253],[0.00064651,-0.00065758,-0.0013959],[0.00022018,-0.00056116,-0.001487],[0.00061756,-0.00040224,-0.0016193],[0.00086232,-6.8475e-06,-0.0015381],[0.00041988,-0.0001466,-0.0015622],[5.3275e-05,-0.0017033,-5.1631e-05],[5.5925e-05,-0.0017012,-4.1113e-05],[4.9715e-05,-0.0017053,-3.1366e-05],[5.2288e-05,-0.0016611,-4.6035e-05],[5.3198e-05,-0.0016948,-1.7377e-05],[2.4979e-05,-0.0015858,0.00011757],[7.6246e-05,-0.0016875,-3.1607e-05],[-2.5381e-05,-0.0014659,0.0006343],[-0.00027357,-0.0010639,0.0013389],[-2.5256e-05,-0.0018169,7.776e-05],[0.00029277,-0.0013332,-0.0011264],[0.00022868,-0.0016875,-8.7128e-05],[0.00043957,-0.0016351,-6.7693e-05],[0.00078726,-0.0014742,-6.3627e-07],[0.00097349,-0.0014614,0.00031251],[0.00091149,-0.00044738,-0.0014016],[0.00083944,-0.0015445,0.00043493],[-0.00087867,-0.00023462,0.0018141],[-0.00069749,0.00063441,0.0012838],[-0.0014596,0.0010439,0.0012695],[-0.0011966,0.0011276,0.0010073],[-0.00051987,-0.0026961,0.00089968],[-8.9924e-05,-0.0015024,-2.848e-05],[-0.00036394,0.000408,0.0012818],[0.00054169,-0.0002573,0.0017769],[-0.00011426,0.0010409,0.0008209],[0.00058579,-0.00054627,0.0017405],[-0.00049384,0.0013106,0.0011302],[0.0004514,-0.00018214,0.0016638],[0.00035628,-0.0011084,-0.00034406],[0.00098958,-0.001516,-0.0002739],[0.00098837,-0.0015584,-0.00019841],[0.0010729,0.00025726,-0.0013956],[0.00093185,0.00012974,-0.0014615],[0.00083518,0.0006007,-0.001342],[0.00085725,0.00050413,-0.0012498],[0.0011716,9.2826e-05,-0.0017267],[0.00058552,4.3238e-05,-0.00093842],[0.00096804,0.00078517,-0.0016934],[0.00064869,0.00097095,-0.0015162],[0.00084717,0.0009672,-0.0016376],[0.00097411,1.1887e-05,-0.0012751],[0.00077625,0.00069152,-0.0014253],[0.0012714,-0.00021608,-0.0011084],[0.00072093,0.00087823,-0.0010164],[0.00071744,0.00097748,-0.00092167],[0.0008996,0.0012451,6.3256e-05],[0.0008349,0.0010118,0.0001651],[0.0011685,0.0016041,-4.2567e-05],[0.0012042,0.0016742,-0.00034096],[0.00066437,0.00086132,0.00020948],[0.0006754,0.0011697,0.00047353],[0.00099319,0.0013332,-0.00044883],[0.00081887,0.0013123,-0.00068076],[0.00071135,0.001242,-0.00085334],[0.00054666,0.0012899,0.00061003],[0.00036071,0.0011965,0.001293],[-3.7193e-05,0.0010484,0.0014595],[0.00055976,0.0012372,0.001102],[0.00028808,0.0010857,0.0013859],[7.014e-05,0.00097834,0.0015124],[-0.00019922,0.0007664,0.0015658],[0.00070032,0.001212,0.00090795],[-0.00016652,0.00068464,0.0015535],[-4.2806e-05,0.00061858,0.0018301],[-0.00027965,0.0017286,0.00078251],[-0.00013306,0.00075325,0.0016252],[-0.0012433,0.00043341,0.0013001],[-0.0011016,0.00092674,0.0010316],[-0.0015316,-0.00012947,0.00098219],[-0.0014859,-0.00048512,0.0008342],[-0.0014932,-0.00069619,0.00081235],[-0.00092026,0.0010477,0.00094589],[-0.00065933,0.0013145,0.00099939],[0.00013542,-0.00031117,0.0015881],[0.00046155,-0.00042691,0.0015131],[-0.00032374,0.00045358,0.0016099],[4.1862e-05,-0.00080193,-0.0014534],[0.00046083,-0.00058635,-0.0014316],[0.00056171,-0.00062405,-0.0014939],[0.00019121,-0.00053454,-0.0014689],[0.00046675,-0.00032959,-0.0017282],[0.00068974,7.7096e-05,-0.001629],[0.00027968,-5.4488e-05,-0.0015795],[5.8973e-05,-0.0017024,-4.7154e-05],[4.079e-05,-0.0017133,-5.3886e-05],[5.0729e-05,-0.0017024,-3.0485e-05],[3.3272e-05,-0.0016931,-6.3564e-05],[1.7012e-05,-0.0016989,-4.3278e-05],[3.2817e-05,-0.0016099,0.00012578],[6.781e-05,-0.0016495,-3.5444e-05],[-7.289e-05,-0.0014781,0.00059792],[-0.00022496,-0.0010679,0.0014013],[2.0136e-06,-0.001934,0.00010402],[0.0001299,-0.0012786,-0.0012457],[0.00066071,-0.0016158,0.00019075],[0.0013632,-0.0013912,0.00047968],[0.00112,-0.0013126,0.00020602],[0.00061017,-0.0013855,8.9793e-05],[0.00068092,-0.00033114,-0.0015706],[0.00025153,-0.0015918,-2.833e-05],[-0.0013008,-0.00049864,0.0023241],[-0.0011269,0.00082872,0.00096083],[-0.0012934,0.00094396,0.001257],[-0.0011412,0.001125,0.0011051],[-0.00044854,-0.0018954,0.00048532],[-7.8152e-06,-0.0013977,5.434e-05],[0.00049611,-0.00040519,0.0018003],[0.00067315,-0.00040152,0.0018527],[-0.00047411,0.001227,0.0010855],[0.00041063,-0.00034934,0.0016929],[-0.0011228,0.00097051,0.0011388],[0.00052783,-0.00027772,0.0015651],[0.00014472,-0.0016758,-7.1778e-05],[0.00014042,-0.0017975,0.00014481],[0.00091599,0.00032285,-0.0015038],[0.00067256,0.00020336,-0.0015422],[0.00064916,0.00062472,-0.0013748],[0.00070117,0.00054472,-0.0013281],[0.00095319,0.00016364,-0.001689],[0.00086423,6.2077e-05,-0.0013942],[0.00068183,0.00053755,-0.0012957],[0.00054779,0.00090979,-0.0014336],[0.0005809,0.00088469,-0.0014892],[0.00087509,8.5013e-05,-0.0014187],[0.00070772,0.00066978,-0.0013981],[0.00083204,2.9051e-05,-0.0014879],[0.00065798,0.00086258,-0.001029],[0.00063028,0.0009714,-0.00098221],[0.00086136,0.0011574,2.4879e-05],[0.0009628,0.001377,1.656e-05],[0.00054659,0.00071075,0.00016472],[0.0010694,0.0016237,-0.00017371],[0.0010434,0.0015011,-0.00033897],[0.00055807,0.00096693,0.00035613],[0.00083108,0.001368,-0.00064922],[0.000704,0.0012958,-0.00077558],[0.00063388,0.0012045,-0.00088024],[0.00052331,0.0012612,0.00058707],[3.0318e-05,0.0011903,0.0013649],[0.00041073,0.0012711,0.0011737],[0.0002767,0.0010622,0.0014031],[0.00059329,0.0012834,0.0010005],[0.00012272,0.0010786,0.0014157],[-0.00012925,0.00094055,0.0014991],[-0.00013709,0.00075029,0.001518],[0.00078886,0.0013232,0.00054028],[-0.00022552,0.00065643,0.0015082],[-0.00030127,0.00058994,0.0018073],[4.7895e-05,0.0010657,0.0012634],[-0.00027815,0.00071085,0.0015328],[-0.001431,0.00036313,0.00093408],[-0.0010771,0.00080419,0.00084438],[-0.0015589,3.2266e-05,0.00085331],[-0.0015736,-0.00023805,0.00069012],[-0.0014635,-0.0004419,0.00067746],[-0.00090893,0.0011057,0.00073227],[-0.0009593,0.0011385,0.00076893],[-4.0156e-05,-0.00013399,0.0016167],[0.00022435,-0.00018178,0.0015896],[-0.00022808,0.00047681,0.0016377],[0.00019152,-0.00073959,-0.0014783],[4.1103e-05,-0.00082778,-0.0014637],[0.00046073,-0.00063987,-0.001423],[0.0005835,-0.00020696,-0.0016135],[0.00024965,-0.00049967,-0.0014619],[0.00045265,-0.00011489,-0.0017279],[0.00070486,-0.00031056,-0.0015573],[0.00023265,-8.1352e-05,-0.0015798],[5.3098e-05,-0.0017123,-4.4953e-05],[6.163e-05,-0.0017142,-6.1944e-05],[4.3159e-05,-0.0017136,-6.7975e-05],[1.3518e-05,-0.0017145,-1.5116e-05],[-7.5872e-06,-0.0016118,0.00014241],[7.2508e-05,-0.0016845,-6.7534e-05],[-5.9395e-05,-0.0014623,0.00058565],[0.00069559,-0.0015599,-0.00032144],[-0.0002057,-0.001087,0.0014108],[-8.7113e-06,-0.0019319,0.00010725],[0.00026489,-0.0012116,-0.0012587],[0.001409,-0.0013793,-0.00019462],[0.0011405,-0.0013499,0.00052333],[0.00058339,-0.0013878,0.00041541],[0.00018814,-0.0015178,0.00022467],[0.00066145,-0.00011533,-0.001591],[-5.0466e-05,-0.0016839,6.7134e-05],[-0.0015649,2.2646e-05,0.0010035],[-0.001507,0.00059638,0.0010077],[-0.0012984,0.0010207,0.0010755],[-0.00073443,-0.0015281,0.00034847],[-0.00038092,-0.0017624,9.2552e-05],[0.00065727,-0.00037164,0.0017664],[0.00063141,-0.00034379,0.0016184],[0.00080156,0.00017056,0.0015589],[0.00077335,-0.00024267,0.0015825],[0.00039883,0.0004153,0.0019789],[0.0003689,-0.00031381,0.001569],[-9.3276e-06,-0.0014718,0.0004374],[0.00022065,-0.0019892,-4.3375e-05],[-2.2857e-05,-0.0017583,-1.6531e-05],[0.00090641,0.00025861,-0.0016481],[0.00085974,0.00041816,-0.001458],[0.00056066,0.0005058,-0.001504],[0.00068062,0.00051501,-0.0013664],[0.00082828,3.8915e-05,-0.0017559],[0.00081283,0.00023639,-0.001802],[0.00091695,0.00052547,-0.0013116],[0.00053956,0.00089325,-0.0014323],[0.0007176,0.00065172,-0.0014863],[0.00064199,0.00099289,-0.0011561],[0.00082147,3.6228e-05,-0.0014711],[0.00082251,0.0010528,-0.0018221],[0.0008325,0.00136,-0.0013541],[0.00070375,0.0012677,3.408e-05],[0.0008394,0.0014742,2.8127e-05],[0.00033799,0.00070766,0.00014088],[0.00095479,0.0016845,-0.00015957],[0.00090969,0.0015892,-0.00031789],[0.00038286,0.0008395,0.00024429],[0.00077042,0.0013965,-0.00063533],[0.00068142,0.0013266,-0.00077729],[0.00066492,0.0012491,-0.00091635],[0.00052694,0.0011961,0.00035292],[1.2294e-05,0.0011969,0.0013387],[0.00024769,0.001421,0.0010372],[0.00014362,0.0012138,0.0013208],[0.00038307,0.0014742,0.0008296],[0.00015205,0.0010543,0.0014438],[-0.00015149,0.00096079,0.0014773],[-0.0001928,0.00081442,0.001493],[0.00069116,0.0014172,0.0004603],[-0.00028743,0.00071732,0.0014396],[-0.00023172,0.0004859,0.0016663],[-0.00050575,0.00099277,0.0010752],[-0.0001463,0.00051872,0.0016521],[-0.0014271,0.00041241,0.00087401],[-0.00091863,0.00060454,0.00085222],[-0.00148,2.9152e-05,0.00079079],[-0.0013744,-0.00021613,0.00078042],[-0.00014792,-0.00043172,0.0015788],[-0.00063188,0.0010581,0.00077942],[-0.00066938,0.0006894,0.0011427],[0.00027496,-5.5515e-05,0.0016308],[0.0001595,-0.00020605,0.0016357],[-0.00035522,0.00062613,0.0014957],[0.0004693,-0.00090614,-0.0013536],[0.00022615,-0.0009032,-0.0013271],[0.00025887,-0.00052985,-0.0014353],[0.0003899,-0.00010624,-0.0018958],[0.00010614,-0.00039319,-0.0014841],[0.00050414,-0.00013988,-0.0017387],[0.00059952,-0.00025505,-0.0016495],[0.00021663,-7.2564e-05,-0.0015356],[5.3085e-05,-0.0017129,-4.4996e-05],[6.5098e-05,-0.0017185,-5.9501e-05],[5.1531e-05,-0.0017342,-6.2461e-05],[5.1705e-06,-0.0017231,-2.1393e-05],[-1.8948e-05,-0.0016764,0.0001385],[-2.6481e-05,-0.0017117,-0.00013547],[-4.4776e-05,-0.0014759,0.00060529],[0.00080119,-0.0015267,-0.00023986],[-0.00025997,-0.0011345,0.0014197],[-1.0277e-05,-0.0020087,0.00011067],[5.5321e-05,-0.0011086,-0.0013664],[0.00080763,-0.0016248,-0.00059465],[-0.00036299,-0.0019641,-0.00021296],[-0.00060167,-0.0023215,-0.00022274],[-0.00030529,-0.001446,-0.00022765],[0.00070186,-0.00013381,-0.0015904],[-7.239e-05,-0.0015951,3.5741e-05],[-0.0016066,0.00015783,0.0013683],[-0.0011003,0.00040022,0.0011526],[-0.0014656,0.00017968,0.0011269],[0.00020232,0.00034631,0.001473],[-0.0005436,-0.0012883,0.00047504],[0.00061883,-0.000361,0.0016028],[0.00072147,-0.00044459,0.0015925],[0.00021934,2.4949e-05,0.0017439],[0.00050262,4.4761e-05,0.0016348],[3.1423e-05,0.00028512,0.0021908],[0.00018831,-0.00011988,0.0016423],[0.00014402,-0.0014841,0.00041166],[-3.4698e-05,-0.0014282,3.578e-05],[7.182e-06,-0.0020137,-2.9936e-05],[0.00085905,0.0002584,-0.0016031],[0.000805,0.00042959,-0.0014593],[0.00059334,0.00047257,-0.0014314],[0.00053101,0.00054254,-0.0014007],[0.00083786,1.8317e-05,-0.00169],[0.00073802,0.00023257,-0.0015384],[0.00067165,0.00022828,-0.0014553],[0.0004396,0.00091986,-0.0014879],[0.0005558,0.00065453,-0.0015234],[0.0003837,0.0012015,-0.0014439],[0.00077812,7.8742e-05,-0.0015719],[0.00088657,0.00097949,-0.0016311],[0.00084287,0.0012925,-0.0012454],[0.00055382,0.0011818,0.00021953],[0.00078138,0.0013947,4.8291e-05],[0.00083021,0.0015605,-5.3344e-05],[0.00086836,0.0016133,-0.00020524],[0.00075392,0.0015404,-0.00044355],[0.00067942,0.0013834,-0.0007172],[0.00064061,0.0013028,-0.00079318],[0.00059714,0.001217,-0.00094497],[0.0005572,0.0013051,0.00036749],[0.00015152,0.0014486,0.0010798],[0.00014178,0.0012082,0.0013206],[0.00032163,0.001515,0.00086948],[0.00023507,0.0012214,0.0013079],[0.00047519,0.0015638,0.00057939],[-0.00013219,0.0010159,0.001461],[-0.00014504,0.00094098,0.001446],[-0.00027644,0.00078009,0.0014559],[0.00069239,0.0013887,0.00037355],[-0.00030794,0.00071742,0.001446],[-0.00016334,0.00050552,0.0016959],[-6.9455e-05,0.00029759,0.0015937],[4.1379e-05,0.00053875,0.0016356],[-0.0011721,0.00041109,0.00096427],[-0.00079084,0.0006208,0.0009106],[-0.0014147,-7.6952e-05,0.00084377],[-0.0013249,-0.00027794,0.00080297],[-0.00036163,-0.00018605,0.001627],[-0.00032565,0.00044022,0.00098611],[-0.00036146,0.00074859,0.0012635],[0.00035974,-0.00014351,0.0016107],[0.0001787,-0.00022609,0.0016376],[-0.00019971,0.00066084,0.0015125],[8.6625e-05,-0.0011151,-0.0013622],[0.00048038,-0.00083935,-0.0014067],[0.00025467,-0.00055786,-0.00149],[0.00027274,-0.00011508,-0.0015044],[0.00038478,-0.00025595,-0.0018572],[0.00020858,-0.00033425,-0.0014841],[0.00050013,-0.00033373,-0.0016892],[0.00059044,-0.00022002,-0.0016329],[0.00010949,-0.00013383,-0.0015131],[5.7073e-06,-0.0017432,-2.5935e-05],[5.5244e-05,-0.0017379,-5.601e-05],[6.7968e-05,-0.001761,-6.9315e-05],[1.5855e-05,-0.001727,0.00012915],[-3.5209e-05,-0.0017873,-4.373e-06],[-0.00014388,-0.0014166,0.00062874],[0.00078943,-0.0014178,-0.00053206],[-0.00020865,-0.0011381,0.0013902],[0.0007522,-0.0015919,-0.00021637],[-7.9125e-06,-0.0020187,0.00011044],[0.0002709,-0.00098531,-0.0013737],[-0.00036662,-0.0018285,2.4271e-05],[-0.0005904,-0.0022133,-0.00013777],[-0.00033261,-0.0015087,-0.00018316],[5.4685e-05,-0.0016135,-0.00032268],[0.00067939,1.0157e-05,-0.0015797],[0.00029949,-0.0017751,-5.34e-05],[-0.00021258,0.00085397,0.0011936],[-0.00049139,0.00063916,0.0011878],[0.00044024,0.00045001,0.0014566],[0.0001652,0.00045122,0.0013481],[-7.655e-05,-0.0013141,0.00045563],[0.00032116,-0.00042586,0.0016536],[-4.0685e-06,-0.00057666,0.0016057],[0.00038739,-0.00019882,0.0016018],[-0.0001671,-0.00010028,0.0016904],[3.6161e-05,0.00017881,0.0014944],[-6.3514e-05,-0.0001848,0.0016454],[-8.9442e-05,-0.001497,0.00016821],[6.2866e-05,-0.0012228,0.00061332],[-3.698e-05,-0.0014286,3.4664e-05],[-4.0339e-05,-0.0019953,8.1198e-05],[0.000173,-0.0017049,0.00025095],[0.00078405,0.00039315,-0.0015213],[0.0007868,0.00018275,-0.0016414],[0.00071511,0.00061303,-0.0013915],[0.00054951,0.00056451,-0.0014032],[0.00084921,8.1574e-05,-0.0015263],[0.00085099,0.00019278,-0.0015552],[0.00037504,0.00028109,-0.0015124],[0.00046638,0.00098285,-0.0014405],[0.00057873,0.00069251,-0.0015431],[0.00064272,0.00058922,-0.0016105],[0.00017381,0.0006473,-0.0013714],[0.00076886,0.0010035,-0.001549],[0.00064141,0.0012166,-0.0010941],[0.00058614,0.0012807,0.00023921],[0.00070407,0.0014383,5.7513e-05],[0.00077682,0.0015704,-4.7355e-05],[0.0008274,0.0016168,-0.00019838],[0.00080009,0.0014997,-0.00044917],[0.00061265,0.001387,-0.00069271],[0.0005253,0.0013629,-0.00076921],[0.00044025,0.0013285,-0.00093082],[3.8542e-05,0.0015255,0.00097164],[-4.6122e-05,0.0013784,0.0011652],[0.00029261,0.0015337,0.0008423],[4.739e-05,0.0013893,0.0011527],[0.00046129,0.001558,0.00056568],[-0.0002633,0.0011368,0.0013488],[-0.00023085,0.0010173,0.0013653],[-0.0003806,0.00086715,0.0013162],[0.00066492,0.0014935,0.00032973],[-0.0003725,0.00076843,0.0012455],[-0.00024678,0.00061343,0.0015998],[-0.00022894,0.00026933,0.0015413],[-0.00012383,0.00074481,0.001495],[-0.0011322,0.00032035,0.0010237],[-0.00079586,-3.7798e-05,0.0017225],[-0.0012579,-6.1647e-05,0.00090207],[-0.00035205,-0.00015431,0.0016443],[0.00039896,-5.3658e-05,0.0016483],[-1.3275e-05,0.0004841,0.0010692],[-0.00014368,0.00047852,0.0014699],[0.00019151,-0.00019079,0.0016445],[-0.00012351,-0.0003024,0.0016261],[-0.00038602,0.00096765,0.0015296],[0.00032418,-0.001283,-0.0012866],[0.00047674,-0.00084424,-0.0014296],[0.0002063,-0.00048914,-0.0013442],[0.00022083,-9.9755e-05,-0.0014004],[0.00055316,-0.00036164,-0.0018282],[0.00022586,-0.00034973,-0.0015046],[0.0005954,-0.00038827,-0.001658],[0.00049659,-0.0001691,-0.0016833],[0.00025274,-0.00023845,-0.0014839],[4.962e-06,-0.0017395,-2.6437e-05],[5.3381e-05,-0.0017476,-5.8001e-05],[6.3467e-05,-0.0017946,-7.5127e-05],[-2.6589e-05,-0.0018443,0.00010298],[6.1806e-06,-0.0017695,2.4508e-05],[7.8174e-06,-0.0014189,0.00078384],[0.00060166,-0.0014069,-0.00062967],[-0.00026537,-0.0011677,0.0013661],[0.00067401,-0.0016147,-0.00027486],[-2.3369e-05,-0.0020408,0.00010101],[9.2572e-05,-0.0008849,-0.0014466],[6.083e-05,-0.0017084,0.00026396],[-0.000431,-0.0020223,-3.5213e-05],[0.00016064,-0.001692,-0.00015697],[0.00016803,-0.0016363,-0.00018332],[0.00070097,3.8494e-06,-0.0015954],[0.00021128,-0.0017137,-0.00016559],[-0.00021453,0.00050268,0.0013419],[0.00046858,0.00047819,0.0015509],[3.3954e-05,0.00077274,0.0013472],[-9.5223e-06,0.000377,0.0013395],[0.00013667,-0.00020564,0.001655],[-0.00024018,-0.00030522,0.0016592],[-0.00011514,-0.00030575,0.001646],[-7.0104e-05,-0.00020457,0.0016442],[-0.00049824,1.0405e-05,0.001542],[-7.7734e-05,-0.00016355,0.001607],[0.00011937,-0.0014783,0.00015777],[9.8705e-05,-0.0014073,0.00070083],[-2.0592e-05,-0.0015004,3.3667e-06],[-3.2846e-05,-0.0013046,5.354e-05],[-6.4459e-05,-0.0019182,0.00034042],[0.001038,0.0002385,-0.0013195],[0.00081384,0.00013366,-0.0015086],[0.00068065,0.00059931,-0.0013543],[0.00048001,0.00055072,-0.0013535],[0.00077044,0.00011716,-0.0015389],[0.00067843,7.3417e-05,-0.0015606],[0.00051,0.00046339,-0.0013535],[5.945e-05,0.0010589,-0.0015513],[0.00049327,0.00068266,-0.0015217],[0.00061049,0.00055842,-0.0015261],[0.00023627,0.00059532,-0.0012255],[0.0005927,0.0009201,-0.0014927],[0.00046919,0.0010951,-0.0010866],[0.00061961,0.0014162,0.0002307],[0.00073499,0.0015576,2.8799e-05],[0.00077224,0.001614,-8.01e-05],[0.0007693,0.001616,-0.00026251],[0.00062467,0.0015,-0.00064349],[0.00052406,0.0013611,-0.00076652],[0.00042096,0.0013318,-0.00085533],[0.00032951,0.0012712,-0.00099302],[6.8797e-05,0.0015875,0.00091896],[4.7191e-05,0.0015387,0.00097835],[4.6756e-05,0.0013854,0.0011513],[0.0003948,0.0016398,0.00061325],[-0.00016146,0.0013653,0.0011764],[0.0005363,0.0016172,0.0003843],[-0.00019554,0.0011307,0.0013255],[-0.0003728,0.00096263,0.0013325],[-0.00037943,0.00085667,0.0013012],[0.00066896,0.0014414,0.00023087],[-0.00032083,0.0007843,0.0012553],[-0.00019846,0.00062017,0.0015964],[-0.0003907,0.00046963,0.0015202],[-0.00031087,0.00070977,0.0014845],[-0.00048958,0.00039092,0.0012522],[0.00010357,0.00012696,0.0020944],[-0.0009086,-0.00051902,0.0010659],[-0.000249,-0.00029109,0.0016771],[0.00061296,-0.00028855,0.0016708],[-5.3216e-05,0.00047209,0.0014293],[-5.0564e-05,6.3335e-05,0.0016878],[-0.00028035,-0.00014998,0.0016536],[-0.00044504,0.0009515,0.0015223],[0.00036249,-0.0009345,-0.0014825],[0.00015977,-0.0013482,-0.0012689],[0.00049987,-0.00044939,-0.0015441],[0.00055068,-0.00040778,-0.002196],[0.00054046,-0.00047691,-0.0017788],[0.00053787,-0.00050076,-0.0021497],[0.00024635,-0.00032882,-0.0014739],[0.00058901,-0.0001093,-0.0016819],[0.00049042,-0.00015485,-0.0016655],[0.0001217,-0.00032047,-0.0015155],[9.6894e-06,-0.001754,-2.8599e-05],[6.4109e-06,-0.0017712,-3.9705e-05],[5.7474e-05,-0.0018245,-7.3715e-05],[4.0932e-05,-0.001939,8.4137e-05],[-0.00024433,-0.0013645,0.00088016],[0.00060876,-0.0015497,-0.00022845],[-0.00014151,-0.0011582,0.0012781],[0.00065169,-0.001448,-0.00066564],[2.7063e-05,-0.0020644,8.6479e-05],[9.7886e-05,-0.0016952,-3.6846e-05],[0.00031186,-0.00076266,-0.0014555],[-0.00054293,-0.001925,0.00059191],[0.00013573,-0.0017298,-0.00027665],[0.00015153,-0.0021882,-0.0003054],[0.00022633,-0.0017174,-0.00019958],[-3.8586e-05,-0.001883,-0.00014081],[-6.3172e-05,0.00065455,0.0014904],[0.00061606,0.00054347,0.0015451],[0.00021558,0.00086471,0.0013679],[-0.00033419,0.00068866,0.0012835],[-0.00029458,-0.00027799,0.0016657],[-0.00052248,-0.00033853,0.0015105],[-0.00025651,-0.00016123,0.0017106],[-0.00062584,-0.00030762,0.0014693],[-0.00091935,0.00033734,0.001546],[-0.00028975,-0.00021139,0.0015708],[-6.5485e-05,-0.0016616,0.00060815],[-4.4146e-06,-0.0015683,3.444e-06],[-1.658e-06,-0.0014986,7.4056e-06],[-3.5297e-06,-0.0013987,0.00011977],[-0.00015861,-0.0018739,0.00015271],[0.00085224,0.00078924,-0.0011935],[0.00080365,1.748e-05,-0.0014635],[0.00076448,9.4736e-05,-0.0015223],[0.00060325,0.00069662,-0.0012654],[0.00074486,0.00010371,-0.0015235],[0.00076215,3.3513e-05,-0.0015457],[0.00030732,0.00045288,-0.0012699],[0.00038928,0.00045589,-0.0017129],[-4.5475e-05,0.0010004,-0.0020375],[0.00057259,0.00050026,-0.0015046],[0.00049296,0.00070964,-0.0014939],[-0.00019555,0.0012495,-0.0018953],[0.00044288,0.00083499,-0.0012639],[0.00033571,0.001155,-0.0010583],[0.00063899,0.001592,4.1515e-05],[0.00064025,0.0016476,-6.1093e-05],[0.00061777,0.0016492,-0.00023484],[0.00047002,0.0015043,-0.0005952],[0.00042893,0.0013779,-0.00073348],[0.00034374,0.00152,-0.00091187],[0.00018646,0.0014336,-0.0010324],[-1.6836e-05,0.0016125,0.00082251],[1.2313e-05,0.0015514,0.00093798],[-6.4626e-05,0.00147,0.0010479],[0.00021854,0.0016871,0.00043615],[-0.00021493,0.0013862,0.0011033],[0.00045919,0.0016953,0.0003017],[-0.00032286,0.0012277,0.001195],[-0.00036021,0.00093534,0.001301],[-0.00038381,0.00079379,0.0010576],[0.00061086,0.0015966,0.00012036],[-0.00032896,0.00080991,0.0010615],[-0.0004259,0.00089708,0.0014487],[-9.8259e-05,0.0005167,0.0015246],[-0.00043817,0.00085098,0.0013824],[-9.7981e-05,-0.00023542,0.001697],[-0.00020599,0.00044309,0.0017667],[-0.00030237,-0.00045729,0.0015677],[0.00067591,-9.2664e-05,0.001671],[-0.00023562,-0.00045507,0.0016673],[-9.9106e-05,0.00058935,0.0016581],[-0.0002311,1.8623e-05,0.0017004],[0.00012807,-3.8557e-05,0.0016329],[-0.00056087,0.0010489,0.0013513],[0.00014479,-0.00081682,-0.0016277],[0.00017889,-0.0013845,-0.0012865],[0.00030394,-0.00031993,-0.0016479],[0.00029495,-0.00033418,-0.0016025],[0.00058786,-0.00050617,-0.0016823],[0.00056413,-0.00052425,-0.0017624],[0.00019239,-0.00029022,-0.0015017],[0.00057609,-0.00010436,-0.0016656],[0.00062493,-0.0002277,-0.0015996],[0.00015296,-0.00034691,-0.0015356],[-1.793e-05,-0.0017603,-4.8422e-05],[-5.3457e-06,-0.0017548,-4.8137e-05],[3.2703e-05,-0.0018086,-9.118e-05],[7.4802e-06,-0.0019602,6.0934e-05],[-0.00010717,-0.0012551,0.00093619],[0.00020364,-0.0016022,-0.00049447],[-0.00021788,-0.0011247,0.0011415],[0.00087643,-0.0013927,-0.00049265],[-2.4161e-05,-0.0020767,5.0763e-05],[0.00033262,-0.0017172,0.00012569],[0.00025081,-0.00073006,-0.0014875],[-0.00053304,-0.0018364,0.00055053],[0.00029712,-0.0015321,-2.5413e-05],[0.00015482,-0.0018137,-0.00025662],[0.00016969,-0.001738,-0.0002901],[4.0144e-05,-0.0018325,-7.0663e-06],[0.00064698,0.00026256,0.0018063],[0.00022192,0.00085974,0.0014887],[-7.2778e-05,0.0010756,0.0012385],[-0.00011493,0.00080711,0.0013507],[-0.00044663,-8.3597e-05,0.0016443],[-0.00060905,-0.00023784,0.0015101],[-0.0010713,-0.00028654,0.0014543],[-0.0003963,-0.0005584,0.0014519],[-0.00064363,0.00049438,0.0019067],[-1.1842e-05,-0.00048733,0.0014917],[0.00033611,-0.0017477,0.00067606],[-2.1439e-05,-0.0017293,2.537e-05],[2.5346e-05,-0.0013698,6.0131e-06],[6.0385e-05,-0.0013375,7.7323e-05],[8.5927e-06,-0.0016978,0.00012397],[0.00089704,0.00075229,-0.0011534],[0.00074612,4.7301e-05,-0.0014915],[0.00059413,0.00017405,-0.0015634],[0.00052887,0.00066857,-0.0012016],[0.0008434,4.8343e-05,-0.0014725],[0.00061035,-7.8657e-05,-0.0015896],[0.00049569,0.00066152,-0.00114],[0.00028635,0.00045385,-0.0016546],[0.00017848,0.00073842,-0.0016316],[0.00040212,0.00053055,-0.0015641],[0.00059606,0.00067995,-0.0013965],[0.00012558,0.001013,-0.00099109],[0.00033915,0.00059901,-0.00089],[0.00021155,0.0010599,-0.0010853],[0.00062838,0.0016754,-5.1342e-06],[0.00059824,0.0017104,-0.00013872],[0.00044008,0.0016624,-0.00044273],[0.00041451,0.0014836,-0.00064296],[0.00030938,0.0013391,-0.00083922],[0.00018574,0.0014404,-0.0010271],[5.4528e-05,0.0013413,-0.0011025],[0.00012306,0.0017511,0.00047188],[6.7506e-05,0.0016304,0.00081555],[-1.5371e-05,0.0015544,0.00094517],[-0.00019102,0.0014563,0.0010635],[0.00026313,0.0017555,0.00034479],[-0.0002711,0.001366,0.0010993],[0.00050265,0.0016698,0.00014443],[-0.00028976,0.0012221,0.0011825],[-0.00046636,0.0008953,0.0012836],[-0.00053718,0.0012312,0.0015616],[-0.00045135,0.00087562,0.0014235],[-0.00033698,0.00078593,0.001415],[-0.00030284,0.00086428,0.0013708],[-0.00022315,-0.00027831,0.0017753],[-0.00014008,0.00060146,0.0017074],[-0.00055127,0.00037755,0.001747],[-0.00028951,-0.00048579,0.0015941],[0.00016725,0.00048433,0.001732],[-0.00058862,-5.1325e-05,0.0017109],[0.00014707,-0.00035959,0.0016762],[0.00045265,-0.00033589,0.0016101],[-0.00063656,0.0011803,0.0015306],[-0.00041134,0.0010638,0.0013313],[0.00026202,-0.00076474,-0.0016267],[0.0001788,-0.00038815,-0.0017021],[6.9916e-05,-0.0013868,-0.001239],[0.00044287,-0.00046016,-0.001688],[0.00055916,-0.00055312,-0.0017476],[0.00059736,-0.00010728,-0.0017237],[0.00042864,-0.00052665,-0.0016414],[0.00015335,-0.00031573,-0.0015201],[0.0005669,-0.00027099,-0.0016208],[0.00027554,-0.00029113,-0.0015706],[-1.8447e-05,-0.0017754,-4.3503e-05],[0.00019313,-0.0016892,-0.00013421],[-2.7542e-06,-0.0018172,-7.7923e-05],[1.7155e-05,-0.0019811,5.8125e-05],[-0.00014921,-0.0012593,0.00096437],[-0.00014287,-0.0011679,0.0011312],[0.00091119,-0.0013714,-0.00079814],[2.518e-05,-0.0020967,3.6164e-05],[0.00035936,-0.0018037,-0.00027906],[0.00033634,-0.00069142,-0.0015055],[-0.00056489,-0.0019227,0.00066029],[3.3421e-05,-0.0024427,-7.1039e-05],[0.00013739,-0.0018006,-0.00027659],[0.00018048,-0.0017698,-0.00029625],[0.00012379,-0.0019399,-1.1127e-05],[0.00069243,0.00028101,0.0019319],[0.00059637,0.0010235,0.0014682],[0.00038118,0.0011904,0.0011527],[-0.00038461,0.0010511,0.0012763],[-0.0011183,-0.0001467,0.0013261],[-0.0010811,-0.00030704,0.0012309],[-0.0012913,-0.00016725,0.0015328],[-0.00057898,-0.00060227,0.0014236],[-0.001433,0.0010858,0.00071339],[-0.00038607,-0.00056933,0.0014423],[0.00016511,-0.0016999,0.00087523],[4.2425e-05,-0.0017371,3.2588e-05],[3.0215e-05,-0.0016681,8.0479e-06],[5.1243e-06,-0.0018096,8.7407e-05],[1.7591e-05,-0.0017562,0.0001118],[2.7045e-05,-0.0016647,0.00015914],[0.00065657,0.00053635,-0.0013866],[0.00075198,0.00088531,-0.00098358],[0.00061286,-6.2192e-05,-0.0015591],[0.00081547,0.00034708,-0.0014584],[0.0007268,-4.6087e-05,-0.0015414],[0.00073017,-0.00014387,-0.0015344],[0.00028076,0.00073123,-0.0011974],[0.00030051,0.00047774,-0.0016716],[0.00031723,0.0003662,-0.0016271],[0.00026073,0.0007835,-0.0013838],[0.00043393,0.0003354,-0.0014163],[0.00030722,0.00049877,-0.00085743],[3.9868e-05,0.0011515,-0.0011019],[0.0002238,0.00085694,-0.0011286],[7.2676e-05,0.001185,-0.0011331],[0.00044665,0.0017021,2.1003e-05],[0.00038027,0.0017324,-0.00010325],[0.00031425,0.0016511,-0.00041071],[0.00027284,0.0014163,-0.00056955],[0.00014853,0.0011126,-0.00062944],[-2.7464e-05,0.0017464,-0.0011313],[-0.00018869,0.0014673,-0.001108],[1.0233e-05,0.0017721,0.00034953],[-0.00016166,0.0017151,0.00056306],[-0.00015894,0.0016311,0.00079308],[-0.00026291,0.0014861,0.00097414],[0.00018396,0.0018075,0.00026492],[-0.00037847,0.0014114,0.0009567],[0.0004431,0.0017498,6.4523e-05],[-0.00046602,0.0013218,0.00097691],[-0.00053651,0.00094709,0.0012267],[-0.0004179,0.0009678,0.001213],[-0.00046335,0.00087871,0.0013771],[-0.0003476,0.00078604,0.0014177],[-0.00031082,0.00087328,0.0013673],[-0.00061066,0.00014576,0.0016922],[-0.00035412,0.00081355,0.0014522],[-0.00055669,0.0003729,0.0018225],[-0.000124,-0.00044894,0.0016516],[-0.00045412,0.00031682,0.0016981],[0.00030107,0.00014945,0.001711],[0.00048452,-0.00026075,0.0016497],[0.00021683,-0.00041451,0.0016441],[-0.00066991,0.0011028,0.0012222],[-0.00045493,0.001077,0.0012351],[0.0003028,-0.00080323,-0.0016451],[0.00031458,-0.00049158,-0.0016838],[-1.3413e-05,-0.0013638,-0.0013027],[0.00030644,-0.00049924,-0.0016458],[0.00033746,-0.00037569,-0.0017413],[0.00046589,-3.0958e-05,-0.0016953],[0.00045123,-0.00054782,-0.0016592],[0.00011694,-0.00029427,-0.00157],[0.00057006,-0.000274,-0.0015835],[0.00030848,-0.0003158,-0.0015967],[1.3994e-05,-0.0017729,-1.9967e-05],[7.7594e-05,-0.0017576,-0.00021969],[4.6736e-05,-0.0017974,-4.1161e-05],[8.1521e-05,-0.0019403,9.9994e-05],[-7.7803e-05,-0.0011897,0.00097971],[-0.00020099,-0.0011815,0.001079],[0.0010599,-0.0013297,-0.00067745],[-4.065e-05,-0.0021335,-9.1832e-06],[0.00041145,-0.0018012,-0.00024011],[0.00031458,-0.00068397,-0.0015326],[-0.0004418,-0.0018108,0.00071066],[0.00036799,-0.0022298,0.00032655],[0.00039538,-0.0017372,-0.00029261],[0.00029572,-0.0017546,-0.00011162],[9.3591e-05,-0.001883,-5.8509e-05],[0.00061156,0.0004066,0.0021198],[0.00039819,0.0012076,0.0014429],[4.5316e-05,0.0014135,0.00095641],[-7.3569e-05,0.0010478,0.0011611],[-0.0011127,-0.00019301,0.0013669],[-0.00062587,-0.00083432,0.0012824],[-0.001535,-0.0001818,0.0012699],[-0.00044653,-0.00074958,0.0014238],[-0.0014505,-1.5601e-05,0.0013522],[-0.0012607,0.0011643,0.00079602],[2.774e-05,-0.00099542,0.0013627],[0.0001913,-0.0016044,0.00083484],[-5.3354e-05,-0.0017438,0.00014035],[4.43e-05,-0.0016828,1.0131e-05],[5.0753e-05,-0.0016105,3.0909e-05],[-1.5518e-05,-0.0011049,7.1285e-05],[-3.6236e-07,-0.001103,0.00010805],[0.00065822,0.00053111,-0.0013777],[0.00068378,0.00096187,-0.0010498],[0.00057044,-4.3894e-05,-0.0015574],[0.00068771,0.00040349,-0.0014842],[0.00043781,0.00010252,-0.0016183],[0.00056858,-0.00025921,-0.0015809],[0.00055929,0.0010035,-0.00086145],[0.00013341,0.00048412,-0.0015936],[0.00011983,0.00038556,-0.0015993],[0.00020427,0.00081243,-0.0014073],[0.00043663,0.0003454,-0.0014549],[0.00025663,0.00065262,-0.0010924],[0.00027704,0.0011999,-0.00059543],[0.00024452,0.00086039,-0.0011006],[-1.4886e-05,0.0011143,-0.0011688],[0.00037491,0.0017617,-7.4177e-05],[0.00026207,0.0017584,-0.0002431],[0.00024944,0.0016358,-0.00047651],[0.00015487,0.0013795,-0.00068197],[-1.3132e-05,0.0010543,-0.00077959],[-0.00020378,0.0016099,-0.0012533],[-0.00029419,0.0013523,-0.001157],[3.6892e-05,0.0018366,0.00029175],[-8.506e-05,0.0018046,0.00037155],[-5.8485e-05,0.0017303,0.00054994],[-0.00018353,0.0016267,0.00079552],[-0.00038562,0.0014497,0.00097526],[0.00024345,0.0017971,9.6192e-05],[-0.00045594,0.0013773,0.0009498],[-0.00044984,0.0013159,0.00096995],[-0.00069775,0.0011911,0.0011039],[-0.00030065,0.00089419,0.0013516],[-0.00029576,0.00070049,0.0013346],[-0.00024679,0.00081924,0.0012713],[-0.00042139,0.00020465,0.0017574],[-0.00036929,0.00096959,0.0017219],[-3.7401e-05,0.0005158,0.0017526],[-0.00049689,-6.5375e-05,0.001694],[-1.7515e-05,-0.0001895,0.001759],[0.00085018,-0.00045201,0.0016541],[0.00053301,-0.00030283,0.0016663],[0.00023276,-0.00043847,0.0017152],[-0.00060866,0.0012044,0.0010948],[-0.00043816,0.0011537,0.0012143],[-0.00023467,0.0011014,0.001205],[7.2877e-05,-0.00087922,-0.0016045],[0.00032646,-0.00048481,-0.001679],[-3.9906e-05,-0.0013528,-0.001279],[0.00033075,-0.00052274,-0.001668],[0.00046888,-0.00029837,-0.0016927],[0.00035088,-0.00010537,-0.0017569],[0.00043753,-0.0003592,-0.0016056],[0.00031631,-0.00058427,-0.0016196],[0.00032104,-0.00020788,-0.0016341],[0.00034424,-0.00030166,-0.0016116],[7.2389e-05,-0.001805,-4.3292e-05],[1.4906e-05,-0.001811,-2.91e-05],[0.0010954,-0.0013818,-0.0006652],[3.5439e-05,-0.0019062,0.00011526],[-1.7751e-05,-0.0012541,0.00099368],[-8.489e-05,-0.0012296,0.0010475],[1.7289e-05,-0.0021489,-2.6554e-05],[0.00040158,-0.0017143,-0.0004094],[0.00035005,-0.00067349,-0.0015491],[-0.00035056,-0.0018194,8.0015e-05],[0.00028574,-0.0018994,0.00046882],[0.0004222,-0.0016904,0.00023029],[0.0005414,-0.0016444,-4.5263e-05],[0.00011596,-0.0017636,-0.00010453],[-5.8312e-05,-0.0019757,-7.2031e-05],[0.00050014,0.00037583,0.0022114],[0.00033968,0.0012761,0.0015671],[5.4556e-05,0.0015257,0.001031],[-0.0010146,0.001476,0.00034619],[-0.0012716,-0.00020156,0.0011391],[-0.00062294,-0.0008477,0.0013149],[-0.0012937,-0.00023775,0.0011321],[-0.00038497,-0.00073347,0.0014288],[-0.0016739,3.9468e-05,0.00067211],[-0.0015628,0.001379,0.0002532],[-9.1597e-05,-0.0010312,0.0013556],[-0.00012123,-0.0015431,0.0011666],[9.6938e-05,-0.0016895,0.00016601],[4.5518e-05,-0.0016978,9.3718e-06],[2.0457e-06,-0.0015636,2.1132e-05],[1.7115e-05,-0.0015165,6.4033e-05],[-8.8856e-05,-0.0020016,0.00016081],[0.00051263,0.00041401,-0.0014772],[0.00054829,0.00084285,-0.0011731],[0.00071608,6.2756e-05,-0.001504],[0.00042713,0.00021046,-0.0016021],[0.00044965,0.00011389,-0.0015882],[0.00045748,-0.00019791,-0.0015878],[0.00043178,0.0010595,-0.00089062],[0.00028606,0.00069436,-0.00153],[0.00015696,0.00044691,-0.0016379],[0.00026183,6.5866e-05,-0.0017366],[0.00073326,0.001542,-0.0011314],[0.00031023,0.00081728,-0.0014116],[0.00030776,0.001229,-0.00061147],[4.8725e-06,0.00099418,-0.0011589],[-0.0002541,0.0012172,-0.0011807],[0.00019137,0.0017866,-4.5168e-05],[0.00015074,0.0017619,-0.00022288],[0.00011295,0.0016285,-0.00044209],[4.7468e-05,0.0012237,-0.00056768],[-0.00010906,0.0011428,-0.00071074],[0.00026719,0.0014275,-0.0012166],[4.8382e-05,0.0012985,-0.001212],[-4.6916e-05,0.0018352,0.0002009],[-0.00016378,0.0017934,0.00027569],[-0.00017931,0.0017408,0.00040015],[-0.00031821,0.0016604,0.0006281],[-0.00051808,0.0014939,0.00080008],[0.00017234,0.0018268,1.7783e-05],[-0.00055012,0.001409,0.00082152],[-0.00053425,0.0013746,0.00089759],[-0.00060506,0.0013296,0.00088267],[-0.00031608,0.00090446,0.0013285],[-0.00018045,0.00047219,0.00089686],[-0.000192,0.00077993,0.0013322],[-0.00020703,-3.7329e-05,0.0017987],[-0.00017822,0.00048051,0.00089598],[5.2002e-05,0.00041866,0.0017804],[7.7803e-05,0.00011177,0.0018271],[0.00098694,0.00016474,0.001591],[0.00048199,-0.00053866,0.0016844],[0.00024586,-0.00040641,0.0017434],[-0.00020998,-0.00056548,0.0016547],[-0.00062077,0.0011874,0.0010319],[-0.00034797,0.0010629,0.001256],[-0.00014,0.0010188,0.0012851],[4.0539e-05,-0.00086732,-0.0016354],[0.00036954,-0.00051939,-0.0016879],[-0.0001885,-0.0012905,-0.0013686],[0.00035927,-0.00049995,-0.0016396],[0.00024362,-0.00012675,-0.0017544],[0.00031557,-8.658e-05,-0.0016831],[0.00028436,-0.00025926,-0.0016135],[0.00037748,-0.00064308,-0.001646],[0.00023877,-0.00014978,-0.0016917],[0.00043318,-0.0003639,-0.0016346],[0.00031679,-0.0017526,0.00013213],[0.00013818,-0.0018227,5.8298e-05],[0.0011637,-0.0013217,-0.00057733],[0.00028071,-0.001828,0.0002735],[9.221e-05,-0.0011935,0.0010575],[-0.0001134,-0.0012071,0.00099773],[-4.4522e-05,-0.0021738,-6.7595e-05],[0.00043892,-0.0016909,-0.00037265],[0.00036402,-0.00069161,-0.001579],[-1.6212e-05,-0.0017275,0.00036121],[0.00020607,-0.0017613,0.00037226],[0.00028945,-0.0016254,7.1335e-05],[0.00025416,-0.0014969,-1.4602e-05],[0.00022893,-0.0017166,8.2897e-05],[3.8786e-05,-0.0019537,0.00011854],[-0.00012814,-0.0019333,5.4483e-05],[0.00016345,0.00062053,0.0022355],[-1.0384e-05,0.0013832,0.0012692],[-0.00034664,0.001743,0.00075561],[-0.00025011,0.0020304,0.00037774],[-0.0006289,-0.00093652,0.0012265],[-0.00043435,-0.00097163,0.0012568],[-0.0015253,-0.00024143,0.00086498],[-8.2127e-05,-0.00098604,0.0013665],[-0.0015616,-0.00034925,0.00079333],[-0.0016074,0.00017136,0.00076687],[-0.0014693,0.0011853,0.00016196],[-0.00030271,-0.0009309,0.0015018],[0.0002183,-0.0013311,0.0011386],[-7.419e-05,-0.0017463,0.00035323],[7.9552e-06,-0.0016346,1.7583e-06],[1.0949e-05,-0.0015714,1.3271e-05],[8.4043e-07,-0.001482,5.9089e-05],[1.1646e-06,-0.0014167,5.7645e-05],[0.00058522,0.00037621,-0.0014452],[0.00069095,0.00076897,-0.0011263],[0.00046395,0.0001788,-0.0015583],[0.00052686,0.00015384,-0.0015465],[0.00044641,0.00010838,-0.0015542],[0.00027947,-0.00030542,-0.0015941],[0.00050813,0.0011002,-0.00075414],[0.00016796,0.00067664,-0.0014442],[0.00028399,0.00040352,-0.001547],[-5.5685e-05,9.9543e-05,-0.0016302],[0.0002415,0.001652,-0.0011462],[7.8989e-05,0.00082908,-0.0013587],[0.00031037,0.0011648,-0.00077665],[-3.1196e-06,0.0010212,-0.0012025],[-0.00025077,0.0011795,-0.0011495],[0.00011793,0.0018051,-0.00012798],[6.3421e-05,0.0017668,-0.0003201],[-7.1276e-06,0.0015965,-0.00056696],[5.2907e-05,0.0016221,-0.0010347],[8.4138e-05,0.0012881,-0.0013531],[0.00014909,0.0016508,-0.00092551],[-9.0172e-05,0.0011762,-0.001301],[1.1643e-05,0.0018338,4.2541e-05],[-0.00014173,0.0018402,0.00021584],[-0.00012756,0.0017968,0.00027068],[-0.00022841,0.0017426,0.00040812],[-0.00045458,0.0016196,0.0006365],[-0.00053735,0.0014529,0.00078397],[-0.00049913,0.0014089,0.00081314],[-0.00061708,0.0013407,0.00087712],[-0.00014515,0.0008937,0.0012615],[-0.00026708,0.00066539,0.0012389],[-0.00025646,0.00070197,0.001228],[0.00025888,0.00011245,0.0017819],[-1.5906e-05,0.0005809,0.0013641],[0.00017077,0.00078857,0.0016413],[0.00026734,0.00046455,0.0016762],[0.00021351,-4.571e-05,0.0018093],[0.00058551,0.00057944,0.0016828],[4.1617e-05,-8.921e-05,0.0018711],[3.3609e-05,-0.00020902,0.0018166],[-0.00038236,-0.00042049,0.001684],[-0.00055295,0.0013388,0.00086382],[-0.00023733,0.0012855,0.0010145],[-0.00010447,0.0010983,0.0012186],[4.606e-06,0.001019,0.0012373],[-0.00011054,-0.00091061,-0.001599],[7.5334e-05,-0.00062344,-0.0016479],[9.9937e-05,-0.0011991,-0.0014253],[0.00022991,-0.0004063,-0.0017306],[0.00023388,-0.00012828,-0.0016876],[0.00044634,-2.8734e-05,-0.0017191],[0.00029801,-0.00025489,-0.0016261],[0.00017774,-0.00071051,-0.0016233],[0.00047373,-4.21e-05,-0.0017099],[0.00039994,-0.00039534,-0.0016767],[0.0012056,-0.0014397,-0.00020366],[0.00032753,-0.0017515,-1.7363e-05],[0.00011276,-0.0017905,0.00033294],[0.00040256,-0.0016006,-0.00025025],[0.00016515,-0.0012411,0.0010604],[0.00011054,-0.0012643,0.00091306],[4.3971e-05,-0.002142,-9.337e-05],[0.00028983,-0.00073735,-0.0015878],[9.3174e-05,-0.0017089,-0.00024246],[0.00021465,-0.0016911,0.00027884],[0.00021108,-0.0017107,0.00035935],[0.00011627,-0.0012661,8.0667e-05],[0.00023423,-0.0015191,-5.0252e-05],[5.1514e-05,-0.0017358,8.0377e-05],[-0.00027293,-0.0020874,7.7126e-05],[-0.0006738,-0.0021082,-1.5474e-05],[-0.00022516,0.00048243,0.0021859],[-0.000568,0.0013368,0.0013859],[-0.00030393,0.0015558,0.00067313],[-0.00068925,0.0020365,-9.1798e-05],[-0.00088535,-0.00097861,0.0010621],[-0.00026255,-0.00095647,0.0013452],[-0.00083202,-0.001036,0.00088129],[-4.1751e-05,-0.00097906,0.0013779],[-0.0020126,-0.00041095,0.00068301],[-0.0017507,0.00021678,0.00021824],[-0.0013199,0.0010819,-0.00022688],[-0.00061388,0.001885,-0.00022319],[-9.3216e-05,-0.00085642,0.0015003],[-0.00061858,-0.00079547,0.0016267],[1.1874e-05,-0.0017801,0.00038136],[-6.2319e-05,-0.0017343,6.5652e-05],[-2.8502e-05,-0.0015892,3.9796e-06],[-1.3401e-05,-0.0014695,6.9973e-05],[-0.00019392,-0.0012548,9.6702e-06],[0.00053357,0.00066434,-0.0013403],[0.00045942,0.00027889,-0.0015689],[0.00052351,0.00022154,-0.0015504],[0.00044253,9.6834e-05,-0.0015393],[0.00035992,5.4858e-05,-0.0015279],[0.00035298,-0.00034566,-0.0015171],[0.00046798,0.0010219,-0.0012814],[0.00029627,0.00042014,-0.001589],[0.00026651,0.0004824,-0.0015798],[-0.00022794,0.00056094,-0.001535],[0.00036111,0.0016404,-0.00095492],[6.6745e-05,0.0014212,-0.0013158],[-0.00021395,0.001086,-0.0012111],[-0.00011344,0.0011921,-0.0012033],[-3.3006e-05,0.0018088,-0.00010276],[-3.5905e-05,0.001775,-0.00030348],[-3.0051e-05,0.0016684,-0.00058803],[-9.6463e-05,0.0016463,-0.00065879],[8.6213e-06,0.0012631,-0.0013073],[-3.5652e-05,0.0016306,-0.00087871],[3.9798e-05,0.0011453,-0.001301],[-5.8682e-05,0.0018309,-3.3518e-05],[-0.00021743,0.0018187,0.00012611],[-0.0002097,0.001785,0.00016868],[-0.00030766,0.001736,0.00030146],[-0.0005154,0.0016058,0.00053638],[-0.00062892,0.0014876,0.00067363],[-0.00061904,0.0014587,0.00066753],[-0.00053441,0.0014795,0.00066356],[-6.5852e-05,0.00084752,0.0013576],[5.332e-05,0.00080472,0.0012788],[6.5656e-05,0.00047887,0.0014698],[0.00018181,0.0001777,0.0016919],[0.00054927,0.00035522,0.0015537],[0.00012442,0.00057081,0.0015816],[0.00045573,2.7517e-05,0.0017604],[0.00016219,0.0004402,0.0018229],[6.4473e-05,-7.7703e-05,0.0017951],[-0.00035513,-0.00031866,0.001695],[-0.00026644,-0.00037716,0.0016538],[-0.00057982,0.0013397,0.00081385],[-0.00012811,0.0012031,0.0011236],[2.6837e-06,0.0010119,0.0013166],[0.00011233,0.00095266,0.00135],[-0.00021801,-0.00085017,-0.0016578],[1.5275e-05,-0.000594,-0.0017127],[-0.0003324,-0.00097971,-0.0016573],[3.3426e-05,-0.00048192,-0.0017181],[0.0003652,-0.00022827,-0.0016553],[0.00035932,3.9369e-05,-0.0017411],[0.00039659,-0.0003199,-0.0015608],[0.00021368,-0.00074497,-0.0016274],[3.0989e-05,0.00028481,-0.0019144],[0.00019968,-0.00027142,-0.0018765],[0.001375,-0.0013272,-3.7757e-05],[0.0012672,-0.0013456,0.00064997],[0.00098908,-0.0013421,0.00088577],[0.00061342,-0.0015516,-8.2846e-05],[0.00017567,-0.0012522,0.001079],[-5.4917e-05,-0.0013432,0.00081203],[0.00016486,-0.0020886,-2.0123e-05],[0.00014341,-0.00040658,-0.00090929],[0.00026862,-0.0019571,-0.00014371],[-1.6828e-05,-0.0014967,5.593e-05],[1.2723e-05,-0.0016174,0.00010797],[8.2795e-05,-0.0012381,2.9429e-05],[5.1301e-05,-0.0013039,-2.6447e-05],[1.8847e-05,-0.0017301,2.2232e-05],[-0.00047899,-0.0019701,0.00042525],[-0.00024579,-0.0021345,0.00014777],[-0.00026316,-0.0012362,-0.00023868],[-0.00064683,0.00075711,0.0015402],[-0.00045666,0.0010954,0.0011533],[-0.0007755,0.0015806,0.00025288],[-0.0007583,0.0017248,-4.0562e-05],[-0.00033765,-0.0013806,0.00098953],[-6.7745e-05,-0.0010961,0.0012961],[-0.00072356,-0.0010278,0.00097793],[-0.00011787,-0.00096159,0.0014521],[-0.00080809,-0.0016273,0.00058418],[-0.0019657,-0.00055032,0.00054135],[-0.0017048,0.00018151,0.00018789],[-0.0014073,0.0010105,-0.00032568],[-0.00079896,0.001658,-0.00048113],[-0.00035439,-0.00069049,0.0016194],[9.2965e-05,-0.00045489,0.001562],[-0.00027447,-0.0015669,0.00060444],[-7.5696e-05,-0.0016659,5.8452e-05],[-7.6174e-05,-0.0016199,4.1439e-05],[-0.00023033,-0.0013608,-1.1916e-06],[-0.00021455,-0.0012879,2.069e-05],[0.00015425,0.00078336,-0.0013468],[0.00047784,0.00026476,-0.0015438],[0.0006164,0.00016895,-0.0015],[0.00057576,1.9129e-05,-0.0014581],[0.00051657,-4.1332e-05,-0.0014439],[0.00037795,-0.00032216,-0.0014907],[-0.00010488,0.0011457,-0.0012848],[0.00035918,0.00039425,-0.0015334],[0.00029065,0.00045944,-0.0015196],[-7.9334e-06,0.00051579,-0.0015226],[0.00057918,0.001274,-0.001281],[0.00015538,0.0016876,-0.00095701],[-0.00022673,0.0010576,-0.0012033],[-0.00015496,0.0011504,-0.0012264],[-9.0307e-05,0.0018207,-0.00016545],[-0.00011834,0.001774,-0.00039421],[-0.00013167,0.0017105,-0.00067891],[8.5252e-06,0.0012426,-0.0012859],[-0.00014993,0.0015383,-0.00097893],[-0.00012028,0.0017082,-0.00066478],[-1.1107e-06,0.0011015,-0.0013219],[-0.00016328,0.0018191,-1.6427e-05],[-0.00019257,0.0018085,0.0001218],[-0.00025778,0.0017781,0.00017513],[-0.00042751,0.0017005,0.0003134],[-0.00057435,0.0015661,0.00053435],[-0.00061855,0.0014662,0.0006637],[-0.00056468,0.0014879,0.00062189],[0.00019526,0.00086863,0.0012802],[0.0004485,0.00054388,0.0014173],[0.00045693,0.00056773,0.0014132],[0.00069571,0.00031721,0.0016366],[2.7274e-05,0.00021478,0.0017147],[0.00064022,0.00062345,0.0014729],[0.00011061,0.00057885,0.0016181],[0.00060024,-0.00012143,0.0017489],[0.00027119,0.00030966,0.0017373],[-6.821e-05,5.2157e-05,0.001846],[-0.0003728,-0.0003159,0.0017305],[-0.0001108,-0.00052483,0.0016739],[-0.00050551,0.0014895,0.00061312],[2.5996e-05,0.0014569,0.00074929],[0.00011081,0.0012444,0.0010959],[0.00014586,0.0010348,0.0012931],[9.8045e-05,0.00096118,0.0013686],[-0.00030795,-0.00088341,-0.0016493],[-0.00017452,-0.00065047,-0.0016455],[-0.00019786,-0.00094875,-0.0017115],[0.00030897,-0.00070335,-0.0016394],[0.00031242,-0.00025446,-0.0016647],[0.00033511,-0.00035607,-0.0015775],[0.00032769,2.6217e-05,-0.0017291],[0.0003065,-0.00071247,-0.0016335],[0.00030921,0.00042598,-0.0020376],[0.0003815,-0.00019023,-0.0019312],[0.00056366,-0.0015804,0.00025135],[0.0012373,-0.001277,0.00061003],[0.0011056,-0.0011745,0.00070345],[0.00089721,-0.0012276,0.00073339],[0.00019353,-0.0018233,9.2205e-05],[0.00023972,-0.0013763,0.00069026],[0.00011131,-0.0019426,-4.4536e-06],[8.6621e-05,-0.00044786,-0.00092479],[0.00014342,-0.00040661,-0.00090929],[5.4166e-06,-0.0014649,-2.2122e-05],[3.2399e-05,-0.001625,5.2982e-05],[0.00010609,-0.0019361,0.00011789],[8.2793e-05,-0.0012381,2.9429e-05],[-0.00012071,-0.0018892,8.2636e-06],[-0.0012934,0.0010937,-0.00065247],[-0.0003076,-0.0020936,0.0001295],[-0.00069341,-0.0010843,-3.4842e-05],[-0.0010525,0.00050264,0.0013854],[-0.0011765,0.00070347,0.00089415],[-0.0012774,0.00091958,0.00041385],[-0.00090652,0.0016285,-0.00022556],[-0.00028627,-0.0014496,0.0010835],[-0.00034218,-0.0011715,0.0012402],[-0.0002885,-0.0014596,0.0009516],[-0.00034383,-0.00099447,0.0013729],[-0.00064167,-0.0018015,0.00084488],[-0.0019532,-0.00056623,4.0155e-05],[-0.0016838,0.00018341,-0.00019099],[-0.0013077,0.00095774,-0.00046839],[-0.00072999,0.0015768,-0.00058451],[-0.00083249,0.0016173,-0.00046485],[-0.00033369,-0.00066663,0.0015715],[-0.00098711,0.00035192,0.0012961],[-0.00042765,-0.0016612,0.00060808],[-0.00047557,-0.0017348,0.00035518],[-0.00012477,-0.0016988,3.091e-05],[-0.00011947,-0.0015037,-9.0475e-05],[-0.0007447,-0.0011572,-0.00016069],[0.00033952,0.00094524,-0.0012807],[0.00060971,0.00036075,-0.0015187],[0.00058607,0.00014742,-0.0015069],[0.00049653,-1.5991e-05,-0.0013875],[0.00039129,-0.00010192,-0.0013852],[0.00039048,-0.00032857,-0.0014489],[0.00023645,0.0014538,-0.0010526],[0.00025099,0.00027586,-0.0016298],[0.00032971,0.00050434,-0.0015464],[0.00019342,0.00071062,-0.0014482],[0.00051264,0.0011552,-0.0011898],[0.00013451,0.0015728,-0.00090547],[-0.00012397,0.0010673,-0.0012446],[-3.1696e-05,0.0011506,-0.001259],[-0.00016625,0.0018175,-0.00015241],[-0.00016292,0.0017892,-0.00038994],[-0.00022202,0.0017643,-0.00045263],[-0.00031369,0.0012454,-0.0012181],[-0.00030266,0.0015074,-0.00092689],[-0.00026228,0.0016955,-0.00063148],[-0.00012429,0.001102,-0.0012895],[-0.0001996,0.0018054,-5.7874e-05],[-0.00026673,0.0017902,3.2026e-05],[-0.00030849,0.0017538,0.00010489],[-0.00042696,0.0016748,0.00030072],[-0.00058051,0.0015502,0.00051088],[-0.00062023,0.0014507,0.00064361],[-0.00057057,0.0014695,0.0006414],[0.00011398,0.0010017,0.0013162],[5.5877e-05,0.0009229,0.0013248],[0.00017494,0.00083524,0.0015827],[-0.00039354,0.00056881,0.0015496],[0.00020063,0.00095213,0.0013213],[-6.0293e-05,0.00073633,0.001571],[0.00013295,-0.00024665,0.0018179],[-2.1471e-05,0.00023001,0.0018317],[-0.00029992,-2.7545e-05,0.0017649],[-3.4065e-05,-0.00020134,0.0017697],[-0.00029658,-0.00058057,0.0016106],[-0.00041663,0.001441,0.00071522],[1.8935e-05,0.001463,0.00074207],[0.00014255,0.00124,0.0011419],[0.00011279,0.0010789,0.001283],[-6.5091e-06,0.0011052,0.0013764],[-0.00053685,-0.000662,-0.001651],[-0.00013173,-0.00067035,-0.0016071],[-0.00086603,-0.00050752,-0.0019044],[-0.0001704,-0.00083576,-0.0015152],[0.00046422,-0.00037305,-0.0015881],[0.00040854,-0.00040757,-0.0015032],[0.00058925,-0.00018329,-0.0016527],[-1.4804e-05,-0.00044616,-0.0017392],[-0.00020955,0.00058878,-0.0014532],[-8.1324e-05,0.00014021,-0.0020117],[0.00028306,-0.0017652,-1.2453e-06],[0.00050235,-0.0017713,-1.0844e-05],[0.0004192,-0.0015618,0.0002267],[0.00026505,-0.0014238,0.00033263],[0.00010907,-0.0017943,5.8976e-07],[-5.2889e-05,-0.0014273,0.00046961],[6.8695e-05,-0.0019154,-2.9279e-05],[0.00055021,-0.0011271,-0.0010855],[0.00047612,-0.00025405,-0.00098423],[1.1069e-05,-0.001465,-1.9493e-05],[-1.572e-06,-0.0014701,1.1843e-05],[3.6672e-05,-0.0019352,7.241e-06],[-2.815e-05,-0.0019095,2.4281e-05],[-7.6274e-05,-0.0018266,8.7396e-05],[-0.0015003,0.00087022,-0.0012136],[-0.0015153,0.00011225,0.0012502],[-0.0002743,-0.0019282,0.00014356],[-0.0013528,0.00019327,0.0013299],[-0.0011586,0.00054832,-0.00070049],[-0.0013519,0.0007879,0.0010279],[-0.0013759,0.001001,0.0004554],[-0.0013526,0.0011164,-3.7686e-05],[-0.00037287,-0.0013149,0.001061],[-0.00037823,-0.0011398,0.0012466],[-0.00047503,-0.0016305,0.00094983],[-0.00042007,-0.00097924,0.0014355],[-0.0006785,-0.0019215,0.00076789],[-0.0004821,-0.0016721,0.00069338],[-0.00070489,-0.0016238,0.00077788],[-0.0017652,-0.00043203,9.2289e-05],[-0.0017756,0.00029772,-0.00012074],[-0.0014291,0.00091494,-0.00058022],[-0.00086806,0.001451,-0.00078001],[-0.00092732,0.0014854,-0.00061093],[-0.00032234,-0.00068745,0.0015885],[-0.0012432,0.00017228,0.0012134],[-0.00048457,-0.0015881,0.00030591],[-0.00048401,-0.0015857,0.0003022],[-0.0011363,-0.001045,-0.00055953],[-0.001144,-0.0010921,6.1138e-05],[0.00056635,0.00089717,-0.0013099],[0.00055774,0.00039608,-0.0015618],[0.0005294,0.00017573,-0.0015233],[0.00032819,7.25e-05,-0.0014085],[0.00022547,-1.077e-06,-0.0014197],[0.00040258,-0.00032339,-0.0014508],[-4.9562e-05,0.0014652,-0.0010116],[0.00021479,0.00027374,-0.0015848],[0.00026208,0.00051553,-0.0015491],[0.00023694,0.00074187,-0.0015235],[0.0002623,0.0012211,-0.0011862],[-4.382e-06,0.0015516,-0.00087021],[1.9626e-06,0.0012014,-0.0011798],[-0.00011757,0.001062,-0.0013096],[-0.00023068,0.0018236,-0.00022229],[-0.00026246,0.001767,-0.00044608],[-0.00041449,0.0013929,-0.0010179],[-0.00023523,0.0013114,-0.0011528],[-0.00037435,0.0015812,-0.00073503],[-0.00030954,0.0017152,-0.00049404],[-0.00012903,0.0010734,-0.0012696],[-0.00023366,0.001789,-5.1478e-05],[-0.00028148,0.0017717,3.4362e-05],[-0.00035182,0.0017323,0.0001102],[-0.00050378,0.0016477,0.00030802],[-0.00057263,0.0015361,0.00050581],[-0.00056384,0.0014436,0.00062471],[7.0042e-05,0.00099418,0.001328],[0.0002052,0.0009692,0.0013019],[-0.00026096,0.00073476,0.0016588],[-0.00031093,0.00077135,0.0013829],[-7.0486e-05,0.00066237,0.0015219],[4.4189e-05,0.00077235,0.0015831],[-0.00038158,0.00026668,0.0017667],[0.00012359,7.9487e-05,0.0018169],[6.816e-05,-0.00037045,0.0017711],[0.00015981,-0.0003732,0.0017643],[-0.00031896,-0.00055966,0.0016085],[-0.00037654,0.0014941,0.00060672],[5.2208e-05,0.0015338,0.00066225],[0.00030412,0.0014841,0.00068596],[0.00018737,0.001233,0.0011224],[-3.1854e-06,0.0011403,0.001399],[-4.0669e-05,0.0011656,0.0014677],[-0.00055347,-0.00079554,-0.0015095],[-0.00085755,-0.00074077,-0.0015632],[-0.0011755,-0.00062081,-0.002089],[-0.00039367,-0.00064747,-0.001559],[0.00044139,-0.00040379,-0.0015288],[0.00054342,-0.00034254,-0.0015879],[-1.2129e-05,-0.00042466,-0.0017019],[-0.00036246,-0.00052913,-0.0015878],[-0.00019948,0.00057222,-0.001367],[7.3424e-05,0.00071588,-0.0014966],[0.00013416,0.00024811,-0.0021864],[0.00025996,-0.0017397,9.0322e-05],[8.4946e-05,-0.001741,6.734e-05],[0.00039533,-0.0015588,0.00023736],[0.00037399,-0.0014212,0.00027973],[0.00027096,-0.0014549,0.0003268],[5.7621e-05,-0.0019797,1.7642e-05],[1.181e-05,-0.0017821,-5.9083e-06],[0.00011016,-0.0013238,-0.0010024],[0.0002205,0.00057184,-0.0017292],[4.9143e-05,-0.00213,5.1009e-05],[2.9088e-05,-0.0017037,1.2799e-05],[-1.631e-06,-0.0018044,5.1709e-06],[-2.206e-05,-0.0019106,3.542e-05],[-0.00021141,-0.0018791,7.2774e-05],[-0.0010176,0.00094327,-0.0011289],[-0.0018263,0.0010002,-0.0012392],[-0.0011443,0.0011985,0.001076],[-0.00037911,-0.0018852,0.00011776],[-0.0014371,0.0002024,0.0014196],[-0.0012345,0.0005265,-0.00050202],[-0.0014154,0.00063643,0.00087614],[-0.0014493,0.00075188,0.00049723],[-0.0014148,0.0010573,-0.00011304],[-0.00039873,-0.0013817,0.0011108],[-0.00047437,-0.0011528,0.0012123],[-0.0004089,-0.0014461,0.00083366],[-0.00046405,-0.00093121,0.0013231],[-0.0004107,-0.0016422,0.00073618],[-0.00042067,-0.0016435,0.00071372],[-0.00064954,-0.0014858,0.00054671],[-0.0016737,-0.00043169,-0.00019534],[-0.00143,0.00025201,-0.00060523],[-0.0015937,0.00099838,-0.00050716],[-0.00084171,0.001436,-0.00083715],[-0.00083365,0.0014253,-0.00076515],[-0.0012603,0.001045,-0.00043824],[-0.00056194,-0.00070315,0.0013619],[-0.0015212,0.00039571,0.00094593],[-0.0014743,0.00032936,0.00084572],[-0.0017148,-0.00018349,-0.00036408],[-0.0018882,-0.00034194,0.00025386],[0.0004681,0.00082844,-0.0013955],[0.00053621,0.00037995,-0.0015921],[0.00043488,6.7025e-05,-0.0018415],[0.00016687,-3.6797e-05,-0.0010537],[0.00014275,-2.0916e-05,-0.0010718],[0.00016681,-0.00015366,-0.0014685],[2.6898e-05,0.0015152,-0.00095247],[0.0003509,0.00041454,-0.0014712],[0.00016726,0.00041375,-0.0015812],[0.00019786,0.00069946,-0.0015325],[0.00017848,0.0011551,-0.0012479],[9.2538e-06,0.0015566,-0.00085622],[-0.00020937,0.0012532,-0.00118],[-0.00011752,0.0010691,-0.0013187],[-0.00025417,0.0018036,-0.00021544],[-0.00032092,0.0017821,-0.00028871],[-0.00046422,0.0013855,-0.0010025],[-0.0005737,0.0013244,-0.0010923],[-0.00045045,0.0015525,-0.00070518],[-0.00037555,0.0016896,-0.00047298],[-9.4795e-05,0.0010868,-0.0012956],[-0.00028632,0.0017684,-0.00011384],[-0.0003354,0.0017438,-3.6001e-05],[-0.00037578,0.0017097,7.2232e-05],[-0.00052703,0.0016134,0.00025452],[-0.00060463,0.0015239,0.00044812],[-0.00049918,0.0015474,0.00043515],[-3.11e-05,0.0011464,0.0013757],[6.2498e-05,0.0011004,0.0012711],[-3.8045e-05,0.00051364,0.0017008],[-0.00012041,0.00061737,0.0014444],[-0.00011619,0.00070244,0.0015042],[0.00026475,0.00057273,0.0016773],[0.00021709,0.00042813,0.0017263],[0.00019372,9.9807e-05,0.0018417],[0.00017361,-0.00034523,0.001825],[-0.00032432,-0.00054067,0.0016917],[-0.00061337,-0.00065082,0.0014765],[-0.0001912,0.0013471,0.00078268],[0.00021246,0.0014222,0.0008446],[0.00051618,0.0013672,0.0009254],[0.00015923,0.0012865,0.0011187],[-0.00012619,0.0010055,0.0010108],[-0.0001405,0.0014236,0.0016457],[-0.00066272,-0.00067346,-0.0014976],[-0.00087812,-0.00062888,-0.0014841],[-0.0011022,-0.00022098,-0.0016325],[-0.00066257,-0.00070738,-0.0014498],[0.0002812,-0.00027695,-0.001549],[0.00041874,-0.00023905,-0.0016411],[3.1287e-05,-0.00046584,-0.0017103],[-0.0004057,-0.00049355,-0.0015979],[-4.3603e-05,0.00059546,-0.0013284],[-7.8392e-06,0.00079195,-0.0014333],[-0.00013411,0.00045881,-0.0018044],[5.6767e-06,-0.0019384,-0.00010058],[-4.8167e-05,-0.0020888,-6.7835e-05],[6.7586e-05,-0.0021847,-6.8382e-05],[9.7781e-05,-0.0011735,6.0188e-05],[-2.0873e-05,-0.0015207,0.00014279],[1.5686e-05,-0.0014386,-3.0741e-05],[-3.3678e-05,-0.0016711,-3.2432e-05],[1.1601e-05,-0.0015313,-0.0012736],[0.00016288,0.00058211,-0.001862],[2.7338e-05,-0.0016265,3.8818e-05],[3.8202e-05,-0.001761,2.7325e-05],[1.6627e-05,-0.001879,3.9218e-05],[-2.1482e-06,-0.001822,3.5401e-05],[-0.00014188,-0.0018059,0.00021174],[-0.001476,0.00085631,0.000707],[-0.00087933,0.001197,-0.00089541],[-0.00117,0.0008529,-0.00072018],[-0.0010414,0.0010349,0.0012895],[-0.0013088,0.00027027,0.0015084],[-0.00022243,-0.0018037,0.00050854],[-0.0010912,0.0010121,0.0012582],[-0.0014314,0.00072549,-0.00050408],[-0.0015314,0.00062317,0.00036076],[-0.0015337,0.0007247,-2.1201e-05],[-0.00049215,-0.0012339,0.0010964],[-0.00050785,-0.0010885,0.0011952],[-0.00056641,-0.001512,0.00079011],[-0.00061941,-0.00084085,0.0013778],[-0.00041698,-0.0016758,0.00060353],[-0.00042629,-0.001662,0.00075385],[-0.00056011,-0.0014947,0.00078372],[-0.00050849,-0.0015704,0.00065563],[-0.00077451,-0.0012653,0.000635],[-0.0015961,-0.0003737,-0.00016402],[-0.00179,0.00078307,-0.00037959],[-0.0011062,0.0015015,-4.6606e-05],[-0.00091133,0.0012718,-0.00094823],[-0.00094645,0.0012793,-0.00093863],[-0.0012718,0.0008632,-0.00054309],[-0.00059273,-0.00068136,0.0013688],[-0.0015697,0.00051024,0.00038194],[-0.001645,0.00070314,-7.62e-05],[0.00025427,0.00090801,-0.0014153],[0.00024781,0.00049034,-0.0016133],[0.00031095,0.00011359,-0.0017591],[0.00022592,-7.7189e-05,-0.00098512],[0.00022591,-7.7191e-05,-0.0009851],[0.00031774,-0.00011394,-0.0015866],[-0.00033461,0.0015055,-0.00087142],[-6.3172e-05,0.00048508,-0.0014324],[-4.6002e-05,0.0004409,-0.0015564],[-0.00023268,0.00076036,-0.0015304],[0.00016968,0.0011505,-0.0012405],[-0.00013007,0.0015287,-0.00081153],[-0.00038471,0.0010634,-0.0012776],[-0.00013325,0.0010317,-0.0013069],[-0.00031997,0.0017631,-0.00028516],[-0.00055772,0.0012925,-0.0010838],[-0.00053331,0.0014456,-0.00078321],[-0.00032803,0.0015001,-0.00087158],[-0.00048542,0.0015685,-0.00059069],[-0.00040371,0.0016865,-0.00038105],[-6.1781e-05,0.0011008,-0.0012513],[-0.00030752,0.001751,-0.00010877],[-0.00033505,0.0017347,-3.56e-05],[-0.00046661,0.0016779,8.533e-05],[-0.00053428,0.0015995,0.00025411],[-0.00047591,0.001501,0.00043429],[6.6e-05,0.0011482,0.0013157],[-5.283e-05,0.0010722,0.0013141],[4.7096e-05,0.0005416,0.0017192],[-9.6071e-05,0.00043029,0.0010208],[0.00037758,0.00086301,0.0016432],[0.00011301,0.00057216,0.0017843],[6.1945e-05,0.00058794,0.0016911],[-8.1303e-05,0.00036659,0.0018021],[-0.00039334,0.00015966,0.0017909],[-0.00060441,-0.00030799,0.0017182],[-0.00079249,-0.00048387,0.0014686],[0.00023389,0.0014097,0.00073811],[-4.405e-05,0.0015333,0.00036478],[0.00056555,0.0014257,0.00078448],[0.00021451,0.0014147,0.0010229],[-0.00010946,0.0010111,0.00095041],[-0.00031519,0.00096388,0.0009658],[9.8921e-06,0.0012618,0.0013492],[-8.2258e-05,0.0018341,0.0013747],[-0.00089132,-0.0007211,-0.0014076],[-0.0011683,-0.0006501,-0.0012695],[-0.0016276,-0.00017427,-0.00094859],[-0.00074786,-0.00060264,-0.0014313],[0.00040573,-0.00024284,-0.0015972],[4.6094e-05,-0.00039108,-0.001699],[-0.00042743,-0.00058958,-0.0015421],[-0.00073088,-0.00056399,-0.0014292],[-0.000613,0.0012755,-0.00082545],[1.1879e-05,0.00083193,-0.0014963],[-5.1361e-05,0.0005078,-0.0018975],[-1.6204e-05,-0.0019471,-1.2018e-05],[-4.1483e-05,-0.002039,-8.2504e-05],[3.5052e-05,-0.001335,-7.5593e-05],[-1.6173e-05,-0.0012766,0.00013387],[0.00012229,-0.0014858,5.9109e-05],[-2.5632e-05,-0.0015706,-3.3972e-05],[-1.395e-05,-0.0015318,-3.5282e-05],[-0.00011472,-0.001498,-0.0011577],[1.5265e-05,0.00082141,-0.0021146],[-1.2126e-05,-0.0016434,-4.2336e-05],[2.2125e-05,-0.0017456,2.691e-05],[7.0169e-06,-0.0017906,3.7151e-05],[2.3153e-06,-0.0018747,4.5259e-05],[3.1162e-06,-0.0018015,0.00023124],[-0.00080802,0.0015639,0.0003199],[-0.00070945,0.0013422,0.0010454],[-0.0014817,0.00039709,0.0014158],[-0.00059002,-0.0016191,0.00037977],[-0.0011571,0.0010752,0.0013226],[-0.00060642,-0.0012753,0.0010821],[-0.00058426,-0.0010996,0.001173],[-0.00063399,-0.0013982,0.000808],[-0.00058022,-0.00074976,0.001205],[-0.00049246,-0.0016518,0.00055322],[-0.0005068,-0.0016213,0.00068874],[-0.00064774,-0.0014872,0.00073127],[-0.00050436,-0.0015834,0.00022007],[-0.00080653,-0.0013171,7.4178e-05],[-0.0015665,-0.00038186,-0.00025275],[-0.00074286,0.00050021,-0.00072807],[-0.0012185,0.0013585,2.6417e-05],[-0.001103,0.0012579,0.00085845],[-0.00060807,-0.00065548,0.0012516],[0.00016579,0.00082141,-0.0014359],[0.00023205,0.00047031,-0.0015739],[0.00038081,0.0001505,-0.001764],[0.00015508,-0.00020303,-0.0016925],[0.00017535,-9.7015e-06,-0.0015565],[5.3111e-05,0.00061403,-0.0013841],[-0.00016876,0.0016554,-0.00076533],[-3.8374e-05,0.00043603,-0.0015116],[-0.0001908,0.0007772,-0.0014702],[0.00011648,0.0010989,-0.0012679],[-0.00013255,0.0015337,-0.00081656],[-0.00013733,0.0010525,-0.0013258],[-1.8755e-05,0.0010284,-0.0013324],[-0.00035547,0.0017292,-0.00016753],[-0.00017792,0.0012952,-0.0011632],[-0.00036786,0.0014407,-0.00081279],[-0.00023831,0.0014996,-0.00089173],[-0.00043133,0.0015375,-0.0005875],[-0.00041066,0.0016525,-0.00037041],[-0.00023776,0.0011316,-0.0012422],[-0.00039009,0.0016964,-0.00011167],[-0.00048694,0.0016384,4.2421e-05],[-0.00055013,0.0015735,0.00021561],[-0.00039,0.0015808,0.00018978],[7.9469e-06,0.00124,0.0013476],[-2.9835e-05,0.0010547,0.0013237],[9.2672e-05,0.00050699,0.0017629],[-0.00027593,0.00057819,0.00094772],[0.00019007,0.0010493,0.001584],[-0.00012939,0.00081006,0.0017025],[-2.1087e-05,0.00057509,0.0017178],[-0.0003617,0.00028912,0.0017861],[-0.00052137,0.00012304,0.0018041],[-0.00081846,-0.00038248,0.0015981],[-0.0008149,-0.00048849,0.0014185],[0.00015508,0.0014944,0.00065196],[-0.00012499,0.0015467,0.00026466],[0.00042843,0.0015781,0.00062874],[-0.00012758,0.0019642,0.00082182],[-0.00044574,0.0014196,0.0010044],[-0.00029446,0.0011841,0.00080213],[0.0010648,0.00013494,0.0012336],[0.0011588,0.00040176,0.0012486],[-0.00096663,-0.00057585,-0.0013568],[-0.0012143,-0.00045158,-0.0011852],[-0.00093627,0.00066504,-0.0013407],[-0.0014436,-0.00010843,-0.00082318],[-0.00096873,-0.00063529,-0.0012975],[0.00027368,-0.00012911,-0.0016277],[2.7333e-05,-0.00037843,-0.0017207],[-0.0004533,-0.00057771,-0.0015613],[-0.00075414,-0.00053675,-0.0014234],[-0.00041008,0.0012582,-0.00078208],[-0.00051742,0.0013283,-0.00097022],[-0.00078395,0.0011996,-0.0012254],[-0.0013688,0.00012446,-0.0006017],[-0.00096195,0.00064661,-0.00099353],[-9.2145e-05,-0.0013795,-0.0001277],[-7.6945e-05,-0.001667,-6.6979e-05],[-9.1247e-05,-0.001685,-5.8122e-05],[-0.0001355,-0.0016763,-0.00010125],[-6.9324e-05,-0.0015108,-0.00018294],[-0.0011336,-0.0010561,-0.0014751],[-0.00091103,0.00050726,-0.0019926],[1.9173e-05,-0.0017101,-4.2697e-05],[1.1977e-06,-0.0017643,-1.7419e-05],[-4.3677e-05,-0.0019526,-5.8786e-05],[-9.5622e-05,-0.0017839,3.6847e-05],[-8.2175e-05,-0.0017547,2.1429e-05],[-0.00075933,0.0013197,0.0010146],[-0.00075474,0.0014364,0.0010288],[-0.0014449,0.00036838,0.0013423],[-0.0012201,0.0010333,0.0012684],[-0.0005651,-0.0015421,0.00035057],[-0.00075212,0.0014514,0.0010419],[-0.00059458,-0.0011653,0.0010211],[-0.00062938,-0.00091101,0.0010932],[-0.00073469,-0.0014197,0.00077377],[-0.00066265,-0.00077191,0.0012921],[-0.00051473,-0.0016739,0.0005714],[-0.00047945,-0.0016372,0.00039182],[-0.00065924,-0.0015049,0.00078],[-0.00072903,-0.0014073,0.00077387],[-0.00036599,-0.0016463,0.00032166],[-0.00059584,-0.001541,0.0002785],[-0.0012215,-0.00090103,0.00036182],[-0.0013493,-0.00018185,-0.00014202],[-0.00066252,0.00086222,0.00061926],[-0.00075652,0.0013696,0.0010249],[-0.0011012,0.0012116,0.00085374],[-0.00070243,-0.00050222,0.0011768],[0.00019991,0.00080147,-0.0014202],[0.00021354,0.00046661,-0.0015448],[0.00032953,0.00015278,-0.0016408],[0.00020619,-0.00023336,-0.0015797],[0.00030793,3.3593e-05,-0.0016292],[-0.00030823,0.00064359,-0.0012778],[-0.00024599,0.0015468,-0.00069424],[-3.3925e-05,0.00043656,-0.0015161],[-0.00019328,0.00078267,-0.0014802],[4.8075e-05,0.0011151,-0.0012641],[-0.00023126,0.0014637,-0.00075305],[-0.00025389,0.00090431,-0.0013697],[-7.2254e-05,0.00095337,-0.0013445],[-0.00037375,0.0017083,-0.00016144],[-0.00012742,0.0013183,-0.0011005],[-0.00033413,0.0014464,-0.00077113],[-0.00044825,0.0015027,-0.00060554],[-0.00036762,0.0013512,-0.00099226],[-0.0004804,0.001561,-0.00045544],[-0.00044888,0.0016493,-0.00025925],[-0.00018945,0.0011599,-0.0011877],[-0.00043693,0.0016717,-0.00010144],[-0.00048961,0.0016223,4.3212e-05],[-0.00039217,0.0015423,0.00016922],[-2.2313e-05,0.0012554,0.0013868],[-0.00024026,0.00042905,0.0017722],[0.00018399,0.001138,0.0018442],[-3.5785e-05,0.0011275,0.0017434],[-0.00016976,0.00081985,0.0017373],[-0.00015158,0.0007018,0.0016747],[-0.00051718,0.00043983,0.0017041],[-0.00069265,0.00030615,0.001682],[-0.00097142,-0.00018595,0.0014996],[-0.00088701,-0.00041822,0.0013994],[0.00031848,0.0015791,0.00019601],[0.00040522,0.0015122,0.00062283],[-7.2248e-05,0.0015422,0.0001141],[-0.00014333,0.0018432,0.00083189],[-0.0003738,0.001737,0.00080162],[0.0012051,0.0005826,0.0010301],[0.0009367,0.00011133,0.0013807],[0.0011092,0.00037984,0.0011263],[0.0011792,0.000585,0.0011455],[-0.0012495,-0.00058565,-0.0011008],[-0.0015345,-0.00039061,-0.0006202],[-0.0012757,0.00072872,-0.00039549],[-0.0015501,-6.7885e-05,-0.0005436],[-0.001036,-0.00054021,-0.0012782],[6.3583e-05,-0.00021266,-0.0016925],[-0.00043353,-0.00051098,-0.0015733],[-0.00077812,-0.000646,-0.001359],[-0.0010425,-0.00059014,-0.0012402],[-0.00059298,0.0013781,-0.00049842],[-0.00058998,0.0012942,-0.00095303],[-0.001113,0.000985,-0.0010507],[-0.001464,0.0002079,-0.00047471],[-0.0015807,9.2679e-05,-0.00060347],[-0.0003436,8.003e-05,-0.001727],[-0.00029005,0.00011945,-0.0018305],[-0.001717,0.00011664,-0.0006256],[-9.4897e-05,-0.0016498,-0.00012768],[-0.00015462,-0.0005842,-0.0018028],[-0.00012836,-0.0016376,-0.00020775],[-0.001186,0.0008979,-0.0011191],[-7.5393e-05,-0.001656,-0.00028422],[-7.0135e-05,-0.0018775,-1.8517e-05],[1.7576e-05,-0.0020092,-6.5093e-05],[-9.2863e-05,-0.0021116,7.9656e-05],[-0.00028813,-0.0017642,-1.0038e-05],[-0.00065359,0.0014466,0.00094113],[-0.0016086,0.00066409,0.00050505],[-0.0014266,0.0013068,0.00056891],[-0.001744,9.1646e-05,-0.00020781],[-0.00085213,0.0016847,0.00066694],[-0.00074958,-0.0011936,0.00098299],[-0.00060651,-0.00094061,0.0011456],[-0.00078535,-0.0013167,0.00078818],[-0.00055983,-0.0005742,0.0008643],[-0.00065727,-0.0016562,0.00048568],[-0.00061831,-0.001622,0.00030267],[-0.00088682,-0.00145,0.00063335],[-0.00098116,-0.0013731,0.00062296],[-0.00034549,-0.001616,1.9882e-06],[-0.00051599,-0.001321,0.00053993],[-0.0012129,-0.00087569,0.00046137],[-0.001209,0.00015714,0.00086247],[-0.00073485,0.0013496,0.00084291],[-0.00091057,0.0009887,0.00079809],[-0.00080668,-0.00054631,0.001048],[0.0001429,0.00074252,-0.0014303],[0.00024698,0.00047047,-0.0014007],[0.00022175,0.00010226,-0.0014768],[6.4338e-05,-0.00028177,-0.0015935],[0.00016109,0.00015673,-0.0016197],[3.8802e-05,0.0010458,-0.0010721],[-0.00013727,0.00033097,-0.0016118],[-0.00024716,0.0014529,-0.00066848],[-0.00018041,0.00078813,-0.0014616],[4.7175e-05,0.0011083,-0.0012575],[-0.00027088,0.0014461,-0.00078984],[-3.9668e-05,0.00089928,-0.0014172],[-8.7699e-05,0.00097155,-0.0013664],[-0.00010407,0.0012983,-0.0010885],[-0.00020744,0.001435,-0.00079151],[-0.00029797,0.001504,-0.00063695],[4.0438e-05,0.0013273,-0.0010683],[-0.0003586,0.0015751,-0.00048385],[-0.00037826,0.0016522,-0.00027273],[-0.00022493,0.0011707,-0.0011904],[-0.00043572,0.001638,-0.0001089],[-0.00047935,0.0015834,4.023e-05],[-0.00034181,0.0015822,1.6337e-05],[0.0010051,0.00022904,0.0014013],[-0.00027938,0.00046743,0.0017773],[0.00067061,0.00066754,0.0019839],[-3.1366e-05,0.0012238,0.0019069],[-0.00016575,0.00088411,0.0019243],[-0.00046086,0.00062263,0.0016591],[-0.00066329,0.0003922,0.0016641],[-0.00091993,0.00022941,0.0016078],[-0.00088747,-0.00015573,0.0015175],[-0.00093069,-0.00043527,0.0013023],[0.0004117,0.0015851,0.00031127],[0.00026831,0.0016422,0.00045285],[-0.00012182,0.0015845,5.6559e-05],[-5.0317e-05,0.0017973,0.00099782],[0.00090545,0.00059579,0.0014361],[0.0011043,0.00059701,0.00090678],[0.0010954,-0.00018421,0.0012264],[0.0013671,-3.5661e-05,0.00094359],[0.0014326,1.8124e-06,0.00087226],[-0.0012687,-0.00050065,-0.0010642],[-0.0015202,-0.00033406,-0.00059442],[-0.0007116,0.0012468,-0.00064413],[-0.0014541,0.00017624,-0.00071391],[-0.0015188,-0.00025152,-0.00060122],[-0.0012755,-0.00055172,-0.0010256],[2.8002e-05,-0.0001822,-0.0017293],[-0.00052068,-0.00043164,-0.0015791],[-0.00082492,-0.00059772,-0.0013527],[-0.0010592,-0.00051422,-0.0011926],[-0.00042618,0.001466,-0.00053428],[-0.0006543,0.0013294,-0.00080399],[-0.0010947,0.00097251,-0.00091941],[-0.0014361,0.00061445,-0.00042777],[-0.0014275,0.00057201,-0.00051819],[-0.00028053,0.00046013,-0.0017077],[-0.00059214,0.00033346,-0.0017637],[-0.0015664,0.00022627,-0.00076587],[-0.0014605,0.00017365,-0.00080202],[-0.0012868,0.00040637,-0.0011643],[-0.00055306,0.00038166,-0.0018767],[-0.0011555,0.00085286,-0.0010644],[-0.0002825,-0.0017406,-0.00031918],[-0.00018813,-0.0015212,-0.00037856],[-0.00050315,-0.0010221,-0.0011364],[-0.00015919,-0.0014424,6.3043e-05],[-0.0001969,-0.0014612,0.00010629],[-0.00069134,0.0016285,0.0005021],[-0.0014208,0.001025,0.00036316],[-0.00119,0.0014458,0.00042432],[-0.0013824,5.2007e-05,-8.5948e-05],[-0.00066317,0.0016815,0.00053861],[-0.00068599,-0.001156,0.00092071],[-0.00063383,-0.00064068,0.00099917],[-0.00088535,-0.0013449,0.000767],[-0.0006218,-0.00047594,0.00079452],[-0.00064271,-0.0016713,0.00031483],[-0.0009012,-0.0014797,0.00062357],[-0.00059075,-0.001643,2.0129e-05],[-0.00099132,-0.0013667,0.00069349],[-0.00090283,-0.0014318,0.00057939],[-0.00082719,-0.0011841,0.00028808],[-0.00042846,-0.0016672,-5.1815e-05],[-0.00079866,-0.0010357,0.00074472],[-0.0014769,0.00027485,0.00084632],[-0.00088532,0.0011181,0.00093039],[-0.0014875,0.00042895,0.0011942],[-0.00088339,-0.00061823,0.0011699],[0.00015961,0.00073855,-0.0014319],[0.00021184,0.00046308,-0.0013519],[0.00011694,0.0001002,-0.0010656],[-3.1989e-05,-0.00018689,-0.0015365],[0.00011165,0.0001438,-0.0016802],[-0.00037472,0.0011032,-0.00099824],[3.7124e-06,0.00033026,-0.0016935],[-0.00032231,0.0014387,-0.00063899],[-7.8028e-05,0.00078458,-0.001483],[-0.00011055,0.0011299,-0.0012326],[-0.00032714,0.0013981,-0.00074323],[-9.9699e-05,0.00082667,-0.0014424],[-2.9795e-05,0.0010302,-0.0013191],[-4.3083e-07,0.0013746,-0.00097648],[-0.00022126,0.0014104,-0.000802],[-0.00028063,0.0014955,-0.00061315],[-0.00035854,0.0015401,-0.00048299],[-0.00018584,0.0010921,-0.0012478],[-0.00042031,0.0016023,-0.00033017],[-0.00028354,0.0011014,-0.0012309],[-0.00042957,0.0016132,-0.00010717],[-0.00036454,0.0015718,-1.6159e-05],[0.00043805,0.00014957,0.0017873],[-0.00022118,0.00048677,0.0017998],[-0.00029887,0.00054154,0.001999],[-2.3056e-05,0.0011711,0.001823],[-0.00063721,0.00076045,0.0018314],[-0.00023199,0.00040202,0.0017385],[-0.00050358,0.00023527,0.0016895],[-0.00063333,-5.6685e-05,0.001671],[-0.00090205,-0.00011244,0.0014631],[-0.00096068,-0.0004571,0.0013545],[0.00030353,0.0015985,0.00032845],[0.00052918,0.0015388,-6.3068e-05],[0.00017271,0.0018001,0.00050974],[-7.6072e-05,0.0015811,-6.8139e-05],[0.0010745,0.0014698,0.00077813],[0.0010252,0.001496,0.00095467],[0.0013331,0.00030271,0.00083008],[0.0011495,-8.6984e-05,0.0012273],[0.00092535,-0.00021054,0.0013536],[0.0014148,-6.4629e-06,0.00096],[0.001462,0.00032391,0.00069473],[-0.0015516,-0.00044655,-0.00051327],[-0.0015425,-0.00032813,-0.00054682],[-0.00088089,0.0013127,-0.00029321],[-0.00159,0.00021367,-0.00045526],[-0.0016062,-0.00023863,-0.00048054],[-0.001265,-0.00051619,-0.00099915],[-0.00048775,-0.00033995,-0.0015756],[-0.00078457,-0.00048863,-0.0013757],[-0.0010816,-0.0006342,-0.001114],[-0.0012752,-0.00053439,-0.00099473],[-0.00050289,0.0014811,-0.00035419],[-0.0008695,0.0011837,-0.00070968],[-0.0010748,0.00089494,-0.0008477],[-0.00097218,6.859e-05,-0.0012767],[-0.0013982,0.00059401,-0.00041173],[-0.0013444,0.00065368,-0.00061698],[-0.00049309,0.00064138,-0.0015057],[-0.00048436,0.0006847,-0.0015881],[-0.0013819,0.00066436,-0.00066013],[-0.00071219,0.00048864,-0.0019439],[-0.0011817,0.00089917,-0.00093114],[-0.00088837,0.0004715,-0.0020365],[-0.0009784,-0.0013094,-0.00038054],[-0.0013703,-0.00051349,-0.00076181],[-0.00108,-0.00034233,-0.00087657],[-0.0013592,-0.00016367,-0.00039587],[-0.00099301,0.001275,0.0006971],[-0.0014461,0.0010645,-0.000196],[-0.0012351,0.0015431,-0.00016722],[-0.0010917,0.00051403,-0.00026331],[-0.00067413,0.0018845,-6.2656e-05],[-0.00087323,-0.0011685,0.0008641],[-0.00083712,-0.00063155,0.00096177],[-0.00093532,-0.0012837,0.0007924],[-0.00076953,-0.00059254,0.00088634],[-0.0007579,-0.0016288,0.00022452],[-0.00095166,-0.0014626,0.0005879],[-0.00061421,-0.0016437,3.4309e-06],[-0.0011379,-0.0012747,0.00055326],[-0.00098951,-0.0014272,0.00053297],[-0.00087779,-0.0013619,-9.1369e-05],[-0.00039096,-0.001587,-0.0001792],[-0.00082893,-0.0010783,0.00074039],[-0.00086576,0.001159,0.00064692],[-0.0013589,0.00041553,0.0010164],[-0.0009692,-0.000669,0.0011578],[0.00016341,0.00070458,-0.0013324],[9.3675e-05,0.00026922,-0.00093902],[2.5637e-05,8.6262e-05,-0.0012689],[5.7748e-05,-0.00016412,-0.0015631],[0.00014682,0.0001054,-0.0016593],[6.1285e-05,0.00040838,-0.0017366],[-0.00020705,0.0011909,-0.0008337],[-0.00016248,0.00069214,-0.0015298],[-0.00034472,0.0014118,-0.0006541],[-0.00019184,0.0010426,-0.0012814],[-0.00043279,0.001361,-0.00083725],[7.7482e-05,0.0008263,-0.0014882],[-0.000256,0.001056,-0.0012982],[-6.9287e-05,0.0013706,-0.00095834],[-0.00012821,0.0014056,-0.00081962],[-0.00020756,0.0014867,-0.00062472],[-0.00031117,0.0015294,-0.00048859],[6.0357e-06,0.0010775,-0.001281],[-0.00037698,0.001578,-0.00033205],[2.3055e-05,0.0010764,-0.0012763],[-0.00044555,0.0015706,-0.00013894],[-0.00030893,0.0015707,-0.00016416],[0.00082784,-0.00052096,0.0014055],[-0.00052422,0.00074194,0.0015596],[0.00065005,-0.00061786,0.0015405],[0.00087937,-0.00025801,0.001277],[-0.00029562,0.00040673,0.0011954],[-0.00049117,0.00034069,0.0017307],[-0.00058545,0.00022239,0.0017245],[-0.00094885,-0.0001701,0.0015184],[-0.00095543,-0.00013114,0.0014492],[-0.0011101,-0.00052844,0.0011522],[0.00027016,0.0016073,0.00028431],[0.00058171,0.001589,-2.3283e-05],[0.00021748,0.0015337,0.00056042],[-3.652e-05,0.0016262,-1.6309e-05],[0.00073824,0.0013243,0.00038794],[0.0013233,0.00059767,0.00062218],[0.0013359,0.00060463,0.0006362],[0.0012192,-0.00013806,0.0012471],[0.0010994,-0.00043648,0.0012988],[0.0012997,0.000309,0.0011442],[0.0014326,0.00050029,0.00078379],[-0.0015338,-0.00043394,-0.00050454],[-0.0015186,-0.00036031,-0.00055207],[-0.0012553,0.00062376,-0.00071668],[-0.00057536,0.0015268,-0.00041785],[-0.0015424,-0.00011459,-0.00055719],[-0.0015146,-0.00043913,-0.00052669],[-0.0015397,-0.0004773,-0.00047046],[-0.0006419,-0.00017454,-0.0015406],[-0.0008922,-0.00037461,-0.0013558],[-0.0011063,-0.0005996,-0.0011028],[-0.0013106,-0.00042859,-0.00094929],[-0.00045777,0.0015548,-0.0003815],[-0.000829,0.0011382,-0.00073626],[-0.0010953,0.00092938,-0.00066725],[-0.00081168,0.00072202,-0.0011614],[-0.0012051,0.00090686,-0.00032732],[-0.0011862,0.00078224,-0.00053429],[-0.00050352,0.00054492,-0.0014941],[-0.00057703,0.00077049,-0.0015416],[-0.00053278,0.00082556,-0.0016265],[-0.0010543,0.00057498,-0.0021193],[-0.001113,0.00044971,-0.0018122],[-0.0013049,0.00060229,-0.0010044],[-0.0013422,-0.00010006,-0.00051635],[-0.0011143,0.0017537,2.5173e-05],[-0.0013176,0.0012071,-0.00029451],[-0.0010021,0.0016254,-0.00027964],[-0.00113,0.00049649,-0.00045482],[-0.0009373,0.0017116,9.2906e-05],[-0.0010623,-0.00086745,0.00095533],[-0.001149,-0.001224,0.000659],[-0.00092635,-0.00073762,0.0010969],[-0.00094955,-0.0015426,0.00031881],[-0.0007345,-0.0016387,5.0715e-05],[-0.0011557,-0.0012499,0.00070216],[-0.0005651,-0.001604,-0.00021336],[-0.00098862,-0.0014458,0.00046953],[-0.0011569,-0.0012426,0.00062762],[-0.0013893,-0.00065369,0.00017252],[-0.00082175,-0.0014757,-8.4537e-07],[-0.00046826,-0.0016125,-0.00024602],[-0.0014116,0.0004018,0.00099447],[-0.0011447,-0.00062359,0.0011702],[-9.3145e-05,0.00076136,-0.0013245],[-0.00010281,0.00032135,-0.0010414],[-0.00015219,0.00030228,-0.0012281],[-0.00031252,0.00024929,-0.0015016],[-0.0006163,-0.00011527,-0.0015321],[0.00020641,0.00041249,-0.0018153],[-0.00038882,0.0011873,-0.00077095],[0.00022276,0.000675,-0.001602],[-0.00046764,0.0013439,-0.0005763],[-0.00027516,0.0010548,-0.0012723],[-0.00051843,0.0012927,-0.00075713],[0.00012115,0.00087148,-0.0014507],[-0.00025663,0.0010433,-0.0012887],[-0.00012344,0.001404,-0.00081192],[-0.00013183,0.0013014,-0.0010112],[-0.00020862,0.0014754,-0.00062349],[-0.00027818,0.0015308,-0.0004464],[-0.00037075,0.0015473,-0.00032699],[1.7605e-05,0.0010785,-0.0012583],[-9.7411e-05,0.00093341,-0.001352],[-0.00028684,0.0015693,-0.000136],[0.0006691,-0.0005468,0.0015371],[-0.00053904,0.00074411,0.001568],[0.00081344,-0.00054189,0.0013248],[-0.00022167,-0.00017867,0.0015557],[-0.00015175,-0.00010406,0.0012209],[-0.00040568,0.00032453,0.0010448],[-0.00060574,0.00046135,0.0015803],[-0.00077907,0.00045334,0.001527],[-0.0011689,0.00010861,0.0013299],[-0.0011996,0.00015524,0.001273],[-0.0013825,-0.00030656,0.0010197],[0.0003814,0.0016067,9.1716e-06],[0.00035589,0.001572,0.00026796],[0.00060254,0.0015178,-0.00013537],[0.00066568,0.0012774,0.00048031],[3.4688e-05,0.0015776,-0.00019976],[0.0013475,0.00074647,0.00028168],[0.0013071,0.00077574,0.00048549],[0.0014303,0.00056107,0.0006396],[0.0011898,-0.00013581,0.0012235],[0.001353,0.00031954,0.0011169],[0.0010565,-0.0004109,0.0012179],[0.0013189,0.00049679,0.001043],[0.0014511,0.0005843,0.00073954],[-0.0015283,-0.0004309,-0.00049557],[-0.0015151,-0.00036731,-0.00058865],[-0.0013882,0.00066335,-0.0002787],[-0.00065895,0.0016278,-0.00032369],[-0.0016631,-0.00011623,-0.00054261],[-0.001557,-0.00045475,-0.0005551],[-0.0014983,-0.0004923,-0.00046864],[-0.00084354,-0.00023546,-0.0013867],[-0.0010227,-0.00040314,-0.0011508],[-0.0013143,-0.00061403,-0.00079549],[-0.0015033,-0.00040502,-0.00054513],[-0.00047536,0.0014767,-0.0002399],[-0.00094832,0.00089697,-0.00057388],[-0.000558,0.00047925,-0.0013677],[-0.0009638,0.00057341,-0.0009764],[-0.0011969,0.00083595,-0.0002744],[-0.00065023,0.00068974,-0.0013493],[-0.00054707,0.0007714,-0.0014759],[-0.00068968,0.00098836,-0.0016082],[-0.00068566,0.0010357,-0.001681],[-0.0013129,0.00071045,-0.0011651],[-0.0013148,0.00070408,-0.00065276],[-0.0021998,0.00063863,0.00069952],[-0.0012721,0.001149,-0.00061377],[-0.00088431,0.0014943,-0.0006765],[-0.00076511,0.0015473,-0.00057775],[-0.0011604,-0.00087009,0.00094655],[-0.0012889,-0.0010426,0.0007359],[-0.0012602,-0.00099717,0.0010326],[-0.0013062,-0.0012984,-2.7131e-05],[-0.0009005,-0.0015418,-0.00010265],[-0.0013924,-0.0011057,0.00048651],[-0.00060053,-0.0016006,-0.00024417],[-0.0011096,-0.0013914,0.00036853],[-0.0012931,-0.0011416,0.000491],[-0.0014697,-0.00054359,0.00071255],[-0.00073174,-0.0013364,-0.00012222],[-0.00045047,-0.001581,-0.00028856],[-0.0015231,0.00058898,0.0007241],[-0.0010905,-0.00059232,0.0010221],[-0.00020788,0.00057496,-0.0013186],[-0.00025279,0.00034362,-0.0014159],[-0.00050821,0.00023123,-0.0015834],[-0.00063133,-0.00011134,-0.0015553],[5.465e-05,0.00023916,-0.0017589],[0.00012181,0.00055896,-0.0016515],[-0.0003877,0.0011777,-0.00076668],[-0.00053621,0.0013228,-0.00062671],[-0.00048667,0.00079142,-0.001386],[-0.00076817,0.0011054,-0.00091601],[-0.00018419,0.00090878,-0.0014387],[-0.00014098,0.0010278,-0.001297],[1.8586e-05,0.0014004,-0.00084163],[-4.1462e-05,0.0012998,-0.0010314],[-2.9876e-06,0.0014711,-0.00066726],[-0.00011545,0.0015241,-0.00047802],[-0.00019961,0.0015405,-0.00035963],[2.3935e-05,0.0010886,-0.0012715],[0.00012824,0.00092634,-0.0014011],[0.0009808,-0.00096304,0.0014484],[-0.00049482,0.00067666,0.0013905],[0.0011344,-0.00094845,0.00099243],[0.00058384,-0.0016556,0.00069898],[-1.0469e-05,-0.00011948,0.0013699],[-4.47e-05,-0.00020425,0.0015518],[-0.00079522,0.00041353,0.0015392],[-0.0011233,0.00033185,0.0013407],[-0.0012056,8.2923e-05,0.0012596],[-0.0013546,7.5754e-05,0.0010865],[-0.0012708,-0.00026468,0.0010415],[0.00036874,0.0016266,-1.1429e-05],[0.00033027,0.0015044,0.00024276],[0.00060257,0.0015599,-0.00015931],[0.00051042,0.0013699,0.00024263],[7.1782e-05,0.0016183,-0.00015618],[0.0013671,0.00084183,0.00021518],[0.0013976,0.00069208,0.00046897],[0.0014307,0.00071885,0.0005195],[0.00098552,0.00023155,0.0014572],[0.0011887,0.00060892,0.0012601],[0.00095712,-0.00016181,0.0014826],[0.0012336,0.00063979,0.0010979],[0.0013053,0.00078114,0.00079523],[-0.0014807,-0.00046624,-0.00049824],[-0.0014669,-0.00043189,-0.00059914],[-0.0010738,0.0013162,-0.00073898],[-0.0014474,-6.045e-05,-0.00050388],[-0.00065028,0.0018128,-0.00037254],[-0.001552,-0.00039734,-0.00060081],[-0.0014312,-0.00069711,-0.00060092],[-0.0014645,-0.00048428,-0.00047241],[-0.00087358,-0.00018989,-0.0013625],[-0.0010798,-0.00029747,-0.0010955],[-0.0013326,-0.00057349,-0.00077909],[-0.0014097,-0.00056687,-0.00059683],[-0.0010682,0.0010067,-0.00012339],[-0.0005903,0.00043902,-0.00142],[-0.00097144,0.001012,-0.001057],[-0.001471,0.0015403,-0.00029363],[-0.00073601,0.00041122,-0.0014261],[-0.00075677,0.00099058,-0.0012689],[-0.00053596,0.0010702,-0.0013746],[-0.0010246,0.0013401,-0.0010745],[-0.00099277,0.0012609,-0.0010059],[-0.0011854,0.0007861,-0.00040195],[-0.0010639,0.0013346,-0.00079779],[-0.00090314,0.0015157,-0.00068447],[-0.0013157,0.0016487,-0.00046483],[-0.0013285,-0.0010388,0.00072318],[-0.001427,-0.0010387,0.0010728],[-0.0013757,-0.0012291,1.1075e-05],[-0.0012998,-0.0012625,8.4089e-05],[-0.00088982,-0.0015414,-0.00015505],[-0.0011141,-0.0014118,0.00032081],[-0.00059462,-0.0016829,-0.00042608],[-0.0012749,-0.0011281,0.00047517],[-0.0012896,-0.0010691,0.00049565],[-0.0016429,-0.00042807,0.0005742],[-0.00074481,-0.0013261,-0.00013617],[-0.00056622,-0.0015863,-0.00040066],[-0.00060009,-0.0005865,0.00047824],[-0.0012563,-0.0005017,0.00096754],[-0.00038919,0.00061384,-0.0013745],[-0.00042002,0.00059792,-0.0014211],[-0.00062678,0.00039575,-0.0015061],[-0.00091779,-0.00019843,-0.0014322],[-0.00015707,0.00025839,-0.0018066],[-2.7083e-05,0.00058943,-0.0016929],[-0.00046228,0.0012124,-0.00076917],[-0.00063619,0.0012681,-0.00055577],[-0.00058757,0.00076576,-0.0013168],[-0.00094572,0.0010533,-0.00080068],[-0.0001769,0.00090197,-0.001416],[-0.00026818,0.00087973,-0.0013794],[-7.4992e-05,0.0013336,-0.00094694],[-3.6774e-05,0.0014506,-0.00070746],[-8.1983e-05,0.0012628,-0.001071],[-9.1613e-05,0.0015373,-0.00044851],[-0.0001741,0.0015474,-0.00032657],[3.7248e-05,0.0011074,-0.0012659],[-0.00012645,0.00062372,-0.0015795],[0.00033335,-0.00040512,0.00065669],[-0.00077926,0.00064039,0.0013601],[0.0010022,-0.00097521,0.0010273],[0.001221,-0.00056839,0.0010644],[0.00078184,-0.001656,0.00065973],[0.00052866,-0.0016713,0.00045677],[-0.00061809,-0.0003558,0.0015276],[-0.00082791,0.0004775,0.0013829],[-0.0010713,0.0003026,0.0013094],[-0.0012013,7.7901e-05,0.0012628],[-0.0013288,1.2823e-05,0.0011552],[-0.0014058,-0.00015832,0.00097772],[0.00040613,0.0015941,-0.00012242],[0.0004496,0.0015304,-2.8282e-05],[0.00053654,0.0013914,0.00020387],[0.00062475,0.0014957,-0.00026026],[0.0013378,0.00079165,9.8265e-05],[0.00016765,0.0015615,-0.00040152],[0.0013662,0.00076739,0.00019778],[0.001361,0.00081354,0.00034133],[0.0013118,0.00088186,0.00056095],[0.0010661,0.00057805,0.0013351],[0.0010481,0.00024598,0.0012968],[0.0011919,0.00065296,0.0012351],[0.0011543,-8.3419e-05,0.0010977],[0.0012542,0.00081966,0.00099093],[0.0012869,0.00091187,0.0006698],[-0.0014527,-0.0004728,-0.00055339],[-0.0013668,-0.00045607,-0.00077315],[-0.00017483,0.00048671,-0.0015569],[-0.00031117,-6.0853e-05,-0.0014874],[0.00011293,0.0010983,-0.0013821],[-0.00040337,-0.00046942,-0.0016883],[-0.00037356,-0.00076741,-0.0017018],[-0.0014386,-0.00049309,-0.00047077],[-0.0010962,-0.00027422,-0.00113],[-0.00097371,-0.00028138,-0.00070379],[-0.0015056,-0.00059485,-0.00064636],[-0.0014601,-0.00054874,-0.0004362],[-0.00044535,0.00045972,-0.0014287],[-0.00058858,0.00043481,-0.0014434],[-0.0010807,0.0011749,-0.0012335],[-0.00086542,0.00053882,-0.0013803],[-0.00094168,0.00057038,-0.0014642],[-0.00083892,0.0014078,-0.0008666],[-0.00080477,0.0014035,-0.00087067],[-0.0014274,7.6917e-05,-1.6248e-05],[-0.0012763,0.00080516,-0.00063121],[-0.00099233,0.0011843,-0.001006],[-0.0012858,-0.0010094,0.0007],[-0.0012109,-0.0013171,0.00016206],[-0.0012643,-0.0012509,9.6441e-05],[-0.0010026,-0.0013758,-0.00030096],[-0.0011308,-0.0013975,0.00030257],[-0.00089073,-0.0016819,-0.00070002],[-0.0013313,-0.0010818,0.00041632],[-0.0013998,-0.00098888,0.00039197],[-0.0016027,-0.00051923,-0.00040148],[-0.00062311,-0.0012635,-0.00049582],[-0.00059936,-0.0016155,-0.00033604],[-0.00072026,-0.00049664,0.00038007],[-0.0012181,-0.00050747,0.00075735],[-0.00050166,0.0006167,-0.0014632],[-0.0010969,0.00029544,-0.0012915],[-0.0011263,0.00012934,-0.0012376],[-0.00011927,0.00026794,-0.0016592],[-0.00019645,0.00035749,-0.0016903],[-0.00057121,0.0010987,-0.00082849],[-0.00091165,0.0010388,-0.0007007],[-0.00059646,0.00077815,-0.0013372],[-0.0010906,0.00082507,-0.0008464],[-0.00027832,0.00090756,-0.0013991],[-0.00025128,0.00089143,-0.0014039],[0.0001294,0.0013317,-0.00099489],[0.00027819,0.0014388,-0.00077292],[2.5714e-05,0.0012602,-0.0010942],[0.00027649,0.0015364,-0.00052896],[-4.1604e-05,0.0011269,-0.0012688],[0.0001063,0.00060492,-0.0016242],[0.00037001,-0.00045836,0.00063693],[-0.00045335,0.00030597,0.0014592],[0.0012306,-0.0013572,0.00044002],[0.0014608,-0.0009395,0.00058584],[0.00075422,-0.0015765,0.00050702],[0.00018364,-0.0015784,0.00051107],[-0.00042349,-0.00038725,0.0013197],[-0.0010455,0.00042734,0.0013128],[-0.0011655,0.00027741,0.0012759],[-0.0013081,2.1583e-05,0.0011394],[-0.0013917,-4.3254e-05,0.00099616],[-0.0014475,-0.0002207,0.00077493],[0.00038384,0.0016459,-0.00016825],[0.00037781,0.0015422,-0.00011018],[0.00045839,0.0013602,0.00012104],[0.00063368,0.0015606,-0.00029462],[0.0013245,0.00086787,1.8184e-05],[0.00020851,0.0015963,-0.00035713],[0.0013676,0.00077267,0.00019392],[0.0013236,0.00088952,0.00035931],[0.001273,0.00091722,0.00049221],[0.00090362,0.00071787,0.0013467],[0.00076333,0.00053443,0.0013474],[0.001077,0.00079888,0.0012781],[0.0011719,-1.9516e-05,0.0012064],[0.001225,0.00084476,0.00099322],[0.0014398,0.00067183,0.00060583],[-0.0013899,-0.00054703,-0.00056606],[-0.001217,-0.00070989,-0.00081516],[-0.00014896,0.00052786,-0.0016748],[-0.00031102,-0.00014024,-0.0017368],[-0.00098807,0.00086935,-0.0011299],[-0.00033935,-0.00061223,-0.0017803],[-0.00027549,-0.00087866,-0.001676],[-0.0013293,-0.00048639,-0.00057536],[-0.0012187,-3.6682e-05,-0.0009873],[-0.0010718,-0.00010066,-0.00062343],[-0.0015782,-0.00031011,-0.00051967],[-0.0015079,-0.00039009,-0.0003779],[-0.00053374,0.00013851,-0.0014797],[-0.0007034,-2.9164e-06,-0.0014952],[-0.0010484,-3.3385e-05,-0.0014686],[-0.0013096,0.0010382,-0.00074539],[-0.0014264,7.0238e-05,-0.00032965],[-0.0014418,0.00016055,-0.00028513],[-0.0014186,-0.00095342,0.00062293],[-0.0011245,-0.0014475,0.00010706],[-0.0011982,-0.0013338,5.8403e-05],[-0.0012006,-0.0012744,-0.00023502],[-0.0011515,-0.0016691,-0.00068369],[-0.0013409,-0.0010839,0.00043887],[-0.00097661,-0.0016302,-0.00040583],[-0.0013834,-0.00095747,0.00045672],[-0.001332,-0.00096188,0.00036982],[-0.0014486,-0.00072569,-0.00014053],[-0.0005073,-0.0015191,-0.00032275],[-0.00070176,-0.0016715,-0.00043209],[-0.001183,-0.00086899,0.00051682],[-0.0013669,-0.00041066,0.00067596],[-0.0011607,0.0004906,-0.0010721],[-0.0013757,7.6691e-05,-0.0011617],[-0.00029542,0.00028418,-0.0017307],[-0.00028118,0.00037012,-0.0017262],[-0.00056038,0.0012045,-0.00092569],[-0.0013723,0.001268,-0.00076401],[-0.0010627,0.00068687,-0.0010693],[-0.0011167,0.00058663,-0.00050514],[-0.00042282,0.00070455,-0.0014605],[-0.00027535,0.00086181,-0.0014197],[2.8892e-05,0.0012525,-0.0010988],[0.0001171,0.0013444,-0.00096425],[0.00021355,0.0015276,-0.00061971],[-7.5698e-05,0.0011879,-0.0012058],[-0.00011437,0.0010732,-0.0013542],[0.00013932,0.00065137,-0.0016181],[0.0016072,-0.00027143,0.00076325],[-0.0010921,0.00017484,0.0012655],[0.0014904,-0.0010299,0.00065499],[0.00086835,-0.0015558,0.00060924],[0.00036465,-0.001648,0.0006073],[0.00012451,-0.0017231,0.000474],[0.00022141,-0.0016562,0.00043479],[-0.0010353,-0.00070426,0.00079121],[-0.0010277,0.00043431,0.001254],[-0.0011559,0.00027692,0.001262],[-0.0012851,-3.6889e-05,0.0012022],[-0.0013519,-0.00014126,0.0010986],[-0.0013894,-0.00032663,0.00087952],[0.00040481,0.0015644,-0.00016868],[0.00041494,0.0016278,-0.00025387],[0.00057622,0.0014052,-0.00014768],[0.001288,0.00082587,-3.5002e-05],[0.0006762,0.001488,-0.0004351],[0.0013681,0.00069561,-9.8848e-06],[0.0014162,0.0006644,0.00017287],[0.001316,0.00074414,0.00048511],[0.0014584,0.00066779,0.00043627],[0.00098562,0.00081402,0.0014486],[0.00084268,0.00070565,0.0014134],[0.0012314,0.00069009,0.00089029],[0.0011442,0.00093298,0.0012918],[0.001747,0.00018425,0.0008384],[0.0011711,0.00073392,0.0010214],[0.001377,0.00067682,0.00054684],[-0.0011685,-0.00058123,-0.00088917],[-0.00024227,-0.00073935,-0.0017196],[0.00010723,0.00014994,-0.0015697],[4.9859e-05,-8.2971e-05,-0.001701],[-0.00032635,0.00018085,-0.0015561],[0.00015191,-0.00046293,-0.0017118],[0.00015437,-0.00073698,-0.0016879],[-0.0013306,-0.00049934,-0.00058128],[-0.0019395,-0.00012446,-0.00086549],[-0.0017045,-0.00032629,-0.00051989],[-0.0013181,-0.00039302,-0.00066735],[-0.00044851,4.2965e-05,-0.0015373],[-0.00076057,1.5801e-05,-0.0015711],[-0.001294,2.7337e-05,-0.0016847],[-0.0014545,0.00018955,-0.00095887],[-0.0014795,-3.4513e-05,-0.00027326],[-0.0012834,-0.0012842,-8.8271e-05],[-0.0012802,-0.001202,-7.4174e-05],[-0.0011749,-0.0011519,-0.00027959],[-0.0008261,-0.0011317,-0.00049755],[-0.0014877,-0.00091462,0.00024819],[-0.0012359,-0.0013881,-0.00072721],[-0.0014602,-0.00085342,0.00034371],[-0.0012819,-0.0010429,0.0004511],[-0.0013484,-0.00075326,-0.00068869],[-0.00044535,-0.0015283,-0.00047071],[-0.00047483,-0.0017863,-0.0010584],[-0.0013521,-0.00074303,0.00037411],[-0.0014815,-0.00050844,0.00048893],[-0.00129,0.00038562,-0.00080393],[-0.0014375,0.00037869,-0.00087674],[-0.00024199,0.00032839,-0.0016468],[-0.00030238,0.00031193,-0.0016703],[-0.001189,0.00071144,-0.0013661],[-0.0010386,0.00045369,-0.00060555],[-0.0012892,0.00057919,-0.00055736],[-0.0001948,0.00068996,-0.0014953],[3.8848e-05,0.00085024,-0.0014872],[0.00053002,0.0011596,-0.0011375],[0.00078072,0.001193,-0.0010259],[0.00078537,0.0014112,-0.00069952],[0.00025443,0.0011405,-0.0012422],[-1.361e-05,0.00106,-0.0013649],[-4.2284e-05,0.00067471,-0.0016074],[0.0017779,-0.00024274,0.00095681],[-0.0014074,0.00031513,0.0014712],[0.0016323,-0.0010933,0.00082311],[0.00080338,-0.0014653,0.00082221],[0.00026132,-0.0015183,0.00085608],[-0.00013324,-0.0014742,0.0004642],[-9.2213e-05,-0.0016026,0.00046759],[-0.0002846,-0.00063422,0.00041038],[-0.001106,0.00042534,0.0012471],[-0.001229,0.00025283,0.0012214],[-0.0013346,-7.4626e-05,0.0011012],[-0.0014093,-0.00019896,0.00092633],[-0.0015008,-0.00044622,0.00054135],[0.00038671,0.0017189,-0.00024474],[0.00038971,0.0017357,-0.00033446],[0.00055759,0.0013865,-0.00015576],[0.0012307,0.00094074,-0.00018787],[0.00070641,0.0015633,-0.0004628],[0.001354,0.00073506,-5.8551e-05],[0.0014221,0.00058005,0.00025964],[0.0014236,0.00059988,0.00046391],[0.0014458,0.00061237,0.00047978],[0.00086188,0.00090472,0.0014421],[0.00089499,0.00060335,0.0013508],[0.0012132,0.0007071,0.00089452],[0.0011177,0.00095067,0.0012891],[0.0015351,0.00058224,0.0011903],[0.0013735,0.0003814,0.00091672],[0.0015556,0.00029705,0.00043789],[-0.001053,-0.00070984,-0.00089264],[-7.2744e-05,-0.00094197,-0.0016897],[-0.0003224,8.0116e-05,-0.0014583],[0.0001944,-0.00025753,-0.0017046],[0.00020018,-0.0005207,-0.0017304],[0.00014013,-0.00072625,-0.0017046],[-0.00096591,-0.00048359,-0.0010042],[-0.0018036,-6.5467e-05,-0.00076911],[-0.0016634,-0.00014931,-0.00042802],[-0.0013238,-0.00048898,-0.00071673],[-0.00048553,-0.0002075,-0.0014649],[-0.00079824,-0.00038646,-0.0014031],[-0.0012078,-0.00040866,-0.0012617],[-0.0019469,-8.171e-05,-0.00097204],[-0.0014354,-3.4118e-05,-0.00070062],[-0.0014835,-0.00096759,4.5323e-05],[-0.001261,-0.0012643,-9.545e-05],[-0.0011777,-0.0011659,-0.00028423],[-0.0010069,-0.0010321,-0.00045237],[-0.0008147,-0.0010902,-0.00056099],[-0.0014739,-0.0008879,0.00025004],[-0.0011468,-0.0015295,-0.00078298],[-0.00072349,-0.00097232,-0.00079678],[-0.0013151,-0.0011262,0.00023897],[-0.0014137,-0.00072898,0.00056466],[-0.0012901,-0.0011396,-0.0003387],[-0.00076885,-0.0014507,-0.00092955],[-0.00039588,-0.0011234,-0.00080821],[-0.001469,-0.00052509,0.00050542],[-0.0014778,0.00035966,-0.00079701],[-0.00031176,0.00034386,-0.0017191],[-0.00032284,0.00031611,-0.0016885],[-0.00045146,0.00070428,-0.0016631],[-0.0009608,0.00056574,-0.00066297],[-0.00098854,0.00057938,-0.00068577],[-0.00037263,0.00043183,-0.0015782],[-0.00022751,0.00047953,-0.0016306],[0.00059908,0.0011045,-0.0012804],[0.00032649,0.0010256,-0.0013874],[0.00073273,0.0014576,-0.00085872],[3.8527e-05,0.00097162,-0.0014695],[-0.00015834,0.00092531,-0.0015016],[-0.00011672,0.00057816,-0.001655],[0.0023684,-0.00033849,0.0013074],[-0.0012665,0.00028059,0.0013161],[0.0010572,-0.0013201,0.0010698],[0.00031599,-0.0015151,0.0009323],[-0.00065847,-0.0013027,0.00088001],[-0.00010473,-0.0014751,0.00040104],[-0.00019461,-0.00083698,0.00015078],[-0.00040834,-0.0011025,0.0011392],[-0.00092492,-0.001003,0.00060059],[-0.0011132,0.00045063,0.0012077],[-0.0012614,0.00032927,0.0011372],[-0.001397,1.182e-05,0.001034],[-0.0014035,-0.00019509,0.00091879],[-0.0013773,-0.00060485,0.00069413],[0.00061941,0.0014222,-0.00029676],[0.00042695,0.0017419,-0.00034536],[0.00043758,0.0017508,-0.00044678],[0.0012885,0.00094102,-0.00030121],[0.0013716,0.00067691,-0.00022479],[0.0014574,0.0004496,-0.000102],[0.0014344,0.00052406,0.00024926],[0.0014109,0.00060268,0.00045148],[0.0015854,0.00042312,0.0004496],[0.00057862,0.00064478,0.00094988],[0.00094404,0.00089306,0.0012868],[0.0012663,0.00068984,0.0009597],[0.0012541,0.0007195,0.00084897],[0.00075539,0.00040478,0.001318],[0.00057862,0.00064478,0.00094991],[0.0014555,0.00055215,0.0011334],[0.001323,0.00037349,0.00087718],[0.0016277,0.00040433,0.00036958],[-1.6449e-05,-0.0006808,-0.0017681],[0.00011084,-0.00084645,-0.0016468],[-0.00014194,-0.00014696,-0.0016205],[-0.00017804,-0.00032066,-0.0015904],[-7.8799e-05,-0.00058521,-0.0016569],[-0.00033583,-0.00081803,-0.0015216],[-0.00085331,-0.00068707,-0.0010646],[-0.0017232,-5.0093e-05,-0.00050796],[-0.0013474,-0.00019244,-0.00098454],[-0.00076057,-0.00038005,-0.0012335],[-0.00042362,-0.00031416,-0.0015688],[-0.0008872,-0.00036451,-0.0014735],[-0.0012752,-0.00030522,-0.0012164],[-0.0015793,-0.00031636,-0.00058281],[-0.001437,-0.0010127,9.9402e-05],[-0.001326,-0.0012216,-0.00016527],[-0.0012332,-0.001116,-0.0003594],[-0.0010705,-0.0011209,-0.00046771],[-0.00094468,-0.0010292,-0.00054177],[-0.0014612,-0.00091393,0.00027532],[-0.0009286,-0.0010621,-0.00076],[-0.00076994,-0.00096044,-0.00090282],[-0.0013704,-0.0011025,0.00019762],[-0.0015235,-0.00058885,0.00041692],[-0.00096021,-0.0011123,-0.00093059],[-0.00045365,-0.0013025,-0.0011512],[-0.0003802,-0.0010866,-0.00088916],[-0.0015282,-0.00057022,0.0004234],[-0.0013737,0.00033586,-0.00073935],[-0.00030752,0.00038152,-0.0017751],[-0.00030756,0.00034098,-0.0016853],[-0.00070712,0.00024666,-0.0016509],[-0.00088659,0.00075724,-0.0013039],[-0.00020384,0.00042965,-0.0016353],[-6.6562e-05,0.00046663,-0.0016513],[0.00043208,0.0011952,-0.0013219],[0.00036737,0.0010413,-0.0014198],[0.00051883,0.0016395,-0.00090224],[0.00020388,0.00093963,-0.0014776],[-0.0002127,0.00093211,-0.0014953],[-0.00018101,0.0005853,-0.0016517],[0.0022762,8.0116e-06,0.0019507],[-0.00095538,0.00030775,0.00074754],[0.00071278,-0.00092883,0.0018528],[9.0382e-05,-0.0009957,0.0010749],[-0.00073469,-0.0011328,0.0010271],[-0.00031942,-0.001537,0.00042846],[-0.00051066,-0.0014229,0.00023059],[-0.00011653,-0.0012021,0.001152],[-0.0011335,-0.0012056,0.00044864],[-0.00057697,-0.0013981,-3.618e-05],[-0.0011968,0.00040364,0.0011027],[-0.0013064,0.00029491,0.0010487],[-0.0013764,-4.6632e-06,0.00095616],[-0.0014388,-0.00022102,0.00084017],[-0.0014849,-0.00069964,0.00043243],[0.00045381,0.0013545,-0.0003751],[0.00031264,0.0020728,-0.00062592],[0.00038243,0.0018337,-0.00057287],[0.0012241,0.00097704,-0.00037666],[0.0014165,0.00068674,-0.00021751],[0.0015323,0.00031017,8.0616e-05],[0.0015242,0.0005112,0.00031311],[0.0015616,0.00025449,0.00040087],[0.0015649,0.00031168,0.00055569],[0.00088696,0.00031984,0.00088661],[0.0012925,0.0004978,0.0011706],[0.0013056,0.00057443,0.00089976],[0.0012277,0.00071026,0.00083485],[0.0010756,-0.00018392,0.0010978],[0.0010402,1.1296e-06,0.00072126],[0.0010491,0.0006609,0.0010511],[0.0013993,0.00031434,0.0012635],[0.0013782,0.00022041,0.00082152],[0.0016297,0.00019943,0.00030147],[-1.1747e-05,-0.0007085,-0.0018227],[-0.00012632,-0.00060931,-0.0017379],[-5.7651e-05,-0.00048334,-0.0016843],[-0.00024669,-0.00042176,-0.0017109],[-0.00047414,-0.00069967,-0.0015749],[2.1948e-05,-0.00053358,-0.0017883],[-0.0015907,-0.00011225,-0.00050669],[-0.0012476,-0.00048817,-0.0010748],[-0.00057654,-0.00062058,-0.0012751],[-0.00046113,-0.00054717,-0.0014686],[-0.00087463,-0.00050762,-0.0013336],[-0.0012773,-0.0005195,-0.0010771],[-0.0014767,-0.00096433,0.00012197],[-0.0014289,-0.001062,-0.0001005],[-0.0012035,-0.0011795,-0.00038558],[-0.0011539,-0.0010758,-0.00044896],[-0.0010308,-0.0011474,-0.00057523],[-0.0013787,-0.0011082,0.00020304],[-0.00076468,-0.0010136,-0.00093647],[-0.00089632,-0.0011232,-0.00078609],[-0.00071259,-0.00092418,-0.00091591],[-0.0015974,-0.00063054,0.00038363],[-0.0010439,-0.00095934,-0.0011443],[-0.00044438,-0.0014396,-0.0011157],[-0.00054425,-0.0014642,-0.0011733],[-0.0014874,-0.00060478,0.00045989],[-0.001456,0.00033768,-0.00068701],[-0.00020681,0.00035276,-0.0016701],[-0.00022912,0.00033495,-0.0016799],[-0.00027278,0.00022842,-0.0016813],[-0.0012353,-0.00016677,-0.00095341],[-0.00024432,0.00036736,-0.0016516],[-0.00015708,0.00032824,-0.0016803],[0.00035398,0.001156,-0.0014374],[0.00032105,0.0015354,-0.0011272],[0.00023301,0.00092596,-0.0015574],[-7.5907e-05,0.00064387,-0.0016637],[-0.00030376,0.00083613,-0.0015624],[-8.6749e-05,0.00072168,-0.0016171],[0.00077361,-0.00065679,0.0018658],[-0.00095769,0.00035191,0.00085819],[-0.00072606,-0.00026652,0.00063839],[0.00027782,-0.00064787,0.0011891],[-0.00047568,-0.00089843,0.0011764],[-0.00059641,-0.0011514,0.001009],[-0.00045232,-0.0015289,0.00012143],[-0.00049249,-0.0015089,0.00070985],[-0.00025564,-0.00094677,0.0014417],[-0.0010001,-0.00087785,0.00033112],[-0.00051747,-0.0013942,-2.4382e-05],[-0.001132,0.00040037,0.0010106],[-0.0013218,0.0003407,0.00099874],[-0.0014722,0.00016344,0.00079691],[-0.0014792,-4.3321e-05,0.00064192],[-0.001554,-0.00043238,0.00018222],[0.001211,0.00087436,-0.00065606],[0.00062599,0.0017967,-0.00067633],[0.00031471,0.0020927,-0.00062965],[0.0013833,0.00058654,-0.00042032],[0.0015184,0.00016093,-0.00027397],[0.001556,0.00042123,9.6759e-05],[0.0015162,0.00023868,0.00024863],[0.0015916,0.00029854,0.00036542],[0.0016936,1.6167e-05,0.00051886],[0.00086108,-4.6176e-05,0.0012134],[0.0012074,0.00043287,0.0011091],[0.0013394,0.00049254,0.00097424],[0.0012622,0.00056577,0.00094335],[0.0010607,0.00067331,0.0010604],[0.0012601,0.00022432,0.00086892],[0.0011386,-0.00018609,0.00096596],[0.00086745,0.00059853,0.0013421],[0.00087228,0.00054356,0.0013722],[0.00143,0.00032479,0.00076518],[0.0017278,7.4385e-06,0.00050224],[-0.00014892,-0.0006938,-0.0016541],[-0.00046436,-0.0006685,-0.0015769],[-0.00058322,-0.00047321,-0.001509],[-0.00062706,-0.0007135,-0.0014709],[7.0904e-06,-0.00050927,-0.0017661],[-0.001131,-0.00011979,-0.0012573],[-0.0005109,-0.00038961,-0.0014481],[1.2922e-06,-0.00051185,-0.0016996],[-0.00058589,-0.00046585,-0.0015266],[-0.00095945,-0.00047389,-0.0013759],[-0.00095493,-0.00093803,-0.0012233],[-0.0015153,-0.00089002,5.0233e-05],[-0.0014558,-0.0010222,-0.00014226],[-0.0012417,-0.0011966,-0.0004097],[-0.001223,-0.001092,-0.00050592],[-0.0011791,-0.001075,-0.00055661],[-0.0014941,-0.0009452,3.564e-05],[-0.00077238,-0.0012337,-0.00078772],[-0.00088995,-0.0013358,-0.00064356],[-0.00076981,-0.0011911,-0.00080934],[-0.0016187,-0.00052676,0.00028409],[-0.00027666,-0.0013019,-0.001142],[-0.00043947,-0.001238,-0.00090049],[-0.0015724,-0.0006877,0.00022523],[-0.00021334,0.00036517,-0.0016862],[-0.00027709,0.00023604,-0.0016826],[-0.00061966,-0.00043134,-0.0015107],[-0.00080658,-0.00014351,-0.0015043],[-0.00017466,0.00036388,-0.001662],[0.00011695,0.00030545,-0.0016962],[0.00014688,0.0011908,-0.0014161],[0.0001707,0.0015925,-0.0011374],[0.00012673,0.00093889,-0.0015423],[-7.2569e-05,0.00064303,-0.0016632],[0.00016757,0.00077339,-0.0015951],[-0.00019609,0.00073175,-0.001611],[0.00019914,-0.00014932,0.001883],[-0.0010766,0.00046209,0.00087844],[-0.00081064,-0.0003478,0.00039684],[-0.0001551,-0.0003776,0.0013889],[-0.00054224,-0.00066033,0.0013467],[-0.00076569,-0.00072582,0.001344],[-0.00047872,-0.0016209,0.00012879],[-0.00050755,-0.0015138,0.00071336],[-0.00025367,-0.00097994,0.001487],[-0.0010172,-0.00089496,0.00035977],[-0.00058293,-0.0014172,-0.00032442],[-0.0005235,-0.0015539,-8.2224e-05],[-0.0012122,0.00037257,0.00093452],[-0.0013818,0.00026432,0.00078245],[-0.001415,0.00012768,0.0006684],[-0.0016015,-0.00012764,0.00029147],[-0.0016144,-0.00043991,0.0002515],[0.0013048,0.00084477,-0.00057837],[0.00074556,0.0017503,-0.00059627],[4.8301e-05,0.0018177,-0.00091229],[0.0014016,0.000548,-0.00035964],[0.001715,0.00015047,-0.0002672],[0.0013432,-1.2037e-05,0.00055972],[0.0014627,-0.00040965,0.00091975],[0.0015728,0.00027294,0.00035713],[0.0016162,0.0002288,0.00025091],[0.00098332,-0.00029725,0.0010854],[0.0013663,7.2747e-05,0.00090033],[0.0014972,0.00012546,0.00077405],[0.0013692,0.00036891,0.00085246],[0.0011554,0.00062483,0.0010679],[0.0014047,1.283e-06,0.00081299],[0.0011185,-0.00013599,0.0009917],[0.00080967,0.00072765,0.0014529],[0.00023821,7.8634e-05,0.0019818],[0.0014959,-2.8507e-05,0.00063655],[0.0010458,0.00029822,0.00040718],[-0.00033511,-0.00049333,-0.0016976],[-0.00047989,-0.00066997,-0.0016022],[-0.00058104,-0.00050633,-0.0015535],[-0.00054973,-0.00085333,-0.0015363],[-0.0003592,-0.00059215,-0.0016231],[-0.00094719,-0.00051875,-0.0013292],[-0.00034751,-0.00061191,-0.0014553],[-1.6117e-05,-0.00047149,-0.0016321],[-0.00098132,-0.0005773,-0.001335],[-0.00093374,-0.00089623,-0.0012101],[-0.0015111,-0.00095744,2.9619e-05],[-0.0015175,-0.00092877,-0.00010351],[-0.0013815,-0.0010234,-0.00033697],[-0.0011753,-0.0011847,-0.00054397],[-0.0011601,-0.0013515,-0.00034827],[-0.0016799,-0.00056753,0.00019187],[-0.00075547,-0.0012486,-0.00083225],[-0.00074776,-0.0013047,-0.00081496],[-0.000987,-0.0013554,-0.00066513],[-0.00063816,-0.001186,-0.0009565],[-0.00021753,-0.001259,-0.00083647],[-0.00040313,-0.0011992,-0.00089599],[-0.0015707,-0.00063599,0.00018099],[-0.00017434,0.00035905,-0.001669],[-0.00021266,0.00023125,-0.0016664],[-0.00026161,-0.00041712,-0.0016244],[-0.00017665,0.00036357,-0.0016698],[8.318e-05,0.00025402,-0.0017185],[5.5583e-05,0.0011129,-0.0014947],[2.7486e-05,0.0014562,-0.00126],[-9.468e-05,0.00069958,-0.0016776],[0.00012276,0.00086895,-0.0015607],[-6.9379e-05,0.00047796,-0.001708],[-0.00018156,0.00076674,-0.0016279],[-0.000173,-0.00034432,0.0016686],[-0.001245,0.00040967,0.00068636],[-0.00092207,-0.00049198,0.00049444],[-0.00065376,-0.00065252,0.0014156],[-0.00040577,-0.00092555,0.0014174],[-0.0006999,-0.001622,0.00039257],[-0.00071538,-0.0013289,0.001037],[-0.00030199,-0.00088634,0.0015934],[-0.00094685,-0.00091553,0.00036552],[-0.00060067,-0.0014329,-0.0003316],[-0.00085623,-0.0015066,-0.00014422],[-0.0012799,0.00057002,0.00075173],[-0.0015121,0.00039184,0.00071866],[-0.0014047,0.00024087,0.00050139],[-0.0015158,-5.2428e-05,0.00020428],[-0.0016536,-0.00022664,6.1653e-05],[0.0011461,0.00063536,-0.00088558],[0.0013407,0.0004285,-0.00058366],[0.00029676,0.0012535,-0.001386],[-1.7064e-07,0.0017735,-0.00088606],[0.0015243,5.1888e-05,-0.00039438],[0.0015428,-0.00038738,-0.00028776],[0.0015168,-0.00056051,0.00060211],[0.0013689,0.00019525,0.0010135],[0.0014152,2.4605e-05,0.00055346],[0.0015726,0.00064832,0.00029635],[0.0011486,-0.00014707,0.0011195],[0.0012225,-0.00029072,0.00099303],[0.0014768,8.6724e-05,0.00078184],[0.0013696,0.00011212,0.00098576],[0.0011532,0.00034836,0.0012088],[0.00079914,0.00053914,0.0015234],[0.0012852,1.9563e-05,0.00099411],[0.0014086,1.1499e-05,0.00080683],[-3.6644e-05,0.00028327,0.0017274],[-4.1889e-05,0.00023877,0.0017394],[0.0011246,0.00025257,0.00027241],[0.0010457,0.0002982,0.00040717],[-0.00042192,-0.00050016,-0.0016132],[-0.0004706,-0.00064194,-0.0015247],[-0.00045135,-0.00051934,-0.00084775],[-0.00018991,-0.00077744,-0.0015898],[-0.00027955,-0.00044292,-0.001524],[-5.5772e-05,-0.00058474,-0.001594],[-0.00010641,-0.00051544,-0.0016966],[-0.00076997,-0.00088733,-0.0014453],[-0.0017026,-0.00060549,-0.00031093],[-0.0016637,-0.00061524,-0.00039369],[-0.0014837,-0.0008082,-0.0005472],[-0.0012228,-0.0011548,-0.0006109],[-0.00139,-0.0011637,-0.00029412],[-0.0017307,-0.00039921,4.0263e-05],[-0.00078362,-0.0011556,-0.00097526],[-0.00079946,-0.001123,-0.001044],[-0.0010865,-0.00078557,-0.0011719],[-0.00065222,-0.0011783,-0.0010113],[-1.4978e-06,-0.00084711,-0.00074485],[-0.00042125,-0.0011779,-0.0010413],[-0.0016194,-0.00070019,-8.0668e-05],[-0.00023198,0.00026439,-0.0016824],[-0.00044106,-0.00017083,-0.0016491],[-0.00028099,-0.00046152,-0.0015614],[3.1084e-05,0.00035317,-0.0016872],[7.209e-05,0.00025774,-0.001737],[-0.00011469,0.0011461,-0.0014896],[-0.00013813,0.0014287,-0.001207],[-0.00012124,0.0007021,-0.0016712],[6.6404e-05,0.00088614,-0.0015715],[-0.00010797,0.00048878,-0.0017283],[-0.0003756,0.0007688,-0.0015825],[-0.00048696,-0.00010883,0.0015961],[-0.0012951,0.0003747,0.00077148],[-0.001062,-0.00059594,0.00027202],[-0.00069269,-0.00022875,0.0015616],[-0.00055985,-0.00035227,0.0017224],[-0.00091339,-0.001565,0.00038105],[-0.0009457,-0.0011074,0.00092646],[-0.00079726,-0.00066023,0.0015388],[-0.00058911,-0.0014753,-0.00018575],[-0.00099436,-0.00095377,0.00029543],[-0.00082827,-0.0014521,-0.00015023],[-0.00098124,-0.0014664,-4.8115e-06],[-0.0013521,0.00056891,0.00071574],[-0.0013426,0.00029201,0.00049942],[-0.0014549,0.00017669,0.00027036],[-0.0016325,-7.2877e-05,0.00011378],[-0.0017168,-0.00022793,0.00011989],[0.0009368,0.00069807,-0.0010026],[0.001151,0.00059593,-0.00086954],[0.00051164,0.0010044,-0.0010952],[-0.00020696,0.0015412,-0.0010984],[0.0011729,0.00035824,-0.00095824],[0.0015098,-0.00073279,5.3304e-05],[0.0012396,-0.00081796,0.0010357],[0.001166,-0.00020507,0.0011927],[0.0013568,0.00045731,0.00060014],[0.0013483,0.00039661,0.00046615],[0.0016666,0.00068149,0.00030417],[0.0010743,0.00011462,0.0012923],[0.0009054,0.00022123,0.0012218],[0.0012337,0.00040973,0.0009149],[0.001159,0.00043004,0.0011142],[0.0010262,0.00052694,0.0012711],[0.00073281,0.00061717,0.0015529],[0.0011675,0.00039812,0.0011855],[0.0013935,0.00039262,0.00097333],[0.00024995,0.00010021,0.0017759],[-0.00049749,-3.0436e-05,0.0016623],[0.0014873,0.00039065,0.00036447],[0.0011371,0.00027008,0.00031117],[1.047e-05,-0.00093938,-0.0015104],[7.1215e-05,-0.0011221,-0.0013893],[1.7346e-05,-0.0008743,-0.00075229],[-0.00045134,-0.00051933,-0.00084778],[8.3401e-05,-0.00070947,-0.0016446],[-5.3369e-05,-0.00059301,-0.0016068],[0.00016922,-0.00080596,-0.0015912],[-0.0017469,-0.00045369,-0.0002475],[-0.0016746,-0.00061235,-0.00039313],[-0.0016286,-0.00061925,-0.00047466],[-0.0015119,-0.00080779,-0.00049014],[-0.0015069,-0.00067823,-0.00073699],[-0.00084513,-0.0011237,-0.00098441],[-0.00072094,-0.0011392,-0.001022],[-0.0011726,-0.00077913,-0.0010023],[-0.0012128,-0.00066062,-0.0011546],[-0.00053506,-0.0011467,-0.0010887],[-8.0415e-06,-0.00119,-0.00099865],[-0.0002335,-0.001056,-0.0010685],[-0.0016021,-0.00070734,-6.896e-05],[-0.00019011,0.00026713,-0.0017091],[-0.00033357,-0.00017368,-0.0017566],[9.8284e-06,-0.0004431,-0.0016583],[-0.00011138,0.000122,-0.0017304],[2.235e-05,0.00018067,-0.0017459],[-0.00025296,0.00097444,-0.0015502],[-0.0002356,0.0013533,-0.0013027],[5.9234e-05,0.00090365,-0.0015688],[-0.00016696,0.00060908,-0.0016804],[-0.00018103,0.00038301,-0.0017257],[-0.00042763,0.00071502,-0.001619],[-0.0014649,0.00031202,0.00053462],[-0.0010263,-0.00082041,0.00037497],[-0.00054277,-0.00034171,0.0016327],[-0.00089263,-0.00017459,0.00166],[-0.0010784,-0.0011752,0.00069747],[-0.0011257,-0.00087804,0.0011533],[-0.00090303,-0.00017509,0.0016876],[-0.0010484,-0.0011087,-0.00032298],[-0.0008917,-0.0011381,0.00036669],[-0.0012353,-0.0011716,-0.00026685],[-0.0014701,-0.0012109,-7.6556e-05],[-0.0014016,0.00067579,0.00064678],[-0.0010967,0.00030664,0.00028605],[-0.0011521,0.00012781,0.00023018],[-0.0016127,6.9691e-05,-1.4365e-05],[-0.001725,0.00013108,-0.00016854],[0.00074367,0.00045684,-0.0012866],[0.0010147,0.0004386,-0.001042],[0.001173,0.00039583,-0.00086593],[0.00020695,0.00064933,-0.0014661],[-0.00018347,0.00146,-0.0010413],[0.0011949,-0.00039012,-0.00086349],[0.00080664,-0.0006601,-0.0010126],[0.0010273,-0.0012192,-8.6553e-05],[0.0012607,-0.00076171,0.0010361],[0.0010935,-0.00020422,0.0013733],[0.00077299,0.0004706,0.0011832],[0.0012306,0.00036638,0.00059861],[0.00069221,0.0011924,0.00032519],[0.001045,0.0012713,0.00017516],[0.0011975,0.00040415,0.0011912],[0.00092408,0.00010255,0.0013489],[0.0012054,0.0002536,0.00097899],[0.0011758,0.0004268,0.0011356],[0.0010159,0.00042164,0.001317],[0.00070118,0.00048337,0.0015615],[0.00031719,0.00050424,0.0017907],[0.0011771,0.00041697,0.0011804],[0.0013278,0.00053126,0.00078295],[-9.0175e-05,-8.6652e-05,0.0018943],[-0.00010409,-0.00029533,0.0017471],[0.00073408,0.0014094,3.006e-05],[0.0001403,-0.00086118,-0.0014665],[0.00015022,-0.00095186,-0.0012244],[0.00023911,-0.00084828,-0.0015639],[0.00026612,-0.00056335,-0.0017055],[0.00026972,-0.00081388,-0.0016527],[-0.0016436,-0.00080095,4.3319e-05],[-0.0016127,-0.00088251,-0.00017105],[-0.00159,-0.0007754,-0.00034353],[-0.0015132,-0.00074622,-0.00053293],[-0.0014325,-0.00076544,-0.00075753],[-0.00089495,-0.00092153,-0.0012381],[-0.00074699,-0.00097479,-0.0012391],[-0.0012113,-0.00072989,-0.0010867],[-0.001197,-0.00068273,-0.0011187],[-0.00052252,-0.00094048,-0.0012912],[-0.00021349,-0.00096432,-0.0012819],[-0.0016725,-0.00071535,7.9174e-05],[-0.00040522,-8.1492e-05,-0.0017879],[-0.0002871,-8.0723e-05,-0.0017627],[5.5168e-05,0.00011777,-0.0017656],[-0.0001535,0.00019013,-0.0017307],[-0.00029257,0.00097727,-0.0015418],[-0.00045454,0.0013235,-0.0012261],[-0.00023662,0.00095544,-0.0015561],[-0.00032407,0.00063077,-0.0016683],[-0.00028539,0.00039716,-0.0017422],[-0.00055035,0.00071689,-0.0015896],[-0.0013715,5.8254e-05,0.00072222],[-0.0010965,-0.00088075,0.00019278],[-0.00057808,-0.00059682,0.0017135],[-0.00088195,-0.00026882,0.0016457],[-0.0013053,-0.0009068,0.00062242],[-0.0011675,-0.00064743,0.00093533],[-0.0011832,2.6612e-05,0.001483],[-0.0012,-0.0011382,-0.00025925],[-0.00093137,-0.001207,0.0003754],[-0.00095149,-0.0012114,0.00037693],[-0.0013683,-0.0011099,-0.00012836],[-0.0013919,-0.00087536,0.00017349],[-0.0016886,0.00069641,0.00050586],[-0.001894,0.00055043,0.00048423],[-0.0015514,0.00013999,1.3659e-05],[-0.0016605,4.9615e-05,-0.00019413],[-0.0017147,0.00010485,-0.00044018],[0.00067988,0.00046591,-0.0012912],[0.00082856,0.00050695,-0.0011544],[0.00095813,0.00050462,-0.0010515],[0.00049864,0.0005797,-0.0013816],[-0.00042094,0.0012707,-0.0012903],[0.00069909,-0.00016688,-0.0012918],[0.00053084,-0.0011842,-0.00046169],[0.00085654,-0.0013962,0.00021558],[0.00094032,-0.00069069,0.0010514],[0.00099163,-2.558e-05,0.0011424],[0.00068788,0.00016428,0.0013546],[0.00078032,0.00099803,0.00053294],[0.00072,0.00102,0.00060517],[0.00095823,0.0011999,0.00011116],[0.00084644,0.0013759,0.00043494],[0.000652,0.0010302,0.0012439],[0.00062556,0.0005097,0.0015012],[0.00096322,0.00061679,0.0011465],[0.00083928,0.00071648,0.001157],[0.00054877,0.00078324,0.0012556],[0.00036784,0.00078974,0.0015255],[-2.7111e-05,0.00076957,0.0017102],[0.00065095,0.0011454,0.0012239],[0.00062785,0.0015784,0.00071525],[-0.00063082,0.00027379,0.0017569],[0.00052059,0.0016092,0.00035144],[0.00011865,-0.00085107,-0.0014913],[3.2131e-06,-0.00084812,-0.0012839],[0.00012681,-0.00088842,-0.0015719],[-8.0281e-05,-0.00020014,-0.0019279],[-5.8289e-05,-0.00045032,-0.0016483],[-0.0016617,-0.0008483,-0.00015331],[-0.0015786,-0.00092007,-0.00040148],[-0.0015344,-0.00080262,-0.0005601],[-0.0014371,-0.00049681,-0.00094702],[-0.00084514,-0.00090285,-0.0012512],[-0.0010791,-0.00071288,-0.0012014],[-0.00057397,-0.0009271,-0.001324],[-0.0012359,-0.00068476,-0.001071],[-0.0013391,-0.00049304,-0.0010762],[-0.00034809,-0.0008657,-0.0013209],[-0.001734,4.657e-05,-0.0004737],[-5.4065e-05,-7.3875e-05,-0.001875],[-1.258e-05,-5.5586e-05,-0.0018605],[-5.6804e-05,-6.1893e-05,-0.0017953],[-0.00021026,8.7556e-05,-0.0016881],[-0.00026131,0.0010025,-0.0015137],[-0.00046528,0.0012991,-0.0012274],[-0.00039186,0.00076848,-0.0016234],[-0.00036482,0.00056061,-0.0016547],[-0.00034355,0.00030535,-0.001731],[-0.00065901,0.0005438,-0.0015921],[-0.001456,1.265e-05,0.00054781],[-0.00091575,-0.0011977,0.00035209],[-0.00099932,-0.00033981,0.0016277],[-0.0013601,0.00011542,0.0011994],[-0.0011248,-0.00063764,0.00061054],[-0.0013978,-0.00017111,0.001143],[-0.001104,0.00028473,0.0013736],[-0.0014439,-0.00058198,-0.00038318],[-0.0012109,-0.0010317,0.00031128],[-0.0011406,-0.0011597,0.00034158],[-0.0016346,-0.00038902,-0.00027992],[-0.0009271,-0.00017255,7.3539e-05],[-0.0015141,0.00028094,0.00069941],[-0.0014709,0.00040407,5.8646e-05],[-0.0015902,0.00020076,-3.2662e-05],[-0.001679,4.5716e-05,-0.00019274],[-0.0017002,-3.256e-05,-0.00033666],[0.00057787,0.00033987,-0.0014156],[0.00072071,0.00037181,-0.0013205],[0.00088272,0.00040587,-0.0011956],[0.0011426,0.00011352,-0.0011798],[0.00035575,0.00040861,-0.001534],[4.4553e-05,0.0012242,-0.0012943],[0.00046962,-0.00094557,-0.00092171],[0.00046597,-0.00133,-0.00052762],[0.0001062,-0.00089147,-0.0008675],[0.00094322,-0.0013514,0.00023498],[0.00096421,-0.00065348,0.0010549],[0.00096781,0.00017114,0.0012636],[0.00072114,-1.6161e-05,0.0013046],[0.00037899,0.00061959,0.001295],[0.00069072,0.00055982,0.00091774],[7.7559e-05,0.0012364,0.00036859],[0.00042637,0.0014713,-4.467e-05],[0.00032572,0.0015865,0.00024689],[0.00069315,0.0010334,0.0012095],[0.00063997,0.0010873,0.0011417],[0.00094511,0.00054834,0.0011638],[0.00083593,0.00059093,0.0012284],[0.00056591,0.00069688,0.0013631],[0.00034855,0.00077705,0.0014068],[-3.0829e-05,0.00069717,0.0016336],[-0.00059407,0.00055555,0.0017181],[0.00060295,0.0015134,0.00068492],[-0.0010259,6.3328e-05,0.0016998],[0.00035548,0.0017227,0.00028906],[2.3059e-05,-0.00086143,-0.001444],[-0.00013086,-0.00061399,-0.001584],[1.6068e-05,-0.00019479,-0.0020196],[-9.0962e-05,-0.00045915,-0.0016568],[-0.0016551,-0.00012343,-0.00064888],[-0.0015639,-0.00052977,-0.00066307],[-0.0015171,-0.00054351,-0.00073087],[-0.0014233,-0.00055834,-0.00097524],[-0.00088555,-0.00072229,-0.0014936],[-0.0011449,-0.0006925,-0.0013194],[-0.00056741,-0.00072968,-0.0015443],[-0.0012676,-0.00071281,-0.0010915],[-0.0013382,-0.0005833,-0.0010143],[-0.00033765,-0.0007598,-0.0014992],[-0.0016934,3.7483e-05,-0.00058158],[-3.1363e-05,-3.6595e-05,-0.0020288],[-0.00013655,-6.1102e-05,-0.0017767],[-0.00041747,9.1852e-05,-0.0016817],[-0.00048951,0.0010153,-0.0014605],[-0.00044498,0.001267,-0.0011994],[-0.00047168,0.00076713,-0.0015892],[-0.00041916,0.00056532,-0.0016448],[-0.00034296,0.00031184,-0.0017715],[-0.00061056,0.00054591,-0.001612],[-0.0011733,-0.00030633,0.00067087],[-0.00098019,-0.0012684,0.00021506],[-0.001369,0.00011135,0.0012076],[-0.00091851,-5.11e-05,0.00026161],[-0.0014564,1.3278e-05,0.00082604],[-0.001328,0.00050026,0.001051],[-0.0016221,-0.00041713,-0.0001757],[-0.0013215,-0.00080814,0.00055144],[-0.00124,-0.0010962,0.00045688],[-0.0012256,-0.0012417,0.00034764],[-0.0018591,-0.00027144,-0.00011687],[-0.00092709,-0.00017255,7.3539e-05],[-0.0015184,9.9202e-05,-0.00028013],[-0.0015961,0.00061721,-5.6351e-05],[-0.0016158,0.00019509,-0.00011783],[-0.0016997,3.4374e-05,-0.00030513],[-0.0016628,-4.9117e-05,-0.00051212],[0.0006035,0.00033114,-0.0014022],[0.00073715,0.00036145,-0.0012993],[0.00094396,0.00040055,-0.0011997],[0.00052399,0.0003868,-0.001708],[0.00043405,0.00038443,-0.0015011],[-0.0002606,0.00089292,-0.0015226],[-7.9349e-05,-0.0003957,-0.0014599],[0.00046764,-0.001413,-0.00051708],[0.00013945,-0.00086399,-0.0009671],[0.00091627,-0.0013889,0.00029182],[0.0008887,-0.00062785,0.0010397],[0.0010089,0.00030698,0.0012124],[0.00060585,-0.00025676,0.0012995],[0.00038904,0.00056021,0.0014761],[-0.00018073,0.0015804,0.00067827],[8.2543e-05,0.001353,0.00038266],[0.00044481,0.0013457,0.0001831],[0.00030512,0.0016312,7.9618e-05],[0.00034001,0.0016629,0.00025991],[0.00021948,0.001368,0.0011165],[6.4138e-05,0.0014788,0.0009743],[0.00014377,0.0012038,0.0012191],[0.00028467,0.0010652,0.0013348],[0.00026152,0.00082579,0.0012194],[6.9898e-05,0.00091441,0.0012008],[-0.00035846,0.00094763,0.0014525],[-0.00086341,0.00075921,0.0014881],[0.00017543,0.001726,0.00052735],[-0.0012227,0.0002641,0.0014016],[0.00016403,0.0017036,0.00043383],[-0.00017603,-0.0006961,-0.0014972],[-0.00015774,-0.00062428,-0.0015926],[-8.3373e-06,-0.00012103,-0.0014361],[-0.00025774,-0.00025322,-0.0016025],[-0.0016999,-0.00011471,-0.00049967],[-0.0015352,-0.00053238,-0.00073092],[-0.0013988,-0.00038692,-0.0010714],[-0.00092665,-0.00066707,-0.0014755],[-0.00065523,-0.00067012,-0.0015762],[-0.0010564,-0.00069203,-0.0012599],[-0.00044285,-0.00068198,-0.0015363],[-0.0013347,-0.00059062,-0.0010522],[-0.0014866,-0.00039178,-0.00098199],[-0.0016773,-4.7856e-05,-0.00051576],[0.00013615,-2.698e-05,-0.0020999],[1.8419e-05,0.00016885,-0.0016799],[-0.0003616,0.00017146,-0.0016521],[-0.00055424,0.00093834,-0.0014888],[-0.00055807,0.0011608,-0.0012753],[-0.00049857,0.00072551,-0.0015896],[-0.00044378,0.00052831,-0.0016451],[-0.00042883,0.00017799,-0.0017644],[-0.00085978,0.00012276,-0.0015963],[-0.0013717,-0.00048981,1.2064e-05],[-0.0010499,-0.0012902,0.00020559],[-0.0015374,0.00027413,0.00094532],[-0.0015831,-0.00020832,0.00043752],[-0.0014691,0.00031266,0.00080699],[-0.0013395,0.00040229,0.0010692],[-0.0016297,-0.00010501,-0.00023715],[-0.0016361,-0.00041566,0.00045896],[-0.0016836,-0.00063145,0.00027983],[-0.0017484,-0.00068474,5.7663e-05],[-0.0017099,0.00012962,-0.00021506],[-0.0010364,-9.327e-05,0.00023911],[-0.0015083,-0.00037551,7.3824e-05],[-0.0015141,0.00057492,-0.00016305],[-0.0015854,0.00039729,-0.00026809],[-0.0016118,0.00020931,-0.0004224],[-0.0015923,6.4792e-05,-0.00057287],[0.00049485,0.00019775,-0.0015219],[0.00063638,0.00024119,-0.0014203],[0.00080284,0.00023763,-0.0013528],[0.00091769,0.00081515,-0.00127],[0.00043399,0.00018796,-0.0013348],[0.00035356,0.00028845,-0.0015639],[0.00017253,0.00084419,-0.0015209],[0.00010494,-0.00043758,-0.0016123],[-0.00018307,-0.00058087,-0.0013259],[-0.00011902,-0.00093416,-0.0010368],[0.00053152,-0.0014714,-0.00053778],[-0.00027033,-0.0005627,-0.0012772],[0.00096463,-0.0013838,0.0003028],[0.00090233,-0.00065342,0.0010631],[0.00044004,0.00030956,0.0016021],[0.00084878,0.0005931,0.001246],[0.00071804,-0.00025462,0.0011655],[-0.00010326,0.001125,0.0012944],[-0.00029101,0.0012619,0.0015378],[-0.00015589,0.0011892,0.00029228],[9.0844e-05,0.0014669,3.5857e-05],[0.00016149,0.0015814,2.3832e-05],[0.00015199,0.0016713,0.00017629],[0.00020232,0.0014923,0.00087163],[0.00013902,0.0012901,0.0011122],[4.661e-05,0.0017103,0.00062047],[0.00026928,0.0012179,0.0011122],[0.00026541,0.00094931,0.0011773],[5.2766e-05,0.00078466,0.0012756],[-0.00031144,0.0008476,0.0012693],[-0.00082751,0.00080499,0.0014295],[-0.001205,0.0005779,0.0013694],[0.00012862,0.0017068,0.00041696],[-0.00035543,-0.00043431,-0.0016492],[-0.00020613,-0.00014718,-0.0013959],[-6.6206e-06,-0.00015152,-0.001531],[-0.00029623,-0.00026155,-0.0016166],[-0.0016903,-1.1618e-05,-0.00056666],[-0.0015415,-0.0003713,-0.00084871],[-0.0014346,-0.0003811,-0.0010893],[-0.00092749,-0.00023271,-0.0018211],[-0.00061285,-0.00039755,-0.00176],[-0.0013443,-0.00030011,-0.0019449],[-0.00039363,-0.00044857,-0.0016746],[-0.0012213,-0.0002628,-0.0011316],[-0.0013956,-0.00010209,-0.0010969],[-0.0016592,-5.1996e-05,-0.00057693],[-0.00035066,0.00016554,-0.0016049],[-0.00086716,0.00015993,-0.0015593],[-0.00064778,0.00093861,-0.0014589],[-0.00057575,0.001167,-0.0012777],[-0.00076755,0.00071129,-0.0014491],[-0.00079864,0.00052981,-0.0014745],[-0.00052094,0.00018105,-0.0017593],[-0.00049749,0.00013543,-0.0017498],[-0.0014046,-0.00084358,0.00018513],[-0.0011653,-0.0013903,-0.0001722],[-0.0015706,0.00039786,0.00095181],[-0.0015501,0.00043782,0.00054781],[-0.0015315,0.0006018,0.00068664],[-0.0017576,8.3382e-05,-5.0288e-05],[-0.0015917,-0.00031579,0.00052381],[-0.001694,-0.00042564,0.00045461],[-0.0017661,-0.00062464,0.00011027],[-0.0018018,-0.00066676,-0.00016035],[-0.0017339,0.00033937,-3.0589e-06],[-0.001646,0.00022142,0.00029405],[-0.0010974,-0.00043541,-0.00053733],[-0.0016022,0.00044665,-5.5164e-05],[-0.0015777,0.00039154,-0.00032386],[-0.0015962,0.00020234,-0.00050599],[-0.0016261,6.9798e-05,-0.00051605],[0.00037194,0.00023117,-0.0015455],[0.00049353,0.00027836,-0.0014403],[0.00059505,0.00026772,-0.0013273],[0.00084062,0.00045328,-0.00067571],[0.00056443,0.00014494,-0.0011914],[0.00022686,0.00032057,-0.0015938],[-0.00025174,0.00033309,-0.0017078],[-0.00020097,0.00023766,-0.0011586],[-0.00047777,-0.0001539,-0.0013334],[-0.00010067,-0.0009024,-0.0010849],[0.00056548,-0.0014382,-0.00059307],[-0.00029295,-0.00061585,-0.0011854],[0.00094726,-0.0013814,0.00031648],[0.00081407,-0.00062709,0.0010593],[0.00070385,0.0010046,0.0013655],[0.00073357,9.8427e-05,0.0014767],[0.00077088,-0.00021325,0.0012123],[-0.00014037,0.0010291,0.0014678],[-0.00086127,0.0016144,0.00089571],[-0.00020641,0.0010927,-0.00018443],[0.00010336,0.0014229,0.0001409],[0.00015438,0.0015357,2.6667e-06],[0.00014026,0.0016644,4.4644e-05],[0.00011591,0.0017324,0.00021179],[1.0344e-05,0.0015932,0.00080153],[-0.00011566,0.0014061,0.0010315],[4.5026e-05,0.0016929,0.00061353],[-8.0601e-05,0.0013489,0.001011],[-0.00018487,0.0011618,0.0011737],[-0.00017469,0.00079412,0.00096306],[-0.0004952,0.0010086,0.0011416],[-0.00096571,0.0009149,0.0012563],[-0.00128,0.0006507,0.0012133],[3.9783e-05,0.0017257,0.00048744],[-0.00026961,-0.00011855,-0.0015924],[-0.00027075,-0.00016078,-0.0014716],[-0.00049666,-2.1177e-05,-0.0016576],[-0.0016552,-1.7875e-05,-0.00070561],[-0.0014259,-0.00037311,-0.0010868],[-0.0008111,-0.00021708,-0.0018077],[-0.0010445,-0.00027713,-0.0020759],[-0.00049644,-0.00037164,-0.0017437],[-0.00086905,-0.00024721,-0.001344],[-0.0013868,-0.00010352,-0.0011578],[-0.001343,-0.00037598,-0.0012297],[-0.0016348,-4.0581e-05,-0.00057587],[-0.00041494,4.5341e-05,-0.0015836],[-0.00086225,0.00018899,-0.0015786],[-0.00069941,0.0010674,-0.0013518],[-0.00080476,0.00074997,-0.001508],[-0.0008436,0.00060658,-0.0014598],[-0.00072664,0.00064373,-0.0014775],[-0.00052206,0.00017571,-0.0017529],[-0.00043373,0.0002668,-0.0017909],[-0.0016273,-0.0010264,-0.0005418],[-0.0015746,-0.00096176,-0.00044873],[-0.0016147,0.00045415,0.00081739],[-0.0016023,0.00048919,0.0005603],[-0.0016023,0.00040229,0.00075188],[-0.0016817,0.00025868,-9.3342e-05],[-0.0017137,7.4882e-05,0.0004369],[-0.0017983,-6.798e-05,0.00033771],[-0.0018813,-0.0001099,-9.5271e-05],[-0.0018637,-0.00020475,-0.00033509],[-0.0017272,0.0004378,-2.8049e-05],[-0.001415,-0.000884,-0.00046588],[-0.0016572,0.00045717,-0.00042023],[-0.001527,0.00045498,-0.00036986],[-0.0015114,0.00038655,-0.00062467],[-0.0015347,0.00032486,-0.00066554],[0.00028093,0.00013389,-0.001601],[0.00038951,0.00016881,-0.0015477],[0.00051538,0.0001814,-0.001431],[0.00047737,8.4513e-05,-0.00098704],[0.00084065,0.00045328,-0.00067573],[0.0017401,0.00084788,-0.0014158],[0.00056443,0.00014494,-0.0011914],[-5.4036e-08,6.7834e-05,-0.0016713],[-8.2264e-05,0.00032449,-0.0017585],[-0.00042352,-0.0001609,-0.0013854],[-0.00057997,-0.00034909,-0.0012974],[-0.00038136,-0.0006604,-0.0012568],[-5.1171e-05,-0.00096113,-0.0011638],[0.00054098,-0.001497,-0.00061589],[-0.00059393,-0.00035679,-0.0013255],[0.0009177,-0.0014204,0.00031259],[0.00090455,-0.00073388,0.0011978],[0.001145,0.00012184,0.0012971],[0.00020623,0.00096058,0.0015921],[4.9518e-05,0.001108,0.0014628],[0.0005654,-0.00019361,0.0012513],[-0.00075469,0.001413,0.00088746],[-0.0010913,0.0015176,0.00085646],[-0.00080719,0.0014008,0.000894],[-0.00020641,0.0010926,-0.00018443],[-0.00019448,0.0011941,2.1474e-05],[6.1475e-05,0.0015613,-4.3055e-05],[7.4408e-05,0.0016484,1.2979e-05],[6.4337e-05,0.0017291,0.00018542],[4.2415e-06,0.0016808,0.00063894],[-0.00012857,0.0015418,0.00087966],[-8.4142e-05,0.0013825,0.00098943],[-0.00018849,0.0012076,0.0010089],[-0.00017725,0.00088753,0.00093081],[-0.00044114,0.00074211,0.00099446],[-0.0008471,0.00089542,0.0011013],[-0.001211,0.00077427,0.0011397],[-7.2942e-05,0.0017246,0.00042589],[-0.00052394,-0.0001415,-0.0015788],[-0.00036402,-6.4798e-06,-0.001764],[-0.0016262,-2.9337e-05,-0.00068533],[-0.0015531,-3.6311e-05,-0.00096355],[-0.00067805,5.2997e-05,-0.0017914],[-0.00076453,0.00012447,-0.0019533],[-0.00092883,3.4353e-05,-0.0013774],[-0.0012795,1.6943e-05,-0.0011401],[-0.0012778,-0.00015004,-0.0013316],[-0.0015989,-4.6163e-05,-0.0006832],[-0.00074272,5.0369e-05,-0.0015195],[-0.0003881,0.00019869,-0.0017495],[-0.00059628,0.0011211,-0.0014603],[-0.00084681,0.00083313,-0.0016875],[-0.00077537,0.00057984,-0.0014087],[-0.00070778,0.00063902,-0.001474],[-0.00075498,0.00017603,-0.0016715],[-0.0006515,0.00026426,-0.0017268],[-0.0014778,-0.0010195,-0.00047472],[-0.0015646,-0.00096104,-0.00041325],[-0.00161,0.00052988,0.00080358],[-0.0016921,0.00064107,0.00016806],[-0.0017421,0.00058934,0.00026236],[-0.0016597,0.00010722,0.00045576],[-0.0017052,0.00041197,4.4742e-05],[-0.0017233,7.2527e-05,0.00045035],[-0.0018418,6.948e-06,1.1165e-05],[-0.0018547,-8.1287e-05,-0.00021366],[-0.0017795,-0.00014336,-0.0005384],[-0.001225,-0.00093996,-0.00075189],[-0.0017247,0.00025494,-0.00025087],[-0.0015082,0.00044549,-0.00059299],[-0.0015109,0.00038646,-0.00061862],[-0.0014393,0.00030875,-0.00084988],[0.00015939,0.00016988,-0.0016399],[0.00028217,0.00020074,-0.0015594],[0.00038593,0.00022022,-0.0014474],[0.00055354,0.00010629,-0.0011713],[0.00046038,7.6165e-05,-0.00097052],[0.0018853,6.2908e-05,-0.00033907],[0.0002123,0.0012509,-0.0014825],[-5.3899e-05,8.029e-05,-0.0017108],[-0.0002765,9.9655e-05,-0.0018202],[-0.00081151,0.00030191,-0.0014343],[-0.00093354,8.2907e-05,-0.0012748],[-0.0003863,-0.00066972,-0.0012663],[-5.8365e-05,-0.00099805,-0.0011837],[0.00057957,-0.0014669,-0.00067588],[-0.00061403,-0.00039137,-0.0013245],[0.00094423,-0.0014511,0.00031361],[0.00093548,-0.00079803,0.0013356],[0.0010639,-0.00030344,0.0015481],[0.00031734,0.0012942,0.0013287],[-9.1609e-06,0.00091276,0.0016755],[0.00067127,0.00042494,0.00096585],[-0.00076315,0.0014463,0.00089025],[-0.0012646,0.0012944,-0.00043105],[-0.001298,0.0012688,-0.00029683],[-0.00011437,0.0016279,-0.0001521],[5.0313e-05,0.0014881,-0.00012639],[6.8713e-05,0.0016095,-4.7482e-05],[5.6592e-05,0.0017155,2.6438e-05],[-7.669e-05,0.0017467,0.0002809],[-0.0001206,0.0016835,0.00056869],[-0.00021544,0.0015512,0.00083074],[-0.00030171,0.0014257,0.00089178],[-0.00048646,0.0012535,0.00089361],[-0.00050891,0.00089887,0.00082836],[-0.00055929,0.00086928,0.00088003],[-0.00088409,0.00092721,0.0010382],[-0.0012334,0.00079775,0.0010707],[-0.00012559,0.0017304,0.00046334],[-0.0007133,0.00013851,-0.0015539],[-0.0004335,6.724e-05,-0.0018295],[-0.0014172,0.00025207,-0.0010727],[-0.00076542,0.00011989,-0.0018889],[-0.00086043,0.0001896,-0.0020878],[-0.0011628,0.00017737,-0.0019267],[-0.0013079,-0.00015087,-0.0012776],[-0.0013782,0.0002525,-0.0011351],[-0.0014692,0.00030818,-0.00087176],[-0.0004142,0.00019761,-0.0018394],[-0.00084603,0.00089963,-0.001575],[-0.00096624,0.00072662,-0.0017455],[-0.00071372,0.00067484,-0.0014095],[-0.00084087,0.00043437,-0.0014684],[-0.00066891,0.00031583,-0.0016828],[-0.0007031,0.00023641,-0.0017992],[-0.0014068,-0.00090719,-0.00074507],[-0.0017599,-0.00028507,-0.00069041],[-0.0016239,0.00054831,0.0007551],[-0.0017542,0.00053006,0.00020398],[-0.0018359,0.00018684,0.00037067],[-0.0016825,0.00046316,0.00037009],[-0.0017252,0.00046478,3.2952e-05],[-0.0017606,0.00041403,0.00035397],[-0.0018404,0.0002398,-7.2481e-05],[-0.0018173,0.00012427,-0.00028258],[-0.0017125,-1.6248e-05,-0.00055817],[-0.001219,-0.00095179,-0.00071814],[-0.0015766,0.00021246,-0.0006958],[-0.0014668,0.00039314,-0.00054606],[-0.001492,0.00033132,-0.00057539],[-0.0014227,0.00027952,-0.00082158],[0.00017707,0.0001001,-0.0016301],[-2.4077e-05,-1.9607e-05,-0.0016813],[0.00030489,0.00014418,-0.0015945],[0.00038855,-3.9512e-05,-0.0012588],[0.00092156,-0.00031993,-0.00072347],[0.0018945,6.5283e-05,-0.00032968],[0.0014603,0.00057093,-0.00012068],[0.0015582,0.0005193,-0.00066184],[-0.00041349,0.00032186,-0.0017966],[-0.00015949,-3.6289e-05,-0.0017179],[-0.00021223,9.448e-05,-0.0018606],[-0.00087366,8.7988e-05,-0.0013364],[-0.00096453,9.7224e-05,-0.001328],[-0.00065114,-0.00040316,-0.0013566],[-0.00036514,-0.00070546,-0.001346],[3.9834e-06,-0.0010383,-0.0012449],[0.00048318,-0.0015249,-0.00069522],[-0.001015,9.7335e-05,-0.0013294],[0.00087606,-0.0016083,0.00031156],[0.00083123,-0.0011507,0.0014968],[0.0013351,-0.00025906,0.0012978],[0.0002419,0.00099818,0.0016344],[0.00041619,0.0013315,0.0013053],[-0.00083326,0.0014796,0.00092506],[0.00017663,0.00055817,0.0014952],[-0.0012961,0.001326,-0.00021916],[-0.0013185,0.0012167,-0.00029924],[-0.00070905,0.0013557,-0.00021105],[-0.001313,0.0012092,-0.00021365],[-0.00012179,0.0015053,-0.00067205],[2.9362e-05,0.0015821,-0.0002259],[3.2887e-05,0.0016357,-0.00015175],[4.3664e-05,0.0016759,-6.4607e-05],[-2.7504e-05,0.0017439,-1.993e-05],[-0.00015858,0.0017298,0.00023241],[-0.0002336,0.0016732,0.00064556],[-0.00031651,0.0014958,0.00088316],[-0.00052031,0.0013355,0.0010287],[-0.00062225,0.0011002,0.00096514],[-0.00081449,0.00096113,0.0009643],[-0.0011528,0.00087066,0.00099606],[-0.00020942,0.0016903,0.0004044],[-0.00037347,0.00015411,-0.0017546],[-0.0013931,0.00030164,-0.0010189],[-0.00066696,0.00019528,-0.0017667],[-0.00069812,0.00023879,-0.0018201],[-0.00091659,0.00013375,-0.0015248],[-0.0012115,0.00013151,-0.0014026],[-0.0012986,0.00034379,-0.0011393],[-0.00054598,0.00093067,-0.0017438],[-0.00060455,0.00051579,-0.0012367],[-0.00069369,0.00064239,-0.0013335],[-0.00076968,0.00042919,-0.0014905],[-0.00069224,0.00031931,-0.0016941],[-0.0016202,-0.00045544,-0.00096798],[-0.000966,-0.00067703,-0.00055661],[-0.0016981,-0.0002533,-0.0007476],[-0.0016197,0.00060798,0.00074302],[-0.0017491,0.00055457,0.00010892],[-0.0018945,0.00030635,2.2396e-05],[-0.0017127,0.00039269,0.00030515],[-0.0016849,0.00045807,0.00039483],[-0.0017578,0.00046116,0.00014041],[-0.001802,0.000254,-0.00015638],[-0.0017529,0.00014954,-0.0004027],[-0.0016172,1.2604e-05,-0.00064764],[-0.00091386,-0.00066193,-0.00047908],[-0.0015495,-1.7647e-05,-0.00046348],[-0.001505,0.00040319,-0.00054272],[-0.0013962,0.00031086,-0.00078433],[-0.0013104,0.00026363,-0.00098662],[-5.0285e-05,0.00017577,-0.0016722],[-0.00017003,1.7689e-05,-0.0016848],[5.2845e-05,0.00023318,-0.0016477],[0.00037471,-1.6114e-05,-0.0014249],[0.00041153,-0.00082161,-0.0010755],[0.0010692,-0.00088945,-0.00077356],[0.0015399,-0.00014166,-0.0001196],[0.0014942,0.00045034,-2.7353e-06],[0.00080816,0.0012673,-0.00032651],[-0.00081455,0.0011547,-0.0011528],[-0.00022712,-2.2737e-05,-0.0017232],[-0.00031714,-2.3426e-05,-0.0018805],[-0.0011651,0.00056333,-0.0010993],[-0.0012063,0.00052048,-0.0011219],[-0.00066119,-0.00039834,-0.0013998],[-0.00037252,-0.00072139,-0.0013492],[-1.6708e-05,-0.0011052,-0.0012443],[0.00052765,-0.0015465,-0.00076127],[-0.0010568,0.00011894,-0.0014016],[0.00095095,-0.0016145,0.00024644],[0.0010329,-0.0011702,0.0013532],[0.0013953,-0.00044451,0.0014897],[0.00019176,0.00084308,0.0018672],[0.00036946,0.0011829,0.0013744],[-0.00084924,0.0015432,0.0009296],[0.00035348,0.0011799,0.00113],[-0.0013215,0.0013124,-0.00017837],[-0.0011263,0.00057688,-0.0011543],[-0.00066572,0.0010314,-0.0010716],[-0.0012233,0.00052187,-0.0011191],[0.00050407,0.0014764,-0.000264],[0.00012585,0.0016317,-0.00017547],[4.0697e-05,0.0016435,-0.00015666],[-2.7225e-05,0.0017217,-2.5202e-05],[-0.00016117,0.0017532,5.9262e-05],[-0.00021539,0.0017309,0.00026802],[-0.00033945,0.0015936,0.00054732],[-0.00048926,0.0014516,0.00074947],[-0.00077433,0.0014501,0.0010006],[-0.00080875,0.0011636,0.00097228],[-0.00085761,0.001012,0.00099549],[-0.0011371,0.00085666,0.001007],[-0.00034917,0.0016398,0.00048689],[-0.0013429,0.0003019,-0.00099005],[-0.0007002,0.00031191,-0.0016993],[-0.00085489,0.00031361,-0.0016336],[-0.0012273,0.0003483,-0.0012478],[-0.0012536,0.00029742,-0.0011259],[-0.00093692,0.00050492,-0.0018139],[-0.00066229,0.00056408,-0.0011854],[-0.00060527,0.00052826,-0.001249],[-0.00077373,0.00052786,-0.001333],[-0.00072838,0.0004954,-0.0014983],[-0.0016463,-0.00051322,-0.00081768],[-0.00083142,-0.00048369,-0.00069794],[-0.0016172,-8.3642e-05,-0.00075531],[-0.0016393,0.00064146,0.00065158],[-0.0018128,0.00042512,0.00015005],[-0.0019258,0.00010468,7.5924e-05],[-0.0016982,0.00068816,0.00022443],[-0.0016792,0.00071207,0.00032726],[-0.0016949,0.00062785,7.7511e-05],[-0.0017088,0.00043405,-0.00021677],[-0.001685,0.00021871,-0.00041386],[-0.0016514,0.00010227,-0.00069092],[-0.00083141,-0.00048366,-0.00069791],[-0.001277,-8.6104e-05,-0.00083632],[-0.0013941,0.00050974,-0.00059871],[-0.0012719,0.00044659,-0.00083494],[-0.0012146,0.00037643,-0.0010163],[-0.00017941,5.2672e-05,-0.0016779],[-4.2904e-05,0.00014781,-0.0016589],[-0.00022193,-3.2977e-05,-0.0017074],[0.00017694,-0.00021308,-0.0015879],[0.00081861,-0.0011169,-0.00073552],[0.0010691,-0.00072069,-0.0001581],[0.0015516,-0.00013003,-4.7193e-05],[0.0015273,0.0004833,0.00015519],[0.00099776,0.0010111,0.00022585],[0.0010277,0.00098748,-3.5961e-05],[0.00075725,0.0012059,-0.00025095],[-0.00036191,-0.0001763,-0.0017028],[-0.00042955,-2.6843e-06,-0.0017553],[-0.0012704,0.00054089,-0.0011774],[-0.00065637,-0.00041005,-0.0014492],[-0.0010818,0.00012176,-0.0014348],[-0.00034965,-0.00073569,-0.0013847],[-1.5268e-06,-0.0011483,-0.0012966],[0.00041169,-0.0016173,-0.00078536],[-0.0012754,0.00054629,-0.001193],[0.00072318,-0.0018227,0.00022407],[0.0010433,-0.0011659,0.001356],[0.0014848,-0.00039196,0.0012477],[0.00052339,0.00089952,0.0016924],[-0.00086316,0.0016234,0.00096912],[0.00052699,0.0010709,0.0010782],[-0.0013485,0.0014139,-0.00018749],[0.00020012,0.00094552,0.00089678],[0.00019762,0.00087173,0.00070167],[0.00047299,0.0014248,-0.000127],[0.00011774,0.0015879,-9.1399e-05],[2.3146e-05,0.0016686,-0.00012896],[2.459e-05,0.0016664,-0.00016863],[-3.4925e-05,0.0017321,-2.9778e-05],[-0.00017551,0.0017548,5.1228e-05],[-0.00026498,0.0016998,0.00023383],[-0.00051727,0.0015432,0.00064986],[-0.0008268,0.0015521,0.0010091],[-0.00085329,0.0012276,0.0010278],[-0.0010918,0.0010114,0.00096859],[-0.00036061,0.0015885,0.00045978],[-0.0008456,0.00042781,-0.001462],[-0.0011797,0.00044736,-0.0012776],[-0.0011558,0.00041121,-0.0011369],[-0.00072349,0.00039936,-0.0014702],[-0.00066311,0.00055878,-0.0011816],[-0.00068875,0.00055682,-0.0012699],[-0.00071382,0.0005285,-0.0013857],[-0.00079706,0.00050094,-0.0014702],[-0.0015476,2.3916e-05,-0.00084839],[-0.0013729,-0.00081395,-0.0011527],[-0.0014714,-2.5046e-05,-0.0008709],[-0.0016018,0.00078845,0.00060578],[-0.0018713,0.00047935,1.1301e-05],[-0.0020042,0.00014423,-2.3276e-05],[-0.0017052,0.0006763,0.00029266],[-0.0016503,0.0007297,0.0001721],[-0.0016541,0.0006346,-3.1542e-05],[-0.0016685,0.00042612,-0.00022275],[-0.0015991,0.00023376,-0.00052317],[-0.0014715,0.00014793,-0.00089982],[-0.001082,-0.00080565,-0.0012718],[-0.0011483,0.00012536,-0.00095369],[-0.0012418,0.00046877,-0.00079479],[-0.0011977,0.000435,-0.00095823],[-0.0011079,0.00036029,-0.0011174],[-0.0003817,0.0001121,-0.0016539],[-0.00036819,0.00025964,-0.0016262],[-0.00047897,3.69e-05,-0.0016426],[-0.00044881,3.6347e-05,-0.0016113],[-0.00050894,-0.0011868,-0.00090657],[0.00060151,-0.0012605,-0.00044405],[0.0012171,-0.00059651,-0.00028682],[0.0014481,-0.00018235,-4.2754e-06],[0.0015329,0.00033006,0.00026518],[0.0012348,0.00086731,0.00046647],[0.00060481,0.0013536,0.00014272],[-0.00052125,-0.00014555,-0.0016825],[-0.00061648,-0.00023601,-0.001673],[-0.00063449,-0.00035104,-0.0015],[-0.001086,0.0001676,-0.0014875],[-0.0003483,-0.00073217,-0.0014088],[-9.4693e-06,-0.0011831,-0.0013077],[0.00046939,-0.0015755,-0.00087311],[-0.0013238,0.00060344,-0.0012609],[0.00089268,-0.0017387,-3.1624e-06],[0.0010788,-0.0011302,0.0012378],[0.0017129,-0.0004351,0.0014257],[0.00046568,0.00071218,0.0018249],[-0.00088831,0.0016734,0.00099645],[0.00048513,0.00097974,0.001059],[-0.0013853,0.00145,-0.00018989],[0.00024991,0.0011231,0.00086326],[0.00019605,0.00086528,0.00069813],[0.00064799,0.0013951,1.3117e-05],[0.00037489,0.0017306,6.3515e-05],[-0.00010824,0.0016369,-0.00020501],[-3.1595e-05,0.0017343,-0.00014743],[-0.00017711,0.0017723,4.3802e-05],[-0.00027723,0.0017706,0.00010842],[-0.00037674,0.0016494,0.00029958],[-0.00046904,0.0013946,0.0005864],[-0.00052471,0.00097689,0.00063206],[-0.0011493,0.0012362,0.00094406],[-0.0011271,0.0010502,0.0009283],[-0.00048882,0.0014522,0.00052066],[-0.0011136,0.00041419,-0.0012135],[-0.00089599,0.00022397,-0.0014756],[-0.00092596,0.00063788,-0.0011425],[-0.00070473,0.00056502,-0.0012951],[-0.00075699,0.00047828,-0.0013926],[-0.0015977,-4.8877e-05,-0.00069948],[-0.0012776,-0.00071209,-0.00122],[-0.0014063,0.00013029,-0.00087318],[-0.0015142,0.00072926,0.00063598],[-0.001965,0.0002531,8.45e-05],[-0.0019501,0.00015461,-2.6489e-05],[-0.0017082,0.00068378,0.00029155],[-0.001594,0.00068334,0.00017257],[-0.0016349,0.00068791,-5.1119e-05],[-0.0014884,0.00057314,-0.00026421],[-0.00089827,8.1827e-05,-0.00027928],[-0.0017915,0.00017739,-0.0010944],[-0.0010309,-0.00069102,-0.0014138],[-0.0011626,0.00013851,-0.00088964],[-0.0012071,0.00046695,-0.00078109],[-0.001077,0.00056015,-0.00099044],[-0.0010188,0.00045962,-0.0011297],[-0.00040535,0.00022372,-0.0016175],[-0.00047695,2.3114e-05,-0.0016501],[-0.00031617,0.00013237,-0.0015291],[-0.00055507,-2.7927e-05,-0.0016704],[-0.00011286,-0.0008343,-0.0013834],[-0.00012353,-0.0016741,-0.0004415],[0.0007218,-0.0014254,-0.00040644],[0.0013112,-0.00062494,-0.00024584],[0.0015975,0.00033552,0.00021698],[0.0014928,-0.00020388,-8.249e-05],[0.0012288,0.00086677,0.00048105],[0.0009191,0.0010735,0.00054662],[0.00095827,0.0011037,0.00052596],[-0.00072945,-0.0004025,-0.0015814],[-0.00080702,-0.00020167,-0.0016048],[-0.0010366,0.00017157,-0.001556],[-0.00066527,-0.00035535,-0.0015119],[-0.0013092,0.00061234,-0.0012987],[-0.00035746,-0.00074977,-0.0014424],[5.2063e-06,-0.0012039,-0.0013343],[0.00034408,-0.0016371,-0.00089247],[0.0006042,-0.0018833,-2.8626e-05],[0.0011911,-0.00080186,0.0010806],[0.0016173,-0.00041073,0.0013457],[0.00080489,0.00075811,0.0016128],[-0.00079586,0.0017377,0.00099353],[-0.0014218,0.0014989,-0.00019586],[0.00049823,0.00098874,0.0010627],[0.00073808,-0.00014932,0.00068924],[0.00052405,0.0012012,0.00088149],[0.00027936,0.0011238,0.00090129],[0.00034907,0.001665,0.00014524],[-0.00014325,0.0016891,0.00029304],[-0.00015537,0.0016881,-0.00019252],[-0.00015491,0.0017018,-0.00022177],[-0.00032475,0.0017799,-4.4313e-05],[-0.00037973,0.00177,5.2542e-05],[-0.00046371,0.0016305,0.00024954],[-0.00054632,0.0010226,0.00054158],[-0.00080506,0.00095451,0.00067282],[-0.0011668,0.0012515,0.00099282],[-0.00059763,0.0014618,0.00047747],[-0.0010408,0.00050161,-0.0012155],[-0.0012508,0.00026339,-0.0014142],[-0.00094118,0.00068754,-0.0011909],[-0.00088603,0.00059005,-0.001188],[-0.0012659,-0.00041431,-0.0011696],[-0.0012795,1.5997e-05,-0.00056152],[-0.0009504,-0.00081773,-0.001366],[-0.0013835,5.2164e-05,-0.00056513],[-0.001402,0.000798,0.00056339],[-0.0020934,0.00031444,-5.8154e-05],[-0.0013822,-6.8881e-05,0.00039803],[-0.0016517,0.00068242,0.00015989],[-0.0016197,0.00072772,-1.7103e-05],[-0.0015402,0.00067168,-0.00021899],[-0.0014286,0.00055646,-0.00028892],[-0.0008373,0.00010841,-0.00047654],[-0.0016465,0.00015771,-0.00096261],[-0.0011938,-0.00072091,-0.0014747],[-0.0010301,0.00038602,-0.0010299],[-0.0008678,0.00032584,-0.00089097],[-0.0009869,0.00053174,-0.0010132],[-0.0010045,0.00046656,-0.0011909],[-0.00070776,0.00031957,-0.0015163],[-0.00074481,0.00010451,-0.0015313],[-0.00053179,0.00021943,-0.0015684],[-0.00067577,9.471e-06,-0.0016023],[-0.00093755,-0.00052204,-0.0012909],[-0.001088,-0.0012968,-0.0002793],[-9.4477e-05,-0.0016198,-0.0004752],[0.00080578,-0.0013891,-0.00052301],[0.0012965,-0.00063181,-0.00023459],[0.00159,0.00020324,0.00030697],[0.0014912,-0.00029782,-1.6894e-05],[0.0012666,0.00082208,0.00053317],[0.00097072,0.0010343,0.00060687],[0.00063106,0.0011859,0.00055172],[-0.0008219,-0.00036427,-0.0014937],[-0.00098804,-0.00046648,-0.0014477],[-0.0010676,0.00012376,-0.0015427],[-0.00067633,-0.00037871,-0.0014997],[-0.0013182,0.00064006,-0.0013239],[-0.00036282,-0.00076103,-0.0014588],[2.9398e-05,-0.0011662,-0.0013742],[0.0004235,-0.0015231,-0.0010113],[0.00080715,-0.0017628,-0.00033116],[0.0010379,-0.00075382,0.0012228],[0.0012201,-0.00016589,0.00095633],[0.00064784,0.00032209,0.0017119],[-0.00078135,0.0017875,0.00095345],[-0.0013741,0.001547,-0.00029419],[0.00045733,0.00088662,0.00109],[0.0007182,-0.00015604,0.0007628],[0.00040519,0.00084851,0.00097583],[0.00026619,0.0010686,0.00083979],[0.00077257,0.0013812,0.00056371],[-0.00055863,0.0017926,9.098e-05],[-0.000283,0.0015189,-0.00026396],[-0.00031128,0.0017513,-0.00016228],[-0.00039407,0.001841,-1.8361e-05],[-0.00050661,0.0017664,0.00012142],[-0.00062769,0.001526,0.00035258],[-0.00072395,0.0012077,0.00061691],[-0.00080534,0.00095653,0.00065473],[-0.00074508,0.0012439,0.00056327],[-0.0010993,0.00053435,-0.0015678],[-0.0008895,0.00061542,-0.0010159],[-0.001014,-0.00020627,-0.0013051],[-0.00095001,0.00010404,-0.0006175],[-0.00076541,-0.00056558,-0.0015397],[-0.0013029,0.00028385,-0.00059674],[-0.0012904,0.00069703,0.00069664],[-0.0020762,-0.00078834,0.0002973],[-0.0013257,0.00031894,0.00028233],[-0.0012521,-3.1542e-05,0.00043405],[-0.0016171,-0.00035962,0.00049179],[-0.0011837,0.00011191,0.00010921],[-0.0017071,0.00060921,-0.0001874],[-0.0017136,0.00076591,-0.0004059],[-0.0015523,0.00073427,-0.0006341],[-0.0015128,0.00034189,-0.00103],[-0.0010484,-0.00050877,-0.0015807],[-0.00086919,0.00032152,-0.00088796],[-0.00078228,0.00025776,-0.0015098],[-0.00065102,0.00010411,-0.0015254],[-0.00071045,0.0001541,-0.001577],[-0.00047567,-8.3522e-05,-0.0015908],[-0.00088217,-0.0002078,-0.0015069],[-0.00074852,-0.0013796,-0.001144],[-0.0012335,-0.0012584,-0.00042506],[-0.00010425,-0.0017425,-0.00051585],[0.00092197,-0.0014469,-0.00035777],[0.0013645,-0.00064296,-0.00016964],[0.0016602,0.0001961,0.00017673],[0.0012939,0.00084029,0.00054723],[0.0015401,-0.00031048,-3.2832e-05],[0.00096997,0.001032,0.00060183],[0.00066583,0.0011975,0.00066549],[0.00069393,0.0012043,0.00061675],[-0.00095516,-0.0005119,-0.0014606],[-0.0010651,0.00012526,-0.0015671],[-0.0013124,0.0006422,-0.0013345],[-0.00065613,-0.00038421,-0.0015311],[-0.00028671,-0.00077487,-0.0015107],[1.8955e-05,-0.0011845,-0.001393],[0.00028662,-0.0015799,-0.0010285],[0.00093225,-0.0023431,-0.00044109],[0.0010387,0.00075321,0.00073697],[0.0011012,-0.0001688,0.0010568],[0.0010709,0.00034742,0.001312],[-0.00082996,0.001885,0.0010075],[-0.0013907,0.0015726,-0.0002988],[0.00055747,0.00095803,0.0011393],[0.00097525,-0.0001012,0.00098039],[0.00028813,0.0012154,0.0009603],[-0.0009042,0.0017493,0.00059651],[-0.00054037,0.00174,0.00026942],[-0.00028242,0.0014372,-2.5597e-05],[-0.00043958,0.0015714,-0.00022482],[-0.00046429,0.0016721,-0.00026594],[-0.00062218,0.0020714,-0.0001229],[-0.00053173,0.001645,8.3306e-05],[-0.00069663,0.0015055,0.00031353],[-0.001123,0.00088834,0.00056152],[-0.00093201,0.0011983,0.00047575],[-0.0010507,-1.2794e-05,-0.0012733],[-0.00086377,-0.0003217,-0.0015591],[-0.0010354,0.00025578,-0.00060309],[-0.0015057,-0.00057644,0.00025449],[-0.001322,0.00028093,0.00052932],[-0.0012436,0.00015073,0.00037137],[-0.0012578,-0.00015145,-9.6767e-05],[-0.0011837,0.00011191,0.00010921],[-0.0017991,6.605e-05,-0.00060472],[-0.0015712,0.0005753,-0.00044266],[-0.0018545,0.00060839,-0.00055286],[-0.0016456,0.00057688,-0.00097729],[-0.0013355,0.00031364,-0.0011019],[-0.0009075,0.00029546,-0.0014286],[-0.00091416,0.00018979,-0.0014576],[-0.0007011,0.00015215,-0.0015541],[-0.0010013,0.00014506,-0.0014368],[-0.00075097,-0.00023347,-0.0015098],[-0.0016829,-0.00067187,-0.00079704],[-0.0012758,-0.0012389,-0.00027157],[-0.0015575,-0.00055045,-0.00011199],[-0.00016476,-0.0018165,-0.0004241],[0.00097135,-0.0014676,-0.00040577],[0.0013928,-0.0006797,-0.0001573],[0.0017092,0.00011651,0.00024439],[0.0013556,0.00085315,0.0005885],[0.0015715,-0.0003207,-3.0668e-05],[0.0010056,0.0010791,0.00062072],[0.00068462,0.0012536,0.00068157],[-0.00077902,0.0015502,0.00076674],[-0.00099055,0.00021071,-0.0015677],[-0.0012528,0.00070367,-0.00134],[-0.00060063,-0.00027467,-0.0015775],[-0.00026461,-0.00073379,-0.0015737],[6.3986e-05,-0.0010825,-0.0014349],[0.00034891,-0.0014353,-0.0011213],[0.00081053,-0.00204,-0.00038182],[0.0011306,0.00058149,0.0015219],[0.0011016,-0.00015345,0.0010485],[0.00088824,-3.903e-05,0.0013265],[-0.00078981,0.0019465,0.00091886],[-0.0012905,0.0016009,-0.00043055],[0.00055319,0.00090936,0.0012973],[0.00092383,-0.00010281,0.0010749],[0.0003085,0.0012985,0.0010119],[-0.00071613,0.0014785,0.00069843],[-0.00054981,0.00057548,-1.3207e-05],[-0.00052239,0.00055315,-0.00017039],[-0.00058595,0.0013883,-0.0003453],[-0.00051245,0.0018225,-0.00028773],[-0.00072353,0.0021725,-9.9628e-05],[-0.00073481,0.0015707,0.00020336],[-0.0009422,0.0012114,0.00049863],[-0.0011084,0.00086938,0.00062021],[-0.00096241,7.5205e-05,-0.0013416],[-0.0013047,0.00053692,-0.00093097],[-0.00159,-0.00023085,0.00011132],[-0.0012327,0.00063818,0.00040125],[-0.0014379,0.00017404,0.00043419],[-0.0015658,-0.00031059,-4.9127e-05],[-0.0012488,-0.00023541,-0.00016303],[-0.0015831,0.00024208,-0.00068434],[-0.0016216,0.00063774,-0.0004847],[-0.0016507,0.00064512,-0.00043169],[-0.0015247,0.00051685,-0.000891],[-0.0012065,0.00049565,-0.0012006],[-0.0007616,0.0004437,-0.0015023],[-0.00087569,0.00019856,-0.0014241],[-0.00088826,0.00020917,-0.0014045],[-0.00086855,2.6121e-05,-0.0015556],[-0.00097536,7.1351e-05,-0.001426],[-0.0013671,-0.00061109,-0.00060898],[-0.0014151,-0.00065674,-0.001014],[-0.0013444,-0.0012698,-0.00031052],[-0.0015828,-0.00024743,-0.00032179],[-0.00024816,-0.0017962,-0.00057707],[0.00098679,-0.0014809,-0.00039362],[0.0014266,-0.00070314,-0.0001888],[0.0017456,0.00010822,0.00013493],[0.001425,0.00088157,0.00053295],[0.0010034,0.001092,0.00066699],[0.0016196,-0.00033936,-8.7002e-05],[0.00069455,0.0012755,0.00069828],[-0.00074006,0.0015761,0.00082331],[-0.00072302,0.0014838,0.00081199],[-0.00091319,0.00020569,-0.0015633],[-0.0011826,0.00069899,-0.0013744],[-0.00058267,-0.00027198,-0.0015657],[-0.00023529,-0.00074284,-0.0016048],[-3.1618e-05,-0.0011155,-0.0014505],[0.00010511,-0.0016402,-0.001232],[0.00073276,-0.0020396,-0.00036902],[0.00043096,0.0013767,0.0010156],[0.0011227,0.00068403,0.0018188],[0.001102,-0.00015408,0.0010548],[0.0010874,-2.1924e-05,0.0011539],[-0.00084906,0.0018823,0.00091929],[-0.0013689,0.0015782,-0.00043055],[0.00069198,0.00094551,0.0012684],[0.0010528,5.9078e-06,0.0011279],[-0.0014387,0.00058919,0.00026833],[-0.001406,0.00098218,-0.00033414],[-0.00054844,0.001326,-0.00041532],[-0.00069303,0.0015029,-0.00036107],[-0.00072532,0.0016201,-0.00048871],[-0.00046631,0.0013531,-6.795e-05],[-0.00080099,0.0015272,0.00015546],[-0.0011359,0.0010254,0.00032303],[-0.0010496,0.00049391,-0.0012259],[-0.0015098,-0.00021135,2.1021e-05],[-0.0012878,0.00066238,0.00048728],[-0.0014056,0.00046554,0.00031813],[-0.0016484,-0.00019642,-0.00031455],[-0.00167,0.00025434,-0.00074691],[-0.0015104,-0.00026529,-0.0010144],[-0.0014683,0.00077616,-0.00051357],[-0.0014429,0.00078357,-0.00069733],[-0.0012148,0.00043168,-0.0012451],[-0.0010547,0.00045585,-0.0012968],[-0.001068,0.00052876,-0.0012765],[-0.0012046,0.00029552,-0.0012236],[-0.0012855,0.00036596,-0.0011033],[-0.0012387,0.00018823,-0.0010404],[-0.001527,-0.00017983,-0.00039664],[-0.001394,5.1652e-05,-0.00060091],[-0.0016871,-0.00018466,0.0001344],[-0.0013835,-0.00133,-0.00039133],[-0.0014009,-0.00046388,-0.00040874],[-0.00030233,-0.00185,-0.00049509],[0.0010024,-0.0014759,-0.00041464],[0.0014535,-0.00069955,-0.00020421],[0.0017853,0.00014525,0.00011126],[0.001457,0.00093263,0.00052574],[0.0010166,0.001162,0.00065554],[0.0016451,-0.00033613,-9.4826e-05],[0.00068896,0.0013522,0.00067868],[-0.00072282,0.0015836,0.00089846],[-0.001535,0.00056823,0.00035774],[-0.00089098,0.00019444,-0.0015179],[-0.00116,0.00072326,-0.0013775],[-0.00057265,-0.00026138,-0.0015521],[-0.00024823,-0.0007673,-0.0015865],[-8.5422e-06,-0.0010394,-0.0014458],[0.00015031,-0.001452,-0.0012721],[0.00080106,-0.0017329,-0.00081819],[0.00040033,0.0012976,0.0012821],[0.0010588,0.00079677,0.0014363],[0.001126,-0.00018809,0.0010981],[0.0010305,-0.00020301,0.0012088],[-0.00075929,0.0019463,0.00074165],[-0.0012246,0.0015889,-0.00058897],[0.00065854,0.00082825,0.0014289],[0.0009747,4.1689e-06,0.0012064],[-0.0014281,0.00058788,0.00029849],[-0.001366,0.00096304,3.986e-05],[-0.0009765,0.0011725,-0.00041757],[-0.00092577,0.0011703,-0.00083564],[-0.0010475,0.0010157,-0.00079311],[-0.00087802,0.0016556,-0.00047153],[-0.00073779,0.0014045,-0.00042363],[-0.00066036,0.0012445,4.0148e-05],[-0.0011655,0.0010525,0.00043934],[-0.0016656,-4.0256e-05,-8.323e-05],[-0.0014651,0.00048529,0.0003307],[-0.0016965,-3.872e-05,-0.00021292],[-0.0015715,0.00022592,-0.0006912],[-0.0015812,-0.0005451,-0.00080357],[-0.0015385,0.00021427,-0.00083524],[-0.0013123,0.00080701,-0.00069638],[-0.0011605,0.0004437,-0.0012239],[-0.0011948,0.00045535,-0.0011443],[-0.0012189,0.00024534,-0.0010839],[-0.0012438,0.00023774,-0.0010283],[-0.0014849,-0.00018698,-0.00044902],[-0.0013908,-0.00041669,-0.00019675],[-0.0013763,2.0579e-05,-0.00090707],[-0.0017636,9.1299e-05,-7.9212e-05],[-0.0014206,-0.0013951,-0.00038024],[-0.0013053,-0.00059811,-0.00028701],[-0.00026248,-0.0018904,-0.00041947],[0.00105,-0.0015184,-0.00037951],[0.0015021,-0.00073032,-0.00024333],[0.0018152,0.0001443,7.5395e-05],[0.0014944,0.0009348,0.00041209],[0.0010639,0.0011913,0.00061103],[0.00067999,0.0013745,0.0007402],[0.001693,-0.00035194,-0.00013954],[-0.00069849,0.0017131,0.00084099],[-0.0016954,0.00062995,0.00045052],[-0.0015929,0.00059375,0.0004221],[-0.0011132,0.00070536,-0.0013574],[-0.00084395,0.00018904,-0.0014961],[-0.00056128,-0.00025981,-0.0015446],[-0.00025851,-0.00077491,-0.0015993],[-0.00018759,-0.0010803,-0.0014442],[0.00012663,-0.0013516,-0.0011798],[0.00019703,-0.0015279,-0.00053007],[0.00053401,0.0013602,0.0012605],[0.00076854,0.00084682,0.0013789],[0.0015169,0.00080495,0.0010834],[0.0011298,-0.00019079,0.0011],[0.0011294,-0.00018594,0.0010791],[-0.00088028,0.001837,0.00075439],[-0.001303,0.0014768,-0.00056454],[0.0010437,9.423e-05,0.0012063],[-0.0014464,-8.6263e-05,-5.1121e-07],[-0.00144,0.00023234,-0.00034563],[-0.0012725,0.00057445,-0.00093034],[-0.0010571,0.0010175,-0.00074967],[-0.0012569,0.00081706,-0.00067077],[-0.0011175,0.00074153,-0.00094632],[-0.0010638,0.0012836,-0.0003166],[-0.0010861,0.0013033,-0.00026443],[-0.0014096,0.0007863,0.00020216],[-0.0016866,0.00023996,-0.00038467],[-0.001459,-0.00057426,-0.0010566],[-0.0017005,0.00024194,-0.0003874],[-0.0014251,0.00022942,-0.0008027],[-0.0013272,0.00019808,-0.00095711],[-0.0011065,0.00074028,-0.0010402],[-0.0013968,0.00032186,-0.0008026],[-0.0012572,0.00024496,-0.00099159],[-0.0014896,0.00025524,-0.00022674],[-0.001463,-9.4239e-05,-4.9212e-05],[-0.0012654,0.00058203,-0.00057446],[-0.0012867,-0.00061208,-0.00038327],[-0.0019861,0.00012525,0.00011055],[-0.0014412,-0.0013731,-0.00027307],[-0.0014486,-0.0002746,-0.00014913],[-0.00024327,-0.0019275,-0.00048008],[0.0010904,-0.0014744,-0.00045032],[0.0015332,-0.0007298,-0.00025941],[0.0018348,0.00017898,4.9549e-05],[0.001506,0.00095234,0.00040856],[0.0010567,0.0012379,0.000584],[0.00063646,0.0014417,0.0006692],[0.0016981,-0.00034929,-0.00014275],[-0.00069327,0.001727,0.00088449],[-0.001782,0.00066189,0.00046756],[-0.0014378,-0.00024259,4.5141e-05],[-0.0010921,0.00068179,-0.0013236],[-0.00081445,0.00017257,-0.0014324],[-0.00055235,-0.00025457,-0.0015227],[-0.00026531,-0.00078509,-0.0015652],[-0.00016504,-0.0010171,-0.0014528],[8.6827e-05,-0.0013408,-0.00097214],[0.00019473,-0.0011974,-0.00087438],[0.00048884,0.0012012,0.0014291],[0.00072767,0.00072063,0.0014923],[0.0013586,0.00038588,0.0013343],[0.0011662,-0.00026995,0.0011817],[0.0011833,-0.00016791,0.0011109],[-0.000787,0.0018817,0.00057577],[-0.0011749,0.0014688,-0.00068467],[0.00098467,9.1163e-05,0.0012673],[-0.001212,0.00051157,-0.00085874],[-0.0011602,0.0004848,-0.00098935],[-0.0013347,0.00015527,-0.00091021],[-0.0011918,0.00082692,-0.0010394],[-0.0013679,0.00090689,-0.00076909],[-0.0014375,0.00080112,3.3233e-05],[-0.0011453,-3.4707e-05,-0.0013505],[-0.0013989,5.5753e-05,-0.00084704],[-0.0013667,5.5035e-05,-0.00084118],[-0.0013608,4.8647e-05,-0.00087427],[-0.0012426,-0.00063231,-0.00034892],[-0.0019352,0.00035818,-8.8756e-05],[-0.0014983,-0.001259,-0.00043251],[-0.0014077,-0.00046226,-2.5935e-05],[-0.0001369,-0.0020566,-0.00028193],[0.0011549,-0.001525,-0.00039712],[0.0015535,-0.00076388,-0.00038076],[0.0018669,0.00018019,2.9986e-05],[0.0015516,0.00096626,0.00033139],[0.0010824,0.0012411,0.00051825],[0.00065159,0.0014644,0.00066688],[-0.00066275,0.0018325,0.00081649],[0.0016928,-0.00036298,-0.00024917],[-0.0018803,0.00074951,0.00045541],[-0.001041,0.00066527,-0.0013126],[-0.00076774,0.0001681,-0.0014303],[-0.00057568,-0.00025309,-0.0015108],[-0.00036455,-0.00079755,-0.0015566],[-0.00040909,-0.0011872,-0.0016188],[4.4165e-05,-0.0020167,-0.0011748],[0.00084606,0.00073819,0.0014244],[0.00064365,0.0012349,0.001343],[0.0015338,0.00038983,0.0011082],[0.0012179,-0.0002575,0.0010737],[-0.00092375,0.00177,0.00059452],[0.0011409,-0.00016999,0.0011478],[-0.001272,0.0013485,-0.00065795],[0.0010186,0.00013785,0.0012678],[-0.0013689,0.00017674,-0.00080479],[-0.0012408,-0.00064968,-0.0004363],[-0.0013542,-0.000488,-0.00028682],[-0.0011599,0.00021482,-0.00011335],[-0.0015092,-0.0011102,-0.00010857],[-0.00116,0.00021483,-0.00011335],[-2.3869e-05,-0.0020156,-0.00049694],[0.0012327,-0.0013873,-0.00056064],[0.0015536,-0.00076498,-0.00038007],[0.0018596,0.00020531,8.8968e-06],[0.0015587,0.00095908,0.00034093],[0.0010797,0.001255,0.00050926],[0.00063209,0.0014789,0.00063707],[-0.00060289,0.0017512,0.00088367],[0.0016687,-0.00035173,-0.00025006],[-0.0019147,0.00076056,0.00034699],[-0.0010296,0.00062143,-0.0012695],[-0.00076819,0.00013868,-0.0013969],[-0.00058022,-0.0002815,-0.0014659],[-0.00036391,-0.00079602,-0.0015097],[-0.00038524,-0.0011171,-0.0015581],[4.7151e-05,-0.0017358,-0.0012716],[0.00079883,0.00062542,0.0014666],[0.0005963,0.0010572,0.0014547],[0.0015166,0.00028113,0.0011961],[0.0012616,-0.00032476,0.0011515],[-0.00080861,0.0017803,0.00038613],[0.0011769,-0.00012227,0.0011431],[-0.0011382,0.0013275,-0.00076588],[0.0010035,0.00013681,0.0012743],[-0.0012863,-0.00087311,-0.00033926],[-0.0013376,-0.00057011,-0.00023335],[-0.00114,0.00010009,-1.5123e-05],[-0.0016414,-0.00072185,-0.00059731],[-0.001177,0.00020799,-0.00010635],[2.8329e-05,-0.0022163,-0.00043696],[0.0013149,-0.0014191,-0.00043219],[0.0015733,-0.0007742,-0.00038216],[0.0018248,0.0001957,-6.1204e-05],[0.0015759,0.00095834,0.0002746],[0.0011062,0.0012622,0.0004537],[0.0006494,0.0014775,0.00059265],[-0.00052972,0.0018449,0.00074654],[-0.0019757,0.00081188,0.0003368],[0.0016559,-0.00035014,-0.00025826],[-0.00097866,0.00060913,-0.0012728],[-0.00078373,0.00013866,-0.0013637],[-0.0006529,-0.0002825,-0.0014604],[-0.00053211,-0.00082404,-0.001506],[-0.00056872,-0.0010752,-0.0013803],[-0.00017446,-0.0014156,-0.00083957],[0.00090254,0.00063643,0.0013897],[0.00085697,0.0011043,0.001286],[0.0017667,0.00026341,0.00062682],[0.0012632,-0.00031523,0.0010817],[-0.00090915,0.0016691,0.00039785],[-0.0012389,0.0012033,-0.00073697],[0.0011151,-0.00012155,0.0011658],[0.0010058,0.00014643,0.0012681],[-0.00102,-0.00074614,-0.00058198],[-0.0012526,-0.00052456,-0.0001698],[-0.0013794,8.6079e-05,-0.00020404],[-0.0015835,0.00021678,-0.0003303],[-0.0018257,-0.00054752,-5.1521e-06],[0.00013609,-0.0021282,-0.00061576],[0.0013549,-0.0012788,-0.00055011],[0.0015598,-0.00072977,-0.00040338],[0.0018081,0.0001866,-5.476e-05],[0.0015779,0.00095403,0.0002789],[0.0011132,0.0012471,0.00046763],[0.00065336,0.0014685,0.0005997],[-0.00042803,0.0017404,0.00091258],[-0.0017699,0.00072334,0.00079641],[0.0016537,-0.00032462,-0.00027533],[-0.00099807,0.00054167,-0.0012361],[-0.00081325,0.00010245,-0.0013693],[-0.00068399,-0.0003228,-0.0014813],[-0.00051632,-0.00079043,-0.0015017],[-0.00053749,-0.0010123,-0.0013145],[-0.00022451,-0.0013508,-0.0011326],[0.00087414,0.0005723,0.0014029],[0.00081836,0.00091755,0.0014225],[0.0018757,0.00018956,0.00073882],[0.0013687,-0.00029757,0.0011403],[-0.0008154,0.0016467,0.00023884],[-0.0011181,0.0011773,-0.00082598],[0.0011601,-2.6766e-05,0.0011328],[0.0010532,0.00014782,0.0012185],[-0.0010767,-0.0010098,-0.00047672],[-0.0013939,-0.00056301,-0.00020052],[-0.0013776,-8.6573e-05,-6.3635e-05],[-0.0015882,8.541e-05,-0.00022069],[-0.0018222,-6.91e-05,-0.00046325],[0.00012623,-0.0021599,-0.00065359],[0.0014648,-0.0012744,-0.00020211],[0.0015802,-0.00072088,-0.0002662],[0.0015591,0.00093601,0.00023112],[0.0017699,0.000181,-7.4401e-05],[0.0011355,0.0012463,0.00039759],[0.00067473,0.0014759,0.0005568],[-0.00033739,0.0018274,0.00073947],[-0.0016995,0.0010475,0.00050783],[0.0016442,-0.00031297,-0.00016803],[-0.00098596,0.00053086,-0.0012028],[-0.00087805,0.00010796,-0.0013325],[-0.00078975,-0.00031905,-0.0014397],[-0.00072363,-0.00073823,-0.0012897],[-0.00088619,-0.0014723,-0.0018368],[0.0017484,0.0001461,0.00010344],[0.00098025,0.00056691,0.0012564],[0.0011322,0.00089778,0.00099143],[0.0013029,-0.00028341,0.0010875],[-0.00093413,0.0015773,0.00025956],[-0.0011983,0.00099627,-0.0007679],[0.0011526,-2.6587e-05,0.0010869],[0.0010842,0.00020385,0.0012099],[-0.0015128,-0.00061215,-0.00023148],[-0.0011785,-0.0010956,-0.00049309],[-0.0015136,-0.00015121,-0.00029377],[-0.0016721,5.3406e-05,-0.00038713],[-0.0020109,1.6906e-05,-0.00012233],[8.7182e-05,-0.0021174,-0.00058502],[0.0015357,-0.0010357,-0.00039551],[0.0016095,-0.0006505,-0.00032324],[0.0015361,0.00095207,0.00020638],[0.0017372,0.00016402,-6.2619e-05],[0.0011061,0.0012788,0.00035439],[0.00064946,0.0014875,0.00052117],[-0.00030011,0.0017857,0.00079571],[-0.0016084,0.0010103,0.00073059],[0.0016551,-0.00027214,-0.00019696],[-0.00099884,0.00045562,-0.0011585],[-0.00092437,6.8206e-05,-0.0013583],[-0.00085598,-0.00040017,-0.0014779],[-0.00071189,-0.00072106,-0.0012799],[-0.00053303,-0.00075924,-0.0011556],[0.0018592,4.6206e-05,0.00019331],[0.00099495,0.00055436,0.0012963],[0.0011145,0.00064122,0.0011696],[0.0015544,2.3955e-05,0.0010047],[-0.00084199,0.0015146,0.00012457],[-0.0010996,0.00097223,-0.00083603],[0.0011713,0.00012968,0.00099138],[0.0011476,0.00020226,0.001126],[-0.0015298,-0.00063999,-0.00021992],[-0.0012268,-0.00099629,-0.00061583],[-0.0015687,-0.00026972,-0.00021544],[-0.0017245,-7.6218e-05,-0.00029204],[-0.0019997,0.00019002,-0.00026627],[-8.2442e-05,-0.0020107,-0.0010217],[0.0016367,-0.001071,-0.00020043],[0.0016152,-0.00063629,-0.00014594],[0.0015084,0.00094232,0.00024808],[0.0010958,0.0012639,0.0003426],[0.0016732,0.00016499,1.2595e-05],[0.00067822,0.0014874,0.00044661],[-0.00023234,0.0018338,0.00066071],[-0.0015865,0.0012241,0.00054822],[0.0016185,-0.00024958,1.1628e-05],[-0.0010503,0.00046559,-0.001143],[-0.0010595,8.3526e-05,-0.0013012],[-0.00099188,-0.00036632,-0.0013201],[-0.00061533,-0.00065333,-0.0011708],[-0.00039595,-0.00061455,-0.0011806],[0.0020463,0.00059422,0.00039068],[0.0012281,0.00058696,0.001199],[0.0010887,5.413e-05,0.00087465],[-0.00084342,0.0014837,0.00012581],[-0.0011508,0.00072572,-0.00074152],[0.0012015,0.00012981,0.00096748],[0.0011984,0.00030586,0.0011078],[-0.0015199,-0.00065107,-0.00043963],[-0.0013512,-0.0010754,-0.00058849],[-0.0016301,-0.00030722,-0.00038179],[-0.0018018,-8.7132e-05,-0.00035575],[-0.0021172,0.00019459,-0.00034318],[-0.00018233,-0.0019254,-0.00083422],[0.0016087,-0.00090102,-0.00027658],[0.001597,-0.00059028,-0.00016689],[0.0014978,0.00091046,0.00026331],[0.0010877,0.0012736,0.00033045],[0.0016204,0.00016656,7.5673e-06],[0.00066417,0.0014897,0.00042888],[-0.00018792,0.0017823,0.00072795],[-0.0015626,0.001226,0.00066865],[0.0016149,-0.00016431,-4.1848e-05],[-0.0011297,0.00035939,-0.0011427],[-0.0012449,-1.3852e-05,-0.0014346],[-0.001089,-0.00040484,-0.0014454],[-0.00059013,-0.00058096,-0.0013074],[-0.00047228,-0.0005422,-0.0012358],[0.0019841,0.00047521,0.00044241],[0.0011571,0.00060069,0.0010964],[0.0012166,0.0004444,0.00057286],[-0.00062664,0.0012565,-0.00071975],[-0.0011072,0.00071322,-0.00076507],[0.0012232,0.00061946,0.00071288],[0.0013464,0.00031252,0.0010089],[-0.0014947,-0.00087833,-0.00026491],[-0.0013623,-0.001019,-0.00063831],[-0.0016278,-0.00037094,-0.00033175],[-0.0017841,-8.8581e-05,-0.00035053],[-0.0019831,0.00019956,-0.00033334],[-0.00032123,-0.001653,-0.0011648],[0.0016281,-0.00090714,-0.00018089],[0.0015686,-0.00057186,-2.4134e-05],[0.0014632,0.00090056,0.0003212],[0.0010758,0.0012914,0.00042001],[0.00065112,0.0014678,0.00043116],[0.0015648,0.00017541,0.00012562],[-0.00011563,0.001803,0.0005766],[-0.0015589,0.0013524,0.00056534],[0.0015755,-0.00014652,9.9968e-05],[-0.001307,0.00039001,-0.0011021],[-0.0013059,-1.243e-06,-0.0013405],[-0.00082004,-0.00040457,-0.0014689],[-0.00033978,-0.00037262,-0.0011849],[0.0018935,0.00066793,0.00049746],[-0.0017659,0.0018525,-0.0006608],[0.0016976,0.00082562,0.00059874],[0.0014227,0.00096343,0.00062954],[-0.0014055,-0.00087136,-0.00056141],[-0.0013889,-0.0010413,-0.00066303],[-0.0016486,-0.00039335,-0.00049436],[-0.0018117,-9.7369e-05,-0.0004678],[-0.0020384,0.0002057,-0.00044779],[-0.00041865,-0.0016086,-0.0009806],[0.0014954,-0.00082018,-0.00017187],[0.0015773,-0.00048128,-7.8353e-05],[0.0014628,0.00084105,0.00035791],[0.0010728,0.0012603,0.0004315],[0.00064598,0.0014142,0.00043778],[0.0015521,0.0002067,0.0001041],[-4.2934e-05,0.0016587,0.00065699],[-0.0015567,0.0013778,0.00071185],[0.0016128,-0.00012442,8.6569e-05],[-0.001335,0.0010318,-0.0013901],[-0.001203,0.00051663,-0.0017488],[-0.00083856,-0.00029252,-0.0018622],[-0.00046704,-0.00019086,-0.0011673],[0.0014009,0.00069554,0.00025653],[-0.0015159,0.0011562,-0.0015386],[0.0014109,0.00083746,0.0003],[-0.0014016,-0.0011029,-0.00040771],[-0.0014379,-0.0010221,-0.00072471],[-0.0016461,-0.00044514,-0.00045576],[-0.0017665,-0.00012761,-0.00043363],[-0.0018823,0.00015394,-0.00039229],[-0.000478,-0.0013452,-0.0011259],[0.00117,-0.0006402,-0.00020449],[0.0016599,-0.00049504,8.4196e-05],[0.0014476,0.00083995,0.00039906],[0.0010284,0.0012476,0.0005289],[0.00063148,0.0014334,0.00050237],[1.0463e-05,0.0016497,0.00054077],[0.0015539,0.00022945,0.00026027],[-0.0014787,0.0014874,0.00054635],[0.0017123,-0.00011622,0.00020306],[-0.001202,0.00050407,-0.0015969],[-0.001261,-0.0010792,-0.00074133],[-0.0014077,-0.0010109,-0.00076743],[-0.0016252,-0.00045174,-0.00061318],[-0.0017709,-0.00012926,-0.00055172],[-0.0019029,0.000167,-0.00053881],[-0.00019721,-0.0011134,-0.0012348],[0.00095799,-0.00064433,-0.00013956],[0.0016604,-0.00042722,4.8356e-05],[0.0014609,0.00074052,0.00046397],[0.0010517,0.0011839,0.00057731],[0.00067592,0.0013368,0.00057486],[0.00017633,0.0014464,0.00076878],[0.0015603,0.00023862,0.00025662],[-0.001407,0.0014726,0.00082452],[0.0016969,-9.5691e-05,0.0001898],[-0.0013015,-0.001238,-0.00069259],[-0.0014282,-0.00096554,-0.00081995],[-0.0016224,-0.00052469,-0.00056119],[-0.0017621,-0.00018586,-0.00051124],[-0.0018516,0.00010051,-0.00049096],[-0.00014053,-0.0012279,-0.0010607],[0.00043396,-0.0006719,-0.0011099],[0.001467,-0.00049045,-0.00086914],[0.001021,0.0011782,0.00064906],[0.0014628,0.00074951,0.00051766],[0.0006552,0.0013369,0.00062655],[0.00026161,0.0014941,0.0006223],[-0.0013326,0.0015407,0.00070077],[0.0016825,0.00026754,0.00035687],[0.0018242,-0.00012318,6.0312e-05],[-0.0011677,-0.0011967,-0.00093109],[-0.0014136,-0.00095659,-0.00081747],[-0.0015792,-0.0005212,-0.00071146],[-0.0017565,-0.00018226,-0.00067244],[-0.0018009,0.00013517,-0.00079206],[0.00050315,-0.0014281,-0.001004],[0.0013135,-0.0012738,-0.00030836],[0.0011027,0.0010632,0.00077387],[0.001447,0.000634,0.00056812],[0.00074083,0.001204,0.00075078],[0.00036303,0.0013206,0.00075618],[-0.0012873,0.0015132,0.00081701],[0.0016257,0.00024339,0.00035331],[0.0019602,-7.7918e-05,2.5338e-05],[-0.0012052,-0.0012991,-0.00092203],[-0.0014034,-0.00089069,-0.00085149],[-0.001558,-0.00051963,-0.00069832],[-0.0017398,-0.00018389,-0.00066389],[-0.0017932,1.3338e-05,-0.0007252],[-0.0014144,0.00010958,-0.0013377],[-0.00047951,-0.00020546,-0.001759],[0.001131,0.001084,0.0007678],[0.00071056,0.0011907,0.00079909],[0.001527,0.00066238,0.0005426],[0.00035477,0.0013085,0.00075967],[-0.0012001,0.0016172,0.00066915],[0.0018984,0.00026396,0.00013581],[-0.00051854,-0.00029391,-0.0017646],[-0.0011174,-0.0012646,-0.0010432],[-0.001372,-0.00088283,-0.00091072],[-0.0015261,-0.00051975,-0.00085102],[-0.0016807,-0.00017233,-0.00083145],[-0.001634,-0.00042141,-0.0011861],[0.0013043,0.0011078,0.00094089],[0.00057822,0.0014029,0.00061447],[0.0015286,0.0005697,0.0005936],[0.00029364,0.0013246,0.00067054],[-0.001104,0.0015164,0.00075009],[0.0016901,0.00031364,7.5325e-05],[-0.00070607,-0.00017523,-0.0020346],[-0.00113,-0.0012859,-0.0010501],[-0.0012986,-0.00080839,-0.00087987],[-0.0014307,-0.00048811,-0.00079729],[-0.0016573,-0.00019059,-0.00080718],[-0.0016352,-0.00018964,-0.00099844],[0.00035385,0.0013155,0.00097978],[0.0012857,0.0010785,0.00085842],[0.00034934,0.0013482,0.00058156],[0.001591,0.00058686,0.00019239],[-0.001003,0.001562,0.00061138],[-0.00047109,0.00015641,-0.0017339],[-0.0016547,-0.00020348,-0.001004],[-0.0010805,-0.0012649,-0.0011072],[-0.001293,-0.00081916,-0.00096727],[-0.00143,-0.00049883,-0.00094098],[0.00016342,0.0017363,0.00081176],[0.0010599,0.0017788,0.00018226],[0.00016649,0.0013558,0.00035676],[0.00060189,0.0013597,-0.00070456],[-0.00081446,0.0012647,0.00048559],[-0.00040122,4.1206e-05,-0.0017276],[-0.0013957,-0.00047704,-0.0010749],[-0.0010207,-0.001193,-0.0010472],[-0.00117,-0.0007147,-0.00089207],[-0.0013408,-0.00044696,-0.0008934],[0.00044215,0.0019752,0.00030584],[3.2206e-05,0.001253,0.00060236],[0.00078171,0.00069185,-0.00079338],[-0.00066357,0.0012221,0.00031534],[-0.0014937,-2.8523e-05,-0.0013289],[-0.0010468,-0.0012432,-0.0011332],[-0.0011825,-0.00073931,-0.001013],[-0.00081455,-2.1015e-05,-0.0011167],[-0.00093334,-0.00012667,-0.00089727],[-0.0008544,-1.5763e-05,-0.0011631],[-0.00077982,-0.0001562,-0.0014862],[-0.0010816,-0.00048194,-0.0014265],[-0.0010715,-0.00045821,-0.0010196],[-0.0013106,-0.00040792,-0.0011442]])

        n_gt =  n_gt/np.tile(np.linalg.norm(n_gt,axis=1)[:,None],(1,3))

        self.assertTrue(np.isclose(n-n_gt,0.0,atol=1e-4).all())


if __name__ == '__main__':
    unittest.main()