# AUTOGENERATED! DO NOT EDIT! File to edit: notebooks/03_graph.ipynb (unless otherwise specified).

__all__ = ['plot_prediction']

# Cell

from typing import *
from pathlib import Path
import pandas as pd

import plotly
import plotly.graph_objs as go

# Internal Cell

from .testing import get_test_dataframe

import numpy as np
import re

# Internal Cell

my_dark_blue = "#003851"
my_bright_blue = "#5ec5ed"
my_red = "#c22828"
my_green = "#71ad3d"
my_white = "#eae4d9"

# Internal Cell

my_graph_color_scheme = dict(
    known_curve=my_dark_blue,
    predict_curv=my_bright_blue,
    break_even_point=my_red,
    known_point=my_green,
)

# Internal Cell

layout = go.Layout(
    # title=f"<B>Nanobit<B>",
    yaxis=dict(title="<i>(Revenues)</i>", color=my_dark_blue),
    xaxis=dict(title="<i>(Days)</i>", color=my_dark_blue),
    font=dict(color=my_dark_blue, size=12),
    plot_bgcolor=my_white,
)

# Internal Cell

days_real = list(range(30)) + list(range(30, 180 + 1, 30))
revenue_columns = [f"revenue_d{i}" for i in days_real]
days = list(range(len(days_real)))

# Internal Cell


def get_last_known_index(xs: np.ndarray) -> int:
    """
    Gets an index of the last known value.

    Args:
        xs (np.ndarray): Revenue values.

    Returns:
        int: Returns index of last known value.
    """
    x = np.where(xs[:-1] > xs[1:])[0]
    if len(x) > 0:
        return x[0]
    else:
        if (xs == 0).all():
            # all unknown
            return 0
        else:
            # all known
            return len(xs) - 1

# Internal Cell


def calculate_break_even_point(revenues: np.ndarray, spend: float) -> float:
    """
    Calculates the intersection of direction and curve and returns the point at which they intersect.

    Args:
        revenue_array (np.ndarray): Revenue values.
        spend (float): The parameter represents how much money was spent that day.

    Returns:
        Optional[float]: Returns intersection point if there is one.
    """

    i = np.argmax(revenues > spend)
    if i == 0:
        return 0 if revenues[0] > spend else None  # type: ignore

    y_1, y_2 = revenues[i - 1], revenues[i]
    delta = (spend - y_1) / (y_2 - y_1)

    return days[i - 1] + delta * (days[i] - days[i - 1])  # type: ignore

# Internal Cell


def get_plot_data(
    df_known: pd.DataFrame,
    df_pred: pd.DataFrame,
    last_n_days: int = 1,
    target_day: int = 0,
) -> Dict[str, Any]:
    """
    Gets and calculates the data needed to draw a graph.

    Args:
        df_known (pd.DataFrame): Previous campaign data.
        df_pred (pd.DataFrame): Predicted data.
        last_n_days (int): Determines how many previous days will be added and drawn together
            starting from target_day.
        target_day (int): Target day.

    Returns:
        Dict[str, Any]: Returns the data needed to draw a graph.
    """
    # agregagtion slice for indexing rows
    ix = slice(target_day - (last_n_days - 1), target_day + 1)

    # known data
    df_known = df_known.iloc[ix, :].sum()

    spend = df_known["spend"]
    known_revenues = df_known[revenue_columns].values

    known_index = get_last_known_index(known_revenues)
    known_revenues = np.maximum.accumulate(known_revenues)
    break_event_point = calculate_break_even_point(known_revenues, spend)

    # predictioons
    predicted_revenues = df_pred.iloc[ix, :].sum()[revenue_columns].values
    if break_event_point == None:
        break_event_point = calculate_break_even_point(predicted_revenues, spend)

    return dict(
        known_revenues=known_revenues,
        predicted_revenues=predicted_revenues,
        days=days,
        known_index=known_index,
        break_event_point=break_event_point,
        spend=spend,
    )

# Internal Cell


def plot_curve(
    x: List[int],
    y: List[float],
    ix: slice,
    color: str,
    name: str,
) -> List[go.Scatter]:
    """
    Creates curve.
    Args:
        x (List[int]): Values on the x axis.
        y (List[float]): Values on the y axis.
        ix (slice): Divides the graph into 2 parts
        color (str): Curve color.
        name (str): Curve name.

    Returns:
        List[go.Scatter]: Returns created curve.
    """

    x = x[: ix + 1], x[ix:]  # type:ignore
    y = y[: ix + 1], y[ix:]  # type:ignore
    dash = [None, "dash"]
    mode = ["lines+markers", "lines"]
    traces = [
        go.Scatter(
            x=x[i],
            y=y[i],
            mode=mode[i],
            name=name,
            marker=dict(size=12),
            line=dict(color=color, width=4, dash=dash[i]),
            opacity=0.75,
            showlegend=False if len(x[i]) <= 1 else None,  # type:ignore
        )
        for i in range(2)
    ]
    return traces

# Internal Cell


def get_plot_text(x, y):
    if x <= 30:
        text = f"({np.round(x, 2)}, {y})"
    else:
        x_text = days_real[int(x)] + (x - int(x)) * 30
        text = f"({np.round(x_text, 2)}, {y})"

    return text

# Internal Cell


def plot_point(
    x: float,
    y: float,
    color: str,
    name: str,
) -> List[go.Scatter]:
    """
    Creates point.

    Args:
        x (List[int]): Value on the x axis.
        y (List[float]): Value on the y axis.
        color (str): Point color.
        name (str): Point name.

    Returns:
        List[go.Scatter]: Returns created point.
    """

    x = [0, x, x]  # type: ignore
    y = [y, y, 0]  # type: ignore

    dash = "dash"
    traces = [
        go.Scatter(
            x=x,
            y=y,
            mode="lines",
            name=name,
            line=dict(color=color, width=2, dash="dot"),
            opacity=0.75,
            hoverinfo="skip",
            showlegend=False,
        ),
        go.Scatter(
            x=x[1:2],  # type: ignore
            y=y[1:2],  # type: ignore
            mode="markers",
            name=name,
            marker=dict(size=15, color=color),
            opacity=0.75,
            text=get_plot_text(x[1], y[1]),  # type: ignore
            hoverinfo="text+name",
        ),
    ]
    return traces

# Internal Cell


def plot_graph(
    plot_data: Dict[str, Any], color_scheme: Dict[str, str]
) -> List[go.Scatter]:
    """
    Creates graph.

    Args:
        plot_data (Dict[str, Any]): The data needed to draw the graph.
        color_scheme (Dict[str, str]): Color scheme to color the graph.

    Returns:
        List[go.Scatter]: Returns created graph.
    """

    x = days
    y = plot_data["known_revenues"]
    ix = plot_data["known_index"]
    traces = plot_curve(x, y, ix, color_scheme["known_curve"], name="known curve")

    y = plot_data["predicted_revenues"]
    ix = 0
    traces = traces + plot_curve(
        x, y, ix, color_scheme["predict_curv"], name="predictions"
    )

    y = plot_data["spend"]
    x = plot_data["break_event_point"]
    if x is None:
        x = days[-1]
    traces = traces + plot_point(
        x, y, color_scheme["break_even_point"], name="break even"  # type:ignore
    )

    y = plot_data["known_revenues"][-1]
    x = plot_data["known_index"]

    traces = traces + plot_point(
        x, y, color_scheme["known_point"], name="known"  # type:ignore
    )

    return traces

# Internal Cell


def get_date_from_df(
    dfs: Union[pd.DataFrame, List[pd.DataFrame]], last_n_days: int, target_day: int
):
    if isinstance(dfs, list):
        dfs = dfs[-1]
    date_tuple = (
        str(dfs.iloc[target_day - last_n_days + 1]["date"].date().strftime("%d.%m.%Y")),
        str(dfs.iloc[target_day]["date"].date().strftime("%d.%m.%Y")),
    )

    if last_n_days == 1:
        return date_tuple[0]
    else:
        return f"{date_tuple[0]} - {date_tuple[1]}"

# Cell


def plot_prediction(
    df_known: Union[List[pd.DataFrame], pd.DataFrame],
    df_pred: Union[List[pd.DataFrame], pd.DataFrame],
    last_n_days: int = 1,
    target_day: int = 0,
    color_scheme: Union[List[Dict[str, str]], Dict[str, str]] = my_graph_color_scheme,
    title: str = None,
    height: Optional[int] = None,
    width: Optional[int] = None,
) -> go.Figure:
    """
    The function draws graphs from known data and predicted data.
    If the function with the values last_n_days = 1 and target_day = 29 is called,
        a graph will be drawn for the 29th day of the campaign.
    If the function with the values last_n_days = 3 and target_day = 29 is called,
        the values from the 29th, 28th and 27th days will be summed and a graph will be drawn.

    Args:
        df_known (pd.DataFrame): Previous campaign data.
        df_pred (pd.DataFrame): Predicted data.
        last_n_days (int): Determines how many previous days will be added and drawn together
            starting from target_day.
        target_day (int): Target day.
        color_scheme (Dict[str, str]): Color scheme used to color the graph.
        title (str): Graph title.
        height optional(int): Height of the figure.
        width optional(int): Width of the figure.

    Returns:
        (plotly.graph_objs.Figure): Returns a graph on which known data and predictions are drawn.
    """
    if title is None:
        title = "Capt'n AI predictions"

    if not isinstance(df_known, list):
        df_known = [df_known]
        df_pred = [df_pred]
    if not isinstance(color_scheme, list):
        color_scheme = [color_scheme] * len(df_known)

    campaign = df_known[0]["campaign"].iloc[0]
    adgroup = df_known[0]["adgroup"].iloc[0]

    traces = []  # type:ignore
    for dfk, dfp, cs in zip(df_known, df_pred, color_scheme):
        plot_data = get_plot_data(
            dfk, dfp, last_n_days=last_n_days, target_day=target_day
        )
        traces = traces + plot_graph(plot_data, cs)

    fig = go.Figure(data=traces, layout=layout)

    fig.update_layout(
        title={"text": title, "x": 0.15, "y": 0.93, "xanchor": "left"},
        title_font_size=25,
        height=height,
        width=width,
    )
    fig.update_xaxes(
        type="linear",
        tickmode="array",
        ticktext=days_real,
        tickvals=days,
        rangemode="tozero",
    )
    fig.update_yaxes(rangemode="tozero")

    dates = get_date_from_df(dfk, last_n_days, target_day)
    text = f"{campaign}/{dates}"
    for i, text in enumerate(
        [f"Campaign: {campaign}", f"Ad group: {adgroup}", f"Date: {dates}"]
    ):
        fig.add_annotation(
            dict(
                font=dict(color=my_dark_blue, size=12),
                x=0.99,
                y=0.01 + 0.06 * (2 - i),
                showarrow=False,
                text=text,
                xanchor="right",
                xref="paper",
                yref="paper",
            )
        )

    fig.add_layout_image(
        dict(
            source="https://nanobit.api.captn.ai/assets/images/company_logo.png",
            xref="paper",
            yref="paper",
            x=0.02,
            y=1.08,
            sizex=0.15,
            sizey=0.15,
            xanchor="right",
            yanchor="bottom",
        )
    )

    return fig