"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IntegTesting = void 0;
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const ifw = require(".");
class IntegTesting {
    constructor() {
        const app = new cdk.App();
        const env = {
            region: process.env.CDK_INTEG_REGION || process.env.CDK_DEFAULT_REGION || 'us-east-1',
            account: process.env.CDK_INTEG_ACCOUNT || process.env.CDK_DEFAULT_ACCOUNT,
        };
        const stack = new cdk.Stack(app, 'integ-stack', { env });
        const databaseName = 'FleetWise';
        const tableName = 'FleetWise';
        const database = new aws_cdk_lib_1.aws_timestream.CfnDatabase(stack, 'Database', {
            databaseName,
        });
        const table = new aws_cdk_lib_1.aws_timestream.CfnTable(stack, 'Table', {
            databaseName,
            tableName,
        });
        table.node.addDependency(database);
        const role = new aws_cdk_lib_1.aws_iam.Role(stack, 'Role', {
            roleName: 'iotfleetwise-role',
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('iotfleetwise.amazonaws.com'),
        });
        role.addToPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
            actions: [
                'timestream:WriteRecords',
                'timestream:Select',
            ],
            resources: ['*'],
        }));
        role.addToPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
            actions: [
                'timestream:DescribeEndpoints',
            ],
            resources: ['*'],
        }));
        const signalCatalog = new ifw.SignalCatalog(stack, 'SignalCatalog', {
            database,
            table,
            role,
            description: 'my signal catalog',
            nodes: [
                new ifw.SignalCatalogBranch('Vehicle', 'Vehicle'),
                new ifw.SignalCatalogSensor('EngineTorque', 'Vehicle.EngineTorque', 'DOUBLE'),
            ],
        });
        const model_a = new ifw.VehicleModel(stack, 'ModelA', {
            signalCatalog,
            name: 'modelA',
            description: 'Model A vehicle',
            networkInterfaces: [new ifw.CanVehicleInterface('1', 'vcan0')],
            signals: [
                new ifw.CanVehicleSignal('EngineTorque', 'Vehicle.EngineTorque', '1', 401, // messageId
                1.0, // factor
                true, // isBigEndian
                false, // isSigned
                8, // length
                0.0, // offset
                9),
            ],
        });
        const vin100 = new ifw.Vehicle(stack, 'vin100', {
            vehicleId: 'vin100',
            vehicleModel: model_a,
            createIotThing: true,
        });
        const vpc = aws_cdk_lib_1.aws_ec2.Vpc.fromLookup(stack, 'VPC', { isDefault: true });
        const securityGroup = new aws_cdk_lib_1.aws_ec2.SecurityGroup(stack, 'SecurityGroup', {
            vpc,
            allowAllOutbound: true,
        });
        securityGroup.addIngressRule(aws_cdk_lib_1.aws_ec2.Peer.anyIpv4(), aws_cdk_lib_1.aws_ec2.Port.tcp(22), 'SSH access');
        securityGroup.addIngressRule(aws_cdk_lib_1.aws_ec2.Peer.anyIpv4(), aws_cdk_lib_1.aws_ec2.Port.allIcmp(), 'ping');
        // EC2 role
        const ec2_role = new aws_cdk_lib_1.aws_iam.Role(stack, 'ec2Role', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('ec2.amazonaws.com'),
            managedPolicies: [
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
            ],
        });
        ec2_role.addToPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
            actions: [
                's3:List*',
                's3:Get*',
            ],
            resources: ['arn:aws:s3:::*'],
        }));
        // Ubuntu 18.04 for Arm64
        const machineImage = aws_cdk_lib_1.aws_ec2.MachineImage.fromSsmParameter('/aws/service/canonical/ubuntu/server/18.04/stable/current/arm64/hvm/ebs-gp2/ami-id', { os: aws_cdk_lib_1.aws_ec2.OperatingSystemType.LINUX });
        // Create the Vehicle simulator
        const keyName = stack.node.tryGetContext('key_name');
        const instance = new aws_cdk_lib_1.aws_ec2.Instance(stack, 'VehicleSim', {
            vpc: vpc,
            instanceType: new aws_cdk_lib_1.aws_ec2.InstanceType('m6g.xlarge'),
            machineImage,
            securityGroup,
            keyName,
            role: ec2_role,
            vpcSubnets: {
                subnetGroupName: 'Public',
            },
            resourceSignalTimeout: cdk.Duration.minutes(30),
        });
        const sourceUrl = 'https://github.com/aws/aws-iot-fleetwise-edge.git';
        const userData = `\
    set -euo pipefail

    # Wait for any existing package install to finish
    i=0
    while true; do
        if sudo fuser /var/{lib/{dpkg,apt/lists},cache/apt/archives}/lock >/dev/null 2>&1; then
            i=0
        else
            i=\`expr $i + 1\`
            if expr $i \\>= 10 > /dev/null; then
                break
            fi
        fi
        sleep 1
    done

    # Upgrade system and reboot if required
    apt update && apt upgrade -y
    if [ -f /var/run/reboot-required ]; then
      # Delete the UserData info file so that we run again after reboot
      rm -f /var/lib/cloud/instances/*/sem/config_scripts_user
      reboot
      exit
    fi

    # Install helper scripts:
    apt update && apt install -y python3-setuptools
    mkdir -p /opt/aws/bin
    wget https://s3.amazonaws.com/cloudformation-examples/aws-cfn-bootstrap-py3-latest.tar.gz
    python3 -m easy_install --script-dir /opt/aws/bin aws-cfn-bootstrap-py3-latest.tar.gz
    rm aws-cfn-bootstrap-py3-latest.tar.gz

    # On error, signal back to cfn:
    error_handler() {
      /opt/aws/bin/cfn-signal --success false --stack ${stack.stackName} --resource ${instance.instance.logicalId} --region ${stack.region}
    }
    trap error_handler ERR

    # Increase pid_max:
    echo 1048576 > /proc/sys/kernel/pid_max
    # Disable syslog:
    systemctl stop syslog.socket rsyslog.service
    # Remove journald rate limiting and set max size:
    printf "RateLimitBurst=0\nSystemMaxUse=1G\n" >> /etc/systemd/journald.conf

    # Install packages
    apt update && apt install -y git ec2-instance-connect htop jq unzip

    # Install AWS CLI:
    curl "https://awscli.amazonaws.com/awscli-exe-linux-aarch64.zip" -o "awscliv2.zip"
    unzip awscliv2.zip
    ./aws/install
    rm awscliv2.zip

    # Download source
    cd /home/ubuntu
    if echo ${sourceUrl} | grep -q 's3://'; then
      sudo -u ubuntu aws s3 cp ${sourceUrl} aws-iot-fleetwise-edge.zip
      sudo -u ubuntu unzip aws-iot-fleetwise-edge.zip -d aws-iot-fleetwise-edge
    else
      sudo -u ubuntu git clone ${sourceUrl} aws-iot-fleetwise-edge
    fi
    cd aws-iot-fleetwise-edge
    
    # Install SocketCAN modules:
    ./tools/install-socketcan.sh --bus-count 1
    
    # Install CAN Simulator
    ./tools/install-cansim.sh --bus-count 1
    
    # Install FWE credentials and config file
    mkdir /etc/aws-iot-fleetwise
    echo -n "${vin100.certificatePem}" > /etc/aws-iot-fleetwise/certificate.pem
    echo -n "${vin100.privateKey}" > /etc/aws-iot-fleetwise/private-key.key
    ./tools/configure-fwe.sh \
      --input-config-file "configuration/static-config.json" \
      --output-config-file "/etc/aws-iot-fleetwise/config-0.json" \
      --vehicle-id vin100 \
      --endpoint-url "${vin100.endpointAddress}" \
      --topic-prefix '$aws/iotfleetwise/' \
      --can-bus0 "vcan0"

    # Install source deps
    ./tools/install-deps-native.sh

    # Build source
    sudo -u ubuntu ./tools/build-fwe-native.sh

    # Install FWE
    ./tools/install-fwe.sh
    
    # Signal init complete:
    /opt/aws/bin/cfn-signal --stack ${stack.stackName} --resource ${instance.instance.logicalId} --region ${stack.region}`;
        instance.addUserData(userData);
        new cdk.CfnOutput(stack, 'Vehicle Sim ssh command', { value: `ssh -i ${keyName}.pem ubuntu@${instance.instancePublicIp}` });
        new ifw.Campaign(stack, 'Campaign', {
            name: 'FwTimeBasedCampaign',
            target: vin100,
            collectionScheme: new ifw.TimeBasedCollectionScheme(cdk.Duration.seconds(10)),
            signals: [
                new ifw.CampaignSignal('Vehicle.EngineTorque'),
            ],
            autoApprove: true,
        });
        new ifw.Fleet(stack, 'Fleet', {
            fleetId: 'fleet',
            signalCatalog,
            vehicles: [vin100],
        });
        this.stack = [stack];
    }
}
exports.IntegTesting = IntegTesting;
new IntegTesting();
//# sourceMappingURL=data:application/json;base64,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