import os
from dataclasses import dataclass


@dataclass
class Config:
    broker_url: str
    result_backend: str


class ConfigBuilder:
    """Logging service configuration"""

    broker_url: str = "amqp://rabbitmq:rabbitmq@localhost:5672/"
    result_backend: str = "redis://localhost:6379"

    @classmethod
    def __build_config(cls) -> Config:
        return Config(
            broker_url=cls.broker_url,
            result_backend=cls.result_backend,
        )

    @classmethod
    def local(cls) -> Config:
        return cls.__build_config()

    @classmethod
    def from_env(cls) -> Config:
        amqp_broker = os.getenv("AMQP_BROKER")
        amqp_port = os.getenv("AMQP_PORT")
        amqp_user = os.getenv("AMQP_USER")
        amqp_pass = os.getenv("AMQP_PASSWORD")

        redis_host = os.getenv("REDIS_HOST")
        redis_port = os.getenv("REDIS_PORT")

        if None in [
            amqp_broker,
            amqp_port,
            amqp_user,
            amqp_pass,
            redis_host,
            redis_port,
        ]:
            return cls.local()

        cls.broker_url = f"amqp://{amqp_user}:{amqp_pass}@{amqp_broker}:{amqp_port}/"
        cls.result_backend = f"redis://{redis_host}:{redis_port}"

        return cls.__build_config()
