"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RecaptchaAuthorizer = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const apigateway = require("aws-cdk-lib/aws-apigateway");
const lambda = require("aws-cdk-lib/aws-lambda");
const lambdaNodeJS = require("aws-cdk-lib/aws-lambda-nodejs");
const MIN_SCORE_THRESHOLD = 0.0;
const MAX_SCORE_THRESHOLD = 1.0;
const DEFAULT_SCORE_THRESHOLD = 0.5;
/**
 * Request-based lambda authorizer that authorizes requests using Google's reCaptcha API
 *
 * @resource AWS::ApiGateway::Authorizer
 */
class RecaptchaAuthorizer extends apigateway.Authorizer {
    constructor(scope, id, props) {
        super(scope, id);
        const scoreThreshold = props.scoreThreshold ?? DEFAULT_SCORE_THRESHOLD;
        if (scoreThreshold < MIN_SCORE_THRESHOLD || scoreThreshold > MAX_SCORE_THRESHOLD) {
            throw new Error('scoreThreshold must be between 0.0 and 1.0');
        }
        const handler = new lambdaNodeJS.NodejsFunction(this, 'function', {
            bundling: {
                minify: true
            },
            environment: {
                ALLOWED_ACTIONS: JSON.stringify(props.allowedActions),
                SCORE_THRESHOLD: scoreThreshold.toString(),
                SECRET_KEY_TYPE: props.reCaptchaSecretKey.secretKeyType,
                ...props.reCaptchaSecretKey.environment
            },
            runtime: lambda.Runtime.NODEJS_14_X,
            tracing: props.tracing
        });
        if (props.reCaptchaSecretKey.grantRead) {
            props.reCaptchaSecretKey.grantRead(handler);
        }
        this.authorizer = new apigateway.RequestAuthorizer(this, 'Authorizer', {
            handler,
            identitySources: [apigateway.IdentitySource.header('X-reCAPTCHA-Token')]
        });
        this.authorizationType = this.authorizer.authorizationType;
    }
    /**
     * The authorizer ID.
     * @attribute
     */
    get authorizerId() {
        return this.authorizer.authorizerId;
    }
    /**
     * Attaches this authorizer to a specific REST API.
     * @internal
     * @returns {void}
     * @param {RestApi} restApi the rest API to attach this authorizer to
     */
    _attachToApi(restApi) {
        // eslint-disable-next-line @typescript-eslint/ban-ts-comment
        // @ts-ignore
        // eslint-disable-next-line no-underscore-dangle
        this.authorizer._attachToApi(restApi);
    }
}
exports.RecaptchaAuthorizer = RecaptchaAuthorizer;
_a = JSII_RTTI_SYMBOL_1;
RecaptchaAuthorizer[_a] = { fqn: "cdk-recaptcha-authorizer.RecaptchaAuthorizer", version: "2.1.4" };
//# sourceMappingURL=data:application/json;base64,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