"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const targets = require("@aws-cdk/aws-events-targets");
const lambda = require("@aws-cdk/aws-lambda");
const logs = require("@aws-cdk/aws-logs");
const cdk = require("@aws-cdk/core");
const rds = require("../lib");
const app = new cdk.App();
class DatabaseInstanceStack extends cdk.Stack {
    constructor(scope, id, props) {
        super(scope, id, props);
        const vpc = new ec2.Vpc(this, 'VPC', { maxAzs: 2 });
        /// !show
        // Set open cursors with parameter group
        const parameterGroup = new rds.ParameterGroup(this, 'ParameterGroup', {
            family: 'oracle-se1-11.2',
            parameters: {
                open_cursors: '2500',
            },
        });
        /// Add XMLDB and OEM with option group
        const optionGroup = new rds.OptionGroup(this, 'OptionGroup', {
            engine: rds.DatabaseInstanceEngine.ORACLE_SE1,
            majorEngineVersion: '11.2',
            configurations: [
                {
                    name: 'XMLDB',
                },
                {
                    name: 'OEM',
                    port: 1158,
                    vpc,
                },
            ],
        });
        // Allow connections to OEM
        optionGroup.optionConnections.OEM.connections.allowDefaultPortFromAnyIpv4();
        // Database instance with production values
        const instance = new rds.DatabaseInstance(this, 'Instance', {
            engine: rds.DatabaseInstanceEngine.ORACLE_SE1,
            licenseModel: rds.LicenseModel.BRING_YOUR_OWN_LICENSE,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE3, ec2.InstanceSize.MEDIUM),
            multiAz: true,
            storageType: rds.StorageType.IO1,
            masterUsername: 'syscdk',
            vpc,
            databaseName: 'ORCL',
            storageEncrypted: true,
            backupRetention: cdk.Duration.days(7),
            monitoringInterval: cdk.Duration.seconds(60),
            enablePerformanceInsights: true,
            cloudwatchLogsExports: [
                'trace',
                'audit',
                'alert',
                'listener',
            ],
            cloudwatchLogsRetention: logs.RetentionDays.ONE_MONTH,
            autoMinorVersionUpgrade: false,
            optionGroup,
            parameterGroup,
        });
        // Allow connections on default port from any IPV4
        instance.connections.allowDefaultPortFromAnyIpv4();
        // Rotate the master user password every 30 days
        instance.addRotationSingleUser();
        // Add alarm for high CPU
        new cloudwatch.Alarm(this, 'HighCPU', {
            metric: instance.metricCPUUtilization(),
            threshold: 90,
            evaluationPeriods: 1,
        });
        // Trigger Lambda function on instance availability events
        const fn = new lambda.Function(this, 'Function', {
            code: lambda.Code.fromInline('exports.handler = (event) => console.log(event);'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const availabilityRule = instance.onEvent('Availability', { target: new targets.LambdaFunction(fn) });
        availabilityRule.addEventPattern({
            detail: {
                EventCategories: [
                    'availability',
                ],
            },
        });
        /// !hide
    }
}
new DatabaseInstanceStack(app, 'aws-cdk-rds-instance');
app.synth();
//# sourceMappingURL=data:application/json;base64,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