"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseClusterEngine = void 0;
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const version_1 = require("./private/version");
/**
 * A database cluster engine. Provides mapping to the serverless application
 * used for secret rotation.
 */
class DatabaseClusterEngine {
    // tslint:disable-next-line max-line-length
    constructor(name, singleUserRotationApplication, multiUserRotationApplication, parameterGroupFamilies) {
        this.name = name;
        this.singleUserRotationApplication = singleUserRotationApplication;
        this.multiUserRotationApplication = multiUserRotationApplication;
        this.parameterGroupFamilies = parameterGroupFamilies;
    }
    /**
     * Get the latest parameter group family for this engine. Latest is determined using semver on the engine major version.
     * When `engineVersion` is specified, return the parameter group family corresponding to that engine version.
     * Return undefined if no parameter group family is defined for this engine or for the requested `engineVersion`.
     */
    parameterGroupFamily(engineVersion) {
        if (this.parameterGroupFamilies === undefined) {
            return undefined;
        }
        if (engineVersion) {
            const family = this.parameterGroupFamilies.find(x => engineVersion.startsWith(x.engineMajorVersion));
            if (family) {
                return family.parameterGroupFamily;
            }
        }
        else if (this.parameterGroupFamilies.length > 0) {
            const sorted = this.parameterGroupFamilies.slice().sort((a, b) => {
                return version_1.compare(a.engineMajorVersion, b.engineMajorVersion);
            }).reverse();
            return sorted[0].parameterGroupFamily;
        }
        return undefined;
    }
}
exports.DatabaseClusterEngine = DatabaseClusterEngine;
/* tslint:disable max-line-length */
DatabaseClusterEngine.AURORA = new DatabaseClusterEngine('aurora', secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER, [
    { engineMajorVersion: '5.6', parameterGroupFamily: 'aurora5.6' },
]);
DatabaseClusterEngine.AURORA_MYSQL = new DatabaseClusterEngine('aurora-mysql', secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER, [
    { engineMajorVersion: '5.7', parameterGroupFamily: 'aurora-mysql5.7' },
]);
DatabaseClusterEngine.AURORA_POSTGRESQL = new DatabaseClusterEngine('aurora-postgresql', secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER, [
    { engineMajorVersion: '9.6', parameterGroupFamily: 'aurora-postgresql9.6' },
    { engineMajorVersion: '10', parameterGroupFamily: 'aurora-postgresql10' },
    { engineMajorVersion: '11', parameterGroupFamily: 'aurora-postgresql11' },
]);
//# sourceMappingURL=data:application/json;base64,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