"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ClusterParameterGroup = exports.ParameterGroup = void 0;
const core_1 = require("@aws-cdk/core");
const rds_generated_1 = require("./rds.generated");
/**
 * A new cluster or instance parameter group
 */
class ParameterGroupBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * Parameters of the parameter group
         */
        this.parameters = {};
    }
    /**
     * Imports a parameter group
     */
    static fromParameterGroupName(scope, id, parameterGroupName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.parameterGroupName = parameterGroupName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add a parameter to this parameter group
     *
     * @param key The key of the parameter to be added
     * @param value The value of the parameter to be added
     */
    addParameter(key, value) {
        if (!this.parameters) {
            this.parameters = {};
        }
        this.parameters[key] = value;
    }
}
/**
 * A parameter group
 *
 * @resource AWS::RDS::DBParameterGroup
 */
class ParameterGroup extends ParameterGroupBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.parameters = props.parameters ? props.parameters : {};
        const resource = new rds_generated_1.CfnDBParameterGroup(this, 'Resource', {
            description: props.description || `Parameter group for ${props.family}`,
            family: props.family,
            parameters: core_1.Lazy.anyValue({ produce: () => this.parameters }),
        });
        this.parameterGroupName = resource.ref;
    }
}
exports.ParameterGroup = ParameterGroup;
/**
 * A cluster parameter group
 *
 * @resource AWS::RDS::DBClusterParameterGroup
 */
class ClusterParameterGroup extends ParameterGroupBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.parameters = props.parameters ? props.parameters : {};
        const resource = new rds_generated_1.CfnDBClusterParameterGroup(this, 'Resource', {
            description: props.description || `Cluster parameter group for ${props.family}`,
            family: props.family,
            parameters: core_1.Lazy.anyValue({ produce: () => this.parameters }),
        });
        this.parameterGroupName = resource.ref;
    }
}
exports.ClusterParameterGroup = ClusterParameterGroup;
//# sourceMappingURL=data:application/json;base64,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