"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OptionGroup = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const rds_generated_1 = require("./rds.generated");
/**
 * An option group
 */
class OptionGroup extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * The connections object for the options.
         */
        this.optionConnections = {};
        const optionGroup = new rds_generated_1.CfnOptionGroup(this, 'Resource', {
            engineName: props.engine.name,
            majorEngineVersion: props.majorEngineVersion,
            optionGroupDescription: props.description || `Option group for ${props.engine.name} ${props.majorEngineVersion}`,
            optionConfigurations: this.renderConfigurations(props.configurations),
        });
        this.optionGroupName = optionGroup.ref;
    }
    /**
     * Import an existing option group.
     */
    static fromOptionGroupName(scope, id, optionGroupName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.optionGroupName = optionGroupName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Renders the option configurations specifications.
     */
    renderConfigurations(configurations) {
        const configs = [];
        for (const config of configurations) {
            let configuration = {
                optionName: config.name,
                optionSettings: config.settings && Object.entries(config.settings).map(([name, value]) => ({ name, value })),
                optionVersion: config.version,
            };
            if (config.port) {
                if (!config.vpc) {
                    throw new Error('`port` and `vpc` must be specified together.');
                }
                const securityGroup = new ec2.SecurityGroup(this, `SecurityGroup${config.name}`, {
                    description: `Security group for ${config.name} option`,
                    vpc: config.vpc,
                });
                this.optionConnections[config.name] = new ec2.Connections({
                    securityGroups: [securityGroup],
                    defaultPort: ec2.Port.tcp(config.port),
                });
                configuration = {
                    ...configuration,
                    port: config.port,
                    vpcSecurityGroupMemberships: [securityGroup.securityGroupId],
                };
            }
            configs.push(configuration);
        }
        return configs;
    }
}
exports.OptionGroup = OptionGroup;
//# sourceMappingURL=data:application/json;base64,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