"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigurationMapper = void 0;
exports.isEmpty = isEmpty;
exports.filter = filter;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const toml_1 = require("@iarna/toml");
const param_case_1 = require("param-case");
const pascal_case_1 = require("pascal-case");
const snake_case_1 = require("snake-case");
class ConfigurationMapper {
    static withDefaults(props) {
        const { globalConfiguration, runnersConfiguration } = props;
        return new ConfigurationMapper({
            globalConfiguration: {
                concurrent: 10,
                checkInterval: 0,
                logFormat: "runner",
                logLevel: "info",
                ...globalConfiguration,
            },
            runnersConfiguration: runnersConfiguration.map((item) => {
                return {
                    url: "https://gitlab.com",
                    limit: 10,
                    outputLimit: 52428800,
                    executor: "docker+machine",
                    environment: [
                        "DOCKER_DRIVER=overlay2",
                        "DOCKER_TLS_CERTDIR=/certs",
                        'DOCKER_AUTH_CONFIG={"credsStore": "ecr-login"}',
                    ],
                    ...item,
                    docker: {
                        tlsVerify: false,
                        image: "docker:19.03.5",
                        privileged: true,
                        capAdd: ["CAP_SYS_ADMIN"],
                        waitForServicesTimeout: 300,
                        disableCache: false,
                        volumes: ["/certs/client", "/cache"],
                        shmSize: 0,
                        ...item.docker,
                    },
                    machine: {
                        idleCount: 0,
                        idleTime: 300,
                        maxBuilds: 20,
                        machineDriver: "amazonec2",
                        machineName: "gitlab-runner-%s",
                        ...item.machine,
                        machineOptions: {
                            requestSpotInstance: true,
                            spotPrice: 0.03,
                            metadataToken: "required",
                            metadataTokenResponseHopLimit: 2,
                            ...item.machine?.machineOptions,
                        },
                        autoscaling: item.machine?.autoscaling
                            ? item.machine?.autoscaling
                            : [
                                {
                                    periods: ["* * 7-22 * * mon-fri *"],
                                    idleCount: 1,
                                    idleTime: 1800,
                                    timezone: "Etc/UTC",
                                },
                            ],
                    },
                    cache: item.cache?.s3 && Object.keys(item.cache?.s3).length
                        ? {
                            type: "s3",
                            shared: true,
                            ...item.cache,
                        }
                        : undefined,
                };
            }),
        });
    }
    static fromProps(props) {
        return new ConfigurationMapper(props);
    }
    constructor(props) {
        this.props = props;
    }
    toToml() {
        return (0, toml_1.stringify)(this._toJsonMap());
    }
    /**
     * @internal
     */
    _toJsonMap() {
        const { globalConfiguration, runnersConfiguration } = this.props;
        const result = toJsonMap(globalConfiguration, snake_case_1.snakeCase);
        result.runners = [];
        for (const config of runnersConfiguration) {
            const runner = toJsonMap(config, snake_case_1.snakeCase);
            // Fix naming convention inconsistencies
            runner["tls-ca-file"] = runner.tls_ca_file;
            delete runner.tls_ca_file;
            runner["tls-cert-file"] = runner.tls_cert_file;
            delete runner.tls_ca_file;
            runner["tls-key-file"] = runner.tls_key_file;
            delete runner.tls_ca_file;
            if (config.docker) {
                runner.docker = toJsonMap(config.docker, snake_case_1.snakeCase);
            }
            runner.machine = toJsonMap(config.machine, pascal_case_1.pascalCase);
            if (config.machine?.machineOptions) {
                runner.machine.MachineOptions = this._mapMachineOptions(config.machine?.machineOptions);
            }
            if (config.machine?.autoscaling?.length) {
                runner.machine.autoscaling = config.machine?.autoscaling.map((autoscaling) => toJsonMap(autoscaling, pascal_case_1.pascalCase));
            }
            delete runner.machine.Autoscaling;
            if (config?.cache?.s3 && Object.keys(config.cache.s3).length) {
                runner.cache = toJsonMap(config.cache, pascal_case_1.pascalCase);
                runner.cache.s3 = toJsonMap(config.cache.s3, pascal_case_1.pascalCase);
                delete runner.cache.s3;
            }
            else {
                delete runner.cache;
            }
            result.runners.push(runner);
        }
        return filter(result, (item) => !isEmpty(item));
    }
    _mapMachineOptions(machineOptions) {
        return toProperties(machineOptions, (key) => {
            switch (true) {
                case key == "engineInstallUrl":
                    return (0, param_case_1.paramCase)(key);
                default:
                    return `amazonec2-${(0, param_case_1.paramCase)(key)}`;
            }
        });
    }
}
exports.ConfigurationMapper = ConfigurationMapper;
_a = JSII_RTTI_SYMBOL_1;
ConfigurationMapper[_a] = { fqn: "@pepperize/cdk-autoscaling-gitlab-runner.ConfigurationMapper", version: "0.2.631" };
/**
 * Transforms configuration objects to JsonMap. Pass an inflector function to transform object keys.
 *
 * @param configuration
 * @param inflector A function to transform the object key
 */
function toJsonMap(configuration, inflector) {
    const result = {};
    for (const key in configuration) {
        const value = configuration[key];
        if (value === undefined) {
            continue;
        }
        result[inflector(key)] = value;
    }
    return result;
}
/**
 * Transforms configuration objects to a property array. Pass an inflector function to transform object keys.
 *
 * @param configuration
 * @param inflector A function to transform the object key
 * @example
 * // returns ["foo=bar"]
 * toProperties({foo: "bar", (s) => s});
 */
function toProperties(configuration, inflector) {
    const result = [];
    for (const key in configuration) {
        const value = configuration[key];
        if (value === undefined) {
            continue;
        }
        result.push(`${inflector(key)}=${value}`);
    }
    return result;
}
function isEmpty(subject) {
    if (Array.isArray(subject)) {
        return !subject.length;
    }
    if (typeof subject === "object" && !(subject instanceof Date)) {
        return !Object.keys(subject).length;
    }
    if (subject === undefined) {
        return true;
    }
    if (subject === null) {
        return true;
    }
    return false;
}
function filter(subject, predicate) {
    if (Array.isArray(subject)) {
        const result = [];
        subject.forEach((element) => {
            const filtered = filter(element, predicate);
            if (predicate.call(subject, filtered)) {
                result.push(filtered);
            }
        });
        return result;
    }
    if (typeof subject === "object" && !(subject instanceof Date)) {
        const result = {};
        for (const key in subject) {
            const value = subject[key];
            const filtered = filter(value, predicate);
            if (predicate.call(subject, filtered)) {
                result[key] = filtered;
            }
        }
        return result;
    }
    return subject;
}
//# sourceMappingURL=data:application/json;base64,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