"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitlabRunnerAutoscaling = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk_security_group_1 = require("@pepperize/cdk-security-group");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_autoscaling_1 = require("aws-cdk-lib/aws-autoscaling");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const constructs_1 = require("constructs");
const cache_1 = require("./cache");
const job_runner_1 = require("./job-runner");
const manager_1 = require("./manager");
const network_1 = require("./network");
/**
 * The Gitlab Runner autoscaling on EC2 by Docker Machine.
 *
 * @example <caption>Provisioning a basic Runner</caption>
 * const app = new cdk.App();
 * const stack = new cdk.Stack(app, "RunnerStack", {
 *   env: {
 *     account: "000000000000",
 *     region: "us-east-1",
 *   }
 * });
 *
 * new GitlabRunnerAutoscaling(stack, "GitlabRunner", {
 *   runners: [{
 *     gitlabToken: "xxxxxxxxxxxxxxxxxxxx"},
 *   }]
 * });
 */
class GitlabRunnerAutoscaling extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { concurrent, checkInterval, logFormat, logLevel, manager, cache, runners, network, } = props;
        /**
         * Global Configuration
         * @link GlobalConfiguration
         */
        this.concurrent = concurrent !== null && concurrent !== void 0 ? concurrent : 10;
        this.checkInterval = checkInterval !== null && checkInterval !== void 0 ? checkInterval : 0;
        this.logFormat = logFormat !== null && logFormat !== void 0 ? logFormat : "runner";
        this.logLevel = logLevel !== null && logLevel !== void 0 ? logLevel : "info";
        /**
         * S3 Bucket for Runners' cache
         */
        this.cacheBucket = (cache === null || cache === void 0 ? void 0 : cache.bucket) || new cache_1.Cache(scope, "Cache", cache === null || cache === void 0 ? void 0 : cache.options).bucket;
        /**
         * Network
         */
        this.network = new network_1.Network(scope, "Network", network);
        /**
         * Security groups
         */
        const runnersSecurityGroupName = `${scope.stackName}-RunnersSecurityGroup`;
        const runnersSecurityGroup = new cdk_security_group_1.SecurityGroup(scope, "RunnersSecurityGroup", {
            securityGroupName: runnersSecurityGroupName,
            description: "Security group for GitLab Runners.",
            vpc: this.network.vpc,
        });
        const managerSecurityGroup = new cdk_security_group_1.SecurityGroup(scope, "ManagerSecurityGroup", {
            vpc: this.network.vpc,
            description: "Security group for GitLab Runners Manager.",
        });
        managerSecurityGroup.connections.allowTo(runnersSecurityGroup, aws_ec2_1.Port.tcp(22), "SSH traffic from Manager");
        managerSecurityGroup.connections.allowTo(runnersSecurityGroup, aws_ec2_1.Port.tcp(2376), "SSH traffic from Docker");
        /**
         * GitLab Manager
         */
        this.manager = new manager_1.GitlabRunnerAutoscalingManager(scope, "Manager", {
            ...manager,
            globalConfiguration: {
                concurrent: this.concurrent,
                checkInterval: this.checkInterval,
                logFormat: this.logFormat,
                logLevel: this.logLevel,
            },
            runnersSecurityGroup: runnersSecurityGroup,
            network: this.network,
            cacheBucket: this.cacheBucket,
            runners: runners.map((runnerProps, index) => {
                return new job_runner_1.GitlabRunnerAutoscalingJobRunner(scope, `GitlabRunnerAutoscalingJobRunner${index}`, runnerProps);
            }),
        });
        new aws_autoscaling_1.AutoScalingGroup(scope, "ManagerAutoscalingGroup", {
            vpc: this.network.vpc,
            vpcSubnets: {
                subnets: [this.network.subnet],
            },
            instanceType: this.manager.instanceType,
            machineImage: this.manager.machineImage,
            keyName: this.manager.keyPairName,
            securityGroup: managerSecurityGroup,
            role: this.manager.role,
            userData: this.manager.userData,
            init: this.manager.initConfig,
            initOptions: {
                ignoreFailures: false,
            },
            maxCapacity: 1,
            minCapacity: 1,
            desiredCapacity: 1,
            signals: aws_autoscaling_1.Signals.waitForCount(1, { timeout: aws_cdk_lib_1.Duration.minutes(15) }),
        });
    }
}
exports.GitlabRunnerAutoscaling = GitlabRunnerAutoscaling;
_a = JSII_RTTI_SYMBOL_1;
GitlabRunnerAutoscaling[_a] = { fqn: "@pepperize/cdk-autoscaling-gitlab-runner.GitlabRunnerAutoscaling", version: "0.1.0" };
//# sourceMappingURL=data:application/json;base64,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