import { Stack } from "aws-cdk-lib";
import { IBucket } from "aws-cdk-lib/aws-s3";
import { Construct } from "constructs";
import { GlobalConfiguration, LogFormat, LogLevel } from "../runner-configuration";
import { CacheProps } from "./cache";
import { GitlabRunnerAutoscalingJobRunnerProps } from "./job-runner";
import { GitlabRunnerAutoscalingManager, GitlabRunnerAutoscalingManagerBaseProps } from "./manager";
import { Network, NetworkProps } from "./network";
/**
 * This is a AWS CDK Construct that may be used to deploy a GitLab runner with Docker executor and auto-scaling.
 *
 * @remarks
 * The `cdk-gitlab-runner` defines the {@link GitlabRunnerAutoscalingProps} interface and {@link GitlabRunnerAutoscaling} construct class,
 * which are used to provision a the runner.
 *
 * @packageDocumentation
 */
/**
 * Properties of the Gitlab Runner. You have to provide at least the GitLab's Runner's authentication token.
 */
export interface GitlabRunnerAutoscalingProps extends GlobalConfiguration {
    readonly cache?: GitlabRunnerAutoscalingCacheProps;
    /**
     * The network configuration for the Runner. If not set, the defaults will be used.
     * @link NetworkProps
     */
    readonly network?: NetworkProps;
    /**
     * The manager EC2 instance configuration. If not set, the defaults will be used.
     * @link GitlabRunnerAutoscalingManagerBaseProps
     */
    readonly manager?: GitlabRunnerAutoscalingManagerBaseProps;
    /**
     * The runner EC2 instances settings. At least one runner should be set up.
     * @link GitlabRunnerAutoscalingJobRunnerProps
     */
    readonly runners: GitlabRunnerAutoscalingJobRunnerProps[];
}
/**
 * The distributed GitLab runner S3 cache. Either pass an existing bucket or override default options.
 * @see https://docs.gitlab.com/runner/configuration/advanced-configuration.html#the-runnerscaches3-section
 */
export interface GitlabRunnerAutoscalingCacheProps {
    /**
     * An existing S3 bucket used as runner's cache.
     */
    readonly bucket?: IBucket;
    /**
     * If no existing S3 bucket is provided, a S3 bucket will be created.
     */
    readonly options?: CacheProps;
}
/**
 * The Gitlab Runner autoscaling on EC2 by Docker Machine.
 *
 * @example <caption>Provisioning a basic Runner</caption>
 * const app = new cdk.App();
 * const stack = new cdk.Stack(app, "RunnerStack", {
 *   env: {
 *     account: "000000000000",
 *     region: "us-east-1",
 *   }
 * });
 *
 * new GitlabRunnerAutoscaling(stack, "GitlabRunner", {
 *   runners: [{
 *     gitlabToken: "xxxxxxxxxxxxxxxxxxxx"},
 *   }]
 * });
 */
export declare class GitlabRunnerAutoscaling extends Construct {
    readonly concurrent?: number;
    readonly checkInterval?: number;
    readonly logFormat?: LogFormat;
    readonly logLevel?: LogLevel;
    readonly network: Network;
    readonly cacheBucket: IBucket;
    readonly manager: GitlabRunnerAutoscalingManager;
    constructor(scope: Stack, id: string, props: GitlabRunnerAutoscalingProps);
}
