# -*- coding: utf-8 -*-
"""`ISIC4 Standard <https://unstats.un.org/unsd/classifications/Econ/Download/In%20Text/ISIC_Rev_4_english_structure.Txt>`_.
"""
from ...types import Category, Classification, Standard, Standards

ISIC4 = Standard(
    standard=Standards.ISIC4,
    classes=[
        Classification("A", "Agriculture, forestry and fishing", Category.SECTION),
        Classification(
            "01",
            "Crop and animal production, hunting and related service activities",
            Category.DIVISION,
        ),
        Classification("011", "Growing of non-perennial crops", Category.GROUP),
        Classification(
            "0111",
            "Growing of cereals (except rice),, leguminous crops and oil seeds",
            Category.CLASS,
        ),
        Classification("0112", "Growing of rice", Category.CLASS),
        Classification("0113", "Growing of vegetables and melons, roots and tubers", Category.CLASS),
        Classification("0114", "Growing of sugar cane", Category.CLASS),
        Classification("0115", "Growing of tobacco", Category.CLASS),
        Classification("0116", "Growing of fibre crops", Category.CLASS),
        Classification("0119", "Growing of other non-perennial crops", Category.CLASS),
        Classification("012", "Growing of perennial crops", Category.GROUP),
        Classification("0121", "Growing of grapes", Category.CLASS),
        Classification("0122", "Growing of tropical and subtropical fruits", Category.CLASS),
        Classification("0123", "Growing of citrus fruits", Category.CLASS),
        Classification("0124", "Growing of pome fruits and stone fruits", Category.CLASS),
        Classification("0125", "Growing of other tree and bush fruits and nuts", Category.CLASS),
        Classification("0126", "Growing of oleaginous fruits", Category.CLASS),
        Classification("0127", "Growing of beverage crops", Category.CLASS),
        Classification(
            "0128",
            "Growing of spices, aromatic, drug and pharmaceutical crops",
            Category.CLASS,
        ),
        Classification("0129", "Growing of other perennial crops", Category.CLASS),
        Classification("013", "Plant propagation", Category.GROUP),
        Classification("0130", "Plant propagation", Category.CLASS),
        Classification("014", "Animal production", Category.GROUP),
        Classification("0141", "Raising of cattle and buffaloes", Category.CLASS),
        Classification("0142", "Raising of horses and other equines", Category.CLASS),
        Classification("0143", "Raising of camels and camelids", Category.CLASS),
        Classification("0144", "Raising of sheep and goats", Category.CLASS),
        Classification("0145", "Raising of swine/pigs", Category.CLASS),
        Classification("0146", "Raising of poultry", Category.CLASS),
        Classification("0149", "Raising of other animals", Category.CLASS),
        Classification("015", "Mixed farming", Category.GROUP),
        Classification("0150", "Mixed farming", Category.CLASS),
        Classification(
            "016",
            "Support activities to agriculture and post-harvest crop activities",
            Category.GROUP,
        ),
        Classification("0161", "Support activities for crop production", Category.CLASS),
        Classification("0162", "Support activities for animal production", Category.CLASS),
        Classification("0163", "Post-harvest crop activities", Category.CLASS),
        Classification("0164", "Seed processing for propagation", Category.CLASS),
        Classification("017", "Hunting, trapping and related service activities", Category.GROUP),
        Classification("0170", "Hunting, trapping and related service activities", Category.CLASS),
        Classification("02", "Forestry and logging", Category.DIVISION),
        Classification("021", "Silviculture and other forestry activities", Category.GROUP),
        Classification("0210", "Silviculture and other forestry activities", Category.CLASS),
        Classification("022", "Logging", Category.GROUP),
        Classification("0220", "Logging", Category.CLASS),
        Classification("023", "Gathering of non-wood forest products", Category.GROUP),
        Classification("0230", "Gathering of non-wood forest products", Category.CLASS),
        Classification("024", "Support services to forestry", Category.GROUP),
        Classification("0240", "Support services to forestry", Category.CLASS),
        Classification("03", "Fishing and aquaculture", Category.DIVISION),
        Classification("031", "Fishing", Category.GROUP),
        Classification("0311", "Marine fishing", Category.CLASS),
        Classification("0312", "Freshwater fishing", Category.CLASS),
        Classification("032", "Aquaculture", Category.GROUP),
        Classification("0321", "Marine aquaculture", Category.CLASS),
        Classification("0322", "Freshwater aquaculture", Category.CLASS),
        Classification("B", "Mining and quarrying", Category.SECTION),
        Classification("05", "Mining of coal and lignite", Category.DIVISION),
        Classification("051", "Mining of hard coal", Category.GROUP),
        Classification("0510", "Mining of hard coal", Category.CLASS),
        Classification("052", "Mining of lignite", Category.GROUP),
        Classification("0520", "Mining of lignite", Category.CLASS),
        Classification("06", "Extraction of crude petroleum and natural gas", Category.DIVISION),
        Classification("061", "Extraction of crude petroleum", Category.GROUP),
        Classification("0610", "Extraction of crude petroleum", Category.CLASS),
        Classification("062", "Extraction of natural gas", Category.GROUP),
        Classification("0620", "Extraction of natural gas", Category.CLASS),
        Classification("07", "Mining of metal ores", Category.DIVISION),
        Classification("071", "Mining of iron ores", Category.GROUP),
        Classification("0710", "Mining of iron ores", Category.CLASS),
        Classification("072", "Mining of non-ferrous metal ores", Category.GROUP),
        Classification("0721", "Mining of uranium and thorium ores", Category.CLASS),
        Classification("0729", "Mining of other non-ferrous metal ores", Category.CLASS),
        Classification("08", "Other mining and quarrying", Category.DIVISION),
        Classification("081", "Quarrying of stone, sand and clay", Category.GROUP),
        Classification("0810", "Quarrying of stone, sand and clay", Category.CLASS),
        Classification("089", "Mining and quarrying n.e.c.", Category.GROUP),
        Classification("0891", "Mining of chemical and fertilizer minerals", Category.CLASS),
        Classification("0892", "Extraction of peat", Category.CLASS),
        Classification("0893", "Extraction of salt", Category.CLASS),
        Classification("0899", "Other mining and quarrying n.e.c.", Category.CLASS),
        Classification("09", "Mining support service activities", Category.DIVISION),
        Classification(
            "091",
            "Support activities for petroleum and natural gas extraction",
            Category.GROUP,
        ),
        Classification(
            "0910",
            "Support activities for petroleum and natural gas extraction",
            Category.CLASS,
        ),
        Classification("099", "Support activities for other mining and quarrying", Category.GROUP),
        Classification("0990", "Support activities for other mining and quarrying", Category.CLASS),
        Classification("C", "Manufacturing", Category.SECTION),
        Classification("10", "Manufacture of food products", Category.DIVISION),
        Classification("101", "Processing and preserving of meat", Category.GROUP),
        Classification("1010", "Processing and preserving of meat", Category.CLASS),
        Classification(
            "102",
            "Processing and preserving of fish, crustaceans and molluscs",
            Category.GROUP,
        ),
        Classification(
            "1020",
            "Processing and preserving of fish, crustaceans and molluscs",
            Category.CLASS,
        ),
        Classification("103", "Processing and preserving of fruit and vegetables", Category.GROUP),
        Classification("1030", "Processing and preserving of fruit and vegetables", Category.CLASS),
        Classification("104", "Manufacture of vegetable and animal oils and fats", Category.GROUP),
        Classification("1040", "Manufacture of vegetable and animal oils and fats", Category.CLASS),
        Classification("105", "Manufacture of dairy products", Category.GROUP),
        Classification("1050", "Manufacture of dairy products", Category.CLASS),
        Classification(
            "106",
            "Manufacture of grain mill products, starches and starch products",
            Category.GROUP,
        ),
        Classification("1061", "Manufacture of grain mill products", Category.CLASS),
        Classification("1062", "Manufacture of starches and starch products", Category.CLASS),
        Classification("107", "Manufacture of other food products", Category.GROUP),
        Classification("1071", "Manufacture of bakery products", Category.CLASS),
        Classification("1072", "Manufacture of sugar", Category.CLASS),
        Classification(
            "1073",
            "Manufacture of cocoa, chocolate and sugar confectionery",
            Category.CLASS,
        ),
        Classification(
            "1074",
            "Manufacture of macaroni, noodles, couscous and similar farinaceous products",
            Category.CLASS,
        ),
        Classification("1075", "Manufacture of prepared meals and dishes", Category.CLASS),
        Classification("1079", "Manufacture of other food products n.e.c.", Category.CLASS),
        Classification("108", "Manufacture of prepared animal feeds", Category.GROUP),
        Classification("1080", "Manufacture of prepared animal feeds", Category.CLASS),
        Classification("11", "Manufacture of beverages", Category.DIVISION),
        Classification("110", "Manufacture of beverages", Category.GROUP),
        Classification("1101", "Distilling, rectifying and blending of spirits", Category.CLASS),
        Classification("1102", "Manufacture of wines", Category.CLASS),
        Classification("1103", "Manufacture of malt liquors and malt", Category.CLASS),
        Classification(
            "1104",
            "Manufacture of soft drinks; production of mineral waters and other bottled waters",
            Category.CLASS,
        ),
        Classification("12", "Manufacture of tobacco products", Category.DIVISION),
        Classification("120", "Manufacture of tobacco products", Category.GROUP),
        Classification("1200", "Manufacture of tobacco products", Category.CLASS),
        Classification("13", "Manufacture of textiles", Category.DIVISION),
        Classification("131", "Spinning, weaving and finishing of textiles", Category.GROUP),
        Classification("1311", "Preparation and spinning of textile fibres", Category.CLASS),
        Classification("1312", "Weaving of textiles", Category.CLASS),
        Classification("1313", "Finishing of textiles", Category.CLASS),
        Classification("139", "Manufacture of other textiles", Category.GROUP),
        Classification("1391", "Manufacture of knitted and crocheted fabrics", Category.CLASS),
        Classification(
            "1392",
            "Manufacture of made-up textile articles, except apparel",
            Category.CLASS,
        ),
        Classification("1393", "Manufacture of carpets and rugs", Category.CLASS),
        Classification("1394", "Manufacture of cordage, rope, twine and netting", Category.CLASS),
        Classification("1399", "Manufacture of other textiles n.e.c.", Category.CLASS),
        Classification("14", "Manufacture of wearing apparel", Category.DIVISION),
        Classification("141", "Manufacture of wearing apparel, except fur apparel", Category.GROUP),
        Classification("1410", "Manufacture of wearing apparel, except fur apparel", Category.CLASS),
        Classification("142", "Manufacture of articles of fur", Category.GROUP),
        Classification("1420", "Manufacture of articles of fur", Category.CLASS),
        Classification("143", "Manufacture of knitted and crocheted apparel", Category.GROUP),
        Classification("1430", "Manufacture of knitted and crocheted apparel", Category.CLASS),
        Classification("15", "Manufacture of leather and related products", Category.DIVISION),
        Classification(
            "151",
            "Tanning and dressing of leather; manufacture of luggage, handbags, saddlery and harness; dressing and dyeing of fur",
            Category.GROUP,
        ),
        Classification(
            "1511",
            "Tanning and dressing of leather; dressing and dyeing of fur",
            Category.CLASS,
        ),
        Classification(
            "1512",
            "Manufacture of luggage, handbags and the like, saddlery and harness",
            Category.CLASS,
        ),
        Classification("152", "Manufacture of footwear", Category.GROUP),
        Classification("1520", "Manufacture of footwear", Category.CLASS),
        Classification(
            "16",
            "Manufacture of wood and of products of wood and cork, except furniture; manufacture of articles of straw and plaiting materials",
            Category.DIVISION,
        ),
        Classification("161", "Sawmilling and planing of wood", Category.GROUP),
        Classification("1610", "Sawmilling and planing of wood", Category.CLASS),
        Classification(
            "162",
            "Manufacture of products of wood, cork, straw and plaiting materials",
            Category.GROUP,
        ),
        Classification("1621", "Manufacture of veneer sheets and wood-based panels", Category.CLASS),
        Classification("1622", "Manufacture of builders' carpentry and joinery", Category.CLASS),
        Classification("1623", "Manufacture of wooden containers", Category.CLASS),
        Classification(
            "1629",
            "Manufacture of other products of wood; manufacture of articles of cork, straw and plaiting materials",
            Category.CLASS,
        ),
        Classification("17", "Manufacture of paper and paper products", Category.DIVISION),
        Classification("170", "Manufacture of paper and paper products", Category.GROUP),
        Classification("1701", "Manufacture of pulp, paper and paperboard", Category.CLASS),
        Classification(
            "1702",
            "Manufacture of corrugated paper and paperboard and of containers of paper and paperboard",
            Category.CLASS,
        ),
        Classification(
            "1709",
            "Manufacture of other articles of paper and paperboard",
            Category.CLASS,
        ),
        Classification("18", "Printing and reproduction of recorded media", Category.DIVISION),
        Classification("181", "Printing and service activities related to printing", Category.GROUP),
        Classification("1811", "Printing", Category.CLASS),
        Classification("1812", "Service activities related to printing", Category.CLASS),
        Classification("182", "Reproduction of recorded media", Category.GROUP),
        Classification("1820", "Reproduction of recorded media", Category.CLASS),
        Classification(
            "19",
            "Manufacture of coke and refined petroleum products",
            Category.DIVISION,
        ),
        Classification("191", "Manufacture of coke oven products", Category.GROUP),
        Classification("1910", "Manufacture of coke oven products", Category.CLASS),
        Classification("192", "Manufacture of refined petroleum products", Category.GROUP),
        Classification("1920", "Manufacture of refined petroleum products", Category.CLASS),
        Classification("20", "Manufacture of chemicals and chemical products", Category.DIVISION),
        Classification(
            "201",
            "Manufacture of basic chemicals, fertilizers and nitrogen compounds, plastics and synthetic rubber in primary forms",
            Category.GROUP,
        ),
        Classification("2011", "Manufacture of basic chemicals", Category.CLASS),
        Classification("2012", "Manufacture of fertilizers and nitrogen compounds", Category.CLASS),
        Classification(
            "2013",
            "Manufacture of plastics and synthetic rubber in primary forms",
            Category.CLASS,
        ),
        Classification("202", "Manufacture of other chemical products", Category.GROUP),
        Classification(
            "2021",
            "Manufacture of pesticides and other agrochemical products",
            Category.CLASS,
        ),
        Classification(
            "2022",
            "Manufacture of paints, varnishes and similar coatings, printing ink and mastics",
            Category.CLASS,
        ),
        Classification(
            "2023",
            "Manufacture of soap and detergents, cleaning and polishing preparations, perfumes and toilet preparations",
            Category.CLASS,
        ),
        Classification("2029", "Manufacture of other chemical products n.e.c.", Category.CLASS),
        Classification("203", "Manufacture of man-made fibres", Category.GROUP),
        Classification("2030", "Manufacture of man-made fibres", Category.CLASS),
        Classification(
            "21",
            "Manufacture of basic pharmaceutical products and pharmaceutical preparations",
            Category.DIVISION,
        ),
        Classification(
            "210",
            "Manufacture of pharmaceuticals, medicinal chemical and botanical products",
            Category.GROUP,
        ),
        Classification(
            "2100",
            "Manufacture of pharmaceuticals, medicinal chemical and botanical products",
            Category.CLASS,
        ),
        Classification("22", "Manufacture of rubber and plastics products", Category.DIVISION),
        Classification("221", "Manufacture of rubber products", Category.GROUP),
        Classification(
            "2211",
            "Manufacture of rubber tyres and tubes; retreading and rebuilding of rubber tyres",
            Category.CLASS,
        ),
        Classification("2219", "Manufacture of other rubber products", Category.CLASS),
        Classification("222", "Manufacture of plastics products", Category.GROUP),
        Classification("2220", "Manufacture of plastics products", Category.CLASS),
        Classification(
            "23",
            "Manufacture of other non-metallic mineral products",
            Category.DIVISION,
        ),
        Classification("231", "Manufacture of glass and glass products", Category.GROUP),
        Classification("2310", "Manufacture of glass and glass products", Category.CLASS),
        Classification("239", "Manufacture of non-metallic mineral products n.e.c.", Category.GROUP),
        Classification("2391", "Manufacture of refractory products", Category.CLASS),
        Classification("2392", "Manufacture of clay building materials", Category.CLASS),
        Classification(
            "2393",
            "Manufacture of other porcelain and ceramic products",
            Category.CLASS,
        ),
        Classification("2394", "Manufacture of cement, lime and plaster", Category.CLASS),
        Classification(
            "2395",
            "Manufacture of articles of concrete, cement and plaster",
            Category.CLASS,
        ),
        Classification("2396", "Cutting, shaping and finishing of stone", Category.CLASS),
        Classification(
            "2399",
            "Manufacture of other non-metallic mineral products n.e.c.",
            Category.CLASS,
        ),
        Classification("24", "Manufacture of basic metals", Category.DIVISION),
        Classification("241", "Manufacture of basic iron and steel", Category.GROUP),
        Classification("2410", "Manufacture of basic iron and steel", Category.CLASS),
        Classification(
            "242",
            "Manufacture of basic precious and other non-ferrous metals",
            Category.GROUP,
        ),
        Classification(
            "2420",
            "Manufacture of basic precious and other non-ferrous metals",
            Category.CLASS,
        ),
        Classification("243", "Casting of metals", Category.GROUP),
        Classification("2431", "Casting of iron and steel", Category.CLASS),
        Classification("2432", "Casting of non-ferrous metals", Category.CLASS),
        Classification(
            "25",
            "Manufacture of fabricated metal products, except machinery and equipment",
            Category.DIVISION,
        ),
        Classification(
            "251",
            "Manufacture of structural metal products, tanks, reservoirs and steam generators",
            Category.GROUP,
        ),
        Classification("2511", "Manufacture of structural metal products", Category.CLASS),
        Classification(
            "2512",
            "Manufacture of tanks, reservoirs and containers of metal",
            Category.CLASS,
        ),
        Classification(
            "2513",
            "Manufacture of steam generators, except central heating hot water boilers",
            Category.CLASS,
        ),
        Classification("252", "Manufacture of weapons and ammunition", Category.GROUP),
        Classification("2520", "Manufacture of weapons and ammunition", Category.CLASS),
        Classification(
            "259",
            "Manufacture of other fabricated metal products; metalworking service activities",
            Category.GROUP,
        ),
        Classification(
            "2591",
            "Forging, pressing, stamping and roll-forming of metal; powder metallurgy",
            Category.CLASS,
        ),
        Classification("2592", "Treatment and coating of metals; machining", Category.CLASS),
        Classification(
            "2593",
            "Manufacture of cutlery, hand tools and general hardware",
            Category.CLASS,
        ),
        Classification(
            "2599",
            "Manufacture of other fabricated metal products n.e.c.",
            Category.CLASS,
        ),
        Classification(
            "26",
            "Manufacture of computer, electronic and optical products",
            Category.DIVISION,
        ),
        Classification("261", "Manufacture of electronic components and boards", Category.GROUP),
        Classification("2610", "Manufacture of electronic components and boards", Category.CLASS),
        Classification("262", "Manufacture of computers and peripheral equipment", Category.GROUP),
        Classification("2620", "Manufacture of computers and peripheral equipment", Category.CLASS),
        Classification("263", "Manufacture of communication equipment", Category.GROUP),
        Classification("2630", "Manufacture of communication equipment", Category.CLASS),
        Classification("264", "Manufacture of consumer electronics", Category.GROUP),
        Classification("2640", "Manufacture of consumer electronics", Category.CLASS),
        Classification(
            "265",
            "Manufacture of measuring, testing, navigating and control equipment; watches and clocks",
            Category.GROUP,
        ),
        Classification(
            "2651",
            "Manufacture of measuring, testing, navigating and control equipment",
            Category.CLASS,
        ),
        Classification("2652", "Manufacture of watches and clocks", Category.CLASS),
        Classification(
            "266",
            "Manufacture of irradiation, electromedical and electrotherapeutic equipment",
            Category.GROUP,
        ),
        Classification(
            "2660",
            "Manufacture of irradiation, electromedical and electrotherapeutic equipment",
            Category.CLASS,
        ),
        Classification(
            "267",
            "Manufacture of optical instruments and photographic equipment",
            Category.GROUP,
        ),
        Classification(
            "2670",
            "Manufacture of optical instruments and photographic equipment",
            Category.CLASS,
        ),
        Classification("268", "Manufacture of magnetic and optical media", Category.GROUP),
        Classification("2680", "Manufacture of magnetic and optical media", Category.CLASS),
        Classification("27", "Manufacture of electrical equipment", Category.DIVISION),
        Classification(
            "271",
            "Manufacture of electric motors, generators, transformers and electricity distribution and control apparatus",
            Category.GROUP,
        ),
        Classification(
            "2710",
            "Manufacture of electric motors, generators, transformers and electricity distribution and control apparatus",
            Category.CLASS,
        ),
        Classification("272", "Manufacture of batteries and accumulators", Category.GROUP),
        Classification("2720", "Manufacture of batteries and accumulators", Category.CLASS),
        Classification("273", "Manufacture of wiring and wiring devices", Category.GROUP),
        Classification("2731", "Manufacture of fibre optic cables", Category.CLASS),
        Classification(
            "2732",
            "Manufacture of other electronic and electric wires and cables",
            Category.CLASS,
        ),
        Classification("2733", "Manufacture of wiring devices", Category.CLASS),
        Classification("274", "Manufacture of electric lighting equipment", Category.GROUP),
        Classification("2740", "Manufacture of electric lighting equipment", Category.CLASS),
        Classification("275", "Manufacture of domestic appliances", Category.GROUP),
        Classification("2750", "Manufacture of domestic appliances", Category.CLASS),
        Classification("279", "Manufacture of other electrical equipment", Category.GROUP),
        Classification("2790", "Manufacture of other electrical equipment", Category.CLASS),
        Classification("28", "Manufacture of machinery and equipment n.e.c.", Category.DIVISION),
        Classification("281", "Manufacture of general-purpose machinery", Category.GROUP),
        Classification(
            "2811",
            "Manufacture of engines and turbines, except aircraft, vehicle and cycle engines",
            Category.CLASS,
        ),
        Classification("2812", "Manufacture of fluid power equipment", Category.CLASS),
        Classification(
            "2813",
            "Manufacture of other pumps, compressors, taps and valves",
            Category.CLASS,
        ),
        Classification(
            "2814",
            "Manufacture of bearings, gears, gearing and driving elements",
            Category.CLASS,
        ),
        Classification("2815", "Manufacture of ovens, furnaces and furnace burners", Category.CLASS),
        Classification("2816", "Manufacture of lifting and handling equipment", Category.CLASS),
        Classification(
            "2817",
            "Manufacture of office machinery and equipment (except computers and peripheral equipment),",
            Category.CLASS,
        ),
        Classification("2818", "Manufacture of power-driven hand tools", Category.CLASS),
        Classification("2819", "Manufacture of other general-purpose machinery", Category.CLASS),
        Classification("282", "Manufacture of special-purpose machinery", Category.GROUP),
        Classification("2821", "Manufacture of agricultural and forestry machinery", Category.CLASS),
        Classification(
            "2822",
            "Manufacture of metal-forming machinery and machine tools",
            Category.CLASS,
        ),
        Classification("2823", "Manufacture of machinery for metallurgy", Category.CLASS),
        Classification(
            "2824",
            "Manufacture of machinery for mining, quarrying and construction",
            Category.CLASS,
        ),
        Classification(
            "2825",
            "Manufacture of machinery for food, beverage and tobacco processing",
            Category.CLASS,
        ),
        Classification(
            "2826",
            "Manufacture of machinery for textile, apparel and leather production",
            Category.CLASS,
        ),
        Classification("2829", "Manufacture of other special-purpose machinery", Category.CLASS),
        Classification(
            "29",
            "Manufacture of motor vehicles, trailers and semi-trailers",
            Category.DIVISION,
        ),
        Classification("291", "Manufacture of motor vehicles", Category.GROUP),
        Classification("2910", "Manufacture of motor vehicles", Category.CLASS),
        Classification(
            "292",
            "Manufacture of bodies (coachwork), for motor vehicles; manufacture of trailers and semi-trailers",
            Category.GROUP,
        ),
        Classification(
            "2920",
            "Manufacture of bodies (coachwork), for motor vehicles; manufacture of trailers and semi-trailers",
            Category.CLASS,
        ),
        Classification(
            "293",
            "Manufacture of parts and accessories for motor vehicles",
            Category.GROUP,
        ),
        Classification(
            "2930",
            "Manufacture of parts and accessories for motor vehicles",
            Category.CLASS,
        ),
        Classification("30", "Manufacture of other transport equipment", Category.DIVISION),
        Classification("301", "Building of ships and boats", Category.GROUP),
        Classification("3011", "Building of ships and floating structures", Category.CLASS),
        Classification("3012", "Building of pleasure and sporting boats", Category.CLASS),
        Classification(
            "302",
            "Manufacture of railway locomotives and rolling stock",
            Category.GROUP,
        ),
        Classification(
            "3020",
            "Manufacture of railway locomotives and rolling stock",
            Category.CLASS,
        ),
        Classification(
            "303",
            "Manufacture of air and spacecraft and related machinery",
            Category.GROUP,
        ),
        Classification(
            "3030",
            "Manufacture of air and spacecraft and related machinery",
            Category.CLASS,
        ),
        Classification("304", "Manufacture of military fighting vehicles", Category.GROUP),
        Classification("3040", "Manufacture of military fighting vehicles", Category.CLASS),
        Classification("309", "Manufacture of transport equipment n.e.c.", Category.GROUP),
        Classification("3091", "Manufacture of motorcycles", Category.CLASS),
        Classification("3092", "Manufacture of bicycles and invalid carriages", Category.CLASS),
        Classification("3099", "Manufacture of other transport equipment n.e.c.", Category.CLASS),
        Classification("31", "Manufacture of furniture", Category.DIVISION),
        Classification("310", "Manufacture of furniture", Category.GROUP),
        Classification("3100", "Manufacture of furniture", Category.CLASS),
        Classification("32", "Other manufacturing", Category.DIVISION),
        Classification(
            "321",
            "Manufacture of jewellery, bijouterie and related articles",
            Category.GROUP,
        ),
        Classification("3211", "Manufacture of jewellery and related articles", Category.CLASS),
        Classification(
            "3212",
            "Manufacture of imitation jewellery and related articles",
            Category.CLASS,
        ),
        Classification("322", "Manufacture of musical instruments", Category.GROUP),
        Classification("3220", "Manufacture of musical instruments", Category.CLASS),
        Classification("323", "Manufacture of sports goods", Category.GROUP),
        Classification("3230", "Manufacture of sports goods", Category.CLASS),
        Classification("324", "Manufacture of games and toys", Category.GROUP),
        Classification("3240", "Manufacture of games and toys", Category.CLASS),
        Classification(
            "325",
            "Manufacture of medical and dental instruments and supplies",
            Category.GROUP,
        ),
        Classification(
            "3250",
            "Manufacture of medical and dental instruments and supplies",
            Category.CLASS,
        ),
        Classification("329", "Other manufacturing n.e.c.", Category.GROUP),
        Classification("3290", "Other manufacturing n.e.c.", Category.CLASS),
        Classification(
            "33",
            "Repair and installation of machinery and equipment",
            Category.DIVISION,
        ),
        Classification(
            "331",
            "Repair of fabricated metal products, machinery and equipment",
            Category.GROUP,
        ),
        Classification("3311", "Repair of fabricated metal products", Category.CLASS),
        Classification("3312", "Repair of machinery", Category.CLASS),
        Classification("3313", "Repair of electronic and optical equipment", Category.CLASS),
        Classification("3314", "Repair of electrical equipment", Category.CLASS),
        Classification(
            "3315",
            "Repair of transport equipment, except motor vehicles",
            Category.CLASS,
        ),
        Classification("3319", "Repair of other equipment", Category.CLASS),
        Classification("332", "Installation of industrial machinery and equipment", Category.GROUP),
        Classification("3320", "Installation of industrial machinery and equipment", Category.CLASS),
        Classification("D", "Electricity, gas, steam and air conditioning supply", Category.SECTION),
        Classification(
            "35",
            "Electricity, gas, steam and air conditioning supply",
            Category.DIVISION,
        ),
        Classification(
            "351",
            "Electric power generation, transmission and distribution",
            Category.GROUP,
        ),
        Classification(
            "3510",
            "Electric power generation, transmission and distribution",
            Category.CLASS,
        ),
        Classification(
            "352",
            "Manufacture of gas; distribution of gaseous fuels through mains",
            Category.GROUP,
        ),
        Classification(
            "3520",
            "Manufacture of gas; distribution of gaseous fuels through mains",
            Category.CLASS,
        ),
        Classification("353", "Steam and air conditioning supply", Category.GROUP),
        Classification("3530", "Steam and air conditioning supply", Category.CLASS),
        Classification(
            "E",
            "Water supply; sewerage, waste management and remediation activities",
            Category.SECTION,
        ),
        Classification("36", "Water collection, treatment and supply", Category.DIVISION),
        Classification("360", "Water collection, treatment and supply", Category.GROUP),
        Classification("3600", "Water collection, treatment and supply", Category.CLASS),
        Classification("37", "Sewerage", Category.DIVISION),
        Classification("370", "Sewerage", Category.GROUP),
        Classification("3700", "Sewerage", Category.CLASS),
        Classification(
            "38",
            "Waste collection, treatment and disposal activities; materials recovery",
            Category.DIVISION,
        ),
        Classification("381", "Waste collection", Category.GROUP),
        Classification("3811", "Collection of non-hazardous waste", Category.CLASS),
        Classification("3812", "Collection of hazardous waste", Category.CLASS),
        Classification("382", "Waste treatment and disposal", Category.GROUP),
        Classification("3821", "Treatment and disposal of non-hazardous waste", Category.CLASS),
        Classification("3822", "Treatment and disposal of hazardous waste", Category.CLASS),
        Classification("383", "Materials recovery", Category.GROUP),
        Classification("3830", "Materials recovery", Category.CLASS),
        Classification(
            "39",
            "Remediation activities and other waste management services",
            Category.DIVISION,
        ),
        Classification(
            "390",
            "Remediation activities and other waste management services",
            Category.GROUP,
        ),
        Classification(
            "3900",
            "Remediation activities and other waste management services",
            Category.CLASS,
        ),
        Classification("F", "Construction", Category.SECTION),
        Classification("41", "Construction of buildings", Category.DIVISION),
        Classification("410", "Construction of buildings", Category.GROUP),
        Classification("4100", "Construction of buildings", Category.CLASS),
        Classification("42", "Civil engineering", Category.DIVISION),
        Classification("421", "Construction of roads and railways", Category.GROUP),
        Classification("4210", "Construction of roads and railways", Category.CLASS),
        Classification("422", "Construction of utility projects", Category.GROUP),
        Classification("4220", "Construction of utility projects", Category.CLASS),
        Classification("429", "Construction of other civil engineering projects", Category.GROUP),
        Classification("4290", "Construction of other civil engineering projects", Category.CLASS),
        Classification("43", "Specialized construction activities", Category.DIVISION),
        Classification("431", "Demolition and site preparation", Category.GROUP),
        Classification("4311", "Demolition", Category.CLASS),
        Classification("4312", "Site preparation", Category.CLASS),
        Classification(
            "432",
            "Electrical, plumbing and other construction installation activities",
            Category.GROUP,
        ),
        Classification("4321", "Electrical installation", Category.CLASS),
        Classification("4322", "Plumbing, heat and air-conditioning installation", Category.CLASS),
        Classification("4329", "Other construction installation", Category.CLASS),
        Classification("433", "Building completion and finishing", Category.GROUP),
        Classification("4330", "Building completion and finishing", Category.CLASS),
        Classification("439", "Other specialized construction activities", Category.GROUP),
        Classification("4390", "Other specialized construction activities", Category.CLASS),
        Classification(
            "G",
            "Wholesale and retail trade; repair of motor vehicles and motorcycles",
            Category.SECTION,
        ),
        Classification(
            "45",
            "Wholesale and retail trade and repair of motor vehicles and motorcycles",
            Category.DIVISION,
        ),
        Classification("451", "Sale of motor vehicles", Category.GROUP),
        Classification("4510", "Sale of motor vehicles", Category.CLASS),
        Classification("452", "Maintenance and repair of motor vehicles", Category.GROUP),
        Classification("4520", "Maintenance and repair of motor vehicles", Category.CLASS),
        Classification("453", "Sale of motor vehicle parts and accessories", Category.GROUP),
        Classification("4530", "Sale of motor vehicle parts and accessories", Category.CLASS),
        Classification(
            "454",
            "Sale, maintenance and repair of motorcycles and related parts and accessories",
            Category.GROUP,
        ),
        Classification(
            "4540",
            "Sale, maintenance and repair of motorcycles and related parts and accessories",
            Category.CLASS,
        ),
        Classification(
            "46",
            "Wholesale trade, except of motor vehicles and motorcycles",
            Category.DIVISION,
        ),
        Classification("461", "Wholesale on a fee or contract basis", Category.GROUP),
        Classification("4610", "Wholesale on a fee or contract basis", Category.CLASS),
        Classification(
            "462",
            "Wholesale of agricultural raw materials and live animals",
            Category.GROUP,
        ),
        Classification(
            "4620",
            "Wholesale of agricultural raw materials and live animals",
            Category.CLASS,
        ),
        Classification("463", "Wholesale of food, beverages and tobacco", Category.GROUP),
        Classification("4630", "Wholesale of food, beverages and tobacco", Category.CLASS),
        Classification("464", "Wholesale of household goods", Category.GROUP),
        Classification("4641", "Wholesale of textiles, clothing and footwear", Category.CLASS),
        Classification("4649", "Wholesale of other household goods", Category.CLASS),
        Classification("465", "Wholesale of machinery, equipment and supplies", Category.GROUP),
        Classification(
            "4651",
            "Wholesale of computers, computer peripheral equipment and software",
            Category.CLASS,
        ),
        Classification(
            "4652",
            "Wholesale of electronic and telecommunications equipment and parts",
            Category.CLASS,
        ),
        Classification(
            "4653",
            "Wholesale of agricultural machinery, equipment and supplies",
            Category.CLASS,
        ),
        Classification("4659", "Wholesale of other machinery and equipment", Category.CLASS),
        Classification("466", "Other specialized wholesale", Category.GROUP),
        Classification(
            "4661",
            "Wholesale of solid, liquid and gaseous fuels and related products",
            Category.CLASS,
        ),
        Classification("4662", "Wholesale of metals and metal ores", Category.CLASS),
        Classification(
            "4663",
            "Wholesale of construction materials, hardware, plumbing and heating equipment and supplies",
            Category.CLASS,
        ),
        Classification(
            "4669",
            "Wholesale of waste and scrap and other products n.e.c.",
            Category.CLASS,
        ),
        Classification("469", "Non-specialized wholesale trade", Category.GROUP),
        Classification("4690", "Non-specialized wholesale trade", Category.CLASS),
        Classification(
            "47",
            "Retail trade, except of motor vehicles and motorcycles",
            Category.DIVISION,
        ),
        Classification("471", "Retail sale in non-specialized stores", Category.GROUP),
        Classification(
            "4711",
            "Retail sale in non-specialized stores with food, beverages or tobacco predominating",
            Category.CLASS,
        ),
        Classification("4719", "Other retail sale in non-specialized stores", Category.CLASS),
        Classification(
            "472",
            "Retail sale of food, beverages and tobacco in specialized stores",
            Category.GROUP,
        ),
        Classification("4721", "Retail sale of food in specialized stores", Category.CLASS),
        Classification("4722", "Retail sale of beverages in specialized stores", Category.CLASS),
        Classification(
            "4723",
            "Retail sale of tobacco products in specialized stores",
            Category.CLASS,
        ),
        Classification(
            "473",
            "Retail sale of automotive fuel in specialized stores",
            Category.GROUP,
        ),
        Classification(
            "4730",
            "Retail sale of automotive fuel in specialized stores",
            Category.CLASS,
        ),
        Classification(
            "474",
            "Retail sale of information and communications equipment in specialized stores",
            Category.GROUP,
        ),
        Classification(
            "4741",
            "Retail sale of computers, peripheral units, software and telecommunications equipment in specialized stores",
            Category.CLASS,
        ),
        Classification(
            "4742",
            "Retail sale of audio and video equipment in specialized stores",
            Category.CLASS,
        ),
        Classification(
            "475",
            "Retail sale of other household equipment in specialized stores",
            Category.GROUP,
        ),
        Classification("4751", "Retail sale of textiles in specialized stores", Category.CLASS),
        Classification(
            "4752",
            "Retail sale of hardware, paints and glass in specialized stores",
            Category.CLASS,
        ),
        Classification(
            "4753",
            "Retail sale of carpets, rugs, wall and floor coverings in specialized stores",
            Category.CLASS,
        ),
        Classification(
            "4759",
            "Retail sale of electrical household appliances, furniture, lighting equipment and other household articles in specialized stores",
            Category.CLASS,
        ),
        Classification(
            "476",
            "Retail sale of cultural and recreation goods in specialized stores",
            Category.GROUP,
        ),
        Classification(
            "4761",
            "Retail sale of books, newspapers and stationary in specialized stores",
            Category.CLASS,
        ),
        Classification(
            "4762",
            "Retail sale of music and video recordings in specialized stores",
            Category.CLASS,
        ),
        Classification(
            "4763",
            "Retail sale of sporting equipment in specialized stores",
            Category.CLASS,
        ),
        Classification(
            "4764",
            "Retail sale of games and toys in specialized stores",
            Category.CLASS,
        ),
        Classification("477", "Retail sale of other goods in specialized stores", Category.GROUP),
        Classification(
            "4771",
            "Retail sale of clothing, footwear and leather articles in specialized stores",
            Category.CLASS,
        ),
        Classification(
            "4772",
            "Retail sale of pharmaceutical and medical goods, cosmetic and toilet articles in specialized stores",
            Category.CLASS,
        ),
        Classification(
            "4773",
            "Other retail sale of new goods in specialized stores",
            Category.CLASS,
        ),
        Classification("4774", "Retail sale of second-hand goods", Category.CLASS),
        Classification("478", "Retail sale via stalls and markets", Category.GROUP),
        Classification(
            "4781",
            "Retail sale via stalls and markets of food, beverages and tobacco products",
            Category.CLASS,
        ),
        Classification(
            "4782",
            "Retail sale via stalls and markets of textiles, clothing and footwear",
            Category.CLASS,
        ),
        Classification("4789", "Retail sale via stalls and markets of other goods", Category.CLASS),
        Classification("479", "Retail trade not in stores, stalls or markets", Category.GROUP),
        Classification("4791", "Retail sale via mail order houses or via Internet", Category.CLASS),
        Classification("4799", "Other retail sale not in stores, stalls or markets", Category.CLASS),
        Classification("H", "Transportation and storage", Category.SECTION),
        Classification("49", "Land transport and transport via pipelines", Category.DIVISION),
        Classification("491", "Transport via railways", Category.GROUP),
        Classification("4911", "Passenger rail transport, interurban", Category.CLASS),
        Classification("4912", "Freight rail transport", Category.CLASS),
        Classification("492", "Other land transport", Category.GROUP),
        Classification("4921", "Urban and suburban passenger land transport", Category.CLASS),
        Classification("4922", "Other passenger land transport", Category.CLASS),
        Classification("4923", "Freight transport by road", Category.CLASS),
        Classification("493", "Transport via pipeline", Category.GROUP),
        Classification("4930", "Transport via pipeline", Category.CLASS),
        Classification("50", "Water transport", Category.DIVISION),
        Classification("501", "Sea and coastal water transport", Category.GROUP),
        Classification("5011", "Sea and coastal passenger water transport", Category.CLASS),
        Classification("5012", "Sea and coastal freight water transport", Category.CLASS),
        Classification("502", "Inland water transport", Category.GROUP),
        Classification("5021", "Inland passenger water transport", Category.CLASS),
        Classification("5022", "Inland freight water transport", Category.CLASS),
        Classification("51", "Air transport", Category.DIVISION),
        Classification("511", "Passenger air transport", Category.GROUP),
        Classification("5110", "Passenger air transport", Category.CLASS),
        Classification("512", "Freight air transport", Category.GROUP),
        Classification("5120", "Freight air transport", Category.CLASS),
        Classification(
            "52",
            "Warehousing and support activities for transportation",
            Category.DIVISION,
        ),
        Classification("521", "Warehousing and storage", Category.GROUP),
        Classification("5210", "Warehousing and storage", Category.CLASS),
        Classification("522", "Support activities for transportation", Category.GROUP),
        Classification(
            "5221",
            "Service activities incidental to land transportation",
            Category.CLASS,
        ),
        Classification(
            "5222",
            "Service activities incidental to water transportation",
            Category.CLASS,
        ),
        Classification(
            "5223",
            "Service activities incidental to air transportation",
            Category.CLASS,
        ),
        Classification("5224", "Cargo handling", Category.CLASS),
        Classification("5229", "Other transportation support activities", Category.CLASS),
        Classification("53", "Postal and courier activities", Category.DIVISION),
        Classification("531", "Postal activities", Category.GROUP),
        Classification("5310", "Postal activities", Category.CLASS),
        Classification("532", "Courier activities", Category.GROUP),
        Classification("5320", "Courier activities", Category.CLASS),
        Classification("I", "Accommodation and food service activities", Category.SECTION),
        Classification("55", "Accommodation", Category.DIVISION),
        Classification("551", "Short term accommodation activities", Category.GROUP),
        Classification("5510", "Short term accommodation activities", Category.CLASS),
        Classification(
            "552",
            "Camping grounds, recreational vehicle parks and trailer parks",
            Category.GROUP,
        ),
        Classification(
            "5520",
            "Camping grounds, recreational vehicle parks and trailer parks",
            Category.CLASS,
        ),
        Classification("559", "Other accommodation", Category.GROUP),
        Classification("5590", "Other accommodation", Category.CLASS),
        Classification("56", "Food and beverage service activities", Category.DIVISION),
        Classification("561", "Restaurants and mobile food service activities", Category.GROUP),
        Classification("5610", "Restaurants and mobile food service activities", Category.CLASS),
        Classification("562", "Event catering and other food service activities", Category.GROUP),
        Classification("5621", "Event catering", Category.CLASS),
        Classification("5629", "Other food service activities", Category.CLASS),
        Classification("563", "Beverage serving activities", Category.GROUP),
        Classification("5630", "Beverage serving activities", Category.CLASS),
        Classification("J", "Information and communication", Category.SECTION),
        Classification("58", "Publishing activities", Category.DIVISION),
        Classification(
            "581",
            "Publishing of books, periodicals and other publishing activities",
            Category.GROUP,
        ),
        Classification("5811", "Book publishing", Category.CLASS),
        Classification("5812", "Publishing of directories and mailing lists", Category.CLASS),
        Classification("5813", "Publishing of newspapers, journals and periodicals", Category.CLASS),
        Classification("5819", "Other publishing activities", Category.CLASS),
        Classification("582", "Software publishing", Category.GROUP),
        Classification("5820", "Software publishing", Category.CLASS),
        Classification(
            "59",
            "Motion picture, video and television programme production, sound recording and music publishing activities",
            Category.DIVISION,
        ),
        Classification(
            "591",
            "Motion picture, video and television programme activities",
            Category.GROUP,
        ),
        Classification(
            "5911",
            "Motion picture, video and television programme production activities",
            Category.CLASS,
        ),
        Classification(
            "5912",
            "Motion picture, video and television programme post-production activities",
            Category.CLASS,
        ),
        Classification(
            "5913",
            "Motion picture, video and television programme distribution activities",
            Category.CLASS,
        ),
        Classification("5914", "Motion picture projection activities", Category.CLASS),
        Classification("592", "Sound recording and music publishing activities", Category.GROUP),
        Classification("5920", "Sound recording and music publishing activities", Category.CLASS),
        Classification("60", "Programming and broadcasting activities", Category.DIVISION),
        Classification("601", "Radio broadcasting", Category.GROUP),
        Classification("6010", "Radio broadcasting", Category.CLASS),
        Classification("602", "Television programming and broadcasting activities", Category.GROUP),
        Classification("6020", "Television programming and broadcasting activities", Category.CLASS),
        Classification("61", "Telecommunications", Category.DIVISION),
        Classification("611", "Wired telecommunications activities", Category.GROUP),
        Classification("6110", "Wired telecommunications activities", Category.CLASS),
        Classification("612", "Wireless telecommunications activities", Category.GROUP),
        Classification("6120", "Wireless telecommunications activities", Category.CLASS),
        Classification("613", "Satellite telecommunications activities", Category.GROUP),
        Classification("6130", "Satellite telecommunications activities", Category.CLASS),
        Classification("619", "Other telecommunications activities", Category.GROUP),
        Classification("6190", "Other telecommunications activities", Category.CLASS),
        Classification(
            "62",
            "Computer programming, consultancy and related activities",
            Category.DIVISION,
        ),
        Classification(
            "620",
            "Computer programming, consultancy and related activities",
            Category.GROUP,
        ),
        Classification("6201", "Computer programming activities", Category.CLASS),
        Classification(
            "6202",
            "Computer consultancy and computer facilities management activities",
            Category.CLASS,
        ),
        Classification(
            "6209",
            "Other information technology and computer service activities",
            Category.CLASS,
        ),
        Classification("63", "Information service activities", Category.DIVISION),
        Classification(
            "631",
            "Data processing, hosting and related activities; web portals",
            Category.GROUP,
        ),
        Classification("6311", "Data processing, hosting and related activities", Category.CLASS),
        Classification("6312", "Web portals", Category.CLASS),
        Classification("639", "Other information service activities", Category.GROUP),
        Classification("6391", "News agency activities", Category.CLASS),
        Classification("6399", "Other information service activities n.e.c.", Category.CLASS),
        Classification("K", "Financial and insurance activities", Category.SECTION),
        Classification(
            "64",
            "Financial service activities, except insurance and pension funding",
            Category.DIVISION,
        ),
        Classification("641", "Monetary intermediation", Category.GROUP),
        Classification("6411", "Central banking", Category.CLASS),
        Classification("6419", "Other monetary intermediation", Category.CLASS),
        Classification("642", "Activities of holding companies", Category.GROUP),
        Classification("6420", "Activities of holding companies", Category.CLASS),
        Classification("643", "Trusts, funds and similar financial entities", Category.GROUP),
        Classification("6430", "Trusts, funds and similar financial entities", Category.CLASS),
        Classification(
            "649",
            "Other financial service activities, except insurance and pension funding activities",
            Category.GROUP,
        ),
        Classification("6491", "Financial leasing", Category.CLASS),
        Classification("6492", "Other credit granting", Category.CLASS),
        Classification(
            "6499",
            "Other financial service activities, except insurance and pension funding activities, n.e.c.",
            Category.CLASS,
        ),
        Classification(
            "65",
            "Insurance, reinsurance and pension funding, except compulsory social security",
            Category.DIVISION,
        ),
        Classification("651", "Insurance", Category.GROUP),
        Classification("6511", "Life insurance", Category.CLASS),
        Classification("6512", "Non-life insurance", Category.CLASS),
        Classification("652", "Reinsurance", Category.GROUP),
        Classification("6520", "Reinsurance", Category.CLASS),
        Classification("653", "Pension funding", Category.GROUP),
        Classification("6530", "Pension funding", Category.CLASS),
        Classification(
            "66",
            "Activities auxiliary to financial service and insurance activities",
            Category.DIVISION,
        ),
        Classification(
            "661",
            "Activities auxiliary to financial service activities, except insurance and pension funding",
            Category.GROUP,
        ),
        Classification("6611", "Administration of financial markets", Category.CLASS),
        Classification("6612", "Security and commodity contracts brokerage", Category.CLASS),
        Classification(
            "6619",
            "Other activities auxiliary to financial service activities",
            Category.CLASS,
        ),
        Classification(
            "662",
            "Activities auxiliary to insurance and pension funding",
            Category.GROUP,
        ),
        Classification("6621", "Risk and damage evaluation", Category.CLASS),
        Classification("6622", "Activities of insurance agents and brokers", Category.CLASS),
        Classification(
            "6629",
            "Other activities auxiliary to insurance and pension funding",
            Category.CLASS,
        ),
        Classification("663", "Fund management activities", Category.GROUP),
        Classification("6630", "Fund management activities", Category.CLASS),
        Classification("L", "Real estate activities", Category.SECTION),
        Classification("68", "Real estate activities", Category.DIVISION),
        Classification("681", "Real estate activities with own or leased property", Category.GROUP),
        Classification("6810", "Real estate activities with own or leased property", Category.CLASS),
        Classification("682", "Real estate activities on a fee or contract basis", Category.GROUP),
        Classification("6820", "Real estate activities on a fee or contract basis", Category.CLASS),
        Classification("M", "Professional, scientific and technical activities", Category.SECTION),
        Classification("69", "Legal and accounting activities", Category.DIVISION),
        Classification("691", "Legal activities", Category.GROUP),
        Classification("6910", "Legal activities", Category.CLASS),
        Classification(
            "692",
            "Accounting, bookkeeping and auditing activities; tax consultancy",
            Category.GROUP,
        ),
        Classification(
            "6920",
            "Accounting, bookkeeping and auditing activities; tax consultancy",
            Category.CLASS,
        ),
        Classification(
            "70",
            "Activities of head offices; management consultancy activities",
            Category.DIVISION,
        ),
        Classification("701", "Activities of head offices", Category.GROUP),
        Classification("7010", "Activities of head offices", Category.CLASS),
        Classification("702", "Management consultancy activities", Category.GROUP),
        Classification("7020", "Management consultancy activities", Category.CLASS),
        Classification(
            "71",
            "Architectural and engineering activities; technical testing and analysis",
            Category.DIVISION,
        ),
        Classification(
            "711",
            "Architectural and engineering activities and related technical consultancy",
            Category.GROUP,
        ),
        Classification(
            "7110",
            "Architectural and engineering activities and related technical consultancy",
            Category.CLASS,
        ),
        Classification("712", "Technical testing and analysis", Category.GROUP),
        Classification("7120", "Technical testing and analysis", Category.CLASS),
        Classification("72", "Scientific research and development", Category.DIVISION),
        Classification(
            "721",
            "Research and experimental development on natural sciences and engineering",
            Category.GROUP,
        ),
        Classification(
            "7210",
            "Research and experimental development on natural sciences and engineering",
            Category.CLASS,
        ),
        Classification(
            "722",
            "Research and experimental development on social sciences and humanities",
            Category.GROUP,
        ),
        Classification(
            "7220",
            "Research and experimental development on social sciences and humanities",
            Category.CLASS,
        ),
        Classification("73", "Advertising and market research", Category.DIVISION),
        Classification("731", "Advertising", Category.GROUP),
        Classification("7310", "Advertising", Category.CLASS),
        Classification("732", "Market research and public opinion polling", Category.GROUP),
        Classification("7320", "Market research and public opinion polling", Category.CLASS),
        Classification(
            "74",
            "Other professional, scientific and technical activities",
            Category.DIVISION,
        ),
        Classification("741", "Specialized design activities", Category.GROUP),
        Classification("7410", "Specialized design activities", Category.CLASS),
        Classification("742", "Photographic activities", Category.GROUP),
        Classification("7420", "Photographic activities", Category.CLASS),
        Classification(
            "749",
            "Other professional, scientific and technical activities n.e.c.",
            Category.GROUP,
        ),
        Classification(
            "7490",
            "Other professional, scientific and technical activities n.e.c.",
            Category.CLASS,
        ),
        Classification("75", "Veterinary activities", Category.DIVISION),
        Classification("750", "Veterinary activities", Category.GROUP),
        Classification("7500", "Veterinary activities", Category.CLASS),
        Classification("N", "Administrative and support service activities", Category.SECTION),
        Classification("77", "Rental and leasing activities", Category.DIVISION),
        Classification("771", "Renting and leasing of motor vehicles", Category.GROUP),
        Classification("7710", "Renting and leasing of motor vehicles", Category.CLASS),
        Classification("772", "Renting and leasing of personal and household goods", Category.GROUP),
        Classification(
            "7721",
            "Renting and leasing of recreational and sports goods",
            Category.CLASS,
        ),
        Classification("7722", "Renting of video tapes and disks", Category.CLASS),
        Classification(
            "7729",
            "Renting and leasing of other personal and household goods",
            Category.CLASS,
        ),
        Classification(
            "773",
            "Renting and leasing of other machinery, equipment and tangible goods",
            Category.GROUP,
        ),
        Classification(
            "7730",
            "Renting and leasing of other machinery, equipment and tangible goods",
            Category.CLASS,
        ),
        Classification(
            "774",
            "Leasing of intellectual property and similar products, except copyrighted works",
            Category.GROUP,
        ),
        Classification(
            "7740",
            "Leasing of intellectual property and similar products, except copyrighted works",
            Category.CLASS,
        ),
        Classification("78", "Employment activities", Category.DIVISION),
        Classification("781", "Activities of employment placement agencies", Category.GROUP),
        Classification("7810", "Activities of employment placement agencies", Category.CLASS),
        Classification("782", "Temporary employment agency activities", Category.GROUP),
        Classification("7820", "Temporary employment agency activities", Category.CLASS),
        Classification("783", "Other human resources provision", Category.GROUP),
        Classification("7830", "Other human resources provision", Category.CLASS),
        Classification(
            "79",
            "Travel agency, tour operator, reservation service and related activities",
            Category.DIVISION,
        ),
        Classification("791", "Travel agency and tour operator activities", Category.GROUP),
        Classification("7911", "Travel agency activities", Category.CLASS),
        Classification("7912", "Tour operator activities", Category.CLASS),
        Classification("799", "Other reservation service and related activities", Category.GROUP),
        Classification("7990", "Other reservation service and related activities", Category.CLASS),
        Classification("80", "Security and investigation activities", Category.DIVISION),
        Classification("801", "Private security activities", Category.GROUP),
        Classification("8010", "Private security activities", Category.CLASS),
        Classification("802", "Security systems service activities", Category.GROUP),
        Classification("8020", "Security systems service activities", Category.CLASS),
        Classification("803", "Investigation activities", Category.GROUP),
        Classification("8030", "Investigation activities", Category.CLASS),
        Classification("81", "Services to buildings and landscape activities", Category.DIVISION),
        Classification("811", "Combined facilities support activities", Category.GROUP),
        Classification("8110", "Combined facilities support activities", Category.CLASS),
        Classification("812", "Cleaning activities", Category.GROUP),
        Classification("8121", "General cleaning of buildings", Category.CLASS),
        Classification("8129", "Other building and industrial cleaning activities", Category.CLASS),
        Classification("813", "Landscape care and maintenance service activities", Category.GROUP),
        Classification("8130", "Landscape care and maintenance service activities", Category.CLASS),
        Classification(
            "82",
            "Office administrative, office support and other business support activities",
            Category.DIVISION,
        ),
        Classification("821", "Office administrative and support activities", Category.GROUP),
        Classification("8211", "Combined office administrative service activities", Category.CLASS),
        Classification(
            "8219",
            "Photocopying, document preparation and other specialized office support activities",
            Category.CLASS,
        ),
        Classification("822", "Activities of call centres", Category.GROUP),
        Classification("8220", "Activities of call centres", Category.CLASS),
        Classification("823", "Organization of conventions and trade shows", Category.GROUP),
        Classification("8230", "Organization of conventions and trade shows", Category.CLASS),
        Classification("829", "Business support service activities n.e.c.", Category.GROUP),
        Classification(
            "8291",
            "Activities of collection agencies and credit bureaus",
            Category.CLASS,
        ),
        Classification("8292", "Packaging activities", Category.CLASS),
        Classification("8299", "Other business support service activities n.e.c.", Category.CLASS),
        Classification(
            "O",
            "Public administration and defence; compulsory social security",
            Category.SECTION,
        ),
        Classification(
            "84",
            "Public administration and defence; compulsory social security",
            Category.DIVISION,
        ),
        Classification(
            "841",
            "Administration of the State and the economic and social policy of the community",
            Category.GROUP,
        ),
        Classification("8411", "General public administration activities", Category.CLASS),
        Classification(
            "8412",
            "Regulation of the activities of providing health care, education, cultural services and other social services, excluding social security",
            Category.CLASS,
        ),
        Classification(
            "8413",
            "Regulation of and contribution to more efficient operation of businesses",
            Category.CLASS,
        ),
        Classification("842", "Provision of services to the community as a whole", Category.GROUP),
        Classification("8421", "Foreign affairs", Category.CLASS),
        Classification("8422", "Defence activities", Category.CLASS),
        Classification("8423", "Public order and safety activities", Category.CLASS),
        Classification("843", "Compulsory social security activities", Category.GROUP),
        Classification("8430", "Compulsory social security activities", Category.CLASS),
        Classification("P", "Education", Category.SECTION),
        Classification("85", "Education", Category.DIVISION),
        Classification("851", "Pre-primary and primary education", Category.GROUP),
        Classification("8510", "Pre-primary and primary education", Category.CLASS),
        Classification("852", "Secondary education", Category.GROUP),
        Classification("8521", "General secondary education", Category.CLASS),
        Classification("8522", "Technical and vocational secondary education", Category.CLASS),
        Classification("853", "Higher education", Category.GROUP),
        Classification("8530", "Higher education", Category.CLASS),
        Classification("854", "Other education", Category.GROUP),
        Classification("8541", "Sports and recreation education", Category.CLASS),
        Classification("8542", "Cultural education", Category.CLASS),
        Classification("8549", "Other education n.e.c.", Category.CLASS),
        Classification("855", "Educational support activities", Category.GROUP),
        Classification("8550", "Educational support activities", Category.CLASS),
        Classification("Q", "Human health and social work activities", Category.SECTION),
        Classification("86", "Human health activities", Category.DIVISION),
        Classification("861", "Hospital activities", Category.GROUP),
        Classification("8610", "Hospital activities", Category.CLASS),
        Classification("862", "Medical and dental practice activities", Category.GROUP),
        Classification("8620", "Medical and dental practice activities", Category.CLASS),
        Classification("869", "Other human health activities", Category.GROUP),
        Classification("8690", "Other human health activities", Category.CLASS),
        Classification("87", "Residential care activities", Category.DIVISION),
        Classification("871", "Residential nursing care facilities", Category.GROUP),
        Classification("8710", "Residential nursing care facilities", Category.CLASS),
        Classification(
            "872",
            "Residential care activities for mental retardation, mental health and substance abuse",
            Category.GROUP,
        ),
        Classification(
            "8720",
            "Residential care activities for mental retardation, mental health and substance abuse",
            Category.CLASS,
        ),
        Classification(
            "873",
            "Residential care activities for the elderly and disabled",
            Category.GROUP,
        ),
        Classification(
            "8730",
            "Residential care activities for the elderly and disabled",
            Category.CLASS,
        ),
        Classification("879", "Other residential care activities", Category.GROUP),
        Classification("8790", "Other residential care activities", Category.CLASS),
        Classification("88", "Social work activities without accommodation", Category.DIVISION),
        Classification(
            "881",
            "Social work activities without accommodation for the elderly and disabled",
            Category.GROUP,
        ),
        Classification(
            "8810",
            "Social work activities without accommodation for the elderly and disabled",
            Category.CLASS,
        ),
        Classification("889", "Other social work activities without accommodation", Category.GROUP),
        Classification("8890", "Other social work activities without accommodation", Category.CLASS),
        Classification("R", "Arts, entertainment and recreation", Category.SECTION),
        Classification("90", "Creative, arts and entertainment activities", Category.DIVISION),
        Classification("900", "Creative, arts and entertainment activities", Category.GROUP),
        Classification("9000", "Creative, arts and entertainment activities", Category.CLASS),
        Classification(
            "91",
            "Libraries, archives, museums and other cultural activities",
            Category.DIVISION,
        ),
        Classification(
            "910",
            "Libraries, archives, museums and other cultural activities",
            Category.GROUP,
        ),
        Classification("9101", "Library and archives activities", Category.CLASS),
        Classification(
            "9102",
            "Museums activities and operation of historical sites and buildings",
            Category.CLASS,
        ),
        Classification(
            "9103",
            "Botanical and zoological gardens and nature reserves activities",
            Category.CLASS,
        ),
        Classification("92", "Gambling and betting activities", Category.DIVISION),
        Classification("920", "Gambling and betting activities", Category.GROUP),
        Classification("9200", "Gambling and betting activities", Category.CLASS),
        Classification(
            "93",
            "Sports activities and amusement and recreation activities",
            Category.DIVISION,
        ),
        Classification("931", "Sports activities", Category.GROUP),
        Classification("9311", "Operation of sports facilities", Category.CLASS),
        Classification("9312", "Activities of sports clubs", Category.CLASS),
        Classification("9319", "Other sports activities", Category.CLASS),
        Classification("932", "Other amusement and recreation activities", Category.GROUP),
        Classification("9321", "Activities of amusement parks and theme parks", Category.CLASS),
        Classification("9329", "Other amusement and recreation activities n.e.c.", Category.CLASS),
        Classification("S", "Other service activities", Category.SECTION),
        Classification("94", "Activities of membership organizations", Category.DIVISION),
        Classification(
            "941",
            "Activities of business, employers and professional membership organizations",
            Category.GROUP,
        ),
        Classification(
            "9411",
            "Activities of business and employers membership organizations",
            Category.CLASS,
        ),
        Classification(
            "9412",
            "Activities of professional membership organizations",
            Category.CLASS,
        ),
        Classification("942", "Activities of trade unions", Category.GROUP),
        Classification("9420", "Activities of trade unions", Category.CLASS),
        Classification("949", "Activities of other membership organizations", Category.GROUP),
        Classification("9491", "Activities of religious organizations", Category.CLASS),
        Classification("9492", "Activities of political organizations", Category.CLASS),
        Classification(
            "9499",
            "Activities of other membership organizations n.e.c.",
            Category.CLASS,
        ),
        Classification(
            "95",
            "Repair of computers and personal and household goods",
            Category.DIVISION,
        ),
        Classification("951", "Repair of computers and communication equipment", Category.GROUP),
        Classification("9511", "Repair of computers and peripheral equipment", Category.CLASS),
        Classification("9512", "Repair of communication equipment", Category.CLASS),
        Classification("952", "Repair of personal and household goods", Category.GROUP),
        Classification("9521", "Repair of consumer electronics", Category.CLASS),
        Classification(
            "9522",
            "Repair of household appliances and home and garden equipment",
            Category.CLASS,
        ),
        Classification("9523", "Repair of footwear and leather goods", Category.CLASS),
        Classification("9524", "Repair of furniture and home furnishings", Category.CLASS),
        Classification("9529", "Repair of other personal and household goods", Category.CLASS),
        Classification("96", "Other personal service activities", Category.DIVISION),
        Classification("960", "Other personal service activities", Category.GROUP),
        Classification(
            "9601",
            "Washing and (dry-), cleaning of textile and fur products",
            Category.CLASS,
        ),
        Classification("9602", "Hairdressing and other beauty treatment", Category.CLASS),
        Classification("9603", "Funeral and related activities", Category.CLASS),
        Classification("9609", "Other personal service activities n.e.c.", Category.CLASS),
        Classification(
            "T",
            "Activities of households as employers; undifferentiated goods- and services-producing activities of households for own use",
            Category.SECTION,
        ),
        Classification(
            "97",
            "Activities of households as employers of domestic personnel",
            Category.DIVISION,
        ),
        Classification(
            "970",
            "Activities of households as employers of domestic personnel",
            Category.GROUP,
        ),
        Classification(
            "9700",
            "Activities of households as employers of domestic personnel",
            Category.CLASS,
        ),
        Classification(
            "98",
            "Undifferentiated goods- and services-producing activities of private households for own use",
            Category.DIVISION,
        ),
        Classification(
            "981",
            "Undifferentiated goods-producing activities of private households for own use",
            Category.GROUP,
        ),
        Classification(
            "9810",
            "Undifferentiated goods-producing activities of private households for own use",
            Category.CLASS,
        ),
        Classification(
            "982",
            "Undifferentiated service-producing activities of private households for own use",
            Category.GROUP,
        ),
        Classification(
            "9820",
            "Undifferentiated service-producing activities of private households for own use",
            Category.CLASS,
        ),
        Classification(
            "U",
            "Activities of extraterritorial organizations and bodies",
            Category.SECTION,
        ),
        Classification(
            "99",
            "Activities of extraterritorial organizations and bodies",
            Category.DIVISION,
        ),
        Classification(
            "990",
            "Activities of extraterritorial organizations and bodies",
            Category.GROUP,
        ),
        Classification(
            "9900",
            "Activities of extraterritorial organizations and bodies",
            Category.CLASS,
        ),
    ],
)
