# -*- coding: utf-8 -*-
# vim: tabstop=4 shiftwidth=4 softtabstop=4
#
# Copyright (C) 2012-2021 GEM Foundation
#
# OpenQuake is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenQuake is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with OpenQuake. If not, see <http://www.gnu.org/licenses/>.

from openquake.hazardlib.gsim.base import CoeffsTable

#: Coefficient table constructed from the electronic suplements of the
#: original paper.

COEFFS_FS_ROCK_SWISS01 = CoeffsTable(sa_damping=5, table="""\
    IMT     k_adj          a1            a2            b1            b2            Rm            phi_11      phi_21      C2          Mc1    Mc2    Rc11    Rc21   mean_phi_ss
    pga     0.845181       9.890787E-01  1.094114E+00  1.000921E+00  6.455198E-01  4.242559E+00  0.58        0.47        0.35        5      7      16      36     0.46
    0.05    0.928791564    7.422166E-01  8.879871E-01  1.000934E+00  6.131160E-01  3.908670E+00  0.5520412   0.4490309   0.4059176   5      7      16      36     0.4530103
    0.1     0.869538666    7.315855E-01  9.854400E-01  1.000728E+00  4.859887E-01  2.551226E+00  0.54        0.44        0.43        5      7      16      36     0.45
    0.15    0.866689265    9.065195E-01  1.137244E+00  1.000716E+00  4.986533E-01  2.688280E+00  0.580947375 0.47509775  0.400751875 5      7      16      36     0.467548875
    0.2     0.870918939    1.030637E+00  1.258931E+00  1.000707E+00  5.076390E-01  2.785522E+00  0.61        0.5         0.38        5      7      16      36     0.48
    0.25    0.890550333    1.097448E+00  1.328856E+00  1.000697E+00  5.132224E-01  2.845948E+00  0.626510191 0.5         0.374496603 5      7      16      36     0.48
    0.3     0.911248241    1.152036E+00  1.388863E+00  1.000689E+00  5.177843E-01  2.895320E+00  0.64        0.5         0.37        5      7      16      36     0.48
    0.35    0.925168921    1.225808E+00  1.467951E+00  1.000686E+00  5.359156E-01  3.090605E+00  0.627929292 0.493964646 0.37        5      7      16      36     0.473964646
    0.4     0.940177136    1.289869E+00  1.540248E+00  1.000683E+00  5.517024E-01  3.260638E+00  0.617473168 0.488736584 0.37        5      7      16      36     0.468736584
    0.45    0.958216066    1.272552E+00  1.515365E+00  1.000683E+00  5.452414E-01  3.190961E+00  0.608250214 0.484125107 0.37        5      7      16      36     0.464125107
    0.5     0.971079656    1.257061E+00  1.493446E+00  1.000683E+00  5.394618E-01  3.128633E+00  0.6         0.48        0.37        5      7      16      36     0.46
    0.55    0.975571902    1.243048E+00  1.473892E+00  1.000683E+00  5.342336E-01  3.072250E+00  0.591749789 0.475874894 0.374125106 5      7      16      36     0.458624965
    0.6     0.98826925     1.230256E+00  1.456264E+00  1.000682E+00  5.294605E-01  3.020777E+00  0.584217936 0.472108968 0.377891032 5      7      16      36     0.457369656
    0.65    0.994322759    1.218487E+00  1.440234E+00  1.000682E+00  5.250697E-01  2.973426E+00  0.577289303 0.468644651 0.381355349 5      7      16      36     0.456214884
    0.7     1.00006189     1.207592E+00  1.425550E+00  1.000682E+00  5.210045E-01  2.929586E+00  0.57087439  0.465437195 0.384562805 5      7      16      36     0.455145732
    0.75    1.001923505    1.197448E+00  1.412014E+00  1.000682E+00  5.172199E-01  2.888772E+00  0.56490225  0.462451125 0.387548875 5      7      16      36     0.454150375
    0.8     1.008323171    1.187959E+00  1.399468E+00  1.000682E+00  5.136796E-01  2.850593E+00  0.559315686 0.459657843 0.390342157 5      7      16      36     0.453219281
    0.85    1.008454694    1.179046E+00  1.387785E+00  1.000682E+00  5.103540E-01  2.814729E+00  0.554067915 0.457033958 0.392966042 5      7      16      36     0.452344653
    0.9     1.007119423    1.170642E+00  1.376859E+00  1.000682E+00  5.072186E-01  2.780916E+00  0.549120186 0.454560093 0.395439907 5      7      16      36     0.451520031
    0.95    1.010757554    1.162693E+00  1.366603E+00  1.000682E+00  5.042527E-01  2.748931E+00  0.544440035 0.452220017 0.397779983 5      7      16      36     0.450740006
    1       1.009517831    1.155152E+00  1.356944E+00  1.000682E+00  5.014390E-01  2.718588E+00  0.54        0.45        0.4         5      7      16      36     0.45
    1.05    1.008523959    1.089756E+00  1.261431E+00  1.000668E+00  4.882475E-01  2.585423E+00  0.539555893 0.447779464 0.4         5      7      16      36     0.448223571
    1.1     1.005447172    1.027402E+00  1.176630E+00  1.000656E+00  4.756698E-01  2.458455E+00  0.539132449 0.445662247 0.4         5      7      16      36     0.446529797
    1.15    1.00259109     9.678210E-01  1.100931E+00  1.000644E+00  4.636512E-01  2.337132E+00  0.538727832 0.44363916  0.4         5      7      16      36     0.444911328
    1.2     1.002753584    9.107759E-01  1.033022E+00  1.000632E+00  4.521443E-01  2.220973E+00  0.538340438 0.441702188 0.4         5      7      16      36     0.443361751
    1.25    1.002181169    8.560599E-01  9.718251E-01  1.000621E+00  4.411072E-01  2.109557E+00  0.53796886  0.439844299 0.4         5      7      16      36     0.441875439
    1.3     1.000805599    8.034901E-01  9.164459E-01  1.000610E+00  4.305030E-01  2.002510E+00  0.537611858 0.438059288 0.4         5      7      16      36     0.44044743
    1.35    0.999180834    7.529046E-01  8.661386E-01  1.000600E+00  4.202990E-01  1.899505E+00  0.53726833  0.436341651 0.4         5      7      16      36     0.439073321
    1.4     0.997612099    7.041590E-01  8.202755E-01  1.000590E+00  4.104662E-01  1.800246E+00  0.536937298 0.434686489 0.4         5      7      16      36     0.437749191
    1.45    0.99634652     6.571241E-01  7.783256E-01  1.000581E+00  4.009785E-01  1.704470E+00  0.536617883 0.433089414 0.4         5      7      16      36     0.436471531
    1.5     0.995286842    6.116840E-01  7.398367E-01  1.000571E+00  3.918124E-01  1.611942E+00  0.536309298 0.431546488 0.4         5      7      16      36     0.43523719
    1.55    0.994262081    5.677339E-01  7.044217E-01  1.000562E+00  3.829470E-01  1.522448E+00  0.536010832 0.430054159 0.4         5      7      16      36     0.434043327
    1.6     0.991773547    5.251794E-01  6.717473E-01  1.000554E+00  3.743630E-01  1.435795E+00  0.535721843 0.428609213 0.4         5      7      16      36     0.43288737
    1.65    0.990228525    4.839344E-01  6.415255E-01  1.000545E+00  3.660432E-01  1.351809E+00  0.535441747 0.427208734 0.4         5      7      16      36     0.431766988
    1.7     0.991380905    4.439207E-01  6.135060E-01  1.000537E+00  3.579718E-01  1.270331E+00  0.535170014 0.425850068 0.4         5      7      16      36     0.430680054
    1.75    0.992034968    4.050671E-01  5.874704E-01  1.000529E+00  3.501344E-01  1.191215E+00  0.534906158 0.424530788 0.4         5      7      16      36     0.42962463
    1.8     0.989343718    3.673081E-01  5.632273E-01  1.000522E+00  3.425177E-01  1.114327E+00  0.534649735 0.423248676 0.4         5      7      16      36     0.428598941
    1.85    0.986733778    3.305837E-01  5.406085E-01  1.000514E+00  3.351098E-01  1.039547E+00  0.534400339 0.422001695 0.4         5      7      16      36     0.427601356
    1.9     0.985938621    2.948387E-01  5.194656E-01  1.000507E+00  3.278994E-01  9.667603E-01  0.534157594 0.420787971 0.4         5      7      16      36     0.426630377
    1.95    0.987982233    2.600223E-01  4.996672E-01  1.000500E+00  3.208764E-01  8.958647E-01  0.533921155 0.419605775 0.4         5      7      16      36     0.42568462
    2       0.987049461    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.533690702 0.418453512 0.4         5      7      16      36     0.42476281
    2.05    0.986014077    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.533465941 0.417329703 0.4         5      7      16      36     0.423863762
    2.1     0.98500662     2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.533246595 0.416232976 0.4         5      7      16      36     0.422986381
    2.15    0.983824939    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.533032411 0.415162057 0.4         5      7      16      36     0.422129646
    2.2     0.979567253    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.532823152 0.414115759 0.4         5      7      16      36     0.421292607
    2.25    0.980977982    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.532618595 0.413092975 0.4         5      7      16      36     0.42047438
    2.3     0.982358258    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.532418534 0.412092672 0.4         5      7      16      36     0.419674138
    2.35    0.983710352    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.532222777 0.411113883 0.4         5      7      16      36     0.418891106
    2.4     0.984009756    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53203114  0.410155701 0.4         5      7      16      36     0.418124561
    2.45    0.981567191    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.531843455 0.409217276 0.4         5      7      16      36     0.417373821
    2.5     0.97882924     2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.531659562 0.408297812 0.4         5      7      16      36     0.416638249
    2.55    0.977128157    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.531479311 0.407396555 0.4         5      7      16      36     0.415917244
    2.6     0.976217642    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53130256  0.4065128   0.4         5      7      16      36     0.41521024
    2.65    0.975327817    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.531129176 0.405645879 0.4         5      7      16      36     0.414516703
    2.7     0.975311743    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.530959033 0.404795164 0.4         5      7      16      36     0.413836131
    2.75    0.975452742    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.530792012 0.403960058 0.4         5      7      16      36     0.413168047
    2.8     0.97558962     2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.530628    0.403140001 0.4         5      7      16      36     0.412512001
    2.85    0.975724225    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.530466892 0.402334458 0.4         5      7      16      36     0.411867567
    2.9     0.975714048    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.530308585 0.401542926 0.4         5      7      16      36     0.411234341
    2.95    0.975258422    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.530152985 0.400764925 0.4         5      7      16      36     0.41061194
    3       0.974810662    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.05    0.974370503    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.1     0.973937695    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.15    0.973512001    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.2     0.973093192    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.25    0.972681052    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.3     0.972275376    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.35    0.971875966    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.4     0.971482633    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.45    0.971095199    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.5     0.97071349     2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.55    0.970337343    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.6     0.9699666      2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.65    0.969601109    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.7     0.969240726    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.75    0.968885311    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.8     0.968534732    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.85    0.968188861    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.9     0.967847573    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.95    0.967510751    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4       0.967178281    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.05    0.966850053    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.1     0.966525963    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.15    0.966205907    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.2     0.965889788    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.25    0.965577512    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.3     0.965268988    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.35    0.964964128    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.4     0.964662846    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.45    0.964365061    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.5     0.964070694    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.55    0.963779668    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.6     0.963491909    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.65    0.963207345    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.7     0.962925908    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.75    0.96264753     2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.8     0.962372146    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.85    0.962099694    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.9     0.961830112    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.95    0.961563341    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5       0.961299324    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.05    0.961038006    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.1     0.960779332    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.15    0.960523251    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.2     0.960269711    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.25    0.960018663    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.3     0.95977006     2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.35    0.959523854    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.4     0.959280001    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.45    0.959038457    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.5     0.958799179    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.55    0.958562125    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.6     0.958327255    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.65    0.958094529    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.7     0.95786391     2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.75    0.957635359    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.8     0.957408841    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.85    0.957184321    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.9     0.956961763    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.95    0.956741135    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6       0.956522403    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.05    0.956305536    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.1     0.956090502    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.15    0.955877271    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.2     0.955665814    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.25    0.955456102    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.3     0.955248106    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.35    0.955041799    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.4     0.954837154    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.45    0.954634146    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.5     0.954432747    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.55    0.954232934    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.6     0.954034682    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.65    0.953837966    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.7     0.953642764    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.75    0.953449054    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.8     0.953256811    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.85    0.953066016    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.9     0.952876645    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.95    0.95268868     2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7       0.952502098    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.05    0.952316881    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.1     0.952133008    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.15    0.95195046     2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.2     0.95176922     2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.25    0.951589267    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.3     0.951410586    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.35    0.951233157    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.4     0.951056964    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.45    0.950881989    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.5     0.950708217    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.55    0.950535631    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.6     0.950364215    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.65    0.950193954    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.7     0.950024833    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.75    0.949856835    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.8     0.949689948    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.85    0.949524156    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.9     0.949359445    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.95    0.949195802    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8       0.949033212    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.05    0.948871664    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.1     0.948711142    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.15    0.948551636    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.2     0.948393132    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.25    0.948235617    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.3     0.94807908     2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.35    0.947923509    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.4     0.947768893    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.45    0.947615218    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.5     0.947462475    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.55    0.947310652    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.6     0.947159739    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.65    0.947009725    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.7     0.946860598    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.75    0.94671235     2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.8     0.946564969    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.85    0.946418446    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.9     0.946272771    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.95    0.946127934    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9       0.945983926    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.05    0.945840738    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.1     0.94569836     2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.15    0.945556783    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.2     0.945415999    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.25    0.945275999    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.3     0.945136775    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.35    0.944998317    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.4     0.944860617    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.45    0.944723668    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.5     0.944587462    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.55    0.94445199     2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.6     0.944317245    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.65    0.944183218    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.7     0.944049903    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.75    0.943917293    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.8     0.943785379    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.85    0.943654155    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.9     0.943523613    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.95    0.943393747    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    10.0    0.943393747    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    """)

COEFFS_FS_ROCK_SWISS04 = CoeffsTable(sa_damping=5, table="""\
    IMT     k_adj         a1             a2            b1            b2            Rm            phi_11      phi_21      C2          Mc1    Mc2    Rc11    Rc21   mean_phi_ss
    pga     0.640173       9.890787E-01  1.094114E+00  1.000921E+00  6.455198E-01  4.242559E+00  0.58        0.47        0.35        5      7      16      36     0.46
    0.05    0.597157584    7.422166E-01  8.879871E-01  1.000934E+00  6.131160E-01  3.908670E+00  0.5520412   0.4490309   0.4059176   5      7      16      36     0.4530103
    0.1     0.673802922    7.315855E-01  9.854400E-01  1.000728E+00  4.859887E-01  2.551226E+00  0.54        0.44        0.43        5      7      16      36     0.45
    0.15    0.726135284    9.065195E-01  1.137244E+00  1.000716E+00  4.986533E-01  2.688280E+00  0.580947375 0.47509775  0.400751875 5      7      16      36     0.467548875
    0.2     0.760991555    1.030637E+00  1.258931E+00  1.000707E+00  5.076390E-01  2.785522E+00  0.61        0.5         0.38        5      7      16      36     0.48
    0.25    0.799375899    1.097448E+00  1.328856E+00  1.000697E+00  5.132224E-01  2.845948E+00  0.626510191 0.5         0.374496603 5      7      16      36     0.48
    0.3     0.83245448     1.152036E+00  1.388863E+00  1.000689E+00  5.177843E-01  2.895320E+00  0.64        0.5         0.37        5      7      16      36     0.48
    0.35    0.855722155    1.225808E+00  1.467951E+00  1.000686E+00  5.359156E-01  3.090605E+00  0.627929292 0.493964646 0.37        5      7      16      36     0.473964646
    0.4     0.878000705    1.289869E+00  1.540248E+00  1.000683E+00  5.517024E-01  3.260638E+00  0.617473168 0.488736584 0.37        5      7      16      36     0.468736584
    0.45    0.901534875    1.272552E+00  1.515365E+00  1.000683E+00  5.452414E-01  3.190961E+00  0.608250214 0.484125107 0.37        5      7      16      36     0.464125107
    0.5     0.919107265    1.257061E+00  1.493446E+00  1.000683E+00  5.394618E-01  3.128633E+00  0.6         0.48        0.37        5      7      16      36     0.46
    0.55    0.928001534    1.243048E+00  1.473892E+00  1.000683E+00  5.342336E-01  3.072250E+00  0.591749789 0.475874894 0.374125106 5      7      16      36     0.458624965
    0.6     0.944047082    1.230256E+00  1.456264E+00  1.000682E+00  5.294605E-01  3.020777E+00  0.584217936 0.472108968 0.377891032 5      7      16      36     0.457369656
    0.65    0.953221765    1.218487E+00  1.440234E+00  1.000682E+00  5.250697E-01  2.973426E+00  0.577289303 0.468644651 0.381355349 5      7      16      36     0.456214884
    0.7     0.961685383    1.207592E+00  1.425550E+00  1.000682E+00  5.210045E-01  2.929586E+00  0.57087439  0.465437195 0.384562805 5      7      16      36     0.455145732
    0.75    0.966082633    1.197448E+00  1.412014E+00  1.000682E+00  5.172199E-01  2.888772E+00  0.56490225  0.462451125 0.387548875 5      7      16      36     0.454150375
    0.8     0.974514653    1.187959E+00  1.399468E+00  1.000682E+00  5.136796E-01  2.850593E+00  0.559315686 0.459657843 0.390342157 5      7      16      36     0.453219281
    0.85    0.976675432    1.179046E+00  1.387785E+00  1.000682E+00  5.103540E-01  2.814729E+00  0.554067915 0.457033958 0.392966042 5      7      16      36     0.452344653
    0.9     0.97714973     1.170642E+00  1.376859E+00  1.000682E+00  5.072186E-01  2.780916E+00  0.549120186 0.454560093 0.395439907 5      7      16      36     0.451520031
    0.95    0.982274187    1.162693E+00  1.366603E+00  1.000682E+00  5.042527E-01  2.748931E+00  0.544440035 0.452220017 0.397779983 5      7      16      36     0.450740006
    1       0.982496383    1.155152E+00  1.356944E+00  1.000682E+00  5.014390E-01  2.718588E+00  0.54        0.45        0.4         5      7      16      36     0.45
    1.05    0.982850994    1.089756E+00  1.261431E+00  1.000668E+00  4.882475E-01  2.585423E+00  0.539555893 0.447779464 0.4         5      7      16      36     0.448223571
    1.1     0.981053597    1.027402E+00  1.176630E+00  1.000656E+00  4.756698E-01  2.458455E+00  0.539132449 0.445662247 0.4         5      7      16      36     0.446529797
    1.15    0.979405438    9.678210E-01  1.100931E+00  1.000644E+00  4.636512E-01  2.337132E+00  0.538727832 0.44363916  0.4         5      7      16      36     0.444911328
    1.2     0.980583735    9.107759E-01  1.033022E+00  1.000632E+00  4.521443E-01  2.220973E+00  0.538340438 0.441702188 0.4         5      7      16      36     0.443361751
    1.25    0.981003986    8.560599E-01  9.718251E-01  1.000621E+00  4.411072E-01  2.109557E+00  0.53796886  0.439844299 0.4         5      7      16      36     0.441875439
    1.3     0.98056576     8.034901E-01  9.164459E-01  1.000610E+00  4.305030E-01  2.002510E+00  0.537611858 0.438059288 0.4         5      7      16      36     0.44044743
    1.35    0.979810678    7.529046E-01  8.661386E-01  1.000600E+00  4.202990E-01  1.899505E+00  0.53726833  0.436341651 0.4         5      7      16      36     0.439073321
    1.4     0.979081467    7.041590E-01  8.202755E-01  1.000590E+00  4.104662E-01  1.800246E+00  0.536937298 0.434686489 0.4         5      7      16      36     0.437749191
    1.45    0.978587593    6.571241E-01  7.783256E-01  1.000581E+00  4.009785E-01  1.704470E+00  0.536617883 0.433089414 0.4         5      7      16      36     0.436471531
    1.5     0.978258611    6.116840E-01  7.398367E-01  1.000571E+00  3.918124E-01  1.611942E+00  0.536309298 0.431546488 0.4         5      7      16      36     0.43523719
    1.55    0.977940012    5.677339E-01  7.044217E-01  1.000562E+00  3.829470E-01  1.522448E+00  0.536010832 0.430054159 0.4         5      7      16      36     0.434043327
    1.6     0.976130904    5.251794E-01  6.717473E-01  1.000554E+00  3.743630E-01  1.435795E+00  0.535721843 0.428609213 0.4         5      7      16      36     0.43288737
    1.65    0.975206197    4.839344E-01  6.415255E-01  1.000545E+00  3.660432E-01  1.351809E+00  0.535441747 0.427208734 0.4         5      7      16      36     0.431766988
    1.7     0.976925074    4.439207E-01  6.135060E-01  1.000537E+00  3.579718E-01  1.270331E+00  0.535170014 0.425850068 0.4         5      7      16      36     0.430680054
    1.75    0.978127313    4.050671E-01  5.874704E-01  1.000529E+00  3.501344E-01  1.191215E+00  0.534906158 0.424530788 0.4         5      7      16      36     0.42962463
    1.8     0.975979818    3.673081E-01  5.632273E-01  1.000522E+00  3.425177E-01  1.114327E+00  0.534649735 0.423248676 0.4         5      7      16      36     0.428598941
    1.85    0.973895833    3.305837E-01  5.406085E-01  1.000514E+00  3.351098E-01  1.039547E+00  0.534400339 0.422001695 0.4         5      7      16      36     0.427601356
    1.9     0.973586963    2.948387E-01  5.194656E-01  1.000507E+00  3.278994E-01  9.667603E-01  0.534157594 0.420787971 0.4         5      7      16      36     0.426630377
    1.95    0.976054855    2.600223E-01  4.996672E-01  1.000500E+00  3.208764E-01  8.958647E-01  0.533921155 0.419605775 0.4         5      7      16      36     0.42568462
    2       0.975530203    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.533690702 0.418453512 0.4         5      7      16      36     0.42476281
    2.05    0.974895061    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.533465941 0.417329703 0.4         5      7      16      36     0.423863762
    2.1     0.974275032    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.533246595 0.416232976 0.4         5      7      16      36     0.422986381
    2.15    0.973472694    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.533032411 0.415162057 0.4         5      7      16      36     0.422129646
    2.2     0.9695248      2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.532823152 0.414115759 0.4         5      7      16      36     0.421292607
    2.25    0.971170255    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.532618595 0.413092975 0.4         5      7      16      36     0.42047438
    2.3     0.972781335    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.532418534 0.412092672 0.4         5      7      16      36     0.419674138
    2.35    0.974363913    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.532222777 0.411113883 0.4         5      7      16      36     0.418891106
    2.4     0.974871807    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53203114  0.410155701 0.4         5      7      16      36     0.418124561
    2.45    0.972596986    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.531843455 0.409217276 0.4         5      7      16      36     0.417373821
    2.5     0.970016734    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.531659562 0.408297812 0.4         5      7      16      36     0.416638249
    2.55    0.968460851    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.531479311 0.407396555 0.4         5      7      16      36     0.415917244
    2.6     0.967688054    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53130256  0.4065128   0.4         5      7      16      36     0.41521024
    2.65    0.966928939    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.531129176 0.405645879 0.4         5      7      16      36     0.414516703
    2.7     0.966992431    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.530959033 0.404795164 0.4         5      7      16      36     0.413836131
    2.75    0.967201619    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.530792012 0.403960058 0.4         5      7      16      36     0.413168047
    2.8     0.967403852    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.530628    0.403140001 0.4         5      7      16      36     0.412512001
    2.85    0.967600194    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.530466892 0.402334458 0.4         5      7      16      36     0.411867567
    2.9     0.967656357    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.530308585 0.401542926 0.4         5      7      16      36     0.411234341
    2.95    0.967270536    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.530152985 0.400764925 0.4         5      7      16      36     0.41061194
    3       0.966891349    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.05    0.966518576    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.1     0.966152004    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.15    0.965791433    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.2     0.965436672    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.25    0.965087539    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.3     0.964743859    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.35    0.964405468    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.4     0.964072206    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.45    0.963743922    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.5     0.963420471    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.55    0.963101714    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.6     0.962787519    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.65    0.962477759    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.7     0.96217231     2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.75    0.961871056    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.8     0.961573886    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.85    0.96128069     2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.9     0.960991364    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    3.95    0.960705811    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4       0.960423932    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.05    0.960145636    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.1     0.959870834    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.15    0.95959944     2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.2     0.959331372    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.25    0.95906655     2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.3     0.958804898    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.35    0.95854634     2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.4     0.958290806    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.45    0.958038226    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.5     0.957788534    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.55    0.957541665    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.6     0.957297557    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.65    0.957056149    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.7     0.956817383    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.75    0.956581202    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.8     0.956347552    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.85    0.956116379    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.9     0.955887632    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    4.95    0.955661262    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5       0.955437219    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.05    0.955215458    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.1     0.954995932    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.15    0.954778597    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.2     0.954563411    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.25    0.954350332    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.3     0.95413932     2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.35    0.953930335    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.4     0.95372334     2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.45    0.953518296    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.5     0.953315169    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.55    0.953113922    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.6     0.952914523    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.65    0.952716937    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.7     0.952521132    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.75    0.952327077    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.8     0.952134741    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.85    0.951944095    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.9     0.951755109    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    5.95    0.951567754    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6       0.951382004    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.05    0.951197832    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.1     0.95101521     2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.15    0.950834114    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.2     0.950654518    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.25    0.950476398    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.3     0.950299731    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.35    0.950124493    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.4     0.94995066     2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.45    0.949778213    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.5     0.949607128    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.55    0.949437384    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.6     0.949268961    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.65    0.949101839    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.7     0.948935997    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.75    0.948771418    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.8     0.948608081    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.85    0.948445969    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.9     0.948285063    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    6.95    0.948125346    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7       0.9479668      2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.05    0.947809409    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.1     0.947653157    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.15    0.947498026    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.2     0.947344002    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.25    0.947191068    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.3     0.947039209    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.35    0.946888412    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.4     0.94673866     2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.45    0.946589941    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.5     0.946442239    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.55    0.946295542    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.6     0.946149835    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.65    0.946005106    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.7     0.945861342    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.75    0.94571853     2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.8     0.945576658    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.85    0.945435713    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.9     0.945295684    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    7.95    0.945156559    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8       0.945018327    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.05    0.944880976    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.1     0.944744495    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.15    0.944608874    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.2     0.944474101    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.25    0.944340167    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.3     0.944207061    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.35    0.944074773    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.4     0.943943293    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.45    0.943812611    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.5     0.943682719    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.55    0.943553606    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.6     0.943425263    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.65    0.943297682    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.7     0.943170853    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.75    0.943044767    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.8     0.942919417    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.85    0.942794794    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.9     0.942670889    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    8.95    0.942547695    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9       0.942425202    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.05    0.942303404    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.1     0.942182293    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.15    0.942061861    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.2     0.941942101    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.25    0.941823004    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.3     0.941704565    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.35    0.941586775    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.4     0.941469629    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.45    0.941353118    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.5     0.941237236    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.55    0.941121977    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.6     0.941007334    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.65    0.9408933      2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.7     0.940779869    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.75    0.940667035    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.8     0.940554792    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.85    0.940443133    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.9     0.940332052    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    9.95    0.940221544    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    10.0    0.940221544    2.260874E-01  4.810965E-01  1.000493E+00  3.140312E-01  8.267642E-01  0.53        0.4         0.4         5      7      16      36     0.41
    """)

COEFFS_FS_ROCK_SWISS08 = CoeffsTable(sa_damping=5, table="""\
    IMT   k_adj       a1               a2           b1           b2           Rm           phi_11      phi_21      C2          Mc1 Mc2 Rc11 Rc21     mean_phi_ss
    pga   0.634915    9.890787E-01     1.094114E+00 1.000921E+00 6.455198E-01 4.242559E+00 0.58        0.47        0.35        5   7   16   36       0.46
    0.05  0.620699347 7.422166E-01     8.879871E-01 1.000934E+00 6.131160E-01 3.908670E+00 0.5520412   0.4490309   0.4059176   5   7   16   36       0.4530103
    0.1   0.683765292 7.315855E-01     9.854400E-01 1.000728E+00 4.859887E-01 2.551226E+00 0.54        0.44        0.43        5   7   16   36       0.45
    0.15  0.713726614 9.065195E-01     1.137244E+00 1.000716E+00 4.986533E-01 2.688280E+00 0.580947375 0.47509775  0.400751875 5   7   16   36       0.467548875
    0.2   0.72398324  1.030637E+00     1.258931E+00 1.000707E+00 5.076390E-01 2.785522E+00 0.61        0.5         0.38        5   7   16   36       0.48
    0.25  0.737445481 1.097448E+00     1.328856E+00 1.000697E+00 5.132224E-01 2.845948E+00 0.626510191 0.5         0.374496603 5   7   16   36       0.48
    0.3   0.748056847 1.152036E+00     1.388863E+00 1.000689E+00 5.177843E-01 2.895320E+00 0.64        0.5         0.37        5   7   16   36       0.48
    0.35  0.751859753 1.225808E+00     1.467951E+00 1.000686E+00 5.359156E-01 3.090605E+00 0.627929292 0.493964646 0.37        5   7   16   36       0.473964646
    0.4   0.756718138 1.289869E+00     1.540248E+00 1.000683E+00 5.517024E-01 3.260638E+00 0.617473168 0.488736584 0.37        5   7   16   36       0.468736584
    0.45  0.765064892 1.272552E+00     1.515365E+00 1.000683E+00 5.452414E-01 3.190961E+00 0.608250214 0.484125107 0.37        5   7   16   36       0.464125107
    0.5   0.770763824 1.257061E+00     1.493446E+00 1.000683E+00 5.394618E-01 3.128633E+00 0.6         0.48        0.37        5   7   16   36       0.46
    0.55  0.771281651 1.243048E+00     1.473892E+00 1.000683E+00 5.342336E-01 3.072250E+00 0.591749789 0.475874894 0.374125106 5   7   16   36       0.458624965
    0.6   0.779840742 1.230256E+00     1.456264E+00 1.000682E+00 5.294605E-01 3.020777E+00 0.584217936 0.472108968 0.377891032 5   7   16   36       0.457369656
    0.65  0.784559105 1.218487E+00     1.440234E+00 1.000682E+00 5.250697E-01 2.973426E+00 0.577289303 0.468644651 0.381355349 5   7   16   36       0.456214884
    0.7   0.790071489 1.207592E+00     1.425550E+00 1.000682E+00 5.210045E-01 2.929586E+00 0.57087439  0.465437195 0.384562805 5   7   16   36       0.455145732
    0.75  0.793332124 1.197448E+00     1.412014E+00 1.000682E+00 5.172199E-01 2.888772E+00 0.56490225  0.462451125 0.387548875 5   7   16   36       0.454150375
    0.8   0.801399028 1.187959E+00     1.399468E+00 1.000682E+00 5.136796E-01 2.850593E+00 0.559315686 0.459657843 0.390342157 5   7   16   36       0.453219281
    0.85  0.804823898 1.179046E+00     1.387785E+00 1.000682E+00 5.103540E-01 2.814729E+00 0.554067915 0.457033958 0.392966042 5   7   16   36       0.452344653
    0.9   0.807757055 1.170642E+00     1.376859E+00 1.000682E+00 5.072186E-01 2.780916E+00 0.549120186 0.454560093 0.395439907 5   7   16   36       0.451520031
    0.95  0.815034482 1.162693E+00     1.366603E+00 1.000682E+00 5.042527E-01 2.748931E+00 0.544440035 0.452220017 0.397779983 5   7   16   36       0.450740006
    1     0.818818579 1.155152E+00     1.356944E+00 1.000682E+00 5.014390E-01 2.718588E+00 0.54        0.45        0.4         5   7   16   36       0.45
    1.05  0.822926232 1.089756E+00     1.261431E+00 1.000668E+00 4.882475E-01 2.585423E+00 0.539555893 0.447779464 0.4         5   7   16   36       0.448223571
    1.1   0.825609755 1.027402E+00     1.176630E+00 1.000656E+00 4.756698E-01 2.458455E+00 0.539132449 0.445662247 0.4         5   7   16   36       0.446529797
    1.15  0.828296319 9.678210E-01     1.100931E+00 1.000644E+00 4.636512E-01 2.337132E+00 0.538727832 0.44363916  0.4         5   7   16   36       0.444911328
    1.2   0.833670749 9.107759E-01     1.033022E+00 1.000632E+00 4.521443E-01 2.220973E+00 0.538340438 0.441702188 0.4         5   7   16   36       0.443361751
    1.25  0.838249017 8.560599E-01     9.718251E-01 1.000621E+00 4.411072E-01 2.109557E+00 0.53796886  0.439844299 0.4         5   7   16   36       0.441875439
    1.3   0.842113846 8.034901E-01     9.164459E-01 1.000610E+00 4.305030E-01 2.002510E+00 0.537611858 0.438059288 0.4         5   7   16   36       0.44044743
    1.35  0.845683379 7.529046E-01     8.661386E-01 1.000600E+00 4.202990E-01 1.899505E+00 0.53726833  0.436341651 0.4         5   7   16   36       0.439073321
    1.4   0.849135019 7.041590E-01     8.202755E-01 1.000590E+00 4.104662E-01 1.800246E+00 0.536937298 0.434686489 0.4         5   7   16   36       0.437749191
    1.45  0.852694829 6.571241E-01     7.783256E-01 1.000581E+00 4.009785E-01 1.704470E+00 0.536617883 0.433089414 0.4         5   7   16   36       0.436471531
    1.5   0.856291008 6.116840E-01     7.398367E-01 1.000571E+00 3.918124E-01 1.611942E+00 0.536309298 0.431546488 0.4         5   7   16   36       0.43523719
    1.55  0.859780241 5.677339E-01     7.044217E-01 1.000562E+00 3.829470E-01 1.522448E+00 0.536010832 0.430054159 0.4         5   7   16   36       0.434043327
    1.6   0.861771482 5.251794E-01     6.717473E-01 1.000554E+00 3.743630E-01 1.435795E+00 0.535721843 0.428609213 0.4         5   7   16   36       0.43288737
    1.65  0.864391666 4.839344E-01     6.415255E-01 1.000545E+00 3.660432E-01 1.351809E+00 0.535441747 0.427208734 0.4         5   7   16   36       0.431766988
    1.7   0.869265354 4.439207E-01     6.135060E-01 1.000537E+00 3.579718E-01 1.270331E+00 0.535170014 0.425850068 0.4         5   7   16   36       0.430680054
    1.75  0.873584094 4.050671E-01     5.874704E-01 1.000529E+00 3.501344E-01 1.191215E+00 0.534906158 0.424530788 0.4         5   7   16   36       0.42962463
    1.8   0.874648223 3.673081E-01     5.632273E-01 1.000522E+00 3.425177E-01 1.114327E+00 0.534649735 0.423248676 0.4         5   7   16   36       0.428598941
    1.85  0.875685046 3.305837E-01     5.406085E-01 1.000514E+00 3.351098E-01 1.039547E+00 0.534400339 0.422001695 0.4         5   7   16   36       0.427601356
    1.9   0.878243917 2.948387E-01     5.194656E-01 1.000507E+00 3.278994E-01 9.667603E-01 0.534157594 0.420787971 0.4         5   7   16   36       0.426630377
    1.95  0.883171134 2.600223E-01     4.996672E-01 1.000500E+00 3.208764E-01 8.958647E-01 0.533921155 0.419605775 0.4         5   7   16   36       0.42568462
    2     0.88513988  2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.533690702 0.418453512 0.4         5   7   16   36       0.42476281
    2.05  0.886944501 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.533465941 0.417329703 0.4         5   7   16   36       0.423863762
    2.1   0.888708461 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.533246595 0.416232976 0.4         5   7   16   36       0.422986381
    2.15  0.890246098 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.533032411 0.415162057 0.4         5   7   16   36       0.422129646
    2.2   0.888576658 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.532823152 0.414115759 0.4         5   7   16   36       0.421292607
    2.25  0.891963655 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.532618595 0.413092975 0.4         5   7   16   36       0.42047438
    2.3   0.895288842 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.532418534 0.412092672 0.4         5   7   16   36       0.419674138
    2.35  0.898555163 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.532222777 0.411113883 0.4         5   7   16   36       0.418891106
    2.4   0.900737102 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53203114  0.410155701 0.4         5   7   16   36       0.418124561
    2.45  0.900132507 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.531843455 0.409217276 0.4         5   7   16   36       0.417373821
    2.5   0.899189919 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.531659562 0.408297812 0.4         5   7   16   36       0.416638249
    2.55  0.899163957 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.531479311 0.407396555 0.4         5   7   16   36       0.415917244
    2.6   0.899834564 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53130256  0.4065128   0.4         5   7   16   36       0.41521024
    2.65  0.90049399  2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.531129176 0.405645879 0.4         5   7   16   36       0.414516703
    2.7   0.901705075 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.530959033 0.404795164 0.4         5   7   16   36       0.413836131
    2.75  0.902993988 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.530792012 0.403960058 0.4         5   7   16   36       0.413168047
    2.8   0.904260801 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.530628    0.403140001 0.4         5   7   16   36       0.412512001
    2.85  0.905506282 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.530466892 0.402334458 0.4         5   7   16   36       0.411867567
    2.9   0.906601424 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.530308585 0.401542926 0.4         5   7   16   36       0.411234341
    2.95  0.907269213 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.530152985 0.400764925 0.4         5   7   16   36       0.41061194
    3     0.907926258 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    3.05  0.908572907 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    3.1   0.90920949  2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    3.15  0.909836322 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    3.2   0.910453705 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    3.25  0.911061925 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    3.3   0.911661255 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    3.35  0.912251959 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    3.4   0.912834285 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    3.45  0.913408473 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    3.5   0.913974752 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    3.55  0.914533343 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    3.6   0.915084455 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    3.65  0.915628291 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    3.7   0.916165044 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    3.75  0.9166949   2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    3.8   0.917218039 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    3.85  0.917734631 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    3.9   0.918244843 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    3.95  0.918748834 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    4     0.919246757 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    4.05  0.919738758 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    4.1   0.920224982 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    4.15  0.920705563 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    4.2   0.921180636 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    4.25  0.921650327 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    4.3   0.92211476  2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    4.35  0.922574054 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    4.4   0.923028323 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    4.45  0.92347768  2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    4.5   0.92392223  2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    4.55  0.924362079 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    4.6   0.924797326 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    4.65  0.92522807  2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    4.7   0.925654404 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    4.75  0.926076419 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    4.8   0.926494206 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    4.85  0.926907848 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    4.9   0.927317429 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    4.95  0.927723031 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    5     0.92812473  2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    5.05  0.928522604 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    5.1   0.928916726 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    5.15  0.929307168 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    5.2   0.929693999 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    5.25  0.930077287 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    5.3   0.930457098 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    5.35  0.930833495 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    5.4   0.931206541 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    5.45  0.931576296 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    5.5   0.931942819 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    5.55  0.932306167 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    5.6   0.932666397 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    5.65  0.933023561 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    5.7   0.933377714 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    5.75  0.933728907 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    5.8   0.934077189 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    5.85  0.934422609 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    5.9   0.934765216 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    5.95  0.935105056 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    6     0.935442174 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    6.05  0.935776615 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    6.1   0.93610842  2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    6.15  0.936437634 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    6.2   0.936764296 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    6.25  0.937088447 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    6.3   0.937410125 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    6.35  0.93772937  2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    6.4   0.938046219 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    6.45  0.938360707 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    6.5   0.938672871 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    6.55  0.938982746 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    6.6   0.939290365 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    6.65  0.939595763 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    6.7   0.939898971 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    6.75  0.940200021 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    6.8   0.940498945 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    6.85  0.940795773 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    6.9   0.941090534 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    6.95  0.941383259 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    7     0.941673976 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    7.05  0.941962712 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    7.1   0.942249495 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    7.15  0.942534353 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    7.2   0.94281731  2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    7.25  0.943098393 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    7.3   0.943377627 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    7.35  0.943655037 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    7.4   0.943930647 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    7.45  0.944204481 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    7.5   0.944476561 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    7.55  0.944746912 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    7.6   0.945015554 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    7.65  0.945282511 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    7.7   0.945547803 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    7.75  0.945811452 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    7.8   0.946073478 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    7.85  0.946333901 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    7.9   0.946592743 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    7.95  0.946850021 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    8     0.947105755 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    8.05  0.947359965 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    8.1   0.947612668 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    8.15  0.947863882 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    8.2   0.948113627 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    8.25  0.948361918 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    8.3   0.948608773 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    8.35  0.94885421  2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    8.4   0.949098244 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    8.45  0.949340892 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    8.5   0.94958217  2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    8.55  0.949822093 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    8.6   0.950060678 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    8.65  0.950297939 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    8.7   0.950533891 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    8.75  0.950768549 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    8.8   0.951001928 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    8.85  0.951234041 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    8.9   0.951464902 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    8.95  0.951694526 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    9     0.951922925 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    9.05  0.952150113 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    9.1   0.952376104 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    9.15  0.952600909 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    9.2   0.952824542 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    9.25  0.953047015 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    9.3   0.95326834  2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    9.35  0.953488529 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    9.4   0.953707595 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    9.45  0.953925548 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    9.5   0.9541424   2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    9.55  0.954358163 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    9.6   0.954572848 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    9.65  0.954786465 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    9.7   0.954999026 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    9.75  0.955210541 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    9.8   0.95542102  2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    9.85  0.955630475 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    9.9   0.955838914 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    9.95  0.956046349 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    10.0  0.956046349 2.260874E-01     4.810965E-01 1.000493E+00 3.140312E-01 8.267642E-01 0.53        0.4         0.4         5   7   16   36       0.41
    """)
