"""
This module contains the Message class, which is an abstract base class for all user-defined 
messages.

Created on Jan 9, 2021

:author: smalbadger
"""

import math
import inspect
from abc import ABC
from typing import Dict
from copy import deepcopy
from terminaltables import AsciiTable

from pymessagelib.field import Field
from pymessagelib._exceptions import (
    InvalidDataFormatException,
    MissingFieldDataException,
    InvalidFieldDataException,
    ConflictingContextsException,
)


class Message(ABC):
    """
    Subclasses of the Message class are dynamically generated by a MessageBuilder object. To
    generate subclasses, user-defined message definitions must be loaded into the MessageBuilder.
    Message definitions map field names to Field objects. The Field objects instantiated in the message
    definitions are not to be used for any other purpose than to define field attributes. Below is an
    example of a message definition:

    msg_defs = {
        "INPUTS" : {
            "service_req": Bit(1),
            "voltage_ready": Bit(1),
            "exit_code": Bytes(2),
            "last_command_id": Bits(2),
            "unused": Byte(1, value="x0"),
        }
    }

    In the example above, Bit, Bytes, Bits, and Byte are all derived from the Field class

    .. seealso:
        The Field class documentation describes valid Field subclasses and parameters in more detail.

    When a class for the above message definition is generated, a Message object can be instantiated
    using the name of the message as the constructor. Only values for writable fields can be specified.
    Writable fields with a default value can be specified if desired, but values for writable fields without a
    default value *MUST* be specified. For example:

    #TODO: Implement default values for writable fields

    INPUTS(service_req='b1', voltage_ready='b0', exit_code='x0001', last_command_id='b11').

    All values will be checked for validity based on criteria of the Field object in the message definition.

    .. seealso:
        The MessageBuilder class describes how to generate subclasses from message definitions

    Message objects can represent either an entire message or they can represent a portion of a message
    by being nested inside a field. Nesting fields allows for:

        - Less repeated code when there are multiple messages with similar components.
        - Assigning additional meaning to fields dynamically.
        - Establishing a hierarchy of information within messages.

    Individual fields can be compared, but comparison at the message level means comparing
    each field of 2 messages.
    """

    def __init__(self, fields: Dict):

        # Fields need to be deep copied so the same field objects aren't shared
        # across all message instances of the same type.
        self._fields = deepcopy(fields)  # maps field names to field objects
        for field in self._fields.values():
            field._parent_message = self
        self._parent_field = None

    def __repr__(self):
        """Return a short string representation of the message"""
        return f"<{type(self).__name__}: {self.render()}>"

    @staticmethod
    def _create_setter(name, field):
        """Used for dynamically creating setters for fields of subclasses."""

        def set_field(self, value):
            """
            Used as the setter function for all writable fields in a message. fields of all other
            types will *not* have an associated setter.

            When this function is called, all auto-update fields of the field's parent message
            will be updated since they might rely on this field.

            :raises: InvalidFieldDataException if the data being set is not valid for this field.
            """
            if not field.value_is_valid(value):
                raise InvalidFieldDataException(f"{value} is not a valid value for {field}")
            self._fields[name].value = value
            self.update_fields()

        return set_field

    @staticmethod
    def _create_getter(name):
        """Used for dynamically creating getters for field properties of subclasses."""

        def get_field(self):
            """
            Used as the getter function for all fields in a message regardless of the field type.

            If this is for a nested-field, a new object of the field context type will be
            instantiated and returned. Else, the Field object belonging to this instance will be
            returned.
            """
            field = self._fields[name]
            if field.context is not None:
                msg = field.value
                msg._parent_field = field
                return msg
            return field

        return get_field

    @property
    def context(self):
        """
        If this Message object belongs to a nested field, the context of the field is returned.
        Else, the class of the message is returned
        """
        if self._parent_field is not None:
            return self._parent_field.context
        return type(self)

    @context.setter
    def context(self, context):
        """
        If this Message object belongs to a nested field, set the context of the field.
        Else, raise an exception.
        """
        assert self._parent_field is not None
        self._parent_field.context = context

    def update_fields(self) -> None:
        """
        Updates all auto-update fields.

        :raises: CircularDependencyException if auto-update fields depend on each other.
        """
        # Update all auto-update fields
        auto_update_fields = [f for f in self._fields.values() if f.is_auto_updated]

        for field in auto_update_fields:
            field._value = None  # erase old values

        # NOTE: This is inefficient because auto-update fields can depend on each other.
        #       This could be improved by forming an update order from the dependency graph.
        for i in range(len(auto_update_fields)):
            for field in auto_update_fields:
                field.value = field.value_updater(
                    *[self._fields[arg] for arg in inspect.getfullargspec(field.value_updater)[0]]
                )

        # Propagate updates to parents
        if self._parent_field is not None:
            if self._parent_field._parent_message is not None:
                self._parent_field._parent_message.update_fields()

    def render(self, fmt=Field.Format.Hex, pad_to_length=0) -> str:
        """Renders entire field object as a hexadecimal value."""
        pad_to_length = pad_to_length if pad_to_length > 0 else math.ceil(len(self) / math.log2(fmt.value))
        bin_data = f"b{''.join([data.render(fmt=Field.Format.Bin)[1:] for data in self._fields.values()])}"
        return Field.render_value(value=bin_data, fmt=fmt, pad_to_length=pad_to_length)

    def get_field_name_mapping(self, expand_nested=False):
        """TODO: Verify this function works"""
        fields = {}

        def recursive_helper(cur_name, cur_field):
            if not cur_field.context:
                fields[cur_name] = cur_field
                return

            for n, f in cur_field._nested_msg._fields.items():
                recursive_helper(f"{cur_name}.{n}", f)

        for name, field in self._fields.items():
            if expand_nested and field.context:
                recursive_helper(name, field)
            else:
                fields[name] = field

        return fields

    def get_context_mapping(self, expand_nested=True):
        """TODO: Verify this function works"""
        fields = {}

        def recursive_helper(cur_name, cur_msg):
            if cur_msg is None:
                fields[cur_name] = None
                return
            fields[cur_name] = type(cur_msg)
            for n, f in cur_msg._fields.items():
                recursive_helper(f"{cur_name}.{n}", f._nested_msg)

        for name, field in self._fields.items():
            if expand_nested and field.context:
                recursive_helper(name, field._nested_msg)
            else:
                fields[name] = field.context

        return fields

    def render_table(self, formats=(Field.Format.Hex, Field.Format.Bin), expand_nested=False) -> str:
        """
        Renders the Message object as an ASCII table. The first column specifies the name of each field
        and each subsequent column contains each field rendered in a specific format. The formats that
        fields are rendered as is dictated by the value of the `formats` parameter.
        """

        title = type(self).__name__

        fields = self.get_field_name_mapping(expand_nested)

        if not formats:
            header = [["Field Name", "Value"]]
            field_values = [[name, field.render()] for name, field in fields.items()]

        else:
            header = [["Field Name"] + [fmt.name for fmt in formats]]
            field_values = []
            for name, field in fields.items():
                field_values.append([name] + [field.render(fmt=fmt) for fmt in formats])

        table_instance = AsciiTable(header + field_values, title)
        return table_instance.table

    def compare_tables(self, other_message, formats=None, expand_nested=True):
        """
        Constructs an ASCII representation of the message comparison. The tables for each
        message are displayed side-by-side. Fields that differ are denoted by `!=` between
        the tables in the corresponding row and fields that are equivalent are denoted
        by `==` in the same manner.
        """

        if type(self) != type(other_message):
            raise ConflictingContextsException(
                f"Cannot render table comparison of message types {type(self)} and {type(other_message)}"
            )

        if expand_nested:
            my_contexts = self.get_context_mapping(expand_nested)
            other_contexts = other_message.get_context_mapping(expand_nested)

            if my_contexts != other_contexts:
                raise ConflictingContextsException(f"{my_contexts} != {other_contexts}")

        my_table = self.render_table(formats=formats, expand_nested=expand_nested).split("\n")
        other_table = other_message.render_table(formats=formats, expand_nested=expand_nested).split("\n")

        comps = {}
        counter = 3
        self_fields = self.get_field_name_mapping(expand_nested).values()
        other_fields = other_message.get_field_name_mapping(expand_nested).values()
        for field1, field2 in zip(self_fields, other_fields):
            comps[counter] = field1 == field2
            counter += 1

        counter = 0
        comparison_str = ""
        for my_line, other_line, comp in zip(my_table, other_table, [None] * 3 + list(comps.values()) + [None]):
            comp_table = {
                None: "  ",
                True: "==",
                False: "!=",
            }
            comparison_str += f"{my_line}  {comp_table[comp]}  {other_line}\n"
            counter += 1

        return False in comps, comparison_str

    def __len__(self):
        """Returns the total number of bits in the message."""
        return type(self).bit_length

    def __eq__(self, other):
        """
        Return True if all fields in the message are equal and false otherwise.

        Can also compare to a string value.
        """
        if isinstance(other, str):
            other = Field.render_value(value=other, fmt=Field.Format.Bin, pad_to_length=len(self))
            return self.render(fmt=Field.Format.Bin, pad_to_length=len(self)) == other
        if type(self) != type(other):
            return False
        for name in self._fields:
            if self._fields[name] != other._fields[name]:
                return False
        return True

    def update(self, data):
        """
        Same as the Message.from_data method except a new object is not constructed.
        All fields of the message are updated with the new data.
        """
        new_msg = type(self).from_data(data)
        for name, field in new_msg._fields.items():
            self._fields[name].value = field.render()

    @classmethod
    def from_data(cls, data):
        """
        Constructs an object of type cls given an entire hex message.

        #TODO: take a context tree as a parameter to allow construction of nested messages

        :raises: InvalidDataFormatException if the object could not be created due to issues with the format of the data
        """

        # 1. Convert the data to binary
        binary_data = Field.render_value(
            value=data, fmt=Field.Format.Bin, pad_to_length=cls.bit_length, check_length=True
        )[1:]

        # 2. chunk into fields
        writable_field_data = {}
        for fieldname, field in cls.format.items():
            field_data = f"b{binary_data[:len(field)]}"
            if field.is_writable:
                writable_field_data[fieldname] = field_data
            elif field.is_auto_updated:
                pass
            else:
                if field.value != field_data:
                    raise InvalidDataFormatException(
                        f"The data '{field_data}' does not match with constant field {fieldname}."
                    )
            binary_data = binary_data[len(field) :]

        # 3. Construct a new message providing data only for writable fields.
        return cls(**writable_field_data)
