"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Portfolio = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("@aws-cdk/core");
const association_manager_1 = require("./private/association-manager");
const util_1 = require("./private/util");
const validation_1 = require("./private/validation");
const servicecatalog_generated_1 = require("./servicecatalog.generated");
class PortfolioBase extends cdk.Resource {
    constructor() {
        super(...arguments);
        this.associatedPrincipals = new Set();
    }
    /**
     * (experimental) Associate portfolio with an IAM Role.
     *
     * @experimental
     */
    giveAccessToRole(role) {
        this.associatePrincipal(role.roleArn, role.node.addr);
    }
    /**
     * (experimental) Associate portfolio with an IAM User.
     *
     * @experimental
     */
    giveAccessToUser(user) {
        this.associatePrincipal(user.userArn, user.node.addr);
    }
    /**
     * (experimental) Associate portfolio with an IAM Group.
     *
     * @experimental
     */
    giveAccessToGroup(group) {
        this.associatePrincipal(group.groupArn, group.node.addr);
    }
    /**
     * (experimental) Associate portfolio with the given product.
     *
     * @experimental
     */
    addProduct(product) {
        association_manager_1.AssociationManager.associateProductWithPortfolio(this, product);
    }
    /**
     * (experimental) Initiate a portfolio share with another account.
     *
     * @experimental
     */
    shareWithAccount(accountId, options = {}) {
        const hashId = this.generateUniqueHash(accountId);
        new servicecatalog_generated_1.CfnPortfolioShare(this, `PortfolioShare${hashId}`, {
            portfolioId: this.portfolioId,
            accountId: accountId,
            shareTagOptions: options.shareTagOptions,
            acceptLanguage: options.messageLanguage,
        });
    }
    /**
     * (experimental) Add a Resource Update Constraint.
     *
     * @experimental
     */
    constrainTagUpdates(product, options = {}) {
        association_manager_1.AssociationManager.constrainTagUpdates(this, product, options);
    }
    /**
     * Associate a principal with the portfolio.
     * If the principal is already associated, it will skip.
     */
    associatePrincipal(principalArn, principalId) {
        if (!this.associatedPrincipals.has(principalArn)) {
            const hashId = this.generateUniqueHash(principalId);
            new servicecatalog_generated_1.CfnPortfolioPrincipalAssociation(this, `PortolioPrincipalAssociation${hashId}`, {
                portfolioId: this.portfolioId,
                principalArn: principalArn,
                principalType: 'IAM',
            });
            this.associatedPrincipals.add(principalArn);
        }
    }
}
/**
 * (experimental) A Service Catalog portfolio.
 *
 * @experimental
 */
class Portfolio extends PortfolioBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.validatePortfolioProps(props);
        this.portfolio = new servicecatalog_generated_1.CfnPortfolio(this, 'Resource', {
            displayName: props.displayName,
            providerName: props.providerName,
            description: props.description,
            acceptLanguage: props.messageLanguage,
        });
        this.portfolioId = this.portfolio.ref;
        this.portfolioArn = cdk.Stack.of(this).formatArn({
            service: 'servicecatalog',
            resource: 'portfolio',
            resourceName: this.portfolioId,
        });
    }
    /**
     * (experimental) Creates a Portfolio construct that represents an external portfolio.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param portfolioArn the Amazon Resource Name of the existing portfolio.
     * @experimental
     */
    static fromPortfolioArn(scope, id, portfolioArn) {
        const arn = cdk.Stack.of(scope).splitArn(portfolioArn, cdk.ArnFormat.SLASH_RESOURCE_NAME);
        const portfolioId = arn.resourceName;
        if (!portfolioId) {
            throw new Error('Missing required Portfolio ID from Portfolio ARN: ' + portfolioArn);
        }
        class Import extends PortfolioBase {
            constructor() {
                super(...arguments);
                this.portfolioArn = portfolioArn;
                this.portfolioId = portfolioId;
            }
            generateUniqueHash(value) {
                return util_1.hashValues(this.portfolioArn, value);
            }
        }
        return new Import(scope, id, {
            environmentFromArn: portfolioArn,
        });
    }
    /**
     * (experimental) Create a unique id based off the L1 CfnPortfolio or the arn of an imported portfolio.
     *
     * @experimental
     */
    generateUniqueHash(value) {
        return util_1.hashValues(cdk.Names.nodeUniqueId(this.portfolio.node), value);
    }
    validatePortfolioProps(props) {
        validation_1.InputValidator.validateLength(this.node.path, 'portfolio display name', 1, 100, props.displayName);
        validation_1.InputValidator.validateLength(this.node.path, 'portfolio provider name', 1, 50, props.providerName);
        validation_1.InputValidator.validateLength(this.node.path, 'portfolio description', 0, 2000, props.description);
    }
}
exports.Portfolio = Portfolio;
_a = JSII_RTTI_SYMBOL_1;
Portfolio[_a] = { fqn: "@aws-cdk/aws-servicecatalog.Portfolio", version: "1.114.0" };
//# sourceMappingURL=data:application/json;base64,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