#!/usr/bin/env python3
# -*- coding: utf-8 -*-
from __future__ import annotations
from typing import List

from refinery.units.crypto.cipher import (
    StandardBlockCipherUnit,
)
from refinery.lib.crypto import (
    BlockCipher,
    BlockCipherFactory,
    BufferType,
)


_SBOX_ = (
    0x4, 0xA, 0x9, 0x2, 0xD, 0x8, 0x0, 0xE, 0x6, 0xB, 0x1, 0xC, 0x7, 0xF, 0x5, 0x3,
    0xE, 0xB, 0x4, 0xC, 0x6, 0xD, 0xF, 0xA, 0x2, 0x3, 0x8, 0x1, 0x0, 0x7, 0x5, 0x9,
    0x5, 0x8, 0x1, 0xD, 0xA, 0x3, 0x4, 0x2, 0xE, 0xF, 0xC, 0x7, 0x6, 0x0, 0x9, 0xB,
    0x7, 0xD, 0xA, 0x1, 0x0, 0x8, 0x9, 0xF, 0xE, 0x4, 0x6, 0xC, 0xB, 0x2, 0x5, 0x3,
    0x6, 0xC, 0x7, 0x1, 0x5, 0xF, 0xD, 0x8, 0x4, 0xA, 0x9, 0xE, 0x0, 0x3, 0xB, 0x2,
    0x4, 0xB, 0xA, 0x0, 0x7, 0x2, 0x1, 0xD, 0x3, 0x6, 0x8, 0x5, 0x9, 0xC, 0xF, 0xE,
    0xD, 0xB, 0x4, 0x1, 0x3, 0xF, 0x5, 0x9, 0x0, 0xA, 0xE, 0x7, 0x6, 0x8, 0x2, 0xC,
    0x1, 0xF, 0xD, 0x0, 0x5, 0x7, 0xA, 0x4, 0x9, 0x2, 0x3, 0xE, 0x6, 0xB, 0x8, 0xC,
)

_SBOX = (
    0xC, 0x4, 0x6, 0x2, 0xA, 0x5, 0xB, 0x9, 0xE, 0x8, 0xD, 0x7, 0x0, 0x3, 0xF, 0x1,
    0x6, 0x8, 0x2, 0x3, 0x9, 0xA, 0x5, 0xC, 0x1, 0xE, 0x4, 0x7, 0xB, 0xD, 0x0, 0xF,
    0xB, 0x3, 0x5, 0x8, 0x2, 0xF, 0xA, 0xD, 0xE, 0x1, 0x7, 0x4, 0xC, 0x9, 0x6, 0x0,
    0xC, 0x8, 0x2, 0x1, 0xD, 0x4, 0xF, 0x6, 0x7, 0x0, 0xA, 0x5, 0x3, 0xE, 0x9, 0xB,
    0x7, 0xF, 0x5, 0xA, 0x8, 0x1, 0x6, 0xD, 0x0, 0x9, 0x3, 0xE, 0xB, 0x4, 0x2, 0xC,
    0x5, 0xD, 0xF, 0x6, 0x9, 0x2, 0xC, 0xA, 0xB, 0x7, 0x8, 0x1, 0x4, 0x3, 0xE, 0x0,
    0x8, 0xE, 0x2, 0x5, 0x6, 0x9, 0x1, 0xC, 0xF, 0x4, 0xB, 0x0, 0xD, 0xA, 0x3, 0x7,
    0x1, 0x7, 0xE, 0xD, 0x0, 0x5, 0x8, 0x3, 0x4, 0xF, 0xA, 0x6, 0x9, 0xC, 0xB, 0x2,
)


def _F(A: int, K: int):
    T = A + K & 0xFFFFFFFF
    F = ((_SBOX[0x00 | ((T >> 0x00) & 0xF)] << 0x00)
       | (_SBOX[0x10 | ((T >> 0x04) & 0xF)] << 0x04)
       | (_SBOX[0x20 | ((T >> 0x08) & 0xF)] << 0x08)
       | (_SBOX[0x30 | ((T >> 0x0C) & 0xF)] << 0x0C)
       | (_SBOX[0x40 | ((T >> 0x10) & 0xF)] << 0x10)
       | (_SBOX[0x50 | ((T >> 0x14) & 0xF)] << 0x14)
       | (_SBOX[0x60 | ((T >> 0x18) & 0xF)] << 0x18)
       | (_SBOX[0x70 | ((T >> 0x1C) & 0xF)] << 0x1C))
    return (F >> 21) | ((F & 0x1FFFFF) << 11)


_BO = 'little'


class GOST(BlockCipher):

    _key_data: List[int]

    block_size = 8
    key_size = frozenset({32})

    def block_decrypt(self, block) -> BufferType:
        A, B = (
            int.from_bytes(block[:4], _BO),
            int.from_bytes(block[4:], _BO))
        K = self._key_data
        for i in range(8):
            B, A = A, B ^ _F(A, K[i])
        for i in range(24 - 1, -1, -1):
            B, A = A, B ^ _F(A, K[i % 8])
        return A.to_bytes(4, _BO) + B.to_bytes(4, _BO)

    def block_encrypt(self, block) -> BufferType:
        A, B = (
            int.from_bytes(block[:4], _BO),
            int.from_bytes(block[4:], _BO))
        K = self._key_data
        for i in range(24):
            A, B = B, A ^ _F(B, K[i % 8])
        for i in range(8 - 1, -1, -1):
            A, B = B, A ^ _F(B, K[i])
        return A.to_bytes(4, _BO) + B.to_bytes(4, _BO)

    @property
    def key(self):
        return self._key_data

    @key.setter
    def key(self, key: bytes):
        key_data = []
        for k in range(0, 32, 4):
            key_data.append(int.from_bytes(key[k:k + 4], _BO))
        self._key_data = key_data


class gost(StandardBlockCipherUnit, cipher=BlockCipherFactory(GOST)):
    """
    GOST encryption and decryption.
    """
    pass
