"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Server = void 0;
const child_process_1 = require("child_process");
const fs_1 = require("fs");
const http_1 = require("http");
const os_1 = require("os");
const path_1 = require("path");
/**
 * @stability stable
 */
class Server {
    /**
     * @stability stable
     */
    constructor(props) {
        this.appCommand = props.appCommand;
        this.server = http_1.createServer((req, res) => this.handleRequest(req, res).catch(e => {
            console.error('server error: ', e);
            res.statusCode = 500;
            res.write(e.message);
            res.end();
        }));
        this.tmpdir = fs_1.mkdtempSync(path_1.join(os_1.tmpdir(), 'cdk8s-operator-'));
    }
    /**
     * Starts HTTP server.
     *
     * @param port The port to listen to.
     * @stability stable
     */
    async listen(port) {
        var _a;
        const lport = (_a = port !== null && port !== void 0 ? port : process.env.PORT) !== null && _a !== void 0 ? _a : 0;
        return new Promise((ok, ko) => {
            this.server.listen(lport, () => {
                const addr = this.server.address();
                if (typeof (addr) === 'string') {
                    throw new Error(`cannot determine port from server address ${addr}`);
                }
                return ok(addr.port);
            });
            this.server.on('error', err => ko(err));
        });
    }
    /**
     * Stop server.
     *
     * @stability stable
     */
    close() {
        this.server.close();
    }
    async handleRequest(req, res) {
        const inputfile = await this.writeInputFile(req);
        const child = child_process_1.spawn(this.appCommand, [inputfile], {
            stdio: ['ignore', 'pipe', 'pipe'],
            shell: true,
        });
        const stderr = new Array();
        res.setHeader('Content-Type', 'application/json');
        // stdout should go directly to the response
        child.stdout.on('data', chunk => {
            process.stderr.write('output: ' + chunk);
            res.write(chunk);
        });
        // for stderr: write to server terminal and only send back if we exited with a non-zero
        child.stderr.on('data', chunk => {
            process.stderr.write(chunk);
            stderr.push(chunk);
        });
        // will be caused by the async handler and 500 will be returned.
        child.on('error', err => {
            throw err;
        });
        child.on('exit', code => {
            if (code !== 0) {
                res.statusCode = 500;
                for (const c of stderr) {
                    res.write(c);
                }
                res.end();
            }
            // success
            return res.end();
        });
    }
    async writeInputFile(req) {
        return new Promise((ok, ko) => {
            const inputfile = path_1.join(this.tmpdir, `input-${Math.round(Math.random() * 999999)}.json`);
            const input = fs_1.createWriteStream(inputfile);
            req.pipe(input);
            input.on('close', () => {
                try {
                    const inputJson = JSON.parse(fs_1.readFileSync(inputfile, 'utf-8'));
                    console.error(`input: ${JSON.stringify(inputJson)}`);
                    return ok(inputfile);
                }
                catch (e) {
                    return ko(new Error(`unable to parse request body as JSON: ${e}`));
                }
            });
            req.on('error', err => ko(err.message));
        });
    }
}
exports.Server = Server;
//# sourceMappingURL=data:application/json;base64,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